---
id: DE-003
slug: cli-integration-for-frontmatter-schema-show-commands
name: Delta - CLI Integration for Frontmatter Schema Show Commands
created: '2025-11-02'
updated: '2025-11-02'
status: completed
kind: delta
aliases: []
tags: ["cli"]
relations:
  - type: implements
    target: PROD-004.FR-004
context_inputs:
  - type: decision
    id: FRONTMATTER_MIGRATION_COMPLETE
applies_to:
  specs:
    - PROD-004
  requirements:
    - PROD-004.FR-004
outcome_summary: >-
  Enable agents and developers to query frontmatter JSON Schema via CLI commands.
  All 13 frontmatter kinds accessible through `spec-driver schema show frontmatter.*`
  commands with multiple output formats (json-schema, yaml-example).
risk_register:
  - id: DE-003.RISK-01
    title: CLI command naming conflicts with existing commands
    exposure: change
    likelihood: low
    impact: low
    mitigation: Follow existing CLI patterns, check for conflicts before implementation
  - id: DE-003.RISK-02
    title: JSON Schema output format incompatibility with agents
    exposure: change
    likelihood: low
    impact: medium
    mitigation: Test with JSON Schema validators, follow JSON Schema Draft 2020-12
---

# DE-003 – CLI Integration for Frontmatter Schema Show Commands

```yaml supekku:delta.relationships@v1
schema: supekku.delta.relationships
version: 1
delta: DE-003
revision_links:
  introduces: []
  supersedes: []
specs:
  primary:
    - PROD-004
  collaborators: []
requirements:
  implements:
    - PROD-004.FR-004
  updates: []
  verifies: []
phases:
  - id: IP-003.PHASE-01
```

## 1. Summary & Context

- **Product Spec**: [PROD-004](../../../specify/product/PROD-004/PROD-004.md) – Frontmatter Metadata Validation
- **Implementation Plan**: [IP-003](./IP-003.md) – CLI integration execution plan
- **Change Drivers**:
  - Frontmatter metadata migration 100% complete (260 tests passing)
  - PROD-004.FR-004 requirement: CLI must support `schema show frontmatter.{kind}` commands
  - Agent need for JSON Schema to write valid frontmatter YAML
  - Developer need for schema discovery and validation

## 2. Motivation

**Current State**:
- All 13 frontmatter schemas implemented with metadata-driven validation
- JSON Schema generation working (`metadata_to_json_schema()` function available)
- No CLI access to schemas - agents and developers must read Python code to understand frontmatter structure

**Pain Points**:
- Agents writing frontmatter YAML lack schema documentation, leading to trial-and-error validation
- Developers extending frontmatter must inspect Python metadata definitions
- No self-service schema discovery mechanism
- JSON Schema benefits (agent understanding, IDE integration) not realized

**Desired State**:
- CLI commands expose all 13 frontmatter schemas in JSON Schema format
- Agents query schema before writing frontmatter: `spec-driver schema show frontmatter.delta --format=json-schema`
- Developers get YAML examples: `spec-driver schema show frontmatter.prod --format=yaml-example`
- Foundation for future IDE plugins, CI linters, external tooling

**Supporting Evidence**:
- PROD-004 Section 2 (Stakeholders & Journeys): Agent queries schema before writing frontmatter
- PROD-004 FR-004: CLI MUST support `schema show frontmatter.{kind}` commands
- Frontmatter migration completion: All schemas ready, JSON Schema generation tested

## 3. Scope & Objectives

**Primary Outcomes**:

1. **CLI Command Structure** (implements PROD-004.FR-004)
   - Add `schema show frontmatter.<kind>` command for all 13 frontmatter kinds
   - Support output formats: `--format=json-schema` (default), `--format=yaml-example`
   - Return formatted output suitable for agent consumption

2. **Schema Discovery**
   - List all available frontmatter kinds: `spec-driver schema list frontmatter`
   - Help text documents available kinds and format options

3. **Agent-Friendly Output**
   - JSON Schema output complies with JSON Schema Draft 2020-12
   - YAML examples demonstrate valid minimal frontmatter structure
   - Error messages guide users to correct kind names

**Operational Constraints**:
- Must follow existing CLI patterns (consistency with `spec-driver` commands)
- No breaking changes to existing CLI commands
- Fast execution (schema generation <100ms per PROD-004.NF-002)
- Zero external dependencies beyond existing project requirements

**Dependencies**:
- ✅ Frontmatter metadata migration complete (260 tests passing)
- ✅ `FRONTMATTER_METADATA_REGISTRY` available in `supekku.scripts.lib.core.frontmatter_metadata`
- ✅ `metadata_to_json_schema()` function tested and working
- Existing CLI infrastructure (`supekku/cli/` using Typer)

## 4. Out of Scope

**Explicitly Deferred**:
- **Schema validation command**: Not implementing `spec-driver schema validate frontmatter.yaml` (future delta)
- **Migration to metadata validator**: Gradual replacement of `validate_frontmatter()` calls (PROD-004 Phase 6E, future work)
- **Deprecation of old validator**: Happens after full ecosystem migration (future work)
- **External tool integration**: IDE plugins, CI linters (enabled by this work, implemented separately)
- **Custom schema overrides**: Users cannot provide custom frontmatter schemas (may be future enhancement)
- **Schema versioning**: All schemas currently v1, versioning strategy deferred

**Adjacent Work**:
- Policy/Standard frontmatter already implemented (completed in earlier phase)
- Cross-artifact validation, relation target validation (separate concern)

## 5. Approach Overview

### System Touchpoints

**Packages Modified**:
- `supekku/cli/` - Add new CLI commands for schema show
- `supekku/scripts/lib/formatters/` - Add schema formatters if needed

**Packages Referenced** (no modifications):
- `supekku/scripts/lib/core/frontmatter_metadata/` - Registry and metadata definitions
- `supekku/scripts/lib/blocks/metadata/` - JSON Schema generation

### Key Changes

**1. New CLI Command Structure** (follows existing patterns)

File: `supekku/cli/schema.py` (new or extend existing)

```python
import typer
from typing import Optional
from supekku.scripts.lib.core.frontmatter_metadata import (
  FRONTMATTER_METADATA_REGISTRY,
  get_frontmatter_metadata,
)
from supekku.scripts.lib.blocks.metadata import metadata_to_json_schema
import json
import yaml

app = typer.Typer()

@app.command("list")
def list_frontmatter_kinds():
  """List all available frontmatter kinds."""
  kinds = sorted(FRONTMATTER_METADATA_REGISTRY.keys())
  for kind in kinds:
    schema = FRONTMATTER_METADATA_REGISTRY[kind]
    typer.echo(f"{kind:20s} - {schema.description}")

@app.command("show")
def show_frontmatter_schema(
  kind: str,
  format: str = typer.Option("json-schema", help="Output format: json-schema or yaml-example")
):
  """Show frontmatter schema for a specific kind.

  Examples:
    spec-driver schema show frontmatter.prod --format=json-schema
    spec-driver schema show frontmatter.delta --format=yaml-example
  """
  # Strip "frontmatter." prefix if provided
  if kind.startswith("frontmatter."):
    kind = kind[12:]

  # Get metadata
  if kind not in FRONTMATTER_METADATA_REGISTRY:
    typer.echo(f"Error: Unknown frontmatter kind: {kind}", err=True)
    typer.echo(f"Available kinds: {', '.join(sorted(FRONTMATTER_METADATA_REGISTRY.keys()))}", err=True)
    raise typer.Exit(1)

  metadata = get_frontmatter_metadata(kind)

  if format == "json-schema":
    # Generate and output JSON Schema
    schema = metadata_to_json_schema(metadata)
    typer.echo(json.dumps(schema, indent=2))
  elif format == "yaml-example":
    # Output minimal YAML example
    if metadata.examples and len(metadata.examples) > 0:
      example = metadata.examples[0]  # Use first (minimal) example
      typer.echo(yaml.dump(example, default_flow_style=False, sort_keys=False))
    else:
      typer.echo(f"Error: No examples available for {kind}", err=True)
      raise typer.Exit(1)
  else:
    typer.echo(f"Error: Unknown format: {format}", err=True)
    typer.echo("Supported formats: json-schema, yaml-example", err=True)
    raise typer.Exit(1)
```

**Integration with main CLI**:

File: `supekku/cli/main.py` (or similar entry point)

```python
from supekku.cli import schema

app.add_typer(schema.app, name="schema", help="Schema inspection commands")
```

**2. Schema Output Formats**

**JSON Schema Format** (`--format=json-schema`):
- Direct output from `metadata_to_json_schema()`
- Complies with JSON Schema Draft 2020-12
- Includes: properties, required fields, types, patterns, enums, descriptions, examples
- Piped to `jq` for pretty-printing if needed

**YAML Example Format** (`--format=yaml-example`):
- Uses first example from `metadata.examples` (minimal example by convention)
- YAML formatted with proper indentation
- Demonstrates minimal valid frontmatter structure

**3. Help Text Updates**

```bash
spec-driver schema --help
# Should show:
#   list      List available frontmatter kinds
#   show      Show schema for specific frontmatter kind
```

```bash
spec-driver schema show --help
# Should show examples and format options
```

### Migration / Rollout Notes

**No Migration Needed**: New functionality, no existing behavior changed

**Rollout**:
1. Implement CLI commands
2. Add comprehensive tests
3. Update documentation (README, help text)
4. Announce to users/agents via changelog

**Testing Strategy**:
- CLI integration tests for all 13 frontmatter kinds
- JSON Schema validation tests (ensure output is valid JSON Schema)
- YAML example tests (ensure examples validate against schema)
- Error handling tests (unknown kinds, invalid formats)

## 6. Verification Strategy

**Requirements Coverage**:

- **PROD-004.FR-004**: CLI MUST support `schema show frontmatter.{kind}` commands
  - ✅ Verify: `spec-driver schema show frontmatter.prod --format=json-schema` returns valid JSON Schema
  - ✅ Verify: `spec-driver schema show frontmatter.delta --format=yaml-example` returns valid YAML
  - ✅ Verify: All 13 frontmatter kinds accessible via CLI

**Planned Verification Artefacts**:

- **VT-CLI-001**: CLI command integration tests
  - Test all 13 frontmatter kinds return valid JSON Schema
  - Test YAML examples validate against generated schemas
  - Test error handling for unknown kinds
  - Test format option validation

- **VT-CLI-002**: End-to-end agent workflow tests
  - Agent queries schema: `spec-driver schema show frontmatter.delta --format=json-schema`
  - Agent uses schema to construct valid frontmatter YAML
  - Frontmatter validates successfully
  - Zero trial-and-error iterations needed

- **VT-CLI-003**: Performance tests
  - Schema generation completes in <100ms (PROD-004.NF-002)
  - CLI response time acceptable for agent workflows

**Acceptance Criteria**:

✅ **Must Pass Before Marking Delta Complete**:
1. All 13 frontmatter kinds accessible via `spec-driver schema show frontmatter.<kind>`
2. JSON Schema output validates against JSON Schema Draft 2020-12 specification
3. YAML examples validate against their corresponding JSON Schemas
4. Error messages guide users to correct syntax
5. Help text documents all available kinds and formats
6. Zero ruff warnings, pylint >8.0
7. All tests passing (existing + new CLI tests)
8. Documentation updated (README, user guide if applicable)

## 7. Risks & Mitigations

**Risk**: CLI command naming conflicts with existing commands
- **Likelihood**: Low
- **Impact**: Low (breaking change to CLI interface)
- **Mitigation**: Review existing CLI structure before implementation, follow established patterns, test for conflicts

**Risk**: JSON Schema output format incompatible with agents/validators
- **Likelihood**: Low
- **Impact**: Medium (defeats purpose if agents can't consume)
- **Mitigation**: Validate output against JSON Schema Draft 2020-12, test with online validators, include examples in schema output

**Risk**: Performance regression for schema generation
- **Likelihood**: Low
- **Impact**: Low (slightly slower CLI, but cached after first call)
- **Mitigation**: Schema generation already tested <100ms in metadata migration, cache schemas if needed

**Risk**: Missing examples for some frontmatter kinds
- **Likelihood**: Low (all current schemas have examples)
- **Impact**: Medium (YAML example format fails)
- **Mitigation**: Verify all schemas have at least one example, fall back to minimal base fields if needed

## 8. Follow-ups & Tracking

**Future Phases / Deltas**:

1. **Schema Validation Command** (future delta)
   - `spec-driver schema validate <file> --kind=delta`
   - Validates frontmatter YAML against schema
   - Provides detailed error messages

2. **Gradual Migration to Metadata Validator** (PROD-004 Phase 6E)
   - Replace `validate_frontmatter()` calls with `validate_frontmatter_metadata()`
   - Opt-in per code path, no breaking changes
   - Tracked in PROD-004 implementation plan

3. **Deprecate Imperative Validator** (PROD-004 final phase)
   - After full ecosystem migration
   - Deprecation warnings, then removal
   - Tracked in PROD-004 backlog

**Backlog Items**: None at this time

**Open Decisions / Questions**:

1. **Command naming convention**: Use `schema show frontmatter.prod` or `schema show-frontmatter prod`?
   - **Recommendation**: `schema show frontmatter.prod` (follows common CLI patterns like `git log origin/main`)
   - **Decision needed**: Review with team for consistency

2. **Output to stdout or file**: Should `--output=file.json` option be supported?
   - **Recommendation**: Start with stdout only (users can redirect), add file output later if needed
   - **Decision**: Defer file output to future enhancement

3. **Schema versioning in output**: Include schema version in JSON Schema output?
   - **Recommendation**: Yes, include `"$id": "supekku.frontmatter.{kind}.v1"` in output
   - **Decision needed**: Confirm versioning strategy

## 9. Implementation Notes

### Development Environment Setup

```bash
# Verify frontmatter metadata available
uv run python -c "
from supekku.scripts.lib.core.frontmatter_metadata import FRONTMATTER_METADATA_REGISTRY
print(f'Registry has {len(FRONTMATTER_METADATA_REGISTRY)} entries')
"

# Test JSON Schema generation
uv run python -c "
from supekku.scripts.lib.core.frontmatter_metadata import PROD_FRONTMATTER_METADATA
from supekku.scripts.lib.blocks.metadata import metadata_to_json_schema
import json
schema = metadata_to_json_schema(PROD_FRONTMATTER_METADATA)
print(json.dumps(schema, indent=2))
"
```

### Implementation Checklist

**Before Starting**:
- [ ] Review existing CLI structure in `supekku/cli/`
- [ ] Identify CLI entry point and command registration pattern
- [ ] Check for existing `schema` command group
- [ ] Review Typer documentation for best practices

**Implementation Steps**:
1. [ ] Create `supekku/cli/schema.py` (or extend existing)
2. [ ] Implement `list` command (list all frontmatter kinds)
3. [ ] Implement `show` command (show schema for specific kind)
4. [ ] Add `--format` option (json-schema, yaml-example)
5. [ ] Integrate with main CLI app
6. [ ] Write CLI integration tests
7. [ ] Update help text and documentation
8. [ ] Run full test suite (ensure no regressions)
9. [ ] Lint (ruff, pylint)
10. [ ] Manual testing with all 13 frontmatter kinds

**Testing**:
- [ ] Test all 13 kinds with `--format=json-schema`
- [ ] Test all 13 kinds with `--format=yaml-example`
- [ ] Test error handling (unknown kind, invalid format)
- [ ] Test help text (`--help`)
- [ ] Validate JSON Schema output with online validator
- [ ] Validate YAML examples against schemas

**Documentation**:
- [ ] Update README with new commands
- [ ] Add examples to user documentation
- [ ] Update CHANGELOG
- [ ] Update PROD-004 status (FR-004 implemented)

### Reference Documentation

**Existing Patterns**:
- Review `supekku/cli/list.py` for list command patterns
- Review `supekku/cli/show.py` for show command patterns (if exists)
- Review `supekku/cli/adr.py` for ADR CLI patterns

**JSON Schema Resources**:
- JSON Schema Draft 2020-12: https://json-schema.org/draft/2020-12/schema
- JSON Schema Validator: https://www.jsonschemavalidator.net/

**Metadata Engine**:
- `supekku/scripts/lib/blocks/metadata/` - Metadata definitions
- `metadata_to_json_schema()` function - JSON Schema generation
- `FRONTMATTER_METADATA_REGISTRY` - All available schemas

### Example CLI Usage

```bash
# List all available frontmatter kinds
spec-driver schema list frontmatter

# Show JSON Schema for product spec
spec-driver schema show frontmatter.prod --format=json-schema

# Show YAML example for delta
spec-driver schema show frontmatter.delta --format=yaml-example

# Pipe to jq for formatting
spec-driver schema show frontmatter.spec --format=json-schema | jq .

# Save to file
spec-driver schema show frontmatter.design_revision --format=json-schema > design_revision_schema.json

# Use in agent workflow
SCHEMA=$(spec-driver schema show frontmatter.prod --format=json-schema)
# Agent parses SCHEMA and constructs valid frontmatter
```

### Agent Integration Pattern

**Agent Workflow**:
1. Agent needs to create delta artifact
2. Agent queries schema: `spec-driver schema show frontmatter.delta --format=json-schema`
3. Agent parses JSON Schema to understand required/optional fields
4. Agent constructs frontmatter YAML conforming to schema
5. Agent writes frontmatter to delta file
6. Validation passes on first attempt (zero trial-and-error)

**Example Agent Prompt Enhancement**:
```
Before writing frontmatter YAML, query the schema:
`spec-driver schema show frontmatter.{kind} --format=json-schema`

Parse the JSON Schema to identify:
- Required fields (in "required" array)
- Field types and patterns
- Enum values for constrained fields
- Nested object structures

Construct frontmatter YAML matching the schema exactly.
```

---

## Next Steps for Implementation

1. **Create implementation plan** in `IP-003.md` with phases
2. **Implement CLI commands** following checklist above
3. **Write comprehensive tests** (VT-CLI-001, VT-CLI-002, VT-CLI-003)
4. **Update documentation** (README, help text, CHANGELOG)
5. **Mark PROD-004.FR-004 as implemented** upon completion

**Estimated Effort**: 4-6 hours
- CLI implementation: 2-3 hours
- Tests: 1-2 hours
- Documentation: 1 hour

**Success Criteria**: All 13 frontmatter schemas accessible via CLI with JSON Schema and YAML example formats, comprehensive tests passing, documentation complete.
