---
id: DE-006
slug: standardize-cli-json-output-support
name: Delta - Standardize CLI JSON output support
created: '2025-11-02'
updated: '2025-11-02'
status: complete
kind: delta
aliases: []
tags: ["cli"]
relations:
  - type: resolves
    target: ISSUE-007
    nature: Fixes inconsistent --json support across CLI commands
applies_to:
  specs:
    - SPEC-003  # supekku/cli/create.py
    - SPEC-004  # supekku/cli/list.py
  requirements: []
context_inputs:
  - ISSUE-007 documentation and evidence
  - Existing --json implementation in create spec command
  - Agent workflow requirements from PROD-001 and PROD-002
outcome_summary: All CLI commands support consistent --json output for machine-readable results
risk_register:
  - id: RISK-001
    description: Breaking changes to existing text output parsers
    likelihood: low
    impact: medium
    mitigation: Add --json as opt-in flag; preserve default text output unchanged
    status: mitigated
  - id: RISK-002
    description: Inconsistent JSON schema across commands
    likelihood: medium
    impact: low
    mitigation: Define standard output format with common fields (id, path, status)
    status: mitigated
---

# DE-006 – Standardize CLI JSON output support

```yaml supekku:delta.relationships@v1
schema: supekku.delta.relationships
version: 1
delta: DE-006
revision_links:
  introduces: []
  supersedes: []
specs:
  primary:
    - SPEC-003  # supekku/cli/create.py
    - SPEC-004  # supekku/cli/list.py
  collaborators: []
requirements:
  implements: []
  updates: []
  verifies: []
phases:
  - IP-006.PHASE-01
```

## 1. Summary & Context

- **Product Spec(s)**: PROD-001 (Streamline spec creation), PROD-002 (Delta creation workflow) – both depend on `--json` for metadata extraction
- **Technical Spec(s)**:
  - SPEC-003 (supekku/cli/create.py) – has partial JSON support (spec only)
  - SPEC-004 (supekku/cli/list.py) – missing JSON support
  - SPEC-006 (supekku/cli/show.py) – missing JSON support
- **Implementation Plan**: [IP-006](./IP-006.md) – Single phase implementation
- **Change Drivers**:
  - ISSUE-007: CLI commands missing --json output support
  - Agent workflows broken by missing JSON output
  - Slash command `/supekku.specify` assumes `--json` available

## 2. Motivation

**Current Pain**:
- Agents expect `--json` output based on documentation but encounter errors
- `spec-driver list specs --json` fails despite being standard pattern
- `spec-driver create issue <title> --json` fails despite create spec supporting it
- Inconsistent CLI interface confuses users and blocks automation

**User Impact**:
- Agents cannot reliably extract metadata for workflows
- External tooling must parse brittle text output
- Slash commands like `/supekku.specify` fail during discovery
- Inconsistent experience across similar commands (create spec works, create issue doesn't)

**Desired State**:
- All `create`, `list`, and `show` commands support `--json` flag
- Consistent JSON output schema: `{"id": "...", "path": "...", "status": "...", ...}`
- Agent workflows complete without text parsing hacks
- Documentation matches actual implementation

## 3. Scope & Objectives

**Primary Outcomes**:
1. Add `--json` flag to all `create` subcommands (issue, problem, risk, etc.)
2. Add `--json` flag to all `list` commands (specs, issues, deltas, etc.)
3. Add `--json` flag to `show` commands where applicable
4. Standardize JSON output format across all commands

**Operational Constraints**:
- Zero breaking changes to existing text output
- `--json` is opt-in; default behavior unchanged
- Implementation must follow existing pattern from `create spec --json`
- All changes must pass existing test suite

**Dependencies**:
- None - standalone improvement

## 4. Out of Scope

- Adding `--json` to commands that don't return data (e.g., `sync`, `validate`)
- Changing default output format (text remains default)
- Adding new CLI commands or subcommands
- GraphQL/REST API endpoints (CLI-only change)

## 5. Approach Overview

**System Touchpoints**:
- `supekku/cli/create.py` - extend existing JSON support to all create subcommands
- `supekku/cli/list.py` - add JSON output for all list commands
- `supekku/cli/show.py` - add JSON output for show commands
- Test files: `supekku/cli/*_test.py` - add JSON output tests

**Key Changes**:

### 5.1 Standard JSON Output Format

All commands returning single items use:
```json
{
  "id": "ISSUE-007",
  "path": "/absolute/path/to/artifact.md",
  "kind": "issue",
  "status": "open",
  "created": "2025-11-02"
}
```

All commands returning lists use:
```json
{
  "count": 42,
  "items": [
    {"id": "...", "path": "...", ...},
    ...
  ]
}
```

### 5.2 Implementation Pattern

Follow existing pattern from `create.py`:
```python
@app.command("issue")
def create_issue(
  title: str,
  json_output: Annotated[
    bool,
    typer.Option("--json", help="Output result as JSON"),
  ] = False,
):
  # ... create logic ...

  if json_output:
    result = {
      "id": issue_id,
      "path": str(issue_path),
      "kind": "issue",
      "status": "open",
    }
    print(json.dumps(result, indent=2))
  else:
    # existing text output
    print(f"Issue created: {issue_id}\n{issue_path}")
```

### 5.3 Commands to Update

**create.py** (extend existing JSON support):
- `create issue` - add `--json` flag
- `create problem` - add `--json` flag
- `create risk` - add `--json` flag
- `create policy` - add `--json` flag (if exists)
- `create standard` - add `--json` flag (if exists)
- `create delta` - add `--json` flag (if exists)

**list.py** (add new JSON support):
- `list specs` - add `--json` flag
- `list issues` - add `--json` flag
- `list deltas` - add `--json` flag
- `list requirements` - add `--json` flag
- `list adrs` - add `--json` flag

**show.py** (add new JSON support):
- `show spec` - add `--json` flag (if applicable)
- Other show commands as needed

**Migration / Rollout Notes**:
- No migration needed - purely additive change
- `--json` is opt-in flag
- Default text output unchanged for backward compatibility
- Can be rolled out incrementally per command

## 6. Verification Strategy

**Requirements Coverage**:
- **Consistency**: All similar commands have same `--json` flag signature
- **Correctness**: JSON output matches documented schema
- **Backward compatibility**: Text output unchanged when `--json` not specified

**Planned Artifacts**:
- VT-CLI-JSON-001: Unit tests for each command's JSON output
- VT-CLI-JSON-002: Integration tests for agent workflow using JSON
- VT-CLI-JSON-003: Regression tests ensuring text output unchanged

**Acceptance Criteria**:
- [ ] All `create` subcommands support `--json` flag
- [ ] All `list` commands support `--json` flag
- [ ] JSON output follows standard schema (id, path, kind, status)
- [ ] Default text output unchanged
- [ ] All tests passing (including new JSON tests)
- [ ] Agent workflow from `/supekku.specify` works end-to-end
- [ ] ISSUE-007 resolved and closed

## 7. Risks & Mitigations

- **RISK-001**: Breaking changes to existing text output parsers
  - **Likelihood**: low
  - **Impact**: medium (affects external scripts)
  - **Mitigation**: Add --json as opt-in flag; preserve default text output unchanged

- **RISK-002**: Inconsistent JSON schema across commands
  - **Likelihood**: medium
  - **Impact**: low (confusion, rework)
  - **Mitigation**: Define standard output format with common fields (id, path, status); document schema

- **RISK-003**: Missing edge cases in JSON serialization
  - **Likelihood**: low
  - **Impact**: low (runtime errors)
  - **Mitigation**: Comprehensive test coverage; use standard json.dumps with default=str fallback

## 8. Follow-ups & Tracking

**Future Phases / Deltas**:
- Consider adding `--format=yaml` option for alternative output
- Consider JSON Schema output for `schema show` commands (separate work)

**Backlog Items**:
- ISSUE-007 – Resolved by this delta

**Open Decisions / Questions**:
- None - straightforward additive change

## 9. Implementation Notes

**Environment Setup**:
```bash
# No special setup needed
uv sync
```

**Testing Commands**:
```bash
# Test JSON output for each command
uv run spec-driver create issue "Test" --json
uv run spec-driver list specs --json
uv run spec-driver show spec SPEC-001 --json

# Run test suite
uv run pytest supekku/cli/
uv run ruff check supekku/cli/
```

**Reference Implementation**:
- See `supekku/cli/create.py` lines 58-79 for existing `--json` pattern
- Standard library `json.dumps()` for serialization
- Typer `Option("--json")` for flag definition

**Code Standards**:
- Follow existing `json_output` parameter naming
- Use `Annotated[bool, typer.Option("--json", help="...")]` pattern
- Print JSON to stdout with `print(json.dumps(result, indent=2))`
- Preserve existing text output logic in else branch
