---
id: DE-009
slug: cli-json-consistency-fixes
name: Delta - CLI JSON consistency fixes
created: '2025-11-03'
updated: '2025-01-04'
status: completed
kind: delta
aliases: []
relations: []
tags: ["cli"]
applies_to:
  specs:
  - SPEC-110
  requirements:
  - PROD-010.FR-001
  - PROD-010.FR-002
  - PROD-010.FR-003
  - PROD-010.NF-003
---

# DE-009 – CLI JSON consistency fixes

```yaml supekku:delta.relationships@v1
schema: supekku.delta.relationships
version: 1
delta: DE-009
revision_links:
  introduces: []
  supersedes: []
specs:
  primary:
    - SPEC-110
  collaborators:
    - SPEC-120
requirements:
  implements:
    - PROD-010.FR-001
    - PROD-010.FR-002
    - PROD-010.FR-003
    - PROD-010.NF-003
  updates: []
  verifies: []
phases:
  - phase-01
```

## 1. Summary & Context

- **Product Spec(s)**: [PROD-010](../../../specify/product/PROD-010/PROD-010.md) – CLI Agent UX improvements for consistent automation patterns
- **Technical Spec(s)**: [SPEC-110](../../../specify/tech/SPEC-110/SPEC-110.md) – supekku/cli implementation target
- **Implementation Plan**: [IP-009](./IP-009.md) – Single phase implementation
- **Change Drivers**:
  - UX Research Report (docs/ux-research-cli-2025-11-03.md) identified critical inconsistencies
  - Agent workflows currently require command-specific special-case handling
  - Missing status filter on specs forces post-processing workarounds
  - JSON output inconsistency (`--json` vs `--format json`) breaks pattern learning

## 2. Motivation

AI agents using spec-driver CLI encounter three critical consistency gaps that prevent pattern-based automation:

1. **JSON flag inconsistency**: `list specs` uses `--json` while all other list commands use `--format json`. Agents must maintain command-specific logic instead of learning universal patterns.

2. **Missing JSON on show commands**: `show delta --json` works but undocumented; `show spec --json` entirely missing. Agents cannot reliably access structured data for detailed artifact information.

3. **Specs lack status filter**: Every artifact type supports `-s`/`--status` filtering except specs. Agents must fetch full list and post-process with jq, consuming 2x tokens unnecessarily.

These gaps violate the "consistency enables automation" principle from PROD-010. The UX research report classified all three as HIGH priority (Section 12: Quick Wins - Priority 1).

**Target State**: Agents learn CLI patterns once and apply everywhere. Zero command-specific logic required for basic list/show/filter operations.

**Supporting Evidence**:
- docs/ux-research-cli-2025-11-03.md Section 2: Command Structure & Consistency
- PROD-010 Section 2: Journey 2 (Agent Filters Specs by Status - currently broken)

## 3. Scope & Objectives

**Primary Outcomes**:

1. **Universal JSON Support** (PROD-010.FR-001, FR-002)
   - All list commands accept both `--json` and `--format=json` with identical behavior
   - All show commands accept `--json` flag for structured output
   - JSON schemas stable and documented
   - Help text updated to document JSON availability

2. **Status Filter Parity** (PROD-010.FR-003)
   - `list specs` command supports `-s`/`--status` filter matching behavior of other list commands
   - Filter works with existing status values: draft, active, deprecated, superseded
   - Consistent with existing status filtering patterns

3. **Validation** (PROD-010.NF-003)
   - All Priority 1 findings from UX research addressed
   - Regression tests confirm no breakage to existing JSON output
   - Help text accurately reflects new capabilities

**Operational Constraints**:
- **Backward Compatibility**: Existing JSON schemas must remain stable (no breaking changes)
- **Timeline**: Single phase, estimated 1-2 days
- **Risk Tolerance**: Low - these are read-only commands; rollback trivial

**Dependencies**:
- None - this delta is self-contained

## 4. Out of Scope

Explicitly deferred to future deltas:

- **Multi-value filters** (PROD-010.FR-004) → DE-010 (Priority 2)
- **Reverse relationship queries** (PROD-010.FR-005) → DE-010 (Priority 2)
- **Enum introspection** (PROD-010.FR-006) → DE-011 (Priority 3)
- **Help text examples** (PROD-010.FR-007) → DE-011 (Priority 3)
- **Machine-readable mode** (PROD-010.FR-008) → DE-012 (Priority 3)
- **Pagination** (PROD-010.FR-009) → DE-012 (Priority 3)
- **Enhanced error messages** (PROD-010.FR-010) → DE-011 (Priority 3)

Rationale: Priority 1 fixes provide highest ROI (80% of agent pain with 20% of implementation effort). Later deltas build on consistent foundation.

## 5. Approach Overview

**System Touchpoints**:
- `supekku/cli/list.py` - Add `--json` shorthand to list commands; add status filter to specs
- `supekku/cli/show.py` - Add `--json` flag to all show commands
- `supekku/cli/common.py` - Shared option types and flag patterns
- `supekku/scripts/lib/formatters/spec_formatters.py` - Status filtering logic for specs
- Test files: `supekku/cli/test_cli.py` - Comprehensive coverage for new flags

**Key Changes**:

1. **Standardize `--json` across list commands** (FR-001)
   - Add `--json` shorthand option to all list commands (deltas, adrs, requirements, revisions, changes)
   - Map `--json` to `--format=json` internally
   - Maintain backward compatibility: both flags work identically
   - Update help text to document both flags

2. **Add `--json` to show commands** (FR-002)
   - Implement JSON formatter for `show spec` (currently missing)
   - Document existing `show delta --json` in help text
   - Ensure consistent JSON structure across all show commands
   - Test JSON parseability and schema stability

3. **Add status filter to specs** (FR-003)
   - Add `-s`/`--status` option to `list specs` command matching other list commands
   - Implement filtering in specs formatter using existing status values
   - Test filter behavior matches deltas/adrs/requirements patterns
   - Support valid spec statuses: draft, active, deprecated, superseded

4. **Update help text and tests**
   - Document JSON availability in all command help text
   - Add regression tests for JSON schema stability
   - Test status filter on specs with all valid values
   - Validate backward compatibility (existing usage unaffected)

**Migration / Rollout Notes**:
- No migration required - purely additive changes
- Existing `--format=json` usage continues to work
- New `--json` shorthand available immediately
- No data migration or registry updates needed

## 6. Verification Strategy

**Requirements Coverage**:

| Requirement | Verification Artifact | Approach |
|-------------|----------------------|----------|
| PROD-010.FR-001 | VT-PROD010-JSON-001 | Unit tests: `--json` flag on all list commands produces valid parseable JSON matching `--format=json` output |
| PROD-010.FR-002 | VT-PROD010-JSON-002 | Unit tests: `--json` flag on all show commands produces structured output; schema matches list output structure |
| PROD-010.FR-003 | VT-PROD010-FILTER-001 | Unit tests: `-s`/`--status` filter on specs command filters correctly; behavior matches deltas/adrs filtering |
| PROD-010.NF-003 | VH-PROD010-UX-001 | Manual validation: Cross-reference implementation against UX research Priority 1 findings; confirm all gaps addressed |

**Planned Artefacts**:
- **VT-CLI-JSON-CONSISTENCY**: Test suite covering all JSON flags
- **VT-CLI-STATUS-SPECS**: Test suite for specs status filtering
- **VT-CLI-REGRESSION**: Backward compatibility tests ensuring no breakage

**Acceptance Criteria**:
- [ ] All list commands support both `--json` and `--format=json` flags
- [ ] All show commands support `--json` flag with documented behavior
- [ ] `list specs -s draft` returns only draft specs (matching other commands)
- [ ] All unit tests passing (`just test`)
- [ ] Both linters passing (`just lint`, `just pylint`)
- [ ] Help text updated and accurate for all modified commands
- [ ] JSON output schemas validated as stable (no breaking changes)
- [ ] Manual agent workflow testing confirms consistency improvements

## 7. Risks & Mitigations

| Risk | Likelihood | Impact | Mitigation |
|------|-----------|--------|------------|
| **Breaking JSON schema changes** | Low | High | Comprehensive regression tests validate schema stability; review existing JSON formatters before modifying |
| **Status filter implementation differs from other commands** | Medium | Medium | Reuse existing status filtering patterns from deltas/adrs; comprehensive test coverage ensures parity |
| **Help text becomes inconsistent** | Low | Low | Systematic review of all modified command help text; test help output in CI |
| **Performance regression on large registries** | Low | Low | Status filtering uses existing registry filtering; no new performance bottlenecks introduced |

## 8. Follow-ups & Tracking

**Future Phases / Deltas**:
- **DE-010**: Enhanced filtering (multi-value filters, reverse relationship queries) - Priority 2
- **DE-011**: Self-documentation (enum introspection, help text examples, error messages) - Priority 3
- **DE-012**: Machine-readable mode (unified flag, pagination) - Priority 3

**Backlog Items**:
- Track implementation progress in [IP-009](./IP-009.md)
- Update PROD-010 verification coverage blocks as tests are implemented
- Consider creating ISSUE for JSON schema versioning strategy (if breaking changes needed in future)

**Open Decisions / Questions**:
- None currently - scope is well-defined from UX research

## 9. Implementation Notes

**Implementation Status**: ✅ **COMPLETED** - 2025-11-03

**What Was Delivered**:

1. **JSON Flag Consistency** (PROD-010.FR-001, FR-002)
   - Added `--json` shorthand to 5 list commands: `deltas`, `adrs`, `requirements`, `revisions`, `changes`
   - Added `--json` flag to 4 show commands: `spec`, `adr`, `requirement`, `revision`
   - Backward compatible: existing `--format=json` continues to work identically
   - Help text auto-generated via typer for consistency

2. **Status Filter Parity** (PROD-010.FR-003)
   - Added `-s`/`--status` filter to `list specs` command
   - Supports statuses: draft, active, deprecated, superseded
   - Uses same `normalize_status()` pattern as other list commands
   - Behavior matches `list deltas` and `list adrs` exactly

3. **Test Coverage** (PROD-010.NF-003)
   - Added 35 new tests across 3 test classes
   - All 74 CLI tests passing (100% pass rate)
   - Comprehensive coverage of new functionality and backward compatibility

**Files Modified**:
- `supekku/cli/list.py` - JSON flags + status filter (5 list commands + specs filter)
- `supekku/cli/show.py` - JSON flags (4 show commands)
- `supekku/cli/test_cli.py` - 35 new tests (3 test classes)

**Quality Metrics**:
- CLI Tests: 74/74 passed ✅
- Ruff: All changes pass ✅
- Pylint: 9.36/10 (acceptable) ✅
- Test Coverage: Comprehensive (35 new tests for ~100 LOC changes)

**Key Implementation Patterns Used**:

1. **Adding `--json` shorthand**:
   ```python
   # In list.py - add to existing format option
   @app.command("list")
   def list_items(
       format: Annotated[str, typer.Option("--format", "-f")] = "table",
       json_flag: Annotated[bool, typer.Option("--json")] = False,
       ...
   ):
       output_format = "json" if json_flag else format
       # Rest of logic uses output_format
   ```

2. **Reusing status filter pattern**:
   ```python
   # Follow existing pattern from list_deltas
   status: Annotated[str | None, typer.Option("--status", "-s")] = None
   if status and normalize_status(spec.status) != normalize_status(status):
       continue  # Filter out non-matching specs
   ```

3. **JSON formatter consistency**:
   - Maintained `{"items": [...]}` wrapper structure for list commands
   - Used `.to_dict()` methods for show commands where available
   - All JSON output parseable and schema-stable

**Development Approach**:
- TDD methodology throughout (red-green-refactor)
- Followed "Skinny CLI" pattern from CLAUDE.md
- Pure functions, no stateful changes
- Formatter separation maintained

**Reference Documentation**:
- UX Research Report: `docs/ux-research-cli-2025-11-03.md`
- PROD-010: `specify/product/PROD-010/PROD-010.md`
- SPEC-110: `specify/tech/SPEC-110/SPEC-110.md`
- CLAUDE.md architectural patterns (Skinny CLI, formatters separation)
- Phase Sheet: `change/deltas/DE-009-cli-json-consistency-fixes/phases/phase-01.md`
