---
id: DR-014
slug: backlog-command-shortcuts
name: Design Revision - backlog command shortcuts
created: '2025-11-04'
updated: '2025-11-04'
status: draft
kind: design_revision
aliases: []
owners: []
relations:
- type: implements
  target: DE-014
delta_ref: DE-014
source_context:
- path: supekku/cli/list.py
  component: list_backlog
  notes: Existing backlog listing function with kind filtering
- path: supekku/cli/create.py
  component: create_issue/create_problem/create_improvement/create_risk
  notes: Pattern to mirror for consistency
code_impacts:
- path: supekku/cli/list.py
  change_type: addition
  component: list_issues/list_problems/list_improvements/list_risks
  current: No shortcut commands; users must use `list backlog -k <kind>`
  target: Four new command functions delegating to list_backlog with fixed kind parameter
verification_alignment:
- id: VT-PROD010-BACKLOG-001
  impact: new
  notes: New test coverage for shortcut commands
design_decisions:
- id: DEC-014-001
  summary: Delegate to existing list_backlog function
  rationale: Avoid code duplication; maintain single source of truth for backlog listing logic
  consequences: Any changes to list_backlog automatically apply to shortcuts
- id: DEC-014-002
  summary: Use plural command names (issues not issue)
  rationale: Consistency with list commands pattern (list specs, list deltas, etc.)
  consequences: Differs from create commands which use singular forms
open_questions: []
---

# DR-014 – backlog command shortcuts

## 1. Executive Summary
- **Delta**: [DE-014](./DE-014.md)
- **Status**: draft
- **Owners / Team**: CLI maintainers
- **Last Updated**: 2025-11-04
- **Synopsis**: Add four shortcut commands for listing backlog items by kind, mirroring the create command pattern

## 2. Problem & Constraints
- **Current Behaviour**: Users must use `list backlog -k <kind>` to filter backlog items by type; inconsistent with create command pattern (`create issue`, `create problem`, etc.)
- **Drivers / Inputs**:
  - PROD-010.FR-011 requirement from requirements-additions-2025-11-04.md
  - UX research identifying "principle of least surprise" violations
  - Command discoverability gap
- **Constraints / Guardrails**:
  - Must not duplicate existing list_backlog logic
  - Must support all existing filter options (status, substring, regexp, format, truncate)
  - Must maintain backward compatibility with `list backlog` command
- **Out of Scope**:
  - Deprecating `list backlog` command (both patterns should coexist)
  - Modifying backlog discovery logic
  - Changes to output formatting

## 3. Architecture Intent
- **Target Outcomes**:
  - Consistent command pattern: `create issue` → `list issues` (instead of `list backlog -k issue`)
  - Improved discoverability: users can discover list commands matching create commands
  - Reduced cognitive load: no need to remember `-k` flag syntax
- **Guiding Principles**:
  - Skinny CLI: commands delegate to existing business logic
  - No duplication: single source of truth for backlog listing
  - Consistency: follow established patterns (list specs, list deltas, etc.)
- **State Transitions / Lifecycle Impact**: None; read-only commands

## 4. Code Impact Summary
| Path | Current State | Target State |
| --- | --- | --- |
| `supekku/cli/list.py` | 9 list commands; backlog requires `-k` flag | 13 list commands; add list_issues, list_problems, list_improvements, list_risks |

**Implementation Pattern** (for each shortcut):
```python
@app.command("issues")
def list_issues(
  root: RootOption = None,
  status: Annotated[str | None, typer.Option(...)] = None,
  substring: Annotated[str | None, typer.Option(...)] = None,
  regexp: RegexpOption = None,
  case_insensitive: CaseInsensitiveOption = False,
  format_type: FormatOption = "table",
  truncate: TruncateOption = False,
) -> None:
  """List backlog issues with optional filtering."""
  list_backlog(
    root=root,
    kind="issue",  # Fixed parameter
    status=status,
    substring=substring,
    regexp=regexp,
    case_insensitive=case_insensitive,
    format_type=format_type,
    truncate=truncate,
  )
```

## 5. Verification Alignment
| Verification | Impact | Notes |
| --- | --- | --- |
| VT-PROD010-BACKLOG-001 | new | Test shortcut commands with various filter combinations |

## 6. Supporting Context
- **Research**: docs/ux-research-cli-2025-11-03.md Section 10 (Principle of Least Surprise)
- **Related Specs**: PROD-010 FR-011 (Backlog Command Shortcuts)
- **Pattern Reference**: create commands at supekku/cli/create.py:404-506

## 7. Design Decisions & Trade-offs

**DEC-014-001: Delegate to existing list_backlog function**
- **Rationale**: Avoid code duplication; maintain single source of truth
- **Trade-offs**:
  - ✓ Easier maintenance: changes to list_backlog automatically apply to shortcuts
  - ✓ Consistent behavior across all list methods
  - ✓ Minimal code changes
  - ✗ Slightly less obvious that shortcuts are wrappers (but documented)
- **Consequences**: Any behavioral changes to list_backlog affect shortcuts

**DEC-014-002: Use plural command names**
- **Rationale**: Consistency with list command pattern (list specs, list deltas, list adrs)
- **Trade-offs**:
  - ✓ Matches existing list command conventions
  - ✓ Clear that command returns multiple items
  - ✗ Creates asymmetry with create commands (create issue vs list issues)
- **Consequences**: Users see `create issue` but `list issues` (justified by convention)

**DEC-014-003: Support all existing filter options**
- **Rationale**: Users expect full filtering capability regardless of entry point
- **Trade-offs**:
  - ✓ Feature parity with list backlog command
  - ✓ Maximum flexibility for users
  - ✗ Verbose function signatures (mitigated by Typer annotations)
- **Consequences**: Each shortcut must accept and pass through all filter parameters

## 8. Open Questions
- [x] Should shortcuts support the `--kind` flag? → No; fixed kind is the point of shortcuts
- [x] Command names: singular vs plural? → Plural (matches list command pattern)
- [x] Should list backlog remain available? → Yes; backward compatibility

## 9. Rollout & Operational Notes
- **Migration / Backfill**: None required; additive change
- **Observability / Alerts**: None required; read-only CLI commands
- **Recovery / Rollback**: Simple revert; no state changes or data migration

## 10. References & Links
- Implementation plan: [IP-014.md](./IP-014.md)
- Requirement: docs/requirements-additions-2025-11-04.md (FR-011)
- Architecture guide: AGENTS.md (Skinny CLI pattern)
- Existing code:
  - list_backlog: supekku/cli/list.py:1168-1245
  - create commands: supekku/cli/create.py:404-506
