#!/usr/bin/env python3
"""Validate workspace metadata and relationships."""

from __future__ import annotations

import argparse
from pathlib import Path

from supekku.scripts.lib.core.cli_utils import add_root_argument
from supekku.scripts.lib.core.repo import find_repo_root
from supekku.scripts.lib.validation.validator import validate_workspace  # type: ignore
from supekku.scripts.lib.workspace import Workspace  # type: ignore


def parse_args(argv: list[str] | None = None) -> argparse.Namespace:
  """Parse command-line arguments for workspace validation.

  Args:
    argv: Optional list of command-line arguments.

  Returns:
    Parsed argument namespace.
  """
  parser = argparse.ArgumentParser(description=__doc__)
  add_root_argument(parser, "Repository root (auto-detected if not provided)")
  parser.add_argument(
    "--sync",
    action="store_true",
    help="Synchronise registries before validation",
  )
  parser.add_argument(
    "--strict",
    action="store_true",
    help="Enable strict validation (warn about deprecated ADR references)",
  )
  return parser.parse_args(argv)


def main(argv: list[str] | None = None) -> int:
  """Validate workspace metadata and relationships.

  Args:
    argv: Optional command-line arguments.

  Returns:
    Exit code: 0 if validation passes, 1 if issues found.
  """
  args = parse_args(argv)
  ws = (
    Workspace(find_repo_root(args.root))
    if hasattr(args, "root")
    else Workspace.from_cwd()
  )
  if args.sync:
    ws.reload_specs()
    ws.sync_change_registries()
    ws.sync_requirements()

  issues = validate_workspace(ws, strict=args.strict)
  if not issues:
    return 0

  for _issue in issues:
    pass
  return 1


def get_repo_root(start: Path) -> Path:
  """Get repository root from starting path.

  Args:
    start: Starting path to search from.

  Returns:
    Repository root path.
  """
  return find_repo_root(start)


if __name__ == "__main__":
  raise SystemExit(main())
