"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_elasticbeanstalk_1 = require("monocdk/aws-elasticbeanstalk");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * EC2 instances in Elastic Beanstalk environments upload rotated logs to S3
 * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/command-options-general.html#command-options-general-elasticbeanstalkhostmanager
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_elasticbeanstalk_1.CfnEnvironment) {
        const optionSettings = monocdk_1.Stack.of(node).resolve(node.optionSettings);
        if (optionSettings == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        let foundEnabled = false;
        for (const optionSetting of optionSettings) {
            const resolvedOptionSetting = monocdk_1.Stack.of(node).resolve(optionSetting);
            const namespace = resolvedOptionSetting.namespace;
            const optionName = resolvedOptionSetting.optionName;
            const value = resolvedOptionSetting.value;
            if (namespace === 'aws:elasticbeanstalk:hostmanager' &&
                optionName === 'LogPublicationControl' &&
                value === 'true') {
                foundEnabled = true;
                break;
            }
        }
        if (!foundEnabled) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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