"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_elasticsearch_1 = require("monocdk/aws-elasticsearch");
const aws_opensearchservice_1 = require("monocdk/aws-opensearchservice");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * OpenSearch Service domains minimally publish SEARCH_SLOW_LOGS and INDEX_SLOW_LOGS to CloudWatch Logs
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_elasticsearch_1.CfnDomain || node instanceof aws_opensearchservice_1.CfnDomain) {
        const logPublishingOptions = monocdk_1.Stack.of(node).resolve(node.logPublishingOptions);
        if (logPublishingOptions == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const requiredSlowLogs = [
            logPublishingOptions === null || logPublishingOptions === void 0 ? void 0 : logPublishingOptions.SEARCH_SLOW_LOGS,
            logPublishingOptions === null || logPublishingOptions === void 0 ? void 0 : logPublishingOptions.INDEX_SLOW_LOGS,
        ];
        for (const log of requiredSlowLogs) {
            const resolvedLog = monocdk_1.Stack.of(node).resolve(log);
            if (resolvedLog == undefined) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
            const enabled = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedLog.enabled);
            if (!enabled) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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