# PyPI Release Plan for QuizGenerator

## Phase 1: Vendor LMSInterface

### Step 1: Copy the code
```bash
# From QuizGeneration directory
mkdir -p QuizGenerator/canvas
cp ../LMSInterface/lms_interface/__init__.py QuizGenerator/canvas/
cp ../LMSInterface/lms_interface/canvas_interface.py QuizGenerator/canvas/
cp ../LMSInterface/lms_interface/classes.py QuizGenerator/canvas/
```

### Step 2: Update internal imports
In `QuizGenerator/canvas/canvas_interface.py`, change:
```python
# Old
from .classes import LMSWrapper, Student, Submission, ...

# New (stays the same - relative import still works)
from .classes import LMSWrapper, Student, Submission, ...
```

### Step 3: Update QuizGenerator imports
In `generate_quiz.py`, change:
```python
# Old
from lms_interface.canvas_interface import CanvasInterface, CanvasCourse

# New
from QuizGenerator.canvas.canvas_interface import CanvasInterface, CanvasCourse
```

### Step 4: Update pyproject.toml dependencies
```toml
[project]
dependencies = [
    # Remove this line:
    # "lms-interface",

    # Add the dependencies that lms-interface needed:
    "canvasapi==3.2.0",
    "requests==2.32.2",
    # (python-dotenv is already in your deps)
]

# Remove this section:
# [tool.uv.sources]
# lms-interface = { path = "../LMSInterface" }
```

### Step 5: Test it works
```bash
uv sync
python generate_quiz.py --help
# Try generating a quiz
python generate_quiz.py --num_pdfs 1
```

---

## Phase 2: Prepare Package Metadata

### Required files to add/update:

#### 1. README.md (root of repo)
Create a user-facing README with:
- What is QuizGenerator?
- Installation: `pip install QuizGenerator`
- Quick example
- Link to documentation

#### 2. LICENSE
Choose and add a license file (MIT is common for academic tools)

#### 3. Update pyproject.toml
```toml
[project]
name = "QuizGenerator"
version = "1.0.0"  # First public release
description = "Generate randomized quiz questions for Canvas LMS and PDF exams"
readme = "README.md"
license = {text = "MIT"}  # or your chosen license
requires-python = ">=3.12"

authors = [
    {name = "Sam Ogden", email = "samuel.s.ogden@gmail.com"},
]

keywords = ["education", "quiz", "canvas", "lms", "assessment", "teaching", "exam"]

classifiers = [
    "Development Status :: 4 - Beta",
    "Intended Audience :: Education",
    "Topic :: Education :: Testing",
    "License :: OSI Approved :: MIT License",
    "Programming Language :: Python :: 3",
    "Programming Language :: Python :: 3.12",
]

[project.urls]
Homepage = "https://github.com/OtterDen-Lab/QuizGenerator"
Documentation = "https://github.com/OtterDen-Lab/QuizGenerator/tree/main/documentation"
Repository = "https://github.com/OtterDen-Lab/QuizGenerator"
"Bug Tracker" = "https://github.com/OtterDen-Lab/QuizGenerator/issues"

[project.scripts]
quizgen = "generate_quiz:main"  # Add main() function to generate_quiz.py
```

#### 4. Add CLI entry point
In `generate_quiz.py`, make sure there's a `main()` function:
```python
def main():
    args = parse_args()

    if args.command == "TEST":
        test()
        return

    # ... rest of current code

if __name__ == "__main__":
    main()
```

---

## Phase 3: Clean Up for Release

### Things to review/clean up:

1. **Dependencies**: Review and relax version pins where safe
   - Change `==` to `>=` for most packages
   - Keep `==` only for packages with breaking changes

2. **Example files**:
   - Ensure `example_files/` has no sensitive data (API keys, real student names, etc.)
   - Add comments/documentation to examples

3. **Documentation**:
   - ✅ Custom questions guide (already done)
   - Add getting started guide
   - Document built-in question types

4. **Environment variables**:
   - Document required `.env` setup for Canvas integration
   - Add `.env.example` file

5. **System requirements**:
   - Document LaTeX requirement (`latexmk`)
   - Document optional Typst requirement

---

## Phase 4: Build and Test

### Local testing:
```bash
# Install build tools
pip install build twine

# Build the package
python -m build

# This creates:
# dist/QuizGenerator-1.0.0-py3-none-any.whl
# dist/QuizGenerator-1.0.0.tar.gz

# Inspect the wheel
unzip -l dist/QuizGenerator-*.whl

# Test install in clean environment
python -m venv test_env
source test_env/bin/activate
pip install dist/QuizGenerator-*.whl

# Test it works
quizgen --help
python -m generate_quiz --help
```

### Test PyPI (optional but recommended):
```bash
# Upload to test.pypi.org first
twine upload --repository testpypi dist/*

# Test install from test PyPI
pip install --index-url https://test.pypi.org/simple/ QuizGenerator
```

---

## Phase 5: Release to PyPI

### One-time setup:
1. Create account on https://pypi.org
2. Enable 2FA
3. Generate API token (Account Settings → API tokens)
4. Save token securely

### Release process:
```bash
# Make sure version is updated in pyproject.toml
# Make sure all changes are committed
git tag v1.0.0
git push origin v1.0.0

# Build fresh
rm -rf dist/
python -m build

# Upload to PyPI
twine upload dist/*
# Enter __token__ as username
# Enter your API token as password

# Or use environment variable:
export TWINE_USERNAME=__token__
export TWINE_PASSWORD=pypi-...your-token...
twine upload dist/*
```

### Verify:
```bash
# Test install from real PyPI
pip install QuizGenerator

# Check it works
quizgen --help
```

---

## Phase 6: Post-Release

1. Create GitHub release with tag and release notes
2. Update documentation with new installation instructions
3. Share with colleagues/community

---

## Quick Reference: Complete Steps

1. ✅ Vendor LMSInterface → QuizGenerator/canvas/
2. ✅ Update imports in generate_quiz.py
3. ✅ Update pyproject.toml dependencies
4. ✅ Test locally
5. Add README.md
6. Add LICENSE
7. Update pyproject.toml metadata
8. Add CLI entry point
9. Clean up examples and docs
10. Build: `python -m build`
11. Test install locally
12. Upload to PyPI: `twine upload dist/*`
13. Test install from PyPI
14. Create GitHub release

---

## Ongoing: Future Releases

When you make changes and want to release a new version:

1. Update version in `pyproject.toml` (e.g., 1.0.0 → 1.1.0)
2. Commit changes
3. Tag: `git tag v1.1.0`
4. Build: `python -m build`
5. Upload: `twine upload dist/*`

### Versioning guide:
- **1.0.0 → 2.0.0**: Breaking changes (API changes, removed features)
- **1.0.0 → 1.1.0**: New features (backward compatible)
- **1.0.0 → 1.0.1**: Bug fixes only

---

## Notes on Vendoring LMSInterface

Since you'll maintain this code in multiple projects:

### Strategy for keeping in sync:
1. Add a version comment in `QuizGenerator/canvas/__init__.py`:
   ```python
   """
   Canvas LMS integration for QuizGenerator.

   Vendored from LMSInterface v0.1.0 (2024-11-17)
   """
   __version__ = "0.1.0"
   ```

2. When you update LMSInterface and want to sync:
   ```bash
   # Copy updated files
   cp ../LMSInterface/lms_interface/*.py QuizGenerator/canvas/

   # Update version comment
   # Test
   # Commit with message: "Update vendored canvas module to v0.2.0"
   ```

3. Consider adding a note in both READMEs about the relationship

This keeps it simple while allowing you to update when needed.