/**
 * Build and bundle the shell web worker and overwrite the file generated by the default
 * labextension build process that does not build it as a web worker.
 */
const fs = require('fs');
const path = require('path');

const rules = [
  {
    test: /\.js$/,
    exclude: /node_modules/,
    loader: 'source-map-loader'
  },
  {
    test: /\.ts$/,
    use: 'ts-loader',
    exclude: /node_modules/
  }
];

const resolve = {
  fallback: {
    fs: false,
    child_process: false,
    crypto: false
  },
  extensions: ['.js', '.ts']
};

const labextensionStaticDir = path.resolve(
  __dirname,
  'jupyterlite_terminal',
  'labextension',
  'static'
);

/**
 * Return the filename that the default labextension build process uses for the web worker.
 */
function getOutputFilename(devMode) {
  const lsProd = fs
    .readdirSync(labextensionStaticDir)
    .filter(f => f.endsWith('js'));
  const regex = /extends (\w+)\.BaseShellWorker/;

  let outputFilename = '';
  for (file of lsProd) {
    const content = fs.readFileSync(path.join(labextensionStaticDir, file));
    if (regex.test(content)) {
      outputFilename = file;
      break;
    }
  }

  if (outputFilename == '') {
    console.error(
      'ERROR: Failed to find JavaScript web worker file to replace'
    );
    process.exit(1);
  }

  console.log('JavaScript web worker file to replace:' + outputFilename);
  return outputFilename;
}

module.exports = env => {
  const devMode = env.dev ?? false;

  const config = {
    entry: './src/worker.ts',
    output: {
      filename: getOutputFilename(devMode),
      path: labextensionStaticDir
    },
    module: {
      rules
    },
    mode: devMode ? 'development' : 'production',
    target: 'webworker',
    resolve
  };

  if (devMode) {
    config.devtool = 'source-map';
  }

  return [config];
};
