"""
test data with implied volatilities for VIX, SQQQ, Bitcoin, Gold, SPY
data are taken around Nov2021
data is used for logsv_model_wtih_quadratic_drift figures
"""

import numpy as np
from numpy import array
from enum import Enum

from stochvolmodels.data.option_chain import OptionChain


def get_btc_test_chain_data() -> OptionChain:
    """
    BTC implied vols of 21Oct2021
    """
    ids = array(['2w', '1m', '2m', '3m'])
    ticker = 'BTC'
    ttms = array([0.04289242541152263, 0.10122575874485597, 0.1984479809670782, 0.4317813143004115])
    forwards = array([67106.44399999999, 67843.219, 68689.48000000001, 70617.77892857141])
    discfactors = array([1.0, 1.0, 1.0, 1.0])
    bid_ivs = (array([0.9231, 0.8835, 0.8695, 0.8621, 0.855 , 0.8589, 0.8822, 0.8856,
                      0.8944, 0.8996, 0.9607, 0.9718]),
               array([0.9475, 0.9211, 0.8917, 0.8863, 0.8873, 0.8913, 0.9012, 0.9102,
                      0.9244, 0.9377, 0.9494, 0.9755, 1.0317]),
               array([0.9882, 0.9595, 0.94  , 0.9247, 0.914 , 0.9131, 0.9109, 0.9168,
                      0.9244, 0.9305, 0.941 , 0.9544, 0.9682, 1.047 , 1.0887]),
               array([1.0052, 0.981 , 0.9593, 0.9722, 0.9924, 1.013 , 1.0419, 1.1222,
                      1.1489]))

    ask_ivs = (array([0.9399, 0.8944, 0.8967, 0.8856, 0.8744, 0.8774, 0.9006, 0.9047,
                      0.9144, 0.9202, 0.9762, 1.0151]),
               array([0.972 , 0.9401, 0.9092, 0.9014, 0.9041, 0.9079, 0.917 , 0.9264,
                      0.9417, 0.952 , 0.9602, 0.9899, 1.0585]),
               array([1.0167, 0.9739, 0.9568, 0.9372, 0.9285, 0.9261, 0.9261, 0.9225,
                      0.9358, 0.9362, 0.9558, 0.9637, 0.9823, 1.0664, 1.1144]),
               array([1.0204, 0.9968, 0.9683, 0.976 , 0.9963, 1.0173, 1.047 , 1.1411,
                      1.1736]))

    strikes_ttms = (array([52000., 56000., 58000., 60000., 64000., 66000., 70000., 72000.,
                           74000., 75000., 90000., 95000.]),
                    array([ 45000.,  48000.,  55000.,  58000.,  64000.,  65000.,  70000.,
                            75000.,  80000.,  85000.,  90000., 100000., 120000.]),
                    array([ 38000.,  42000.,  46000.,  52000.,  56000.,  60000.,  64000.,
                            70000.,  75000.,  80000.,  85000.,  90000., 100000., 140000.,
                            160000.]),
                    array([ 35000.,  40000.,  60000.,  80000., 100000., 120000., 150000.,
                            250000., 300000.]))

    optiontypes_ttms = (array(['P', 'P', 'P', 'P', 'P', 'P', 'C', 'C', 'C', 'C', 'C', 'C'],
                              dtype='<U1'),
                        array(['P', 'P', 'P', 'P', 'P', 'P', 'C', 'C', 'C', 'C', 'C', 'C', 'C'],
                              dtype='<U1'),
                        array(['P', 'P', 'P', 'P', 'P', 'P', 'P', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C'], dtype='<U1'),
                        array(['P', 'P', 'P', 'C', 'C', 'C', 'C', 'C', 'C'], dtype='<U1'))

    data = OptionChain(ids=ids,
                       ttms=ttms,
                       ticker=ticker,
                       forwards=forwards,
                       strikes_ttms=strikes_ttms,
                       optiontypes_ttms=optiontypes_ttms,
                       discfactors=discfactors,
                       bid_ivs=bid_ivs,
                       ask_ivs=ask_ivs)

    return data


def get_vix_test_chain_data() -> OptionChain:
    """
    VIX implied vols 15_Jul_2022_16_23_02
    """
    ids = array(['2w', '1m', '2m', '6m'])
    ticker = 'VIX'
    ttms = array([0.05088164965679655, 0.08977053854568544, 0.18699276076790766, 0.4397705385456855])
    forwards = array([27.03333333333333, 27.19166666666666, 28.266666666666666, 28.541666666666664])
    discfactors = array([0.9989373488895353, 0.9981385743802681, 0.9955766521308749, 0.9877198727151])
    bid_ivs = (array([0.56835182, 0.61038868, 0.67019286, 0.72738467, 0.77129258,
                      0.81958001, 0.85305811, 0.89729795, 0.93616669, 0.99789698,
                      1.08298867, 1.13360005, 1.1883643 , 1.23382656, 1.28412618,
                      1.34131985]),
               array([0.5444521 , 0.55126086, 0.60467879, 0.66830862, 0.7064206 ,
                      0.76288524, 0.79335457, 0.85159729, 0.87526777, 0.91572401,
                      0.96852567, 1.03500855, 1.08075232, 1.13855152, 1.17229788,
                      1.21215437, 1.26824298, 1.31384972, 1.37491966, 1.44876616]),
               array([0.52432547, 0.54973039, 0.59746211, 0.61649596, 0.65440702,
                      0.68553427, 0.70242763, 0.72892705, 0.76167413, 0.77856581,
                      0.8414647 , 0.89188721, 0.92780338, 0.96691013, 0.98955065,
                      1.02540273, 1.06586968, 1.08214485, 1.1419159 , 1.17895471,
                      1.21823635, 1.24170569, 1.2799203 , 1.30897156, 1.32903795]),
               array([0.46081894, 0.48042652, 0.50771811, 0.51839194, 0.54595747,
                      0.56131066, 0.57570321, 0.59724762, 0.61163043, 0.62020978,
                      0.64457913, 0.65542545, 0.6860472 , 0.7108894 , 0.74137097,
                      0.76002296, 0.7835153 , 0.79893258, 0.82362204, 0.83429229,
                      0.85778578, 0.88469198, 0.91043128, 0.92613176, 0.94624771,
                      0.95934269, 0.98227221, 0.98299831, 0.99654157, 1.02718443,
                      1.03813564]))

    ask_ivs = (array([0.66930902, 0.70288836, 0.75358012, 0.79219914, 0.85534302,
                      0.92458897, 0.94964133, 0.9795049 , 1.00453637, 1.08203571,
                      1.15337307, 1.21207582, 1.28245343, 1.33709956, 1.37161703,
                      1.47835206]),
               array([0.59715166, 0.62026032, 0.6534268 , 0.68785087, 0.74073201,
                      0.79475553, 0.82410262, 0.86679802, 0.90582516, 0.94677219,
                      1.00210206, 1.07204692, 1.1223848 , 1.18484204, 1.2247449 ,
                      1.27083907, 1.33182324, 1.34938026, 1.4182737 , 1.49878657]),
               array([0.55107868, 0.58872003, 0.61299989, 0.64327837, 0.66644723,
                      0.69674635, 0.72384736, 0.77060909, 0.80277835, 0.79902528,
                      0.8623003 , 0.91361753, 0.95083615, 0.97921486, 1.01598685,
                      1.05360603, 1.08089787, 1.11438942, 1.16020099, 1.19983655,
                      1.24166476, 1.29379226, 1.30917289, 1.3411977 , 1.36491654]),
               array([0.48014239, 0.50945479, 0.51955576, 0.53873739, 0.56393312,
                      0.5777923 , 0.59883718, 0.61193095, 0.63997505, 0.64784122,
                      0.67176679, 0.66887161, 0.71281541, 0.73807772, 0.75535419,
                      0.78176518, 0.79861773, 0.81475287, 0.83187918, 0.85161514,
                      0.87695232, 0.90569193, 0.93329409, 0.95130205, 0.97352534,
                      0.98916572, 1.01389412, 1.01827342, 1.03440252, 1.04696366,
                      1.06154015]))

    strikes_ttms = (array([22. , 23. , 24. , 25. , 26. , 27. , 28. , 29. , 30. , 32.5, 35. ,
                           37.5, 40. , 42.5, 45. , 47.5]),
                    array([21. , 22. , 23. , 24. , 25. , 26. , 27. , 28. , 29. , 30. , 32.5,
                           35. , 37.5, 40. , 42.5, 45. , 47.5, 50. , 55. , 60. ]),
                    array([21. , 22. , 23. , 24. , 25. , 26. , 27. , 28. , 29. , 30. , 32.5,
                           35. , 37.5, 40. , 42.5, 45. , 47.5, 50. , 55. , 60. , 65. , 70. ,
                           75. , 80. , 85. ]),
                    array([ 19. ,  20. ,  21. ,  22. ,  23. ,  24. ,  25. ,  26. ,  27. ,
                            28. ,  29. ,  30. ,  32.5,  35. ,  37.5,  40. ,  42.5,  45. ,
                            47.5,  50. ,  55. ,  60. ,  65. ,  70. ,  75. ,  80. ,  85. ,
                            90. ,  95. , 100. , 110. ]))

    optiontypes_ttms = (array(['P', 'P', 'P', 'P', 'P', 'P', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C'], dtype='<U1'),
                        array(['P', 'P', 'P', 'P', 'P', 'P', 'P', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C'], dtype='<U1'),
                        array(['P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C'],
                              dtype='<U1'),
                        array(['P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C'], dtype='<U1'))

    data = OptionChain(ids=ids,
                       ttms=ttms,
                       ticker=ticker,
                       forwards=forwards,
                       strikes_ttms=strikes_ttms,
                       optiontypes_ttms=optiontypes_ttms,
                       discfactors=discfactors,
                       bid_ivs=bid_ivs,
                       ask_ivs=ask_ivs)

    return data


def get_gld_test_chain_data_6m() -> OptionChain:
    """
    GLD implied vols 15_Jul_2022_16_23_17
    """
    ids = array(['2w', '1m', '2m', '6m'])
    ticker = 'GLD'
    ttms = array([0.056437205212352105, 0.09532609410124099, 0.17310387187901877, 0.4258816496567966])
    forwards = array([159.2092857142857, 159.31156249999998, 159.5546875, 160.60000000000005])
    discfactors = array([0.9988269169205103, 0.9980054593706332, 0.9959230968030202, 0.9882261172126616])
    bid_ivs = (array([0.21416116, 0.20881313, 0.20308286, 0.1979989 , 0.19300358,
                      0.19015356, 0.1867255 , 0.18265313, 0.18057913, 0.17817519,
                      0.1765541 , 0.17135631, 0.17201628, 0.17037751, 0.16807024,
                      0.16761692, 0.16796547, 0.17064928, 0.17104001, 0.17138833,
                      0.17231145, 0.17561304, 0.17704071, 0.17964647, 0.18040263,
                      0.1826215 , 0.18553302, 0.19216536, 0.19529925]),
               array([0.22202523, 0.21722313, 0.21295024, 0.20761746, 0.20254541,
                      0.19751882, 0.19333188, 0.18958455, 0.18597085, 0.18370514,
                      0.17960691, 0.17634409, 0.17471453, 0.17206715, 0.16996323,
                      0.16922419, 0.16664042, 0.16758403, 0.1671996 , 0.16700824,
                      0.16930958, 0.16886411, 0.17027834, 0.17085714, 0.17279068,
                      0.17514339, 0.17830399, 0.1808898 , 0.18191485, 0.18576891,
                      0.18851229, 0.19139953, 0.19469393, 0.19696987]),
               array([0.21397793, 0.19363323, 0.1910246 , 0.18822983, 0.18520102,
                      0.18248039, 0.17991924, 0.17739859, 0.17578215, 0.17438375,
                      0.17266291, 0.17017468, 0.17053468, 0.1691491 , 0.16763514,
                      0.16860087, 0.16797746, 0.16611696, 0.16862731, 0.16990598,
                      0.17114975, 0.17119102, 0.17340031, 0.17410525, 0.17505932,
                      0.17737987, 0.17930308, 0.18140703, 0.18264842, 0.18419929,
                      0.18755057, 0.18960827, 0.19169614, 0.19390304, 0.19633311,
                      0.19910635, 0.2023569 , 0.20415449, 0.20759689, 0.20951864,
                      0.2121876 , 0.21446088, 0.21774704]),
               array([0.21866178, 0.20688834, 0.19545688, 0.18687374, 0.18006473,
                      0.17644716, 0.17881794, 0.18065602, 0.1816837 , 0.1822525 ,
                      0.18236353, 0.18328057, 0.1850558 , 0.1864424 , 0.1874404 ,
                      0.18940837, 0.18964535, 0.19090108, 0.19326846, 0.19445691,
                      0.19592936, 0.19709704, 0.19860161, 0.20049014, 0.20212751,
                      0.20315541, 0.20499086, 0.20622463, 0.20797644, 0.20951308,
                      0.21123989, 0.21276531, 0.21408346, 0.21518698, 0.21700404,
                      0.21913362, 0.22012779, 0.22193804, 0.22306704, 0.22510091,
                      0.22700785, 0.2282049 , 0.22983186, 0.23132148, 0.23872331,
                      0.24604502]))

    ask_ivs = (array([0.21771043, 0.21192634, 0.2044636 , 0.20042504, 0.19515812,
                      0.19320574, 0.1896274 , 0.18526601, 0.18295222, 0.18037009,
                      0.17862099, 0.17405497, 0.17466602, 0.1736271 , 0.17133154,
                      0.17029685, 0.17077994, 0.17209943, 0.17254436, 0.17303295,
                      0.1750499 , 0.17663715, 0.1793721 , 0.18211451, 0.18173628,
                      0.18569113, 0.18732641, 0.19416289, 0.1976177 ]),
               array([0.22379163, 0.21881239, 0.21437475, 0.21018589, 0.20487329,
                      0.20067612, 0.19524318, 0.19131659, 0.18833237, 0.1851432 ,
                      0.18160037, 0.17881508, 0.17645152, 0.17426404, 0.17207174,
                      0.17178582, 0.17168995, 0.17010758, 0.16924612, 0.16858616,
                      0.17094958, 0.17059451, 0.17273206, 0.17350109, 0.17565321,
                      0.17748437, 0.18000884, 0.18182929, 0.18503878, 0.18805925,
                      0.191049  , 0.19420945, 0.19625857, 0.20043478]),
               array([0.21627998, 0.19598792, 0.19319435, 0.19023784, 0.18706765,
                      0.18421896, 0.18154402, 0.17892369, 0.17769483, 0.17574249,
                      0.17438962, 0.17224741, 0.17253501, 0.17109675, 0.16954573,
                      0.17048951, 0.16985938, 0.16989707, 0.17053944, 0.17185386,
                      0.17314693, 0.17284198, 0.17511123, 0.17633751, 0.17740135,
                      0.17934793, 0.18137839, 0.18305403, 0.18498247, 0.1866868 ,
                      0.18952531, 0.19171392, 0.19394408, 0.19709308, 0.19973388,
                      0.20092911, 0.20428711, 0.20622774, 0.20869788, 0.21186702,
                      0.21468554, 0.21712914, 0.21916622]),
               array([0.22258674, 0.2092668 , 0.19774427, 0.18867106, 0.18301492,
                      0.17902469, 0.18124083, 0.1830854 , 0.1841301 , 0.18595594,
                      0.18611108, 0.1858143 , 0.18762754, 0.18905781, 0.18877427,
                      0.19076805, 0.19242555, 0.19374563, 0.19472409, 0.19624778,
                      0.19746102, 0.19898513, 0.2005416 , 0.20215101, 0.20383525,
                      0.20526739, 0.20716261, 0.20846482, 0.20989992, 0.21188985,
                      0.21328205, 0.2148714 , 0.21625865, 0.21832825, 0.21978077,
                      0.22104194, 0.22308245, 0.22447799, 0.22621529, 0.22780269,
                      0.22923592, 0.23107914, 0.23279399, 0.23437734, 0.24228739,
                      0.25018411]))

    strikes_ttms = (array([148. , 149. , 150. , 151. , 152. , 152.5, 153. , 154. , 155. ,
                           156. , 157. , 157.5, 158. , 159. , 160. , 161. , 162. , 162.5,
                           163. , 164. , 165. , 166. , 167. , 167.5, 168. , 169. , 170. ,
                           171. , 172. ]),
                    array([143., 144., 145., 146., 147., 148., 149., 150., 151., 152., 153.,
                           154., 155., 156., 157., 158., 159., 160., 161., 162., 163., 164.,
                           165., 166., 167., 168., 169., 170., 171., 172., 173., 174., 175.,
                           176.]),
                    array([140., 145., 146., 147., 148., 149., 150., 151., 152., 153., 154.,
                           155., 156., 157., 158., 159., 160., 161., 162., 163., 164., 165.,
                           166., 167., 168., 169., 170., 171., 172., 173., 174., 175., 176.,
                           177., 178., 179., 180., 181., 182., 183., 184., 185., 186.]),
                    array([130., 135., 140., 145., 150., 155., 160., 164., 165., 166., 167.,
                           168., 169., 170., 171., 172., 173., 174., 175., 176., 177., 178.,
                           179., 180., 181., 182., 183., 184., 185., 186., 187., 188., 189.,
                           190., 191., 192., 193., 194., 195., 196., 197., 198., 199., 200.,
                           205., 210.]))

    optiontypes_ttms = (array(['P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'P', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C'], dtype='<U1'),
                        array(['P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'P', 'P', 'P', 'P', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C'], dtype='<U1'),
                        array(['P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'P', 'P', 'P', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C'], dtype='<U1'),
                        array(['P', 'P', 'P', 'P', 'P', 'P', 'P', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C'], dtype='<U1'))

    data = OptionChain(ids=ids,
                       ttms=ttms,
                       ticker=ticker,
                       forwards=forwards,
                       strikes_ttms=strikes_ttms,
                       optiontypes_ttms=optiontypes_ttms,
                       discfactors=discfactors,
                       bid_ivs=bid_ivs,
                       ask_ivs=ask_ivs)

    return data


def get_gld_test_chain_data() -> OptionChain:
    """
    GLD implied vols 15_Jul_2022_16_23_17
    """
    ids = array(['1m', '2m', '5m', '12m'])
    ticker = 'GLD'
    ttms = array([0.09532609410124099, 0.17310387187901877, 0.4258816496567966, 0.9314372052123521])
    forwards = array([159.31156249999998, 159.5546875, 160.60000000000005, 162.9916666666667])
    discfactors = array([0.9980054593706332, 0.9959230968030202, 0.9882261172126616, 0.9712488834817177])
    bid_ivs = (array([0.22202523, 0.21722313, 0.21295024, 0.20761746, 0.20254541,
                      0.19751882, 0.19333188, 0.18958455, 0.18597085, 0.18370514,
                      0.17960691, 0.17634409, 0.17471453, 0.17206715, 0.16996323,
                      0.16922419, 0.16664042, 0.16758403, 0.1671996 , 0.16700824,
                      0.16930958, 0.16886411, 0.17027834, 0.17085714, 0.17279068,
                      0.17514339, 0.17830399, 0.1808898 , 0.18191485, 0.18576891,
                      0.18851229, 0.19139953, 0.19469393, 0.19696987]),
               array([0.21397793, 0.19363323, 0.1910246 , 0.18822983, 0.18520102,
                      0.18248039, 0.17991924, 0.17739859, 0.17578215, 0.17438375,
                      0.17266291, 0.17017468, 0.17053468, 0.1691491 , 0.16763514,
                      0.16860087, 0.16797746, 0.16611696, 0.16862731, 0.16990598,
                      0.17114975, 0.17119102, 0.17340031, 0.17410525, 0.17505932,
                      0.17737987, 0.17930308, 0.18140703, 0.18264842, 0.18419929,
                      0.18755057, 0.18960827, 0.19169614, 0.19390304, 0.19633311,
                      0.19910635, 0.2023569 , 0.20415449, 0.20759689, 0.20951864,
                      0.2121876 , 0.21446088, 0.21774704]),
               array([0.21866178, 0.20688834, 0.19545688, 0.18687374, 0.18006473,
                      0.17644716, 0.17881794, 0.18065602, 0.1816837 , 0.1822525 ,
                      0.18236353, 0.18328057, 0.1850558 , 0.1864424 , 0.1874404 ,
                      0.18940837, 0.18964535, 0.19090108, 0.19326846, 0.19445691,
                      0.19592936, 0.19709704, 0.19860161, 0.20049014, 0.20212751,
                      0.20315541, 0.20499086, 0.20622463, 0.20797644, 0.20951308,
                      0.21123989, 0.21276531, 0.21408346, 0.21518698, 0.21700404,
                      0.21913362, 0.22012779, 0.22193804, 0.22306704, 0.22510091,
                      0.22700785, 0.2282049 , 0.22983186, 0.23132148, 0.23872331,
                      0.24604502]),
               array([0.209464  , 0.20243303, 0.19762663, 0.1918023 , 0.18740263,
                      0.18262234, 0.18285964, 0.18324906, 0.18533701, 0.18914492,
                      0.19108385, 0.19408288, 0.19889031, 0.20366875, 0.21013068,
                      0.21602766, 0.22180264, 0.22742505, 0.23357328, 0.23899032,
                      0.24509863, 0.24996135, 0.25683815, 0.25971804, 0.2646542 ,
                      0.26867146, 0.27468827, 0.27748659, 0.28157316, 0.28616449,
                      0.28952359]))

    ask_ivs = (array([0.22379163, 0.21881239, 0.21437475, 0.21018589, 0.20487329,
                      0.20067612, 0.19524318, 0.19131659, 0.18833237, 0.1851432 ,
                      0.18160037, 0.17881508, 0.17645152, 0.17426404, 0.17207174,
                      0.17178582, 0.17168995, 0.17010758, 0.16924612, 0.16858616,
                      0.17094958, 0.17059451, 0.17273206, 0.17350109, 0.17565321,
                      0.17748437, 0.18000884, 0.18182929, 0.18503878, 0.18805925,
                      0.191049  , 0.19420945, 0.19625857, 0.20043478]),
               array([0.21627998, 0.19598792, 0.19319435, 0.19023784, 0.18706765,
                      0.18421896, 0.18154402, 0.17892369, 0.17769483, 0.17574249,
                      0.17438962, 0.17224741, 0.17253501, 0.17109675, 0.16954573,
                      0.17048951, 0.16985938, 0.16989707, 0.17053944, 0.17185386,
                      0.17314693, 0.17284198, 0.17511123, 0.17633751, 0.17740135,
                      0.17934793, 0.18137839, 0.18305403, 0.18498247, 0.1866868 ,
                      0.18952531, 0.19171392, 0.19394408, 0.19709308, 0.19973388,
                      0.20092911, 0.20428711, 0.20622774, 0.20869788, 0.21186702,
                      0.21468554, 0.21712914, 0.21916622]),
               array([0.22258674, 0.2092668 , 0.19774427, 0.18867106, 0.18301492,
                      0.17902469, 0.18124083, 0.1830854 , 0.1841301 , 0.18595594,
                      0.18611108, 0.1858143 , 0.18762754, 0.18905781, 0.18877427,
                      0.19076805, 0.19242555, 0.19374563, 0.19472409, 0.19624778,
                      0.19746102, 0.19898513, 0.2005416 , 0.20215101, 0.20383525,
                      0.20526739, 0.20716261, 0.20846482, 0.20989992, 0.21188985,
                      0.21328205, 0.2148714 , 0.21625865, 0.21832825, 0.21978077,
                      0.22104194, 0.22308245, 0.22447799, 0.22621529, 0.22780269,
                      0.22923592, 0.23107914, 0.23279399, 0.23437734, 0.24228739,
                      0.25018411]),
               array([0.2202844 , 0.21193891, 0.20201797, 0.19540524, 0.19066581,
                      0.18694498, 0.18572717, 0.18675882, 0.18867534, 0.19324236,
                      0.19438976, 0.19749178, 0.20246161, 0.208399  , 0.21316118,
                      0.2203523 , 0.22529735, 0.23168852, 0.23788869, 0.24392832,
                      0.24977714, 0.25566598, 0.26148901, 0.26555004, 0.27170952,
                      0.27712495, 0.28086494, 0.28810084, 0.29289634, 0.29707323,
                      0.30176663]))

    strikes_ttms = (array([143., 144., 145., 146., 147., 148., 149., 150., 151., 152., 153.,
                           154., 155., 156., 157., 158., 159., 160., 161., 162., 163., 164.,
                           165., 166., 167., 168., 169., 170., 171., 172., 173., 174., 175.,
                           176.]),
                    array([140., 145., 146., 147., 148., 149., 150., 151., 152., 153., 154.,
                           155., 156., 157., 158., 159., 160., 161., 162., 163., 164., 165.,
                           166., 167., 168., 169., 170., 171., 172., 173., 174., 175., 176.,
                           177., 178., 179., 180., 181., 182., 183., 184., 185., 186.]),
                    array([130., 135., 140., 145., 150., 155., 160., 164., 165., 166., 167.,
                           168., 169., 170., 171., 172., 173., 174., 175., 176., 177., 178.,
                           179., 180., 181., 182., 183., 184., 185., 186., 187., 188., 189.,
                           190., 191., 192., 193., 194., 195., 196., 197., 198., 199., 200.,
                           205., 210.]),
                    array([120., 125., 130., 135., 140., 145., 150., 155., 160., 165., 170.,
                           175., 180., 185., 190., 195., 200., 205., 210., 215., 220., 225.,
                           230., 235., 240., 245., 250., 255., 260., 265., 270.]))

    optiontypes_ttms = (array(['P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'P', 'P', 'P', 'P', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C'], dtype='<U1'),
                        array(['P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'P', 'P', 'P', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C'], dtype='<U1'),
                        array(['P', 'P', 'P', 'P', 'P', 'P', 'P', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C'], dtype='<U1'),
                        array(['P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C'], dtype='<U1'))

    data = OptionChain(ids=ids,
                       ttms=ttms,
                       ticker=ticker,
                       forwards=forwards,
                       strikes_ttms=strikes_ttms,
                       optiontypes_ttms=optiontypes_ttms,
                       discfactors=discfactors,
                       bid_ivs=bid_ivs,
                       ask_ivs=ask_ivs)

    return data


def get_sqqq_test_chain_data() -> OptionChain:
    """
    SQQQ implied vols of 15_Jul_2022_16_23_12
    """
    ids = array(['2w', '1m', '2m', '6m'])
    ticker = 'SQQQ'
    ttms = array([0.056437205212352105, 0.09532609410124099, 0.17310387187901877, 0.4258816496567966])
    forwards = array([52.6275, 52.71, 52.69, 52.87])
    discfactors = array([0.9988269169205103, 0.9980054593706331, 0.9959230968030199, 0.9882261172126616])
    bid_ivs = (array([0.77016505, 0.80369003, 0.8203842 , 0.82466014, 0.84856724,
                      0.84941631, 0.86148104, 0.87507298, 0.87967748, 0.90256447,
                      0.91178085, 0.90876063, 0.92813003, 0.95112527, 0.94303437,
                      0.92357608, 0.93205502, 0.93907858, 0.95462792, 0.95893254,
                      0.97196604, 0.93913315, 0.94539118, 0.93584246, 0.94796674,
                      0.95482465, 0.97358884, 0.96311877, 0.96673153, 0.97836071,
                      0.9869026 , 0.99226539, 0.98476308, 0.99540579, 0.99222018,
                      1.02518285, 1.01772072, 1.02336954, 1.02272535, 1.03807112,
                      1.0582176 , 1.06642608, 1.05354585, 1.08071118, 1.11546531,
                      1.13942312]),
               array([0.7567628 , 0.80217671, 0.86211441, 0.87951953, 0.87639827,
                      0.89047693, 0.91683592, 0.92395449, 0.92741243, 0.9427074 ,
                      0.93163772, 0.96399475, 0.97818735, 0.97378723, 0.97454173,
                      0.98089626, 0.97458865, 1.00379823, 0.99623989, 0.9950129 ,
                      1.00082651, 1.01247569, 1.0364264 , 0.98770745, 1.04049225,
                      1.0310359 , 1.04372526, 1.04846334, 1.05887778, 1.06581443,
                      1.07907452, 1.0812889 , 1.08477736, 1.08976912, 1.096501  ,
                      1.10228542, 1.11612336, 1.10779199, 1.05911568, 1.14185689,
                      1.13680049, 1.13262141, 1.13425294, 1.14686542, 1.14161598,
                      1.14823088, 1.14510656, 1.17367674, 1.18830949, 1.1692348 ,
                      1.19803291]),
               array([0.73117855, 0.73828639, 0.73812035, 0.74587889, 0.7516982 ,
                      0.76844138, 0.77592894, 0.7798126 , 0.78897266, 0.79626859,
                      0.81080203, 0.82190525, 0.83550837, 0.84505429, 0.85734785,
                      0.86620929, 0.87802338, 0.88687432, 0.89888042, 0.91509895,
                      0.92162711, 0.92024957, 0.92824823, 0.95136195, 0.94411243,
                      0.94647853, 0.96435462, 0.98073893, 0.97833298, 0.98606467,
                      0.98655885, 1.00349731, 1.00138861, 0.99180842, 1.01153735,
                      1.02425281, 1.03609187, 1.04708953, 1.03798418, 1.04716443,
                      1.06876367, 1.06323414, 1.0633559 , 1.06253793, 1.06218374,
                      1.06236092, 1.07631415, 1.07207362, 1.08199437, 1.08685489,
                      1.09112266, 1.09638491, 1.09950139, 1.10366948, 1.10228645,
                      1.10870949, 1.12331002, 1.11677827, 1.12546596, 1.12477344,
                      1.14008334, 1.1459776 , 1.12665111, 1.13541713, 1.15184317,
                      1.15422154, 1.16632257]),
               array([0.41075308, 0.59765035, 0.38355723, 0.54056091, 0.71651608,
                      0.50976381, 0.54676967, 0.57746752, 0.61122317, 0.63425441,
                      0.69065396, 0.67767877, 0.69289817, 0.70966315, 0.71875897,
                      0.73445423, 0.74762871, 0.76283264, 0.77593172, 0.78312721,
                      0.80073688, 0.80484014, 0.80768577, 0.81702495, 0.89584776,
                      0.83710893, 0.96247354, 0.84237708, 0.84918383, 0.86615999,
                      0.85646826, 0.98589344, 0.99650708, 0.87065241, 0.87638117,
                      0.8851118 , 0.92257111, 0.89704324, 0.89291468, 0.93618432,
                      0.90881884, 0.91044366, 0.91902311, 0.91592066, 0.91983141,
                      0.99468364, 0.92634609, 0.94043147, 0.92734534, 0.9291455 ,
                      0.93834931, 1.00912088, 0.94801515, 0.9563249 , 0.95641754,
                      0.96016512, 0.95148515, 0.9627238 , 0.969675  , 0.96816489,
                      0.97047506, 0.97251743, 0.96582574, 0.9800656 , 0.99188166,
                      0.98853811, 0.99288279, 0.99705726]))

    ask_ivs = (array([0.88526163, 0.88282578, 0.91060993, 0.97053297, 0.94899908,
                      0.93318711, 0.93061221, 0.96871521, 0.96174454, 0.98363133,
                      1.0213158 , 0.98792221, 1.00696988, 1.01010305, 1.01175941,
                      1.00215772, 1.0206152 , 1.01805293, 1.01411614, 1.06842255,
                      1.05212716, 1.0645258 , 1.04550053, 1.06582887, 1.04406711,
                      1.07285073, 1.07581492, 1.09473336, 1.09583487, 1.10676891,
                      1.08832575, 1.09966063, 1.11030086, 1.07558589, 1.12224905,
                      1.10559224, 1.12574881, 1.14652672, 1.14057617, 1.13663356,
                      1.1413914 , 1.14030809, 1.16285168, 1.15464895, 1.20302671,
                      1.26032864]),
               array([0.80991266, 0.86611665, 0.91034987, 0.91101113, 0.92291469,
                      0.95188027, 0.96262559, 0.9696489 , 0.98079924, 0.97335909,
                      0.98571531, 1.00294386, 1.00970678, 0.98982885, 1.02335385,
                      1.03058964, 1.03555036, 1.03996772, 1.07003639, 1.08488042,
                      1.08562728, 1.09171105, 1.07134894, 1.10648007, 1.10164851,
                      1.11715832, 1.11503032, 1.1239703 , 1.1317576 , 1.13613483,
                      1.13455502, 1.13858571, 1.13890833, 1.15062995, 1.15379207,
                      1.1612195 , 1.16502832, 1.17636886, 1.17839159, 1.1853838 ,
                      1.19472845, 1.20522333, 1.21599802, 1.22584066, 1.22793306,
                      1.22545162, 1.24487339, 1.23709188, 1.24816388, 1.25484652,
                      1.26942945]),
               array([0.77271373, 0.76781008, 0.77546342, 0.80564801, 0.80344294,
                      0.81900683, 0.82245903, 0.84518106, 0.85496546, 0.85562193,
                      0.85421916, 0.89093768, 0.86881739, 0.89660431, 0.92003524,
                      0.94587598, 0.91409775, 0.92831347, 0.94566466, 0.96144819,
                      0.96764066, 0.95457271, 0.9738881 , 0.9912558 , 1.0182288 ,
                      1.0036184 , 0.99875591, 1.01527389, 1.01887375, 1.01523444,
                      1.03942415, 1.04492242, 1.04921584, 1.07031651, 1.06036957,
                      1.04898191, 1.06726837, 1.07230844, 1.08275913, 1.08602688,
                      1.09489221, 1.08985431, 1.10380474, 1.11046093, 1.11643553,
                      1.12174007, 1.11933498, 1.13037386, 1.14238467, 1.13935239,
                      1.15334821, 1.15951421, 1.16518541, 1.17190862, 1.17193581,
                      1.18403749, 1.17818665, 1.20414623, 1.2023095 , 1.22303038,
                      1.21367575, 1.2085173 , 1.2114009 , 1.212085  , 1.21410437,
                      1.23035608, 1.22436457]),
               array([1.1745287 , 1.0490971 , 1.08726886, 1.07426015, 1.1279145 ,
                      1.08123911, 0.88911331, 1.07475458, 1.05613038, 1.06284614,
                      1.07764713, 1.06113738, 1.05832705, 1.06821556, 1.07644632,
                      1.08318633, 1.07989372, 1.0841998 , 1.08736284, 1.09774846,
                      1.0906045 , 1.10695701, 1.11013007, 1.12429137, 1.09660955,
                      1.08883841, 1.08430815, 1.08675156, 1.11119997, 1.1158915 ,
                      1.12366667, 1.115833  , 1.12608538, 1.14313521, 1.13364757,
                      1.15322151, 1.14642505, 1.15388413, 1.16453058, 1.16736547,
                      1.15509658, 1.17906924, 1.17697059, 1.19646882, 1.19726596,
                      1.01703831, 1.20511283, 1.2011764 , 1.21895338, 1.16273821,
                      1.21682039, 1.2152826 , 1.22823018, 1.23722395, 1.23484726,
                      1.23218381, 1.25529792, 1.25215182, 1.24499439, 1.26004014,
                      1.25238475, 1.27080507, 1.27776574, 1.2694329 , 1.2657045 ,
                      1.28705703, 1.29643685, 1.30182069]))

    strikes_ttms = (array([40. , 44. , 45. , 46. , 47. , 48. , 49. , 50. , 50.5, 51. , 52. ,
                           52.5, 53. , 53.5, 54. , 54.5, 55. , 55.5, 56. , 56.5, 57. , 57.5,
                           58. , 58.5, 59. , 59.5, 60. , 60.5, 61. , 61.5, 62. , 62.5, 63. ,
                           64. , 65. , 66. , 67. , 68. , 69. , 70. , 71. , 72. , 73. , 75. ,
                           80. , 85. ]),
                    array([ 40.,  45.,  50.,  51.,  52.,  53.,  54.,  55.,  56.,  57.,  58.,
                            59.,  60.,  61.,  62.,  63.,  64.,  65.,  66.,  67.,  68.,  69.,
                            70.,  71.,  72.,  73.,  74.,  75.,  76.,  77.,  78.,  79.,  80.,
                            81.,  82.,  83.,  84.,  85.,  86.,  87.,  88.,  90.,  91.,  93.,
                            94.,  95.,  96.,  97.,  98.,  99., 100.]),
                    array([ 34.,  35.,  36.,  37.,  38.,  39.,  40.,  41.,  42.,  43.,  44.,
                            45.,  46.,  47.,  48.,  49.,  50.,  51.,  52.,  53.,  54.,  55.,
                            56.,  57.,  58.,  59.,  60.,  61.,  62.,  63.,  64.,  65.,  66.,
                            67.,  68.,  69.,  70.,  71.,  72.,  73.,  74.,  75.,  76.,  77.,
                            78.,  79.,  80.,  81.,  82.,  83.,  84.,  85.,  86.,  87.,  88.,
                            89.,  90.,  91.,  92.,  93.,  94.,  95.,  96.,  97.,  98.,  99.,
                            100.]),
                    array([ 27.,  30.,  31.,  32.,  33.,  34.,  35.,  36.,  37.,  38.,  39.,
                            40.,  41.,  42.,  43.,  44.,  45.,  46.,  47.,  48.,  49.,  50.,
                            51.,  52.,  53.,  54.,  55.,  56.,  57.,  58.,  59.,  60.,  61.,
                            62.,  63.,  64.,  65.,  66.,  67.,  68.,  69.,  70.,  71.,  72.,
                            73.,  74.,  75.,  76.,  77.,  78.,  79.,  80.,  81.,  82.,  83.,
                            84.,  85.,  86.,  87.,  88.,  89.,  90.,  91.,  92.,  95.,  98.,
                            99., 100.]))

    optiontypes_ttms = (array(['P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C'], dtype='<U1'),
                        array(['P', 'P', 'P', 'P', 'P', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C'],
                              dtype='<U1'),
                        array(['P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'P', 'P', 'P', 'P', 'P', 'P', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C'], dtype='<U1'),
                        array(['P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C'], dtype='<U1'))

    data = OptionChain(ids=ids,
                       ttms=ttms,
                       ticker=ticker,
                       forwards=forwards,
                       strikes_ttms=strikes_ttms,
                       optiontypes_ttms=optiontypes_ttms,
                       discfactors=discfactors,
                       bid_ivs=bid_ivs,
                       ask_ivs=ask_ivs)

    return data


def get_spy_test_chain_data() -> OptionChain:
    """
    SPY implied vols of 15_Jul_2022_16_23_09
    """
    ids = array(['2w', '1m', '2m', '6m'])
    ticker = 'SPY'
    ttms = array([0.056437205212352105, 0.09532609410124099, 0.17310387187901877, 0.4258816496567966])
    forwards = array([382.83312499999994, 382.9625, 382.97565789473686, 384.54092105263146])
    discfactors = array([0.9988269169205102, 0.9980054593706331, 0.9959230968030199, 0.9882261172126617])
    bid_ivs = (array([0.29607985, 0.29731007, 0.29593791, 0.29425093, 0.29003112,
                      0.28805134, 0.28051652, 0.28066925, 0.28083241, 0.27953194,
                      0.27727477, 0.2752893 , 0.27220861, 0.27091042, 0.26923654,
                      0.26100521, 0.26659025, 0.26555018, 0.26504592, 0.26443846,
                      0.26357189, 0.26070013, 0.256557  , 0.25730068, 0.25703232,
                      0.2533645 , 0.25438713, 0.24710238, 0.25010514, 0.24552158,
                      0.24532887, 0.24585186, 0.2462478 , 0.23523199, 0.24125643,
                      0.23862202, 0.23132075, 0.23380836, 0.23531829, 0.22605816,
                      0.22710072, 0.22956673, 0.22664138, 0.22743158, 0.22586901,
                      0.21922076, 0.2129792 , 0.21454468, 0.21323002, 0.21069765,
                      0.2078028 , 0.20650794, 0.20230557, 0.20136065, 0.19655681,
                      0.19463683, 0.1895964 , 0.19144884, 0.18944655, 0.18657273,
                      0.18772489]),
               array([0.32121802, 0.31852404, 0.31648031, 0.31431019, 0.31270748,
                      0.31094641, 0.30514354, 0.30698538, 0.30540038, 0.30365034,
                      0.30174459, 0.29969132, 0.29854699, 0.29568047, 0.29071779,
                      0.29395945, 0.29208117, 0.28915001, 0.28914659, 0.2880262 ,
                      0.28669497, 0.28555557, 0.28382435, 0.28191296, 0.28162193,
                      0.27581979, 0.27448201, 0.27393215, 0.27473777, 0.27458424,
                      0.27324143, 0.27138484, 0.27078483, 0.26964635, 0.26716791,
                      0.26641187, 0.2654037 , 0.26519172, 0.26266678, 0.25567833,
                      0.25901421, 0.25927441, 0.25066357, 0.25541192, 0.25371644,
                      0.25041288, 0.25140573, 0.24452324, 0.2471859 , 0.24645104,
                      0.24392777, 0.24158206, 0.24025345, 0.23674076, 0.23359756,
                      0.2335404 , 0.23166191, 0.23075822, 0.23060324, 0.22405105,
                      0.22263983, 0.22090817, 0.21948746, 0.21773535, 0.21433314,
                      0.21582673, 0.2141457 , 0.20983448, 0.20782971, 0.20778686,
                      0.20501548, 0.20327387, 0.20340132, 0.20008691, 0.20028045,
                      0.19710432, 0.19625967, 0.19592102, 0.19241685, 0.19349771,
                      0.19230045, 0.19065747, 0.18813165, 0.18818007, 0.18747655,
                      0.18635472, 0.18522832, 0.18611534, 0.18413163, 0.1843673 ,
                      0.18547994, 0.18386696, 0.18440508, 0.18322647, 0.18387931]),
               array([0.3478838 , 0.34500926, 0.3426879 , 0.34144519, 0.34010873,
                      0.3370858 , 0.3335153 , 0.33355355, 0.33043313, 0.32975018,
                      0.32655702, 0.32429015, 0.32471658, 0.32318495, 0.31983346,
                      0.31987494, 0.31644558, 0.31504289, 0.31433945, 0.31273871,
                      0.3102991 , 0.30855078, 0.31066756, 0.30833474, 0.30798776,
                      0.30350007, 0.30394912, 0.30389011, 0.3027428 , 0.30027394,
                      0.29982917, 0.29779249, 0.29767032, 0.29544903, 0.29452059,
                      0.29267039, 0.29308456, 0.29255311, 0.28988363, 0.29010061,
                      0.28846691, 0.28744255, 0.28629685, 0.28503362, 0.28276215,
                      0.28304525, 0.28208058, 0.2790805 , 0.2799839 , 0.27803173,
                      0.27598081, 0.27602314, 0.27414113, 0.27389356, 0.27311334,
                      0.2703177 , 0.26854933, 0.26886583, 0.26793026, 0.26703741,
                      0.26388689, 0.26379329, 0.26250525, 0.25920059, 0.26038108,
                      0.25633784, 0.25670608, 0.25393237, 0.25235444, 0.25194278,
                      0.246194  , 0.24486198, 0.24898572, 0.2430286 , 0.24537102,
                      0.2410524 , 0.24006653, 0.24033839, 0.23611782, 0.23726023,
                      0.23541611, 0.23278094, 0.23311828, 0.22983069, 0.22636184,
                      0.22714012, 0.22712252, 0.22149985, 0.22321244, 0.21959144,
                      0.21525168, 0.21463178, 0.2138293 , 0.21114815, 0.21028582,
                      0.20818671, 0.20920461, 0.20578578, 0.20499562, 0.20659932,
                      0.20166515, 0.2006052 , 0.19991994, 0.19821735, 0.19708929,
                      0.19743692, 0.19740709, 0.19405013, 0.19450075, 0.19240581,
                      0.19148646, 0.19288825, 0.18998019, 0.1913487 , 0.18808909,
                      0.18943793, 0.18917417, 0.18647328, 0.18671203, 0.18539452,
                      0.18524776, 0.18605149, 0.18634916, 0.18526447, 0.1838923 ,
                      0.18540066, 0.18354752, 0.18433781, 0.18398173, 0.18451371,
                      0.18377685]),
               array([0.38723977, 0.37891047, 0.36943703, 0.36367155, 0.3545992 ,
                      0.34710497, 0.34130384, 0.33460932, 0.32814719, 0.32104371,
                      0.31674204, 0.31097539, 0.30565289, 0.30017854, 0.29225279,
                      0.28949757, 0.28571658, 0.28449118, 0.28334678, 0.28165855,
                      0.2807965 , 0.27927143, 0.27964431, 0.27754188, 0.27743265,
                      0.27631343, 0.27409037, 0.27462723, 0.27324279, 0.2712397 ,
                      0.27124705, 0.26779881, 0.2682599 , 0.26731067, 0.26697204,
                      0.2645917 , 0.26381498, 0.2630903 , 0.26263235, 0.26232865,
                      0.26100091, 0.2583453 , 0.2587194 , 0.25649665, 0.25674509,
                      0.25431595, 0.25444096, 0.25222082, 0.25088053, 0.24938115,
                      0.24659526, 0.24396465, 0.24464775, 0.24729618, 0.24479171,
                      0.24101618, 0.24356132, 0.23667549, 0.24118966, 0.23917673,
                      0.2388148 , 0.23657432, 0.23638525, 0.23361059, 0.23399611,
                      0.23290333, 0.23032708, 0.23061567, 0.22738879, 0.22825451,
                      0.22640678, 0.22693944, 0.22679462, 0.22380235, 0.22258412,
                      0.2168135 , 0.22044214, 0.21983557, 0.21799529, 0.21606765,
                      0.21426376, 0.2147441 , 0.21103322, 0.21147703, 0.21020308,
                      0.20973474, 0.20817888, 0.20732447, 0.20765191, 0.20583131,
                      0.20543262, 0.20365225, 0.20248598, 0.20208008, 0.20110416,
                      0.20041326, 0.2006544 , 0.19877647, 0.19795511, 0.19784216,
                      0.19712069, 0.1963113 , 0.19470994, 0.19498247, 0.19461996,
                      0.19359065, 0.19215558, 0.19167593, 0.19079878, 0.19094315,
                      0.18939327, 0.1888834 , 0.18880081, 0.18899839, 0.18789245,
                      0.18740171, 0.1870114 , 0.18768504, 0.18656839, 0.18693481,
                      0.18540812, 0.1850047 , 0.18495151, 0.18438684, 0.18441508,
                      0.18414719, 0.1833191 , 0.18362255, 0.18337064, 0.18251758,
                      0.18236251, 0.18241393, 0.1821206 , 0.18204191, 0.18279784,
                      0.18230142, 0.18203106, 0.18135318, 0.18159097, 0.18128285]))

    ask_ivs = (array([0.29801977, 0.29816088, 0.29674701, 0.29502264, 0.29144236,
                      0.2886924 , 0.28231371, 0.28175304, 0.28237941, 0.28101741,
                      0.27819045, 0.27656223, 0.27339881, 0.2720625 , 0.27035402,
                      0.26173881, 0.26764382, 0.2665747 , 0.26639335, 0.26576857,
                      0.2645448 , 0.2616523 , 0.25780204, 0.25851781, 0.25793612,
                      0.25426172, 0.25556045, 0.24797076, 0.25152842, 0.24664656,
                      0.24644704, 0.24751948, 0.24762403, 0.23632527, 0.24261435,
                      0.23997457, 0.23294177, 0.23488824, 0.23693825, 0.22768219,
                      0.22846025, 0.23120855, 0.2282909 , 0.22853635, 0.22726603,
                      0.22007332, 0.21414015, 0.21572726, 0.21444076, 0.21163197,
                      0.20876831, 0.20750651, 0.20302876, 0.20214931, 0.19744431,
                      0.19563842, 0.19069986, 0.1926083 , 0.19079987, 0.18738816,
                      0.18867061]),
               array([0.32199724, 0.31928134, 0.31721275, 0.31501936, 0.31339146,
                      0.31160702, 0.30579965, 0.3076039 , 0.30599754, 0.30422768,
                      0.30230346, 0.30023292, 0.29906926, 0.29618918, 0.29171896,
                      0.29443211, 0.2929969 , 0.29004291, 0.29000552, 0.2888574 ,
                      0.2870981 , 0.28594616, 0.2842039 , 0.2826509 , 0.28233607,
                      0.27652443, 0.27516681, 0.27459648, 0.27538037, 0.27520832,
                      0.27384994, 0.27227617, 0.27165367, 0.27021224, 0.26772241,
                      0.26722461, 0.26619911, 0.26648833, 0.2634313 , 0.25618207,
                      0.26024365, 0.25999864, 0.25161956, 0.25588028, 0.25440897,
                      0.25109679, 0.25208023, 0.24541393, 0.24850554, 0.2477577 ,
                      0.24479156, 0.24243934, 0.24110506, 0.23801129, 0.23465174,
                      0.23459123, 0.23229121, 0.23159676, 0.23165216, 0.22489212,
                      0.22348398, 0.22154461, 0.22012821, 0.21838155, 0.21520414,
                      0.21648683, 0.21481482, 0.21051557, 0.20852335, 0.2084931 ,
                      0.2054976 , 0.20401486, 0.20390696, 0.20087023, 0.20081695,
                      0.19766178, 0.19741207, 0.19651791, 0.19304422, 0.19414568,
                      0.19297816, 0.19137004, 0.18851015, 0.18857618, 0.1883104 ,
                      0.18679755, 0.18569934, 0.18709867, 0.18466298, 0.18492868,
                      0.18606779, 0.18450357, 0.18507792, 0.18395526, 0.18465031]),
               array([0.34847876, 0.34559441, 0.34326119, 0.34255828, 0.34065151,
                      0.33762013, 0.33404319, 0.33406293, 0.33093486, 0.33023681,
                      0.32703652, 0.32475991, 0.32516944, 0.32362683, 0.32026921,
                      0.32029643, 0.3172762 , 0.31585318, 0.31512715, 0.31350829,
                      0.31067707, 0.30965759, 0.31102241, 0.30868292, 0.30866447,
                      0.30417062, 0.30459927, 0.30420641, 0.30305188, 0.30088091,
                      0.30042097, 0.29837315, 0.29823626, 0.2960049 , 0.29506428,
                      0.29320392, 0.2936046 , 0.29306178, 0.29038438, 0.29058976,
                      0.28894732, 0.28791378, 0.28675938, 0.28548789, 0.28343289,
                      0.28370232, 0.28251112, 0.27971759, 0.28040024, 0.27864702,
                      0.27658737, 0.27642072, 0.27551259, 0.27447251, 0.27349381,
                      0.2706937 , 0.26910619, 0.26941482, 0.26847241, 0.26757307,
                      0.26441722, 0.26501601, 0.26319681, 0.26022857, 0.26088966,
                      0.25701056, 0.25720602, 0.25459386, 0.25284685, 0.25292044,
                      0.24684202, 0.24550607, 0.24962598, 0.24398468, 0.24600562,
                      0.24184285, 0.24069702, 0.24096742, 0.23674581, 0.23788763,
                      0.23588654, 0.23356537, 0.23406071, 0.2304603 , 0.22715125,
                      0.2277738 , 0.22775887, 0.22214028, 0.22369537, 0.22024055,
                      0.21590719, 0.21512793, 0.21449752, 0.21452792, 0.21079971,
                      0.20888201, 0.20973247, 0.2063238 , 0.20554268, 0.20733835,
                      0.20223467, 0.20118677, 0.20051399, 0.19862363, 0.19750593,
                      0.19786261, 0.19827803, 0.19450271, 0.19496429, 0.1928866 ,
                      0.19223119, 0.19339585, 0.19051195, 0.19162092, 0.18866291,
                      0.19002564, 0.18978335, 0.18743883, 0.18704612, 0.18609548,
                      0.18561322, 0.18680452, 0.18674063, 0.18567711, 0.18476463,
                      0.18584918, 0.1845037 , 0.18483438, 0.18450403, 0.18505684,
                      0.18435235]),
               array([0.38795822, 0.37957913, 0.37006381, 0.36424896, 0.35514061,
                      0.34786141, 0.34200505, 0.33504539, 0.32896076, 0.32161596,
                      0.31727515, 0.31164236, 0.30627999, 0.30106577, 0.29281515,
                      0.290162  , 0.28635529, 0.28512408, 0.28384855, 0.28215614,
                      0.28141279, 0.28268952, 0.28024947, 0.27814253, 0.2781469 ,
                      0.27702197, 0.2746767 , 0.27520837, 0.27381969, 0.27192726,
                      0.27181555, 0.26836405, 0.2703905 , 0.2679792 , 0.26763604,
                      0.26525187, 0.26447103, 0.2637424 , 0.2632806 , 0.26286579,
                      0.2616421 , 0.25898343, 0.25935423, 0.25712862, 0.25737413,
                      0.2549424 , 0.25506481, 0.25294594, 0.25149983, 0.25226142,
                      0.25172261, 0.24672548, 0.24536161, 0.24800836, 0.24550237,
                      0.24172546, 0.24426953, 0.244859  , 0.24199709, 0.24008435,
                      0.23952038, 0.2374813 , 0.23719152, 0.23431638, 0.23460147,
                      0.23391328, 0.23113623, 0.23142606, 0.22820105, 0.22896671,
                      0.22722305, 0.22755329, 0.22741038, 0.22452385, 0.22320517,
                      0.22274604, 0.22106913, 0.22057077, 0.21852375, 0.21681251,
                      0.21490691, 0.21539103, 0.21549056, 0.21213488, 0.21086676,
                      0.21029247, 0.21087949, 0.2080072 , 0.20822567, 0.20641214,
                      0.2060195 , 0.20424709, 0.20320898, 0.20268981, 0.20172223,
                      0.20103968, 0.20116134, 0.19929247, 0.19847911, 0.19837334,
                      0.19779523, 0.19699733, 0.19555001, 0.19555062, 0.19519768,
                      0.19432722, 0.1927586 , 0.19229047, 0.19158331, 0.19158153,
                      0.19004878, 0.18972   , 0.18931301, 0.18951974, 0.18860591,
                      0.18794968, 0.18775852, 0.1882545 , 0.18715479, 0.18733326,
                      0.18602607, 0.18563865, 0.18560014, 0.18505388, 0.18509754,
                      0.18484767, 0.183802  , 0.18411574, 0.18387745, 0.18278026,
                      0.18290153, 0.18296663, 0.18268985, 0.18233495, 0.18309588,
                      0.18291639, 0.18266518, 0.18201105, 0.18192885, 0.18166336]))

    strikes_ttms = (array([342. , 344. , 345. , 346. , 348. , 350. , 352. , 354. , 355. ,
                           356. , 358. , 360. , 362. , 363. , 364. , 365. , 366. , 367. ,
                           367.5, 368. , 369. , 370. , 371. , 372. , 372.5, 373. , 374. ,
                           375. , 376. , 377. , 377.5, 378. , 379. , 380. , 381. , 382. ,
                           382.5, 383. , 384. , 385. , 386. , 387. , 387.5, 388. , 389. ,
                           390. , 391. , 392. , 393. , 394. , 395. , 396. , 398. , 400. ,
                           402. , 404. , 405. , 406. , 408. , 410. , 412. ]),
                    array([327., 328., 329., 330., 331., 332., 333., 334., 335., 336., 337.,
                           338., 339., 340., 341., 342., 343., 344., 345., 346., 347., 348.,
                           349., 350., 351., 352., 353., 354., 355., 356., 357., 358., 359.,
                           360., 361., 362., 363., 364., 365., 366., 367., 368., 369., 370.,
                           371., 372., 373., 374., 375., 376., 377., 378., 379., 380., 381.,
                           382., 383., 384., 385., 386., 387., 388., 389., 390., 391., 392.,
                           393., 394., 395., 396., 397., 398., 399., 400., 401., 402., 403.,
                           404., 405., 406., 407., 408., 409., 410., 411., 412., 413., 414.,
                           415., 416., 417., 418., 419., 420., 421.]),
                    array([305., 306., 307., 308., 309., 310., 311., 312., 313., 314., 315.,
                           316., 317., 318., 319., 320., 321., 322., 323., 324., 325., 326.,
                           327., 328., 329., 330., 331., 332., 333., 334., 335., 336., 337.,
                           338., 339., 340., 341., 342., 343., 344., 345., 346., 347., 348.,
                           349., 350., 351., 352., 353., 354., 355., 356., 357., 358., 359.,
                           360., 361., 362., 363., 364., 365., 366., 367., 368., 369., 370.,
                           371., 372., 373., 374., 375., 376., 377., 378., 379., 380., 381.,
                           382., 383., 384., 385., 386., 387., 388., 389., 390., 391., 392.,
                           393., 394., 395., 396., 397., 398., 399., 400., 401., 402., 403.,
                           404., 405., 406., 407., 408., 409., 410., 411., 412., 413., 414.,
                           415., 416., 417., 418., 419., 420., 421., 422., 423., 424., 425.,
                           426., 427., 428., 429., 430., 431., 432., 433., 434., 435.]),
                    array([265., 270., 275., 280., 285., 290., 295., 300., 305., 310., 315.,
                           320., 325., 330., 335., 340., 344., 345., 346., 347., 348., 349.,
                           350., 351., 352., 353., 354., 355., 356., 357., 358., 359., 360.,
                           361., 362., 363., 364., 365., 366., 367., 368., 369., 370., 371.,
                           372., 373., 374., 375., 376., 377., 378., 379., 380., 381., 382.,
                           383., 384., 385., 386., 387., 388., 389., 390., 391., 392., 393.,
                           394., 395., 396., 397., 398., 399., 400., 401., 402., 403., 404.,
                           405., 406., 407., 408., 409., 410., 411., 412., 413., 414., 415.,
                           416., 417., 418., 419., 420., 421., 422., 423., 424., 425., 426.,
                           427., 428., 429., 430., 431., 432., 433., 434., 435., 436., 437.,
                           438., 439., 440., 441., 442., 443., 444., 445., 446., 447., 448.,
                           449., 450., 451., 452., 453., 454., 455., 456., 457., 458., 459.,
                           460., 461., 462., 463., 464., 465., 466., 470.]))

    optiontypes_ttms = (array(['P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C'], dtype='<U1'),
                        array(['P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'P', 'P', 'P', 'P', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C'], dtype='<U1'),
                        array(['P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C'], dtype='<U1'),
                        array(['P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                               'P', 'P', 'P', 'P', 'P', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C',
                               'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C', 'C'], dtype='<U1'))

    data = OptionChain(ids=ids,
                       ttms=ttms,
                       ticker=ticker,
                       forwards=forwards,
                       strikes_ttms=strikes_ttms,
                       optiontypes_ttms=optiontypes_ttms,
                       discfactors=discfactors,
                       bid_ivs=bid_ivs,
                       ask_ivs=ask_ivs)

    return data


def get_qv_options_test_chain_data(num_strikes: int = 21) -> OptionChain:
    """
    BTC implied vols of 21Oct2021
    """
    ids = array(['1w', '2w', '1m', '3m', '6m', '12m'])
    ttms = array([7.0/365.0, 14.0/365.0, 0.083333333, 0.25, 0.5, 1.0])
    forwards = array([1.0, 1.0, 1.0, 1.0, 1.0, 1.0])
    discfactors = array([1.0, 1.0, 1.0, 1.0, 1.0, 1.0])
    strikes_ttm = np.linspace(0.75, 1.5, num_strikes)
    strikes_ttms = (strikes_ttm, strikes_ttm, strikes_ttm, strikes_ttm, strikes_ttm, strikes_ttm)

    optiontypes_ttm = np.full(strikes_ttm.shape, 'C')
    optiontypes_ttms = (optiontypes_ttm, optiontypes_ttm, optiontypes_ttm, optiontypes_ttm, optiontypes_ttm, optiontypes_ttm)

    data = OptionChain(ids=ids,
                       ttms=ttms,
                       ticker='BTC',
                       forwards=forwards,
                       strikes_ttms=strikes_ttms,
                       optiontypes_ttms=optiontypes_ttms,
                       discfactors=discfactors,
                       bid_ivs=None,
                       ask_ivs=None)

    return data


class LocalTests(Enum):
    BTC = 1
    VIX = 2
    GLD = 3
    SQQQ = 4
    SPY = 5


def run_local_test(local_test: LocalTests):
    """Run local tests for development and debugging purposes.

    These are integration tests that download real data and generate reports.
    Use for quick verification during development.
    """

    if local_test == LocalTests.BTC:
        option_chain = get_btc_test_chain_data()
        print(option_chain)

    elif local_test == LocalTests.VIX:
        option_chain = get_vix_test_chain_data()
        print(option_chain)

    elif local_test == LocalTests.GLD:
        option_chain = get_gld_test_chain_data()
        print(option_chain)

    elif local_test == LocalTests.SQQQ:
        option_chain = get_sqqq_test_chain_data()
        print(option_chain)

    elif local_test == LocalTests.SPY:
        option_chain = get_spy_test_chain_data()
        print(option_chain)


if __name__ == '__main__':

    run_local_test(local_test=LocalTests.BTC)
