"""A simple calculator class with memory and basic operations.

This module defines a `Calculator` that maintains an internal memory and
supports arithmetic operations while validating inputs. Docstrings follow
the Google style guide.
"""

import math

class Calculator:
    """Calculator with an internal memory register.

    The calculator stores a floating-point `memory` value and exposes methods
    to mutate it via arithmetic operations (add, subtract, multiply, divide)
    as well as utility operations like square root accumulation and memory reset.
    """
      
    def __init__(self) -> None:
        """Initialize the calculator with memory set to 0.0."""
        self._memory: float = 0.0 #._memory is non-public variable named with PEP8 convention
    
    def __str__(self) -> str:
        """Return a readable representation of the calculator.

        Returns:
            str: A string showing the current memory value.
        """
        return f"Calculator with memory {self._memory}"
    
    @property
    def memory(self) -> float:
         """Current value stored in the calculator's memory.

        Returns:
            float: The current memory value.
        """
         return self._memory
    
    @memory.setter
    def memory(self, value: float) -> None:
        """Disallow direct assignment to memory.

        Attempting to set the memory directly raises an AttributeError.
        Use the provided arithmetic methods or `memory_reset()` instead.

        Args:
            value (float): Ignored.

        Raises:
            AttributeError: Always raised to enforce read-only memory property.
        """
        raise AttributeError("memory is read-only")
    
    @staticmethod
    def _to_float(x: any) -> float:
        """Validate and convert a numeric input to float.

        Accepts `int` or `float`. Other types raise a `TypeError`.

        Args:
            x (Any): The value to validate and convert.

        Returns:
            float: The converted floating-point value.

        Raises:
            TypeError: If `x` is not an `int` or `float`.
        """
        if not isinstance(x, (int, float)):
            raise TypeError("Argument must be an int or float")
        x = float(x)
        return x
     
    def add(self, n: float) -> float:
        """Add `n` to memory.

        Args:
            n (float): The value to add.

        Returns:
            float: The updated memory value.
        """
        n = self._to_float(n)
        self._memory += n
        return self._memory

    def subtract(self, n: float) -> float:
        """Subtract `n` from memory.

        Args:
            n (float): The value to subtract.

        Returns:
            float: The updated memory value.
        """
        n = self._to_float(n)
        self._memory -= n
        return self._memory

    def multiply(self, n: float) -> float:
        """Multiply memory by `n`.

        Args:
            n (float): The multiplier.

        Returns:
            float: The updated memory value.
        """
        n = self._to_float(n)
        self._memory*=n
        return self._memory

    def divide(self, n: float) -> float:
        """Divide memory by `n`.

        Args:
            n (float): The divisor.

        Returns:
            float: The updated memory value.

        Raises:
            ZeroDivisionError: If `n` equals zero.
        """
        n = self._to_float(n)
        if n == 0.0:
            raise ZeroDivisionError("division by zero")
        else:
            self._memory/=n
        return self._memory
    
    def sqrt(self, n: float) -> float:
         """Add the square root of n to memory.

        Sqrt() does not replace memory with the square root; it adds it to
        the current memory value.

        Args:
            n (float): The value whose square root will be added.

        Returns:
            float: The updated memory value.

        Raises:
            ValueError: If `n` is negative (no real square root).
        """
         n = self._to_float(n)
         if n <0:
            raise ValueError("Cannot take square root of a negative real number")
         self._memory += math.sqrt(n)
         return self._memory
    
    def memory_reset(self) -> float:
        """Reset memory to 0.0.

        Returns:
            float: The reset memory value (0.0).
        """
        self._memory = 0.0
        return self._memory

def main():
        c = Calculator()

if __name__ == "__main__":
        main()