# Calculator — Simple Memory-Based Arithmetic Module

*A tiny Python class that keeps a running **memory** and performs validated arithmetic operations.*

> This document explains the module, how to install/use it. It also includes a mini‑tutorial in English

---

## Features

- `Calculator.memory` — read‑only property exposing the current memory value.
- Validated inputs: only `int` or `float` are accepted; other types raise `TypeError`.
- Arithmetic: `add`, `subtract`, `multiply`, `divide`.
- Utilities: `sqrt` (adds √n to memory), `memory_reset` (resets to `0.0`).
- Clear, Google‑style docstrings and helpful error messages.

---

## Installation

This is a single‑file module. You can:
- drop `calculator.py` into your project, or
- install it as an editable local package if you later add `pyproject.toml`/`setup.cfg`.

**Python version:** 3.9+ recommended.

---

## Quick Start (Tutorial)

### 1) Import and construct
```python
from calculator import Calculator

c = Calculator()
print(c.memory)   # 0.0
```

### 2) Basic operations
```python
c.add(10)         # -> 10.0
c.subtract(3.5)   # -> 6.5
c.multiply(2)     # -> 13.0
c.divide(4)       # -> 3.25
```

### 3) Square root accumulation
```python
c.sqrt(9)         # adds 3, -> 6.25
```

### 4) Reset
```python
c.memory_reset()  # -> 0.0
```

### 5) Defensive behavior (errors)
```python

try:
    c.add("3")  # not allowed
except TypeError as e:
    print(e)    # Argument must be an int or float

try:
    c.divide(0) # division by zero
except ZeroDivisionError as e:
    print(e)

try:
    c.sqrt(-1)  # negative sqrt
except ValueError as e:
    print(e)    # Cannot take square root of a negative real number

# read-only property
try:
    c.memory = 42
except AttributeError as e:
    print(e)    # memory is read-only
```


## API Reference

### Class: `Calculator()`
- **Properties**
  - `memory: float` — read‑only; returns the current memory value.
- **Methods**
  - `add(n: float) -> float` — Adds `n` to memory.
  - `subtract(n: float) -> float` — Subtracts `n` from memory.
  - `multiply(n: float) -> float` — Multiplies memory by `n`.
  - `divide(n: float) -> float` — Divides memory by `n`. Raises `ZeroDivisionError` if `n == 0`.
  - `sqrt(n: float) -> float` — Adds `sqrt(n)` to memory. Raises `ValueError` if `n < 0`.
  - `memory_reset() -> float` — Resets memory to `0.0`.

**Validation rule:** a private helper `_to_float` converts and validates inputs. Only `int` and `float` are accepted

---