import sys

# Version
try:
    from importlib.metadata import version as _pkg_version
    __version__ = _pkg_version("histotuner")
except Exception:
    __version__ = "unknown"

# Print only histotuner version on import
print(f"histotuner {__version__}")

# Provide a friendly greeting

def hello() -> str:
    return "Hello from histotuner!"

# Lightweight import (safe)
try:
    from .crop_image import cropImage  # lightweight
except Exception:
    # Lazy fallback to avoid noisy import-time errors
    def cropImage(*args, **kwargs):
        from .crop_image import cropImage as _fn
        return _fn(*args, **kwargs)

# Heavy deps: define lazy wrappers to avoid import-time initialization/prints (e.g., cellpose)

def segmentImage(*args, **kwargs):
    from .segment import segmentImage as _fn
    return _fn(*args, **kwargs)

# Tissue helpers (cv2/geopandas/shapely may be heavy): lazy wrappers

def tissueMask(*args, **kwargs):
    from .tissue import tissueMask as _fn
    return _fn(*args, **kwargs)

def generateTissueMask(*args, **kwargs):
    from .tissue import generateTissueMask as _fn
    return _fn(*args, **kwargs)

def BinaryMask(*args, **kwargs):
    from .tissue import BinaryMask as _fn
    return _fn(*args, **kwargs)

def binaryMaskToPolygons(*args, **kwargs):
    from .tissue import binaryMaskToPolygons as _fn
    return _fn(*args, **kwargs)

def binaryMaskToPolygonsWithProb(*args, **kwargs):
    from .tissue import binaryMaskToPolygonsWithProb as _fn
    return _fn(*args, **kwargs)

def overlayTissueThumbnail(*args, **kwargs):
    from .tissue import overlayTissueThumbnail as _fn
    return _fn(*args, **kwargs)

# Embedding pipeline: lazy wrappers

def load_config(*args, **kwargs):
    from .embedImage import load_config as _fn
    return _fn(*args, **kwargs)

def loadHistoEmbedConfig(*args, **kwargs):
    """Alias for embedding config loader to avoid confusion with training config."""
    from .embedImage import load_config as _fn
    return _fn(*args, **kwargs)

def runEmbeddingPipeline(*args, **kwargs):
    from .embedImage import runEmbeddingPipeline as _fn
    return _fn(*args, **kwargs)

def embedImageFromDict(*args, **kwargs):
    from .embedImage import embedImageFromDict as _fn
    return _fn(*args, **kwargs)

def embedImageFromYaml(*args, **kwargs):
    from .embedImage import embedImageFromYaml as _fn
    return _fn(*args, **kwargs)

# Training helpers (histotune): lazy wrappers

def loadHistotuneConfig(*args, **kwargs):
    from .histotune import load_config as _fn
    return _fn(*args, **kwargs)

def runHistotunePipeline(*args, **kwargs):
    from .histotune import runTrainingPipeline as _fn
    return _fn(*args, **kwargs)

def runHistotuneFromYaml(*args, **kwargs):
    from .histotune import runTrainingFromYaml as _fn
    return _fn(*args, **kwargs)

def runHistotuneTraining(*args, **kwargs):
    from .histotune import run_training as _fn
    return _fn(*args, **kwargs)

# Spatial feature table helpers: lazy wrappers

def spatialFeatureTable(*args, **kwargs):
    from .spatialfeaturetable import spatialFeatureTable as _fn
    return _fn(*args, **kwargs)

def spatialFeatureTableFromDict(*args, **kwargs):
    from .spatialfeaturetable import spatialFeatureTableFromDict as _fn
    return _fn(*args, **kwargs)

# Token→cell mapper utilities: lazy wrappers

def token_to_cell_mapper(*args, **kwargs):
    from .token_cell_mapper import token_to_cell_mapper as _fn
    return _fn(*args, **kwargs)

def attach_token_cell_alignment(*args, **kwargs):
    from .token_cell_mapper import attach_token_cell_alignment as _fn
    return _fn(*args, **kwargs)

def save_spatialdata_to_zarr(*args, **kwargs):
    from .token_cell_mapper import save_spatialdata_to_zarr as _fn
    return _fn(*args, **kwargs)

def tokenCellMapper(*args, **kwargs):
    from .token_cell_mapper import tokenCellMapper as _fn
    return _fn(*args, **kwargs)

# Cell phenotype mapper utilities: lazy wrappers

def attach_cell_phenotype(*args, **kwargs):
    from .cell_phenotype_mapper import attach_cell_phenotype as _fn
    return _fn(*args, **kwargs)

def attach_cell_phenotype_to_zarr(*args, **kwargs):
    from .cell_phenotype_mapper import attach_cell_phenotype_to_zarr as _fn
    return _fn(*args, **kwargs)

def cellPhenotypeMapper(*args, **kwargs):
    from .cell_phenotype_mapper import cellPhenotypeMapper as _fn
    return _fn(*args, **kwargs)

def phenotypeCellMapper(*args, **kwargs):
    from .cell_phenotype_mapper import phenotypeCellMapper as _fn
    return _fn(*args, **kwargs)

__all__ = [
    "__version__",
    "hello",
    "cropImage",
    "segmentImage",
    "tissueMask",
    "generateTissueMask",
    "BinaryMask",
    "binaryMaskToPolygons",
    "binaryMaskToPolygonsWithProb",
    "overlayTissueThumbnail",
    # Embedding pipeline (image)
    "load_config",
    "loadHistoEmbedConfig",
    "runEmbeddingPipeline",
    "embedImageFromDict",
    "embedImageFromYaml",
    # Training pipeline (histotune)
    "loadHistotuneConfig",
    "runHistotunePipeline",
    "runHistotuneFromYaml",
    "runHistotuneTraining",
    # Spatial feature table and mappers
    "spatialFeatureTable",
    "spatialFeatureTableFromDict",
    "token_to_cell_mapper",
    "attach_token_cell_alignment",
    "save_spatialdata_to_zarr",
    "tokenCellMapper",
    "attach_cell_phenotype",
    "attach_cell_phenotype_to_zarr",
    "cellPhenotypeMapper",
    "phenotypeCellMapper",
]
