"""
add OrderIntakeDaily, DeliveriesByDay, ChangeLogCommitted tables

Revision ID: c76cf026488e
Revises: 3fa4e62e787b
Create Date: 2025-09-18 16:52:48.058722

"""

from collections.abc import Sequence

import sqlalchemy as sa
from sqlalchemy.dialects import postgresql

from alembic import op

# revision identifiers, used by Alembic.
revision: str = "c76cf026488e"
down_revision: str | Sequence[str] | None = "3fa4e62e787b"
branch_labels: str | Sequence[str] | None = None
depends_on: str | Sequence[str] | None = None


def upgrade() -> None:
    """Upgrade schema."""
    # ### commands auto generated by Alembic - please adjust! ###

    # Ensure TimescaleDB extension exists
    bind = op.get_bind()
    bind.execute(sa.text("CREATE EXTENSION IF NOT EXISTS timescaledb;"))
    op.create_table(
        "change_log_committed",
        sa.Column("company_id", sa.Integer(), nullable=False),
        sa.Column("sku_id", sa.Integer(), nullable=False),
        sa.Column("delivery_day", sa.Date(), nullable=False),
        sa.Column("valid_from", postgresql.TIMESTAMP(timezone=True), nullable=False),
        sa.Column("committed_qty", sa.DOUBLE_PRECISION(), nullable=False),
        sa.ForeignKeyConstraint(
            ["company_id", "sku_id"],
            ["sku_table.company_id", "sku_table.sku_id"],
            name="fk_clc_sku",
            onupdate="RESTRICT",
            ondelete="CASCADE",
        ),
        sa.PrimaryKeyConstraint("company_id", "sku_id", "delivery_day", "valid_from", name="pk_change_log_committed"),
        info={
            "timescale": {
                "time_column": "valid_from",
                "space_column": "sku_id",
                "number_partitions": 32,
                "chunk_interval": "14 days",
                "compression": {
                    "enabled": True,
                    "segmentby": ["company_id", "sku_id", "delivery_day"],
                    "orderby": "valid_from DESC",
                    "compress_after": "30 days",
                },
                "caggs": [],
            }
        },
    )
    op.create_index(
        "ix_clc_asof", "change_log_committed", ["company_id", "sku_id", "delivery_day", "valid_from"], unique=False
    )
    op.create_index(
        "ix_clc_company_delivery", "change_log_committed", ["company_id", "delivery_day", "valid_from"], unique=False
    )
    op.create_table(
        "deliveries_by_day",
        sa.Column("company_id", sa.Integer(), nullable=False),
        sa.Column("sku_id", sa.Integer(), nullable=False),
        sa.Column("delivery_day", sa.Date(), nullable=False),
        sa.Column("total_qty", sa.DOUBLE_PRECISION(), nullable=False),
        sa.ForeignKeyConstraint(
            ["company_id", "sku_id"],
            ["sku_table.company_id", "sku_table.sku_id"],
            name="fk_dbd_sku",
            onupdate="RESTRICT",
            ondelete="CASCADE",
        ),
        sa.PrimaryKeyConstraint("company_id", "sku_id", "delivery_day", name="pk_deliveries_by_day"),
        info={
            "timescale": {
                "time_column": "delivery_day",
                "space_column": "sku_id",
                "number_partitions": 32,
                "chunk_interval": "14 days",
                "compression": {
                    "enabled": True,
                    "segmentby": ["company_id", "sku_id"],
                    "orderby": "delivery_day DESC",
                    "compress_after": "30 days",
                },
                "caggs": [],
            }
        },
    )
    op.create_index("ix_dbd_company_day", "deliveries_by_day", ["company_id", "delivery_day"], unique=False)
    op.create_index(
        "ix_dbd_company_sku_day", "deliveries_by_day", ["company_id", "sku_id", "delivery_day"], unique=False
    )
    op.create_table(
        "order_intake_daily",
        sa.Column("company_id", sa.Integer(), nullable=False),
        sa.Column("sku_id", sa.Integer(), nullable=False),
        sa.Column("order_day", sa.Date(), nullable=False),
        sa.Column("delivery_day", sa.Date(), nullable=False),
        sa.Column("qty_total", sa.DOUBLE_PRECISION(), nullable=False),
        sa.ForeignKeyConstraint(
            ["company_id", "sku_id"],
            ["sku_table.company_id", "sku_table.sku_id"],
            name="fk_oid_sku",
            onupdate="RESTRICT",
            ondelete="CASCADE",
        ),
        sa.PrimaryKeyConstraint("company_id", "sku_id", "order_day", "delivery_day", name="pk_order_intake_daily"),
        info={
            "timescale": {
                "time_column": "order_day",
                "space_column": "sku_id",
                "number_partitions": 32,
                "chunk_interval": "14 days",
                "compression": {
                    "enabled": True,
                    "segmentby": ["company_id", "sku_id"],
                    "orderby": "order_day DESC",
                    "compress_after": "30 days",
                },
                "caggs": [],
            }
        },
    )
    op.create_index("ix_oid_company_orderday", "order_intake_daily", ["company_id", "order_day"], unique=False)
    op.create_index(
        "ix_oid_company_sku_orderday", "order_intake_daily", ["company_id", "sku_id", "order_day"], unique=False
    )
    op.drop_index(op.f("ix_tsf_company_feature_ts"), table_name="time_store_features")
    op.drop_index(op.f("ix_tsf_company_store_ts"), table_name="time_store_features")
    op.drop_table("time_store_features")

    # TimescaleDB: Convert new tables to hypertables

    # change_log_committed
    bind.execute(
        sa.text("""
      SELECT create_hypertable(
        'change_log_committed', 'valid_from',
        partitioning_column => 'sku_id',
        number_partitions   => 32,
        create_default_indexes => FALSE,
        if_not_exists => TRUE
      );
    """)
    )
    bind.execute(sa.text("SELECT set_chunk_time_interval('change_log_committed', INTERVAL '14 days');"))
    bind.execute(
        sa.text("""
      ALTER TABLE "change_log_committed"
      SET (timescaledb.compress = 'on',
           timescaledb.compress_orderby  = 'valid_from DESC',
           timescaledb.compress_segmentby = 'company_id,sku_id,delivery_day');
    """)
    )
    bind.execute(sa.text("SELECT add_compression_policy('change_log_committed', INTERVAL '30 days');"))

    # deliveries_by_day
    bind.execute(
        sa.text("""
      SELECT create_hypertable(
        'deliveries_by_day', 'delivery_day',
        partitioning_column => 'sku_id',
        number_partitions   => 32,
        create_default_indexes => FALSE,
        if_not_exists => TRUE
      );
    """)
    )
    bind.execute(sa.text("SELECT set_chunk_time_interval('deliveries_by_day', INTERVAL '14 days');"))
    bind.execute(
        sa.text("""
      ALTER TABLE "deliveries_by_day"
      SET (timescaledb.compress = 'on',
           timescaledb.compress_orderby  = 'delivery_day DESC',
           timescaledb.compress_segmentby = 'company_id,sku_id');
    """)
    )
    bind.execute(sa.text("SELECT add_compression_policy('deliveries_by_day', INTERVAL '30 days');"))

    # order_intake_daily
    bind.execute(
        sa.text("""
      SELECT create_hypertable(
        'order_intake_daily', 'order_day',
        partitioning_column => 'sku_id',
        number_partitions   => 32,
        create_default_indexes => FALSE,
        if_not_exists => TRUE
      );
    """)
    )
    bind.execute(sa.text("SELECT set_chunk_time_interval('order_intake_daily', INTERVAL '14 days');"))
    bind.execute(
        sa.text("""
      ALTER TABLE "order_intake_daily"
      SET (timescaledb.compress = 'on',
           timescaledb.compress_orderby  = 'order_day DESC',
           timescaledb.compress_segmentby = 'company_id,sku_id');
    """)
    )
    bind.execute(sa.text("SELECT add_compression_policy('order_intake_daily', INTERVAL '30 days');"))

    # ### end Alembic commands ###


def downgrade() -> None:
    """Downgrade schema."""
    # ### commands auto generated by Alembic - please adjust! ###
    op.create_table(
        "time_store_features",
        sa.Column("company_id", sa.INTEGER(), autoincrement=False, nullable=False),
        sa.Column("store_id", sa.INTEGER(), autoincrement=False, nullable=False),
        sa.Column("feature_id", sa.INTEGER(), autoincrement=False, nullable=False),
        sa.Column("ts", postgresql.TIMESTAMP(timezone=True), autoincrement=False, nullable=False),
        sa.Column("value", sa.DOUBLE_PRECISION(precision=53), autoincrement=False, nullable=True),
        sa.ForeignKeyConstraint(
            ["company_id", "feature_id"],
            ["feature_descriptions.company_id", "feature_descriptions.feature_id"],
            name=op.f("fk_tsf_features"),
            onupdate="RESTRICT",
            ondelete="CASCADE",
        ),
        sa.ForeignKeyConstraint(
            ["company_id", "store_id"],
            ["stores.company_id", "stores.store_id"],
            name=op.f("fk_tsf_stores"),
            onupdate="RESTRICT",
            ondelete="CASCADE",
        ),
        sa.PrimaryKeyConstraint("company_id", "store_id", "feature_id", "ts", name=op.f("pk_time_store_features")),
    )
    op.create_index(
        op.f("ix_tsf_company_store_ts"), "time_store_features", ["company_id", "store_id", "ts"], unique=False
    )
    op.create_index(
        op.f("ix_tsf_company_feature_ts"), "time_store_features", ["company_id", "feature_id", "ts"], unique=False
    )
    op.drop_index("ix_oid_company_sku_orderday", table_name="order_intake_daily")
    op.drop_index("ix_oid_company_orderday", table_name="order_intake_daily")
    op.drop_table("order_intake_daily")
    op.drop_index("ix_dbd_company_sku_day", table_name="deliveries_by_day")
    op.drop_index("ix_dbd_company_day", table_name="deliveries_by_day")
    op.drop_table("deliveries_by_day")
    op.drop_index("ix_clc_company_delivery", table_name="change_log_committed")
    op.drop_index("ix_clc_asof", table_name="change_log_committed")
    op.drop_table("change_log_committed")
    # ### end Alembic commands ###
