"""File management tools for Pydantic AI agents.

These tools are restricted to the .shotgun directory for security.
"""

from pathlib import Path
from typing import Literal

from pydantic_ai import RunContext

from shotgun.agents.models import AgentDeps, AgentType, FileOperationType
from shotgun.logging_config import get_logger
from shotgun.utils.file_system_utils import get_shotgun_base_path

logger = get_logger(__name__)

# Map agent modes to their allowed directories/files (in workflow order)
# Values can be:
# - A Path: exact file (e.g., Path("research.md"))
# - A list of Paths: multiple allowed files/directories (e.g., [Path("specification.md"), Path("contracts")])
# - "*": any file except protected files (for export agent)
AGENT_DIRECTORIES: dict[AgentType, str | Path | list[Path]] = {
    AgentType.RESEARCH: Path("research.md"),
    AgentType.SPECIFY: [
        Path("specification.md"),
        Path("contracts"),
    ],  # Specify can write specs and contract files
    AgentType.PLAN: Path("plan.md"),
    AgentType.TASKS: Path("tasks.md"),
    AgentType.EXPORT: "*",  # Export agent can write anywhere except protected files
}

# Files protected from export agent modifications
PROTECTED_AGENT_FILES = {
    "research.md",
    "specification.md",
    "plan.md",
    "tasks.md",
}


def _validate_agent_scoped_path(filename: str, agent_mode: AgentType | None) -> Path:
    """Validate and resolve a file path within the agent's scoped directory.

    Args:
        filename: Relative filename
        agent_mode: The current agent mode

    Returns:
        Absolute path to the file within the agent's scoped directory

    Raises:
        ValueError: If the path attempts to access files outside the agent's scope
    """
    base_path = get_shotgun_base_path()

    if agent_mode and agent_mode in AGENT_DIRECTORIES:
        # For export mode, allow writing to any file except protected agent files
        if agent_mode == AgentType.EXPORT:
            # Check if trying to write to a protected file
            if filename in PROTECTED_AGENT_FILES:
                raise ValueError(
                    f"Export agent cannot write to protected file '{filename}'. "
                    f"Protected files are: {', '.join(sorted(PROTECTED_AGENT_FILES))}"
                )

            # Allow writing anywhere else in .shotgun directory
            full_path = (base_path / filename).resolve()
        else:
            # For other agents, check if they have access to the requested file
            allowed_paths_raw = AGENT_DIRECTORIES[agent_mode]

            # Convert single Path/string to list of Paths for uniform handling
            if isinstance(allowed_paths_raw, str):
                # Special case: "*" means export agent
                allowed_paths = (
                    [Path(allowed_paths_raw)] if allowed_paths_raw != "*" else []
                )
            elif isinstance(allowed_paths_raw, Path):
                allowed_paths = [allowed_paths_raw]
            else:
                # Already a list
                allowed_paths = allowed_paths_raw

            # Check if filename matches any allowed path
            is_allowed = False
            for allowed_path in allowed_paths:
                allowed_str = str(allowed_path)

                # Check if it's a directory (no .md extension or suffix)
                # Directories: Path("contracts") has no suffix, files: Path("spec.md") has .md suffix
                if not allowed_path.suffix or (
                    allowed_path.suffix and not allowed_str.endswith(".md")
                ):
                    # Directory - allow any file within this directory
                    # Check both "contracts/file.py" and "contracts" prefix
                    if (
                        filename.startswith(allowed_str + "/")
                        or filename == allowed_str
                    ):
                        is_allowed = True
                        break
                else:
                    # Exact file match
                    if filename == allowed_str:
                        is_allowed = True
                        break

            if not is_allowed:
                allowed_display = ", ".join(f"'{p}'" for p in allowed_paths)
                raise ValueError(
                    f"{agent_mode.value.capitalize()} agent can only write to {allowed_display}. "
                    f"Attempted to write to '{filename}'"
                )

            full_path = (base_path / filename).resolve()
    else:
        # No agent mode specified, fall back to old validation
        full_path = (base_path / filename).resolve()

    # Ensure the resolved path is within the .shotgun directory
    try:
        full_path.relative_to(base_path.resolve())
    except ValueError as e:
        raise ValueError(
            f"Access denied: Path '{filename}' is outside .shotgun directory"
        ) from e

    return full_path


def _validate_shotgun_path(filename: str) -> Path:
    """Validate and resolve a file path within the .shotgun directory.

    Args:
        filename: Relative filename within .shotgun directory

    Returns:
        Absolute path to the file within .shotgun directory

    Raises:
        ValueError: If the path attempts to access files outside .shotgun directory
    """
    base_path = get_shotgun_base_path()

    # Create the full path
    full_path = (base_path / filename).resolve()

    # Ensure the resolved path is within the .shotgun directory
    try:
        full_path.relative_to(base_path.resolve())
    except ValueError as e:
        raise ValueError(
            f"Access denied: Path '{filename}' is outside .shotgun directory"
        ) from e

    return full_path


async def read_file(ctx: RunContext[AgentDeps], filename: str) -> str:
    """Read a file from the .shotgun directory.

    Args:
        filename: Relative path to file within .shotgun directory

    Returns:
        File contents as string

    Raises:
        ValueError: If path is outside .shotgun directory
        FileNotFoundError: If file does not exist
    """
    logger.debug("🔧 Reading file: %s", filename)

    try:
        file_path = _validate_shotgun_path(filename)

        if not file_path.exists():
            raise FileNotFoundError(f"File not found: {filename}")

        content = file_path.read_text(encoding="utf-8")
        logger.debug("📄 Read %d characters from %s", len(content), filename)
        return content

    except Exception as e:
        error_msg = f"Error reading file '{filename}': {str(e)}"
        logger.error("❌ File read failed: %s", error_msg)
        return error_msg


async def write_file(
    ctx: RunContext[AgentDeps],
    filename: str,
    content: str,
    mode: Literal["w", "a"] = "w",
) -> str:
    """Write content to a file in the .shotgun directory.

    Args:
        filename: Relative path to file within .shotgun directory
        content: Content to write to the file
        mode: Write mode - 'w' for overwrite, 'a' for append

    Returns:
        Success message or error message

    Raises:
        ValueError: If path is outside .shotgun directory or invalid mode
    """
    logger.debug("🔧 Writing file: %s (mode: %s)", filename, mode)

    if mode not in ["w", "a"]:
        raise ValueError(f"Invalid mode '{mode}'. Use 'w' for write or 'a' for append")

    try:
        # Use agent-scoped validation for write operations
        file_path = _validate_agent_scoped_path(filename, ctx.deps.agent_mode)

        # Determine operation type
        if mode == "a":
            operation = FileOperationType.UPDATED
        else:
            operation = (
                FileOperationType.CREATED
                if not file_path.exists()
                else FileOperationType.UPDATED
            )

        # Ensure parent directory exists
        file_path.parent.mkdir(parents=True, exist_ok=True)

        # Write content
        if mode == "a":
            with open(file_path, "a", encoding="utf-8") as f:
                f.write(content)
            logger.debug("📄 Appended %d characters to %s", len(content), filename)
            result = f"Successfully appended {len(content)} characters to {filename}"
        else:
            file_path.write_text(content, encoding="utf-8")
            logger.debug("📄 Wrote %d characters to %s", len(content), filename)
            result = f"Successfully wrote {len(content)} characters to {filename}"

        # Track the file operation
        ctx.deps.file_tracker.add_operation(file_path, operation)

        return result

    except Exception as e:
        error_msg = f"Error writing file '{filename}': {str(e)}"
        logger.error("❌ File write failed: %s", error_msg)
        return error_msg


async def append_file(ctx: RunContext[AgentDeps], filename: str, content: str) -> str:
    """Append content to a file in the .shotgun directory.

    Args:
        filename: Relative path to file within .shotgun directory
        content: Content to append to the file

    Returns:
        Success message or error message
    """
    return await write_file(ctx, filename, content, mode="a")
