import pytest
from collections import namedtuple
from eolib.data.number_encoding_utils import encode_number, decode_number


TCase = namedtuple("TCase", ["decoded", "encoded"])


TEST_CASES = [
    TCase(0, [0x01, 0xFE, 0xFE, 0xFE]),
    TCase(1, [0x02, 0xFE, 0xFE, 0xFE]),
    TCase(28, [0x1D, 0xFE, 0xFE, 0xFE]),
    TCase(100, [0x65, 0xFE, 0xFE, 0xFE]),
    TCase(128, [0x81, 0xFE, 0xFE, 0xFE]),
    TCase(252, [0xFD, 0xFE, 0xFE, 0xFE]),
    TCase(253, [0x01, 0x02, 0xFE, 0xFE]),
    TCase(254, [0x02, 0x02, 0xFE, 0xFE]),
    TCase(255, [0x03, 0x02, 0xFE, 0xFE]),
    TCase(32003, [0x7E, 0x7F, 0xFE, 0xFE]),
    TCase(32004, [0x7F, 0x7F, 0xFE, 0xFE]),
    TCase(32005, [0x80, 0x7F, 0xFE, 0xFE]),
    TCase(64008, [0xFD, 0xFD, 0xFE, 0xFE]),
    TCase(64009, [0x01, 0x01, 0x02, 0xFE]),
    TCase(64010, [0x02, 0x01, 0x02, 0xFE]),
    TCase(10_000_000, [0xB0, 0x3A, 0x9D, 0xFE]),
    TCase(16_194_276, [0xFD, 0xFD, 0xFD, 0xFE]),
    TCase(16_194_277, [0x01, 0x01, 0x01, 0x02]),
    TCase(16_194_278, [0x02, 0x01, 0x01, 0x02]),
    TCase(2_048_576_039, [0x7E, 0x7F, 0x7F, 0x7F]),
    TCase(2_048_576_040, [0x7F, 0x7F, 0x7F, 0x7F]),
    TCase(2_048_576_041, [0x80, 0x7F, 0x7F, 0x7F]),
    TCase(4_097_152_079, [0xFC, 0xFD, 0xFD, 0xFD]),
    TCase(4_097_152_080, [0xFD, 0xFD, 0xFD, 0xFD]),
]


def idfn(test_case: TCase) -> str:
    return str(test_case.decoded)


@pytest.mark.parametrize(
    "test_case",
    TEST_CASES,
    ids=idfn,
)
def test_encode_number(test_case: TCase) -> None:
    assert encode_number(test_case.decoded) == bytes(test_case.encoded)


@pytest.mark.parametrize(
    "test_case",
    TEST_CASES,
    ids=idfn,
)
def test_decode_number(test_case: TCase) -> None:
    assert decode_number(test_case.encoded) == test_case.decoded
