import pandas as pd

def rnaseh1_dict(label) :
    if label == 'R4a':
        
        # Define weights from the R4a experiment (positions 1 to 14)
        weights = {
            'A': [-0.01032282, 0.003136005, -0.00305681, 0.007715657, 0.0, 0.0, -0.01667942, -0.07365609,
                -0.01803916, -0.054889, -0.043132147, -0.007929123, -0.00125416, 0.002206189],
            'C': [0.012265437, -0.005505833, -0.015755671, 0.020965877, 0.0, 1.8e-15, 0.06663978, 0.06678766,
                -0.10665088, 0.02336113, 0.024941758, -0.033295411, 0.005095898, -0.012100435],
            'G': [-0.002194694, 4.49701e-05, 0.007817385, -0.044064409, 1.8e-15, 0.0, -0.01224785, 0.04965035,
                0.05090317, 0.0453844, 0.058558984, -0.053753251, -0.025738673, 0.015062984],
            'T': [0.003921987, 0.00169841, 0.008358475, 0.05226517, 0.0, 0.0, 0.02785011, -0.03534451,
                0.09438379, 0.02260767, 0.006754902, 0.050369343, 0.013656149, -0.002621679]
        }
        return weights
    
    if label == "R4a_dinuc":
        # R4a dinucleotide weight matrix (positions 1 to 13)
        dinuc_weights = {
            'AA': [-0.011145051, -0.006067481, -0.007890538, 0, 0.0, 0, -0.07337747, -0.06851626,
                -0.010306159, -0.094417048, -0.05058309, -0.0168, -0.010695708],
            'AC': [-0.004447214, -0.008866372, 0.021128335, 0, 0.0, 0, 0.07088278, -0.1525785,
                -0.044078913, -0.016465339, -0.075775455, 0.0121, -0.00902432],
            'AG': [-0.011652815, 0.012412212, -0.042339535, 0.007717149, 0.0, 0, 0.02535729, -0.04410562,
                0.036961377, 0.007966314, -0.109548632, -0.0291, 0.012377923],
            'AT': [-0.014661642, 0.013936723, 0.048128461, 0, 0.0, 0, -0.05891855, -0.01396531,
                -0.014798928, -0.070465864, 0.013196108, -0.00000713, 0.009006838],
            'CA': [0.020816251, -0.008566963, -0.007651668, 0, 0.0, -0.01667931, -0.01899446, 0.02661876,
                -0.197637331, -0.034399076, 0.011969527, -0.0292, 0.013777801],
            'CC': [0.011068607, -0.012480093, 0.000156886, 0, 0.0, 0.06663685, 0.23241208, -0.02829809,
                0.008031078, 0.088154422, 0.010987723, -0.0314, -0.003837145],
            'CG': [0.002897998, -0.009800158, -0.05712865, 0.020966283, 0.0, -0.01225109, 0.11504828, 0.048322,
                -0.023436983, 0.101680984, -0.045403135, -0.072, 0.013297102],
            'CT': [0.015692507, 0.019640334, 0.041232207, 0, 0.0, 0.02785783, 0.05687383, 0.1748351,
                -0.108577974, 0.018424432, 0.068804131, -0.0086, -0.009253344],
            'GA': [-0.015326425, -0.000473612, 0.026303393, 0, 0.0, 0, -0.12058821, 0.04786808,
                -0.039598722, -0.013355767, 0.036976505, -0.0714, -0.022117792],
            'GC': [-0.008938635, -0.02234771, 0.02612862, 0, 1.8e-15, 0, 0.03592721, -0.08329603,
                0.095519987, 0.063964234, 0.041290001, -0.0532, -0.031887997],
            'GG': [0.00506995, 0.010606765, -0.039558916, -0.044062832, 0.0, 0, 0.03770482, 0.14672315,
                0.049224694, 0.046869673, 0.061370459, -0.0323, -0.002909972],
            'GT': [0.011026739, 0.00235329, 0.055641554, 0, 0.0, 0, -0.03902255, 0.135601,
                0.060115731, 0.079379173, 0.103596682, -0.0432, -0.03805112],
            'TA': [0.015878317, 0.002939182, 0.016805845, 0, 0.0, 0, -0.03057776, -0.08951474,
                0.043641921, -0.002714079, 0.008931228, 0.0627, 0.034444806],
            'TC': [-0.022119476, -0.020515622, 0.028231483, 0, 0.0, 0, 0.10451626, -0.12514098,
                0.068869986, 0.009395615, -0.051668893, 0.0443, -0.009542745],
            'TG': [0.009728738, 0.015238746, -0.040777529, 0.052261172, 0.0, 0, 0.09233641, -0.04319589,
                0.143892551, 0.063370269, -0.02844665, 0.0325, 0.032088368],
            'TT': [0.00472374, -0.003331569, 0.06683977, 0, 0.0, 0, -0.02119269, 0.09665211,
                0.150923932, 0.04770594, 0.057774192, 0.0493, 0.006159874]
        }
        return dinuc_weights
    
    if label == 'R4b':
        weights = {
        'A': [-0.002354787, 0.003294872, 0.006780463, 0.0, 0.0, -0.01114242, 0.01506308, 0.07217545,
            0.07568231, -0.043834023, -0.12996776, -0.03597764, -0.07797353, -0.067145423,
            -0.001961368, 0.000131249],
        'C': [0.004994686, -0.006945926, -0.018221328, 0.018770769, -1.47e-15, 0.00623508, 0.05518087,
            0.08449527, 0.06051392, 0.101195572, 0.07935141, -0.10888559, 0.01620083, 0.023277155,
            -0.031108229, 0.011287499],
        'G': [-0.009785106, -0.000962357, 0.007479555, -0.009842844, 0.0, -0.0285429, -0.06796898,
            -0.09867123, -0.07455191, -0.023284051, 0.06047636, 0.06222209, 0.04648892, 0.053950661,
            -0.053136271, -0.01013757],
        'T': [0.014523727, 0.005043385, -0.006490964, 0.0, 0.0, 0.05924024, 0.06116798, 0.05436943,
            0.01077339, 0.001273785, -0.07412014, 0.11244672, 0.01139138, -0.008432221,
            0.065214638, 0.004101607]
        }
        return weights
    
    if label == "R7":
        weights = {
        'A': [-0.003055021, 0.02156569, 0.076145086, 0.059064563, -0.01358513,
            -0.034385764, -0.05425748, -0.06438475, -0.029782176, -0.004193173,
            -0.004533779, 0.00502, 0.002129393],
        'C': [-0.002721742, 0.02677823, 0.004240495, 0.004228382, 0.03099521,
            0.012450965, -0.03341468, 0.02480901, -0.002817492, -0.010539598,
            0.009975088, -0.00807, -0.003694151],
        'G': [-0.024870337, -0.07909881, -0.102823731, -0.075071303, -0.0535504,
            -0.002308598, 0.0153482, 0.01957649, 0.002622478, -0.044473455,
            0.009235409, -0.000191, -0.007289507],
        'T': [0.038159302, 0.04298555, 0.007136523, -0.015604777, 0.02061486,
            0.027951969, 0.08944129, 0.03150905, 0.037332804, 0.038902627,
            -0.01508523, -0.00000544, 0.009366819]
        }


        return weights

    if label == "R4b_dinuc":
        dinuc_weights = {
        'AA': [-0.00543, 0.004283488, 0, 0, 0, -0.009522305, 0.072130987, 0.126768829, 0.011363718,
            -0.18538821, -0.16348208, -0.03856405, -0.149361809, -0.077219692, -0.015567376],
        'AC': [-0.00397, -0.01312702, 0.03484197, 0, 0, 0.062748291, 0.126789177, 0.146143769,
            0.162159634, 0.04588116, -0.20509574, -0.0817389, -0.045611282, -0.104644007, 0.031495467],
        'AG': [-0.000378, 0.014252352, -0.007480984, 0, 0, -0.074673607, -0.071779421, -0.006093003,
            0.065834431, 0.01515899, -0.0817777, 0.03034085, 0.002040677, -0.110411569, -0.017962542],
        'AT': [0.00143, -0.00425201, 0, 0, 0, 0.045505634, 0.039005913, 0.102050449, 0.090197303,
            -0.10922156, -0.03617314, -0.02909581, -0.116448795, 0.006404936, 0.006821267],
        'CA': [0.0109, -0.000577155, 0, 0, -0.011144808, 0.019866681, 0.112734937, 0.166467536,
            0.023867952, -0.01242086, 0.04442879, -0.22519636, -0.065117058, 0.015147464, -0.028647697],
        'CC': [0.0015, -0.02927809, 0.008212147, 0.018770994, 0.006235185, 0.036813409, 0.103896886,
            0.082043208, 0.139220915, 0.21498211, -0.00635486, -0.01584372, 0.066876833,
            0.011610443, -0.008439025],
        'CG': [-0.00282, -0.002202767, -0.032077217, 0, -0.028542785, -0.048494666, -0.036089654,
            0.007860852, 0.042339635, 0.15221119, 0.08093335, -0.04598857, 0.08726532,
            -0.063182316, -0.063004892],
        'CT': [0.0132, -0.002168181, 0, 0, 0.059238522, 0.061439031, 0.126049786, 0.121479104,
            0.081918838, 0.04447931, 0.2069162, -0.14804715, -0.004805387, 0.087041447, -0.00792387],
        'GA': [-0.0064, 0.011872802, 0, 0, 0, -0.024059662, -0.00503588, -0.066217923, -0.154121101,
            -0.19828033, 0.03502903, -0.0157952, -0.024273441, 0.058617309, -0.044690048],
        'GC': [-0.0203, -0.02081156, 0.01434027, -0.009842453, 0, 0.042361668, 0.037128607,
            0.000191676, 0.052317918, 0.0388746, -0.08623692, 0.07971978, 0.060579975,
            0.018986939, -0.059867284],
        'GG': [-0.00859, 0.004669302, 0.00366422, 0, 0, -0.106397757, -0.177483795, -0.167332953,
            -0.106955224, 0.05023473, 0.17994785, 0.05797816, 0.051906268, 0.045972052, -0.037411787],
        'GT': [-0.00425, -0.008486544, 0, 0, 0, 0.038453389, -0.004292496, -0.097860748,
            -0.074362022, -0.10099033, 0.14524248, 0.09016484, 0.06890931, 0.111802725, -0.06880036],
        'TA': [0.0232, 0.012047228, 0, 0, 0, 0.082731471, 0.130425552, 0.143849946, -0.016044997,
            -0.098102, -0.13255233, 0.05628669, -0.022079335, 0.006050205, 0.071322068],
        'TC': [-0.00000435, -0.007348133, 0.015023827, 0, 0, 0.093607182, 0.141686594, 0.115182071,
            0.121599896, 0.10164178, -0.16083653, 0.06752028, -0.006992164, -0.062854824, 0.069002871],
        'TG': [0.0141, 0.012923515, -0.016716362, 0, 0, -0.002642735, -0.033763896, -0.024930341,
            -0.009859235, 0.0499366, -0.06925341, 0.15341265, 0.059492085, -0.054688374, 0.063014776],
        'TT': [0.0177, -0.01243655, 0, 0, 0, 0.122115793, 0.120970014, 0.061715236, -0.005222877,
            -0.08899389, 0.07600838, 0.18025393, 0.026954761, 0.059708054, 0.059339477]
        }
        return dinuc_weights

    if label == "R7_dinuc":
        dinuc_weights = {
        'AA': [0.009081411, 0.087778434, 0.154159324, 0.09141531, 0.0029742, -0.050968079, -0.080101303, -0.103107912, -0.031777122, -0.017510513, -0.00331, 0.007606967],
        'AC': [0.035017676, 0.033503366, 0.073575187, 0.07572152, -0.006389322, -0.060141008, -0.05010873, -0.058971992, -0.049130859, 0.011343901, -0.00819, 0.000357329],
        'AG': [-0.084497637, -0.069390977, -0.002043154, 0.01119738, -0.023798874, -0.013266516, -0.022619033, -0.054198475, -0.092043541, 0.008640449, -0.00526, -0.000887778],
        'AT': [0.045971299, 0.030973446, 0.050978338, 0.03572171, -0.035385509, -0.001174494, -0.050685047, -0.029060379, 0.024231604, -0.013966992, -0.00147, 0.012182348],
        'CA': [0.008389612, 0.092189709, 0.057132147, -0.03544149, -0.036685202, -0.06395038, -0.125153888, -0.001004514, -0.009623127, -0.017279695, 0.0154, -0.006833143],
        'CC': [0.012593763, -0.002411931, -0.040929122, 0.04824559, 0.076564814, 0.014287539, 0.041989015, 0.035681438, 0.002130942, 0.00874438, 0.0111, -0.000858851],
        'CG': [-0.064891784, -0.04474141, -0.010070678, -0.03993763, 0.030321195, 0.025332536, -0.016425931, 0.019051606, -0.05506201, -0.005866804, 0.000061, -0.025199587],
        'CT': [0.044395592, 0.027324876, -0.003589745, 0.04495715, 0.070275125, 0.09831353, -0.008604034, 0.052594728, 0.027359893, -0.025314094, 0.0101, 0.004106841],
        'GA': [0.016382139, 0.030309937, -0.045976887, -0.09743956, -0.09952783, -0.048026708, -0.079382144, -0.015967358, -0.004537817, -0.04537268, 0.0104, 0.000170652],
        'GC': [0.009503221, -0.064815252, -0.070268435, -0.04083228, -0.050954636, -0.055066878, 0.048826935, 0.011534141, -0.00650564, -0.049393756, -0.00781, -0.009703637],
        'GG': [-0.121551835, -0.216446123, -0.20757596, -0.14751025, -0.028349851, 0.031811408, -0.001523036, 0.037665588, 0.005285288, -0.021653404, 0.0242, -0.002963063],
        'GT': [0.01184916, -0.082920407, -0.121300976, -0.05610021, -0.033219176, 0.101497182, 0.060215168, 0.055946057, 0.022295787, -0.060448133, 0.00675, 0.015480354],
        'TA': [0.060507082, 0.098414385, 0.048197099, -0.06980566, -0.04116122, -0.049054633, 0.038531065, 0.012981482, 0.037555698, 0.044343643, -0.000561, 0.005326662],
        'TC': [0.047997697, 0.060325751, 0.029055058, 0.04234167, 0.032044778, -0.033002797, 0.067977716, 0.0050741, 0.019220529, 0.045840932, -0.0291, -0.006319487],
        'TG': [-0.028551982, -0.061329424, -0.067021223, -0.07699481, 0.002964694, 0.025995407, 0.122281992, 0.027539061, -0.008035706, 0.041132195, -0.0227, -0.003740494],
        'TT': [0.075586262, 0.068103273, -0.005930056, 0.03129569, 0.090986229, 0.170092685, 0.143294716, 0.081620065, 0.078419051, 0.018446183, -0.0164, 0.001870845]
        }
        return dinuc_weights
    
    if label == "R4a_krel":
        krel_weights = {
        'A': [1.032029719, 0.99026958, 1.009484692, 0.976059805, 1, 1, 1.051753022, 1.228540635,
              1.055972033, 1.170309976, 1.133830729, 1.024602539, 1.003891416, 0.993154621],
        'C': [0.961942716, 1.017083538, 1.048886807, 0.934946929, 1, 1, 0.793229622, 0.792770778,
              1.330917101, 0.927514921, 0.922610538, 1.103309236, 0.984188412, 1.037545315],
        'G': [1.006809712, 0.999860466, 0.975744163, 1.136723358, 1, 1, 1.038002715, 0.845944545,
              0.842057286, 0.859180964, 0.818302772, 1.166785966, 1.079862136, 0.953262467],
        'T': [0.987830831, 0.994730161, 0.974065265, 0.837831263, 1, 1, 0.913586483, 1.109667194,
              0.707145312, 0.929852762, 0.979040843, 0.843713648, 0.957627605, 1.008134564]
        }
        return krel_weights
    
    if label == "R4a_krel_dinuc":
        dinuc_krel_r4a_weights = {
        'AA': [1.034580943, 1.018826222, 1.024482817, 1, 1, 1, 1.22767613, 1.212592734, 1.031978023, 1.292957881, 1.156949568, 1.052192596, 1.033186718],
        'AC': [1.013798847, 1.027510643, 0.934442853, 1, 1, 1, 0.780064411, 1.473421644, 1.136768362, 1.05108877, 1.235116615, 0.962446851, 1.028000724],
        'AG': [1.036156436, 0.9614873, 1.131371407, 0.976055176, 1, 1, 0.921321222, 1.136851228, 0.885315979, 0.975282065, 1.339908267, 1.090384521, 0.961593693],
        'AT': [1.045492246, 0.956757039, 0.850666672, 1, 1, 1, 1.182812892, 1.043331662, 1.045918218, 1.218641979, 0.959055023, 1.000022125, 0.97205352],
        'CA': [0.93541119, 1.026581632, 1.02374165, 1, 1, 1.05175268, 1.058936144, 0.917407124, 1.613230502, 1.106733695, 0.962860867, 1.090527777, 0.957250142],
        'CC': [0.965656248, 1.038723322, 0.999513212, 1, 1, 0.793238713, 0.278870162, 1.087803513, 0.975081115, 0.726473839, 0.965907216, 1.09737468, 1.011905921],
        'CG': [0.991008071, 1.030408, 1.177259178, 0.93494567, 1, 1.038012768, 0.643027387, 0.850066158, 1.072720436, 0.684503527, 1.140877167, 1.223276971, 0.958741658],
        'CT': [0.951309179, 0.939059834, 0.872064418, 1, 1, 0.913562529, 0.82353148, 0.457520421, 1.336896502, 0.942832543, 0.786514058, 1.026693092, 1.028711341],
        'GA': [1.047554939, 1.001469525, 0.918385648, 1, 1, 1, 1.374161947, 0.851474584, 1.122867193, 1.041440368, 0.88526904, 1.221600245, 1.068627241],
        'GC': [1.02773486, 1.069340632, 0.918927935, 1, 1, 1, 0.8885248, 1.258451508, 0.703619912, 0.801531324, 0.871885094, 1.164950559, 1.098942302],
        'GG': [0.984268923, 0.967089256, 1.122743682, 1.136718465, 1, 1, 0.883009219, 0.544746377, 0.847265273, 0.854572449, 0.809579308, 1.100296655, 1.009029082],
        'GT': [0.965786157, 0.992698196, 0.827354996, 1, 1, 1, 1.121079443, 0.579256263, 0.813472487, 0.753701743, 0.678559486, 1.134080515, 1.118065283],
        'TA': [0.950732646, 0.990880285, 0.947854706, 1, 1, 1, 1.094876889, 1.277746965, 0.86458754, 1.008421263, 0.972288123, 0.805586985, 0.893124413],
        'TC': [1.068632466, 1.063656018, 0.912403156, 1, 1, 1, 0.675706213, 1.388288314, 0.786309723, 0.97084722, 1.160318605, 0.862661042, 1.029609296],
        'TG': [0.969813603, 0.95271711, 1.126524805, 0.837843668, 1, 1, 0.713497937, 1.134028512, 0.55352918, 0.803374283, 1.088264466, 0.89911859, 0.900435986],
        'TT': [0.985343146, 1.010337215, 0.792609092, 1, 1, 1, 1.065756828, 0.700107153, 0.531712161, 0.851977674, 0.82073783, 0.846883206, 0.9808871]
        }
        return dinuc_krel_r4a_weights
    
    if label == "R4b_krel":
        krel_weights = {
        'A': [1.005876944, 0.991776846, 0.983077704, 1.0, 1.0, 1.027808622, 0.962406416, 0.819868592, 0.811116369, 1.109398477, 1.324366188, 1.08979096, 1.194601928, 1.167577751, 1.004895071, 0.999672436],
        'C': [0.987534545, 1.01733525, 1.04547576, 0.953152976, 1.0, 0.98443884, 0.862282704, 0.789121482, 0.848972779, 0.747441812, 0.801959237, 1.2717505, 0.959566884, 0.941906192, 1.077638159, 0.971829298],
        'G': [1.024421114, 1.002401796, 0.981332949, 1.024565214, 1.0, 1.071235756, 1.169633138, 1.246258078, 1.186062443, 1.058111018, 0.849066519, 0.844709625, 0.883975581, 0.86535299, 1.132614501, 1.025300774],
        'T': [0.963752503, 0.987413006, 1.016199781, 1.0, 1.0, 0.852151558, 0.847340413, 0.86430785, 0.973112382, 0.996820959, 1.184984855, 0.719361833, 0.971570038, 1.02104466, 0.837240993, 0.989763441]
        }
        return krel_weights
    
    if label == "R4b_krel_dinuc":
        krel_dinucleotide_weights_rb = {
        'AA': [1.013561294, 0.989309514, 1.0, 1.0, 1.0, 1.023765231, 0.81997956, 0.683617522, 0.971639075, 1.462681415, 1.408009334, 1.096245976, 1.37276876, 1.192720542, 1.038852177],
        'AC': [1.009896451, 1.032761675, 0.913043381, 1.0, 1.0, 0.843396362, 0.683566739, 0.635262644, 0.595291152, 0.885492395, 1.511866354, 1.203999326, 1.113834059, 1.261164597, 0.921395394],
        'AG': [1.000942255, 0.964429786, 1.018670617, 1.0, 1.0, 1.186366167, 1.179143021, 1.015206573, 0.835694149, 0.96216705, 1.204096161, 0.924277022, 0.994906994, 1.275558952, 1.044829897],
        'AT': [0.996428594, 1.010611927, 1.0, 1.0, 1.0, 0.886429611, 0.902651247, 0.745308259, 0.77489067, 1.272588995, 1.090278878, 1.072615678, 1.29062632, 0.984014923, 0.982975867],
        'CA': [0.972840646, 1.00144043, 1.0, 1.0, 1.027814582, 0.950417861, 0.718642517, 0.584539735, 0.940431715, 1.03099928, 0.889117137, 1.562032345, 1.162515472, 0.962195816, 1.071497303],
        'CC': [0.996252294, 1.073070601, 0.979504588, 0.953152414, 0.984438578, 0.908123175, 0.740700025, 0.795241199, 0.65254031, 0.463459803, 1.0158601, 1.039541861, 0.833092581, 0.971023313, 1.021061642],
        'CG': [1.00703743, 1.005497542, 1.080056505, 1.0, 1.071235469, 1.121030246, 1.090070518, 0.98038133, 0.894331132, 0.620119916, 0.79801112, 1.114775673, 0.782208147, 1.157686853, 1.157244048],
        'CT': [0.967122673, 1.005411224, 1.0, 1.0, 0.852155846, 0.84666394, 0.68541207, 0.696819321, 0.795551595, 0.888991053, 0.483590245, 1.369487708, 1.011993013, 0.782766877, 1.019775947],
        'GA': [1.015965773, 0.970368532, 1.0, 1.0, 1.0, 1.060046744, 1.012568265, 1.165262949, 1.384646731, 1.494856839, 0.912576527, 1.039420767, 1.060580282, 0.853706234, 1.111534895],
        'GC': [1.050589219, 1.051940314, 0.964210365, 1.024564238, 1.0, 0.894276144, 0.907336522, 0.999521627, 0.869427898, 0.902978971, 1.21522523, 0.80103988, 0.848807923, 0.952613471, 1.149413383],
        'GG': [1.021427408, 0.988346621, 0.990855047, 1.0, 1.0, 1.265541507, 1.442954022, 1.417620124, 1.266932802, 0.874627001, 0.550896328, 0.855301386, 0.870455271, 0.885265551, 1.093370223],
        'GT': [1.010613234, 1.021180237, 1.0, 1.0, 1.0, 0.904030205, 1.010712969, 1.244235322, 1.185588531, 1.252045957, 0.637512029, 0.774971689, 0.82802004, 0.720969079, 1.171708049],
        'TA': [0.942018214, 0.96993321, 1.0, 1.0, 1.0, 0.793523471, 0.67449128, 0.640987437, 1.040044196, 1.244837426, 1.330816612, 0.859522861, 1.055104356, 0.98490024, 0.821998415],
        'TC': [1.000010862, 1.018339054, 0.962504382, 1.0, 1.0, 0.766380487, 0.646386608, 0.712535098, 0.696517854, 0.746328191, 1.40140672, 0.831486702, 1.017450648, 1.156869517, 0.827786535],
        'TG': [0.964754986, 0.967746222, 1.041719751, 1.0, 1.0, 1.006595589, 1.084266023, 1.062219737, 1.024606121, 0.875371057, 1.172838746, 0.617121381, 0.851523017, 1.136488152, 0.842731284],
        'TT': [0.955806073, 1.031038437, 1.0, 1.0, 1.0, 0.695230307, 0.698089877, 0.845974603, 1.013034961, 1.22210592, 0.810302582, 0.550132431, 0.932727831, 0.850984014, 0.851903888]
        }
        return krel_dinucleotide_weights_rb
    
    if label == "R7_krel":
        logfc_nucleotide_weights_r7 = {
        'A': [1.008030101, 0.943314734, 0.799853173, 0.844749208, 1.035708419, 1.090382741, 1.142615407, 1.169234866, 1.078282242, 1.011021726, 1.011917006, 0.986792185, 0.994402905],
        'C': [1.00715408, 0.929613609, 0.988853888, 0.988885727, 0.918529307, 0.967272725, 1.087830253, 0.93478969, 1.007405758, 1.02770326, 0.973780551, 1.021207444, 1.00971005],
        'G': [1.065371507, 1.20791067, 1.270271459, 1.197324396, 1.140756853, 1.006068134, 0.959657363, 0.948543333, 0.993106835, 1.116898166, 0.975724792, 1.000502299, 1.019160418],
        'T': [0.899698541, 0.887012782, 0.981241699, 1.041017047, 0.945813984, 0.926528445, 0.764904194, 0.917178682, 0.901870985, 0.897744716, 1.039651422, 1.000014297, 0.975379382]
        }
        return logfc_nucleotide_weights_r7
    
    if label == "R7_krel_dinuc":
        logfc_dinucleotide_weights_r7 = {
        'AA': [0.976129575, 0.769274999, 0.594793293, 0.759715496, 0.992182336, 1.13396924, 1.210545716, 1.271018427, 1.083525943, 1.046026261, 1.008705054, 0.980005141],
        'AC': [0.907956283, 0.911936636, 0.806608134, 0.800966513, 1.016794288, 1.15808022, 1.131710447, 1.155007469, 1.12914012, 0.970182635, 1.021538395, 0.999060764],
        'AG': [1.222101449, 1.182393699, 1.005370416, 0.970567765, 1.062555174, 1.034870945, 1.05945397, 1.142460313, 1.241935805, 0.97728864, 1.013818227, 1.002333518],
        'AT': [0.879164761, 0.918586513, 0.866003794, 0.906105735, 1.093010563, 1.003087149, 1.133225292, 1.076385003, 0.936307398, 1.036712141, 1.003869617, 0.967978783],
        'CA': [0.977947963, 0.757679993, 0.84982855, 1.093157709, 1.096426797, 1.168093127, 1.328966121, 1.002640358, 1.025294322, 1.045419558, 0.959597276, 1.017960869],
        'CC': [0.966897382, 1.006339744, 1.107581912, 0.873186803, 0.798749921, 0.962445303, 0.889632167, 0.906211589, 0.994398834, 0.977015458, 0.970928762, 1.002257484],
        'CG': [1.170567601, 1.117602484, 1.026470707, 1.104975782, 0.920300951, 0.933413606, 1.043175445, 0.949922987, 1.144730109, 1.015420853, 0.999839676, 1.066236939],
        'CT': [0.883306496, 0.928176754, 1.00943562, 0.881830445, 0.815282324, 0.741583573, 1.022615643, 0.861755125, 0.928084712, 1.066537919, 0.973371552, 0.989205198],
        'GA': [0.956939662, 0.920330542, 1.120849927, 1.256119203, 1.261608206, 1.126237867, 1.208655411, 1.041970089, 1.01192762, 1.119261773, 0.972556246, 0.999551443],
        'GC': [0.97502085, 1.170366437, 1.184700091, 1.107327363, 1.133933905, 1.144742904, 0.871658742, 0.969682591, 1.017100029, 1.129831143, 1.020539972, 1.025505941],
        'GG': [1.31949815, 1.568927124, 1.545611962, 1.387729663, 1.074517385, 0.916383936, 1.004003289, 0.900996265, 0.986107657, 1.056915821, 0.936439096, 1.007788389],
        'GT': [0.968854566, 1.217955711, 1.318838769, 1.14745901, 1.087316372, 0.733215366, 0.841724851, 0.85294618, 0.941395679, 1.158887496, 0.982256655, 0.959309998],
        'TA': [0.840957557, 0.741318476, 0.873314262, 1.18348369, 1.108191979, 1.12893976, 0.898721365, 0.965878265, 0.90128511, 0.883443044, 1.001473352, 0.985998907],
        'TC': [0.873838388, 0.841434185, 0.923628983, 0.888705216, 0.915770525, 1.086747622, 0.821321048, 0.986662764, 0.949478974, 0.87950743, 1.07651856, 1.016610726],
        'TG': [1.075048685, 1.161203963, 1.176164817, 1.202380321, 0.992207323, 0.931671254, 0.678582638, 0.92761377, 1.021121797, 0.891884313, 1.05973048, 1.009831862],
        'TT': [0.801322038, 0.820991022, 1.015587111, 0.917739497, 0.760843332, 0.552912565, 0.623350839, 0.785462229, 0.793876072, 0.951514337, 1.043144452, 0.995082498]
        }
        return logfc_dinucleotide_weights_r7


#################################################################################################
def score_window(subseq: str, weights_df) -> float:
    if not subseq:
        return 0.0

    score = 0.0
    for i, base in enumerate(subseq):
        if base not in weights_df.index:
            raise ValueError(f"Invalid base '{base}' at position {i}")
        col = f'Pos_{i+1}'
        if col in weights_df.columns:
            score += weights_df.loc[base, col]
    return score / len(subseq)

#################################################################################################
def score_window_dinuc(subseq: str, weights_df) -> float:
    """
    Computes a dinucleotide-based score for a given subsequence using position-specific weights.
    Protects against empty subsequence and short lengths.
    """
    if not subseq or len(subseq) < 2:
        return 0.0

    score = 0.0
    for i in range(len(subseq) - 1):
        dimer = subseq[i:i+2]
        col = f'Pos_{i+1}'
        if dimer in weights_df.index and col in weights_df.columns:
            score += weights_df.loc[dimer, col]

    return score / len(subseq)
      
        
##############################################################################################
def old_compute_rnaseh1_score(aso_sequence: str, weights: dict) -> float:
    """
    General RNase H1 cleavage score based on single nucleotide weights.
    Adapts to variable-length weight tables.

    Parameters:
    -----------
    aso_sequence : str
        ASO sequence (A/C/G/T only)
    weights : dict
        Dictionary mapping bases (A/C/G/T) to a list of position-specific weights

    Returns:
    --------
    float
        Cleavage score based on supplied weights
    """
    import pandas as pd

    weights_df = pd.DataFrame(weights).T
    weights_df.columns = [f'Pos_{i+1}' for i in range(weights_df.shape[1])]
    max_window = weights_df.shape[1]  # e.g. 14

    seq = aso_sequence.upper().replace("U", "T")
    L = len(seq)

    if L == 0:
        return 0.0

    if L < max_window:
        return score_window(seq, weights_df)

    elif L % 2 == 0:
        start = (L - max_window) // 2
        subseq = seq[start:start + max_window]
        if len(subseq) == 0:
            return 0.0
        return score_window(subseq, weights_df)

    else:
        offset1 = (L - max_window) // 2
        offset2 = offset1 + 1
        s1 = score_window(seq[offset1:offset1 + max_window], weights_df)
        s2 = score_window(seq[offset2:offset2 + max_window], weights_df)
        return (s1 + s2) / 2


#################################################################################################

def old_compute_rnaseh1_dinucleotide_score(aso_sequence: str, dinuc_weights) -> float:
    """
    Computes RNase H1 cleavage score based on dinucleotide weights from R4a experiment.

    Uses a  central window (or shorter if sequence is shorter).
    For odd-length sequences, averages two overlapping central windows.
    """

    import pandas as pd

    weights_df = pd.DataFrame(dinuc_weights).T
    weights_df.columns = [f'Pos_{i+1}' for i in range(weights_df.shape[1])]

    max_window = weights_df.shape[1]  # usually 13
    seq = aso_sequence.upper().replace("U", "T")
    L = len(seq)

    if L == 0:
        return 0.0

    if L < max_window:
        return score_window_dinuc(seq, weights_df)

    elif L % 2 == 0:
        start = (L - max_window) // 2
        subseq = seq[start:start + max_window]
        if len(subseq) < 2:
            return 0.0
        return score_window_dinuc(subseq, weights_df)

    else:
        offset1 = (L - max_window) // 2
        offset2 = offset1 + 1
        s1 = score_window_dinuc(seq[offset1:offset1 + max_window], weights_df)
        s2 = score_window_dinuc(seq[offset2:offset2 + max_window], weights_df)
        return (s1 + s2) / 2


#######################################################################################################
def check_motif_presence(aso_sequence: str, motif: str) -> float:
    """
    Checks whether a specific motif is present in the ASO sequence.

    Parameters:
    -----------
    aso_sequence : str
        The DNA ASO sequence (A/C/G/T)

    motif : str
        A motif (sub-sequence) to search for (e.g., 'TCCC')

    Returns:
    --------
    float
        1.0 if the motif is found in the sequence, otherwise 0.0
        ---------
    good motif examples: 'TCCC', 'GGGA', 'CGCG', 'AGGA', 'TGCC' , 'CCCG'  
    bad motif examples: 'TTTT', 'AAAA', 'CCCC', 'GGGG', 'TTAA', 'GCGC'
    """

    seq = aso_sequence.upper()
    motif = motif.upper()
    return 1.0 if motif in seq else 0.0

#########################################################################################################
def compute_rnaseh1_score(aso_sequence: str, weights: dict, window_start: int = None) -> float:
    """
    Computes the RNase H1 cleavage score for a given ASO sequence using single-nucleotide weights.

    Behavior:
    ---------
    - If the sequence is shorter than the weight window, score the entire sequence.
    - If `window_start` is provided:
        • If it's within bounds but the window extends past the end, slice until the end.
        • If it's invalid (negative or beyond sequence length), return 0.
    - If `window_start` is not provided:
        • Use the default behavior: centered window (averaging if sequence length is odd).

    Parameters:
    -----------
    aso_sequence : str
        The antisense oligonucleotide (ASO) sequence (A/C/G/T/U only).
    weights : dict
        Dictionary mapping nucleotides (A/C/G/T) to lists of position-specific weights.
    window_start : int, optional
        Start index for the scoring window. If None, a centered window will be used (default behavior).

    Returns:
    --------
    float
        The cleavage score computed using the provided nucleotide weights.
    """
    import pandas as pd

    weights_df = pd.DataFrame(weights).T
    weights_df.columns = [f'Pos_{i+1}' for i in range(weights_df.shape[1])]
    max_window = weights_df.shape[1]  # e.g., 14 positions

    seq = aso_sequence.upper().replace("U", "T")
    L = len(seq)

    if L == 0:
        return 0.0

    # If sequence is shorter than the window, score the full sequence
    if L < max_window:
        return score_window(seq, weights_df)

    # If a specific window_start is given
    if window_start is not None:
        if window_start < 0 or window_start >= L:
            return 0.0
        subseq = seq[window_start:window_start + max_window]  # may be shorter near the end
        return score_window(subseq, weights_df)

    # Default behavior — center-aligned window
    if L % 2 == 0:
        start = (L - max_window) // 2
        subseq = seq[start:start + max_window]
        if len(subseq) == 0:
            return 0.0
        return score_window(subseq, weights_df)
    else:
        offset1 = (L - max_window) // 2
        offset2 = offset1 + 1
        s1 = score_window(seq[offset1:offset1 + max_window], weights_df)
        s2 = score_window(seq[offset2:offset2 + max_window], weights_df)
        return (s1 + s2) / 2
#########################################################################################################3
def compute_rnaseh1_dinucleotide_score(aso_sequence: str, dinuc_weights, window_start: int = None) -> float:
    """
    Computes the RNase H1 cleavage score for a given ASO sequence using dinucleotide weights.

    Behavior:
    ---------
    - If the sequence is shorter than the weight window, use the full sequence.
    - If `window_start` is provided:
        • If it's within bounds but extends beyond the end, slice until the end (no padding or truncation to window size).
        • If it's invalid (negative or beyond sequence length), return 0.
    - If `window_start` is not provided:
        • Use the default behavior: central window (averaging if sequence length is odd).

    Parameters:
    -----------
    aso_sequence : str
        The antisense oligonucleotide (ASO) sequence (A/C/G/T/U only).
    dinuc_weights : dict
        Dictionary mapping dinucleotides (e.g., "AA", "AC", ...) to lists of position-specific weights.
    window_start : int, optional
        Start index for the scoring window. If None, a centered window will be used (default behavior).

    Returns:
    --------
    float
        The cleavage score computed using the provided dinucleotide weights.
    """
    import pandas as pd

    weights_df = pd.DataFrame(dinuc_weights).T
    weights_df.columns = [f'Pos_{i+1}' for i in range(weights_df.shape[1])]
    max_window = weights_df.shape[1]  # e.g., 13 positions

    seq = aso_sequence.upper().replace("U", "T")
    L = len(seq)

    if L == 0:
        return 0.0

    # If sequence is shorter than the window, score the entire sequence
    if L < max_window:
        return score_window_dinuc(seq, weights_df)

    # If a specific window_start is given
    if window_start is not None:
        if window_start < 0 or window_start >= L:
            return 0.0
        subseq = seq[window_start:window_start + max_window]  # may be shorter than max_window near the end
        return score_window_dinuc(subseq, weights_df)

    # Default behavior — central window logic
    if L % 2 == 0:
        start = (L - max_window) // 2
        subseq = seq[start:start + max_window]
        if len(subseq) < 2:
            return 0.0
        return score_window_dinuc(subseq, weights_df)
    else:
        offset1 = (L - max_window) // 2
        offset2 = offset1 + 1
        s1 = score_window_dinuc(seq[offset1:offset1 + max_window], weights_df)
        s2 = score_window_dinuc(seq[offset2:offset2 + max_window], weights_df)
        return (s1 + s2) / 2


# ==================== Global wrapper for RNaseH features ====================

# Default best window_start maps (length -> position)
BEST_WIN_NT = {
    'R4a': {10:0,11:0,12:0,13:0,14:0,15:0,16:0,17:2,18:2,19:4,20:3,21:0,22:0,25:0},
    'R4b': {10:0,11:0,12:0,13:0,14:0,15:0,16:0,17:0,18:0,19:1,20:3,21:0,22:0,25:0},
    'R7' : {10:0,11:0,12:0,13:0,14:0,15:0,16:3,17:2,18:4,19:1,20:4,21:0,22:0,25:0},
}

BEST_WIN_DINUC = {
    'R4a_dinuc': {10:0,11:0,12:0,13:0,14:0,15:0,16:3,17:3,18:2,19:4,20:6,21:0,22:0,25:0},
    'R4b_dinuc': {10:0,11:0,12:0,13:0,14:0,15:0,16:0,17:0,18:3,19:1,20:3,21:0,22:0,25:0},
    'R7_dinuc' : {10:0,11:0,12:0,13:0,14:0,15:0,16:0,17:3,18:4,19:6,20:3,21:0,22:0,25:0},
}

BEST_WIN_KREL_NT = {
    'R4a_krel': {10:0,11:0,12:0,13:0,14:0,15:0,16:0,17:3,18:2,19:4,20:3,21:0,22:0,25:0},
    'R4b_krel': {10:0,11:0,12:0,13:0,14:0,15:0,16:0,17:1,18:3,19:1,20:3,21:0,22:0,25:0},
    'R7_krel' : {10:0,11:0,12:0,13:0,14:0,15:0,16:3,17:2,18:4,19:6,20:4,21:0,22:0,25:0},
}

BEST_WIN_KREL_DINUC = {
    'R4a_krel_dinuc': {10:0,11:0,12:0,13:0,14:0,15:0,16:2,17:1,18:2,19:4,20:6,21:0,22:0,25:0},
    'R4b_krel_dinuc': {10:0,11:0,12:0,13:0,14:0,15:0,16:0,17:0,18:3,19:1,20:3,21:0,22:0,25:0},
    'R7_krel_dinuc' : {10:0,11:0,12:0,13:0,14:0,15:0,16:0,17:3,18:4,19:6,20:3,21:0,22:0,25:0},
}


def score_row_general(seq, exp, wset, weights, pos_maps):
    """
    Compute a single RNaseH score for one sequence using:
      - experiment (exp): "R4a" | "R4b" | "R7"
      - weight set (wset): "nt" | "dinuc" | "krel_nt" | "krel_dinuc"
      - the appropriate window_start picked by length from pos_maps.

    Returns a float score. If the sequence is missing/invalid, returns 0.0.
    """
    if not isinstance(seq, str) or len(seq) == 0:
        return 0.0
    L = len(seq)

    if wset == "nt":
        pos = pos_maps["nt"].get(exp, {}).get(L, 0)
        return compute_rnaseh1_score(seq, weights, window_start=pos)

    elif wset == "dinuc":
        key = f"{exp}_dinuc"
        if L < 2:
            return 0.0
        pos = pos_maps["dinuc"].get(key, {}).get(L, 0)
        return compute_rnaseh1_dinucleotide_score(seq, weights, window_start=pos)

    elif wset == "krel_nt":
        key = f"{exp}_krel"
        pos = pos_maps["krel_nt"].get(key, {}).get(L, 0)
        return compute_rnaseh1_score(seq, weights, window_start=pos)

    elif wset == "krel_dinuc":
        key = f"{exp}_krel_dinuc"
        if L < 2:
            return 0.0
        pos = pos_maps["krel_dinuc"].get(key, {}).get(L, 0)
        return compute_rnaseh1_dinucleotide_score(seq, weights, window_start=pos)

    else:
        raise ValueError(f"Unknown weight set: {wset}")

#####################################################################################
#use this function to compute all requested RNaseH features for a DataFrame
def compute_rnaseh_features(df,seq_col="Sequence",experiments=("R4a", "R4b", "R7"),weight_sets=("nt", "dinuc", "krel_nt", "krel_dinuc"),pos_map_nt=None,pos_map_dinuc=None,pos_map_krel_nt=None,pos_map_krel_dinuc=None,):
    """
    Compute RNaseH features for the given DataFrame.

    Parameters
    ----------
    df : pandas.DataFrame
        Input DataFrame that must contain a column of sequences.
    seq_col : str
        Name of the sequence column (default: "Sequence").
    experiments : tuple[str]
        Which experiments to include (any of "R4a", "R4b", "R7").
    weight_sets : tuple[str]
        Which weight sets to compute. Options are:
          - "nt"
          - "dinuc"
          - "krel_nt"
          - "krel_dinuc"
        You may pass one or multiple.
    pos_map_* : dict | None
        Optional custom maps of (length -> best window_start).
        If None, the defaults defined above are used.

    Returns
    -------
    pandas.DataFrame
        The same DataFrame with new RNaseH feature columns appended.

    Examples
    --------
    >>> import pandas as pd
    >>> df = pd.DataFrame({"Sequence": ["ATGCGTACGTAG", "TTGCCGGTGGTGCAGATGAA"]})
    >>> # 1) All default weight sets for all experiments
    >>> df = compute_rnaseh_features(df)
    >>> # 2) Only nt for R4a and R7
    >>> df = compute_rnaseh_features(df, experiments=("R4a","R7"), weight_sets=("nt",))
    >>> # 3) nt + dinuc for all experiments
    >>> df = compute_rnaseh_features(df, weight_sets=("nt","dinuc"))
    >>> # 4) Custom nt position map (others default)
    >>> custom_nt = {"R4a": {16:2, 17:2}, "R7": {16:3}}
    >>> df = compute_rnaseh_features(df, experiments=("R4a","R7"), weight_sets=("nt",), pos_map_nt=custom_nt)
    """
    # Resolve window maps (use defaults if none provided)
    pos_maps = {
        "nt": pos_map_nt or BEST_WIN_NT,
        "dinuc": pos_map_dinuc or BEST_WIN_DINUC,
        "krel_nt": pos_map_krel_nt or BEST_WIN_KREL_NT,
        "krel_dinuc": pos_map_krel_dinuc or BEST_WIN_KREL_DINUC,
    }

    # For each requested set and experiment, compute a column via a single generic scorer
    for wset in weight_sets:
        for exp in experiments:
            if wset == "nt":
                col_name = f"RNaseH1_score_{exp}"
                weights = rnaseh1_dict(exp)
            elif wset == "dinuc":
                col_name = f"RNaseH1_score_dinucleotide_{exp}_dinuc"
                weights = rnaseh1_dict(f"{exp}_dinuc")
            elif wset == "krel_nt":
                col_name = f"RNaseH1_Krel_score_{exp}_krel"
                weights = rnaseh1_dict(f"{exp}_krel")
            elif wset == "krel_dinuc":
                col_name = f"RNaseH1_Krel_dinucleotide_score_{exp}_krel_dinuc"
                weights = rnaseh1_dict(f"{exp}_krel_dinuc")
            else:
                raise ValueError(f"Unknown weight set: {wset}")

            df[col_name] = df.apply(
                lambda row: score_row_general(
                    row.get(seq_col, None),  # sequence
                    exp,                     # experiment
                    wset,                    # weight set
                    weights,                 # weights vector/dict
                    pos_maps,                # position maps
                ),
                axis=1,
            )

    return df
