#!/usr/bin/python3

version = __version__ = '5.0.0rc1'

_change_log = ''

__version__ = version.split()[0]  # For PEP 396 and PEP 345

# The shortened version of version
try:
    ver = version.split(' ')[0]
except:
    ver = ''

# __version__ = version

port = 'PySimpleGUI'

# all of the tkinter involved imports
import tkinter as tk
from tkinter import filedialog
from tkinter.colorchooser import askcolor
from tkinter import ttk
# import tkinter.scrolledtext as tkst
import tkinter.font
from uuid import uuid4

# end of tkinter specific imports
# get the tkinter detailed version
tclversion_detailed = tkinter.Tcl().eval('info patchlevel')
framework_version = tclversion_detailed

import time
import pickle
import calendar
import datetime
import textwrap

import socket
from hashlib import sha256 as hh
import inspect
import traceback
import difflib
import copy
import pprint
try:  # Because Raspberry Pi is still on 3.4....it's not critical if this module isn't imported on the Pi
    from typing import List, Any, Union, Tuple, Dict, SupportsAbs, Optional  # because this code has to run on 2.7 can't use real type hints.  Must do typing only in comments
except:
    print('*** Skipping import of Typing module. "pip3 install typing" to remove this warning ***')
import random
import warnings
from math import floor
from math import fabs
from functools import wraps

try:  # Because Raspberry Pi is still on 3.4....
    # from subprocess import run, PIPE, Popen
    import subprocess
except Exception as e:
    print('** Import error {} **'.format(e))

import threading
import itertools
import json
import configparser
import queue

try:
    import webbrowser

    webbrowser_available = True
except:
    webbrowser_available = False
# used for github upgrades
import urllib.request
import urllib.error
import urllib.parse
import pydoc
from urllib import request
import os
import sys
import re
import tempfile
import ctypes
import platform

pil_import_attempted = pil_imported = False

warnings.simplefilter('always', UserWarning)

g_time_start = 0
g_time_end = 0
g_time_delta = 0


# These timer routines are to help you quickly time portions of code.  Place the timer_start call at the point
# you want to start timing and the timer_stop at the end point. The delta between the start and stop calls
# is returned from calling timer_stop

def timer_start():
    """
    Time your code easily.... starts the timer.
    Uses the time.time value, a technique known to not be terribly accurage, but tis' gclose enough for our purposes
    """
    global g_time_start

    g_time_start = time.time()


def timer_stop():
    """
    Time your code easily.... stop the timer and print the number of MILLISECONDS since the timer start

    :return: delta in MILLISECONDS from timer_start was called
    :rtype:  int
    """
    global g_time_delta, g_time_end

    g_time_end = time.time()
    g_time_delta = g_time_end - g_time_start
    return int(g_time_delta * 1000)

def timer_stop_usec():
    """
    Time your code easily.... stop the timer and print the number of MICROSECONDS since the timer start

    :return: delta in MICROSECONDS from timer_start was called
    :rtype:  int
    """
    global g_time_delta, g_time_end

    g_time_end = time.time()
    g_time_delta = g_time_end - g_time_start
    return int(g_time_delta * 1000000)


def _timeit(func):
    """
    Put @_timeit as a decorator to a function to get the time spent in that function printed out

    :param func: Decorated function
    :type func:
    :return:     Execution time for the decorated function
    :rtype:
    """

    @wraps(func)
    def wrapper(*args, **kwargs):
        start = time.time()
        result = func(*args, **kwargs)
        end = time.time()
        print('{} executed in {:.4f} seconds'.format(func.__name__, end - start))
        return result

    return wrapper


_timeit_counter = 0
MAX_TIMEIT_COUNT = 1000
_timeit_total = 0


def _timeit_summary(func):
    """
    Same as the timeit decorator except that the value is shown as an averave
    Put @_timeit_summary as a decorator to a function to get the time spent in that function printed out

    :param func: Decorated function
    :type func:
    :return:     Execution time for the decorated function
    :rtype:
    """

    @wraps(func)
    def wrapper(*args, **kwargs):
        global _timeit_counter, _timeit_total

        start = time.time()
        result = func(*args, **kwargs)
        end = time.time()
        _timeit_counter += 1
        _timeit_total += end - start
        if _timeit_counter > MAX_TIMEIT_COUNT:
            print('{} executed in {:.4f} seconds'.format(func.__name__, _timeit_total / MAX_TIMEIT_COUNT))
            _timeit_counter = 0
            _timeit_total = 0
        return result

    return wrapper


def formatted_datetime_now():
    """
    Returns a string with current date and time formatted YYYY-MM-DD HH:MM:SS for easy logging

    :return:    String with date and time formatted YYYY-MM-DD  HH:MM:SS
    :rtype:     (str)
    """
    now = datetime.datetime.now()
    current_time = now.strftime('%Y-%m-%d %H:%M:%S')
    return current_time



def running_linux():
    """
    Determines the OS is Linux by using sys.platform

    Returns True if Linux

    :return: True if sys.platform indicates running Linux
    :rtype:  (bool)
    """
    return sys.platform.startswith('linux')


def running_mac():
    """
    Determines the OS is Mac by using sys.platform

    Returns True if Mac

    :return: True if sys.platform indicates running Mac
    :rtype:  (bool)
    """
    return sys.platform.startswith('darwin')


def running_windows():
    """
    Determines the OS is Windows by using sys.platform

    Returns True if Windows

    :return: True if sys.platform indicates running Windows
    :rtype:  (bool)
    """
    return sys.platform.startswith('win')


def running_trinket():
    """
    A special case for Trinket.  Checks both the OS and the number of environment variables
    Currently, Trinket only has ONE environment variable.  This fact is used to figure out if Trinket is being used.

    Returns True if "Trinket" (in theory)

    :return: True if sys.platform indicates Linux and the number of environment variables is 1
    :rtype:  (bool)
    """
    if sys.platform.startswith('linux') and socket.gethostname().startswith('pygame-'):
        return True
    return False


def running_replit():
    """
    A special case for REPLIT.  Checks both the OS and for the existance of the number of environment variable REPL_OWNER
    Currently, Trinket only has ONE environment variable.  This fact is used to figure out if Trinket is being used.

    Returns True if running on "replit"

    :return: True if sys.platform indicates Linux and setting REPL_OWNER is found in the environment variables
    :rtype:  (bool)
    """
    if 'REPL_OWNER' in os.environ and sys.platform.startswith('linux'):
        return True
    return False




# Handy python statements to increment and decrement with wrapping that I don't want to forget
# count = (count + (MAX - 1)) % MAX           # Decrement - roll over to MAX from 0
# count = (count + 1) % MAX                   # Increment to MAX then roll over to 0

"""
    Welcome to the "core" PySimpleGUI code....

    It's a mess.... really... it's a mess internally... it's the external-facing interfaces that
    are not a mess.  The Elements and the methods for them are well-designed.
    PEP8 - this code is far far from PEP8 compliant.
    It was written PRIOR to learning that PEP8 existed.

    I'll be honest.... started learning Python in Nov 2017, started writing PySimpleGUI in Feb 2018.
    Released PySimpleGUI in July 2018.  I knew so little about Python that my parameters were all named
    using CamelCase.  DOH!  Someone on Reddit set me straight on that.  So overnight I renamed all of the
    parameters to lower case.  Unfortunately, the internal naming conventions have been set.  Mixing them
    with PEP8 at this moment would be even MORE confusing.

    Code I write now, outside PySimpleGUI, IS PEP8 compliant.

    The variable and function naming in particular are not compliant.  There is
    liberal use of CamelVariableAndFunctionNames, but for anything externally facing, there are aliases
    available for all functions.  If you've got a serious enough problem with 100% PEP8 compliance
    that you'll pass on this package, then that's your right and I invite you to do so.  However, if
    perhaps you're a practical thinker where it's the results that matter, then you'll have no
    trouble with this code base.  There is consisency however.

    I truly hope you get a lot of enjoyment out of using PySimpleGUI.  It came from good intentions.
"""

# ----====----====----==== Constants the user CAN safely change ====----====----====----#

# Base64 encoded GIF file
DEFAULT_BASE64_ICON = b'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'

DEFAULT_BASE64_ICON_16_BY_16 = b'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAAAXNSR0IArs4c6QAAAARnQU1BAACxjwv8YQUAAAAJcEhZcwAADsMAAA7DAcdvqGQAAAKCSURBVDhPVZNbSFRRFIb35YwXItBIGtDsiqENEUTRjJlZkJggPSUYBD0UhULElE6hBY6ID/ZSpD1IDxaCEPhUaFLRQyWRNxIJe8syMxCjMCbB07fOsaMt+GftvWf//7/2Whyt1sTei/fCpDqQBTrGOi9Myrk7URwhnQUfQLeOvErJuUQgADlK6gObvAOl5sHx0doHljwARFRiCpxG5J1sjPxALiYNgn9kiQ3gafdYUYzseCd+FICX7sShw7LR++q6cl3XHaXQHFdOJLxFsJtvKHnbUr1nqp01hhStpXAzo7TZZXOjJ+9orT9pY74aY3ZobZZYW8D/GpjM19Ob088fmJxW2tkC4AJt17Oeg2MLrHX6jXWes16w1sbBkrFWBTB2nTLpv5VJg7wGNhRDwCS0tR1cbECkidwMQohAdoScqiz8/FCZUKlPCgSWlQ71elOI1fcco9hCXp1kS7dX3u+qVOm2L4nW8qE4Neetvl8v83NOb++9703BcUI/cU3imuWV7JedKtv5LdFaMRzHLW+N+zJoVDZzRLj6SFNfPlMYwy5bDiRcCojmz15tKx+6hKPv7LvjrG/Q2RoOwjSyzNDlahyzA2dAJeNtFcMHA2cfLn24STNr6P4I728jJ7hvf/lEGuaXLnkRAp0PyFK+hlyLSJGyGWnKyeBi2oJU0IPIjNd15uuL2f2PJgueQBKhVRETCgNeYU+xaeEpnWaw8cQPRM7g/McT8eF0De9u7P+49TqXF7no98BDEEkdvvXem8LAtfJniFRB/A5XeiAiG2+/icgHVQUW5d5KyAhl3M2y+U+ysv1FDukyKGQW3Y+vHJWvU7mz8RJSPZgDd3H2RqiUUn8BSQuaBvGjGpsAAAAASUVORK5CYII='

DEFAULT_BASE64_LOADING_GIF = b'R0lGODlhQABAAKUAAAQCBJyenERCRNTS1CQiJGRmZLS2tPTy9DQyNHR2dAwODKyqrFRSVNze3GxubMzKzPz6/Dw6PAwKDKSmpExKTNza3CwqLLy+vHx+fBQWFLSytAQGBKSipERGRNTW1CQmJGxqbLy6vPT29DQ2NHx6fBQSFKyurFRWVOTi5HRydPz+/Dw+PP7+/gAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH/C05FVFNDQVBFMi4wAwEAAAAh+QQJCQAsACwAAAAAQABAAAAG/kCWcEgsGo/IpHLJbDqf0CjxwEmkJgepdrvIAL6A0mJLdi7AaMC4zD4eSmlwKduuCwNxdMDOfEw4D0oOeWAOfEkmBGgEJkgphF8ph0cYhCRHeJB7SCgJAgIJKFpnkGtTCoQKdEYGEmgSBlEqipAEEEakcROcqGkSok8PkGCBRhNwcrtICYQJUJnDm0YHASkpAatHK4Qrz8Nf0mTbed3B3wDFZY95kk8QtIS2bQ29r8BPE8PKbRquYBuxpJCwdKhBghUrQpFZAA8AgX2T7DwIACiixYsYM2rc+OSAhwrZOEa5QGHDlw0dLoiEAqEAoQK3VjJxCQmEzCUhzgXciOKE/gIFJ+4NEXBOAEcPyL6UqEBExLkvIjYyiMOAyICnAAZs9IdGgVWsWjWaTON1yAGsUTVOTUOhyLhh5TQi7cqUyIVzKjmiYCBBQtAjNAnZvKmk5cuYhJVc6DAWZd7ETTx6CAm5suXLRQY4sPDTQoqwmIlAADE2DYi0oUUQhbQC8WUQ5wZf9oDVA58KdaPAflqgTgMEXxA0iPIB64c6I9AgiFL624Y2FeLkbtJ82HM2tNPYfmLBOHLlUQJ/6z0POADhUa4+3V7HA/vw58gfEaFBA+qMIt6Su9/UPAL+F4mwWxwwJZGLGitp9kFfHzgAGhIHmhKaESIkB8AIrk1YBAQmDJiQoYYghijiiFAEAQAh+QQJCQApACwAAAAAQABAAIUEAgSEgoREQkTU0tRkYmQ0MjSkpqTs6ux0cnQUEhSMjozc3ty0trT09vRUUlRsamw8OjwMCgxMSkx8fnwcGhyUlpTk5uS8vrz8/vwEBgSMioxERkTc2txkZmQ0NjS0srT08vR0dnQUFhSUkpTk4uS8urz8+vxsbmw8Pjz+/v4AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAG/sCUcEgsGo/IpHLJbDqf0Kh0Sl0aPACAx1DtOh/ZMODhLSMNYjHXzBZi01lPm42BizHz5CAk2YQGSSYZdll4eUUYCHAhJkhvcAWHRiGECGeEa0gNAR4QEw1TA4RZgEcdcB1KBwViBQdSiqOWZ6wABZlIE3ATUhujAAJsj2FyUQK/wWbDcVInvydsumm8UaKjpWWrra+whNBtDRMeHp9UJs5pJ4aSXgMnGxsI2Oz09fb3+Pn6+/xEJh8KRjBo1M/JiARiEowoyIQAIQIMk1T4tXAfBw6aEI5KAArfgjcFFhj58CsLg3zDIhXRUBKABnwc4GAkoqDly3vWxMxLQbLk/kl8tbKoJAJCIyGO+RbUCnlkxC8F/DjsLOLQDsSISRREEBMBKlYlDRgoUMCg49ezaNOqVQJCqtm1Qy5IGAQgw4YLcFOYOGWnA8G0fAmRSVui5c+zx0omM2NBgwYLUhq0zPKWSIMFHCojsUAhiwjIUHKWnPpBAF27H5YEEBOg2mQA80A4ICQBRBJpWVpDAfHabAMUv1BoFkJChGcSUoCXREGEUslZRxoHAB3lQku8Qg7Q/ZWB26HAdgYLmTi5Aru9hPwSqdryKrsLG07fNTJ7soN7IAZwsH2EfUn3ETk1WUVYWbDdKBlQh1Usv0D3VQPLpOHBcAyBIAFt/K31AQrbBqGQWhtBAAAh+QQJCQAyACwAAAAAQABAAIUEAgSEgoTEwsREQkTk4uQsLiykoqRkYmQUEhTU0tRUUlT08vS0srSMjox8enwMCgzMysw8OjwcGhxcWlz8+vy8urxMSkzs6uysqqxsamzc2tyUlpQEBgSMiozExsTk5uQ0NjSkpqRkZmQUFhRUVlT09vS0trSUkpR8fnwMDgzMzsw8PjwcHhxcXlz8/vy8vrxMTkzc3tz+/v4AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAG/kCZcEgsGo/IpHLJbDqf0Kh0Sq1ar8nEgMOxqLBgZCIFKAMeibB6aDGbB2u1i+Muc1xxJSWmoSwpdHUcfnlGJSgIZSkoJUptdXCFRRQrdQArhEcqD24PX0wUmVMOlmUOSiqPXkwLLQ8PLQtTFCOlAAiiVyRuJFMatmVpYIB1jVEJwADCWCWBdsZQtLa4artmvaO2p2oXrhyxVCWVdSvQahR4ViUOZAApDuaSVhQaGvHy+Pn6+/z9/v8AAzrxICJCBBEeBII6YOnAPYVDWthqAfGIgGQC/H3o0OEDEonAKPL7IKHMCI9GQCQD0S+AmwBHVAJjyQ/FyyMgJ/YjUAvA/ggCFjFqDNAxSc46IitOOlqmRS6lQwSIABHhwAuoWLNq3cq1ogcHLVqgyFiFAoMGJ0w8teJBphsQCaWcaFcGwYkwITiV4hAiCsNSB7B4cLYXwpMNye5WcVEgWZkC6ZaUSAQMwUMnFRybqdCEgWYTVUhpBrBtSQfNHZC48BDCgIfIRKxpxrakAWojLjaUNCNhA2wZsh3TVuLZMWgiJRTYgiFKtObSShbQLZUinohkIohkHs25yYnERVRo/iSDQmPHBdYi+Wsp6ZDrjrNH1Uz2SYPpKRocOZ+sQJEQhLnBgQFTlHBWAyZcxoJmEhjRliVw4cMfMP4ZQYEADpDQggMvJ/yWB3zYYQWBZnFBxV4p8mFVAgzLqacQBSf0ZNIJLla0mgGu1ThFEAAh+QQJCQAqACwAAAAAQABAAIUEAgSUkpRERkTMyswkIiTs6uy0trRkZmQ0MjTU1tQcGhykpqRUVlT09vTEwsQsKix8enwMCgycnpzU0tS8vrw8Ojzc3txcXlz8/vwEBgSUlpRMSkzMzswkJiT08vS8urxsamw0NjTc2twcHhysqqz8+vzExsQsLix8fnxkYmT+/v4AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAG/kCVcEgsGo/IpHLJbDqf0Kh0Sq1ar8tEAstdWk4AwMnSLRfBYbF5nUint+tu2w2Ax5OFghMdPt2TBg9hDwZMImgnIn9HH3QAhUxaTw0LCw1WHY4dax6CAA8eVAWOYXplEm4SoqQApl2oaapUmXSbZgW0HaFUBo6QZpQLu1UGub+LWHnIy8zNzs/Q0dLTzSYQFxcoDtRMAwiOCCZJDRwDl88kGawZC0YlEOoAGRDnywPx6wNEHnxpJ8N/SvRjdaLEkAOsDiyjwMrRByEe8NHJADAOhIZ0IAgZgFHcIgYY3TAQYqIjMpAhw4xUEXFdxTUXUwLQKAQhKYXIGsl8CHGg/piXa0p4wvgAA5EG8MLMq4esZEiPRRoMMMGU2QKJbthxQ2LiG51wW5NgcACBwQUIFIyGXcu2bdgGGjZ06LBBQ1UoJg5UqHAAKhcTBByN8OukRApHKe5OcYA1TQbCTC6wuoClQeCGIxQjcYBxm5UAKQM8kdyQshUBKQU8CYERwZURKUc88crKNZIJZRlAmIAEdkjZTkhPPtLAppsDd1GHVO2Ec0PPREoodyTAIBHQIUWPHm5EA0btQxoowKgAaJISwtNcsF7ENyvgRCg0Vgq5iYMDISqkoIDEQkoyRZjgXhojQHcHRyHpYwRcAhBAgAB2LeNfSACyNaBgbqngXUPgGLElHSvVZahCA4fRcYFma3GQGwQciAhNEAAh+QQJCQAwACwAAAAAQABAAIUEAgSEgoTEwsRERkTk4uQkIiSkpqRsamwUEhTU0tT08vSUkpRUUlQ0MjS0trQMCgzMyszs6ux8enwcGhzc2tz8+vyMioxMTkysrqw8OjwEBgSEhoTExsRMSkzk5uQkJiSsqqxsbmwUFhTU1tT09vSUlpRUVlQ0NjS8vrwMDgzMzszs7ux8fnwcHhzc3tz8/vz+/v4AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAG/kCYcEgsGo/IpHLJbDqf0Kh0Sq1ar9hs1sNiebRgowsBACBczJcKA1K9wkxWucxSVgKTOUC0qcCTcnN1SBEnenoZX39iZAApaEcVhod6J35SFSgoJE4EXYpHFpSUAVIqBWUFKlkVIqOHIpdOJHlzE5xXEK+UHFAClChYBruHBlAowMLEesZPtHoiuFa6y2W9UBAtZS2rWK3VsVIkmtJYosuDi1Ekk68n5epPhe4R8VR3rnN8svZTLxAg2vDrR7CgwYMItZAo0eHDhw4l4CVMwgHVoRbXjrygMOLNQQEaXmnISARErQnNCFbQtqsFPBCUUtpbUG0BkRe19EzwaG9A/rUBREa8GkHQIrEWRCgMJcjyKJFvsHjG87kMaMmYBWkus1nEwEmZ9p7tmqBA44gRA/uhCDlq5MQlHJrOaSHgLZOFAwoUGBDRrt+/gAMLhkMiwYiyV0iogCARCwUTbDWYoHBPQmQJjak4eEDpgQMpKxpQarAiCwXOox4QhXLg1YEsDIgxgKKALSUNiKvUXpb5CLVXJKeoqNatCQdiwY2QyH0kAfEnu9syJ0Jiw4dUGxorqNb7SOtRr4+saDeH9BETsqOEHl36yIVXF46MQN15NRQSlstowIzk+K7kMGzW2WdUKAABB90FQEwp8l1g2wX2xfOda0oolkB3YWyw4GBCIfgHHIdCvDdKByAKsd4h5pUIAwkBsNRCdioWoUB7MRoUBAAh+QQJCQAuACwAAAAAQABAAIUEAgSEhoTMzsxMSkykpqQcHhz08vRkYmQUEhSUlpS0trTc3twsLixsbmwMCgzU1tSsrqz8+vycnpyMjoxUUlQkJiRsamwcGhy8vrw0NjR0dnQEBgTU0tSsqqz09vRkZmQUFhScmpy8urzk5uQ0MjR0cnQMDgzc2ty0srT8/vykoqSUkpRUVlQsKiz+/v4AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAG/kCXcEgsGo8RRWlAaSgix6h0Sp2KKoCstiKqer/fkHasTYDP6KFoQ25303BqBNsmV6DxvBFSr0P0gEMNfW0WgYEDhGQDRwsTFhYTC4dTiYpajEQeB2xjBx6URxaXWoZDHiR9JKChRHykAH9DB4oHcQIlJQJRc6R3Qwukk2gcnRscUSKkb0ITpBNpo6VSCZ11ZkS0l7Zo0lmmUQp0YxUKRtq1aQLGyFNJDUxOeEXOl9DqDbqhJ6QnrYDo6nD7l8cDgz4MWBHMYyBglgMGFh46MeHDhwn+JGrcyLGjx48gO3rg8CBiSDQnWBhjkfFkFQUO2jgwF8UACgUmPz6IWcfB/oMjGBBkQYABJAVFFIwYMDEGQc6NBqz1USjk1RhZHAWQ2kUERRsUHrVe4jpk6RgTTzV6IEVVCAamAEwU/XiUUNIjNlGk5bizj0+XVGDKpAl4yoO6WSj8LOzFgwAObRlLnky5suXLEg2o0FCCwF40KU48SEGwg1AtCDrk6XAhywUCrTr0UZ1GNhnYhwycbuMUdGsyF0gHkqBIApoHfRYDKqGoAcrkhzQoKoEmAog2IIRHSSEiQAAR84wQJ2Qcje0xuKOcaDGmhfIiZuughUPg9+spI66TATEiyvnbeaTwwAPhidLHB1IQsBsACKS3kX7YTWGABLlI8BlBEShSIGUQIO6HmRDekIHgh/lh19+HLjzA3hbvfZiEdwpoh+KMjAUBACH5BAkJACYALAAAAABAAEAAhQQCBISGhMzKzERCRDQyNKSmpOzq7GRiZBQSFHRydJyanNTW1LS2tPz6/Dw6PAwODLSytPTy9GxubBweHHx6fKSipNze3AQGBIyKjMzOzExOTDQ2NKyqrOzu7GRmZBQWFHR2dJyenNza3Ly+vPz+/Dw+PP7+/gAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAb+QJNwSCwaj8ikcslsmjoYx+fjwHSc2KyS8QF4vwiGdjxmXL5or5jMXnYQ6TTi2q4bA/F4wM60UDZTGxQWRw55aRt8SSQUhyAkRQ+HaA+KRw0akwAaDUSSmgCVRg0hA1MDCp1ZIKAACUQbrYlFBrGIBlgirV4LQ3ige0QNtnEbqkwSuwASQ2+aD3RDCpoKTgTKBEQMmmtEhpMlTp+tokMMcGkP3UToh+VL46DvQh0BGwgIGwHRkc/W2HW+HQrXJNkuZm2mTarWZIGyXm2GHTKGhRWoV3ZqFcOFBZMmTooaKCiBr0SqMQ0sxgFxzJIiESAI4CMAQoTLmzhz6tzJs6f+z59Ah0SoACJBgQhByXDoAoZD0iwcDjlFIuDAAQFPOzCNM+dIhjMALmRIGkJTiCMe0BxIavAQwiIH1CZNoAljka9exJI1iySDVaxJneV5gPQpk6h5Chh2UqAdAASKFzvpEKJoCH6SM2vezLmz58+gQ7fhsOHCBQeR20SAwKDwzbZf3o4ZgQ7BiJsFDqXOEiFeV0sCEZGBEGcqHxKaIGkhngaCJRJg41xQnkWwF8IuiQknM+LTg9tMBAQIADhJ7sRtOrDGfIRE3C8HWhqB7UV2Twx6lhQofWHDbp8TxDGBaEIgl4d8nwWYxoAEmvALGsEQ6J5aCIYmHnkNZqghgUEBAAAh+QQJCQAnACwAAAAAQABAAIUEAgSEgoRERkTEwsTk4uRkYmQ0MjQUFhRUVlTU1tT08vSkpqQMCgxMTkzMysxsbmz8+vzs6uwcHhxcXlzc3tysrqwEBgSEhoRMSkzExsRkZmQ8OjwcGhxcWlzc2tz09vSsqqwMDgxUUlTMzsx0dnT8/vzs7uz+/v4AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAG/sCTcEgsGo/IpHLJbA5NjozJSa02RxiAFiAYWb/g08Ky3VoW4TRzxCiXLV613Jh1lwVzJ4RCgCQjdnZTeUkZImQAFiIZRxmBbgOERyUkjyQlRQOPZZFIFCAVHmGVmyRFgJtag0UUAncUVpqpAJ1Drpt4RhQHdgewVHWpGEUOiHZwR7d2uU0fbbMWfkRjx2hGHqkJTtizWqLEylwOSAup1kzc3d9GERlSShWpIE4fxpvRaumB2k7BuHPh7lSRlapWml29flEhZYkQARF31lGBwNANCWmEPIAAwS9MhgaILDQwKEnSHgoYS6pcqRJCSpZzMhTgBeBAAZIwrXzo8AjB/oecXxQYSGVgFdAmCLohODoEhAELFjacE+KoGy2mD+w8IJLU6lKgIB6d42C15tENjwwMKatFQc4SqTCdYAvALcwS9t7IpdntwNGhgdQK4en1aNhA5wjOwrkyq5utXJUyFbLgqQUDU4UIJWp3MhMFXe0gMOqZyYAJZAFwmMC4dBMIP13Lnk27tu3buHPnSYABKoaOYRwUKMBIZYJnWhgAtzIiZBxJ/rQw+6KhTIGSEPImkvulgPWSeI+9pNJcC7KS0bmoGTFhwnNJx8sod10BAYIKTRLcErD86IUyAeiGhAn2WECagCeMYMd7CJ5A4BsHIhgAgA0eUd99FWao4YYcAy4RBAA7OEloRWRqYW9jdzhOTjdUeHV4MTVCcmpRRWxDKzdGSWtiWnV5UUlCY0t5QTlKYmUzU25OM3ArSDd0K3JOMEtOTw=='

# Old debugger logo
# PSG_DEBUGGER_LOGO = b'R0lGODlhMgAtAPcAAAAAADD/2akK/4yz0pSxyZWyy5u3zZ24zpW30pG52J250J+60aC60KS90aDC3a3E163F2K3F2bPI2bvO3rzP3qvJ4LHN4rnR5P/zuf/zuv/0vP/0vsDS38XZ6cnb6f/xw//zwv/yxf/1w//zyP/1yf/2zP/3z//30wAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAEAAP8ALAAAAAAyAC0AAAj/AP8JHEiwoMGDCBMqXMiwoUOFAiJGXBigYoAPDxlK3CigwUGLIAOEyIiQI8cCBUOqJFnQpEkGA1XKZPlPgkuXBATK3JmRws2bB3TuXNmQw8+jQoeCbHj0qIGkSgNobNoUqlKIVJs++BfV4oiEWalaHVpyosCwJidw7Sr1YMQFBDn+y4qSbUW3AiDElXiWqoK1bPEKGLixr1jAXQ9GuGn4sN22Bl02roo4Kla+c8OOJbsQM9rNPJlORlr5asbPpTk/RP2YJGu7rjWnDm2RIQLZrSt3zgp6ZmqwmkHAng3ccWDEMe8Kpnw8JEHlkXnPdh6SxHPILaU/dp60LFUP07dfRq5aYntohAO0m+c+nvT6pVMPZ3jv8AJu8xktyNbw+ATJDtKFBx9NlA20gWU0DVQBYwZhsJMICRrkwEYJJGRCSBtEqGGCAQEAOw=='

PSG_DEBUGGER_LOGO = b'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'

UDEMY_ICON = b'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'

BLANK_BASE64 = b'iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAAAXNSR0IArs4c6QAAAARnQU1BAACxjwv8YQUAAAAJcEhZcwAADsMAAA7DAcdvqGQAAAANSURBVBhXY2BgYGAAAAAFAAGKM+MAAAAAAElFTkSuQmCC'
BLANK_BASE64 = b'iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAQAAAC1HAwCAAAAC0lEQVR42mNkYAAAAAYAAjCB0C8AAAAASUVORK5CYII='


DEFAULT_WINDOW_ICON = DEFAULT_BASE64_ICON

DEFAULT_ELEMENT_SIZE = (45, 1)  # In CHARACTERS
DEFAULT_BUTTON_ELEMENT_SIZE = (10, 1)  # In CHARACTERS
DEFAULT_MARGINS = (10, 5)  # Margins for each LEFT/RIGHT margin is first term
DEFAULT_ELEMENT_PADDING = (5, 3)  # Padding between elements (row, col) in pixels
DEFAULT_AUTOSIZE_TEXT = True
DEFAULT_AUTOSIZE_BUTTONS = True
DEFAULT_FONT = ('Helvetica', 10)
DEFAULT_TEXT_JUSTIFICATION = 'left'
DEFAULT_BORDER_WIDTH = 1
DEFAULT_AUTOCLOSE_TIME = 3  # time in seconds to show an autoclose form
DEFAULT_DEBUG_WINDOW_SIZE = (80, 20)
DEFAULT_WINDOW_LOCATION = (None, None)
MAX_SCROLLED_TEXT_BOX_HEIGHT = 50
DEFAULT_TOOLTIP_TIME = 400
DEFAULT_TOOLTIP_OFFSET = (0, -20)
DEFAULT_KEEP_ON_TOP = None
DEFAULT_SCALING = None
DEFAULT_ALPHA_CHANNEL = 1.0
DEFAULT_HIDE_WINDOW_WHEN_CREATING = True
TOOLTIP_BACKGROUND_COLOR = '#ffffe0'
TOOLTIP_FONT = None
DEFAULT_USE_BUTTON_SHORTCUTS = False
#################### COLOR STUFF ####################
BLUES = ('#082567', '#0A37A3', '#00345B')
PURPLES = ('#480656', '#4F2398', '#380474')
GREENS = ('#01826B', '#40A860', '#96D2AB', '#00A949', '#003532')
YELLOWS = ('#F3FB62', '#F0F595')
TANS = ('#FFF9D5', '#F4EFCF', '#DDD8BA')
NICE_BUTTON_COLORS = ((GREENS[3], TANS[0]),
                      ('#000000', '#FFFFFF'),
                      ('#FFFFFF', '#000000'),
                      (YELLOWS[0], PURPLES[1]),
                      (YELLOWS[0], GREENS[3]),
                      (YELLOWS[0], BLUES[2]))

COLOR_SYSTEM_DEFAULT = '1234567890'  # A Magic Number kind of signal to PySimpleGUI that the color should not be set at all
DEFAULT_BUTTON_COLOR = ('white', BLUES[0])  # Foreground, Background (None, None) == System Default
OFFICIAL_PYSIMPLEGUI_BUTTON_COLOR = ('white', BLUES[0])

# The "default PySimpleGUI theme"
OFFICIAL_PYSIMPLEGUI_THEME = CURRENT_LOOK_AND_FEEL = 'Dark Blue 3'

DEFAULT_ERROR_BUTTON_COLOR = ('#FFFFFF', '#FF0000')
DEFAULT_BACKGROUND_COLOR = None
DEFAULT_ELEMENT_BACKGROUND_COLOR = None
DEFAULT_ELEMENT_TEXT_COLOR = COLOR_SYSTEM_DEFAULT
DEFAULT_TEXT_ELEMENT_BACKGROUND_COLOR = None
DEFAULT_TEXT_COLOR = COLOR_SYSTEM_DEFAULT
DEFAULT_INPUT_ELEMENTS_COLOR = COLOR_SYSTEM_DEFAULT
DEFAULT_INPUT_TEXT_COLOR = COLOR_SYSTEM_DEFAULT
DEFAULT_SCROLLBAR_COLOR = None
# DEFAULT_BUTTON_COLOR = (YELLOWS[0], PURPLES[0])    # (Text, Background) or (Color "on", Color) as a way to remember
# DEFAULT_BUTTON_COLOR = (GREENS[3], TANS[0])    # Foreground, Background (None, None) == System Default
# DEFAULT_BUTTON_COLOR = (YELLOWS[0], GREENS[4])    # Foreground, Background (None, None) == System Default
# DEFAULT_BUTTON_COLOR = ('white', 'black')    # Foreground, Background (None, None) == System Default
# DEFAULT_BUTTON_COLOR = (YELLOWS[0], PURPLES[2])    # Foreground, Background (None, None) == System Default
# DEFAULT_PROGRESS_BAR_COLOR = (GREENS[2], GREENS[0])     # a nice green progress bar
# DEFAULT_PROGRESS_BAR_COLOR = (BLUES[1], BLUES[1])     # a nice green progress bar
# DEFAULT_PROGRESS_BAR_COLOR = (BLUES[0], BLUES[0])     # a nice green progress bar
# DEFAULT_PROGRESS_BAR_COLOR = (PURPLES[1],PURPLES[0])    # a nice purple progress bar


# A transparent button is simply one that matches the background
# TRANSPARENT_BUTTON = 'This constant has been depricated. You must set your button background = background it is on for it to be transparent appearing'


# --------------------------------------------------------------------------------
# Progress Bar Relief Choices
RELIEF_RAISED = 'raised'
RELIEF_SUNKEN = 'sunken'
RELIEF_FLAT = 'flat'
RELIEF_RIDGE = 'ridge'
RELIEF_GROOVE = 'groove'
RELIEF_SOLID = 'solid'
RELIEF_LIST = (RELIEF_RAISED, RELIEF_FLAT, RELIEF_SUNKEN, RELIEF_RIDGE, RELIEF_SOLID, RELIEF_GROOVE)

# These are the spepific themes that tkinter offers
THEME_DEFAULT = 'default'  # this is a TTK theme, not a PSG theme!!!
THEME_WINNATIVE = 'winnative'
THEME_CLAM = 'clam'
THEME_ALT = 'alt'
THEME_CLASSIC = 'classic'
THEME_VISTA = 'vista'
THEME_XPNATIVE = 'xpnative'

# The theme to use by default for all windows
DEFAULT_TTK_THEME = THEME_DEFAULT
ttk_theme_in_use = None
# TTK_THEME_LIST = ('default', 'winnative', 'clam', 'alt', 'classic', 'vista', 'xpnative')


USE_TTK_BUTTONS = None

DEFAULT_PROGRESS_BAR_COLOR = ('#01826B', '#D0D0D0')  # a nice green progress bar
DEFAULT_PROGRESS_BAR_COMPUTE = ('#000000', '#000000')  # Means that the progress bar colors should be computed from other colors
DEFAULT_PROGRESS_BAR_COLOR_OFFICIAL = ('#01826B', '#D0D0D0')  # a nice green progress bar
DEFAULT_PROGRESS_BAR_SIZE = (20, 20)  # Size of Progress Bar (characters for length, pixels for width)
DEFAULT_PROGRESS_BAR_BORDER_WIDTH = 1
DEFAULT_PROGRESS_BAR_RELIEF = RELIEF_GROOVE
# PROGRESS_BAR_STYLES = ('default', 'winnative', 'clam', 'alt', 'classic', 'vista', 'xpnative')
DEFAULT_PROGRESS_BAR_STYLE = DEFAULT_TTK_THEME
DEFAULT_METER_ORIENTATION = 'Horizontal'
DEFAULT_SLIDER_ORIENTATION = 'vertical'
DEFAULT_SLIDER_BORDER_WIDTH = 1
DEFAULT_SLIDER_RELIEF = tk.FLAT
DEFAULT_FRAME_RELIEF = tk.GROOVE

DEFAULT_LISTBOX_SELECT_MODE = tk.SINGLE
SELECT_MODE_MULTIPLE = tk.MULTIPLE
LISTBOX_SELECT_MODE_MULTIPLE = 'multiple'
SELECT_MODE_BROWSE = tk.BROWSE
LISTBOX_SELECT_MODE_BROWSE = 'browse'
SELECT_MODE_EXTENDED = tk.EXTENDED
LISTBOX_SELECT_MODE_EXTENDED = 'extended'
SELECT_MODE_SINGLE = tk.SINGLE
LISTBOX_SELECT_MODE_SINGLE = 'single'

TABLE_SELECT_MODE_NONE = tk.NONE
TABLE_SELECT_MODE_BROWSE = tk.BROWSE
TABLE_SELECT_MODE_EXTENDED = tk.EXTENDED
DEFAULT_TABLE_SELECT_MODE = TABLE_SELECT_MODE_EXTENDED
TABLE_CLICKED_INDICATOR = '+CLICKED+'           # Part of the tuple returned as an event when a Table element has click events enabled
DEFAULT_MODAL_WINDOWS_ENABLED = True
DEFAULT_MODAL_WINDOWS_FORCED = False

TAB_LOCATION_TOP = 'top'
TAB_LOCATION_TOP_LEFT = 'topleft'
TAB_LOCATION_TOP_RIGHT = 'topright'
TAB_LOCATION_LEFT = 'left'
TAB_LOCATION_LEFT_TOP = 'lefttop'
TAB_LOCATION_LEFT_BOTTOM = 'leftbottom'
TAB_LOCATION_RIGHT = 'right'
TAB_LOCATION_RIGHT_TOP = 'righttop'
TAB_LOCATION_RIGHT_BOTTOM = 'rightbottom'
TAB_LOCATION_BOTTOM = 'bottom'
TAB_LOCATION_BOTTOM_LEFT = 'bottomleft'
TAB_LOCATION_BOTTOM_RIGHT = 'bottomright'


TITLE_LOCATION_TOP = tk.N
TITLE_LOCATION_BOTTOM = tk.S
TITLE_LOCATION_LEFT = tk.W
TITLE_LOCATION_RIGHT = tk.E
TITLE_LOCATION_TOP_LEFT = tk.NW
TITLE_LOCATION_TOP_RIGHT = tk.NE
TITLE_LOCATION_BOTTOM_LEFT = tk.SW
TITLE_LOCATION_BOTTOM_RIGHT = tk.SE

TEXT_LOCATION_TOP = tk.N
TEXT_LOCATION_BOTTOM = tk.S
TEXT_LOCATION_LEFT = tk.W
TEXT_LOCATION_RIGHT = tk.E
TEXT_LOCATION_TOP_LEFT = tk.NW
TEXT_LOCATION_TOP_RIGHT = tk.NE
TEXT_LOCATION_BOTTOM_LEFT = tk.SW
TEXT_LOCATION_BOTTOM_RIGHT = tk.SE
TEXT_LOCATION_CENTER = tk.CENTER

GRAB_ANYWHERE_IGNORE_THESE_WIDGETS = (ttk.Sizegrip, tk.Scale, ttk.Scrollbar, tk.Scrollbar, tk.Entry, tk.Text, tk.PanedWindow, tk.Listbox, tk.OptionMenu, ttk.Treeview)

# ----====----====----==== Constants the user should NOT f-with ====----====----====----#
ThisRow = 555666777  # magic number

# DEFAULT_WINDOW_ICON = ''
MESSAGE_BOX_LINE_WIDTH = 60

# "Special" Key Values.. reserved
# Key representing a Read timeout
EVENT_TIMEOUT = TIMEOUT_EVENT = TIMEOUT_KEY = '__TIMEOUT__'
EVENT_TIMER = TIMER_KEY = '__TIMER EVENT__'
WIN_CLOSED = WINDOW_CLOSED = None
WINDOW_CLOSE_ATTEMPTED_EVENT = WIN_X_EVENT = WIN_CLOSE_ATTEMPTED_EVENT = '-WINDOW CLOSE ATTEMPTED-'
WINDOW_CONFIG_EVENT = '__WINDOW CONFIG__'
TITLEBAR_MINIMIZE_KEY = '__TITLEBAR MINIMIZE__'
TITLEBAR_MAXIMIZE_KEY = '__TITLEBAR MAXIMIZE__'
TITLEBAR_CLOSE_KEY = '__TITLEBAR CLOSE__'
TITLEBAR_IMAGE_KEY = '__TITLEBAR IMAGE__'
TITLEBAR_TEXT_KEY = '__TITLEBAR TEXT__'
TITLEBAR_DO_NOT_USE_AN_ICON = '__TITLEBAR_NO_ICON__'

# Key indicating should not create any return values for element
WRITE_ONLY_KEY = '__WRITE ONLY__'

MENU_DISABLED_CHARACTER = '!'
MENU_SHORTCUT_CHARACTER = '&'
MENU_KEY_SEPARATOR = '::'
MENU_SEPARATOR_LINE = '---'
MENU_RIGHT_CLICK_EDITME_EXIT = ['', ['Edit Me', 'Exit']]
MENU_RIGHT_CLICK_EDITME_VER_EXIT = ['', ['Edit Me', 'Version', 'Exit']]
MENU_RIGHT_CLICK_EDITME_VER_LOC_EXIT = ['', ['Edit Me', 'Version', 'File Location', 'Exit']]
MENU_RIGHT_CLICK_EDITME_VER_SETTINGS_EXIT = ['', ['Edit Me', 'Settings', 'Version', 'Exit']]
MENU_RIGHT_CLICK_EXIT = ['', ['Exit']]
MENU_RIGHT_CLICK_DISABLED = ['', []]
_MENU_RIGHT_CLICK_TABGROUP_DEFAULT = ['TABGROUP DEFAULT', []]
ENABLE_TK_WINDOWS = False

USE_CUSTOM_TITLEBAR = None
CUSTOM_TITLEBAR_BACKGROUND_COLOR = None
CUSTOM_TITLEBAR_TEXT_COLOR = None
CUSTOM_TITLEBAR_ICON = None
CUSTOM_TITLEBAR_FONT = None
TITLEBAR_METADATA_MARKER = 'This window has a titlebar'

CUSTOM_MENUBAR_METADATA_MARKER = 'This is a custom menubar'

SUPPRESS_ERROR_POPUPS = False
SUPPRESS_RAISE_KEY_ERRORS = True
SUPPRESS_KEY_GUESSING = False
SUPPRESS_WIDGET_NOT_FINALIZED_WARNINGS = False
ENABLE_TREEVIEW_869_PATCH = True

# These are now set based on the global settings file
ENABLE_MAC_NOTITLEBAR_PATCH = False
ENABLE_MAC_MODAL_DISABLE_PATCH = False
ENABLE_MAC_DISABLE_GRAB_ANYWHERE_WITH_TITLEBAR = True
ENABLE_MAC_ALPHA_99_PATCH= False

OLD_TABLE_TREE_SELECTED_ROW_COLORS = ('#FFFFFF', '#4A6984')
ALTERNATE_TABLE_AND_TREE_SELECTED_ROW_COLORS = ('SystemHighlightText', 'SystemHighlight')

# Some handy unicode symbols
SYMBOL_SQUARE = '█'
SYMBOL_CIRCLE = '⚫'
SYMBOL_CIRCLE_OUTLINE = '◯'
SYMBOL_BULLET = '•'
SYMBOL_UP = '▲'
SYMBOL_RIGHT = '►'
SYMBOL_LEFT = '◄'
SYMBOL_DOWN = '▼'
SYMBOL_X = '❎'
SYMBOL_CHECK = '✅'
SYMBOL_CHECK_SMALL = '✓'
SYMBOL_X_SMALL = '✗'
SYMBOL_BALLOT_X = '☒'
SYMBOL_BALLOT_CHECK = '☑'
SYMBOL_LEFT_DOUBLE = '«'
SYMBOL_RIGHT_DOUBLE = '»'
SYMBOL_LEFT_ARROWHEAD = '⮜'
SYMBOL_RIGHT_ARROWHEAD = '⮞'
SYMBOL_UP_ARROWHEAD = '⮝'
SYMBOL_DOWN_ARROWHEAD = '⮟'

if sum([int(i) for i in tclversion_detailed.split('.')]) > 19:
    SYMBOL_TITLEBAR_MINIMIZE = '_'
    SYMBOL_TITLEBAR_MAXIMIZE = '◻'
    SYMBOL_TITLEBAR_CLOSE = 'Ｘ'
else:
    SYMBOL_TITLEBAR_MINIMIZE = '_'
    SYMBOL_TITLEBAR_MAXIMIZE = 'O'
    SYMBOL_TITLEBAR_CLOSE = 'X'

#################### PATHS for user_settings APIs ####################
# These paths are passed to os.path.expanduser to get the default path for user_settings
# They can be changed using set_options

DEFAULT_USER_SETTINGS_WIN_PATH = r'~\AppData\Local\PySimpleGUI\settings'
DEFAULT_USER_SETTINGS_LINUX_PATH = r'~/.config/PySimpleGUI/settings'
DEFAULT_USER_SETTINGS_MAC_PATH = r'~/Library/Application Support/PySimpleGUI/settings'
DEFAULT_USER_SETTINGS_TRINKET_PATH = r'.'
DEFAULT_USER_SETTINGS_REPLIT_PATH = r'.'
DEFAULT_USER_SETTINGS_UNKNOWN_OS_PATH = r'~/Library/Application Support/PySimpleGUI/settings'
DEFAULT_USER_SETTINGS_PATH = None  # value set by user to override all paths above
DEFAULT_USER_SETTINGS_PYSIMPLEGUI_PATH = None  # location of the global PySimpleGUI settings
DEFAULT_USER_SETTINGS_PYSIMPLEGUI_FILENAME = '_PySimpleGUI_settings_global_.json'  # location of the global PySimpleGUI settings


# ====================================================================== #
# One-liner functions that are handy as f_ck                             #
# ====================================================================== #
def rgb(red, green, blue):
    """
    Given integer values of Red, Green, Blue, return a color string "#RRGGBB"
    :param red:   Red portion from 0 to 255
    :type red:    (int)
    :param green: Green portion from 0 to 255
    :type green:  (int)
    :param blue:  Blue portion from 0 to 255
    :type  blue:  (int)
    :return:      A single RGB String in the format "#RRGGBB" where each pair is a hex number.
    :rtype:       (str)
    """
    red = min(int(red), 255) if red > 0 else 0
    blue = min(int(blue), 255) if blue > 0 else 0
    green = min(int(green), 255) if green > 0 else 0
    return '#%02x%02x%02x' % (red, green, blue)


# ====================================================================== #
# Enums for types                                                        #
# ====================================================================== #
# -------------------------  Button types  ------------------------- #
# uncomment this line and indent to go back to using Enums
BUTTON_TYPE_BROWSE_FOLDER = 1
BUTTON_TYPE_BROWSE_FILE = 2
BUTTON_TYPE_BROWSE_FILES = 21
BUTTON_TYPE_SAVEAS_FILE = 3
BUTTON_TYPE_CLOSES_WIN = 5
BUTTON_TYPE_CLOSES_WIN_ONLY = 6
BUTTON_TYPE_READ_FORM = 7
BUTTON_TYPE_REALTIME = 9
BUTTON_TYPE_CALENDAR_CHOOSER = 30
BUTTON_TYPE_COLOR_CHOOSER = 40
BUTTON_TYPE_SHOW_DEBUGGER = 50

BROWSE_FILES_DELIMITER = ';'  # the delimiter to be used between each file in the returned string

FILE_TYPES_ALL_FILES = (('ALL Files', '*.* *'),)

BUTTON_DISABLED_MEANS_IGNORE = 'ignore'

# -------------------------  Element types  ------------------------- #

ELEM_TYPE_TEXT = 'text'
ELEM_TYPE_INPUT_TEXT = 'input'
ELEM_TYPE_INPUT_COMBO = 'combo'
ELEM_TYPE_INPUT_OPTION_MENU = 'option menu'
ELEM_TYPE_INPUT_RADIO = 'radio'
ELEM_TYPE_INPUT_MULTILINE = 'multiline'
ELEM_TYPE_INPUT_CHECKBOX = 'checkbox'
ELEM_TYPE_INPUT_SPIN = 'spind'
ELEM_TYPE_BUTTON = 'button'
ELEM_TYPE_IMAGE = 'image'
ELEM_TYPE_CANVAS = 'canvas'
ELEM_TYPE_FRAME = 'frame'
ELEM_TYPE_GRAPH = 'graph'
ELEM_TYPE_TAB = 'tab'
ELEM_TYPE_TAB_GROUP = 'tabgroup'
ELEM_TYPE_INPUT_SLIDER = 'slider'
ELEM_TYPE_INPUT_LISTBOX = 'listbox'
ELEM_TYPE_OUTPUT = 'output'
ELEM_TYPE_COLUMN = 'column'
ELEM_TYPE_MENUBAR = 'menubar'
ELEM_TYPE_PROGRESS_BAR = 'progressbar'
ELEM_TYPE_BLANK = 'blank'
ELEM_TYPE_TABLE = 'table'
ELEM_TYPE_TREE = 'tree'
ELEM_TYPE_ERROR = 'error'
ELEM_TYPE_SEPARATOR = 'separator'
ELEM_TYPE_STATUSBAR = 'statusbar'
ELEM_TYPE_PANE = 'pane'
ELEM_TYPE_BUTTONMENU = 'buttonmenu'
ELEM_TYPE_TITLEBAR = 'titlebar'
ELEM_TYPE_SIZEGRIP = 'sizegrip'

# STRETCH == ERROR ELEMENT as a filler

# -------------------------  Popup Buttons Types  ------------------------- #
POPUP_BUTTONS_YES_NO = 1
POPUP_BUTTONS_CANCELLED = 2
POPUP_BUTTONS_ERROR = 3
POPUP_BUTTONS_OK_CANCEL = 4
POPUP_BUTTONS_OK = 0
POPUP_BUTTONS_NO_BUTTONS = 5



PSG_THEME_PART_BUTTON_TEXT = 'Button Text Color'
PSG_THEME_PART_BUTTON_BACKGROUND = 'Button Background Color'
PSG_THEME_PART_BACKGROUND = 'Background Color'
PSG_THEME_PART_INPUT_BACKGROUND = 'Input Element Background Color'
PSG_THEME_PART_INPUT_TEXT = 'Input Element Text Color'
PSG_THEME_PART_TEXT = 'Text Color'
PSG_THEME_PART_SLIDER = 'Slider Color'
PSG_THEME_PART_LIST = [PSG_THEME_PART_BACKGROUND, PSG_THEME_PART_BUTTON_BACKGROUND, PSG_THEME_PART_BUTTON_TEXT,PSG_THEME_PART_INPUT_BACKGROUND, PSG_THEME_PART_INPUT_TEXT, PSG_THEME_PART_TEXT, PSG_THEME_PART_SLIDER ]

# theme_button

TTK_SCROLLBAR_PART_TROUGH_COLOR = 'Trough Color'
TTK_SCROLLBAR_PART_BACKGROUND_COLOR = 'Background Color'
TTK_SCROLLBAR_PART_ARROW_BUTTON_ARROW_COLOR = 'Arrow Button Arrow Color'
TTK_SCROLLBAR_PART_FRAME_COLOR = 'Frame Color'
TTK_SCROLLBAR_PART_SCROLL_WIDTH = 'Frame Width'
TTK_SCROLLBAR_PART_ARROW_WIDTH = 'Arrow Width'
TTK_SCROLLBAR_PART_RELIEF = 'Relief'
TTK_SCROLLBAR_PART_LIST = [TTK_SCROLLBAR_PART_TROUGH_COLOR, TTK_SCROLLBAR_PART_BACKGROUND_COLOR, TTK_SCROLLBAR_PART_ARROW_BUTTON_ARROW_COLOR,
                           TTK_SCROLLBAR_PART_FRAME_COLOR, TTK_SCROLLBAR_PART_SCROLL_WIDTH, TTK_SCROLLBAR_PART_ARROW_WIDTH, TTK_SCROLLBAR_PART_RELIEF]
TTK_SCROLLBAR_PART_THEME_BASED_LIST = [TTK_SCROLLBAR_PART_TROUGH_COLOR, TTK_SCROLLBAR_PART_BACKGROUND_COLOR, TTK_SCROLLBAR_PART_ARROW_BUTTON_ARROW_COLOR,
                           TTK_SCROLLBAR_PART_FRAME_COLOR]
DEFAULT_TTK_PART_MAPPING_DICT = {TTK_SCROLLBAR_PART_TROUGH_COLOR: PSG_THEME_PART_SLIDER,
                                 TTK_SCROLLBAR_PART_BACKGROUND_COLOR : PSG_THEME_PART_BUTTON_BACKGROUND,
                                 TTK_SCROLLBAR_PART_ARROW_BUTTON_ARROW_COLOR :PSG_THEME_PART_BUTTON_TEXT,
                                 TTK_SCROLLBAR_PART_FRAME_COLOR : PSG_THEME_PART_BACKGROUND,
                                 TTK_SCROLLBAR_PART_SCROLL_WIDTH : 12,
                                 TTK_SCROLLBAR_PART_ARROW_WIDTH: 12,
                                 TTK_SCROLLBAR_PART_RELIEF: RELIEF_RAISED}

ttk_part_mapping_dict = copy.copy(DEFAULT_TTK_PART_MAPPING_DICT)

class TTKPartOverrides():
    """
    This class contains "overrides" to the defaults for ttk scrollbars that are defined in the global settings file.
    This class is used in every element, in the Window class and there's a global one that is used by set_options.
    """
    def __init__(self, sbar_trough_color=None, sbar_background_color=None, sbar_arrow_color=None, sbar_width=None, sbar_arrow_width=None, sbar_frame_color=None, sbar_relief=None):
        self.sbar_trough_color = sbar_trough_color
        self.sbar_background_color = sbar_background_color
        self.sbar_arrow_color = sbar_arrow_color
        self.sbar_width = sbar_width
        self.sbar_arrow_width = sbar_arrow_width
        self.sbar_frame_color = sbar_frame_color
        self.sbar_relief = sbar_relief

ttk_part_overrides_from_options = TTKPartOverrides()


# -------------------------  tkinter BASIC cursors... there are some OS dependent ones too  ------------------------- #
TKINTER_CURSORS = ['X_cursor', 'arrow', 'based_arrow_down', 'based_arrow_up', 'boat',
                   'bogosity', 'bottom_left_corner', 'bottom_right_corner', 'bottom_side',
                   'bottom_tee', 'box_spiral', 'center_ptr', 'circle', 'clock',
                   'coffee_mug', 'cross', 'cross_reverse', 'crosshair', 'diamond_cross',
                   'dot', 'dotbox', 'double_arrow', 'draft_large', 'draft_small', 'draped_box',
                   'exchange', 'fleur', 'gobbler', 'gumby', 'hand1', 'hand2', 'heart',
                   'icon', 'iron_cross', 'left_ptr', 'left_side', 'left_tee', 'leftbutton',
                   'll_angle', 'lr_angle', 'man', 'middlebutton', 'mouse', 'pencil', 'pirate',
                   'plus', 'question_arrow', 'right_ptr', 'right_side', 'right_tee',
                   'rightbutton', 'rtl_logo', 'sailboat', 'sb_down_arrow', 'sb_h_double_arrow',
                   'sb_left_arrow', 'sb_right_arrow', 'sb_up_arrow', 'sb_v_double_arrow',
                   'shuttle', 'sizing', 'spider', 'spraycan', 'star', 'target', 'tcross',
                   'top_left_arrow', 'top_left_corner', 'top_right_corner', 'top_side', 'top_tee',
                   'trek', 'ul_angle', 'umbrella', 'ur_angle', 'watch', 'xterm']


TKINTER_CURSORS = ['X_cursor', 'arrow', 'based_arrow_down', 'based_arrow_up', 'boat', 'bogosity', 'bottom_left_corner', 'bottom_right_corner', 'bottom_side', 'bottom_tee', 'box_spiral', 'center_ptr', 'circle', 'clock', 'coffee_mug', 'cross', 'cross_reverse', 'crosshair', 'diamond_cross', 'dot', 'dotbox', 'double_arrow', 'draft_large', 'draft_small', 'draped_box', 'exchange', 'fleur', 'gobbler', 'gumby', 'hand1', 'hand2', 'heart', 'ibeam', 'icon', 'iron_cross', 'left_ptr', 'left_side', 'left_tee', 'leftbutton', 'll_angle', 'lr_angle', 'man', 'middlebutton', 'mouse', 'no', 'none', 'pencil', 'pirate', 'plus', 'question_arrow', 'right_ptr', 'right_side', 'right_tee', 'rightbutton', 'rtl_logo', 'sailboat', 'sb_down_arrow', 'sb_h_double_arrow', 'sb_left_arrow', 'sb_right_arrow', 'sb_up_arrow', 'sb_v_double_arrow', 'shuttle', 'size', 'size_ne_sw', 'size_ns', 'size_nw_se', 'size_we', 'sizing', 'spider', 'spraycan', 'star', 'starting', 'target', 'tcross', 'top_left_arrow', 'top_left_corner', 'top_right_corner', 'top_side', 'top_tee', 'trek', 'ul_angle', 'umbrella', 'uparrow', 'ur_angle', 'wait', 'watch', 'xterm']
# -------------------------  tkinter key codes for bindings  ------------------------- #

# The keycode that when pressed will take a snapshot of the current window
DEFAULT_WINDOW_SNAPSHOT_KEY_CODE = None
DEFAULT_WINDOW_SNAPSHOT_KEY = '--SCREENSHOT THIS WINDOW--'

tkinter_keysyms = ('space', 'exclam', 'quotedbl', 'numbersign', 'dollar', 'percent', 'ampersand', 'quoteright', 'parenleft', 'parenright', 'asterisk', 'plus', 'comma', 'minus', 'period', 'slash', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'colon', 'semicolon', 'less', 'equal', 'greater', 'question', 'at', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 'bracketleft', 'backslash', 'bracketright', 'asciicircum', 'underscore', 'quoteleft', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', 'braceleft', 'bar', 'braceright', 'asciitilde', 'nobreakspace', 'exclamdown', 'cent', 'sterling', 'currency', 'yen', 'brokenbar', 'section', 'diaeresis', 'copyright', 'ordfeminine', 'guillemotleft', 'notsign', 'hyphen', 'registered', 'macron', 'degree', 'plusminus', 'twosuperior', 'threesuperior', 'acute', 'mu', 'paragraph', 'periodcentered', 'cedilla', 'onesuperior', 'masculine', 'guillemotright', 'onequarter', 'onehalf', 'threequarters', 'questiondown', 'Agrave', 'Aacute', 'Acircumflex', 'Atilde', 'Adiaeresis', 'Aring', 'AE', 'Ccedilla', 'Egrave', 'Eacute', 'Ecircumflex', 'Ediaeresis', 'Igrave', 'Iacute', 'Icircumflex', 'Idiaeresis', 'Eth', 'Ntilde', 'Ograve', 'Oacute', 'Ocircumflex', 'Otilde', 'Odiaeresis', 'multiply', 'Ooblique', 'Ugrave', 'Uacute', 'Ucircumflex', 'Udiaeresis', 'Yacute', 'Thorn', 'ssharp', 'agrave', 'aacute', 'acircumflex', 'atilde', 'adiaeresis', 'aring', 'ae', 'ccedilla', 'egrave', 'eacute', 'ecircumflex', 'ediaeresis', 'igrave', 'iacute', 'icircumflex', 'idiaeresis', 'eth', 'ntilde', 'ograve', 'oacute', 'ocircumflex', 'otilde', 'odiaeresis', 'division', 'oslash', 'ugrave', 'uacute', 'ucircumflex', 'udiaeresis', 'yacute', 'thorn', 'ydiaeresis', 'Aogonek', 'breve', 'Lstroke', 'Lcaron', 'Sacute', 'Scaron', 'Scedilla', 'Tcaron', 'Zacute', 'Zcaron', 'Zabovedot', 'aogonek', 'ogonek', 'lstroke', 'lcaron', 'sacute', 'caron', 'scaron', 'scedilla', 'tcaron', 'zacute', 'doubleacute', 'zcaron', 'zabovedot', 'Racute', 'Abreve', 'Cacute', 'Ccaron', 'Eogonek', 'Ecaron', 'Dcaron', 'Nacute', 'Ncaron', 'Odoubleacute', 'Rcaron', 'Uring', 'Udoubleacute', 'Tcedilla', 'racute', 'abreve', 'cacute', 'ccaron', 'eogonek', 'ecaron', 'dcaron', 'nacute', 'ncaron', 'odoubleacute', 'rcaron', 'uring', 'udoubleacute', 'tcedilla', 'abovedot', 'Hstroke', 'Hcircumflex', 'Iabovedot', 'Gbreve', 'Jcircumflex', 'hstroke', 'hcircumflex', 'idotless', 'gbreve', 'jcircumflex', 'Cabovedot', 'Ccircumflex', 'Gabovedot', 'Gcircumflex', 'Ubreve', 'Scircumflex', 'cabovedot', 'ccircumflex', 'gabovedot', 'gcircumflex', 'ubreve', 'scircumflex', 'kappa', 'Rcedilla', 'Itilde', 'Lcedilla', 'Emacron', 'Gcedilla', 'Tslash', 'rcedilla', 'itilde', 'lcedilla', 'emacron', 'gacute', 'tslash', 'ENG', 'eng', 'Amacron', 'Iogonek', 'Eabovedot', 'Imacron', 'Ncedilla', 'Omacron', 'Kcedilla', 'Uogonek', 'Utilde', 'Umacron', 'amacron', 'iogonek', 'eabovedot', 'imacron', 'ncedilla', 'omacron', 'kcedilla', 'uogonek', 'utilde', 'umacron', 'overline', 'kana_fullstop', 'kana_openingbracket', 'kana_closingbracket', 'kana_comma', 'kana_middledot', 'kana_WO', 'kana_a', 'kana_i', 'kana_u', 'kana_e', 'kana_o', 'kana_ya', 'kana_yu', 'kana_yo', 'kana_tu', 'prolongedsound', 'kana_A', 'kana_I', 'kana_U', 'kana_E', 'kana_O', 'kana_KA', 'kana_KI', 'kana_KU', 'kana_KE', 'kana_KO', 'kana_SA', 'kana_SHI', 'kana_SU', 'kana_SE', 'kana_SO', 'kana_TA', 'kana_TI', 'kana_TU', 'kana_TE', 'kana_TO', 'kana_NA', 'kana_NI', 'kana_NU', 'kana_NE', 'kana_NO', 'kana_HA', 'kana_HI', 'kana_HU', 'kana_HE', 'kana_HO', 'kana_MA', 'kana_MI', 'kana_MU', 'kana_ME', 'kana_MO', 'kana_YA', 'kana_YU', 'kana_YO', 'kana_RA', 'kana_RI', 'kana_RU', 'kana_RE', 'kana_RO', 'kana_WA', 'kana_N', 'voicedsound', 'semivoicedsound', 'Arabic_comma', 'Arabic_semicolon', 'Arabic_question_mark', 'Arabic_hamza', 'Arabic_maddaonalef', 'Arabic_hamzaonalef', 'Arabic_hamzaonwaw', 'Arabic_hamzaunderalef', 'Arabic_hamzaonyeh', 'Arabic_alef', 'Arabic_beh', 'Arabic_tehmarbuta', 'Arabic_teh', 'Arabic_theh', 'Arabic_jeem', 'Arabic_hah', 'Arabic_khah', 'Arabic_dal', 'Arabic_thal', 'Arabic_ra', 'Arabic_zain', 'Arabic_seen', 'Arabic_sheen', 'Arabic_sad', 'Arabic_dad', 'Arabic_tah', 'Arabic_zah', 'Arabic_ain', 'Arabic_ghain', 'Arabic_tatweel', 'Arabic_feh', 'Arabic_qaf', 'Arabic_kaf', 'Arabic_lam', 'Arabic_meem', 'Arabic_noon', 'Arabic_heh', 'Arabic_waw', 'Arabic_alefmaksura', 'Arabic_yeh', 'Arabic_fathatan', 'Arabic_dammatan', 'Arabic_kasratan', 'Arabic_fatha', 'Arabic_damma', 'Arabic_kasra', 'Arabic_shadda', 'Arabic_sukun', 'Serbian_dje', 'Macedonia_gje', 'Cyrillic_io', 'Ukranian_je', 'Macedonia_dse', 'Ukranian_i', 'Ukranian_yi', 'Serbian_je', 'Serbian_lje', 'Serbian_nje', 'Serbian_tshe', 'Macedonia_kje', 'Byelorussian_shortu', 'Serbian_dze', 'numerosign', 'Serbian_DJE', 'Macedonia_GJE', 'Cyrillic_IO', 'Ukranian_JE', 'Macedonia_DSE', 'Ukranian_I', 'Ukranian_YI', 'Serbian_JE', 'Serbian_LJE', 'Serbian_NJE', 'Serbian_TSHE', 'Macedonia_KJE', 'Byelorussian_SHORTU', 'Serbian_DZE', 'Cyrillic_yu', 'Cyrillic_a', 'Cyrillic_be', 'Cyrillic_tse', 'Cyrillic_de', 'Cyrillic_ie', 'Cyrillic_ef', 'Cyrillic_ghe', 'Cyrillic_ha', 'Cyrillic_i', 'Cyrillic_shorti', 'Cyrillic_ka', 'Cyrillic_el', 'Cyrillic_em', 'Cyrillic_en', 'Cyrillic_o', 'Cyrillic_pe', 'Cyrillic_ya', 'Cyrillic_er', 'Cyrillic_es', 'Cyrillic_te', 'Cyrillic_u', 'Cyrillic_zhe', 'Cyrillic_ve', 'Cyrillic_softsign', 'Cyrillic_yeru', 'Cyrillic_ze', 'Cyrillic_sha', 'Cyrillic_e', 'Cyrillic_shcha', 'Cyrillic_che', 'Cyrillic_hardsign', 'Cyrillic_YU', 'Cyrillic_A', 'Cyrillic_BE', 'Cyrillic_TSE', 'Cyrillic_DE', 'Cyrillic_IE', 'Cyrillic_EF', 'Cyrillic_GHE', 'Cyrillic_HA', 'Cyrillic_I', 'Cyrillic_SHORTI', 'Cyrillic_KA', 'Cyrillic_EL', 'Cyrillic_EM', 'Cyrillic_EN', 'Cyrillic_O', 'Cyrillic_PE', 'Cyrillic_YA', 'Cyrillic_ER', 'Cyrillic_ES', 'Cyrillic_TE', 'Cyrillic_U', 'Cyrillic_ZHE', 'Cyrillic_VE', 'Cyrillic_SOFTSIGN', 'Cyrillic_YERU', 'Cyrillic_ZE', 'Cyrillic_SHA', 'Cyrillic_E', 'Cyrillic_SHCHA', 'Cyrillic_CHE', 'Cyrillic_HARDSIGN', 'Greek_ALPHAaccent', 'Greek_EPSILONaccent', 'Greek_ETAaccent', 'Greek_IOTAaccent', 'Greek_IOTAdiaeresis', 'Greek_IOTAaccentdiaeresis', 'Greek_OMICRONaccent', 'Greek_UPSILONaccent', 'Greek_UPSILONdieresis', 'Greek_UPSILONaccentdieresis', 'Greek_OMEGAaccent', 'Greek_alphaaccent', 'Greek_epsilonaccent', 'Greek_etaaccent', 'Greek_iotaaccent', 'Greek_iotadieresis', 'Greek_iotaaccentdieresis', 'Greek_omicronaccent', 'Greek_upsilonaccent', 'Greek_upsilondieresis', 'Greek_upsilonaccentdieresis', 'Greek_omegaaccent', 'Greek_ALPHA', 'Greek_BETA', 'Greek_GAMMA', 'Greek_DELTA', 'Greek_EPSILON', 'Greek_ZETA', 'Greek_ETA', 'Greek_THETA', 'Greek_IOTA', 'Greek_KAPPA', 'Greek_LAMBDA', 'Greek_MU', 'Greek_NU', 'Greek_XI', 'Greek_OMICRON', 'Greek_PI', 'Greek_RHO', 'Greek_SIGMA', 'Greek_TAU', 'Greek_UPSILON', 'Greek_PHI', 'Greek_CHI', 'Greek_PSI', 'Greek_OMEGA', 'Greek_alpha', 'Greek_beta', 'Greek_gamma', 'Greek_delta', 'Greek_epsilon', 'Greek_zeta', 'Greek_eta', 'Greek_theta', 'Greek_iota', 'Greek_kappa', 'Greek_lambda', 'Greek_mu', 'Greek_nu', 'Greek_xi', 'Greek_omicron', 'Greek_pi', 'Greek_rho', 'Greek_sigma', 'Greek_finalsmallsigma', 'Greek_tau', 'Greek_upsilon', 'Greek_phi', 'Greek_chi', 'Greek_psi', 'Greek_omega', 'leftradical', 'topleftradical', 'horizconnector', 'topintegral', 'botintegral', 'vertconnector', 'topleftsqbracket', 'botleftsqbracket', 'toprightsqbracket', 'botrightsqbracket', 'topleftparens', 'botleftparens', 'toprightparens', 'botrightparens', 'leftmiddlecurlybrace', 'rightmiddlecurlybrace', 'topleftsummation', 'botleftsummation', 'topvertsummationconnector', 'botvertsummationconnector', 'toprightsummation', 'botrightsummation', 'rightmiddlesummation', 'lessthanequal', 'notequal', 'greaterthanequal', 'integral', 'therefore', 'variation', 'infinity', 'nabla', 'approximate', 'similarequal', 'ifonlyif', 'implies', 'identical', 'radical', 'includedin', 'includes', 'intersection', 'union', 'logicaland', 'logicalor', 'partialderivative', 'function', 'leftarrow', 'uparrow', 'rightarrow', 'downarrow', 'blank', 'soliddiamond', 'checkerboard', 'ht', 'ff', 'cr', 'lf', 'nl', 'vt', 'lowrightcorner', 'uprightcorner', 'upleftcorner', 'lowleftcorner', 'crossinglines', 'horizlinescan1', 'horizlinescan3', 'horizlinescan5', 'horizlinescan7', 'horizlinescan9', 'leftt', 'rightt', 'bott', 'topt', 'vertbar', 'emspace', 'enspace', 'em3space', 'em4space', 'digitspace', 'punctspace', 'thinspace', 'hairspace', 'emdash', 'endash', 'signifblank', 'ellipsis', 'doubbaselinedot', 'onethird', 'twothirds', 'onefifth', 'twofifths', 'threefifths', 'fourfifths', 'onesixth', 'fivesixths', 'careof', 'figdash', 'leftanglebracket', 'decimalpoint', 'rightanglebracket', 'marker', 'oneeighth', 'threeeighths', 'fiveeighths', 'seveneighths', 'trademark', 'signaturemark', 'trademarkincircle', 'leftopentriangle', 'rightopentriangle', 'emopencircle', 'emopenrectangle', 'leftsinglequotemark', 'rightsinglequotemark', 'leftdoublequotemark', 'rightdoublequotemark', 'prescription', 'minutes', 'seconds', 'latincross', 'hexagram', 'filledrectbullet', 'filledlefttribullet', 'filledrighttribullet', 'emfilledcircle', 'emfilledrect', 'enopencircbullet', 'enopensquarebullet', 'openrectbullet', 'opentribulletup', 'opentribulletdown', 'openstar', 'enfilledcircbullet', 'enfilledsqbullet', 'filledtribulletup', 'filledtribulletdown', 'leftpointer', 'rightpointer', 'club', 'diamond', 'heart', 'maltesecross', 'dagger', 'doubledagger', 'checkmark', 'ballotcross', 'musicalsharp', 'musicalflat', 'malesymbol', 'femalesymbol', 'telephone', 'telephonerecorder', 'phonographcopyright', 'caret', 'singlelowquotemark', 'doublelowquotemark', 'cursor', 'leftcaret', 'rightcaret', 'downcaret', 'upcaret', 'overbar', 'downtack', 'upshoe', 'downstile', 'underbar', 'jot', 'quad', 'uptack', 'circle', 'upstile', 'downshoe', 'rightshoe', 'leftshoe', 'lefttack', 'righttack', 'hebrew_aleph', 'hebrew_beth', 'hebrew_gimmel', 'hebrew_daleth', 'hebrew_he', 'hebrew_waw', 'hebrew_zayin', 'hebrew_het', 'hebrew_teth', 'hebrew_yod', 'hebrew_finalkaph', 'hebrew_kaph', 'hebrew_lamed', 'hebrew_finalmem', 'hebrew_mem', 'hebrew_finalnun', 'hebrew_nun', 'hebrew_samekh', 'hebrew_ayin', 'hebrew_finalpe', 'hebrew_pe', 'hebrew_finalzadi', 'hebrew_zadi', 'hebrew_kuf', 'hebrew_resh', 'hebrew_shin', 'hebrew_taf', 'BackSpace', 'Tab', 'Linefeed', 'Clear', 'Return', 'Pause', 'Scroll_Lock', 'Sys_Req', 'Escape', 'Multi_key', 'Kanji', 'Home', 'Left', 'Up', 'Right', 'Down', 'Prior', 'Next', 'End', 'Begin', 'Win_L', 'Win_R', 'App', 'Select', 'Print', 'Execute', 'Insert', 'Undo', 'Redo', 'Menu', 'Find', 'Cancel', 'Help', 'Break', 'Hebrew_switch', 'Num_Lock', 'KP_Space', 'KP_Tab', 'KP_Enter', 'KP_F1', 'KP_F2', 'KP_F3', 'KP_F4', 'KP_Multiply', 'KP_Add', 'KP_Separator', 'KP_Subtract', 'KP_Decimal', 'KP_Divide', 'KP_0', 'KP_1', 'KP_2', 'KP_3', 'KP_4', 'KP_5', 'KP_6', 'KP_7', 'KP_8', 'KP_9', 'KP_Equal', 'F1', 'F2', 'F3', 'F4', 'F5', 'F6', 'F7', 'F8', 'F9', 'F10', 'F11', 'F12', 'L1', 'L2', 'L3', 'L4', 'L5', 'L6', 'L7', 'L8', 'L9', 'L10', 'R1', 'R2', 'R3', 'R4', 'R5', 'R6', 'R7', 'R8', 'R9', 'R10', 'R11', 'R12', 'F33', 'R14', 'R15', 'Shift_L', 'Shift_R', 'Control_L', 'Control_R', 'Caps_Lock', 'Shift_Lock', 'Meta_L', 'Meta_R', 'Alt_L', 'Alt_R', 'Super_L', 'Super_R', 'Hyper_L', 'Hyper_R', 'Delete')





# ------------------------------------------------------------------------- #
#                       ToolTip used by the Elements                        #
# ------------------------------------------------------------------------- #

class ToolTip:
    """
    Create a tooltip for a given widget
    (inspired by https://stackoverflow.com/a/36221216)
    This is an INTERNALLY USED only class.  Users should not refer to this class at all.
    """

    def __init__(self, widget, text, timeout=DEFAULT_TOOLTIP_TIME):
        """
        :param widget:  The tkinter widget
        :type widget:   widget type varies
        :param text:    text for the tooltip. It can inslude \n
        :type text:     (str)
        :param timeout: Time in milliseconds that mouse must remain still before tip is shown
        :type timeout:  (int)
        """
        self.widget = widget
        self.text = text
        self.timeout = timeout
        # self.wraplength = wraplength if wraplength else widget.winfo_screenwidth() // 2
        self.tipwindow = None
        self.id = None
        self.x = self.y = 0
        self.widget.bind('<Enter>', self.enter)
        self.widget.bind('<Leave>', self.leave)
        self.widget.bind('<ButtonPress>', self.leave)

    def enter(self, event=None):
        """
        Called by tkinter when mouse enters a widget
        :param event: from tkinter.  Has x,y coordinates of mouse
        :type event:

        """
        self.x = event.x
        self.y = event.y
        self.schedule()

    def leave(self, event=None):
        """
        Called by tktiner when mouse exits a widget
        :param event: from tkinter.  Event info that's not used by function.
        :type event:

        """
        self.unschedule()
        self.hidetip()

    def schedule(self):
        """
        Schedule a timer to time how long mouse is hovering
        """
        self.unschedule()
        self.id = self.widget.after(self.timeout, self.showtip)

    def unschedule(self):
        """
        Cancel timer used to time mouse hover
        """
        if self.id:
            self.widget.after_cancel(self.id)
        self.id = None

    def showtip(self):
        """
        Creates a topoltip window with the tooltip text inside of it
        """
        if self.tipwindow:
            return
        x = self.widget.winfo_rootx() + self.x + DEFAULT_TOOLTIP_OFFSET[0]
        y = self.widget.winfo_rooty() + self.y + DEFAULT_TOOLTIP_OFFSET[1]
        self.tipwindow = tk.Toplevel(self.widget)
        # if not sys.platform.startswith('darwin'):
        try:
            self.tipwindow.wm_overrideredirect(True)
            # if running_mac() and ENABLE_MAC_NOTITLEBAR_PATCH:
            if _mac_should_apply_notitlebar_patch():
                self.tipwindow.wm_overrideredirect(False)
        except Exception as e:
            print('* Error performing wm_overrideredirect in showtip *', e)
        self.tipwindow.wm_geometry('+%d+%d' % (x, y))
        self.tipwindow.wm_attributes('-topmost', 1)

        label = ttk.Label(self.tipwindow, text=self.text, justify=tk.LEFT,
                          background=TOOLTIP_BACKGROUND_COLOR, relief=tk.SOLID, borderwidth=1)
        if TOOLTIP_FONT is not None:
            label.config(font=TOOLTIP_FONT)
        label.pack()

    def hidetip(self):
        """
        Destroy the tooltip window
        """
        if self.tipwindow:
            self.tipwindow.destroy()
        self.tipwindow = None


# ---------------------------------------------------------------------- #
# Cascading structure.... Objects get larger                             #
#   Button                                                               #
#       Element                                                          #
#           Row                                                          #
#               Form                                                     #
# ---------------------------------------------------------------------- #
# ------------------------------------------------------------------------- #
#                       Element CLASS                                       #
# ------------------------------------------------------------------------- #
class Element():
    """ The base class for all Elements. Holds the basic description of an Element like size and colors """

    def __init__(self, type, size=(None, None), auto_size_text=None, font=None, background_color=None, text_color=None, key=None, pad=None, tooltip=None,
                 visible=True, metadata=None,
                 sbar_trough_color=None, sbar_background_color=None, sbar_arrow_color=None, sbar_width=None, sbar_arrow_width=None, sbar_frame_color=None, sbar_relief=None):
        """
        Element base class. Only used internally.  User will not create an Element object by itself

        :param type:                        The type of element. These constants all start with "ELEM_TYPE_"
        :type type:                         (int) (could be enum)
        :param size:                        w=characters-wide, h=rows-high. If an int instead of a tuple is supplied, then height is auto-set to 1
        :type size:                         (int, int) | (None, None) | int
        :param auto_size_text:              True if the Widget should be shrunk to exactly fit the number of chars to show
        :type auto_size_text:               bool
        :param font:                        specifies the font family, size. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:                         (str or (str, int[, str]) or None)
        :param background_color:            color of background. Can be in #RRGGBB format or a color name "black"
        :type background_color:             (str)
        :param text_color:                  element's text color. Can be in #RRGGBB format or a color name "black"
        :type text_color:                   (str)
        :param key:                         Identifies an Element. Should be UNIQUE to this window.
        :type key:                          str | int | tuple | object
        :param pad:                         Amount of padding to put around element in pixels (left/right, top/bottom). If an int is given, then auto-converted to tuple (int, int)
        :type pad:                          (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param tooltip:                     text, that will appear when mouse hovers over the element
        :type tooltip:                      (str)
        :param visible:                     set visibility state of the element (Default = True)
        :type visible:                      (bool)
        :param metadata:                    User metadata that can be set to ANYTHING
        :type metadata:                     (Any)
        :param sbar_trough_color:           Scrollbar color of the trough
        :type sbar_trough_color:            (str)
        :param sbar_background_color:       Scrollbar color of the background of the arrow buttons at the ends AND the color of the "thumb" (the thing you grab and slide). Switches to arrow color when mouse is over
        :type sbar_background_color:        (str)
        :param sbar_arrow_color:            Scrollbar color of the arrow at the ends of the scrollbar (it looks like a button). Switches to background color when mouse is over
        :type sbar_arrow_color:             (str)
        :param sbar_width:                  Scrollbar width in pixels
        :type sbar_width:                   (int)
        :param sbar_arrow_width:            Scrollbar width of the arrow on the scrollbar. It will potentially impact the overall width of the scrollbar
        :type sbar_arrow_width:             (int)
        :param sbar_frame_color:            Scrollbar Color of frame around scrollbar (available only on some ttk themes)
        :type sbar_frame_color:             (str)
        :param sbar_relief:                 Scrollbar relief that will be used for the "thumb" of the scrollbar (the thing you grab that slides). Should be a constant that is defined at starting with "RELIEF_" - RELIEF_RAISED, RELIEF_SUNKEN, RELIEF_FLAT, RELIEF_RIDGE, RELIEF_GROOVE, RELIEF_SOLID
        :type sbar_relief:                  (str)
        """

        if size is not None and size != (None, None):
            if isinstance(size, int):
                size = (size, 1)
            if isinstance(size, tuple) and len(size) == 1:
                size = (size[0],  1)

        if pad is not None and pad != (None, None):
            if isinstance(pad, int):
                pad = (pad, pad)


        self.Size = size
        self.Type = type
        self.AutoSizeText = auto_size_text

        self.Pad = pad
        self.Font = font

        self.TKStringVar = None
        self.TKIntVar = None
        self.TKText = None
        self.TKEntry = None
        self.TKImage = None
        self.ttk_style_name = ''        # The ttk style name (if this is a ttk widget)
        self.ttk_style = None           # The ttk Style object (if this is a ttk widget)
        self._metadata = None  # type: Any

        self.ParentForm = None  # type: Window
        self.ParentContainer = None  # will be a Form, Column, or Frame element # UNBIND
        self.TextInputDefault = None
        self.Position = (0, 0)  # Default position Row 0, Col 0
        self.BackgroundColor = background_color if background_color is not None else DEFAULT_ELEMENT_BACKGROUND_COLOR
        self.TextColor = text_color if text_color is not None else DEFAULT_ELEMENT_TEXT_COLOR
        self.Key = key  # dictionary key for return values
        self.Tooltip = tooltip
        self.TooltipObject = None
        self._visible = visible
        self.TKRightClickMenu = None
        self.Widget = None  # Set when creating window. Has the main tkinter widget for element
        self.Tearoff = False  # needed because of right click menu code
        self.ParentRowFrame = None  # type tk.Frame
        self.metadata = metadata
        self.user_bind_dict = {}  # Used when user defines a tkinter binding using bind method - convert bind string to key modifier
        self.user_bind_event = None  # Used when user defines a tkinter binding using bind method - event data from tkinter
        # self.pad_used = (0, 0)  # the amount of pad used when was inserted into the layout
        self._popup_menu_location = (None, None)
        self.pack_settings = None
        self.vsb_style_name = None           # ttk style name used for the verical scrollbar if one is attached to element
        self.hsb_style_name = None           # ttk style name used for the horizontal scrollbar if one is attached to element
        self.vsb_style = None                # The ttk style used for the vertical scrollbar if one is attached to element
        self.hsb_style = None                # The ttk style used for the horizontal scrollbar if one is attached to element
        self.hsb = None                      # The horizontal scrollbar if one is attached to element
        self.vsb = None                      # The vertical scrollbar if one is attached to element
        ## TTK Scrollbar Settings
        self.ttk_part_overrides = TTKPartOverrides(sbar_trough_color=sbar_trough_color, sbar_background_color=sbar_background_color, sbar_arrow_color=sbar_arrow_color, sbar_width=sbar_width, sbar_arrow_width=sbar_arrow_width, sbar_frame_color=sbar_frame_color, sbar_relief=sbar_relief)

        PSG_THEME_PART_FUNC_MAP = {PSG_THEME_PART_BACKGROUND: theme_background_color,
                                       PSG_THEME_PART_BUTTON_BACKGROUND: theme_button_color_background,
                                       PSG_THEME_PART_BUTTON_TEXT: theme_button_color_text,
                                       PSG_THEME_PART_INPUT_BACKGROUND: theme_input_background_color,
                                       PSG_THEME_PART_INPUT_TEXT: theme_input_text_color,
                                       PSG_THEME_PART_TEXT: theme_text_color,
                                       PSG_THEME_PART_SLIDER: theme_slider_color}

        # class Theme_Parts():
        #     PSG_THEME_PART_FUNC_MAP = {PSG_THEME_PART_BACKGROUND: theme_background_color,
        if sbar_trough_color is not None:
            self.scroll_trough_color = sbar_trough_color
        else:
            self.scroll_trough_color = PSG_THEME_PART_FUNC_MAP.get(ttk_part_mapping_dict[TTK_SCROLLBAR_PART_TROUGH_COLOR], ttk_part_mapping_dict[TTK_SCROLLBAR_PART_TROUGH_COLOR])
            if callable(self.scroll_trough_color):
                self.scroll_trough_color = self.scroll_trough_color()


        if sbar_background_color is not None:
            self.scroll_background_color = sbar_background_color
        else:
            self.scroll_background_color = PSG_THEME_PART_FUNC_MAP.get(ttk_part_mapping_dict[TTK_SCROLLBAR_PART_BACKGROUND_COLOR], ttk_part_mapping_dict[TTK_SCROLLBAR_PART_BACKGROUND_COLOR])
            if callable(self.scroll_background_color):
                    self.scroll_background_color = self.scroll_background_color()


        if sbar_arrow_color is not None:
            self.scroll_arrow_color = sbar_arrow_color
        else:
            self.scroll_arrow_color = PSG_THEME_PART_FUNC_MAP.get(ttk_part_mapping_dict[TTK_SCROLLBAR_PART_ARROW_BUTTON_ARROW_COLOR], ttk_part_mapping_dict[TTK_SCROLLBAR_PART_ARROW_BUTTON_ARROW_COLOR])
            if callable(self.scroll_arrow_color):
                self.scroll_arrow_color = self.scroll_arrow_color()


        if sbar_frame_color is not None:
            self.scroll_frame_color = sbar_frame_color
        else:
            self.scroll_frame_color = PSG_THEME_PART_FUNC_MAP.get(ttk_part_mapping_dict[TTK_SCROLLBAR_PART_FRAME_COLOR], ttk_part_mapping_dict[TTK_SCROLLBAR_PART_FRAME_COLOR])
            if callable(self.scroll_frame_color):
                self.scroll_frame_color = self.scroll_frame_color()

        if sbar_relief is not None:
            self.scroll_relief = sbar_relief
        else:
            self.scroll_relief = ttk_part_mapping_dict[TTK_SCROLLBAR_PART_RELIEF]

        if sbar_width is not None:
            self.scroll_width = sbar_width
        else:
            self.scroll_width = ttk_part_mapping_dict[TTK_SCROLLBAR_PART_SCROLL_WIDTH]

        if sbar_arrow_width is not None:
            self.scroll_arrow_width = sbar_arrow_width
        else:
            self.scroll_arrow_width = ttk_part_mapping_dict[TTK_SCROLLBAR_PART_ARROW_WIDTH]


        if not hasattr(self, 'DisabledTextColor'):
            self.DisabledTextColor = None
        if not hasattr(self, 'ItemFont'):
            self.ItemFont = None
        if not hasattr(self, 'RightClickMenu'):
            self.RightClickMenu = None
        if not hasattr(self, 'Disabled'):
            self.Disabled = None        # in case the element hasn't defined this, add it here

    @property
    def visible(self):
        """
        Returns visibility state for the element.  This is a READONLY property
        :return: Visibility state for element
        :rtype:  (bool)
        """
        return self._visible

    @property
    def metadata(self):
        """
        Metadata is an Element property that you can use at any time to hold any value
        :return: the current metadata value
        :rtype:  (Any)
        """
        return self._metadata

    @metadata.setter
    def metadata(self, value):
        """
         Metadata is an Element property that you can use at any time to hold any value
        :param value: Anything you want it to be
        :type value:  (Any)
        """
        self._metadata = value

    @property
    def key(self):
        """
        Returns key for the element.  This is a READONLY property.
        Keys can be any hashable object (basically anything except a list... tuples are ok, but not lists)
        :return: The window's Key
        :rtype:  (Any)
        """
        return self.Key



    @property
    def widget(self):
        """
        Returns tkinter widget for the element.  This is a READONLY property.
        The implementation is that the Widget member variable is returned. This is a backward compatible addition
        :return:    The element's underlying tkinter widget
        :rtype:     (tkinter.Widget)
        """
        return self.Widget




    def _RightClickMenuCallback(self, event):
        """
        Callback function that's called when a right click happens. Shows right click menu as result

        :param event: information provided by tkinter about the event including x,y location of click
        :type event:

        """
        if self.Type == ELEM_TYPE_TAB_GROUP:
            try:
                index  = self.Widget.index('@{},{}'.format(event.x,event.y))
                tab = self.Widget.tab(index, 'text')
                key = self.find_key_from_tab_name(tab)
                tab_element = self.ParentForm.key_dict[key]
                if tab_element.RightClickMenu is None:      # if this tab didn't explicitly have a menu, then don't show anything
                    return
                tab_element.TKRightClickMenu.tk_popup(event.x_root, event.y_root, 0)
                self.TKRightClickMenu.grab_release()
            except:
                pass
            return
        self.TKRightClickMenu.tk_popup(event.x_root, event.y_root, 0)
        self.TKRightClickMenu.grab_release()
        if self.Type == ELEM_TYPE_GRAPH:
            self._update_position_for_returned_values(event)

    def _tearoff_menu_callback(self, parent, menu):
        """
        Callback function that's called when a right click menu is torn off.
        The reason for this function is to relocate the torn-off menu. It will default to 0,0 otherwise
        This callback moves the right click menu window to the location of the current window

        :param parent: information provided by tkinter - the parent of the Meny
        :type parent:
        :param menu:   information provided by tkinter - the menu window
        :type menu:

        """
        if self._popup_menu_location == (None, None):
            winx, winy = self.ParentForm.current_location()
        else:
            winx, winy = self._popup_menu_location
        # self.ParentForm.TKroot.update()
        self.ParentForm.TKroot.tk.call('wm', 'geometry', menu, '+{}+{}'.format(winx, winy))

    def _MenuItemChosenCallback(self, item_chosen):  # TEXT Menu item callback
        """
        Callback function called when user chooses a menu item from menubar, Button Menu or right click menu

        :param item_chosen: String holding the value chosen.
        :type item_chosen:  str

        """
        # print('IN MENU ITEM CALLBACK', item_chosen)
        self.MenuItemChosen = item_chosen
        self.ParentForm.LastButtonClicked = self.MenuItemChosen
        self.ParentForm.FormRemainedOpen = True
        _exit_mainloop(self.ParentForm)
        # Window._window_that_exited = self.ParentForm
        # self.ParentForm.TKroot.quit()  # kick the users out of the mainloop

    def _FindReturnKeyBoundButton(self, form):
        """
        Searches for which Button has the flag Button.BindReturnKey set.  It is called recursively when a
        "Container Element" is encountered. Func has to walk entire window including these "sub-forms"

        :param form: the Window object to search
        :type form:
        :return:     Button Object if a button is found, else None
        :rtype:      Button | None
        """
        for row in form.Rows:
            for element in row:
                if element.Type == ELEM_TYPE_BUTTON:
                    if element.BindReturnKey:
                        return element
                if element.Type == ELEM_TYPE_COLUMN:
                    rc = self._FindReturnKeyBoundButton(element)
                    if rc is not None:
                        return rc
                if element.Type == ELEM_TYPE_FRAME:
                    rc = self._FindReturnKeyBoundButton(element)
                    if rc is not None:
                        return rc
                if element.Type == ELEM_TYPE_TAB_GROUP:
                    rc = self._FindReturnKeyBoundButton(element)
                    if rc is not None:
                        return rc
                if element.Type == ELEM_TYPE_TAB:
                    rc = self._FindReturnKeyBoundButton(element)
                    if rc is not None:
                        return rc
                if element.Type == ELEM_TYPE_PANE:
                    rc = self._FindReturnKeyBoundButton(element)
                    if rc is not None:
                        return rc
        return None

    def _TextClickedHandler(self, event):
        """
        Callback that's called when a text element is clicked on with events enabled on the Text Element.
        Result is that control is returned back to user (quits mainloop).

        :param event:
        :type event:

        """
        # If this is a minimize button for a custom titlebar, then minimize the window
        if self.Key in (TITLEBAR_MINIMIZE_KEY, TITLEBAR_MAXIMIZE_KEY, TITLEBAR_CLOSE_KEY):
            self.ParentForm._custom_titlebar_callback(self.Key)
        self._generic_callback_handler(self.DisplayText)
        return


    def _ReturnKeyHandler(self, event):
        """
        Internal callback for the ENTER / RETURN key. Results in calling the ButtonCallBack for element that has the return key bound to it, just as if button was clicked.

        :param event:
        :type event:

        """
        # if the element is disabled, ignore the event
        if self.Disabled:
            return

        MyForm = self.ParentForm
        button_element = self._FindReturnKeyBoundButton(MyForm)
        if button_element is not None:
            # if the Button has been disabled, then don't perform the callback
            if button_element.Disabled:
                return
            button_element.ButtonCallBack()

    def _generic_callback_handler(self, alternative_to_key=None, force_key_to_be=None):
        """
        Peforms the actions that were in many of the callback functions previously.  Combined so that it's
        easier to modify and is in 1 place now

        :param event:            From tkinter and is not used
        :type event:             Any
        :param alternate_to_key: If key is None, then use this value instead
        :type alternate_to_key:  Any
        """
        if force_key_to_be is not None:
            self.ParentForm.LastButtonClicked = force_key_to_be
        elif self.Key is not None:
            self.ParentForm.LastButtonClicked = self.Key
        else:
            self.ParentForm.LastButtonClicked = alternative_to_key
        self.ParentForm.FormRemainedOpen = True

        _exit_mainloop(self.ParentForm)
        # if self.ParentForm.CurrentlyRunningMainloop:
        #     Window._window_that_exited = self.ParentForm
        #     self.ParentForm.TKroot.quit()  # kick the users out of the mainloop

    def _ListboxSelectHandler(self, event):
        """
        Internal callback function for when a listbox item is selected

        :param event: Information from tkinter about the callback
        :type event:

        """
        self._generic_callback_handler('')

    def _ComboboxSelectHandler(self, event):
        """
        Internal callback function for when an entry is selected in a Combobox.
        :param event: Event data from tkinter (not used)
        :type event:

        """
        self._generic_callback_handler('')


    def _SpinboxSelectHandler(self, event=None):
        """
        Internal callback function for when an entry is selected in a Spinbox.
        Note that the parm is optional because it's not used if arrows are used to change the value
        but if the return key is pressed, it will include the event parm
        :param event: Event data passed in by tkinter (not used)
        :type event:
        """
        self._generic_callback_handler('')

    def _RadioHandler(self):
        """
        Internal callback for when a radio button is selected and enable events was set for radio
        """
        self._generic_callback_handler('')

    def _CheckboxHandler(self):
        """
        Internal callback for when a checkbnox is selected and enable events was set for checkbox
        """
        self._generic_callback_handler('')

    def _TabGroupSelectHandler(self, event):
        """
        Internal callback for when a Tab is selected and enable events was set for TabGroup

        :param event: Event data passed in by tkinter (not used)
        :type event:
        """
        self._generic_callback_handler('')

    def _KeyboardHandler(self, event):
        """
        Internal callback for when a key is pressed andd return keyboard events was set for window

        :param event: Event data passed in by tkinter (not used)
        :type event:
        """

        # if the element is disabled, ignore the event
        if self.Disabled:
            return
        self._generic_callback_handler('')

    def _ClickHandler(self, event):
        """
        Internal callback for when a mouse was clicked... I think.

        :param event: Event data passed in by tkinter (not used)
        :type event:
        """
        self._generic_callback_handler('')

    def _this_elements_window_closed(self, quick_check=True):
        if self.ParentForm is not None:
            return self.ParentForm.is_closed(quick_check=quick_check)

        return True

    def _user_bind_callback(self, bind_string, event, propagate=True):
        """
        Used when user binds a tkinter event directly to an element

        :param bind_string: The event that was bound so can lookup the key modifier
        :type bind_string:  (str)
        :param event:       Event data passed in by tkinter (not used)
        :type event:        (Any)
        :param propagate:   If True then tkinter will be told to propagate the event to the element
        :type propagate:    (bool)
        """
        key_suffix = self.user_bind_dict.get(bind_string, '')
        self.user_bind_event = event
        if self.Type == ELEM_TYPE_GRAPH:
            self._update_position_for_returned_values(event)
        if self.Key is not None:
            if isinstance(self.Key, str):
                key = self.Key + str(key_suffix)
            else:
                key = (self.Key, key_suffix)  # old way (pre 2021) was to make a brand new tuple
                # key = self.Key + (key_suffix,)   # in 2021 tried this. It will break existing applications though - if key is a tuple, add one more item
        else:
            key = bind_string

        self._generic_callback_handler(force_key_to_be=key)

        return 'break' if propagate is not True else None


    def bind(self, bind_string, key_modifier, propagate=True):
        """
        Used to add tkinter events to an Element.
        The tkinter specific data is in the Element's member variable user_bind_event
        :param bind_string:  The string tkinter expected in its bind function
        :type bind_string:   (str)
        :param key_modifier: Additional data to be added to the element's key when event is returned
        :type key_modifier:  (str)
        :param propagate:    If True then tkinter will be told to propagate the event to the element
        :type propagate:     (bool)
        """
        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        try:
            self.Widget.bind(bind_string, lambda evt: self._user_bind_callback(bind_string, evt, propagate))
        except Exception as e:
            self.Widget.unbind_all(bind_string)
            return

        self.user_bind_dict[bind_string] = key_modifier

    def unbind(self, bind_string):
        """
        Removes a previously bound tkinter event from an Element.
        :param bind_string: The string tkinter expected in its bind function
        :type bind_string:  (str)
        """
        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return
        self.Widget.unbind(bind_string)
        self.user_bind_dict.pop(bind_string, None)

    def set_tooltip(self, tooltip_text):
        """
        Called by application to change the tooltip text for an Element.  Normally invoked using the Element Object such as: window.Element('key').SetToolTip('New tip').

        :param tooltip_text: the text to show in tooltip.
        :type tooltip_text:  (str)
        """

        if self.TooltipObject:
            try:
                self.TooltipObject.leave()
            except:
                pass

        self.TooltipObject = ToolTip(self.Widget, text=tooltip_text, timeout=DEFAULT_TOOLTIP_TIME)

    def set_focus(self, force=False):
        """
        Sets the current focus to be on this element

        :param force: if True will call focus_force otherwise calls focus_set
        :type force:  bool
        """
        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return
        try:
            if force:
                self.Widget.focus_force()
            else:
                self.Widget.focus_set()
        except Exception as e:
            _error_popup_with_traceback("Exception blocking focus. Check your element's Widget", e)


    def block_focus(self, block=True):
        """
        Enable or disable the element from getting focus by using the keyboard.
        If the block parameter is True, then this element will not be given focus by using
        the keyboard to go from one element to another.
        You CAN click on the element and utilize it.

        :param block: if True the element will not get focus via the keyboard
        :type block:  bool
        """
        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return
        try:
            self.ParentForm.TKroot.focus_force()
            if block:
                self.Widget.configure(takefocus=0)
            else:
                self.Widget.configure(takefocus=1)
        except Exception as e:
            _error_popup_with_traceback("Exception blocking focus. Check your element's Widget", e)


    def get_next_focus(self):
        """
        Gets the next element that should get focus after this element.

        :return:    Element that will get focus after this one
        :rtype:     (Element)
        """
        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return None

        try:
            next_widget_focus = self.widget.tk_focusNext()
            return self.ParentForm.widget_to_element(next_widget_focus)
        except Exception as e:
            _error_popup_with_traceback("Exception getting next focus. Check your element's Widget", e)


    def get_previous_focus(self):
        """
        Gets the element that should get focus previous to this element.

        :return:    Element that should get the focus before this one
        :rtype:     (Element)
        """
        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return None
        try:
            next_widget_focus = self.widget.tk_focusPrev()      # tkinter.Widget
            return self.ParentForm.widget_to_element(next_widget_focus)
        except Exception as e:
            _error_popup_with_traceback("Exception getting previous focus. Check your element's Widget", e)


    def set_size(self, size=(None, None)):
        """
        Changes the size of an element to a specific size.
        It's possible to specify None for one of sizes so that only 1 of the element's dimensions are changed.

        :param size: The size in characters, rows typically. In some cases they are pixels
        :type size:  (int, int)
        """
        try:
            if size[0] != None:
                self.Widget.config(width=size[0])
        except:
            print('Warning, error setting width on element with key=', self.Key)
        try:
            if size[1] != None:
                self.Widget.config(height=size[1])
        except:
            try:
                self.Widget.config(length=size[1])
            except:
                print('Warning, error setting height on element with key=', self.Key)

        if self.Type == ELEM_TYPE_GRAPH:
            self.CanvasSize = size


    def get_size(self):
        """
        Return the size of an element in Pixels.  Care must be taken as some elements use characters to specify their size but will return pixels when calling this get_size method.
        :return: width and height of the element
        :rtype:  (int, int)
        """
        try:
            w = self.Widget.winfo_width()
            h = self.Widget.winfo_height()
        except:
            print('Warning, error getting size of element', self.Key)
            w = h = None
        return w, h

    def hide_row(self):
        """
        Hide the entire row an Element is located on.
        Use this if you must have all space removed when you are hiding an element, including the row container
        """
        try:
            self.ParentRowFrame.pack_forget()
        except:
            print('Warning, error hiding element row for key =', self.Key)

    def unhide_row(self):
        """
        Unhides (makes visible again) the row container that the Element is located on.
        Note that it will re-appear at the bottom of the window / container, most likely.
        """
        try:
            self.ParentRowFrame.pack()
        except:
            print('Warning, error hiding element row for key =', self.Key)

    def expand(self, expand_x=False, expand_y=False, expand_row=True):
        """
        Causes the Element to expand to fill available space in the X and Y directions.  Can specify which or both directions

        :param expand_x:   If True Element will expand in the Horizontal directions
        :type expand_x:    (bool)
        :param expand_y:   If True Element will expand in the Vertical directions
        :type expand_y:    (bool)
        :param expand_row: If True the row containing the element will also expand. Without this your element is "trapped" within the row
        :type expand_row:  (bool)
        """
        if expand_x and expand_y:
            fill = tk.BOTH
        elif expand_x:
            fill = tk.X
        elif expand_y:
            fill = tk.Y
        else:
            return

        if not self._widget_was_created():
            return
        self.Widget.pack(expand=True, fill=fill)
        self.ParentRowFrame.pack(expand=expand_row, fill=fill)
        if self.element_frame is not None:
            self.element_frame.pack(expand=True, fill=fill)

    def set_cursor(self, cursor=None, cursor_color=None):
        """
        Sets the cursor for the current Element.
        "Cursor" is used in 2 different ways in this call.
        For the parameter "cursor" it's actually the mouse pointer.
        If you do not want any mouse pointer, then use the string "none"
        For the parameter "cursor_color" it's the color of the beam used when typing into an input element

        :param cursor:       The tkinter cursor name
        :type cursor:        (str)
        :param cursor_color: color to set the "cursor" to
        :type cursor_color:  (str)
        """
        if not self._widget_was_created():
            return
        if cursor is not None:
            try:
                self.Widget.config(cursor=cursor)
            except Exception as e:
                print('Warning bad cursor specified ', cursor)
                print(e)
        if cursor_color is not None:
            try:
                self.Widget.config(insertbackground=cursor_color)
            except Exception as e:
                print('Warning bad cursor color', cursor_color)
                print(e)

    def set_vscroll_position(self, percent_from_top):
        """
        Attempts to set the vertical scroll postition for an element's Widget
        :param percent_from_top: From 0 to 1.0, the percentage from the top to move scrollbar to
        :type percent_from_top:  (float)
        """
        if self.Type == ELEM_TYPE_COLUMN and self.Scrollable:
            widget = self.widget.canvas     # scrollable column is a special case
        else:
            widget = self.widget

        try:
            widget.yview_moveto(percent_from_top)
        except Exception as e:
            print('Warning setting the vertical scroll (yview_moveto failed)')
            print(e)

    def _widget_was_created(self):
        """
        Determines if a Widget was created for this element.

        :return: True if a Widget has been created previously (Widget is not None)
        :rtype:  (bool)
        """
        if self.Widget is not None:
            return True
        else:
            if SUPPRESS_WIDGET_NOT_FINALIZED_WARNINGS:
                return False

            warnings.warn('You cannot Update element with key = {} until the window.read() is called or set finalize=True when creating window'.format(self.Key), UserWarning)
            if not SUPPRESS_ERROR_POPUPS:
                _error_popup_with_traceback('Unable to complete operation on element with key {}'.format(self.Key),
                                            'You cannot perform operations (such as calling update) on an Element until:',
                                            ' window.read() is called or finalize=True when Window created.',
                                            'Adding a "finalize=True" parameter to your Window creation will likely fix this.',
                                            _create_error_message(),
                                            )
            return False


    def _grab_anywhere_on_using_control_key(self):
        """
        Turns on Grab Anywhere functionality AFTER a window has been created.  Don't try on a window that's not yet
        been Finalized or Read.
        """
        self.Widget.bind('<Control-Button-1>', self.ParentForm._StartMove)
        self.Widget.bind('<Control-ButtonRelease-1>', self.ParentForm._StopMove)
        self.Widget.bind('<Control-B1-Motion>', self.ParentForm._OnMotion)


    def _grab_anywhere_on(self):
        """
        Turns on Grab Anywhere functionality AFTER a window has been created.  Don't try on a window that's not yet
        been Finalized or Read.
        """
        self.Widget.bind('<ButtonPress-1>', self.ParentForm._StartMove)
        self.Widget.bind('<ButtonRelease-1>', self.ParentForm._StopMove)
        self.Widget.bind('<B1-Motion>', self.ParentForm._OnMotion)

    def _grab_anywhere_off(self):
        """
        Turns off Grab Anywhere functionality AFTER a window has been created.  Don't try on a window that's not yet
        been Finalized or Read.
        """
        self.Widget.unbind('<ButtonPress-1>')
        self.Widget.unbind('<ButtonRelease-1>')
        self.Widget.unbind('<B1-Motion>')

    def grab_anywhere_exclude(self):
        """
        Excludes this element from being used by the grab_anywhere feature
        Handy for elements like a Graph element when dragging is enabled. You want the Graph element to get the drag events instead of the window dragging.
        """
        self.ParentForm._grab_anywhere_ignore_these_list.append(self.Widget)

    def grab_anywhere_include(self):
        """
        Includes this element in the grab_anywhere feature
        This will allow you to make a Multline element drag a window for example
        """
        self.ParentForm._grab_anywhere_include_these_list.append(self.Widget)



    def set_right_click_menu(self, menu=None):
        """
        Sets a right click menu for an element.
        If a menu is already set for the element, it will call the tkinter destroy method to remove it
        :param menu:                   A list of lists of Menu items to show when this element is right clicked. See user docs for exact format.
        :type menu:                    List[List[ List[str] | str ]]
        """
        if menu == MENU_RIGHT_CLICK_DISABLED:
            return
        if menu is None:
            menu = self.ParentForm.RightClickMenu
            if menu is None:
                return
        if menu:
            # If previously had a menu destroy it
            if self.TKRightClickMenu:
                try:
                    self.TKRightClickMenu.destroy()
                except:
                    pass
            top_menu = tk.Menu(self.ParentForm.TKroot, tearoff=self.ParentForm.right_click_menu_tearoff, tearoffcommand=self._tearoff_menu_callback)

            if self.ParentForm.right_click_menu_background_color not in (COLOR_SYSTEM_DEFAULT, None):
                top_menu.config(bg=self.ParentForm.right_click_menu_background_color)
            if self.ParentForm.right_click_menu_text_color not in (COLOR_SYSTEM_DEFAULT, None):
                top_menu.config(fg=self.ParentForm.right_click_menu_text_color)
            if self.ParentForm.right_click_menu_disabled_text_color not in (COLOR_SYSTEM_DEFAULT, None):
                top_menu.config(disabledforeground=self.ParentForm.right_click_menu_disabled_text_color)
            if self.ParentForm.right_click_menu_font is not None:
                top_menu.config(font=self.ParentForm.right_click_menu_font)

            if self.ParentForm.right_click_menu_selected_colors[0] not in (COLOR_SYSTEM_DEFAULT, None):
                top_menu.config(activeforeground=self.ParentForm.right_click_menu_selected_colors[0])
            if self.ParentForm.right_click_menu_selected_colors[1] not in (COLOR_SYSTEM_DEFAULT, None):
                top_menu.config(activebackground=self.ParentForm.right_click_menu_selected_colors[1])
            AddMenuItem(top_menu, menu[1], self, right_click_menu=True)
            self.TKRightClickMenu = top_menu
            if self.ParentForm.RightClickMenu:            # if the top level has a right click menu, then setup a callback for the Window itself
                if self.ParentForm.TKRightClickMenu is None:
                    self.ParentForm.TKRightClickMenu = top_menu
                    if (running_mac()):
                        self.ParentForm.TKroot.bind('<ButtonRelease-2>', self.ParentForm._RightClickMenuCallback)
                    else:
                        self.ParentForm.TKroot.bind('<ButtonRelease-3>', self.ParentForm._RightClickMenuCallback)
            if (running_mac()):
                self.Widget.bind('<ButtonRelease-2>', self._RightClickMenuCallback)
            else:
                self.Widget.bind('<ButtonRelease-3>', self._RightClickMenuCallback)


    def save_element_screenshot_to_disk(self, filename=None):
        """
        Saves an image of the PySimpleGUI window provided into the filename provided

        :param filename:        Optional filename to save screenshot to. If not included, the User Settinds are used to get the filename
        :return:                A PIL ImageGrab object that can be saved or manipulated
        :rtype:                 (PIL.ImageGrab | None)
        """
        global pil_import_attempted, pil_imported, PIL, ImageGrab, Image

        if not pil_import_attempted:
            try:
                import PIL as PIL
                from PIL import ImageGrab
                from PIL import Image
                pil_imported = True
                pil_import_attempted = True
            except:
                pil_imported = False
                pil_import_attempted = True
                print('FAILED TO IMPORT PIL!')
                return None
        try:
            # Add a little to the X direction if window has a titlebar
            rect = (self.widget.winfo_rootx(), self.widget.winfo_rooty(), self.widget.winfo_rootx() + self.widget.winfo_width(), self.widget.winfo_rooty() + self.widget.winfo_height())

            grab = ImageGrab.grab(bbox=rect)
            # Save the grabbed image to disk
        except Exception as e:
            # print(e)
            popup_error_with_traceback('Screen capture failure', 'Error happened while trying to save screencapture of an element', e)
            return None

        # return grab
        if filename is None:
            folder = pysimplegui_user_settings.get('-screenshots folder-', '')
            filename = pysimplegui_user_settings.get('-screenshots filename-', '')
            full_filename = os.path.join(folder, filename)
        else:
            full_filename = filename
        if full_filename:
            try:
                grab.save(full_filename)
            except Exception as e:
                popup_error_with_traceback('Screen capture failure', 'Error happened while trying to save screencapture', e)
        else:
            popup_error_with_traceback('Screen capture failure', 'You have attempted a screen capture but have not set up a good filename to save to')
        return grab




    def _pack_forget_save_settings(self, alternate_widget=None):
        """
        Performs a pack_forget which will make a widget invisible.
        This method saves the pack settings so that they can be restored if the element is made visible again

        :param alternate_widget:   Widget to use that's different than the one defined in Element.Widget. These are usually Frame widgets
        :type alternate_widget:    (tk.Widget)
        """

        if alternate_widget is not None and self.Widget is None:
            return

        widget = alternate_widget if alternate_widget is not None else self.Widget
        # if the widget is already invisible (i.e. not packed) then will get an error
        try:
            pack_settings = widget.pack_info()
            self.pack_settings = pack_settings
            widget.pack_forget()
        except:
            pass

    def _pack_restore_settings(self, alternate_widget=None):
        """
        Restores a previously packated widget which will make it visible again.
        If no settings were saved, then the widget is assumed to have not been unpacked and will not try to pack it again

        :param alternate_widget:   Widget to use that's different than the one defined in Element.Widget. These are usually Frame widgets
        :type alternate_widget:    (tk.Widget)
        """

        # if there are no saved pack settings, then assume it hasnb't been packaed before. The request will be ignored
        if self.pack_settings is None:
            return

        widget = alternate_widget if alternate_widget is not None else self.Widget
        if widget is not None:
            widget.pack(**self.pack_settings)


    def update(self, *args, **kwargs):
        """
        A dummy update call.  This will only be called if an element hasn't implemented an update method
        It is provided here for docstring purposes.  If you got here by browing code via PyCharm, know
        that this is not the function that will be called.  Your actual element's update method will be called.

        If you call update, you must call window.refresh if you want the change to happen prior to your next
        window.read() call. Normally uou don't do this as the window.read call is likely going to happen next.
        """
        print('* Base Element Class update was called. Your element does not seem to have an update method')


    def __call__(self, *args, **kwargs):
        """
        Makes it possible to "call" an already existing element.  When you do make the "call", it actually calls
        the Update method for the element.
        Example:    If this text element was in yoiur layout:
                    sg.Text('foo', key='T')
                    Then you can call the Update method for that element by writing:
                    window.find_element('T')('new text value')
        """
        return self.update(*args, **kwargs)






    SetTooltip = set_tooltip
    SetFocus = set_focus


# ---------------------------------------------------------------------- #
#                           Input Class                                  #
# ---------------------------------------------------------------------- #
class Input(Element):
    """
    Display a single text input field.  Based on the tkinter Widget `Entry`
    """

    def __init__(self, default_text='', size=(None, None), s=(None, None), disabled=False, password_char='',
                 justification=None, background_color=None, text_color=None, font=None, tooltip=None, border_width=None,
                 change_submits=False, enable_events=False, do_not_clear=True, key=None, k=None, focus=False, pad=None, p=None,
                 use_readonly_for_disable=True, readonly=False, disabled_readonly_background_color=None, disabled_readonly_text_color=None, selected_text_color=None, selected_background_color=None, expand_x=False, expand_y=False,
                 right_click_menu=None, visible=True, metadata=None):
        """
        :param default_text:                       Text initially shown in the input box as a default value(Default value = ''). Will automatically be converted to string
        :type default_text:                        (Any)
        :param size:                               w=characters-wide, h=rows-high. If an int is supplied rather than a tuple, then a tuple is created width=int supplied and heigh=1
        :type size:                                (int, int) |  (int, None) | int
        :param s:                                  Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
        :type s:                                   (int, int)  | (None, None) | int
        :param disabled:                           set disable state for element (Default = False)
        :type disabled:                            (bool)
        :param password_char:                      Password character if this is a password field (Default value = '')
        :type password_char:                       (char)
        :param justification:                      justification for data display. Valid choices - left, right, center
        :type justification:                       (str)
        :param background_color:                   color of background in one of the color formats
        :type background_color:                    (str)
        :param text_color:                         color of the text
        :type text_color:                          (str)
        :param font:                               specifies the font family, size. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:                                (str or (str, int[, str]) or None)
        :param tooltip:                            text, that will appear when mouse hovers over the element
        :type tooltip:                             (str)
        :param border_width:                       width of border around element in pixels
        :type border_width:                        (int)
        :param change_submits:                     * DEPRICATED DO NOT USE. Use `enable_events` instead
        :type change_submits:                      (bool)
        :param enable_events:                      If True then changes to this element are immediately reported as an event. Use this instead of change_submits (Default = False)
        :type enable_events:                       (bool)
        :param do_not_clear:                       If False then the field will be set to blank after ANY event (button, any event) (Default = True)
        :type do_not_clear:                        (bool)
        :param key:                                Value that uniquely identifies this element from all other elements. Used when Finding an element or in return values. Must be unique to the window
        :type key:                                 str | int | tuple | object
        :param k:                                  Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                                   str | int | tuple | object
        :param focus:                              Determines if initial focus should go to this element.
        :type focus:                               (bool)
        :param pad:                                Amount of padding to put around element. Normally (horizontal pixels, vertical pixels) but can be split apart further into ((horizontal left, horizontal right), (vertical above, vertical below)). If int is given, then converted to tuple (int, int) with the value provided duplicated
        :type pad:                                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                                  Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                                   (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param use_readonly_for_disable:           If True (the default) tkinter state set to 'readonly'. Otherwise state set to 'disabled'
        :type use_readonly_for_disable:            (bool)
        :param readonly:                           If True tkinter state set to 'readonly'.  Use this in place of use_readonly_for_disable as another way of achieving readonly.  Note cannot set BOTH readonly and disabled as tkinter only supplies a single flag
        :type readonly:                            (bool)
        :param disabled_readonly_background_color: If state is set to readonly or disabled, the color to use for the background
        :type disabled_readonly_background_color:  (str)
        :param disabled_readonly_text_color:       If state is set to readonly or disabled, the color to use for the text
        :type disabled_readonly_text_color:        (str)
        :param selected_text_color:                Color of text when it is selected (using mouse or control+A, etc)
        :type selected_text_color:                 (str)
        :param selected_background_color:          Color of background when it is selected (using mouse or control+A, etc)
        :type selected_background_color:           (str)
        :param expand_x:                           If True the element will automatically expand in the X direction to fill available space
        :type expand_x:                            (bool)
        :param expand_y:                           If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:                            (bool)
        :param right_click_menu:                   A list of lists of Menu items to show when this element is right clicked. See user docs for exact format.
        :type right_click_menu:                    List[List[ List[str] | str ]]
        :param visible:                            set visibility state of the element (Default = True)
        :type visible:                             (bool)
        :param metadata:                           User metadata that can be set to ANYTHING
        :type metadata:                            (Any)
        """


        self.DefaultText = default_text if default_text is not None else ''
        self.PasswordCharacter = password_char
        bg = background_color if background_color is not None else DEFAULT_INPUT_ELEMENTS_COLOR
        fg = text_color if text_color is not None else DEFAULT_INPUT_TEXT_COLOR
        self.selected_text_color = selected_text_color
        self.selected_background_color = selected_background_color
        self.Focus = focus
        self.do_not_clear = do_not_clear
        self.Justification = justification
        self.Disabled = disabled
        self.ChangeSubmits = change_submits or enable_events
        self.RightClickMenu = right_click_menu
        self.UseReadonlyForDisable = use_readonly_for_disable
        self.disabled_readonly_background_color = disabled_readonly_background_color
        self.disabled_readonly_text_color = disabled_readonly_text_color
        self.ReadOnly = readonly
        self.BorderWidth = border_width if border_width is not None else DEFAULT_BORDER_WIDTH
        self.TKEntry = self.Widget = None  # type: tk.Entry
        key = key if key is not None else k
        sz = size if size != (None, None) else s
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y

        super().__init__(ELEM_TYPE_INPUT_TEXT, size=sz, background_color=bg, text_color=fg, key=key, pad=pad,
                         font=font, tooltip=tooltip, visible=visible, metadata=metadata)

    def update(self, value=None, disabled=None, select=None, visible=None, text_color=None, background_color=None, font=None, move_cursor_to='end', password_char=None, paste=None, readonly=None):
        """
        Changes some of the settings for the Input Element. Must call `Window.Read` or `Window.Finalize` prior.
        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param value:            new text to display as default text in Input field
        :type value:             (str)
        :param disabled:         disable or enable state of the element (sets Entry Widget to readonly or normal)
        :type disabled:          (bool)
        :param select:           if True, then the text will be selected
        :type select:            (bool)
        :param visible:          change visibility of element
        :type visible:           (bool)
        :param text_color:       change color of text being typed
        :type text_color:        (str)
        :param background_color: change color of the background
        :type background_color:  (str)
        :param font:             specifies the font family, size. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:              (str or (str, int[, str]) or None)
        :param move_cursor_to:   Moves the cursor to a particular offset. Defaults to 'end'
        :type move_cursor_to:    int | str
        :param password_char:    Password character if this is a password field
        :type password_char:     str
        :param paste:            If True "Pastes" the value into the element rather than replacing the entire element. If anything is selected it is replaced. The text is inserted at the current cursor location.
        :type paste:             bool
        :param readonly:         if True make element readonly (user cannot change any choices). Enables the element if either choice are made.
        :type readonly:          (bool)
        """
        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in Input.update - The window was closed')
            return

        if background_color not in (None, COLOR_SYSTEM_DEFAULT):
            self.TKEntry.configure(background=background_color)
            self.BackgroundColor = background_color
        if text_color not in (None, COLOR_SYSTEM_DEFAULT):
            self.TKEntry.configure(fg=text_color)
            self.TextColor = text_color

        if disabled is True:
            if self.UseReadonlyForDisable:
                if self.disabled_readonly_text_color not in (None, COLOR_SYSTEM_DEFAULT):
                    self.TKEntry.configure(fg=self.disabled_readonly_text_color)
                self.TKEntry['state'] = 'readonly'
            else:
                if self.TextColor not in (None, COLOR_SYSTEM_DEFAULT):
                    self.TKEntry.configure(fg=self.TextColor)
                self.TKEntry['state'] = 'disabled'
            self.Disabled = True
        elif disabled is False:
            self.TKEntry['state'] = 'normal'
            if self.TextColor not in (None, COLOR_SYSTEM_DEFAULT):
                self.TKEntry.configure(fg=self.TextColor)
            self.Disabled = False

        if readonly is True:
            self.TKEntry['state'] = 'readonly'
        elif readonly is False:
            self.TKEntry['state'] = 'normal'




        if value is not None:
            if paste is not True:
                try:
                    self.TKStringVar.set(value)
                except:
                    pass
            self.DefaultText = value
            if paste is True:
                try:
                    self.TKEntry.delete('sel.first', 'sel.last')
                except:
                    pass
                self.TKEntry.insert('insert', value)
            if move_cursor_to == 'end':
                self.TKEntry.icursor(tk.END)
            elif move_cursor_to is not None:
                self.TKEntry.icursor(move_cursor_to)
        if select:
            self.TKEntry.select_range(0, 'end')
        if visible is False:
            self._pack_forget_save_settings()
            # self.TKEntry.pack_forget()
        elif visible is True:
            self._pack_restore_settings()
            # self.TKEntry.pack(padx=self.pad_used[0], pady=self.pad_used[1])
            # self.TKEntry.pack(padx=self.pad_used[0], pady=self.pad_used[1], in_=self.ParentRowFrame)
        if visible is not None:
            self._visible = visible
        if password_char is not None:
            self.TKEntry.configure(show=password_char)
            self.PasswordCharacter = password_char
        if font is not None:
            self.TKEntry.configure(font=font)



    def set_ibeam_color(self, ibeam_color=None):
        """
        Sets the color of the I-Beam that is used to "insert" characters. This is oftens called a "Cursor" by
        many users.  To keep from being confused with tkinter's definition of cursor (the mouse pointer), the term
        ibeam is used in this case.
        :param ibeam_color: color to set the "I-Beam" used to indicate where characters will be inserted
        :type ibeam_color:  (str)
        """

        if not self._widget_was_created():
            return
        if ibeam_color is not None:
            try:
                self.Widget.config(insertbackground=ibeam_color)
            except Exception as e:
                _error_popup_with_traceback('Error setting I-Beam color in set_ibeam_color',
                           'The element has a key:', self.Key,
                            'The color passed in was:', ibeam_color)




    def get(self):
        """
        Read and return the current value of the input element. Must call `Window.Read` or `Window.Finalize` prior

        :return: current value of Input field or '' if error encountered
        :rtype:  (str)
        """
        try:
            text = self.TKStringVar.get()
        except:
            text = ''
        return text

    Get = get
    Update = update


# -------------------------  INPUT TEXT Element lazy functions  ------------------------- #
In = Input
InputText = Input
I = Input


# ---------------------------------------------------------------------- #
#                           Combo                                        #
# ---------------------------------------------------------------------- #
class Combo(Element):
    """
    ComboBox Element - A combination of a single-line input and a drop-down menu. User can type in their own value or choose from list.
    """

    def __init__(self, values, default_value=None, size=(None, None), s=(None, None), auto_size_text=None, background_color=None, text_color=None, button_background_color=None, button_arrow_color=None, bind_return_key=False, change_submits=False, enable_events=False, enable_per_char_events=None, disabled=False, key=None, k=None, pad=None, p=None, expand_x=False, expand_y=False, tooltip=None, readonly=False, font=None, visible=True, metadata=None):
        """
        :param values:                  values to choose. While displayed as text, the items returned are what the caller supplied, not text
        :type values:                   List[Any] or Tuple[Any]
        :param default_value:           Choice to be displayed as initial value. Must match one of values variable contents
        :type default_value:            (Any)
        :param size:                    width, height. Width = characters-wide, height = NOTE it's the number of entries to show in the list. If an Int is passed rather than a tuple, then height is auto-set to 1 and width is value of the int
        :type size:                     (int, int)  | (None, None) | int
        :param s:                       Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
        :type s:                        (int, int)  | (None, None) | int
        :param auto_size_text:          True if element should be the same size as the contents
        :type auto_size_text:           (bool)
        :param background_color:        color of background
        :type background_color:         (str)
        :param text_color:              color of the text
        :type text_color:               (str)
        :param button_background_color: The color of the background of the button on the combo box
        :type button_background_color:  (str)
        :param button_arrow_color:      The color of the arrow on the button on the combo box
        :type button_arrow_color:       (str)
        :param bind_return_key:         If True, then the return key will cause a the Combo to generate an event when return key is pressed
        :type bind_return_key:          (bool)
        :param change_submits:          DEPRICATED DO NOT USE. Use `enable_events` instead
        :type change_submits:           (bool)
        :param enable_events:           Turns on the element specific events. Combo event is when a choice is made
        :type enable_events:            (bool)
        :param enable_per_char_events:  Enables generation of events for every character that's input. This is like the Input element's events
        :type enable_per_char_events:   (bool)
        :param disabled:                set disable state for element
        :type disabled:                 (bool)
        :param key:                     Used with window.find_element and with return values to uniquely identify this element
        :type key:                      str | int | tuple | object
        :param k:                       Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                        str | int | tuple | object
        :param pad:                     Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:                      (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                       Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                        (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param expand_x:                If True the element will automatically expand in the X direction to fill available space
        :type expand_x:                 (bool)
        :param expand_y:                If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:                 (bool)
        :param tooltip:                 text that will appear when mouse hovers over this element
        :type tooltip:                  (str)
        :param readonly:                make element readonly (user can't change). True means user cannot change
        :type readonly:                 (bool)
        :param font:                    specifies the font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:                     (str or (str, int[, str]) or None)
        :param visible:                 set visibility state of the element
        :type visible:                  (bool)
        :param metadata:                User metadata that can be set to ANYTHING
        :type metadata:                 (Any)
        """


        self.Values = values
        self.DefaultValue = default_value
        self.ChangeSubmits = change_submits or enable_events
        self.Widget = self.TKCombo = None  # type: ttk.Combobox
        self.Disabled = disabled
        self.Readonly = readonly
        self.BindReturnKey = bind_return_key
        bg = background_color if background_color else DEFAULT_INPUT_ELEMENTS_COLOR
        fg = text_color if text_color is not None else DEFAULT_INPUT_TEXT_COLOR
        key = key if key is not None else k
        sz = size if size != (None, None) else s
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y
        if button_background_color is None:
            self.button_background_color = theme_button_color()[1]
        else:
            self.button_background_color = button_background_color
        if button_arrow_color is None:
            self.button_arrow_color = theme_button_color()[0]
        else:
            self.button_arrow_color = button_arrow_color
        self.enable_per_char_events = enable_per_char_events

        super().__init__(ELEM_TYPE_INPUT_COMBO, size=sz, auto_size_text=auto_size_text, background_color=bg,
                         text_color=fg, key=key, pad=pad, tooltip=tooltip, font=font or DEFAULT_FONT, visible=visible, metadata=metadata)

    def update(self, value=None, values=None, set_to_index=None, disabled=None, readonly=None, font=None, visible=None, size=(None, None), select=None, text_color=None, background_color=None):
        """
        Changes some of the settings for the Combo Element. Must call `Window.Read` or `Window.Finalize` prior.
        Note that the state can be in 3 states only.... enabled, disabled, readonly even
        though more combinations are available. The easy way to remember is that if you
        change the readonly parameter then you are enabling the element.

        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param value:            change which value is current selected based on new list of previous list of choices
        :type value:             (Any)
        :param values:           change list of choices
        :type values:            List[Any]
        :param set_to_index:     change selection to a particular choice starting with index = 0
        :type set_to_index:      (int)
        :param disabled:         disable or enable state of the element
        :type disabled:          (bool)
        :param readonly:         if True make element readonly (user cannot change any choices). Enables the element if either choice are made.
        :type readonly:          (bool)
        :param font:             specifies the font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:              (str or (str, int[, str]) or None)
        :param visible:          control visibility of element
        :type visible:           (bool)
        :param size:             width, height. Width = characters-wide, height = NOTE it's the number of entries to show in the list
        :type size:              (int, int)
        :param select:           if True, then the text will be selected, if False then selection will be cleared
        :type select:            (bool)
        :param background_color: color of background
        :type background_color:  (str)
        :param text_color:       color of the text
        :type text_color:        (str)
        """

        if size != (None, None):
            if isinstance(size, int):
                size = (size, 1)
            if isinstance(size, tuple) and len(size) == 1:
                size = (size[0],  1)

        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in Combo.update - The window was closed')
            return


        if values is not None:
            try:
                self.TKCombo['values'] = values
                # self.TKCombo.current(0)       # don't set any value if a new set of values was made
            except:
                pass
            self.Values = values
            if value is None:
                self.TKCombo.set('')
            if size == (None, None):
                max_line_len = max([len(str(l)) for l in self.Values]) if len(self.Values) else 0
                if self.AutoSizeText is False:
                    width = self.Size[0]
                else:
                    width = max_line_len + 1
                self.TKCombo.configure(width=width)
            else:
                self.TKCombo.configure(height=size[1])
                self.TKCombo.configure(width=size[0])
        if value is not None:
            if value not in self.Values:
                self.TKCombo.set(value)
            else:
                for index, v in enumerate(self.Values):
                    if v == value:
                        try:
                            self.TKCombo.current(index)
                        except:
                            pass
                        self.DefaultValue = value
                        break
        if set_to_index is not None:
            try:
                self.TKCombo.current(set_to_index)
                self.DefaultValue = self.Values[set_to_index]
            except:
                pass
        if readonly:
            self.Readonly = True
            self.TKCombo['state'] = 'readonly'
        elif readonly is False:
            self.Readonly = False
            self.TKCombo['state'] = 'enable'
        if disabled is True:
            self.TKCombo['state'] = 'disable'
        elif disabled is False and self.Readonly is True:
                self.TKCombo['state'] = 'readonly'
        elif disabled is False and self.Readonly is False:
                self.TKCombo['state'] = 'enable'
        self.Disabled = disabled if disabled is not None else self.Disabled

        combostyle = self.ttk_style
        style_name = self.ttk_style_name
        if text_color is not None:
            combostyle.configure(style_name, foreground=text_color)
            combostyle.configure(style_name, selectforeground=text_color)
            combostyle.configure(style_name, insertcolor=text_color)
            combostyle.map(style_name, fieldforeground=[('readonly', text_color)])
            self.TextColor = text_color
        if background_color is not None:
            combostyle.configure(style_name, selectbackground=background_color)
            combostyle.map(style_name, fieldbackground=[('readonly', background_color)])
            combostyle.configure(style_name, fieldbackground=background_color)
            self.BackgroundColor = background_color

        if self.Readonly is True:
            if text_color not in (None, COLOR_SYSTEM_DEFAULT):
                combostyle.configure(style_name, selectforeground=text_color)
            if background_color not in (None, COLOR_SYSTEM_DEFAULT):
                combostyle.configure(style_name, selectbackground=background_color)


        if font is not None:
            self.Font = font
            self.TKCombo.configure(font=font)
            self._dropdown_newfont = tkinter.font.Font(font=font)
            self.ParentRowFrame.option_add('*TCombobox*Listbox*Font', self._dropdown_newfont)


        # make tcl call to deal with colors for the drop-down formatting
        try:
            if self.BackgroundColor not in (None, COLOR_SYSTEM_DEFAULT) and \
                self.TextColor not in (None, COLOR_SYSTEM_DEFAULT):
                self.Widget.tk.eval(
            '[ttk::combobox::PopdownWindow {}].f.l configure -foreground {} -background {} -selectforeground {} -selectbackground {} -font {}'.format(self.Widget, self.TextColor, self.BackgroundColor, self.BackgroundColor, self.TextColor, self._dropdown_newfont))
        except Exception as e:
            pass    # going to let this one slide

        if visible is False:
            self._pack_forget_save_settings()
            # self.TKCombo.pack_forget()
        elif visible is True:
            self._pack_restore_settings()
            # self.TKCombo.pack(padx=self.pad_used[0], pady=self.pad_used[1])
        if visible is not None:
            self._visible = visible
        if select is True:
           self.TKCombo.select_range(0, tk.END)
        elif select is False:
           self.TKCombo.select_clear()


    def get(self):
        """
        Returns the current (right now) value of the Combo.  DO NOT USE THIS AS THE NORMAL WAY OF READING A COMBO!
        You should be using values from your call to window.read instead.  Know what you're doing if you use it.

        :return: Returns the value of what is currently chosen
        :rtype:  Any | None
        """
        try:
            if self.TKCombo.current() == -1:  # if the current value was not in the original list
                value = self.TKCombo.get()  # then get the value typed in by user
            else:
                value = self.Values[self.TKCombo.current()]  # get value from original list given index
        except:
            value = None  # only would happen if user closes window
        return value

    Get = get
    Update = update


# -------------------------  INPUT COMBO Element lazy functions  ------------------------- #
InputCombo = Combo
DropDown = InputCombo
Drop = InputCombo
DD = Combo


# ---------------------------------------------------------------------- #
#                           Option Menu                                  #
# ---------------------------------------------------------------------- #
class OptionMenu(Element):
    """
    Option Menu is an Element available ONLY on the tkinter port of PySimpleGUI.  It's is a widget that is unique
    to tkinter.  However, it looks much like a ComboBox.  Instead of an arrow to click to pull down the list of
    choices, another little graphic is shown on the widget to indicate where you click.  After clicking to activate,
    it looks like a Combo Box that you scroll to select a choice.
    """

    def __init__(self, values, default_value=None, size=(None, None), s=(None, None), disabled=False, auto_size_text=None, expand_x=False, expand_y=False,
                 background_color=None, text_color=None, key=None, k=None, pad=None, p=None, tooltip=None, visible=True, metadata=None):
        """
        :param values:           Values to be displayed
        :type values:            List[Any] or Tuple[Any]
        :param default_value:    the value to choose by default
        :type default_value:     (Any)
        :param size:             (width, height) size in characters (wide), height is ignored and present to be consistent with other elements
        :type size:              (int, int) (width, UNUSED)
        :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
        :type s:                 (int, int)  | (None, None) | int
        :param disabled:         control enabled / disabled
        :type disabled:          (bool)
        :param auto_size_text:   True if size of Element should match the contents of the items
        :type auto_size_text:    (bool)
        :param expand_x:         If True the element will automatically expand in the X direction to fill available space
        :type expand_x:          (bool)
        :param expand_y:         If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:          (bool)
        :param background_color: color of background
        :type background_color:  (str)
        :param text_color:       color of the text
        :type text_color:        (str)
        :param key:              Used with window.find_element and with return values to uniquely identify this element
        :type key:               str | int | tuple | object
        :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                 str | int | tuple | object
        :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param tooltip:          text that will appear when mouse hovers over this element
        :type tooltip:           (str)
        :param visible:          set visibility state of the element
        :type visible:           (bool)
        :param metadata:         User metadata that can be set to ANYTHING
        :type metadata:          (Any)
        """

        self.Values = values
        self.DefaultValue = default_value
        self.Widget = self.TKOptionMenu = None  # type: tk.OptionMenu
        self.Disabled = disabled
        bg = background_color if background_color else DEFAULT_INPUT_ELEMENTS_COLOR
        fg = text_color if text_color is not None else DEFAULT_INPUT_TEXT_COLOR
        key = key if key is not None else k
        sz = size if size != (None, None) else s
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y


        super().__init__(ELEM_TYPE_INPUT_OPTION_MENU, size=sz, auto_size_text=auto_size_text, background_color=bg,
                         text_color=fg, key=key, pad=pad, tooltip=tooltip, visible=visible, metadata=metadata)

    def update(self, value=None, values=None, disabled=None, visible=None, size=(None, None)):
        """
        Changes some of the settings for the OptionMenu Element. Must call `Window.Read` or `Window.Finalize` prior

        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param value:    the value to choose by default
        :type value:     (Any)
        :param values:   Values to be displayed
        :type values:    List[Any]
        :param disabled: disable or enable state of the element
        :type disabled:  (bool)
        :param visible:  control visibility of element
        :type visible:   (bool)
        :param size:     (width, height) size in characters (wide), height is ignored and present to be consistent with other elements
        :type size:      (int, int) (width, UNUSED)
        """
        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in OptionMenu.update - The window was closed')
            return


        if values is not None:
            self.Values = values
            self.TKOptionMenu['menu'].delete(0, 'end')

            # Insert list of new options (tk._setit hooks them up to var)
            # self.TKStringVar.set(self.Values[0])
            for new_value in self.Values:
                self.TKOptionMenu['menu'].add_command(label=new_value, command=tk._setit(self.TKStringVar, new_value))
            if value is None:
                self.TKStringVar.set('')

            if size == (None, None):
                max_line_len = max([len(str(l)) for l in self.Values]) if len(self.Values) else 0
                if self.AutoSizeText is False:
                    width = self.Size[0]
                else:
                    width = max_line_len + 1
                self.TKOptionMenu.configure(width=width)
            else:
                self.TKOptionMenu.configure(width=size[0])

        if value is not None:
            self.DefaultValue = value
            self.TKStringVar.set(value)

        if disabled is True:
            self.TKOptionMenu['state'] = 'disabled'
        elif disabled is False:
            self.TKOptionMenu['state'] = 'normal'
        self.Disabled = disabled if disabled is not None else self.Disabled
        if visible is False:
            self._pack_forget_save_settings()
            # self.TKOptionMenu.pack_forget()
        elif visible is True:
            self._pack_restore_settings()
            # self.TKOptionMenu.pack(padx=self.pad_used[0], pady=self.pad_used[1])
        if visible is not None:
            self._visible = visible

    Update = update


# -------------------------  OPTION MENU Element lazy functions  ------------------------- #
InputOptionMenu = OptionMenu


# ---------------------------------------------------------------------- #
#                           Listbox                                      #
# ---------------------------------------------------------------------- #
class Listbox(Element):
    """
    A List Box.  Provide a list of values for the user to choose one or more of.   Returns a list of selected rows
    when a window.read() is executed.
    """

    def __init__(self, values, default_values=None, select_mode=None, change_submits=False, enable_events=False,
                 bind_return_key=False, size=(None, None), s=(None, None), disabled=False, justification=None, auto_size_text=None, font=None, no_scrollbar=False, horizontal_scroll=False,
                 background_color=None, text_color=None, highlight_background_color=None, highlight_text_color=None,
                 sbar_trough_color=None, sbar_background_color=None, sbar_arrow_color=None, sbar_width=None, sbar_arrow_width=None, sbar_frame_color=None, sbar_relief=None,
                 key=None, k=None, pad=None, p=None, tooltip=None, expand_x=False, expand_y=False,right_click_menu=None, visible=True, metadata=None):
        """
        :param values:                     list of values to display. Can be any type including mixed types as long as they have __str__ method
        :type values:                      List[Any] or Tuple[Any]
        :param default_values:             which values should be initially selected
        :type default_values:              List[Any]
        :param select_mode:                Select modes are used to determine if only 1 item can be selected or multiple and how they can be selected.   Valid choices begin with "LISTBOX_SELECT_MODE_" and include: LISTBOX_SELECT_MODE_SINGLE LISTBOX_SELECT_MODE_MULTIPLE LISTBOX_SELECT_MODE_BROWSE LISTBOX_SELECT_MODE_EXTENDED
        :type select_mode:                 [enum]
        :param change_submits:             DO NOT USE. Only listed for backwards compat - Use enable_events instead
        :type change_submits:              (bool)
        :param enable_events:              Turns on the element specific events. Listbox generates events when an item is clicked
        :type enable_events:               (bool)
        :param bind_return_key:            If True, then the return key will cause a the Listbox to generate an event when return key is pressed
        :type bind_return_key:             (bool)
        :param size:                       w=characters-wide, h=rows-high. If an int instead of a tuple is supplied, then height is auto-set to 1
        :type size:                        (int, int) |  (int, None) | int
        :param s:                          Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
        :type s:                           (int, int)  | (None, None) | int
        :param disabled:                   set disable state for element
        :type disabled:                    (bool)
        :param justification:              justification for items in listbox. Valid choices - left, right, center.  Default is left. NOTE - on some older versions of tkinter, not available
        :type justification:               (str)
        :param auto_size_text:             True if element should be the same size as the contents
        :type auto_size_text:              (bool)
        :param font:                       specifies the font family, size, etc.  Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:                        (str or (str, int[, str]) or None)
        :param no_scrollbar:               Controls if a scrollbar should be shown.  If True, no scrollbar will be shown
        :type no_scrollbar:                (bool)
        :param horizontal_scroll:          Controls if a horizontal scrollbar should be shown.  If True a horizontal scrollbar will be shown in addition to vertical
        :type horizontal_scroll:           (bool)
        :param background_color:           color of background
        :type background_color:            (str)
        :param text_color:                 color of the text
        :type text_color:                  (str)
        :param highlight_background_color: color of the background when an item is selected. Defaults to normal text color (a reverse look)
        :type highlight_background_color:  (str)
        :param highlight_text_color:       color of the text when an item is selected. Defaults to the normal background color (a rerverse look)
        :type highlight_text_color:        (str)
        :param sbar_trough_color:           Scrollbar color of the trough
        :type sbar_trough_color:            (str)
        :param sbar_background_color:       Scrollbar color of the background of the arrow buttons at the ends AND the color of the "thumb" (the thing you grab and slide). Switches to arrow color when mouse is over
        :type sbar_background_color:        (str)
        :param sbar_arrow_color:            Scrollbar color of the arrow at the ends of the scrollbar (it looks like a button). Switches to background color when mouse is over
        :type sbar_arrow_color:             (str)
        :param sbar_width:                  Scrollbar width in pixels
        :type sbar_width:                   (int)
        :param sbar_arrow_width:            Scrollbar width of the arrow on the scrollbar. It will potentially impact the overall width of the scrollbar
        :type sbar_arrow_width:             (int)
        :param sbar_frame_color:            Scrollbar Color of frame around scrollbar (available only on some ttk themes)
        :type sbar_frame_color:             (str)
        :param sbar_relief:                 Scrollbar relief that will be used for the "thumb" of the scrollbar (the thing you grab that slides). Should be a constant that is defined at starting with "RELIEF_" - RELIEF_RAISED, RELIEF_SUNKEN, RELIEF_FLAT, RELIEF_RIDGE, RELIEF_GROOVE, RELIEF_SOLID
        :type sbar_relief:                  (str)
        :param key:                        Used with window.find_element and with return values to uniquely identify this element
        :type key:                         str | int | tuple | object
        :param k:                          Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                           str | int | tuple | object
        :param pad:                        Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:                         (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                          Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                           (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param tooltip:                    text, that will appear when mouse hovers over the element
        :type tooltip:                     (str)
        :param expand_x:                   If True the element will automatically expand in the X direction to fill available space
        :type expand_x:                    (bool)
        :param expand_y:                   If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:                    (bool)
        :param right_click_menu:           A list of lists of Menu items to show when this element is right clicked. See user docs for exact format.
        :type right_click_menu:            List[List[ List[str] | str ]]
        :param visible:                    set visibility state of the element
        :type visible:                     (bool)
        :param metadata:                   User metadata that can be set to ANYTHING
        :type metadata:                    (Any)
        """

        if values is None:
            _error_popup_with_traceback('Error in your Listbox definition - The values parameter cannot be None', 'Use an empty list if you want no values in your Listbox')

        self.Values = values
        self.DefaultValues = default_values
        self.TKListbox = None
        self.ChangeSubmits = change_submits or enable_events
        self.BindReturnKey = bind_return_key
        self.Disabled = disabled
        if select_mode == LISTBOX_SELECT_MODE_BROWSE:
            self.SelectMode = SELECT_MODE_BROWSE
        elif select_mode == LISTBOX_SELECT_MODE_EXTENDED:
            self.SelectMode = SELECT_MODE_EXTENDED
        elif select_mode == LISTBOX_SELECT_MODE_MULTIPLE:
            self.SelectMode = SELECT_MODE_MULTIPLE
        elif select_mode == LISTBOX_SELECT_MODE_SINGLE:
            self.SelectMode = SELECT_MODE_SINGLE
        else:
            self.SelectMode = DEFAULT_LISTBOX_SELECT_MODE
        bg = background_color if background_color is not None else theme_input_background_color()
        fg = text_color if text_color is not None else theme_input_text_color()
        self.HighlightBackgroundColor = highlight_background_color if highlight_background_color is not None else fg
        self.HighlightTextColor = highlight_text_color if highlight_text_color is not None else bg
        self.RightClickMenu = right_click_menu
        self.vsb = None  # type: tk.Scrollbar or None
        self.hsb = None  # type: tk.Scrollbar | None
        self.TKListbox = self.Widget = None  # type: tk.Listbox
        self.element_frame = None  # type: tk.Frame
        self.NoScrollbar = no_scrollbar
        self.HorizontalScroll = horizontal_scroll
        key = key if key is not None else k
        sz = size if size != (None, None) else s
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y
        self.justification = justification

        super().__init__(ELEM_TYPE_INPUT_LISTBOX, size=sz, auto_size_text=auto_size_text, font=font,
                         background_color=bg, text_color=fg, key=key, pad=pad, tooltip=tooltip, visible=visible, metadata=metadata,
                         sbar_trough_color=sbar_trough_color, sbar_background_color=sbar_background_color, sbar_arrow_color=sbar_arrow_color, sbar_width=sbar_width, sbar_arrow_width=sbar_arrow_width, sbar_frame_color=sbar_frame_color, sbar_relief=sbar_relief)

    def update(self, values=None, disabled=None, set_to_index=None, scroll_to_index=None, select_mode=None, visible=None):
        """
        Changes some of the settings for the Listbox Element. Must call `Window.Read` or `Window.Finalize` prior
        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param values:          new list of choices to be shown to user
        :type values:           List[Any]
        :param disabled:        disable or enable state of the element
        :type disabled:         (bool)
        :param set_to_index:    highlights the item(s) indicated. If parm is an int one entry will be set. If is a list, then each entry in list is highlighted
        :type set_to_index:     int | list | tuple
        :param scroll_to_index: scroll the listbox so that this index is the first shown
        :type scroll_to_index:  (int)
        :param select_mode:     changes the select mode according to tkinter's listbox widget
        :type select_mode:      (str)
        :param visible:         control visibility of element
        :type visible:          (bool)
        """

        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in Listbox.update - The window was closed')
            return

        if disabled is True:
            self.TKListbox.configure(state='disabled')
        elif disabled is False:
            self.TKListbox.configure(state='normal')
        self.Disabled = disabled if disabled is not None else self.Disabled

        if values is not None:
            self.TKListbox.delete(0, 'end')
            for item in list(values):
                self.TKListbox.insert(tk.END, item)
            # self.TKListbox.selection_set(0, 0)
            self.Values = list(values)
        if set_to_index is not None:
            self.TKListbox.selection_clear(0, len(self.Values))  # clear all listbox selections
            if type(set_to_index) in (tuple, list):
                for i in set_to_index:
                    try:
                        self.TKListbox.selection_set(i, i)
                    except:
                        warnings.warn('* Listbox Update selection_set failed with index {}*'.format(set_to_index))
            else:
                try:
                    self.TKListbox.selection_set(set_to_index, set_to_index)
                except:
                    warnings.warn('* Listbox Update selection_set failed with index {}*'.format(set_to_index))
        if visible is False:
            self._pack_forget_save_settings(self.element_frame)
        elif visible is True:
            self._pack_restore_settings(self.element_frame)
        if scroll_to_index is not None and len(self.Values):
            self.TKListbox.yview_moveto(scroll_to_index / len(self.Values))
        if select_mode is not None:
            try:
                self.TKListbox.config(selectmode=select_mode)
            except:
                print('Listbox.update error trying to change mode to: ', select_mode)
        if visible is not None:
            self._visible = visible

    def set_value(self, values):
        """
        Set listbox highlighted choices

        :param values: new values to choose based on previously set values
        :type values:  List[Any] | Tuple[Any]

        """
        for index, item in enumerate(self.Values):
            try:
                if item in values:
                    self.TKListbox.selection_set(index)
                else:
                    self.TKListbox.selection_clear(index)
            except:
                pass
        self.DefaultValues = values

    def get_list_values(self):
        # type: (Listbox) -> List[Any]
        """
        Returns list of Values provided by the user in the user's format

        :return: List of values. Can be any / mixed types -> []
        :rtype:  List[Any]
        """
        return self.Values

    def get_indexes(self):
        """
        Returns the items currently selected as a list of indexes

        :return: A list of offsets into values that is currently selected
        :rtype:  List[int]
        """
        return self.TKListbox.curselection()

    def get(self):
        """
        Returns the list of items currently selected in this listbox.  It should be identical
        to the value you would receive when performing a window.read() call.

        :return: The list of currently selected items. The actual items are returned, not the indexes
        :rtype:  List[Any]
        """
        try:
            items = self.TKListbox.curselection()
            value = [self.Values[int(item)] for item in items]
        except:
            value = []
        return value


    def select_index(self, index, highlight_text_color=None, highlight_background_color=None):
        """
        Selects an index while providing capability to setting the selected color for the index to specific text/background color

        :param index:                      specifies which item to change. index starts at 0 and goes to length of values list minus one
        :type  index:                      (int)
        :param highlight_text_color:       color of the text when this item is selected.
        :type  highlight_text_color:        (str)
        :param highlight_background_color: color of the background when this item is selected
        :type  highlight_background_color:  (str)
        """

        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in Listbox.select_item - The window was closed')
            return

        if index >= len(self.Values):
            _error_popup_with_traceback('Index {} is out of range for Listbox.select_index. Max allowed index is {}.'.format(index, len(self.Values)-1))
            return

        self.TKListbox.selection_set(index, index)

        if highlight_text_color is not None:
            self.widget.itemconfig(index, selectforeground=highlight_text_color)
        if highlight_background_color is not None:
            self.widget.itemconfig(index, selectbackground=highlight_background_color)


    def set_index_color(self, index, text_color=None, background_color=None, highlight_text_color=None, highlight_background_color=None):
        """
        Sets the color of a specific item without selecting it

        :param index:                      specifies which item to change. index starts at 0 and goes to length of values list minus one
        :type  index:                      (int)
        :param text_color:                 color of the text for this item
        :type  text_color:                 (str)
        :param background_color:           color of the background for this item
        :type  background_color:           (str)
        :param highlight_text_color:       color of the text when this item is selected.
        :type  highlight_text_color:       (str)
        :param highlight_background_color: color of the background when this item is selected
        :type  highlight_background_color: (str)
        """

        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in Listbox.set_item_color - The window was closed')
            return

        if index >= len(self.Values):
            _error_popup_with_traceback('Index {} is out of range for Listbox.set_index_color. Max allowed index is {}.'.format(index, len(self.Values)-1))
            return

        if text_color is not None:
            self.widget.itemconfig(index, fg=text_color)
        if background_color is not None:
            self.widget.itemconfig(index, bg=background_color)
        if highlight_text_color is not None:
            self.widget.itemconfig(index, selectforeground=highlight_text_color)
        if highlight_background_color is not None:
            self.widget.itemconfig(index, selectbackground=highlight_background_color)




    GetIndexes = get_indexes
    GetListValues = get_list_values
    SetValue = set_value
    Update = update


LBox = Listbox
LB = Listbox


# ---------------------------------------------------------------------- #
#                           Radio                                        #
# ---------------------------------------------------------------------- #
class Radio(Element):
    """
    Radio Button Element - Used in a group of other Radio Elements to provide user with ability to select only
    1 choice in a list of choices.
    """

    def __init__(self, text, group_id, default=False, disabled=False, size=(None, None), s=(None, None), auto_size_text=None,
                 background_color=None, text_color=None, circle_color=None, font=None, key=None, k=None, pad=None, p=None, tooltip=None,
                 change_submits=False, enable_events=False, right_click_menu=None, expand_x=False, expand_y=False, visible=True, metadata=None):
        """
        :param text:             Text to display next to button
        :type text:              (str)
        :param group_id:         Groups together multiple Radio Buttons. Any type works
        :type group_id:          (Any)
        :param default:          Set to True for the one element of the group you want initially selected
        :type default:           (bool)
        :param disabled:         set disable state
        :type disabled:          (bool)
        :param size:             (w, h) w=characters-wide, h=rows-high. If an int instead of a tuple is supplied, then height is auto-set to 1
        :type size:              (int, int)  | (None, None) | int
        :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
        :type s:                 (int, int)  | (None, None) | int
        :param auto_size_text:   if True will size the element to match the length of the text
        :type auto_size_text:    (bool)
        :param background_color: color of background
        :type background_color:  (str)
        :param text_color:       color of the text
        :type text_color:        (str)
        :param circle_color:     color of background of the circle that has the dot selection indicator in it
        :type circle_color:      (str)
        :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:              (str or (str, int[, str]) or None)
        :param key:              Used with window.find_element and with return values to uniquely identify this element
        :type key:               str | int | tuple | object
        :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                 str | int | tuple | object
        :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param tooltip:          text, that will appear when mouse hovers over the element
        :type tooltip:           (str)
        :param change_submits:   DO NOT USE. Only listed for backwards compat - Use enable_events instead
        :type change_submits:    (bool)
        :param enable_events:    Turns on the element specific events. Radio Button events happen when an item is selected
        :type enable_events:     (bool)
        :param right_click_menu: A list of lists of Menu items to show when this element is right clicked. See user docs for exact format.
        :type right_click_menu:  List[List[ List[str] | str ]]
        :param expand_x:         If True the element will automatically expand in the X direction to fill available space
        :type expand_x:          (bool)
        :param expand_y:         If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:          (bool)
        :param visible:          set visibility state of the element
        :type visible:           (bool)
        :param metadata:         User metadata that can be set to ANYTHING
        :type metadata:          (Any)
        """


        self.InitialState = default
        self.Text = text
        self.Widget = self.TKRadio = None  # type: tk.Radiobutton
        self.GroupID = group_id
        self.Value = None
        self.Disabled = disabled
        self.TextColor = text_color if text_color else theme_text_color()
        self.RightClickMenu = right_click_menu

        if circle_color is None:
            # ---- compute color of circle background ---
            try:  # something in here will fail if a color is not specified in Hex
                text_hsl = _hex_to_hsl(self.TextColor)
                background_hsl = _hex_to_hsl(background_color if background_color else theme_background_color())
                l_delta = abs(text_hsl[2] - background_hsl[2]) / 10
                if text_hsl[2] > background_hsl[2]:  # if the text is "lighter" than the background then make background darker
                    bg_rbg = _hsl_to_rgb(background_hsl[0], background_hsl[1], background_hsl[2] - l_delta)
                else:
                    bg_rbg = _hsl_to_rgb(background_hsl[0], background_hsl[1], background_hsl[2] + l_delta)
                self.CircleBackgroundColor = rgb(*bg_rbg)
            except:
                self.CircleBackgroundColor = background_color if background_color else theme_background_color()
        else:
            self.CircleBackgroundColor = circle_color
        self.ChangeSubmits = change_submits or enable_events
        self.EncodedRadioValue = None
        key = key if key is not None else k
        sz = size if size != (None, None) else s
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y

        super().__init__(ELEM_TYPE_INPUT_RADIO, size=sz, auto_size_text=auto_size_text, font=font,
                         background_color=background_color, text_color=self.TextColor, key=key, pad=pad,
                         tooltip=tooltip, visible=visible, metadata=metadata)

    def update(self, value=None, text=None, background_color=None, text_color=None, circle_color=None, disabled=None, visible=None):
        """
        Changes some of the settings for the Radio Button Element. Must call `Window.read` or `Window.finalize` prior

        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param value:            if True change to selected and set others in group to unselected
        :type value:             (bool)
        :param text:             Text to display next to radio button
        :type text:              (str)
        :param background_color: color of background
        :type background_color:  (str)
        :param text_color:       color of the text. Note this also changes the color of the selection dot
        :type text_color:        (str)
        :param circle_color:     color of background of the circle that has the dot selection indicator in it
        :type circle_color:      (str)
        :param disabled:         disable or enable state of the element
        :type disabled:          (bool)
        :param visible:          control visibility of element
        :type visible:           (bool)
        """

        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in Radio.update - The window was closed')
            return


        if value is not None:
            try:
                if value is True:
                    self.TKIntVar.set(self.EncodedRadioValue)
                elif value is False:
                    if self.TKIntVar.get() == self.EncodedRadioValue:
                        self.TKIntVar.set(0)
            except:
                print('Error updating Radio')
            self.InitialState = value
        if text is not None:
            self.Text = str(text)
            self.TKRadio.configure(text=self.Text)
        if background_color not in (None, COLOR_SYSTEM_DEFAULT):
            self.TKRadio.configure(background=background_color)
            self.BackgroundColor = background_color
        if text_color not in (None, COLOR_SYSTEM_DEFAULT):
            self.TKRadio.configure(fg=text_color)
            self.TextColor = text_color

        if circle_color not in (None, COLOR_SYSTEM_DEFAULT):
            self.CircleBackgroundColor = circle_color
            self.TKRadio.configure(selectcolor=self.CircleBackgroundColor)  # The background of the radio button
        elif text_color or background_color:
            if self.TextColor not in (None, COLOR_SYSTEM_DEFAULT) and self.BackgroundColor not in (None, COLOR_SYSTEM_DEFAULT) and self.TextColor.startswith(
                    '#') and self.BackgroundColor.startswith('#'):
                # ---- compute color of circle background ---
                text_hsl = _hex_to_hsl(self.TextColor)
                background_hsl = _hex_to_hsl(self.BackgroundColor if self.BackgroundColor else theme_background_color())
                l_delta = abs(text_hsl[2] - background_hsl[2]) / 10
                if text_hsl[2] > background_hsl[2]:  # if the text is "lighter" than the background then make background darker
                    bg_rbg = _hsl_to_rgb(background_hsl[0], background_hsl[1], background_hsl[2] - l_delta)
                else:
                    bg_rbg = _hsl_to_rgb(background_hsl[0], background_hsl[1], background_hsl[2] + l_delta)
                self.CircleBackgroundColor = rgb(*bg_rbg)
                self.TKRadio.configure(selectcolor=self.CircleBackgroundColor)  # The background of the checkbox

        if disabled is True:
            self.TKRadio['state'] = 'disabled'
        elif disabled is False:
            self.TKRadio['state'] = 'normal'
        self.Disabled = disabled if disabled is not None else self.Disabled

        if visible is False:
            self._pack_forget_save_settings()
        elif visible is True:
            self._pack_restore_settings()
        if visible is not None:
            self._visible = visible

    def reset_group(self):
        """
        Sets all Radio Buttons in the group to not selected
        """
        self.TKIntVar.set(0)

    def get(self):
        # type: (Radio) -> bool
        """
        A snapshot of the value of Radio Button -> (bool)

        :return: True if this radio button is selected
        :rtype:  (bool)
        """
        return self.TKIntVar.get() == self.EncodedRadioValue

    Get = get
    ResetGroup = reset_group
    Update = update


R = Radio
Rad = Radio


# ---------------------------------------------------------------------- #
#                           Checkbox                                     #
# ---------------------------------------------------------------------- #
class Checkbox(Element):
    """
    Checkbox Element - Displays a checkbox and text next to it
    """

    def __init__(self, text, default=False, size=(None, None), s=(None, None), auto_size_text=None, font=None, background_color=None,
                 text_color=None, checkbox_color=None, highlight_thickness=1, change_submits=False, enable_events=False, disabled=False, key=None, k=None, pad=None, p=None, tooltip=None,
                 right_click_menu=None, expand_x=False, expand_y=False, visible=True, metadata=None):
        """
        :param text:                Text to display next to checkbox
        :type text:                 (str)
        :param default:             Set to True if you want this checkbox initially checked
        :type default:              (bool)
        :param size:                (w, h) w=characters-wide, h=rows-high. If an int instead of a tuple is supplied, then height is auto-set to 1
        :type size:                 (int, int)  | (None, None) | int
        :param s:                   Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
        :type s:                    (int, int)  | (None, None) | int
        :param auto_size_text:      if True will size the element to match the length of the text
        :type auto_size_text:       (bool)
        :param font:                specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:                 (str or (str, int[, str]) or None)
        :param background_color:    color of background
        :type background_color:     (str)
        :param text_color:          color of the text
        :type text_color:           (str)
        :param checkbox_color:      color of background of the box that has the check mark in it. The checkmark is the same color as the text
        :type checkbox_color:       (str)
        :param highlight_thickness: thickness of border around checkbox when gets focus
        :type highlight_thickness:  (int)
        :param change_submits:      DO NOT USE. Only listed for backwards compat - Use enable_events instead
        :type change_submits:       (bool)
        :param enable_events:       Turns on the element specific events. Checkbox events happen when an item changes
        :type enable_events:        (bool)
        :param disabled:            set disable state
        :type disabled:             (bool)
        :param key:                 Used with window.find_element and with return values to uniquely identify this element
        :type key:                  str | int | tuple | object
        :param k:                   Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                    str | int | tuple | object
        :param pad:                 Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:                  (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                   Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                    (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param tooltip:             text, that will appear when mouse hovers over the element
        :type tooltip:              (str)
        :param right_click_menu:    A list of lists of Menu items to show when this element is right clicked. See user docs for exact format.
        :type right_click_menu:     List[List[ List[str] | str ]]
        :param expand_x:            If True the element will automatically expand in the X direction to fill available space
        :type expand_x:             (bool)
        :param expand_y:            If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:             (bool)
        :param visible:             set visibility state of the element
        :type visible:              (bool)
        :param metadata:            User metadata that can be set to ANYTHING
        :type metadata:             (Any)
        """



        self.Text = text
        self.InitialState = bool(default)
        self.Value = None
        self.TKCheckbutton = self.Widget = None  # type: tk.Checkbutton
        self.Disabled = disabled
        self.TextColor = text_color if text_color else theme_text_color()
        self.RightClickMenu = right_click_menu
        self.highlight_thickness = highlight_thickness

        # ---- compute color of circle background ---
        if checkbox_color is None:
            try:  # something in here will fail if a color is not specified in Hex
                text_hsl = _hex_to_hsl(self.TextColor)
                background_hsl = _hex_to_hsl(background_color if background_color else theme_background_color())
                l_delta = abs(text_hsl[2] - background_hsl[2]) / 10
                if text_hsl[2] > background_hsl[2]:  # if the text is "lighter" than the background then make background darker
                    bg_rbg = _hsl_to_rgb(background_hsl[0], background_hsl[1], background_hsl[2] - l_delta)
                else:
                    bg_rbg = _hsl_to_rgb(background_hsl[0], background_hsl[1], background_hsl[2] + l_delta)
                self.CheckboxBackgroundColor = rgb(*bg_rbg)
            except:
                self.CheckboxBackgroundColor = background_color if background_color else theme_background_color()
        else:
            self.CheckboxBackgroundColor = checkbox_color
        self.ChangeSubmits = change_submits or enable_events
        key = key if key is not None else k
        sz = size if size != (None, None) else s
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y

        super().__init__(ELEM_TYPE_INPUT_CHECKBOX, size=sz, auto_size_text=auto_size_text, font=font,
                         background_color=background_color, text_color=self.TextColor, key=key, pad=pad,
                         tooltip=tooltip, visible=visible, metadata=metadata)

    def get(self):
        # type: (Checkbox) -> bool
        """
        Return the current state of this checkbox

        :return: Current state of checkbox
        :rtype:  (bool)
        """
        return self.TKIntVar.get() != 0

    def update(self, value=None, text=None, background_color=None, text_color=None, checkbox_color=None, disabled=None, visible=None):
        """
        Changes some of the settings for the Checkbox Element. Must call `Window.Read` or `Window.Finalize` prior.
        Note that changing visibility may cause element to change locations when made visible after invisible

        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param value:            if True checks the checkbox, False clears it
        :type value:             (bool)
        :param text:             Text to display next to checkbox
        :type text:              (str)
        :param background_color: color of background
        :type background_color:  (str)
        :param text_color:       color of the text. Note this also changes the color of the checkmark
        :type text_color:        (str)
        :param disabled:         disable or enable element
        :type disabled:          (bool)
        :param visible:          control visibility of element
        :type visible:           (bool)
        """

        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in Checkbox.update - The window was closed')
            return


        if value is not None:
            value = bool(value)
            try:
                self.TKIntVar.set(value)
                self.InitialState = value
            except:
                print('Checkbox update failed')
        if disabled is True:
            self.TKCheckbutton.configure(state='disabled')
        elif disabled is False:
            self.TKCheckbutton.configure(state='normal')
        self.Disabled = disabled if disabled is not None else self.Disabled

        if text is not None:
            self.Text = str(text)
            self.TKCheckbutton.configure(text=self.Text)
        if background_color not in (None, COLOR_SYSTEM_DEFAULT):
            self.TKCheckbutton.configure(background=background_color)
            self.BackgroundColor = background_color
        if text_color not in (None, COLOR_SYSTEM_DEFAULT):
            self.TKCheckbutton.configure(fg=text_color)
            self.TextColor = text_color
        # Color the checkbox itself
        if checkbox_color not in (None, COLOR_SYSTEM_DEFAULT):
            self.CheckboxBackgroundColor = checkbox_color
            self.TKCheckbutton.configure(selectcolor=self.CheckboxBackgroundColor)  # The background of the checkbox
        elif text_color or background_color:
            if self.CheckboxBackgroundColor is not None and self.TextColor is not None and self.BackgroundColor is not None and self.TextColor.startswith(
                    '#') and self.BackgroundColor.startswith('#'):
                # ---- compute color of checkbox background ---
                text_hsl = _hex_to_hsl(self.TextColor)
                background_hsl = _hex_to_hsl(self.BackgroundColor if self.BackgroundColor else theme_background_color())
                l_delta = abs(text_hsl[2] - background_hsl[2]) / 10
                if text_hsl[2] > background_hsl[2]:  # if the text is "lighter" than the background then make background darker
                    bg_rbg = _hsl_to_rgb(background_hsl[0], background_hsl[1], background_hsl[2] - l_delta)
                else:
                    bg_rbg = _hsl_to_rgb(background_hsl[0], background_hsl[1], background_hsl[2] + l_delta)
                self.CheckboxBackgroundColor = rgb(*bg_rbg)
                self.TKCheckbutton.configure(selectcolor=self.CheckboxBackgroundColor)  # The background of the checkbox

        if visible is False:
            self._pack_forget_save_settings()
        elif visible is True:
            self._pack_restore_settings()

        if visible is not None:
            self._visible = visible

    Get = get
    Update = update


# -------------------------  CHECKBOX Element lazy functions  ------------------------- #
CB = Checkbox
CBox = Checkbox
Check = Checkbox


# ---------------------------------------------------------------------- #
#                           Spin                                         #
# ---------------------------------------------------------------------- #

class Spin(Element):
    """
    A spinner with up/down buttons and a single line of text. Choose 1 values from list
    """

    def __init__(self, values, initial_value=None, disabled=False, change_submits=False, enable_events=False, readonly=False,
                 size=(None, None), s=(None, None), auto_size_text=None, bind_return_key=None, font=None, background_color=None, text_color=None, key=None, k=None, pad=None, p=None, wrap=None,
                 tooltip=None, right_click_menu=None, expand_x=False, expand_y=False, visible=True, metadata=None):
        """
        :param values:           List of valid values
        :type values:            Tuple[Any] or List[Any]
        :param initial_value:    Initial item to show in window. Choose from list of values supplied
        :type initial_value:     (Any)
        :param disabled:         set disable state
        :type disabled:          (bool)
        :param change_submits:   DO NOT USE. Only listed for backwards compat - Use enable_events instead
        :type change_submits:    (bool)
        :param enable_events:    Turns on the element specific events. Spin events happen when an item changes
        :type enable_events:     (bool)
        :param readonly:         If True, then users cannot type in values. Only values from the values list are allowed.
        :type readonly:          (bool)
        :param size:             (w, h) w=characters-wide, h=rows-high. If an int instead of a tuple is supplied, then height is auto-set to 1
        :type size:              (int, int)  | (None, None) | int
        :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
        :type s:                 (int, int)  | (None, None) | int
        :param auto_size_text:   if True will size the element to match the length of the text
        :type auto_size_text:    (bool)
        :param bind_return_key:  If True, then the return key will cause a the element to generate an event when return key is pressed
        :type bind_return_key:   (bool)
        :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:              (str or (str, int[, str]) or None)
        :param background_color: color of background
        :type background_color:  (str)
        :param text_color:       color of the text
        :type text_color:        (str)
        :param key:              Used with window.find_element and with return values to uniquely identify this element
        :type key:               str | int | tuple | object
        :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                 str | int | tuple | object
        :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param wrap:             Determines if the values should "Wrap". Default is False. If True, when reaching last value, will continue back to the first value.
        :type wrap:              (bool)
        :param tooltip:          text, that will appear when mouse hovers over the element
        :type tooltip:           (str)
        :param right_click_menu: A list of lists of Menu items to show when this element is right clicked. See user docs for exact format.
        :type right_click_menu:  List[List[ List[str] | str ]]
        :param expand_x:         If True the element will automatically expand in the X direction to fill available space
        :type expand_x:          (bool)
        :param expand_y:         If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:          (bool)
        :param visible:          set visibility state of the element
        :type visible:           (bool)
        :param metadata:         User metadata that can be set to ANYTHING
        :type metadata:          (Any)
        """


        self.Values = values
        self.DefaultValue = initial_value
        self.ChangeSubmits = change_submits or enable_events
        self.TKSpinBox = self.Widget = None  # type: tk.Spinbox
        self.Disabled = disabled
        self.Readonly = readonly
        self.RightClickMenu = right_click_menu
        self.BindReturnKey = bind_return_key
        self.wrap = wrap

        bg = background_color if background_color else DEFAULT_INPUT_ELEMENTS_COLOR
        fg = text_color if text_color is not None else DEFAULT_INPUT_TEXT_COLOR
        key = key if key is not None else k
        sz = size if size != (None, None) else s
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y


        super().__init__(ELEM_TYPE_INPUT_SPIN, size=sz, auto_size_text=auto_size_text, font=font, background_color=bg, text_color=fg,
                         key=key, pad=pad, tooltip=tooltip, visible=visible, metadata=metadata)
        return

    def update(self, value=None, values=None, disabled=None, readonly=None, visible=None):
        """
        Changes some of the settings for the Spin Element. Must call `Window.Read` or `Window.Finalize` prior
        Note that the state can be in 3 states only.... enabled, disabled, readonly even
        though more combinations are available. The easy way to remember is that if you
        change the readonly parameter then you are enabling the element.

        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param value:    set the current value from list of choices
        :type value:     (Any)
        :param values:   set available choices
        :type values:    List[Any]
        :param disabled: disable. Note disabled and readonly cannot be mixed. It must be one OR the other
        :type disabled:  (bool)
        :param readonly: make element readonly.  Note disabled and readonly cannot be mixed. It must be one OR the other
        :type readonly:  (bool)
        :param visible:  control visibility of element
        :type visible:   (bool)
        """

        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in Spin.update - The window was closed')
            return

        if values != None:
            old_value = self.TKStringVar.get()
            self.Values = values
            self.TKSpinBox.configure(values=values)
            self.TKStringVar.set(old_value)
        if value is not None:
            try:
                self.TKStringVar.set(value)
                self.DefaultValue = value
            except:
                pass

        if readonly is True:
            self.Readonly = True
            self.TKSpinBox['state'] = 'readonly'
        elif readonly is False:
            self.Readonly = False
            self.TKSpinBox['state'] = 'normal'
        if disabled is True:
            self.TKSpinBox['state'] = 'disable'
        elif disabled is False:
            if self.Readonly:
                self.TKSpinBox['state'] = 'readonly'
            else:
                self.TKSpinBox['state'] = 'normal'
        self.Disabled = disabled if disabled is not None else self.Disabled

        if visible is False:
            self._pack_forget_save_settings()
        elif visible is True:
            self._pack_restore_settings()
        if visible is not None:
            self._visible = visible


    def _SpinChangedHandler(self, event):
        """
        Callback function. Used internally only. Called by tkinter when Spinbox Widget changes.  Results in Window.Read() call returning

        :param event: passed in from tkinter
        :type event:
        """
        # first, get the results table built
        if self.Key is not None:
            self.ParentForm.LastButtonClicked = self.Key
        else:
            self.ParentForm.LastButtonClicked = ''
        self.ParentForm.FormRemainedOpen = True
        _exit_mainloop(self.ParentForm)
        # if self.ParentForm.CurrentlyRunningMainloop:
        #     Window._window_that_exited = self.ParentForm
        #     self.ParentForm.TKroot.quit()  # kick the users out of the mainloop




    def set_ibeam_color(self, ibeam_color=None):
        """
        Sets the color of the I-Beam that is used to "insert" characters. This is oftens called a "Cursor" by
        many users.  To keep from being confused with tkinter's definition of cursor (the mouse pointer), the term
        ibeam is used in this case.
        :param ibeam_color: color to set the "I-Beam" used to indicate where characters will be inserted
        :type ibeam_color:  (str)
        """

        if not self._widget_was_created():
            return
        if ibeam_color is not None:
            try:
                self.Widget.config(insertbackground=ibeam_color)
            except Exception as e:
                _error_popup_with_traceback('Error setting I-Beam color in set_ibeam_color',
                           'The element has a key:', self.Key,
                            'The color passed in was:', ibeam_color)



    def get(self):
        """
        Return the current chosen value showing in spinbox.
        This value will be the same as what was provided as list of choices.  If list items are ints, then the
        item returned will be an int (not a string)

        :return: The currently visible entry
        :rtype:  (Any)
        """
        value = self.TKStringVar.get()
        for v in self.Values:
            if str(v) == value:
                value = v
                break
        return value

    Get = get
    Update = update


Sp = Spin  # type: Spin


# ---------------------------------------------------------------------- #
#                           Multiline                                    #
# ---------------------------------------------------------------------- #
class Multiline(Element):
    """
    Multiline Element - Display and/or read multiple lines of text.  This is both an input and output element.
    Other PySimpleGUI ports have a separate MultilineInput and MultilineOutput elements.  May want to split this
    one up in the future too.
    """

    def __init__(self, default_text='', enter_submits=False, disabled=False, autoscroll=False, autoscroll_only_at_bottom=False, border_width=None,
                 size=(None, None), s=(None, None), auto_size_text=None, background_color=None, text_color=None, selected_text_color=None, selected_background_color=None, horizontal_scroll=False, change_submits=False,
                 enable_events=False, do_not_clear=True, key=None, k=None, write_only=False, auto_refresh=False, reroute_stdout=False, reroute_stderr=False, reroute_cprint=False, echo_stdout_stderr=False, focus=False, font=None, pad=None, p=None, tooltip=None, justification=None, no_scrollbar=False, wrap_lines=None,
                 sbar_trough_color=None, sbar_background_color=None, sbar_arrow_color=None, sbar_width=None, sbar_arrow_width=None, sbar_frame_color=None, sbar_relief=None,
                 expand_x=False, expand_y=False, rstrip=True, right_click_menu=None, visible=True, metadata=None):
        """
        :param default_text:                 Initial text to show
        :type default_text:                  (Any)
        :param enter_submits:                if True, the Window.read call will return is enter key is pressed in this element
        :type enter_submits:                 (bool)
        :param disabled:                     set disable state
        :type disabled:                      (bool)
        :param autoscroll:                   If True the contents of the element will automatically scroll as more data added to the end
        :type autoscroll:                    (bool)
        :param autoscroll_only_at_bottom:    If True the contents of the element will automatically scroll only if the scrollbar is at the bottom of the multiline
        :type autoscroll_only_at_bottom:     (bool)
        :param border_width:                 width of border around element in pixels
        :type border_width:                  (int)
        :param size:                         (w, h) w=characters-wide, h=rows-high. If an int instead of a tuple is supplied, then height is auto-set to 1
        :type size:                          (int, int)  | (None, None) | int
        :param s:                            Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
        :type s:                             (int, int)  | (None, None) | int
        :param auto_size_text:               if True will size the element to match the length of the text
        :type auto_size_text:                (bool)
        :param background_color:             color of background
        :type background_color:              (str)
        :param text_color:                   color of the text
        :type text_color:                    (str)
        :param selected_text_color:          Color of text when it is selected (using mouse or control+A, etc)
        :type selected_text_color:           (str)
        :param selected_background_color:    Color of background when it is selected (using mouse or control+A, etc)
        :type selected_background_color:     (str)
        :param horizontal_scroll:            Controls if a horizontal scrollbar should be shown.  If True a horizontal scrollbar will be shown in addition to vertical
        :type horizontal_scroll:             (bool)
        :param change_submits:               DO NOT USE. Only listed for backwards compat - Use enable_events instead
        :type change_submits:                (bool)
        :param enable_events:                If True then any key press that happens when the element has focus will generate an event.
        :type enable_events:                 (bool)
        :param do_not_clear:                 if False the element will be cleared any time the Window.read call returns
        :type do_not_clear:                  (bool)
        :param key:                          Used with window.find_element and with return values to uniquely identify this element to uniquely identify this element
        :type key:                           str | int | tuple | object
        :param k:                            Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                             str | int | tuple | object
        :param write_only:                   If True then no entry will be added to the values dictionary when the window is read
        :type write_only:                    bool
        :param auto_refresh:                 If True then anytime the element is updated, the window will be refreshed so that the change is immediately displayed
        :type auto_refresh:                  (bool)
        :param reroute_stdout:               If True then all output to stdout will be output to this element
        :type reroute_stdout:                (bool)
        :param reroute_stderr:               If True then all output to stderr will be output to this element
        :type reroute_stderr:                (bool)
        :param reroute_cprint:               If True your cprint calls will output to this element. It's the same as you calling cprint_set_output_destination
        :type reroute_cprint:                (bool)
        :param echo_stdout_stderr:           If True then output to stdout and stderr will be output to this element AND also to the normal console location
        :type echo_stdout_stderr:            (bool)
        :param focus:                        if True initial focus will go to this element
        :type focus:                         (bool)
        :param font:                         specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:                          (str or (str, int[, str]) or None)
        :param pad:                          Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:                           (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                            Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                             (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param tooltip:                      text, that will appear when mouse hovers over the element
        :type tooltip:                       (str)
        :param justification:                text justification. left, right, center. Can use single characters l, r, c.
        :type justification:                 (str)
        :param no_scrollbar:                 If False then a vertical scrollbar will be shown (the default)
        :type no_scrollbar:                  (bool)
        :param wrap_lines:                   If True, the lines will be wrapped automatically. Other parms affect this setting, but this one will override them all. Default is it does nothing and uses previous settings for wrapping.
        :type wrap_lines:                    (bool)
        :param sbar_trough_color:           Scrollbar color of the trough
        :type sbar_trough_color:            (str)
        :param sbar_background_color:       Scrollbar color of the background of the arrow buttons at the ends AND the color of the "thumb" (the thing you grab and slide). Switches to arrow color when mouse is over
        :type sbar_background_color:        (str)
        :param sbar_arrow_color:            Scrollbar color of the arrow at the ends of the scrollbar (it looks like a button). Switches to background color when mouse is over
        :type sbar_arrow_color:             (str)
        :param sbar_width:                  Scrollbar width in pixels
        :type sbar_width:                   (int)
        :param sbar_arrow_width:            Scrollbar width of the arrow on the scrollbar. It will potentially impact the overall width of the scrollbar
        :type sbar_arrow_width:             (int)
        :param sbar_frame_color:            Scrollbar Color of frame around scrollbar (available only on some ttk themes)
        :type sbar_frame_color:             (str)
        :param sbar_relief:                 Scrollbar relief that will be used for the "thumb" of the scrollbar (the thing you grab that slides). Should be a constant that is defined at starting with "RELIEF_" - RELIEF_RAISED, RELIEF_SUNKEN, RELIEF_FLAT, RELIEF_RIDGE, RELIEF_GROOVE, RELIEF_SOLID
        :type sbar_relief:                  (str)
        :param expand_x:                     If True the element will automatically expand in the X direction to fill available space
        :type expand_x:                      (bool)
        :param expand_y:                     If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:                      (bool)
        :param rstrip:                       If True the value returned in will have whitespace stripped from the right side
        :type rstrip:                        (bool)
        :param right_click_menu:             A list of lists of Menu items to show when this element is right clicked. See user docs for exact format.
        :type right_click_menu:              List[List[ List[str] | str ]]
        :param visible:                      set visibility state of the element
        :type visible:                       (bool)
        :param metadata:                     User metadata that can be set to ANYTHING
        :type metadata:                      (Any)
        """


        self.DefaultText = str(default_text)
        self.EnterSubmits = enter_submits
        bg = background_color if background_color else DEFAULT_INPUT_ELEMENTS_COLOR
        self.Focus = focus
        self.do_not_clear = do_not_clear
        fg = text_color if text_color is not None else DEFAULT_INPUT_TEXT_COLOR
        fg = text_color if text_color is not None else DEFAULT_INPUT_TEXT_COLOR
        self.selected_text_color = selected_text_color
        self.selected_background_color = selected_background_color
        self.Autoscroll = autoscroll
        self.Disabled = disabled
        self.ChangeSubmits = change_submits or enable_events
        self.RightClickMenu = right_click_menu
        self.BorderWidth = border_width if border_width is not None else DEFAULT_BORDER_WIDTH
        self.TagCounter = 0
        self.TKText = self.Widget = None  # type: tk.Text
        self.element_frame = None  # type: tk.Frame
        self.HorizontalScroll = horizontal_scroll
        self.tags = set()
        self.WriteOnly = write_only
        self.AutoRefresh = auto_refresh
        key = key if key is not None else k
        self.reroute_cprint = reroute_cprint
        self.echo_stdout_stderr = echo_stdout_stderr
        self.Justification = 'left' if justification is None else justification
        self.justification_tag = self.just_center_tag = self.just_left_tag = self.just_right_tag = None
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y
        self.rstrip = rstrip
        self.wrap_lines = wrap_lines
        self.reroute_stdout = reroute_stdout
        self.reroute_stderr = reroute_stderr
        self.no_scrollbar = no_scrollbar
        self.hscrollbar = None      # The horizontal scrollbar
        self.auto_scroll_only_at_bottom = autoscroll_only_at_bottom
        sz = size if size != (None, None) else s

        super().__init__(ELEM_TYPE_INPUT_MULTILINE, size=sz, auto_size_text=auto_size_text, background_color=bg,
                         text_color=fg, key=key, pad=pad, tooltip=tooltip, font=font or DEFAULT_FONT, visible=visible, metadata=metadata,
                         sbar_trough_color=sbar_trough_color, sbar_background_color=sbar_background_color, sbar_arrow_color=sbar_arrow_color, sbar_width=sbar_width, sbar_arrow_width=sbar_arrow_width, sbar_frame_color=sbar_frame_color, sbar_relief=sbar_relief)
        return

    def update(self, value=None, disabled=None, append=False, font=None, text_color=None, background_color=None, text_color_for_value=None,
               background_color_for_value=None, visible=None, autoscroll=None, justification=None, font_for_value=None):
        """
        Changes some of the settings for the Multiline Element. Must call `Window.read` or set finalize=True when creating window.

        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param value:                      new text to display
        :type value:                       (Any)
        :param disabled:                   disable or enable state of the element
        :type disabled:                    (bool)
        :param append:                     if True then new value will be added onto the end of the current value. if False then contents will be replaced.
        :type append:                      (bool)
        :param font:                       specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike for the entire element
        :type font:                        (str or (str, int[, str]) or None)
        :param text_color:                 color of the text
        :type text_color:                  (str)
        :param background_color:           color of background
        :type background_color:            (str)
        :param text_color_for_value:       color of the new text being added (the value paramter)
        :type text_color_for_value:        (str)
        :param background_color_for_value: color of the new background of the text being added (the value paramter)
        :type background_color_for_value:  (str)
        :param visible:                    set visibility state of the element
        :type visible:                     (bool)
        :param autoscroll:                 if True then contents of element are scrolled down when new text is added to the end
        :type autoscroll:                  (bool)
        :param justification:              text justification. left, right, center. Can use single characters l, r, c. Sets only for this value, not entire element
        :type justification:               (str)
        :param font_for_value:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike for the value being updated
        :type font_for_value:              str | (str, int)
        """

        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            # _error_popup_with_traceback('Error in Multiline.update - The window was closed')
            return


        if autoscroll is not None:
            self.Autoscroll = autoscroll
        current_scroll_position = self.TKText.yview()[1]

        if justification is not None:
            if justification.startswith('l'):
                just_tag = 'left'
            if justification.startswith('r'):
                just_tag = 'right'
            if justification.startswith('c'):
                just_tag = 'center'
        else:
            just_tag = self.justification_tag

        starting_point = self.Widget.index(tk.INSERT)
        tag = None
        if value is not None:
            value = str(value)
            if background_color_for_value is not None or text_color_for_value is not None or font_for_value is not None:
                try:
                    tag = 'Multiline(' + str(text_color_for_value) + ',' + str(background_color_for_value) + ',' + str(font_for_value) + ')'
                    if tag not in self.tags:
                        self.tags.add(tag)
                    if background_color_for_value is not None:
                        self.TKText.tag_configure(tag, background=background_color_for_value)
                    if text_color_for_value is not None:
                        self.TKText.tag_configure(tag, foreground=text_color_for_value)
                    if font_for_value is not None:
                        self.TKText.tag_configure(tag, font=font_for_value)
                except Exception as e:
                    print('* Multiline.update - bad color likely specified:', e)
            if self.Disabled:
                self.TKText.configure(state='normal')
            try:
                if not append:
                    self.TKText.delete('1.0', tk.END)
                if tag is not None or just_tag is not None:
                    self.TKText.insert(tk.END, value, (just_tag, tag))
                else:
                    self.TKText.insert(tk.END, value)

                # self.TKText.tag_add(just_tag, starting_point, starting_point)

            except Exception as e:
                print('* Error setting multiline *', e)
            if self.Disabled:
                self.TKText.configure(state='disabled')
            self.DefaultText = value

        # if self.Autoscroll:
        #     self.TKText.see(tk.END)
        if self.Autoscroll:
            if not self.auto_scroll_only_at_bottom or (self.auto_scroll_only_at_bottom and current_scroll_position == 1.0):
                self.TKText.see(tk.END)
        if disabled is True:
            self.TKText.configure(state='disabled')
        elif disabled is False:
            self.TKText.configure(state='normal')
        self.Disabled = disabled if disabled is not None else self.Disabled

        if background_color not in (None, COLOR_SYSTEM_DEFAULT):
            self.TKText.configure(background=background_color)
        if text_color not in (None, COLOR_SYSTEM_DEFAULT):
            self.TKText.configure(fg=text_color)
        if font is not None:
            self.TKText.configure(font=font)


        if visible is False:
            self._pack_forget_save_settings(alternate_widget=self.element_frame)
            # self.element_frame.pack_forget()
        elif visible is True:
            self._pack_restore_settings(alternate_widget=self.element_frame)
            # self.element_frame.pack(padx=self.pad_used[0], pady=self.pad_used[1])

        if self.AutoRefresh and self.ParentForm:
            try:  # in case the window was destroyed
                self.ParentForm.refresh()
            except:
                pass
        if visible is not None:
            self._visible = visible

    def get(self):
        """
        Return current contents of the Multiline Element

        :return: current contents of the Multiline Element (used as an input type of Multiline
        :rtype:  (str)
        """
        value = str(self.TKText.get(1.0, tk.END))
        if self.rstrip:
            return value.rstrip()
        return value


    def print(self, *args, end=None, sep=None, text_color=None, background_color=None, justification=None, font=None, colors=None, t=None, b=None, c=None,
              autoscroll=True):
        """
        Print like Python normally prints except route the output to a multiline element and also add colors if desired

        colors -(str, str) or str.  A combined text/background color definition in a single parameter

        There are also "aliases" for text_color, background_color and colors (t, b, c)
        t - An alias for color of the text (makes for shorter calls)
        b - An alias for the background_color parameter
        c - (str, str) - "shorthand" way of specifying color. (foreground, backgrouned)
        c - str - can also be a string of the format "foreground on background"  ("white on red")

        With the aliases it's possible to write the same print but in more compact ways:
        cprint('This will print white text on red background', c=('white', 'red'))
        cprint('This will print white text on red background', c='white on red')
        cprint('This will print white text on red background', text_color='white', background_color='red')
        cprint('This will print white text on red background', t='white', b='red')

        :param args:             The arguments to print
        :type args:              (Any)
        :param end:              The end char to use just like print uses
        :type end:               (str)
        :param sep:              The separation character like print uses
        :type sep:               (str)
        :param text_color:       The color of the text
        :type text_color:        (str)
        :param background_color: The background color of the line
        :type background_color:  (str)
        :param justification:    text justification. left, right, center. Can use single characters l, r, c. Sets only for this value, not entire element
        :type justification:     (str)
        :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike for the args being printed
        :type font:              (str or (str, int[, str]) or None)
        :param colors:           Either a tuple or a string that has both the text and background colors. Or just the text color
        :type colors:            (str) or (str, str)
        :param t:                Color of the text
        :type t:                 (str)
        :param b:                The background color of the line
        :type b:                 (str)
        :param c:                Either a tuple or a string that has both the text and background colors or just tex color (same as the color parm)
        :type c:                 (str) or (str, str)
        :param autoscroll:       If True the contents of the element will automatically scroll as more data added to the end
        :type autoscroll:        (bool)
        """

        kw_text_color = text_color or t
        kw_background_color = background_color or b
        dual_color = colors or c
        try:
            if isinstance(dual_color, tuple):
                kw_text_color = dual_color[0]
                kw_background_color = dual_color[1]
            elif isinstance(dual_color, str):
                if ' on ' in dual_color:  # if has "on" in the string, then have both text and background
                    kw_text_color = dual_color.split(' on ')[0]
                    kw_background_color = dual_color.split(' on ')[1]
                else:  # if no "on" then assume the color string is just the text color
                    kw_text_color = dual_color
        except Exception as e:
            print('* multiline print warning * you messed up with color formatting', e)

        _print_to_element(self, *args, end=end, sep=sep, text_color=kw_text_color, background_color=kw_background_color, justification=justification,
                          autoscroll=autoscroll, font=font)

    def reroute_stdout_to_here(self):
        """
        Sends stdout (prints) to this element
        """
        # if nothing on the stack, then need to save the very first stdout
        if len(Window._rerouted_stdout_stack) == 0:
            Window._original_stdout = sys.stdout
        Window._rerouted_stdout_stack.insert(0, (self.ParentForm, self))
        sys.stdout = self

    def reroute_stderr_to_here(self):
        """
        Sends stderr to this element
        """
        if len(Window._rerouted_stderr_stack) == 0:
            Window._original_stderr = sys.stderr
        Window._rerouted_stderr_stack.insert(0, (self.ParentForm, self))
        sys.stderr = self

    def restore_stdout(self):
        """
        Restore a previously re-reouted stdout back to the original destination
        """
        Window._restore_stdout()

    def restore_stderr(self):
        """
        Restore a previously re-reouted stderr back to the original destination
        """
        Window._restore_stderr()

    def write(self, txt):
        """
        Called by Python (not tkinter?) when stdout or stderr wants to write

        :param txt: text of output
        :type txt:  (str)
        """
        try:
            self.update(txt, append=True)
            # if need to echo, then send the same text to the destinatoin that isn't thesame as this one
            if self.echo_stdout_stderr:
                if sys.stdout != self:
                    sys.stdout.write(txt)
                elif sys.stderr != self:
                    sys.stderr.write(txt)
        except:
            pass

    def flush(self):
        """
        Flush parameter was passed into a print statement.
        For now doing nothing.  Not sure what action should be taken to ensure a flush happens regardless.
        """
        # try:
        #     self.previous_stdout.flush()
        # except:
        #     pass
        return





    def set_ibeam_color(self, ibeam_color=None):
        """
        Sets the color of the I-Beam that is used to "insert" characters. This is oftens called a "Cursor" by
        many users.  To keep from being confused with tkinter's definition of cursor (the mouse pointer), the term
        ibeam is used in this case.
        :param ibeam_color: color to set the "I-Beam" used to indicate where characters will be inserted
        :type ibeam_color:  (str)
        """

        if not self._widget_was_created():
            return
        if ibeam_color is not None:
            try:
                self.Widget.config(insertbackground=ibeam_color)
            except Exception as e:
                _error_popup_with_traceback('Error setting I-Beam color in set_ibeam_color',
                           'The element has a key:', self.Key,
                            'The color passed in was:', ibeam_color)



    def __del__(self):
        """
        AT ONE TIME --- If this Widget is deleted, be sure and restore the old stdout, stderr
        Now the restore is done differently. Do not want to RELY on Python to call this method
        in order for stdout and stderr to be restored.  Instead explicit restores are called.

        """

        return


    Get = get
    Update = update


ML = Multiline
MLine = Multiline


# ---------------------------------------------------------------------- #
#                                       Text                             #
# ---------------------------------------------------------------------- #
class Text(Element):
    """
    Text - Display some text in the window.  Usually this means a single line of text.  However, the text can also be multiple lines.  If multi-lined there are no scroll bars.
    """

    def __init__(self, text='', size=(None, None), s=(None, None), auto_size_text=None, click_submits=False, enable_events=False, relief=None, font=None,
                 text_color=None, background_color=None, border_width=None, justification=None, pad=None, p=None, key=None, k=None, right_click_menu=None, expand_x=False, expand_y=False, grab=None,
                 tooltip=None, visible=True, metadata=None):
        """
        :param text:             The text to display. Can include /n to achieve multiple lines.  Will convert (optional) parameter into a string
        :type text:              Any
        :param size:             (w, h) w=characters-wide, h=rows-high. If an int instead of a tuple is supplied, then height is auto-set to 1
        :type size:              (int, int) |  (int, None) | (None, None) | (int, ) | int
        :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
        :type s:                 (int, int) |  (int, None) | (None, None) | (int, ) | int
        :param auto_size_text:   if True size of the Text Element will be sized to fit the string provided in 'text' parm
        :type auto_size_text:    (bool)
        :param click_submits:    DO NOT USE. Only listed for backwards compat - Use enable_events instead
        :type click_submits:     (bool)
        :param enable_events:    Turns on the element specific events. Text events happen when the text is clicked
        :type enable_events:     (bool)
        :param relief:           relief style around the text. Values are same as progress meter relief values. Should be a constant that is defined at starting with RELIEF - RELIEF_RAISED, RELIEF_SUNKEN, RELIEF_FLAT, RELIEF_RIDGE, RELIEF_GROOVE, RELIEF_SOLID
        :type relief:            (str)
        :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:              (str or (str, int[, str]) or None)
        :param text_color:       color of the text
        :type text_color:        (str)
        :param background_color: color of background
        :type background_color:  (str)
        :param border_width:     number of pixels for the border (if using a relief)
        :type border_width:      (int)
        :param justification:    how string should be aligned within space provided by size. Valid choices = `left`, `right`, `center`
        :type justification:     (str)
        :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param key:              Used with window.find_element and with return values to uniquely identify this element to uniquely identify this element
        :type key:               str or int or tuple or object
        :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                 str | int | tuple | object
        :param right_click_menu: A list of lists of Menu items to show when this element is right clicked. See user docs for exact format.
        :type right_click_menu:  List[List[ List[str] | str ]]
        :param expand_x:         If True the element will automatically expand in the X direction to fill available space
        :type expand_x:          (bool)
        :param expand_y:         If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:          (bool)
        :param grab:             If True can grab this element and move the window around. Default is False
        :type grab:              (bool)
        :param tooltip:          text, that will appear when mouse hovers over the element
        :type tooltip:           (str)
        :param visible:          set visibility state of the element
        :type visible:           (bool)
        :param metadata:         User metadata that can be set to ANYTHING
        :type metadata:          (Any)
        """

        self.DisplayText = str(text)
        self.TextColor = text_color if text_color else DEFAULT_TEXT_COLOR
        self.Justification = justification
        self.Relief = relief
        self.ClickSubmits = click_submits or enable_events
        if background_color is None:
            bg = DEFAULT_TEXT_ELEMENT_BACKGROUND_COLOR
        else:
            bg = background_color
        self.RightClickMenu = right_click_menu
        self.TKRightClickMenu = None
        self.BorderWidth = border_width
        self.Grab = grab
        key = key if key is not None else k
        sz = size if size != (None, None) else s
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y

        super().__init__(ELEM_TYPE_TEXT, auto_size_text=auto_size_text, size=sz, background_color=bg, font=font if font else DEFAULT_FONT,
                         text_color=self.TextColor, pad=pad, key=key, tooltip=tooltip, visible=visible, metadata=metadata)

    def update(self, value=None, background_color=None, text_color=None, font=None, visible=None):
        """
        Changes some of the settings for the Text Element. Must call `Window.Read` or `Window.Finalize` prior

        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param value:            new text to show
        :type value:             (Any)
        :param background_color: color of background
        :type background_color:  (str)
        :param text_color:       color of the text
        :type text_color:        (str)
        :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:              (str or (str, int[, str]) or None)
        :param visible:          set visibility state of the element
        :type visible:           (bool)
        """

        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in Text.update - The window was closed')
            return

        if value is not None:
            self.DisplayText = str(value)
            self.TKStringVar.set(str(value))
        if background_color not in (None, COLOR_SYSTEM_DEFAULT):
            self.TKText.configure(background=background_color)
        if text_color not in (None, COLOR_SYSTEM_DEFAULT):
            self.TKText.configure(fg=text_color)
        if font is not None:
            self.TKText.configure(font=font)
        if visible is False:
            self._pack_forget_save_settings()
            # self.TKText.pack_forget()
        elif visible is True:
            self._pack_restore_settings()
            # self.TKText.pack(padx=self.pad_used[0], pady=self.pad_used[1])
        if visible is not None:
            self._visible = visible

    def get(self):
        """
        Gets the current value of the displayed text

        :return: The current value
        :rtype:  (str)
        """
        try:
            text = self.TKStringVar.get()
        except:
            text = ''
        return text


    @classmethod
    def fonts_installed_list(cls):
        """
        Returns a list of strings that tkinter reports as the installed fonts

        :return:          List of the installed font names
        :rtype:           List[str]
        """
        # A window must exist before can perform this operation. Create the hidden master root if it doesn't exist
        _get_hidden_master_root()

        fonts = list(tkinter.font.families())
        fonts.sort()

        return fonts


    @classmethod
    def char_width_in_pixels(cls, font, character='W'):
        """
        Get the with of the character "W" in pixels for the font being passed in or
        the character of your choosing if "W" is not a good representative character.
        Cannot be used until a window has been created.
        If an error occurs, 0 will be returned
        :param font:      specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike, to be measured
        :type font:       (str or (str, int[, str]) or None)
        :param character: specifies a SINGLE CHARACTER character to measure
        :type character:  (str)
        :return:          Width in pixels of "A"
        :rtype:           (int)
        """
        # A window must exist before can perform this operation. Create the hidden master root if it doesn't exist
        _get_hidden_master_root()

        size = 0
        try:
            size = tkinter.font.Font(font=font).measure(character)  # single character width
        except Exception as e:
            _error_popup_with_traceback('Exception retrieving char width in pixels', e)

        return size

    @classmethod
    def char_height_in_pixels(cls, font):
        """
        Get the height of a string if using the supplied font in pixels.
        Cannot be used until a window has been created.
        If an error occurs, 0 will be returned
        :param font: specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike, to be measured
        :type font:  (str or (str, int[, str]) or None)
        :return:     Height in pixels of "A"
        :rtype:      (int)
        """

        # A window must exist before can perform this operation. Create the hidden master root if it doesn't exist
        _get_hidden_master_root()


        size = 0
        try:
            size = tkinter.font.Font(font=font).metrics('linespace')
        except Exception as e:
            _error_popup_with_traceback('Exception retrieving char height in pixels', e)

        return size

    @classmethod
    def string_width_in_pixels(cls, font, string):
        """
        Get the with of the supplied string in pixels for the font being passed in.
        If an error occurs, 0 will be returned
        :param font:   specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike, to be measured
        :type font:    (str or (str, int[, str]) or None)
        :param string: the string to measure
        :type string:  str
        :return:       Width in pixels of string
        :rtype:        (int)
        """

        # A window must exist before can perform this operation. Create the hidden master root if it doesn't exist
        _get_hidden_master_root()

        size = 0
        try:
            size = tkinter.font.Font(font=font).measure(string)  # string's  width
        except Exception as e:
            _error_popup_with_traceback('Exception retrieving string width in pixels', e)

        return size

    def _print_to_element(self, *args, end=None, sep=None, text_color=None, background_color=None, autoscroll=None, justification=None, font=None, append=None):
        """
        Print like Python normally prints except route the output to a multiline element and also add colors if desired

        :param multiline_element: The multiline element to be output to
        :type multiline_element:  (Multiline)
        :param args:              The arguments to print
        :type args:               List[Any]
        :param end:               The end char to use just like print uses
        :type end:                (str)
        :param sep:               The separation character like print uses
        :type sep:                (str)
        :param text_color:        color of the text
        :type text_color:         (str)
        :param background_color:  The background color of the line
        :type background_color:   (str)
        :param autoscroll:        If True (the default), the element will scroll to bottom after updating
        :type autoscroll:         (bool)
        :param font:              specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike for the value being updated
        :type font:               str | (str, int)
        """
        end_str = str(end) if end is not None else '\n'
        sep_str = str(sep) if sep is not None else ' '

        outstring = ''
        num_args = len(args)
        for i, arg in enumerate(args):
            outstring += str(arg)
            if i != num_args - 1:
                outstring += sep_str
        outstring += end_str
        if append:
            outstring = self.get() + outstring

        self.update(outstring, text_color=text_color, background_color=background_color, font=font)

        try:  # if the element is set to autorefresh, then refresh the parent window
            if self.AutoRefresh:
                self.ParentForm.refresh()
        except:
            pass

    def print(self, *args, end=None, sep=None, text_color=None, background_color=None, justification=None, font=None, colors=None, t=None, b=None, c=None, autoscroll=True, append=True):
        """
        Print like Python normally prints except route the output to a multiline element and also add colors if desired

        colors -(str, str) or str.  A combined text/background color definition in a single parameter

        There are also "aliases" for text_color, background_color and colors (t, b, c)
        t - An alias for color of the text (makes for shorter calls)
        b - An alias for the background_color parameter
        c - (str, str) - "shorthand" way of specifying color. (foreground, backgrouned)
        c - str - can also be a string of the format "foreground on background"  ("white on red")

        With the aliases it's possible to write the same print but in more compact ways:
        cprint('This will print white text on red background', c=('white', 'red'))
        cprint('This will print white text on red background', c='white on red')
        cprint('This will print white text on red background', text_color='white', background_color='red')
        cprint('This will print white text on red background', t='white', b='red')

        :param args:             The arguments to print
        :type args:              (Any)
        :param end:              The end char to use just like print uses
        :type end:               (str)
        :param sep:              The separation character like print uses
        :type sep:               (str)
        :param text_color:       The color of the text
        :type text_color:        (str)
        :param background_color: The background color of the line
        :type background_color:  (str)
        :param justification:    text justification. left, right, center. Can use single characters l, r, c. Sets only for this value, not entire element
        :type justification:     (str)
        :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike for the args being printed
        :type font:              (str or (str, int[, str]) or None)
        :param colors:           Either a tuple or a string that has both the text and background colors. Or just the text color
        :type colors:            (str) or (str, str)
        :param t:                Color of the text
        :type t:                 (str)
        :param b:                The background color of the line
        :type b:                 (str)
        :param c:                Either a tuple or a string that has both the text and background colors or just tex color (same as the color parm)
        :type c:                 (str) or (str, str)
        :param autoscroll:       If True the contents of the element will automatically scroll as more data added to the end
        :type autoscroll:        (bool)
        """

        kw_text_color = text_color or t
        kw_background_color = background_color or b
        dual_color = colors or c
        try:
            if isinstance(dual_color, tuple):
                kw_text_color = dual_color[0]
                kw_background_color = dual_color[1]
            elif isinstance(dual_color, str):
                if ' on ' in dual_color:  # if has "on" in the string, then have both text and background
                    kw_text_color = dual_color.split(' on ')[0]
                    kw_background_color = dual_color.split(' on ')[1]
                else:  # if no "on" then assume the color string is just the text color
                    kw_text_color = dual_color
        except Exception as e:
            print('* multiline print warning * you messed up with color formatting', e)

        self._print_to_element( *args, end=end, sep=sep, text_color=kw_text_color, background_color=kw_background_color, justification=justification, autoscroll=autoscroll, font=font, append=append)


    Get = get
    Update = update


# -------------------------  Text Element lazy functions  ------------------------- #

Txt = Text  # type: Text
T = Text  # type: Text


# ---------------------------------------------------------------------- #
#                                       StatusBar                        #
# ---------------------------------------------------------------------- #
class StatusBar(Element):
    """
    A StatusBar Element creates the sunken text-filled strip at the bottom. Many Windows programs have this line
    """

    def __init__(self, text, size=(None, None), s=(None, None), auto_size_text=None, click_submits=None, enable_events=False,
                 relief=RELIEF_SUNKEN, font=None, text_color=None, background_color=None, justification=None, pad=None, p=None,
                 key=None, k=None, right_click_menu=None, expand_x=False, expand_y=False, tooltip=None, visible=True, metadata=None):
        """
        :param text:             Text that is to be displayed in the widget
        :type text:              (str)
        :param size:             (w, h) w=characters-wide, h=rows-high. If an int instead of a tuple is supplied, then height is auto-set to 1
        :type size:              (int, int) |  (int, None) | int
        :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
        :type s:                 (int, int)  | (None, None) | int
        :param auto_size_text:   True if size should fit the text length
        :type auto_size_text:    (bool)
        :param click_submits:    DO NOT USE. Only listed for backwards compat - Use enable_events instead
        :type click_submits:     (bool)
        :param enable_events:    Turns on the element specific events. StatusBar events occur when the bar is clicked
        :type enable_events:     (bool)
        :param relief:           relief style. Values are same as progress meter relief values.  Can be a constant or a string: `RELIEF_RAISED RELIEF_SUNKEN RELIEF_FLAT RELIEF_RIDGE RELIEF_GROOVE RELIEF_SOLID`
        :type relief:            (enum)
        :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:              (str or (str, int[, str]) or None)
        :param text_color:       color of the text
        :type text_color:        (str)
        :param background_color: color of background
        :type background_color:  (str)
        :param justification:    how string should be aligned within space provided by size. Valid choices = `left`, `right`, `center`
        :type justification:     (str)
        :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param key:              Used with window.find_element and with return values to uniquely identify this element to uniquely identify this element
        :type key:               str | int | tuple | object
        :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                 str | int | tuple | object
        :param right_click_menu: A list of lists of Menu items to show when this element is right clicked. See user docs for exact format.
        :type right_click_menu:  List[List[ List[str] | str ]]
        :param expand_x:         If True the element will automatically expand in the X direction to fill available space
        :type expand_x:          (bool)
        :param expand_y:         If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:          (bool)
        :param tooltip:          text, that will appear when mouse hovers over the element
        :type tooltip:           (str)
        :param visible:          set visibility state of the element
        :type visible:           (bool)
        :param metadata:         User metadata that can be set to ANYTHING
        :type metadata:          (Any)
        """


        self.DisplayText = text
        self.TextColor = text_color if text_color else DEFAULT_TEXT_COLOR
        self.Justification = justification
        self.Relief = relief
        self.ClickSubmits = click_submits or enable_events
        if background_color is None:
            bg = DEFAULT_TEXT_ELEMENT_BACKGROUND_COLOR
        else:
            bg = background_color
        self.TKText = self.Widget = None  # type: tk.Label
        key = key if key is not None else k
        self.RightClickMenu = right_click_menu
        sz = size if size != (None, None) else s
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y

        super().__init__(ELEM_TYPE_STATUSBAR, size=sz, auto_size_text=auto_size_text, background_color=bg,
                         font=font or DEFAULT_FONT, text_color=self.TextColor, pad=pad, key=key, tooltip=tooltip,
                         visible=visible, metadata=metadata)
        return

    def update(self, value=None, background_color=None, text_color=None, font=None, visible=None):
        """
        Changes some of the settings for the Status Bar Element. Must call `Window.Read` or `Window.Finalize` prior

        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param value:            new text to show
        :type value:             (str)
        :param background_color: color of background
        :type background_color:  (str)
        :param text_color:       color of the text
        :type text_color:        (str)
        :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:              (str or (str, int[, str]) or None)
        :param visible:          set visibility state of the element
        :type visible:           (bool)
        """

        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in StatusBar.update - The window was closed')
            return

        if value is not None:
            self.DisplayText = value
            stringvar = self.TKStringVar
            stringvar.set(value)
        if background_color not in (None, COLOR_SYSTEM_DEFAULT):
            self.TKText.configure(background=background_color)
        if text_color not in (None, COLOR_SYSTEM_DEFAULT):
            self.TKText.configure(fg=text_color)
        if font is not None:
            self.TKText.configure(font=font)
        if visible is False:
            self._pack_forget_save_settings()
            # self.TKText.pack_forget()
        elif visible is True:
            self._pack_restore_settings()
            # self.TKText.pack(padx=self.pad_used[0], pady=self.pad_used[1])
        if visible is not None:
            self._visible = visible

    Update = update


SBar = StatusBar


# ---------------------------------------------------------------------- #
#                       TKProgressBar                                    #
#  Emulate the TK ProgressBar using canvas and rectangles
# ---------------------------------------------------------------------- #

class TKProgressBar():
    uniqueness_counter = 0

    def __init__(self, root, max, length=400, width=DEFAULT_PROGRESS_BAR_SIZE[1], ttk_theme=DEFAULT_TTK_THEME, style_name='',
                 relief=DEFAULT_PROGRESS_BAR_RELIEF, border_width=DEFAULT_PROGRESS_BAR_BORDER_WIDTH,
                 orientation='horizontal', BarColor=(None, None), key=None):
        """
        :param root:         The root window bar is to be shown in
        :type root:          tk.Tk | tk.TopLevel
        :param max:          Maximum value the bar will be measuring
        :type max:           (int)
        :param length:       length in pixels of the bar
        :type length:        (int)
        :param width:        width in pixels of the bar
        :type width:         (int)
        :param style_name:   Progress bar style to use.  Set in the packer function
        :type style_name:    (str)
        :param ttk_theme:    Progress bar style defined as one of these 'default', 'winnative', 'clam', 'alt', 'classic', 'vista', 'xpnative'
        :type ttk_theme:     (str)
        :param relief:       relief style. Values are same as progress meter relief values.  Can be a constant or a string: `RELIEF_RAISED RELIEF_SUNKEN RELIEF_FLAT RELIEF_RIDGE RELIEF_GROOVE RELIEF_SOLID` (Default value = DEFAULT_PROGRESS_BAR_RELIEF)
        :type relief:        (str)
        :param border_width: The amount of pixels that go around the outside of the bar
        :type border_width:  (int)
        :param orientation:  'horizontal' or 'vertical' ('h' or 'v' work) (Default value = 'vertical')
        :type orientation:   (str)
        :param BarColor:     The 2 colors that make up a progress bar. One is the background, the other is the bar
        :type BarColor:      (str, str)
        :param key:          Used with window.find_element and with return values to uniquely identify this element to uniquely identify this element
        :type key:           str | int | tuple | object
        """

        self.Length = length
        self.Width = width
        self.Max = max
        self.Orientation = orientation
        self.Count = None
        self.PriorCount = 0
        self.style_name = style_name

        TKProgressBar.uniqueness_counter += 1

        if orientation.lower().startswith('h'):
            s = ttk.Style()
            _change_ttk_theme(s, ttk_theme)

            # self.style_name = str(key) + str(TKProgressBar.uniqueness_counter) + "my.Horizontal.TProgressbar"
            if BarColor != COLOR_SYSTEM_DEFAULT and BarColor[0] != COLOR_SYSTEM_DEFAULT:
                s.configure(self.style_name, background=BarColor[0], troughcolor=BarColor[1],
                            troughrelief=relief, borderwidth=border_width, thickness=width)
            else:
                s.configure(self.style_name, troughrelief=relief, borderwidth=border_width, thickness=width)

            self.TKProgressBarForReal = ttk.Progressbar(root, maximum=self.Max, style=self.style_name, length=length, orient=tk.HORIZONTAL, mode='determinate')
        else:
            s = ttk.Style()
            _change_ttk_theme(s, ttk_theme)
            # self.style_name = str(key) + str(TKProgressBar.uniqueness_counter) + "my.Vertical.TProgressbar"
            if BarColor != COLOR_SYSTEM_DEFAULT and BarColor[0] != COLOR_SYSTEM_DEFAULT:

                s.configure(self.style_name, background=BarColor[0],
                            troughcolor=BarColor[1], troughrelief=relief, borderwidth=border_width, thickness=width)
            else:
                s.configure(self.style_name, troughrelief=relief, borderwidth=border_width, thickness=width)

            self.TKProgressBarForReal = ttk.Progressbar(root, maximum=self.Max, style=self.style_name, length=length, orient=tk.VERTICAL, mode='determinate')

    def Update(self, count=None, max=None):
        """
        Update the current value of the bar and/or update the maximum value the bar can reach
        :param count: current value
        :type count:  (int)
        :param max:   the maximum value
        :type max:    (int)
        """
        if max is not None:
            self.Max = max
            try:
                self.TKProgressBarForReal.config(maximum=max)
            except:
                return False
        if count is not None:
            try:
                self.TKProgressBarForReal['value'] = count
            except:
                return False
        return True


# ---------------------------------------------------------------------- #
#                           Output                                       #
#  Routes stdout, stderr to a scrolled window                            #
# ---------------------------------------------------------------------- #
class Output(Multiline):
    """
    Output Element - a multi-lined text area to where stdout, stderr, cprint are rerouted.

    The Output Element is now based on the Multiline Element.  When you make an Output Element, you're
    creating a Multiline Element with some specific settings set:
        auto_refresh = True
        auto_scroll = True
        reroute_stdout = True
        reroute_stderr = True
        reroute_cprint = True
        write_only = True

    If you choose to use a Multiline element to replace an Output element, be sure an turn on the write_only paramter in the Multiline
    so that an item is not included in the values dictionary on every window.read call
    """

    def __init__(self, size=(None, None), s=(None, None), background_color=None, text_color=None, pad=None, p=None, autoscroll_only_at_bottom=False, echo_stdout_stderr=False, font=None, tooltip=None,
                 key=None, k=None, right_click_menu=None, expand_x=False, expand_y=False, visible=True, metadata=None, wrap_lines=None, horizontal_scroll=None,
                 sbar_trough_color=None, sbar_background_color=None, sbar_arrow_color=None,  sbar_width=None, sbar_arrow_width=None, sbar_frame_color=None, sbar_relief=None):
        """
        :param size:                        (w, h) w=characters-wide, h=rows-high. If an int instead of a tuple is supplied, then height is auto-set to 1
        :type size:                         (int, int)  | (None, None) | int
        :param s:                           Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
        :type s:                            (int, int)  | (None, None) | int
        :param background_color:            color of background
        :type background_color:             (str)
        :param text_color:                  color of the text
        :type text_color:                   (str)
        :param pad:                         Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:                          (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                           Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                            (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param autoscroll_only_at_bottom:   If True the contents of the element will automatically scroll only if the scrollbar is at the bottom of the multiline
        :type autoscroll_only_at_bottom:    (bool)
        :param echo_stdout_stderr:          If True then output to stdout will be output to this element AND also to the normal console location
        :type echo_stdout_stderr:           (bool)
        :param font:                        specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:                         (str or (str, int[, str]) or None)
        :param tooltip:                     text, that will appear when mouse hovers over the element
        :type tooltip:                      (str)
        :param key:                         Used with window.find_element and with return values to uniquely identify this element to uniquely identify this element
        :type key:                          str | int | tuple | object
        :param k:                           Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                            str | int | tuple | object
        :param right_click_menu:            A list of lists of Menu items to show when this element is right clicked. See user docs for exact format.
        :type right_click_menu:             List[List[ List[str] | str ]]
        :param expand_x:                    If True the element will automatically expand in the X direction to fill available space
        :type expand_x:                     (bool)
        :param expand_y:                    If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:                     (bool)
        :param visible:                     set visibility state of the element
        :type visible:                      (bool)
        :param metadata:                    User metadata that can be set to ANYTHING
        :type metadata:                     (Any)
        :param wrap_lines:                  If True, the lines will be wrapped automatically. Other parms affect this setting, but this one will override them all. Default is it does nothing and uses previous settings for wrapping.
        :type wrap_lines:                   (bool)
        :param horizontal_scroll:           Controls if a horizontal scrollbar should be shown. If True, then line wrapping will be off by default
        :type horizontal_scroll:            (bool)
        :param sbar_trough_color:           Scrollbar color of the trough
        :type sbar_trough_color:            (str)
        :param sbar_background_color:       Scrollbar color of the background of the arrow buttons at the ends AND the color of the "thumb" (the thing you grab and slide). Switches to arrow color when mouse is over
        :type sbar_background_color:        (str)
        :param sbar_arrow_color:            Scrollbar color of the arrow at the ends of the scrollbar (it looks like a button). Switches to background color when mouse is over
        :type sbar_arrow_color:             (str)
        :param sbar_width:                  Scrollbar width in pixels
        :type sbar_width:                   (int)
        :param sbar_arrow_width:            Scrollbar width of the arrow on the scrollbar. It will potentially impact the overall width of the scrollbar
        :type sbar_arrow_width:             (int)
        :param sbar_frame_color:            Scrollbar Color of frame around scrollbar (available only on some ttk themes)
        :type sbar_frame_color:             (str)
        :param sbar_relief:                 Scrollbar relief that will be used for the "thumb" of the scrollbar (the thing you grab that slides). Should be a constant that is defined at starting with "RELIEF_" - RELIEF_RAISED, RELIEF_SUNKEN, RELIEF_FLAT, RELIEF_RIDGE, RELIEF_GROOVE, RELIEF_SOLID
        :type sbar_relief:                  (str)
        """


        super().__init__(size=size, s=s, background_color=background_color, autoscroll_only_at_bottom=autoscroll_only_at_bottom, text_color=text_color, pad=pad, p=p, echo_stdout_stderr=echo_stdout_stderr, font=font, tooltip=tooltip, wrap_lines=wrap_lines, horizontal_scroll=horizontal_scroll, key=key, k=k, right_click_menu=right_click_menu, write_only=True, reroute_stdout=True, reroute_stderr=True, reroute_cprint=True, autoscroll=True, auto_refresh=True, expand_x=expand_x, expand_y=expand_y, visible=visible, metadata=metadata, sbar_trough_color=sbar_trough_color, sbar_background_color=sbar_background_color, sbar_arrow_color=sbar_arrow_color, sbar_width=sbar_width, sbar_arrow_width=sbar_arrow_width, sbar_frame_color=sbar_frame_color, sbar_relief=sbar_relief)



# ---------------------------------------------------------------------- #
#                           Button Class                                 #
# ---------------------------------------------------------------------- #
class Button(Element):
    """
    Button Element - Defines all possible buttons. The shortcuts such as Submit, FileBrowse, ... each create a Button
    """

    def __init__(self, button_text='', button_type=BUTTON_TYPE_READ_FORM, target=(None, None), tooltip=None,
                 file_types=FILE_TYPES_ALL_FILES, initial_folder=None, default_extension='', disabled=False, change_submits=False,
                 enable_events=False, image_filename=None, image_data=None, image_size=(None, None),
                 image_subsample=None, image_zoom=None, image_source=None, border_width=None, size=(None, None), s=(None, None), auto_size_button=None, button_color=None,
                 disabled_button_color=None,
                 highlight_colors=None, mouseover_colors=(None, None), use_ttk_buttons=None, font=None, bind_return_key=False, focus=False, pad=None, p=None, key=None,
                 k=None, right_click_menu=None, expand_x=False, expand_y=False, visible=True, metadata=None):
        """
        :param button_text:           Text to be displayed on the button
        :type button_text:            (str)
        :param button_type:           You  should NOT be setting this directly. ONLY the shortcut functions set this
        :type button_type:            (int)
        :param target:                key or (row,col) target for the button. Note that -1 for column means 1 element to the left of this one. The constant ThisRow is used to indicate the current row. The Button itself is a valid target for some types of button
        :type target:                 str | (int, int)
        :param tooltip:               text, that will appear when mouse hovers over the element
        :type tooltip:                (str)
        :param file_types:            the filetypes that will be used to match files. To indicate all files: (("ALL Files", "*.* *"),).
        :type file_types:             Tuple[(str, str), ...]
        :param initial_folder:        starting path for folders and files
        :type initial_folder:         (str)
        :param default_extension:     If no extension entered by user, add this to filename (only used in saveas dialogs)
        :type default_extension:      (str)
        :param disabled:              If True button will be created disabled. If BUTTON_DISABLED_MEANS_IGNORE then the button will be ignored rather than disabled using tkinter
        :type disabled:               (bool | str)
        :param change_submits:        DO NOT USE. Only listed for backwards compat - Use enable_events instead
        :type change_submits:         (bool)
        :param enable_events:         Turns on the element specific events. If this button is a target, should it generate an event when filled in
        :type enable_events:          (bool)
        :param image_source:          Image to place on button. Use INSTEAD of the image_filename and image_data. Unifies these into 1 easier to use parm
        :type image_source:           (str | bytes)
        :param image_filename:        image filename if there is a button image. GIFs and PNGs only.
        :type image_filename:         (str)
        :param image_data:            Raw or Base64 representation of the image to put on button. Choose either filename or data
        :type image_data:             bytes | str
        :param image_size:            Size of the image in pixels (width, height)
        :type image_size:             (int, int)
        :param image_subsample:       amount to reduce the size of the image. Divides the size by this number. 2=1/2, 3=1/3, 4=1/4, etc
        :type image_subsample:        (int)
        :param image_zoom:            amount to increase the size of the image. 2=twice size, 3=3 times, etc
        :type image_zoom:             (int)
        :param border_width:          width of border around button in pixels
        :type border_width:           (int)
        :param size:                  (w, h) w=characters-wide, h=rows-high. If an int instead of a tuple is supplied, then height is auto-set to 1
        :type size:                   (int | None, int | None)  | (None, None) | int
        :param s:                     Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
        :type s:                      (int | None, int | None)  | (None, None) | int
        :param auto_size_button:      if True the button size is sized to fit the text
        :type auto_size_button:       (bool)
        :param button_color:          Color of button. default is from theme or the window. Easy to remember which is which if you say "ON" between colors. "red" on "green". Normally a tuple, but can be a simplified-button-color-string "foreground on background". Can be a single color if want to set only the background.
        :type button_color:           (str, str) | str
        :param disabled_button_color: colors to use when button is disabled (text, background). Use None for a color if don't want to change. Only ttk buttons support both text and background colors. tk buttons only support changing text color
        :type disabled_button_color:  (str, str) | str
        :param highlight_colors:      colors to use when button has focus (has focus, does not have focus). None will use colors based on theme. Only used by Linux and only for non-TTK button
        :type highlight_colors:       (str, str)
        :param mouseover_colors:      Important difference between Linux & Windows! Linux - Colors when mouse moved over button.  Windows - colors when button is pressed. The default is to switch the text and background colors (an inverse effect)
        :type mouseover_colors:       (str, str) | str
        :param use_ttk_buttons:       True = use ttk buttons. False = do not use ttk buttons.  None (Default) = use ttk buttons only if on a Mac and not with button images
        :type use_ttk_buttons:        (bool)
        :param font:                  specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:                   (str or (str, int[, str]) or None)
        :param bind_return_key:       If True then pressing the return key in an Input or Multiline Element will cause this button to appear to be clicked (generates event with this button's key
        :type bind_return_key:        (bool)
        :param focus:                 if True, initial focus will be put on this button
        :type focus:                  (bool)
        :param pad:                   Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:                    (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                     Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                      (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param key:                   Used with window.find_element and with return values to uniquely identify this element to uniquely identify this element
        :type key:                    str | int | tuple | object
        :param k:                     Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                      str | int | tuple | object
        :param right_click_menu:      A list of lists of Menu items to show when this element is right clicked. See user docs for exact format.
        :type right_click_menu:       List[List[ List[str] | str ]]
        :param expand_x:              If True the element will automatically expand in the X direction to fill available space
        :type expand_x:               (bool)
        :param expand_y:              If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:               (bool)
        :param visible:               set visibility state of the element
        :type visible:                (bool)
        :param metadata:              User metadata that can be set to ANYTHING
        :type metadata:               (Any)
        """


        self.AutoSizeButton = auto_size_button
        self.BType = button_type
        if file_types is not None and len(file_types) == 2 and isinstance(file_types[0], str) and isinstance(file_types[1], str):
            warnings.warn('file_types parameter not correctly specified. This parameter is a LIST of TUPLES. You have passed (str,str) rather than ((str, str),). Fixing it for you this time.\nchanging {} to {}\nPlease correct your code'.format(file_types, ((file_types[0], file_types[1]),)), UserWarning)
            file_types = ((file_types[0], file_types[1]),)
        self.FileTypes = file_types
        self.Widget = self.TKButton = None  # type: tk.Button
        self.Target = target
        self.ButtonText = str(button_text)
        self.RightClickMenu = right_click_menu
        # Button colors can be a tuple (text, background) or a string with format "text on background"
        # bc = button_color
        # if button_color is None:
        #     bc = DEFAULT_BUTTON_COLOR
        # else:
        #     try:
        #         if isinstance(button_color,str):
        #             bc = button_color.split(' on ')
        #     except Exception as e:
        #         print('* cprint warning * you messed up with color formatting', e)
        #     if bc[1] is None:
        #         bc = (bc[0], theme_button_color()[1])
        # self.ButtonColor = bc
        self.ButtonColor = button_color_to_tuple(button_color)

        # experimental code to compute disabled button text color
        # if disabled_button_color is None:
        #     try:
        #         disabled_button_color = (get_complimentary_hex(theme_button_color()[0]), theme_button_color()[1])
        #         # disabled_button_color = disabled_button_color
        #     except:
        #         print('* Problem computing disabled button color *')
        self.DisabledButtonColor = button_color_to_tuple(disabled_button_color) if disabled_button_color is not None else (None, None)
        if image_source is not None:
            if isinstance(image_source, bytes):
                image_data = image_source
            elif isinstance(image_source, str):
                image_filename = image_source
        self.ImageFilename = image_filename
        self.ImageData = image_data
        self.ImageSize = image_size
        self.ImageSubsample = image_subsample
        self.zoom = int(image_zoom) if image_zoom is not None else None
        self.UserData = None
        self.BorderWidth = border_width if border_width is not None else DEFAULT_BORDER_WIDTH
        self.BindReturnKey = bind_return_key
        self.Focus = focus
        self.TKCal = None
        self.calendar_default_date_M_D_Y = (None, None, None)
        self.calendar_close_when_chosen = False
        self.calendar_locale = None
        self.calendar_format = None
        self.calendar_location = (None, None)
        self.calendar_no_titlebar = True
        self.calendar_begin_at_sunday_plus = 0
        self.calendar_month_names = None
        self.calendar_day_abbreviations = None
        self.calendar_title = ''
        self.calendar_selection = ''
        self.default_button = None
        self.InitialFolder = initial_folder
        self.DefaultExtension = default_extension
        self.Disabled = disabled
        self.ChangeSubmits = change_submits or enable_events
        self.UseTtkButtons = use_ttk_buttons
        self._files_delimiter = BROWSE_FILES_DELIMITER  # used by the file browse button. used when multiple files are selected by user
        if use_ttk_buttons is None and running_mac():
            self.UseTtkButtons = True
        # if image_filename or image_data:
        #     self.UseTtkButtons = False              # if an image is to be displayed, then force the button to not be a TTK Button
        if key is None and k is None:
            _key = self.ButtonText
            if DEFAULT_USE_BUTTON_SHORTCUTS is True:
                pos = _key.find(MENU_SHORTCUT_CHARACTER)
                if pos != -1:
                    if pos < len(MENU_SHORTCUT_CHARACTER) or _key[pos - len(MENU_SHORTCUT_CHARACTER)] != '\\':
                        _key = _key[:pos] + _key[pos + len(MENU_SHORTCUT_CHARACTER):]
                    else:
                        _key = _key.replace('\\'+MENU_SHORTCUT_CHARACTER, MENU_SHORTCUT_CHARACTER)
        else:
            _key = key if key is not None else k
        if highlight_colors is not None:
            self.HighlightColors = highlight_colors
        else:
            self.HighlightColors = self._compute_highlight_colors()

        if mouseover_colors != (None, None):
            self.MouseOverColors = button_color_to_tuple(mouseover_colors)
        elif button_color != None:
            self.MouseOverColors = (self.ButtonColor[1], self.ButtonColor[0])
        else:
            self.MouseOverColors = (theme_button_color()[1], theme_button_color()[0])
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y

        sz = size if size != (None, None) else s
        super().__init__(ELEM_TYPE_BUTTON, size=sz, font=font, pad=pad, key=_key, tooltip=tooltip, visible=visible, metadata=metadata)
        return

    def _compute_highlight_colors(self):
        """
        Determines the color to use to indicate the button has focus. This setting is only used by Linux.
        :return: Pair of colors. (Highlight, Highlight Background)
        :rtype:  (str, str)
        """
        highlight_color = highlight_background = COLOR_SYSTEM_DEFAULT
        if self.ButtonColor != COLOR_SYSTEM_DEFAULT and theme_background_color() != COLOR_SYSTEM_DEFAULT:
            highlight_background = theme_background_color()
        if self.ButtonColor != COLOR_SYSTEM_DEFAULT and self.ButtonColor[0] != COLOR_SYSTEM_DEFAULT:
            if self.ButtonColor[0] != theme_background_color():
                highlight_color = self.ButtonColor[0]
            else:
                highlight_color = 'red'
        return (highlight_color, highlight_background)

        # Realtime button release callback

    def ButtonReleaseCallBack(self, parm):
        """
        Not a user callable function.  Called by tkinter when a "realtime" button is released

        :param parm: the event info from tkinter
        :type parm:

        """
        self.LastButtonClickedWasRealtime = False
        self.ParentForm.LastButtonClicked = None

    # Realtime button callback
    def ButtonPressCallBack(self, parm):
        """
        Not a user callable method. Callback called by tkinter when a "realtime" button is pressed

        :param parm: Event info passed in by tkinter
        :type parm:

        """
        self.ParentForm.LastButtonClickedWasRealtime = True
        if self.Key is not None:
            self.ParentForm.LastButtonClicked = self.Key
        else:
            self.ParentForm.LastButtonClicked = self.ButtonText
        # if self.ParentForm.CurrentlyRunningMainloop:
        #     Window._window_that_exited = self.ParentForm
        #     self.ParentForm.TKroot.quit()  # kick out of loop if read was called
        _exit_mainloop(self.ParentForm)

    def _find_target(self):
        target = self.Target
        target_element = None

        if target[0] == ThisRow:
            target = [self.Position[0], target[1]]
            if target[1] < 0:
                target[1] = self.Position[1] + target[1]
        strvar = None
        should_submit_window = False
        if target == (None, None):
            strvar = self.TKStringVar
        else:
            # Need a try-block because if the target is not hashable, the "in" test will raise exception
            try:
                if target in self.ParentForm.AllKeysDict:
                    target_element = self.ParentForm.AllKeysDict[target]
            except:
                pass
            # if target not found or the above try got exception, then keep looking....
            if target_element is None:
                if not isinstance(target, str):
                    if target[0] < 0:
                        target = [self.Position[0] + target[0], target[1]]
                    target_element = self.ParentContainer._GetElementAtLocation(target)
                else:
                    target_element = self.ParentForm.find_element(target)
            try:
                strvar = target_element.TKStringVar
            except:
                pass
            try:
                if target_element.ChangeSubmits:
                    should_submit_window = True
            except:
                pass
        return target_element, strvar, should_submit_window

    # -------  Button Callback  ------- #
    def ButtonCallBack(self):
        """
        Not user callable! Called by tkinter when a button is clicked.  This is where all the fun begins!
        """

        if self.Disabled == BUTTON_DISABLED_MEANS_IGNORE:
            return
        target_element, strvar, should_submit_window = self._find_target()

        filetypes = FILE_TYPES_ALL_FILES if self.FileTypes is None else self.FileTypes

        if self.BType == BUTTON_TYPE_BROWSE_FOLDER:
            if running_mac():  # macs don't like seeing the parent window (go firgure)
                folder_name = tk.filedialog.askdirectory(initialdir=self.InitialFolder)  # show the 'get folder' dialog box
            else:
                folder_name = tk.filedialog.askdirectory(initialdir=self.InitialFolder, parent=self.ParentForm.TKroot)  # show the 'get folder' dialog box
            if folder_name:
                try:
                    strvar.set(folder_name)
                    self.TKStringVar.set(folder_name)
                except:
                    pass
            else:  # if "cancel" button clicked, don't generate an event
                should_submit_window = False
        elif self.BType == BUTTON_TYPE_BROWSE_FILE:
            if running_mac():
                # Workaround for the "*.*" issue on Mac
                is_all = [(x, y) for (x, y) in filetypes if all(ch in '* .' for ch in y)]
                if not len(set(filetypes)) > 1 and (len(is_all) != 0 or filetypes == FILE_TYPES_ALL_FILES):
                    file_name = tk.filedialog.askopenfilename(initialdir=self.InitialFolder)
                else:
                    file_name = tk.filedialog.askopenfilename(initialdir=self.InitialFolder, filetypes=filetypes)  # show the 'get file' dialog box
                # elif _mac_allow_filetypes():
                    # file_name = tk.filedialog.askopenfilename(initialdir=self.InitialFolder, filetypes=filetypes)  # show the 'get file' dialog box
                # else:
                #     file_name = tk.filedialog.askopenfilename(initialdir=self.InitialFolder)  # show the 'get file' dialog box
            else:
                file_name = tk.filedialog.askopenfilename(filetypes=filetypes, initialdir=self.InitialFolder, parent=self.ParentForm.TKroot)  # show the 'get file' dialog box

            if file_name:
                strvar.set(file_name)
                self.TKStringVar.set(file_name)
            else:           # if "cancel" button clicked, don't generate an event
                should_submit_window = False
        elif self.BType == BUTTON_TYPE_COLOR_CHOOSER:
            color = tk.colorchooser.askcolor(parent=self.ParentForm.TKroot, color=self.default_color)  # show the 'get file' dialog box
            color = color[1]  # save only the #RRGGBB portion
            if color is not None:
                strvar.set(color)
                self.TKStringVar.set(color)
        elif self.BType == BUTTON_TYPE_BROWSE_FILES:
            if running_mac():
                # Workaround for the "*.*" issue on Mac
                is_all = [(x, y) for (x, y) in filetypes if all(ch in '* .' for ch in y)]
                if not len(set(filetypes)) > 1 and (len(is_all) != 0 or filetypes == FILE_TYPES_ALL_FILES):
                    file_name = tk.filedialog.askopenfilenames(initialdir=self.InitialFolder)
                else:
                    file_name = tk.filedialog.askopenfilenames(filetypes=filetypes, initialdir=self.InitialFolder)
                # elif _mac_allow_filetypes():
                #     file_name = tk.filedialog.askopenfilenames(filetypes=filetypes, initialdir=self.InitialFolder)
                # else:
                #     file_name = tk.filedialog.askopenfilenames(initialdir=self.InitialFolder)
            else:
                file_name = tk.filedialog.askopenfilenames(filetypes=filetypes, initialdir=self.InitialFolder, parent=self.ParentForm.TKroot)

            if file_name:
                file_name = self._files_delimiter.join(file_name)  # normally a ';'
                strvar.set(file_name)
                self.TKStringVar.set(file_name)
            else:           # if "cancel" button clicked, don't generate an event
                should_submit_window = False
        elif self.BType == BUTTON_TYPE_SAVEAS_FILE:
            # show the 'get file' dialog box
            if running_mac():
                # Workaround for the "*.*" issue on Mac
                is_all = [(x, y) for (x, y) in filetypes if all(ch in '* .' for ch in y)]
                if not len(set(filetypes)) > 1 and (len(is_all) != 0 or filetypes == FILE_TYPES_ALL_FILES):
                    file_name = tk.filedialog.asksaveasfilename(defaultextension=self.DefaultExtension, initialdir=self.InitialFolder)
                else:
                    file_name = tk.filedialog.asksaveasfilename(filetypes=filetypes, defaultextension=self.DefaultExtension, initialdir=self.InitialFolder)
                # elif _mac_allow_filetypes():
                #     file_name = tk.filedialog.asksaveasfilename(filetypes=filetypes, defaultextension=self.DefaultExtension, initialdir=self.InitialFolder)
                # else:
                #     file_name = tk.filedialog.asksaveasfilename(defaultextension=self.DefaultExtension, initialdir=self.InitialFolder)
            else:
                file_name = tk.filedialog.asksaveasfilename(filetypes=filetypes, defaultextension=self.DefaultExtension, initialdir=self.InitialFolder, parent=self.ParentForm.TKroot)

            if file_name:
                strvar.set(file_name)
                self.TKStringVar.set(file_name)
            else:           # if "cancel" button clicked, don't generate an event
                should_submit_window = False
        elif self.BType == BUTTON_TYPE_CLOSES_WIN:  # this is a return type button so GET RESULTS and destroy window
            # first, get the results table built
            # modify the Results table in the parent FlexForm object
            if self.Key is not None:
                self.ParentForm.LastButtonClicked = self.Key
            else:
                self.ParentForm.LastButtonClicked = self.ButtonText
            self.ParentForm.FormRemainedOpen = False
            self.ParentForm._Close()
            _exit_mainloop(self.ParentForm)

            if self.ParentForm.NonBlocking:
                self.ParentForm.TKroot.destroy()
                Window._DecrementOpenCount()
        elif self.BType == BUTTON_TYPE_READ_FORM:  # LEAVE THE WINDOW OPEN!! DO NOT CLOSE
            # This is a PLAIN BUTTON
            # first, get the results table built
            # modify the Results table in the parent FlexForm object
            if self.Key is not None:
                self.ParentForm.LastButtonClicked = self.Key
            else:
                self.ParentForm.LastButtonClicked = self.ButtonText
            self.ParentForm.FormRemainedOpen = True
            _exit_mainloop(self.ParentForm)
        elif self.BType == BUTTON_TYPE_CLOSES_WIN_ONLY:  # special kind of button that does not exit main loop
            self.ParentForm._Close(without_event=True)
            self.ParentForm.TKroot.destroy()  # close the window with tkinter
            Window._DecrementOpenCount()
        elif self.BType == BUTTON_TYPE_CALENDAR_CHOOSER:  # this is a return type button so GET RESULTS and destroy window
            # ------------ new chooser code -------------
            self.ParentForm.LastButtonClicked = self.Key  # key should have been generated already if not set by user
            self.ParentForm.FormRemainedOpen = True
            should_submit_window = False
            _exit_mainloop(self.ParentForm)
        # elif self.BType == BUTTON_TYPE_SHOW_DEBUGGER:
            # **** DEPRICATED *****
            # if self.ParentForm.DebuggerEnabled:
                # show_debugger_popout_window()

        if should_submit_window:
            self.ParentForm.LastButtonClicked = target_element.Key
            self.ParentForm.FormRemainedOpen = True
            _exit_mainloop(self.ParentForm)

        return

    def update(self, text=None, button_color=(None, None), disabled=None, image_source=None, image_data=None, image_filename=None,
               visible=None, image_subsample=None, image_zoom=None, disabled_button_color=(None, None), image_size=None):
        """
        Changes some of the settings for the Button Element. Must call `Window.Read` or `Window.Finalize` prior

        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param text:                  sets button text
        :type text:                   (str)
        :param button_color:          Color of button. default is from theme or the window. Easy to remember which is which if you say "ON" between colors. "red" on "green". Normally a tuple, but can be a simplified-button-color-string "foreground on background". Can be a single color if want to set only the background.
        :type button_color:           (str, str) | str
        :param disabled:              True/False to enable/disable at the GUI level. Use BUTTON_DISABLED_MEANS_IGNORE to ignore clicks (won't change colors)
        :type disabled:               (bool | str)
        :param image_source:          Image to place on button. Use INSTEAD of the image_filename and image_data. Unifies these into 1 easier to use parm
        :type image_source:           (str | bytes)
        :param image_data:            Raw or Base64 representation of the image to put on button. Choose either filename or data
        :type image_data:             bytes | str
        :param image_filename:        image filename if there is a button image. GIFs and PNGs only.
        :type image_filename:         (str)
        :param disabled_button_color: colors to use when button is disabled (text, background). Use None for a color if don't want to change. Only ttk buttons support both text and background colors. tk buttons only support changing text color
        :type disabled_button_color:  (str, str)
        :param visible:               control visibility of element
        :type visible:                (bool)
        :param image_subsample:       amount to reduce the size of the image. Divides the size by this number. 2=1/2, 3=1/3, 4=1/4, etc
        :type image_subsample:        (int)
        :param image_zoom:            amount to increase the size of the image. 2=twice size, 3=3 times, etc
        :type image_zoom:             (int)
        :param image_size:            Size of the image in pixels (width, height)
        :type image_size:             (int, int)
        """

        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in Button.update - The window was closed')
            return

        if image_source is not None:
            if isinstance(image_source, bytes):
                image_data = image_source
            elif isinstance(image_source, str):
                image_filename = image_source

        if self.UseTtkButtons:
            style_name = self.ttk_style_name        # created when made initial window (in the pack)
            # style_name = str(self.Key) + 'custombutton.TButton'
            button_style = ttk.Style()
        if text is not None:
            btext = text
            if DEFAULT_USE_BUTTON_SHORTCUTS is True:
                pos = btext.find(MENU_SHORTCUT_CHARACTER)
                if pos != -1:
                    if pos < len(MENU_SHORTCUT_CHARACTER) or btext[pos - len(MENU_SHORTCUT_CHARACTER)] != '\\':
                        btext = btext[:pos] + btext[pos + len(MENU_SHORTCUT_CHARACTER):]
                    else:
                        btext = btext.replace('\\'+MENU_SHORTCUT_CHARACTER, MENU_SHORTCUT_CHARACTER)
                        pos = -1
                if pos != -1:
                    self.TKButton.config(underline=pos)
            self.TKButton.configure(text=btext)
            self.ButtonText = text
        if button_color != (None, None) and button_color != COLOR_SYSTEM_DEFAULT:
            bc = button_color_to_tuple(button_color, self.ButtonColor)
            # if isinstance(button_color, str):
            #     try:
            #         button_color = button_color.split(' on ')
            #     except Exception as e:
            #         print('** Error in formatting your button color **', button_color, e)
            if self.UseTtkButtons:
                if bc[0] not in (None, COLOR_SYSTEM_DEFAULT):
                    button_style.configure(style_name, foreground=bc[0])
                if bc[1] not in (None, COLOR_SYSTEM_DEFAULT):
                    button_style.configure(style_name, background=bc[1])
            else:
                if bc[0] not in (None, COLOR_SYSTEM_DEFAULT):
                    self.TKButton.config(foreground=bc[0], activebackground=bc[0])
                if bc[1] not in (None, COLOR_SYSTEM_DEFAULT):
                    self.TKButton.config(background=bc[1], activeforeground=bc[1])
            self.ButtonColor = bc
        if disabled is True:
            self.TKButton['state'] = 'disabled'
        elif disabled is False:
            self.TKButton['state'] = 'normal'
        elif disabled == BUTTON_DISABLED_MEANS_IGNORE:
            self.TKButton['state'] = 'normal'
        self.Disabled = disabled if disabled is not None else self.Disabled

        if image_data is not None:
            image = tk.PhotoImage(data=image_data)
            if image_subsample:
                image = image.subsample(image_subsample)
            if image_zoom is not None:
                image = image.zoom(int(image_zoom))
            if image_size is not None:
                width, height = image_size
            else:
                width, height = image.width(), image.height()
            if self.UseTtkButtons:
                button_style.configure(style_name, image=image, width=width, height=height)
            else:
                self.TKButton.config(image=image, width=width, height=height)
            self.TKButton.image = image
        if image_filename is not None:
            image = tk.PhotoImage(file=image_filename)
            if image_subsample:
                image = image.subsample(image_subsample)
            if image_zoom is not None:
                image = image.zoom(int(image_zoom))
            if image_size is not None:
                width, height = image_size
            else:
                width, height = image.width(), image.height()
            if self.UseTtkButtons:
                button_style.configure(style_name, image=image, width=width, height=height)
            else:
                self.TKButton.config(highlightthickness=0, image=image, width=width, height=height)
            self.TKButton.image = image
        if visible is False:
            self._pack_forget_save_settings()
        elif visible is True:
            self._pack_restore_settings()
        if disabled_button_color != (None, None) and disabled_button_color != COLOR_SYSTEM_DEFAULT:
            if not self.UseTtkButtons:
                self.TKButton['disabledforeground'] = disabled_button_color[0]
            else:
                if disabled_button_color[0] is not None:
                    button_style.map(style_name, foreground=[('disabled', disabled_button_color[0])])
                if disabled_button_color[1] is not None:
                    button_style.map(style_name, background=[('disabled', disabled_button_color[1])])
            self.DisabledButtonColor = (disabled_button_color[0] if disabled_button_color[0] is not None else self.DisabledButtonColor[0],
                                        disabled_button_color[1] if disabled_button_color[1] is not None else self.DisabledButtonColor[1])

        if visible is not None:
            self._visible = visible

    def get_text(self):
        """
        Returns the current text shown on a button

        :return: The text currently displayed on the button
        :rtype:  (str)
        """
        return self.ButtonText

    def click(self):
        """
        Generates a click of the button as if the user clicked the button
        Calls the tkinter invoke method for the button
        """
        try:
            self.TKButton.invoke()
        except:
            print('Exception clicking button')

    Click = click
    GetText = get_text
    Update = update


# -------------------------  Button lazy functions  ------------------------- #
B = Button
Btn = Button


# ---------------------------------------------------------------------- #
#                           ButtonMenu Class                             #
# ---------------------------------------------------------------------- #
class ButtonMenu(Element):
    """
    The Button Menu Element.  Creates a button that when clicked will show a menu similar to right click menu
    """

    def __init__(self, button_text, menu_def, tooltip=None, disabled=False, image_source=None,
                 image_filename=None, image_data=None, image_size=(None, None), image_subsample=None, image_zoom=None, border_width=None,
                 size=(None, None), s=(None, None), auto_size_button=None, button_color=None, text_color=None, background_color=None, disabled_text_color=None,
                 font=None, item_font=None, pad=None, p=None, expand_x=False, expand_y=False, key=None, k=None, tearoff=False, visible=True,
                 metadata=None):
        """
        :param button_text:               Text to be displayed on the button
        :type button_text:                (str)
        :param menu_def:                  A list of lists of Menu items to show when this element is clicked. See docs for format as they are the same for all menu types
        :type menu_def:                   List[List[str]]
        :param tooltip:                   text, that will appear when mouse hovers over the element
        :type tooltip:                    (str)
        :param disabled:                  If True button will be created disabled
        :type disabled:                   (bool)
        :param image_source:              Image to place on button. Use INSTEAD of the image_filename and image_data. Unifies these into 1 easier to use parm
        :type image_source:               (str | bytes)
        :param image_filename:            image filename if there is a button image. GIFs and PNGs only.
        :type image_filename:             (str)
        :param image_data:                Raw or Base64 representation of the image to put on button. Choose either filename or data
        :type image_data:                 bytes | str
        :param image_size:                Size of the image in pixels (width, height)
        :type image_size:                 (int, int)
        :param image_subsample:           amount to reduce the size of the image. Divides the size by this number. 2=1/2, 3=1/3, 4=1/4, etc
        :type image_subsample:            (int)
        :param image_zoom:                amount to increase the size of the image. 2=twice size, 3=3 times, etc
        :type image_zoom:                 (int)
        :param border_width:              width of border around button in pixels
        :type border_width:               (int)
        :param size:                      (w, h) w=characters-wide, h=rows-high. If an int instead of a tuple is supplied, then height is auto-set to 1
        :type size:                       (int, int)  | (None, None) | int
        :param s:                         Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
        :type s:                          (int, int)  | (None, None) | int
        :param auto_size_button:          if True the button size is sized to fit the text
        :type auto_size_button:           (bool)
        :param button_color:              of button. Easy to remember which is which if you say "ON" between colors. "red" on "green"
        :type button_color:               (str, str) | str
        :param background_color:          color of the background
        :type background_color:           (str)
        :param text_color:                element's text color. Can be in #RRGGBB format or a color name "black"
        :type text_color:                 (str)
        :param disabled_text_color:       color to use for text when item is disabled. Can be in #RRGGBB format or a color name "black"
        :type disabled_text_color:        (str)
        :param font:                      specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:                       (str or (str, int[, str]) or None)
        :param item_font:                 specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike, for the menu items
        :type item_font:                  (str or (str, int[, str]) or None)
        :param pad:                       Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:                        (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                         Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                          (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param expand_x:                  If True the element will automatically expand in the X direction to fill available space
        :type expand_x:                   (bool)
        :param expand_y:                  If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:                   (bool)
        :param key:                       Used with window.find_element and with return values to uniquely identify this element to uniquely identify this element
        :type key:                        str | int | tuple | object
        :param k:                         Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                          str | int | tuple | object
        :param tearoff:                   Determines if menus should allow them to be torn off
        :type tearoff:                    (bool)
        :param visible:                   set visibility state of the element
        :type visible:                    (bool)
        :param metadata:                  User metadata that can be set to ANYTHING
        :type metadata:                   (Any)
        """


        self.MenuDefinition = copy.deepcopy(menu_def)

        self.AutoSizeButton = auto_size_button
        self.ButtonText = button_text
        self.ButtonColor = button_color_to_tuple(button_color)
        # self.TextColor = self.ButtonColor[0]
        # self.BackgroundColor = self.ButtonColor[1]
        self.BackgroundColor = background_color if background_color is not None else theme_input_background_color()
        self.TextColor = text_color if text_color is not None else theme_input_text_color()
        self.DisabledTextColor = disabled_text_color if disabled_text_color is not None else COLOR_SYSTEM_DEFAULT
        self.ItemFont = item_font
        self.BorderWidth = border_width if border_width is not None else DEFAULT_BORDER_WIDTH
        if image_source is not None:
            if isinstance(image_source, str):
                image_filename = image_source
            elif isinstance(image_source, bytes):
                image_data = image_source
            else:
                warnings.warn('ButtonMenu element - image_source is not a valid type: {}'.format(type(image_source)), UserWarning)

        self.ImageFilename = image_filename
        self.ImageData = image_data
        self.ImageSize = image_size
        self.ImageSubsample = image_subsample
        self.zoom = int(image_zoom) if image_zoom is not None else None
        self.Disabled = disabled
        self.IsButtonMenu = True
        self.MenuItemChosen = None
        self.Widget = self.TKButtonMenu = None  # type: tk.Menubutton
        self.TKMenu = None  # type: tk.Menu
        self.part_of_custom_menubar = False
        self.custom_menubar_key = None
        # self.temp_size = size if size != (NONE, NONE) else
        key = key if key is not None else k
        sz = size if size != (None, None) else s
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y

        super().__init__(ELEM_TYPE_BUTTONMENU, size=sz, font=font, pad=pad, key=key, tooltip=tooltip,
                         text_color=self.TextColor, background_color=self.BackgroundColor, visible=visible, metadata=metadata)
        self.Tearoff = tearoff


    def _MenuItemChosenCallback(self, item_chosen):  # ButtonMenu Menu Item Chosen Callback
        """
        Not a user callable function.  Called by tkinter when an item is chosen from the menu.

        :param item_chosen: The menu item chosen.
        :type item_chosen:  (str)
        """
        # print('IN MENU ITEM CALLBACK', item_chosen)
        self.MenuItemChosen = item_chosen
        self.ParentForm.LastButtonClicked = self.Key
        self.ParentForm.FormRemainedOpen = True
        # if self.ParentForm.CurrentlyRunningMainloop:
        #     self.ParentForm.TKroot.quit()  # kick the users out of the mainloop
        _exit_mainloop(self.ParentForm)


    def update(self, menu_definition=None, visible=None, image_source=None, image_size=(None, None), image_subsample=None, image_zoom=None, button_text=None, button_color=None):
        """
        Changes some of the settings for the ButtonMenu Element. Must call `Window.Read` or `Window.Finalize` prior

        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param menu_definition: (New menu definition (in menu definition format)
        :type menu_definition:  List[List]
        :param visible:         control visibility of element
        :type visible:          (bool)
        :param image_source:    new image if image is to be changed. Can be a filename or a base64 encoded byte-string
        :type image_source:     (str | bytes)
        :param image_size:      Size of the image in pixels (width, height)
        :type image_size:       (int, int)
        :param image_subsample: amount to reduce the size of the image. Divides the size by this number. 2=1/2, 3=1/3, 4=1/4, etc
        :type image_subsample:  (int)
        :param image_zoom:      amount to increase the size of the image. 2=twice size, 3=3 times, etc
        :type image_zoom:       (int)
        :param button_text:     Text to be shown on the button
        :type button_text:      (str)
        :param button_color:    Normally a tuple, but can be a simplified-button-color-string "foreground on background". Can be a single color if want to set only the background.
        :type button_color:     (str, str) | str
        """

        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in ButtonMenu.update - The window was closed')
            return


        if menu_definition is not None:
            self.MenuDefinition = copy.deepcopy(menu_definition)
            top_menu = self.TKMenu = tk.Menu(self.TKButtonMenu, tearoff=self.Tearoff, font=self.ItemFont, tearoffcommand=self._tearoff_menu_callback)

            if self.BackgroundColor not in (COLOR_SYSTEM_DEFAULT, None):
                top_menu.config(bg=self.BackgroundColor)
            if self.TextColor not in (COLOR_SYSTEM_DEFAULT, None):
                top_menu.config(fg=self.TextColor)
            if self.DisabledTextColor not in (COLOR_SYSTEM_DEFAULT, None):
                top_menu.config(disabledforeground=self.DisabledTextColor)
            if self.ItemFont is not None:
                top_menu.config(font=self.ItemFont)
            AddMenuItem(self.TKMenu, self.MenuDefinition[1], self)
            self.TKButtonMenu.configure(menu=self.TKMenu)
        if image_source is not None:
            filename = data = None
            if image_source is not None:
                if isinstance(image_source, bytes):
                    data = image_source
                elif isinstance(image_source, str):
                    filename = image_source
                else:
                    warnings.warn('ButtonMenu element - image_source is not a valid type: {}'.format(type(image_source)), UserWarning)
            image = None
            if filename is not None:
                image = tk.PhotoImage(file=filename)
                if image_subsample is not None:
                    image = image.subsample(image_subsample)
                if image_zoom is not None:
                    image = image.zoom(int(image_zoom))
            elif data is not None:
                # if type(data) is bytes:
                try:
                    image = tk.PhotoImage(data=data)
                    if image_subsample is not None:
                        image = image.subsample(image_subsample)
                    if image_zoom is not None:
                        image = image.zoom(int(image_zoom))
                except Exception as e:
                    image = data

            if image is not None:
                if type(image) is not bytes:
                    width, height = image_size[0] if image_size[0] is not None else image.width(), image_size[1] if image_size[1] is not None else image.height()
                else:
                    width, height = image_size

                self.TKButtonMenu.config(image=image, compound=tk.CENTER, width=width, height=height)
                self.TKButtonMenu.image = image
        if button_text is not None:
            self.TKButtonMenu.configure(text=button_text)
            self.ButtonText = button_text
        if visible is False:
            self._pack_forget_save_settings()
        elif visible is True:
            self._pack_restore_settings()
        if visible is not None:
            self._visible = visible
        if button_color != (None, None) and button_color != COLOR_SYSTEM_DEFAULT:
            bc = button_color_to_tuple(button_color, self.ButtonColor)
            if bc[0] not in (None, COLOR_SYSTEM_DEFAULT):
                self.TKButtonMenu.config(foreground=bc[0], activeforeground=bc[0])
            if bc[1] not in (None, COLOR_SYSTEM_DEFAULT):
                self.TKButtonMenu.config(background=bc[1], activebackground=bc[1])
            self.ButtonColor = bc

    def click(self):
        """
        Generates a click of the button as if the user clicked the button
        Calls the tkinter invoke method for the button
        """
        try:
            self.TKMenu.invoke(1)
        except:
            print('Exception clicking button')

    Update = update
    Click = click

BMenu = ButtonMenu
BM = ButtonMenu


# ---------------------------------------------------------------------- #
#                           ProgreessBar                                 #
# ---------------------------------------------------------------------- #
class ProgressBar(Element):
    """
    Progress Bar Element - Displays a colored bar that is shaded as progress of some operation is made
    """

    def __init__(self, max_value, orientation=None, size=(None, None), s=(None, None), size_px=(None, None), auto_size_text=None, bar_color=None, style=None, border_width=None,
                 relief=None, key=None, k=None, pad=None, p=None, right_click_menu=None, expand_x=False, expand_y=False, visible=True, metadata=None):
        """
        :param max_value:        max value of progressbar
        :type max_value:         (int)
        :param orientation:      'horizontal' or 'vertical'
        :type orientation:       (str)
        :param size:             Size of the bar.  If horizontal (chars long, pixels wide), vert (chars high, pixels wide). Vert height measured using horizontal chars units.
        :type size:              (int, int) |  (int, None)
        :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
        :type s:                 (int, int)  | (None, None)
        :param size_px:          Size in pixels (length, width). Will be used in place of size parm if specified
        :type size_px:           (int, int) | (None, None)
        :param auto_size_text:   Not sure why this is here
        :type auto_size_text:    (bool)
        :param bar_color:        The 2 colors that make up a progress bar. Either a tuple of 2 strings or a string. Tuple - (bar, background). A string with 1 color changes the background of the bar only. A string with 2 colors separated by "on" like "red on blue" specifies a red bar on a blue background.
        :type bar_color:         (str, str) or str
        :param style:            Progress bar style defined as one of these 'default', 'winnative', 'clam', 'alt', 'classic', 'vista', 'xpnative'
        :type style:             (str)
        :param border_width:     The amount of pixels that go around the outside of the bar
        :type border_width:      (int)
        :param relief:           relief style. Values are same as progress meter relief values.  Can be a constant or a string: `RELIEF_RAISED RELIEF_SUNKEN RELIEF_FLAT RELIEF_RIDGE RELIEF_GROOVE RELIEF_SOLID` (Default value = DEFAULT_PROGRESS_BAR_RELIEF)
        :type relief:            (str)
        :param key:              Used with window.find_element and with return values to uniquely identify this element to uniquely identify this element
        :type key:               str | int | tuple | object
        :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                 str | int | tuple | object
        :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param right_click_menu: A list of lists of Menu items to show when this element is right clicked. See user docs for exact format.
        :type right_click_menu:  List[List[ List[str] | str ]]
        :param expand_x:         If True the element will automatically expand in the X direction to fill available space
        :type expand_x:          (bool)
        :param expand_y:         If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:          (bool)
        :param visible:          set visibility state of the element
        :type visible:           (bool)
        :param metadata:         User metadata that can be set to ANYTHING
        :type metadata:          (Any)
        """


        self.MaxValue = max_value
        self.TKProgressBar = None  # type: TKProgressBar
        self.Cancelled = False
        self.NotRunning = True
        self.Orientation = orientation if orientation else DEFAULT_METER_ORIENTATION
        self.RightClickMenu = right_click_menu
        # Progress Bar colors can be a tuple (text, background) or a string with format "bar on background" - examples "red on white" or ("red", "white")
        if bar_color is None:
            bar_color = DEFAULT_PROGRESS_BAR_COLOR
        else:
            bar_color = _simplified_dual_color_to_tuple(bar_color, default=DEFAULT_PROGRESS_BAR_COLOR)

        self.BarColor = bar_color  # should be a tuple at this point
        self.BarStyle = style if style else DEFAULT_TTK_THEME
        self.BorderWidth = border_width if border_width else DEFAULT_PROGRESS_BAR_BORDER_WIDTH
        self.Relief = relief if relief else DEFAULT_PROGRESS_BAR_RELIEF
        self.BarExpired = False
        key = key if key is not None else k
        sz = size if size != (None, None) else s
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y
        self.size_px = size_px

        super().__init__(ELEM_TYPE_PROGRESS_BAR, size=sz, auto_size_text=auto_size_text, key=key, pad=pad,
                         visible=visible, metadata=metadata)

    # returns False if update failed
    def update_bar(self, current_count, max=None):
        """
        DEPRECATED BUT STILL USABLE - has been combined with the normal ProgressBar.update method.
        Change what the bar shows by changing the current count and optionally the max count

        :param current_count: sets the current value
        :type current_count:  (int)
        :param max:           changes the max value
        :type max:            (int)
        """

        if self.ParentForm.TKrootDestroyed:
            return False
        self.TKProgressBar.Update(current_count, max=max)
        try:
            self.ParentForm.TKroot.update()
        except:
            Window._DecrementOpenCount()
            # _my_windows.Decrement()
            return False
        return True

    def update(self, current_count=None, max=None, bar_color=None, visible=None):
        """
        Changes some of the settings for the ProgressBar Element. Must call `Window.Read` or `Window.Finalize` prior
        Now has the ability to modify the count so that the update_bar method is not longer needed separately

        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param current_count: sets the current value
        :type current_count:  (int)
        :param max:           changes the max value
        :type max:            (int)
        :param bar_color:     The 2 colors that make up a progress bar. Easy to remember which is which if you say "ON" between colors. "red" on "green".
        :type bar_color:      (str, str) or str
        :param visible:       control visibility of element
        :type visible:        (bool)
        :return:              Returns True if update was OK.  False means something wrong with window or it was closed
        :rtype:               (bool)
        """
        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return False

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in ProgressBar.update - The window was closed')
            return


        if self.ParentForm.TKrootDestroyed:
            return False

        if visible is False:
            self._pack_forget_save_settings()
        elif visible is True:
            self._pack_restore_settings()

        if visible is not None:
            self._visible = visible
        if bar_color is not None:
            bar_color = _simplified_dual_color_to_tuple(bar_color, default=DEFAULT_PROGRESS_BAR_COLOR)
            self.BarColor = bar_color
            style = ttk.Style()
            style.configure(self.ttk_style_name, background=bar_color[0], troughcolor=bar_color[1])
        if current_count is not None:
            self.TKProgressBar.Update(current_count, max=max)

        try:
            self.ParentForm.TKroot.update()
        except:
            # Window._DecrementOpenCount()
            # _my_windows.Decrement()
            return False
        return True

    Update = update
    UpdateBar = update_bar


PBar = ProgressBar
Prog = ProgressBar
Progress = ProgressBar


# ---------------------------------------------------------------------- #
#                           Image                                        #
# ---------------------------------------------------------------------- #
class Image(Element):
    """
    Image Element - show an image in the window. Should be a GIF or a PNG only
    """

    def __init__(self, source=None, filename=None, data=None, background_color=None, size=(None, None), s=(None, None), pad=None, p=None, key=None, k=None, tooltip=None, subsample=None, zoom=None, right_click_menu=None, expand_x=False, expand_y=False, visible=True, enable_events=False, metadata=None):
        """
        :param source:           A filename or a base64 bytes. Will automatically detect the type and fill in filename or data for you.
        :type source:            str | bytes | None
        :param filename:         image filename if there is a button image. GIFs and PNGs only.
        :type filename:          str | None
        :param data:             Raw or Base64 representation of the image to put on button. Choose either filename or data
        :type data:              bytes | str | None
        :param background_color: color of background
        :type background_color:
        :param size:             (width, height) size of image in pixels
        :type size:              (int, int)
        :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
        :type s:                 (int, int)  | (None, None) | int
        :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param key:              Used with window.find_element and with return values to uniquely identify this element to uniquely identify this element
        :type key:               str | int | tuple | object
        :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                 str | int | tuple | object
        :param tooltip:          text, that will appear when mouse hovers over the element
        :type tooltip:           (str)
        :param subsample:        amount to reduce the size of the image. Divides the size by this number. 2=1/2, 3=1/3, 4=1/4, etc
        :type subsample:         (int)
        :param zoom:             amount to increase the size of the image.
        :type zoom:              (int)
        :param right_click_menu: A list of lists of Menu items to show when this element is right clicked. See user docs for exact format.
        :type right_click_menu:  List[List[ List[str] | str ]]
        :param expand_x:         If True the element will automatically expand in the X direction to fill available space
        :type expand_x:          (bool)
        :param expand_y:         If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:          (bool)
        :param visible:          set visibility state of the element
        :type visible:           (bool)
        :param enable_events:    Turns on the element specific events. For an Image element, the event is "image clicked"
        :type enable_events:     (bool)
        :param metadata:         User metadata that can be set to ANYTHING
        :type metadata:          (Any)
        """

        if source is not None:
            if isinstance(source, bytes):
                data = source
            elif isinstance(source, str):
                filename = source
            else:
                warnings.warn('Image element - source is not a valid type: {}'.format(type(source)), UserWarning)

        self.Filename = filename
        self.Data = data
        self.Widget = self.tktext_label = None  # type: tk.Label
        self.BackgroundColor = background_color
        if data is None and filename is None:
            self.Filename = ''
        self.EnableEvents = enable_events
        self.RightClickMenu = right_click_menu
        self.AnimatedFrames = None
        self.CurrentFrameNumber = 0
        self.TotalAnimatedFrames = 0
        self.LastFrameTime = 0
        self.ImageSubsample = subsample
        self.zoom = int(zoom) if zoom is not None else None

        self.Source = filename if filename is not None else data
        key = key if key is not None else k
        sz = size if size != (None, None) else s
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y


        super().__init__(ELEM_TYPE_IMAGE, size=sz, background_color=background_color, pad=pad, key=key,
                         tooltip=tooltip, visible=visible, metadata=metadata)
        return

    def update(self, source=None, filename=None, data=None, size=(None, None), subsample=None, zoom=None, visible=None):
        """
        Changes some of the settings for the Image Element. Must call `Window.Read` or `Window.Finalize` prior.
        To clear an image that's been displayed, call with NONE of the options set.  A blank update call will
        delete the previously shown image.

        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param source:   A filename or a base64 bytes. Will automatically detect the type and fill in filename or data for you.
        :type source:    str | bytes | None
        :param filename: filename to the new image to display.
        :type filename:  (str)
        :param data:     Base64 encoded string OR a tk.PhotoImage object
        :type data:      str | tkPhotoImage
        :param size:     (width, height) size of image in pixels
        :type size:      Tuple[int,int]
        :param subsample: amount to reduce the size of the image. Divides the size by this number. 2=1/2, 3=1/3, 4=1/4, etc
        :type subsample: (int)
        :param zoom:     amount to increase the size of the image
        :type zoom:      (int)
        :param visible:  control visibility of element
        :type visible:   (bool)
        """

        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in Image.update - The window was closed')
            return


        if source is not None:
            if isinstance(source, bytes):
                data = source
            elif isinstance(source, str):
                filename = source
            else:
                warnings.warn('Image element - source is not a valid type: {}'.format(type(source)), UserWarning)

        image = None
        if filename is not None:
            try:
                image = tk.PhotoImage(file=filename)
                if subsample is not None:
                    image = image.subsample(subsample)
                if zoom is not None:
                    image = image.zoom(int(zoom))
            except Exception as e:
                _error_popup_with_traceback('Exception updating Image element', e)

        elif data is not None:
            # if type(data) is bytes:
            try:
                image = tk.PhotoImage(data=data)
                if subsample is not None:
                    image = image.subsample(subsample)
                if zoom is not None:
                    image = image.zoom(int(zoom))
            except Exception as e:
                image = data
                # return  # an error likely means the window has closed so exit

        if image is not None:
            self.tktext_label.configure(image='')           # clear previous image
            if self.tktext_label.image is not None:
                del self.tktext_label.image
            if type(image) is not bytes:
                width, height = size[0] if size[0] is not None else image.width(), size[1] if size[1] is not None else image.height()
            else:
                width, height = size
            try:  # sometimes crashes if user closed with X
                self.tktext_label.configure(image=image, width=width, height=height)
            except Exception as e:
                _error_popup_with_traceback('Exception updating Image element', e)
            self.tktext_label.image = image
        if visible is False:
            self._pack_forget_save_settings()
        elif visible is True:
            self._pack_restore_settings()

        # if everything is set to None, then delete the image
        if filename is None and image is None and visible is None and size == (None, None):
            # Using a try because the image may have been previously deleted and don't want an error if that's happened
            try:
                self.tktext_label.configure(image='', width=1, height=1, bd=0)
                self.tktext_label.image = None
            except:
                pass

        if visible is not None:
            self._visible = visible

    def update_animation(self, source, time_between_frames=0):
        """
        Show an Animated GIF. Call the function as often as you like. The function will determine when to show the next frame and will automatically advance to the next frame at the right time.
        NOTE - does NOT perform a sleep call to delay
        :param source:              Filename or Base64 encoded string containing Animated GIF
        :type source:               str | bytes | None
        :param time_between_frames: Number of milliseconds to wait between showing frames
        :type time_between_frames:  (int)
        """

        if self.Source != source:
            self.AnimatedFrames = None
            self.Source = source

        if self.AnimatedFrames is None:
            self.TotalAnimatedFrames = 0
            self.AnimatedFrames = []
            # Load up to 1000 frames of animation.  stops when a bad frame is returns by tkinter
            for i in range(1000):
                if type(source) is not bytes:
                    try:
                        self.AnimatedFrames.append(tk.PhotoImage(file=source, format='gif -index %i' % (i)))
                    except Exception as e:
                        break
                else:
                    try:
                        self.AnimatedFrames.append(tk.PhotoImage(data=source, format='gif -index %i' % (i)))
                    except Exception as e:
                        break
            self.TotalAnimatedFrames = len(self.AnimatedFrames)
            self.LastFrameTime = time.time()
            self.CurrentFrameNumber = -1  # start at -1 because it is incremented before every frame is shown
        # show the frame

        now = time.time()

        if time_between_frames:
            if (now - self.LastFrameTime) * 1000 > time_between_frames:
                self.LastFrameTime = now
                self.CurrentFrameNumber = (self.CurrentFrameNumber + 1) % self.TotalAnimatedFrames
            else:  # don't reshow the frame again if not time for new frame
                return
        else:
            self.CurrentFrameNumber = (self.CurrentFrameNumber + 1) % self.TotalAnimatedFrames
        image = self.AnimatedFrames[self.CurrentFrameNumber]
        try:  # needed in case the window was closed with an "X"
            self.tktext_label.configure(image=image, width=image.width(), heigh=image.height())
        except Exception as e:
            print('Exception in update_animation', e)


    def update_animation_no_buffering(self, source, time_between_frames=0):
        """
        Show an Animated GIF. Call the function as often as you like. The function will determine when to show the next frame and will automatically advance to the next frame at the right time.
        NOTE - does NOT perform a sleep call to delay

        :param source:              Filename or Base64 encoded string containing Animated GIF
        :type source:               str | bytes
        :param time_between_frames: Number of milliseconds to wait between showing frames
        :type time_between_frames:  (int)
        """

        if self.Source != source:
            self.AnimatedFrames = None
            self.Source = source
            self.frame_num = 0

        now = time.time()

        if time_between_frames:
            if (now - self.LastFrameTime) * 1000 > time_between_frames:
                self.LastFrameTime = now
            else:  # don't reshow the frame again if not time for new frame
                return

        # read a frame
        while True:
            if type(source) is not bytes:
                try:
                    self.image = tk.PhotoImage(file=source, format='gif -index %i' % (self.frame_num))
                    self.frame_num += 1
                except:
                    self.frame_num = 0
            else:
                try:
                    self.image = tk.PhotoImage(data=source, format='gif -index %i' % (self.frame_num))
                    self.frame_num += 1
                except:
                    self.frame_num = 0
            if self.frame_num:
                break

        try:  # needed in case the window was closed with an "X"
            self.tktext_label.configure(image=self.image, width=self.image.width(), heigh=self.image.height())

        except:
            pass

    Update = update
    UpdateAnimation = update_animation


Im = Image


# ---------------------------------------------------------------------- #
#                           Canvas                                       #
# ---------------------------------------------------------------------- #
class Canvas(Element):

    def __init__(self, canvas=None, background_color=None, size=(None, None), s=(None, None), pad=None, p=None, key=None, k=None, tooltip=None,
                 right_click_menu=None, expand_x=False, expand_y=False, visible=True, border_width=0, metadata=None):
        """
        :param canvas:           Your own tk.Canvas if you already created it. Leave blank to create a Canvas
        :type canvas:            (tk.Canvas)
        :param background_color: color of background
        :type background_color:  (str)
        :param size:             (width in char, height in rows) size in pixels to make canvas
        :type size:              (int,int) | (None, None)
        :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
        :type s:                 (int, int)  | (None, None) | int
        :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param key:              Used with window.find_element and with return values to uniquely identify this element
        :type key:               str | int | tuple | object
        :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                 str | int | tuple | object
        :param tooltip:          text, that will appear when mouse hovers over the element
        :type tooltip:           (str)
        :param right_click_menu: A list of lists of Menu items to show when this element is right clicked. See user docs for exact format.
        :type right_click_menu:  List[List[ List[str] | str ]]
        :param expand_x:         If True the element will automatically expand in the X direction to fill available space
        :type expand_x:          (bool)
        :param expand_y:         If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:          (bool)
        :param visible:          set visibility state of the element
        :type visible:           (bool)
        :param border_width:     width of border around element in pixels. Not normally used with Canvas element
        :type border_width:      (int)
        :param metadata:         User metadata that can be set to ANYTHING
        :type metadata:          (Any)
        """


        self.BackgroundColor = background_color if background_color is not None else DEFAULT_BACKGROUND_COLOR
        self._TKCanvas = self.Widget = canvas
        self.RightClickMenu = right_click_menu
        self.BorderWidth = border_width
        key = key if key is not None else k
        sz = size if size != (None, None) else s
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y

        super().__init__(ELEM_TYPE_CANVAS, background_color=background_color, size=sz, pad=pad, key=key,
                         tooltip=tooltip, visible=visible, metadata=metadata)
        return


    def update(self,  background_color=None, visible=None):
        """

        :param background_color: color of background
        :type background_color:  (str)
        :param visible:          set visibility state of the element
        :type visible:           (bool)
        """

        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in Canvas.update - The window was closed')
            return

        if background_color not in (None, COLOR_SYSTEM_DEFAULT):
            self._TKCanvas.configure(background=background_color)
        if visible is False:
            self._pack_forget_save_settings()
        elif visible is True:
            self._pack_restore_settings()
        if visible is not None:
            self._visible = visible


    @property
    def tk_canvas(self):
        """
        Returns the underlying tkiner Canvas widget

        :return: The tkinter canvas widget
        :rtype:  (tk.Canvas)
        """
        if self._TKCanvas is None:
            print('*** Did you forget to call Finalize()? Your code should look something like: ***')
            print('*** window = sg.Window("My Form", layout, finalize=True) ***')
        return self._TKCanvas

    TKCanvas = tk_canvas


# ---------------------------------------------------------------------- #
#                           Graph                                        #
# ---------------------------------------------------------------------- #
class Graph(Element):
    """
    Creates an area for you to draw on.  The MAGICAL property this Element has is that you interact
    with the element using your own coordinate system.  This is an important point!!  YOU define where the location
    is for (0,0).  Want (0,0) to be in the middle of the graph like a math 4-quadrant graph?  No problem!  Set your
    lower left corner to be (-100,-100) and your upper right to be (100,100) and you've got yourself a graph with
    (0,0) at the center.
    One of THE coolest of the Elements.
    You can also use float values. To do so, be sure and set the float_values parameter.
    Mouse click and drag events are possible and return the (x,y) coordinates of the mouse
    Drawing primitives return an "id" that is referenced when you want to operation on that item (e.g. to erase it)
    """

    def __init__(self, canvas_size, graph_bottom_left, graph_top_right, background_color=None, pad=None, p=None,
                 change_submits=False, drag_submits=False, enable_events=False, motion_events=False, key=None, k=None, tooltip=None,
                 right_click_menu=None, expand_x=False, expand_y=False, visible=True, float_values=False, border_width=0, metadata=None):
        """
        :param canvas_size:       size of the canvas area in pixels
        :type canvas_size:        (int, int)
        :param graph_bottom_left: (x,y) The bottoms left corner of your coordinate system
        :type graph_bottom_left:  (int, int)
        :param graph_top_right:   (x,y) The top right corner of  your coordinate system
        :type graph_top_right:    (int, int)
        :param background_color:  background color of the drawing area
        :type background_color:   (str)
        :param pad:               Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:                (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                 Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                  (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param change_submits:    * DEPRICATED DO NOT USE. Use `enable_events` instead
        :type change_submits:     (bool)
        :param drag_submits:      if True and Events are enabled for the Graph, will report Events any time the mouse moves while button down.  When the mouse button is released, you'll get an event = graph key + '+UP' (if key is a string.. if not a string, it'll be made into a tuple)
        :type drag_submits:       (bool)
        :param enable_events:     If True then clicks on the Graph are immediately reported as an event. Use this instead of change_submits
        :type enable_events:      (bool)
        :param motion_events:     If True then if no button is down and the mouse is moved, an event is generated with key = graph key + '+MOVE' (if key is a string, it not a string then a tuple is returned)
        :type motion_events:      (bool)
        :param key:               Value that uniquely identifies this element from all other elements. Used when Finding an element or in return values. Must be unique to the window
        :type key:                str | int | tuple | object
        :param k:                 Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                  str | int | tuple | object
        :param tooltip:           text, that will appear when mouse hovers over the element
        :type tooltip:            (str)
        :param right_click_menu:  A list of lists of Menu items to show when this element is right clicked. See user docs for exact format.
        :type right_click_menu:   List[List[ List[str] | str ]]
        :param expand_x:          If True the element will automatically expand in the X direction to fill available space
        :type expand_x:           (bool)
        :param expand_y:          If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:           (bool)
        :param visible:           set visibility state of the element (Default = True)
        :type visible:            (bool)
        :param float_values:      If True x,y coordinates are returned as floats, not ints
        :type float_values:       (bool)
        :param border_width:      width of border around element in pixels. Not normally used for Graph Elements
        :type border_width:       (int)
        :param metadata:          User metadata that can be set to ANYTHING
        :type metadata:           (Any)
        """

        self.CanvasSize = canvas_size
        self.BottomLeft = graph_bottom_left
        self.TopRight = graph_top_right
        # self._TKCanvas = None               # type: tk.Canvas
        self._TKCanvas2 = self.Widget = None  # type: tk.Canvas
        self.ChangeSubmits = change_submits or enable_events
        self.DragSubmits = drag_submits
        self.ClickPosition = (None, None)
        self.MouseButtonDown = False
        self.Images = {}
        self.RightClickMenu = right_click_menu
        self.FloatValues = float_values
        self.BorderWidth = border_width
        key = key if key is not None else k
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y
        self.motion_events = motion_events

        super().__init__(ELEM_TYPE_GRAPH, background_color=background_color, size=canvas_size, pad=pad, key=key,
                         tooltip=tooltip, visible=visible, metadata=metadata)
        return

    def _convert_xy_to_canvas_xy(self, x_in, y_in):
        """
        Not user callable.  Used to convert user's coordinates into the ones used by tkinter
        :param x_in: The x coordinate to convert
        :type x_in:  int | float
        :param y_in: The y coordinate to convert
        :type y_in:  int | float
        :return:     (int, int) The converted canvas coordinates
        :rtype:      (int, int)
        """
        if None in (x_in, y_in):
            return None, None
        try:
            scale_x = (self.CanvasSize[0] - 0) / (self.TopRight[0] - self.BottomLeft[0])
            scale_y = (0 - self.CanvasSize[1]) / (self.TopRight[1] - self.BottomLeft[1])
        except:
            scale_x = scale_y = 0

        new_x = 0 + scale_x * (x_in - self.BottomLeft[0])
        new_y = self.CanvasSize[1] + scale_y * (y_in - self.BottomLeft[1])
        return new_x, new_y

    def _convert_canvas_xy_to_xy(self, x_in, y_in):
        """
        Not user callable.  Used to convert tkinter Canvas coords into user's coordinates

        :param x_in: The x coordinate in canvas coordinates
        :type x_in:  (int)
        :param y_in: (int) The y coordinate in canvas coordinates
        :type y_in:
        :return:     The converted USER coordinates
        :rtype:      (int, int) | Tuple[float, float]
        """
        if None in (x_in, y_in):
            return None, None
        scale_x = (self.CanvasSize[0] - 0) / (self.TopRight[0] - self.BottomLeft[0])
        scale_y = (0 - self.CanvasSize[1]) / (self.TopRight[1] - self.BottomLeft[1])

        new_x = x_in / scale_x + self.BottomLeft[0]
        new_y = (y_in - self.CanvasSize[1]) / scale_y + self.BottomLeft[1]
        if self.FloatValues:
            return new_x, new_y
        else:
            return floor(new_x), floor(new_y)

    def draw_line(self, point_from, point_to, color='black', width=1):
        """
        Draws a line from one point to another point using USER'S coordinates. Can set the color and width of line
        :param point_from: Starting point for line
        :type point_from:  (int, int) | Tuple[float, float]
        :param point_to:   Ending point for line
        :type point_to:    (int, int) | Tuple[float, float]
        :param color:      Color of the line
        :type color:       (str)
        :param width:      width of line in pixels
        :type width:       (int)
        :return:           id returned from tktiner or None if user closed the window. id is used when you
        :rtype:            int | None
        """
        if point_from == (None, None):
            return
        converted_point_from = self._convert_xy_to_canvas_xy(point_from[0], point_from[1])
        converted_point_to = self._convert_xy_to_canvas_xy(point_to[0], point_to[1])
        if self._TKCanvas2 is None:
            print('*** WARNING - The Graph element has not been finalized and cannot be drawn upon ***')
            print('Call Window.Finalize() prior to this operation')
            return None
        try:  # in case window was closed with an X
            id = self._TKCanvas2.create_line(converted_point_from, converted_point_to, width=width, fill=color)
        except:
            id = None
        return id

    def draw_lines(self, points, color='black', width=1):
        """
        Draw a series of lines given list of points

        :param points: list of points that define the polygon
        :type points:  List[(int, int) | Tuple[float, float]]
        :param color:  Color of the line
        :type color:   (str)
        :param width:  width of line in pixels
        :type width:   (int)
        :return:       id returned from tktiner or None if user closed the window. id is used when you
        :rtype:        int | None
        """
        converted_points = [self._convert_xy_to_canvas_xy(point[0], point[1]) for point in points]

        try:  # in case window was closed with an X
            id = self._TKCanvas2.create_line(*converted_points, width=width, fill=color)
        except:
            if self._TKCanvas2 is None:
                print('*** WARNING - The Graph element has not been finalized and cannot be drawn upon ***')
                print('Call Window.Finalize() prior to this operation')
            id = None
        return id

    def draw_point(self, point, size=2, color='black'):
        """
        Draws a "dot" at the point you specify using the USER'S coordinate system
        :param point: Center location using USER'S coordinate system
        :type point:  (int, int) | Tuple[float, float]
        :param size:  Radius? (Or is it the diameter?) in user's coordinate values.
        :type size:   int | float
        :param color: color of the point to draw
        :type color:  (str)
        :return:      id returned from tkinter that you'll need if you want to manipulate the point
        :rtype:       int | None
        """
        if point == (None, None):
            return
        converted_point = self._convert_xy_to_canvas_xy(point[0], point[1])
        size_converted = self._convert_xy_to_canvas_xy(point[0] + size, point[1])
        size = size_converted[0] - converted_point[0]
        if self._TKCanvas2 is None:
            print('*** WARNING - The Graph element has not been finalized and cannot be drawn upon ***')
            print('Call Window.Finalize() prior to this operation')
            return None
        try:  # needed in case window was closed with an X
            point1 = converted_point[0] - size // 2, converted_point[1] - size // 2
            point2 = converted_point[0] + size // 2, converted_point[1] + size // 2
            id = self._TKCanvas2.create_oval(point1[0], point1[1],
                                             point2[0], point2[1],
                                             width=0,
                                             fill=color,
                                             outline=color)
        except:
            id = None
        return id

    def draw_circle(self, center_location, radius, fill_color=None, line_color='black', line_width=1):
        """
        Draws a circle, cenetered at the location provided.  Can set the fill and outline colors
        :param center_location: Center location using USER'S coordinate system
        :type center_location:  (int, int) | Tuple[float, float]
        :param radius:          Radius in user's coordinate values.
        :type radius:           int | float
        :param fill_color:      color of the point to draw
        :type fill_color:       (str)
        :param line_color:      color of the outer line that goes around the circle (sorry, can't set thickness)
        :type line_color:       (str)
        :param line_width:      width of the line around the circle, the outline, in pixels
        :type line_width:       (int)
        :return:                id returned from tkinter that you'll need if you want to manipulate the circle
        :rtype:                 int | None
        """
        if center_location == (None, None):
            return
        converted_point = self._convert_xy_to_canvas_xy(center_location[0], center_location[1])
        radius_converted = self._convert_xy_to_canvas_xy(center_location[0] + radius, center_location[1])
        radius = radius_converted[0] - converted_point[0]
        # radius = radius_converted[1]-5
        if self._TKCanvas2 is None:
            print('*** WARNING - The Graph element has not been finalized and cannot be drawn upon ***')
            print('Call Window.Finalize() prior to this operation')
            return None
        # print('Oval parms', int(converted_point[0]) - int(radius), int(converted_point[1]) - int(radius),
        #                                      int(converted_point[0]) + int(radius), int(converted_point[1]) + int(radius))
        try:  # needed in case the window was closed with an X
            id = self._TKCanvas2.create_oval(int(converted_point[0]) - int(radius), int(converted_point[1]) - int(radius),
                                             int(converted_point[0]) + int(radius), int(converted_point[1]) + int(radius), fill=fill_color,
                                             outline=line_color, width=line_width)
        except:
            id = None
        return id

    def draw_oval(self, top_left, bottom_right, fill_color=None, line_color=None, line_width=1):
        """
        Draws an oval based on coordinates in user coordinate system. Provide the location of a "bounding rectangle"
        :param top_left:     the top left point of bounding rectangle
        :type top_left:      (int, int) | Tuple[float, float]
        :param bottom_right: the bottom right point of bounding rectangle
        :type bottom_right:  (int, int) | Tuple[float, float]
        :param fill_color:   color of the interrior
        :type fill_color:    (str)
        :param line_color:   color of outline of oval
        :type line_color:    (str)
        :param line_width:   width of the line around the oval, the outline, in pixels
        :type line_width:    (int)
        :return:             id returned from tkinter that you'll need if you want to manipulate the oval
        :rtype:              int | None
        """
        converted_top_left = self._convert_xy_to_canvas_xy(top_left[0], top_left[1])
        converted_bottom_right = self._convert_xy_to_canvas_xy(bottom_right[0], bottom_right[1])
        if self._TKCanvas2 is None:
            print('*** WARNING - The Graph element has not been finalized and cannot be drawn upon ***')
            print('Call Window.Finalize() prior to this operation')
            return None
        try:  # in case windows close with X
            id = self._TKCanvas2.create_oval(converted_top_left[0], converted_top_left[1], converted_bottom_right[0],
                                             converted_bottom_right[1], fill=fill_color, outline=line_color, width=line_width)
        except:
            id = None

        return id

    def draw_arc(self, top_left, bottom_right, extent, start_angle, style=None, arc_color='black', line_width=1, fill_color=None):
        """
        Draws different types of arcs.  Uses a "bounding box" to define location
        :param top_left:     the top left point of bounding rectangle
        :type top_left:      (int, int) | Tuple[float, float]
        :param bottom_right: the bottom right point of bounding rectangle
        :type bottom_right:  (int, int) | Tuple[float, float]
        :param extent:       Andle to end drawing. Used in conjunction with start_angle
        :type extent:        (float)
        :param start_angle:  Angle to begin drawing. Used in conjunction with extent
        :type start_angle:   (float)
        :param style:        Valid choices are One of these Style strings- 'pieslice', 'chord', 'arc', 'first', 'last', 'butt', 'projecting', 'round', 'bevel', 'miter'
        :type style:         (str)
        :param arc_color:    color to draw arc with
        :type arc_color:     (str)
        :param fill_color:   color to fill the area
        :type fill_color:    (str)
        :return:             id returned from tkinter that you'll need if you want to manipulate the arc
        :rtype:              int | None
        """
        converted_top_left = self._convert_xy_to_canvas_xy(top_left[0], top_left[1])
        converted_bottom_right = self._convert_xy_to_canvas_xy(bottom_right[0], bottom_right[1])
        tkstyle = tk.PIESLICE if style is None else style
        if self._TKCanvas2 is None:
            print('*** WARNING - The Graph element has not been finalized and cannot be drawn upon ***')
            print('Call Window.Finalize() prior to this operation')
            return None
        try:  # in case closed with X
            id = self._TKCanvas2.create_arc(converted_top_left[0], converted_top_left[1], converted_bottom_right[0],
                                            converted_bottom_right[1], extent=extent, start=start_angle, style=tkstyle,
                                            outline=arc_color, width=line_width, fill=fill_color)
        except Exception as e:
            print('Error encountered drawing arc.', e)
            id = None
        return id

    def draw_rectangle(self, top_left, bottom_right, fill_color=None, line_color=None, line_width=None):
        """
        Draw a rectangle given 2 points. Can control the line and fill colors

        :param top_left:     the top left point of rectangle
        :type top_left:      (int, int) | Tuple[float, float]
        :param bottom_right: the bottom right point of rectangle
        :type bottom_right:  (int, int) | Tuple[float, float]
        :param fill_color:   color of the interior
        :type fill_color:    (str)
        :param line_color:   color of outline
        :type line_color:    (str)
        :param line_width:   width of the line in pixels
        :type line_width:    (int)
        :return:             int | None id returned from tkinter that you'll need if you want to manipulate the rectangle
        :rtype:              int | None
        """

        converted_top_left = self._convert_xy_to_canvas_xy(top_left[0], top_left[1])
        converted_bottom_right = self._convert_xy_to_canvas_xy(bottom_right[0], bottom_right[1])
        if self._TKCanvas2 is None:
            print('*** WARNING - The Graph element has not been finalized and cannot be drawn upon ***')
            print('Call Window.Finalize() prior to this operation')
            return None
        if line_width is None:
            line_width = 1
        try:  # in case closed with X
            id = self._TKCanvas2.create_rectangle(converted_top_left[0], converted_top_left[1],
                                                  converted_bottom_right[0],
                                                  converted_bottom_right[1], fill=fill_color, outline=line_color, width=line_width)
        except:
            id = None
        return id

    def draw_polygon(self, points, fill_color=None, line_color=None, line_width=None):
        """
        Draw a polygon given list of points

        :param points:     list of points that define the polygon
        :type points:      List[(int, int) | Tuple[float, float]]
        :param fill_color: color of the interior
        :type fill_color:  (str)
        :param line_color: color of outline
        :type line_color:  (str)
        :param line_width: width of the line in pixels
        :type line_width:  (int)
        :return:           id returned from tkinter that you'll need if you want to manipulate the rectangle
        :rtype:            int | None
        """

        converted_points = [self._convert_xy_to_canvas_xy(point[0], point[1]) for point in points]
        if self._TKCanvas2 is None:
            print('*** WARNING - The Graph element has not been finalized and cannot be drawn upon ***')
            print('Call Window.Finalize() prior to this operation')
            return None
        try:  # in case closed with X
            id = self._TKCanvas2.create_polygon(converted_points, fill=fill_color, outline=line_color, width=line_width)
        except:
            id = None
        return id

    def draw_text(self, text, location, color='black', font=None, angle=0, text_location=TEXT_LOCATION_CENTER):
        """
        Draw some text on your graph.  This is how you label graph number lines for example

        :param text:          text to display
        :type text:           (Any)
        :param location:      location to place first letter
        :type location:       (int, int) | Tuple[float, float]
        :param color:         text color
        :type color:          (str)
        :param font:          specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:           (str or (str, int[, str]) or None)
        :param angle:         Angle 0 to 360 to draw the text.  Zero represents horizontal text
        :type angle:          (float)
        :param text_location: "anchor" location for the text. Values start with TEXT_LOCATION_
        :type text_location:  (enum)
        :return:              id returned from tkinter that you'll need if you want to manipulate the text
        :rtype:               int | None
        """
        text = str(text)
        if location == (None, None):
            return
        converted_point = self._convert_xy_to_canvas_xy(location[0], location[1])
        if self._TKCanvas2 is None:
            print('*** WARNING - The Graph element has not been finalized and cannot be drawn upon ***')
            print('Call Window.Finalize() prior to this operation')
            return None
        try:  # in case closed with X
            id = self._TKCanvas2.create_text(converted_point[0], converted_point[1], text=text, font=font, fill=color, angle=angle, anchor=text_location)
        except:
            id = None
        return id

    def draw_image(self, filename=None, data=None, location=(None, None)):
        """
        Places an image onto your canvas.  It's a really important method for this element as it enables so much

        :param filename: if image is in a file, path and filename for the image. (GIF and PNG only!)
        :type filename:  (str)
        :param data:     if image is in Base64 format or raw? format then use instead of filename
        :type data:      str | bytes
        :param location: the (x,y) location to place image's top left corner
        :type location:  (int, int) | Tuple[float, float]
        :return:         id returned from tkinter that you'll need if you want to manipulate the image
        :rtype:          int | None
        """
        if location == (None, None):
            return
        if filename is not None:
            image = tk.PhotoImage(file=filename)
        elif data is not None:
            # if type(data) is bytes:
            try:
                image = tk.PhotoImage(data=data)
            except:
                return None  # an error likely means the window has closed so exit
        converted_point = self._convert_xy_to_canvas_xy(location[0], location[1])
        if self._TKCanvas2 is None:
            print('*** WARNING - The Graph element has not been finalized and cannot be drawn upon ***')
            print('Call Window.Finalize() prior to this operation')
            return None
        try:  # in case closed with X
            id = self._TKCanvas2.create_image(converted_point, image=image, anchor=tk.NW)
            self.Images[id] = image
        except:
            id = None
        return id

    def erase(self):
        """
        Erase the Graph - Removes all figures previously "drawn" using the Graph methods (e.g. DrawText)
        """
        if self._TKCanvas2 is None:
            print('*** WARNING - The Graph element has not been finalized and cannot be drawn upon ***')
            print('Call Window.Finalize() prior to this operation')
            return None
        self.Images = {}
        try:  # in case window was closed with X
            self._TKCanvas2.delete('all')
        except:
            pass

    def delete_figure(self, id):
        """
        Remove from the Graph the figure represented by id. The id is given to you anytime you call a drawing primitive

        :param id: the id returned to you when calling one of the drawing methods
        :type id:  (int)
        """
        try:
            self._TKCanvas2.delete(id)
        except:
            print('DeleteFigure - bad ID {}'.format(id))
        try:
            del self.Images[id]  # in case was an image. If wasn't an image, then will get exception
        except:
            pass

    def update(self, background_color=None, visible=None):
        """
        Changes some of the settings for the Graph Element. Must call `Window.Read` or `Window.Finalize` prior

        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param background_color: color of background
        :type background_color:  ???
        :param visible:          control visibility of element
        :type visible:           (bool)
        """
        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in Graph.update - The window was closed')
            return

        if background_color is not None and background_color != COLOR_SYSTEM_DEFAULT:
            self._TKCanvas2.configure(background=background_color)

        if visible is False:
            self._pack_forget_save_settings()
        elif visible is True:
            self._pack_restore_settings()

        if visible is not None:
            self._visible = visible

    def move(self, x_direction, y_direction):
        """
        Moves the entire drawing area (the canvas) by some delta from the current position.  Units are indicated in your coordinate system indicated number of ticks in your coordinate system

        :param x_direction: how far to move in the "X" direction in your coordinates
        :type x_direction:  int | float
        :param y_direction: how far to move in the "Y" direction in your coordinates
        :type y_direction:  int | float
        """
        zero_converted = self._convert_xy_to_canvas_xy(0, 0)
        shift_converted = self._convert_xy_to_canvas_xy(x_direction, y_direction)
        shift_amount = (shift_converted[0] - zero_converted[0], shift_converted[1] - zero_converted[1])
        if self._TKCanvas2 is None:
            print('*** WARNING - The Graph element has not been finalized and cannot be drawn upon ***')
            print('Call Window.Finalize() prior to this operation')
            return None
        self._TKCanvas2.move('all', shift_amount[0], shift_amount[1])

    def move_figure(self, figure, x_direction, y_direction):
        """
        Moves a previously drawn figure using a "delta" from current position

        :param figure:      Previously obtained figure-id. These are returned from all Draw methods
        :type figure:       (id)
        :param x_direction: delta to apply to position in the X direction
        :type x_direction:  int | float
        :param y_direction: delta to apply to position in the Y direction
        :type y_direction:  int | float
        """
        zero_converted = self._convert_xy_to_canvas_xy(0, 0)
        shift_converted = self._convert_xy_to_canvas_xy(x_direction, y_direction)
        shift_amount = (shift_converted[0] - zero_converted[0], shift_converted[1] - zero_converted[1])
        if figure is None:
            print('* move_figure warning - your figure is None *')
            return None
        self._TKCanvas2.move(figure, shift_amount[0], shift_amount[1])

    def relocate_figure(self, figure, x, y):
        """
        Move a previously made figure to an arbitrary (x,y) location. This differs from the Move methods because it
        uses absolute coordinates versus relative for Move

        :param figure: Previously obtained figure-id. These are returned from all Draw methods
        :type figure:  (id)
        :param x:      location on X axis (in user coords) to move the upper left corner of the figure
        :type x:       int | float
        :param y:      location on Y axis (in user coords) to move the upper left corner of the figure
        :type y:       int | float
        """

        zero_converted = self._convert_xy_to_canvas_xy(0, 0)
        shift_converted = self._convert_xy_to_canvas_xy(x, y)
        shift_amount = (shift_converted[0] - zero_converted[0], shift_converted[1] - zero_converted[1])
        if figure is None:
            print('*** WARNING - Your figure is None. It most likely means your did not Finalize your Window ***')
            print('Call Window.Finalize() prior to all graph operations')
            return None
        xy = self._TKCanvas2.coords(figure)
        self._TKCanvas2.move(figure, shift_converted[0] - xy[0], shift_converted[1] - xy[1])

    def send_figure_to_back(self, figure):
        """
        Changes Z-order of figures on the Graph.  Sends the indicated figure to the back of all other drawn figures

        :param figure: value returned by tkinter when creating the figure / drawing
        :type figure:  (int)
        """
        self.TKCanvas.tag_lower(figure)  # move figure to the "bottom" of all other figure

    def bring_figure_to_front(self, figure):
        """
        Changes Z-order of figures on the Graph.  Brings the indicated figure to the front of all other drawn figures

        :param figure: value returned by tkinter when creating the figure / drawing
        :type figure:  (int)
        """
        self.TKCanvas.tag_raise(figure)  # move figure to the "top" of all other figures

    def get_figures_at_location(self, location):
        """
        Returns a list of figures located at a particular x,y location within the Graph

        :param location: point to check
        :type location:  (int, int) | Tuple[float, float]
        :return:         a list of previously drawn "Figures" (returned from the drawing primitives)
        :rtype:          List[int]
        """
        x, y = self._convert_xy_to_canvas_xy(location[0], location[1])
        ids = self.TKCanvas.find_overlapping(x, y, x, y)
        return ids

    def get_bounding_box(self, figure):
        """
        Given a figure, returns the upper left and lower right bounding box coordinates

        :param figure: a previously drawing figure
        :type figure:  object
        :return:       upper left x, upper left y, lower right x, lower right y
        :rtype:        Tuple[int, int, int, int] | Tuple[float, float, float, float]
        """
        box = self.TKCanvas.bbox(figure)
        top_left = self._convert_canvas_xy_to_xy(box[0], box[1])
        bottom_right = self._convert_canvas_xy_to_xy(box[2], box[3])
        return top_left, bottom_right

    def change_coordinates(self, graph_bottom_left, graph_top_right):
        """
        Changes the corrdinate system to a new one.  The same 2 points in space are used to define the coorinate
        system - the bottom left and the top right values of your graph.

        :param graph_bottom_left: The bottoms left corner of your coordinate system
        :type graph_bottom_left:  (int, int) (x,y)
        :param graph_top_right:   The top right corner of  your coordinate system
        :type graph_top_right:    (int, int)  (x,y)
        """
        self.BottomLeft = graph_bottom_left
        self.TopRight = graph_top_right

    @property
    def tk_canvas(self):
        """
        Returns the underlying tkiner Canvas widget

        :return: The tkinter canvas widget
        :rtype:  (tk.Canvas)
        """
        if self._TKCanvas2 is None:
            print('*** Did you forget to call Finalize()? Your code should look something like: ***')
            print('*** form = sg.Window("My Form").Layout(layout).Finalize() ***')
        return self._TKCanvas2

    # button release callback
    def button_release_call_back(self, event):
        """
        Not a user callable method.  Used to get Graph click events. Called by tkinter when button is released

        :param event: (event) event info from tkinter. Note not used in this method
        :type event:
        """
        if not self.DragSubmits:
            return  # only report mouse up for drag operations
        self.ClickPosition = self._convert_canvas_xy_to_xy(event.x, event.y)
        self.ParentForm.LastButtonClickedWasRealtime = False
        if self.Key is not None:
            self.ParentForm.LastButtonClicked = self.Key
        else:
            self.ParentForm.LastButtonClicked = '__GRAPH__'  # need to put something rather than None
        _exit_mainloop(self.ParentForm)
        if isinstance(self.ParentForm.LastButtonClicked, str):
            self.ParentForm.LastButtonClicked = self.ParentForm.LastButtonClicked + '+UP'
        else:
            self.ParentForm.LastButtonClicked = (self.ParentForm.LastButtonClicked, '+UP')
        self.MouseButtonDown = False


    # button callback
    def button_press_call_back(self, event):
        """
        Not a user callable method.  Used to get Graph click events. Called by tkinter when button is released

        :param event: (event) event info from tkinter. Contains the x and y coordinates of a click
        :type event:
        """

        self.ClickPosition = self._convert_canvas_xy_to_xy(event.x, event.y)
        self.ParentForm.LastButtonClickedWasRealtime = self.DragSubmits
        if self.Key is not None:
            self.ParentForm.LastButtonClicked = self.Key
        else:
            self.ParentForm.LastButtonClicked = '__GRAPH__'  # need to put something rather than None
        # if self.ParentForm.CurrentlyRunningMainloop:
        #     self.ParentForm.TKroot.quit()  # kick out of loop if read was called
        _exit_mainloop(self.ParentForm)
        self.MouseButtonDown = True

    def _update_position_for_returned_values(self, event):
        """
        Updates the variable that's used when the values dictionary is returned from a window read.

        Not called by the user.  It's called from another method/function that tkinter calledback

        :param event: (event) event info from tkinter. Contains the x and y coordinates of a click
        :type event:
        """
        """
        Updates the variable that's used when the values dictionary is returned from a window read.

        Not called by the user.  It's called from another method/function that tkinter calledback

        :param event: (event) event info from tkinter. Contains the x and y coordinates of a click
        :type event:
        """

        self.ClickPosition = self._convert_canvas_xy_to_xy(event.x, event.y)

    # button callback
    def motion_call_back(self, event):
        """
        Not a user callable method.  Used to get Graph mouse motion events. Called by tkinter when mouse moved

        :param event: (event) event info from tkinter. Contains the x and y coordinates of a mouse
        :type event:
        """

        if not self.MouseButtonDown and not self.motion_events:
            return
        self.ClickPosition = self._convert_canvas_xy_to_xy(event.x, event.y)
        self.ParentForm.LastButtonClickedWasRealtime = self.DragSubmits
        if self.Key is not None:
            self.ParentForm.LastButtonClicked = self.Key
        else:
            self.ParentForm.LastButtonClicked = '__GRAPH__'  # need to put something rather than None
        # if self.ParentForm.CurrentlyRunningMainloop:
        #     self.ParentForm.TKroot.quit()  # kick out of loop if read was called
        if self.motion_events and not self.MouseButtonDown:
            if isinstance(self.ParentForm.LastButtonClicked, str):
                self.ParentForm.LastButtonClicked = self.ParentForm.LastButtonClicked + '+MOVE'
            else:
                self.ParentForm.LastButtonClicked = (self.ParentForm.LastButtonClicked, '+MOVE')
        _exit_mainloop(self.ParentForm)

    BringFigureToFront = bring_figure_to_front
    ButtonPressCallBack = button_press_call_back
    ButtonReleaseCallBack = button_release_call_back
    DeleteFigure = delete_figure
    DrawArc = draw_arc
    DrawCircle = draw_circle
    DrawImage = draw_image
    DrawLine = draw_line
    DrawOval = draw_oval
    DrawPoint = draw_point
    DrawPolygon = draw_polygon
    DrawLines = draw_lines
    DrawRectangle = draw_rectangle
    DrawText = draw_text
    GetFiguresAtLocation = get_figures_at_location
    GetBoundingBox = get_bounding_box
    Erase = erase
    MotionCallBack = motion_call_back
    Move = move
    MoveFigure = move_figure
    RelocateFigure = relocate_figure
    SendFigureToBack = send_figure_to_back
    TKCanvas = tk_canvas
    Update = update


G = Graph


# ---------------------------------------------------------------------- #
#                           Frame                                        #
# ---------------------------------------------------------------------- #
class Frame(Element):
    """
    A Frame Element that contains other Elements. Encloses with a line around elements and a text label.
    """

    def __init__(self, title, layout, title_color=None, background_color=None, title_location=None,
                 relief=DEFAULT_FRAME_RELIEF, size=(None, None), s=(None, None), font=None, pad=None, p=None, border_width=None, key=None, k=None,
                 tooltip=None, right_click_menu=None, expand_x=False, expand_y=False, grab=None, visible=True, element_justification='left', vertical_alignment=None, metadata=None):
        """
        :param title:                 text that is displayed as the Frame's "label" or title
        :type title:                  (str)
        :param layout:                The layout to put inside the Frame
        :type layout:                 List[List[Elements]]
        :param title_color:           color of the title text
        :type title_color:            (str)
        :param background_color:      background color of the Frame
        :type background_color:       (str)
        :param title_location:        location to place the text title.  Choices include: TITLE_LOCATION_TOP TITLE_LOCATION_BOTTOM TITLE_LOCATION_LEFT TITLE_LOCATION_RIGHT TITLE_LOCATION_TOP_LEFT TITLE_LOCATION_TOP_RIGHT TITLE_LOCATION_BOTTOM_LEFT TITLE_LOCATION_BOTTOM_RIGHT
        :type title_location:         (enum)
        :param relief:                relief style. Values are same as other elements with reliefs. Choices include RELIEF_RAISED RELIEF_SUNKEN RELIEF_FLAT RELIEF_RIDGE RELIEF_GROOVE RELIEF_SOLID
        :type relief:                 (enum)
        :param size:                  (width, height) Sets an initial hard-coded size for the Frame. This used to be a problem, but was fixed in 4.53.0 and works better than Columns when using the size paramter
        :type size:                   (int, int)
        :param s:                     Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
        :type s:                      (int, int)  | (None, None) | int
        :param font:                  specifies the  font family, size, etc. for the TITLE. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:                   (str or (str, int[, str]) or None)
        :param pad:                   Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:                    (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                     Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                      (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param border_width:          width of border around element in pixels
        :type border_width:           (int)
        :param key:                   Value that uniquely identifies this element from all other elements. Used when Finding an element or in return values. Must be unique to the window
        :type key:                    str | int | tuple | object
        :param k:                     Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                      str | int | tuple | object
        :param tooltip:               text, that will appear when mouse hovers over the element
        :type tooltip:                (str)
        :param right_click_menu:      A list of lists of Menu items to show when this element is right clicked. See user docs for exact format.
        :type right_click_menu:       List[List[ List[str] | str ]]
        :param expand_x:              If True the element will automatically expand in the X direction to fill available space
        :type expand_x:               (bool)
        :param expand_y:              If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:               (bool)
        :param grab:                  If True can grab this element and move the window around. Default is False
        :type grab:                   (bool)
        :param visible:               set visibility state of the element
        :type visible:                (bool)
        :param element_justification: All elements inside the Frame will have this justification 'left', 'right', 'center' are valid values
        :type element_justification:  (str)
        :param vertical_alignment:    Place the Frame at the 'top', 'center', 'bottom' of the row (can also use t,c,r). Defaults to no setting (tkinter decides)
        :type vertical_alignment:     (str)
        :param metadata:              User metadata that can be set to ANYTHING
        :type metadata:               (Any)
        """


        self.UseDictionary = False
        self.ReturnValues = None
        self.ReturnValuesList = []
        self.ReturnValuesDictionary = {}
        self.DictionaryKeyCounter = 0
        self.ParentWindow = None
        self.Rows = []
        # self.ParentForm = None
        self.TKFrame = None
        self.Title = title
        self.Relief = relief
        self.TitleLocation = title_location
        self.BorderWidth = border_width
        self.BackgroundColor = background_color if background_color is not None else DEFAULT_BACKGROUND_COLOR
        self.RightClickMenu = right_click_menu
        self.ContainerElemementNumber = Window._GetAContainerNumber()
        self.ElementJustification = element_justification
        self.VerticalAlignment = vertical_alignment
        self.Widget = None  # type: tk.LabelFrame
        self.Grab = grab
        self.Layout(layout)
        key = key if key is not None else k
        sz = size if size != (None, None) else s
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y

        super().__init__(ELEM_TYPE_FRAME, background_color=background_color, text_color=title_color, size=sz,
                         font=font, pad=pad, key=key, tooltip=tooltip, visible=visible, metadata=metadata)
        return


    def add_row(self, *args):
        """
        Not recommended user call.  Used to add rows of Elements to the Frame Element.

        :param *args: The list of elements for this row
        :type *args:  List[Element]
        """
        NumRows = len(self.Rows)  # number of existing rows is our row number
        CurrentRowNumber = NumRows  # this row's number
        CurrentRow = []  # start with a blank row and build up
        # -------------------------  Add the elements to a row  ------------------------- #
        for i, element in enumerate(args):  # Loop through list of elements and add them to the row
            if type(element) == list:
                PopupError('Error creating Frame layout',
                           'Layout has a LIST instead of an ELEMENT',
                           'This sometimes means you have a badly placed ]',
                           'The offensive list is:',
                           element,
                           'This list will be stripped from your layout',
                           keep_on_top=True
                           )
                continue
            elif callable(element) and not isinstance(element, Element):
                PopupError('Error creating Frame layout',
                           'Layout has a FUNCTION instead of an ELEMENT',
                           'This likely means you are missing () from your layout',
                           'The offensive list is:',
                           element,
                           'This item will be stripped from your layout',
                           keep_on_top=True)
                continue
            if element.ParentContainer is not None:
                warnings.warn(
                    '*** YOU ARE ATTEMPTING TO REUSE AN ELEMENT IN YOUR LAYOUT! Once placed in a layout, an element cannot be used in another layout. ***',
                    UserWarning)
                _error_popup_with_traceback('Error creating Frame layout',
                                            'The layout specified has already been used',
                                            'You MUST start witha "clean", unused layout every time you create a window',
                                            'The offensive Element = ',
                                            element,
                                            'and has a key = ', element.Key,
                                            'This item will be stripped from your layout',
                                            'Hint - try printing your layout and matching the IDs "print(layout)"',
                                            )
                continue
            element.Position = (CurrentRowNumber, i)
            element.ParentContainer = self
            CurrentRow.append(element)
            if element.Key is not None:
                self.UseDictionary = True
        # -------------------------  Append the row to list of Rows  ------------------------- #
        self.Rows.append(CurrentRow)

    def layout(self, rows):
        """
        Can use like the Window.Layout method, but it's better to use the layout parameter when creating

        :param rows: The rows of Elements
        :type rows:  List[List[Element]]
        :return:     Used for chaining
        :rtype:      (Frame)
        """

        for row in rows:
            try:
                iter(row)
            except TypeError:
                PopupError('Error creating Frame layout',
                           'Your row is not an iterable (e.g. a list)',
                           'Instead of a list, the type found was {}'.format(type(row)),
                           'The offensive row = ',
                           row,
                           'This item will be stripped from your layout', keep_on_top=True, image=_random_error_emoji())
                continue
            self.AddRow(*row)
        return self

    def _GetElementAtLocation(self, location):
        """
        Not user callable. Used to find the Element at a row, col position within the layout

        :param location: (row, column) position of the element to find in layout
        :type location:  (int, int)
        :return:         (Element) The element found at the location
        :rtype:          (Element)
        """

        (row_num, col_num) = location
        row = self.Rows[row_num]
        element = row[col_num]
        return element

    def update(self, value=None, visible=None):
        """
        Changes some of the settings for the Frame Element. Must call `Window.Read` or `Window.Finalize` prior

        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param value:   New text value to show on frame
        :type value:    (Any)
        :param visible: control visibility of element
        :type visible:  (bool)
        """
        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in Frame.update - The window was closed')
            return

        if visible is False:
            self._pack_forget_save_settings()
            # self.TKFrame.pack_forget()
        elif visible is True:
            self._pack_restore_settings()
            # self.TKFrame.pack(padx=self.pad_used[0], pady=self.pad_used[1])
        if value is not None:
            self.TKFrame.config(text=str(value))
        if visible is not None:
            self._visible = visible

    AddRow = add_row
    Layout = layout
    Update = update


Fr = Frame


# ---------------------------------------------------------------------- #
#                           Vertical Separator                           #
# ---------------------------------------------------------------------- #
class VerticalSeparator(Element):
    """
    Vertical Separator Element draws a vertical line at the given location. It will span 1 "row". Usually paired with
    Column Element if extra height is needed
    """

    def __init__(self, color=None, pad=None, p=None, key=None, k=None):
        """
        :param color: Color of the line. Defaults to theme's text color. Can be name or #RRGGBB format
        :type color:  (str)
        :param pad:   Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:    (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:     Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:      (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param key:   Value that uniquely identifies this element from all other elements. Used when Finding an element or in return values. Must be unique to the window
        :type key:    str | int | tuple | object
        :param k:     Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:      str | int | tuple | object
        """
        key = key if key is not None else k
        pad = pad if pad is not None else p
        self.expand_x = None
        self.expand_y = None
        self.Orientation = 'vertical'  # for now only vertical works
        self.color = color if color is not None else theme_text_color()
        super().__init__(ELEM_TYPE_SEPARATOR, pad=pad, key=key)


VSeperator = VerticalSeparator
VSeparator = VerticalSeparator
VSep = VerticalSeparator


# ---------------------------------------------------------------------- #
#                           Horizontal Separator                           #
# ---------------------------------------------------------------------- #
class HorizontalSeparator(Element):
    """
    Horizontal Separator Element draws a Horizontal line at the given location.
    """

    def __init__(self, color=None, pad=None, p=None, key=None, k=None):
        """
        :param color: Color of the line. Defaults to theme's text color. Can be name or #RRGGBB format
        :type color:  (str)
        :param pad:   Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:    (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:     Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:      (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param key:   Value that uniquely identifies this element from all other elements. Used when Finding an element or in return values. Must be unique to the window
        :type key:    str | int | tuple | object
        :param k:     Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:      str | int | tuple | object
        """

        self.Orientation = 'horizontal'  # for now only vertical works
        self.color = color if color is not None else theme_text_color()
        self.expand_x = True
        self.expand_y = None
        key = key if key is not None else k
        pad = pad if pad is not None else p

        super().__init__(ELEM_TYPE_SEPARATOR, pad=pad, key=key)


HSeparator = HorizontalSeparator
HSep = HorizontalSeparator


# ---------------------------------------------------------------------- #
#                           Sizegrip                                     #
# ---------------------------------------------------------------------- #
class Sizegrip(Element):
    """
        Sizegrip element will be added to the bottom right corner of your window.
        It should be placed on the last row of your window along with any other elements on that row.
        The color will match the theme's background color.
    """

    def __init__(self, background_color=None, pad=None, p=(0,0), key=None, k=None):
        """
        Sizegrip Element
        :param background_color: color to use for the background of the grip
        :type background_color:  str
        :param pad:   Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:    (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:     Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:      (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param key:   Value that uniquely identifies this element from all other elements. Used when Finding an element or in return values. Must be unique to the window
        :type key:    str | int | tuple | object
        :param k:     Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:      str | int | tuple | object
        """

        bg = background_color if background_color is not None else theme_background_color()
        pad = pad if pad is not None else p
        key = key if key is not None else k


        super().__init__(ELEM_TYPE_SIZEGRIP, background_color=bg,key=key, pad=pad)


SGrip = Sizegrip


# ---------------------------------------------------------------------- #
#                           Tab                                          #
# ---------------------------------------------------------------------- #
class Tab(Element):
    """
    Tab Element is another "Container" element that holds a layout and displays a tab with text. Used with TabGroup only
    Tabs are never placed directly into a layout.  They are always "Contained" in a TabGroup layout
    """

    def __init__(self, title, layout, title_color=None, background_color=None, font=None, pad=None, p=None, disabled=False,
                 border_width=None, key=None, k=None, tooltip=None, right_click_menu=None, expand_x=False, expand_y=False, visible=True, element_justification='left', image_source=None, image_subsample=None, image_zoom=None, metadata=None):
        """
        :param title:                 text to show on the tab
        :type title:                  (str)
        :param layout:                The element layout that will be shown in the tab
        :type layout:                 List[List[Element]]
        :param title_color:           color of the tab text (note not currently working on tkinter)
        :type title_color:            (str)
        :param background_color:      color of background of the entire layout
        :type background_color:       (str)
        :param font:                  NOT USED in the tkinter port
        :type font:                   (str or (str, int[, str]) or None)
        :param pad:                   Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:                    (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                     Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                      (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param disabled:              If True button will be created disabled
        :type disabled:               (bool)
        :param border_width:          NOT USED in tkinter port
        :type border_width:           (int)
        :param key:                   Value that uniquely identifies this element from all other elements. Used when Finding an element or in return values. Must be unique to the window
        :type key:                    str | int | tuple | object
        :param k:                     Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                      str | int | tuple | object
        :param tooltip:               text, that will appear when mouse hovers over the element
        :type tooltip:                (str)
        :param right_click_menu:      A list of lists of Menu items to show when this element is right clicked. See user docs for exact format.
        :type right_click_menu:       List[List[ List[str] | str ]]
        :param expand_x:              If True the element will automatically expand in the X direction to fill available space
        :type expand_x:               (bool)
        :param expand_y:              If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:               (bool)
        :param visible:               set visibility state of the element
        :type visible:                (bool)
        :param element_justification: All elements inside the Tab will have this justification 'left', 'right', 'center' are valid values
        :type element_justification:  (str)
        :param image_source:          A filename or a base64 bytes of an image to place on the Tab
        :type image_source:            str | bytes | None
        :param image_subsample:       amount to reduce the size of the image. Divides the size by this number. 2=1/2, 3=1/3, 4=1/4, etc
        :type image_subsample:        (int)
        :param image_zoom:            amount to increase the size of the image. 2=twice size, 3=3 times, etc
        :type image_zoom:             (int)
        :param metadata:              User metadata that can be set to ANYTHING
        :type metadata:               (Any)
        """

        filename = data = None
        if image_source is not None:
            if isinstance(image_source, bytes):
                data = image_source
            elif isinstance(image_source, str):
                filename = image_source
            else:
                warnings.warn('Image element - source is not a valid type: {}'.format(type(image_source)), UserWarning)

        self.Filename = filename
        self.Data = data
        self.ImageSubsample = image_subsample
        self.zoom = int(image_zoom) if image_zoom is not None else None
        self.UseDictionary = False
        self.ReturnValues = None
        self.ReturnValuesList = []
        self.ReturnValuesDictionary = {}
        self.DictionaryKeyCounter = 0
        self.ParentWindow = None
        self.Rows = []
        self.TKFrame = None
        self.Widget = None  # type: tk.Frame
        self.Title = title
        self.BorderWidth = border_width
        self.Disabled = disabled
        self.ParentNotebook = None
        self.TabID = None
        self.BackgroundColor = background_color if background_color is not None else DEFAULT_BACKGROUND_COLOR
        self.RightClickMenu = right_click_menu
        self.ContainerElemementNumber = Window._GetAContainerNumber()
        self.ElementJustification = element_justification
        key = key if key is not None else k
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y

        self.Layout(layout)

        super().__init__(ELEM_TYPE_TAB, background_color=background_color, text_color=title_color, font=font, pad=pad, key=key, tooltip=tooltip,
                         visible=visible, metadata=metadata)
        return

    def add_row(self, *args):
        """
        Not recommended use call.  Used to add rows of Elements to the Frame Element.

        :param *args: The list of elements for this row
        :type *args:  List[Element]
        """
        NumRows = len(self.Rows)  # number of existing rows is our row number
        CurrentRowNumber = NumRows  # this row's number
        CurrentRow = []  # start with a blank row and build up
        # -------------------------  Add the elements to a row  ------------------------- #
        for i, element in enumerate(args):  # Loop through list of elements and add them to the row
            if type(element) == list:
                popup_error_with_traceback('Error creating Tab layout',
                           'Layout has a LIST instead of an ELEMENT',
                           'This sometimes means you have a badly placed ]',
                           'The offensive list is:',
                           element,
                           'This list will be stripped from your layout')
                continue
            elif callable(element) and not isinstance(element, Element):
                popup_error_with_traceback('Error creating Tab layout',
                           'Layout has a FUNCTION instead of an ELEMENT',
                           'This likely means you are missing () from your layout',
                           'The offensive list is:',
                           element,
                           'This item will be stripped from your layout')
                continue
            if element.ParentContainer is not None:
                warnings.warn(
                    '*** YOU ARE ATTEMPTING TO REUSE AN ELEMENT IN YOUR LAYOUT! Once placed in a layout, an element cannot be used in another layout. ***',
                    UserWarning)
                popup_error_with_traceback('Error creating Tab layout',
                           'The layout specified has already been used',
                           'You MUST start witha "clean", unused layout every time you create a window',
                           'The offensive Element = ',
                           element,
                           'and has a key = ', element.Key,
                           'This item will be stripped from your layout',
                           'Hint - try printing your layout and matching the IDs "print(layout)"')
                continue
            element.Position = (CurrentRowNumber, i)
            element.ParentContainer = self
            CurrentRow.append(element)
            if element.Key is not None:
                self.UseDictionary = True
        # -------------------------  Append the row to list of Rows  ------------------------- #
        self.Rows.append(CurrentRow)

    def layout(self, rows):
        """
        Not user callable.  Use layout parameter instead. Creates the layout using the supplied rows of Elements

        :param rows: List[List[Element]] The list of rows
        :type rows:  List[List[Element]]
        :return:     (Tab) used for chaining
        :rtype:
        """

        for row in rows:
            try:
                iter(row)
            except TypeError:
                PopupError('Error creating Tab layout',
                           'Your row is not an iterable (e.g. a list)',
                           'Instead of a list, the type found was {}'.format(type(row)),
                           'The offensive row = ',
                           row,
                           'This item will be stripped from your layout', keep_on_top=True, image=_random_error_emoji())
                continue
            self.AddRow(*row)
        return self

    def update(self, title=None, disabled=None, visible=None):
        """
        Changes some of the settings for the Tab Element. Must call `Window.Read` or `Window.Finalize` prior

        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param title:    tab title
        :type title:     (str)
        :param disabled: disable or enable state of the element
        :type disabled:  (bool)
        :param visible:  control visibility of element
        :type visible:   (bool)
        """
        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in Tab.update - The window was closed')
            return


        state = 'normal'
        if disabled is not None:
            self.Disabled = disabled
            if disabled:
                state = 'disabled'
        if visible is False:
            state = 'hidden'
        if visible is not None:
            self._visible = visible

        self.ParentNotebook.tab(self.TabID, state=state)

        if title is not None:
            self.Title = str(title)
            self.ParentNotebook.tab(self.TabID, text=self.Title)
            # self.ParentNotebook.tab(self.ContainerElemementNumber-1, text=self.Title)

        # if visible is False:
        #     self.ParentNotebook.pack_forget()
        # elif visible is True:
        #     self.ParentNotebook.pack()
        return self

    def _GetElementAtLocation(self, location):
        """
        Not user callable. Used to find the Element at a row, col position within the layout

        :param location: (row, column) position of the element to find in layout
        :type location:  (int, int)
        :return:         The element found at the location
        :rtype:          (Element)
        """

        (row_num, col_num) = location
        row = self.Rows[row_num]
        element = row[col_num]
        return element

    def select(self):
        """
        Create a tkinter event that mimics user clicking on a tab. Must have called window.Finalize / Read first!

        """
        # Use a try in case the window has been destoyed
        try:
            self.ParentNotebook.select(self.TabID)
        except Exception as e:
            print('Exception Selecting Tab {}'.format(e))

    AddRow = add_row
    Layout = layout
    Select = select
    Update = update


# ---------------------------------------------------------------------- #
#                           TabGroup                                     #
# ---------------------------------------------------------------------- #
class TabGroup(Element):
    """
    TabGroup Element groups together your tabs into the group of tabs you see displayed in your window
    """

    def __init__(self, layout, tab_location=None, title_color=None, tab_background_color=None, selected_title_color=None, selected_background_color=None,
                 background_color=None, focus_color=None, font=None, change_submits=False, enable_events=False, pad=None, p=None, border_width=None, tab_border_width=None, theme=None, key=None, k=None,
                 size=(None, None), s=(None, None), tooltip=None, right_click_menu=None, expand_x=False, expand_y=False, visible=True, metadata=None):
        """
        :param layout:                    Layout of Tabs. Different than normal layouts. ALL Tabs should be on first row
        :type layout:                     List[List[Tab]]
        :param tab_location:              location that tabs will be displayed. Choices are left, right, top, bottom, lefttop, leftbottom, righttop, rightbottom, bottomleft, bottomright, topleft, topright
        :type tab_location:               (str)
        :param title_color:               color of text on tabs
        :type title_color:                (str)
        :param tab_background_color:      color of all tabs that are not selected
        :type tab_background_color:       (str)
        :param selected_title_color:      color of tab text when it is selected
        :type selected_title_color:       (str)
        :param selected_background_color: color of tab when it is selected
        :type selected_background_color:  (str)
        :param background_color:          color of background area that tabs are located on
        :type background_color:           (str)
        :param focus_color:               color of focus indicator on the tabs
        :type focus_color:                (str)
        :param font:                      specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:                       (str or (str, int[, str]) or None)
        :param change_submits:            * DEPRICATED DO NOT USE. Use `enable_events` instead
        :type change_submits:             (bool)
        :param enable_events:             If True then switching tabs will generate an Event
        :type enable_events:              (bool)
        :param pad:                       Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:                        (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                         Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                          (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param border_width:              width of border around element in pixels
        :type border_width:               (int)
        :param tab_border_width:          width of border around the tabs
        :type tab_border_width:           (int)
        :param theme:                     DEPRICATED - You can only specify themes using set options or when window is created. It's not possible to do it on an element basis
        :type theme:                      (enum)
        :param key:                       Value that uniquely identifies this element from all other elements. Used when Finding an element or in return values. Must be unique to the window
        :type key:                        str | int | tuple | object
        :param k:                         Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                          str | int | tuple | object
        :param size:                      (width, height) w=pixels-wide, h=pixels-high. Either item in tuple can be None to indicate use the computed value and set only 1 direction
        :type size:                       (int|None, int|None)
        :param s:                         Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
        :type s:                          (int|None, int|None)
        :param tooltip:                   text, that will appear when mouse hovers over the element
        :type tooltip:                    (str)
        :param right_click_menu:          A list of lists of Menu items to show when this element is right clicked. See user docs for exact format.
        :type right_click_menu:           List[List[ List[str] | str ]]
        :param expand_x:                  If True the element will automatically expand in the X direction to fill available space
        :type expand_x:                   (bool)
        :param expand_y:                  If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:                   (bool)
        :param visible:                   DEPRECATED  - Should you need to control visiblity for the TabGroup as a whole, place it into a Column element
        :type visible:                    (bool)
        :param metadata:                  User metadata that can be set to ANYTHING
        :type metadata:                   (Any)
        """


        self.UseDictionary = False
        self.ReturnValues = None
        self.ReturnValuesList = []
        self.ReturnValuesDictionary = {}
        self.DictionaryKeyCounter = 0
        self.ParentWindow = None
        self.SelectedTitleColor = selected_title_color if selected_title_color is not None else LOOK_AND_FEEL_TABLE[CURRENT_LOOK_AND_FEEL]['TEXT']
        self.SelectedBackgroundColor = selected_background_color if selected_background_color is not None else LOOK_AND_FEEL_TABLE[CURRENT_LOOK_AND_FEEL][
            'BACKGROUND']
        title_color = title_color if title_color is not None else LOOK_AND_FEEL_TABLE[CURRENT_LOOK_AND_FEEL]['TEXT_INPUT']
        self.TabBackgroundColor = tab_background_color if tab_background_color is not None else LOOK_AND_FEEL_TABLE[CURRENT_LOOK_AND_FEEL]['INPUT']
        self.Rows = []
        self.TKNotebook = None  # type: ttk.Notebook
        self.Widget = None  # type: ttk.Notebook
        self.tab_index_to_key = {}      # has a list of the tabs in the notebook and their associated key
        self.TabCount = 0
        self.BorderWidth = border_width
        self.BackgroundColor = background_color if background_color is not None else DEFAULT_BACKGROUND_COLOR
        self.ChangeSubmits = change_submits or enable_events
        self.TabLocation = tab_location
        self.ElementJustification = 'left'
        self.RightClickMenu = right_click_menu
        self.TabBorderWidth = tab_border_width
        self.FocusColor = focus_color

        key = key if key is not None else k
        sz = size if size != (None, None) else s
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y

        self.Layout(layout)

        super().__init__(ELEM_TYPE_TAB_GROUP, size=sz, background_color=background_color, text_color=title_color, font=font,
                         pad=pad, key=key, tooltip=tooltip, visible=visible, metadata=metadata)
        return

    def add_row(self, *args):
        """
        Not recommended user call.  Used to add rows of Elements to the Frame Element.

        :param *args:     The list of elements for this row
        :type *args:      List[Element]
        """

        NumRows = len(self.Rows)  # number of existing rows is our row number
        CurrentRowNumber = NumRows  # this row's number
        CurrentRow = []  # start with a blank row and build up
        # -------------------------  Add the elements to a row  ------------------------- #
        for i, element in enumerate(args):  # Loop through list of elements and add them to the row
            if type(element) == list:
                PopupError('Error creating Tab layout',
                           'Layout has a LIST instead of an ELEMENT',
                           'This sometimes means you have a badly placed ]',
                           'The offensive list is:',
                           element,
                           'This list will be stripped from your layout', keep_on_top=True, image=_random_error_emoji()
                           )
                continue
            elif callable(element) and not isinstance(element, Element):
                PopupError('Error creating Tab layout',
                           'Layout has a FUNCTION instead of an ELEMENT',
                           'This likely means you are missing () from your layout',
                           'The offensive list is:',
                           element,
                           'This item will be stripped from your layout', keep_on_top=True, image=_random_error_emoji())
                continue
            if element.ParentContainer is not None:
                warnings.warn(
                    '*** YOU ARE ATTEMPTING TO REUSE AN ELEMENT IN YOUR LAYOUT! Once placed in a layout, an element cannot be used in another layout. ***',
                    UserWarning)
                PopupError('Error creating Tab layout',
                           'The layout specified has already been used',
                           'You MUST start witha "clean", unused layout every time you create a window',
                           'The offensive Element = ',
                           element,
                           'and has a key = ', element.Key,
                           'This item will be stripped from your layout',
                           'Hint - try printing your layout and matching the IDs "print(layout)"', keep_on_top=True, image=_random_error_emoji())
                continue
            element.Position = (CurrentRowNumber, i)
            element.ParentContainer = self
            CurrentRow.append(element)
            if element.Key is not None:
                self.UseDictionary = True
        # -------------------------  Append the row to list of Rows  ------------------------- #
        self.Rows.append(CurrentRow)

    def layout(self, rows):
        """
        Can use like the Window.Layout method, but it's better to use the layout parameter when creating

        :param rows: The rows of Elements
        :type rows:  List[List[Element]]
        :return:     Used for chaining
        :rtype:      (Frame)
        """
        for row in rows:
            try:
                iter(row)
            except TypeError:
                PopupError('Error creating Tab layout',
                           'Your row is not an iterable (e.g. a list)',
                           'Instead of a list, the type found was {}'.format(type(row)),
                           'The offensive row = ',
                           row,
                           'This item will be stripped from your layout', keep_on_top=True, image=_random_error_emoji())
                continue
            self.AddRow(*row)
        return self

    def _GetElementAtLocation(self, location):
        """
        Not user callable. Used to find the Element at a row, col position within the layout

        :param location: (row, column) position of the element to find in layout
        :type location:  (int, int)
        :return:         The element found at the location
        :rtype:          (Element)
        """

        (row_num, col_num) = location
        row = self.Rows[row_num]
        element = row[col_num]
        return element

    def find_key_from_tab_name(self, tab_name):
        """
        Searches through the layout to find the key that matches the text on the tab. Implies names should be unique

        :param tab_name: name of a tab
        :type tab_name:  str
        :return:         Returns the key or None if no key found
        :rtype:          key | None
        """
        for row in self.Rows:
            for element in row:
                if element.Title == tab_name:
                    return element.Key
        return None


    def find_currently_active_tab_key(self):
        """
        Returns the key for the currently active tab in this TabGroup
        :return:    Returns the key or None of no key found
        :rtype:     key | None
        """
        try:
            current_index = self.TKNotebook.index('current')
            key = self.tab_index_to_key.get(current_index, None)
        except:
            key = None

        return key

    def get(self):
        """
        Returns the current value for the Tab Group, which will be the currently selected tab's KEY or the text on
        the tab if no key is defined.  Returns None if an error occurs.
        Note that this is exactly the same data that would be returned from a call to Window.read. Are you sure you
        are using this method correctly?

        :return: The key of the currently selected tab or None if there is an error
        :rtype:  Any | None
        """

        try:
            current_index = self.TKNotebook.index('current')
            key = self.tab_index_to_key.get(current_index, None)
        except:
            key = None

        return key

    def add_tab(self, tab_element):
        """
        Add a new tab to an existing TabGroup
        This call was written so that tabs can be added at runtime as your user performs operations.
        Your Window should already be created and finalized.

        :param tab_element: A Tab Element that has a layout in it
        :type tab_element:  Tab
        """

        self.add_row(tab_element)
        tab_element.TKFrame = tab_element.Widget = tk.Frame(self.TKNotebook)
        form = self.ParentForm
        form._BuildKeyDictForWindow(form, tab_element, form.AllKeysDict)
        form.AllKeysDict[tab_element.Key] = tab_element
        # Pack the tab's layout into the tab. NOTE - This does NOT pack the Tab itself... for that see below...
        PackFormIntoFrame(tab_element, tab_element.TKFrame, self.ParentForm)

        # - This is below -    Perform the same operation that is performed when a Tab is packed into the window.
        # If there's an image in the tab, then do the imagey-stuff
        # ------------------- start of imagey-stuff -------------------
        try:
            if tab_element.Filename is not None:
                photo = tk.PhotoImage(file=tab_element.Filename)
            elif tab_element.Data is not None:
                photo = tk.PhotoImage(data=tab_element.Data)
            else:
                photo = None

            if tab_element.ImageSubsample and photo is not None:
                photo = photo.subsample(tab_element.ImageSubsample)
                # print('*ERROR laying out form.... Image Element has no image specified*')
        except Exception as e:
            photo = None
            _error_popup_with_traceback('Your Window has an Tab Element with an IMAGE problem',
                                        'The traceback will show you the Window with the problem layout',
                                        'Look in this Window\'s layout for an Image tab_element that has a key of {}'.format(tab_element.Key),
                                        'The error occuring is:', e)

        tab_element.photo = photo
        # add the label
        if photo is not None:
            width, height = photo.width(), photo.height()
            tab_element.tktext_label = tk.Label(tab_element.ParentRowFrame, image=photo, width=width, height=height, bd=0)
        else:
            tab_element.tktext_label = tk.Label(tab_element.ParentRowFrame, bd=0)
        # ------------------- end of imagey-stuff -------------------

        state = 'normal'
        if tab_element.Disabled:
            state = 'disabled'
        if tab_element.visible is False:
            state = 'hidden'
        if photo is not None:
            self.TKNotebook.add(tab_element.TKFrame, text=tab_element.Title, compound=tk.LEFT, state=state, image=photo)
        else:
            self.TKNotebook.add(tab_element.TKFrame, text=tab_element.Title, state=state)
        tab_element.ParentNotebook = self.TKNotebook
        tab_element.TabID = self.TabCount
        tab_element.ParentForm = self.ParentForm
        self.TabCount += 1
        if tab_element.BackgroundColor != COLOR_SYSTEM_DEFAULT and tab_element.BackgroundColor is not None:
            tab_element.TKFrame.configure(background=tab_element.BackgroundColor, highlightbackground=tab_element.BackgroundColor,
                                          highlightcolor=tab_element.BackgroundColor)
        if tab_element.BorderWidth is not None:
            tab_element.TKFrame.configure(borderwidth=tab_element.BorderWidth)
        if tab_element.Tooltip is not None:
            tab_element.TooltipObject = ToolTip(tab_element.TKFrame, text=tab_element.Tooltip, timeout=DEFAULT_TOOLTIP_TIME)
        _add_right_click_menu(tab_element, form)


    def update(self, visible=None):
        """
        Enables changing the visibility

        :param visible:  control visibility of element
        :type visible:   (bool)
        """
        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in TabGroup.update - The window was closed')
            return

        if visible is False:
            self._pack_forget_save_settings()
        elif visible is True:
            self._pack_restore_settings()

        if visible is not None:
            self._visible = visible




    AddRow = add_row
    FindKeyFromTabName = find_key_from_tab_name
    Get = get
    Layout = layout


# ---------------------------------------------------------------------- #
#                           Slider                                       #
# ---------------------------------------------------------------------- #
class Slider(Element):
    """
    A slider, horizontal or vertical
    """

    def __init__(self, range=(None, None), default_value=None, resolution=None, tick_interval=None, orientation=None,
                 disable_number_display=False, border_width=None, relief=None, change_submits=False,
                 enable_events=False, disabled=False, size=(None, None), s=(None, None), font=None, background_color=None,
                 text_color=None, trough_color=None, key=None, k=None, pad=None, p=None, expand_x=False, expand_y=False, tooltip=None, visible=True, metadata=None):
        """
        :param range:                  slider's range (min value, max value)
        :type range:                   (int, int) | Tuple[float, float]
        :param default_value:          starting value for the slider
        :type default_value:           int | float
        :param resolution:             the smallest amount the slider can be moved
        :type resolution:              int | float
        :param tick_interval:          how often a visible tick should be shown next to slider
        :type tick_interval:           int | float
        :param orientation:            'horizontal' or 'vertical' ('h' or 'v' also work)
        :type orientation:             (str)
        :param disable_number_display: if True no number will be displayed by the Slider Element
        :type disable_number_display:  (bool)
        :param border_width:           width of border around element in pixels
        :type border_width:            (int)
        :param relief:                 relief style. Use constants - RELIEF_RAISED RELIEF_SUNKEN RELIEF_FLAT RELIEF_RIDGE RELIEF_GROOVE RELIEF_SOLID
        :type relief:                  str | None
        :param change_submits:         * DEPRICATED DO NOT USE. Use `enable_events` instead
        :type change_submits:          (bool)
        :param enable_events:          If True then moving the slider will generate an Event
        :type enable_events:           (bool)
        :param disabled:               set disable state for element
        :type disabled:                (bool)
        :param size:                   (l=length chars/rows, w=width pixels)
        :type size:                    (int, int)
        :param s:                      Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
        :type s:                       (int, int)  | (None, None)
        :param font:                   specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:                    (str or (str, int[, str]) or None)
        :param background_color:       color of slider's background
        :type background_color:        (str)
        :param text_color:             color of the slider's text
        :type text_color:              (str)
        :param trough_color:           color of the slider's trough
        :type trough_color:            (str)
        :param key:                    Value that uniquely identifies this element from all other elements. Used when Finding an element or in return values. Must be unique to the window
        :type key:                     str | int | tuple | object
        :param k:                      Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                       str | int | tuple | object
        :param pad:                    Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:                     (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                      Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                       (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param expand_x:               If True the element will automatically expand in the X direction to fill available space
        :type expand_x:                (bool)
        :param expand_y:               If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:                (bool)
        :param tooltip:                text, that will appear when mouse hovers over the element
        :type tooltip:                 (str)
        :param visible:                set visibility state of the element
        :type visible:                 (bool)
        :param metadata:               User metadata that can be set to ANYTHING
        :type metadata:                (Any)
        """

        self.TKScale = self.Widget = None  # type: tk.Scale
        self.Range = (1, 10) if range == (None, None) else range
        self.DefaultValue = self.Range[0] if default_value is None else default_value
        self.Orientation = orientation if orientation else DEFAULT_SLIDER_ORIENTATION
        self.BorderWidth = border_width if border_width else DEFAULT_SLIDER_BORDER_WIDTH
        self.Relief = relief if relief else DEFAULT_SLIDER_RELIEF
        self.Resolution = 1 if resolution is None else resolution
        self.ChangeSubmits = change_submits or enable_events
        self.Disabled = disabled
        self.TickInterval = tick_interval
        self.DisableNumericDisplay = disable_number_display
        self.TroughColor = trough_color or DEFAULT_SCROLLBAR_COLOR
        sz = size if size != (None, None) else s
        temp_size = sz
        if temp_size == (None, None):
            temp_size = (20, 20) if self.Orientation.startswith('h') else (8, 20)
        key = key if key is not None else k
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y

        super().__init__(ELEM_TYPE_INPUT_SLIDER, size=temp_size, font=font, background_color=background_color,
                         text_color=text_color, key=key, pad=pad, tooltip=tooltip, visible=visible, metadata=metadata)
        return

    def update(self, value=None, range=(None, None), disabled=None, visible=None):
        """
        Changes some of the settings for the Slider Element. Must call `Window.Read` or `Window.Finalize` prior

        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param value:    sets current slider value
        :type value:     int | float
        :param range:    Sets a new range for slider
        :type range:     (int, int) | Tuple[float, float
        :param disabled: disable or enable state of the element
        :type disabled:  (bool)
        :param visible:  control visibility of element
        :type visible:   (bool)
        """
        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in Slider.update - The window was closed')
            return


        if range != (None, None):
            self.TKScale.config(from_=range[0], to_=range[1])
        if value is not None:
            try:
                self.TKIntVar.set(value)
            except:
                pass
            self.DefaultValue = value
        if disabled is True:
            self.TKScale['state'] = 'disabled'
        elif disabled is False:
            self.TKScale['state'] = 'normal'
        self.Disabled = disabled if disabled is not None else self.Disabled

        if visible is False:
            self._pack_forget_save_settings()
        elif visible is True:
            self._pack_restore_settings()

        if visible is not None:
            self._visible = visible


    def _SliderChangedHandler(self, event):
        """
        Not user callable.  Callback function for when slider is moved.

        :param event: (event) the event data provided by tkinter. Unknown format. Not used.
        :type event:
        """

        if self.Key is not None:
            self.ParentForm.LastButtonClicked = self.Key
        else:
            self.ParentForm.LastButtonClicked = ''
        self.ParentForm.FormRemainedOpen = True
        # if self.ParentForm.CurrentlyRunningMainloop:
        #     self.ParentForm.TKroot.quit()  # kick the users out of the mainloop
        _exit_mainloop(self.ParentForm)

    Update = update


Sl = Slider


# ---------------------------------------------------------------------- #
#                          TkFixedFrame (Used by Column)                 #
# ---------------------------------------------------------------------- #
class TkFixedFrame(tk.Frame):
    """
    A tkinter frame that is used with Column Elements that do not have a scrollbar
    """

    def __init__(self, master, **kwargs):
        """
        :param master:   The parent widget
        :type master:    (tk.Widget)
        :param **kwargs: The keyword args
        :type **kwargs:
        """
        tk.Frame.__init__(self, master, **kwargs)

        self.canvas = tk.Canvas(self)

        self.canvas.pack(side='left', fill='both', expand=True)

        # reset the view
        self.canvas.xview_moveto(0)
        self.canvas.yview_moveto(0)

        # create a frame inside the canvas which will be scrolled with it
        self.TKFrame = tk.Frame(self.canvas, **kwargs)
        self.frame_id = self.canvas.create_window(0, 0, window=self.TKFrame, anchor='nw')
        self.canvas.config(borderwidth=0, highlightthickness=0)
        self.TKFrame.config(borderwidth=0, highlightthickness=0)
        self.config(borderwidth=0, highlightthickness=0)


# ---------------------------------------------------------------------- #
#                          TkScrollableFrame (Used by Column)            #
# ---------------------------------------------------------------------- #
class TkScrollableFrame(tk.Frame):
    """
    A frame with one or two scrollbars.  Used to make Columns with scrollbars
    """

    def __init__(self, master, vertical_only, element, window, **kwargs):
        """
        :param master:        The parent widget
        :type master:         (tk.Widget)
        :param vertical_only: if True the only a vertical scrollbar will be shown
        :type vertical_only:  (bool)
        :param element:       The element containing this object
        :type element:        (Column)
        """
        tk.Frame.__init__(self, master, **kwargs)
        # create a canvas object and a vertical scrollbar for scrolling it

        self.canvas = tk.Canvas(self)
        element.Widget = self.canvas
        # Okay, we're gonna make a list. Containing the y-min, x-min, y-max, and x-max of the frame
        element.element_frame = self
        _make_ttk_scrollbar(element, 'v', window)
        # element.vsb = tk.Scrollbar(self, orient=tk.VERTICAL)
        element.vsb.pack(side='right', fill='y', expand='false')

        if not vertical_only:
            _make_ttk_scrollbar(element, 'h', window)
            # self.hscrollbar = tk.Scrollbar(self, orient=tk.HORIZONTAL)
            element.hsb.pack(side='bottom', fill='x', expand='false')
            self.canvas.config(xscrollcommand=element.hsb.set)
            # self.canvas = tk.Canvas(self, )
        # else:
        #     self.canvas = tk.Canvas(self)

        self.canvas.config(yscrollcommand=element.vsb.set)
        self.canvas.pack(side='left', fill='both', expand=True)
        element.vsb.config(command=self.canvas.yview)
        if not vertical_only:
            element.hsb.config(command=self.canvas.xview)

        # reset the view
        self.canvas.xview_moveto(0)
        self.canvas.yview_moveto(0)

        # create a frame inside the canvas which will be scrolled with it
        self.TKFrame = tk.Frame(self.canvas, **kwargs)
        self.frame_id = self.canvas.create_window(0, 0, window=self.TKFrame, anchor='nw')
        self.canvas.config(borderwidth=0, highlightthickness=0)
        self.TKFrame.config(borderwidth=0, highlightthickness=0)
        self.config(borderwidth=0, highlightthickness=0)

        # Canvas can be: master, canvas, TKFrame

        # Chr0nic

        # self.unhookMouseWheel(None)
        # self.TKFrame.bind("<Enter>", self.hookMouseWheel)
        # self.TKFrame.bind("<Leave>", self.unhookMouseWheel)
        # self.bind('<Configure>', self.set_scrollregion)


        self.unhookMouseWheel(None)
        self.canvas.bind('<Enter>', self.hookMouseWheel)
        self.canvas.bind('<Leave>', self.unhookMouseWheel)
        self.bind('<Configure>', self.set_scrollregion)


    # Chr0nic
    def hookMouseWheel(self, e):
        # print("enter")
        VarHolder.canvas_holder = self.canvas
        self.canvas.bind_all('<4>', self.yscroll, add='+')
        self.canvas.bind_all('<5>', self.yscroll, add='+')
        self.canvas.bind_all('<MouseWheel>', self.yscroll, add='+')
        self.canvas.bind_all('<Shift-MouseWheel>', self.xscroll, add='+')

    # Chr0nic
    def unhookMouseWheel(self, e):
        # print("leave")
        VarHolder.canvas_holder = None
        self.canvas.unbind_all('<4>')
        self.canvas.unbind_all('<5>')
        self.canvas.unbind_all('<MouseWheel>')
        self.canvas.unbind_all('<Shift-MouseWheel>')

    def resize_frame(self, e):
        self.canvas.itemconfig(self.frame_id, height=e.height, width=e.width)

    def yscroll(self, event):
        if self.canvas.yview() == (0.0, 1.0):
            return
        if event.num == 5 or event.delta < 0:
            self.canvas.yview_scroll(1, 'unit')
        elif event.num == 4 or event.delta > 0:
            self.canvas.yview_scroll(-1, 'unit')

    def xscroll(self, event):
        if event.num == 5 or event.delta < 0:
            self.canvas.xview_scroll(1, 'unit')
        elif event.num == 4 or event.delta > 0:
            self.canvas.xview_scroll(-1, 'unit')

    def bind_mouse_scroll(self, parent, mode):
        # ~~ Windows only
        parent.bind('<MouseWheel>', mode)
        # ~~ Unix only
        parent.bind('<Button-4>', mode)
        parent.bind('<Button-5>', mode)

    def set_scrollregion(self, event=None):
        """ Set the scroll region on the canvas """
        self.canvas.configure(scrollregion=self.canvas.bbox('all'))


# ---------------------------------------------------------------------- #
#                           Column                                       #
# ---------------------------------------------------------------------- #
class Column(Element):
    """
    A container element that is used to create a layout within your window's layout
    """

    def __init__(self, layout, background_color=None, size=(None, None), s=(None, None), size_subsample_width=1, size_subsample_height=2, pad=None, p=None, scrollable=False,
                 vertical_scroll_only=False, right_click_menu=None, key=None, k=None, visible=True, justification=None, element_justification=None,
                 vertical_alignment=None, grab=None, expand_x=None, expand_y=None, metadata=None,
                 sbar_trough_color=None, sbar_background_color=None, sbar_arrow_color=None, sbar_width=None, sbar_arrow_width=None,
                 sbar_frame_color=None, sbar_relief=None):
        """
        :param layout:                      Layout that will be shown in the Column container
        :type layout:                       List[List[Element]]
        :param background_color:            color of background of entire Column
        :type background_color:             (str)
        :param size:                        (width, height) size in pixels (doesn't work quite right, sometimes only 1 dimension is set by tkinter. Use a Sizer Element to help set sizes
        :type size:                         (int | None, int | None)
        :param s:                           Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
        :type s:                            (int | None, int | None)
        :param size_subsample_width:        Determines the size of a scrollable column width based on 1/size_subsample * required size. 1 = match the contents exactly, 2 = 1/2 contents size, 3 = 1/3. Can be a fraction to make larger than required.
        :type size_subsample_width:         (float)
        :param size_subsample_height:       Determines the size of a scrollable height based on 1/size_subsample * required size. 1 = match the contents exactly, 2 = 1/2 contents size, 3 = 1/3. Can be a fraction to make larger than required..
        :type size_subsample_height:        (float)
        :param pad:                         Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:                          (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                           Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                            (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param scrollable:                  if True then scrollbars will be added to the column. If you update the contents of a scrollable column, be sure and call Column.contents_changed also
        :type scrollable:                   (bool)
        :param vertical_scroll_only:        if True then no horizontal scrollbar will be shown if a scrollable column
        :type vertical_scroll_only:         (bool)
        :param right_click_menu:            A list of lists of Menu items to show when this element is right clicked. See user docs for exact format.
        :type right_click_menu:             List[List[ List[str] | str ]]
        :param key:                         Value that uniquely identifies this element from all other elements. Used when Finding an element or in return values. Must be unique to the window
        :type key:                          str | int | tuple | object
        :param k:                           Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                            str | int | tuple | object
        :param visible:                     set visibility state of the element
        :type visible:                      (bool)
        :param justification:               set justification for the Column itself. Note entire row containing the Column will be affected
        :type justification:                (str)
        :param element_justification:       All elements inside the Column will have this justification 'left', 'right', 'center' are valid values
        :type element_justification:        (str)
        :param vertical_alignment:          Place the column at the 'top', 'center', 'bottom' of the row (can also use t,c,r). Defaults to no setting (tkinter decides)
        :type vertical_alignment:           (str)
        :param grab:                        If True can grab this element and move the window around. Default is False
        :type grab:                         (bool)
        :param expand_x:                    If True the column will automatically expand in the X direction to fill available space
        :type expand_x:                     (bool)
        :param expand_y:                    If True the column will automatically expand in the Y direction to fill available space
        :type expand_y:                     (bool)
        :param metadata:                    User metadata that can be set to ANYTHING
        :type metadata:                     (Any)
        :param sbar_trough_color:           Scrollbar color of the trough
        :type sbar_trough_color:            (str)
        :param sbar_background_color:       Scrollbar color of the background of the arrow buttons at the ends AND the color of the "thumb" (the thing you grab and slide). Switches to arrow color when mouse is over
        :type sbar_background_color:        (str)
        :param sbar_arrow_color:            Scrollbar color of the arrow at the ends of the scrollbar (it looks like a button). Switches to background color when mouse is over
        :type sbar_arrow_color:             (str)
        :param sbar_width:                  Scrollbar width in pixels
        :type sbar_width:                   (int)
        :param sbar_arrow_width:            Scrollbar width of the arrow on the scrollbar. It will potentially impact the overall width of the scrollbar
        :type sbar_arrow_width:             (int)
        :param sbar_frame_color:            Scrollbar Color of frame around scrollbar (available only on some ttk themes)
        :type sbar_frame_color:             (str)
        :param sbar_relief:                 Scrollbar relief that will be used for the "thumb" of the scrollbar (the thing you grab that slides). Should be a constant that is defined at starting with "RELIEF_" - RELIEF_RAISED, RELIEF_SUNKEN, RELIEF_FLAT, RELIEF_RIDGE, RELIEF_GROOVE, RELIEF_SOLID
        :type sbar_relief:                  (str)
        """


        self.UseDictionary = False
        self.ReturnValues = None
        self.ReturnValuesList = []
        self.ReturnValuesDictionary = {}
        self.DictionaryKeyCounter = 0
        self.ParentWindow = None
        self.ParentPanedWindow = None
        self.Rows = []
        self.TKFrame = None
        self.TKColFrame = None  # type: tk.Frame
        self.Scrollable = scrollable
        self.VerticalScrollOnly = vertical_scroll_only

        self.RightClickMenu = right_click_menu
        bg = background_color if background_color is not None else DEFAULT_BACKGROUND_COLOR
        self.ContainerElemementNumber = Window._GetAContainerNumber()
        self.ElementJustification = element_justification
        self.Justification = justification
        self.VerticalAlignment = vertical_alignment
        key = key if key is not None else k
        self.Grab = grab
        self.expand_x = expand_x
        self.expand_y = expand_y
        self.Layout(layout)
        sz = size if size != (None, None) else s
        pad = pad if pad is not None else p
        self.size_subsample_width = size_subsample_width
        self.size_subsample_height = size_subsample_height

        super().__init__(ELEM_TYPE_COLUMN, background_color=bg, size=sz, pad=pad, key=key, visible=visible, metadata=metadata,
                         sbar_trough_color=sbar_trough_color, sbar_background_color=sbar_background_color, sbar_arrow_color=sbar_arrow_color, sbar_width=sbar_width, sbar_arrow_width=sbar_arrow_width, sbar_frame_color=sbar_frame_color, sbar_relief=sbar_relief)
        return

    def add_row(self, *args):
        """
        Not recommended user call.  Used to add rows of Elements to the Column Element.

        :param *args: The list of elements for this row
        :type *args:  List[Element]
        """

        NumRows = len(self.Rows)  # number of existing rows is our row number
        CurrentRowNumber = NumRows  # this row's number
        CurrentRow = []  # start with a blank row and build up
        # -------------------------  Add the elements to a row  ------------------------- #
        for i, element in enumerate(args):  # Loop through list of elements and add them to the row
            if type(element) == list:
                PopupError('Error creating Column layout',
                           'Layout has a LIST instead of an ELEMENT',
                           'This sometimes means you have a badly placed ]',
                           'The offensive list is:',
                           element,
                           'This list will be stripped from your layout', keep_on_top=True, image=_random_error_emoji()
                           )
                continue
            elif callable(element) and not isinstance(element, Element):
                PopupError('Error creating Column layout',
                           'Layout has a FUNCTION instead of an ELEMENT',
                           'This likely means you are missing () from your layout',
                           'The offensive list is:',
                           element,
                           'This item will be stripped from your layout', keep_on_top=True, image=_random_error_emoji())
                continue
            if element.ParentContainer is not None:
                warnings.warn(
                    '*** YOU ARE ATTEMPTING TO REUSE AN ELEMENT IN YOUR LAYOUT! Once placed in a layout, an element cannot be used in another layout. ***',
                    UserWarning)
                PopupError('Error creating Column layout',
                           'The layout specified has already been used',
                           'You MUST start witha "clean", unused layout every time you create a window',
                           'The offensive Element = ',
                           element,
                           'and has a key = ', element.Key,
                           'This item will be stripped from your layout',
                           'Hint - try printing your layout and matching the IDs "print(layout)"', keep_on_top=True, image=_random_error_emoji())
                continue
            element.Position = (CurrentRowNumber, i)
            element.ParentContainer = self
            CurrentRow.append(element)
            if element.Key is not None:
                self.UseDictionary = True
        # -------------------------  Append the row to list of Rows  ------------------------- #
        self.Rows.append(CurrentRow)

    def layout(self, rows):
        """
        Can use like the Window.Layout method, but it's better to use the layout parameter when creating

        :param rows: The rows of Elements
        :type rows:  List[List[Element]]
        :return:     Used for chaining
        :rtype:      (Column)
        """

        for row in rows:
            try:
                iter(row)
            except TypeError:
                PopupError('Error creating Column layout',
                           'Your row is not an iterable (e.g. a list)',
                           'Instead of a list, the type found was {}'.format(type(row)),
                           'The offensive row = ',
                           row,
                           'This item will be stripped from your layout', keep_on_top=True, image=_random_error_emoji())
                continue
            self.AddRow(*row)
        return self

    def _GetElementAtLocation(self, location):
        """
        Not user callable. Used to find the Element at a row, col position within the layout

        :param location:     (row, column) position of the element to find in layout
        :type  location:     (int, int)
        :return:             The element found at the location
        :rtype:              (Element)
        """

        (row_num, col_num) = location
        row = self.Rows[row_num]
        element = row[col_num]
        return element

    def update(self, visible=None):
        """
        Changes some of the settings for the Column Element. Must call `Window.Read` or `Window.Finalize` prior

        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param visible: control visibility of element
        :type visible:  (bool)
        """
        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in Column.update - The window was closed')
            return

        if self.expand_x and self.expand_y:
            expand = tk.BOTH
        elif self.expand_x:
            expand = tk.X
        elif self.expand_y:
            expand = tk.Y
        else:
            expand = None

        if visible is False:
            if self.TKColFrame:
                self._pack_forget_save_settings()
                # self.TKColFrame.pack_forget()
            if self.ParentPanedWindow:
                self.ParentPanedWindow.remove(self.TKColFrame)
        elif visible is True:
            if self.TKColFrame:
                self._pack_restore_settings()
                # self.TKColFrame.pack(padx=self.pad_used[0], pady=self.pad_used[1], fill=expand)
            if self.ParentPanedWindow:
                self.ParentPanedWindow.add(self.TKColFrame)
        if visible is not None:
            self._visible = visible

    def contents_changed(self):
        """
        When a scrollable column has part of its layout changed by making elements visible or invisible or the
        layout is extended for the Column, then this method needs to be called so that the new scroll area
        is computed to match the new contents.
        """
        self.TKColFrame.canvas.config(scrollregion=self.TKColFrame.canvas.bbox('all'))

    AddRow = add_row
    Layout = layout
    Update = update


Col = Column


# ---------------------------------------------------------------------- #
#                           Pane                                         #
# ---------------------------------------------------------------------- #
class Pane(Element):
    """
    A sliding Pane that is unique to tkinter.  Uses Columns to create individual panes
    """

    def __init__(self, pane_list, background_color=None, size=(None, None), s=(None, None), pad=None, p=None, orientation='vertical',
                 show_handle=True, relief=RELIEF_RAISED, handle_size=None, border_width=None, key=None, k=None,  expand_x=None, expand_y=None, visible=True, metadata=None):
        """
        :param pane_list:        Must be a list of Column Elements. Each Column supplied becomes one pane that's shown
        :type pane_list:         List[Column] | Tuple[Column]
        :param background_color: color of background
        :type background_color:  (str)
        :param size:             (width, height) w=characters-wide, h=rows-high How much room to reserve for the Pane
        :type size:              (int, int)
        :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
        :type s:                 (int, int)  | (None, None)
        :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param orientation:      'horizontal' or 'vertical' or ('h' or 'v'). Direction the Pane should slide
        :type orientation:       (str)
        :param show_handle:      if True, the handle is drawn that makes it easier to grab and slide
        :type show_handle:       (bool)
        :param relief:           relief style. Values are same as other elements that use relief values. RELIEF_RAISED RELIEF_SUNKEN RELIEF_FLAT RELIEF_RIDGE RELIEF_GROOVE RELIEF_SOLID
        :type relief:            (enum)
        :param handle_size:      Size of the handle in pixels
        :type handle_size:       (int)
        :param border_width:     width of border around element in pixels
        :type border_width:      (int)
        :param key:              Value that uniquely identifies this element from all other elements. Used when Finding an element or in return values. Must be unique to the window
        :type key:               str | int | tuple | object
        :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                 str | int | tuple | object
        :param expand_x:         If True the column will automatically expand in the X direction to fill available space
        :type expand_x:          (bool)
        :param expand_y:         If True the column will automatically expand in the Y direction to fill available space
        :type expand_y:          (bool)
        :param visible:          set visibility state of the element
        :type visible:           (bool)
        :param metadata:         User metadata that can be set to ANYTHING
        :type metadata:          (Any)
        """


        self.UseDictionary = False
        self.ReturnValues = None
        self.ReturnValuesList = []
        self.ReturnValuesDictionary = {}
        self.DictionaryKeyCounter = 0
        self.ParentWindow = None
        self.Rows = []
        self.TKFrame = None
        self.PanedWindow = None
        self.Orientation = orientation
        self.PaneList = pane_list
        self.ShowHandle = show_handle
        self.Relief = relief
        self.HandleSize = handle_size or 8
        self.BorderDepth = border_width
        bg = background_color if background_color is not None else DEFAULT_BACKGROUND_COLOR

        self.Rows = [pane_list]
        key = key if key is not None else k
        sz = size if size != (None, None) else s
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y

        super().__init__(ELEM_TYPE_PANE, background_color=bg, size=sz, pad=pad, key=key, visible=visible, metadata=metadata)
        return

    def update(self, visible=None):
        """
        Changes some of the settings for the Pane Element. Must call `Window.Read` or `Window.Finalize` prior

        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param visible: control visibility of element
        :type visible:  (bool)
        """
        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in Pane.update - The window was closed')
            return

        if visible is False:
            self._pack_forget_save_settings()
        elif visible is True:
            self._pack_restore_settings()

        if visible is not None:
            self._visible = visible

    Update = update


# ---------------------------------------------------------------------- #
#                           Calendar                                     #
# ---------------------------------------------------------------------- #
class TKCalendar(ttk.Frame):
    """
    This code was shamelessly lifted from moshekaplan's repository - moshekaplan/tkinter_components
    NONE of this code is user callable.  Stay away!
    """
    # XXX ToDo: cget and configure

    datetime = calendar.datetime.datetime
    timedelta = calendar.datetime.timedelta

    def __init__(self, master=None, target_element=None, close_when_chosen=True, default_date=(None, None, None), **kw):
        """WIDGET-SPECIFIC OPTIONS: locale, firstweekday, year, month, selectbackground, selectforeground """
        self._TargetElement = target_element
        default_mon, default_day, default_year = default_date
        # remove custom options from kw before initializating ttk.Frame
        fwday = kw.pop('firstweekday', calendar.MONDAY)
        year = kw.pop('year', default_year or self.datetime.now().year)
        month = kw.pop('month', default_mon or self.datetime.now().month)
        locale = kw.pop('locale', None)
        sel_bg = kw.pop('selectbackground', '#ecffc4')
        sel_fg = kw.pop('selectforeground', '#05640e')
        self.format = kw.pop('format')
        if self.format is None:
            self.format = '%Y-%m-%d %H:%M:%S'

        self._date = self.datetime(year, month, default_day or 1)
        self._selection = None  # no date selected
        self._master = master
        self.close_when_chosen = close_when_chosen
        ttk.Frame.__init__(self, master, **kw)

        # instantiate proper calendar class
        if locale is None:
            self._cal = calendar.TextCalendar(fwday)
        else:
            self._cal = calendar.LocaleTextCalendar(fwday, locale)

        self.__setup_styles()  # creates custom styles
        self.__place_widgets()  # pack/grid used widgets
        self.__config_calendar()  # adjust calendar columns and setup tags
        # configure a canvas, and proper bindings, for selecting dates
        self.__setup_selection(sel_bg, sel_fg)

        # store items ids, used for insertion later
        self._items = [self._calendar.insert('', 'end', values='')
                       for _ in range(6)]
        # insert dates in the currently empty calendar
        self._build_calendar()

    def __setitem__(self, item, value):
        if item in ('year', 'month'):
            raise AttributeError("attribute '%s' is not writeable" % item)
        elif item == 'selectbackground':
            self._canvas['background'] = value
        elif item == 'selectforeground':
            self._canvas.itemconfigure(self._canvas.text, item=value)
        else:
            ttk.Frame.__setitem__(self, item, value)

    def __getitem__(self, item):
        if item in ('year', 'month'):
            return getattr(self._date, item)
        elif item == 'selectbackground':
            return self._canvas['background']
        elif item == 'selectforeground':
            return self._canvas.itemcget(self._canvas.text, 'fill')
        else:
            r = ttk.tclobjs_to_py({item: ttk.Frame.__getitem__(self, item)})
            return r[item]

    def __setup_styles(self):
        # custom ttk styles
        style = ttk.Style(self.master)
        arrow_layout = lambda dir: (
            [('Button.focus', {'children': [('Button.%sarrow' % dir, None)]})]
        )
        style.layout('L.TButton', arrow_layout('left'))
        style.layout('R.TButton', arrow_layout('right'))

    def __place_widgets(self):
        # header frame and its widgets
        hframe = ttk.Frame(self)
        lbtn = ttk.Button(hframe, style='L.TButton', command=self._prev_month)
        rbtn = ttk.Button(hframe, style='R.TButton', command=self._next_month)
        self._header = ttk.Label(hframe, width=15, anchor='center')
        # the calendar
        self._calendar = ttk.Treeview(self, show='', selectmode='none', height=7)

        # pack the widgets
        hframe.pack(in_=self, side='top', pady=4, anchor='center')
        lbtn.grid(in_=hframe)
        self._header.grid(in_=hframe, column=1, row=0, padx=12)
        rbtn.grid(in_=hframe, column=2, row=0)
        self._calendar.pack(in_=self, expand=1, fill='both', side='bottom')

    def __config_calendar(self):
        cols = self._cal.formatweekheader(3).split()
        self._calendar['columns'] = cols
        self._calendar.tag_configure('header', background='grey90')
        self._calendar.insert('', 'end', values=cols, tag='header')
        # adjust its columns width
        font = tkinter.font.Font()
        maxwidth = max(font.measure(col) for col in cols)
        for col in cols:
            self._calendar.column(col, width=maxwidth, minwidth=maxwidth,
                                  anchor='e')

    def __setup_selection(self, sel_bg, sel_fg):
        self._font = tkinter.font.Font()
        self._canvas = canvas = tk.Canvas(self._calendar,
                                          background=sel_bg, borderwidth=0, highlightthickness=0)
        canvas.text = canvas.create_text(0, 0, fill=sel_fg, anchor='w')

        canvas.bind('<ButtonPress-1>', lambda evt: canvas.place_forget())
        self._calendar.bind('<Configure>', lambda evt: canvas.place_forget())
        self._calendar.bind('<ButtonPress-1>', self._pressed)

    def __minsize(self, evt):
        width, height = self._calendar.master.geometry().split('x')
        height = height[:height.index('+')]
        self._calendar.master.minsize(width, height)

    def _build_calendar(self):
        year, month = self._date.year, self._date.month

        # update header text (Month, YEAR)
        header = self._cal.formatmonthname(year, month, 0)
        self._header['text'] = header.title()

        # update calendar shown dates
        cal = self._cal.monthdayscalendar(year, month)
        for indx, item in enumerate(self._items):
            week = cal[indx] if indx < len(cal) else []
            fmt_week = [('%02d' % day) if day else '' for day in week]
            self._calendar.item(item, values=fmt_week)

    def _show_selection(self, text, bbox):
        """ Configure canvas for a new selection. """
        x, y, width, height = bbox

        textw = self._font.measure(text)

        canvas = self._canvas
        canvas.configure(width=width, height=height)
        canvas.coords(canvas.text, width - textw, height / 2 - 1)
        canvas.itemconfigure(canvas.text, text=text)
        canvas.place(in_=self._calendar, x=x, y=y)

    # Callbacks

    def _pressed(self, evt):
        """ Clicked somewhere in the calendar. """
        x, y, widget = evt.x, evt.y, evt.widget
        item = widget.identify_row(y)
        column = widget.identify_column(x)

        if not column or not item in self._items:
            # clicked in the weekdays row or just outside the columns
            return

        item_values = widget.item(item)['values']
        if not len(item_values):  # row is empty for this month
            return

        text = item_values[int(column[1]) - 1]
        if not text:  # date is empty
            return

        bbox = widget.bbox(item, column)
        if not bbox:  # calendar not visible yet
            return

        # update and then show selection
        text = '%02d' % text
        self._selection = (text, item, column)
        self._show_selection(text, bbox)
        year, month = self._date.year, self._date.month
        now = self.datetime.now()
        try:
            self._TargetElement.Update(
                self.datetime(year, month, int(self._selection[0]), now.hour, now.minute, now.second).strftime(
                    self.format))
            if self._TargetElement.ChangeSubmits:
                self._TargetElement.ParentForm.LastButtonClicked = self._TargetElement.Key
                self._TargetElement.ParentForm.FormRemainedOpen = True
                self._TargetElement.ParentForm.TKroot.quit()  # kick the users out of the mainloop
        except:
            pass
        if self.close_when_chosen:
            self._master.destroy()

    def _prev_month(self):
        """Updated calendar to show the previous month."""
        self._canvas.place_forget()

        self._date = self._date - self.timedelta(days=1)
        self._date = self.datetime(self._date.year, self._date.month, 1)
        self._build_calendar()  # reconstuct calendar

    def _next_month(self):
        """Update calendar to show the next month."""
        self._canvas.place_forget()

        year, month = self._date.year, self._date.month
        self._date = self._date + self.timedelta(
            days=calendar.monthrange(year, month)[1] + 1)
        self._date = self.datetime(self._date.year, self._date.month, 1)
        self._build_calendar()  # reconstruct calendar

    # Properties

    @property
    def selection(self):
        if not self._selection:
            return None

        year, month = self._date.year, self._date.month
        return self.datetime(year, month, int(self._selection[0]))


# ---------------------------------------------------------------------- #
#                           Menu                                         #
# ---------------------------------------------------------------------- #
class Menu(Element):
    """
    Menu Element is the Element that provides a Menu Bar that goes across the top of the window, just below titlebar.
    Here is an example layout.  The "&" are shortcut keys ALT+key.
    Is a List of -  "Item String" + List
    Where Item String is what will be displayed on the Menubar itself.
    The List that follows the item represents the items that are shown then Menu item is clicked
    Notice how an "entry" in a mennu can be a list which means it branches out and shows another menu, etc. (resursive)
    menu_def = [['&File', ['!&Open', '&Save::savekey', '---', '&Properties', 'E&xit']],
                ['!&Edit', ['!&Paste', ['Special', 'Normal', ], 'Undo'], ],
                ['&Debugger', ['Popout', 'Launch Debugger']],
                ['&Toolbar', ['Command &1', 'Command &2', 'Command &3', 'Command &4']],
                ['&Help', '&About...'], ]
    Important Note!  The colors, font, look of the Menubar itself cannot be changed, only the menus shown AFTER clicking the menubar
    can be changed.  If you want to change the style/colors the Menubar, then you will have to use the MenubarCustom element.
    Finally, "keys" can be added to entries so make them unique.  The "Save" entry has a key associated with it. You
    can see it has a "::" which signifies the beginning of a key.  The user will not see the key portion when the
    menu is shown.  The key portion is returned as part of the event.
    """

    def __init__(self, menu_definition, background_color=None, text_color=None, disabled_text_color=None, size=(None, None), s=(None, None), tearoff=False,
                 font=None, pad=None, p=None, key=None, k=None, visible=True, metadata=None):
        """
        :param menu_definition:           The Menu definition specified using lists (docs explain the format)
        :type menu_definition:            List[List[Tuple[str, List[str]]]
        :param background_color:          color of the background of menus, NOT the Menubar
        :type background_color:           (str)
        :param text_color:                text color for menus, NOT the Menubar. Can be in #RRGGBB format or a color name "black".
        :type text_color:                 (str)
        :param disabled_text_color:       color to use for text when item in submenu, not the menubar itself, is disabled. Can be in #RRGGBB format or a color name "black"
        :type disabled_text_color:        (str)
        :param size:                      Not used in the tkinter port
        :type size:                       (int, int)
        :param s:                         Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
        :type s:                          (int, int)  | (None, None)
        :param tearoff:                   if True, then can tear the menu off from the window ans use as a floating window. Very cool effect
        :type tearoff:                    (bool)
        :param pad:                       Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:                        (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                         Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                          (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param font:                      specifies the  font family, size, etc. of submenus. Does NOT apply to the Menubar itself. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:                       (str or (str, int[, str]) or None)
        :param key:                       Value that uniquely identifies this element from all other elements. Used when Finding an element or in return values. Must be unique to the window
        :type key:                        str | int | tuple | object
        :param k:                         Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                          str | int | tuple | object
        :param visible:                   set visibility state of the element
        :type visible:                    (bool)
        :param metadata:                  User metadata that can be set to ANYTHING
        :type metadata:                   (Any)
        """

        self.BackgroundColor = background_color if background_color is not None else theme_input_background_color()
        self.TextColor = text_color if text_color is not None else theme_input_text_color()

        self.DisabledTextColor = disabled_text_color if disabled_text_color is not None else COLOR_SYSTEM_DEFAULT
        self.MenuDefinition = copy.deepcopy(menu_definition)
        self.Widget = self.TKMenu = None  # type: tk.Menu
        self.MenuItemChosen = None
        key = key if key is not None else k
        sz = size if size != (None, None) else s
        pad = pad if pad is not None else p

        super().__init__(ELEM_TYPE_MENUBAR, background_color=self.BackgroundColor, text_color=self.TextColor, size=sz, pad=pad, key=key, visible=visible,
                         font=font, metadata=metadata)
        # super().__init__(ELEM_TYPE_MENUBAR, background_color=COLOR_SYSTEM_DEFAULT, text_color=COLOR_SYSTEM_DEFAULT, size=sz, pad=pad, key=key, visible=visible, font=None, metadata=metadata)

        self.Tearoff = tearoff

        return

    def _MenuItemChosenCallback(self, item_chosen):  # Menu Menu Item Chosen Callback
        """
        Not user callable.  Called when some end-point on the menu (an item) has been clicked.  Send the information back to the application as an event.  Before event can be sent

        :param item_chosen: the text that was clicked on / chosen from the menu
        :type item_chosen:  (str)
        """
        # print('IN MENU ITEM CALLBACK', item_chosen)
        self.MenuItemChosen = item_chosen
        self.ParentForm.LastButtonClicked = item_chosen
        self.ParentForm.FormRemainedOpen = True
        # if self.ParentForm.CurrentlyRunningMainloop:
        #     self.ParentForm.TKroot.quit()  # kick the users out of the mainloop
        _exit_mainloop(self.ParentForm)

    def update(self, menu_definition=None, visible=None):
        """
        Update a menubar - can change the menu definition and visibility.  The entire menu has to be specified

        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param menu_definition: The menu definition list
        :type menu_definition:  List[List[Tuple[str, List[str]]]
        :param visible:         control visibility of element
        :type visible:          (bool)
        """
        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in Menu.update - The window was closed')
            return

        if menu_definition is not None:
            self.MenuDefinition = copy.deepcopy(menu_definition)
            if self.TKMenu is None:     # if no menu exists, make one
                self.TKMenu = tk.Menu(self.ParentForm.TKroot, tearoff=self.Tearoff, tearoffcommand=self._tearoff_menu_callback)  # create the menubar
            menubar = self.TKMenu
            # Delete all the menu items (assuming 10000 should be a high enough number to cover them all)
            menubar.delete(0, 10000)
            self.Widget = self.TKMenu   # same the new menu so user can access to extend PySimpleGUI
            for menu_entry in self.MenuDefinition:
                baritem = tk.Menu(menubar, tearoff=self.Tearoff, tearoffcommand=self._tearoff_menu_callback)

                if self.BackgroundColor not in (COLOR_SYSTEM_DEFAULT, None):
                    baritem.config(bg=self.BackgroundColor)
                if self.TextColor not in (COLOR_SYSTEM_DEFAULT, None):
                    baritem.config(fg=self.TextColor)
                if self.DisabledTextColor not in (COLOR_SYSTEM_DEFAULT, None):
                    baritem.config(disabledforeground=self.DisabledTextColor)
                if self.Font is not None:
                    baritem.config(font=self.Font)

                if self.Font is not None:
                    baritem.config(font=self.Font)
                pos = menu_entry[0].find(MENU_SHORTCUT_CHARACTER)
                # print(pos)
                if pos != -1:
                    if pos == 0 or menu_entry[0][pos - len(MENU_SHORTCUT_CHARACTER)] != '\\':
                        menu_entry[0] = menu_entry[0][:pos] + menu_entry[0][pos + len(MENU_SHORTCUT_CHARACTER):]
                if menu_entry[0][0] == MENU_DISABLED_CHARACTER:
                    menubar.add_cascade(label=menu_entry[0][len(MENU_DISABLED_CHARACTER):], menu=baritem, underline=pos)
                    menubar.entryconfig(menu_entry[0][len(MENU_DISABLED_CHARACTER):], state='disabled')
                else:
                    menubar.add_cascade(label=menu_entry[0], menu=baritem, underline=pos)

                if len(menu_entry) > 1:
                    AddMenuItem(baritem, menu_entry[1], self)

        if visible is False:
            self.ParentForm.TKroot.configure(menu=[])  # this will cause the menubar to disappear
        elif self.TKMenu is not None:
            self.ParentForm.TKroot.configure(menu=self.TKMenu)
        if visible is not None:
            self._visible = visible

    Update = update


MenuBar = Menu  # another name for Menu to make it clear it's the Menu Bar
Menubar = Menu  # another name for Menu to make it clear it's the Menu Bar


# ---------------------------------------------------------------------- #
#                           Table                                        #
# ---------------------------------------------------------------------- #
class Table(Element):

    def __init__(self, values, headings=None, visible_column_map=None, col_widths=None, cols_justification=None, def_col_width=10,
                 auto_size_columns=True, max_col_width=20, select_mode=None, display_row_numbers=False, starting_row_number=0, num_rows=None,
                 row_height=None, font=None, justification='right', text_color=None, background_color=None,
                 alternating_row_color=None, selected_row_colors=(None, None), header_text_color=None, header_background_color=None, header_font=None, header_border_width=None, header_relief=None,
                 row_colors=None, vertical_scroll_only=True, hide_vertical_scroll=False, border_width=None,
                 sbar_trough_color=None, sbar_background_color=None, sbar_arrow_color=None, sbar_width=None, sbar_arrow_width=None, sbar_frame_color=None, sbar_relief=None,
                 size=(None, None), s=(None, None), change_submits=False, enable_events=False, enable_click_events=False, right_click_selects=False, bind_return_key=False, pad=None, p=None,
                 key=None, k=None, tooltip=None, right_click_menu=None, expand_x=False, expand_y=False, visible=True, metadata=None):
        """
        :param values:                  Your table data represented as a 2-dimensions table... a list of rows, with each row representing a row in your table.
        :type values:                   List[List[str | int | float]]
        :param headings:                The headings to show on the top line
        :type headings:                 List[str]
        :param visible_column_map:      One entry for each column. False indicates the column is not shown
        :type visible_column_map:       List[bool]
        :param col_widths:              Number of characters that each column will occupy
        :type col_widths:               List[int]
        :param cols_justification:      Justification for EACH column. Is a list of strings with the value 'l', 'r', 'c' that indicates how the column will be justified. Either no columns should be set, or have to have one for every colun
        :type cols_justification:       List[str] or Tuple[str] or None
        :param def_col_width:           Default column width in characters
        :type def_col_width:            (int)
        :param auto_size_columns:       if True columns will be sized automatically
        :type auto_size_columns:        (bool)
        :param max_col_width:           Maximum width for all columns in characters
        :type max_col_width:            (int)
        :param select_mode:             Select Mode. Valid values start with "TABLE_SELECT_MODE_".  Valid values are: TABLE_SELECT_MODE_NONE TABLE_SELECT_MODE_BROWSE TABLE_SELECT_MODE_EXTENDED
        :type select_mode:              (enum)
        :param display_row_numbers:     if True, the first column of the table will be the row #
        :type display_row_numbers:      (bool)
        :param starting_row_number:     The row number to use for the first row. All following rows will be based on this starting value. Default is 0.
        :type starting_row_number:      (int)
        :param num_rows:                The number of rows of the table to display at a time
        :type num_rows:                 (int)
        :param row_height:              height of a single row in pixels
        :type row_height:               (int)
        :param font:                    specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:                     (str or (str, int[, str]) or None)
        :param justification:           'left', 'right', 'center' are valid choices
        :type justification:            (str)
        :param text_color:              color of the text
        :type text_color:               (str)
        :param background_color:        color of background
        :type background_color:         (str)
        :param alternating_row_color:   if set then every other row will have this color in the background.
        :type alternating_row_color:    (str)
        :param selected_row_colors:     Sets the text color and background color for a selected row. Same format as button colors - tuple ('red', 'yellow') or string 'red on yellow'. Defaults to theme's button color
        :type selected_row_colors:      str or (str, str)
        :param header_text_color:       sets the text color for the header
        :type header_text_color:        (str)
        :param header_background_color: sets the background color for the header
        :type header_background_color:  (str)
        :param header_font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type header_font:              (str or (str, int[, str]) or None)
        :param header_border_width:     Border width for the header portion
        :type header_border_width:      (int | None)
        :param header_relief:           Relief style for the header. Values are same as other elements that use relief. RELIEF_RAISED RELIEF_SUNKEN RELIEF_FLAT RELIEF_RIDGE RELIEF_GROOVE RELIEF_SOLID
        :type header_relief:            (str | None)
        :param row_colors:              list of tuples of (row, background color) OR (row, foreground color, background color). Sets the colors of listed rows to the color(s) provided (note the optional foreground color)
        :type row_colors:               List[Tuple[int, str] | Tuple[Int, str, str]]
        :param vertical_scroll_only:    if True only the vertical scrollbar will be visible
        :type vertical_scroll_only:     (bool)
        :param hide_vertical_scroll:    if True vertical scrollbar will be hidden
        :type hide_vertical_scroll:     (bool)
        :param border_width:            Border width/depth in pixels
        :type border_width:             (int)
        :param sbar_trough_color:           Scrollbar color of the trough
        :type sbar_trough_color:            (str)
        :param sbar_background_color:       Scrollbar color of the background of the arrow buttons at the ends AND the color of the "thumb" (the thing you grab and slide). Switches to arrow color when mouse is over
        :type sbar_background_color:        (str)
        :param sbar_arrow_color:            Scrollbar color of the arrow at the ends of the scrollbar (it looks like a button). Switches to background color when mouse is over
        :type sbar_arrow_color:             (str)
        :param sbar_width:                  Scrollbar width in pixels
        :type sbar_width:                   (int)
        :param sbar_arrow_width:            Scrollbar width of the arrow on the scrollbar. It will potentially impact the overall width of the scrollbar
        :type sbar_arrow_width:             (int)
        :param sbar_frame_color:            Scrollbar Color of frame around scrollbar (available only on some ttk themes)
        :type sbar_frame_color:             (str)
        :param sbar_relief:                 Scrollbar relief that will be used for the "thumb" of the scrollbar (the thing you grab that slides). Should be a constant that is defined at starting with "RELIEF_" - RELIEF_RAISED, RELIEF_SUNKEN, RELIEF_FLAT, RELIEF_RIDGE, RELIEF_GROOVE, RELIEF_SOLID
        :type sbar_relief:                  (str)
        :param size:                    DO NOT USE! Use num_rows instead
        :type size:                     (int, int)
        :param change_submits:          DO NOT USE. Only listed for backwards compat - Use enable_events instead
        :type change_submits:           (bool)
        :param enable_events:           Turns on the element specific events. Table events happen when row is clicked
        :type enable_events:            (bool)
        :param enable_click_events:     Turns on the element click events that will give you (row, col) click data when the table is clicked
        :type enable_click_events:      (bool)
        :param right_click_selects:     If True, then right clicking a row will select that row if multiple rows are not currently selected
        :type right_click_selects:      (bool)
        :param bind_return_key:         if True, pressing return key will cause event coming from Table, ALSO a left button double click will generate an event if this parameter is True
        :type bind_return_key:          (bool)
        :param pad:                     Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:                      (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                       Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                        (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param key:                     Used with window.find_element and with return values to uniquely identify this element to uniquely identify this element
        :type key:                      str | int | tuple | object
        :param k:                       Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                        str | int | tuple | object
        :param tooltip:                 text, that will appear when mouse hovers over the element
        :type tooltip:                  (str)
        :param right_click_menu:        A list of lists of Menu items to show when this element is right clicked. See user docs for exact format.
        :type right_click_menu:         List[List[ List[str] | str ]]
        :param expand_x:                If True the element will automatically expand in the X direction to fill available space
        :type expand_x:                 (bool)
        :param expand_y:                If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:                 (bool)
        :param visible:                 set visibility state of the element
        :type visible:                  (bool)
        :param metadata:                User metadata that can be set to ANYTHING
        :type metadata:                 (Any)
        """


        self.Values = values
        self.ColumnHeadings = headings
        self.ColumnsToDisplay = visible_column_map
        self.ColumnWidths = col_widths
        self.cols_justification = cols_justification
        self.MaxColumnWidth = max_col_width
        self.DefaultColumnWidth = def_col_width
        self.AutoSizeColumns = auto_size_columns
        self.BackgroundColor = background_color if background_color is not None else DEFAULT_BACKGROUND_COLOR
        self.TextColor = text_color
        self.HeaderTextColor = header_text_color if header_text_color is not None else LOOK_AND_FEEL_TABLE[CURRENT_LOOK_AND_FEEL]['TEXT_INPUT']
        self.HeaderBackgroundColor = header_background_color if header_background_color is not None else LOOK_AND_FEEL_TABLE[CURRENT_LOOK_AND_FEEL]['INPUT']
        self.HeaderFont = header_font
        self.Justification = justification
        self.InitialState = None
        self.SelectMode = select_mode
        self.DisplayRowNumbers = display_row_numbers
        self.NumRows = num_rows if num_rows is not None else size[1]
        self.RowHeight = row_height
        self.Widget = self.TKTreeview = None  # type: ttk.Treeview
        self.AlternatingRowColor = alternating_row_color
        self.VerticalScrollOnly = vertical_scroll_only
        self.HideVerticalScroll = hide_vertical_scroll
        self.SelectedRows = []
        self.ChangeSubmits = change_submits or enable_events
        self.BindReturnKey = bind_return_key
        self.StartingRowNumber = starting_row_number  # When displaying row numbers, where to start
        self.RowHeaderText = 'Row'
        self.enable_click_events = enable_click_events
        self.right_click_selects = right_click_selects
        self.last_clicked_position = (None, None)
        self.HeaderBorderWidth = header_border_width
        self.BorderWidth = border_width
        self.HeaderRelief = header_relief
        self.table_ttk_style_name = None        # the ttk style name for the Table itself
        if selected_row_colors == (None, None):
            # selected_row_colors = DEFAULT_TABLE_AND_TREE_SELECTED_ROW_COLORS
            selected_row_colors = theme_button_color()
        else:
            try:
                if isinstance(selected_row_colors, str):
                    selected_row_colors = selected_row_colors.split(' on ')
            except Exception as e:
                print('* Table Element Warning * you messed up with color formatting of Selected Row Color', e)
        self.SelectedRowColors = selected_row_colors

        self.RightClickMenu = right_click_menu
        self.RowColors = row_colors
        self.tree_ids = []  # ids returned when inserting items into table - will use to delete colors
        key = key if key is not None else k
        sz = size if size != (None, None) else s
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y

        super().__init__(ELEM_TYPE_TABLE, text_color=text_color, background_color=background_color, font=font,
                         size=sz, pad=pad, key=key, tooltip=tooltip, visible=visible, metadata=metadata,
                         sbar_trough_color=sbar_trough_color, sbar_background_color=sbar_background_color, sbar_arrow_color=sbar_arrow_color, sbar_width=sbar_width, sbar_arrow_width=sbar_arrow_width, sbar_frame_color=sbar_frame_color, sbar_relief=sbar_relief)
        return

    def update(self, values=None, num_rows=None, visible=None, select_rows=None, alternating_row_color=None, row_colors=None):
        """
        Changes some of the settings for the Table Element. Must call `Window.Read` or `Window.Finalize` prior

        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param values:                A new 2-dimensional table to show
        :type values:                 List[List[str | int | float]]
        :param num_rows:              How many rows to display at a time
        :type num_rows:               (int)
        :param visible:               if True then will be visible
        :type visible:                (bool)
        :param select_rows:           List of rows to select as if user did
        :type select_rows:            List[int]
        :param alternating_row_color: the color to make every other row
        :type alternating_row_color:  (str)
        :param row_colors:            list of tuples of (row, background color) OR (row, foreground color, background color). Changes the colors of listed rows to the color(s) provided (note the optional foreground color)
        :type row_colors:             List[Tuple[int, str] | Tuple[Int, str, str]]
        """
        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in Table.update - The window was closed')
            return

        if values is not None:
            for id in self.tree_ids:
                self.TKTreeview.item(id, tags=())
                if self.BackgroundColor is not None and self.BackgroundColor != COLOR_SYSTEM_DEFAULT:
                    self.TKTreeview.tag_configure(id, background=self.BackgroundColor)
                else:
                    self.TKTreeview.tag_configure(id, background='#FFFFFF', foreground='#000000')
                if self.TextColor is not None and self.TextColor != COLOR_SYSTEM_DEFAULT:
                    self.TKTreeview.tag_configure(id, foreground=self.TextColor)
                else:
                    self.TKTreeview.tag_configure(id, foreground='#000000')

            children = self.TKTreeview.get_children()
            for i in children:
                self.TKTreeview.detach(i)
                self.TKTreeview.delete(i)
            children = self.TKTreeview.get_children()

            self.tree_ids = []
            for i, value in enumerate(values):
                if self.DisplayRowNumbers:
                    value = [i + self.StartingRowNumber] + value
                id = self.TKTreeview.insert('', 'end', text=value, iid=i + 1, values=value, tag=i)
                if self.BackgroundColor is not None and self.BackgroundColor != COLOR_SYSTEM_DEFAULT:
                    self.TKTreeview.tag_configure(id, background=self.BackgroundColor)
                else:
                    self.TKTreeview.tag_configure(id, background='#FFFFFF')
                self.tree_ids.append(id)
            self.Values = values
            self.SelectedRows = []
        if visible is False:
            self._pack_forget_save_settings(self.element_frame)
        elif visible is True:
            self._pack_restore_settings(self.element_frame)

        if num_rows is not None:
            self.TKTreeview.config(height=num_rows)
        if select_rows is not None:
            rows_to_select = [i + 1 for i in select_rows]
            self.TKTreeview.selection_set(rows_to_select)

        if alternating_row_color is not None:  # alternating colors
            self.AlternatingRowColor = alternating_row_color

        if self.AlternatingRowColor is not None:
            for row in range(0, len(self.Values), 2):
                self.TKTreeview.tag_configure(row, background=self.AlternatingRowColor)
        if row_colors is not None:  # individual row colors
            self.RowColors = row_colors
            for row_def in self.RowColors:
                if len(row_def) == 2:  # only background is specified
                    self.TKTreeview.tag_configure(row_def[0], background=row_def[1])
                else:
                    self.TKTreeview.tag_configure(row_def[0], background=row_def[2], foreground=row_def[1])
        if visible is not None:
            self._visible = visible

    def _treeview_selected(self, event):
        """
        Not user callable.  Callback function that is called when something is selected from Table.
        Stores the selected rows in Element as they are being selected. If events enabled, then returns from Read

        :param event: event information from tkinter
        :type event:  (unknown)
        """
        # print('**-- in treeview selected --**')
        selections = self.TKTreeview.selection()
        self.SelectedRows = [int(x) - 1 for x in selections]
        if self.ChangeSubmits:
            if self.Key is not None:
                self.ParentForm.LastButtonClicked = self.Key
            else:
                self.ParentForm.LastButtonClicked = ''
            self.ParentForm.FormRemainedOpen = True
            # if self.ParentForm.CurrentlyRunningMainloop:
            #     self.ParentForm.TKroot.quit()
            _exit_mainloop(self.ParentForm)

    def _treeview_double_click(self, event):
        """
        Not user callable.  Callback function that is called when something is selected from Table.
        Stores the selected rows in Element as they are being selected. If events enabled, then returns from Read

        :param event: event information from tkinter
        :type event:  (unknown)
        """
        selections = self.TKTreeview.selection()
        self.SelectedRows = [int(x) - 1 for x in selections]
        if self.BindReturnKey:  # Signifies BOTH a return key AND a double click
            if self.Key is not None:
                self.ParentForm.LastButtonClicked = self.Key
            else:
                self.ParentForm.LastButtonClicked = ''
            self.ParentForm.FormRemainedOpen = True
            # if self.ParentForm.CurrentlyRunningMainloop:
            #     self.ParentForm.TKroot.quit()
            _exit_mainloop(self.ParentForm)


    def _table_clicked(self, event):
        """
        Not user callable.  Callback function that is called a click happens on a table.
        Stores the selected rows in Element as they are being selected. If events enabled, then returns from Read

        :param event: event information from tkinter
        :type event:  (unknown)
        """
        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return
        # popup(obj_to_string_single_obj(event))
        try:
            region = self.Widget.identify('region', event.x, event.y)
            if region == 'heading':
                row = -1
            elif region == 'cell':
                row = int(self.Widget.identify_row(event.y))-1
            elif region == 'separator':
                row = None
            else:
                row = None
            col_identified = self.Widget.identify_column(event.x)
            if col_identified:      # Sometimes tkinter returns a value of '' which would cause an error if cast to an int
                column = int(self.Widget.identify_column(event.x)[1:])-1-int(self.DisplayRowNumbers is True)
            else:
                column = None
        except Exception as e:
            warnings.warn('Error getting table click data for table with key= {}\nError: {}'.format(self.Key, e), UserWarning)
            if not SUPPRESS_ERROR_POPUPS:
                _error_popup_with_traceback('Unable to complete operation getting the clicked event for table with key {}'.format(self.Key), _create_error_message(), e, 'Event data:', obj_to_string_single_obj(event))
            row = column = None

        self.last_clicked_position = (row, column)

        # update the rows being selected if appropriate
        self.ParentForm.TKroot.update()
        # self.TKTreeview.()
        selections = self.TKTreeview.selection()
        if self.right_click_selects and len(selections) <= 1:
            if (event.num == 3 and not running_mac()) or (event.num == 2 and running_mac()):
                if row != -1 and row is not None:
                    selections = [row+1]
                    self.TKTreeview.selection_set(selections)
        # print(selections)
        self.SelectedRows = [int(x) - 1 for x in selections]
        # print('The new selected rows = ', self.SelectedRows, 'selections =', selections)
        if self.enable_click_events is True:
            if self.Key is not None:
                self.ParentForm.LastButtonClicked = (self.Key, TABLE_CLICKED_INDICATOR, (row, column))
            else:
                self.ParentForm.LastButtonClicked = ''
            self.ParentForm.FormRemainedOpen = True
            _exit_mainloop(self.ParentForm)



    def get(self):
        """
        Get the selected rows using tktiner's selection method.  Returns a list of the selected rows.

        :return: a list of the index of the selected rows (a list of ints)
        :rtype:  List[int]
        """

        selections = self.TKTreeview.selection()
        selected_rows = [int(x) - 1 for x in selections]
        return selected_rows

    def get_last_clicked_position(self):
        """
        Returns a tuple with the row and column of the cell that was last clicked.
        Headers will have a row == -1 and the Row Number Column (if present) will have a column == -1
        :return: The (row,col) position of the last cell clicked in the table
        :rtype:  (int | None, int | None)
        """
        return self.last_clicked_position




    Update = update
    Get = get


# ---------------------------------------------------------------------- #
#                           Tree                                         #
# ---------------------------------------------------------------------- #
class Tree(Element):
    """
    Tree Element - Presents data in a tree-like manner, much like a file/folder browser.  Uses the TreeData class
    to hold the user's data and pass to the element for display.
    """

    def __init__(self, data=None, headings=None, visible_column_map=None, col_widths=None, col0_width=10, col0_heading='',
                 def_col_width=10, auto_size_columns=True, max_col_width=20, select_mode=None, show_expanded=False,
                 change_submits=False, enable_events=False, click_toggles_select=None, font=None, justification='right', text_color=None, border_width=None,
                 background_color=None, selected_row_colors=(None, None), header_text_color=None, header_background_color=None, header_font=None, header_border_width=None, header_relief=None, num_rows=None,
                 sbar_trough_color=None, sbar_background_color=None, sbar_arrow_color=None, sbar_width=None, sbar_arrow_width=None, sbar_frame_color=None, sbar_relief=None,
                 row_height=None, vertical_scroll_only=True, hide_vertical_scroll=False, pad=None, p=None, key=None, k=None, tooltip=None,
                 right_click_menu=None, expand_x=False, expand_y=False, visible=True, metadata=None):
        """
        :param data:                    The data represented using a PySimpleGUI provided TreeData class
        :type data:                     (TreeData)
        :param headings:                List of individual headings for each column
        :type headings:                 List[str]
        :param visible_column_map:      Determines if a column should be visible. If left empty, all columns will be shown
        :type visible_column_map:       List[bool]
        :param col_widths:              List of column widths so that individual column widths can be controlled
        :type col_widths:               List[int]
        :param col0_width:              Size of Column 0 which is where the row numbers will be optionally shown
        :type col0_width:               (int)
        :param col0_heading:            Text to be shown in the header for the left-most column
        :type col0_heading:             (str)
        :param def_col_width:           default column width
        :type def_col_width:            (int)
        :param auto_size_columns:       if True, the size of a column is determined  using the contents of the column
        :type auto_size_columns:        (bool)
        :param max_col_width:           the maximum size a column can be
        :type max_col_width:            (int)
        :param select_mode:             Use same values as found on Table Element.  Valid values include: TABLE_SELECT_MODE_NONE TABLE_SELECT_MODE_BROWSE TABLE_SELECT_MODE_EXTENDED
        :type select_mode:              (enum)
        :param show_expanded:           if True then the tree will be initially shown with all nodes completely expanded
        :type show_expanded:            (bool)
        :param change_submits:          DO NOT USE. Only listed for backwards compat - Use enable_events instead
        :type change_submits:           (bool)
        :param enable_events:           Turns on the element specific events. Tree events happen when row is clicked
        :type enable_events:            (bool)
        :param click_toggles_select:    If True then clicking a row will cause the selection for that row to toggle between selected and deselected
        :type click_toggles_select:     (bool)
        :param font:                    specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:                     (str or (str, int[, str]) or None)
        :param justification:           'left', 'right', 'center' are valid choices
        :type justification:            (str)
        :param text_color:              color of the text
        :type text_color:               (str)
        :param border_width:            Border width/depth in pixels
        :type border_width:             (int)
        :param background_color:        color of background
        :type background_color:         (str)
        :param selected_row_colors:     Sets the text color and background color for a selected row. Same format as button colors - tuple ('red', 'yellow') or string 'red on yellow'. Defaults to theme's button color
        :type selected_row_colors:      str or (str, str)
        :param header_text_color:       sets the text color for the header
        :type header_text_color:        (str)
        :param header_background_color: sets the background color for the header
        :type header_background_color:  (str)
        :param header_font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type header_font:              (str or (str, int[, str]) or None)
        :param header_border_width:     Border width for the header portion
        :type header_border_width:      (int | None)
        :param header_relief:           Relief style for the header. Values are same as other elements that use relief. RELIEF_RAISED RELIEF_SUNKEN RELIEF_FLAT RELIEF_RIDGE RELIEF_GROOVE RELIEF_SOLID
        :type header_relief:            (str | None)
        :param num_rows:                The number of rows of the table to display at a time
        :type num_rows:                 (int)
        :param row_height:              height of a single row in pixels
        :type row_height:               (int)
        :param vertical_scroll_only:    if True only the vertical scrollbar will be visible
        :type vertical_scroll_only:     (bool)
        :param hide_vertical_scroll:    if True vertical scrollbar will be hidden
        :type hide_vertical_scroll:     (bool)
        :param sbar_trough_color:           Scrollbar color of the trough
        :type sbar_trough_color:            (str)
        :param sbar_background_color:       Scrollbar color of the background of the arrow buttons at the ends AND the color of the "thumb" (the thing you grab and slide). Switches to arrow color when mouse is over
        :type sbar_background_color:        (str)
        :param sbar_arrow_color:            Scrollbar color of the arrow at the ends of the scrollbar (it looks like a button). Switches to background color when mouse is over
        :type sbar_arrow_color:             (str)
        :param sbar_width:                  Scrollbar width in pixels
        :type sbar_width:                   (int)
        :param sbar_arrow_width:            Scrollbar width of the arrow on the scrollbar. It will potentially impact the overall width of the scrollbar
        :type sbar_arrow_width:             (int)
        :param sbar_frame_color:            Scrollbar Color of frame around scrollbar (available only on some ttk themes)
        :type sbar_frame_color:             (str)
        :param sbar_relief:                 Scrollbar relief that will be used for the "thumb" of the scrollbar (the thing you grab that slides). Should be a constant that is defined at starting with "RELIEF_" - RELIEF_RAISED, RELIEF_SUNKEN, RELIEF_FLAT, RELIEF_RIDGE, RELIEF_GROOVE, RELIEF_SOLID
        :type sbar_relief:                  (str)
        :param pad:                     Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
        :type pad:                      (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param p:                       Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
        :type p:                        (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
        :param key:                     Used with window.find_element and with return values to uniquely identify this element to uniquely identify this element
        :type key:                      str | int | tuple | object
        :param k:                       Same as the Key. You can use either k or key. Which ever is set will be used.
        :type k:                        str | int | tuple | object
        :param tooltip:                 text, that will appear when mouse hovers over the element
        :type tooltip:                  (str)
        :param right_click_menu:        A list of lists of Menu items to show when this element is right clicked. See user docs for exact format.
        :type right_click_menu:         List[List[str] | str]]
        :param expand_x:                If True the element will automatically expand in the X direction to fill available space
        :type expand_x:                 (bool)
        :param expand_y:                If True the element will automatically expand in the Y direction to fill available space
        :type expand_y:                 (bool)
        :param visible:                 set visibility state of the element
        :type visible:                  (bool)
        :param metadata:                User metadata that can be set to ANYTHING
        :type metadata:                 (Any)
        """

        self.image_dict = {}

        self.TreeData = data
        self.ColumnHeadings = headings
        self.ColumnsToDisplay = visible_column_map
        self.ColumnWidths = col_widths
        self.MaxColumnWidth = max_col_width
        self.DefaultColumnWidth = def_col_width
        self.AutoSizeColumns = auto_size_columns
        self.BackgroundColor = background_color if background_color is not None else DEFAULT_BACKGROUND_COLOR
        self.TextColor = text_color
        self.HeaderTextColor = header_text_color if header_text_color is not None else LOOK_AND_FEEL_TABLE[CURRENT_LOOK_AND_FEEL]['TEXT_INPUT']
        self.HeaderBackgroundColor = header_background_color if header_background_color is not None else LOOK_AND_FEEL_TABLE[CURRENT_LOOK_AND_FEEL]['INPUT']
        self.HeaderBorderWidth = header_border_width
        self.BorderWidth = border_width
        self.HeaderRelief = header_relief
        self.click_toggles_select = click_toggles_select
        if selected_row_colors == (None, None):
            # selected_row_colors = DEFAULT_TABLE_AND_TREE_SELECTED_ROW_COLORS
            selected_row_colors = theme_button_color()
        else:
            try:
                if isinstance(selected_row_colors, str):
                    selected_row_colors = selected_row_colors.split(' on ')
            except Exception as e:
                print('* Table Element Warning * you messed up with color formatting of Selected Row Color', e)
        self.SelectedRowColors = selected_row_colors

        self.HeaderFont = header_font
        self.Justification = justification
        self.InitialState = None
        self.SelectMode = select_mode
        self.ShowExpanded = show_expanded
        self.NumRows = num_rows
        self.Col0Width = col0_width
        self.col0_heading = col0_heading
        self.TKTreeview = None  # type: ttk.Treeview
        self.element_frame = None  # type: tk.Frame
        self.VerticalScrollOnly = vertical_scroll_only
        self.HideVerticalScroll = hide_vertical_scroll
        self.SelectedRows = []
        self.ChangeSubmits = change_submits or enable_events
        self.RightClickMenu = right_click_menu
        self.RowHeight = row_height
        self.IconList = {}
        self.IdToKey = {'': ''}
        self.KeyToID = {'': ''}
        key = key if key is not None else k
        pad = pad if pad is not None else p
        self.expand_x = expand_x
        self.expand_y = expand_y

        super().__init__(ELEM_TYPE_TREE, text_color=text_color, background_color=background_color, font=font, pad=pad, key=key, tooltip=tooltip,
                         visible=visible, metadata=metadata,
                         sbar_trough_color=sbar_trough_color, sbar_background_color=sbar_background_color, sbar_arrow_color=sbar_arrow_color, sbar_width=sbar_width, sbar_arrow_width=sbar_arrow_width, sbar_frame_color=sbar_frame_color, sbar_relief=sbar_relief)
        return

    def _treeview_selected(self, event):
        """
        Not a user function.  Callback function that happens when an item is selected from the tree.  In this
        method, it saves away the reported selections so they can be properly returned.

        :param event: An event parameter passed in by tkinter.  Not used
        :type event:  (Any)
        """

        selections = self.TKTreeview.selection()
        selected_rows = [self.IdToKey[x] for x in selections]
        if self.click_toggles_select:
            if set(self.SelectedRows) == set(selected_rows):
                for item in selections:
                    self.TKTreeview.selection_remove(item)
                selections = []
        self.SelectedRows = [self.IdToKey[x] for x in selections]

        if self.ChangeSubmits:
            MyForm = self.ParentForm
            if self.Key is not None:
                self.ParentForm.LastButtonClicked = self.Key
            else:
                self.ParentForm.LastButtonClicked = ''
            self.ParentForm.FormRemainedOpen = True
            # if self.ParentForm.CurrentlyRunningMainloop:
            #     self.ParentForm.TKroot.quit()
            _exit_mainloop(self.ParentForm)


    def add_treeview_data(self, node):
        """
        Not a user function.  Recursive method that inserts tree data into the tkinter treeview widget.

        :param node: The node to insert.  Will insert all nodes from starting point downward, recursively
        :type node:  (TreeData)
        """
        if node.key != '':
            if node.icon:
                try:
                    if node.icon not in self.image_dict:
                        if type(node.icon) is bytes:
                            photo = tk.PhotoImage(data=node.icon)
                        else:
                            photo = tk.PhotoImage(file=node.icon)
                        self.image_dict[node.icon] = photo
                    else:
                        photo = self.image_dict.get(node.icon)

                    node.photo = photo
                    id = self.TKTreeview.insert(self.KeyToID[node.parent], 'end', iid=None, text=node.text,
                                                values=node.values, open=self.ShowExpanded, image=node.photo)
                    self.IdToKey[id] = node.key
                    self.KeyToID[node.key] = id
                except:
                    self.photo = None
            else:
                id = self.TKTreeview.insert(self.KeyToID[node.parent], 'end', iid=None, text=node.text,
                                            values=node.values, open=self.ShowExpanded)
                self.IdToKey[id] = node.key
                self.KeyToID[node.key] = id

        for node in node.children:
            self.add_treeview_data(node)

    def update(self, values=None, key=None, value=None, text=None, icon=None, visible=None):
        """
        Changes some of the settings for the Tree Element. Must call `Window.Read` or `Window.Finalize` prior

        Changes will not be visible in your window until you call window.read or window.refresh.

        If you change visibility, your element may MOVE. If you want it to remain stationary, use the "layout helper"
        function "pin" to ensure your element is "pinned" to that location in your layout so that it returns there
        when made visible.

        :param values:  Representation of the tree
        :type values:   (TreeData)
        :param key:     identifies a particular item in tree to update
        :type key:      str | int | tuple | object
        :param value:   sets the node identified by key to a particular value
        :type value:    (Any)
        :param text:    sets the node identified by key to this string
        :type text:     (str)
        :param icon:    can be either a base64 icon or a filename for the icon
        :type icon:     bytes | str
        :param visible: control visibility of element
        :type visible:  (bool)
        """
        if not self._widget_was_created():  # if widget hasn't been created yet, then don't allow
            return

        if self._this_elements_window_closed():
            _error_popup_with_traceback('Error in Tree.update - The window was closed')
            return

        if values is not None:
            children = self.TKTreeview.get_children()
            for i in children:
                self.TKTreeview.detach(i)
                self.TKTreeview.delete(i)
            children = self.TKTreeview.get_children()
            self.TreeData = values
            self.IdToKey = {'': ''}
            self.KeyToID = {'': ''}
            self.add_treeview_data(self.TreeData.root_node)
            self.SelectedRows = []
        if key is not None:
            for id in self.IdToKey.keys():
                if key == self.IdToKey[id]:
                    break
            else:
                id = None
                print('** Key not found **')
        else:
            id = None
        if id:
            # item = self.TKTreeview.item(id)
            if value is not None:
                self.TKTreeview.item(id, values=value)
            if text is not None:
                self.TKTreeview.item(id, text=text)
            if icon is not None:
                try:
                    if type(icon) is bytes:
                        photo = tk.PhotoImage(data=icon)
                    else:
                        photo = tk.PhotoImage(file=icon)
                    self.TKTreeview.item(id, image=photo)
                    self.IconList[key] = photo  # save so that it's not deleted (save reference)
                except:
                    pass
            # item = self.TKTreeview.item(id)
        if visible is False:
            self._pack_forget_save_settings(self.element_frame)
        elif visible is True:
            self._pack_restore_settings(self.element_frame)

        if visible is not None:
            self._visible = visible

        return self

    Update = update


class TreeData(object):
    """
    Class that user fills in to represent their tree data. It's a very simple tree representation with a root "Node"
    with possibly one or more children "Nodes".  Each Node contains a key, text to display, list of values to display
    and an icon.  The entire tree is built using a single method, Insert.  Nothing else is required to make the tree.
    """

    class Node(object):
        """
        Contains information about the individual node in the tree
        """

        def __init__(self, parent, key, text, values, icon=None):
            """
            Represents a node within the TreeData class

            :param parent: The parent Node
            :type parent:  (TreeData.Node)
            :param key:    Used to uniquely identify this node
            :type key:     str | int | tuple | object
            :param text:   The text that is displayed at this node's location
            :type text:    (str)
            :param values: The list of values that are displayed at this node
            :type values:  List[Any]
            :param icon:   just a icon
            :type icon:    str | bytes
            """

            self.parent = parent  # type: TreeData.Node
            self.children = []  # type: List[TreeData.Node]
            self.key = key  # type: str
            self.text = text  # type: str
            self.values = values  # type: List[Any]
            self.icon = icon  # type: str | bytes

        def _Add(self, node):
            self.children.append(node)

    def __init__(self):
        """
        Instantiate the object, initializes the Tree Data, creates a root node for you
        """
        self.tree_dict = {}  # type: Dict[str, TreeData.Node]
        self.root_node = self.Node('', '', 'root', [], None)  # The root node
        self.tree_dict[''] = self.root_node  # Start the tree out with the root node

    def _AddNode(self, key, node):
        """
        Adds a node to tree dictionary (not user callable)

        :param key:  Uniquely identifies this Node
        :type key:   (str)
        :param node: Node being added
        :type node:  (TreeData.Node)
        """
        self.tree_dict[key] = node

    def insert(self, parent, key, text, values, icon=None):
        """
        Inserts a node into the tree. This is how user builds their tree, by Inserting Nodes
        This is the ONLY user callable method in the TreeData class

        :param parent: the parent Node
        :type parent:  (Node)
        :param key:    Used to uniquely identify this node
        :type key:     str | int | tuple | object
        :param text:   The text that is displayed at this node's location
        :type text:    (str)
        :param values: The list of values that are displayed at this node
        :type values:  List[Any]
        :param icon:   icon
        :type icon:    str | bytes
        """

        node = self.Node(parent, key, text, values, icon)
        self.tree_dict[key] = node
        parent_node = self.tree_dict[parent]
        parent_node._Add(node)

    def __repr__(self):
        """
        Converts the TreeData into a printable version, nicely formatted

        :return: (str) A formatted, text version of the TreeData
        :rtype:
        """
        return self._NodeStr(self.root_node, 1)

    def _NodeStr(self, node, level):
        """
        Does the magic of converting the TreeData into a nicely formatted string version

        :param node:  The node to begin printing the tree
        :type node:   (TreeData.Node)
        :param level: The indentation level for string formatting
        :type level:  (int)
        """
        return '\n'.join(
            [str(node.key) + ' : ' + str(node.text) + ' [ ' +  ', '.join([str(v) for v in node.values])  +' ]'] +
            [' ' * 4 * level + self._NodeStr(child, level + 1) for child in node.children])

    Insert = insert


# ---------------------------------------------------------------------- #
#                           Error Element                                #
# ---------------------------------------------------------------------- #
class ErrorElement(Element):
    """
    A "dummy Element" that is returned when there are error conditions, like trying to find an element that's invalid
    """

    def __init__(self, key=None, metadata=None):
        """
        :param key: Used with window.find_element and with return values to uniquely identify this element
        :type key:
        """
        self.Key = key

        super().__init__(ELEM_TYPE_ERROR, key=key, metadata=metadata)

    def update(self, silent_on_error=True, *args, **kwargs):
        """
        Update method for the Error Element, an element that should not be directly used by developer

        :param silent_on_error: if False, then a Popup window will be shown
        :type silent_on_error:  (bool)
        :param *args:           meant to "soak up" any normal parameters passed in
        :type *args:            (Any)
        :param **kwargs:        meant to "soak up" any keyword parameters that were passed in
        :type **kwargs:         (Any)
        :return:                returns 'self' so call can be chained
        :rtype:                 (ErrorElement)
        """
        print('** Your update is being ignored because you supplied a bad key earlier **')
        return self

    def get(self):
        """
        One of the method names found in other Elements. Used here to return an error string in case it's called

        :return: A warning text string.
        :rtype:  (str)
        """
        return 'This is NOT a valid Element!\nSTOP trying to do things with it or I will have to crash at some point!'

    Get = get
    Update = update


# ---------------------------------------------------------------------- #
#                           Stretch Element                              #
# ---------------------------------------------------------------------- #
# This is for source code compatibility with tkinter version. No tkinter equivalent but you can fake it using a Text element that expands in the X direction
def Push(background_color=None):
    """
    Acts like a Stretch element found in the Qt port.
    Used in a Horizontal fashion.  Placing one on each side of an element will enter the element.
    Place one to the left and the element to the right will be right justified.  See VStretch for vertical type
    :param background_color: color of background may be needed because of how this is implemented
    :type background_color:  (str)
    :return:                 (Text)
    """
    return Text(font='_ 1', background_color=background_color, pad=(0,0), expand_x=True)

P = Push
Stretch = Push

def VPush(background_color=None):
    """
    Acts like a Stretch element found in the Qt port.
    Used in a Vertical fashion.
    :param background_color: color of background may be needed because of how this is implemented
    :type background_color:  (str)
    :return:                 (Text)
    """
    return Text(font='_ 1', background_color=background_color, pad=(0,0), expand_y=True)


VStretch = VPush
VP = VPush




# ------------------------------------------------------------------------- #
#                       _TimerPeriodic CLASS                                #
# ------------------------------------------------------------------------- #

class _TimerPeriodic:
    id_counter = 1
    # Dictionary containing the active timers.  Format is {id : _TimerPeriodic object}
    active_timers = {}         #type: dict[int:_TimerPeriodic]

    def __init__(self, window, frequency_ms, key=EVENT_TIMER, repeating=True):
        """
        :param window:          The window to send events to
        :type window:           Window
        :param frequency_ms:    How often to send events in milliseconds
        :type frequency_ms:     int
        :param repeating:       If True then the timer will run, repeatedly sending events, until stopped
        :type repeating:        bool
        """
        self.window = window
        self.frequency_ms = frequency_ms
        self.repeating = repeating
        self.key = key
        self.id = _TimerPeriodic.id_counter
        _TimerPeriodic.id_counter += 1
        self.start()


    @classmethod
    def stop_timer_with_id(cls, timer_id):
        """
        Not user callable!
        :return: A simple counter that makes each container element unique
        :rtype:
        """
        timer = cls.active_timers.get(timer_id, None)
        if timer is not None:
            timer.stop()


    @classmethod
    def stop_all_timers_for_window(cls, window):
        """
        Stops all timers for a given window
        :param window:      The window to stop timers for
        :type window:       Window
        """
        for timer in _TimerPeriodic.active_timers.values():
            if timer.window == window:
                timer.running = False

    @classmethod
    def get_all_timers_for_window(cls, window):
        """
        Returns a list of timer IDs for a given window
        :param window:      The window to find timers for
        :type window:       Window
        :return:            List of timer IDs for the window
        :rtype:             List[int]
        """
        timers = []
        for timer in _TimerPeriodic.active_timers.values():
            if timer.window == window:
                timers.append(timer.id)

        return timers



    def timer_thread(self):
        """
        The thread that sends events to the window.  Runs either once or in a loop until timer is stopped
        """

        if not self.running:            # if timer has been cancelled, abort
            del _TimerPeriodic.active_timers[self.id]
            return
        while True:
            time.sleep(self.frequency_ms/1000)
            if not self.running:        # if timer has been cancelled, abort
                del _TimerPeriodic.active_timers[self.id]
                return
            self.window.write_event_value(self.key, self.id)

            if not self.repeating:      # if timer does not repeat, then exit thread
                del _TimerPeriodic.active_timers[self.id]
                return


    def start(self):
        """
        Starts a timer by starting a timer thread
        Adds timer to the list of active timers
        """
        self.running = True
        self.thread = threading.Thread(target=self.timer_thread, daemon=True)
        self.thread.start()
        _TimerPeriodic.active_timers[self.id] = self


    def stop(self):
        """
        Stops a timer
        """
        self.running = False





# ------------------------------------------------------------------------- #
#                       Window CLASS                                        #
# ------------------------------------------------------------------------- #
class Window:
    """
    Represents a single Window
    """
    NumOpenWindows = 0
    _user_defined_icon = None
    hidden_master_root = None  # type: tk.Tk
    _animated_popup_dict = {}  # type: Dict
    _active_windows = {}  # type: Dict[Window, tk.Tk()]
    _move_all_windows = False  # if one window moved, they will move
    _window_that_exited = None  # type: Window
    _root_running_mainloop = None  # type: tk.Tk()    # (may be the hidden root or a window's root)
    _timeout_key = None
    _TKAfterID = None  # timer that is used to run reads with timeouts
    _window_running_mainloop = None  # The window that is running the mainloop
    _container_element_counter = 0  # used to get a number of Container Elements (Frame, Column, Tab)
    _read_call_from_debugger = False
    _timeout_0_counter = 0  # when timeout=0 then go through each window one at a time
    _counter_for_ttk_widgets = 0
    _floating_debug_window_build_needed = False
    _main_debug_window_build_needed = False
    # rereouted stdout info. List of tuples (window, element, previous destination)
    _rerouted_stdout_stack = []             # type: List[Tuple[Window, Element]]
    _rerouted_stderr_stack = []             # type: List[Tuple[Window, Element]]
    _original_stdout = None
    _original_stderr = None
    _watermark = None
    _watermark_temp_forced = False
    _watermark_user_text = ''

    def __init__(self, title, layout=None, default_element_size=None,
                 default_button_element_size=(None, None),
                 auto_size_text=None, auto_size_buttons=None, location=(None, None), relative_location=(None, None), size=(None, None),
                 element_padding=None, margins=(None, None), button_color=None, font=None,
                 progress_bar_color=(None, None), background_color=None, border_depth=None, auto_close=False,
                 auto_close_duration=DEFAULT_AUTOCLOSE_TIME, icon=None, force_toplevel=False,
                 alpha_channel=None, return_keyboard_events=False, use_default_focus=True, text_justification=None,
                 no_titlebar=False, grab_anywhere=False, grab_anywhere_using_control=True, keep_on_top=None, resizable=False, disable_close=False,
                 disable_minimize=False, right_click_menu=None, transparent_color=None, debugger_enabled=True,
                 right_click_menu_background_color=None, right_click_menu_text_color=None, right_click_menu_disabled_text_color=None, right_click_menu_selected_colors=(None, None),
                 right_click_menu_font=None, right_click_menu_tearoff=False,
                 finalize=False, element_justification='left', ttk_theme=None, use_ttk_buttons=None, modal=False, enable_close_attempted_event=False, enable_window_config_events=False,
                 titlebar_background_color=None, titlebar_text_color=None, titlebar_font=None, titlebar_icon=None,
                 use_custom_titlebar=None, scaling=None,
                 sbar_trough_color=None, sbar_background_color=None, sbar_arrow_color=None, sbar_width=None, sbar_arrow_width=None, sbar_frame_color=None, sbar_relief=None, watermark=None,
                 metadata=None):
        """
        :param title:                                The title that will be displayed in the Titlebar and on the Taskbar
        :type title:                                 (str)
        :param layout:                               The layout for the window. Can also be specified in the Layout method
        :type layout:                                List[List[Element]] | Tuple[Tuple[Element]]
        :param default_element_size:                 size in characters (wide) and rows (high) for all elements in this window
        :type default_element_size:                  (int, int) - (width, height)
        :param default_button_element_size:          (width, height) size in characters (wide) and rows (high) for all Button elements in this window
        :type default_button_element_size:           (int, int)
        :param auto_size_text:                       True if Elements in Window should be sized to exactly fir the length of text
        :type auto_size_text:                        (bool)
        :param auto_size_buttons:                    True if Buttons in this Window should be sized to exactly fit the text on this.
        :type auto_size_buttons:                     (bool)
        :param relative_location:                    (x,y) location relative to the default location of the window, in pixels. Normally the window centers.  This location is relative to the location the window would be created. Note they can be negative.
        :type relative_location:                     (int, int)
        :param location:                             (x,y) location, in pixels, to locate the upper left corner of the window on the screen. Default is to center on screen. None will not set any location meaning the OS will decide
        :type location:                              (int, int) or (None, None) or None
        :param size:                                 (width, height) size in pixels for this window. Normally the window is autosized to fit contents, not set to an absolute size by the user. Try not to set this value. You risk, the contents being cut off, etc. Let the layout determine the window size instead
        :type size:                                  (int, int)
        :param element_padding:                      Default amount of padding to put around elements in window (left/right, top/bottom) or ((left, right), (top, bottom)), or an int. If an int, then it's converted into a tuple (int, int)
        :type element_padding:                       (int, int) or ((int, int),(int,int)) or int
        :param margins:                              (left/right, top/bottom) Amount of pixels to leave inside the window's frame around the edges before your elements are shown.
        :type margins:                               (int, int)
        :param button_color:                         Default button colors for all buttons in the window
        :type button_color:                          (str, str) | str
        :param font:                                 specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
        :type font:                                  (str or (str, int[, str]) or None)
        :param progress_bar_color:                   (bar color, background color) Sets the default colors for all progress bars in the window
        :type progress_bar_color:                    (str, str)
        :param background_color:                     color of background
        :type background_color:                      (str)
        :param border_depth:                         Default border depth (width) for all elements in the window
        :type border_depth:                          (int)
        :param auto_close:                           If True, the window will automatically close itself
        :type auto_close:                            (bool)
        :param auto_close_duration:                  Number of seconds to wait before closing the window
        :type auto_close_duration:                   (int)
        :param icon:                                 Can be either a filename or Base64 value. For Windows if filename, it MUST be ICO format. For Linux, must NOT be ICO. Most portable is to use a Base64 of a PNG file. This works universally across all OS's
        :type icon:                                  (str | bytes)
        :param force_toplevel:                       If True will cause this window to skip the normal use of a hidden master window
        :type force_toplevel:                        (bool)
        :param alpha_channel:                        Sets the opacity of the window. 0 = invisible 1 = completely visible. Values bewteen 0 & 1 will produce semi-transparent windows in SOME environments (The Raspberry Pi always has this value at 1 and cannot change.
        :type alpha_channel:                         (float)
        :param return_keyboard_events:               if True key presses on the keyboard will be returned as Events from Read calls
        :type return_keyboard_events:                (bool)
        :param use_default_focus:                    If True will use the default focus algorithm to set the focus to the "Correct" element
        :type use_default_focus:                     (bool)
        :param text_justification:                   Default text justification for all Text Elements in window
        :type text_justification:                    'left' | 'right' | 'center'
        :param no_titlebar:                          If true, no titlebar nor frame will be shown on window. This means you cannot minimize the window and it will not show up on the taskbar
        :type no_titlebar:                           (bool)
        :param grab_anywhere:                        If True can use mouse to click and drag to move the window. Almost every location of the window will work except input fields on some systems
        :type grab_anywhere:                         (bool)
        :param grab_anywhere_using_control:          If True can use CONTROL key + left mouse mouse to click and drag to move the window. DEFAULT is TRUE. Unlike normal grab anywhere, it works on all elements.
        :type grab_anywhere_using_control:           (bool)
        :param keep_on_top:                          If True, window will be created on top of all other windows on screen. It can be bumped down if another window created with this parm
        :type keep_on_top:                           (bool)
        :param resizable:                            If True, allows the user to resize the window. Note the not all Elements will change size or location when resizing.
        :type resizable:                             (bool)
        :param disable_close:                        If True, the X button in the top right corner of the window will no work.  Use with caution and always give a way out toyour users
        :type disable_close:                         (bool)
        :param disable_minimize:                     if True the user won't be able to minimize window.  Good for taking over entire screen and staying that way.
        :type disable_minimize:                      (bool)
        :param right_click_menu:                     A list of lists of Menu items to show when this element is right clicked. See user docs for exact format.
        :type right_click_menu:                      List[List[ List[str] | str ]]
        :param transparent_color:                    Any portion of the window that has this color will be completely transparent. You can even click through these spots to the window under this window.
        :type transparent_color:                     (str)
        :param debugger_enabled:                     If True then the internal debugger will be enabled
        :type debugger_enabled:                      (bool)
        :param right_click_menu_background_color:    Background color for right click menus
        :type right_click_menu_background_color:     (str)
        :param right_click_menu_text_color:          Text color for right click menus
        :type right_click_menu_text_color:           (str)
        :param right_click_menu_disabled_text_color: Text color for disabled right click menu items
        :type right_click_menu_disabled_text_color:  (str)
        :param right_click_menu_selected_colors:     Text AND background colors for a selected item. Can be a Tuple OR a color string. simplified-button-color-string "foreground on background". Can be a single color if want to set only the background. Normally a tuple, but can be a simplified-dual-color-string "foreground on background". Can be a single color if want to set only the background.
        :type right_click_menu_selected_colors:      (str, str) | str | Tuple
        :param right_click_menu_font:                Font for right click menus
        :type right_click_menu_font:                 (str or (str, int[, str]) or None)
        :param right_click_menu_tearoff:             If True then all right click menus can be torn off
        :type right_click_menu_tearoff:              bool
        :param finalize:                             If True then the Finalize method will be called. Use this rather than chaining .Finalize for cleaner code
        :type finalize:                              (bool)
        :param element_justification:                All elements in the Window itself will have this justification 'left', 'right', 'center' are valid values
        :type element_justification:                 (str)
        :param ttk_theme:                            Set the tkinter ttk "theme" of the window.  Default = DEFAULT_TTK_THEME.  Sets all ttk widgets to this theme as their default
        :type ttk_theme:                             (str)
        :param use_ttk_buttons:                      Affects all buttons in window. True = use ttk buttons. False = do not use ttk buttons.  None = use ttk buttons only if on a Mac
        :type use_ttk_buttons:                       (bool)
        :param modal:                                If True then this window will be the only window a user can interact with until it is closed
        :type modal:                                 (bool)
        :param enable_close_attempted_event:         If True then the window will not close when "X" clicked. Instead an event WINDOW_CLOSE_ATTEMPTED_EVENT if returned from window.read
        :type enable_close_attempted_event:          (bool)
        :param enable_window_config_events:          If True then window configuration events (resizing or moving the window) will return WINDOW_CONFIG_EVENT from window.read. Note you will get several when Window is created.
        :type enable_window_config_events:           (bool)
        :param titlebar_background_color:            If custom titlebar indicated by use_custom_titlebar, then use this as background color
        :type titlebar_background_color:             (str | None)
        :param titlebar_text_color:                  If custom titlebar indicated by use_custom_titlebar, then use this as text color
        :type titlebar_text_color:                   (str | None)
        :param titlebar_font:                        If custom titlebar indicated by use_custom_titlebar, then use this as title font
        :type titlebar_font:                         (str or (str, int[, str]) or None)
        :param titlebar_icon:                        If custom titlebar indicated by use_custom_titlebar, then use this as the icon (file or base64 bytes)
        :type titlebar_icon:                         (bytes | str)
        :param use_custom_titlebar:                  If True, then a custom titlebar will be used instead of the normal titlebar
        :type use_custom_titlebar:                   bool
        :param scaling:                              Apply scaling to the elements in the window. Can be set on a global basis using set_options
        :type scaling:                               float
        :param sbar_trough_color:                    Scrollbar color of the trough
        :type sbar_trough_color:                     (str)
        :param sbar_background_color:                Scrollbar color of the background of the arrow buttons at the ends AND the color of the "thumb" (the thing you grab and slide). Switches to arrow color when mouse is over
        :type sbar_background_color:                 (str)
        :param sbar_arrow_color:                     Scrollbar color of the arrow at the ends of the scrollbar (it looks like a button). Switches to background color when mouse is over
        :type sbar_arrow_color:                      (str)
        :param sbar_width:                           Scrollbar width in pixels
        :type sbar_width:                            (int)
        :param sbar_arrow_width:                     Scrollbar width of the arrow on the scrollbar. It will potentially impact the overall width of the scrollbar
        :type sbar_arrow_width:                      (int)
        :param sbar_frame_color:                     Scrollbar Color of frame around scrollbar (available only on some ttk themes)
        :type sbar_frame_color:                      (str)
        :param sbar_relief:                          Scrollbar relief that will be used for the "thumb" of the scrollbar (the thing you grab that slides). Should be a constant that is defined at starting with "RELIEF_" - RELIEF_RAISED, RELIEF_SUNKEN, RELIEF_FLAT, RELIEF_RIDGE, RELIEF_GROOVE, RELIEF_SOLID
        :type sbar_relief:                           (str)
        :param watermark:                            If True, then turns on watermarking temporarily for ALL windows created from this point forward. See global settings doc for more info
        :type watermark:                             bool
        :param metadata:                             User metadata that can be set to ANYTHING
        :type metadata:                              (Any)
        """


        self._metadata = None  # type: Any
        self.AutoSizeText = auto_size_text if auto_size_text is not None else DEFAULT_AUTOSIZE_TEXT
        self.AutoSizeButtons = auto_size_buttons if auto_size_buttons is not None else DEFAULT_AUTOSIZE_BUTTONS
        self.Title = str(title)
        self.Rows = []  # a list of ELEMENTS for this row
        self.DefaultElementSize = default_element_size if default_element_size is not None else DEFAULT_ELEMENT_SIZE
        self.DefaultButtonElementSize = default_button_element_size if default_button_element_size != (
            None, None) else DEFAULT_BUTTON_ELEMENT_SIZE
        if DEFAULT_WINDOW_LOCATION != (None, None) and location == (None, None):
            self.Location = DEFAULT_WINDOW_LOCATION
        else:
            self.Location = location
        self.RelativeLoction = relative_location
        self.ButtonColor = button_color_to_tuple(button_color)
        self.BackgroundColor = background_color if background_color else DEFAULT_BACKGROUND_COLOR
        self.ParentWindow = None
        self.Font = font if font else DEFAULT_FONT
        self.RadioDict = {}
        self.BorderDepth = border_depth
        if icon:
            self.WindowIcon = icon
        elif Window._user_defined_icon is not None:
            self.WindowIcon = Window._user_defined_icon
        else:
            self.WindowIcon = DEFAULT_WINDOW_ICON
        self.AutoClose = auto_close
        self.NonBlocking = False
        self.TKroot = None  # type: tk.Tk
        self.TKrootDestroyed = False
        self.CurrentlyRunningMainloop = False
        self.FormRemainedOpen = False
        self.TKAfterID = None
        self.ProgressBarColor = progress_bar_color
        self.AutoCloseDuration = auto_close_duration
        self.RootNeedsDestroying = False
        self.Shown = False
        self.ReturnValues = None
        self.ReturnValuesList = []
        self.ReturnValuesDictionary = {}
        self.DictionaryKeyCounter = 0
        self.LastButtonClicked = None
        self.LastButtonClickedWasRealtime = False
        self.UseDictionary = False
        self.UseDefaultFocus = use_default_focus
        self.ReturnKeyboardEvents = return_keyboard_events
        self.LastKeyboardEvent = None
        self.TextJustification = text_justification
        self.NoTitleBar = no_titlebar
        self.Grab = grab_anywhere
        self.GrabAnywhere = grab_anywhere
        self.GrabAnywhereUsingControlKey = grab_anywhere_using_control
        if keep_on_top is None and DEFAULT_KEEP_ON_TOP is not None:
            keep_on_top = DEFAULT_KEEP_ON_TOP
        elif keep_on_top is None:
            keep_on_top = False
        self.KeepOnTop = keep_on_top
        self.ForceTopLevel = force_toplevel
        self.Resizable = resizable
        self._AlphaChannel = alpha_channel if alpha_channel is not None else DEFAULT_ALPHA_CHANNEL
        self.Timeout = None
        self.TimeoutKey = TIMEOUT_KEY
        self.TimerCancelled = False
        self.DisableClose = disable_close
        self.DisableMinimize = disable_minimize
        self._Hidden = False
        self._Size = size
        self.XFound = False
        if element_padding is not None:
            if isinstance(element_padding, int):
                element_padding = (element_padding, element_padding)

        if element_padding is None:
            self.ElementPadding = DEFAULT_ELEMENT_PADDING
        else:
            self.ElementPadding = element_padding
        self.RightClickMenu = right_click_menu
        self.Margins = margins if margins != (None, None) else DEFAULT_MARGINS
        self.ContainerElemementNumber = Window._GetAContainerNumber()
        # The dictionary containing all elements and keys for the window
        # The keys are the keys for the elements and the values are the elements themselves.
        self.AllKeysDict = {}
        self.TransparentColor = transparent_color
        self.UniqueKeyCounter = 0
        self.DebuggerEnabled = debugger_enabled
        self.WasClosed = False
        self.ElementJustification = element_justification
        self.FocusSet = False
        self.metadata = metadata
        self.TtkTheme = ttk_theme or DEFAULT_TTK_THEME
        self.UseTtkButtons = use_ttk_buttons if use_ttk_buttons is not None else USE_TTK_BUTTONS
        self.user_bind_dict = {}  # Used when user defines a tkinter binding using bind method - convert bind string to key modifier
        self.user_bind_event = None  # Used when user defines a tkinter binding using bind method - event data from tkinter
        self.modal = modal
        self.thread_queue = None  # type: queue.Queue
        self.thread_lock = None  # type: threading.Lock
        self.thread_timer = None  # type: tk.Misc
        self.thread_strvar = None  # type: tk.StringVar
        self.read_closed_window_count = 0
        self.config_last_size = (None, None)
        self.config_last_location = (None, None)
        self.starting_window_position = (None, None)
        self.not_completed_initial_movement = True
        self.config_count = 0
        self.saw_00 = False
        self.maximized = False
        self.right_click_menu_background_color = right_click_menu_background_color if right_click_menu_background_color is not None else theme_input_background_color()
        self.right_click_menu_text_color = right_click_menu_text_color if right_click_menu_text_color is not None else theme_input_text_color()
        self.right_click_menu_disabled_text_color = right_click_menu_disabled_text_color if right_click_menu_disabled_text_color is not None else COLOR_SYSTEM_DEFAULT
        self.right_click_menu_font = right_click_menu_font if right_click_menu_font is not None else self.Font
        self.right_click_menu_tearoff = right_click_menu_tearoff
        self.auto_close_timer_needs_starting = False
        self.finalize_in_progress = False
        self.close_destroys_window = not enable_close_attempted_event if enable_close_attempted_event is not None else None
        self.enable_window_config_events = enable_window_config_events
        self.override_custom_titlebar = False
        self.use_custom_titlebar = use_custom_titlebar or theme_use_custom_titlebar()
        self.titlebar_background_color = titlebar_background_color
        self.titlebar_text_color = titlebar_text_color
        self.titlebar_font = titlebar_font
        self.titlebar_icon = titlebar_icon
        self.right_click_menu_selected_colors = _simplified_dual_color_to_tuple(right_click_menu_selected_colors,
                                                                                (self.right_click_menu_background_color, self.right_click_menu_text_color))
        self.TKRightClickMenu = None
        self._grab_anywhere_ignore_these_list = []
        self._grab_anywhere_include_these_list = []
        self._has_custom_titlebar = use_custom_titlebar
        self._mousex = self._mousey = 0
        self._startx = self._starty = 0
        self.scaling = scaling if scaling is not None else DEFAULT_SCALING
        if self.use_custom_titlebar:
            self.Margins = (0, 0)
            self.NoTitleBar = True
        self._mouse_offset_x = self._mouse_offset_y = 0

        if watermark is True:
            Window._watermark_temp_forced = True
            _global_settings_get_watermark_info()
        elif watermark is False:
            Window._watermark = None
            Window._watermark_temp_forced = False


        self.ttk_part_overrides = TTKPartOverrides(sbar_trough_color=sbar_trough_color, sbar_background_color=sbar_background_color, sbar_arrow_color=sbar_arrow_color, sbar_width=sbar_width, sbar_arrow_width=sbar_arrow_width, sbar_frame_color=sbar_frame_color, sbar_relief=sbar_relief)

        if no_titlebar is True:
            self.override_custom_titlebar = True

        if layout is not None and type(layout) not in (list, tuple):
            warnings.warn('Your layout is not a list or tuple... this is not good!')

        if layout is not None:
            self.Layout(layout)
            if finalize:
                self.Finalize()

        if CURRENT_LOOK_AND_FEEL == 'Default':
            print('Window will be a boring gray. Try removing the theme call entirely\n',
                  'You will get the default theme or the one set in global settings\n'
                  "If you seriously want this gray window and no more nagging, add  theme('DefaultNoMoreNagging')  or theme('Gray Gray Gray') for completely gray/System Defaults")


    @classmethod
    def _GetAContainerNumber(cls):
        """
        Not user callable!
        :return: A simple counter that makes each container element unique
        :rtype:
        """
        cls._container_element_counter += 1
        return cls._container_element_counter

    @classmethod
    def _IncrementOpenCount(self):
        """
        Not user callable!  Increments the number of open windows
        Note - there is a bug where this count easily gets out of sync. Issue has been opened already. No ill effects
        """
        self.NumOpenWindows += 1
        # print('+++++ INCREMENTING Num Open Windows = {} ---'.format(Window.NumOpenWindows))

    @classmethod
    def _DecrementOpenCount(self):
        """
        Not user callable!  Decrements the number of open windows
        """
        self.NumOpenWindows -= 1 * (self.NumOpenWindows != 0)  # decrement if not 0
        # print('----- DECREMENTING Num Open Windows = {} ---'.format(Window.NumOpenWindows))

    @classmethod
    def get_screen_size(self):
        """
        This is a "Class Method" meaning you call it by writing: width, height = Window.get_screen_size()
        Returns the size of the "screen" as determined by tkinter.  This can vary depending on your operating system and the number of monitors installed on your system.  For Windows, the primary monitor's size is returns. On some multi-monitored Linux systems, the monitors are combined and the total size is reported as if one screen.

        :return: Size of the screen in pixels as determined by tkinter
        :rtype:  (int, int)
        """
        root = _get_hidden_master_root()
        screen_width = root.winfo_screenwidth()
        screen_height = root.winfo_screenheight()
        return screen_width, screen_height

    @property
    def metadata(self):
        """
        Metadata is available for all windows. You can set to any value.
        :return: the current metadata value
        :rtype:  (Any)
        """
        return self._metadata

    @metadata.setter
    def metadata(self, value):
        """
        Metadata is available for all windows. You can set to any value.
        :param value: Anything you want it to be
        :type value:  (Any)
        """
        self._metadata = value

    # ------------------------- Add ONE Row to Form ------------------------- #
    def add_row(self, *args):
        """
        Adds a single row of elements to a window's self.Rows variables.
        Generally speaking this is NOT how users should be building Window layouts.
        Users, create a single layout (a list of lists) and pass as a parameter to Window object, or call Window.Layout(layout)

        :param *args: List[Elements]
        :type *args:
        """
        NumRows = len(self.Rows)  # number of existing rows is our row number
        CurrentRowNumber = NumRows  # this row's number
        CurrentRow = []  # start with a blank row and build up
        # -------------------------  Add the elements to a row  ------------------------- #
        for i, element in enumerate(args):  # Loop through list of elements and add them to the row

            if isinstance(element, tuple) or isinstance(element, list):
                self.add_row(*element)
                continue
                _error_popup_with_traceback('Error creating Window layout',
                                            'Layout has a LIST instead of an ELEMENT',
                                            'This sometimes means you have a badly placed ]',
                                            'The offensive list is:',
                                            element,
                                            'This list will be stripped from your layout'
                                            )
                continue
            elif callable(element) and not isinstance(element, Element):
                _error_popup_with_traceback('Error creating Window layout',
                                            'Layout has a FUNCTION instead of an ELEMENT',
                                            'This likely means you are missing () from your layout',
                                            'The offensive list is:',
                                            element,
                                            'This item will be stripped from your layout')
                continue
            if element.ParentContainer is not None:
                warnings.warn(
                    '*** YOU ARE ATTEMPTING TO REUSE AN ELEMENT IN YOUR LAYOUT! Once placed in a layout, an element cannot be used in another layout. ***',
                    UserWarning)
                _error_popup_with_traceback('Error detected in layout - Contains an element that has already been used.',
                                            'You have attempted to reuse an element in your layout.',
                                            "The layout specified has an element that's already been used.",
                                            'You MUST start with a "clean", unused layout every time you create a window',
                                            'The offensive Element = ',
                                            element,
                                            'and has a key = ', element.Key,
                                            'This item will be stripped from your layout',
                                            'Hint - try printing your layout and matching the IDs "print(layout)"')
                continue
            element.Position = (CurrentRowNumber, i)
            element.ParentContainer = self
            CurrentRow.append(element)
            # if this element is a titlebar, then automatically set the window margins to (0,0) and turn off normal titlebar
            if element.metadata == TITLEBAR_METADATA_MARKER:
                self.Margins = (0, 0)
                self.NoTitleBar = True
        # -------------------------  Append the row to list of Rows  ------------------------- #
        self.Rows.append(CurrentRow)

    # ------------------------- Add Multiple Rows to Form ------------------------- #
    def add_rows(self, rows):
        """
        Loops through a list of lists of elements and adds each row, list, to the layout.
        This is NOT the best way to go about creating a window.  Sending the entire layout at one time and passing
        it as a parameter to the Window call is better.

        :param rows: A list of a list of elements
        :type rows:  List[List[Elements]]
        """
        for row in rows:
            try:
                iter(row)
            except TypeError:
                _error_popup_with_traceback('Error Creating Window Layout', 'Error creating Window layout',
                           'Your row is not an iterable (e.g. a list)',
                           'Instead of a list, the type found was {}'.format(type(row)),
                           'The offensive row = ',
                           row,
                           'This item will be stripped from your layout')
                continue
            self.add_row(*row)
        # if _optional_window_data(self) is not None:
        #     self.add_row(_optional_window_data(self))
        if Window._watermark is not None:
            self.add_row(Window._watermark(self))



    def layout(self, rows):
        """
        Second of two preferred ways of telling a Window what its layout is. The other way is to pass the layout as
        a parameter to Window object.  The parameter method is the currently preferred method. This call to Layout
        has been removed from examples contained in documents and in the Demo Programs. Trying to remove this call
        from history and replace with sending as a parameter to Window.

        :param rows: Your entire layout
        :type rows:  List[List[Elements]]
        :return:     self so that you can chain method calls
        :rtype:      (Window)
        """
        if self.use_custom_titlebar and not self.override_custom_titlebar:
            if self.titlebar_icon is not None:
                icon = self.titlebar_icon
            elif CUSTOM_TITLEBAR_ICON is not None:
                icon = CUSTOM_TITLEBAR_ICON
            elif self.titlebar_icon is not None:
                icon = self.titlebar_icon
            elif self.WindowIcon == DEFAULT_WINDOW_ICON:
                icon = DEFAULT_BASE64_ICON_16_BY_16
            else:
                icon = None

            new_rows = [[Titlebar(title=self.Title, icon=icon, text_color=self.titlebar_text_color, background_color=self.titlebar_background_color,
                                  font=self.titlebar_font)]] + rows
        else:
            new_rows = rows
        self.add_rows(new_rows)
        self._BuildKeyDict()

        if self._has_custom_titlebar_element():
            self.Margins = (0, 0)
            self.NoTitleBar = True
            self._has_custom_titlebar = True
        return self

    def extend_layout(self, container, rows):
        """
        Adds new rows to an existing container element inside of this window
        If the container is a scrollable Column, you need to also call the contents_changed() method

        :param container: The container Element the layout will be placed inside of
        :type container:  Frame | Column | Tab
        :param rows:      The layout to be added
        :type rows:       (List[List[Element]])
        :return:          (Window) self so could be chained
        :rtype:           (Window)
        """
        column = Column(rows, pad=(0, 0), background_color=container.BackgroundColor)
        if self == container:
            frame = self.TKroot
        elif isinstance(container.Widget, TkScrollableFrame):
            frame = container.Widget.TKFrame
        else:
            frame = container.Widget
        PackFormIntoFrame(column, frame, self)
        # sg.PackFormIntoFrame(col, window.TKroot, window)
        self.AddRow(column)
        self.AllKeysDict = self._BuildKeyDictForWindow(self, column, self.AllKeysDict)
        return self

    def LayoutAndRead(self, rows, non_blocking=False):
        """
        Deprecated!!  Now your layout your window's rows (layout) and then separately call Read.

        :param rows:         The layout of the window
        :type rows:          List[List[Element]]
        :param non_blocking: if True the Read call will not block
        :type non_blocking:  (bool)
        """
        _error_popup_with_traceback('LayoutAndRead Depricated', 'Wow!  You have been using PySimpleGUI for a very long time.',
                                                                'The Window.LayoutAndRead call is no longer supported')

        raise DeprecationWarning(
            'LayoutAndRead is no longer supported... change your call window.Layout(layout).Read()\nor window(title, layout).Read()')
        # self.AddRows(rows)
        # self._Show(non_blocking=non_blocking)
        # return self.ReturnValues

    def LayoutAndShow(self, rows):
        """
        Deprecated - do not use any longer.  Layout your window and then call Read.  Or can add a Finalize call before the Read
        """
        raise DeprecationWarning('LayoutAndShow is no longer supported... ')

    def _Show(self, non_blocking=False):
        """
        NOT TO BE CALLED BY USERS.  INTERNAL ONLY!
        It's this method that first shows the window to the user, collects results

        :param non_blocking: if True, this is a non-blocking call
        :type non_blocking:  (bool)
        :return:             Tuple[Any, Dict] The event, values turple that is returned from Read calls
        :rtype:
        """
        self.Shown = True
        # Compute num rows & num cols (it'll come in handy debugging)
        self.NumRows = len(self.Rows)
        self.NumCols = max(len(row) for row in self.Rows)
        self.NonBlocking = non_blocking

        # Search through entire form to see if any elements set the focus
        # if not, then will set the focus to the first input element
        found_focus = False
        for row in self.Rows:
            for element in row:
                try:
                    if element.Focus:
                        found_focus = True
                except:
                    pass
                try:
                    if element.Key is not None:
                        self.UseDictionary = True
                except:
                    pass

        if not found_focus and self.UseDefaultFocus:
            self.UseDefaultFocus = True
        else:
            self.UseDefaultFocus = False
        # -=-=-=-=-=-=-=-=- RUN the GUI -=-=-=-=-=-=-=-=- ##
        StartupTK(self)
        # If a button or keyboard event happened but no results have been built, build the results
        if self.LastKeyboardEvent is not None or self.LastButtonClicked is not None:
            return _BuildResults(self, False, self)
        return self.ReturnValues

    # ------------------------- SetIcon - set the window's fav icon ------------------------- #
    def set_icon(self, icon=None, pngbase64=None):
        """
        Changes the icon that is shown on the title bar and on the task bar.
        NOTE - The file type is IMPORTANT and depends on the OS!
        Can pass in:
        * filename which must be a .ICO icon file for windows, PNG file for Linux
        * bytes object
        * BASE64 encoded file held in a variable

        :param icon:      Filename or bytes object
        :type icon:       (str)
        :param pngbase64: Base64 encoded image
        :type pngbase64:  (bytes)
        """
        if type(icon) is bytes or pngbase64 is not None:
            wicon = tkinter.PhotoImage(data=icon if icon is not None else pngbase64)
            try:
                self.TKroot.tk.call('wm', 'iconphoto', self.TKroot._w, wicon)
            except:
                wicon = tkinter.PhotoImage(data=DEFAULT_BASE64_ICON)
                try:
                    self.TKroot.tk.call('wm', 'iconphoto', self.TKroot._w, wicon)
                except:
                    pass
            self.WindowIcon = wicon
            return

        wicon = icon
        try:
            self.TKroot.iconbitmap(icon)
        except:
            try:
                wicon = tkinter.PhotoImage(file=icon)
                self.TKroot.tk.call('wm', 'iconphoto', self.TKroot._w, wicon)
            except:
                try:
                    wicon = tkinter.PhotoImage(data=DEFAULT_BASE64_ICON)
                    try:
                        self.TKroot.tk.call('wm', 'iconphoto', self.TKroot._w, wicon)
                    except:
                        pass
                except:
                    pass
        self.WindowIcon = wicon

    def _GetElementAtLocation(self, location):
        """
        Given a (row, col) location in a layout, return the element located at that position

        :param location: (int, int) Return the element located at (row, col) in layout
        :type location:
        :return:         (Element) The Element located at that position in this window
        :rtype:
        """

        (row_num, col_num) = location
        row = self.Rows[row_num]
        element = row[col_num]
        return element

    def _GetDefaultElementSize(self):
        """
        Returns the default elementSize

        :return: (width, height) of the default element size
        :rtype:  (int, int)
        """

        return self.DefaultElementSize

    def _AutoCloseAlarmCallback(self):
        """
        Function that's called by tkinter when autoclode timer expires.  Closes the window

        """
        try:
            window = self
            if window:
                if window.NonBlocking:
                    self.Close()
                else:
                    window._Close()
                    self.TKroot.quit()
                    self.RootNeedsDestroying = True
        except:
            pass

    def _TimeoutAlarmCallback(self):
        """
        Read Timeout Alarm callback. Will kick a mainloop call out of the tkinter event loop and cause it to return
        """
        # first, get the results table built
        # modify the Results table in the parent FlexForm object
        # print('TIMEOUT CALLBACK')
        if self.TimerCancelled:
            # print('** timer was cancelled **')
            return
        self.LastButtonClicked = self.TimeoutKey
        self.FormRemainedOpen = True
        self.TKroot.quit()  # kick the users out of the mainloop

    def _calendar_chooser_button_clicked(self, elem):
        """

        :param elem:
        :type elem:
        :return:
        :rtype:
        """
        target_element, strvar, should_submit_window = elem._find_target()

        if elem.calendar_default_date_M_D_Y == (None, None, None):
            now = datetime.datetime.now()
            cur_month, cur_day, cur_year = now.month, now.day, now.year
        else:
            cur_month, cur_day, cur_year = elem.calendar_default_date_M_D_Y

        date_chosen = popup_get_date(start_mon=cur_month, start_day=cur_day, start_year=cur_year, close_when_chosen=elem.calendar_close_when_chosen,
                                     no_titlebar=elem.calendar_no_titlebar, begin_at_sunday_plus=elem.calendar_begin_at_sunday_plus,
                                     locale=elem.calendar_locale, location=elem.calendar_location, month_names=elem.calendar_month_names,
                                     day_abbreviations=elem.calendar_day_abbreviations, title=elem.calendar_title)
        if date_chosen is not None:
            month, day, year = date_chosen
            now = datetime.datetime.now()
            hour, minute, second = now.hour, now.minute, now.second
            try:
                date_string = calendar.datetime.datetime(year, month, day, hour, minute, second).strftime(elem.calendar_format)
            except Exception as e:
                print('Bad format string in calendar chooser button', e)
                date_string = 'Bad format string'

            if target_element is not None and target_element != elem:
                target_element.update(date_string)
            elif target_element == elem:
                elem.calendar_selection = date_string

            strvar.set(date_string)
            elem.TKStringVar.set(date_string)
            if should_submit_window:
                self.LastButtonClicked = target_element.Key
                results = _BuildResults(self, False, self)
        else:
            should_submit_window = False
        return should_submit_window

    # @_timeit_summary
    def read(self, timeout=None, timeout_key=TIMEOUT_KEY, close=False):
        """
        THE biggest deal method in the Window class! This is how you get all of your data from your Window.
            Pass in a timeout (in milliseconds) to wait for a maximum of timeout milliseconds. Will return timeout_key
            if no other GUI events happen first.

        :param timeout:     Milliseconds to wait until the Read will return IF no other GUI events happen first
        :type timeout:      (int)
        :param timeout_key: The value that will be returned from the call if the timer expired
        :type timeout_key:  (Any)
        :param close:       if True the window will be closed prior to returning
        :type close:        (bool)
        :return:            (event, values)
        :rtype:             Tuple[(Any), Dict[Any, Any], List[Any], None]
        """

        if Window._floating_debug_window_build_needed is True:
            Window._floating_debug_window_build_needed = False
            _Debugger.debugger._build_floating_window()

        if Window._main_debug_window_build_needed is True:
            Window._main_debug_window_build_needed = False
            _Debugger.debugger._build_main_debugger_window()

        # ensure called only 1 time through a single read cycle
        if not Window._read_call_from_debugger:
            _refresh_debugger()

        # if the user has not added timeout and a debug window is open, then set a timeout for them so the debugger continuously refreshes
        if _debugger_window_is_open() and not Window._read_call_from_debugger:
            if timeout is None or timeout > 3000:
                timeout = 200


        while True:
            Window._root_running_mainloop = self.TKroot
            results = self._read(timeout=timeout, timeout_key=timeout_key)
            if results is not None:
                if results[0] == DEFAULT_WINDOW_SNAPSHOT_KEY:
                    self.save_window_screenshot_to_disk()
                    popup_quick_message('Saved window screenshot to disk', background_color='#1c1e23', text_color='white', keep_on_top=True, font='_ 30')
                    continue
            # Post processing for Calendar Chooser Button
            try:
                if results[0] == timeout_key:  # if a timeout, then not a calendar button
                    break
                elem = self.find_element(results[0], silent_on_error=True)  # get the element that caused the event
                if elem.Type == ELEM_TYPE_BUTTON:
                    if elem.BType == BUTTON_TYPE_CALENDAR_CHOOSER:
                        if self._calendar_chooser_button_clicked(elem):  # returns True if should break out
                            # results[0] = self.LastButtonClicked
                            results = self.ReturnValues
                            break
                        else:
                            continue
                break
            except:
                break  # wasn't a calendar button for sure


        if close:
            self.close()

        return results

    # @_timeit
    def _read(self, timeout=None, timeout_key=TIMEOUT_KEY):
        """
        THE biggest deal method in the Window class! This is how you get all of your data from your Window.
            Pass in a timeout (in milliseconds) to wait for a maximum of timeout milliseconds. Will return timeout_key
            if no other GUI events happen first.

        :param timeout:     Milliseconds to wait until the Read will return IF no other GUI events happen first
        :type timeout:      (int)
        :param timeout_key: The value that will be returned from the call if the timer expired
        :type timeout_key:  (Any)
        :return:            (event, values) (event or timeout_key or None, Dictionary of values or List of values from all elements in the Window)
        :rtype:             Tuple[(Any), Dict[Any, Any], List[Any], None]
        """

        # if there are events in the thread event queue, then return those events before doing anything else.
        if self._queued_thread_event_available():
            self.ReturnValues = results = _BuildResults(self, False, self)
            return results

        if self.finalize_in_progress and self.auto_close_timer_needs_starting:
            self._start_autoclose_timer()
            self.auto_close_timer_needs_starting = False

        timeout = int(timeout) if timeout is not None else None
        if timeout == 0:  # timeout of zero runs the old readnonblocking
            event, values = self._ReadNonBlocking()
            if event is None:
                event = timeout_key
            if values is None:
                event = None
            return event, values  # make event None if values was None and return
        # Read with a timeout
        self.Timeout = timeout
        self.TimeoutKey = timeout_key
        self.NonBlocking = False
        if self.TKrootDestroyed:
            self.read_closed_window_count += 1
            if self.read_closed_window_count > 100:
                popup_error_with_traceback('Trying to read a closed window', 'You have tried 100 times to read a closed window.', 'You need to add a check for event == WIN_CLOSED',)
            return None, None
        if not self.Shown:
            self._Show()
        else:
            # if already have a button waiting, the return previously built results
            if self.LastButtonClicked is not None and not self.LastButtonClickedWasRealtime:
                results = _BuildResults(self, False, self)
                self.LastButtonClicked = None
                return results
            InitializeResults(self)

            if self._queued_thread_event_available():
                self.ReturnValues = results = _BuildResults(self, False, self)
                return results

            # if the last button clicked was realtime, emulate a read non-blocking
            # the idea is to quickly return realtime buttons without any blocks until released
            if self.LastButtonClickedWasRealtime:
                # clear the realtime flag if the element is not a button element (for example a graph element that is dragging)
                if self.AllKeysDict.get(self.LastButtonClicked, None):
                    if self.AllKeysDict.get(self.LastButtonClicked).Type != ELEM_TYPE_BUTTON:
                        self.LastButtonClickedWasRealtime = False  # stops from generating events until something changes
                else:  # it is possible for the key to not be in the dicitonary because it has a modifier. If so, then clear the realtime button flag
                    self.LastButtonClickedWasRealtime = False  # stops from generating events until something changes

                try:
                    rc = self.TKroot.update()
                except:
                    self.TKrootDestroyed = True
                    Window._DecrementOpenCount()
                    # _my_windows.Decrement()
                    # print('ROOT Destroyed')
                results = _BuildResults(self, False, self)
                if results[0] != None and results[0] != timeout_key:
                    return results
                else:
                    pass

                # else:
                #     print("** REALTIME PROBLEM FOUND **", results)

            if self.RootNeedsDestroying:
                # print('*** DESTROYING really late***')
                try:
                    self.TKroot.destroy()
                except:
                    pass
                # _my_windows.Decrement()
                self.LastButtonClicked = None
                return None, None

            # normal read blocking code....
            if timeout != None:
                self.TimerCancelled = False
                self.TKAfterID = self.TKroot.after(timeout, self._TimeoutAlarmCallback)
            self.CurrentlyRunningMainloop = True
            # self.TKroot.protocol("WM_DESTROY_WINDOW", self._OnClosingCallback)
            # self.TKroot.protocol("WM_DELETE_WINDOW", self._OnClosingCallback)
            Window._window_running_mainloop = self
            try:
                Window._root_running_mainloop.mainloop()
            except:
                print('**** EXITING ****')
                exit(-1)
            # print('Out main')
            self.CurrentlyRunningMainloop = False
            # if self.LastButtonClicked != TIMEOUT_KEY:
            try:
                self.TKroot.after_cancel(self.TKAfterID)
                del self.TKAfterID
            except:
                pass
                # print('** tkafter cancel failed **')
            self.TimerCancelled = True
            if self.RootNeedsDestroying:
                # print('*** DESTROYING LATE ***')
                try:
                    self.TKroot.destroy()
                except:
                    pass
                Window._DecrementOpenCount()
                # _my_windows.Decrement()
                self.LastButtonClicked = None
                return None, None
            # if form was closed with X
            if self.LastButtonClicked is None and self.LastKeyboardEvent is None and self.ReturnValues[0] is None:
                Window._DecrementOpenCount()
                # _my_windows.Decrement()
        # Determine return values
        if self.LastKeyboardEvent is not None or self.LastButtonClicked is not None:
            results = _BuildResults(self, False, self)
            if not self.LastButtonClickedWasRealtime:
                self.LastButtonClicked = None
            return results
        else:
            if self._queued_thread_event_available():
                self.ReturnValues = results = _BuildResults(self, False, self)
                return results
            if not self.XFound and self.Timeout != 0 and self.Timeout is not None and self.ReturnValues[
                0] is None:  # Special Qt case because returning for no reason so fake timeout
                self.ReturnValues = self.TimeoutKey, self.ReturnValues[1]  # fake a timeout
            elif not self.XFound and self.ReturnValues[0] is None:  # Return a timeout event... can happen when autoclose used on another window
                # print("*** Faking timeout ***")
                self.ReturnValues = self.TimeoutKey, self.ReturnValues[1]  # fake a timeout
            return self.ReturnValues

    def _ReadNonBlocking(self):
        """
        Should be NEVER called directly by the user.  The user can call Window.read(timeout=0) to get same effect

        :return: (event, values). (event or timeout_key or None, Dictionary of values or List of values from all elements in the Window)
        :rtype:  Tuple[(Any), Dict[Any, Any] | List[Any] | None]
        """
        if self.TKrootDestroyed:
            try:
                self.TKroot.quit()
                self.TKroot.destroy()
            except:
                pass
                # print('DESTROY FAILED')
            return None, None
        if not self.Shown:
            self._Show(non_blocking=True)
        try:
            rc = self.TKroot.update()
        except:
            self.TKrootDestroyed = True
            Window._DecrementOpenCount()
            # _my_windows.Decrement()
            # print("read failed")
            # return None, None
        if self.RootNeedsDestroying:
            # print('*** DESTROYING LATE ***', self.ReturnValues)
            self.TKroot.destroy()
            Window._DecrementOpenCount()
            # _my_windows.Decrement()
            self.Values = None
            self.LastButtonClicked = None
            return None, None
        return _BuildResults(self, False, self)

    def _start_autoclose_timer(self):
        duration = DEFAULT_AUTOCLOSE_TIME if self.AutoCloseDuration is None else self.AutoCloseDuration
        self.TKAfterID = self.TKroot.after(int(duration * 1000), self._AutoCloseAlarmCallback)

    def finalize(self):
        """
        Use this method to cause your layout to built into a real tkinter window.  In reality this method is like
        Read(timeout=0).  It doesn't block and uses your layout to create tkinter widgets to represent the elements.
        Lots of action!

        :return: Returns 'self' so that method "Chaining" can happen (read up about it as it's very cool!)
        :rtype:  (Window)
        """

        if self.TKrootDestroyed:
            return self
        self.finalize_in_progress = True

        self.Read(timeout=1)

        if self.AutoClose:
            self.auto_close_timer_needs_starting = True
        # add the window to the list of active windows
        Window._active_windows[self] = Window.hidden_master_root
        return self
        # OLD CODE FOLLOWS
        if not self.Shown:
            self._Show(non_blocking=True)
        try:
            rc = self.TKroot.update()
        except:
            self.TKrootDestroyed = True
            Window._DecrementOpenCount()
            print('** Finalize failed **')
            # _my_windows.Decrement()
            # return None, None
        return self

    def refresh(self):
        """
        Refreshes the window by calling tkroot.update().  Can sometimes get away with a refresh instead of a Read.
        Use this call when you want something to appear in your Window immediately (as soon as this function is called).
        If you change an element in a window, your change will not be visible until the next call to Window.read
        or a call to Window.refresh()

        :return: `self` so that method calls can be easily "chained"
        :rtype:  (Window)
        """

        if self.TKrootDestroyed:
            return self
        try:
            rc = self.TKroot.update()
        except:
            pass
        return self

    def fill(self, values_dict):
        """
        Fill in elements that are input fields with data based on a 'values dictionary'

        :param values_dict: pairs
        :type values_dict:  (Dict[Any, Any]) - {Element_key : value}
        :return:            returns self so can be chained with other methods
        :rtype:             (Window)
        """

        FillFormWithValues(self, values_dict)
        return self

    def _find_closest_key(self, search_key):
        if not isinstance(search_key, str):
            search_key = str(search_key)
        matches = difflib.get_close_matches(search_key, [str(k) for k in self.AllKeysDict.keys()])
        if not len(matches):
            return None
        for k in self.AllKeysDict.keys():
            if matches[0] == str(k):
                return k
        return matches[0] if len(matches) else None

    def FindElement(self, key, silent_on_error=False):
        """
        ** Warning ** This call will eventually be depricated. **

        It is suggested that you modify your code to use the recommended window[key] lookup or the PEP8 compliant window.find_element(key)

        For now, you'll only see a message printed and the call will continue to funcation as before.

        :param key:             Used with window.find_element and with return values to uniquely identify this element
        :type key:              str | int | tuple | object
        :param silent_on_error: If True do not display popup nor print warning of key errors
        :type silent_on_error:  (bool)
        :return:                Return value can be: the Element that matches the supplied key if found; an Error Element if silent_on_error is False; None if silent_on_error True;
        :rtype:                 Element | Error Element | None
        """

        warnings.warn('Use of FindElement is not recommended.\nEither switch to the recommended window[key] format\nor the PEP8 compliant find_element',
                      UserWarning)
        print('** Warning - FindElement should not be used to look up elements. window[key] or window.find_element are recommended. **')

        return self.find_element(key, silent_on_error=silent_on_error)

    def find_element(self, key, silent_on_error=False, supress_guessing=None, supress_raise=None):
        """
        Find element object associated with the provided key.
        THIS METHOD IS NO LONGER NEEDED to be called by the user

        You can perform the same operation by writing this statement:
        element = window[key]

        You can drop the entire "find_element" function name and use [ ] instead.

        However, if you wish to perform a lookup without error checking, and don't have error popups turned
        off globally, you'll need to make this call so that you can disable error checks on this call.

        find_element is typically used in combination with a call to element's update method (or any other element method!):
        window[key].update(new_value)

        Versus the "old way"
        window.FindElement(key).Update(new_value)

        This call can be abbreviated to any of these:
        find_element = FindElement == Element == Find
        With find_element being the PEP8 compliant call that should be used.

        Rememeber that this call will return None if no match is found which may cause your code to crash if not
        checked for.

        :param key:              Used with window.find_element and with return values to uniquely identify this element
        :type key:               str | int | tuple | object
        :param silent_on_error:  If True do not display popup nor print warning of key errors
        :type silent_on_error:   (bool)
        :param supress_guessing: Override for the global key guessing setting.
        :type supress_guessing:  (bool | None)
        :param supress_raise:    Override for the global setting that determines if a key error should raise an exception
        :type supress_raise:     (bool | None)
        :return:                 Return value can be: the Element that matches the supplied key if found; an Error Element if silent_on_error is False; None if silent_on_error True
        :rtype:                  Element | ErrorElement | None
        """

        key_error = False
        closest_key = None
        supress_guessing = supress_guessing if supress_guessing is not None else SUPPRESS_KEY_GUESSING
        supress_raise = supress_raise if supress_raise is not None else SUPPRESS_RAISE_KEY_ERRORS
        try:
            element = self.AllKeysDict[key]
        except KeyError:
            key_error = True
            closest_key = self._find_closest_key(key)
            if not silent_on_error:
                print('** Error looking up your element using the key: ', key, 'The closest matching key: ', closest_key)
                _error_popup_with_traceback('Key Error', 'Problem finding your key ' + str(key), 'Closest match = ' + str(closest_key), emoji=EMOJI_BASE64_KEY)
                element = ErrorElement(key=key)
            else:
                element = None
            if not supress_raise:
                raise KeyError(key)

        if key_error:
            if not supress_guessing and closest_key is not None:
                element = self.AllKeysDict[closest_key]

        return element

    Element = find_element  # Shortcut function
    Find = find_element  # Shortcut function, most likely not used by many people.
    Elem = find_element  # NEW for 2019!  More laziness... Another shortcut

    def find_element_with_focus(self):
        """
        Returns the Element that currently has focus as reported by tkinter. If no element is found None is returned!
        :return: An Element if one has been found with focus or None if no element found
        :rtype:  Element | None
        """
        element = _FindElementWithFocusInSubForm(self)
        return element


    def widget_to_element(self, widget):
        """
        Returns the element that matches a supplied tkinter widget.
        If no matching element is found, then None is returned.


        :return:    Element that uses the specified widget
        :rtype:     Element | None
        """
        if self.AllKeysDict is None or len(self.AllKeysDict) == 0:
            return None
        for key, element in self.AllKeysDict.items():
            if element.Widget == widget:
                return element
        return None


    def _BuildKeyDict(self):
        """
        Used internally only! Not user callable
        Builds a dictionary containing all elements with keys for this window.
        """
        dict = {}
        self.AllKeysDict = self._BuildKeyDictForWindow(self, self, dict)

    def _BuildKeyDictForWindow(self, top_window, window, key_dict):
        """
        Loop through all Rows and all Container Elements for this window and create the keys for all of them.
        Note that the calls are recursive as all pathes must be walked

        :param top_window: The highest level of the window
        :type top_window:  (Window)
        :param window:     The "sub-window" (container element) to be searched
        :type window:      Column | Frame | TabGroup | Pane | Tab
        :param key_dict:   The dictionary as it currently stands.... used as part of recursive call
        :type key_dict:
        :return:           (dict) Dictionary filled with all keys in the window
        :rtype:
        """
        for row_num, row in enumerate(window.Rows):
            for col_num, element in enumerate(row):
                if element.Type == ELEM_TYPE_COLUMN:
                    key_dict = self._BuildKeyDictForWindow(top_window, element, key_dict)
                if element.Type == ELEM_TYPE_FRAME:
                    key_dict = self._BuildKeyDictForWindow(top_window, element, key_dict)
                if element.Type == ELEM_TYPE_TAB_GROUP:
                    key_dict = self._BuildKeyDictForWindow(top_window, element, key_dict)
                if element.Type == ELEM_TYPE_PANE:
                    key_dict = self._BuildKeyDictForWindow(top_window, element, key_dict)
                if element.Type == ELEM_TYPE_TAB:
                    key_dict = self._BuildKeyDictForWindow(top_window, element, key_dict)
                if element.Key is None:  # if no key has been assigned.... create one for input elements
                    if element.Type == ELEM_TYPE_BUTTON:
                        element.Key = element.ButtonText
                    elif element.Type == ELEM_TYPE_TAB:
                        element.Key = element.Title
                    if element.Type in (ELEM_TYPE_MENUBAR, ELEM_TYPE_BUTTONMENU,
                                        ELEM_TYPE_INPUT_SLIDER, ELEM_TYPE_GRAPH, ELEM_TYPE_IMAGE,
                                        ELEM_TYPE_INPUT_CHECKBOX, ELEM_TYPE_INPUT_LISTBOX, ELEM_TYPE_INPUT_COMBO,
                                        ELEM_TYPE_INPUT_MULTILINE, ELEM_TYPE_INPUT_OPTION_MENU, ELEM_TYPE_INPUT_SPIN,
                                        ELEM_TYPE_INPUT_RADIO, ELEM_TYPE_INPUT_TEXT, ELEM_TYPE_PROGRESS_BAR,
                                        ELEM_TYPE_TABLE, ELEM_TYPE_TREE,
                                        ELEM_TYPE_TAB_GROUP, ELEM_TYPE_SEPARATOR):
                        element.Key = top_window.DictionaryKeyCounter
                        top_window.DictionaryKeyCounter += 1
                if element.Key is not None:
                    if element.Key in key_dict.keys():
                        if element.Type == ELEM_TYPE_BUTTON and WARN_DUPLICATE_BUTTON_KEY_ERRORS:  # for Buttons see if should complain
                            warnings.warn('*** Duplicate key found in your layout {} ***'.format(element.Key), UserWarning)
                            warnings.warn('*** Replaced new key with {} ***'.format(str(element.Key) + str(self.UniqueKeyCounter)))
                            if not SUPPRESS_ERROR_POPUPS:
                                _error_popup_with_traceback('Duplicate key found in your layout', 'Dupliate key: {}'.format(element.Key),
                                                            'Is being replaced with: {}'.format(str(element.Key) + str(self.UniqueKeyCounter)),
                                                            'The line of code above shows you which layout, but does not tell you exactly where the element was defined',
                                                            'The element type is {}'.format(element.Type))
                        element.Key = str(element.Key) + str(self.UniqueKeyCounter)
                        self.UniqueKeyCounter += 1
                    key_dict[element.Key] = element
        return key_dict

    def element_list(self):
        """
        Returns a list of all elements in the window

        :return: List of all elements in the window and container elements in the window
        :rtype:  List[Element]
        """
        return self._build_element_list()

    def _build_element_list(self):
        """
        Used internally only! Not user callable
        Builds a dictionary containing all elements with keys for this window.
        """
        elem_list = []
        elem_list = self._build_element_list_for_form(self, self, elem_list)
        return elem_list

    def _build_element_list_for_form(self, top_window, window, elem_list):
        """
        Loop through all Rows and all Container Elements for this window and create a list
        Note that the calls are recursive as all pathes must be walked

        :param top_window: The highest level of the window
        :type top_window:  (Window)
        :param window:     The "sub-window" (container element) to be searched
        :type window:      Column | Frame | TabGroup | Pane | Tab
        :param elem_list:  The element list as it currently stands.... used as part of recursive call
        :type elem_list:   ???
        :return:           List of all elements in this sub-window
        :rtype:            List[Element]
        """
        for row_num, row in enumerate(window.Rows):
            for col_num, element in enumerate(row):
                elem_list.append(element)
                if element.Type in (ELEM_TYPE_COLUMN, ELEM_TYPE_FRAME, ELEM_TYPE_TAB_GROUP, ELEM_TYPE_PANE, ELEM_TYPE_TAB):
                    elem_list = self._build_element_list_for_form(top_window, element, elem_list)
        return elem_list

    def save_to_disk(self, filename):
        """
        Saves the values contained in each of the input areas of the form. Basically saves what would be returned from a call to Read.  It takes these results and saves them to disk using pickle.
         Note that every element in your layout that is to be saved must have a key assigned to it.

        :param filename: Filename to save the values to in pickled form
        :type filename:  str
        """
        try:
            event, values = _BuildResults(self, False, self)
            remove_these = []
            for key in values:
                if self.Element(key).Type == ELEM_TYPE_BUTTON:
                    remove_these.append(key)
            for key in remove_these:
                del values[key]
            with open(filename, 'wb') as sf:
                pickle.dump(values, sf)
        except:
            print('*** Error saving Window contents to disk ***')

    def load_from_disk(self, filename):
        """
        Restore values from a previous call to SaveToDisk which saves the returned values dictionary in Pickle format

        :param filename: Pickle Filename to load
        :type filename:  (str)
        """
        try:
            with open(filename, 'rb') as df:
                self.Fill(pickle.load(df))
        except:
            print('*** Error loading form to disk ***')

    def get_screen_dimensions(self):
        """
        Get the screen dimensions.  NOTE - you must have a window already open for this to work (blame tkinter not me)

        :return: Tuple containing width and height of screen in pixels
        :rtype:  Tuple[None, None] | Tuple[width, height]
        """

        if self.TKrootDestroyed or self.TKroot is None:
            return Window.get_screen_size()
        screen_width = self.TKroot.winfo_screenwidth()  # get window info to move to middle of screen
        screen_height = self.TKroot.winfo_screenheight()
        return screen_width, screen_height

    def move(self, x, y):
        """
        Move the upper left corner of this window to the x,y coordinates provided
        :param x: x coordinate in pixels
        :type x:  (int)
        :param y: y coordinate in pixels
        :type y:  (int)
        """
        try:
            self.TKroot.geometry('+%s+%s' % (x, y))
            self.config_last_location = (int(x), (int(y)))

        except:
            pass


    def move_to_center(self):
        """
        Recenter your window after it's been moved or the size changed.

        This is a conveinence method. There are no tkinter calls involved, only pure PySimpleGUI API calls.
        """
        if not self._is_window_created('tried Window.move_to_center'):
            return
        screen_width, screen_height = self.get_screen_dimensions()
        win_width, win_height = self.size
        x, y = (screen_width - win_width)//2, (screen_height - win_height)//2
        self.move(x, y)



    def minimize(self):
        """
        Minimize this window to the task bar
        """
        if not self._is_window_created('tried Window.minimize'):
            return
        if self.use_custom_titlebar is True:
            self._custom_titlebar_minimize()
        else:
            self.TKroot.iconify()
        self.maximized = False


    def maximize(self):
        """
        Maximize the window. This is done differently on a windows system versus a linux or mac one.  For non-Windows
        the root attribute '-fullscreen' is set to True.  For Windows the "root" state is changed to "zoomed"
        The reason for the difference is the title bar is removed in some cases when using fullscreen option
        """

        if not self._is_window_created('tried Window.maximize'):
            return
        if not running_linux():
            self.TKroot.state('zoomed')
        else:
            self.TKroot.attributes('-fullscreen', True)
        # this method removes the titlebar too
        # self.TKroot.attributes('-fullscreen', True)
        self.maximized = True

    def normal(self):
        """
        Restore a window to a non-maximized state.  Does different things depending on platform.  See Maximize for more.
        """
        if not self._is_window_created('tried Window.normal'):
            return
        if self.use_custom_titlebar:
            self._custom_titlebar_restore()
        else:
            if self.TKroot.state() == 'iconic':
                self.TKroot.deiconify()
            else:
                if not running_linux():
                    self.TKroot.state('normal')
                else:
                    self.TKroot.attributes('-fullscreen', False)
            self.maximized = False


    def _StartMoveUsingControlKey(self, event):
        """
        Used by "Grab Anywhere" style windows. This function is bound to mouse-down. It marks the beginning of a drag.
        :param event: event information passed in by tkinter. Contains x,y position of mouse
        :type event:  (event)
        """
        self._start_move_save_offset(event)
        return


    def _StartMoveGrabAnywhere(self, event):


        """
        Used by "Grab Anywhere" style windows. This function is bound to mouse-down. It marks the beginning of a drag.
        :param event: event information passed in by tkinter. Contains x,y position of mouse
        :type event:  (event)
        """
        if (isinstance(event.widget, GRAB_ANYWHERE_IGNORE_THESE_WIDGETS) or event.widget in self._grab_anywhere_ignore_these_list) and event.widget not in self._grab_anywhere_include_these_list:
            # print('Found widget to ignore in grab anywhere...')
            return
        self._start_move_save_offset(event)

    def _StartMove(self, event):
        self._start_move_save_offset(event)
        return

    def _StopMove(self, event):
        """
        Used by "Grab Anywhere" style windows. This function is bound to mouse-up. It marks the ending of a drag.
        Sets the position of the window to this final x,y coordinates
        :param event: event information passed in by tkinter. Contains x,y position of mouse
        :type event:  (event)
        """
        return

    def _start_move_save_offset(self, event):
        self._mousex = event.x + event.widget.winfo_rootx()
        self._mousey = event.y + event.widget.winfo_rooty()
        geometry = self.TKroot.geometry()
        location = geometry[geometry.find('+') + 1:].split('+')
        self._startx = int(location[0])
        self._starty = int(location[1])
        self._mouse_offset_x = self._mousex - self._startx
        self._mouse_offset_y = self._mousey - self._starty
        # ------ Move All Windows code ------
        if Window._move_all_windows:
            # print('Moving all')
            for win in Window._active_windows:
                if win == self:
                    continue
                geometry = win.TKroot.geometry()
                location = geometry[geometry.find('+') + 1:].split('+')
                _startx = int(location[0])
                _starty = int(location[1])
                win._mouse_offset_x = event.x_root - _startx
                win._mouse_offset_y = event.y_root - _starty


    def _OnMotionUsingControlKey(self, event):
        self._OnMotion(event)


    def _OnMotionGrabAnywhere(self, event):

        """
        Used by "Grab Anywhere" style windows. This function is bound to mouse motion. It actually moves the window
        :param event: event information passed in by tkinter. Contains x,y position of mouse
        :type event:  (event)
        """
        if (isinstance(event.widget, GRAB_ANYWHERE_IGNORE_THESE_WIDGETS) or event.widget in self._grab_anywhere_ignore_these_list) and event.widget not in self._grab_anywhere_include_these_list:
            # print('Found widget to ignore in grab anywhere...')
            return

        self._OnMotion(event)


    def _OnMotion(self, event):

        self.TKroot.geometry(f'+{event.x_root-self._mouse_offset_x}+{event.y_root-self._mouse_offset_y}')
        # print(f"+{event.x_root}+{event.y_root}")
        # ------ Move All Windows code ------
        try:
            if Window._move_all_windows:
                for win in Window._active_windows:
                    if win == self:
                        continue
                    win.TKroot.geometry(f'+{event.x_root-win._mouse_offset_x}+{event.y_root-win._mouse_offset_y}')
        except Exception as e:
            print('on motion error', e)

    def _focus_callback(self, event):
        print('Focus event = {} window = {}'.format(event, self.Title))

    def _config_callback(self, event):
        """
        Called when a config event happens for the window

        :param event:            From tkinter and is not used
        :type event:             Any
        """
        self.LastButtonClicked = WINDOW_CONFIG_EVENT
        self.FormRemainedOpen = True
        self.user_bind_event = event
        _exit_mainloop(self)


    def _move_callback(self, event):
        """
        Called when a control + arrow key is pressed.
        This is a built-in window positioning key sequence

        :param event:            From tkinter and is not used
        :type event:             Any
        """
        if not self._is_window_created('Tried to move window using arrow keys'):
            return
        x,y = self.current_location()
        if event.keysym == 'Up':
            self.move(x, y-1)
        elif event.keysym == 'Down':
            self.move(x, y+1)
        elif event.keysym == 'Left':
            self.move(x-1, y)
        elif event.keysym == 'Right':
            self.move(x+1, y)

    """
    def _config_callback(self, event):
        new_x = event.x
        new_y = event.y


        if self.not_completed_initial_movement:
            if self.starting_window_position != (new_x, new_y):
                return
            self.not_completed_initial_movement = False
            return

        if not self.saw_00:
            if new_x == 0 and new_y == 0:
                self.saw_00 = True

        # self.config_count += 1
        # if self.config_count < 40:
        #     return

        print('Move LOGIC')

        if self.config_last_size != (event.width, event.height):
            self.config_last_size = (event.width, event.height)

        if self.config_last_location[0] != new_x or self.config_last_location[1] != new_y:
            if self.config_last_location == (None, None):
                self.config_last_location = (new_x, new_y)
                return

        deltax = self.config_last_location[0] - event.x
        deltay = self.config_last_location[1] - event.y
        if deltax == 0 and deltay == 0:
            print('not moving so returning')
            return
        if Window._move_all_windows:
            print('checking all windows')
            for window in Window._active_windows:
                if window == self:
                    continue
                x = window.TKroot.winfo_x() + deltax
                y = window.TKroot.winfo_y() + deltay
                # window.TKroot.geometry("+%s+%s" % (x, y))  # this is what really moves the window
                # window.config_last_location = (x,y)
    """

    def _KeyboardCallback(self, event):
        """
        Window keyboard callback. Called by tkinter.  Will kick user out of the tkinter event loop. Should only be
        called if user has requested window level keyboard events

        :param event: object provided by tkinter that contains the key information
        :type event:  (event)
        """
        self.LastButtonClicked = None
        self.FormRemainedOpen = True
        if event.char != '':
            self.LastKeyboardEvent = event.char
        else:
            self.LastKeyboardEvent = str(event.keysym) + ':' + str(event.keycode)
        # if not self.NonBlocking:
        #     _BuildResults(self, False, self)
        _exit_mainloop(self)

    def _MouseWheelCallback(self, event):
        """
        Called by tkinter when a mouse wheel event has happened. Only called if keyboard events for the window
        have been enabled

        :param event: object sent in by tkinter that has the wheel direction
        :type event:  (event)
        """
        self.LastButtonClicked = None
        self.FormRemainedOpen = True
        self.LastKeyboardEvent = 'MouseWheel:Down' if event.delta < 0 or event.num == 5 else 'MouseWheel:Up'
        # if not self.NonBlocking:
        #     _BuildResults(self, False, self)
        _exit_mainloop(self)

    def _Close(self, without_event=False):
        """
        The internal close call that does the real work of building. This method basically sets up for closing
        but doesn't destroy the window like the User's version of Close does

        :parm without_event: if True, then do not cause an event to be generated, "silently" close the window
        :type without_event: (bool)
        """

        try:
            self.TKroot.update()
        except:
            pass

        if not self.NonBlocking or not without_event:
            _BuildResults(self, False, self)
        if self.TKrootDestroyed:
            return
        self.TKrootDestroyed = True
        self.RootNeedsDestroying = True
        return

    def close(self):
        """
        Closes window.  Users can safely call even if window has been destroyed.   Should always call when done with
        a window so that resources are properly freed up within your thread.
        """

        try:
            del Window._active_windows[self]  # will only be in the list if window was explicitly finalized
        except:
            pass

        try:
            self.TKroot.update()  # On Linux must call update if the user closed with X or else won't actually close the window
        except:
            pass

        self._restore_stdout()
        self._restore_stderr()

        _TimerPeriodic.stop_all_timers_for_window(self)

        if self.TKrootDestroyed:
            return
        try:
            self.TKroot.destroy()
            self.TKroot.update()
            Window._DecrementOpenCount()
        except:
            pass
        # if down to 1 window, try and destroy the hidden window, if there is one
        # if Window.NumOpenWindows == 1:
        #     try:
        #         Window.hidden_master_root.destroy()
        #         Window.NumOpenWindows = 0  # if no hidden window, then this won't execute
        #     except:
        #         pass
        self.TKrootDestroyed = True

        # Free up anything that was held in the layout and the root variables
        self.Rows = None
        self.TKroot = None




    def is_closed(self, quick_check=None):
        """
        Returns True is the window is maybe closed.  Can be difficult to tell sometimes
        NOTE - the call to TKroot.update was taking over 500 ms sometimes so added a flag to bypass the lengthy call.
        :param quick_quick: If True, then don't use the root.update call, only check the flags
        :type quick_check:  bool
        :return:            True if the window was closed or destroyed
        :rtype:             (bool)
        """

        if self.TKrootDestroyed or self.TKroot is None:
            return True

        # if performing a quick check only, then skip calling tkinter for performance reasons
        if quick_check is True:
            return False

        # see if can do an update... if not, then it's been destroyed
        try:
            rc = self.TKroot.update()
        except:
            return True
        return False


    # IT FINALLY WORKED! 29-Oct-2018 was the first time this damned thing got called
    def _OnClosingCallback(self):
        """
        Internally used method ONLY. Not sure callable.  tkinter calls this when the window is closed by clicking X
        """
        # global _my_windows
        # print('Got closing callback', self.DisableClose)
        if self.DisableClose:
            return
        if self.CurrentlyRunningMainloop:  # quit if this is the current mainloop, otherwise don't quit!
            _exit_mainloop(self)
            if self.close_destroys_window:
                self.TKroot.destroy()  # destroy this window
                self.TKrootDestroyed = True
                self.XFound = True
            else:
                self.LastButtonClicked = WINDOW_CLOSE_ATTEMPTED_EVENT
        elif Window._root_running_mainloop == Window.hidden_master_root:
            _exit_mainloop(self)
        else:
            if self.close_destroys_window:
                self.TKroot.destroy()  # destroy this window
                self.XFound = True
            else:
                self.LastButtonClicked = WINDOW_CLOSE_ATTEMPTED_EVENT
        if self.close_destroys_window:
            self.RootNeedsDestroying = True
        self._restore_stdout()
        self._restore_stderr()

    def disable(self):
        """
        Disables window from taking any input from the user
        """
        if not self._is_window_created('tried Window.disable'):
            return
        self.TKroot.attributes('-disabled', 1)
        # self.TKroot.grab_set_global()

    def enable(self):
        """
        Re-enables window to take user input after having it be Disabled previously
        """
        if not self._is_window_created('tried Window.enable'):
            return
        self.TKroot.attributes('-disabled', 0)
        # self.TKroot.grab_release()

    def hide(self):
        """
        Hides the window from the screen and the task bar
        """
        if not self._is_window_created('tried Window.hide'):
            return
        self._Hidden = True
        self.TKroot.withdraw()

    def un_hide(self):
        """
        Used to bring back a window that was previously hidden using the Hide method
        """
        if not self._is_window_created('tried Window.un_hide'):
            return
        if self._Hidden:
            self.TKroot.deiconify()
            self._Hidden = False

    def is_hidden(self):
        """
            Returns True if the window is currently hidden
        :return:    Returns True if the window is currently hidden
        :rtype:     bool
        """
        return self._Hidden

    def disappear(self):
        """
        Causes a window to "disappear" from the screen, but remain on the taskbar. It does this by turning the alpha
        channel to 0.  NOTE that on some platforms alpha is not supported. The window will remain showing on these
        platforms.  The Raspberry Pi for example does not have an alpha setting
        """
        if not self._is_window_created('tried Window.disappear'):
            return
        self.TKroot.attributes('-alpha', 0)

    def reappear(self):
        """
        Causes a window previously made to "Disappear" (using that method). Does this by restoring the alpha channel
        """
        if not self._is_window_created('tried Window.reappear'):
            return
        self.TKroot.attributes('-alpha', 255)

    def set_alpha(self, alpha):
        """
        Sets the Alpha Channel for a window.  Values are between 0 and 1 where 0 is completely transparent

        :param alpha: 0 to 1. 0 is completely transparent.  1 is completely visible and solid (can't see through)
        :type alpha:  (float)
        """
        if not self._is_window_created('tried Window.set_alpha'):
            return
        self._AlphaChannel = alpha
        self.TKroot.attributes('-alpha', alpha)

    @property
    def alpha_channel(self):
        """
        A property that changes the current alpha channel value (internal value)
        :return: the current alpha channel setting according to self, not read directly from tkinter
        :rtype:  (float)
        """
        return self._AlphaChannel

    @alpha_channel.setter
    def alpha_channel(self, alpha):
        """
        The setter method for this "property".
        Planning on depricating so that a Set call is always used by users. This is more in line with the SDK
        :param alpha: 0 to 1. 0 is completely transparent.  1 is completely visible and solid (can't see through)
        :type alpha:  (float)
        """
        if not self._is_window_created('tried Window.alpha_channel'):
            return
        self._AlphaChannel = alpha
        self.TKroot.attributes('-alpha', alpha)

    def bring_to_front(self):
        """
        Brings this window to the top of all other windows (perhaps may not be brought before a window made to "stay
        on top")
        """
        if not self._is_window_created('tried Window.bring_to_front'):
            return
        if running_windows():
            try:
                self.TKroot.wm_attributes('-topmost', 0)
                self.TKroot.wm_attributes('-topmost', 1)
                if not self.KeepOnTop:
                    self.TKroot.wm_attributes('-topmost', 0)
            except Exception as e:
                warnings.warn('Problem in Window.bring_to_front' + str(e), UserWarning)
        else:
            try:
                self.TKroot.lift()
            except:
                pass

    def send_to_back(self):
        """
        Pushes this window to the bottom of the stack of windows. It is the opposite of BringToFront
        """
        if not self._is_window_created('tried Window.send_to_back'):
            return
        try:
            self.TKroot.lower()
        except:
            pass


    def keep_on_top_set(self):
        """
        Sets keep_on_top after a window has been created.  Effect is the same
        as if the window was created with this set.  The Window is also brought
        to the front
        """
        if not self._is_window_created('tried Window.keep_on_top_set'):
            return
        self.KeepOnTop = True
        self.bring_to_front()
        try:
            self.TKroot.wm_attributes('-topmost', 1)
        except Exception as e:
            warnings.warn('Problem in Window.keep_on_top_set trying to set wm_attributes topmost' + str(e), UserWarning)


    def keep_on_top_clear(self):
        """
        Clears keep_on_top after a window has been created.  Effect is the same
        as if the window was created with this set.
        """
        if not self._is_window_created('tried Window.keep_on_top_clear'):
            return
        self.KeepOnTop = False
        try:
            self.TKroot.wm_attributes('-topmost', 0)
        except Exception as e:
            warnings.warn('Problem in Window.keep_on_top_clear trying to clear wm_attributes topmost' + str(e), UserWarning)



    def current_location(self, more_accurate=False, without_titlebar=False):
        """
        Get the current location of the window's top left corner.
        Sometimes, depending on the environment, the value returned does not include the titlebar,etc
        A new option, more_accurate, can be used to get the theoretical upper leftmost corner of the window.
        The titlebar and menubar are crated by the OS. It gets really confusing when running in a webpage (repl, trinket)
        Thus, the values can appear top be "off" due to the sometimes unpredictable way the location is calculated.
        If without_titlebar is set then the location of the root x,y is used which should not include the titlebar but
            may be OS dependent.

        :param more_accurate:    If True, will use the window's geometry to get the topmost location with titlebar, menubar taken into account
        :type more_accurate:     (bool)
        :param without_titlebar: If True, return location of top left of main window area without the titlebar (may be OS dependent?)
        :type without_titlebar:  (bool)
        :return:                 The x and y location in tuple form (x,y)
        :rtype:                  Tuple[(int | None), (int | None)]
        """


        if not self._is_window_created('tried Window.current_location'):
            return (None, None)
        try:
            if without_titlebar is True:
                x, y = self.TKroot.winfo_rootx(), self.TKroot.winfo_rooty()
            elif more_accurate:
                geometry = self.TKroot.geometry()
                location = geometry[geometry.find('+') + 1:].split('+')
                x, y = int(location[0]), int(location[1])
            else:
                x, y =  int(self.TKroot.winfo_x()), int(self.TKroot.winfo_y())
        except Exception as e:
            warnings.warn('Error in Window.current_location. Trouble getting x,y location\n' + str(e), UserWarning)
            x, y = (None, None)
        return (x,y)


    def current_size_accurate(self):
        """
        Get the current location of the window based on tkinter's geometry setting

        :return:              The x and y size in tuple form (x,y)
        :rtype:               Tuple[(int | None), (int | None)]
        """

        if not self._is_window_created('tried Window.current_location'):
            return (None, None)
        try:
            geometry = self.TKroot.geometry()
            geometry_tuple = geometry.split('+')
            window_size = geometry_tuple[0].split('x')
            x, y = int(window_size[0]), int(window_size[1])
        except Exception as e:
            warnings.warn('Error in Window.current_size_accurate. Trouble getting x,y size\n{} {}'.format(geometry, geometry_tuple) + str(e), UserWarning)
            x, y = (None, None)
        return (x,y)

    @property
    def size(self):
        """
        Return the current size of the window in pixels

        :return: (width, height) of the window
        :rtype:  Tuple[(int), (int)] or Tuple[None, None]
        """
        if not self._is_window_created('Tried to use Window.size property'):
            return (None, None)
        win_width = self.TKroot.winfo_width()
        win_height = self.TKroot.winfo_height()
        return win_width, win_height

    @size.setter
    def size(self, size):
        """
        Changes the size of the window, if possible

        :param size: (width, height) of the desired window size
        :type size:  (int, int)
        """
        try:
            self.TKroot.geometry('%sx%s' % (size[0], size[1]))
            self.TKroot.update_idletasks()
        except:
            pass


    def set_size(self, size):
        """
        Changes the size of the window, if possible. You can also use the Window.size prooerty
        to set/get the size.

        :param size: (width, height) of the desired window size
        :type size:  (int, int)
        """
        if not self._is_window_created('Tried to change the size of the window prior to creation.'):
            return
        try:
            self.TKroot.geometry('%sx%s' % (size[0], size[1]))
            self.TKroot.update_idletasks()
        except:
            pass



    def set_min_size(self, size):
        """
        Changes the minimum size of the window. Note Window must be read or finalized first.

        :param size: (width, height) tuple (int, int) of the desired window size in pixels
        :type size:  (int, int)
        """
        if not self._is_window_created('tried Window.set_min_size'):
            return
        self.TKroot.minsize(size[0], size[1])
        self.TKroot.update_idletasks()


    def set_resizable(self, x_axis_enable, y_axis_enable):
        """
        Changes if a window can be resized in either the X or the Y direction.
        Note Window must be read or finalized first.

        :param x_axis_enable: If True, the window can be changed in the X-axis direction. If False, it cannot
        :type x_axis_enable: (bool)
        :param y_axis_enable: If True, the window can be changed in the Y-axis direction. If False, it cannot
        :type y_axis_enable: (bool)
        """

        if not self._is_window_created('tried Window.set_resixable'):
            return
        try:
            self.TKroot.resizable(x_axis_enable, y_axis_enable)
        except Exception as e:
            _error_popup_with_traceback('Window.set_resizable - tkinter reported error', e)

    def visibility_changed(self):
        """
        When making an element in a column or someplace that has a scrollbar, then you'll want to call this function
        prior to the column's contents_changed() method.
        """
        self.refresh()

    def set_transparent_color(self, color):
        """
        Set the color that will be transparent in your window. Areas with this color will be SEE THROUGH.

        :param color: Color string that defines the transparent color
        :type color:  (str)
        """
        if not self._is_window_created('tried Window.set_transparent_color'):
            return
        try:
            self.TKroot.attributes('-transparentcolor', color)
            self.TransparentColor = color
        except:
            print('Transparent color not supported on this platform (windows only)')

    def mouse_location(self):
        """
        Return the (x,y) location of the mouse relative to the entire screen.  It's the same location that
        you would use to create a window, popup, etc.

        :return:    The location of the mouse pointer
        :rtype:     (int, int)
        """
        if not self._is_window_created('tried Window.mouse_location'):
            return (0,0)

        return (self.TKroot.winfo_pointerx(), self.TKroot.winfo_pointery())

    def grab_any_where_on(self):
        """
        Turns on Grab Anywhere functionality AFTER a window has been created.  Don't try on a window that's not yet
        been Finalized or Read.
        """
        if not self._is_window_created('tried Window.grab_any_where_on'):
            return
        self.TKroot.bind('<ButtonPress-1>', self._StartMoveGrabAnywhere)
        self.TKroot.bind('<ButtonRelease-1>', self._StopMove)
        self.TKroot.bind('<B1-Motion>', self._OnMotionGrabAnywhere)

    def grab_any_where_off(self):
        """
        Turns off Grab Anywhere functionality AFTER a window has been created.  Don't try on a window that's not yet
        been Finalized or Read.
        """
        if not self._is_window_created('tried Window.grab_any_where_off'):
            return
        self.TKroot.unbind('<ButtonPress-1>')
        self.TKroot.unbind('<ButtonRelease-1>')
        self.TKroot.unbind('<B1-Motion>')

    def _user_bind_callback(self, bind_string, event, propagate=True):
        """
        Used when user binds a tkinter event directly to an element

        :param bind_string: The event that was bound so can lookup the key modifier
        :type bind_string:  (str)
        :param event:       Event data passed in by tkinter (not used)
        :type event:
        :param propagate:   If True then tkinter will be told to propagate the event
        :type propagate:    (bool)
        """
        # print('bind callback', bind_string, event)
        key = self.user_bind_dict.get(bind_string, '')
        self.user_bind_event = event
        if key is not None:
            self.LastButtonClicked = key
        else:
            self.LastButtonClicked = bind_string
        self.FormRemainedOpen = True
        # if self.CurrentlyRunningMainloop:
        #     self.TKroot.quit()
        _exit_mainloop(self)
        return 'break' if propagate is not True else None


    def bind(self, bind_string, key, propagate=True):
        """
        Used to add tkinter events to a Window.
        The tkinter specific data is in the Window's member variable user_bind_event
        :param bind_string: The string tkinter expected in its bind function
        :type bind_string:  (str)
        :param key:         The event that will be generated when the tkinter event occurs
        :type key:          str | int | tuple | object
        :param propagate:   If True then tkinter will be told to propagate the event
        :type propagate:    (bool)
        """
        if not self._is_window_created('tried Window.bind'):
            return
        try:
            self.TKroot.bind(bind_string, lambda evt: self._user_bind_callback(bind_string, evt, propagate))
        except Exception as e:
            self.TKroot.unbind_all(bind_string)
            return
            # _error_popup_with_traceback('Window.bind error', e)
        self.user_bind_dict[bind_string] = key


    def unbind(self, bind_string):
        """
        Used to remove tkinter events to a Window.
        This implementation removes ALL of the binds of the bind_string from the Window.  If there
        are multiple binds for the Window itself, they will all be removed.  This can be extended later if there
        is a need.
        :param bind_string: The string tkinter expected in its bind function
        :type bind_string:  (str)
        """
        if not self._is_window_created('tried Window.unbind'):
            return
        self.TKroot.unbind(bind_string)



    def _callback_main_debugger_window_create_keystroke(self, event):
        """
        Called when user presses the key that creates the main debugger window
        March 2022 - now causes the user reads to return timeout events automatically
        :param event: (event) not used. Passed in event info
        :type event:
        """
        Window._main_debug_window_build_needed = True
        # exit the event loop in a way that resembles a timeout occurring
        self.LastButtonClicked = self.TimeoutKey
        self.FormRemainedOpen = True
        self.TKroot.quit()  # kick the users out of the mainloop

    def _callback_popout_window_create_keystroke(self, event):
        """
        Called when user presses the key that creates the floating debugger window
        March 2022 - now causes the user reads to return timeout events automatically
        :param event: (event) not used. Passed in event info
        :type event:
        """
        Window._floating_debug_window_build_needed = True
        # exit the event loop in a way that resembles a timeout occurring
        self.LastButtonClicked = self.TimeoutKey
        self.FormRemainedOpen = True
        self.TKroot.quit()  # kick the users out of the mainloop

    def enable_debugger(self):
        """
        Enables the internal debugger. By default, the debugger IS enabled
        """
        if not self._is_window_created('tried Window.enable_debugger'):
            return
        self.TKroot.bind('<Cancel>', self._callback_main_debugger_window_create_keystroke)
        self.TKroot.bind('<Pause>', self._callback_popout_window_create_keystroke)
        self.DebuggerEnabled = True

    def disable_debugger(self):
        """
        Disable the internal debugger. By default the debugger is ENABLED
        """
        if not self._is_window_created('tried Window.disable_debugger'):
            return
        self.TKroot.unbind('<Cancel>')
        self.TKroot.unbind('<Pause>')
        self.DebuggerEnabled = False

    def set_title(self, title):
        """
        Change the title of the window

        :param title: The string to set the title to
        :type title:  (str)
        """
        if not self._is_window_created('tried Window.set_title'):
            return
        if self._has_custom_titlebar:
            try:    # just in case something goes badly, don't crash
                self.find_element(TITLEBAR_TEXT_KEY).update(title)
            except:
                pass
        # even with custom titlebar, set the main window's title too so it'll match when minimized
        self.TKroot.wm_title(str(title))

    def make_modal(self):
        """
        Makes a window into a "Modal Window"
        This means user will not be able to interact with other windows until this one is closed

        NOTE - Sorry Mac users - you can't have modal windows.... lobby your tkinter Mac devs
        """
        if not self._is_window_created('tried Window.make_modal'):
            return

        if running_mac() and ENABLE_MAC_MODAL_DISABLE_PATCH:
            return

        # if modal windows have been disabled globally
        if not DEFAULT_MODAL_WINDOWS_ENABLED and not DEFAULT_MODAL_WINDOWS_FORCED:
        # if not DEFAULT_MODAL_WINDOWS_ENABLED:
            return

        try:
            self.TKroot.transient()
            self.TKroot.grab_set()
            self.TKroot.focus_force()
        except Exception as e:
            print('Exception trying to make modal', e)

    def force_focus(self):
        """
        Forces this window to take focus
        """
        if not self._is_window_created('tried Window.force_focus'):
            return
        self.TKroot.focus_force()

    def was_closed(self):
        """
        Returns True if the window was closed

        :return: True if the window is closed
        :rtype:  bool
        """
        return self.TKrootDestroyed

    def set_cursor(self, cursor):
        """
        Sets the cursor for the window.
        If you do not want any mouse pointer, then use the string "none"

        :param cursor: The tkinter cursor name
        :type cursor:  (str)
        """

        if not self._is_window_created('tried Window.set_cursor'):
            return
        try:
            self.TKroot.config(cursor=cursor)
        except Exception as e:
            print('Warning bad cursor specified ', cursor)
            print(e)

    def ding(self, display_number=0):
        """
        Make a "bell" sound. A capability provided by tkinter.  Your window needs to be finalized prior to calling.
        Ring a display's bell is the tkinter description of the call.
        :param display_number: Passed to tkinter's bell method as parameter "displayof".
        :type display_number:  int
        """
        if not self._is_window_created('tried Window.ding'):
            return
        try:
            self.TKroot.bell(display_number)
        except Exception as e:
            if not SUPPRESS_ERROR_POPUPS:
                _error_popup_with_traceback('Window.ding() - tkinter reported error from bell() call', e)

    def _window_tkvar_changed_callback(self, *args):
        """
        Internal callback function for when the thread

        :param event: Information from tkinter about the callback
        :type event:

        """
        # print('Thread callback info', threading.current_thread())
        # print(event)
        # trace_details = traceback.format_stack()
        # print(''.join(trace_details))
        # self.thread_lock.acquire()
        # if self.thread_timer:
        # self.TKroot.after_cancel(id=self.thread_timer)
        # self.thread_timer = None
        # self.thread_lock.release()

        if self._queued_thread_event_available():
            self.FormRemainedOpen = True
            _exit_mainloop(self)

    def _create_thread_queue(self):
        """
        Creates the queue used by threads to communicate with this window
        """

        if self.thread_queue is None:
            self.thread_queue = queue.Queue()

        if self.thread_lock is None:
            self.thread_lock = threading.Lock()

        if self.thread_strvar is None:
            self.thread_strvar = tk.StringVar()
            self.thread_strvar.trace('w', self._window_tkvar_changed_callback)

    def write_event_value(self, key, value):
        """
        Adds a key & value tuple to the queue that is used by threads to communicate with the window

        :param key:   The key that will be returned as the event when reading the window
        :type key:    Any
        :param value: The value that will be in the values dictionary
        :type value:  Any
        """

        if self.thread_queue is None:
            print('*** Warning Window.write_event_value - no thread queue found ***')
            return
        # self.thread_lock.acquire()  # first lock the critical section
        self.thread_queue.put(item=(key, value))
        self.TKroot.tk.willdispatch()  # brilliant bit of code provided by Giuliano who I owe a million thank yous!
        self.thread_strvar.set('new item')

        # self.thread_queue.put(item=(key, value))
        # self.thread_strvar.set('new item')
        # March 28 2021 - finally found a solution!  It needs a little more work and a lock
        # if no timer is running, then one should be started
        # if self.thread_timer is None:
        #     print('Starting a timer')
        #     self.thread_timer = self.TKroot.after(1, self._window_tkvar_changed_callback)
        # self.thread_lock.release()

    def _queued_thread_event_read(self):
        if self.thread_queue is None:
            return None

        try:  # see if something has been posted to Queue
            message = self.thread_queue.get_nowait()
        except queue.Empty:  # get_nowait() will get exception when Queue is empty
            return None

        return message

    def _queued_thread_event_available(self):

        if self.thread_queue is None:
            return False
        # self.thread_lock.acquire()
        qsize = self.thread_queue.qsize()
        if qsize == 0:
            self.thread_timer = None
        # self.thread_lock.release()
        return qsize != 0



    def _RightClickMenuCallback(self, event):
        """
        When a right click menu is specified for an entire window, then this callback catches right clicks
        that happen to the window itself, when there are no elements that are in that area.

        The only portion that is not currently covered correctly is the row frame itself.  There will still
        be parts of the window, at the moment, that don't respond to a right click.  It's getting there, bit
        by bit.

        Callback function that's called when a right click happens. Shows right click menu as result.

        :param event: information provided by tkinter about the event including x,y location of click
        :type event:
        """
        # if there are widgets under the mouse, then see if it's the root only.  If not, then let the widget (element) show their menu instead
        x, y = self.TKroot.winfo_pointerxy()
        widget = self.TKroot.winfo_containing(x, y)
        if widget != self.TKroot:
            return
        self.TKRightClickMenu.tk_popup(event.x_root, event.y_root, 0)
        self.TKRightClickMenu.grab_release()


    def save_window_screenshot_to_disk(self, filename=None):
        """
        Saves an image of the PySimpleGUI window provided into the filename provided

        :param filename:        Optional filename to save screenshot to. If not included, the User Settinds are used to get the filename
        :return:                A PIL ImageGrab object that can be saved or manipulated
        :rtype:                 (PIL.ImageGrab | None)
        """
        global pil_import_attempted, pil_imported, PIL, ImageGrab, Image

        if not pil_import_attempted:
            try:
                import PIL as PIL
                from PIL import ImageGrab
                from PIL import Image
                pil_imported = True
                pil_import_attempted = True
            except:
                pil_imported = False
                pil_import_attempted = True
                print('FAILED TO IMPORT PIL!')
                return None
        try:
            # Get location of window to save
            pos = self.current_location()
            # Add a little to the X direction if window has a titlebar
            if not self.NoTitleBar:
                pos = (pos[0]+7, pos[1])
            # Get size of wiondow
            size = self.current_size_accurate()
            # Get size of the titlebar
            titlebar_height = self.TKroot.winfo_rooty() - self.TKroot.winfo_y()
            # Add titlebar to size of window so that titlebar and window will be saved
            size = (size[0], size[1] + titlebar_height)
            if not self.NoTitleBar:
                size_adjustment = (2,1)
            else:
                size_adjustment = (0,0)
            # Make the "Bounding rectangle" used by PLK to do the screen grap "operation
            rect = (pos[0], pos[1], pos[0] + size[0]+size_adjustment[0], pos[1] + size[1]+size_adjustment[1])
            # Grab the image
            grab = ImageGrab.grab(bbox=rect)
            # Save the grabbed image to disk
        except Exception as e:
            # print(e)
            popup_error_with_traceback('Screen capture failure', 'Error happened while trying to save screencapture', e)

            return None
        # return grab
        if filename is None:
            folder = pysimplegui_user_settings.get('-screenshots folder-', '')
            filename = pysimplegui_user_settings.get('-screenshots filename-', '')
            full_filename = os.path.join(folder, filename)
        else:
            full_filename = filename
        if full_filename:
            try:
                grab.save(full_filename)
            except Exception as e:
                popup_error_with_traceback('Screen capture failure', 'Error happened while trying to save screencapture', e)
        else:
            popup_error_with_traceback('Screen capture failure', 'You have attempted a screen capture but have not set up a good filename to save to')
        return grab


    def perform_long_operation(self, func, end_key=None):
        """
        Call your function that will take a long time to execute.  When it's complete, send an event
        specified by the end_key.

        Starts a thread on your behalf.

        This is a way for you to "ease into" threading without learning the details of threading.
        Your function will run, and when it returns 2 things will happen:
        1. The value you provide for end_key will be returned to you when you call window.read()
        2. If your function returns a value, then the value returned will also be included in your windows.read call in the values dictionary

        importANT - This method uses THREADS... this means you CANNOT make any FreeSimpleGUI calls from
        the function you provide with the exception of one function, Window.write_event_value.

        :param func:    A lambda or a function name with no parms
        :type func:     Any
        :param end_key: Optional key that will be generated when the function returns
        :type end_key:  (Any | None)
        :return:        The id of the thread
        :rtype:         threading.Thread
        """

        thread = threading.Thread(target=_long_func_thread, args=(self, end_key, func), daemon=True)
        thread.start()
        return thread

    @property
    def key_dict(self):
        """
        Returns a dictionary with all keys and their corresponding elements
        { key : Element }
        :return: Dictionary of keys and elements
        :rtype:  Dict[Any, Element]
        """
        return self.AllKeysDict


    def key_is_good(self, key):
        """
        Checks to see if this is a good key for this window
        If there's an element with the key provided, then True is returned
        :param key:     The key to check
        :type key:      str | int | tuple | object
        :return:        True if key is an element in this window
        :rtype:         bool
        """
        if key in self.key_dict:
            return True
        return False

    def get_scaling(self):
        """
        Returns the current scaling value set for this window

        :return:    Scaling according to tkinter. Returns DEFAULT_SCALING if error
        :rtype:     float
        """

        if not self._is_window_created('Tried Window.set_scaling'):
            return DEFAULT_SCALING
        try:
            scaling = self.TKroot.tk.call('tk', 'scaling')
        except Exception as e:
            if not SUPPRESS_ERROR_POPUPS:
                _error_popup_with_traceback('Window.get_scaling() - tkinter reported error', e)
            scaling = DEFAULT_SCALING

        return scaling


    def _custom_titlebar_restore_callback(self, event):
        self._custom_titlebar_restore()


    def _custom_titlebar_restore(self):
        if running_linux():
            # if self._skip_first_restore_callback:
            #     self._skip_first_restore_callback = False
            #     return
            self.TKroot.unbind('<Button-1>')
            self.TKroot.deiconify()

            # self.ParentForm.TKroot.wm_overrideredirect(True)
            self.TKroot.wm_attributes('-type', 'dock')

        else:
            self.TKroot.unbind('<Expose>')
            self.TKroot.wm_overrideredirect(True)
        if self.TKroot.state() == 'iconic':
            self.TKroot.deiconify()
        else:
            if not running_linux():
                self.TKroot.state('normal')
            else:
                self.TKroot.attributes('-fullscreen', False)
        self.maximized = False


    def _custom_titlebar_minimize(self):
        if running_linux():
            self.TKroot.wm_attributes('-type', 'normal')
            # self.ParentForm.TKroot.state('icon')
            # return
            # self.ParentForm.maximize()
            self.TKroot.wm_overrideredirect(False)
            # self.ParentForm.minimize()
            # self.ParentForm.TKroot.wm_overrideredirect(False)
            self.TKroot.iconify()
            # self._skip_first_restore_callback = True
            self.TKroot.bind('<Button-1>', self._custom_titlebar_restore_callback)
        else:
            self.TKroot.wm_overrideredirect(False)
            self.TKroot.iconify()
            self.TKroot.bind('<Expose>', self._custom_titlebar_restore_callback)


    def _custom_titlebar_callback(self, key):
        """
        One of the Custom Titlbar buttons was clicked
        :param key:
        :return:
        """
        if key == TITLEBAR_MINIMIZE_KEY:
            if not self.DisableMinimize:
                self._custom_titlebar_minimize()
        elif key == TITLEBAR_MAXIMIZE_KEY:
            if self.Resizable:
                if self.maximized:
                    self.normal()
                else:
                    self.maximize()
        elif key == TITLEBAR_CLOSE_KEY:
            if not self.DisableClose:
                self._OnClosingCallback()


    def timer_start(self, frequency_ms, key=EVENT_TIMER, repeating=True):
        """
        Starts a timer that gnerates Timer Events.  The default is to repeat the timer events until timer is stopped.
        You can provide your own key or a default key will be used.  The default key is defined
        with the constants EVENT_TIMER or TIMER_KEY.  They both equal the same value.
        The values dictionary will contain the timer ID that is returned from this function.

        :param frequency_ms:    How often to generate timer events in milliseconds
        :type frequency_ms:     int
        :param key:             Key to be returned as the timer event
        :type key:              str | int | tuple | object
        :param repeating:       If True then repeat timer events until timer is explicitly stopped
        :type repeating:        bool
        :return:                Timer ID for the timer
        :rtype:                 int
        """
        timer = _TimerPeriodic(self, frequency_ms=frequency_ms, key=key, repeating=repeating)
        return timer.id


    def timer_stop(self, timer_id):
        """
        Stops a timer with a given ID

        :param timer_id:        Timer ID of timer to stop
        :type timer_id:         int
        :return:
        """
        _TimerPeriodic.stop_timer_with_id(timer_id)


    def timer_stop_all(self):
        """
        Stops all timers for THIS window
        """
        _TimerPeriodic.stop_all_timers_for_window(self)


    def timer_get_active_timers(self):
        """
        Returns a list of currently active timers for a window
        :return:    List of timers for the window
        :rtype:     List[int]
        """
        return _TimerPeriodic.get_all_timers_for_window(self)


    @classmethod
    def _restore_stdout(cls):
        for item in cls._rerouted_stdout_stack:
            (window, element) = item   # type: (Window, Element)
            if not window.is_closed():
                sys.stdout = element
                break
        cls._rerouted_stdout_stack = [item for item in cls._rerouted_stdout_stack if not item[0].is_closed()]
        if len(cls._rerouted_stdout_stack) == 0 and cls._original_stdout is not None:
            sys.stdout = cls._original_stdout
        # print('Restored stdout... new stack:',  [item[0].Title for item in cls._rerouted_stdout_stack ])


    @classmethod
    def _restore_stderr(cls):
        for item in cls._rerouted_stderr_stack:
            (window, element) = item   # type: (Window, Element)
            if not window.is_closed():
                sys.stderr = element
                break
        cls._rerouted_stderr_stack = [item for item in cls._rerouted_stderr_stack if not item[0].is_closed()]
        if len(cls._rerouted_stderr_stack) == 0 and cls._original_stderr is not None:
            sys.stderr = cls._original_stderr
        # print('Restored stderr... new stack:',  [item[0].Title for item in cls._rerouted_stderr_stack ])





    # def __enter__(self):
    #     """
    #     WAS used with context managers which are no longer needed nor advised.  It is here for legacy support and
    #     am afraid of removing right now
    #     :return: (window)
    #      :rtype:
    #     """
    #     return self

    def __getitem__(self, key):
        """
        Returns Element that matches the passed in key.
        This is "called" by writing code as thus:
        window['element key'].update

        :param key: The key to find
        :type key:  str | int | tuple | object
        :return:    The element found
        :rtype:     Element | Input | Combo | OptionMenu | Listbox | Radio | Checkbox | Spin | Multiline | Text | StatusBar | Output | Button | ButtonMenu | ProgressBar | Image | Canvas | Graph | Frame | VerticalSeparator | HorizontalSeparator | Tab | TabGroup | Slider | Column | Pane | Menu | Table | Tree | ErrorElement | None
        """

        return self.find_element(key)

    def __call__(self, *args, **kwargs):
        """
        Call window.read but without having to type it out.
        window() == window.read()
        window(timeout=50) == window.read(timeout=50)

        :return: The famous event, values that read returns.
        :rtype:  Tuple[Any, Dict[Any, Any]]
        """
        return self.read(*args, **kwargs)

    def _is_window_created(self, additional_message=''):
        msg = str(additional_message)
        if self.TKroot is None:
            warnings.warn(
                'You cannot perform operations on a Window until it is read or finalized. Adding a "finalize=True" parameter to your Window creation will fix this. ' + msg,
                UserWarning)
            if not SUPPRESS_ERROR_POPUPS:
                _error_popup_with_traceback('You cannot perform operations on a Window until it is read or finalized.',
                                            'Adding a "finalize=True" parameter to your Window creation will likely fix this', msg)
            return False
        return True

    def _has_custom_titlebar_element(self):
        for elem in self.AllKeysDict.values():
            if elem.Key in (TITLEBAR_MAXIMIZE_KEY, TITLEBAR_CLOSE_KEY, TITLEBAR_IMAGE_KEY):
                return True
            if elem.metadata == TITLEBAR_METADATA_MARKER:
                return True
        return False

    AddRow = add_row
    AddRows = add_rows
    AlphaChannel = alpha_channel
    BringToFront = bring_to_front
    Close = close
    CurrentLocation = current_location
    Disable = disable
    DisableDebugger = disable_debugger
    Disappear = disappear
    Enable = enable
    EnableDebugger = enable_debugger
    Fill = fill
    Finalize = finalize
    # FindElement = find_element
    FindElementWithFocus = find_element_with_focus
    GetScreenDimensions = get_screen_dimensions
    GrabAnyWhereOff = grab_any_where_off
    GrabAnyWhereOn = grab_any_where_on
    Hide = hide
    Layout = layout
    LoadFromDisk = load_from_disk
    Maximize = maximize
    Minimize = minimize
    Move = move
    Normal = normal
    Read = read
    Reappear = reappear
    Refresh = refresh
    SaveToDisk = save_to_disk
    SendToBack = send_to_back
    SetAlpha = set_alpha
    SetIcon = set_icon
    SetTransparentColor = set_transparent_color
    Size = size
    UnHide = un_hide
    VisibilityChanged = visibility_changed
    CloseNonBlocking = close
    CloseNonBlockingForm = close
    start_thread = perform_long_operation
    #
    # def __exit__(self, *a):
    #     """
    #     WAS used with context managers which are no longer needed nor advised.  It is here for legacy support and
    #     am afraid of removing right now
    #     :param *a: (?) Not sure what's passed in.
    #      :type *a:
    #     :return:   Always returns False which was needed for context manager to work
    #      :rtype:
    #     """
    #     self.__del__()
    #     return False
    #
    # def __del__(self):
    #     # print('DELETING WINDOW')
    #     for row in self.Rows:
    #         for element in row:
    #             element.__del__()


# -------------------------------- PEP8-ify the Window Class USER Interfaces -------------------------------- #


FlexForm = Window


def _long_func_thread(window, end_key, original_func):
    """
    Used to run long operations on the user's behalf. Called by the window object

    :param window:        The window that will get the event
    :type window:         (Window)
    :param end_key:       The event that will be sent when function returns. If None then no event will be sent when exiting thread
    :type end_key:        (Any|None)
    :param original_func: The user's function that is called. Can be a function with no arguments or a lambda experession
    :type original_func:  (Any)
    """

    return_value = original_func()
    if end_key is not None:
        window.write_event_value(end_key, return_value)


def _exit_mainloop(exiting_window):
    if exiting_window == Window._window_running_mainloop or Window._root_running_mainloop == Window.hidden_master_root:
        Window._window_that_exited = exiting_window
        if Window._root_running_mainloop is not None:
            Window._root_running_mainloop.quit()
        # print('** Exited window mainloop **')


def _timeout_alarm_callback_hidden():
    """
    Read Timeout Alarm callback. Will kick a mainloop call out of the tkinter event loop and cause it to return
    """

    del Window._TKAfterID

    # first, get the results table built
    # modify the Results table in the parent FlexForm object
    # print('TIMEOUT CALLBACK')
    Window._root_running_mainloop.quit()  # kick the users out of the mainloop

    # Get window that caused return
    Window._window_that_exited = None


def read_all_windows(timeout=None, timeout_key=TIMEOUT_KEY):
    """
    Reads all windows that are "active" when the call is made. "Active" means that it's been finalized or read.
    If a window has not been finalized then it will not be considered an "active window"

    If any of the active windows returns a value then the window and its event and values
    are returned.

    If no windows are open, then the value (None, WIN_CLOSED, None) will be returned
        Since WIN_CLOSED is None, it means (None, None, None) is what's returned when no windows remain opened

    :param timeout:     Time in milliseconds to delay before a returning a timeout event
    :type timeout:      (int)
    :param timeout_key: Key to return when a timeout happens. Defaults to the standard TIMEOUT_KEY
    :type timeout_key:  (Any)
    :return:            A tuple with the  (Window, event, values dictionary/list)
    :rtype:             (Window, Any, Dict | List)
    """

    if len(Window._active_windows) == 0:
        return None, WIN_CLOSED, None

    # first see if any queued events are waiting for any of the windows
    for window in Window._active_windows.keys():
        if window._queued_thread_event_available():
            _BuildResults(window, False, window)
            event, values = window.ReturnValues
            return window, event, values

    Window._root_running_mainloop = Window.hidden_master_root
    Window._timeout_key = timeout_key

    if timeout == 0:
        window = list(Window._active_windows.keys())[Window._timeout_0_counter]
        event, values = window._ReadNonBlocking()
        if event is None:
            event = timeout_key
        if values is None:
            event = None
        Window._timeout_0_counter = (Window._timeout_0_counter + 1) % len(Window._active_windows)
        return window, event, values

    Window._timeout_0_counter = 0  # reset value if not reading with timeout 0 so ready next time needed

    # setup timeout timer
    if timeout != None:
        try:
            Window.hidden_master_root.after_cancel(Window._TKAfterID)
            del Window._TKAfterID
        except:
            pass

        Window._TKAfterID = Window.hidden_master_root.after(timeout, _timeout_alarm_callback_hidden)

    # ------------ Call Mainloop ------------
    Window._root_running_mainloop.mainloop()

    try:
        Window.hidden_master_root.after_cancel(Window._TKAfterID)
        del Window._TKAfterID
    except:
        pass
        # print('** tkafter cancel failed **')

    # Get window that caused return

    window = Window._window_that_exited

    if window is None:
        return None, timeout_key, None

    if window.XFound:
        event, values = None, None
        window.close()
        try:
            del Window._active_windows[window]
        except:
            pass
            # print('Error deleting window, but OK')
    else:
        _BuildResults(window, False, window)
        event, values = window.ReturnValues

    return window, event, values

# MP""""""`MM                     dP
# M  mmmmm..M                     88
# M.      `YM dP    dP .d8888b. d8888P .d8888b. 88d8b.d8b.
# MMMMMMM.  M 88    88 Y8ooooo.   88   88ooood8 88'`88'`88
# M. .MMM'  M 88.  .88       88   88   88.  ... 88  88  88
# Mb.     .dM `8888P88 `88888P'   dP   `88888P' dP  dP  dP
# MMMMMMMMMMM      .88
#              d8888P
# M""""""""M
# Mmmm  mmmM
# MMMM  MMMM 88d888b. .d8888b. dP    dP
# MMMM  MMMM 88'  `88 88'  `88 88    88
# MMMM  MMMM 88       88.  .88 88.  .88
# MMMM  MMMM dP       `88888P8 `8888P88
# MMMMMMMMMM                        .88
#                               d8888P

# ------------------------------------------------------------------------- #
#                       SystemTray - class for implementing a psyeudo tray  #
# ------------------------------------------------------------------------- #

# -------------------------------- System Tray Begins Here -------------------------------- #
# Feb 2020 - Just starting on this so code commented out for now. Basing on PySimpleGUIQt's implementation / call format


# -------------------------------------------------------------------
# fade in/out info and default window alpha
SYSTEM_TRAY_WIN_MARGINS = 160, 60  # from right edge of screen, from bottom of screen
SYSTEM_TRAY_MESSAGE_MAX_LINE_LENGTH = 50
# colors
SYSTEM_TRAY_MESSAGE_WIN_COLOR = '#282828'
SYSTEM_TRAY_MESSAGE_TEXT_COLOR = '#ffffff'

SYSTEM_TRAY_MESSAGE_DISPLAY_DURATION_IN_MILLISECONDS = 3000  # how long to display the window
SYSTEM_TRAY_MESSAGE_FADE_IN_DURATION = 1000  # how long to fade in / fade out the window

EVENT_SYSTEM_TRAY_ICON_DOUBLE_CLICKED = '__DOUBLE_CLICKED__'
EVENT_SYSTEM_TRAY_ICON_ACTIVATED = '__ACTIVATED__'
EVENT_SYSTEM_TRAY_MESSAGE_CLICKED = '__MESSAGE_CLICKED__'

# Base64 Images to use as icons in the window
_tray_icon_error = b'iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAMAAABEpIrGAAAAA3NCSVQICAjb4U/gAAAACXBIWXMAAADlAAAA5QGP5Zs8AAAAGXRFWHRTb2Z0d2FyZQB3d3cuaW5rc2NhcGUub3Jnm+48GgAAAIpQTFRF////20lt30Bg30pg4FJc409g4FBe4E9f4U9f4U9g4U9f4E9g31Bf4E9f4E9f4E9f4E9f4E9f4FFh4Vdm4lhn42Bv5GNx5W575nJ/6HqH6HyI6YCM6YGM6YGN6oaR8Kev9MPI9cbM9snO9s3R+Nfb+dzg+d/i++vt/O7v/fb3/vj5//z8//7+////KofnuQAAABF0Uk5TAAcIGBktSYSXmMHI2uPy8/XVqDFbAAAA8UlEQVQ4y4VT15LCMBBTQkgPYem9d9D//x4P2I7vILN68kj2WtsAhyDO8rKuyzyLA3wjSnvi0Eujf3KY9OUP+kno651CvlB0Gr1byQ9UXff+py5SmRhhIS0oPj4SaUUCAJHxP9+tLb/ezU0uEYDUsCc+l5/T8smTIVMgsPXZkvepiMj0Tm5txQLENu7gSF7HIuMreRxYNkbmHI0u5Hk4PJOXkSMz5I3nyY08HMjbpOFylF5WswdJPmYeVaL28968yNfGZ2r9gvqFalJNUy2UWmq1Wa7di/3Kxl3tF1671YHRR04dWn3s9cXRV09f3vb1fwPD7z9j1WgeRgAAAABJRU5ErkJggg=='
_tray_icon_success = b'iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAMAAABEpIrGAAAAA3NCSVQICAjb4U/gAAAACXBIWXMAAAEKAAABCgEWpLzLAAAAGXRFWHRTb2Z0d2FyZQB3d3cuaW5rc2NhcGUub3Jnm+48GgAAAHJQTFRF////ZsxmbbZJYL9gZrtVar9VZsJcbMRYaMZVasFYaL9XbMFbasRZaMFZacRXa8NYasFaasJaasFZasJaasNZasNYasJYasJZasJZasJZasJZasJZasJYasJZasJZasJZasJZasJaasJZasJZasJZasJZ2IAizQAAACV0Uk5TAAUHCA8YGRobHSwtPEJJUVtghJeYrbDByNjZ2tvj6vLz9fb3/CyrN0oAAADnSURBVDjLjZPbWoUgFIQnbNPBIgNKiwwo5v1fsQvMvUXI5oqPf4DFOgCrhLKjC8GNVgnsJY3nKm9kgTsduVHU3SU/TdxpOp15P7OiuV/PVzk5L3d0ExuachyaTWkAkLFtiBKAqZHPh/yuAYSv8R7XE0l6AVXnwBNJUsE2+GMOzWL8k3OEW7a/q5wOIS9e7t5qnGExvF5Bvlc4w/LEM4Abt+d0S5BpAHD7seMcf7+ZHfclp10TlYZc2y2nOqc6OwruxUWx0rDjNJtyp6HkUW4bJn0VWdf/a7nDpj1u++PBOR694+Ftj/8PKNdnDLn/V8YAAAAASUVORK5CYII='
_tray_icon_halt = b'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'
_tray_icon_notallowed = b'iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAMAAABEpIrGAAAAAXNSR0IArs4c6QAAAARnQU1BAACxjwv8YQUAAAMAUExURQAAAPcICPcLC/cMDPcQEPcSEvcXF/cYGPcaGvcbG/ccHPgxMfgyMvg0NPg5Ofg6Ovg7O/hBQfhCQvlFRflGRvljY/pkZPplZfpnZ/p2dgAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAMgEwNYAAAEAdFJOU////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////wBT9wclAAAACXBIWXMAAA7DAAAOwwHHb6hkAAABE0lEQVQ4T4WT65bDIAiExWbbtN0m3Uua+P4P6g4jGtN4NvNL4DuCCC5WWobe++uwmEmtwNxJUTebcwWCt5jJBwsYcKf3NE4hTOOJxj1FEnBTz4NH6qH2jUcCGr/QLLpkQgHe/6VWJXVqFgBB4yI/KVCkBCoFgPrPHw0CWbwCL8RibBFwzQDQH62/QeAtHQBeADUIDbkF/UnmnkB1ixtERrN3xCgyuF5kMntHTCJXh2vyv+wIdMhvgTeCQJ0C2hBMgSKfZlM1wSLXZ5oqgs8sjSpaCQ2VVlfKhLU6fdZGSvyWz9JMb+NE4jt/Nwfm0yJZSkBpYDg7TcJGrjm0Z7jK0B6P/fHiHK8e9Pp/eSmuf1+vf4x/ralnCN9IrncAAAAASUVORK5CYII='
_tray_icon_stop = b'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'
_tray_icon_exclamation = b'iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAMAAABEpIrGAAAAAXNSR0IArs4c6QAAAARnQU1BAACxjwv8YQUAAAMAUExURQAAAN0zM900NN01Nd02Nt03N944ON45Od46Ot47O98/P99BQd9CQt9DQ+FPT+JSUuJTU+JUVOJVVeJWVuNbW+ReXuVjY+Zra+dxceh4eOl7e+l8fOl+ful/f+qBgeqCguqDg+qFheuJieuLi+yPj+yQkO2Wlu+cnO+hofGqqvGtrfre3vrf3/ri4vvn5/75+f76+v/+/v///wAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAMQ8SQkAAAEAdFJOU////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////wBT9wclAAAACXBIWXMAAA7DAAAOwwHHb6hkAAABJElEQVQ4T4WS63KCMBBGsyBai62X0otY0aq90ZZa3v/dtpvsJwTijOfXt7tnILOJYY9tNonjNCtQOlqhuKKG0RrNVjgkmIHBHgMId+h7zHSiwg2a9FNVVYScupETmjkd67o+CWpYwft+R6CpCgeUlq5AOyf45+8JsRUKFI6eQLkI3n5CIREBUekLxGaLpATCymRISiAszARJCYSxiZGUQKDLQoqgnPnFhUPOTWeRoZD3FvVZlmVHkE2OEM9iV71GVoZDBGUpAg9QWN5/jx+Ilsi9hz0q4VHOWD+hEF70yc1QEr1a4Q0F0S3eJDfLuv8T4QEFXduZE1rj+et7g6hzCDxF08N+X4DAu+6lUSTnc5wE5tx73ckSTV8QVoux3N88Rykw/wP3i+vwPKk17AAAAABJRU5ErkJggg=='
_tray_icon_none = None

SYSTEM_TRAY_MESSAGE_ICON_INFORMATION = _tray_icon_success
SYSTEM_TRAY_MESSAGE_ICON_WARNING = _tray_icon_exclamation
SYSTEM_TRAY_MESSAGE_ICON_CRITICAL = _tray_icon_stop
SYSTEM_TRAY_MESSAGE_ICON_NOICON = _tray_icon_none


# ------------------------------------------------------------------------- #
#                       Tray CLASS                                      #
# ------------------------------------------------------------------------- #
class SystemTray:
    """
    A "Simulated System Tray" that duplicates the API calls available to PySimpleGUIWx and PySimpleGUIQt users.

    All of the functionality works. The icon is displayed ABOVE the system tray rather than inside of it.
    """

    def __init__(self, menu=None, filename=None, data=None, data_base64=None, tooltip=None, metadata=None):
        """
        SystemTray - create an icon in the system tray
        :param menu:        Menu definition. Example - ['UNUSED', ['My', 'Simple', '---', 'Menu', 'Exit']]
        :type menu:         List[List[List[str] or str]]
        :param filename:    filename for icon
        :type filename:     (str)
        :param data:        in-ram image for icon (same as data_base64 parm)
        :type data:         (bytes)
        :param data_base64: base-64 data for icon
        :type data_base64:  (bytes)
        :param tooltip:     tooltip string
        :type tooltip:      (str)
        :param metadata:    User metadata that can be set to ANYTHING
        :type metadata:     (Any)
        """
        self._metadata = None
        self.Menu = menu
        self.TrayIcon = None
        self.Shown = False
        self.MenuItemChosen = TIMEOUT_KEY
        self.metadata = metadata
        self.last_message_event = None

        screen_size = Window.get_screen_size()

        if filename:
            image_elem = Image(filename=filename, background_color='red', enable_events=True, tooltip=tooltip, key='-IMAGE-')
        elif data_base64:
            image_elem = Image(data=data_base64, background_color='red', enable_events=True, tooltip=tooltip, key='-IMAGE-')
        elif data:
            image_elem = Image(data=data, background_color='red', enable_events=True, tooltip=tooltip, key='-IMAGE-')
        else:
            image_elem = Image(background_color='red', enable_events=True, tooltip=tooltip, key='-IMAGE-')
        layout = [
            [image_elem],
        ]
        self.window = Window('Window Title', layout, element_padding=(0, 0), margins=(0, 0), grab_anywhere=True, no_titlebar=True, transparent_color='red',
                             keep_on_top=True, right_click_menu=menu, location=(screen_size[0] - 100, screen_size[1] - 100), finalize=True)

        self.window['-IMAGE-'].bind('<Double-Button-1>', '+DOUBLE_CLICK')

    @property
    def metadata(self):
        """
        Metadata is an SystemTray property that you can use at any time to hold any value
        :return: the current metadata value
        :rtype:  (Any)
        """
        return self._metadata

    @metadata.setter
    def metadata(self, value):
        """
        Metadata is an SystemTray property that you can use at any time to hold any value
        :param value: Anything you want it to be
        :type value:  (Any)
        """
        self._metadata = value

    def read(self, timeout=None):
        """
        Reads the context menu
        :param timeout: Optional.  Any value other than None indicates a non-blocking read
        :type timeout:
        :return:
        :rtype:
        """
        if self.last_message_event != TIMEOUT_KEY and self.last_message_event is not None:
            event = self.last_message_event
            self.last_message_event = None
            return event
        event, values = self.window.read(timeout=timeout)
        if event.endswith('DOUBLE_CLICK'):
            return EVENT_SYSTEM_TRAY_ICON_DOUBLE_CLICKED
        elif event == '-IMAGE-':
            return EVENT_SYSTEM_TRAY_ICON_ACTIVATED

        return event

    def hide(self):
        """
        Hides the icon
        """
        self.window.hide()

    def un_hide(self):
        """
        Restores a previously hidden icon
        """
        self.window.un_hide()

    def show_message(self, title, message, filename=None, data=None, data_base64=None, messageicon=None,
                     time=(SYSTEM_TRAY_MESSAGE_FADE_IN_DURATION, SYSTEM_TRAY_MESSAGE_DISPLAY_DURATION_IN_MILLISECONDS)):
        """
        Shows a balloon above icon in system tray
        :param title:       Title shown in balloon
        :type title:        str
        :param message:     Message to be displayed
        :type message:      str
        :param filename:    Optional icon filename
        :type filename:     str
        :param data:        Optional in-ram icon
        :type data:         b''
        :param data_base64: Optional base64 icon
        :type data_base64:  b''
        :param time:        Amount of time to display message in milliseconds. If tuple, first item is fade in/out duration
        :type time:         int | (int, int)
        :return:            The event that happened during the display such as user clicked on message
        :rtype:             Any
        """

        if isinstance(time, tuple):
            fade_duration, display_duration = time
        else:
            fade_duration = SYSTEM_TRAY_MESSAGE_FADE_IN_DURATION
            display_duration = time

        user_icon = data_base64 or filename or data or messageicon

        event = self.notify(title, message, icon=user_icon, fade_in_duration=fade_duration, display_duration_in_ms=display_duration)
        self.last_message_event = event
        return event

    def close(self):
        """
        Close the system tray window
        """
        self.window.close()

    def update(self, menu=None, tooltip=None, filename=None, data=None, data_base64=None, ):
        """
        Updates the menu, tooltip or icon
        :param menu:        menu defintion
        :type menu:         ???
        :param tooltip:     string representing tooltip
        :type tooltip:      ???
        :param filename:    icon filename
        :type filename:     ???
        :param data:        icon raw image
        :type data:         ???
        :param data_base64: icon base 64 image
        :type data_base64:  ???
        """
        # Menu
        if menu is not None:
            top_menu = tk.Menu(self.window.TKroot, tearoff=False)
            AddMenuItem(top_menu, menu[1], self.window['-IMAGE-'])
            self.window['-IMAGE-'].TKRightClickMenu = top_menu

        if filename:
            self.window['-IMAGE-'].update(filename=filename)
        elif data_base64:
            self.window['-IMAGE-'].update(data=data_base64)
        elif data:
            self.window['-IMAGE-'].update(data=data)

        if tooltip:
            self.window['-IMAGE-'].set_tooltip(tooltip)

    @classmethod
    def notify(cls, title, message, icon=_tray_icon_success, display_duration_in_ms=SYSTEM_TRAY_MESSAGE_DISPLAY_DURATION_IN_MILLISECONDS,
               fade_in_duration=SYSTEM_TRAY_MESSAGE_FADE_IN_DURATION, alpha=0.9, location=None):
        """
        Displays a "notification window", usually in the bottom right corner of your display.  Has an icon, a title, and a message
        The window will slowly fade in and out if desired.  Clicking on the window will cause it to move through the end the current "phase". For example, if the window was fading in and it was clicked, then it would immediately stop fading in and instead be fully visible.  It's a way for the user to quickly dismiss the window.
        :param title:                  Text to be shown at the top of the window in a larger font
        :type title:                   (str)
        :param message:                Text message that makes up the majority of the window
        :type message:                 (str)
        :param icon:                   A base64 encoded PNG/GIF image or PNG/GIF filename that will be displayed in the window
        :type icon:                    bytes | str
        :param display_duration_in_ms: Number of milliseconds to show the window
        :type display_duration_in_ms:  (int)
        :param fade_in_duration:       Number of milliseconds to fade window in and out
        :type fade_in_duration:        (int)
        :param alpha:                  Alpha channel. 0 - invisible 1 - fully visible
        :type alpha:                   (float)
        :param location:               Location on the screen to display the window
        :type location:                (int, int)
        :return:                       (int) reason for returning
        :rtype:                        (int)
        """

        messages = message.split('\n')
        full_msg = ''
        for m in messages:
            m_wrap = textwrap.fill(m, SYSTEM_TRAY_MESSAGE_MAX_LINE_LENGTH)
            full_msg += m_wrap + '\n'
        message = full_msg[:-1]

        win_msg_lines = message.count('\n') + 1
        max_line = max(message.split('\n'))

        screen_res_x, screen_res_y = Window.get_screen_size()
        win_margin = SYSTEM_TRAY_WIN_MARGINS  # distance from screen edges
        win_width, win_height = 364, 66 + (14.8 * win_msg_lines)

        layout = [[Graph(canvas_size=(win_width, win_height), graph_bottom_left=(0, win_height), graph_top_right=(win_width, 0), key='-GRAPH-',
                         background_color=SYSTEM_TRAY_MESSAGE_WIN_COLOR, enable_events=True)]]

        win_location = location if location is not None else (screen_res_x - win_width - win_margin[0], screen_res_y - win_height - win_margin[1])
        window = Window(title, layout, background_color=SYSTEM_TRAY_MESSAGE_WIN_COLOR, no_titlebar=True,
                        location=win_location, keep_on_top=True, alpha_channel=0, margins=(0, 0), element_padding=(0, 0), grab_anywhere=True, finalize=True)

        window['-GRAPH-'].draw_rectangle((win_width, win_height), (-win_width, -win_height), fill_color=SYSTEM_TRAY_MESSAGE_WIN_COLOR,
                                         line_color=SYSTEM_TRAY_MESSAGE_WIN_COLOR)
        if type(icon) is bytes:
            window['-GRAPH-'].draw_image(data=icon, location=(20, 20))
        elif icon is not None:
            window['-GRAPH-'].draw_image(filename=icon, location=(20, 20))
        window['-GRAPH-'].draw_text(title, location=(64, 20), color=SYSTEM_TRAY_MESSAGE_TEXT_COLOR, font=('Helvetica', 12, 'bold'),
                                    text_location=TEXT_LOCATION_TOP_LEFT)
        window['-GRAPH-'].draw_text(message, location=(64, 44), color=SYSTEM_TRAY_MESSAGE_TEXT_COLOR, font=('Helvetica', 9),
                                    text_location=TEXT_LOCATION_TOP_LEFT)
        window['-GRAPH-'].set_cursor('hand2')

        if fade_in_duration:
            for i in range(1, int(alpha * 100)):  # fade in
                window.set_alpha(i / 100)
                event, values = window.read(timeout=fade_in_duration // 100)
                if event != TIMEOUT_KEY:
                    window.set_alpha(1)
                    break
            if event != TIMEOUT_KEY:
                window.close()
                return EVENT_SYSTEM_TRAY_MESSAGE_CLICKED if event == '-GRAPH-' else event
            event, values = window(timeout=display_duration_in_ms)
            if event == TIMEOUT_KEY:
                for i in range(int(alpha * 100), 1, -1):  # fade out
                    window.set_alpha(i / 100)
                    event, values = window.read(timeout=fade_in_duration // 100)
                    if event != TIMEOUT_KEY:
                        break
        else:
            window.set_alpha(alpha)
            event, values = window(timeout=display_duration_in_ms)
        window.close()

        return EVENT_SYSTEM_TRAY_MESSAGE_CLICKED if event == '-GRAPH-' else event

    Close = close
    Hide = hide
    Read = read
    ShowMessage = show_message
    UnHide = un_hide
    Update = update


# ################################################################################
# ################################################################################
#  END OF ELEMENT DEFINITIONS
# ################################################################################
# ################################################################################


# =========================================================================== #
# Button Lazy Functions so the caller doesn't have to define a bunch of stuff #
# =========================================================================== #

# ------------------------- A fake Element... the Pad Element ------------------------- #
def Sizer(h_pixels=0, v_pixels=0):
    """
    "Pushes" out the size of whatever it is placed inside of.  This includes Columns, Frames, Tabs and Windows

    :param h_pixels: number of horizontal pixels
    :type h_pixels:  (int)
    :param v_pixels: number of vertical pixels
    :type v_pixels:  (int)
    :return:         (Canvas) A canvas element that has a pad setting set according to parameters
    :rtype:          (Canvas)
    """

    return Canvas(size=(0, 0), pad=((h_pixels, 0), (v_pixels, 0)))

def pin(elem, vertical_alignment=None, shrink=True, expand_x=None, expand_y=None):
    """
    Pin's an element provided into a layout so that when it's made invisible and visible again, it will
     be in the correct place.  Otherwise it will be placed at the end of its containing window/column.

     The element you want to pin is the element that you'll be making visibile/invisible.

    The pin helper function also causes containers to shrink to fit the contents correct after something inside
     has changed visiblity.  Note that setting a hardcoded size on your window can impact this ability to shrink.

    :param elem:               the element to put into the layout
    :type elem:                Element
    :param vertical_alignment: Aligns elements vertically. 'top', 'center', 'bottom'. Can be shortened to 't', 'c', 'b'
    :type vertical_alignment:  str | None
    :param shrink:             If True, then the space will shrink down to a single pixel when hidden. False leaves the area large and blank
    :type shrink:              bool
    :param expand_x:           If True/False the value will be passed to the Column Elements used to make this feature
    :type expand_x:            (bool)
    :param expand_y:           If True/False the value will be passed to the Column Elements used to make this feature
    :type expand_y:            (bool)
    :return:                   A column element containing the provided element
    :rtype:                    Column
    """
    if shrink:
        # return Column([[elem, Canvas(size=(0, 0),background_color=elem.BackgroundColor, pad=(0, 0))]], pad=(0, 0), vertical_alignment=vertical_alignment, expand_x=expand_x, expand_y=expand_y)
        return Column([[elem, Column([[]],pad=(0,0))]], pad=(0, 0), vertical_alignment=vertical_alignment, expand_x=expand_x, expand_y=expand_y)
    else:
        return Column([[elem]], pad=(0, 0), vertical_alignment=vertical_alignment, expand_x=expand_x, expand_y=expand_y)


def vtop(elem_or_row, expand_x=None, expand_y=None, background_color=None):
    """
    Align an element or a row of elements to the top of the row that contains it

    :param elem_or_row:      the element or row of elements
    :type elem_or_row:       Element | List[Element] | Tuple[Element]
    :param expand_x:         If True/False the value will be passed to the Column Elements used to make this feature
    :type expand_x:          (bool)
    :param expand_y:         If True/False the value will be passed to the Column Elements used to make this feature
    :type expand_y:          (bool)
    :param background_color: Background color for container that is used by vtop to do the alignment
    :type background_color:  str | None
    :return:                 A column element containing the provided element aligned to the top or list of elements (a row)
    :rtype:                  Column | List[Column]
    """


    if isinstance(elem_or_row, list) or isinstance(elem_or_row, tuple):
        return [Column([[e]], pad=(0, 0), vertical_alignment='top', expand_x=expand_x, expand_y=expand_y, background_color=background_color) for e in elem_or_row]

    return Column([[elem_or_row]], pad=(0, 0), vertical_alignment='top', expand_x=expand_x, expand_y=expand_y, background_color=background_color)


def vcenter(elem_or_row, expand_x=None, expand_y=None, background_color=None):
    """
    Align an element or a row of elements to the center of the row that contains it

    :param elem_or_row:      the element or row of elements
    :type elem_or_row:       Element | List[Element] | Tuple[Element]
    :param expand_x:         If True/False the value will be passed to the Column Elements used to make this feature
    :type expand_x:          (bool)
    :param expand_y:         If True/False the value will be passed to the Column Elements used to make this feature
    :type expand_y:          (bool)
    :param background_color: Background color for container that is used by vcenter to do the alignment
    :type background_color:  str | None
    :return:                 A column element containing the provided element aligned to the center or list of elements (a row)
    :rtype:                  Column | List[Column]
    """

    if isinstance(elem_or_row, list) or isinstance(elem_or_row, tuple):
        return [Column([[e]], pad=(0, 0), vertical_alignment='center',expand_x=expand_x, expand_y=expand_y, background_color=background_color) for e in elem_or_row]

    return Column([[elem_or_row]], pad=(0, 0), vertical_alignment='center', expand_x=expand_x, expand_y=expand_y,background_color=background_color)


def vbottom(elem_or_row, expand_x=None, expand_y=None, background_color=None):
    """
    Align an element or a row of elements to the bottom of the row that contains it

    :param elem_or_row:      the element or row of elements
    :type elem_or_row:       Element | List[Element] | Tuple[Element]
    :param expand_x:         If True/False the value will be passed to the Column Elements used to make this feature
    :type expand_x:          (bool)
    :param expand_y:         If True/False the value will be passed to the Column Elements used to make this feature
    :type expand_y:          (bool)
    :param background_color: Background color for container that is used by vcenter to do the alignment
    :type background_color:  str | None
    :return:                 A column element containing the provided element aligned to the bottom or list of elements (a row)
    :rtype:                  Column | List[Column]
    """


    if isinstance(elem_or_row, list) or isinstance(elem_or_row, tuple):
        return [Column([[e]], pad=(0, 0), vertical_alignment='bottom', expand_x=expand_x, expand_y=expand_y, background_color=background_color) for e in elem_or_row]

    return Column([[elem_or_row]], pad=(0, 0), vertical_alignment='bottom', expand_x=expand_x, expand_y=expand_y,background_color=background_color)


def Titlebar(title='', icon=None, text_color=None, background_color=None, font=None, key=None, k=None):
    """
    A custom titlebar that replaces the OS provided titlebar, thus giving you control
    the is not possible using the OS provided titlebar such as the color.

    NOTE LINUX USERS - at the moment the minimize function is not yet working.  Windows users
    should have no problem and it should function as a normal window would.

    This titlebar is created from a row of elements that is then encapsulated into a
    one Column element which is what this Titlebar function returns to you.

    A custom titlebar removes the margins from your window.  If you want the  remainder
    of your Window to have margins, place the layout after the Titlebar into a Column and
    set the pad of that Column to the dimensions you would like your margins to have.

    The Titlebar is a COLUMN element.  You can thus call the update method for the column and
    perform operations such as making the column visible/invisible

    :param icon:             Can be either a filename or Base64 byte string of a PNG or GIF. This is used in an Image element to create the titlebar
    :type icon:              str or bytes or None
    :param title:            The "title" to show in the titlebar
    :type title:             str
    :param text_color:       Text color for titlebar
    :type text_color:        str | None
    :param background_color: Background color for titlebar
    :type background_color:  str | None
    :param font:             Font to be used for the text and the symbols
    :type font:              (str or (str, int[, str]) or None)
    :param key:              Identifies an Element. Should be UNIQUE to this window.
    :type key:               str | int | tuple | object | None
    :param k:                Exactly the same as key.  Choose one of them to use
    :type k:                 str | int | tuple | object | None
    :return:                 A single Column element that has eveything in 1 element
    :rtype:                  Column
    """
    bc = background_color or CUSTOM_TITLEBAR_BACKGROUND_COLOR or theme_button_color()[1]
    tc = text_color or CUSTOM_TITLEBAR_TEXT_COLOR or theme_button_color()[0]
    font = font or CUSTOM_TITLEBAR_FONT or ('Helvetica', 12)
    key = k or key

    if isinstance(icon, bytes):
        icon_and_text_portion = [Image(data=icon, background_color=bc, key=TITLEBAR_IMAGE_KEY)]
    elif icon == TITLEBAR_DO_NOT_USE_AN_ICON:
        icon_and_text_portion = []
    elif icon is not None:
        icon_and_text_portion = [Image(filename=icon, background_color=bc, key=TITLEBAR_IMAGE_KEY)]
    elif CUSTOM_TITLEBAR_ICON is not None:
        if isinstance(CUSTOM_TITLEBAR_ICON, bytes):
            icon_and_text_portion = [Image(data=CUSTOM_TITLEBAR_ICON, background_color=bc, key=TITLEBAR_IMAGE_KEY)]
        else:
            icon_and_text_portion = [Image(filename=CUSTOM_TITLEBAR_ICON, background_color=bc, key=TITLEBAR_IMAGE_KEY)]
    else:
        icon_and_text_portion = [Image(data=DEFAULT_BASE64_ICON_16_BY_16, background_color=bc, key=TITLEBAR_IMAGE_KEY)]

    icon_and_text_portion += [T(title, text_color=tc, background_color=bc, font=font, grab=True, key=TITLEBAR_TEXT_KEY)]

    return Column([[Column([icon_and_text_portion], pad=(0, 0), background_color=bc),
                    Column([[T(SYMBOL_TITLEBAR_MINIMIZE, text_color=tc, background_color=bc, enable_events=True, font=font, key=TITLEBAR_MINIMIZE_KEY),
                             Text(SYMBOL_TITLEBAR_MAXIMIZE, text_color=tc, background_color=bc, enable_events=True, font=font, key=TITLEBAR_MAXIMIZE_KEY),
                             Text(SYMBOL_TITLEBAR_CLOSE, text_color=tc, background_color=bc, font=font, enable_events=True, key=TITLEBAR_CLOSE_KEY), ]],
                           element_justification='r', expand_x=True, grab=True, pad=(0, 0), background_color=bc)]], expand_x=True, grab=True,
                  background_color=bc, pad=(0, 0), metadata=TITLEBAR_METADATA_MARKER, key=key)


def MenubarCustom(menu_definition, disabled_text_color=None, bar_font=None, font=None, tearoff=False, pad=0, p=None, background_color=None, text_color=None,
                  bar_background_color=None, bar_text_color=None, key=None, k=None):
    """
    A custom Menubar that replaces the OS provided Menubar

    Why?
    Two reasons - 1. they look great (see custom titlebar) 2. if you have a custom titlebar, then you have to use a custom menubar if you want a menubar

    :param menu_definition:      The Menu definition specified using lists (docs explain the format)
    :type menu_definition:       List[List[Tuple[str, List[str]]]
    :param disabled_text_color:  color to use for text when item is disabled. Can be in #RRGGBB format or a color name "black"
    :type disabled_text_color:   (str)
    :param bar_font:             specifies the font family, size to be used for the chars in the bar itself
    :type bar_font:              (str or (str, int[, str]) or None)
    :param font:                 specifies the font family, size to be used for the menu items
    :type font:                  (str or (str, int[, str]) or None)
    :param tearoff:              if True, then can tear the menu off from the window ans use as a floating window. Very cool effect
    :type tearoff:               (bool)
    :param pad:                  Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int).  TIP - 0 will make flush with titlebar
    :type pad:                   (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                    Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                     (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param background_color:     color to use for background of the menus that are displayed after making a section. Can be in #RRGGBB format or a color name "black". Defaults to the color of the bar text
    :type background_color:      (str)
    :param text_color:           color to use for the text of the many items in the displayed menus. Can be in #RRGGBB format or a color name "black". Defaults to the bar background
    :type text_color:            (str)
    :param bar_background_color: color to use for the menubar. Can be in #RRGGBB format or a color name "black". Defaults to theme's button text color
    :type bar_background_color:  (str)
    :param bar_text_color:       color to use for the menu items text when item is disabled. Can be in #RRGGBB format or a color name "black". Defaults to theme's button background color
    :type bar_text_color:        (str)
    :param key:                  Value that uniquely identifies this element from all other elements. Used when Finding an element or in return values. Must be unique to the window
    :type key:                   str | int | tuple | object
    :param k:                    Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                     str | int | tuple | object
    :returns:                    A Column element that has a series of ButtonMenu elements
    :rtype:                      Column
    """

    bar_bg = bar_background_color if bar_background_color is not None else theme_button_color()[0]
    bar_text = bar_text_color if bar_text_color is not None else theme_button_color()[1]
    menu_bg = background_color if background_color is not None else bar_text
    menu_text = text_color if text_color is not None else bar_bg
    pad = pad if pad is not None else p

    row = []
    for menu in menu_definition:
        text = menu[0]
        if MENU_SHORTCUT_CHARACTER in text:
            text = text.replace(MENU_SHORTCUT_CHARACTER, '')
        if text.startswith(MENU_DISABLED_CHARACTER):
            disabled = True
            text = text[len(MENU_DISABLED_CHARACTER):]
        else:
            disabled = False

        button_menu = ButtonMenu(text, menu, border_width=0, button_color=(bar_text, bar_bg), key=text, pad=(0, 0), disabled=disabled, font=bar_font,
                                 item_font=font, disabled_text_color=disabled_text_color, text_color=menu_text, background_color=menu_bg, tearoff=tearoff)
        button_menu.part_of_custom_menubar = True
        button_menu.custom_menubar_key = key if key is not None else k
        row += [button_menu]
    return Column([row], pad=pad, background_color=bar_bg, expand_x=True, key=key if key is not None else k)


# -------------------------  FOLDER BROWSE Element lazy function  ------------------------- #
def FolderBrowse(button_text='Browse', target=(ThisRow, -1), initial_folder=None, tooltip=None, size=(None, None), s=(None, None),
                 auto_size_button=None, button_color=None, disabled=False, change_submits=False, enable_events=False,
                 font=None, pad=None, p=None, key=None, k=None, visible=True, metadata=None, expand_x=False, expand_y=False):
    """
    :param button_text:      text in the button (Default value = 'Browse')
    :type button_text:       (str)
    :param target:           target for the button (Default value = (ThisRow, -1))
    :type target:            str | (int, int)
    :param initial_folder:   starting path for folders and files
    :type initial_folder:    (str)
    :param tooltip:          text, that will appear when mouse hovers over the element
    :type tooltip:           (str)
    :param size:             (w,h) w=characters-wide, h=rows-high
    :type size:              (int, int)
    :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                 (int, int)  | (None, None) | int
    :param auto_size_button: True if button size is determined by button text
    :type auto_size_button:  (bool)
    :param button_color:     button color (foreground, background)
    :type button_color:      (str, str) | str
    :param disabled:         set disable state for element (Default = False)
    :type disabled:          (bool)
    :param change_submits:   If True, pressing Enter key submits window (Default = False)
    :type enable_events:     (bool)
    :param enable_events:    Turns on the element specific events.(Default = False)
    :type enable_events:     (bool)
    :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:              (str or (str, int[, str]) or None)
    :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:              Used with window.find_element and with return values to uniquely identify this element
    :type key:               str | int | tuple | object
    :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                 str | int | tuple | object
    :param visible:          set initial visibility state of the Button
    :type visible:           (bool)
    :param metadata:         Anything you want to store along with this button
    :type metadata:          (Any)
    :param expand_x:         If True Element will expand in the Horizontal directions
    :type expand_x:          (bool)
    :param expand_y:         If True Element will expand in the Vertical directions
    :type expand_y:          (bool)
    :return:                 The Button created
    :rtype:                  (Button)
    """

    return Button(button_text=button_text, button_type=BUTTON_TYPE_BROWSE_FOLDER, target=target,
                  initial_folder=initial_folder, tooltip=tooltip, size=size, s=s, auto_size_button=auto_size_button,
                  disabled=disabled, button_color=button_color, change_submits=change_submits,
                  enable_events=enable_events, font=font, pad=pad, p=p, key=key, k=k, visible=visible, metadata=metadata, expand_x=expand_x, expand_y=expand_y)


# -------------------------  FILE BROWSE Element lazy function  ------------------------- #
def FileBrowse(button_text='Browse', target=(ThisRow, -1), file_types=FILE_TYPES_ALL_FILES, initial_folder=None,
               tooltip=None, size=(None, None), s=(None, None), auto_size_button=None, button_color=None, change_submits=False,
               enable_events=False, font=None, disabled=False,
               pad=None, p=None, key=None, k=None, visible=True, metadata=None, expand_x=False, expand_y=False):
    """

    :param button_text:      text in the button (Default value = 'Browse')
    :type button_text:       (str)
    :param target:           key or (row,col) target for the button (Default value = (ThisRow, -1))
    :type target:            str | (int, int)
    :param file_types:       filter file types Default value = (("ALL Files", "*.* *"),).
    :type file_types:        Tuple[(str, str), ...]
    :param initial_folder:   starting path for folders and files
    :type initial_folder:
    :param tooltip:          text, that will appear when mouse hovers over the element
    :type tooltip:           (str)
    :param size:             (w,h) w=characters-wide, h=rows-high
    :type size:              (int, int)
    :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                 (int, int)  | (None, None) | int
    :param auto_size_button: True if button size is determined by button text
    :type auto_size_button:  (bool)
    :param button_color:     button color (foreground, background)
    :type button_color:      (str, str) | str
    :param change_submits:   If True, pressing Enter key submits window (Default = False)
    :type change_submits:    (bool)
    :param enable_events:    Turns on the element specific events.(Default = False)
    :type enable_events:     (bool)
    :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:              (str or (str, int[, str]) or None)
    :param disabled:         set disable state for element (Default = False)
    :type disabled:          (bool)
    :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:              key for uniquely identify this element (for window.find_element)
    :type key:               str | int | tuple | object
    :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                 str | int | tuple | object
    :param visible:          set initial visibility state of the Button
    :type visible:           (bool)
    :param metadata:         Anything you want to store along with this button
    :type metadata:          (Any)
    :param expand_x:         If True Element will expand in the Horizontal directions
    :type expand_x:          (bool)
    :param expand_y:         If True Element will expand in the Vertical directions
    :type expand_y:          (bool)
    :return:                 returns a button
    :rtype:                  (Button)
    """
    return Button(button_text=button_text, button_type=BUTTON_TYPE_BROWSE_FILE, target=target, file_types=file_types,
                  initial_folder=initial_folder, tooltip=tooltip, size=size, s=s, auto_size_button=auto_size_button,
                  change_submits=change_submits, enable_events=enable_events, disabled=disabled,
                  button_color=button_color, font=font, pad=pad, p=p, key=key, k=k, visible=visible, metadata=metadata, expand_x=expand_x, expand_y=expand_y)


# -------------------------  FILES BROWSE Element (Multiple file selection) lazy function  ------------------------- #
def FilesBrowse(button_text='Browse', target=(ThisRow, -1), file_types=FILE_TYPES_ALL_FILES, disabled=False,
                initial_folder=None, tooltip=None, size=(None, None), s=(None, None), auto_size_button=None, button_color=None,
                change_submits=False, enable_events=False,
                font=None, pad=None, p=None, key=None, k=None, visible=True, files_delimiter=BROWSE_FILES_DELIMITER, metadata=None, expand_x=False, expand_y=False):
    """
    Allows browsing of multiple files. File list is returned as a single list with the delimiter defined using the files_delimiter parameter.

    :param button_text:      text in the button (Default value = 'Browse')
    :type button_text:       (str)
    :param target:           key or (row,col) target for the button (Default value = (ThisRow, -1))
    :type target:            str | (int, int)
    :param file_types:       Default value = (("ALL Files", "*.* *"),).
    :type file_types:        Tuple[(str, str), ...]
    :param disabled:         set disable state for element (Default = False)
    :type disabled:          (bool)
    :param initial_folder:   starting path for folders and files
    :type initial_folder:    (str)
    :param tooltip:          text, that will appear when mouse hovers over the element
    :type tooltip:           (str)
    :param size:             (w,h) w=characters-wide, h=rows-high
    :type size:              (int, int)
    :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                 (int, int)  | (None, None) | int
    :param auto_size_button: True if button size is determined by button text
    :type auto_size_button:  (bool)
    :param button_color:     button color (foreground, background)
    :type button_color:      (str, str) | str
    :param change_submits:   If True, pressing Enter key submits window (Default = False)
    :type change_submits:    (bool)
    :param enable_events:    Turns on the element specific events.(Default = False)
    :type enable_events:     (bool)
    :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:              (str or (str, int[, str]) or None)
    :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:              key for uniquely identify this element (for window.find_element)
    :type key:               str | int | tuple | object
    :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                 str | int | tuple | object
    :param visible:          set initial visibility state of the Button
    :type visible:           (bool)
    :param files_delimiter:  String to place between files when multiple files are selected. Normally a ;
    :type files_delimiter:   str
    :param metadata:         Anything you want to store along with this button
    :type metadata:          (Any)
    :param expand_x:         If True Element will expand in the Horizontal directions
    :type expand_x:          (bool)
    :param expand_y:         If True Element will expand in the Vertical directions
    :type expand_y:          (bool)
    :return:                 returns a button
    :rtype:                  (Button)
    """
    button = Button(button_text=button_text, button_type=BUTTON_TYPE_BROWSE_FILES, target=target, file_types=file_types,
                    initial_folder=initial_folder, change_submits=change_submits, enable_events=enable_events,
                    tooltip=tooltip, size=size, s=s, auto_size_button=auto_size_button,
                    disabled=disabled, button_color=button_color, font=font, pad=pad, p=p, key=key, k=k, visible=visible, metadata=metadata, expand_x=expand_x, expand_y=expand_y)
    button._files_delimiter = files_delimiter
    return button


# -------------------------  FILE BROWSE Element lazy function  ------------------------- #
def FileSaveAs(button_text='Save As...', target=(ThisRow, -1), file_types=FILE_TYPES_ALL_FILES, initial_folder=None,
               default_extension='', disabled=False, tooltip=None, size=(None, None), s=(None, None), auto_size_button=None, button_color=None,
               change_submits=False, enable_events=False, font=None,
               pad=None, p=None, key=None, k=None, visible=True, metadata=None, expand_x=False, expand_y=False):
    """

    :param button_text:       text in the button (Default value = 'Save As...')
    :type button_text:        (str)
    :param target:            key or (row,col) target for the button (Default value = (ThisRow, -1))
    :type target:             str | (int, int)
    :param file_types:        Default value = (("ALL Files", "*.* *"),).
    :type file_types:         Tuple[(str, str), ...]
    :param default_extension: If no extension entered by user, add this to filename (only used in saveas dialogs)
    :type default_extension:  (str)
    :param initial_folder:    starting path for folders and files
    :type initial_folder:     (str)
    :param disabled:          set disable state for element (Default = False)
    :type disabled:           (bool)
    :param tooltip:           text, that will appear when mouse hovers over the element
    :type tooltip:            (str)
    :param size:              (w,h) w=characters-wide, h=rows-high
    :type size:               (int, int)
    :param s:                 Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                  (int, int)  | (None, None) | int
    :param auto_size_button:  True if button size is determined by button text
    :type auto_size_button:   (bool)
    :param button_color:      button color (foreground, background)
    :type button_color:       (str, str) | str
    :param change_submits:    If True, pressing Enter key submits window (Default = False)
    :type change_submits:     (bool)
    :param enable_events:     Turns on the element specific events.(Default = False)
    :type enable_events:      (bool)
    :param font:              specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:               (str or (str, int[, str]) or None)
    :param pad:               Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:                (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                 Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                  (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:               key for uniquely identify this element (for window.find_element)
    :type key:                str | int | tuple | object
    :param k:                 Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                  str | int | tuple | object
    :param visible:           set initial visibility state of the Button
    :type visible:            (bool)
    :param metadata:          Anything you want to store along with this button
    :type metadata:           (Any)
    :param expand_x:          If True Element will expand in the Horizontal directions
    :type expand_x:           (bool)
    :param expand_y:          If True Element will expand in the Vertical directions
    :type expand_y:           (bool)        :return:                  returns a button
    :rtype:                   (Button)
    """
    return Button(button_text=button_text, button_type=BUTTON_TYPE_SAVEAS_FILE, target=target, file_types=file_types,
                  initial_folder=initial_folder, default_extension=default_extension, tooltip=tooltip, size=size, s=s, disabled=disabled,
                  auto_size_button=auto_size_button, button_color=button_color, change_submits=change_submits,
                  enable_events=enable_events, font=font, pad=pad, p=p, key=key, k=k, visible=visible, metadata=metadata, expand_x=expand_x, expand_y=expand_y)


# -------------------------  SAVE AS Element lazy function  ------------------------- #
def SaveAs(button_text='Save As...', target=(ThisRow, -1), file_types=FILE_TYPES_ALL_FILES, initial_folder=None, default_extension='',
           disabled=False, tooltip=None, size=(None, None), s=(None, None), auto_size_button=None, button_color=None,
           change_submits=False, enable_events=False, font=None,
           pad=None, p=None, key=None, k=None, visible=True, metadata=None, expand_x=False, expand_y=False):
    """

    :param button_text:       text in the button (Default value = 'Save As...')
    :type button_text:        (str)
    :param target:            key or (row,col) target for the button (Default value = (ThisRow, -1))
    :type target:             str | (int, int)
    :param file_types:        Default value = (("ALL Files", "*.* *"),).
    :type file_types:         Tuple[(str, str), ...]
    :param default_extension: If no extension entered by user, add this to filename (only used in saveas dialogs)
    :type default_extension:  (str)
    :param initial_folder:    starting path for folders and files
    :type initial_folder:     (str)
    :param disabled:          set disable state for element (Default = False)
    :type disabled:           (bool)
    :param tooltip:           text, that will appear when mouse hovers over the element
    :type tooltip:            (str)
    :param size:              (w,h) w=characters-wide, h=rows-high
    :type size:               (int, int)
    :param s:                 Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                  (int, int)  | (None, None) | int
    :param auto_size_button:  True if button size is determined by button text
    :type auto_size_button:   (bool)
    :param button_color:      button color (foreground, background)
    :type button_color:       (str, str) or str
    :param change_submits:    If True, pressing Enter key submits window (Default = False)
    :type change_submits:     (bool)
    :param enable_events:     Turns on the element specific events.(Default = False)
    :type enable_events:      (bool)
    :param font:              specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:               (str or (str, int[, str]) or None)
    :param pad:               Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:                (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                 Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                  (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int    :param key:               key for uniquely identify this element (for window.find_element)
    :type key:                str | int | tuple | object
    :param k:                 Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                  str | int | tuple | object
    :param visible:           set initial visibility state of the Button
    :type visible:            (bool)
    :param metadata:          Anything you want to store along with this button
    :type metadata:           (Any)
    :param expand_x:          If True Element will expand in the Horizontal directions
    :type expand_x:           (bool)
    :param expand_y:          If True Element will expand in the Vertical directions
    :type expand_y:           (bool)
    :return:                  returns a button
    :rtype:                   (Button)
    """
    return Button(button_text=button_text, button_type=BUTTON_TYPE_SAVEAS_FILE, target=target, file_types=file_types,
                  initial_folder=initial_folder, default_extension=default_extension, tooltip=tooltip, size=size, s=s, disabled=disabled,
                  auto_size_button=auto_size_button, button_color=button_color, change_submits=change_submits,
                  enable_events=enable_events, font=font, pad=pad, p=p, key=key, k=k, visible=visible, metadata=metadata, expand_x=expand_x, expand_y=expand_y)


# -------------------------  SAVE BUTTON Element lazy function  ------------------------- #
def Save(button_text='Save', size=(None, None), s=(None, None), auto_size_button=None, button_color=None, bind_return_key=True,
         disabled=False, tooltip=None, font=None, focus=False, pad=None, p=None, key=None, k=None, visible=True, metadata=None, expand_x=False, expand_y=False):
    """

    :param button_text:      text in the button (Default value = 'Save')
    :type button_text:       (str)
    :param size:             (w,h) w=characters-wide, h=rows-high
    :type size:              (int, int)
    :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                 (int, int)  | (None, None) | int
    :param auto_size_button: True if button size is determined by button text
    :type auto_size_button:  (bool)
    :param button_color:     button color (foreground, background)
    :type button_color:      (str, str) | str
    :param bind_return_key:  (Default = True) If True, this button will appear to be clicked when return key is pressed in other elements such as Input and elements with return key options
    :type bind_return_key:   (bool)
    :param disabled:         set disable state for element (Default = False)
    :type disabled:          (bool)
    :param tooltip:          text, that will appear when mouse hovers over the element
    :type tooltip:           (str)
    :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:              (str or (str, int[, str]) or None)
    :param focus:            if focus should be set to this
    :type focus:             idk_yetReally
    :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:              key for uniquely identify this element (for window.find_element)
    :type key:               str | int | tuple | object
    :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                 str | int | tuple | object
    :param visible:          set initial visibility state of the Button
    :type visible:           (bool)
    :param metadata:         Anything you want to store along with this button
    :type metadata:          (Any)
    :param expand_x:         If True Element will expand in the Horizontal directions
    :type expand_x:          (bool)
    :param expand_y:         If True Element will expand in the Vertical directions
    :type expand_y:          (bool)
    :return:                 returns a button
    :rtype:                  (Button)
    """
    return Button(button_text=button_text, button_type=BUTTON_TYPE_READ_FORM, tooltip=tooltip, size=size, s=s,
                  auto_size_button=auto_size_button, button_color=button_color, font=font, disabled=disabled,
                  bind_return_key=bind_return_key, focus=focus, pad=pad, p=p, key=key, k=k, visible=visible, metadata=metadata, expand_x=expand_x, expand_y=expand_y)


# -------------------------  SUBMIT BUTTON Element lazy function  ------------------------- #
def Submit(button_text='Submit', size=(None, None), s=(None, None), auto_size_button=None, button_color=None, disabled=False,
           bind_return_key=True, tooltip=None, font=None, focus=False, pad=None, p=None, key=None, k=None, visible=True, metadata=None, expand_x=False, expand_y=False):
    """

    :param button_text:      text in the button (Default value = 'Submit')
    :type button_text:       (str)
    :param size:             (w,h) w=characters-wide, h=rows-high
    :type size:              (int, int)
    :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                 (int, int)  | (None, None) | int
    :param auto_size_button: True if button size is determined by button text
    :type auto_size_button:  (bool)
    :param button_color:     button color (foreground, background)
    :type button_color:      (str, str) | str
    :param disabled:         set disable state for element (Default = False)
    :type disabled:          (bool)
    :param bind_return_key:  (Default = True) If True, this button will appear to be clicked when return key is pressed in other elements such as Input and elements with return key options
    :type bind_return_key:   (bool)
    :param tooltip:          text, that will appear when mouse hovers over the element
    :type tooltip:           (str)
    :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:              (str or (str, int[, str]) or None)
    :param focus:            if focus should be set to this
    :type focus:             idk_yetReally
    :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:              key for uniquely identify this element (for window.find_element)
    :type key:               str | int | tuple | object
    :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                 str | int | tuple | object
    :param visible:          set initial visibility state of the Button
    :type visible:           (bool)
    :param metadata:         Anything you want to store along with this button
    :type metadata:          (Any)
    :param expand_x:         If True Element will expand in the Horizontal directions
    :type expand_x:          (bool)
    :param expand_y:         If True Element will expand in the Vertical directions
    :type expand_y:          (bool)
    :return:                 returns a button
    :rtype:                  (Button)
    """
    return Button(button_text=button_text, button_type=BUTTON_TYPE_READ_FORM, tooltip=tooltip, size=size, s=s,
                  auto_size_button=auto_size_button, button_color=button_color, font=font, disabled=disabled,
                  bind_return_key=bind_return_key, focus=focus, pad=pad, p=p, key=key, k=k, visible=visible, metadata=metadata, expand_x=expand_x, expand_y=expand_y)


# -------------------------  OPEN BUTTON Element lazy function  ------------------------- #
# -------------------------  OPEN BUTTON Element lazy function  ------------------------- #
def Open(button_text='Open', size=(None, None), s=(None, None), auto_size_button=None, button_color=None, disabled=False,
         bind_return_key=True, tooltip=None, font=None, focus=False, pad=None, p=None, key=None, k=None, visible=True, metadata=None, expand_x=False, expand_y=False):
    """

    :param button_text:      text in the button (Default value = 'Open')
    :type button_text:       (str)
    :param size:             (w,h) w=characters-wide, h=rows-high
    :type size:              (int, int)
    :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                 (int, int)  | (None, None) | int
    :param auto_size_button: True if button size is determined by button text
    :type auto_size_button:  (bool)
    :param button_color:     button color (foreground, background)
    :type button_color:      (str, str) | str
    :param disabled:         set disable state for element (Default = False)
    :type disabled:          (bool)
    :param bind_return_key:  (Default = True) If True, this button will appear to be clicked when return key is pressed in other elements such as Input and elements with return key options
    :type bind_return_key:   (bool)
    :param tooltip:          text, that will appear when mouse hovers over the element
    :type tooltip:           (str)
    :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:              (str or (str, int[, str]) or None)
    :param focus:            if focus should be set to this
    :type focus:             idk_yetReally
    :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:              key for uniquely identify this element (for window.find_element)
    :type key:               str | int | tuple | object
    :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                 str | int | tuple | object
    :param visible:          set initial visibility state of the Button
    :type visible:           (bool)
    :param metadata:         Anything you want to store along with this button
    :type metadata:          (Any)
    :param expand_x:         If True Element will expand in the Horizontal directions
    :type expand_x:          (bool)
    :param expand_y:         If True Element will expand in the Vertical directions
    :type expand_y:          (bool)
    :return:                 returns a button
    :rtype:                  (Button)
    """
    return Button(button_text=button_text, button_type=BUTTON_TYPE_READ_FORM, tooltip=tooltip, size=size, s=s,
                  auto_size_button=auto_size_button, button_color=button_color, font=font, disabled=disabled,
                  bind_return_key=bind_return_key, focus=focus, pad=pad, p=p, key=key, k=k, visible=visible, metadata=metadata, expand_x=expand_x, expand_y=expand_y)


# -------------------------  OK BUTTON Element lazy function  ------------------------- #
def OK(button_text='OK', size=(None, None), s=(None, None), auto_size_button=None, button_color=None, disabled=False,
       bind_return_key=True, tooltip=None, font=None, focus=False, pad=None, p=None, key=None, k=None, visible=True, metadata=None, expand_x=False, expand_y=False):
    """

    :param button_text:      text in the button (Default value = 'OK')
    :type button_text:       (str)
    :param size:             (w,h) w=characters-wide, h=rows-high
    :type size:              (int, int)
    :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                 (int, int)  | (None, None) | int
    :param auto_size_button: True if button size is determined by button text
    :type auto_size_button:  (bool)
    :param button_color:     button color (foreground, background)
    :type button_color:      (str, str) | str
    :param disabled:         set disable state for element (Default = False)
    :type disabled:          (bool)
    :param bind_return_key:  (Default = True) If True, this button will appear to be clicked when return key is pressed in other elements such as Input and elements with return key options
    :type bind_return_key:   (bool)
    :param tooltip:          text, that will appear when mouse hovers over the element
    :type tooltip:           (str)
    :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:              (str or (str, int[, str]) or None)
    :param focus:            if focus should be set to this
    :type focus:             idk_yetReally
    :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:              key for uniquely identify this element (for window.find_element)
    :type key:               str | int | tuple | object
    :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                 str | int | tuple | object
    :param visible:          set initial visibility state of the Button
    :type visible:           (bool)
    :param metadata:         Anything you want to store along with this button
    :type metadata:          (Any)
    :param expand_x:         If True Element will expand in the Horizontal directions
    :type expand_x:          (bool)
    :param expand_y:         If True Element will expand in the Vertical directions
    :type expand_y:          (bool)
    :return:                 returns a button
    :rtype:                  (Button)
    """
    return Button(button_text=button_text, button_type=BUTTON_TYPE_READ_FORM, tooltip=tooltip, size=size, s=s,
                  auto_size_button=auto_size_button, button_color=button_color, font=font, disabled=disabled,
                  bind_return_key=bind_return_key, focus=focus, pad=pad, p=p, key=key, k=k, visible=visible, metadata=metadata, expand_x=expand_x, expand_y=expand_y)


# -------------------------  YES BUTTON Element lazy function  ------------------------- #
def Ok(button_text='Ok', size=(None, None), s=(None, None), auto_size_button=None, button_color=None, disabled=False,
       bind_return_key=True, tooltip=None, font=None, focus=False, pad=None, p=None, key=None, k=None, visible=True, metadata=None, expand_x=False, expand_y=False):
    """

    :param button_text:      text in the button (Default value = 'Ok')
    :type button_text:       (str)
    :param size:             (w,h) w=characters-wide, h=rows-high
    :type size:              (int, int)
    :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                 (int, int)  | (None, None) | int
    :param auto_size_button: True if button size is determined by button text
    :type auto_size_button:  (bool)
    :param button_color:     button color (foreground, background)
    :type button_color:      (str, str) | str
    :param disabled:         set disable state for element (Default = False)
    :type disabled:          (bool)
    :param bind_return_key:  (Default = True) If True, this button will appear to be clicked when return key is pressed in other elements such as Input and elements with return key options
    :type bind_return_key:   (bool)
    :param tooltip:          text, that will appear when mouse hovers over the element
    :type tooltip:           (str)
    :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:              (str or (str, int[, str]) or None)
    :param focus:            if focus should be set to this
    :type focus:             idk_yetReally
    :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:              key for uniquely identify this element (for window.find_element)
    :type key:               str | int | tuple | object
    :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                 str | int | tuple | object
    :param visible:          set initial visibility state of the Button
    :type visible:           (bool)
    :param metadata:         Anything you want to store along with this button
    :type metadata:          (Any)
    :param expand_x:         If True Element will expand in the Horizontal directions
    :type expand_x:          (bool)
    :param expand_y:         If True Element will expand in the Vertical directions
    :type expand_y:          (bool)
    :return:                 returns a button
    :rtype:                  (Button)
    """
    return Button(button_text=button_text, button_type=BUTTON_TYPE_READ_FORM, tooltip=tooltip, size=size, s=s,
                  auto_size_button=auto_size_button, button_color=button_color, font=font, disabled=disabled,
                  bind_return_key=bind_return_key, focus=focus, pad=pad, p=p, key=key, k=k, visible=visible, metadata=metadata, expand_x=expand_x, expand_y=expand_y)


# -------------------------  CANCEL BUTTON Element lazy function  ------------------------- #
def Cancel(button_text='Cancel', size=(None, None), s=(None, None), auto_size_button=None, button_color=None, disabled=False,
           tooltip=None, font=None, bind_return_key=False, focus=False, pad=None, p=None, key=None, k=None, visible=True, metadata=None, expand_x=False, expand_y=False):
    """

    :param button_text:      text in the button (Default value = 'Cancel')
    :type button_text:       (str)
    :param size:             (w,h) w=characters-wide, h=rows-high
    :type size:              (int, int)
    :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                 (int, int)  | (None, None) | int
    :param auto_size_button: True if button size is determined by button text
    :type auto_size_button:  (bool)
    :param button_color:     button color (foreground, background)
    :type button_color:      (str, str) | str
    :param disabled:         set disable state for element (Default = False)
    :type disabled:          (bool)
    :param tooltip:          text, that will appear when mouse hovers over the element
    :type tooltip:           (str)
    :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:              (str or (str, int[, str]) or None)
    :param bind_return_key:  (Default = False) If True, this button will appear to be clicked when return key is pressed in other elements such as Input and elements with return key options
    :type bind_return_key:   (bool)
    :param focus:            if focus should be set to this
    :type focus:
    :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:              key for uniquely identify this element (for window.find_element)
    :type key:               str | int | tuple | object
    :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                 str | int | tuple | object
    :param visible:          set initial visibility state of the Button
    :type visible:           (bool)
    :param metadata:         Anything you want to store along with this button
    :type metadata:          (Any)
    :param expand_x:         If True Element will expand in the Horizontal directions
    :type expand_x:          (bool)
    :param expand_y:         If True Element will expand in the Vertical directions
    :type expand_y:          (bool)
    :return:                 returns a button
    :rtype:                  (Button)
    """
    return Button(button_text=button_text, button_type=BUTTON_TYPE_READ_FORM, tooltip=tooltip, size=size, s=s,
                  auto_size_button=auto_size_button, button_color=button_color, font=font, disabled=disabled,
                  bind_return_key=bind_return_key, focus=focus, pad=pad, p=p, key=key, k=k, visible=visible, metadata=metadata, expand_x=expand_x, expand_y=expand_y)


# -------------------------  QUIT BUTTON Element lazy function  ------------------------- #
def Quit(button_text='Quit', size=(None, None), s=(None, None), auto_size_button=None, button_color=None, disabled=False, tooltip=None,
         font=None, bind_return_key=False, focus=False, pad=None, p=None, key=None, k=None, visible=True, metadata=None, expand_x=False, expand_y=False):
    """

    :param button_text:      text in the button (Default value = 'Quit')
    :type button_text:       (str)
    :param size:             (w,h) w=characters-wide, h=rows-high
    :type size:              (int, int)
    :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                 (int, int)  | (None, None) | int
    :param auto_size_button: True if button size is determined by button text
    :type auto_size_button:  (bool)
    :param button_color:     button color (foreground, background)
    :type button_color:      (str, str) | str
    :param disabled:         set disable state for element (Default = False)
    :type disabled:          (bool)
    :param tooltip:          text, that will appear when mouse hovers over the element
    :type tooltip:           (str)
    :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:              (str or (str, int[, str]) or None)
    :param bind_return_key:  (Default = False) If True, this button will appear to be clicked when return key is pressed in other elements such as Input and elements with return key options
    :type bind_return_key:   (bool)
    :param focus:            if focus should be set to this
    :type focus:             (bool)
    :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:              key for uniquely identify this element (for window.find_element)
    :type key:               str | int | tuple | object
    :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                 str | int | tuple | object
    :param visible:          set initial visibility state of the Button
    :type visible:           (bool)
    :param metadata:         Anything you want to store along with this button
    :type metadata:          (Any)
    :param expand_x:         If True Element will expand in the Horizontal directions
    :type expand_x:          (bool)
    :param expand_y:         If True Element will expand in the Vertical directions
    :type expand_y:          (bool)
    :return:                 returns a button
    :rtype:                  (Button)
    """
    return Button(button_text=button_text, button_type=BUTTON_TYPE_READ_FORM, tooltip=tooltip, size=size, s=s,
                  auto_size_button=auto_size_button, button_color=button_color, font=font, disabled=disabled,
                  bind_return_key=bind_return_key, focus=focus, pad=pad, p=p, key=key, k=k, visible=visible, metadata=metadata, expand_x=expand_x, expand_y=expand_y)


# -------------------------  Exit BUTTON Element lazy function  ------------------------- #
def Exit(button_text='Exit', size=(None, None), s=(None, None), auto_size_button=None, button_color=None, disabled=False, tooltip=None,
         font=None, bind_return_key=False, focus=False, pad=None, p=None, key=None, k=None, visible=True, metadata=None, expand_x=False, expand_y=False):
    """

    :param button_text:      text in the button (Default value = 'Exit')
    :type button_text:       (str)
    :param size:             (w,h) w=characters-wide, h=rows-high
    :type size:              (int, int)
    :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                 (int, int)  | (None, None) | int
    :param auto_size_button: True if button size is determined by button text
    :type auto_size_button:  (bool)
    :param button_color:     button color (foreground, background)
    :type button_color:      (str, str) | str
    :param disabled:         set disable state for element (Default = False)
    :type disabled:          (bool)
    :param tooltip:          text, that will appear when mouse hovers over the element
    :type tooltip:           (str)
    :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:              (str or (str, int[, str]) or None)
    :param bind_return_key:  (Default = False) If True, this button will appear to be clicked when return key is pressed in other elements such as Input and elements with return key options
    :type bind_return_key:   (bool)
    :param focus:            if focus should be set to this
    :type focus:
    :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:              key for uniquely identify this element (for window.find_element)
    :type key:               str | int | tuple | object
    :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                 str | int | tuple | object
    :param visible:          set initial visibility state of the Button
    :type visible:           (bool)
    :param metadata:         Anything you want to store along with this button
    :type metadata:          (Any)
    :param expand_x:         If True Element will expand in the Horizontal directions
    :type expand_x:          (bool)
    :param expand_y:         If True Element will expand in the Vertical directions
    :type expand_y:          (bool)
    :return:                 returns a button
    :rtype:                  (Button)
    """
    return Button(button_text=button_text, button_type=BUTTON_TYPE_READ_FORM, tooltip=tooltip, size=size, s=s,
                  auto_size_button=auto_size_button, button_color=button_color, font=font, disabled=disabled,
                  bind_return_key=bind_return_key, focus=focus, pad=pad, p=p, key=key, k=k, visible=visible, metadata=metadata, expand_x=expand_x, expand_y=expand_y)


# -------------------------  YES BUTTON Element lazy function  ------------------------- #
def Yes(button_text='Yes', size=(None, None), s=(None, None), auto_size_button=None, button_color=None, disabled=False, tooltip=None,
        font=None, bind_return_key=True, focus=False, pad=None, p=None, key=None, k=None, visible=True, metadata=None, expand_x=False, expand_y=False):
    """

    :param button_text:      text in the button (Default value = 'Yes')
    :type button_text:       (str)
    :param size:             (w,h) w=characters-wide, h=rows-high
    :type size:              (int, int)
    :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                 (int, int)  | (None, None) | int
    :param auto_size_button: True if button size is determined by button text
    :type auto_size_button:  (bool)
    :param button_color:     button color (foreground, background)
    :type button_color:      (str, str) | str
    :param disabled:         set disable state for element (Default = False)
    :type disabled:          (bool)
    :param tooltip:          text, that will appear when mouse hovers over the element
    :type tooltip:           (str)
    :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:              (str or (str, int[, str]) or None)
    :param bind_return_key:  (Default = True) If True, this button will appear to be clicked when return key is pressed in other elements such as Input and elements with return key options
    :type bind_return_key:   (bool)
    :param focus:            if focus should be set to this
    :type focus:
    :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:              key for uniquely identify this element (for window.find_element)
    :type key:               str | int | tuple | object
    :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                 str | int | tuple | object
    :param visible:          set initial visibility state of the Button
    :type visible:           (bool)
    :param metadata:         Anything you want to store along with this button
    :type metadata:          (Any)
    :param expand_x:         If True Element will expand in the Horizontal directions
    :type expand_x:          (bool)
    :param expand_y:         If True Element will expand in the Vertical directions
    :type expand_y:          (bool)
    :return:                 returns a button
    :rtype:                  (Button)
    """
    return Button(button_text=button_text, button_type=BUTTON_TYPE_READ_FORM, tooltip=tooltip, size=size, s=s,
                  auto_size_button=auto_size_button, button_color=button_color, font=font, disabled=disabled,
                  bind_return_key=bind_return_key, focus=focus, pad=pad, p=p, key=key, k=k, visible=visible, metadata=metadata, expand_x=expand_x, expand_y=expand_y)


# -------------------------  NO BUTTON Element lazy function  ------------------------- #
def No(button_text='No', size=(None, None), s=(None, None), auto_size_button=None, button_color=None, disabled=False, tooltip=None,
       font=None, bind_return_key=False, focus=False, pad=None, p=None, key=None, k=None, visible=True, metadata=None, expand_x=False, expand_y=False):
    """

    :param button_text:      text in the button (Default value = 'No')
    :type button_text:       (str)
    :param size:             (w,h) w=characters-wide, h=rows-high
    :type size:              (int, int)
    :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                 (int, int)  | (None, None) | int
    :param auto_size_button: True if button size is determined by button text
    :type auto_size_button:  (bool)
    :param button_color:     button color (foreground, background)
    :type button_color:      (str, str) | str
    :param disabled:         set disable state for element (Default = False)
    :type disabled:          (bool)
    :param tooltip:          text, that will appear when mouse hovers over the element
    :type tooltip:           (str)
    :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:              (str or (str, int[, str]) or None)
    :param bind_return_key:  (Default = False) If True, then the return key will cause a the Listbox to generate an event
    :type bind_return_key:   (bool)
    :param focus:            if focus should be set to this
    :type focus:
    :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:              key for uniquely identify this element (for window.find_element)
    :type key:               str | int | tuple | object
    :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                 str | int | tuple | object
    :param visible:          set initial visibility state of the Button
    :type visible:           (bool)
    :param metadata:         Anything you want to store along with this button
    :type metadata:          (Any)
    :param expand_x:         If True Element will expand in the Horizontal directions
    :type expand_x:          (bool)
    :param expand_y:         If True Element will expand in the Vertical directions
    :type expand_y:          (bool)
    :return:                 returns a button
    :rtype:                  (Button)
    """
    return Button(button_text=button_text, button_type=BUTTON_TYPE_READ_FORM, tooltip=tooltip, size=size, s=s,
                  auto_size_button=auto_size_button, button_color=button_color, font=font, disabled=disabled,
                  bind_return_key=bind_return_key, focus=focus, pad=pad, p=p, key=key, k=k, visible=visible, metadata=metadata, expand_x=expand_x, expand_y=expand_y)


# -------------------------  NO BUTTON Element lazy function  ------------------------- #
def Help(button_text='Help', size=(None, None), s=(None, None), auto_size_button=None, button_color=None, disabled=False, font=None,
         tooltip=None, bind_return_key=False, focus=False, pad=None, p=None, key=None, k=None, visible=True, metadata=None, expand_x=False, expand_y=False):
    """

    :param button_text:      text in the button (Default value = 'Help')
    :type button_text:       (str)
    :param size:             (w,h) w=characters-wide, h=rows-high
    :type size:              (int, int)
    :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                 (int, int)  | (None, None) | int
    :param auto_size_button: True if button size is determined by button text
    :type auto_size_button:  (bool)
    :param button_color:     button color (foreground, background)
    :type button_color:      (str, str) | str
    :param disabled:         set disable state for element (Default = False)
    :type disabled:          (bool)
    :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:              (str or (str, int[, str]) or None)
    :param tooltip:          text, that will appear when mouse hovers over the element
    :type tooltip:           (str)
    :param bind_return_key:  (Default = False) If True, this button will appear to be clicked when return key is pressed in other elements such as Input and elements with return key options
    :type bind_return_key:   (bool)
    :param focus:            if focus should be set to this
    :type focus:
    :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:              key for uniquely identify this element (for window.find_element)
    :type key:               str | int | tuple | object
    :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                 str | int | tuple | object
    :param visible:          set initial visibility state of the Button
    :type visible:           (bool)
    :param metadata:         Anything you want to store along with this button
    :type metadata:          (Any)
    :param expand_x:         If True Element will expand in the Horizontal directions
    :type expand_x:          (bool)
    :param expand_y:         If True Element will expand in the Vertical directions
    :type expand_y:          (bool)
    :return:                 returns a button
    :rtype:                  (Button)
    """
    return Button(button_text=button_text, button_type=BUTTON_TYPE_READ_FORM, tooltip=tooltip, size=size, s=s,
                  auto_size_button=auto_size_button, button_color=button_color, font=font, disabled=disabled,
                  bind_return_key=bind_return_key, focus=focus, pad=pad, p=p, key=key, k=k, visible=visible, metadata=metadata, expand_x=expand_x, expand_y=expand_y)


# -------------------------  NO BUTTON Element lazy function  ------------------------- #
def Debug(button_text='', size=(None, None), s=(None, None), auto_size_button=None, button_color=None, disabled=False, font=None,
          tooltip=None, bind_return_key=False, focus=False, pad=None, p=None, key=None, k=None, visible=True, metadata=None, expand_x=False, expand_y=False):
    """
    This Button has been changed in how it works!!
    Your button has been replaced with a normal button that has the PySimpleGUI Debugger buggon logo on it.
    In your event loop, you will need to check for the event of this button and then call:
            show_debugger_popout_window()
    :param button_text:      text in the button (Default value = '')
    :type button_text:       (str)
    :param size:             (w,h) w=characters-wide, h=rows-high
    :type size:              (int, int)
    :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                 (int, int)  | (None, None) | int
    :param auto_size_button: True if button size is determined by button text
    :type auto_size_button:  (bool)
    :param button_color:     button color (foreground, background)
    :type button_color:      (str, str) | str
    :param disabled:         set disable state for element (Default = False)
    :type disabled:          (bool)
    :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:              (str or (str, int[, str]) or None)
    :param tooltip:          text, that will appear when mouse hovers over the element
    :type tooltip:           (str)
    :param bind_return_key:  (Default = False) If True, this button will appear to be clicked when return key is pressed in other elements such as Input and elements with return key options
    :type bind_return_key:   (bool)
    :param focus:            if focus should be set to this
    :type focus:
    :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:              key for uniquely identify this element (for window.find_element)
    :type key:               str | int | tuple | object
    :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                 str | int | tuple | object
    :param visible:          set initial visibility state of the Button
    :type visible:           (bool)
    :param metadata:         Anything you want to store along with this button
    :type metadata:          (Any)
    :param expand_x:         If True Element will expand in the Horizontal directions
    :type expand_x:          (bool)
    :param expand_y:         If True Element will expand in the Vertical directions
    :type expand_y:          (bool)
    :return:                 returns a button
    :rtype:                  (Button)
    """


    user_key = key if key is not None else k if k is not None else button_text

    return Button(button_text='', button_type=BUTTON_TYPE_READ_FORM, tooltip=tooltip, size=size, s=s,
                  auto_size_button=auto_size_button, button_color=theme_button_color(), font=font, disabled=disabled,
                  bind_return_key=bind_return_key, focus=focus, pad=pad, p=p, key=user_key, k=k, visible=visible, image_data=PSG_DEBUGGER_LOGO,
                  image_subsample=2, border_width=0, metadata=metadata, expand_x=expand_x, expand_y=expand_y)


# -------------------------  GENERIC BUTTON Element lazy function  ------------------------- #
def SimpleButton(button_text, image_filename=None, image_data=None, image_size=(None, None), image_subsample=None,
                 border_width=None, tooltip=None, size=(None, None), s=(None, None), auto_size_button=None, button_color=None,
                 font=None, bind_return_key=False, disabled=False, focus=False, pad=None, p=None, key=None, k=None, metadata=None, expand_x=False, expand_y=False):
    """
    DEPIRCATED

    This Button should not be used.

    :param button_text:      text in the button
    :type button_text:       (str)
    :param image_filename:   image filename if there is a button image
    :type image_filename:    image filename if there is a button image
    :param image_data:       in-RAM image to be displayed on button
    :type image_data:        in-RAM image to be displayed on button
    :param image_size:       image size (O.K.)
    :type image_size:        (Default = (None))
    :param image_subsample:  amount to reduce the size of the image
    :type image_subsample:   amount to reduce the size of the image
    :param tooltip:          text, that will appear when mouse hovers over the element
    :type tooltip:           (str)
    :param size:             (w,h) w=characters-wide, h=rows-high
    :type size:              (int, int)
    :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                 (int, int)  | (None, None) | int
    :param auto_size_button: True if button size is determined by button text
    :type auto_size_button:  (bool)
    :param button_color:     button color (foreground, background)
    :type button_color:      (str, str) | str
    :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:              (str or (str, int[, str]) or None)
    :param bind_return_key:  (Default = False) If True, this button will appear to be clicked when return key is pressed in other elements such as Input and elements with return key options
    :type bind_return_key:   (bool)
    :param disabled:         set disable state for element (Default = False)
    :type disabled:          (bool)
    :param focus:            if focus should be set to this
    :type focus:             idk_yetReally
    :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:              key for uniquely identify this element (for window.find_element)
    :type key:               str | int | tuple | object
    :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                 str | int | tuple | object
    :param metadata:         Anything you want to store along with this button
    :type metadata:          (Any)
    :param expand_x:         If True Element will expand in the Horizontal directions
    :type expand_x:          (bool)
    :param expand_y:         If True Element will expand in the Vertical directions
    :type expand_y:          (bool)
    :return:                 returns a button
    :rtype:                  (Button)
    """
    return Button(button_text=button_text, button_type=BUTTON_TYPE_CLOSES_WIN, image_filename=image_filename,
                  image_data=image_data, image_size=image_size, image_subsample=image_subsample,
                  border_width=border_width, tooltip=tooltip, disabled=disabled, size=size, s=s,
                  auto_size_button=auto_size_button, button_color=button_color, font=font,
                  bind_return_key=bind_return_key, focus=focus, pad=pad, p=p, key=key, k=k, metadata=metadata, expand_x=expand_x, expand_y=expand_y)


# -------------------------  CLOSE BUTTON Element lazy function  ------------------------- #
def CloseButton(button_text, image_filename=None, image_data=None, image_size=(None, None), image_subsample=None,
                border_width=None, tooltip=None, size=(None, None), s=(None, None), auto_size_button=None, button_color=None, font=None,
                bind_return_key=False, disabled=False, focus=False, pad=None, p=None, key=None, k=None, metadata=None, expand_x=False, expand_y=False):
    """
    DEPRICATED

    This button should not be used.  Instead explicitly close your windows by calling window.close() or by using
    the close parameter in window.read

    :param button_text:      text in the button
    :type button_text:       (str)
    :param image_filename:   image filename if there is a button image
    :type image_filename:    image filename if there is a button image
    :param image_data:       in-RAM image to be displayed on button
    :type image_data:        in-RAM image to be displayed on button
    :param image_size:       image size (O.K.)
    :type image_size:        (Default = (None))
    :param image_subsample:  amount to reduce the size of the image
    :type image_subsample:   amount to reduce the size of the image
    :param tooltip:          text, that will appear when mouse hovers over the element
    :type tooltip:           (str)
    :param size:             (w,h) w=characters-wide, h=rows-high
    :type size:              (int, int)
    :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                 (int, int)  | (None, None) | int
    :param auto_size_button: True if button size is determined by button text
    :type auto_size_button:  (bool)
    :param button_color:     button color (foreground, background)
    :type button_color:      (str, str) | str
    :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:              (str or (str, int[, str]) or None)
    :param bind_return_key:  (Default = False) If True, this button will appear to be clicked when return key is pressed in other elements such as Input and elements with return key options
    :type bind_return_key:   (bool)
    :param disabled:         set disable state for element (Default = False)
    :type disabled:          (bool)
    :param focus:            if focus should be set to this
    :type focus:             idk_yetReally
    :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:              key for uniquely identify this element (for window.find_element)
    :type key:               str | int | tuple | object
    :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                 str | int | tuple | object
    :param metadata:         Anything you want to store along with this button
    :type metadata:          (Any)
    :param expand_x:         If True Element will expand in the Horizontal directions
    :type expand_x:          (bool)
    :param expand_y:         If True Element will expand in the Vertical directions
    :type expand_y:          (bool)
    :return:                 returns a button
    :rtype:                  (Button)
    """
    return Button(button_text=button_text, button_type=BUTTON_TYPE_CLOSES_WIN, image_filename=image_filename,
                  image_data=image_data, image_size=image_size, image_subsample=image_subsample,
                  border_width=border_width, tooltip=tooltip, disabled=disabled, size=size, s=s,
                  auto_size_button=auto_size_button, button_color=button_color, font=font,
                  bind_return_key=bind_return_key, focus=focus, pad=pad, p=p, key=key, k=k, metadata=metadata, expand_x=expand_x, expand_y=expand_y)


CButton = CloseButton


# -------------------------  GENERIC BUTTON Element lazy function  ------------------------- #
def ReadButton(button_text, image_filename=None, image_data=None, image_size=(None, None), image_subsample=None,
               border_width=None, tooltip=None, size=(None, None), s=(None, None), auto_size_button=None, button_color=None, font=None,
               bind_return_key=False, disabled=False, focus=False, pad=None, p=None, key=None, k=None, metadata=None, expand_x=False, expand_y=False):
    """
    :param button_text:      text in the button
    :type button_text:       (str)
    :param image_filename:   image filename if there is a button image
    :type image_filename:    image filename if there is a button image
    :param image_data:       in-RAM image to be displayed on button
    :type image_data:        in-RAM image to be displayed on button
    :param image_size:       image size (O.K.)
    :type image_size:        (Default = (None))
    :param image_subsample:  amount to reduce the size of the image
    :type image_subsample:   amount to reduce the size of the image
    :param tooltip:          text, that will appear when mouse hovers over the element
    :type tooltip:           (str)
    :param size:             (w,h) w=characters-wide, h=rows-high
    :type size:              (int, int)
    :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                 (int, int)  | (None, None) | int
    :param auto_size_button: True if button size is determined by button text
    :type auto_size_button:  (bool)
    :param button_color:     button color (foreground, background)
    :type button_color:      (str, str) | str
    :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:              (str or (str, int[, str]) or None)
    :param bind_return_key:  (Default = False) If True, this button will appear to be clicked when return key is pressed in other elements such as Input and elements with return key options
    :type bind_return_key:   (bool)
    :param disabled:         set disable state for element (Default = False)
    :type disabled:          (bool)
    :param focus:            if focus should be set to this
    :type focus:             idk_yetReally
    :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:              key for uniquely identify this element (for window.find_element)
    :type key:               str | int | tuple | object
    :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                 str | int | tuple | object
    :param border_width:     width of border around element
    :type border_width:      (int)
    :param metadata:         Anything you want to store along with this button
    :type metadata:          (Any)
    :param expand_x:         If True Element will expand in the Horizontal directions
    :type expand_x:          (bool)
    :param expand_y:         If True Element will expand in the Vertical directions
    :type expand_y:          (bool)
    :return:                 Button created
    :rtype:                  (Button)
    """

    return Button(button_text=button_text, button_type=BUTTON_TYPE_READ_FORM, image_filename=image_filename,
                  image_data=image_data, image_size=image_size, image_subsample=image_subsample,
                  border_width=border_width, tooltip=tooltip, size=size, s=s, disabled=disabled,
                  auto_size_button=auto_size_button, button_color=button_color, font=font,
                  bind_return_key=bind_return_key, focus=focus, pad=pad, p=p, key=key, k=k, metadata=metadata, expand_x=expand_x, expand_y=expand_y)


ReadFormButton = ReadButton
RButton = ReadFormButton


# -------------------------  Realtime BUTTON Element lazy function  ------------------------- #
def RealtimeButton(button_text, image_filename=None, image_data=None, image_size=(None, None), image_subsample=None,
                   border_width=None, tooltip=None, size=(None, None), s=(None, None), auto_size_button=None, button_color=None,
                   font=None, disabled=False, bind_return_key=False, focus=False, pad=None, p=None, key=None, k=None, visible=True, metadata=None, expand_x=False, expand_y=False):
    """

    :param button_text:      text in the button
    :type button_text:       (str)
    :param image_filename:   image filename if there is a button image
    :type image_filename:    image filename if there is a button image
    :param image_data:       in-RAM image to be displayed on button
    :type image_data:        in-RAM image to be displayed on button
    :param image_size:       image size (O.K.)
    :type image_size:        (Default = (None))
    :param image_subsample:  amount to reduce the size of the image
    :type image_subsample:   amount to reduce the size of the image
    :param border_width:     width of border around element
    :type border_width:      (int)
    :param tooltip:          text, that will appear when mouse hovers over the element
    :type tooltip:           (str)
    :param size:             (w,h) w=characters-wide, h=rows-high
    :type size:              (int, int)
    :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                 (int, int)  | (None, None) | int
    :param auto_size_button: True if button size is determined by button text
    :type auto_size_button:  (bool)
    :param button_color:     button color (foreground, background)
    :type button_color:      (str, str) | str
    :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:              (str or (str, int[, str]) or None)
    :param disabled:         set disable state for element (Default = False)
    :type disabled:          (bool)
    :param bind_return_key:  (Default = False) If True, this button will appear to be clicked when return key is pressed in other elements such as Input and elements with return key options
    :type bind_return_key:   (bool)
    :param focus:            if focus should be set to this
    :type focus:             (bool)
    :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:              key for uniquely identify this element (for window.find_element)
    :type key:               str | int | tuple | object
    :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                 str | int | tuple | object
    :param visible:          set initial visibility state of the Button
    :type visible:           (bool)
    :param metadata:         Anything you want to store along with this button
    :type metadata:          (Any)
    :param expand_x:         If True Element will expand in the Horizontal directions
    :type expand_x:          (bool)
    :param expand_y:         If True Element will expand in the Vertical directions
    :type expand_y:          (bool)
    :return:                 Button created
    :rtype:                  (Button)
    """
    return Button(button_text=button_text, button_type=BUTTON_TYPE_REALTIME, image_filename=image_filename,
                  image_data=image_data, image_size=image_size, image_subsample=image_subsample,
                  border_width=border_width, tooltip=tooltip, disabled=disabled, size=size, s=s,
                  auto_size_button=auto_size_button, button_color=button_color, font=font,
                  bind_return_key=bind_return_key, focus=focus, pad=pad, p=p, key=key, k=k, visible=visible, metadata=metadata, expand_x=expand_x, expand_y=expand_y)


# -------------------------  Dummy BUTTON Element lazy function  ------------------------- #
def DummyButton(button_text, image_filename=None, image_data=None, image_size=(None, None), image_subsample=None,
                border_width=None, tooltip=None, size=(None, None), s=(None, None), auto_size_button=None, button_color=None, font=None,
                disabled=False, bind_return_key=False, focus=False, pad=None, p=None, key=None, k=None, visible=True, metadata=None, expand_x=False, expand_y=False):
    """
    This is a special type of Button.

    It will close the window but NOT send an event that the window has been closed.

    It's used in conjunction with non-blocking windows to silently close them.  They are used to
    implement the non-blocking popup windows. They're also found in some Demo Programs, so look there for proper use.

    :param button_text:      text in the button
    :type button_text:       (str)
    :param image_filename:   image filename if there is a button image
    :type image_filename:    image filename if there is a button image
    :param image_data:       in-RAM image to be displayed on button
    :type image_data:        in-RAM image to be displayed on button
    :param image_size:       image size (O.K.)
    :type image_size:        (Default = (None))
    :param image_subsample:  amount to reduce the size of the image
    :type image_subsample:   amount to reduce the size of the image
    :param border_width:     width of border around element
    :type border_width:      (int)
    :param tooltip:          text, that will appear when mouse hovers over the element
    :type tooltip:           (str)
    :param size:             (w,h) w=characters-wide, h=rows-high
    :type size:              (int, int)
    :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                 (int, int)  | (None, None) | int
    :param auto_size_button: True if button size is determined by button text
    :type auto_size_button:  (bool)
    :param button_color:     button color (foreground, background)
    :type button_color:      (str, str) | str
    :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:              (str or (str, int[, str]) or None)
    :param disabled:         set disable state for element (Default = False)
    :type disabled:          (bool)
    :param bind_return_key:  (Default = False) If True, this button will appear to be clicked when return key is pressed in other elements such as Input and elements with return key options
    :type bind_return_key:   (bool)
    :param focus:            if focus should be set to this
    :type focus:             (bool)
    :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:              key for uniquely identify this element (for window.find_element)
    :type key:               str | int | tuple | object
    :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                 str | int | tuple | object
    :param visible:          set initial visibility state of the Button
    :type visible:           (bool)
    :param metadata:         Anything you want to store along with this button
    :type metadata:          (Any)
    :param expand_x:         If True Element will expand in the Horizontal directions
    :type expand_x:          (bool)
    :param expand_y:         If True Element will expand in the Vertical directions
    :type expand_y:          (bool)
    :return:                 returns a button
    :rtype:                  (Button)
    """
    return Button(button_text=button_text, button_type=BUTTON_TYPE_CLOSES_WIN_ONLY, image_filename=image_filename,
                  image_data=image_data, image_size=image_size, image_subsample=image_subsample,
                  border_width=border_width, tooltip=tooltip, size=size, s=s, auto_size_button=auto_size_button,
                  button_color=button_color, font=font, disabled=disabled, bind_return_key=bind_return_key, focus=focus,
                  pad=pad, p=p, key=key, k=k, visible=visible, metadata=metadata, expand_x=expand_x, expand_y=expand_y)


# -------------------------  Calendar Chooser Button lazy function  ------------------------- #
def CalendarButton(button_text, target=(ThisRow, -1), close_when_date_chosen=True, default_date_m_d_y=(None, None, None),
                   image_filename=None, image_data=None, image_size=(None, None),
                   image_subsample=None, tooltip=None, border_width=None, size=(None, None), s=(None, None), auto_size_button=None,
                   button_color=None, disabled=False, font=None, bind_return_key=False, focus=False, pad=None, p=None, enable_events=None,
                   key=None, k=None, visible=True, locale=None, format='%Y-%m-%d %H:%M:%S', begin_at_sunday_plus=0, month_names=None, day_abbreviations=None,
                   title='Choose Date',
                   no_titlebar=True, location=(None, None), metadata=None, expand_x=False, expand_y=False):
    """
    Button that will show a calendar chooser window.  Fills in the target element with result

    :param button_text:            text in the button
    :type button_text:             (str)
    :param target:                 Key or "coordinate" (see docs) of target element
    :type target:                  (int, int) | Any
    :param close_when_date_chosen: (Default = True)
    :type close_when_date_chosen:  bool
    :param default_date_m_d_y:     Beginning date to show
    :type default_date_m_d_y:      (int, int or None, int)
    :param image_filename:         image filename if there is a button image
    :type image_filename:          image filename if there is a button image
    :param image_data:             in-RAM image to be displayed on button
    :type image_data:              in-RAM image to be displayed on button
    :param image_size:             image size (O.K.)
    :type image_size:              (Default = (None))
    :param image_subsample:        amount to reduce the size of the image
    :type image_subsample:         amount to reduce the size of the image
    :param tooltip:                text, that will appear when mouse hovers over the element
    :type tooltip:                 (str)
    :param border_width:           width of border around element
    :type border_width:            width of border around element
    :param size:                   (w,h) w=characters-wide, h=rows-high
    :type size:                    (int, int)
    :param s:                      Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                       (int, int)  | (None, None) | int
    :param auto_size_button:       True if button size is determined by button text
    :type auto_size_button:        (bool)
    :param button_color:           button color (foreground, background)
    :type button_color:            (str, str) | str
    :param disabled:               set disable state for element (Default = False)
    :type disabled:                (bool)
    :param font:                   specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:                    (str or (str, int[, str]) or None)
    :param bind_return_key:  (Default = False) If True, this button will appear to be clicked when return key is pressed in other elements such as Input and elements with return key options
    :type bind_return_key:         bool
    :param focus:                  if focus should be set to this
    :type focus:                   bool
    :param pad:                    Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:                     (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                      Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                       (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:                    key for uniquely identify this element (for window.find_element)
    :type key:                     str | int | tuple | object
    :param k:                      Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                       str | int | tuple | object
    :param locale:                 defines the locale used to get day names
    :type locale:                  str
    :param format:                 formats result using this strftime format
    :type format:                  str
    :param begin_at_sunday_plus:   Determines the left-most day in the display. 0=sunday, 1=monday, etc
    :type begin_at_sunday_plus:    (int)
    :param month_names:            optional list of month names to use (should be 12 items)
    :type month_names:             List[str]
    :param day_abbreviations:      optional list of abbreviations to display as the day of week
    :type day_abbreviations:       List[str]
    :param title:                  Title shown on the date chooser window
    :type title:                   (str)
    :param no_titlebar:            if True no titlebar will be shown on the date chooser window
    :type no_titlebar:             bool
    :param location:               Location on the screen (x,y) to show the calendar popup window
    :type location:                (int, int)
    :param visible:                set initial visibility state of the Button
    :type visible:                 (bool)
    :param metadata:               Anything you want to store along with this button
    :type metadata:                (Any)
    :param expand_x:               If True Element will expand in the Horizontal directions
    :type expand_x:                (bool)
    :param expand_y:               If True Element will expand in the Vertical directions
    :type expand_y:                (bool)
    :return:                       returns a button
    :rtype:                        (Button)
    """
    button = Button(button_text=button_text, button_type=BUTTON_TYPE_CALENDAR_CHOOSER, target=target,
                    image_filename=image_filename, image_data=image_data, image_size=image_size,
                    image_subsample=image_subsample, border_width=border_width, tooltip=tooltip, size=size, s=s,
                    auto_size_button=auto_size_button, button_color=button_color, font=font, disabled=disabled, enable_events=enable_events,
                    bind_return_key=bind_return_key, focus=focus, pad=pad, p=p, key=key, k=k, visible=visible, metadata=metadata, expand_x=expand_x, expand_y=expand_y)
    button.calendar_close_when_chosen = close_when_date_chosen
    button.calendar_default_date_M_D_Y = default_date_m_d_y
    button.calendar_locale = locale
    button.calendar_format = format
    button.calendar_no_titlebar = no_titlebar
    button.calendar_location = location
    button.calendar_begin_at_sunday_plus = begin_at_sunday_plus
    button.calendar_month_names = month_names
    button.calendar_day_abbreviations = day_abbreviations
    button.calendar_title = title

    return button


# -------------------------  Calendar Chooser Button lazy function  ------------------------- #
def ColorChooserButton(button_text, target=(ThisRow, -1), image_filename=None, image_data=None, image_size=(None, None),
                       image_subsample=None, tooltip=None, border_width=None, size=(None, None), s=(None, None), auto_size_button=None,
                       button_color=None, disabled=False, font=None, bind_return_key=False, focus=False, pad=None, p=None,
                       key=None, k=None, default_color=None, visible=True, metadata=None, expand_x=False, expand_y=False):
    """

    :param button_text:      text in the button
    :type button_text:       (str)
    :param target:           key or (row,col) target for the button. Note that -1 for column means 1 element to the left of this one. The constant ThisRow is used to indicate the current row. The Button itself is a valid target for some types of button
    :type target:            str | (int, int)
    :type image_filename:    (str)
    :param image_filename:   image filename if there is a button image. GIFs and PNGs only.
    :type image_filename:    (str)
    :param image_data:       Raw or Base64 representation of the image to put on button. Choose either filename or data
    :type image_data:        bytes | str
    :param image_size:       Size of the image in pixels (width, height)
    :type image_size:        (int, int)
    :param image_subsample:  amount to reduce the size of the image. Divides the size by this number. 2=1/2, 3=1/3, 4=1/4, etc
    :type image_subsample:   (int)
    :param tooltip:          text, that will appear when mouse hovers over the element
    :type tooltip:           (str)
    :param border_width:     width of border around element
    :type border_width:      (int)
    :param size:             (w,h) w=characters-wide, h=rows-high
    :type size:              (int, int)
    :param s:                Same as size parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, size will be used
    :type s:                 (int, int)  | (None, None) | int
    :param auto_size_button: True if button size is determined by button text
    :type auto_size_button:  (bool)
    :param button_color:     button color (foreground, background)
    :type button_color:      (str, str) | str
    :param disabled:         set disable state for element (Default = False)
    :type disabled:          (bool)
    :param font:             specifies the  font family, size, etc. Tuple or Single string format 'name size styles'. Styles: italic * roman bold normal underline overstrike
    :type font:              (str or (str, int[, str]) or None)
    :param bind_return_key:  (Default = False) If True, this button will appear to be clicked when return key is pressed in other elements such as Input and elements with return key options
    :type bind_return_key:   (bool)
    :param focus:            Determines if initial focus should go to this element.
    :type focus:             (bool)
    :param pad:              Amount of padding to put around element in pixels (left/right, top/bottom) or ((left, right), (top, bottom)) or an int. If an int, then it's converted into a tuple (int, int)
    :type pad:               (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param p:                Same as pad parameter.  It's an alias. If EITHER of them are set, then the one that's set will be used. If BOTH are set, pad will be used
    :type p:                 (int, int) or ((int, int),(int,int)) or (int,(int,int)) or  ((int, int),int) | int
    :param key:              key for uniquely identify this element (for window.find_element)
    :type key:               str | int | tuple | object
    :param k:                Same as the Key. You can use either k or key. Which ever is set will be used.
    :type k:                 str | int | tuple | object
    :param default_color:    Color to be sent to tkinter to use as the default color
    :type default_color:     str
    :param visible:          set initial visibility state of the Button
    :type visible:           (bool)
    :param metadata:         User metadata that can be set to ANYTHING
    :type metadata:          (Any)
    :param expand_x:         If True Element will expand in the Horizontal directions
    :type expand_x:          (bool)
    :param expand_y:         If True Element will expand in the Vertical directions
    :type expand_y:          (bool)
    :return:                 returns a button
    :rtype:                  (Button)
    """
    button = Button(button_text=button_text, button_type=BUTTON_TYPE_COLOR_CHOOSER, target=target,
                  image_filename=image_filename, image_data=image_data, image_size=image_size,
                  image_subsample=image_subsample, border_width=border_width, tooltip=tooltip, size=size, s=s,
                  auto_size_button=auto_size_button, button_color=button_color, font=font, disabled=disabled,
                  bind_return_key=bind_return_key, focus=focus, pad=pad, p=p, key=key, k=k, visible=visible, metadata=metadata, expand_x=expand_x, expand_y=expand_y)
    button.default_color = default_color
    return button

#####################################  -----  BUTTON Functions   ------ ##################################################

def button_color_to_tuple(color_tuple_or_string, default=(None, None)):
    """
    Convert a color tuple or color string into 2 components and returns them as a tuple
    (Text Color, Button Background Color)
    If None is passed in as the first parameter, then the theme's button color is
    returned

    :param color_tuple_or_string: Button color - tuple or a simplied color string with word "on" between color
    :type  color_tuple_or_string: str | (str, str)
    :param default:               The 2 colors to use if there is a problem. Otherwise defaults to the theme's button color
    :type  default:               (str, str)
    :return:                      (str | (str, str)
    :rtype:                       str | (str, str)
    """
    if default == (None, None):
        color_tuple = _simplified_dual_color_to_tuple(color_tuple_or_string, default=theme_button_color())
    elif color_tuple_or_string == COLOR_SYSTEM_DEFAULT:
        color_tuple = (COLOR_SYSTEM_DEFAULT, COLOR_SYSTEM_DEFAULT)
    else:
        color_tuple = _simplified_dual_color_to_tuple(color_tuple_or_string, default=default)

    return color_tuple


def _simplified_dual_color_to_tuple(color_tuple_or_string, default=(None, None)):
    """
    Convert a color tuple or color string into 2 components and returns them as a tuple
    (Text Color, Button Background Color)
    If None is passed in as the first parameter, theme_

    :param color_tuple_or_string: Button color - tuple or a simplied color string with word "on" between color
    :type  color_tuple_or_string: str | (str, str} | (None, None)
    :param default:               The 2 colors to use if there is a problem. Otherwise defaults to the theme's button color
    :type  default:               (str, str)
    :return:                      (str | (str, str)
    :rtype:                       str | (str, str)
    """
    if color_tuple_or_string is None or color_tuple_or_string == (None, None):
        color_tuple_or_string = default
    if color_tuple_or_string == COLOR_SYSTEM_DEFAULT:
        return (COLOR_SYSTEM_DEFAULT, COLOR_SYSTEM_DEFAULT)
    text_color = background_color = COLOR_SYSTEM_DEFAULT
    try:
        if isinstance(color_tuple_or_string, (tuple, list)):
            if len(color_tuple_or_string) >= 2:
                text_color = color_tuple_or_string[0] or default[0]
                background_color = color_tuple_or_string[1] or default[1]
            elif len(color_tuple_or_string) == 1:
                background_color = color_tuple_or_string[0] or default[1]
        elif isinstance(color_tuple_or_string, str):
            color_tuple_or_string = color_tuple_or_string.lower()
            split_colors = color_tuple_or_string.split(' on ')
            if len(split_colors) >= 2:
                text_color = split_colors[0].strip() or default[0]
                background_color = split_colors[1].strip() or default[1]
            elif len(split_colors) == 1:
                split_colors = color_tuple_or_string.split('on')
                if len(split_colors) == 1:
                    text_color, background_color = default[0], split_colors[0].strip()
                else:
                    split_colors = split_colors[0].strip(), split_colors[1].strip()
                    text_color = split_colors[0] or default[0]
                    background_color = split_colors[1] or default[1]
                    # text_color, background_color = color_tuple_or_string, default[1]
            else:
                text_color, background_color = default
        else:
            if not SUPPRESS_ERROR_POPUPS:
                _error_popup_with_traceback('** Badly formatted dual-color... not a tuple nor string **', color_tuple_or_string)
            else:
                print('** Badly formatted dual-color... not a tuple nor string **', color_tuple_or_string)
            text_color, background_color = default
    except Exception as e:
        if not SUPPRESS_ERROR_POPUPS:
            _error_popup_with_traceback('** Badly formatted button color **', color_tuple_or_string, e)
        else:
            print('** Badly formatted button color... not a tuple nor string **', color_tuple_or_string, e)
        text_color, background_color = default
    if isinstance(text_color, int):
        text_color = '#%06X' % text_color
    if isinstance(background_color, int):
        background_color = '#%06X' % background_color
    # print('converted button color', color_tuple_or_string, 'to', (text_color, background_color))

    return (text_color, background_color)


#####################################  -----  RESULTS   ------ ##################################################

def AddToReturnDictionary(form, element, value):
    form.ReturnValuesDictionary[element.Key] = value
    # if element.Key is None:
    #     form.ReturnValuesDictionary[form.DictionaryKeyCounter] = value
    #     element.Key = form.DictionaryKeyCounter
    #     form.DictionaryKeyCounter += 1
    # else:
    #     form.ReturnValuesDictionary[element.Key] = value


def AddToReturnList(form, value):
    form.ReturnValuesList.append(value)


# ----------------------------------------------------------------------------#
# -------  FUNCTION InitializeResults.  Sets up form results matrix  --------#
def InitializeResults(form):
    _BuildResults(form, True, form)
    return


# =====  Radio Button RadVar encoding and decoding =====#
# =====  The value is simply the row * 1000 + col  =====#
def DecodeRadioRowCol(RadValue):
    container = RadValue // 100000
    row = RadValue // 1000
    col = RadValue % 1000
    return container, row, col


def EncodeRadioRowCol(container, row, col):
    RadValue = container * 100000 + row * 1000 + col
    return RadValue


# -------  FUNCTION BuildResults.  Form exiting so build the results to pass back  ------- #
# format of return values is
# (Button Pressed, input_values)
def _BuildResults(form, initialize_only, top_level_form):
    # Results for elements are:
    #   TEXT - Nothing
    #   INPUT - Read value from TK
    #   Button - Button Text and position as a Tuple

    # Get the initialized results so we don't have to rebuild
    # form.DictionaryKeyCounter = 0
    form.ReturnValuesDictionary = {}
    form.ReturnValuesList = []
    _BuildResultsForSubform(form, initialize_only, top_level_form)
    if not top_level_form.LastButtonClickedWasRealtime:
        top_level_form.LastButtonClicked = None
    return form.ReturnValues


def _BuildResultsForSubform(form, initialize_only, top_level_form):
    event = top_level_form.LastButtonClicked
    for row_num, row in enumerate(form.Rows):
        for col_num, element in enumerate(row):
            if element.Key is not None and WRITE_ONLY_KEY in str(element.Key):
                continue
            value = None
            if element.Type == ELEM_TYPE_COLUMN:
                element.DictionaryKeyCounter = top_level_form.DictionaryKeyCounter
                element.ReturnValuesList = []
                element.ReturnValuesDictionary = {}
                _BuildResultsForSubform(element, initialize_only, top_level_form)
                for item in element.ReturnValuesList:
                    AddToReturnList(top_level_form, item)
                if element.UseDictionary:
                    top_level_form.UseDictionary = True
                if element.ReturnValues[0] is not None:  # if a button was clicked
                    event = element.ReturnValues[0]

            if element.Type == ELEM_TYPE_FRAME:
                element.DictionaryKeyCounter = top_level_form.DictionaryKeyCounter
                element.ReturnValuesList = []
                element.ReturnValuesDictionary = {}
                _BuildResultsForSubform(element, initialize_only, top_level_form)
                for item in element.ReturnValuesList:
                    AddToReturnList(top_level_form, item)
                if element.UseDictionary:
                    top_level_form.UseDictionary = True
                if element.ReturnValues[0] is not None:  # if a button was clicked
                    event = element.ReturnValues[0]

            if element.Type == ELEM_TYPE_PANE:
                element.DictionaryKeyCounter = top_level_form.DictionaryKeyCounter
                element.ReturnValuesList = []
                element.ReturnValuesDictionary = {}
                _BuildResultsForSubform(element, initialize_only, top_level_form)
                for item in element.ReturnValuesList:
                    AddToReturnList(top_level_form, item)
                if element.UseDictionary:
                    top_level_form.UseDictionary = True
                if element.ReturnValues[0] is not None:  # if a button was clicked
                    event = element.ReturnValues[0]

            if element.Type == ELEM_TYPE_TAB_GROUP:
                element.DictionaryKeyCounter = top_level_form.DictionaryKeyCounter
                element.ReturnValuesList = []
                element.ReturnValuesDictionary = {}
                _BuildResultsForSubform(element, initialize_only, top_level_form)
                for item in element.ReturnValuesList:
                    AddToReturnList(top_level_form, item)
                if element.UseDictionary:
                    top_level_form.UseDictionary = True
                if element.ReturnValues[0] is not None:  # if a button was clicked
                    event = element.ReturnValues[0]

            if element.Type == ELEM_TYPE_TAB:
                element.DictionaryKeyCounter = top_level_form.DictionaryKeyCounter
                element.ReturnValuesList = []
                element.ReturnValuesDictionary = {}
                _BuildResultsForSubform(element, initialize_only, top_level_form)
                for item in element.ReturnValuesList:
                    AddToReturnList(top_level_form, item)
                if element.UseDictionary:
                    top_level_form.UseDictionary = True
                if element.ReturnValues[0] is not None:  # if a button was clicked
                    event = element.ReturnValues[0]

            if not initialize_only:
                if element.Type == ELEM_TYPE_INPUT_TEXT:
                    try:
                        value = element.TKStringVar.get()
                    except:
                        value = ''
                    if not top_level_form.NonBlocking and not element.do_not_clear and not top_level_form.ReturnKeyboardEvents:
                        element.TKStringVar.set('')
                elif element.Type == ELEM_TYPE_INPUT_CHECKBOX:
                    value = element.TKIntVar.get()
                    value = (value != 0)
                elif element.Type == ELEM_TYPE_INPUT_RADIO:
                    RadVar = element.TKIntVar.get()
                    this_rowcol = EncodeRadioRowCol(form.ContainerElemementNumber, row_num, col_num)
                    # this_rowcol = element.EncodedRadioValue       # could use the saved one
                    value = RadVar == this_rowcol
                elif element.Type == ELEM_TYPE_BUTTON:
                    if top_level_form.LastButtonClicked == element.Key:
                        event = top_level_form.LastButtonClicked
                        if element.BType != BUTTON_TYPE_REALTIME:  # Do not clear realtime buttons
                            top_level_form.LastButtonClicked = None
                    if element.BType == BUTTON_TYPE_CALENDAR_CHOOSER:
                        # value = None
                        value = element.calendar_selection
                    else:
                        try:
                            value = element.TKStringVar.get()
                        except:
                            value = None
                elif element.Type == ELEM_TYPE_INPUT_COMBO:
                    element = element  # type: Combo
                    # value = element.TKStringVar.get()
                    try:
                        if element.TKCombo.current() == -1:  # if the current value was not in the original list
                            value = element.TKCombo.get()
                        else:
                            value = element.Values[element.TKCombo.current()]  # get value from original list given index
                    except:
                        value = '*Exception occurred*'
                elif element.Type == ELEM_TYPE_INPUT_OPTION_MENU:
                    value = element.TKStringVar.get()
                elif element.Type == ELEM_TYPE_INPUT_LISTBOX:
                    try:
                        items = element.TKListbox.curselection()
                        value = [element.Values[int(item)] for item in items]
                    except Exception as e:
                        value = ''
                elif element.Type == ELEM_TYPE_INPUT_SPIN:
                    try:
                        value = element.TKStringVar.get()
                        for v in element.Values:
                            if str(v) == value:
                                value = v
                                break
                    except:
                        value = 0
                elif element.Type == ELEM_TYPE_INPUT_SLIDER:
                    try:
                        value = float(element.TKScale.get())
                    except:
                        value = 0
                elif element.Type == ELEM_TYPE_INPUT_MULTILINE:
                    if element.WriteOnly:  # if marked as "write only" when created, then don't include with the values being returned
                        continue
                    try:
                        value = element.TKText.get(1.0, tk.END)
                        if element.rstrip:
                            value = value.rstrip()
                        if not top_level_form.NonBlocking and not element.do_not_clear and not top_level_form.ReturnKeyboardEvents:
                            element.TKText.delete('1.0', tk.END)
                    except:
                        value = None
                elif element.Type == ELEM_TYPE_TAB_GROUP:
                    try:
                        value = element.TKNotebook.tab(element.TKNotebook.index('current'))['text']
                        tab_key = element.find_currently_active_tab_key()
                        # tab_key = element.FindKeyFromTabName(value)
                        if tab_key is not None:
                            value = tab_key
                    except:
                        value = None
                elif element.Type == ELEM_TYPE_TABLE:
                    value = element.SelectedRows
                elif element.Type == ELEM_TYPE_TREE:
                    value = element.SelectedRows
                elif element.Type == ELEM_TYPE_GRAPH:
                    value = element.ClickPosition
                elif element.Type == ELEM_TYPE_MENUBAR:
                    if element.MenuItemChosen is not None:
                        event = top_level_form.LastButtonClicked = element.MenuItemChosen
                    value = element.MenuItemChosen
                    element.MenuItemChosen = None
                elif element.Type == ELEM_TYPE_BUTTONMENU:
                    element = element  # type: ButtonMenu
                    value = element.MenuItemChosen
                    if element.part_of_custom_menubar:
                        if element.MenuItemChosen is not None:
                            value = event = element.MenuItemChosen
                            top_level_form.LastButtonClicked = element.MenuItemChosen
                            if element.custom_menubar_key is not None:
                                top_level_form.ReturnValuesDictionary[element.custom_menubar_key] = value
                            element.MenuItemChosen = None
                        else:
                            if element.custom_menubar_key not in top_level_form.ReturnValuesDictionary:
                                top_level_form.ReturnValuesDictionary[element.custom_menubar_key] = None
                            value = None

                    # if element.MenuItemChosen is not None:
                    #     button_pressed_text = top_level_form.LastButtonClicked = element.MenuItemChosen
                    # value = element.MenuItemChosen
                    # element.MenuItemChosen = None
            else:
                value = None

            # if an input type element, update the results
            if element.Type not in (
            ELEM_TYPE_BUTTON, ELEM_TYPE_TEXT, ELEM_TYPE_IMAGE, ELEM_TYPE_OUTPUT, ELEM_TYPE_PROGRESS_BAR, ELEM_TYPE_COLUMN, ELEM_TYPE_FRAME, ELEM_TYPE_SEPARATOR,
            ELEM_TYPE_TAB):
                if not (element.Type == ELEM_TYPE_BUTTONMENU and element.part_of_custom_menubar):
                    AddToReturnList(form, value)
                    AddToReturnDictionary(top_level_form, element, value)
            elif (element.Type == ELEM_TYPE_BUTTON and
                  element.BType == BUTTON_TYPE_COLOR_CHOOSER and
                  element.Target == (None, None)) or \
                    (element.Type == ELEM_TYPE_BUTTON
                     and element.Key is not None and
                     (element.BType in (BUTTON_TYPE_SAVEAS_FILE, BUTTON_TYPE_BROWSE_FILE, BUTTON_TYPE_BROWSE_FILES,
                                        BUTTON_TYPE_BROWSE_FOLDER, BUTTON_TYPE_CALENDAR_CHOOSER))):
                AddToReturnList(form, value)
                AddToReturnDictionary(top_level_form, element, value)

    # if this is a column, then will fail so need to wrap with try
    try:
        if form.ReturnKeyboardEvents and form.LastKeyboardEvent is not None:
            event = form.LastKeyboardEvent
            form.LastKeyboardEvent = None
    except:
        pass

    try:
        form.ReturnValuesDictionary.pop(None, None)  # clean up dictionary include None was included
    except:
        pass

    # if no event was found
    if not initialize_only and event is None and form == top_level_form:
        queued_event_value = form._queued_thread_event_read()
        if queued_event_value is not None:
            event, value = queued_event_value
            AddToReturnList(form, value)
            form.ReturnValuesDictionary[event] = value

    if not form.UseDictionary:
        form.ReturnValues = event, form.ReturnValuesList
    else:
        form.ReturnValues = event, form.ReturnValuesDictionary

    return form.ReturnValues


def fill_form_with_values(window, values_dict):
    """
    Fills a window with values provided in a values dictionary { element_key : new_value }

    :param window:      The window object to fill
    :type window:       (Window)
    :param values_dict: A dictionary with element keys as key and value is values parm for Update call
    :type values_dict:  (Dict[Any, Any])
    :return:            None
    :rtype:             None
    """

    for element_key in values_dict:
        try:
            window.AllKeysDict[element_key].Update(values_dict[element_key])
        except Exception as e:
            print('Problem filling form. Perhaps bad key?  This is a suspected bad key: {}'.format(element_key))


def _FindElementWithFocusInSubForm(form):
    """
    Searches through a "sub-form" (can be a window or container) for the current element with focus

    :param form: a Window, Column, Frame, or TabGroup (container elements)
    :type form:  container elements
    :return:     Element
    :rtype:      Element | None
    """
    for row_num, row in enumerate(form.Rows):
        for col_num, element in enumerate(row):
            if element.Type == ELEM_TYPE_COLUMN:
                matching_elem = _FindElementWithFocusInSubForm(element)
                if matching_elem is not None:
                    return matching_elem
            elif element.Type == ELEM_TYPE_FRAME:
                matching_elem = _FindElementWithFocusInSubForm(element)
                if matching_elem is not None:
                    return matching_elem
            elif element.Type == ELEM_TYPE_TAB_GROUP:
                matching_elem = _FindElementWithFocusInSubForm(element)
                if matching_elem is not None:
                    return matching_elem
            elif element.Type == ELEM_TYPE_TAB:
                matching_elem = _FindElementWithFocusInSubForm(element)
                if matching_elem is not None:
                    return matching_elem
            elif element.Type == ELEM_TYPE_PANE:
                matching_elem = _FindElementWithFocusInSubForm(element)
                if matching_elem is not None:
                    return matching_elem
            elif element.Type == ELEM_TYPE_INPUT_TEXT:
                if element.TKEntry is not None:
                    if element.TKEntry is element.TKEntry.focus_get():
                        return element
            elif element.Type == ELEM_TYPE_INPUT_MULTILINE:
                if element.TKText is not None:
                    if element.TKText is element.TKText.focus_get():
                        return element
            elif element.Type == ELEM_TYPE_BUTTON:
                if element.TKButton is not None:
                    if element.TKButton is element.TKButton.focus_get():
                        return element
            else:  # The "Catch All" - if type isn't one of the above, try generic element.Widget
                try:
                    if element.Widget is not None:
                        if element.Widget is element.Widget.focus_get():
                            return element
                except:
                    return None

    return None


def AddMenuItem(top_menu, sub_menu_info, element, is_sub_menu=False, skip=False, right_click_menu=False):
    """
    Only to be used internally. Not user callable
    :param top_menu:      ???
    :type top_menu:       ???
    :param sub_menu_info: ???
    :type sub_menu_info:
    :param element:       ???
    :type element:        idk_yetReally
    :param is_sub_menu:   (Default = False)
    :type is_sub_menu:    (bool)
    :param skip:          (Default = False)
    :type skip:           (bool)

    """
    return_val = None
    if type(sub_menu_info) is str:
        if not is_sub_menu and not skip:
            pos = sub_menu_info.find(MENU_SHORTCUT_CHARACTER)
            if pos != -1:
                if pos < len(MENU_SHORTCUT_CHARACTER) or sub_menu_info[pos - len(MENU_SHORTCUT_CHARACTER)] != '\\':
                    sub_menu_info = sub_menu_info[:pos] + sub_menu_info[pos + len(MENU_SHORTCUT_CHARACTER):]
            if sub_menu_info == '---':
                top_menu.add('separator')
            else:
                try:
                    item_without_key = sub_menu_info[:sub_menu_info.index(MENU_KEY_SEPARATOR)]
                except:
                    item_without_key = sub_menu_info

                if item_without_key[0] == MENU_DISABLED_CHARACTER:
                    top_menu.add_command(label=item_without_key[len(MENU_DISABLED_CHARACTER):], underline=pos - 1,
                                         command=lambda: element._MenuItemChosenCallback(sub_menu_info))
                    top_menu.entryconfig(item_without_key[len(MENU_DISABLED_CHARACTER):], state='disabled')
                else:
                    top_menu.add_command(label=item_without_key, underline=pos,
                                         command=lambda: element._MenuItemChosenCallback(sub_menu_info))
    else:
        i = 0
        while i < (len(sub_menu_info)):
            item = sub_menu_info[i]
            if i != len(sub_menu_info) - 1:
                if type(sub_menu_info[i + 1]) == list:
                    new_menu = tk.Menu(top_menu, tearoff=element.Tearoff)
                    # if a right click menu, then get styling from the top-level window
                    if right_click_menu:
                        window = element.ParentForm
                        if window.right_click_menu_background_color not in (COLOR_SYSTEM_DEFAULT, None):
                            new_menu.config(bg=window.right_click_menu_background_color)
                            new_menu.config(activeforeground=window.right_click_menu_background_color)
                        if window.right_click_menu_text_color not in (COLOR_SYSTEM_DEFAULT, None):
                            new_menu.config(fg=window.right_click_menu_text_color)
                            new_menu.config(activebackground=window.right_click_menu_text_color)
                        if window.right_click_menu_disabled_text_color not in (COLOR_SYSTEM_DEFAULT, None):
                            new_menu.config(disabledforeground=window.right_click_menu_disabled_text_color)
                        if window.right_click_menu_font is not None:
                            new_menu.config(font=window.right_click_menu_font)
                    else:
                        if element.Font is not None:
                            new_menu.config(font=element.Font)
                        if element.BackgroundColor not in (COLOR_SYSTEM_DEFAULT, None):
                            new_menu.config(bg=element.BackgroundColor)
                            new_menu.config(activeforeground=element.BackgroundColor)
                        if element.TextColor not in (COLOR_SYSTEM_DEFAULT, None):
                            new_menu.config(fg=element.TextColor)
                            new_menu.config(activebackground=element.TextColor)
                        if element.DisabledTextColor not in (COLOR_SYSTEM_DEFAULT, None):
                            new_menu.config(disabledforeground=element.DisabledTextColor)
                        if element.ItemFont is not None:
                            new_menu.config(font=element.ItemFont)
                    return_val = new_menu
                    pos = sub_menu_info[i].find(MENU_SHORTCUT_CHARACTER)
                    if pos != -1:
                        if pos < len(MENU_SHORTCUT_CHARACTER) or sub_menu_info[i][pos - len(MENU_SHORTCUT_CHARACTER)] != '\\':
                            sub_menu_info[i] = sub_menu_info[i][:pos] + sub_menu_info[i][pos + len(MENU_SHORTCUT_CHARACTER):]
                    if sub_menu_info[i][0] == MENU_DISABLED_CHARACTER:
                        top_menu.add_cascade(label=sub_menu_info[i][len(MENU_DISABLED_CHARACTER):], menu=new_menu,
                                             underline=pos, state='disabled')
                    else:
                        top_menu.add_cascade(label=sub_menu_info[i], menu=new_menu, underline=pos)
                    AddMenuItem(new_menu, sub_menu_info[i + 1], element, is_sub_menu=True, right_click_menu=right_click_menu)
                    i += 1  # skip the next one
                else:
                    AddMenuItem(top_menu, item, element, right_click_menu=right_click_menu)
            else:
                AddMenuItem(top_menu, item, element, right_click_menu=right_click_menu)
            i += 1
    return return_val


# 888    888      d8b          888
# 888    888      Y8P          888
# 888    888                   888
# 888888 888  888 888 88888b.  888888  .d88b.  888d888
# 888    888 .88P 888 888 "88b 888    d8P  Y8b 888P"
# 888    888888K  888 888  888 888    88888888 888
# Y88b.  888 "88b 888 888  888 Y88b.  Y8b.     888
#  "Y888 888  888 888 888  888  "Y888  "Y8888  888

# My crappy tkinter code starts here.  (search for "crappy" to get here quickly... that's the purpose if you hadn't caught on

"""
         )
        (
          ,
       ___)\
      (_____)
      (_______)

"""


# Chr0nic || This is probably *very* bad practice. But it works. Simple, but it works...
class VarHolder(object):
    canvas_holder = None

    def __init__(self):
        self.canvas_holder = None


# Also, to get to the point in the code where each element's widget is created, look for element + "p lacement" (without the space)


# ========================   TK CODE STARTS HERE ========================================= #
def _fixed_map(style, style_name, option, highlight_colors=(None, None)):
    # Fix for setting text colour for Tkinter 8.6.9
    # From: https://core.tcl.tk/tk/info/509cafafae

    # default_map = [elm for elm in style.map("Treeview", query_opt=option) if '!' not in elm[0]]
    # custom_map = [elm for elm in style.map(style_name, query_opt=option) if '!' not in elm[0]]
    default_map = [elm for elm in style.map('Treeview', query_opt=option) if '!' not in elm[0] and 'selected' not in elm[0]]
    custom_map = [elm for elm in style.map(style_name, query_opt=option) if '!' not in elm[0] and 'selected' not in elm[0]]
    if option == 'background':
        custom_map.append(('selected', highlight_colors[1] if highlight_colors[1] is not None else ALTERNATE_TABLE_AND_TREE_SELECTED_ROW_COLORS[1]))
    elif option == 'foreground':
        custom_map.append(('selected', highlight_colors[0] if highlight_colors[0] is not None else ALTERNATE_TABLE_AND_TREE_SELECTED_ROW_COLORS[0]))

    new_map = custom_map + default_map
    return new_map
    #
    # new_map = [elm for elm in style.map(style_name, query_opt=option) if elm[:2] != ('!disabled', '!selected')]
    #
    # if option == 'background':
    #     new_map.append(('selected', highlight_colors[1] if highlight_colors[1] is not None else ALTERNATE_TABLE_AND_TREE_SELECTED_ROW_COLORS[1]))
    # elif option == 'foreground':
    #     new_map.append(('selected', highlight_colors[0] if highlight_colors[0] is not None else ALTERNATE_TABLE_AND_TREE_SELECTED_ROW_COLORS[0]))
    # return new_map
    #

def _add_right_click_menu(element, toplevel_form):
    if element.RightClickMenu == MENU_RIGHT_CLICK_DISABLED:
        return
    if element.RightClickMenu or toplevel_form.RightClickMenu:
        menu = element.RightClickMenu or toplevel_form.RightClickMenu
        top_menu = tk.Menu(toplevel_form.TKroot, tearoff=toplevel_form.right_click_menu_tearoff, tearoffcommand=element._tearoff_menu_callback)

        if toplevel_form.right_click_menu_background_color not in (COLOR_SYSTEM_DEFAULT, None):
            top_menu.config(bg=toplevel_form.right_click_menu_background_color)
        if toplevel_form.right_click_menu_text_color not in (COLOR_SYSTEM_DEFAULT, None):
            top_menu.config(fg=toplevel_form.right_click_menu_text_color)
        if toplevel_form.right_click_menu_disabled_text_color not in (COLOR_SYSTEM_DEFAULT, None):
            top_menu.config(disabledforeground=toplevel_form.right_click_menu_disabled_text_color)
        if toplevel_form.right_click_menu_font is not None:
            top_menu.config(font=toplevel_form.right_click_menu_font)

        if toplevel_form.right_click_menu_selected_colors[0] not in (COLOR_SYSTEM_DEFAULT, None):
            top_menu.config(activeforeground=toplevel_form.right_click_menu_selected_colors[0])
        if toplevel_form.right_click_menu_selected_colors[1] not in (COLOR_SYSTEM_DEFAULT, None):
            top_menu.config(activebackground=toplevel_form.right_click_menu_selected_colors[1])
        AddMenuItem(top_menu, menu[1], element, right_click_menu=True)
        element.TKRightClickMenu = top_menu
        if (running_mac()):
            element.Widget.bind('<ButtonRelease-2>', element._RightClickMenuCallback)
        else:
            element.Widget.bind('<ButtonRelease-3>', element._RightClickMenuCallback)


def _change_ttk_theme(style, theme_name):
    global ttk_theme_in_use
    if theme_name not in style.theme_names():
        _error_popup_with_traceback('You are trying to use TTK theme "{}"'.format(theme_name),
                                        'This is not legal for your system',
                                        'The valid themes to choose from are: {}'.format(', '.join(style.theme_names())))
        return False

    style.theme_use(theme_name)
    ttk_theme_in_use = theme_name
    return True

# class Stylist:
#     """
#     A class to help get information about ttk styles
#     """
#     @staticmethod
#     def get_elements(layout):
#         """Return a list of elements contained in the style"""
#         elements = []
#         element = layout[0][0]
#         elements.append(element)
#         sublayout = layout[0][1]
#
#         if 'children' in sublayout:
#             child_elements = Stylist.get_elements(sublayout['children'])
#             elements.extend(child_elements)
#         return elements
#
#     @staticmethod
#     def get_options(ttkstyle, theme=None):
#         style = ttk.Style()
#         if theme is not None:
#             style.theme_use(theme)
#         layout = style.layout(ttkstyle)
#         elements = Stylist.get_elements(layout)
#         options = []
#         for e in elements:
#             _opts = style.element_options(e)
#             if _opts:
#                 options.extend(list(_opts))
#         return list(set(options))
#
#     @staticmethod
#     def create_style(base_style: str, theme=None, **kwargs):
#         style = ttk.Style()
#         if theme is not None:
#             style.theme_use(theme)
#         style_id = uuid4()
#         ttkstyle = '{}.{}'.format(style_id, base_style)
#         style.configure(ttkstyle, **kwargs)
#         return ttkstyle

def _make_ttk_style_name(base_style, element, primary_style=False):
    Window._counter_for_ttk_widgets += 1
    style_name = str(Window._counter_for_ttk_widgets) + '___' + str(element.Key) + base_style
    if primary_style:
        element.ttk_style_name = style_name
    return style_name


def _make_ttk_scrollbar(element, orientation, window):
    """
    Creates a ttk scrollbar for elements as they are being added to the layout

    :param element:     The element
    :type element:      (Element)
    :param orientation: The orientation vertical ('v') or horizontal ('h')
    :type orientation:  (str)
    :param window:      The window containing the scrollbar
    :type window:       (Window)
    """

    style = ttk.Style()
    _change_ttk_theme(style, window.TtkTheme)
    if orientation[0].lower() == 'v':
        orient = 'vertical'
        style_name = _make_ttk_style_name('.Vertical.TScrollbar', element)
        # style_name_thumb = _make_ttk_style_name('.Vertical.TScrollbar.thumb', element)
        element.vsb_style = style
        element.vsb = ttk.Scrollbar(element.element_frame, orient=orient, command=element.Widget.yview, style=style_name)
        element.vsb_style_name = style_name
    else:
        orient = 'horizontal'
        style_name = _make_ttk_style_name('.Horizontal.TScrollbar', element)
        element.hsb_style = style
        element.hsb = ttk.Scrollbar(element.element_frame, orient=orient, command=element.Widget.xview, style=style_name)
        element.hsb_style_name = style_name


    # ------------------ Get the colors using heirarchy of element, window, options, settings ------------------
    # Trough Color
    if element.ttk_part_overrides.sbar_trough_color is not None:
        trough_color = element.ttk_part_overrides.sbar_trough_color
    elif window.ttk_part_overrides.sbar_trough_color is not None:
        trough_color = window.ttk_part_overrides.sbar_trough_color
    elif ttk_part_overrides_from_options.sbar_trough_color is not None:
        trough_color = ttk_part_overrides_from_options.sbar_trough_color
    else:
        trough_color = element.scroll_trough_color
    # Relief
    if element.ttk_part_overrides.sbar_relief is not None:
        scroll_relief = element.ttk_part_overrides.sbar_relief
    elif window.ttk_part_overrides.sbar_relief is not None:
        scroll_relief = window.ttk_part_overrides.sbar_relief
    elif ttk_part_overrides_from_options.sbar_relief is not None:
        scroll_relief = ttk_part_overrides_from_options.sbar_relief
    else:
        scroll_relief = element.scroll_relief
    # Frame Color
    if element.ttk_part_overrides.sbar_frame_color is not None:
        frame_color = element.ttk_part_overrides.sbar_frame_color
    elif window.ttk_part_overrides.sbar_frame_color is not None:
        frame_color = window.ttk_part_overrides.sbar_frame_color
    elif ttk_part_overrides_from_options.sbar_frame_color is not None:
        frame_color = ttk_part_overrides_from_options.sbar_frame_color
    else:
        frame_color = element.scroll_frame_color
    # Background Color
    if element.ttk_part_overrides.sbar_background_color is not None:
        background_color = element.ttk_part_overrides.sbar_background_color
    elif window.ttk_part_overrides.sbar_background_color is not None:
        background_color = window.ttk_part_overrides.sbar_background_color
    elif ttk_part_overrides_from_options.sbar_background_color is not None:
        background_color = ttk_part_overrides_from_options.sbar_background_color
    else:
        background_color = element.scroll_background_color
    # Arrow Color
    if element.ttk_part_overrides.sbar_arrow_color is not None:
        arrow_color = element.ttk_part_overrides.sbar_arrow_color
    elif window.ttk_part_overrides.sbar_arrow_color is not None:
        arrow_color = window.ttk_part_overrides.sbar_arrow_color
    elif ttk_part_overrides_from_options.sbar_arrow_color is not None:
        arrow_color = ttk_part_overrides_from_options.sbar_arrow_color
    else:
        arrow_color = element.scroll_arrow_color
    # Arrow Width
    if element.ttk_part_overrides.sbar_arrow_width is not None:
        arrow_width = element.ttk_part_overrides.sbar_arrow_width
    elif window.ttk_part_overrides.sbar_arrow_width is not None:
        arrow_width = window.ttk_part_overrides.sbar_arrow_width
    elif ttk_part_overrides_from_options.sbar_arrow_width is not None:
        arrow_width = ttk_part_overrides_from_options.sbar_arrow_width
    else:
        arrow_width = element.scroll_arrow_width
    # Scroll Width
    if element.ttk_part_overrides.sbar_width is not None:
        scroll_width = element.ttk_part_overrides.sbar_width
    elif window.ttk_part_overrides.sbar_width is not None:
        scroll_width = window.ttk_part_overrides.sbar_width
    elif ttk_part_overrides_from_options.sbar_width is not None:
        scroll_width = ttk_part_overrides_from_options.sbar_width
    else:
        scroll_width = element.scroll_width


    if trough_color not in (None, COLOR_SYSTEM_DEFAULT):
        style.configure(style_name, troughcolor=trough_color)

    if frame_color not in (None, COLOR_SYSTEM_DEFAULT):
        style.configure(style_name, framecolor=frame_color)
    if frame_color not in (None, COLOR_SYSTEM_DEFAULT):
        style.configure(style_name, bordercolor=frame_color)

    if (background_color not in (None, COLOR_SYSTEM_DEFAULT)) and \
        (arrow_color not in (None, COLOR_SYSTEM_DEFAULT)):
        style.map(style_name, background=[('selected', background_color), ('active', arrow_color), ('background', background_color), ('!focus', background_color)])
    if (background_color not in (None, COLOR_SYSTEM_DEFAULT)) and \
        (arrow_color not in (None, COLOR_SYSTEM_DEFAULT)):
        style.map(style_name, arrowcolor=[('selected', arrow_color), ('active', background_color), ('background', background_color),('!focus', arrow_color)])

    if scroll_width not in (None, COLOR_SYSTEM_DEFAULT):
        style.configure(style_name, width=scroll_width)
    if arrow_width not in (None, COLOR_SYSTEM_DEFAULT):
        style.configure(style_name, arrowsize=arrow_width)

    if scroll_relief not in (None, COLOR_SYSTEM_DEFAULT):
        style.configure(style_name, relief=scroll_relief)

# if __name__ == '__main__':
#     root = tk.Tk()
#
#     # find out what options are available for the theme and widget style
#     options = Stylist.get_options('TFrame', 'default')
#     print('The options for this style and theme are', options)
#
#     # create a new style
#     frame_style = Stylist.create_style('TFrame', 'alt', relief=tk.RAISED, borderwidth=1)
#
#     # apply the new style
#     ttk.Frame(style=frame_style, width=100, height=100).pack(padx=10, pady=10)
#
#     root.mainloop()

# @_timeit
def PackFormIntoFrame(form, containing_frame, toplevel_form):
    """

    :param form:             a window class
    :type form:              (Window)
    :param containing_frame: ???
    :type containing_frame:  ???
    :param toplevel_form:    ???
    :type toplevel_form:     (Window)

    """

    # Old bindings
    def yscroll_old(event):
        try:
            if event.num == 5 or event.delta < 0:
                VarHolder.canvas_holder.yview_scroll(1, 'unit')
            elif event.num == 4 or event.delta > 0:
                VarHolder.canvas_holder.yview_scroll(-1, 'unit')
        except:
            pass

    def xscroll_old(event):
        try:
            if event.num == 5 or event.delta < 0:
                VarHolder.canvas_holder.xview_scroll(1, 'unit')
            elif event.num == 4 or event.delta > 0:
                VarHolder.canvas_holder.xview_scroll(-1, 'unit')
        except:
            pass

    # Chr0nic
    def testMouseHook2(em):
        combo = em.TKCombo
        combo.unbind_class('TCombobox', '<MouseWheel>')
        combo.unbind_class('TCombobox', '<ButtonPress-4>')
        combo.unbind_class('TCombobox', '<ButtonPress-5>')
        containing_frame.unbind_all('<4>')
        containing_frame.unbind_all('<5>')
        containing_frame.unbind_all('<MouseWheel>')
        containing_frame.unbind_all('<Shift-MouseWheel>')

    # Chr0nic
    def testMouseUnhook2(em):
        containing_frame.bind_all('<4>', yscroll_old, add='+')
        containing_frame.bind_all('<5>', yscroll_old, add='+')
        containing_frame.bind_all('<MouseWheel>', yscroll_old, add='+')
        containing_frame.bind_all('<Shift-MouseWheel>', xscroll_old, add='+')

    # Chr0nic
    def testMouseHook(em):
        containing_frame.unbind_all('<4>')
        containing_frame.unbind_all('<5>')
        containing_frame.unbind_all('<MouseWheel>')
        containing_frame.unbind_all('<Shift-MouseWheel>')

    # Chr0nic
    def testMouseUnhook(em):
        containing_frame.bind_all('<4>', yscroll_old, add='+')
        containing_frame.bind_all('<5>', yscroll_old, add='+')
        containing_frame.bind_all('<MouseWheel>', yscroll_old, add='+')
        containing_frame.bind_all('<Shift-MouseWheel>', xscroll_old, add='+')

    def _char_width_in_pixels(font):
        return tkinter.font.Font(font=font).measure('A')  # single character width

    def _char_height_in_pixels(font):
        return tkinter.font.Font(font=font).metrics('linespace')

    def _string_width_in_pixels(font, string):
        return tkinter.font.Font(font=font).measure(string)  # single character width




    # def _valid_theme(style, theme_name):
    #     if theme_name in style.theme_names():
    #         return True
    #     _error_popup_with_traceback('Your Window has an invalid ttk theme specified',
    #                                 'The traceback will show you the Window with the problem layout',
    #                                 '** Invalid ttk theme specified {} **'.format(theme_name),
    #                                 '\nValid choices include: {}'.format(style.theme_names()))
    #
    #     # print('** Invalid ttk theme specified {} **'.format(theme_name),
    #     #       '\nValid choices include: {}'.format(style.theme_names()))
    #     return False



    def _add_grab(element):

        try:
            if form.Grab is True or element.Grab is True:
                # if something already about to the button, then don't do the grab stuff
                if '<Button-1>' not in element.Widget.bind():
                    element.Widget.bind('<ButtonPress-1>', toplevel_form._StartMoveGrabAnywhere)
                    element.Widget.bind('<ButtonRelease-1>', toplevel_form._StopMove)
                    element.Widget.bind('<B1-Motion>', toplevel_form._OnMotionGrabAnywhere)
                element.ParentRowFrame.bind('<ButtonPress-1>', toplevel_form._StartMoveGrabAnywhere)
                element.ParentRowFrame.bind('<ButtonRelease-1>', toplevel_form._StopMove)
                element.ParentRowFrame.bind('<B1-Motion>', toplevel_form._OnMotionGrabAnywhere)
                if element.Type == ELEM_TYPE_COLUMN:
                    element.TKColFrame.canvas.bind('<ButtonPress-1>', toplevel_form._StartMoveGrabAnywhere)
                    element.TKColFrame.canvas.bind('<ButtonRelease-1>', toplevel_form._StopMove)
                    element.TKColFrame.canvas.bind('<B1-Motion>', toplevel_form._OnMotionGrabAnywhere)
        except Exception as e:
            pass
            # print(e)

    def _add_right_click_menu_and_grab(element):
        if element.RightClickMenu == MENU_RIGHT_CLICK_DISABLED:
            return
        if element.Type == ELEM_TYPE_TAB_GROUP:   # unless everything disabled, then need to always set a right click menu for tabgroups
            if toplevel_form.RightClickMenu == MENU_RIGHT_CLICK_DISABLED:
                return
            menu = _MENU_RIGHT_CLICK_TABGROUP_DEFAULT
        else:
            menu = element.RightClickMenu or form.RightClickMenu or toplevel_form.RightClickMenu

        if menu:
            top_menu = tk.Menu(toplevel_form.TKroot, tearoff=toplevel_form.right_click_menu_tearoff, tearoffcommand=element._tearoff_menu_callback)

            if toplevel_form.right_click_menu_background_color not in (COLOR_SYSTEM_DEFAULT, None):
                top_menu.config(bg=toplevel_form.right_click_menu_background_color)
            if toplevel_form.right_click_menu_text_color not in (COLOR_SYSTEM_DEFAULT, None):
                top_menu.config(fg=toplevel_form.right_click_menu_text_color)
            if toplevel_form.right_click_menu_disabled_text_color not in (COLOR_SYSTEM_DEFAULT, None):
                top_menu.config(disabledforeground=toplevel_form.right_click_menu_disabled_text_color)
            if toplevel_form.right_click_menu_font is not None:
                top_menu.config(font=toplevel_form.right_click_menu_font)

            if toplevel_form.right_click_menu_selected_colors[0] not in (COLOR_SYSTEM_DEFAULT, None):
                top_menu.config(activeforeground=toplevel_form.right_click_menu_selected_colors[0])
            if toplevel_form.right_click_menu_selected_colors[1] not in (COLOR_SYSTEM_DEFAULT, None):
                top_menu.config(activebackground=toplevel_form.right_click_menu_selected_colors[1])
            AddMenuItem(top_menu, menu[1], element, right_click_menu=True)
            element.TKRightClickMenu = top_menu
            if toplevel_form.RightClickMenu:            # if the top level has a right click menu, then setup a callback for the Window itself
                if toplevel_form.TKRightClickMenu is None:
                    toplevel_form.TKRightClickMenu = top_menu
                    if (running_mac()):
                        toplevel_form.TKroot.bind('<ButtonRelease-2>', toplevel_form._RightClickMenuCallback)
                    else:
                        toplevel_form.TKroot.bind('<ButtonRelease-3>', toplevel_form._RightClickMenuCallback)
            if (running_mac()):
                element.Widget.bind('<ButtonRelease-2>', element._RightClickMenuCallback)
            else:
                element.Widget.bind('<ButtonRelease-3>', element._RightClickMenuCallback)
                try:
                    if element.Type == ELEM_TYPE_COLUMN:
                        element.TKColFrame.canvas.bind('<ButtonRelease-3>', element._RightClickMenuCallback)
                except:
                    pass
        _add_grab(element)


    def _add_expansion(element, row_should_expand, row_fill_direction):
        expand = True
        if element.expand_x and element.expand_y:
            fill = tk.BOTH
            row_fill_direction = tk.BOTH
            row_should_expand = True
        elif element.expand_x:
            fill = tk.X
            row_fill_direction = tk.X if row_fill_direction == tk.NONE else tk.BOTH if row_fill_direction == tk.Y else tk.X
        elif element.expand_y:
            fill = tk.Y
            row_fill_direction = tk.Y if row_fill_direction == tk.NONE else tk.BOTH if row_fill_direction == tk.X else tk.Y
            row_should_expand = True
        else:
            fill = tk.NONE
            expand = False
        return expand, fill, row_should_expand, row_fill_direction

    tclversion_detailed = tkinter.Tcl().eval('info patchlevel')


    # --------------------------------------------------------------------------- #
    # ****************  Use FlexForm to build the tkinter window ********** ----- #
    # Building is done row by row.                                                #
    # WARNING - You can't use print in this function. If the user has rerouted   #
    # stdout then there will be an error saying the window isn't finalized        #
    # --------------------------------------------------------------------------- #
    ######################### LOOP THROUGH ROWS #########################
    # *********** -------  Loop through ROWS  ------- ***********#
    for row_num, flex_row in enumerate(form.Rows):
        ######################### LOOP THROUGH ELEMENTS ON ROW #########################
        # *********** -------  Loop through ELEMENTS  ------- ***********#
        # *********** Make TK Row                             ***********#
        tk_row_frame = tk.Frame(containing_frame)
        row_should_expand = False
        row_fill_direction = tk.NONE

        if form.ElementJustification is not None:
            row_justify = form.ElementJustification
        else:
            row_justify = 'l'

        for col_num, element in enumerate(flex_row):
            element.ParentRowFrame = tk_row_frame
            element.element_frame = None  # for elements that have a scrollbar too
            element.ParentForm = toplevel_form  # save the button's parent form object
            if toplevel_form.Font and (element.Font == DEFAULT_FONT or element.Font is None):
                font = toplevel_form.Font
            elif element.Font is not None:
                font = element.Font
            else:
                font = DEFAULT_FONT
            # -------  Determine Auto-Size setting on a cascading basis ------- #
            if element.AutoSizeText is not None:  # if element overide
                auto_size_text = element.AutoSizeText
            elif toplevel_form.AutoSizeText is not None:  # if form override
                auto_size_text = toplevel_form.AutoSizeText
            else:
                auto_size_text = DEFAULT_AUTOSIZE_TEXT
            element_type = element.Type
            # Set foreground color
            text_color = element.TextColor
            elementpad = element.Pad if element.Pad is not None else toplevel_form.ElementPadding
            # element.pad_used = elementpad  # store the value used back into the element
            # Determine Element size
            element_size = element.Size
            if (element_size == (None, None) and element_type not in (
                    ELEM_TYPE_BUTTON, ELEM_TYPE_BUTTONMENU)):  # user did not specify a size
                element_size = toplevel_form.DefaultElementSize
            elif (element_size == (None, None) and element_type in (ELEM_TYPE_BUTTON, ELEM_TYPE_BUTTONMENU)):
                element_size = toplevel_form.DefaultButtonElementSize
            else:
                auto_size_text = False  # if user has specified a size then it shouldn't autosize

            border_depth = toplevel_form.BorderDepth if toplevel_form.BorderDepth is not None else DEFAULT_BORDER_WIDTH
            try:
                if element.BorderWidth is not None:
                    border_depth = element.BorderWidth
            except:
                pass

            # -------------------------  COLUMN placement element  ------------------------- #
            if element_type == ELEM_TYPE_COLUMN:
                element = element  # type: Column
                # ----------------------- SCROLLABLE Column ----------------------
                if element.Scrollable:
                    element.Widget = element.TKColFrame = TkScrollableFrame(tk_row_frame, element.VerticalScrollOnly, element, toplevel_form)  # do not use yet!  not working
                    PackFormIntoFrame(element, element.TKColFrame.TKFrame, toplevel_form)
                    element.TKColFrame.TKFrame.update()
                    if element.Size == (None, None):  # if no size specified, use column width x column height/2
                        element.TKColFrame.canvas.config(width=element.TKColFrame.TKFrame.winfo_reqwidth() // element.size_subsample_width,
                                                         height=element.TKColFrame.TKFrame.winfo_reqheight() // element.size_subsample_height)
                    else:
                        element.TKColFrame.canvas.config(width=element.TKColFrame.TKFrame.winfo_reqwidth() // element.size_subsample_width,
                                                         height=element.TKColFrame.TKFrame.winfo_reqheight() // element.size_subsample_height)
                        if None not in (element.Size[0], element.Size[1]):
                            element.TKColFrame.canvas.config(width=element.Size[0], height=element.Size[1])
                        elif element.Size[1] is not None:
                            element.TKColFrame.canvas.config(height=element.Size[1])
                        elif element.Size[0] is not None:
                            element.TKColFrame.canvas.config(width=element.Size[0])
                    if not element.BackgroundColor in (None, COLOR_SYSTEM_DEFAULT):
                        element.TKColFrame.canvas.config(background=element.BackgroundColor)
                        element.TKColFrame.TKFrame.config(background=element.BackgroundColor, borderwidth=0, highlightthickness=0)
                        element.TKColFrame.config(background=element.BackgroundColor, borderwidth=0,
                                                  highlightthickness=0)
                # ----------------------- PLAIN Column ----------------------
                else:
                    if element.Size != (None, None):
                        element.Widget = element.TKColFrame = TkFixedFrame(tk_row_frame)
                        PackFormIntoFrame(element, element.TKColFrame.TKFrame, toplevel_form)
                        element.TKColFrame.TKFrame.update()
                        if None not in (element.Size[0], element.Size[1]):
                            element.TKColFrame.canvas.config(width=element.Size[0], height=element.Size[1])
                        elif element.Size[1] is not None:
                            element.TKColFrame.canvas.config(height=element.Size[1])
                        elif element.Size[0] is not None:
                            element.TKColFrame.canvas.config(width=element.Size[0])
                        if not element.BackgroundColor in (None, COLOR_SYSTEM_DEFAULT):
                            element.TKColFrame.canvas.config(background=element.BackgroundColor)
                            element.TKColFrame.TKFrame.config(background=element.BackgroundColor, borderwidth=0, highlightthickness=0)
                    else:
                        element.Widget = element.TKColFrame = tk.Frame(tk_row_frame)
                        PackFormIntoFrame(element, element.TKColFrame, toplevel_form)
                        if element.BackgroundColor not in (None, COLOR_SYSTEM_DEFAULT):
                            element.TKColFrame.config(background=element.BackgroundColor, borderwidth=0, highlightthickness=0)

                if element.Justification is None:
                    pass
                elif element.Justification.lower().startswith('l'):
                    row_justify = 'l'
                elif element.Justification.lower().startswith('c'):
                    row_justify = 'c'
                elif element.Justification.lower().startswith('r'):
                    row_justify = 'r'

                # anchor=tk.NW
                # side = tk.LEFT
                # row_justify = element.Justification

                # element.Widget = element.TKColFrame

                expand = True
                if element.expand_x and element.expand_y:
                    fill = tk.BOTH
                    row_fill_direction = tk.BOTH
                    row_should_expand = True
                elif element.expand_x:
                    fill = tk.X
                    row_fill_direction = tk.X
                elif element.expand_y:
                    fill = tk.Y
                    row_fill_direction = tk.Y
                    row_should_expand = True
                else:
                    fill = tk.NONE
                    expand = False

                if element.VerticalAlignment is not None:
                    anchor = tk.CENTER  # Default to center if a bad choice is made

                    if element.VerticalAlignment.lower().startswith('t'):
                        anchor = tk.N
                    if element.VerticalAlignment.lower().startswith('c'):
                        anchor = tk.CENTER
                    if element.VerticalAlignment.lower().startswith('b'):
                        anchor = tk.S
                    element.TKColFrame.pack(side=tk.LEFT, anchor=anchor, padx=elementpad[0], pady=elementpad[1], expand=expand, fill=fill)
                else:
                    element.TKColFrame.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], expand=expand, fill=fill)

                # element.TKColFrame.pack(side=side, padx=elementpad[0], pady=elementpad[1], expand=True, fill='both')
                if element.visible is False:
                    element._pack_forget_save_settings()
                    # element.TKColFrame.pack_forget()

                _add_right_click_menu_and_grab(element)
                # if element.Grab:
                #     element._grab_anywhere_on()
                # row_should_expand = True
            # -------------------------  Pane placement element  ------------------------- #
            if element_type == ELEM_TYPE_PANE:
                bd = element.BorderDepth if element.BorderDepth is not None else border_depth
                element.PanedWindow = element.Widget = tk.PanedWindow(tk_row_frame,
                                                                      orient=tk.VERTICAL if element.Orientation.startswith(
                                                                          'v') else tk.HORIZONTAL,
                                                                      borderwidth=bd,
                                                                      bd=bd,
                                                                      )
                if element.Relief is not None:
                    element.PanedWindow.configure(relief=element.Relief)
                element.PanedWindow.configure(handlesize=element.HandleSize)
                if element.ShowHandle:
                    element.PanedWindow.config(showhandle=True)
                if element.Size != (None, None):
                    element.PanedWindow.config(width=element.Size[0], height=element.Size[1])
                if element.BackgroundColor is not None and element.BackgroundColor != COLOR_SYSTEM_DEFAULT:
                    element.PanedWindow.configure(background=element.BackgroundColor)
                for pane in element.PaneList:
                    pane.Widget = pane.TKColFrame = tk.Frame(element.PanedWindow)
                    pane.ParentPanedWindow = element.PanedWindow
                    PackFormIntoFrame(pane, pane.TKColFrame, toplevel_form)
                    if pane.visible:
                        element.PanedWindow.add(pane.TKColFrame)
                    if pane.BackgroundColor != COLOR_SYSTEM_DEFAULT and pane.BackgroundColor is not None:
                        pane.TKColFrame.configure(background=pane.BackgroundColor,
                                                  highlightbackground=pane.BackgroundColor,
                                                  highlightcolor=pane.BackgroundColor)
                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)
                element.PanedWindow.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], expand=expand, fill=fill)
                # element.PanedWindow.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], expand=True, fill='both')
                if element.visible is False:
                    element._pack_forget_save_settings()
                    # element.PanedWindow.pack_forget()
            # -------------------------  TEXT placement element  ------------------------- #
            elif element_type == ELEM_TYPE_TEXT:
                # auto_size_text = element.AutoSizeText
                element = element  # type: Text
                display_text = element.DisplayText  # text to display
                if auto_size_text is False:
                    width, height = element_size
                else:
                    width, height = None, None
                    # lines = display_text.split('\n')
                    # max_line_len = max([len(l) for l in lines])
                    # num_lines = len(lines)
                    # if max_line_len > element_size[0]:  # if text exceeds element size, the will have to wrap
                    #     width = element_size[0]
                    # else:
                    #     width = max_line_len
                    # height = num_lines
                # ---===--- LABEL widget create and place --- #
                element = element  # type: Text
                bd = element.BorderWidth if element.BorderWidth is not None else border_depth
                stringvar = tk.StringVar()
                element.TKStringVar = stringvar
                stringvar.set(str(display_text))
                if auto_size_text:
                    width = 0
                if element.Justification is not None:
                    justification = element.Justification
                elif toplevel_form.TextJustification is not None:
                    justification = toplevel_form.TextJustification
                else:
                    justification = DEFAULT_TEXT_JUSTIFICATION
                justify = tk.LEFT if justification.startswith('l') else tk.CENTER if justification.startswith('c') else tk.RIGHT
                anchor = tk.NW if justification.startswith('l') else tk.N if justification.startswith('c') else tk.NE
                tktext_label = element.Widget = tk.Label(tk_row_frame, textvariable=stringvar, width=width,
                                                         height=height, justify=justify, bd=bd, font=font)
                # Set wrap-length for text (in PIXELS) == PAIN IN THE ASS
                wraplen = tktext_label.winfo_reqwidth()  # width of widget in Pixels
                if auto_size_text or (not auto_size_text and height == 1):  # if just 1 line high, ensure no wrap happens
                    wraplen = 0
                tktext_label.configure(anchor=anchor, wraplen=wraplen)  # set wrap to width of widget
                if element.Relief is not None:
                    tktext_label.configure(relief=element.Relief)
                if element.BackgroundColor is not None and element.BackgroundColor != COLOR_SYSTEM_DEFAULT:
                    tktext_label.configure(background=element.BackgroundColor)
                if element.TextColor != COLOR_SYSTEM_DEFAULT and element.TextColor is not None:
                    tktext_label.configure(fg=element.TextColor)
                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)
                tktext_label.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], expand=expand, fill=fill)
                if element.visible is False:
                    element._pack_forget_save_settings()
                    # tktext_label.pack_forget()
                element.TKText = tktext_label
                if element.ClickSubmits:
                    tktext_label.bind('<Button-1>', element._TextClickedHandler)
                if element.Tooltip is not None:
                    element.TooltipObject = ToolTip(element.TKText, text=element.Tooltip, timeout=DEFAULT_TOOLTIP_TIME)
                _add_right_click_menu_and_grab(element)
                if element.Grab:
                    element._grab_anywhere_on()
            # -------------------------  BUTTON placement element non-ttk version  ------------------------- #
            elif (element_type == ELEM_TYPE_BUTTON and element.UseTtkButtons is False) or \
                    (element_type == ELEM_TYPE_BUTTON and element.UseTtkButtons is not True and toplevel_form.UseTtkButtons is not True):
                element = element  # type: Button
                element.UseTtkButtons = False  # indicate that ttk button was not used
                stringvar = tk.StringVar()
                element.TKStringVar = stringvar
                element.Location = (row_num, col_num)
                btext = element.ButtonText
                btype = element.BType
                if element.AutoSizeButton is not None:
                    auto_size = element.AutoSizeButton
                else:
                    auto_size = toplevel_form.AutoSizeButtons
                if auto_size is False or element.Size[0] is not None:
                    width, height = element_size
                else:
                    width = 0
                    height = toplevel_form.DefaultButtonElementSize[1]
                if element.ButtonColor != (None, None) and element.ButtonColor != DEFAULT_BUTTON_COLOR:
                    bc = element.ButtonColor
                elif toplevel_form.ButtonColor != (None, None) and toplevel_form.ButtonColor != DEFAULT_BUTTON_COLOR:
                    bc = toplevel_form.ButtonColor
                else:
                    bc = DEFAULT_BUTTON_COLOR

                bd = element.BorderWidth
                pos = -1
                if DEFAULT_USE_BUTTON_SHORTCUTS is True:
                    pos = btext.find(MENU_SHORTCUT_CHARACTER)
                    if pos != -1:
                        if pos < len(MENU_SHORTCUT_CHARACTER) or btext[pos - len(MENU_SHORTCUT_CHARACTER)] != '\\':
                            btext = btext[:pos] + btext[pos + len(MENU_SHORTCUT_CHARACTER):]
                        else:
                            btext = btext.replace('\\'+MENU_SHORTCUT_CHARACTER, MENU_SHORTCUT_CHARACTER)
                            pos = -1
                tkbutton = element.Widget = tk.Button(tk_row_frame, text=btext, width=width, height=height, justify=tk.CENTER, bd=bd, font=font)
                if pos != -1:
                    tkbutton.config(underline=pos)
                try:
                    if btype != BUTTON_TYPE_REALTIME:
                        tkbutton.config( command=element.ButtonCallBack)

                    else:
                        tkbutton.bind('<ButtonRelease-1>', element.ButtonReleaseCallBack)
                        tkbutton.bind('<ButtonPress-1>', element.ButtonPressCallBack)
                    if bc != (None, None) and COLOR_SYSTEM_DEFAULT not in bc:
                        tkbutton.config(foreground=bc[0], background=bc[1])
                    else:
                        if bc[0] != COLOR_SYSTEM_DEFAULT:
                            tkbutton.config(foreground=bc[0])
                        if bc[1] != COLOR_SYSTEM_DEFAULT:
                            tkbutton.config(background=bc[1])
                except Exception as e:
                    _error_popup_with_traceback('Button has a problem....',
                                                'The traceback information will not show the line in your layout with the problem, but it does tell you which window.',
                                                'Error {}'.format(e),
                                                # 'Button Text: {}'.format(btext),
                                                # 'Button key: {}'.format(element.Key),
                                                # 'Color string: {}'.format(bc),
                                                "Parent Window's Title: {}".format(toplevel_form.Title))

                if bd == 0 and not running_mac():
                    tkbutton.config(relief=tk.FLAT)

                element.TKButton = tkbutton  # not used yet but save the TK button in case
                if elementpad[0] == 0 or elementpad[1] == 0:
                    tkbutton.config(highlightthickness=0)

                ## -------------- TK Button With Image -------------- ##
                if element.ImageFilename:  # if button has an image on it
                    tkbutton.config(highlightthickness=0)
                    try:
                        photo = tk.PhotoImage(file=element.ImageFilename)
                        if element.ImageSubsample:
                            photo = photo.subsample(element.ImageSubsample)
                        if element.zoom:
                            photo = photo.zoom(element.zoom)
                        if element.ImageSize != (None, None):
                            width, height = element.ImageSize
                        else:
                            width, height = photo.width(), photo.height()
                    except Exception as e:
                        _error_popup_with_traceback('Button Element error {}'.format(e), 'Image filename: {}'.format(element.ImageFilename),
                                                     'NOTE - file format must be PNG or GIF!',
                                                    'Button element key: {}'.format(element.Key),
                                                    "Parent Window's Title: {}".format(toplevel_form.Title))
                    tkbutton.config(image=photo, compound=tk.CENTER, width=width, height=height)
                    tkbutton.image = photo
                if element.ImageData:  # if button has an image on it
                    tkbutton.config(highlightthickness=0)
                    try:
                        photo = tk.PhotoImage(data=element.ImageData)
                        if element.ImageSubsample:
                            photo = photo.subsample(element.ImageSubsample)
                        if element.zoom:
                            photo = photo.zoom(element.zoom)
                        if element.ImageSize != (None, None):
                            width, height = element.ImageSize
                        else:
                            width, height = photo.width(), photo.height()
                        tkbutton.config(image=photo, compound=tk.CENTER, width=width, height=height)
                        tkbutton.image = photo
                    except Exception as e:
                        _error_popup_with_traceback('Button Element error {}'.format(e),
                                                    'Problem using BASE64 Image data Image Susample',
                                                    'Buton element key: {}'.format(element.Key),
                                                    "Parent Window's Title: {}".format(toplevel_form.Title))

                if width != 0:
                    wraplen = width * _char_width_in_pixels(font)
                    tkbutton.configure(wraplength=wraplen)  # set wrap to width of widget
                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)

                tkbutton.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], expand=expand, fill=fill)
                if element.visible is False:
                    element._pack_forget_save_settings()
                    # tkbutton.pack_forget()
                if element.BindReturnKey:
                    element.TKButton.bind('<Return>', element._ReturnKeyHandler)
                if element.Focus is True or (toplevel_form.UseDefaultFocus and not toplevel_form.FocusSet):
                    toplevel_form.FocusSet = True
                    element.TKButton.bind('<Return>', element._ReturnKeyHandler)
                    element.TKButton.focus_set()
                    toplevel_form.TKroot.focus_force()
                if element.Disabled is True:
                    element.TKButton['state'] = 'disabled'
                if element.DisabledButtonColor != (None, None) and element.DisabledButtonColor != (COLOR_SYSTEM_DEFAULT, COLOR_SYSTEM_DEFAULT):
                    if element.DisabledButtonColor[0] not in (None, COLOR_SYSTEM_DEFAULT):
                        element.TKButton['disabledforeground'] = element.DisabledButtonColor[0]
                if element.MouseOverColors[1] not in (COLOR_SYSTEM_DEFAULT, None):
                    tkbutton.config(activebackground=element.MouseOverColors[1])
                if element.MouseOverColors[0] not in (COLOR_SYSTEM_DEFAULT, None):
                    tkbutton.config(activeforeground=element.MouseOverColors[0])

                if element.Tooltip is not None:
                    element.TooltipObject = ToolTip(element.TKButton, text=element.Tooltip,
                                                    timeout=DEFAULT_TOOLTIP_TIME)
                try:
                    if element.HighlightColors[1] != COLOR_SYSTEM_DEFAULT:
                        tkbutton.config(highlightbackground=element.HighlightColors[1])
                    if element.HighlightColors[0] != COLOR_SYSTEM_DEFAULT:
                        tkbutton.config(highlightcolor=element.HighlightColors[0])
                except Exception as e:
                    _error_popup_with_traceback('Button Element error {}'.format(e),
                                                'Button element key: {}'.format(element.Key),
                                                'Button text: {}'.format(btext),
                                                'Has a bad highlight color {}'.format(element.HighlightColors),
                                                "Parent Window's Title: {}".format(toplevel_form.Title))
                    # print('Button with text: ', btext, 'has a bad highlight color', element.HighlightColors)
                _add_right_click_menu_and_grab(element)

            # -------------------------  BUTTON placement element ttk version ------------------------- #
            elif element_type == ELEM_TYPE_BUTTON:
                element = element  # type: Button
                element.UseTtkButtons = True  # indicate that ttk button was used
                stringvar = tk.StringVar()
                element.TKStringVar = stringvar
                element.Location = (row_num, col_num)
                btext = element.ButtonText
                pos = -1
                if DEFAULT_USE_BUTTON_SHORTCUTS is True:
                    pos = btext.find(MENU_SHORTCUT_CHARACTER)
                    if pos != -1:
                        if pos < len(MENU_SHORTCUT_CHARACTER) or btext[pos - len(MENU_SHORTCUT_CHARACTER)] != '\\':
                            btext = btext[:pos] + btext[pos + len(MENU_SHORTCUT_CHARACTER):]
                        else:
                            btext = btext.replace('\\'+MENU_SHORTCUT_CHARACTER, MENU_SHORTCUT_CHARACTER)
                            pos = -1
                btype = element.BType
                if element.AutoSizeButton is not None:
                    auto_size = element.AutoSizeButton
                else:
                    auto_size = toplevel_form.AutoSizeButtons
                if auto_size is False or element.Size[0] is not None:
                    width, height = element_size
                else:
                    width = 0
                    height = toplevel_form.DefaultButtonElementSize[1]
                if element.ButtonColor != (None, None) and element.ButtonColor != COLOR_SYSTEM_DEFAULT:
                    bc = element.ButtonColor
                elif toplevel_form.ButtonColor != (None, None) and toplevel_form.ButtonColor != COLOR_SYSTEM_DEFAULT:
                    bc = toplevel_form.ButtonColor
                else:
                    bc = DEFAULT_BUTTON_COLOR
                bd = element.BorderWidth
                tkbutton = element.Widget = ttk.Button(tk_row_frame, text=btext, width=width)
                if pos != -1:
                    tkbutton.config(underline=pos)
                if btype != BUTTON_TYPE_REALTIME:
                    tkbutton.config(command=element.ButtonCallBack)
                else:
                    tkbutton.bind('<ButtonRelease-1>', element.ButtonReleaseCallBack)
                    tkbutton.bind('<ButtonPress-1>', element.ButtonPressCallBack)
                style_name = _make_ttk_style_name('.TButton', element, primary_style=True)
                button_style = ttk.Style()
                element.ttk_style = button_style
                _change_ttk_theme(button_style, toplevel_form.TtkTheme)
                button_style.configure(style_name, font=font)

                if bc != (None, None) and COLOR_SYSTEM_DEFAULT not in bc:
                    button_style.configure(style_name, foreground=bc[0], background=bc[1])
                elif bc[0] != COLOR_SYSTEM_DEFAULT:
                    button_style.configure(style_name, foreground=bc[0])
                elif bc[1] != COLOR_SYSTEM_DEFAULT:
                    button_style.configure(style_name, background=bc[1])

                if bd == 0 and not running_mac():
                    button_style.configure(style_name, relief=tk.FLAT)
                    button_style.configure(style_name, borderwidth=0)
                else:
                    button_style.configure(style_name, borderwidth=bd)
                button_style.configure(style_name, justify=tk.CENTER)

                if element.MouseOverColors[1] not in (COLOR_SYSTEM_DEFAULT, None):
                    button_style.map(style_name, background=[('active', element.MouseOverColors[1])])
                if element.MouseOverColors[0] not in (COLOR_SYSTEM_DEFAULT, None):
                    button_style.map(style_name, foreground=[('active', element.MouseOverColors[0])])

                if element.DisabledButtonColor[0] not in (COLOR_SYSTEM_DEFAULT, None):
                    button_style.map(style_name, foreground=[('disabled', element.DisabledButtonColor[0])])
                if element.DisabledButtonColor[1] not in (COLOR_SYSTEM_DEFAULT, None):
                    button_style.map(style_name, background=[('disabled', element.DisabledButtonColor[1])])

                if height > 1:
                    button_style.configure(style_name, padding=height * _char_height_in_pixels(font))  # should this be height instead?
                if width != 0:
                    wraplen = width * _char_width_in_pixels(font) # width of widget in Pixels
                    button_style.configure(style_name, wraplength=wraplen)  # set wrap to width of widget

                ## -------------- TTK Button With Image -------------- ##
                if element.ImageFilename:  # if button has an image on it
                    button_style.configure(style_name, borderwidth=0)
                    # tkbutton.configure(highlightthickness=0)
                    photo = tk.PhotoImage(file=element.ImageFilename)
                    if element.ImageSubsample:
                        photo = photo.subsample(element.ImageSubsample)
                    if element.zoom:
                        photo = photo.zoom(element.zoom)
                    if element.ImageSize != (None, None):
                        width, height = element.ImageSize
                    else:
                        width, height = photo.width(), photo.height()
                    button_style.configure(style_name, image=photo, compound=tk.CENTER, width=width, height=height)
                    tkbutton.image = photo
                if element.ImageData:  # if button has an image on it
                    # tkbutton.configure(highlightthickness=0)
                    button_style.configure(style_name, borderwidth=0)

                    photo = tk.PhotoImage(data=element.ImageData)
                    if element.ImageSubsample:
                        photo = photo.subsample(element.ImageSubsample)
                    if element.zoom:
                        photo = photo.zoom(element.zoom)
                    if element.ImageSize != (None, None):
                        width, height = element.ImageSize
                    else:
                        width, height = photo.width(), photo.height()
                    button_style.configure(style_name, image=photo, compound=tk.CENTER, width=width, height=height)
                    # tkbutton.configure(image=photo, compound=tk.CENTER, width=width, height=height)
                    tkbutton.image = photo

                element.TKButton = tkbutton  # not used yet but save the TK button in case
                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)
                tkbutton.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], expand=expand, fill=fill)
                if element.visible is False:
                    element._pack_forget_save_settings()
                    # tkbutton.pack_forget()
                if element.BindReturnKey:
                    element.TKButton.bind('<Return>', element._ReturnKeyHandler)
                if element.Focus is True or (toplevel_form.UseDefaultFocus and not toplevel_form.FocusSet):
                    toplevel_form.FocusSet = True
                    element.TKButton.bind('<Return>', element._ReturnKeyHandler)
                    element.TKButton.focus_set()
                    toplevel_form.TKroot.focus_force()
                if element.Disabled is True:
                    element.TKButton['state'] = 'disabled'

                tkbutton.configure(style=style_name)  # IMPORTANT!  Apply the style to the button!
                _add_right_click_menu_and_grab(element)

                if element.Tooltip is not None:
                    element.TooltipObject = ToolTip(element.TKButton, text=element.Tooltip,
                                                    timeout=DEFAULT_TOOLTIP_TIME)
            # -------------------------  BUTTONMENU placement element  ------------------------- #
            elif element_type == ELEM_TYPE_BUTTONMENU:
                element = element  # type: ButtonMenu
                element.Location = (row_num, col_num)
                btext = element.ButtonText
                if element.AutoSizeButton is not None:
                    auto_size = element.AutoSizeButton
                else:
                    auto_size = toplevel_form.AutoSizeButtons
                if auto_size is False or element.Size[0] is not None:
                    width, height = element_size
                else:
                    width = 0
                    height = toplevel_form.DefaultButtonElementSize[1]
                if element.ButtonColor != (None, None) and element.ButtonColor != DEFAULT_BUTTON_COLOR:
                    bc = element.ButtonColor
                elif toplevel_form.ButtonColor != (None, None) and toplevel_form.ButtonColor != DEFAULT_BUTTON_COLOR:
                    bc = toplevel_form.ButtonColor
                else:
                    bc = DEFAULT_BUTTON_COLOR
                bd = element.BorderWidth
                if element.ItemFont is None:
                    element.ItemFont = font
                tkbutton = element.Widget = tk.Menubutton(tk_row_frame, text=btext, width=width, height=height, justify=tk.LEFT, bd=bd, font=font)
                element.TKButtonMenu = tkbutton
                if bc != (None, None) and bc != COLOR_SYSTEM_DEFAULT and bc[1] != COLOR_SYSTEM_DEFAULT:
                    tkbutton.config(foreground=bc[0], background=bc[1])
                    tkbutton.config(activebackground=bc[0])
                    tkbutton.config(activeforeground=bc[1])
                elif bc[0] != COLOR_SYSTEM_DEFAULT:
                    tkbutton.config(foreground=bc[0])
                    tkbutton.config(activebackground=bc[0])
                if bd == 0 and not running_mac():
                    tkbutton.config(relief=RELIEF_FLAT)
                elif bd != 0:
                    tkbutton.config(relief=RELIEF_RAISED)

                element.TKButton = tkbutton  # not used yet but save the TK button in case
                wraplen = tkbutton.winfo_reqwidth()  # width of widget in Pixels
                if element.ImageFilename:  # if button has an image on it
                    photo = tk.PhotoImage(file=element.ImageFilename)
                    if element.ImageSubsample:
                        photo = photo.subsample(element.ImageSubsample)
                    if element.zoom:
                        photo = photo.zoom(element.zoom)
                    if element.ImageSize != (None, None):
                        width, height = element.ImageSize
                    else:
                        width, height = photo.width(), photo.height()
                    tkbutton.config(image=photo, compound=tk.CENTER, width=width, height=height)
                    tkbutton.image = photo
                if element.ImageData:  # if button has an image on it
                    photo = tk.PhotoImage(data=element.ImageData)
                    if element.ImageSubsample:
                        photo = photo.subsample(element.ImageSubsample)
                    if element.zoom:
                        photo = photo.zoom(element.zoom)
                    if element.ImageSize != (None, None):
                        width, height = element.ImageSize
                    else:
                        width, height = photo.width(), photo.height()
                    tkbutton.config(image=photo, compound=tk.CENTER, width=width, height=height)
                    tkbutton.image = photo
                if width != 0:
                    tkbutton.configure(wraplength=wraplen + 10)  # set wrap to width of widget
                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)
                tkbutton.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], expand=expand, fill=fill)

                menu_def = element.MenuDefinition

                element.TKMenu = top_menu = tk.Menu(tkbutton, tearoff=element.Tearoff, font=element.ItemFont, tearoffcommand=element._tearoff_menu_callback)

                if element.BackgroundColor not in (COLOR_SYSTEM_DEFAULT, None):
                    top_menu.config(bg=element.BackgroundColor)
                    top_menu.config(activeforeground=element.BackgroundColor)
                if element.TextColor not in (COLOR_SYSTEM_DEFAULT, None):
                    top_menu.config(fg=element.TextColor)
                    top_menu.config(activebackground=element.TextColor)
                if element.DisabledTextColor not in (COLOR_SYSTEM_DEFAULT, None):
                    top_menu.config(disabledforeground=element.DisabledTextColor)
                if element.ItemFont is not None:
                    top_menu.config(font=element.ItemFont)

                AddMenuItem(top_menu, menu_def[1], element)
                if elementpad[0] == 0 or elementpad[1] == 0:
                    tkbutton.config(highlightthickness=0)
                tkbutton.configure(menu=top_menu)
                element.TKMenu = top_menu
                if element.visible is False:
                    element._pack_forget_save_settings()
                    # tkbutton.pack_forget()
                if element.Disabled == True:
                    element.TKButton['state'] = 'disabled'
                if element.Tooltip is not None:
                    element.TooltipObject = ToolTip(element.TKButton, text=element.Tooltip, timeout=DEFAULT_TOOLTIP_TIME)

            # -------------------------  INPUT placement element  ------------------------- #
            elif element_type == ELEM_TYPE_INPUT_TEXT:
                element = element  # type: InputText
                default_text = element.DefaultText
                element.TKStringVar = tk.StringVar()
                element.TKStringVar.set(default_text)
                show = element.PasswordCharacter if element.PasswordCharacter else ''
                bd = border_depth
                if element.Justification is not None:
                    justification = element.Justification
                else:
                    justification = DEFAULT_TEXT_JUSTIFICATION
                justify = tk.LEFT if justification.startswith('l') else tk.CENTER if justification.startswith('c') else tk.RIGHT
                # anchor = tk.NW if justification == 'left' else tk.N if justification == 'center' else tk.NE
                element.TKEntry = element.Widget = tk.Entry(tk_row_frame, width=element_size[0],
                                                            textvariable=element.TKStringVar, bd=bd,
                                                            font=font, show=show, justify=justify)
                if element.ChangeSubmits:
                    element.TKEntry.bind('<Key>', element._KeyboardHandler)
                element.TKEntry.bind('<Return>', element._ReturnKeyHandler)


                if element.BackgroundColor not in (None, COLOR_SYSTEM_DEFAULT):
                    element.TKEntry.configure(background=element.BackgroundColor, selectforeground=element.BackgroundColor)

                if text_color not in (None, COLOR_SYSTEM_DEFAULT):
                    element.TKEntry.configure(fg=text_color, selectbackground=text_color)
                    element.TKEntry.config(insertbackground=text_color)
                if element.selected_background_color not in (None, COLOR_SYSTEM_DEFAULT):
                    element.TKEntry.configure(selectbackground=element.selected_background_color)
                if element.selected_text_color not in (None, COLOR_SYSTEM_DEFAULT):
                    element.TKEntry.configure(selectforeground=element.selected_text_color)
                if element.disabled_readonly_background_color not in (None, COLOR_SYSTEM_DEFAULT):
                    element.TKEntry.config(readonlybackground=element.disabled_readonly_background_color)
                if element.disabled_readonly_text_color not in (None, COLOR_SYSTEM_DEFAULT) and element.Disabled:
                    element.TKEntry.config(fg=element.disabled_readonly_text_color)

                element.Widget.config(highlightthickness=0)
                # element.pack_keywords = {'side':tk.LEFT, 'padx':elementpad[0], 'pady':elementpad[1], 'expand':False, 'fill':tk.NONE }
                # element.TKEntry.pack(**element.pack_keywords)
                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)
                element.TKEntry.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], expand=expand, fill=fill)
                if element.visible is False:
                    element._pack_forget_save_settings()
                    # element.TKEntry.pack_forget()
                if element.Focus is True or (toplevel_form.UseDefaultFocus and not toplevel_form.FocusSet):
                    toplevel_form.FocusSet = True
                    element.TKEntry.focus_set()
                if element.Disabled:
                    element.TKEntry['state'] = 'readonly' if element.UseReadonlyForDisable else 'disabled'
                if element.ReadOnly:
                    element.TKEntry['state'] = 'readonly'

                if element.Tooltip is not None:
                    element.TooltipObject = ToolTip(element.TKEntry, text=element.Tooltip, timeout=DEFAULT_TOOLTIP_TIME)
                _add_right_click_menu_and_grab(element)

                # row_should_expand = True

            # -------------------------  COMBO placement element  ------------------------- #
            elif element_type == ELEM_TYPE_INPUT_COMBO:
                element = element  # type: Combo
                max_line_len = max([len(str(l)) for l in element.Values]) if len(element.Values) else 0
                if auto_size_text is False:
                    width = element_size[0]
                else:
                    width = max_line_len + 1
                element.TKStringVar = tk.StringVar()
                style_name = _make_ttk_style_name('.TCombobox', element, primary_style=True)
                combostyle = ttk.Style()
                element.ttk_style = combostyle
                _change_ttk_theme(combostyle, toplevel_form.TtkTheme)

                # Creates a unique name for each field element(Sure there is a better way to do this)
                # unique_field = _make_ttk_style_name('.TCombobox.field', element)


                # Set individual widget options
                try:
                    if element.TextColor not in (None, COLOR_SYSTEM_DEFAULT):
                        combostyle.configure(style_name, foreground=element.TextColor)
                        combostyle.configure(style_name, selectbackground=element.TextColor)
                        combostyle.configure(style_name, insertcolor=element.TextColor)
                        combostyle.map(style_name, fieldforeground=[('readonly', element.TextColor)])
                    if element.BackgroundColor not in (None, COLOR_SYSTEM_DEFAULT):
                        combostyle.configure(style_name, selectforeground=element.BackgroundColor)
                        combostyle.map(style_name, fieldbackground=[('readonly', element.BackgroundColor)])
                        combostyle.configure(style_name, fieldbackground=element.BackgroundColor)

                    if element.button_arrow_color not in (None, COLOR_SYSTEM_DEFAULT):
                        combostyle.configure(style_name, arrowcolor=element.button_arrow_color)
                    if element.button_background_color not in (None, COLOR_SYSTEM_DEFAULT):
                        combostyle.configure(style_name, background=element.button_background_color)
                    if element.Readonly is True:
                        if element.TextColor not in (None, COLOR_SYSTEM_DEFAULT):
                            combostyle.configure(style_name, selectforeground=element.TextColor)
                        if element.BackgroundColor not in (None, COLOR_SYSTEM_DEFAULT):
                            combostyle.configure(style_name, selectbackground=element.BackgroundColor)


                except Exception as e:
                    _error_popup_with_traceback('Combo Element error {}'.format(e),
                                                'Combo element key: {}'.format(element.Key),
                                                'One of your colors is bad. Check the text, background, button background and button arrow colors',
                                                "Parent Window's Title: {}".format(toplevel_form.Title))

                # Strange code that is needed to set the font for the drop-down list
                element._dropdown_newfont = tkinter.font.Font(font=font)
                tk_row_frame.option_add('*TCombobox*Listbox*Font', element._dropdown_newfont)

                element.TKCombo = element.Widget = ttk.Combobox(tk_row_frame, width=width, textvariable=element.TKStringVar, font=font, style=style_name)

                # make tcl call to deal with colors for the drop-down formatting
                try:
                    if element.BackgroundColor not in (None, COLOR_SYSTEM_DEFAULT) and \
                        element.TextColor not in (None, COLOR_SYSTEM_DEFAULT):
                        element.Widget.tk.eval(
                    '[ttk::combobox::PopdownWindow {}].f.l configure -foreground {} -background {} -selectforeground {} -selectbackground {}'.format(element.Widget, element.TextColor, element.BackgroundColor, element.BackgroundColor, element.TextColor))
                except Exception as e:
                    pass    # going to let this one slide

                # Chr0nic
                element.TKCombo.bind('<Enter>', lambda event, em=element: testMouseHook2(em))
                element.TKCombo.bind('<Leave>', lambda event, em=element: testMouseUnhook2(em))

                if toplevel_form.UseDefaultFocus and not toplevel_form.FocusSet:
                    toplevel_form.FocusSet = True
                    element.TKCombo.focus_set()

                if element.Size[1] != 1 and element.Size[1] is not None:
                    element.TKCombo.configure(height=element.Size[1])
                element.TKCombo['values'] = element.Values
                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)
                element.TKCombo.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], expand=expand, fill=fill)
                if element.visible is False:
                    element._pack_forget_save_settings()
                    # element.TKCombo.pack_forget()
                if element.DefaultValue is not None:
                    element.TKCombo.set(element.DefaultValue)
                    # for i, v in enumerate(element.Values):
                    #     if v == element.DefaultValue:
                    #         element.TKCombo.current(i)
                    #         break
                # elif element.Values:
                #     element.TKCombo.current(0)
                if element.ChangeSubmits:
                    element.TKCombo.bind('<<ComboboxSelected>>', element._ComboboxSelectHandler)
                if element.BindReturnKey:
                    element.TKCombo.bind('<Return>', element._ComboboxSelectHandler)
                if element.enable_per_char_events:
                    element.TKCombo.bind('<Key>', element._KeyboardHandler)
                if element.Readonly:
                    element.TKCombo['state'] = 'readonly'
                if element.Disabled is True:  # note overrides readonly if disabled
                    element.TKCombo['state'] = 'disabled'
                if element.Tooltip is not None:
                    element.TooltipObject = ToolTip(element.TKCombo, text=element.Tooltip, timeout=DEFAULT_TOOLTIP_TIME)
                _add_right_click_menu_and_grab(element)

            # -------------------------  OPTIONMENU placement Element (Like ComboBox but different) element  ------------------------- #
            elif element_type == ELEM_TYPE_INPUT_OPTION_MENU:
                max_line_len = max([len(str(l)) for l in element.Values])
                if auto_size_text is False:
                    width = element_size[0]
                else:
                    width = max_line_len
                element.TKStringVar = tk.StringVar()
                if element.DefaultValue:
                    element.TKStringVar.set(element.DefaultValue)
                element.TKOptionMenu = element.Widget = tk.OptionMenu(tk_row_frame, element.TKStringVar, *element.Values)
                element.TKOptionMenu.config(highlightthickness=0, font=font, width=width)
                element.TKOptionMenu['menu'].config(font=font)
                element.TKOptionMenu.config(borderwidth=border_depth)
                if element.BackgroundColor is not None and element.BackgroundColor != COLOR_SYSTEM_DEFAULT:
                    element.TKOptionMenu.configure(background=element.BackgroundColor)
                    element.TKOptionMenu['menu'].config(background=element.BackgroundColor)
                if element.TextColor != COLOR_SYSTEM_DEFAULT and element.TextColor is not None:
                    element.TKOptionMenu.configure(fg=element.TextColor)
                    element.TKOptionMenu['menu'].config(fg=element.TextColor)
                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)
                element.TKOptionMenu.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], expand=expand, fill=fill)
                if element.visible is False:
                    element._pack_forget_save_settings()
                    # element.TKOptionMenu.pack_forget()
                if element.Disabled == True:
                    element.TKOptionMenu['state'] = 'disabled'
                if element.Tooltip is not None:
                    element.TooltipObject = ToolTip(element.TKOptionMenu, text=element.Tooltip,
                                                    timeout=DEFAULT_TOOLTIP_TIME)
            # -------------------------  LISTBOX placement element  ------------------------- #
            elif element_type == ELEM_TYPE_INPUT_LISTBOX:
                element = element  # type: Listbox
                max_line_len = max([len(str(l)) for l in element.Values]) if len(element.Values) else 0
                if auto_size_text is False:
                    width = element_size[0]
                else:
                    width = max_line_len
                element_frame = tk.Frame(tk_row_frame)
                element.element_frame = element_frame

                justification = tk.LEFT
                if element.justification is not None:
                    if element.justification.startswith('l'):
                        justification = tk.LEFT
                    elif element.justification.startswith('r'):
                        justification = tk.RIGHT
                    elif element.justification.startswith('c'):
                        justification = tk.CENTER

                element.TKStringVar = tk.StringVar()
                element.TKListbox = element.Widget = tk.Listbox(element_frame, height=element_size[1], width=width,
                                                                selectmode=element.SelectMode, font=font, exportselection=False)
                # On OLD versions of tkinter the justify option isn't available
                try:
                    element.Widget.config(justify=justification)
                except:
                    pass

                element.Widget.config(highlightthickness=0)
                for index, item in enumerate(element.Values):
                    element.TKListbox.insert(tk.END, item)
                    if element.DefaultValues is not None and item in element.DefaultValues:
                        element.TKListbox.selection_set(index)
                if element.BackgroundColor is not None and element.BackgroundColor != COLOR_SYSTEM_DEFAULT:
                    element.TKListbox.configure(background=element.BackgroundColor)
                if element.HighlightBackgroundColor is not None and element.HighlightBackgroundColor != COLOR_SYSTEM_DEFAULT:
                    element.TKListbox.config(selectbackground=element.HighlightBackgroundColor)
                if text_color is not None and text_color != COLOR_SYSTEM_DEFAULT:
                    element.TKListbox.configure(fg=text_color)
                if element.HighlightTextColor is not None and element.HighlightTextColor != COLOR_SYSTEM_DEFAULT:
                    element.TKListbox.config(selectforeground=element.HighlightTextColor)
                if element.ChangeSubmits:
                    element.TKListbox.bind('<<ListboxSelect>>', element._ListboxSelectHandler)

                if not element.NoScrollbar:
                    _make_ttk_scrollbar(element, 'v', toplevel_form)
                    element.Widget.configure(yscrollcommand=element.vsb.set)
                    element.vsb.pack(side=tk.RIGHT, fill='y')

                # Horizontal scrollbar
                if element.HorizontalScroll:
                    _make_ttk_scrollbar(element, 'h', toplevel_form)
                    element.hsb.pack(side=tk.BOTTOM, fill='x')
                    element.Widget.configure(xscrollcommand=element.hsb.set)

                if not element.NoScrollbar or element.HorizontalScroll:
                    # Chr0nic
                    element.Widget.bind('<Enter>', lambda event, em=element: testMouseHook(em))
                    element.Widget.bind('<Leave>', lambda event, em=element: testMouseUnhook(em))

                # else:
                #     element.TKText.config(wrap='word')

                # if not element.NoScrollbar:
                #     # Vertical scrollbar
                #     element.vsb = tk.Scrollbar(element_frame, orient="vertical", command=element.TKListbox.yview)
                #     element.TKListbox.configure(yscrollcommand=element.vsb.set)
                #     element.vsb.pack(side=tk.RIGHT, fill='y')

                # Horizontal scrollbar
                # if element.HorizontalScroll:
                #     hscrollbar = tk.Scrollbar(element_frame, orient=tk.HORIZONTAL)
                #     hscrollbar.pack(side=tk.BOTTOM, fill='x')
                #     hscrollbar.config(command=element.Widget.xview)
                #     element.Widget.configure(xscrollcommand=hscrollbar.set)
                #     element.hsb = hscrollbar
                #
                #     # Chr0nic
                #     element.TKListbox.bind("<Enter>", lambda event, em=element: testMouseHook(em))
                #     element.TKListbox.bind("<Leave>", lambda event, em=element: testMouseUnhook(em))
                #
                #



                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)
                element_frame.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], fill=fill, expand=expand)
                element.TKListbox.pack(side=tk.LEFT, fill=fill, expand=expand)
                if element.visible is False:
                    element._pack_forget_save_settings(alternate_widget=element_frame)
                    # element_frame.pack_forget()
                if element.BindReturnKey:
                    element.TKListbox.bind('<Return>', element._ListboxSelectHandler)
                    element.TKListbox.bind('<Double-Button-1>', element._ListboxSelectHandler)
                if element.Disabled is True:
                    element.TKListbox['state'] = 'disabled'
                if element.Tooltip is not None:
                    element.TooltipObject = ToolTip(element.TKListbox, text=element.Tooltip,
                                                    timeout=DEFAULT_TOOLTIP_TIME)
                _add_right_click_menu_and_grab(element)
            # -------------------------  MULTILINE placement element  ------------------------- #
            elif element_type == ELEM_TYPE_INPUT_MULTILINE:
                element = element  # type: Multiline
                width, height = element_size
                bd = element.BorderWidth
                element.element_frame = element_frame = tk.Frame(tk_row_frame)

                # if element.no_scrollbar:
                element.TKText = element.Widget = tk.Text(element_frame, width=width, height=height,  bd=bd, font=font, relief=RELIEF_SUNKEN)
                # else:
                #     element.TKText = element.Widget = tk.scrolledtext.ScrolledText(element_frame, width=width, height=height, bd=bd, font=font, relief=RELIEF_SUNKEN)

                if not element.no_scrollbar:
                    _make_ttk_scrollbar(element, 'v', toplevel_form)

                    element.Widget.configure(yscrollcommand=element.vsb.set)
                    element.vsb.pack(side=tk.RIGHT, fill='y')

                # Horizontal scrollbar
                if element.HorizontalScroll:
                    element.TKText.config(wrap='none')
                    _make_ttk_scrollbar(element, 'h', toplevel_form)
                    element.hsb.pack(side=tk.BOTTOM, fill='x')
                    element.Widget.configure(xscrollcommand=element.hsb.set)
                else:
                    element.TKText.config(wrap='word')

                if element.wrap_lines is True:
                    element.TKText.config(wrap='word')
                elif element.wrap_lines is False:
                    element.TKText.config(wrap='none')

                if not element.no_scrollbar or element.HorizontalScroll:
                    # Chr0nic
                    element.TKText.bind('<Enter>', lambda event, em=element: testMouseHook(em))
                    element.TKText.bind('<Leave>', lambda event, em=element: testMouseUnhook(em))

                if element.DefaultText:
                    element.TKText.insert(1.0, element.DefaultText)  # set the default text
                element.TKText.config(highlightthickness=0)
                if text_color is not None and text_color != COLOR_SYSTEM_DEFAULT:
                    element.TKText.configure(fg=text_color, selectbackground=text_color)
                    element.TKText.config(insertbackground=text_color)
                if element.BackgroundColor is not None and element.BackgroundColor != COLOR_SYSTEM_DEFAULT:
                    element.TKText.configure(background=element.BackgroundColor, selectforeground=element.BackgroundColor)
                if element.selected_background_color not in (None, COLOR_SYSTEM_DEFAULT):
                    element.TKText.configure(selectbackground=element.selected_background_color)
                if element.selected_text_color not in (None, COLOR_SYSTEM_DEFAULT):
                    element.TKText.configure(selectforeground=element.selected_text_color)
                element.TKText.tag_configure('center', justify='center')
                element.TKText.tag_configure('left', justify='left')
                element.TKText.tag_configure('right', justify='right')

                if element.Justification.startswith('l'):
                    element.TKText.tag_add('left', 1.0, 'end')
                    element.justification_tag = 'left'
                elif element.Justification.startswith('r'):
                    element.TKText.tag_add('right', 1.0, 'end')
                    element.justification_tag = 'right'
                elif element.Justification.startswith('c'):
                    element.TKText.tag_add('center', 1.0, 'end')
                    element.justification_tag = 'center'
                # if DEFAULT_SCROLLBAR_COLOR not in (None, COLOR_SYSTEM_DEFAULT):               # only works on Linux so not including it
                #     element.TKText.vbar.config(troughcolor=DEFAULT_SCROLLBAR_COLOR)
                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)

                element.element_frame.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], fill=fill, expand=expand)
                element.Widget.pack(side=tk.LEFT, fill=fill, expand=expand)

                if element.visible is False:
                    element._pack_forget_save_settings(alternate_widget=element_frame)
                    # element.element_frame.pack_forget()
                else:
                    # Chr0nic
                    element.TKText.bind('<Enter>', lambda event, em=element: testMouseHook(em))
                    element.TKText.bind('<Leave>', lambda event, em=element: testMouseUnhook(em))
                if element.ChangeSubmits:
                    element.TKText.bind('<Key>', element._KeyboardHandler)
                if element.EnterSubmits:
                    element.TKText.bind('<Return>', element._ReturnKeyHandler)
                if element.Focus is True or (toplevel_form.UseDefaultFocus and not toplevel_form.FocusSet):
                    toplevel_form.FocusSet = True
                    element.TKText.focus_set()


                if element.Disabled is True:
                    element.TKText['state'] = 'disabled'
                if element.Tooltip is not None:
                    element.TooltipObject = ToolTip(element.TKText, text=element.Tooltip, timeout=DEFAULT_TOOLTIP_TIME)

                if element.reroute_cprint:
                    cprint_set_output_destination(toplevel_form, element.Key)

                _add_right_click_menu_and_grab(element)

                if element.reroute_stdout:
                    element.reroute_stdout_to_here()
                if element.reroute_stderr:
                    element.reroute_stderr_to_here()

                # row_should_expand = True
            # -------------------------  CHECKBOX pleacement element  ------------------------- #
            elif element_type == ELEM_TYPE_INPUT_CHECKBOX:
                element = element  # type: Checkbox
                width = 0 if auto_size_text else element_size[0]
                default_value = element.InitialState
                element.TKIntVar = tk.IntVar()
                element.TKIntVar.set(default_value if default_value is not None else 0)

                element.TKCheckbutton = element.Widget = tk.Checkbutton(tk_row_frame, anchor=tk.NW,
                                                                            text=element.Text, width=width,
                                                                            variable=element.TKIntVar, bd=border_depth,
                                                                            font=font)
                if element.ChangeSubmits:
                    element.TKCheckbutton.configure(command=element._CheckboxHandler)
                if element.Disabled:
                    element.TKCheckbutton.configure(state='disable')
                if element.BackgroundColor is not None and element.BackgroundColor != COLOR_SYSTEM_DEFAULT:
                    element.TKCheckbutton.configure(background=element.BackgroundColor)
                    element.TKCheckbutton.configure(selectcolor=element.CheckboxBackgroundColor)  # The background of the checkbox
                    element.TKCheckbutton.configure(activebackground=element.BackgroundColor)
                if text_color is not None and text_color != COLOR_SYSTEM_DEFAULT:
                    element.TKCheckbutton.configure(fg=text_color)
                    element.TKCheckbutton.configure(activeforeground=element.TextColor)

                element.Widget.configure(highlightthickness=element.highlight_thickness)
                if element.BackgroundColor != COLOR_SYSTEM_DEFAULT:
                    element.TKCheckbutton.config(highlightbackground=element.BackgroundColor)
                if element.TextColor != COLOR_SYSTEM_DEFAULT:
                    element.TKCheckbutton.config(highlightcolor=element.TextColor)
                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)
                element.TKCheckbutton.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], expand=expand, fill=fill)
                if element.visible is False:
                    element._pack_forget_save_settings()
                    # element.TKCheckbutton.pack_forget()
                if element.Tooltip is not None:
                    element.TooltipObject = ToolTip(element.TKCheckbutton, text=element.Tooltip,
                                                    timeout=DEFAULT_TOOLTIP_TIME)
                _add_right_click_menu_and_grab(element)

            # -------------------------  PROGRESS placement element  ------------------------- #
            elif element_type == ELEM_TYPE_PROGRESS_BAR:
                element = element  # type: ProgressBar
                if element.size_px != (None, None):
                    progress_length, progress_width = element.size_px
                else:
                    width = element_size[0]
                    fnt = tkinter.font.Font()
                    char_width = fnt.measure('A')  # single character width
                    progress_length = width * char_width
                    progress_width = element_size[1]
                direction = element.Orientation
                if element.BarColor != (None, None):  # if element has a bar color, use it
                    bar_color = element.BarColor
                else:
                    bar_color = DEFAULT_PROGRESS_BAR_COLOR
                if element.Orientation.lower().startswith('h'):
                    base_style_name = '.Horizontal.TProgressbar'
                else:
                    base_style_name = '.Vertical.TProgressbar'
                style_name = _make_ttk_style_name(base_style_name, element, primary_style=True)
                element.TKProgressBar = TKProgressBar(tk_row_frame, element.MaxValue, progress_length, progress_width,
                                                      orientation=direction, BarColor=bar_color,
                                                      border_width=element.BorderWidth, relief=element.Relief,
                                                      ttk_theme=toplevel_form.TtkTheme, key=element.Key, style_name=style_name)
                element.Widget = element.TKProgressBar.TKProgressBarForReal
                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)
                element.TKProgressBar.TKProgressBarForReal.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], expand=expand, fill=fill)
                if element.visible is False:
                    element._pack_forget_save_settings(alternate_widget=element.TKProgressBar.TKProgressBarForReal)
                    # element.TKProgressBar.TKProgressBarForReal.pack_forget()
                _add_right_click_menu_and_grab(element)

                # -------------------------  RADIO placement element  ------------------------- #
            elif element_type == ELEM_TYPE_INPUT_RADIO:
                element = element  # type: Radio
                width = 0 if auto_size_text else element_size[0]
                default_value = element.InitialState
                ID = element.GroupID
                # see if ID has already been placed
                value = EncodeRadioRowCol(form.ContainerElemementNumber, row_num,
                                          col_num)  # value to set intvar to if this radio is selected
                element.EncodedRadioValue = value
                if ID in toplevel_form.RadioDict:
                    RadVar = toplevel_form.RadioDict[ID]
                else:
                    RadVar = tk.IntVar()
                    toplevel_form.RadioDict[ID] = RadVar
                element.TKIntVar = RadVar  # store the RadVar in Radio object
                if default_value:  # if this radio is the one selected, set RadVar to match
                    element.TKIntVar.set(value)
                element.TKRadio = element.Widget = tk.Radiobutton(tk_row_frame, anchor=tk.NW, text=element.Text,
                                                                  width=width, variable=element.TKIntVar, value=value,
                                                                  bd=border_depth, font=font)
                if element.ChangeSubmits:
                    element.TKRadio.configure(command=element._RadioHandler)
                if not element.BackgroundColor in (None, COLOR_SYSTEM_DEFAULT):
                    element.TKRadio.configure(background=element.BackgroundColor)
                    element.TKRadio.configure(selectcolor=element.CircleBackgroundColor)
                    element.TKRadio.configure(activebackground=element.BackgroundColor)
                if text_color is not None and text_color != COLOR_SYSTEM_DEFAULT:
                    element.TKRadio.configure(fg=text_color)
                    element.TKRadio.configure(activeforeground=text_color)

                element.Widget.configure(highlightthickness=1)
                if element.BackgroundColor != COLOR_SYSTEM_DEFAULT:
                    element.TKRadio.config(highlightbackground=element.BackgroundColor)
                if element.TextColor != COLOR_SYSTEM_DEFAULT:
                    element.TKRadio.config(highlightcolor=element.TextColor)

                if element.Disabled:
                    element.TKRadio['state'] = 'disabled'
                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)
                element.TKRadio.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], expand=expand, fill=fill)
                if element.visible is False:
                    element._pack_forget_save_settings()
                    # element.TKRadio.pack_forget()
                if element.Tooltip is not None:
                    element.TooltipObject = ToolTip(element.TKRadio, text=element.Tooltip, timeout=DEFAULT_TOOLTIP_TIME)
                _add_right_click_menu_and_grab(element)

                # -------------------------  SPIN placement element  ------------------------- #
            elif element_type == ELEM_TYPE_INPUT_SPIN:
                element = element  # type: Spin
                width, height = element_size
                width = 0 if auto_size_text else element_size[0]
                element.TKStringVar = tk.StringVar()
                element.TKSpinBox = element.Widget = tk.Spinbox(tk_row_frame, values=element.Values, textvariable=element.TKStringVar, width=width, bd=border_depth)
                if element.DefaultValue is not None:
                    element.TKStringVar.set(element.DefaultValue)
                element.TKSpinBox.configure(font=font)  # set wrap to width of widget
                if element.BackgroundColor is not None and element.BackgroundColor != COLOR_SYSTEM_DEFAULT:
                    element.TKSpinBox.configure(background=element.BackgroundColor)
                    element.TKSpinBox.configure(buttonbackground=element.BackgroundColor)
                if text_color  not in (None, COLOR_SYSTEM_DEFAULT):
                    element.TKSpinBox.configure(fg=text_color)
                    element.TKSpinBox.config(insertbackground=text_color)
                element.Widget.config(highlightthickness=0)
                if element.wrap is True:
                    element.Widget.configure(wrap=True)
                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)
                element.TKSpinBox.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], expand=expand, fill=fill)
                if element.visible is False:
                    element._pack_forget_save_settings()
                    # element.TKSpinBox.pack_forget()
                if element.ChangeSubmits:
                    element.TKSpinBox.configure(command=element._SpinboxSelectHandler)
                    # element.TKSpinBox.bind('<ButtonRelease-1>', element._SpinChangedHandler)
                    # element.TKSpinBox.bind('<Up>', element._SpinChangedHandler)
                    # element.TKSpinBox.bind('<Down>', element._SpinChangedHandler)
                if element.Readonly:
                    element.TKSpinBox['state'] = 'readonly'
                if element.Disabled is True:  # note overrides readonly if disabled
                    element.TKSpinBox['state'] = 'disabled'
                if element.Tooltip is not None:
                    element.TooltipObject = ToolTip(element.TKSpinBox, text=element.Tooltip, timeout=DEFAULT_TOOLTIP_TIME)
                if element.BindReturnKey:
                    element.TKSpinBox.bind('<Return>', element._SpinboxSelectHandler)
                _add_right_click_menu_and_grab(element)
                # -------------------------  IMAGE placement element  ------------------------- #
            elif element_type == ELEM_TYPE_IMAGE:
                element = element  # type: Image
                try:
                    if element.Filename is not None:
                        photo = tk.PhotoImage(file=element.Filename)
                    elif element.Data is not None:
                        photo = tk.PhotoImage(data=element.Data)
                    else:
                        photo = None

                    if photo is not None:
                        if element.ImageSubsample:
                            photo = photo.subsample(element.ImageSubsample)
                        if element.zoom:
                            photo = photo.zoom(element.zoom)
                        # print('*ERROR laying out form.... Image Element has no image specified*')
                except Exception as e:
                    photo = None
                    _error_popup_with_traceback('Your Window has an Image Element with a problem',
                                                'The traceback will show you the Window with the problem layout',
                                                'Look in this Window\'s layout for an Image element that has a key of {}'.format(element.Key),
                                                'The error occuring is:', e)

                element.tktext_label = element.Widget = tk.Label(tk_row_frame, bd=0)

                if photo is not None:
                    if element_size == (None, None) or element_size is None or element_size == toplevel_form.DefaultElementSize:
                        width, height = photo.width(), photo.height()
                    else:
                        width, height = element_size
                    element.tktext_label.config(image=photo, width=width, height=height)


                if not element.BackgroundColor in (None, COLOR_SYSTEM_DEFAULT):
                    element.tktext_label.config(background=element.BackgroundColor)

                element.tktext_label.image = photo
                # tktext_label.configure(anchor=tk.NW, image=photo)
                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)
                element.tktext_label.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], expand=expand, fill=fill)

                if element.visible is False:
                    element._pack_forget_save_settings()
                    # element.tktext_label.pack_forget()
                if element.Tooltip is not None:
                    element.TooltipObject = ToolTip(element.tktext_label, text=element.Tooltip,
                                                    timeout=DEFAULT_TOOLTIP_TIME)
                if element.EnableEvents and element.tktext_label is not None:
                    element.tktext_label.bind('<ButtonPress-1>', element._ClickHandler)

                _add_right_click_menu_and_grab(element)

                # -------------------------  Canvas placement element  ------------------------- #
            elif element_type == ELEM_TYPE_CANVAS:
                element = element  # type: Canvas
                width, height = element_size
                if element._TKCanvas is None:
                    element._TKCanvas = tk.Canvas(tk_row_frame, width=width, height=height, bd=border_depth)
                else:
                    element._TKCanvas.master = tk_row_frame
                element.Widget = element._TKCanvas

                if element.BackgroundColor is not None and element.BackgroundColor != COLOR_SYSTEM_DEFAULT:
                    element._TKCanvas.configure(background=element.BackgroundColor, highlightthickness=0)
                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)
                element._TKCanvas.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], expand=expand, fill=fill)
                if element.visible is False:
                    element._pack_forget_save_settings()
                    # element._TKCanvas.pack_forget()
                if element.Tooltip is not None:
                    element.TooltipObject = ToolTip(element._TKCanvas, text=element.Tooltip,
                                                    timeout=DEFAULT_TOOLTIP_TIME)
                _add_right_click_menu_and_grab(element)

                # -------------------------  Graph placement element  ------------------------- #
            elif element_type == ELEM_TYPE_GRAPH:
                element = element  # type: Graph
                width, height = element_size
                # I don't know why TWO canvases were being defined, on inside the other.  Was it so entire canvas can move?
                # if element._TKCanvas is None:
                #     element._TKCanvas = tk.Canvas(tk_row_frame, width=width, height=height, bd=border_depth)
                # else:
                #     element._TKCanvas.master = tk_row_frame
                element._TKCanvas2 = element.Widget = tk.Canvas(tk_row_frame, width=width, height=height,
                                                                bd=border_depth)
                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)
                element._TKCanvas2.pack(side=tk.LEFT, expand=expand, fill=fill)
                element._TKCanvas2.addtag_all('mytag')
                if element.BackgroundColor is not None and element.BackgroundColor != COLOR_SYSTEM_DEFAULT:
                    element._TKCanvas2.configure(background=element.BackgroundColor, highlightthickness=0)
                    # element._TKCanvas.configure(background=element.BackgroundColor, highlightthickness=0)
                element._TKCanvas2.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], expand=expand, fill=fill)
                if element.visible is False:
                    element._pack_forget_save_settings()
                    # element._TKCanvas2.pack_forget()
                if element.Tooltip is not None:
                    element.TooltipObject = ToolTip(element._TKCanvas2, text=element.Tooltip,
                                                    timeout=DEFAULT_TOOLTIP_TIME)
                if element.ChangeSubmits:
                    element._TKCanvas2.bind('<ButtonRelease-1>', element.ButtonReleaseCallBack)
                    element._TKCanvas2.bind('<ButtonPress-1>', element.ButtonPressCallBack)
                if element.DragSubmits:
                    element._TKCanvas2.bind('<Motion>', element.MotionCallBack)
                _add_right_click_menu_and_grab(element)
            # -------------------------  MENU placement element  ------------------------- #
            elif element_type == ELEM_TYPE_MENUBAR:
                element = element  # type: MenuBar
                menu_def = element.MenuDefinition
                element.TKMenu = element.Widget = tk.Menu(toplevel_form.TKroot, tearoff=element.Tearoff,
                                                          tearoffcommand=element._tearoff_menu_callback)  # create the menubar
                menubar = element.TKMenu
                if font is not None:  # if a font is used, make sure it's saved in the element
                    element.Font = font
                for menu_entry in menu_def:
                    baritem = tk.Menu(menubar, tearoff=element.Tearoff, tearoffcommand=element._tearoff_menu_callback)
                    if element.BackgroundColor not in (COLOR_SYSTEM_DEFAULT, None):
                        baritem.config(bg=element.BackgroundColor)
                        baritem.config(activeforeground=element.BackgroundColor)
                    if element.TextColor not in (COLOR_SYSTEM_DEFAULT, None):
                        baritem.config(fg=element.TextColor)
                        baritem.config(activebackground=element.TextColor)
                    if element.DisabledTextColor not in (COLOR_SYSTEM_DEFAULT, None):
                        baritem.config(disabledforeground=element.DisabledTextColor)
                    if font is not None:
                        baritem.config(font=font)
                    pos = menu_entry[0].find(MENU_SHORTCUT_CHARACTER)
                    # print(pos)
                    if pos != -1:
                        if pos == 0 or menu_entry[0][pos - len(MENU_SHORTCUT_CHARACTER)] != '\\':
                            menu_entry[0] = menu_entry[0][:pos] + menu_entry[0][pos + 1:]
                    if menu_entry[0][0] == MENU_DISABLED_CHARACTER:
                        menubar.add_cascade(label=menu_entry[0][len(MENU_DISABLED_CHARACTER):], menu=baritem,
                                            underline=pos - 1)
                        menubar.entryconfig(menu_entry[0][len(MENU_DISABLED_CHARACTER):], state='disabled')
                    else:
                        menubar.add_cascade(label=menu_entry[0], menu=baritem, underline=pos)

                    if len(menu_entry) > 1:
                        AddMenuItem(baritem, menu_entry[1], element)
                toplevel_form.TKroot.configure(menu=element.TKMenu)
            # -------------------------  Frame placement element  ------------------------- #
            elif element_type == ELEM_TYPE_FRAME:
                element = element  # type: Frame
                labeled_frame = element.Widget = tk.LabelFrame(tk_row_frame, text=element.Title, relief=element.Relief)
                element.TKFrame = labeled_frame
                PackFormIntoFrame(element, labeled_frame, toplevel_form)
                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)
                if element.VerticalAlignment is not None:
                    anchor = tk.CENTER  # Default to center if a bad choice is made
                    if element.VerticalAlignment.lower().startswith('t'):
                        anchor = tk.N
                    if element.VerticalAlignment.lower().startswith('c'):
                        anchor = tk.CENTER
                    if element.VerticalAlignment.lower().startswith('b'):
                        anchor = tk.S
                    labeled_frame.pack(side=tk.LEFT, anchor=anchor, padx=elementpad[0], pady=elementpad[1], expand=expand, fill=fill)
                else:
                    labeled_frame.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], expand=expand, fill=fill)

                if element.Size != (None, None):
                    labeled_frame.config(width=element.Size[0], height=element.Size[1])
                    labeled_frame.pack_propagate(0)
                if not element.visible:
                    element._pack_forget_save_settings()
                    # labeled_frame.pack_forget()
                if element.BackgroundColor != COLOR_SYSTEM_DEFAULT and element.BackgroundColor is not None:
                    labeled_frame.configure(background=element.BackgroundColor,
                                            highlightbackground=element.BackgroundColor,
                                            highlightcolor=element.BackgroundColor)
                if element.TextColor != COLOR_SYSTEM_DEFAULT and element.TextColor is not None:
                    labeled_frame.configure(foreground=element.TextColor)
                if font is not None:
                    labeled_frame.configure(font=font)
                if element.TitleLocation is not None:
                    labeled_frame.configure(labelanchor=element.TitleLocation)
                if element.BorderWidth is not None:
                    labeled_frame.configure(borderwidth=element.BorderWidth)
                if element.Tooltip is not None:
                    element.TooltipObject = ToolTip(labeled_frame, text=element.Tooltip, timeout=DEFAULT_TOOLTIP_TIME)
                _add_right_click_menu_and_grab(element)
                # row_should_expand=True
            # -------------------------  Tab placement element  ------------------------- #
            elif element_type == ELEM_TYPE_TAB:
                element = element  # type: Tab
                form = form   # type: TabGroup
                element.TKFrame = element.Widget = tk.Frame(form.TKNotebook)
                PackFormIntoFrame(element, element.TKFrame, toplevel_form)
                state = 'normal'
                if element.Disabled:
                    state = 'disabled'
                if element.visible is False:
                    state = 'hidden'
                # this code will add an image to the tab. Use it when adding the image on a tab enhancement
                try:
                    if element.Filename is not None:
                        photo = tk.PhotoImage(file=element.Filename)
                    elif element.Data is not None:
                        photo = tk.PhotoImage(data=element.Data)
                    else:
                        photo = None

                    if element.ImageSubsample and photo is not None:
                        photo = photo.subsample(element.ImageSubsample)
                    if element.zoom and photo is not None:
                        photo = photo.zoom(element.zoom)
                        # print('*ERROR laying out form.... Image Element has no image specified*')
                except Exception as e:
                    photo = None
                    _error_popup_with_traceback('Your Window has an Tab Element with an IMAGE problem',
                                                'The traceback will show you the Window with the problem layout',
                                                'Look in this Window\'s layout for an Image element that has a key of {}'.format(element.Key),
                                                'The error occuring is:', e)

                element.photo = photo
                if photo is not None:
                    if element_size == (None, None) or element_size is None or element_size == toplevel_form.DefaultElementSize:
                        width, height = photo.width(), photo.height()
                    else:
                        width, height = element_size
                    element.tktext_label = tk.Label(tk_row_frame, image=photo, width=width, height=height, bd=0)
                else:
                    element.tktext_label = tk.Label(tk_row_frame, bd=0)
                if photo is not None:
                    form.TKNotebook.add(element.TKFrame, text=element.Title, compound=tk.LEFT, state=state,image=photo)

                # element.photo_image = tk.PhotoImage(data=DEFAULT_BASE64_ICON)
                # form.TKNotebook.add(element.TKFrame, text=element.Title, compound=tk.LEFT, state=state,image = element.photo_image)


                form.TKNotebook.add(element.TKFrame, text=element.Title, state=state)
                # July 28 2022 removing the expansion and pack as a test
                # expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)
                # form.TKNotebook.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], fill=fill, expand=expand)

                element.ParentNotebook = form.TKNotebook
                element.TabID = form.TabCount
                form.tab_index_to_key[element.TabID] = element.key      # has a list of the tabs in the notebook and their associated key
                form.TabCount += 1
                if element.BackgroundColor not in (COLOR_SYSTEM_DEFAULT, None):
                    element.TKFrame.configure(background=element.BackgroundColor, highlightbackground=element.BackgroundColor, highlightcolor=element.BackgroundColor)

                # if element.BorderWidth is not None:
                #     element.TKFrame.configure(borderwidth=element.BorderWidth)
                if element.Tooltip is not None:
                    element.TooltipObject = ToolTip(element.TKFrame, text=element.Tooltip, timeout=DEFAULT_TOOLTIP_TIME)
                _add_right_click_menu_and_grab(element)
                # row_should_expand = True
            # -------------------------  TabGroup placement element  ------------------------- #
            elif element_type == ELEM_TYPE_TAB_GROUP:
                element = element  # type: TabGroup
                # custom_style = str(element.Key) + 'customtab.TNotebook'
                custom_style = _make_ttk_style_name('.TNotebook', element, primary_style=True)
                style = ttk.Style()
                _change_ttk_theme(style, toplevel_form.TtkTheme)

                if element.TabLocation is not None:
                    position_dict = {'left': 'w', 'right': 'e', 'top': 'n', 'bottom': 's', 'lefttop': 'wn',
                                     'leftbottom': 'ws', 'righttop': 'en', 'rightbottom': 'es', 'bottomleft': 'sw',
                                     'bottomright': 'se', 'topleft': 'nw', 'topright': 'ne'}
                    try:
                        tab_position = position_dict[element.TabLocation]
                    except:
                        tab_position = position_dict['top']
                    style.configure(custom_style, tabposition=tab_position)

                if element.BackgroundColor is not None and element.BackgroundColor != COLOR_SYSTEM_DEFAULT:
                    style.configure(custom_style, background=element.BackgroundColor)

                # FINALLY the proper styling to get tab colors!
                if element.SelectedTitleColor is not None and element.SelectedTitleColor != COLOR_SYSTEM_DEFAULT:
                    style.map(custom_style + '.Tab', foreground=[('selected', element.SelectedTitleColor)])
                if element.SelectedBackgroundColor is not None and element.SelectedBackgroundColor != COLOR_SYSTEM_DEFAULT:
                    style.map(custom_style + '.Tab', background=[('selected', element.SelectedBackgroundColor)])
                if element.TabBackgroundColor is not None and element.TabBackgroundColor != COLOR_SYSTEM_DEFAULT:
                    style.configure(custom_style + '.Tab', background=element.TabBackgroundColor)
                if element.TextColor is not None and element.TextColor != COLOR_SYSTEM_DEFAULT:
                    style.configure(custom_style + '.Tab', foreground=element.TextColor)
                if element.BorderWidth is not None:
                    style.configure(custom_style, borderwidth=element.BorderWidth)
                if element.TabBorderWidth is not None:
                    style.configure(custom_style + '.Tab', borderwidth=element.TabBorderWidth)       # if ever want to get rid of border around the TABS themselves
                if element.FocusColor not in (None, COLOR_SYSTEM_DEFAULT):
                    style.configure(custom_style + '.Tab', focuscolor=element.FocusColor)

                style.configure(custom_style + '.Tab', font=font)
                element.Style = style
                element.StyleName = custom_style
                element.TKNotebook = element.Widget = ttk.Notebook(tk_row_frame, style=custom_style)

                PackFormIntoFrame(element, toplevel_form.TKroot, toplevel_form)

                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)
                element.TKNotebook.pack(anchor=tk.SW, side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], fill=fill, expand=expand)

                if element.ChangeSubmits:
                    element.TKNotebook.bind('<<NotebookTabChanged>>', element._TabGroupSelectHandler)
                if element.Tooltip is not None:
                    element.TooltipObject = ToolTip(element.TKNotebook, text=element.Tooltip, timeout=DEFAULT_TOOLTIP_TIME)
                if element.Size != (None, None):
                    element.TKNotebook.configure(width=element.Size[0], height=element.Size[1])
                _add_right_click_menu_and_grab(element)
                if element.visible is False:
                    element._pack_forget_save_settings()
                # row_should_expand = True
                # -------------------  SLIDER placement element  ------------------------- #
            elif element_type == ELEM_TYPE_INPUT_SLIDER:
                element = element  # type: Slider
                slider_length = element_size[0] * _char_width_in_pixels(font)
                slider_width = element_size[1]
                element.TKIntVar = tk.IntVar()
                element.TKIntVar.set(element.DefaultValue)
                if element.Orientation.startswith('v'):
                    range_from = element.Range[1]
                    range_to = element.Range[0]
                    slider_length += DEFAULT_MARGINS[1] * (element_size[0] * 2)  # add in the padding
                else:
                    range_from = element.Range[0]
                    range_to = element.Range[1]
                tkscale = element.Widget = tk.Scale(tk_row_frame, orient=element.Orientation,
                                                        variable=element.TKIntVar,
                                                        from_=range_from, to_=range_to, resolution=element.Resolution,
                                                        length=slider_length, width=slider_width,
                                                        bd=element.BorderWidth,
                                                        relief=element.Relief, font=font,
                                                        tickinterval=element.TickInterval)
                tkscale.config(highlightthickness=0)
                if element.ChangeSubmits:
                    tkscale.config(command=element._SliderChangedHandler)
                if element.BackgroundColor not in (None, COLOR_SYSTEM_DEFAULT):
                    tkscale.configure(background=element.BackgroundColor)
                if element.TroughColor != COLOR_SYSTEM_DEFAULT:
                    tkscale.config(troughcolor=element.TroughColor)
                if element.DisableNumericDisplay:
                    tkscale.config(showvalue=0)
                if text_color not in (None, COLOR_SYSTEM_DEFAULT):
                    tkscale.configure(fg=text_color)
                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)
                tkscale.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], expand=expand, fill=fill)
                if element.visible is False:
                    element._pack_forget_save_settings()
                    # tkscale.pack_forget()
                element.TKScale = tkscale
                if element.Disabled == True:
                    element.TKScale['state'] = 'disabled'
                if element.Tooltip is not None:
                    element.TooltipObject = ToolTip(element.TKScale, text=element.Tooltip, timeout=DEFAULT_TOOLTIP_TIME)
                _add_right_click_menu_and_grab(element)

            # -------------------------  TABLE placement element  ------------------------- #
            elif element_type == ELEM_TYPE_TABLE:
                element = element  # type: Table
                element.element_frame = frame = tk.Frame(tk_row_frame)
                element.table_frame = frame
                height = element.NumRows
                if element.Justification.startswith('l'):
                    anchor = tk.W
                elif element.Justification.startswith('r'):
                    anchor = tk.E
                else:
                    anchor = tk.CENTER
                column_widths = {}
                # create column width list
                for row in element.Values:
                    for i, col in enumerate(row):
                        col_width = min(len(str(col)), element.MaxColumnWidth)
                        try:
                            if col_width > column_widths[i]:
                                column_widths[i] = col_width
                        except:
                            column_widths[i] = col_width

                if element.ColumnsToDisplay is None:
                    displaycolumns = element.ColumnHeadings if element.ColumnHeadings is not None else element.Values[0]
                else:
                    displaycolumns = []
                    for i, should_display in enumerate(element.ColumnsToDisplay):
                        if should_display:
                            if element.ColumnHeadings is not None:
                                displaycolumns.append(element.ColumnHeadings[i])
                            else:
                                displaycolumns.append(str(i))

                column_headings = element.ColumnHeadings if element.ColumnHeadings is not None else displaycolumns
                if element.DisplayRowNumbers:  # if display row number, tack on the numbers to front of columns
                    displaycolumns = [element.RowHeaderText, ] + displaycolumns
                    if column_headings is not None:
                        column_headings = [element.RowHeaderText, ] + element.ColumnHeadings
                    else:
                        column_headings = [element.RowHeaderText, ] + displaycolumns
                element.TKTreeview = element.Widget = ttk.Treeview(frame, columns=column_headings,
                                                                   displaycolumns=displaycolumns, show='headings',
                                                                   height=height,
                                                                   selectmode=element.SelectMode, )
                treeview = element.TKTreeview
                if element.DisplayRowNumbers:
                    treeview.heading(element.RowHeaderText, text=element.RowHeaderText)  # make a dummy heading
                    row_number_header_width =_string_width_in_pixels(element.HeaderFont, element.RowHeaderText) + 10
                    row_number_width = _string_width_in_pixels(font, str(len(element.Values))) + 10
                    row_number_width = max(row_number_header_width, row_number_width)
                    treeview.column(element.RowHeaderText, width=row_number_width, minwidth=10, anchor=anchor, stretch=0)

                headings = element.ColumnHeadings if element.ColumnHeadings is not None else element.Values[0]
                for i, heading in enumerate(headings):
                    # heading = str(heading)
                    treeview.heading(heading, text=heading)
                    if element.AutoSizeColumns:
                        col_width = column_widths.get(i, len(heading))      # in case more headings than there are columns of data
                        width = max(col_width * _char_width_in_pixels(font), len(heading)*_char_width_in_pixels(element.HeaderFont))
                    else:
                        try:
                            width = element.ColumnWidths[i] * _char_width_in_pixels(font)
                        except:
                            width = element.DefaultColumnWidth * _char_width_in_pixels(font)
                    if element.cols_justification is not None:
                        try:
                            if element.cols_justification[i].startswith('l'):
                                col_anchor = tk.W
                            elif element.cols_justification[i].startswith('r'):
                                col_anchor = tk.E
                            elif element.cols_justification[i].startswith('c'):
                                col_anchor = tk.CENTER
                            else:
                                col_anchor = anchor

                        except:             # likely didn't specify enough entries (must be one per col)
                            col_anchor = anchor
                    else:
                        col_anchor = anchor
                    treeview.column(heading, width=width, minwidth=10, anchor=col_anchor, stretch=element.expand_x)
                # Insert values into the tree
                for i, value in enumerate(element.Values):
                    if element.DisplayRowNumbers:
                        value = [i + element.StartingRowNumber] + value
                    id = treeview.insert('', 'end', text=value, iid=i + 1, values=value, tag=i)
                    element.tree_ids.append(id)
                if element.AlternatingRowColor not in (None, COLOR_SYSTEM_DEFAULT):  # alternating colors
                    for row in range(0, len(element.Values), 2):
                        treeview.tag_configure(row, background=element.AlternatingRowColor)
                if element.RowColors is not None:  # individual row colors
                    for row_def in element.RowColors:
                        if len(row_def) == 2:  # only background is specified
                            treeview.tag_configure(row_def[0], background=row_def[1])
                        else:
                            treeview.tag_configure(row_def[0], background=row_def[2], foreground=row_def[1])
                # ------ Do Styling of Colors -----
                # style_name = str(element.Key) + 'customtable.Treeview'
                style_name = _make_ttk_style_name( '.Treeview', element, primary_style=True)
                element.table_ttk_style_name = style_name
                table_style = ttk.Style()
                element.ttk_style = table_style

                _change_ttk_theme(table_style, toplevel_form.TtkTheme)

                if element.BackgroundColor is not None and element.BackgroundColor != COLOR_SYSTEM_DEFAULT:
                    table_style.configure(style_name, background=element.BackgroundColor, fieldbackground=element.BackgroundColor, )
                    if element.SelectedRowColors[1] is not None:
                        table_style.map(style_name, background=_fixed_map(table_style, style_name, 'background', element.SelectedRowColors))
                if element.TextColor is not None and element.TextColor != COLOR_SYSTEM_DEFAULT:
                    table_style.configure(style_name, foreground=element.TextColor)
                    if element.SelectedRowColors[0] is not None:
                        table_style.map(style_name, foreground=_fixed_map(table_style, style_name, 'foreground', element.SelectedRowColors))
                if element.RowHeight is not None:
                    table_style.configure(style_name, rowheight=element.RowHeight)
                else:
                    table_style.configure(style_name, rowheight=_char_height_in_pixels(font))
                if element.HeaderTextColor is not None and element.HeaderTextColor != COLOR_SYSTEM_DEFAULT:
                    table_style.configure(style_name + '.Heading', foreground=element.HeaderTextColor)
                if element.HeaderBackgroundColor is not None and element.HeaderBackgroundColor != COLOR_SYSTEM_DEFAULT:
                    table_style.configure(style_name + '.Heading', background=element.HeaderBackgroundColor)
                if element.HeaderFont is not None:
                    table_style.configure(style_name + '.Heading', font=element.HeaderFont)
                else:
                    table_style.configure(style_name + '.Heading', font=font)
                if element.HeaderBorderWidth is not None:
                    table_style.configure(style_name + '.Heading', borderwidth=element.HeaderBorderWidth)
                if element.HeaderRelief is not None:
                    table_style.configure(style_name + '.Heading', relief=element.HeaderRelief)
                table_style.configure(style_name, font=font)
                if element.BorderWidth is not None:
                    table_style.configure(style_name, borderwidth=element.BorderWidth)

                if element.HeaderBackgroundColor not in  (None, COLOR_SYSTEM_DEFAULT) and  element.HeaderTextColor not in  (None, COLOR_SYSTEM_DEFAULT):
                    table_style.map(style_name + '.Heading', background=[('pressed', '!focus', element.HeaderBackgroundColor),
                                                                         ('active', element.HeaderTextColor),])
                    table_style.map(style_name + '.Heading', foreground=[('pressed', '!focus', element.HeaderTextColor),
                                                                         ('active', element.HeaderBackgroundColor),])

                treeview.configure(style=style_name)
                # scrollable_frame.pack(side=tk.LEFT,  padx=elementpad[0], pady=elementpad[1], expand=True, fill='both')
                if element.enable_click_events is True:
                    treeview.bind('<ButtonRelease-1>', element._table_clicked)
                if element.right_click_selects:
                    if running_mac():
                        treeview.bind('<Button-2>', element._table_clicked)
                    else:
                        treeview.bind('<Button-3>', element._table_clicked)
                treeview.bind('<<TreeviewSelect>>', element._treeview_selected)
                if element.BindReturnKey:
                    treeview.bind('<Return>', element._treeview_double_click)
                    treeview.bind('<Double-Button-1>', element._treeview_double_click)




                if not element.HideVerticalScroll:
                    _make_ttk_scrollbar(element, 'v', toplevel_form)

                    element.Widget.configure(yscrollcommand=element.vsb.set)
                    element.vsb.pack(side=tk.RIGHT, fill='y')

                # Horizontal scrollbar
                if not element.VerticalScrollOnly:
                    # element.Widget.config(wrap='none')
                    _make_ttk_scrollbar(element, 'h', toplevel_form)
                    element.hsb.pack(side=tk.BOTTOM, fill='x')
                    element.Widget.configure(xscrollcommand=element.hsb.set)

                if not element.HideVerticalScroll or not element.VerticalScrollOnly:
                    # Chr0nic
                    element.Widget.bind('<Enter>', lambda event, em=element: testMouseHook(em))
                    element.Widget.bind('<Leave>', lambda event, em=element: testMouseUnhook(em))



                # if not element.HideVerticalScroll:
                #     scrollbar = tk.Scrollbar(frame)
                #     scrollbar.pack(side=tk.RIGHT, fill='y')
                #     scrollbar.config(command=treeview.yview)
                #     treeview.configure(yscrollcommand=scrollbar.set)

                # if not element.VerticalScrollOnly:
                #     hscrollbar = tk.Scrollbar(frame, orient=tk.HORIZONTAL)
                #     hscrollbar.pack(side=tk.BOTTOM, fill='x')
                #     hscrollbar.config(command=treeview.xview)
                #     treeview.configure(xscrollcommand=hscrollbar.set)






                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)
                element.TKTreeview.pack(side=tk.LEFT, padx=0, pady=0, expand=expand, fill=fill)
                frame.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], expand=expand, fill=fill)
                if element.visible is False:
                    element._pack_forget_save_settings(alternate_widget=element.element_frame)       # seems like it should be the frame if following other elements conventions
                    # element.TKTreeview.pack_forget()
                if element.Tooltip is not None:
                    element.TooltipObject = ToolTip(element.TKTreeview, text=element.Tooltip,
                                                    timeout=DEFAULT_TOOLTIP_TIME)
                _add_right_click_menu_and_grab(element)

                if tclversion_detailed == '8.6.9' and ENABLE_TREEVIEW_869_PATCH:
                    # print('*** tk version 8.6.9 detected.... patching ttk treeview code ***')
                    table_style.map(style_name,
                                    foreground=_fixed_map(table_style, style_name, 'foreground', element.SelectedRowColors),
                                    background=_fixed_map(table_style, style_name, 'background', element.SelectedRowColors))
            # -------------------------  Tree placement element  ------------------------- #
            elif element_type == ELEM_TYPE_TREE:
                element = element  # type: Tree
                element.element_frame = element_frame = tk.Frame(tk_row_frame)

                height = element.NumRows
                if element.Justification.startswith('l'):  # justification
                    anchor = tk.W
                elif element.Justification.startswith('r'):
                    anchor = tk.E
                else:
                    anchor = tk.CENTER

                if element.ColumnsToDisplay is None:  # Which cols to display
                    displaycolumns = element.ColumnHeadings
                else:
                    displaycolumns = []
                    for i, should_display in enumerate(element.ColumnsToDisplay):
                        if should_display:
                            displaycolumns.append(element.ColumnHeadings[i])
                column_headings = element.ColumnHeadings
                # ------------- GET THE TREEVIEW WIDGET -------------
                element.TKTreeview = element.Widget = ttk.Treeview(element_frame, columns=column_headings,
                                                                   displaycolumns=displaycolumns,
                                                                   show='tree headings' if column_headings is not None else 'tree',
                                                                   height=height,
                                                                   selectmode=element.SelectMode)
                treeview = element.TKTreeview
                max_widths = {}
                for key, node in element.TreeData.tree_dict.items():
                    for i, value in enumerate(node.values):
                        max_width = max_widths.get(i, 0)
                        if len(str(value)) > max_width:
                            max_widths[i] = len(str(value))

                if element.ColumnHeadings is not None:
                    for i, heading in enumerate(element.ColumnHeadings):  # Configure cols + headings
                        treeview.heading(heading, text=heading)
                        if element.AutoSizeColumns:
                            max_width = max_widths.get(i, 0)
                            max_width = max(max_width, len(heading))
                            width = min(element.MaxColumnWidth, max_width+1)
                        else:
                            try:
                                width = element.ColumnWidths[i]
                            except:
                                width = element.DefaultColumnWidth
                        treeview.column(heading, width=width * _char_width_in_pixels(font) + 10, anchor=anchor)

                def add_treeview_data(node):
                    """

                    :param node:
                    :type node:

                    """
                    if node.key != '':
                        if node.icon:
                            if node.icon not in element.image_dict:
                                if type(node.icon) is bytes:
                                    photo = tk.PhotoImage(data=node.icon)
                                else:
                                    photo = tk.PhotoImage(file=node.icon)
                                element.image_dict[node.icon] = photo
                            else:
                                photo = element.image_dict.get(node.icon)

                            node.photo = photo
                            try:
                                id = treeview.insert(element.KeyToID[node.parent], 'end', iid=None, text=node.text, values=node.values, open=element.ShowExpanded, image=node.photo)
                                element.IdToKey[id] = node.key
                                element.KeyToID[node.key] = id
                            except Exception as e:
                                print('Error inserting image into tree', e)
                        else:
                            id = treeview.insert(element.KeyToID[node.parent], 'end', iid=None, text=node.text, values=node.values, open=element.ShowExpanded)
                            element.IdToKey[id] = node.key
                            element.KeyToID[node.key] = id

                    for node in node.children:
                        add_treeview_data(node)

                add_treeview_data(element.TreeData.root_node)
                treeview.column('#0', width=element.Col0Width * _char_width_in_pixels(font), anchor=tk.W)
                treeview.heading('#0', text=element.col0_heading)

                # ----- configure colors -----
                # style_name = str(element.Key) + '.Treeview'
                style_name = _make_ttk_style_name('.Treeview', element, primary_style=True)
                tree_style = ttk.Style()
                _change_ttk_theme(tree_style, toplevel_form.TtkTheme)

                if element.BackgroundColor is not None and element.BackgroundColor != COLOR_SYSTEM_DEFAULT:
                    tree_style.configure(style_name, background=element.BackgroundColor, fieldbackground=element.BackgroundColor)
                    if element.SelectedRowColors[1] is not None:
                        tree_style.map(style_name, background=_fixed_map(tree_style, style_name, 'background', element.SelectedRowColors))
                if element.TextColor is not None and element.TextColor != COLOR_SYSTEM_DEFAULT:
                    tree_style.configure(style_name, foreground=element.TextColor)
                    if element.SelectedRowColors[0] is not None:
                        tree_style.map(style_name, foreground=_fixed_map(tree_style, style_name, 'foreground', element.SelectedRowColors))
                if element.HeaderTextColor is not None and element.HeaderTextColor != COLOR_SYSTEM_DEFAULT:
                    tree_style.configure(style_name + '.Heading', foreground=element.HeaderTextColor)
                if element.HeaderBackgroundColor is not None and element.HeaderBackgroundColor != COLOR_SYSTEM_DEFAULT:
                    tree_style.configure(style_name + '.Heading', background=element.HeaderBackgroundColor)
                if element.HeaderFont is not None:
                    tree_style.configure(style_name + '.Heading', font=element.HeaderFont)
                else:
                    tree_style.configure(style_name + '.Heading', font=font)
                if element.HeaderBorderWidth is not None:
                    tree_style.configure(style_name + '.Heading', borderwidth=element.HeaderBorderWidth)
                if element.HeaderRelief is not None:
                    tree_style.configure(style_name + '.Heading', relief=element.HeaderRelief)
                tree_style.configure(style_name, font=font)
                if element.RowHeight:
                    tree_style.configure(style_name, rowheight=element.RowHeight)
                else:
                    tree_style.configure(style_name, rowheight=_char_height_in_pixels(font))
                if element.BorderWidth is not None:
                    tree_style.configure(style_name, borderwidth=element.BorderWidth)

                treeview.configure(style=style_name)  # IMPORTANT! Be sure and set the style name for this widget



                if not element.HideVerticalScroll:
                    _make_ttk_scrollbar(element, 'v', toplevel_form)

                    element.Widget.configure(yscrollcommand=element.vsb.set)
                    element.vsb.pack(side=tk.RIGHT, fill='y')

                # Horizontal scrollbar
                if not element.VerticalScrollOnly:
                    # element.Widget.config(wrap='none')
                    _make_ttk_scrollbar(element, 'h', toplevel_form)
                    element.hsb.pack(side=tk.BOTTOM, fill='x')
                    element.Widget.configure(xscrollcommand=element.hsb.set)

                if not element.HideVerticalScroll or not element.VerticalScrollOnly:
                    # Chr0nic
                    element.Widget.bind('<Enter>', lambda event, em=element: testMouseHook(em))
                    element.Widget.bind('<Leave>', lambda event, em=element: testMouseUnhook(em))


                # Horizontal scrollbar
                # if not element.VerticalScrollOnly:
                #     element.TKText.config(wrap='none')
                #     _make_ttk_scrollbar(element, 'h')
                #     element.hsb.pack(side=tk.BOTTOM, fill='x')
                #     element.Widget.configure(xscrollcommand=element.hsb.set)

                # if not element.HideVerticalScroll or not element.VerticalScrollOnly:
                    # Chr0nic
                # element.Widget.bind("<Enter>", lambda event, em=element: testMouseHook(em))
                # element.Widget.bind("<Leave>", lambda event, em=element: testMouseUnhook(em))





                # element.scrollbar = scrollbar = tk.Scrollbar(element_frame)
                # scrollbar.pack(side=tk.RIGHT, fill='y')
                # scrollbar.config(command=treeview.yview)
                # treeview.configure(yscrollcommand=scrollbar.set)


                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)
                element.TKTreeview.pack(side=tk.LEFT, padx=0, pady=0, expand=expand, fill=fill)
                element_frame.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], expand=expand, fill=fill)
                if element.visible is False:
                    element._pack_forget_save_settings(alternate_widget=element.element_frame)       # seems like it should be the frame if following other elements conventions
                    # element.TKTreeview.pack_forget()
                treeview.bind('<<TreeviewSelect>>', element._treeview_selected)
                if element.Tooltip is not None:  # tooltip
                    element.TooltipObject = ToolTip(element.TKTreeview, text=element.Tooltip,
                                                    timeout=DEFAULT_TOOLTIP_TIME)
                _add_right_click_menu_and_grab(element)

                if tclversion_detailed == '8.6.9' and ENABLE_TREEVIEW_869_PATCH:
                    # print('*** tk version 8.6.9 detected.... patching ttk treeview code ***')
                    tree_style.map(style_name,
                                   foreground=_fixed_map(tree_style, style_name, 'foreground', element.SelectedRowColors),
                                   background=_fixed_map(tree_style, style_name, 'background', element.SelectedRowColors))

            # -------------------------  Separator placement element  ------------------------- #
            elif element_type == ELEM_TYPE_SEPARATOR:
                element = element  # type: VerticalSeparator
                # style_name = str(element.Key) + "Line.TSeparator"
                style_name = _make_ttk_style_name('.Line.TSeparator', element, primary_style=True)
                style = ttk.Style()

                _change_ttk_theme(style, toplevel_form.TtkTheme)

                if element.color not in (None, COLOR_SYSTEM_DEFAULT):
                    style.configure(style_name, background=element.color)
                separator = element.Widget = ttk.Separator(tk_row_frame, orient=element.Orientation, )

                expand, fill, row_should_expand, row_fill_direction = _add_expansion(element, row_should_expand, row_fill_direction)

                if element.Orientation.startswith('h'):
                    separator.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], fill=tk.X, expand=True)
                else:
                    separator.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], fill=tk.Y, expand=False)
                element.Widget.configure(style=style_name)  # IMPORTANT!  Apply the style
            # -------------------------  SizeGrip placement element  ------------------------- #
            elif element_type == ELEM_TYPE_SIZEGRIP:
                element = element  # type: Sizegrip
                style_name = 'Sizegrip.TSizegrip'
                style = ttk.Style()

                _change_ttk_theme(style, toplevel_form.TtkTheme)

                size_grip = element.Widget = ttk.Sizegrip(tk_row_frame)
                toplevel_form.sizegrip_widget = size_grip
                # if no size is specified, then use the background color for the window
                if element.BackgroundColor != COLOR_SYSTEM_DEFAULT:
                    style.configure(style_name, background=element.BackgroundColor)
                else:
                    style.configure(style_name, background=toplevel_form.TKroot['bg'])
                size_grip.configure(style=style_name)

                size_grip.pack(side=tk.BOTTOM, anchor='se', padx=elementpad[0], pady=elementpad[1], fill=tk.X, expand=True)
                # tricky part of sizegrip... it shouldn't cause the row to expand, but should expand and should add X axis if
                # not already filling in that direction.  Otherwise, leaves things alone!
                # row_should_expand = True
                row_fill_direction = tk.BOTH if row_fill_direction in (tk.Y, tk.BOTH) else tk.X
            # -------------------------  StatusBar placement element  ------------------------- #
            elif element_type == ELEM_TYPE_STATUSBAR:
                # auto_size_text = element.AutoSizeText
                display_text = element.DisplayText  # text to display
                if auto_size_text is False:
                    width, height = element_size
                else:
                    lines = display_text.split('\n')
                    max_line_len = max([len(l) for l in lines])
                    num_lines = len(lines)
                    if max_line_len > element_size[0]:  # if text exceeds element size, the will have to wrap
                        width = element_size[0]
                    else:
                        width = max_line_len
                    height = num_lines
                # ---===--- LABEL widget create and place --- #
                stringvar = tk.StringVar()
                element.TKStringVar = stringvar
                stringvar.set(display_text)
                if auto_size_text:
                    width = 0
                if element.Justification is not None:
                    justification = element.Justification
                elif toplevel_form.TextJustification is not None:
                    justification = toplevel_form.TextJustification
                else:
                    justification = DEFAULT_TEXT_JUSTIFICATION
                justify = tk.LEFT if justification.startswith('l') else tk.CENTER if justification.startswith('c') else tk.RIGHT
                anchor = tk.NW if justification.startswith('l') else tk.N if justification.startswith('c') else tk.NE
                # tktext_label = tk.Label(tk_row_frame, textvariable=stringvar, width=width, height=height,
                #                         justify=justify, bd=border_depth, font=font)
                tktext_label = element.Widget = tk.Label(tk_row_frame, textvariable=stringvar, width=width,
                                                         height=height,
                                                         justify=justify, bd=border_depth, font=font)
                # Set wrap-length for text (in PIXELS) == PAIN IN THE ASS
                wraplen = tktext_label.winfo_reqwidth() + 40  # width of widget in Pixels
                if not auto_size_text and height == 1:
                    wraplen = 0
                # print("wraplen, width, height", wraplen, width, height)
                tktext_label.configure(anchor=anchor, wraplen=wraplen)  # set wrap to width of widget
                if element.Relief is not None:
                    tktext_label.configure(relief=element.Relief)
                if element.BackgroundColor is not None and element.BackgroundColor != COLOR_SYSTEM_DEFAULT:
                    tktext_label.configure(background=element.BackgroundColor)
                if element.TextColor != COLOR_SYSTEM_DEFAULT and element.TextColor is not None:
                    tktext_label.configure(fg=element.TextColor)
                tktext_label.pack(side=tk.LEFT, padx=elementpad[0], pady=elementpad[1], fill=tk.X, expand=True)
                row_fill_direction = tk.X
                if element.visible is False:
                    element._pack_forget_save_settings()
                    # tktext_label.pack_forget()
                element.TKText = tktext_label
                if element.ClickSubmits:
                    tktext_label.bind('<Button-1>', element._TextClickedHandler)
                if element.Tooltip is not None:
                    element.TooltipObject = ToolTip(element.TKText, text=element.Tooltip, timeout=DEFAULT_TOOLTIP_TIME)
                _add_right_click_menu_and_grab(element)

        # ............................DONE WITH ROW pack the row of widgets ..........................#
        # done with row, pack the row of widgets
        # tk_row_frame.grid(row=row_num+2, sticky=tk.NW, padx=DEFAULT_MARGINS[0])

        anchor = 'nw'

        if row_justify.lower().startswith('c'):
            anchor = 'n'
            side = tk.LEFT
        elif row_justify.lower().startswith('r'):
            anchor = 'ne'
            side = tk.RIGHT
        elif row_justify.lower().startswith('l'):
            anchor = 'nw'
            side = tk.LEFT
        # elif toplevel_form.ElementJustification.lower().startswith('c'):
        #     anchor = 'n'
        #     side = tk.TOP
        # elif toplevel_form.ElementJustification.lower().startswith('r'):
        #     anchor = 'ne'
        #     side = tk.TOP
        # else:
        #     anchor = 'nw'
        #     side = tk.TOP

        # row_should_expand = False

        # if form.RightClickMenu:
        #     menu = form.RightClickMenu
        #     top_menu = tk.Menu(toplevel_form.TKroot, tearoff=False)
        #     AddMenuItem(top_menu, menu[1], form)
        #     tk_row_frame.bind('<Button-3>', form._RightClickMenuCallback)

        tk_row_frame.pack(side=tk.TOP, anchor=anchor, padx=0, pady=0, expand=row_should_expand, fill=row_fill_direction)
        if form.BackgroundColor is not None and form.BackgroundColor != COLOR_SYSTEM_DEFAULT:
            tk_row_frame.configure(background=form.BackgroundColor)

    return


def _get_hidden_master_root():
    """
    Creates the hidden master root window.  This window is never visible and represents the overall "application"
    """

    # if one is already made, then skip making another
    if Window.hidden_master_root is None:
        Window._IncrementOpenCount()
        Window.hidden_master_root = tk.Tk()
        Window.hidden_master_root.attributes('-alpha', 0)  # HIDE this window really really really
        # if not running_mac():
        try:
            Window.hidden_master_root.wm_overrideredirect(True)
        except Exception as e:
            if not running_mac():
                print('* Error performing wm_overrideredirect while hiding the hidden master root*', e)
        Window.hidden_master_root.withdraw()
    return Window.hidden_master_root


def _no_titlebar_setup(window):
    """
    Does the operations required to turn off the titlebar for the