import asyncio, inspect
from typing import List
from fastapi import Request

from ..protocol import Embed
from ..base import get_embed, exec_embed, compute_tokens
from . import helpers
from .protocol import Embedding, EmbeddingUsageInfo, EmbeddingsRequest, EmbeddingsResponse

def get_arguments(request: EmbeddingsRequest, sig: inspect.Signature) -> dict:
    arguments = {}
    if 'model' in sig.parameters:
        arguments['model'] = request.model
    if 'engine' in sig.parameters:
        arguments['engine'] = request.engine
    if 'user' in sig.parameters:
        arguments['user'] = request.user
    if 'encoding_format' in sig.parameters:
        arguments['encoding_format'] = request.encoding_format
    return arguments

async def create_embeddings(request: EmbeddingsRequest, raw_request: Request) -> EmbeddingsResponse:
    try:
        embed = get_embed()
        if not embed:
            return helpers.create_501_error('embed')

        embed_sig = inspect.signature(embed)
        arguments = get_arguments(request, embed_sig)

        if 'request' in embed_sig.parameters:
            arguments['request'] = raw_request

        input = request.input if isinstance(request.input, list) else [request.input]
        if 'input' in embed_sig.parameters:
            arguments['input'] = input

        embeddings: List[Embedding] = []
        for (i, data) in enumerate(await exec_embed(embed, arguments)):
            embeddings.append(Embedding(embedding=data, index=i))

        prompt_tokens = 0
        inputs = request.input if isinstance(request.input, list) else [request.input]
        for input in inputs:
            prompt_tokens += await compute_tokens(input, request.model)

        return EmbeddingsResponse(
            model=request.model or '',
            data=embeddings,
            usage=EmbeddingUsageInfo(
                prompt_tokens=prompt_tokens,
                total_tokens=prompt_tokens,
            ),
        )
    except Exception as e:
        return helpers.create_500_error(str(e))
