

from fastapi import APIRouter
from pro_craft import Intel
from pro_craft.utils import create_session

def create_router(database_url: str,
                  slave_database_url: str,
                  model_name: str):
    """
    创建一个包含 ProCraft 路由的 FastAPI APIRouter 实例。

    Args:
        database_url (str): 数据库连接字符串。
        model_name (str): 用于 Intel 实例的模型名称。
        api_key_secret (str, optional): 用于验证 API Key 的秘密字符串。
                                        如果提供，它将覆盖环境变量 PRO_CRAFT_API_KEY。
                                        如果都不提供，会使用硬编码的 'your_default_secret_key'。
    Returns:
        APIRouter: 配置好的 FastAPI APIRouter 实例。
    """

    intels = Intel(
        database_url=database_url,
        model_name=model_name
        )

    router = APIRouter(
        tags=["prompt"] # 这里使用 Depends 确保每次请求都验证
    )

    @router.get("/push_order")
    async def push_order(demand: str, prompt_id: str, action_type: str = "train"):
        result = intels.push_action_order(
            demand=demand,
            prompt_id=prompt_id,
            action_type=action_type
        )
        return {"message": "success", "result": result}

    @router.get("/get_latest_prompt")
    async def get_latest_prompt(prompt_id: str):
        with create_session(intels.engine) as session:
            result = intels.get_prompts_from_sql(
                prompt_id=prompt_id,
                session=session
            )
        return {"message": "success", "result": result}

    @router.get("/sync_database")
    async def sync_database():
        result = intels.sync_prompt_data_to_database(slave_database_url)
        return {"message": "success","result":result}
    
    return router
