"""Streaming LLM Provider - Enhanced streaming support for PydanticAI"""

# Copyright 2025-present aiiware.com
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.



from collections.abc import AsyncIterator
from typing import Optional

from pydantic_ai import Agent
from pydantic_ai.models import Model

from .llm_provider import LLMProvider


class StreamingLLMProvider:
    """Wrapper for PydanticAI with enhanced streaming support

    This provider wraps PydanticAI agents to provide robust streaming capabilities
    with automatic fallback for providers that don't support streaming.
    """

    def __init__(self, model: Model, agent: Agent):
        """Initialize streaming provider

        Args:
            model: PydanticAI model instance
            agent: PydanticAI agent instance
        """
        self.model = model
        self.agent = agent
        self.supports_streaming = self._check_streaming_support()

    def _check_streaming_support(self) -> bool:
        """Check if provider supports streaming

        Returns:
            True if provider supports streaming, False otherwise
        """
        # Get provider name from model
        model_name = str(self.model.name()) if hasattr(self.model, 'name') else str(self.model)

        # Extract provider from model name (e.g., "anthropic:claude-3-5-sonnet" -> "anthropic")
        if ":" in model_name:
            provider_name = model_name.split(":")[0].lower()
        else:
            # Try to infer from model object
            provider_name = type(self.model).__name__.lower()

        # Supported streaming providers
        streaming_providers = ["anthropic", "openai", "gemini", "google"]

        return any(provider in provider_name for provider in streaming_providers)

    async def generate_streaming(
        self,
        prompt: str,
        system_prompt: Optional[str] = None
    ) -> AsyncIterator[str]:
        """Generate response with token streaming

        Args:
            prompt: User prompt to send to LLM
            system_prompt: Optional system prompt (not used with agent)

        Yields:
            str: Tokens as they're generated by the LLM

        Raises:
            RuntimeError: If streaming fails
        """
        if not self.supports_streaming:
            # Fallback: yield complete response in one chunk
            try:
                response = await self.agent.run(prompt)
                yield response.output if hasattr(response, 'output') else str(response.data)
            except Exception as e:
                raise RuntimeError(f"Non-streaming completion failed: {str(e)}") from e
            return

        # Stream tokens from PydanticAI
        try:
            async with self.agent.run_stream(prompt) as stream:
                # Track accumulated content to yield deltas
                accumulated = ""

                async for message in stream:
                    # Extract content from streaming message
                    current_content = None

                    # Try different message formats from PydanticAI
                    if hasattr(message, 'snapshot'):
                        # Snapshot-based streaming
                        if hasattr(message.snapshot, 'all_messages') and message.snapshot.all_messages:
                            last_message = message.snapshot.all_messages[-1]
                            if hasattr(last_message, 'content'):
                                current_content = last_message.content
                    elif hasattr(message, 'content'):
                        # Direct content message
                        current_content = message.content
                    elif hasattr(message, 'delta'):
                        # Delta content (incremental updates) - ideal for streaming
                        yield message.delta
                        continue

                    # If we got full content, yield the delta
                    if current_content and current_content != accumulated:
                        delta = current_content[len(accumulated):]
                        accumulated = current_content
                        yield delta

        except Exception as e:
            # If streaming fails, try fallback to non-streaming
            try:
                response = await self.agent.run(prompt)
                yield response.output if hasattr(response, 'output') else str(response.data)
            except Exception as fallback_error:
                raise RuntimeError(
                    f"Streaming failed and fallback failed. "
                    f"Streaming error: {str(e)}, Fallback error: {str(fallback_error)}"
                ) from e

    async def generate(
        self,
        prompt: str,
        system_prompt: Optional[str] = None
    ) -> str:
        """Non-streaming generation (backward compatibility)

        Args:
            prompt: User prompt to send to LLM
            system_prompt: Optional system prompt (not used with agent)

        Returns:
            str: Complete response from LLM

        Raises:
            RuntimeError: If generation fails
        """
        try:
            response = await self.agent.run(prompt)
            return response.output if hasattr(response, 'output') else str(response.data)
        except Exception as e:
            raise RuntimeError(f"Non-streaming generation failed: {str(e)}") from e
