#!/usr/bin/env python3
"""
Simple test to demonstrate the real-time message display feature.

This example shows how messages are displayed in real-time as agents communicate.
"""

import asyncio
import os
from dotenv import load_dotenv

# Load environment variables
load_dotenv()

# Import synqed components
from synqed.agent import Agent, AgentLogicContext
from synqed.workspace_manager import WorkspaceManager, AgentRuntimeRegistry
from synqed.execution_engine import WorkspaceExecutionEngine
from synqed.planner import PlannerLLM, TaskTreeNode


# ============================================================================
# Agent Logic Functions
# ============================================================================

async def writer_logic(context: AgentLogicContext) -> dict:
    """Writer agent logic - writes content and sends to Editor."""
    messages = context.messages
    
    # Get the last message
    if not messages:
        return context.build_response("Editor", "I need a task to work on.")
    
    last_msg = messages[-1]
    
    # If receiving from USER, write a draft
    if last_msg.from_agent == "USER" or last_msg.from_agent == "SYSTEM":
        return context.build_response(
            "Editor",
            f"I'd love to write that story! Here's a draft:\n\n*The Color of Understanding*\n\nIn a world where emotions..."
        )
    
    # If receiving from Editor, finalize
    if last_msg.from_agent == "Editor":
        return context.build_response(
            "USER",
            "Thank you for the thoughtful feedback! I apologize for the abrupt cutoff—let me complete that final sentence..."
        )
    
    return context.build_response("Editor", "Draft ready for review.")


async def editor_logic(context: AgentLogicContext) -> dict:
    """Editor agent logic - reviews and provides feedback."""
    messages = context.messages
    
    if not messages:
        return context.build_response("Writer", "Waiting for content to review.")
    
    last_msg = messages[-1]
    
    # Review the draft
    if last_msg.from_agent == "Writer":
        return context.build_response(
            "Writer",
            "This is a wonderful start! The premise is engaging, and I love the vivid imagery. I think a quiet, contemplative ending would work beautifully here—it suits the story's gentle pace and thoughtful tone."
        )
    
    return context.build_response("Writer", "Content reviewed.")


# ============================================================================
# Main Execution
# ============================================================================

async def main():
    print("\n" + "="*80)
    print("  SYNQED - Real-Time Message Display Demo")
    print("="*80 + "\n")
    
    # Step 1: Create agent prototypes
    writer_agent = Agent(
        name="Writer",
        description="Creative writer agent",
        logic=writer_logic,
        default_target="Editor"
    )
    
    editor_agent = Agent(
        name="Editor",
        description="Editorial review agent",
        logic=editor_logic,
        default_target="Writer"
    )
    
    # Step 2: Register agent prototypes
    AgentRuntimeRegistry.register("Writer", writer_agent)
    AgentRuntimeRegistry.register("Editor", editor_agent)
    
    # Step 3: Create workspace manager and execution engine
    workspace_manager = WorkspaceManager()
    
    # Get API key from environment
    api_key = os.getenv("OPENAI_API_KEY")
    if not api_key:
        raise ValueError("OPENAI_API_KEY not found in environment")
    
    planner = PlannerLLM(
        provider="openai",
        api_key=api_key,
        model="gpt-4o-mini"
    )
    
    # Create execution engine with display enabled (default)
    execution_engine = WorkspaceExecutionEngine(
        planner=planner,
        workspace_manager=workspace_manager,
        enable_display=True  # This enables real-time display
    )
    
    # Step 4: Create workspace
    task_node = TaskTreeNode(
        id="collaboration-task",
        description="Writer and Editor collaborating on a story",
        required_agents=["Writer", "Editor"],
        may_need_subteams=False
    )
    
    workspace = await workspace_manager.create_workspace(
        task_tree_node=task_node,
        parent_workspace_id=None
    )
    
    # Step 5: Send initial task to Writer
    task = "write a short story about a robot learning to paint."
    
    await workspace.route_message("USER", "Writer", task, manager=workspace_manager)
    
    # Step 6: Execute the workspace with real-time display
    await execution_engine.run(workspace.workspace_id)
    
    print("\n✅ Demo completed!\n")


if __name__ == "__main__":
    asyncio.run(main())

