"""
Two Agents Collaborating - Workspace-based multi-agent communication

This demonstrates the new Workspace-based multi-agent messaging system:
1. Agents run in a Workspace (logical routing domain)
2. Agents maintain their own internal conversation memory
3. Workspace routes messages between agents using MessageRouter
4. WorkspaceExecutionEngine executes agents with event-driven scheduling
5. True agent-to-agent communication via structured actions

Architecture:
- Agents are created with logic functions
- Agents are registered with AgentRuntimeRegistry
- WorkspaceManager creates workspaces with agent instances
- WorkspaceExecutionEngine executes agents in workspaces
- Agents respond with: {"send_to": "AgentName", "content": "text"}

Setup:
1. install: pip install synqed anthropic python-dotenv
2. create .env file with: ANTHROPIC_API_KEY='your-key-here'
3. run: python two_agents_collaborating.py
"""
import asyncio
import os
import json
import logging
from pathlib import Path
from dotenv import load_dotenv

# Import the new Workspace-based API
import synqed
from synqed.workspace_manager import WorkspaceManager, AgentRuntimeRegistry
from synqed.execution_engine import WorkspaceExecutionEngine
from synqed.planner import PlannerLLM, TaskTreeNode

# Load environment variables
load_dotenv()
load_dotenv(dotenv_path=Path(__file__).parent / '.env')

# Enable debug logging to see what's happening
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s | %(name)s | %(levelname)s | %(message)s'
)


# ============================================================================
# Agent Logic Functions
# ============================================================================

async def writer_logic(context: synqed.AgentLogicContext) -> dict:
    """
    Writer agent logic.
    
    This function receives:
    - context.memory: Agent's message memory
    - context.latest_message: Latest incoming message
    - context.build_response(): Helper to build structured responses
    
    Returns:
        dict with "send_to" and "content" keys
    """
    import anthropic
    
    client = anthropic.AsyncAnthropic(api_key=os.environ["ANTHROPIC_API_KEY"])
    
    system_prompt = (
        "You are a creative writer collaborating with an editor. "
        "You receive messages one at a time and respond naturally. "
        "When you want to send a message to the Editor, respond with JSON: "
        '{"send_to": "Editor", "content": "your message here"}. '
        "Be brief and creative."
    )
    
    # Get the latest message from memory
    latest_message = context.latest_message
    if not latest_message:
        return context.build_response("Editor", "I'm ready to start writing!")
    
    # Build conversation context from agent's own memory
    # Only include messages this agent has received
    conversation_parts = []
    for msg in context.memory.get_messages():
        if msg.from_agent == "USER":
            conversation_parts.append(f"User: {msg.content}")
        elif msg.from_agent == "Editor":
            conversation_parts.append(f"Editor: {msg.content}")
        elif msg.from_agent == "SYSTEM":
            continue  # Skip system messages
        else:
            conversation_parts.append(f"{msg.from_agent}: {msg.content}")
    
    # Add instruction to respond
    conversation_parts.append("\n\nRespond with JSON: {\"send_to\": \"Editor\", \"content\": \"your response\"}")
    
    conversation_text = "\n\n".join(conversation_parts)
    
    # Call Anthropic API (async)
    resp = await client.messages.create(
        model="claude-sonnet-4-5",
        max_tokens=300,
        system=system_prompt,
        messages=[{"role": "user", "content": conversation_text}],
    )
    
    response_text = resp.content[0].text.strip()
    
    # Try to parse as JSON, if not valid JSON, wrap it
    try:
        parsed = json.loads(response_text)
        if "send_to" in parsed and "content" in parsed:
            return parsed
        else:
            # Invalid structure, wrap it
            return context.build_response("Editor", response_text)
    except json.JSONDecodeError:
        # Not JSON, wrap it
        return context.build_response("Editor", response_text)


async def editor_logic(context: synqed.AgentLogicContext) -> dict:
    """
    Editor agent logic.
    
    This function receives:
    - context.memory: Agent's message memory
    - context.latest_message: Latest incoming message
    - context.build_response(): Helper to build structured responses
    
    Returns:
        dict with "send_to" and "content" keys
    """
    import anthropic
    
    client = anthropic.AsyncAnthropic(api_key=os.environ["ANTHROPIC_API_KEY"])
    
    system_prompt = (
        "You are an editor collaborating with a writer. "
        "You receive messages one at a time and provide feedback. "
        "When you want to send a message to the Writer, respond with JSON: "
        '{"send_to": "Writer", "content": "your message here"}. '
        "Be concise and helpful."
    )
    
    # Get the latest message from memory
    latest_message = context.latest_message
    if not latest_message:
        return context.build_response("Writer", "I'm ready to edit!")
    
    # Build conversation context from agent's own memory
    # Only include messages this agent has received
    conversation_parts = []
    for msg in context.memory.get_messages():
        if msg.from_agent == "USER":
            conversation_parts.append(f"User: {msg.content}")
        elif msg.from_agent == "Writer":
            conversation_parts.append(f"Writer: {msg.content}")
        elif msg.from_agent == "SYSTEM":
            continue  # Skip system messages
        else:
            conversation_parts.append(f"{msg.from_agent}: {msg.content}")
    
    # Add instruction to respond
    conversation_parts.append("\n\nRespond with JSON: {\"send_to\": \"Writer\", \"content\": \"your response\"}")
    
    conversation_text = "\n\n".join(conversation_parts)
    
    # Call Anthropic API (async)
    resp = await client.messages.create(
        model="claude-sonnet-4-5",
        max_tokens=300,
        system=system_prompt,
        messages=[{"role": "user", "content": conversation_text}],
    )
    
    response_text = resp.content[0].text.strip()
    
    # Try to parse as JSON, if not valid JSON, wrap it
    try:
        parsed = json.loads(response_text)
        if "send_to" in parsed and "content" in parsed:
            return parsed
        else:
            # Invalid structure, wrap it
            return context.build_response("Writer", response_text)
    except json.JSONDecodeError:
        # Not JSON, wrap it
        return context.build_response("Writer", response_text)


# ============================================================================
# Main Execution
# ============================================================================

async def main():
    print("\n" + "="*80)
    print("  Multi-Agent Collaboration - Workspace-based Communication")
    print("  Workspace routing | Event-driven execution | Agent memory")
    print("="*80 + "\n")
    
    # Step 1: Create agent prototypes
    print("📝 Creating agent prototypes...\n")
    
    writer_agent = synqed.Agent(
        name="Writer",
        description="a creative writer specializing in storytelling and narrative",
        logic=writer_logic,
        default_target="Editor"
    )
    
    editor_agent = synqed.Agent(
        name="Editor",
        description="an editor who reviews and improves written content",
        logic=editor_logic,
        default_target="Writer"
    )
    
    print(f"✓ created agent: {writer_agent.name}")
    print(f"✓ created agent: {editor_agent.name}\n")
    
    # Step 2: Register agents with AgentRuntimeRegistry
    print("📋 Registering agents with AgentRuntimeRegistry...\n")
    
    AgentRuntimeRegistry.register("Writer", writer_agent)
    AgentRuntimeRegistry.register("Editor", editor_agent)
    
    print(f"✓ registered agents: {AgentRuntimeRegistry.list_roles()}\n")
    
    # Step 3: Create workspace manager and execution engine
    print("🏗️  Creating workspace manager and execution engine...\n")
    
    workspace_manager = WorkspaceManager(workspaces_root=Path("/tmp/synqed_workspaces"))
    
    # Create a simple planner (we'll create a simple task node manually)
    planner = PlannerLLM(
        provider="anthropic",
        api_key=os.environ["ANTHROPIC_API_KEY"],
        model="claude-sonnet-4-5"
    )
    
    execution_engine = WorkspaceExecutionEngine(
        planner=planner,
        workspace_manager=workspace_manager
    )
    
    print("✓ workspace manager created")
    print("✓ execution engine created\n")
    
    # Step 4: Create a workspace with Writer and Editor agents
    print("🌐 Creating workspace...\n")
    
    # Create a simple task node for the workspace
    task_node = TaskTreeNode(
        id="collaboration-task",
        description="Writer and Editor collaborating on a story",
        required_agents=["Writer", "Editor"],
        may_need_subteams=False
    )
    
    workspace = await workspace_manager.create_workspace(
        task_tree_node=task_node,
        parent_workspace_id=None
    )
    
    print(f"✓ workspace created: {workspace.workspace_id}")
    print(f"✓ agents in workspace: {list(workspace.agents.keys())}\n")
    
    # Step 5: Send initial task to Writer
    print("="*80)
    print("  Conversation structure:")
    print("  Writer receives user task, collaborates with Editor")
    print("  Agents will exchange messages until completion")
    print("="*80 + "\n")
    
    task = "write a short story about a robot learning to paint."
    
    print(f"📋 task: {task}\n")
    print("="*80)
    print("  starting workspace execution...")
    print("="*80 + "\n")
    
    # Send initial message to Writer
    print(f"\n[Initial] Sending task to Writer...")
    await workspace.route_message("USER", "Writer", task, manager=workspace_manager)
    
    # Step 6: Execute the workspace
    print("\n[Execution] Running workspace execution engine...\n")
    
    try:
        await execution_engine.run(workspace.workspace_id)
        print("\n[Execution] Execution engine completed.\n")
    except Exception as e:
        print(f"\n[Execution] Error during execution: {e}\n")
        import traceback
        traceback.print_exc()
        raise
    
    # Step 7: Display results
    print("\n" + "="*80)
    print("  conversation complete")
    print("="*80 + "\n")
    
    print("✅ task completed successfully!\n")
    
    # Get transcript from workspace router
    transcript = workspace.router.get_transcript()
    
    # Print final message transcript
    print("📜 full conversation transcript (from workspace router):")
    print("-" * 80)
    for i, entry in enumerate(transcript, 1):
        print(f"\n[{i}] {entry['from']} → {entry['to']}:")
        content = entry['content']
        # Truncate very long messages
        if len(content) > 500:
            content = content[:500] + "..."
        print(content)
    print("-" * 80 + "\n")
    
    # Clean up
    print("🧹 cleaning up...")
    
    await workspace_manager.destroy_workspace(workspace.workspace_id)
    
    print("✓ workspace destroyed\n")
    
    print("="*80)
    print("  demo complete!")
    print("  architecture:")
    print("  • agents run in Workspace (logical routing domain)")
    print("  • each agent maintains server-side message memory")
    print("  • WorkspaceExecutionEngine executes agents with event-driven scheduling")
    print("  • MessageRouter routes messages between agents")
    print("  • true agent-to-agent communication")
    print("="*80 + "\n")


if __name__ == "__main__":
    asyncio.run(main())
