"""Data models for recommendation tracking and management."""

from datetime import datetime
from enum import Enum

from pydantic import BaseModel


class RecommendationStatus(str, Enum):
    """Status values for recommendation lifecycle."""

    PENDING = "pending"  # Generated by AI, not yet reviewed
    NO_CHANGE_NEEDED = "no_change_needed"  # Current labels match recommendation
    APPROVED = "approved"  # Human-approved, ready for application
    REJECTED = "rejected"  # Human-rejected, will not be applied
    NEEDS_MODIFICATION = "needs_modification"  # Requires changes before approval
    APPLIED = "applied"  # Successfully applied to GitHub
    FAILED = "failed"  # Application to GitHub failed
    ARCHIVED = "archived"  # Moved to historical archive


class ReviewAction(str, Enum):
    """Available review actions."""

    APPROVE = "approve"
    REJECT = "reject"
    MODIFY = "modify"
    REQUEST_CHANGES = "request_changes"


class RecommendationMetadata(BaseModel):
    """Extended metadata for recommendation tracking."""

    # Core identification
    org: str
    repo: str
    issue_number: int
    processor_name: str = "product-labeling"

    # AI analysis details
    original_confidence: float
    ai_reasoning: str
    root_cause_analysis: str | None = None  # Root cause analysis from AI
    root_cause_confidence: float | None = None  # Confidence in root cause (0-1)
    recommended_labels: list[str]
    labels_to_remove: list[str]
    current_labels: list[str] = []  # Current labels from GitHub issue

    # Status tracking
    status: RecommendationStatus = RecommendationStatus.PENDING
    status_updated_at: datetime

    # Review information
    reviewed_by: str | None = None
    reviewed_at: datetime | None = None
    review_confidence: float | None = None  # Human-adjusted confidence
    review_notes: str | None = None
    modified_labels: list[str] | None = None  # Human-modified recommendations

    # Reprocessing control (Phase 1)
    # Note: Application tracking fields will be added in Phase 2

    # File references
    ai_result_file: str
    issue_file: str

    # Computed properties for filtering
    @property
    def primary_product(self) -> str | None:
        """Extract primary product from recommended labels."""
        product_labels = [
            label for label in self.recommended_labels if label.startswith("product::")
        ]
        return product_labels[0] if product_labels else None

    @property
    def confidence_tier(self) -> str:
        """Categorize confidence level."""
        confidence = self.review_confidence or self.original_confidence
        if confidence >= 0.9:
            return "high"
        elif confidence >= 0.7:
            return "medium"
        else:
            return "low"


class RecommendationFilter(BaseModel):
    """Filter criteria for recommendation queries."""

    # Basic filters
    org: str | None = None
    repo: str | None = None
    status: list[RecommendationStatus] | None = None

    # Confidence filters
    min_confidence: float | None = None
    max_confidence: float | None = None
    confidence_tier: list[str] | None = None  # high, medium, low

    # Product filters
    product: list[str] | None = None  # product::kots, product::vendor, etc.

    # Date filters
    created_after: datetime | None = None
    created_before: datetime | None = None
    reviewed_after: datetime | None = None
    reviewed_before: datetime | None = None

    # Text search
    search_text: str | None = None  # Search in reasoning, notes

    # Pagination
    limit: int | None = None
    offset: int = 0
