# Task: AI Recommendation Management and Review System - Phase 1: Core Management and Interactive Review

**Status:** complete

## Description
Implement the foundational recommendation management system with interactive review workflow. This phase focuses on discovering AI recommendations, tracking their status through a review lifecycle, and providing an interactive CLI for efficient recommendation review. Phase 2 will handle bulk operations, GitHub integration, and archival.

## New Functionality Overview (Phase 1)
When this task is complete, users will be able to:
- **Discover and track recommendations**: Automatically scan AI results and create trackable recommendation metadata
- **Interactive review workflow**: Start a review session that guides through pending recommendations one-by-one
- **Status tracking**: Persistent storage of recommendation status (pending → approved/rejected/needs-modification)
- **Review dashboard**: View recommendations filtered by status, product, confidence level, repository
- **Reprocessing control**: Skip already-reviewed recommendations during AI analysis with `--reprocess` override option

**Out of Scope for Phase 1:**
- Bulk operations (Phase 2)
- GitHub label application (Phase 2) 
- Archival system (Phase 2)
- Manual correction detection (future enhancement)

## Prerequisites
- **AI Analysis System**: `clarify-confidence-scores.md` must be completed (provides standardized confidence model)
- **Data Storage**: Existing JSON storage for issues and AI results must be functional

**Note**: Label update integration will be handled in Phase 2.

## Files to Create (Phase 1)
```
github_issue_analysis/
├── recommendation/
│   ├── __init__.py                    # Package initialization
│   ├── models.py                      # Recommendation status tracking models
│   ├── manager.py                     # Core recommendation management logic
│   ├── status_tracker.py              # Status persistence and querying
│   └── review_session.py              # Interactive review workflow
├── cli/
│   └── recommendations.py             # CLI commands: discover, list, summary, review-session
├── docs/
│   └── recommendation-workflow.md     # User guide for recommendation management
└── tests/
    ├── test_recommendation/
    │   ├── test_models.py             # Test recommendation data models
    │   ├── test_manager.py            # Test core management logic
    │   ├── test_status_tracker.py     # Test status persistence
    │   └── test_review_session.py     # Test interactive review workflow
    └── test_cli/
        └── test_recommendations.py    # Test CLI commands
```

## Files to Modify (Phase 1)
```
github_issue_analysis/
├── cli/
│   ├── main.py                        # Add recommendations command group
│   ├── batch.py                       # Add --reprocess flag to batch processing
│   └── process.py                     # Add --reprocess flag to individual processing  
└── ai/
    └── processors.py                  # Add reprocessing filter integration
```

## Implementation Details

### 1. Data Models for Recommendation Tracking

```python
# recommendation/models.py
from enum import Enum
from datetime import datetime
from typing import Optional, List, Dict, Any
from pydantic import BaseModel, Field

class RecommendationStatus(str, Enum):
    """Status values for recommendation lifecycle."""
    PENDING = "pending"           # Generated by AI, not yet reviewed
    APPROVED = "approved"         # Human-approved, ready for application
    REJECTED = "rejected"         # Human-rejected, will not be applied
    NEEDS_MODIFICATION = "needs_modification"  # Requires changes before approval
    APPLIED = "applied"           # Successfully applied to GitHub
    FAILED = "failed"             # Application to GitHub failed
    ARCHIVED = "archived"         # Moved to historical archive

class ReviewAction(str, Enum):
    """Available review actions."""
    APPROVE = "approve"
    REJECT = "reject"
    MODIFY = "modify"
    REQUEST_CHANGES = "request_changes"

class RecommendationMetadata(BaseModel):
    """Extended metadata for recommendation tracking."""
    
    # Core identification
    org: str
    repo: str
    issue_number: int
    processor_name: str = "product-labeling"
    
    # AI analysis details
    original_confidence: float
    ai_reasoning: str
    recommended_labels: List[str]
    labels_to_remove: List[str]
    
    # Status tracking
    status: RecommendationStatus = RecommendationStatus.PENDING
    status_updated_at: datetime
    
    # Review information
    reviewed_by: Optional[str] = None
    reviewed_at: Optional[datetime] = None
    review_confidence: Optional[float] = None  # Human-adjusted confidence
    review_notes: Optional[str] = None
    modified_labels: Optional[List[str]] = None  # Human-modified recommendations
    
    # Reprocessing control (Phase 1)
    # Note: Application tracking fields will be added in Phase 2
    
    # File references
    ai_result_file: str
    issue_file: str
    
    # Computed properties for filtering
    @property
    def primary_product(self) -> Optional[str]:
        """Extract primary product from recommended labels."""
        product_labels = [label for label in self.recommended_labels if label.startswith("product::")]
        return product_labels[0] if product_labels else None
    
    @property
    def confidence_tier(self) -> str:
        """Categorize confidence level."""
        confidence = self.review_confidence or self.original_confidence
        if confidence >= 0.9:
            return "high"
        elif confidence >= 0.7:
            return "medium"
        else:
            return "low"

class RecommendationFilter(BaseModel):
    """Filter criteria for recommendation queries."""
    
    # Basic filters
    org: Optional[str] = None
    repo: Optional[str] = None
    status: Optional[List[RecommendationStatus]] = None
    
    # Confidence filters
    min_confidence: Optional[float] = None
    max_confidence: Optional[float] = None
    confidence_tier: Optional[List[str]] = None  # high, medium, low
    
    # Product filters
    product: Optional[List[str]] = None  # product::kots, product::vendor, etc.
    
    # Date filters
    created_after: Optional[datetime] = None
    created_before: Optional[datetime] = None
    reviewed_after: Optional[datetime] = None
    reviewed_before: Optional[datetime] = None
    
    # Text search
    search_text: Optional[str] = None  # Search in reasoning, notes
    
    # Pagination
    limit: Optional[int] = None
    offset: Optional[int] = 0
```

### 2. Core Management Logic

```python
# recommendation/manager.py
from pathlib import Path
from typing import List, Dict, Optional
import json
from datetime import datetime

from ..ai.models import ProductLabelingResponse
from ..github_client.models import StoredIssue
from .models import RecommendationMetadata, RecommendationFilter, RecommendationStatus
from .status_tracker import StatusTracker

class RecommendationManager:
    """Central coordinator for recommendation management operations."""
    
    def __init__(self, data_dir: Path = Path("data")):
        self.data_dir = data_dir
        self.issues_dir = data_dir / "issues"
        self.results_dir = data_dir / "results"
        self.status_tracker = StatusTracker(data_dir / "recommendation_status")
    
    def discover_recommendations(self, force_refresh: bool = False) -> List[RecommendationMetadata]:
        """Scan for new AI results and create recommendation metadata."""
        recommendations = []
        
        # Scan results directory for AI analysis files
        for result_file in self.results_dir.glob("*_product-labeling.json"):
            try:
                # Parse filename to extract org, repo, issue_number
                parts = result_file.stem.split("_")
                if len(parts) >= 4:
                    org = parts[0]
                    repo = parts[1]
                    issue_number = int(parts[3])
                    
                    # Check if we already have status for this recommendation
                    existing = self.status_tracker.get_recommendation(org, repo, issue_number)
                    if existing and not force_refresh:
                        recommendations.append(existing)
                        continue
                    
                    # Load AI result
                    with open(result_file) as f:
                        result_data = json.load(f)
                    
                    ai_analysis = ProductLabelingResponse.model_validate(result_data["analysis"])
                    
                    # Find corresponding issue file
                    issue_file = self.issues_dir / f"{org}_{repo}_issue_{issue_number}.json"
                    if not issue_file.exists():
                        continue
                    
                    # Create recommendation metadata
                    recommendation = self._create_recommendation_metadata(
                        org, repo, issue_number, ai_analysis, str(result_file), str(issue_file)
                    )
                    
                    # Save status if new
                    if not existing:
                        self.status_tracker.save_recommendation(recommendation)
                    
                    recommendations.append(recommendation)
                    
            except Exception as e:
                print(f"Error processing {result_file}: {e}")
                continue
        
        return recommendations
    
    def _create_recommendation_metadata(
        self, 
        org: str, 
        repo: str, 
        issue_number: int,
        ai_analysis: ProductLabelingResponse,
        result_file: str,
        issue_file: str
    ) -> RecommendationMetadata:
        """Create recommendation metadata from AI analysis."""
        
        # Extract recommended labels
        recommended_labels = [rec.label.value for rec in ai_analysis.recommended_labels]
        
        # Extract labels to remove (incorrect current labels)
        labels_to_remove = [
            assessment.label 
            for assessment in ai_analysis.current_labels_assessment 
            if not assessment.correct
        ]
        
        return RecommendationMetadata(
            org=org,
            repo=repo,
            issue_number=issue_number,
            processor_name="product-labeling",
            original_confidence=ai_analysis.recommendation_confidence,
            ai_reasoning=ai_analysis.reasoning,
            recommended_labels=recommended_labels,
            labels_to_remove=labels_to_remove,
            status=RecommendationStatus.PENDING,
            status_updated_at=datetime.now(),
            ai_result_file=result_file,
            issue_file=issue_file
        )
    
    def get_recommendations(self, filter: RecommendationFilter) -> List[RecommendationMetadata]:
        """Get recommendations matching filter criteria."""
        return self.status_tracker.query_recommendations(filter)
    
    def should_reprocess_issue(self, org: str, repo: str, issue_number: int, force_reprocess: bool = False) -> bool:
        """Determine if an issue should be reprocessed for AI analysis."""
        
        if force_reprocess:
            return True  # --reprocess flag processes everything
        
        # Check if we have an existing recommendation
        recommendation = self.status_tracker.get_recommendation(org, repo, issue_number)
        
        if not recommendation:
            return True  # No existing recommendation, process it
        
        # Process if explicitly marked for reanalysis
        if recommendation.status == RecommendationStatus.NEEDS_MODIFICATION:
            return True
        
        # Skip if already analyzed and reviewed
        if recommendation.status in [RecommendationStatus.PENDING, RecommendationStatus.APPROVED, RecommendationStatus.REJECTED]:
            return False
        
        return True  # Default to processing

    def get_recommendation_summary(self) -> Dict[str, Any]:
        """Get summary statistics for recommendations."""
        all_recommendations = self.status_tracker.get_all_recommendations()
        
        summary = {
            "total_recommendations": len(all_recommendations),
            "by_status": {},
            "by_product": {},
            "by_confidence_tier": {},
            "by_repository": {},
            "pending_high_confidence": 0,
            "recently_applied": 0
        }
        
        # Calculate statistics
        for rec in all_recommendations:
            # Status distribution
            status = rec.status.value
            summary["by_status"][status] = summary["by_status"].get(status, 0) + 1
            
            # Product distribution
            product = rec.primary_product or "unknown"
            summary["by_product"][product] = summary["by_product"].get(product, 0) + 1
            
            # Confidence tier distribution
            tier = rec.confidence_tier
            summary["by_confidence_tier"][tier] = summary["by_confidence_tier"].get(tier, 0) + 1
            
            # Repository distribution
            repo_key = f"{rec.org}/{rec.repo}"
            summary["by_repository"][repo_key] = summary["by_repository"].get(repo_key, 0) + 1
            
            # Special counts
            if rec.status == RecommendationStatus.PENDING and rec.confidence_tier == "high":
                summary["pending_high_confidence"] += 1
            
            if (rec.status == RecommendationStatus.APPLIED and 
                rec.applied_at and 
                (datetime.now() - rec.applied_at).days <= 7):
                summary["recently_applied"] += 1
        
        return summary
```

### 3. Status Persistence and Querying

```python
# recommendation/status_tracker.py
import json
from pathlib import Path
from typing import List, Optional, Dict
from datetime import datetime

from .models import RecommendationMetadata, RecommendationFilter, RecommendationStatus

class StatusTracker:
    """Handles persistence and querying of recommendation status."""
    
    def __init__(self, status_dir: Path):
        self.status_dir = status_dir
        self.status_dir.mkdir(parents=True, exist_ok=True)
        
        # Store all recommendations in main directory (archival added in Phase 2)
        self.recommendations_dir = status_dir
        self.recommendations_dir.mkdir(exist_ok=True)
    
    def save_recommendation(self, recommendation: RecommendationMetadata) -> None:
        """Save or update recommendation status."""
        file_path = self._get_status_file_path(recommendation)
        
        with open(file_path, 'w') as f:
            json.dump(recommendation.model_dump(), f, indent=2, default=str)
    
    def get_recommendation(self, org: str, repo: str, issue_number: int) -> Optional[RecommendationMetadata]:
        """Get recommendation by identifier."""
        file_path = self.recommendations_dir / f"{org}_{repo}_issue_{issue_number}_status.json"
        
        if not file_path.exists():
            return None
        
        try:
            with open(file_path) as f:
                data = json.load(f)
            return RecommendationMetadata.model_validate(data)
        except Exception as e:
            print(f"Error loading recommendation {file_path}: {e}")
            return None
    
    def query_recommendations(self, filter: RecommendationFilter) -> List[RecommendationMetadata]:
        """Query recommendations with filtering."""
        recommendations = self.get_all_recommendations()
        filtered = []
        
        for rec in recommendations:
            if self._matches_filter(rec, filter):
                filtered.append(rec)
        
        # Apply pagination
        if filter.offset:
            filtered = filtered[filter.offset:]
        if filter.limit:
            filtered = filtered[:filter.limit]
        
        return filtered
    
    def _matches_filter(self, rec: RecommendationMetadata, filter: RecommendationFilter) -> bool:
        """Check if recommendation matches filter criteria."""
        
        # Basic filters
        if filter.org and rec.org != filter.org:
            return False
        if filter.repo and rec.repo != filter.repo:
            return False
        if filter.status and rec.status not in filter.status:
            return False
        
        # Confidence filters
        effective_confidence = rec.review_confidence or rec.original_confidence
        if filter.min_confidence and effective_confidence < filter.min_confidence:
            return False
        if filter.max_confidence and effective_confidence > filter.max_confidence:
            return False
        if filter.confidence_tier and rec.confidence_tier not in filter.confidence_tier:
            return False
        
        # Product filters
        if filter.product:
            rec_product = rec.primary_product
            if not rec_product or rec_product not in filter.product:
                return False
        
        # Date filters
        if filter.created_after and rec.status_updated_at < filter.created_after:
            return False
        if filter.created_before and rec.status_updated_at > filter.created_before:
            return False
        if filter.reviewed_after and (not rec.reviewed_at or rec.reviewed_at < filter.reviewed_after):
            return False
        if filter.reviewed_before and (not rec.reviewed_at or rec.reviewed_at > filter.reviewed_before):
            return False
        
        # Text search
        if filter.search_text:
            search_lower = filter.search_text.lower()
            searchable_text = f"{rec.ai_reasoning} {rec.review_notes or ''}".lower()
            if search_lower not in searchable_text:
                return False
        
        return True
    
    def get_all_recommendations(self) -> List[RecommendationMetadata]:
        """Get all recommendations."""
        recommendations = []
        
        for status_file in self.recommendations_dir.glob("*_status.json"):
            try:
                with open(status_file) as f:
                    data = json.load(f)
                recommendations.append(RecommendationMetadata.model_validate(data))
            except Exception as e:
                print(f"Error loading {status_file}: {e}")
        
        return recommendations
    
    def _get_status_file_path(self, recommendation: RecommendationMetadata) -> Path:
        """Get file path for recommendation status."""
        return self.recommendations_dir / f"{recommendation.org}_{recommendation.repo}_issue_{recommendation.issue_number}_status.json"
```

### 4. Interactive Review Session

```python
# recommendation/review_session.py
from typing import List, Optional, Dict, Any
from datetime import datetime
from rich.console import Console
from rich.panel import Panel
from rich.prompt import Prompt, Confirm
from rich.table import Table
from rich import print as rprint

from .models import RecommendationMetadata, RecommendationStatus, RecommendationFilter
from .manager import RecommendationManager

console = Console()

class ReviewSession:
    """Interactive recommendation review session."""
    
    def __init__(self, manager: RecommendationManager):
        self.manager = manager
        self.session_stats = {
            "reviewed": 0,
            "approved": 0,
            "rejected": 0,
            "needs_modification": 0,
            "skipped": 0
        }
    
    def start_session(self, filter_criteria: RecommendationFilter) -> Dict[str, int]:
        """Start interactive review session."""
        
        # Get recommendations to review
        recommendations = self.manager.get_recommendations(filter_criteria)
        
        if not recommendations:
            console.print("[yellow]No recommendations found matching criteria[/yellow]")
            return self.session_stats
        
        # Show session overview
        self._display_session_overview(recommendations)
        
        if not Confirm.ask(f"Start reviewing {len(recommendations)} recommendations?"):
            console.print("Review session cancelled")
            return self.session_stats
        
        # Review each recommendation
        for i, rec in enumerate(recommendations, 1):
            console.print(f"\n[bold blue]--- Reviewing {i} of {len(recommendations)} ---[/bold blue]")
            
            action = self._review_single_recommendation(rec)
            
            if action == "quit":
                console.print(f"\n[yellow]Session ended. Reviewed {self.session_stats['reviewed']} recommendations.[/yellow]")
                break
            elif action == "skip":
                self.session_stats["skipped"] += 1
            else:
                self.session_stats["reviewed"] += 1
                self.session_stats[action] += 1
        
        # Show final summary
        self._display_session_summary()
        return self.session_stats
    
    def _display_session_overview(self, recommendations: List[RecommendationMetadata]) -> None:
        """Display overview of recommendations to be reviewed."""
        
        # Count by product
        by_product = {}
        by_confidence = {"high": 0, "medium": 0, "low": 0}
        
        for rec in recommendations:
            product = rec.primary_product or "unknown"
            by_product[product] = by_product.get(product, 0) + 1
            by_confidence[rec.confidence_tier] += 1
        
        overview = f"""
[bold]Review Session Overview[/bold]

Total recommendations: {len(recommendations)}

By Product:
{chr(10).join(f"  {product}: {count}" for product, count in by_product.items())}

By Confidence:
  High (≥0.9): {by_confidence['high']}
  Medium (0.7-0.9): {by_confidence['medium']}  
  Low (<0.7): {by_confidence['low']}
"""
        console.print(Panel(overview))
    
    def _review_single_recommendation(self, rec: RecommendationMetadata) -> str:
        """Review a single recommendation interactively."""
        
        # Display recommendation details
        self._display_recommendation_details(rec)
        
        # Get user action
        console.print("\n[bold]Actions:[/bold]")
        console.print("  1. Approve")
        console.print("  2. Reject") 
        console.print("  3. Needs Modification")
        console.print("  4. Skip (review later)")
        console.print("  5. Quit session")
        
        choice = Prompt.ask("Choose action", choices=["1", "2", "3", "4", "5"], default="4")
        
        action_map = {
            "1": ("approved", RecommendationStatus.APPROVED),
            "2": ("rejected", RecommendationStatus.REJECTED),
            "3": ("needs_modification", RecommendationStatus.NEEDS_MODIFICATION),
            "4": ("skip", None),
            "5": ("quit", None)
        }
        
        action_name, new_status = action_map[choice]
        
        if action_name in ["skip", "quit"]:
            return action_name
        
        # Get review notes
        notes = Prompt.ask("Review notes (optional)", default="")
        
        # Update recommendation status
        rec.status = new_status
        rec.status_updated_at = datetime.now()
        rec.reviewed_at = datetime.now()
        rec.review_notes = notes if notes else None
        
        self.manager.status_tracker.save_recommendation(rec)
        
        console.print(f"[green]✓[/green] Recommendation {action_name}")
        return action_name
    
    def _display_recommendation_details(self, rec: RecommendationMetadata) -> None:
        """Display detailed view of recommendation."""
        
        # Create labels table
        labels_table = Table(title="Label Changes")
        labels_table.add_column("Action", style="cyan")
        labels_table.add_column("Label", style="white")
        labels_table.add_column("Current", style="dim")
        
        for label in rec.recommended_labels:
            labels_table.add_row("ADD", label, "No")
        
        for label in rec.labels_to_remove:
            labels_table.add_row("REMOVE", label, "Yes")
        
        # Main details panel
        details = f"""
[bold]Issue:[/bold] {rec.org}/{rec.repo}#{rec.issue_number}
[bold]Confidence:[/bold] {rec.original_confidence:.2f} ({rec.confidence_tier})
[bold]Product:[/bold] {rec.primary_product or 'unknown'}

[bold]AI Reasoning:[/bold]
{rec.ai_reasoning}
"""
        
        console.print(Panel(details, title="Recommendation Details"))
        console.print(labels_table)
    
    def _display_session_summary(self) -> None:
        """Display final session summary."""
        
        summary_table = Table(title="Review Session Summary")
        summary_table.add_column("Action", style="cyan")
        summary_table.add_column("Count", style="white")
        
        summary_table.add_row("Approved", str(self.session_stats["approved"]))
        summary_table.add_row("Rejected", str(self.session_stats["rejected"]))
        summary_table.add_row("Needs Modification", str(self.session_stats["needs_modification"]))
        summary_table.add_row("Skipped", str(self.session_stats["skipped"]))
        summary_table.add_row("Total Reviewed", str(self.session_stats["reviewed"]))
        
        console.print(summary_table)
```

### 5. AI Processing Integration

The existing AI processing commands need to be modified to check recommendation status before sending issues for analysis:

```python
# Modify ai/processors.py to add reprocessing check
from ..recommendation.manager import RecommendationManager

class ProductLabelingProcessor:
    def __init__(self, data_dir: Path = Path("data")):
        # ... existing initialization ...
        self.recommendation_manager = RecommendationManager(data_dir)
    
    def process_issues(self, issues: List[Dict], force_reprocess: bool = False) -> List[Dict]:
        """Process issues through AI, respecting recommendation status."""
        
        filtered_issues = []
        
        for issue_data in issues:
            org = issue_data.get("org")
            repo = issue_data.get("repo")
            issue_number = issue_data.get("issue", {}).get("number")
            
            if not all([org, repo, issue_number]):
                filtered_issues.append(issue_data)
                continue
            
            # Check if this issue should be reprocessed
            if self.recommendation_manager.should_reprocess_issue(org, repo, issue_number, force_reprocess):
                filtered_issues.append(issue_data)
            else:
                print(f"Skipping {org}/{repo}#{issue_number} - already reviewed (use --reprocess to override)")
        
        # Process the filtered issues with existing logic
        return self._process_filtered_issues(filtered_issues)
```

**CLI Command Modifications:**
- Add `--reprocess` flag to `uv run gh-analysis process product-labeling` 
- Add `--reprocess` flag to `uv run gh-analysis batch submit product-labeling`
- Both commands check recommendation status before AI submission

### 6. CLI Interface Implementation

```python
# cli/recommendations.py
import typer
from typing import Optional, List
from rich.console import Console
from rich.table import Table
from rich import print as rprint
from pathlib import Path

from ..recommendation.manager import RecommendationManager
from ..recommendation.models import RecommendationFilter, RecommendationStatus
from ..recommendation.review_session import ReviewSession

console = Console()
app = typer.Typer(help="Manage AI recommendation review workflow (Phase 1)")

@app.command()
def discover(
    force_refresh: bool = typer.Option(False, help="Force refresh of existing recommendations"),
    data_dir: str = typer.Option("data", help="Data directory path")
):
    """Discover new AI recommendations and initialize tracking."""
    
    manager = RecommendationManager(Path(data_dir))
    
    with console.status("[bold green]Scanning for recommendations..."):
        recommendations = manager.discover_recommendations(force_refresh)
    
    console.print(f"[green]✓[/green] Found {len(recommendations)} recommendations")
    
    # Show summary
    if recommendations:
        summary_cmd.callback(data_dir=data_dir)

@app.command()
def list(
    org: Optional[str] = typer.Option(None, help="Filter by organization"),
    repo: Optional[str] = typer.Option(None, help="Filter by repository"),
    status: Optional[List[str]] = typer.Option(None, help="Filter by status (pending, approved, applied, etc.)"),
    product: Optional[List[str]] = typer.Option(None, help="Filter by product (kots, vendor, troubleshoot, etc.)"),
    min_confidence: Optional[float] = typer.Option(None, help="Minimum confidence threshold"),
    confidence_tier: Optional[List[str]] = typer.Option(None, help="Filter by confidence tier (high, medium, low)"),
    limit: Optional[int] = typer.Option(20, help="Maximum number of results"),
    format: str = typer.Option("table", help="Output format: table, json, summary"),
    data_dir: str = typer.Option("data", help="Data directory path")
):
    """List recommendations with filtering options."""
    
    manager = RecommendationManager(Path(data_dir))
    
    # Parse status filter
    status_filter = None
    if status:
        try:
            status_filter = [RecommendationStatus(s) for s in status]
        except ValueError as e:
            console.print(f"[red]Invalid status value: {e}[/red]")
            return
    
    filter_criteria = RecommendationFilter(
        org=org,
        repo=repo,
        status=status_filter,
        product=product,
        min_confidence=min_confidence,
        confidence_tier=confidence_tier,
        limit=limit
    )
    
    recommendations = manager.get_recommendations(filter_criteria)
    
    if format == "json":
        import json
        rprint(json.dumps([rec.model_dump() for rec in recommendations], indent=2, default=str))
    elif format == "summary":
        _display_recommendations_summary(recommendations)
    else:
        _display_recommendations_table(recommendations)

@app.command()
def summary(
    data_dir: str = typer.Option("data", help="Data directory path")
):
    """Show recommendation dashboard with statistics."""
    
    manager = RecommendationManager(Path(data_dir))
    summary_data = manager.get_recommendation_summary()
    
    # Main statistics panel
    stats_table = Table(title="Recommendation Statistics")
    stats_table.add_column("Metric", style="cyan")
    stats_table.add_column("Count", style="white")
    
    stats_table.add_row("Total Recommendations", str(summary_data["total_recommendations"]))
    stats_table.add_row("Pending High Confidence", str(summary_data["pending_high_confidence"]))
    stats_table.add_row("Recently Applied", str(summary_data["recently_applied"]))
    
    console.print(stats_table)
    
    # Status breakdown
    if summary_data["by_status"]:
        status_table = Table(title="By Status")
        status_table.add_column("Status", style="cyan")
        status_table.add_column("Count", style="white")
        
        for status, count in summary_data["by_status"].items():
            status_table.add_row(status.title(), str(count))
        
        console.print(status_table)
    
    # Product breakdown  
    if summary_data["by_product"]:
        product_table = Table(title="By Product")
        product_table.add_column("Product", style="cyan")
        product_table.add_column("Count", style="white")
        
        for product, count in sorted(summary_data["by_product"].items(), key=lambda x: x[1], reverse=True)[:10]:
            product_table.add_row(product, str(count))
        
        console.print(product_table)

@app.command()
def review(
    org: str = typer.Option(..., help="Organization name"),
    repo: str = typer.Option(..., help="Repository name"), 
    issue_number: int = typer.Option(..., help="Issue number"),
    action: Optional[str] = typer.Option(None, help="Review action: approve, reject, modify"),
    confidence: Optional[float] = typer.Option(None, help="Override confidence score"),
    notes: Optional[str] = typer.Option(None, help="Review notes"),
    data_dir: str = typer.Option("data", help="Data directory path")
):
    """Review a specific recommendation interactively or with provided action."""
    
    manager = RecommendationManager(Path(data_dir))
    review_interface = ReviewInterface(manager)
    
    # Get recommendation
    recommendation = manager.status_tracker.get_recommendation(org, repo, issue_number)
    if not recommendation:
        console.print(f"[red]No recommendation found for {org}/{repo}#{issue_number}[/red]")
        return
    
    # Display recommendation details
    _display_recommendation_detail(recommendation)
    
    # Perform review
    if action:
        try:
            review_action = ReviewAction(action)
            result = review_interface.review_recommendation(
                recommendation, review_action, confidence, notes
            )
            if result:
                console.print(f"[green]✓[/green] Recommendation {action}ed successfully")
            else:
                console.print(f"[red]✗[/red] Failed to {action} recommendation")
        except ValueError:
            console.print(f"[red]Invalid action: {action}. Use: approve, reject, modify[/red]")
    else:
        # Interactive review
        _interactive_review(recommendation, review_interface)

@app.command()
def review_session(
    org: Optional[str] = typer.Option(None, help="Limit to specific organization"),
    repo: Optional[str] = typer.Option(None, help="Limit to specific repository"),
    status: Optional[List[str]] = typer.Option(["pending"], help="Filter by status (default: pending)"),
    min_confidence: Optional[float] = typer.Option(None, help="Minimum confidence threshold"),
    product: Optional[List[str]] = typer.Option(None, help="Filter by product (kots, vendor, etc.)"),
    data_dir: str = typer.Option("data", help="Data directory path")
):
    """Start interactive review session for recommendations."""
    
    manager = RecommendationManager(Path(data_dir))
    review_session = ReviewSession(manager)
    
    # Parse status filter
    status_filter = None
    if status:
        try:
            status_filter = [RecommendationStatus(s) for s in status]
        except ValueError as e:
            console.print(f"[red]Invalid status value: {e}[/red]")
            console.print("Valid statuses: pending, approved, rejected, needs_modification")
            return
    
    filter_criteria = RecommendationFilter(
        org=org,
        repo=repo,
        status=status_filter,
        min_confidence=min_confidence,
        product=product
    )
    
    # Start the interactive review session
    results = review_session.start_session(filter_criteria)
    
    if results["reviewed"] > 0:
        console.print(f"\n[green]✓[/green] Session complete. Reviewed {results['reviewed']} recommendations")
    else:
        console.print("\n[yellow]No recommendations were reviewed[/yellow]")

def _display_recommendations_table(recommendations: List) -> None:
    """Display recommendations in a table format."""
    if not recommendations:
        console.print("[yellow]No recommendations found[/yellow]")
        return
    
    table = Table()
    table.add_column("Issue", style="cyan")
    table.add_column("Product", style="green")
    table.add_column("Confidence", style="yellow")
    table.add_column("Status", style="white")
    table.add_column("Labels", style="blue")
    table.add_column("Reviewed", style="dim")
    
    for rec in recommendations:
        confidence_display = f"{(rec.review_confidence or rec.original_confidence):.2f}"
        product = rec.primary_product or "unknown"
        labels_display = ", ".join(rec.recommended_labels[:2])
        if len(rec.recommended_labels) > 2:
            labels_display += "..."
        
        reviewed_display = ""
        if rec.reviewed_at:
            reviewed_display = rec.reviewed_at.strftime("%m/%d")
        
        table.add_row(
            f"{rec.org}/{rec.repo}#{rec.issue_number}",
            product.replace("product::", ""),
            confidence_display,
            rec.status.value,
            labels_display,
            reviewed_display
        )
    
    console.print(table)

def _display_recommendation_detail(recommendation) -> None:
    """Display detailed view of a single recommendation."""
    
    panel_content = f"""
**Issue:** {recommendation.org}/{recommendation.repo}#{recommendation.issue_number}
**Status:** {recommendation.status.value}
**Original Confidence:** {recommendation.original_confidence:.2f}
**Products:** {', '.join(recommendation.recommended_labels)}

**AI Reasoning:**
{recommendation.ai_reasoning}

**Labels to Add:** {', '.join(recommendation.recommended_labels)}
**Labels to Remove:** {', '.join(recommendation.labels_to_remove) if recommendation.labels_to_remove else 'None'}
"""
    
    if recommendation.reviewed_at:
        panel_content += f"""
**Review Information:**
- Reviewed: {recommendation.reviewed_at.strftime('%Y-%m-%d %H:%M')}
- Review Confidence: {recommendation.review_confidence or 'Not set'}
- Notes: {recommendation.review_notes or 'None'}
"""
    
    console.print(Panel(panel_content, title="Recommendation Details"))

def _interactive_review(recommendation, review_interface) -> None:
    """Handle interactive review of a recommendation."""
    
    console.print("\n[bold]Review Actions:[/bold]")
    console.print("1. Approve")
    console.print("2. Reject") 
    console.print("3. Request Modifications")
    console.print("4. Cancel")
    
    choice = Prompt.ask("Choose action", choices=["1", "2", "3", "4"])
    
    if choice == "4":
        console.print("Review cancelled")
        return
    
    action_map = {
        "1": ReviewAction.APPROVE,
        "2": ReviewAction.REJECT,
        "3": ReviewAction.REQUEST_CHANGES
    }
    
    action = action_map[choice]
    
    # Get additional inputs
    confidence = None
    if choice in ["1", "3"]:  # Approve or modify
        confidence_input = Prompt.ask(
            f"Override confidence (current: {recommendation.original_confidence:.2f})", 
            default=""
        )
        if confidence_input:
            try:
                confidence = float(confidence_input)
            except ValueError:
                console.print("[red]Invalid confidence value[/red]")
                return
    
    notes = Prompt.ask("Review notes (optional)", default="")
    
    # Execute review
    result = review_interface.review_recommendation(recommendation, action, confidence, notes)
    
    if result:
        console.print(f"[green]✓[/green] Recommendation {action.value}d successfully")
    else:
        console.print(f"[red]✗[/red] Failed to {action.value} recommendation")
```

## Testing Strategy

### Unit Tests (Required for Phase 1)

**tests/test_recommendation/test_models.py:**
```python
class TestRecommendationModels:
    def test_recommendation_metadata_creation(self):
        """Test RecommendationMetadata model validation and creation."""
        # Create valid RecommendationMetadata with all required fields
        # Verify model validation works correctly
        # Test optional field handling
        
    def test_recommendation_filter_validation(self):
        """Test RecommendationFilter model with various filter combinations."""
        # Test valid filter combinations
        # Test invalid date ranges
        # Test confidence range validation
        
    def test_confidence_tier_property(self):
        """Test confidence_tier computed property returns correct tiers."""
        # Test confidence 0.95 → "high"
        # Test confidence 0.75 → "medium" 
        # Test confidence 0.5 → "low"
        
    def test_primary_product_extraction(self):
        """Test primary_product property extracts correct product label."""
        # Test with ["product::kots", "other"] → "product::kots"
        # Test with no product labels → None
        # Test with multiple product labels → first one
```

**tests/test_recommendation/test_status_tracker.py:**
```python
class TestStatusTracker:
    def test_save_and_retrieve_recommendation(self):
        """Test saving recommendation and retrieving by org/repo/issue_number."""
        # Create temp directory
        # Save RecommendationMetadata 
        # Retrieve by identifiers and verify content matches
        # Test file path format: {org}_{repo}_issue_{issue_number}_status.json
        
    def test_get_all_recommendations(self):
        """Test loading all recommendations from directory."""
        # Save 3 different recommendations
        # Call get_all_recommendations()
        # Verify returns list of 3 RecommendationMetadata objects
        
    def test_query_with_filters(self):
        """Test filtering recommendations by various criteria."""
        # Create recommendations with different org, repo, status, confidence
        # Test org filter returns only matching org
        # Test status filter returns only matching statuses  
        # Test confidence range filter
        # Test combined filters
        
    def test_invalid_json_handling(self):
        """Test graceful handling of corrupted status files."""
        # Create invalid JSON file in status directory
        # Verify get_all_recommendations() doesn't crash
        # Verify error is logged but other files still load
```

**tests/test_recommendation/test_manager.py:**
```python
class TestRecommendationManager:
    def test_discover_recommendations(self):
        """Test discovery of AI results and metadata creation."""
        # Create mock AI result files in data/results/
        # Create corresponding issue files in data/issues/
        # Call discover_recommendations()
        # Verify correct number of RecommendationMetadata created
        # Verify metadata fields populated correctly from AI results
        
    def test_should_reprocess_issue_logic(self):
        """Test reprocessing decision logic for different statuses."""
        # Test no existing recommendation → True
        # Test PENDING status → False (skip)
        # Test APPROVED status → False (skip) 
        # Test REJECTED status → False (skip)
        # Test NEEDS_MODIFICATION status → True (reprocess)
        # Test force_reprocess=True → True (always)
        
    def test_recommendation_summary_statistics(self):
        """Test summary statistics generation."""
        # Create recommendations with various statuses, products, confidence
        # Call get_recommendation_summary()
        # Verify counts by status are correct
        # Verify counts by product are correct
        # Verify confidence tier distribution is correct
```

**tests/test_recommendation/test_review_session.py:**
```python
class TestReviewSession:
    def test_session_overview_display(self):
        """Test session overview shows correct statistics."""
        # Create recommendations with different products and confidence
        # Mock console output
        # Call _display_session_overview()
        # Verify product counts and confidence distribution displayed
        
    def test_review_single_recommendation_approve(self):
        """Test approving a single recommendation updates status correctly."""
        # Create RecommendationMetadata with PENDING status
        # Mock user input to select "approve" and add notes
        # Call _review_single_recommendation()
        # Verify status changed to APPROVED
        # Verify reviewed_at timestamp set
        # Verify notes saved
        
    def test_session_stats_tracking(self):
        """Test session tracks statistics correctly."""
        # Start session with 5 recommendations
        # Mock approving 2, rejecting 1, skipping 2
        # Verify session_stats reflects correct counts
```

### CLI Tests (Required for Phase 1)

**tests/test_cli/test_recommendations.py:**
```python
class TestRecommendationsCLI:
    def test_discover_command(self):
        """Test recommendation discovery CLI command."""
        # Create mock AI result and issue files
        # Run: uv run gh-analysis recommendations discover
        # Verify command succeeds
        # Verify recommendation status files created
        
    def test_list_command_no_filters(self):
        """Test list command with no filters shows all recommendations."""
        # Create 3 recommendation status files
        # Run: uv run gh-analysis recommendations list
        # Verify output shows all 3 recommendations in table format
        
    def test_list_command_with_filters(self):
        """Test list command with various filter combinations."""
        # Create recommendations with different org, status, confidence
        # Test --org filter: uv run gh-analysis recommendations list --org testorg
        # Test --status filter: uv run gh-analysis recommendations list --status pending
        # Test --min-confidence filter
        # Verify each filter returns only matching recommendations
        
    def test_summary_command(self):
        """Test summary dashboard command."""
        # Create recommendations with various statuses and products
        # Run: uv run gh-analysis recommendations summary
        # Verify statistics table displayed
        # Verify counts match created recommendations
        
    def test_review_session_command_no_recommendations(self):
        """Test review session with no pending recommendations."""
        # Ensure no recommendation files exist
        # Run: uv run gh-analysis recommendations review-session
        # Verify shows "No recommendations found" message
        
    def test_review_session_command_with_filters(self):
        """Test review session command filters work correctly."""
        # Create recommendations with different orgs and confidence levels
        # Run: uv run gh-analysis recommendations review-session --org testorg --min-confidence 0.8
        # Verify only matching recommendations are included in session
```

### Integration Tests (Required for Phase 1)

**tests/test_integration/test_recommendation_workflow.py:**
```python
class TestRecommendationWorkflow:
    def test_full_discovery_to_review_workflow(self):
        """Test complete workflow from discovery through review."""
        # Create realistic AI result files with actual ProductLabelingResponse format
        # Create corresponding issue files
        # Run discover command
        # Verify recommendations created correctly
        # Run review-session and simulate approving/rejecting
        # Verify status changes persisted correctly
        
    def test_reprocessing_integration_with_ai_processor(self):
        """Test that AI processing respects recommendation status."""
        # Create recommendations with various statuses
        # Mock AI processor.process_issues() call
        # Verify PENDING/APPROVED/REJECTED issues are skipped
        # Verify NEEDS_MODIFICATION issues are processed
        # Verify --reprocess flag processes all issues
```

### Test Coverage Requirements
- **Minimum 85% line coverage** for all recommendation/ package code
- **100% coverage** for critical logic: reprocessing decisions, status updates, data persistence
- **CLI integration tests** must use actual CLI commands via subprocess, not just function calls
- **Error handling tests** for corrupted files, missing dependencies, invalid data

### Test Data Requirements
- **Realistic test fixtures** using actual AI result JSON format from existing system
- **Multiple org/repo combinations** to test filtering and organization
- **Various confidence levels** (0.1 to 1.0) to test confidence-based logic
- **All status transitions** represented in test data

## Documentation Requirements

Create comprehensive user documentation at `docs/recommendation-workflow.md` covering:

1. **Overview**: Purpose and benefits of the recommendation management system
2. **Basic Workflow**: Complete workflow from AI analysis to review to application (Phase 2)
3. **Command Reference**: All CLI commands with examples and options
4. **Interactive Review Guide**: How to use the review session effectively
5. **Status Management**: Understanding recommendation statuses and filtering
6. **Reprocessing Control**: When and how to use --reprocess flag
7. **Troubleshooting**: Common issues and solutions

## Validation Steps (Phase 1)

### Setup and Discovery
```bash
# 1. Ensure we have AI results to work with (use your actual org/repo)
uv run gh-analysis collect --org YOUR_ORG --repo YOUR_REPO --limit 5
uv run gh-analysis process product-labeling --org YOUR_ORG --repo YOUR_REPO

# 2. Test recommendation discovery
uv run gh-analysis recommendations discover
uv run gh-analysis recommendations summary

# 3. Verify status files are created
ls -la data/recommendation_status/
```

### Core Functionality
```bash
# 4. Test recommendation listing and filtering
uv run gh-analysis recommendations list
uv run gh-analysis recommendations list --status pending --min-confidence 0.8
uv run gh-analysis recommendations list --product kots --confidence-tier high

# 5. Test interactive review session (main feature)
uv run gh-analysis recommendations review-session
uv run gh-analysis recommendations review-session --org YOUR_ORG --min-confidence 0.8

# 6. Verify status changes persist
uv run gh-analysis recommendations list --status approved
uv run gh-analysis recommendations summary
```

### Reprocessing Safety
```bash
# 7. Test reprocessing safety (should skip reviewed recommendations)
uv run gh-analysis process product-labeling --org YOUR_ORG --repo YOUR_REPO
# Should skip issues with reviewed recommendations

# 8. Test reprocess override
uv run gh-analysis process product-labeling --org YOUR_ORG --repo YOUR_REPO --reprocess
# Should reprocess all issues regardless of status
```

### Quality Assurance
```bash
# 9. Run tests
uv run pytest tests/test_recommendation/ -v
uv run pytest tests/test_cli/test_recommendations.py -v

# 10. Code quality checks
uv run black . && uv run ruff check --fix --unsafe-fixes && uv run mypy . && uv run pytest
```

## Acceptance Criteria (Phase 1)

### Core Functionality
- [ ] **Recommendation Discovery**: `recommendations discover` scans AI results and creates trackable metadata
- [ ] **Status Tracking**: Persistent storage of recommendation status (pending → approved/rejected/needs-modification)
- [ ] **CLI Dashboard**: `recommendations summary` shows overview statistics by status, product, confidence
- [ ] **Filtering and Listing**: `recommendations list` with filtering by org, repo, status, product, confidence
- [ ] **Interactive Review**: `recommendations review-session` provides guided review workflow with progress tracking
- [ ] **Status Persistence**: Review decisions survive restarts and can be resumed

### Integration and Safety
- [ ] **Reprocessing Control**: AI processing skips reviewed recommendations unless `--reprocess` flag used
- [ ] **CLI Integration**: New commands integrate with existing CLI structure and patterns
- [ ] **File Organization**: Status files organized in logical directory structure

### Quality and Documentation
- [ ] **Error Handling**: Graceful handling of missing files, invalid data, parsing errors
- [ ] **Unit Test Coverage**: >85% test coverage for recommendation management logic
- [ ] **Integration Tests**: Full workflow testing from discovery through review session
- [ ] **Code Quality**: All quality checks pass (ruff, black, mypy, pytest)
- [ ] **User Documentation**: Complete workflow guide in `docs/recommendation-workflow.md`

### CLI Commands (Phase 1)
- [ ] `uv run gh-analysis recommendations discover` - Scan for new recommendations
- [ ] `uv run gh-analysis recommendations summary` - Dashboard view of recommendations
- [ ] `uv run gh-analysis recommendations list [filters]` - List and filter recommendations  
- [ ] `uv run gh-analysis recommendations review-session [filters]` - Interactive review workflow

## Success Metrics (Phase 1)

**User Workflow Achieved:**
1. User runs AI analysis on issues (existing functionality)
2. User runs `recommendations discover` to scan for new recommendations
3. User runs `recommendations summary` to see overview of pending work
4. User runs `recommendations review-session` to interactively review recommendations
5. User can resume review sessions and see progress tracking
6. Future AI analysis automatically skips already-reviewed recommendations (unless `--reprocess` used)

**System Benefits:**
- **Visibility**: Clear dashboard of all pending recommendations across products and repositories
- **Guided Review**: Interactive workflow makes reviewing many recommendations efficient
- **Control**: Manual approval workflow prevents automatic application of uncertain recommendations
- **Persistence**: Review decisions are saved and sessions can be resumed
- **Safety**: Reprocessing protection saves AI costs and processing time

## Phase 2 Preview

**Planned Phase 2 Features:**
- Bulk operations for high-confidence recommendations
- Integration with existing `update-labels` for GitHub application
- Archival system for completed recommendations
- GitHub API integration for direct label application
- Batch approve/apply workflows

This phase establishes the foundation that Phase 2 will build upon for the complete recommendation management system.

## Agent Notes
[Document implementation decisions, data model choices, CLI interface design, and integration approach with existing systems]

## Dependencies and Integration

**Builds on Existing Tasks:**
- **AI Analysis**: Uses existing `ProductLabelingResponse` model and AI result files
- **Label Updates**: Integrates with existing `update-labels` command for GitHub application
- **Storage**: Uses existing JSON storage system with additional status tracking

**No Breaking Changes:**
- All existing functionality continues to work unchanged
- Recommendation system is additive layer on top of current workflow
- Users can continue using `update-labels` directly if desired

**Future Extensibility:**
- Data models support additional recommendation types beyond product-labeling
- Status tracking system can be extended for other AI analysis workflows
- CLI interface designed for easy addition of new bulk operations