"""Functional tests for troubleshoot command that work in CI without API keys.

# type: ignore

These tests verify the command can be executed without encountering errors like
Union instantiation issues, and work without real API keys for CI compatibility.
"""

import json
import os
import tempfile
from pathlib import Path
from unittest.mock import patch

import pytest

from gh_analysis.ai.models import ResolvedAnalysis
from gh_analysis.ai.troubleshooting_agents import create_troubleshooting_agent


class TestTroubleshootFunctional:
    """Functional tests that verify the troubleshoot command works without API keys."""

    def test_create_all_agent_types_without_union_errors(self):
        """Test that all agent types can be created without Union instantiation errors.

        This test would catch the Union bug because agent creation triggers the error
        when PydanticAI processes the output_type with Union fields.
        """
        # Use fake API keys - agent creation doesn't validate them
        os.environ["OPENAI_API_KEY"] = "sk-test-key"
        os.environ["ANTHROPIC_API_KEY"] = "sk-ant-test-key"
        os.environ["SBCTL_TOKEN"] = "test-token"

        for agent_type in [
            "gpt5_mini_medium",
            "gpt5_mini_high",
            "gpt5_medium",
            "gpt5_high",
            "o3_medium",
            "o3_high",
        ]:
            try:
                agent = create_troubleshooting_agent(agent_type, "test-token", None)
                assert agent is not None
                # Just verify the agent was created successfully
                assert hasattr(agent, "output_type")

            except TypeError as e:
                # Fail if we get the Union instantiation error
                if "union" in str(e).lower() and "instantiate" in str(e).lower():
                    pytest.fail(
                        f"Union instantiation error when creating {agent_type}: {e}\n"
                        f"This indicates the patch is not working correctly."
                    )
                else:
                    # Re-raise other TypeErrors
                    raise

    def test_pydantic_ai_patch_prevents_union_error(self):
        """Test that our patch fixes the Union instantiation issue.

        This directly tests the fix by calling the patched method with tool calls,
        which is where the Union instantiation error occurs.
        """
        from pydantic_ai.messages import ToolCallPart
        from pydantic_ai.models.openai import OpenAIModel

        # The patch should already be applied on import
        from gh_analysis.ai import troubleshooting_agents  # noqa: F401

        # Create a tool call that would trigger the Union error
        tool_call = ToolCallPart(
            tool_name="test_function",
            args={"test_arg": "test_value"},
            tool_call_id="test_id_123",
        )

        # This should NOT raise "Cannot instantiate typing.Union"
        try:
            result = OpenAIModel._map_tool_call(tool_call)

            # Verify it returns the expected structure
            assert isinstance(result, dict)
            assert result["id"] == "test_id_123"
            assert result["type"] == "function"
            assert result["function"]["name"] == "test_function"

        except TypeError as e:
            if "union" in str(e).lower() and "instantiate" in str(e).lower():
                pytest.fail(
                    f"Union instantiation error still occurs: {e}\n"
                    f"The patch is not working correctly."
                )
            else:
                raise

    @patch("gh_analysis.ai.analysis.analyze_troubleshooting_issue")
    def test_troubleshoot_workflow_without_api_calls(self, mock_analyze):
        """Test the troubleshoot workflow without making actual API calls.

        This simulates the full execution path from loading an issue to getting results,
        ensuring no Union errors occur during the process.
        """
        # Mock the analysis result - return a resolved analysis
        mock_analyze.return_value = ResolvedAnalysis(
            status="resolved",
            root_cause="Test root cause",
            evidence=["Finding 1", "Finding 2"],
            solution="Test remediation",
            validation="Test explanation",
        )

        with tempfile.TemporaryDirectory() as temp_dir:
            # Create test issue data
            issue_data = {
                "org": "test-org",
                "repo": "test-repo",
                "issue": {
                    "number": 123,
                    "title": "Test issue",
                    "body": "Test body",
                    "labels": [],
                    "attachments": [],
                    "comments": [],
                },
            }

            # Save issue to file
            issue_file = Path(temp_dir) / "test-org_test-repo_issue_123.json"
            with open(issue_file, "w") as f:
                json.dump(issue_data, f)

            # Set fake credentials
            os.environ["OPENAI_API_KEY"] = "sk-test"
            os.environ["SBCTL_TOKEN"] = "test-token"

            # Import and run the troubleshoot workflow

            # This should complete without Union errors
            try:
                # Create agent directly to test the workflow
                agent = create_troubleshooting_agent("o3_medium", "test-token", None)
                assert agent is not None

                # The workflow would call analyze_troubleshooting_issue
                # which is mocked to return our test response
                assert mock_analyze.return_value is not None

            except TypeError as e:
                if "union" in str(e).lower() and "instantiate" in str(e).lower():
                    pytest.fail(f"Union error in workflow: {e}")

    def test_models_can_be_instantiated_and_serialized(self):
        """Test that all models work correctly with Union fields.

        This ensures the models themselves don't have Union issues during
        instantiation or serialization (which PydanticAI does internally).
        """
        from gh_analysis.ai.models import (
            ProductLabel,
            ProductLabelingResponse,
            RecommendedLabel,
        )

        # Test ProductLabelingResponse with its Union field (root_cause_confidence)
        response = ProductLabelingResponse(
            root_cause_analysis="Test analysis",
            root_cause_confidence=0.75,  # Union field: float | None
            recommendation_confidence=0.85,
            recommended_labels=[
                RecommendedLabel(
                    label=ProductLabel.TROUBLESHOOT, reasoning="Test reasoning"
                )
            ],
            current_labels_assessment=[],
            summary="Test summary",
            reasoning="Test reasoning",
        )

        # Test serialization (what PydanticAI does internally)
        serialized = response.model_dump()
        assert serialized["root_cause_confidence"] == 0.75

        # Test with None value
        response_none = ProductLabelingResponse(
            root_cause_analysis="Test",
            root_cause_confidence=None,  # Test None case
            recommendation_confidence=0.85,
            recommended_labels=[],
            current_labels_assessment=[],
            summary="Test",
            reasoning="Test",
        )

        serialized_none = response_none.model_dump()
        assert serialized_none["root_cause_confidence"] is None

        # Test ResolvedAnalysis (part of TechnicalAnalysis union)
        tr = ResolvedAnalysis(
            status="resolved",
            root_cause="Test",
            evidence=["Finding"],
            solution="Fix",
            validation="Explain",
        )

        tr_serialized = tr.model_dump()
        assert tr_serialized["status"] == "resolved"
