"""Shell completion script generator.

This module generates shell-specific completion scripts (bash, zsh, fish)
based on the function registry.
"""

# Copyright 2025-present aiiware.com
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.



from dataclasses import dataclass, field
from typing import Dict, List, Any
from aii.core.registry.function_registry import FunctionRegistry


@dataclass
class CompletionSpec:
    """Specification for shell completions.

    Attributes:
        commands: List of all available command names
        flags: Mapping of command names to their available flags
        flag_values: Mapping of flag names to suggested values
    """
    commands: List[str] = field(default_factory=list)
    flags: Dict[str, List[str]] = field(default_factory=dict)
    flag_values: Dict[str, List[str]] = field(default_factory=dict)


class CompletionGenerator:
    """Generate shell completion scripts from function registry.

    This class builds a completion specification by inspecting the function
    registry and generates shell-specific completion scripts for bash, zsh,
    and fish.
    """

    # Common flag values used across multiple functions
    COMMON_LANGUAGES = ["spanish", "french", "german", "chinese", "japanese", "korean",
                        "italian", "portuguese", "russian", "arabic", "hindi"]
    COMMON_FORMATS = ["markdown", "plain", "json", "html"]
    COMMON_STYLES = ["professional", "casual", "technical", "creative"]

    # Global flags available for all commands
    GLOBAL_FLAGS = ["--help", "--version", "--debug", "--verbose", "--no-streaming", "--no-cache"]

    def __init__(self, registry: FunctionRegistry):
        """Initialize completion generator.

        Args:
            registry: Function registry to build completion spec from
        """
        self.registry = registry
        self.spec = self._build_spec()

    def _build_spec(self) -> CompletionSpec:
        """Build completion specification from function registry.

        Returns:
            CompletionSpec with commands, flags, and flag values
        """
        commands = []
        flags: Dict[str, List[str]] = {}
        flag_values: Dict[str, List[str]] = {
            "--to": self.COMMON_LANGUAGES,
            "--from": self.COMMON_LANGUAGES,
            "--format": self.COMMON_FORMATS,
            "--style": self.COMMON_STYLES,
        }

        # Discover all functions from registry
        for func_name, func_def in self.registry.functions.items():
            commands.append(func_name)

            # Extract flags from parameter schema (ParameterSchema dict)
            func_flags = []

            for param_name, param_schema in func_def.parameters.items():
                flag_name = f"--{param_name.replace('_', '-')}"
                func_flags.append(flag_name)

                # Add enum values to flag_values if present
                # ParameterSchema has 'choices' field
                if param_schema.choices:
                    flag_values[flag_name] = param_schema.choices

            # Add global flags
            func_flags.extend(self.GLOBAL_FLAGS)
            flags[func_name] = func_flags

        # Add meta commands (config, cache, etc.)
        meta_commands = ["config", "cache", "install-completion", "uninstall-completion"]
        commands.extend(meta_commands)

        # Add flags for meta commands
        flags["config"] = ["init", "model", "provider", "web-search", "--help"]
        flags["cache"] = ["clear", "stats", "--help"]
        flags["install-completion"] = ["--shell", "--help"]
        flags["uninstall-completion"] = ["--shell", "--help"]

        return CompletionSpec(
            commands=sorted(commands),
            flags=flags,
            flag_values=flag_values
        )

    def generate_bash(self) -> str:
        """Generate bash completion script.

        Returns:
            Bash completion script as string
        """
        commands = " ".join(self.spec.commands)

        # Build case statements for command-specific flags
        case_statements = []
        for cmd, cmd_flags in self.spec.flags.items():
            flags_str = " ".join(cmd_flags)
            case_statements.append(f'        {cmd})\n            flags="{flags_str}"\n            ;;')

        cases = "\n".join(case_statements)

        return f'''# Bash completion for aii
# Generated by aii completion generator

_aii_completion() {{
    local cur prev commands flags
    cur="${{COMP_WORDS[COMP_CWORD]}}"
    prev="${{COMP_WORDS[COMP_CWORD-1]}}"
    commands="{commands}"

    # Complete commands (first argument)
    if [[ $COMP_CWORD -eq 1 ]]; then
        COMPREPLY=($(compgen -W "$commands" -- "$cur"))
        return 0
    fi

    # Complete flags based on command
    local cmd="${{COMP_WORDS[1]}}"
    case "$cmd" in
{cases}
        *)
            flags="--help"
            ;;
    esac

    COMPREPLY=($(compgen -W "$flags" -- "$cur"))
    return 0
}}

complete -F _aii_completion aii
'''

    def generate_zsh(self) -> str:
        """Generate zsh completion script with descriptions.

        Returns:
            Zsh completion script as string
        """
        # Build command list with descriptions
        commands_with_desc = []
        for cmd in self.spec.commands:
            func_def = self.registry.functions.get(cmd)
            if func_def:
                desc = func_def.description.replace("'", "\\'").replace('"', '\\"')
                commands_with_desc.append(f'        "{cmd}:{desc}"')
            else:
                # Meta commands
                meta_descriptions = {
                    "config": "Configure aii settings",
                    "cache": "Manage result cache",
                    "install-completion": "Install shell tab completion",
                    "uninstall-completion": "Uninstall shell tab completion"
                }
                desc = meta_descriptions.get(cmd, cmd)
                commands_with_desc.append(f'        "{cmd}:{desc}"')

        # Build argument completions for each command
        arg_completions = []
        for cmd, cmd_flags in self.spec.flags.items():
            arg_lines = []
            for flag in cmd_flags:
                if flag in self.spec.flag_values:
                    values = ":".join(self.spec.flag_values[flag])
                    # Extract flag name without --
                    flag_name = flag[2:]
                    arg_lines.append(f"                        '{flag}[{flag_name}]:{flag_name}:({values})'")
                else:
                    flag_name = flag[2:] if flag.startswith("--") else flag
                    arg_lines.append(f"                        '{flag}[{flag_name}]'")

            args_str = " \\\\\n".join(arg_lines)
            arg_completions.append(f'''                {cmd})
                    _arguments \\
{args_str}
                    ;;''')

        args_cases = "\n".join(arg_completions)
        commands_list = "\n".join(commands_with_desc)

        return f'''#compdef aii
# Zsh completion for aii
# Generated by aii completion generator

_aii() {{
    local -a commands
    commands=(
{commands_list}
    )

    _arguments -C \\
        '1: :->command' \\
        '*::arg:->args'

    case $state in
        command)
            _describe -t commands 'aii commands' commands
            ;;
        args)
            case $line[1] in
{args_cases}
                *)
                    _arguments '--help[Show help]'
                    ;;
            esac
            ;;
    esac
}}

_aii "$@"
'''

    def generate_fish(self) -> str:
        """Generate fish completion script with inline suggestions.

        Returns:
            Fish completion script as string
        """
        completions = []

        # Add command completions
        for cmd in self.spec.commands:
            func_def = self.registry.functions.get(cmd)
            if func_def:
                desc = func_def.description.replace('"', '\\"')
                completions.append(f'complete -c aii -n "__fish_use_subcommand" -a "{cmd}" -d "{desc}"')
            else:
                # Meta commands
                meta_descriptions = {
                    "config": "Configure aii settings",
                    "cache": "Manage result cache",
                    "install-completion": "Install shell tab completion",
                    "uninstall-completion": "Uninstall shell tab completion"
                }
                desc = meta_descriptions.get(cmd, cmd)
                completions.append(f'complete -c aii -n "__fish_use_subcommand" -a "{cmd}" -d "{desc}"')

        # Add flag completions for each command
        for cmd, cmd_flags in self.spec.flags.items():
            for flag in cmd_flags:
                if flag.startswith("--"):
                    flag_name = flag[2:]
                    # Check if we have value suggestions
                    if flag in self.spec.flag_values:
                        values = self.spec.flag_values[flag]
                        for value in values:
                            completions.append(
                                f'complete -c aii -n "__fish_seen_subcommand_from {cmd}" '
                                f'-l {flag_name} -a "{value}"'
                            )
                    else:
                        completions.append(
                            f'complete -c aii -n "__fish_seen_subcommand_from {cmd}" '
                            f'-l {flag_name} -d "{flag_name}"'
                        )

        return "# Fish completion for aii\n# Generated by aii completion generator\n\n" + "\n".join(completions) + "\n"
