"""
MCP Content Formatter - Add AII signatures to MCP-generated content (v0.4.10)

This module provides consistent branding for content created through MCP servers,
adding configurable AII signatures to GitHub issues, PRs, comments, etc.
"""

# Copyright 2025-present aiiware.com
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.



from enum import Enum
from typing import Optional, Set


class SignatureStyle(Enum):
    """Signature style options"""
    FULL = "full"      # Full signature with emoji and link
    MINIMAL = "minimal"  # Simple text signature
    NONE = "none"      # No signature


class MCPContentFormatter:
    """
    Formatter for adding AII signatures to MCP-generated content (v0.4.10).

    Signatures help users track what content was generated by AII,
    providing transparency and branding.

    Usage:
        formatter = MCPContentFormatter(enabled=True, style=SignatureStyle.FULL)
        content_with_sig = formatter.add_signature(original_content, "github_create_issue")
    """

    # Functions that should add signatures by default
    SIGNED_FUNCTIONS: Set[str] = {
        # GitHub functions
        "github_create_issue",
        "github_create_pull_request",
        "github_create_comment",
        "github_create_or_update_file",

        # Future: Other MCP content generation
        # "slack_post_message",
        # "notion_create_page",
    }

    # Signature templates
    SIGNATURES = {
        SignatureStyle.FULL: "\n\n---\n🤖 Generated with [aii](https://pypi.org/project/aiiware-cli)",
        SignatureStyle.MINIMAL: "\n\n*Generated with aii*",
        SignatureStyle.NONE: "",
    }

    def __init__(
        self,
        enabled: bool = True,
        style: SignatureStyle = SignatureStyle.FULL,
        function_overrides: Optional[dict[str, SignatureStyle]] = None
    ):
        """
        Initialize the content formatter.

        Args:
            enabled: Whether signatures are enabled globally
            style: Default signature style to use
            function_overrides: Per-function style overrides
                Example: {"github_create_issue": SignatureStyle.MINIMAL}
        """
        self.enabled = enabled
        self.default_style = style
        self.function_overrides = function_overrides or {}

    def should_add_signature(self, function_name: str) -> bool:
        """
        Determine if a signature should be added for this function.

        Args:
            function_name: Name of the function being called

        Returns:
            True if signature should be added
        """
        # Check if signatures are globally disabled
        if not self.enabled:
            return False

        # Check if function has override to "none"
        if function_name in self.function_overrides:
            return self.function_overrides[function_name] != SignatureStyle.NONE

        # Check if function is in the signed functions list
        return function_name in self.SIGNED_FUNCTIONS

    def get_signature_style(self, function_name: str) -> SignatureStyle:
        """
        Get the signature style for a specific function.

        Args:
            function_name: Name of the function being called

        Returns:
            SignatureStyle to use for this function
        """
        # Check for function-specific override
        if function_name in self.function_overrides:
            return self.function_overrides[function_name]

        # Use default style
        return self.default_style

    def add_signature(
        self,
        content: str,
        function_name: str,
        force: bool = False
    ) -> str:
        """
        Add AII signature to content if appropriate.

        Args:
            content: Original content text
            function_name: Name of the function generating the content
            force: Force adding signature even if not in SIGNED_FUNCTIONS

        Returns:
            Content with signature appended (if applicable)
        """
        # Check if we should add signature
        if not force and not self.should_add_signature(function_name):
            return content

        # Get signature style
        style = self.get_signature_style(function_name)

        # Get signature text
        signature = self.SIGNATURES.get(style, "")

        # Avoid duplicate signatures
        if signature and signature.strip() in content:
            return content

        # Add signature
        return content + signature

    def remove_signature(self, content: str) -> str:
        """
        Remove AII signature from content (if present).

        Useful for editing existing content without duplicating signatures.

        Args:
            content: Content that may contain a signature

        Returns:
            Content with signature removed
        """
        for signature in self.SIGNATURES.values():
            if signature and signature.strip() in content:
                content = content.replace(signature, "")

        return content.rstrip()


def create_formatter_from_config(config_manager) -> MCPContentFormatter:
    """
    Create MCPContentFormatter from configuration settings (v0.4.10).

    Reads from config.yaml:
        mcp:
          signature:
            enabled: true
            style: "full"  # full, minimal, or none
            overrides:
              github_create_issue: "minimal"

    Args:
        config_manager: ConfigManager instance

    Returns:
        Configured MCPContentFormatter instance
    """
    # Get signature config (with defaults)
    config = config_manager.get_all_config()
    enabled = config.get("mcp", {}).get("signature", {}).get("enabled", True)
    style_str = config.get("mcp", {}).get("signature", {}).get("style", "full")
    overrides_config = config.get("mcp", {}).get("signature", {}).get("overrides", {})

    # Parse style
    try:
        style = SignatureStyle(style_str)
    except ValueError:
        # Invalid style, use default
        style = SignatureStyle.FULL

    # Parse overrides
    function_overrides = {}
    for func_name, override_style_str in overrides_config.items():
        try:
            function_overrides[func_name] = SignatureStyle(override_style_str)
        except ValueError:
            # Invalid override style, skip
            continue

    return MCPContentFormatter(
        enabled=enabled,
        style=style,
        function_overrides=function_overrides
    )
