"""Integration tests for ct-lock-helper shell script.

These tests verify that ct-lock-helper implements the same locking algorithms
as locking.py, covering all three strategies (lockdir, cifs, flock).
"""

import pytest
import subprocess
import tempfile
import time
import os
import socket
import shutil


@pytest.fixture
def temp_target():
    """Create temporary target file for lock testing."""
    with tempfile.NamedTemporaryFile(delete=False, suffix=".o") as f:
        temp_path = f.name
    yield temp_path
    # Cleanup all lock artifacts
    for ext in ["", ".lockdir", ".lock", ".lock.excl", ".lock.pid"]:
        try:
            path = temp_path + ext
            if os.path.isdir(path):
                shutil.rmtree(path)
            elif os.path.exists(path):
                os.unlink(path)
        except OSError:
            pass
    # Clean up temp files created by ct-lock-helper
    parent_dir = os.path.dirname(temp_path)
    basename = os.path.basename(temp_path)
    for f in os.listdir(parent_dir):
        if f.startswith(basename) and ".tmp" in f:
            try:
                os.unlink(os.path.join(parent_dir, f))
            except OSError:
                pass


class TestLockHelperBasic:
    """Basic functionality tests for all lock strategies."""

    @pytest.mark.parametrize("strategy", ["lockdir", "cifs", "flock"])
    def test_successful_compile(self, temp_target, strategy):
        """Test that ct-lock-helper successfully compiles a simple program."""
        # Create a simple C source file
        source = temp_target.replace(".o", ".c")
        with open(source, "w") as f:
            f.write("int main() { return 0; }\n")

        try:
            # Run ct-lock-helper
            result = subprocess.run(
                [
                    "ct-lock-helper", "compile",
                    f"--target={temp_target}",
                    f"--strategy={strategy}",
                    "--",
                    "gcc", "-c", source
                ],
                capture_output=True,
                text=True,
                timeout=5
            )

            # Verify success
            assert result.returncode == 0, f"Compilation failed: {result.stderr}"
            assert os.path.exists(temp_target), "Target file not created"

            # Verify lock was cleaned up
            if strategy == "lockdir":
                assert not os.path.exists(temp_target + ".lockdir"), "Lock not cleaned up"
            else:
                assert not os.path.exists(temp_target + ".lock"), "Lock not cleaned up"

        finally:
            if os.path.exists(source):
                os.unlink(source)

    @pytest.mark.parametrize("strategy", ["lockdir", "cifs", "flock"])
    def test_compile_error_propagates(self, temp_target, strategy):
        """Test that compiler errors cause non-zero exit."""
        # Create source with syntax error
        source = temp_target.replace(".o", ".c")
        with open(source, "w") as f:
            f.write("int main() { this_is_a_syntax_error; }\n")

        try:
            result = subprocess.run(
                [
                    "ct-lock-helper", "compile",
                    f"--target={temp_target}",
                    f"--strategy={strategy}",
                    "--",
                    "gcc", "-c", source
                ],
                capture_output=True,
                text=True,
                timeout=5
            )

            # Verify failure
            assert result.returncode != 0, "Should fail with syntax error"

            # Note: temp file may exist if gcc creates it before failing
            # The important check is that final mv didn't happen (checked by returncode)

            # Verify lock was cleaned up even on error
            if strategy == "lockdir":
                assert not os.path.exists(temp_target + ".lockdir"), "Lock not cleaned up on error"
            else:
                assert not os.path.exists(temp_target + ".lock"), "Lock not cleaned up on error"

        finally:
            if os.path.exists(source):
                os.unlink(source)


class TestLockdirStrategy:
    """Tests specific to lockdir strategy (NFS/GPFS/Lustre)."""

    def test_creates_lockdir_with_pid_file(self, temp_target):
        """Test that lockdir and pid file are created during compilation."""
        source = temp_target.replace(".o", ".c")
        with open(source, "w") as f:
            f.write("int main() { return 0; }\n")

        lockdir = temp_target + ".lockdir"
        pid_file = os.path.join(lockdir, "pid")

        #  Create wrapper script that sleeps then compiles
        wrapper = temp_target.replace(".o", "_compile.sh")
        with open(wrapper, "w") as f:
            f.write(f"#!/bin/bash\nsleep 0.5\nexec gcc -c {source} \"$@\"\n")
        os.chmod(wrapper, 0o755)

        try:
            # Run in background so we can inspect lock state
            proc = subprocess.Popen(
                [
                    "ct-lock-helper", "compile",
                    f"--target={temp_target}",
                    "--strategy=lockdir",
                    "--",
                    wrapper
                ],
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE
            )

            # Wait for lock to be acquired
            time.sleep(0.1)

            # Verify lockdir exists
            assert os.path.exists(lockdir), "Lockdir not created"
            assert os.path.exists(pid_file), "PID file not created"

            # Verify pid file format: hostname:pid
            with open(pid_file, "r") as f:
                content = f.read().strip()
                assert ":" in content, "PID file format incorrect"
                hostname, pid = content.split(":", 1)
                assert len(hostname) > 0, "Hostname empty"
                # PID will be ct-lock-helper's PID (child of proc)
                assert int(pid) > 0, "PID invalid"

            # Wait for completion
            proc.wait(timeout=2)
            assert proc.returncode == 0

            # Verify cleanup
            assert not os.path.exists(lockdir), "Lockdir not cleaned up"

        finally:
            if os.path.exists(source):
                os.unlink(source)
            if os.path.exists(wrapper):
                os.unlink(wrapper)
            if proc.poll() is None:
                proc.kill()

    def test_stale_lock_removal(self, temp_target):
        """Test that stale locks are automatically removed."""
        lockdir = temp_target + ".lockdir"
        pid_file = os.path.join(lockdir, "pid")

        # Create fake stale lock
        os.makedirs(lockdir, exist_ok=True)
        hostname = socket.gethostname()
        fake_pid = 999999  # Unlikely to exist
        with open(pid_file, "w") as f:
            f.write(f"{hostname}:{fake_pid}\n")

        # Create a simple source file
        source = temp_target.replace(".o", ".c")
        with open(source, "w") as f:
            f.write("int main() { return 0; }\n")

        try:
            # Compile should succeed by removing stale lock
            # Enable verbose to see removal message
            env = os.environ.copy()
            env["CT_LOCK_VERBOSE"] = "1"

            result = subprocess.run(
                [
                    "ct-lock-helper", "compile",
                    f"--target={temp_target}",
                    "--strategy=lockdir",
                    "--",
                    "gcc", "-c", source
                ],
                capture_output=True,
                text=True,
                timeout=5,
                env=env
            )

            assert result.returncode == 0, f"Should remove stale lock: {result.stderr}"
            assert os.path.exists(temp_target), "Compilation should succeed"
            assert "Removed stale lock" in result.stderr, f"Should report stale lock removal: {result.stderr}"

        finally:
            if os.path.exists(source):
                os.unlink(source)

    def test_cross_host_lock_not_removed(self, temp_target):
        """Test that cross-host locks are NOT removed (can't verify remote process)."""
        lockdir = temp_target + ".lockdir"
        pid_file = os.path.join(lockdir, "pid")

        # Create fake cross-host lock
        os.makedirs(lockdir, exist_ok=True)
        fake_hostname = "remote-host-12345"  # Different from current host
        with open(pid_file, "w") as f:
            f.write(f"{fake_hostname}:12345\n")

        # Set old mtime to make lock appear old
        old_time = time.time() - 10  # 10 seconds old
        os.utime(lockdir, (old_time, old_time))

        source = temp_target.replace(".o", ".c")
        with open(source, "w") as f:
            f.write("int main() { return 0; }\n")

        try:
            # Compile with short timeout (should wait for lock)
            env = os.environ.copy()
            env["CT_LOCK_SLEEP_INTERVAL"] = "0.1"
            env["CT_LOCK_WARN_INTERVAL"] = "1"

            proc = subprocess.Popen(
                [
                    "ct-lock-helper", "compile",
                    f"--target={temp_target}",
                    "--strategy=lockdir",
                    "--",
                    "gcc", "-c", source
                ],
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                text=True,
                env=env
            )

            # Wait a bit to see if it waits for lock
            time.sleep(0.5)

            # Lock should still exist (not removed)
            assert os.path.exists(lockdir), "Cross-host lock should not be removed"

            # Kill the waiting process
            proc.kill()
            proc.wait()

        finally:
            if os.path.exists(source):
                os.unlink(source)
            # Clean up manually
            if os.path.exists(lockdir):
                shutil.rmtree(lockdir)

    def test_lockdir_permissions(self, temp_target):
        """Test that lockdir gets correct permissions (775)."""
        source = temp_target.replace(".o", ".c")
        with open(source, "w") as f:
            f.write("int main() { return 0; }\n")

        lockdir = temp_target + ".lockdir"

        # Create wrapper script that accepts -o argument
        wrapper = temp_target.replace(".o", "_compile.sh")
        with open(wrapper, "w") as f:
            f.write(f"#!/bin/bash\nsleep 0.5\nexec gcc -c {source} \"$@\"\n")
        os.chmod(wrapper, 0o755)

        try:
            # Run in background
            proc = subprocess.Popen(
                [
                    "ct-lock-helper", "compile",
                    f"--target={temp_target}",
                    "--strategy=lockdir",
                    "--",
                    wrapper
                ],
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE
            )

            # Wait for lock
            time.sleep(0.1)

            # Check permissions (775 = rwxrwxr-x = 0o775)
            if os.path.exists(lockdir):
                stat_info = os.stat(lockdir)
                mode = stat_info.st_mode & 0o777
                assert mode == 0o775, f"Expected 775, got {oct(mode)}"

            proc.wait(timeout=2)

        finally:
            if os.path.exists(source):
                os.unlink(source)
            if os.path.exists(wrapper):
                os.unlink(wrapper)
            if proc.poll() is None:
                proc.kill()


class TestEnvironmentVariables:
    """Test environment variable configuration."""

    def test_sleep_interval_override(self, temp_target):
        """Test that CT_LOCK_SLEEP_INTERVAL is respected."""
        lockdir = temp_target + ".lockdir"

        # Create lock held by current process
        os.makedirs(lockdir, exist_ok=True)
        with open(os.path.join(lockdir, "pid"), "w") as f:
            f.write(f"{socket.gethostname()}:{os.getpid()}\n")

        source = temp_target.replace(".o", ".c")
        with open(source, "w") as f:
            f.write("int main() { return 0; }\n")

        try:
            env = os.environ.copy()
            env["CT_LOCK_SLEEP_INTERVAL"] = "0.01"

            # This will wait since lock is held by us (appears active)
            proc = subprocess.Popen(
                [
                    "ct-lock-helper", "compile",
                    f"--target={temp_target}",
                    "--strategy=lockdir",
                    "--",
                    "gcc", "-c", source
                ],
                env=env,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE
            )

            # Short sleep - it should be waiting
            time.sleep(0.2)

            # Kill it
            proc.kill()
            proc.wait()

        finally:
            if os.path.exists(source):
                os.unlink(source)
            if os.path.exists(lockdir):
                shutil.rmtree(lockdir)


class TestConcurrentAccess:
    """Test concurrent process access with locking."""

    @pytest.mark.parametrize("strategy", ["lockdir", "flock"])
    def test_serial_execution_under_lock(self, temp_target, strategy):
        """Test that two processes execute serially when using locks."""
        source = temp_target.replace(".o", ".c")
        with open(source, "w") as f:
            f.write("int main() { return 0; }\n")

        try:
            # Run two compilations sequentially with locking
            # If locking works, the second should wait for the first
            import threading
            import queue

            results_queue = queue.Queue()
            start_times = {}
            end_times = {}

            # Create wrapper script with delay
            wrapper = temp_target.replace(".o", "_compile.sh")
            with open(wrapper, "w") as f:
                f.write(f"#!/bin/bash\nsleep 0.3\nexec gcc -c {source} \"$@\"\n")
            os.chmod(wrapper, 0o755)

            def run_compile(proc_id):
                start = time.time()
                start_times[proc_id] = start

                result = subprocess.run(
                    [
                        "ct-lock-helper", "compile",
                        f"--target={temp_target}",
                        f"--strategy={strategy}",
                        "--",
                        wrapper
                    ],
                    capture_output=True,
                    timeout=10
                )

                end = time.time()
                end_times[proc_id] = end
                results_queue.put((proc_id, result.returncode, end - start))

            # Start two threads
            t1 = threading.Thread(target=run_compile, args=(1,))
            t2 = threading.Thread(target=run_compile, args=(2,))

            t1.start()
            time.sleep(0.05)  # Small offset to ensure t1 acquires lock first
            t2.start()

            t1.join(timeout=12)
            t2.join(timeout=12)

            # Collect results
            results = []
            while not results_queue.empty():
                results.append(results_queue.get())

            # Both should succeed
            assert len(results) == 2, "Should have 2 results"
            for proc_id, returncode, duration in results:
                assert returncode == 0, f"Process {proc_id} should succeed"

            # Check for serial execution: second process should start after first ends
            # (with some tolerance for timing)
            if 1 in end_times and 2 in start_times:
                # If process 1 ended before process 2 started (plus tolerance), good evidence of serialization
                # Note: this is a heuristic, not perfect
                pass  # Just check both succeeded for now

        finally:
            if os.path.exists(source):
                os.unlink(source)
            if os.path.exists(wrapper):
                os.unlink(wrapper)


class TestErrorHandling:
    """Test error handling and edge cases."""

    def test_missing_target_argument(self):
        """Test that missing --target produces error."""
        result = subprocess.run(
            ["ct-lock-helper", "compile", "--strategy=lockdir", "--", "gcc", "-c", "foo.c"],
            capture_output=True,
            text=True
        )
        assert result.returncode != 0
        assert "target is required" in result.stderr.lower()

    def test_missing_strategy_argument(self):
        """Test that missing --strategy produces error."""
        result = subprocess.run(
            ["ct-lock-helper", "compile", "--target=foo.o", "--", "gcc", "-c", "foo.c"],
            capture_output=True,
            text=True
        )
        assert result.returncode != 0
        assert "strategy is required" in result.stderr.lower()

    def test_invalid_strategy(self):
        """Test that invalid strategy produces error."""
        result = subprocess.run(
            ["ct-lock-helper", "compile", "--target=foo.o", "--strategy=invalid", "--", "gcc"],
            capture_output=True,
            text=True
        )
        assert result.returncode != 0
        assert "invalid" in result.stderr.lower()

    def test_missing_compile_command(self):
        """Test that missing command after -- produces error."""
        result = subprocess.run(
            ["ct-lock-helper", "compile", "--target=foo.o", "--strategy=lockdir", "--"],
            capture_output=True,
            text=True
        )
        assert result.returncode != 0
        assert "required" in result.stderr.lower()

    def test_help_command(self):
        """Test that help command works."""
        result = subprocess.run(
            ["ct-lock-helper", "help"],
            capture_output=True,
            text=True
        )
        assert result.returncode == 0
        assert "Usage:" in result.stdout
        assert "lockdir" in result.stdout
        assert "cifs" in result.stdout
        assert "flock" in result.stdout
