# SPDX-FileCopyrightText: Copyright (c) 2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
# SPDX-License-Identifier: Apache-2.0
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import io
from contextlib import nullcontext

import torch
import torch.nn as nn
import torch.nn.functional as F
from einops import rearrange

from cosmos_predict2.tokenizers.interface import VideoTokenizerInterface
from imaginaire.utils import log
from imaginaire.utils.distributed import broadcast, get_rank, sync_model_states
from imaginaire.utils.easy_io import easy_io

__all__ = [
    "VAE",
]

CACHE_T = 2


class CausalConv3d(nn.Conv3d):
    """
    Causal 3d convolusion.
    """

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._padding = (self.padding[2], self.padding[2], self.padding[1], self.padding[1], 2 * self.padding[0], 0)
        self.padding = (0, 0, 0)

    def forward(self, x, cache_x=None):
        padding = list(self._padding)
        if cache_x is not None and self._padding[4] > 0:
            cache_x = cache_x.to(x.device)
            x = torch.cat([cache_x, x], dim=2)
            padding[4] -= cache_x.shape[2]
        x = F.pad(x, padding)

        return super().forward(x)


class RMS_norm(nn.Module):
    def __init__(self, dim, channel_first=True, images=True, bias=False):
        super().__init__()
        broadcastable_dims = (1, 1, 1) if not images else (1, 1)
        shape = (dim, *broadcastable_dims) if channel_first else (dim,)

        self.channel_first = channel_first
        self.scale = dim**0.5
        self.gamma = nn.Parameter(torch.ones(shape))
        self.bias = nn.Parameter(torch.zeros(shape)) if bias else 0.0

    def forward(self, x):
        return F.normalize(x, dim=(1 if self.channel_first else -1)) * self.scale * self.gamma + self.bias


class Upsample(nn.Upsample):
    def forward(self, x):
        """
        Fix bfloat16 support for nearest neighbor interpolation.
        """
        return super().forward(x.float()).type_as(x)


class Resample(nn.Module):
    def __init__(self, dim, mode):
        assert mode in ("none", "upsample2d", "upsample3d", "downsample2d", "downsample3d")
        super().__init__()
        self.dim = dim
        self.mode = mode

        # layers
        if mode == "upsample2d":
            self.resample = nn.Sequential(
                Upsample(scale_factor=(2.0, 2.0), mode="nearest-exact"), nn.Conv2d(dim, dim // 2, 3, padding=1)
            )
        elif mode == "upsample3d":
            self.resample = nn.Sequential(
                Upsample(scale_factor=(2.0, 2.0), mode="nearest-exact"), nn.Conv2d(dim, dim // 2, 3, padding=1)
            )
            self.time_conv = CausalConv3d(dim, dim * 2, (3, 1, 1), padding=(1, 0, 0))

        elif mode == "downsample2d":
            self.resample = nn.Sequential(nn.ZeroPad2d((0, 1, 0, 1)), nn.Conv2d(dim, dim, 3, stride=(2, 2)))
        elif mode == "downsample3d":
            self.resample = nn.Sequential(nn.ZeroPad2d((0, 1, 0, 1)), nn.Conv2d(dim, dim, 3, stride=(2, 2)))
            self.time_conv = CausalConv3d(dim, dim, (3, 1, 1), stride=(2, 1, 1), padding=(0, 0, 0))

        else:
            self.resample = nn.Identity()

    def forward(self, x, feat_cache=None, feat_idx=[0]):  # noqa: B006
        b, c, t, h, w = x.size()
        if self.mode == "upsample3d":
            if feat_cache is not None:
                idx = feat_idx[0]
                if feat_cache[idx] is None:
                    feat_cache[idx] = "Rep"
                    feat_idx[0] += 1
                else:
                    cache_x = x[:, :, -CACHE_T:, :, :].clone()
                    if cache_x.shape[2] < 2 and feat_cache[idx] is not None and feat_cache[idx] != "Rep":
                        # cache last frame of last two chunk
                        cache_x = torch.cat(
                            [feat_cache[idx][:, :, -1, :, :].unsqueeze(2).to(cache_x.device), cache_x], dim=2
                        )
                    if cache_x.shape[2] < 2 and feat_cache[idx] is not None and feat_cache[idx] == "Rep":
                        cache_x = torch.cat([torch.zeros_like(cache_x).to(cache_x.device), cache_x], dim=2)
                    if feat_cache[idx] == "Rep":
                        x = self.time_conv(x)
                    else:
                        x = self.time_conv(x, feat_cache[idx])
                    feat_cache[idx] = cache_x
                    feat_idx[0] += 1

                    x = x.reshape(b, 2, c, t, h, w)
                    x = torch.stack((x[:, 0, :, :, :, :], x[:, 1, :, :, :, :]), 3)
                    x = x.reshape(b, c, t * 2, h, w)
        t = x.shape[2]
        x = rearrange(x, "b c t h w -> (b t) c h w")
        x = self.resample(x)
        x = rearrange(x, "(b t) c h w -> b c t h w", t=t)

        if self.mode == "downsample3d":
            if feat_cache is not None:
                idx = feat_idx[0]
                if feat_cache[idx] is None:
                    feat_cache[idx] = x.clone()
                    feat_idx[0] += 1
                else:
                    cache_x = x[:, :, -1:, :, :].clone()
                    # if cache_x.shape[2] < 2 and feat_cache[idx] is not None and feat_cache[idx]!='Rep':
                    #     # cache last frame of last two chunk
                    #     cache_x = torch.cat([feat_cache[idx][:, :, -1, :, :].unsqueeze(2).to(cache_x.device), cache_x], dim=2)

                    x = self.time_conv(torch.cat([feat_cache[idx][:, :, -1:, :, :], x], 2))
                    feat_cache[idx] = cache_x
                    feat_idx[0] += 1
        return x


class ResidualBlock(nn.Module):
    def __init__(self, in_dim, out_dim, dropout=0.0):
        super().__init__()
        self.in_dim = in_dim
        self.out_dim = out_dim

        # layers
        self.residual = nn.Sequential(
            RMS_norm(in_dim, images=False),
            nn.SiLU(),
            CausalConv3d(in_dim, out_dim, 3, padding=1),
            RMS_norm(out_dim, images=False),
            nn.SiLU(),
            nn.Dropout(dropout),
            CausalConv3d(out_dim, out_dim, 3, padding=1),
        )
        self.shortcut = CausalConv3d(in_dim, out_dim, 1) if in_dim != out_dim else nn.Identity()

    def forward(self, x, feat_cache=None, feat_idx=[0]):  # noqa: B006
        h = self.shortcut(x)
        for layer in self.residual:
            if isinstance(layer, CausalConv3d) and feat_cache is not None:
                idx = feat_idx[0]
                cache_x = x[:, :, -CACHE_T:, :, :].clone()
                if cache_x.shape[2] < 2 and feat_cache[idx] is not None:
                    # cache last frame of last two chunk
                    cache_x = torch.cat(
                        [feat_cache[idx][:, :, -1, :, :].unsqueeze(2).to(cache_x.device), cache_x], dim=2
                    )
                x = layer(x, feat_cache[idx])
                feat_cache[idx] = cache_x
                feat_idx[0] += 1
            else:
                x = layer(x)
        return x + h


class AttentionBlock(nn.Module):
    """
    Causal self-attention with a single head.
    """

    def __init__(self, dim):
        super().__init__()
        self.dim = dim

        # layers
        self.norm = RMS_norm(dim)
        self.to_qkv = nn.Conv2d(dim, dim * 3, 1)
        self.proj = nn.Conv2d(dim, dim, 1)

        # zero out the last layer params
        nn.init.zeros_(self.proj.weight)

    def forward(self, x):
        identity = x
        b, c, t, h, w = x.size()
        x = rearrange(x, "b c t h w -> (b t) c h w")
        x = self.norm(x)
        # compute query, key, value
        q, k, v = self.to_qkv(x).reshape(b * t, 1, c * 3, -1).permute(0, 1, 3, 2).contiguous().chunk(3, dim=-1)

        # apply attention
        x = F.scaled_dot_product_attention(
            q,
            k,
            v,
        )
        x = x.squeeze(1).permute(0, 2, 1).reshape(b * t, c, h, w)

        # output
        x = self.proj(x)
        x = rearrange(x, "(b t) c h w-> b c t h w", t=t)
        return x + identity


class Encoder3d(nn.Module):
    def __init__(
        self,
        dim=128,
        z_dim=4,
        dim_mult=[1, 2, 4, 4],  # noqa: B006
        num_res_blocks=2,
        attn_scales=[],  # noqa: B006
        temperal_downsample=[True, True, False],  # noqa: B006
        dropout=0.0,
    ):
        super().__init__()
        self.dim = dim
        self.z_dim = z_dim
        self.dim_mult = dim_mult
        self.num_res_blocks = num_res_blocks
        self.attn_scales = attn_scales
        self.temperal_downsample = temperal_downsample

        # dimensions
        dims = [dim * u for u in [1] + dim_mult]  # noqa: RUF005
        scale = 1.0

        # init block
        self.conv1 = CausalConv3d(3, dims[0], 3, padding=1)

        # downsample blocks
        downsamples = []
        for i, (in_dim, out_dim) in enumerate(zip(dims[:-1], dims[1:], strict=False)):  # noqa: RUF007
            # residual (+attention) blocks
            for _ in range(num_res_blocks):
                downsamples.append(ResidualBlock(in_dim, out_dim, dropout))
                if scale in attn_scales:
                    downsamples.append(AttentionBlock(out_dim))
                in_dim = out_dim

            # downsample block
            if i != len(dim_mult) - 1:
                mode = "downsample3d" if temperal_downsample[i] else "downsample2d"
                downsamples.append(Resample(out_dim, mode=mode))
                scale /= 2.0
        self.downsamples = nn.Sequential(*downsamples)

        # middle blocks
        self.middle = nn.Sequential(
            ResidualBlock(out_dim, out_dim, dropout), AttentionBlock(out_dim), ResidualBlock(out_dim, out_dim, dropout)
        )

        # output blocks
        self.head = nn.Sequential(
            RMS_norm(out_dim, images=False), nn.SiLU(), CausalConv3d(out_dim, z_dim, 3, padding=1)
        )

    def forward(self, x, feat_cache=None, feat_idx=[0]):  # noqa: B006
        if feat_cache is not None:
            idx = feat_idx[0]
            cache_x = x[:, :, -CACHE_T:, :, :].clone()
            if cache_x.shape[2] < 2 and feat_cache[idx] is not None:
                # cache last frame of last two chunk
                cache_x = torch.cat([feat_cache[idx][:, :, -1, :, :].unsqueeze(2).to(cache_x.device), cache_x], dim=2)
            x = self.conv1(x, feat_cache[idx])
            feat_cache[idx] = cache_x
            feat_idx[0] += 1
        else:
            x = self.conv1(x)

        # downsamples
        for layer in self.downsamples:
            if feat_cache is not None:
                x = layer(x, feat_cache, feat_idx)
            else:
                x = layer(x)

        # middle
        for layer in self.middle:
            if isinstance(layer, ResidualBlock) and feat_cache is not None:
                x = layer(x, feat_cache, feat_idx)
            else:
                x = layer(x)

        # head
        for layer in self.head:
            if isinstance(layer, CausalConv3d) and feat_cache is not None:
                idx = feat_idx[0]
                cache_x = x[:, :, -CACHE_T:, :, :].clone()
                if cache_x.shape[2] < 2 and feat_cache[idx] is not None:
                    # cache last frame of last two chunk
                    cache_x = torch.cat(
                        [feat_cache[idx][:, :, -1, :, :].unsqueeze(2).to(cache_x.device), cache_x], dim=2
                    )
                x = layer(x, feat_cache[idx])
                feat_cache[idx] = cache_x
                feat_idx[0] += 1
            else:
                x = layer(x)
        return x


class Decoder3d(nn.Module):
    def __init__(
        self,
        dim=128,
        z_dim=4,
        dim_mult=[1, 2, 4, 4],  # noqa: B006
        num_res_blocks=2,
        attn_scales=[],  # noqa: B006
        temperal_upsample=[False, True, True],  # noqa: B006
        dropout=0.0,
    ):
        super().__init__()
        self.dim = dim
        self.z_dim = z_dim
        self.dim_mult = dim_mult
        self.num_res_blocks = num_res_blocks
        self.attn_scales = attn_scales
        self.temperal_upsample = temperal_upsample

        # dimensions
        dims = [dim * u for u in [dim_mult[-1]] + dim_mult[::-1]]  # noqa: RUF005
        scale = 1.0 / 2 ** (len(dim_mult) - 2)

        # init block
        self.conv1 = CausalConv3d(z_dim, dims[0], 3, padding=1)

        # middle blocks
        self.middle = nn.Sequential(
            ResidualBlock(dims[0], dims[0], dropout), AttentionBlock(dims[0]), ResidualBlock(dims[0], dims[0], dropout)
        )

        # upsample blocks
        upsamples = []
        for i, (in_dim, out_dim) in enumerate(zip(dims[:-1], dims[1:], strict=False)):  # noqa: RUF007
            # residual (+attention) blocks
            if i == 1 or i == 2 or i == 3:
                in_dim = in_dim // 2
            for _ in range(num_res_blocks + 1):
                upsamples.append(ResidualBlock(in_dim, out_dim, dropout))
                if scale in attn_scales:
                    upsamples.append(AttentionBlock(out_dim))
                in_dim = out_dim

            # upsample block
            if i != len(dim_mult) - 1:
                mode = "upsample3d" if temperal_upsample[i] else "upsample2d"
                upsamples.append(Resample(out_dim, mode=mode))
                scale *= 2.0
        self.upsamples = nn.Sequential(*upsamples)

        # output blocks
        self.head = nn.Sequential(RMS_norm(out_dim, images=False), nn.SiLU(), CausalConv3d(out_dim, 3, 3, padding=1))

    def forward(self, x, feat_cache=None, feat_idx=[0]):  # noqa: B006
        # conv1
        if feat_cache is not None:
            idx = feat_idx[0]
            cache_x = x[:, :, -CACHE_T:, :, :].clone()
            if cache_x.shape[2] < 2 and feat_cache[idx] is not None:
                # cache last frame of last two chunk
                cache_x = torch.cat([feat_cache[idx][:, :, -1, :, :].unsqueeze(2).to(cache_x.device), cache_x], dim=2)
            x = self.conv1(x, feat_cache[idx])
            feat_cache[idx] = cache_x
            feat_idx[0] += 1
        else:
            x = self.conv1(x)

        # middle
        for layer in self.middle:
            if isinstance(layer, ResidualBlock) and feat_cache is not None:
                x = layer(x, feat_cache, feat_idx)
            else:
                x = layer(x)

        # upsamples
        for layer in self.upsamples:
            if feat_cache is not None:
                x = layer(x, feat_cache, feat_idx)
            else:
                x = layer(x)

        # head
        for layer in self.head:
            if isinstance(layer, CausalConv3d) and feat_cache is not None:
                idx = feat_idx[0]
                cache_x = x[:, :, -CACHE_T:, :, :].clone()
                if cache_x.shape[2] < 2 and feat_cache[idx] is not None:
                    # cache last frame of last two chunk
                    cache_x = torch.cat(
                        [feat_cache[idx][:, :, -1, :, :].unsqueeze(2).to(cache_x.device), cache_x], dim=2
                    )
                x = layer(x, feat_cache[idx])
                feat_cache[idx] = cache_x
                feat_idx[0] += 1
            else:
                x = layer(x)
        return x


def count_conv3d(model):
    count = 0
    for m in model.modules():
        if isinstance(m, CausalConv3d):
            count += 1
    return count


class VAE_(nn.Module):
    def __init__(
        self,
        dim=128,
        z_dim=4,
        dim_mult=[1, 2, 4, 4],  # noqa: B006
        num_res_blocks=2,
        attn_scales=[],  # noqa: B006
        temperal_downsample=[True, True, False],  # noqa: B006
        dropout=0.0,
        temporal_window=4,
    ):
        super().__init__()
        self.dim = dim
        self.z_dim = z_dim
        self.dim_mult = dim_mult
        self.num_res_blocks = num_res_blocks
        self.attn_scales = attn_scales
        self.temperal_downsample = temperal_downsample
        self.temperal_upsample = temperal_downsample[::-1]
        self.temporal_window = temporal_window
        # modules
        self.encoder = Encoder3d(
            dim, z_dim * 2, dim_mult, num_res_blocks, attn_scales, self.temperal_downsample, dropout
        )
        self.conv1 = CausalConv3d(z_dim * 2, z_dim * 2, 1)
        self.conv2 = CausalConv3d(z_dim, z_dim, 1)
        self.decoder = Decoder3d(dim, z_dim, dim_mult, num_res_blocks, attn_scales, self.temperal_upsample, dropout)

    def forward(self, x):
        mu, log_var = self.encode(x)
        z = self.reparameterize(mu, log_var)
        x_recon = self.decode(z)
        return x_recon, mu, log_var

    def encode(self, x, scale):
        self.clear_cache()
        # cache
        t = x.shape[2]
        iter_ = 1 + (t - 1) // self.temporal_window
        for i in range(iter_):
            self._enc_conv_idx = [0]
            if i == 0:
                out = self.encoder(x[:, :, :1, :, :], feat_cache=self._enc_feat_map, feat_idx=self._enc_conv_idx)
            else:
                out_ = self.encoder(
                    x[:, :, 1 + self.temporal_window * (i - 1) : 1 + self.temporal_window * i, :, :],
                    feat_cache=self._enc_feat_map,
                    feat_idx=self._enc_conv_idx,
                )
                out = torch.cat([out, out_], 2)
        if (t - 1) % self.temporal_window:
            self._enc_conv_idx = [0]
            out_ = self.encoder(
                x[:, :, 1 + self.temporal_window * (iter_ - 1) :, :, :],
                feat_cache=self._enc_feat_map,
                feat_idx=self._enc_conv_idx,
            )
            out = torch.cat([out, out_], 2)
        mu, log_var = self.conv1(out).chunk(2, dim=1)
        if isinstance(scale[0], torch.Tensor):
            mu = (mu - scale[0].view(1, self.z_dim, 1, 1, 1)) * scale[1].view(1, self.z_dim, 1, 1, 1)
        else:
            mu = (mu - scale[0]) * scale[1]
        self.clear_cache()
        return mu

    def decode(self, z, scale):
        self.clear_cache()
        # z: [b,c,t,h,w]
        if isinstance(scale[0], torch.Tensor):
            z = z / scale[1].view(1, self.z_dim, 1, 1, 1) + scale[0].view(1, self.z_dim, 1, 1, 1)
        else:
            z = z / scale[1] + scale[0]
        iter_ = z.shape[2]
        x = self.conv2(z)
        for i in range(iter_):
            self._conv_idx = [0]
            if i == 0:
                out = self.decoder(x[:, :, i : i + 1, :, :], feat_cache=self._feat_map, feat_idx=self._conv_idx)
            else:
                out_ = self.decoder(x[:, :, i : i + 1, :, :], feat_cache=self._feat_map, feat_idx=self._conv_idx)
                out = torch.cat([out, out_], 2)
        self.clear_cache()
        return out

    def reparameterize(self, mu, log_var):
        std = torch.exp(0.5 * log_var)
        eps = torch.randn_like(std)
        return eps * std + mu

    def sample(self, imgs, deterministic=False):
        mu, log_var = self.encode(imgs)
        if deterministic:
            return mu
        std = torch.exp(0.5 * log_var.clamp(-30.0, 20.0))
        return mu + std * torch.randn_like(std)

    def clear_cache(self):
        self._conv_num = count_conv3d(self.decoder)
        self._conv_idx = [0]
        self._feat_map = [None] * self._conv_num
        # cache encode
        self._enc_conv_num = count_conv3d(self.encoder)
        self._enc_conv_idx = [0]
        self._enc_feat_map = [None] * self._enc_conv_num


def _video_vae(
    pretrained_path=None,
    z_dim=None,
    device="cpu",
    load_mean_std=False,
    mean_std_path: str = "",
    **kwargs,
):
    """
    Autoencoder3d adapted from Stable Diffusion 1.x, 2.x and XL.
    """
    # params
    cfg = dict(
        dim=96,
        z_dim=z_dim,
        dim_mult=[1, 2, 4, 4],
        num_res_blocks=2,
        attn_scales=[],
        temperal_downsample=[False, True, True],
        dropout=0.0,
    )
    cfg.update(**kwargs)

    # init model
    with torch.device("meta"):
        model = VAE_(**cfg)

    if pretrained_path is None:
        model.to_empty(device=device)
        if load_mean_std:
            img_mean, img_std = torch.randn(1, 16, 1, 1, 1, device=device), torch.randn(1, 16, 1, 1, 1, device=device)
            video_mean, video_std = (
                torch.randn(1, 16, 32, 1, 1, device=device),
                torch.randn(1, 16, 32, 1, 1, device=device),
            )
    else:
        if get_rank() == 0:
            ckpt = easy_io.load(
                pretrained_path,
                backend_key=None,
                map_location=device,
            )
            if load_mean_std:
                img_mean_std = mean_std_path.replace("mean_std.pt", "images_mean_std.pt")
                video_mean_std = mean_std_path.replace("mean_std.pt", "video_mean_std.pt")
                img_mean, img_std = easy_io.load(img_mean_std, backend_key=backend_key, map_location=device)  # noqa: F821
                video_mean, video_std = easy_io.load(video_mean_std, backend_key=backend_key, map_location=device)  # noqa: F821
                img_mean = img_mean.reshape(1, 16, 1, 1, 1)
                img_std = img_std.reshape(1, 16, 1, 1, 1)
                video_mean = video_mean.reshape(1, 16, 32, 1, 1)
                video_std = video_std.reshape(1, 16, 32, 1, 1)

            # load checkpoint
            log.info(f"Loading {pretrained_path}")
            model.load_state_dict(ckpt, assign=True)
            log.success(f"Successfully loaded {pretrained_path}")
        else:
            model.to_empty(device=device)
            if load_mean_std:
                img_mean, img_std = (
                    torch.randn(1, 16, 1, 1, 1, device=device),
                    torch.randn(1, 16, 1, 1, 1, device=device),
                )
                video_mean, video_std = (
                    torch.randn(1, 16, 32, 1, 1, device=device),
                    torch.randn(1, 16, 32, 1, 1, device=device),
                )
    sync_model_states(model)

    if load_mean_std:
        log.info("broadcast mean and std")
        broadcast(img_mean, 0)
        broadcast(img_std, 0)
        broadcast(video_mean, 0)
        broadcast(video_std, 0)
        return model, img_mean, img_std, video_mean, video_std

    return (
        model,
        torch.zeros(1, 1, 1, 1, 1, device=device),
        torch.ones(1, 1, 1, 1, 1, device=device),
        torch.zeros(1, 1, 50, 1, 1, device=device),
        torch.ones(1, 1, 50, 1, 1, device=device),
    )


class VAE:
    def __init__(
        self,
        z_dim=16,
        vae_pth="",
        load_mean_std=False,
        mean_std_path: str = "",
        dtype=torch.float,
        device="cuda",
        is_amp=True,
        temporal_window: int = 4,
    ):
        self.dtype = dtype
        self.device = device
        self.temporal_window = temporal_window

        mean = [
            -0.7571,
            -0.7089,
            -0.9113,
            0.1075,
            -0.1745,
            0.9653,
            -0.1517,
            1.5508,
            0.4134,
            -0.0715,
            0.5517,
            -0.3632,
            -0.1922,
            -0.9497,
            0.2503,
            -0.2921,
        ]
        std = [
            2.8184,
            1.4541,
            2.3275,
            2.6558,
            1.2196,
            1.7708,
            2.6052,
            2.0743,
            3.2687,
            2.1526,
            2.8652,
            1.5579,
            1.6382,
            1.1253,
            2.8251,
            1.9160,
        ]
        self.mean = torch.tensor(mean, dtype=dtype, device=device)
        self.std = torch.tensor(std, dtype=dtype, device=device)
        self.scale = [self.mean, 1.0 / self.std]

        # init model
        self.model, self.img_mean, self.img_std, self.video_mean, self.video_std = _video_vae(
            pretrained_path=vae_pth,
            z_dim=z_dim,
            load_mean_std=load_mean_std,
            mean_std_path=mean_std_path,
            device=device,
            temporal_window=temporal_window,
        )
        self.model = self.model.eval().requires_grad_(False)
        self.is_amp = is_amp
        if not is_amp:
            self.model = self.model.to(dtype=dtype)
            self.context = nullcontext()
        else:
            self.context = torch.amp.autocast("cuda", dtype=dtype)

    def count_param(self):
        return sum(p.numel() for p in self.model.parameters())

    @torch.no_grad()
    def encode(self, videos):
        """
        videos: A list of videos each with shape [C, T, H, W].
        """
        in_dtype = videos.dtype
        with self.context:
            if not self.is_amp:
                videos = videos.to(self.dtype)
            latent = self.model.encode(videos, self.scale)
        latent = latent.to(in_dtype)
        return latent

    @torch.no_grad()
    def decode(self, zs):
        in_dtype = zs.dtype
        with self.context:
            if not self.is_amp:
                zs = zs.to(self.dtype)
            video_recon = self.model.decode(zs, self.scale)
        video_recon = video_recon.to(in_dtype)
        return video_recon


class CosmosImageTokenizer(torch.nn.Module, VideoTokenizerInterface):
    """
    Cosmos Image Tokenizer class.
    """

    def __init__(
        self,
        vae_pth: str,
        name: str = "",
        dtype: torch.dtype = torch.bfloat16,
        pixel_chunk_duration: int = 32,
        num_overlap_latent: int = 0,
        spatial_compression_factor: int = 8,
        temporal_compression_factor: int = 8,
        latent_ch: int = 16,
        max_enc_batch_size: int = 4,
        max_dec_batch_size: int = 2,
        spatial_resolution: str = "720",
        apply_mean_std: bool = True,
        is_image: bool = False,
        state_dict: bool = False,
        squeeze_for_image: bool = True,
    ):
        """Cosmos Image tokenizer.
        Args:
            vae_pth (str): The file path to the vae model.
            name (str): The name of the model, which include the hyper-parameters passed to the model.
            dtype (torch.dtype): The data type of the model.
            pixel_chunk_duration (int): The number of pixel frames in a chunk.
            num_overlap_latent (int): The number of overlapping latent frames between two chunks.
            spatial_compression_factor (int): The spatial compression factor.
            temporal_compression_factor (int): The temporal compression factor.
            latent_ch (int): The number of channels in the latent tensor.
            max_enc_batch_size (int): The maximum batch size for encoding.
            max_dec_batch_size (int): The maximum batch size for decoding.
            spatial_resolution (str): The spatial resolution of the input data.
            apply_mean_std (bool): Whether to apply mean and std normalization.
            is_image (bool): Whether the input data is image.
            state_dict (bool): Whether to use the state_dict module. If True, module_lazy_class parameter also needs to be passed. Defaults to False.
            squeeze_for_image (bool): Whether to squeeze the image data. Defaults to False.
        """
        super().__init__()

        self.dtype = dtype
        self.is_image = is_image
        self.is_casual = False

        self.apply_mean_std = apply_mean_std
        self.num_overlap_latent = num_overlap_latent
        self.name = name
        self._spatial_compression_factor = spatial_compression_factor
        self._temporal_compression_factor = temporal_compression_factor
        self._pixel_chunk_duration = pixel_chunk_duration
        self._spatial_resolution = spatial_resolution
        self.max_enc_batch_size = max_enc_batch_size
        self.max_dec_batch_size = max_dec_batch_size
        self._latent_ch = latent_ch
        self.use_state_dict = state_dict
        self.squeeze_for_image = squeeze_for_image

        tokenizer_ckpt = torch.load(vae_pth)

        self.encoder = torch.jit.load(io.BytesIO(tokenizer_ckpt["encoder"]))
        self.decoder = torch.jit.load(io.BytesIO(tokenizer_ckpt["decoder"]))
        self.encoder.eval()
        self.decoder.eval()
        mean_std = tokenizer_ckpt["mean_std"]
        self.register_mean_std(mean_std[0], mean_std[1])

        log.info(f"Built tokenizer {self.name}")

    @property
    def latent_ch(self):
        return self._latent_ch

    def register_mean_std(self, mean, std) -> None:
        latent_mean, latent_std = mean, std
        latent_mean = latent_mean.to(torch.device(torch.cuda.current_device()))
        latent_std = latent_std.to(torch.device(torch.cuda.current_device()))
        latent_mean = latent_mean.view(self.latent_ch, -1)
        latent_std = latent_std.view(self.latent_ch, -1)
        target_shape = [1, self.latent_ch, 1, 1, 1]
        latent_mean = latent_mean.reshape(*target_shape)
        latent_std = latent_std.reshape(*target_shape)

        self.register_buffer(
            "latent_mean",
            latent_mean.to(self.dtype),
            persistent=False,
        )
        self.register_buffer(
            "latent_std",
            latent_std.to(self.dtype),
            persistent=False,
        )

    def reset_dtype(self, *args, **kwargs):
        """
        Resets the data type of the encoder and decoder to the model's default data type.

        Args:
            *args, **kwargs: Unused, present to allow flexibility in method calls.
        """
        del args, kwargs
        self.decoder.cuda().to(self.dtype)
        self.encoder.cuda().to(self.dtype)
        self.latent_mean = self.latent_mean.cuda().to(self.dtype)
        self.latent_std = self.latent_std.cuda().to(self.dtype)

    def normalize_latent(self, latent: torch.Tensor) -> torch.Tensor:
        num_frames = latent.shape[2]
        latent_mean = self.latent_mean.to(latent.dtype)
        latent_std = self.latent_std.to(latent.dtype)
        return (latent - latent_mean[:, :, :num_frames]) / latent_std[:, :, :num_frames]

    def denormalize_latent(self, latent: torch.Tensor) -> torch.Tensor:
        latent_mean = self.latent_mean.to(latent.dtype)
        latent_std = self.latent_std.to(latent.dtype)
        num_frames = latent.shape[2]
        return latent * latent_std[:, :, :num_frames] + latent_mean[:, :, :num_frames]

    @property
    def spatial_compression_factor(self):
        """hyper-parameter of vae"""
        return self._spatial_compression_factor

    @property
    def temporal_compression_factor(self):
        """hyper-parameter of vae"""
        return self._temporal_compression_factor

    @property
    def spatial_resolution(self):
        return self._spatial_resolution

    @property
    def pixel_chunk_duration(self):
        """number of pixel frames in a chunk, inference time hyper-parameter"""
        return self._pixel_chunk_duration

    @property
    def latent_chunk_duration(self):
        """number of latent frames in a chunk, computed from pixel_chunk_duration and temporal_compression_factor"""
        return self.pixel_chunk_duration // self.temporal_compression_factor

    @property
    def num_overlap_pixels(self):
        """number of overlapping pixel frames between two chunks, computed from num_overlap_latent and temporal_compression_factor"""
        return self.num_overlap_latent * self.temporal_compression_factor

    def get_latent_num_frames(self, num_pixel_frames: int) -> int:
        """Given total number of pixel frames, compute the number of latent frames."""
        return 1

    def get_pixel_num_frames(self, num_latent_frames: int) -> int:
        return 1

    def compute_num_chunks(self, num_pixel_frames: int = None, num_latent_frames: int = None) -> int:  # noqa: RUF013
        """Compute the number of chunks given the number of pixel frames or latent frames.
        The number of chunks is computed based on the pixel_chunk_duration and num_overlap_pixels.
        It will raise an error if the number of frames is not divisible by the chunk size.

        Args:
            num_pixel_frames (int): The number of pixel frames.
            num_latent_frames (int): The number of latent frames, if num_pixel_frames is not provided.
        Returns:
            num_chunks (int): The number of chunks.

        """
        if num_pixel_frames is None:
            assert num_latent_frames is not None, "Either num_pixel_frames or num_latent_frames should be provided"
            num_pixel_frames = self.get_pixel_num_frames(num_latent_frames)
        assert num_pixel_frames >= self.pixel_chunk_duration, (
            f"Expect the number of frames {num_pixel_frames} to be larger than the chunk size {self.pixel_chunk_duration}"
        )

        # Check if the number of frames is divisible by the chunk size
        if (num_pixel_frames - self.pixel_chunk_duration) % (self.pixel_chunk_duration - self.num_overlap_pixels) != 0:
            num_generate_pixels_per_chunk = self.pixel_chunk_duration - self.num_overlap_pixels
            # Suggest the number of frames that is divisible by the chunk size
            suggested_num_pixel_frames = (
                num_pixel_frames - self.pixel_chunk_duration
            ) // num_generate_pixels_per_chunk * num_generate_pixels_per_chunk + self.pixel_chunk_duration
            raise ValueError(
                f"Expect the number of frames {num_pixel_frames} - {self.pixel_chunk_duration} to be divisible "
                f"by the chunk size {self.pixel_chunk_duration} - {self.num_overlap_pixels}, "
                f"suggested number of frames is {suggested_num_pixel_frames}"
            )

        num_chunks = 1 + (num_pixel_frames - self.pixel_chunk_duration) // (
            self.pixel_chunk_duration - self.num_overlap_pixels
        )
        log.debug(
            f"num_chunks={num_chunks} given num_pixel_frames: {num_pixel_frames}, num_overlap_pixels: {self.num_overlap_pixels}, pixel_chunk_duration: {self.pixel_chunk_duration}"
        )
        return num_chunks

    @torch.no_grad()
    def encode(self, state: torch.Tensor) -> torch.Tensor:
        if not self.squeeze_for_image:
            assert state.shape[2] == 1, f"Expect the number of frames {state.shape[2]} to be 1"
        in_dtype = state.dtype

        if self.squeeze_for_image:
            latent = self.encoder(state.to(self.dtype).squeeze(2))
        else:
            latent = self.encoder(state.to(self.dtype))
        if isinstance(latent, tuple):
            latent = latent[0]
        if self.squeeze_for_image:
            latent = latent.unsqueeze(2)
        assert isinstance(latent, torch.Tensor), "Invalid type of encoded state"

        if self.apply_mean_std:
            latent = self.normalize_latent(latent)

        latent = latent.to(in_dtype)
        return latent

    @torch.no_grad()
    def decode(self, state: torch.Tensor) -> torch.Tensor:
        assert state.shape[2] == 1, f"Expect the number of frames {state.shape[2]} to be 1"
        in_dtype = state.dtype
        if self.apply_mean_std:
            state = self.denormalize_latent(state.to(self.dtype))
        if self.squeeze_for_image:
            state = state.squeeze(2)
        decoded_state = self.decoder(state.to(self.dtype))
        if self.squeeze_for_image:
            decoded_state = decoded_state.unsqueeze(2)
        decoded_state = decoded_state.to(in_dtype)
        return decoded_state


class TokenizerInterface(VideoTokenizerInterface):
    def __init__(self, chunk_duration: int = 81, load_mean_std=False, **kwargs):
        self.model = VAE(
            dtype=torch.bfloat16,
            is_amp=False,
            load_mean_std=load_mean_std,
            vae_pth=kwargs.get(
                "vae_pth",
                "",
            ),
            temporal_window=kwargs.get("temporal_window", 4),
        )
        del kwargs
        self.chunk_duration = chunk_duration

    @property
    def dtype(self):
        return self.model.dtype

    def to(self, **kwargs) -> VAE_:
        return self.model.model.to(**kwargs)

    def reset_dtype(self):
        pass

    # @torch.compile(dynamic=False); it will significantly slow down the training job
    def encode(self, state: torch.Tensor) -> torch.Tensor:
        latents = self.model.encode(state)
        num_frames = latents.shape[2]
        if num_frames == 1:
            return (latents - self.model.img_mean.type_as(latents)) / self.model.img_std.type_as(latents)
        else:
            return (latents - self.model.video_mean[:, :, :num_frames].type_as(latents)) / self.model.video_std[
                :, :, :num_frames
            ].type_as(latents)

    def decode(self, latent: torch.Tensor) -> torch.Tensor:
        num_frames = latent.shape[2]
        if num_frames == 1:
            return self.model.decode(
                (latent * self.model.img_std.type_as(latent)) + self.model.img_mean.type_as(latent)
            )
        else:
            return self.model.decode(
                (latent * self.model.video_std[:, :, :num_frames].type_as(latent))
                + self.model.video_mean[:, :, :num_frames].type_as(latent)
            )

    def get_latent_num_frames(self, num_pixel_frames: int) -> int:
        return 1 + (num_pixel_frames - 1) // 4

    def get_pixel_num_frames(self, num_latent_frames: int) -> int:
        return (num_latent_frames - 1) * 4 + 1

    @property
    def spatial_compression_factor(self):
        return 8

    @property
    def temporal_compression_factor(self):
        return 4

    @property
    def pixel_chunk_duration(self):
        return self.chunk_duration

    @property
    def latent_chunk_duration(self):
        return self.get_latent_num_frames(self.chunk_duration)

    @property
    def latent_ch(self):
        return 16

    @property
    def spatial_resolution(self):
        return 512

    @property
    def name(self):
        return "tokenizer"
