"""
High-level integration coverage for the PyO3-backed Runtime.

The suite exercises the public Python API end to end, ensuring we can spawn
isolates, run sync/async JS, preserve state, enforce shutdown semantics, and
surface promise/timeouts/errors consistently with the Rust core.
"""

import asyncio
import math
import time
from datetime import datetime, timedelta, timezone

import pytest
from jsrun import (
    JavaScriptError,
    JsFunction,
    JsStream,
    JsUndefined,
    Runtime,
    RuntimeConfig,
    RuntimeStats,
    RuntimeTerminated,
    SnapshotBuilder,
    undefined,
)


class TestRuntimeStats:
    def test_runtime_stats_initial_values(self):
        runtime = Runtime()
        try:
            stats = runtime.get_stats()
            assert isinstance(stats, RuntimeStats)
            assert stats.total_execution_time_ms == 0
            assert stats.last_execution_time_ms == 0
            assert stats.last_execution_kind is None
            assert stats.eval_sync_count == 0
            assert stats.eval_async_count == 0
        finally:
            runtime.close()

    def test_runtime_stats_after_eval(self):
        runtime = Runtime()
        try:
            before = runtime.get_stats()
            runtime.eval("Array.from({length: 1_000}).reduce((a, b) => a + b, 0)")
            after_first = runtime.get_stats()

            assert after_first.eval_sync_count == before.eval_sync_count + 1
            assert after_first.total_execution_time_ms >= before.total_execution_time_ms
            assert after_first.last_execution_time_ms >= 0
            assert after_first.last_execution_kind == "eval_sync"

            runtime.eval("Array(10).fill(0).map((_, i) => i).join(',')")
            after_second = runtime.get_stats()

            assert after_second.eval_sync_count == after_first.eval_sync_count + 1
            assert (
                after_second.total_execution_time_ms
                >= after_first.total_execution_time_ms
            )
            assert after_second.last_execution_kind == "eval_sync"
        finally:
            runtime.close()

    @pytest.mark.asyncio
    async def test_runtime_stats_async_paths(self):
        runtime = Runtime()
        try:
            before = runtime.get_stats()
            await runtime.eval_async("Promise.resolve(123)")
            after_async = runtime.get_stats()

            assert after_async.eval_async_count == before.eval_async_count + 1
            assert after_async.total_execution_time_ms >= before.total_execution_time_ms

            js_fn = runtime.eval("(a, b) => a + b")
            result = js_fn(4, 6)
            assert result == 10

            after_call = runtime.get_stats()
            assert (
                after_call.call_function_sync_count
                == after_async.call_function_sync_count + 1
            )

            async_fn = runtime.eval("async (value) => value * 2")
            assert await async_fn.call_async(5) == 10

            final_stats = runtime.get_stats()
            assert (
                final_stats.call_function_async_count
                == after_call.call_function_async_count + 1
            )
            assert (
                final_stats.total_execution_time_ms
                >= after_async.total_execution_time_ms
            )
        finally:
            runtime.close()

    @pytest.mark.asyncio
    async def test_runtime_stats_module_paths(self):
        runtime_sync = Runtime()
        try:
            runtime_sync.add_static_module(
                "stats_mod",
                "export const value = 42; export const doubled = value * 2;",
            )

            before = runtime_sync.get_stats()
            namespace = runtime_sync.eval_module("stats_mod")
            assert namespace["value"] == 42

            after_sync = runtime_sync.get_stats()
            assert (
                after_sync.eval_module_sync_count == before.eval_module_sync_count + 1
            )
        finally:
            runtime_sync.close()

        runtime_async = Runtime()
        try:
            runtime_async.add_static_module(
                "stats_mod_async",
                "export const delayed = await Promise.resolve(21);",
            )

            before_async = runtime_async.get_stats()
            namespace = await runtime_async.eval_module_async("stats_mod_async")
            assert namespace["delayed"] == 21

            after_async = runtime_async.get_stats()
            assert (
                after_async.eval_module_async_count
                == before_async.eval_module_async_count + 1
            )
        finally:
            runtime_async.close()


class TestRuntimeBasics:
    """Basic Runtime creation and lifecycle tests."""

    def test_runtime_spawn(self):
        """Test that Runtime() creates a new runtime."""
        runtime = Runtime()
        assert runtime is not None
        assert not runtime.is_closed()
        runtime.close()

    def test_sanitized_globals(self):
        """Test that internal globals are sanitized from the runtime."""
        runtime = Runtime()
        try:
            result = runtime.eval("typeof Deno")
            assert result == "undefined"
        finally:
            runtime.close()

    def test_runtime_eval_simple(self):
        """Test basic JavaScript evaluation."""
        runtime = Runtime()
        try:
            result = runtime.eval("2 + 2")
            assert result == 4
        finally:
            runtime.close()

    def test_runtime_eval_string(self):
        """Test string evaluation."""
        runtime = Runtime()
        try:
            result = runtime.eval("'hello world'")
            assert result == "hello world"
        finally:
            runtime.close()

    def test_runtime_eval_expression(self):
        """Test more complex expressions."""
        runtime = Runtime()
        try:
            result = runtime.eval("Math.max(10, 20, 30)")
            assert result == 30
        finally:
            runtime.close()

    def test_runtime_eval_javascript_error_metadata(self):
        """JavaScript exceptions should raise JavaScriptError with metadata."""
        runtime = Runtime()
        try:
            with pytest.raises(JavaScriptError) as exc_info:
                runtime.eval(
                    """
                    (() => {
                        const error = new TypeError("boom");
                        error.custom = 123;
                        throw error;
                    })()
                    """
                )
            js_error = exc_info.value
            assert js_error.name == "TypeError"
            assert "boom" in js_error.message
            assert js_error.stack is not None
            assert "<eval>" in js_error.stack
            assert "TypeError" in str(js_error)
        finally:
            runtime.close()

    def test_runtime_eval_javascript_error_frames(self):
        """JavaScript exceptions should capture stack frame information."""
        runtime = Runtime()
        try:
            with pytest.raises(JavaScriptError) as exc_info:
                runtime.eval(
                    """
                    function causeError() {
                        throw new TypeError("boom");
                    }
                    causeError()
                    """
                )
            js_error = exc_info.value
            assert js_error.name == "TypeError"
            assert "boom" in js_error.message

            # Validate frame structure
            assert len(js_error.frames) > 0
            frame = js_error.frames[0]
            assert "function_name" in frame
            assert frame.get("function_name") == "causeError"
            assert "file_name" in frame
            assert "line_number" in frame
        finally:
            runtime.close()

    def test_runtime_close(self):
        """Test runtime shutdown."""
        runtime = Runtime()
        assert not runtime.is_closed()

        runtime.close()
        assert runtime.is_closed()

    def test_runtime_close_idempotent(self):
        """Test that close() can be called multiple times safely."""
        runtime = Runtime()
        runtime.close()
        runtime.close()  # Should not raise
        assert runtime.is_closed()

    def test_runtime_eval_after_close(self):
        """Test that eval after close raises an error."""
        runtime = Runtime()
        runtime.close()

        with pytest.raises(RuntimeError) as exc_info:
            runtime.eval("1 + 1")
        assert "closed" in str(exc_info.value).lower()


class TestRuntimeStatePersistence:
    """Tests for state persistence across evaluations."""

    def test_variable_persistence(self):
        """Test that variables persist across eval calls."""
        runtime = Runtime()
        try:
            runtime.eval("var x = 10;")
            result = runtime.eval("x")
            assert result == 10

            runtime.eval("x = 20;")
            result = runtime.eval("x")
            assert result == 20
        finally:
            runtime.close()

    def test_function_persistence(self):
        """Test that functions persist across eval calls."""
        runtime = Runtime()
        try:
            runtime.eval("function add(a, b) { return a + b; }")
            result = runtime.eval("add(5, 7)")
            assert result == 12
        finally:
            runtime.close()

    def test_object_persistence(self):
        """Test that objects persist across eval calls."""
        runtime = Runtime()
        try:
            runtime.eval("var obj = {name: 'test', value: 42};")
            result = runtime.eval("obj.name")
            assert result == "test"

            result = runtime.eval("obj.value")
            assert result == 42

            runtime.eval("obj.value = 100;")
            result = runtime.eval("obj.value")
            assert result == 100
        finally:
            runtime.close()

    def test_multiple_sequential_evals(self):
        """Test multiple sequential evaluations."""
        runtime = Runtime()
        try:
            runtime.eval("var counter = 0;")

            for i in range(1, 6):
                runtime.eval("counter++;")
                result = runtime.eval("counter")
                assert result == i
        finally:
            runtime.close()


class TestRuntimeContextManager:
    """Tests for context manager protocol support."""

    def test_context_manager_basic(self):
        """Test basic context manager usage."""
        with Runtime() as runtime:
            result = runtime.eval("3 + 3")
            assert result == 6
        # Runtime should be closed after exiting context

    def test_context_manager_auto_close(self):
        """Test that context manager closes runtime automatically."""
        runtime = Runtime()
        assert not runtime.is_closed()

        with runtime:
            result = runtime.eval("5 * 5")
            assert result == 25
            assert not runtime.is_closed()

        # Should be closed after exiting
        assert runtime.is_closed()

    def test_context_manager_with_exception(self):
        """Test that runtime is closed even if exception occurs."""
        runtime = Runtime()

        try:
            with runtime:
                runtime.eval("var x = 1;")
                raise ValueError("Test exception")
        except ValueError:
            pass

        # Runtime should still be closed
        assert runtime.is_closed()


class TestRuntimeTermination:
    """Tests for graceful runtime termination."""

    @pytest.mark.asyncio
    async def test_runtime_terminate_interrupts_execution(self):
        runtime = Runtime()

        async def kill_runtime():
            await asyncio.sleep(0.1)
            runtime.terminate()

        killer = asyncio.create_task(kill_runtime())

        try:
            with pytest.raises(RuntimeTerminated) as async_exc:
                await runtime.eval_async("while (true) {}")

            assert runtime.is_closed()
            assert "host request" in str(async_exc.value).lower()

            with pytest.raises(RuntimeTerminated) as sync_exc:
                runtime.eval("1 + 1")
            assert "host request" in str(sync_exc.value).lower()
        finally:
            await killer
            runtime.close()

    @pytest.mark.asyncio
    async def test_runtime_terminate_concurrent_calls(self):
        runtime = Runtime()

        async def terminate_later(delay: float) -> None:
            await asyncio.sleep(delay)
            runtime.terminate()

        async def trigger():
            await asyncio.gather(
                terminate_later(0.05),
                terminate_later(0.06),
                terminate_later(0.07),
            )

        killer = asyncio.create_task(trigger())

        try:
            with pytest.raises(RuntimeTerminated):
                await runtime.eval_async("while (true) {}")

            await killer
            assert runtime.is_closed()
        finally:
            runtime.close()

    def test_runtime_terminate_idle(self):
        runtime = Runtime()
        try:
            runtime.terminate()
            assert runtime.is_closed()

            with pytest.raises(RuntimeTerminated) as exc_info:
                runtime.eval("2 + 2")
            assert "host request" in str(exc_info.value).lower()

            runtime.terminate()  # Idempotent
        finally:
            runtime.close()


class TestRuntimeHeapLimits:
    """Ensure configured heap limits terminate runtimes before V8 aborts."""

    def test_sync_eval_triggers_heap_termination(self):
        config = RuntimeConfig(
            max_heap_size=8 * 1024 * 1024, initial_heap_size=2 * 1024 * 1024
        )
        runtime = Runtime(config)
        try:
            with pytest.raises(RuntimeTerminated) as exc_info:
                runtime.eval(
                    "const allocations = []; while (true) { allocations.push(new Uint8Array(256 * 1024)); }"
                )
            assert runtime.is_closed()
            assert "heap limit" in str(exc_info.value).lower()
        finally:
            runtime.close()

    @pytest.mark.asyncio
    async def test_async_eval_triggers_heap_termination(self):
        config = RuntimeConfig(
            max_heap_size=8 * 1024 * 1024, initial_heap_size=2 * 1024 * 1024
        )
        runtime = Runtime(config)
        try:
            with pytest.raises(RuntimeTerminated) as exc_info:
                await runtime.eval_async(
                    "const allocations = []; while (true) { allocations.push(new Uint8Array(256 * 1024)); }"
                )
            assert runtime.is_closed()
            assert "heap limit" in str(exc_info.value).lower()
        finally:
            runtime.close()


class TestRuntimeConcurrent:
    """Tests for multiple concurrent runtimes."""

    def test_multiple_runtimes_independent(self):
        """Test that multiple runtimes have independent state."""
        runtime1 = Runtime()
        runtime2 = Runtime()

        try:
            runtime1.eval("var x = 'runtime1';")
            runtime2.eval("var x = 'runtime2';")

            result1 = runtime1.eval("x")
            result2 = runtime2.eval("x")

            assert result1 == "runtime1"
            assert result2 == "runtime2"
        finally:
            runtime1.close()
            runtime2.close()

    def test_sequential_runtime_creation(self):
        """Test creating multiple runtimes sequentially."""
        for i in range(3):
            runtime = Runtime()
            try:
                result = runtime.eval(f"{i} * 2")
                assert result == i * 2
            finally:
                runtime.close()


class TestRuntimeConversions:
    """Tests covering rich value conversions between Python and JavaScript."""

    def test_uint8array_to_bytes(self):
        with Runtime() as runtime:
            result = runtime.eval("new Uint8Array([1, 2, 3, 4])")
            assert isinstance(result, bytes)
            assert result == b"\x01\x02\x03\x04"

    def test_bytes_round_trip(self):
        runtime = Runtime()
        try:
            echo = runtime.eval("(input) => input")
            result = echo(b"\x00\xff")
            assert isinstance(result, bytes)
            assert result == b"\x00\xff"
        finally:
            runtime.close()

    def test_js_date_to_python_datetime(self):
        with Runtime() as runtime:
            result = runtime.eval("new Date(1704067200000)")
            expected = datetime.fromtimestamp(1704067200, tz=timezone.utc)
            assert isinstance(result, datetime)
            assert result == expected
            assert result.tzinfo == timezone.utc

    def test_python_datetime_to_js_date(self):
        runtime = Runtime()
        try:
            millis = runtime.eval("(value) => value.getTime()")
            dt = datetime(2024, 1, 1, 12, 0, tzinfo=timezone.utc)
            result = millis(dt)
            assert result == int(dt.timestamp() * 1000)
        finally:
            runtime.close()

    def test_js_set_to_python_set(self):
        with Runtime() as runtime:
            result = runtime.eval("new Set([1, 2, 3, 4])")
            assert isinstance(result, set)
            assert result == {1, 2, 3, 4}

    def test_python_set_to_js_set(self):
        runtime = Runtime()
        try:
            to_array = runtime.eval("(value) => Array.from(value)")
            result = to_array({3, 1, 2})
            assert isinstance(result, list)
            assert set(result) == {1, 2, 3}
        finally:
            runtime.close()

    def test_js_undefined_singleton(self):
        with Runtime() as runtime:
            result = runtime.eval("undefined")
            assert isinstance(result, JsUndefined)
            assert result is undefined
            assert bool(result) is False

    def test_python_undefined_to_js(self):
        runtime = Runtime()
        try:
            check = runtime.eval("(value) => value === undefined")
            result = check(undefined)
            assert result is True
        finally:
            runtime.close()

    def test_js_bigint_to_python_int(self):
        with Runtime() as runtime:
            result = runtime.eval("2n ** 200n")
            assert isinstance(result, int)
            assert result == 2**200

    def test_python_int_to_js_bigint(self):
        runtime = Runtime()
        try:
            check = runtime.eval(
                "(value) => (typeof value === 'bigint') && value === 2n ** 200n"
            )
            result = check(2**200)
            assert result is True
        finally:
            runtime.close()

    def test_python_to_js_respects_serialization_bytes_limit(self):
        config = RuntimeConfig(max_serialization_bytes=32)
        with Runtime(config) as runtime:
            identity = runtime.eval("(value) => value")
            payload = "abcdefghij" * 4  # 40 bytes
            with pytest.raises(
                RuntimeError, match=r"String size limit exceeded: \d+ > \d+"
            ):
                identity(payload)

    def test_python_to_js_respects_serialization_depth_limit(self):
        config = RuntimeConfig(max_serialization_depth=1)
        with Runtime(config) as runtime:
            identity = runtime.eval("(value) => value")
            nested = [[1]]
            with pytest.raises(
                RuntimeError, match=r"Depth exceeded maximum limit of \d+"
            ):
                identity(nested)

    def test_js_to_python_respects_serialization_bytes_limit(self):
        config = RuntimeConfig(max_serialization_bytes=32)
        with Runtime(config) as runtime:
            with pytest.raises(
                RuntimeError,
                match=r"Size \(\d+ bytes\) exceeded maximum limit of \d+ bytes",
            ):
                runtime.eval("'x'.repeat(64)")

    def test_js_to_python_respects_serialization_depth_limit(self):
        config = RuntimeConfig(max_serialization_depth=2)
        with Runtime(config) as runtime:
            with pytest.raises(
                RuntimeError, match=r"Depth exceeded maximum limit of \d+"
            ):
                runtime.eval("({a: {b: {c: 1}}})")


class TestRuntimeBindings:
    """Tests for the decorator-style binding helper."""

    def test_bind_decorator_uses_function_name(self):
        runtime = Runtime()
        try:

            @runtime.bind
            def add(a, b):
                return a + b

            assert runtime.eval("add(2, 3)") == 5
        finally:
            runtime.close()

    def test_bind_decorator_custom_name(self):
        runtime = Runtime()
        try:

            @runtime.bind(name="pyAdd")
            def add(a, b):
                return a + b

            assert runtime.eval("pyAdd(4, 6)") == 10
        finally:
            runtime.close()

    def test_bind_direct_call_returns_callable(self):
        runtime = Runtime()
        try:

            def multiply(a, b):
                return a * b

            bound = runtime.bind(multiply)
            assert bound is multiply
            assert runtime.eval("multiply(3, 4)") == 12
        finally:
            runtime.close()

    @pytest.mark.asyncio
    async def test_bind_decorator_supports_async_callables(self):
        runtime = Runtime()
        try:
            calls = []

            @runtime.bind
            async def add_async(a, b):
                calls.append((a, b))
                return a + b

            result = await runtime.eval_async("add_async(5, 7)")
            assert result == 12
            assert calls == [(5, 7)]
        finally:
            runtime.close()

    @pytest.mark.asyncio
    async def test_bind_direct_call_supports_async_callables(self):
        runtime = Runtime()
        try:

            async def multiply_async(a, b):
                return a * b

            bound = runtime.bind(multiply_async, name="pyMultiplyAsync")
            assert bound is multiply_async

            result = await runtime.eval_async("pyMultiplyAsync(3, 9)")
            assert result == 27
        finally:
            runtime.close()


class TestRuntimeEdgeCases:
    """Edge case tests for runtime behavior."""

    def test_eval_undefined(self):
        """Test evaluation of undefined."""
        runtime = Runtime()
        try:
            result = runtime.eval("undefined")
            assert result is undefined
        finally:
            runtime.close()

    def test_eval_null(self):
        """Test evaluation of null."""
        runtime = Runtime()
        try:
            result = runtime.eval("null")
            assert result is None
        finally:
            runtime.close()

    def test_eval_boolean(self):
        """Test evaluation of booleans."""
        runtime = Runtime()
        try:
            assert runtime.eval("true") is True
            assert runtime.eval("false") is False
        finally:
            runtime.close()

    def test_eval_array(self):
        """Test evaluation of arrays."""
        runtime = Runtime()
        try:
            result = runtime.eval("[1, 2, 3]")
            # Should return native Python list
            assert result == [1, 2, 3]
        finally:
            runtime.close()

    def test_eval_object(self):
        """Test evaluation of objects."""
        runtime = Runtime()
        try:
            result = runtime.eval("({a: 1, b: 2})")
            # Should return native Python dict
            assert result == {"a": 1, "b": 2}
        finally:
            runtime.close()

    def test_eval_empty_string(self):
        """Test evaluation of empty statement."""
        runtime = Runtime()
        try:
            result = runtime.eval("")
            # Empty script returns undefined sentinel
            assert result is undefined
        finally:
            runtime.close()


class TestRuntimeAsync:
    """Test async evaluation with promises."""

    @pytest.mark.asyncio
    async def test_eval_async_without_promise(self):
        """Test async eval with non-promise value."""
        with Runtime() as runtime:
            result = await runtime.eval_async("10 + 20")
            assert result == 30

    @pytest.mark.asyncio
    async def test_eval_async_with_promise_resolved(self):
        """Test async eval with resolved promise."""
        with Runtime() as runtime:
            result = await runtime.eval_async("Promise.resolve(42)")
            assert result == 42

    @pytest.mark.asyncio
    async def test_eval_async_with_promise_string(self):
        """Test async eval with promise resolving to string."""
        with Runtime() as runtime:
            result = await runtime.eval_async("Promise.resolve('async result')")
            assert result == "async result"

    @pytest.mark.asyncio
    async def test_eval_async_with_deferred_resolution(self):
        """Test async eval with deferred promise resolution."""
        with Runtime() as runtime:
            code = """
                new Promise((resolve) => {
                    // Immediately queued microtask via then
                    Promise.resolve().then(() => resolve('deferred result'));
                })
            """
            result = await runtime.eval_async(code)
            assert result == "deferred result"

    @pytest.mark.asyncio
    async def test_eval_async_promise_chain(self):
        """Test async eval with promise chain."""
        with Runtime() as runtime:
            code = """
                Promise.resolve(10)
                    .then(x => x * 2)
                    .then(x => x + 5)
            """
            result = await runtime.eval_async(code)
            assert result == 25

    @pytest.mark.asyncio
    async def test_eval_async_returns_js_function(self):
        """Ensure eval_async can round-trip JavaScript functions safely."""
        with Runtime() as runtime:
            js_func = await runtime.eval_async("Promise.resolve((value) => value + 5)")
            assert isinstance(js_func, JsFunction)
            assert js_func(37) == 42

    @pytest.mark.asyncio
    async def test_eval_async_promise_rejection(self):
        """Test async eval with rejected promise."""
        with Runtime() as runtime:
            with pytest.raises(JavaScriptError) as exc_info:
                await runtime.eval_async("Promise.reject(new Error('test error'))")
            js_error = exc_info.value
            assert js_error.name == "Error"
            assert "test error" in js_error.message
            assert js_error.stack is not None
            assert "test error" in js_error.stack

    @pytest.mark.asyncio
    async def test_eval_async_promise_rejection_value(self):
        """Test async eval with rejected promise (non-Error value)."""
        with Runtime() as runtime:
            with pytest.raises(JavaScriptError) as exc_info:
                await runtime.eval_async("Promise.reject('custom error')")
            js_error = exc_info.value
            assert js_error.name is None or js_error.name == "Error"
            assert "custom error" in js_error.message
            if js_error.stack:
                assert "custom error" in js_error.stack


class TestRuntimeTimeout:
    """Test timeout functionality."""

    def test_eval_sync_timeout(self):
        """Sync eval should respect RuntimeConfig timeout."""
        config = RuntimeConfig(timeout=0.05)
        with Runtime(config) as runtime:
            with pytest.raises(RuntimeError) as exc_info:
                runtime.eval("while (true) {}")
            assert "timed out" in str(exc_info.value)

    def test_eval_sync_completes_before_timeout(self):
        """Operations that complete quickly should not time out."""
        config = RuntimeConfig(timeout=1.0)
        with Runtime(config) as runtime:
            result = runtime.eval("2 + 2")
            assert result == 4

    def test_eval_sync_error_before_timeout(self):
        """Errors should be preserved when they occur before timeout."""
        config = RuntimeConfig(timeout=1.0)
        with Runtime(config) as runtime:
            with pytest.raises(JavaScriptError) as exc_info:
                runtime.eval("throw new Error('immediate error')")
            assert "immediate error" in str(exc_info.value)
            assert "timed out" not in str(exc_info.value)

    def test_concurrent_sync_operations_different_timeouts(self):
        """Multiple runtimes with different timeouts should work concurrently."""
        import threading

        results = {}
        errors = {}

        def run_with_timeout(name: str, timeout: float, code: str):
            try:
                config = RuntimeConfig(timeout=timeout)
                with Runtime(config) as rt:
                    results[name] = rt.eval(code)
            except Exception as e:
                errors[name] = str(e)

        # Fast operation with short timeout
        t1 = threading.Thread(target=run_with_timeout, args=("fast", 1.0, "42"))
        # Slow operation with long timeout
        t2 = threading.Thread(
            target=run_with_timeout,
            args=(
                "slow",
                1.0,
                "let sum = 0; for(let i = 0; i < 10000000; i++) sum += i; sum",
            ),
        )
        # Infinite loop with short timeout
        t3 = threading.Thread(
            target=run_with_timeout, args=("timeout", 0.05, "while(true) {}")
        )

        t1.start()
        t2.start()
        t3.start()
        t1.join()
        t2.join()
        t3.join()

        assert results.get("fast") == 42
        assert "slow" in results or "slow" in errors  # May complete or timeout
        assert "timeout" in errors
        assert "timed out" in errors["timeout"]

    @pytest.mark.asyncio
    async def test_eval_async_timeout_success(self):
        """Test that fast promises complete before timeout."""
        with Runtime() as runtime:
            result = await runtime.eval_async("Promise.resolve('quick')", timeout=1.0)
            assert result == "quick"

    @pytest.mark.asyncio
    async def test_eval_async_timeout_expiry(self):
        """Test that timeout is enforced for slow operations."""
        with Runtime() as runtime:
            # Promise that never resolves
            code = "new Promise(() => {})"
            with pytest.raises(RuntimeError) as exc_info:
                await runtime.eval_async(code, timeout=0.1)
            message = str(exc_info.value)
            assert "Evaluation failed" in message
            assert "pending" in message

    @pytest.mark.asyncio
    async def test_eval_async_no_timeout(self):
        """Test async eval without timeout."""
        with Runtime() as runtime:
            # Should complete without timeout
            result = await runtime.eval_async("Promise.resolve(123)")
            assert result == 123

    @pytest.mark.asyncio
    async def test_eval_async_timeout_with_promise_chain(self):
        """Test timeout with promise chains that complete in time."""
        with Runtime() as runtime:
            code = """
                Promise.resolve()
                    .then(() => Promise.resolve())
                    .then(() => 'nested')
            """
            result = await runtime.eval_async(code, timeout=1.0)
            assert result == "nested"

    @pytest.mark.asyncio
    async def test_eval_async_timeout_with_timedelta(self):
        """Test timeout with datetime.timedelta."""
        from datetime import timedelta

        with Runtime() as runtime:
            result = await runtime.eval_async(
                "Promise.resolve('timedelta')", timeout=timedelta(seconds=1)
            )
            assert result == "timedelta"

    @pytest.mark.asyncio
    async def test_eval_async_timeout_with_int_seconds(self):
        """Test timeout with integer seconds."""
        with Runtime() as runtime:
            result = await runtime.eval_async("Promise.resolve('int')", timeout=1)
            assert result == "int"


class TestTimeoutValidation:
    """Unified tests for timeout parameter validation across all async methods."""

    @pytest.mark.parametrize(
        "timeout_value,expected_error",
        [
            (-1.0, "Timeout cannot be negative"),
            (-5, "Timeout cannot be negative"),
            (timedelta(seconds=-1), "Timeout cannot be negative"),
            (0, "Timeout cannot be zero"),
            (0.0, "Timeout cannot be zero"),
            (timedelta(seconds=0), "Timeout cannot be zero"),
            (float("nan"), "Timeout must be finite"),
            (float("inf"), "Timeout must be finite"),
            (float("-inf"), "Timeout must be finite"),
            ("invalid", "Timeout must be a number"),
            ([1, 2, 3], "Timeout must be a number"),
            ({"timeout": 1}, "Timeout must be a number"),
        ],
    )
    @pytest.mark.asyncio
    async def test_eval_async_timeout_validation(self, timeout_value, expected_error):
        """Test eval_async rejects invalid timeout values."""
        with Runtime() as runtime:
            with pytest.raises(ValueError, match=expected_error):
                await runtime.eval_async("Promise.resolve(1)", timeout=timeout_value)

    @pytest.mark.parametrize(
        "timeout_value,expected_error",
        [
            (-1.0, "Timeout cannot be negative"),
            (-5, "Timeout cannot be negative"),
            (timedelta(seconds=-1), "Timeout cannot be negative"),
            (0, "Timeout cannot be zero"),
            (0.0, "Timeout cannot be zero"),
            (timedelta(seconds=0), "Timeout cannot be zero"),
            (float("nan"), "Timeout must be finite"),
            (float("inf"), "Timeout must be finite"),
            (float("-inf"), "Timeout must be finite"),
            ("invalid", "Timeout must be a number"),
            ([1, 2, 3], "Timeout must be a number"),
            ({"timeout": 1}, "Timeout must be a number"),
        ],
    )
    @pytest.mark.asyncio
    async def test_eval_module_async_timeout_validation(
        self, timeout_value, expected_error
    ):
        """Test eval_module_async rejects invalid timeout values."""
        with Runtime() as runtime:
            runtime.add_static_module("test", "export const value = 42;")
            with pytest.raises(ValueError, match=expected_error):
                await runtime.eval_module_async("test", timeout=timeout_value)

    @pytest.mark.parametrize(
        "timeout_value,expected_error",
        [
            (-1.0, "Timeout cannot be negative"),
            (-5, "Timeout cannot be negative"),
            (timedelta(seconds=-1), "Timeout cannot be negative"),
            (0, "Timeout cannot be zero"),
            (0.0, "Timeout cannot be zero"),
            (timedelta(seconds=0), "Timeout cannot be zero"),
            (float("nan"), "Timeout must be finite"),
            (float("inf"), "Timeout must be finite"),
            (float("-inf"), "Timeout must be finite"),
            ("invalid", "Timeout must be a number"),
            ([1, 2, 3], "Timeout must be a number"),
            ({"timeout": 1}, "Timeout must be a number"),
        ],
    )
    @pytest.mark.asyncio
    async def test_js_function_call_timeout_validation(
        self, timeout_value, expected_error
    ):
        """Test JsFunction.__call__ rejects invalid timeout values."""
        with Runtime() as runtime:
            fn = runtime.eval("(x) => x * 2")
            with pytest.raises(ValueError, match=expected_error):
                await fn(21, timeout=timeout_value)


class TestRuntimeAsyncConcurrency:
    @pytest.mark.asyncio
    async def test_multiple_async_evals_sequential(self):
        """Test sequential async evaluations."""
        with Runtime() as runtime:
            result1 = await runtime.eval_async("Promise.resolve(1)")
            result2 = await runtime.eval_async("Promise.resolve(2)")
            result3 = await runtime.eval_async("Promise.resolve(3)")

            assert result1 == 1
            assert result2 == 2
            assert result3 == 3

    @pytest.mark.asyncio
    async def test_multiple_runtimes_async(self):
        """Test multiple runtimes with async operations."""
        runtime1 = Runtime()
        runtime2 = Runtime()

        try:
            # Run async evals concurrently on different runtimes
            result1 = await runtime1.eval_async("Promise.resolve('runtime1')")
            result2 = await runtime2.eval_async("Promise.resolve('runtime2')")

            assert result1 == "runtime1"
            assert result2 == "runtime2"
        finally:
            runtime1.close()
            runtime2.close()

    @pytest.mark.asyncio
    async def test_mixed_sync_async_eval(self):
        """Test mixing sync and async eval calls."""
        with Runtime() as runtime:
            # Sync eval
            sync_result = runtime.eval("10 + 10")
            assert sync_result == 20

            # Async eval
            async_result = await runtime.eval_async("Promise.resolve(30)")
            assert async_result == 30

            # Another sync eval to verify state
            final_result = runtime.eval("5 * 5")
            assert final_result == 25

    @pytest.mark.asyncio
    async def test_async_state_persistence(self):
        """Test that state persists across async evaluations."""
        with Runtime() as runtime:
            # Set up state
            runtime.eval("var asyncCounter = 0;")

            # Multiple async evals modifying state
            await runtime.eval_async("Promise.resolve(++asyncCounter)")
            result = await runtime.eval_async("Promise.resolve(asyncCounter)")
            assert result == 1

            await runtime.eval_async("Promise.resolve(++asyncCounter)")
            result = runtime.eval("asyncCounter")
            assert result == 2


class TestRuntimeAsyncErrors:
    """Test error handling in async evaluation."""

    @pytest.mark.asyncio
    async def test_eval_async_syntax_error(self):
        """Test that syntax errors are properly reported in async eval."""
        with Runtime() as runtime:
            with pytest.raises(JavaScriptError) as exc_info:
                await runtime.eval_async("this is not valid javascript {{{")
            js_error = exc_info.value
            assert js_error.name == "SyntaxError"
            assert js_error.stack is not None

    @pytest.mark.asyncio
    async def test_eval_async_runtime_error(self):
        """Test that runtime errors in promises are caught."""
        with Runtime() as runtime:
            code = """
                new Promise((resolve, reject) => {
                    reject(new Error('Runtime error in promise'));
                })
            """
            with pytest.raises(JavaScriptError) as exc_info:
                await runtime.eval_async(code)
            js_error = exc_info.value
            assert js_error.name == "Error"
            assert "Runtime error in promise" in js_error.message
            assert js_error.stack is not None

    @pytest.mark.asyncio
    async def test_eval_async_after_close(self):
        """Test that async eval after close raises an error."""
        runtime = Runtime()
        runtime.close()

        with pytest.raises(RuntimeError, match="closed"):
            await runtime.eval_async("Promise.resolve(1)")


class TestRuntimeConfig:
    """Tests for RuntimeConfig functionality."""

    def test_runtime_config_default(self):
        """Test default RuntimeConfig creation."""
        from jsrun import RuntimeConfig

        config = RuntimeConfig()
        assert config is not None
        # Should have default values
        assert "RuntimeConfig" in repr(config)

    def test_runtime_config_constructor_with_kwargs(self):
        """Test RuntimeConfig constructor with keyword arguments."""
        from jsrun import RuntimeConfig

        config = RuntimeConfig(
            max_heap_size=100 * 1024 * 1024,
            initial_heap_size=50 * 1024 * 1024,
            bootstrap='console.log("Bootstrap executed")',
            timeout=30.0,
        )

        assert config is not None
        assert "RuntimeConfig" in repr(config)
        assert config.max_heap_size == 100 * 1024 * 1024
        assert config.initial_heap_size == 50 * 1024 * 1024
        assert config.bootstrap == 'console.log("Bootstrap executed")'
        assert config.timeout == 30.0

    def test_runtime_config_property_methods(self):
        """Test RuntimeConfig with property setters."""
        from jsrun import RuntimeConfig

        config = RuntimeConfig()
        config.max_heap_size = 100 * 1024 * 1024
        config.initial_heap_size = 50 * 1024 * 1024
        config.bootstrap = 'console.log("Bootstrap executed")'
        config.timeout = 30.0

        assert config is not None
        assert "RuntimeConfig" in repr(config)
        assert config.max_heap_size == 100 * 1024 * 1024
        assert config.initial_heap_size == 50 * 1024 * 1024
        assert config.bootstrap == 'console.log("Bootstrap executed")'
        assert config.timeout == 30.0

    def test_runtime_config_serialization_kwargs(self):
        """RuntimeConfig should accept serialization limits via kwargs."""
        from jsrun import RuntimeConfig

        config = RuntimeConfig(
            max_serialization_depth=200,
            max_serialization_bytes=2048,
        )

        assert config.max_serialization_depth == 200
        assert config.max_serialization_bytes == 2048

    def test_runtime_config_serialization_setters(self):
        """Serialization limit setters should validate values."""
        from jsrun import RuntimeConfig

        config = RuntimeConfig()
        config.max_serialization_depth = 128
        config.max_serialization_bytes = 4096

        assert config.max_serialization_depth == 128
        assert config.max_serialization_bytes == 4096

        with pytest.raises(ValueError, match="positive integer"):
            config.max_serialization_depth = 0
        with pytest.raises(ValueError, match="positive integer"):
            config.max_serialization_bytes = 0

    def test_runtime_config_with_bootstrap(self):
        """Test Runtime with bootstrap script."""
        from jsrun import Runtime, RuntimeConfig

        config = RuntimeConfig(bootstrap="globalThis.bootstrapped = true;")

        with Runtime(config) as runtime:
            result = runtime.eval("bootstrapped")
            assert result is True

    def test_runtime_config_without_bootstrap(self):
        """Test Runtime without bootstrap script (should not have bootstrapped variable)."""
        from jsrun import Runtime, RuntimeConfig

        config = RuntimeConfig()  # No bootstrap

        with Runtime(config) as runtime:
            # Should not have bootstrapped variable
            result = runtime.eval("typeof bootstrapped")
            assert result == "undefined"

    def test_runtime_config_initial_requires_max(self):
        """Providing initial heap size without max should raise on runtime creation."""
        from jsrun import Runtime, RuntimeConfig

        config = RuntimeConfig(initial_heap_size=1 * 1024 * 1024)
        with pytest.raises(RuntimeError) as exc_info:
            Runtime(config)
        message = str(exc_info.value).lower()
        assert "initial_heap_size" in message and "max_heap_size" in message

    def test_runtime_config_initial_exceeds_max(self):
        """initial_heap_size greater than max_heap_size should raise."""
        from jsrun import Runtime, RuntimeConfig

        config = RuntimeConfig(
            initial_heap_size=8 * 1024 * 1024, max_heap_size=4 * 1024 * 1024
        )
        with pytest.raises(RuntimeError) as exc_info:
            Runtime(config)
        message = str(exc_info.value).lower()
        assert "cannot exceed" in message

    def test_runtime_config_timeout_formats(self):
        """Test RuntimeConfig timeout with different formats."""
        from jsrun import RuntimeConfig

        # Test float timeout
        config1 = RuntimeConfig(timeout=30.5)
        assert config1 is not None
        assert config1.timeout == 30.5

        # Test integer timeout
        config2 = RuntimeConfig(timeout=60)
        assert config2 is not None
        assert config2.timeout == 60.0

        # Test timedelta timeout
        import datetime

        config3 = RuntimeConfig(timeout=datetime.timedelta(seconds=45))
        assert config3 is not None
        assert config3.timeout == 45.0

    def test_runtime_config_invalid_timeout(self):
        """Test RuntimeConfig with invalid timeout."""
        from jsrun import RuntimeConfig

        config = RuntimeConfig()

        with pytest.raises(ValueError, match="Timeout must be a number"):
            config.timeout = "invalid"

    @pytest.mark.parametrize(
        "timeout_value,expected_error",
        [
            (-1.0, "Timeout cannot be negative"),
            (-5, "Timeout cannot be negative"),
            (timedelta(seconds=-1), "Timeout cannot be negative"),
            (0, "Timeout cannot be zero"),
            (0.0, "Timeout cannot be zero"),
            (timedelta(seconds=0), "Timeout cannot be zero"),
            (float("nan"), "Timeout must be finite"),
            (float("inf"), "Timeout must be finite"),
            (float("-inf"), "Timeout must be finite"),
            ("invalid", "Timeout must be a number"),
            ([1, 2, 3], "Timeout must be a number"),
            ({"timeout": 1}, "Timeout must be a number"),
        ],
    )
    def test_runtime_config_timeout_validation_constructor(
        self, timeout_value, expected_error
    ):
        """Test RuntimeConfig constructor rejects invalid timeout values."""
        from jsrun import RuntimeConfig

        with pytest.raises(ValueError, match=expected_error):
            RuntimeConfig(timeout=timeout_value)

    @pytest.mark.parametrize(
        "timeout_value,expected_error",
        [
            (-1.0, "Timeout cannot be negative"),
            (-5, "Timeout cannot be negative"),
            (timedelta(seconds=-1), "Timeout cannot be negative"),
            (0, "Timeout cannot be zero"),
            (0.0, "Timeout cannot be zero"),
            (timedelta(seconds=0), "Timeout cannot be zero"),
            (float("nan"), "Timeout must be finite"),
            (float("inf"), "Timeout must be finite"),
            (float("-inf"), "Timeout must be finite"),
            ("invalid", "Timeout must be a number"),
            ([1, 2, 3], "Timeout must be a number"),
            ({"timeout": 1}, "Timeout must be a number"),
        ],
    )
    def test_runtime_config_timeout_validation_setter(
        self, timeout_value, expected_error
    ):
        """Test RuntimeConfig timeout setter rejects invalid timeout values."""
        from jsrun import RuntimeConfig

        config = RuntimeConfig()
        with pytest.raises(ValueError, match=expected_error):
            config.timeout = timeout_value

    @pytest.mark.asyncio
    async def test_runtime_config_execution_timeout_applies_to_eval_async(self):
        """RuntimeConfig.timeout should act as default timeout for eval_async."""
        from jsrun import Runtime, RuntimeConfig

        config = RuntimeConfig(timeout=0.1)
        with Runtime(config) as runtime:
            start = time.monotonic()
            with pytest.raises(RuntimeError) as exc_info:
                await runtime.eval_async("new Promise(() => {})")
            elapsed = time.monotonic() - start
            assert elapsed < 1.0
            message = str(exc_info.value).lower()
            assert "timed out" in message or "pending" in message

    @pytest.mark.asyncio
    async def test_eval_async_timeout_interrupts_sync_loop(self):
        """Explicit eval_async timeout should terminate blocking JavaScript loops."""
        from jsrun import Runtime

        with Runtime() as runtime:
            start = time.monotonic()
            with pytest.raises(RuntimeError) as exc_info:
                await runtime.eval_async("while (true) {}", timeout=0.1)
            elapsed = time.monotonic() - start
            assert elapsed < 1.0
            assert "timed out" in str(exc_info.value).lower()

    @pytest.mark.asyncio
    async def test_eval_module_async_timeout_interrupts_sync_loop(self):
        """Explicit eval_module_async timeout should terminate blocking module execution."""
        from jsrun import Runtime

        with Runtime() as runtime:
            runtime.add_static_module("loop", "while (true) {}")
            start = time.monotonic()
            with pytest.raises(RuntimeError) as exc_info:
                await runtime.eval_module_async("static:loop", timeout=0.1)
            elapsed = time.monotonic() - start
            assert elapsed < 1.0
            assert "timed out" in str(exc_info.value).lower()

    def test_runtime_with_config(self):
        """Test Runtime creation with RuntimeConfig."""
        from jsrun import Runtime, RuntimeConfig

        config = RuntimeConfig(bootstrap="globalThis.configured = true;")

        with Runtime(config) as runtime:
            result = runtime.eval("configured")
            assert result is True

    def test_runtime_without_config(self):
        """Test Runtime creation without RuntimeConfig (default behavior)."""
        from jsrun import Runtime

        # Should work the same as before
        with Runtime() as runtime:
            result = runtime.eval("2 + 2")
            assert result == 4

    def test_runtime_config_property_chaining(self):
        """Test that RuntimeConfig property setters work correctly."""
        from jsrun import RuntimeConfig

        config = RuntimeConfig()
        config.max_heap_size = 100 * 1024 * 1024
        config.initial_heap_size = 50 * 1024 * 1024
        config.bootstrap = 'console.log("property chaining")'
        config.timeout = 30.0

        assert config is not None
        assert config.max_heap_size == 100 * 1024 * 1024
        assert config.initial_heap_size == 50 * 1024 * 1024
        assert config.bootstrap == 'console.log("property chaining")'
        assert config.timeout == 30.0

    def test_runtime_config_multiple_instances(self):
        """Test that multiple RuntimeConfig instances are independent."""
        from jsrun import Runtime, RuntimeConfig

        config1 = RuntimeConfig(bootstrap="globalThis.instance1 = true;")
        config2 = RuntimeConfig(bootstrap="globalThis.instance2 = true;")

        with Runtime(config1) as runtime1:
            result1 = runtime1.eval("instance1")
            assert result1 is True

            # instance2 should not be defined in runtime1
            result2 = runtime1.eval("typeof instance2")
            assert result2 == "undefined"

        with Runtime(config2) as runtime2:
            result1 = runtime2.eval("instance2")
            assert result1 is True

            # instance1 should not be defined in runtime2
            result2 = runtime2.eval("typeof instance1")
            assert result2 == "undefined"

    def test_runtime_config_console_disabled_by_default(self):
        """Test that console is disabled by default."""
        from jsrun import Runtime, RuntimeConfig

        config = RuntimeConfig()
        assert config.enable_console is False

        with Runtime(config) as runtime:
            # Console should either be undefined or a stub
            console_type = runtime.eval("typeof console")
            # When disabled, console is either deleted (undefined) or replaced with stub
            assert console_type in ("undefined", "object")

            # If console exists (stub), methods should be no-ops
            if console_type == "object":
                # Methods should exist but be no-ops
                runtime.eval("console.log('test')")  # Should not raise
                runtime.eval("console.error('test')")  # Should not raise
                runtime.eval("console.warn('test')")  # Should not raise

    def test_runtime_config_console_can_be_enabled(self):
        """Test that console can be explicitly enabled."""
        from jsrun import Runtime, RuntimeConfig

        config = RuntimeConfig(enable_console=True)
        assert config.enable_console is True

        with Runtime(config) as runtime:
            # Console should exist
            console_type = runtime.eval("typeof console")
            assert console_type == "object"

            # Console methods should be callable
            result = runtime.eval("typeof console.log")
            assert result == "function"

            # Test that calling console methods doesn't raise errors
            runtime.eval("console.log('test')")
            runtime.eval("console.error('test')")
            runtime.eval("console.warn('test')")

    def test_runtime_config_console_disabled(self):
        """Test that console can be disabled."""
        from jsrun import Runtime, RuntimeConfig

        config = RuntimeConfig(enable_console=False)
        assert config.enable_console is False

        with Runtime(config) as runtime:
            # Console should either be undefined or a stub
            console_type = runtime.eval("typeof console")
            # When disabled, console is either deleted (undefined) or replaced with stub
            assert console_type in ("undefined", "object")

            # If console exists (stub), methods should be no-ops
            if console_type == "object":
                # Methods should exist but be no-ops
                runtime.eval("console.log('test')")  # Should not raise
                runtime.eval("console.error('test')")  # Should not raise
                runtime.eval("console.warn('test')")  # Should not raise


class TestRuntimeNativeTypes:
    """Tests for native Python type return from eval."""

    @pytest.mark.parametrize(
        "source, expected",
        [
            ("1 + 1", 2),
            ("'hello world'", "hello world"),
            ("true", True),
            ("false", False),
            ("null", None),
            ("undefined", undefined),
            ("[1, 2, 3]", [1, 2, 3]),
            ("({foo: 'bar'})", {"foo": "bar"}),
            ("({nested: {value: 42}})", {"nested": {"value": 42}}),
            ("3.14", 3.14),
            ("-5", -5),
        ],
    )
    def test_eval_returns_native(self, source, expected):
        """Test that eval returns native Python types."""
        with Runtime() as runtime:
            result = runtime.eval(source)
            assert result == expected
            assert isinstance(result, type(expected))

    @pytest.mark.parametrize(
        "source, expected",
        [
            ("Promise.resolve(42)", 42),
            ("Promise.resolve('async result')", "async result"),
            ("Promise.resolve([1, 2, 3])", [1, 2, 3]),
            ("Promise.resolve({key: 'value'})", {"key": "value"}),
        ],
    )
    @pytest.mark.asyncio
    async def test_eval_async_returns_native(self, source, expected):
        """Test that eval_async returns native Python types."""
        with Runtime() as runtime:
            result = await runtime.eval_async(source)
            assert result == expected
            assert isinstance(result, type(expected))

    def test_eval_function_raises_error(self):
        """Test that functions raise appropriate errors."""
        with Runtime() as runtime:
            with pytest.raises(JavaScriptError) as exc_info:
                runtime.eval("function() { return 42; }")
            js_error = exc_info.value
            assert js_error.name == "SyntaxError"
            assert "function" in js_error.message.lower()

    def test_eval_circular_reference_raises_error(self):
        """Test that circular references raise appropriate errors."""
        with Runtime() as runtime:
            # Create circular object without returning it (by adding void 0 at the end)
            runtime.eval("var obj = {}; obj.self = obj; void 0")
            # Now try to eval the circular object - should raise error
            with pytest.raises(RuntimeError) as exc_info:
                runtime.eval("obj")
            assert (
                "circular" in str(exc_info.value).lower()
                or "depth" in str(exc_info.value).lower()
            )

    def test_eval_bigint_handles_large_values(self):
        """Test that BigInt values round-trip without overflow."""
        with Runtime() as runtime:
            result = runtime.eval(f"{2**63}n")
            assert isinstance(result, int)
            assert result == 2**63

    def test_eval_nan_infinity_survives(self):
        """Test that NaN and Infinity survive round-trip."""
        with Runtime() as runtime:
            nan_result = runtime.eval("NaN")
            assert math.isnan(nan_result)

            pos_inf_result = runtime.eval("Infinity")
            assert pos_inf_result == float("inf")

            neg_inf_result = runtime.eval("-Infinity")
            assert neg_inf_result == float("-inf")


class TestRuntimeOpsNativeTypes:
    """Test that ops can handle native JavaScript types including NaN/Infinity"""

    def test_sync_op_echo_nan(self):
        """Test that sync ops can receive and return NaN"""

        def echo(value):
            return value

        with Runtime() as runtime:
            runtime.register_op("echo", echo, mode="sync")
            result = runtime.eval("__host_op_sync__(0, NaN)")
            assert math.isnan(result), f"Expected NaN but got {result}"

    def test_sync_op_echo_infinity(self):
        """Test that sync ops can receive and return Infinity"""

        def echo(value):
            return value

        with Runtime() as runtime:
            runtime.register_op("echo", echo, mode="sync")
            result = runtime.eval("__host_op_sync__(0, Infinity)")
            assert result == float("inf"), f"Expected Infinity but got {result}"

    def test_sync_op_returns_nan_from_python(self):
        """Test that sync ops can return NaN from Python"""

        def return_nan():
            return float("nan")

        with Runtime() as runtime:
            runtime.register_op("returnNaN", return_nan, mode="sync")
            result = runtime.eval("__host_op_sync__(0)")
            assert math.isnan(result), f"Expected NaN but got {result}"

    def test_sync_op_python_circular_list_raises_error(self):
        """Python handlers returning circular lists should raise conversion errors"""

        def make_circular_list():
            data = []
            data.append(data)
            return data

        with Runtime() as runtime:
            runtime.register_op("makeCircularList", make_circular_list, mode="sync")
            with pytest.raises(JavaScriptError) as exc_info:
                runtime.eval("__host_op_sync__(0)")
            message = str(exc_info.value).lower()
            assert "circular" in message and "list" in message

    def test_sync_op_python_circular_dict_raises_error(self):
        """Python handlers returning circular dicts should raise conversion errors"""

        def make_circular_dict():
            data = {}
            data["self"] = data
            return data

        with Runtime() as runtime:
            runtime.register_op("makeCircularDict", make_circular_dict, mode="sync")
            with pytest.raises(JavaScriptError) as exc_info:
                runtime.eval("__host_op_sync__(0)")
            message = str(exc_info.value).lower()
            assert "circular" in message and "dict" in message

    def test_sync_op_python_large_payload_hits_size_limit(self):
        """Very large Python payloads should trigger size limit errors."""

        def make_large_list():
            # Approximately 12MB of data (6k strings x 2KB)
            return ["x" * 2048 for _ in range(6000)]

        with Runtime() as runtime:
            runtime.register_op("makeLargeList", make_large_list, mode="sync")
            with pytest.raises(JavaScriptError) as exc_info:
                runtime.eval("__host_op_sync__(0)")
            message = str(exc_info.value).lower()
            assert "size" in message or "limit" in message


class TestStreaming:
    @pytest.mark.asyncio
    async def test_js_stream_round_trip(self):
        runtime = Runtime()
        try:
            js_stream = await runtime.eval_async(
                """
                (async () => {
                    const stream = new ReadableStream({
                        start(controller) {
                            controller.enqueue("alpha");
                            controller.enqueue("beta");
                            controller.close();
                        }
                    });
                    return stream;
                })()
                """
            )
            assert isinstance(js_stream, JsStream)

            chunks = []
            async for chunk in js_stream:
                chunks.append(chunk)

            assert chunks == ["alpha", "beta"]
        finally:
            runtime.close()

    @pytest.mark.asyncio
    async def test_python_stream_consumed_in_js(self):
        runtime = Runtime()

        async def producer():
            yield "left"
            await asyncio.sleep(0)
            yield "right"

        try:
            py_stream = runtime.stream_from_async_iterable(producer())
            setter = runtime.eval("(stream) => { globalThis.py_stream = stream; }")
            setter(py_stream)

            result = await runtime.eval_async(
                """
                (async () => {
                    const reader = py_stream.getReader();
                    const parts = [];
                    while (true) {
                        const { done, value } = await reader.read();
                        if (done) break;
                        parts.push(value);
                    }
                    return parts.join("|");
                })()
                """
            )

            assert result == "left|right"
        finally:
            runtime.close()

    @pytest.mark.asyncio
    async def test_python_stream_error_rejects_js_reader(self):
        runtime = Runtime()

        async def faulty():
            yield "start"
            raise RuntimeError("py boom")

        try:
            py_stream = runtime.stream_from_async_iterable(faulty())
            setter = runtime.eval("(stream) => { globalThis.py_faulty = stream; }")
            setter(py_stream)

            with pytest.raises(JavaScriptError) as exc_info:
                await runtime.eval_async(
                    """
                    (async () => {
                        const reader = py_faulty.getReader();
                        await reader.read();
                        await reader.read();
                        return "never";
                    })()
                    """
                )
            assert "py boom" in str(exc_info.value)
        finally:
            runtime.close()

    @pytest.mark.asyncio
    async def test_python_stream_cancelled_from_js(self):
        runtime = Runtime()
        cancel_event = asyncio.Event()

        class CancelAware:
            def __aiter__(self):
                return self

            async def __anext__(self):
                await asyncio.sleep(0)
                return "chunk"

            async def aclose(self):
                cancel_event.set()

        try:
            py_stream = runtime.stream_from_async_iterable(CancelAware())
            setter = runtime.eval("(stream) => { globalThis.py_cancel = stream; }")
            setter(py_stream)

            await runtime.eval_async(
                """
                (async () => {
                    const reader = py_cancel.getReader();
                    await reader.read();
                    await reader.cancel("stop");
                })()
                """
            )

            await asyncio.wait_for(cancel_event.wait(), timeout=2)
        finally:
            runtime.close()

    @pytest.mark.asyncio
    async def test_js_stream_cancelled_from_python(self):
        runtime = Runtime()
        try:
            js_stream = runtime.eval(
                """
                (() => {
                    globalThis._js_stream_cancelled = false;
                    return new ReadableStream({
                        start(controller) {
                            controller.enqueue("one");
                            controller.enqueue("two");
                        },
                        cancel() {
                            globalThis._js_stream_cancelled = true;
                        }
                    });
                })()
                """
            )

            chunks = []
            async for chunk in js_stream:
                chunks.append(chunk)
                break

            js_stream.close()

            assert runtime.eval("_js_stream_cancelled") is True
        finally:
            runtime.close()

    @pytest.mark.asyncio
    async def test_js_stream_error_raises_in_python(self):
        runtime = Runtime()
        try:
            js_stream = runtime.eval(
                """
                new ReadableStream({
                    start(controller) {
                        controller.enqueue("first");
                        controller.error(new Error("boom"));
                    }
                })
                """
            )

            async def consume():
                async for _ in js_stream:
                    await asyncio.sleep(0)

            with pytest.raises(JavaScriptError) as exc_info:
                await consume()

            assert "boom" in str(exc_info.value)
        finally:
            runtime.close()


class TestSnapshots:
    def test_snapshot_builder_roundtrip(self):
        builder = SnapshotBuilder()
        builder.execute_script("setup.js", "globalThis.answer = 42;")
        snapshot_bytes = builder.build()

        config = RuntimeConfig(snapshot=snapshot_bytes)
        runtime = Runtime(config)
        try:
            assert runtime.eval("answer") == 42
        finally:
            runtime.close()

    def test_runtime_config_rejects_bootstrap_plus_snapshot(self):
        builder = SnapshotBuilder()
        builder.execute_script("setup.js", "globalThis.answer = 42;")
        snapshot_bytes = builder.build()

        with pytest.raises(ValueError):
            RuntimeConfig(snapshot=snapshot_bytes, bootstrap="console.log('nope');")
