# Changelog

All notable changes to Groggy will be documented in this file.

The format is based on [Keep a Changelog](https://keepachangelog.com/en/1.0.0/),
and this project adheres to [Semantic Versioning](https://semver.org/spec/v2.0.0.html).

## [0.5.1] - 2025-10-21

### 🎉 Release - Getting to a working state

This release represents a fundamental transformation of Groggy from a graph library into a comprehensive graph analytics platform with visualization, the foundation for neural networks, and graph<->table operations.

---

### ✨ Added

#### **Comprehensive Testing Infrastructure**
- Complete test suite in `tests/modules/` with 17 specialized test files
- **406 passing tests** covering all major objects (Graph, Table, Array, Matrix, Subgraph)
- Smart fixtures system with real-world datasets (Karate Club, social networks)
- Test coverage validation and reporting tools
- Integration tests for scientific computing workflows

#### **Python Type Stubs (.pyi) for IDE Support**
- 9,302-line comprehensive type stub file (`_groggy.pyi`)
- Full IDE autocomplete and type checking support across all APIs
- Automatic stub generator with return type inference
- Manual mappings for complex chaining patterns
- VSCode, PyCharm, and Jupyter integration

#### **Visualization System** 🎨
- Complete visualization module (`src/viz/`) with 25,000+ lines
- **Real-time streaming visualization engine** with WebSocket server
- **Multiple embedding algorithms**:
  - Spectral embeddings with eigenvalue decomposition
  - Force-directed layouts with energy minimization
  - Honeycomb 3D projections with quality metrics
  - Random and debug embeddings
- **Interactive web viewer** with pan, zoom, and globe navigation
- **CSS template system** with 5 professionally designed themes (dark, light, minimal, publication, sleek)
- Jupyter notebook integration with custom widgets
- Pandas-like `.viz` accessor for graph objects

#### **Advanced Matrix Operations** 🔢
- Unified matrix system with 3,262-line core implementation
- **Auto-differentiation engine** for neural networks (1,530 lines)
  - Forward and backward pass computation
  - Computational graph tracking
  - Gradient accumulation and optimization
- **80+ matrix operations**:
  - Linear algebra (transpose, inverse, decompositions)
  - Statistical operations (mean, std, correlation)
  - Advanced indexing and slicing
  - Broadcasting and reshaping
- **SIMD optimizations** for NumArray performance
- **Neural network operations**:
  - Activation functions (ReLU, Sigmoid, Tanh, Softmax, GELU)
  - Convolutional operations with padding and stride
  - Operation fusion for performance
- Scientific computing integration (NumPy, SciPy sparse matrices)

#### **Table System Overhaul** 📊
- **BaseTable** comprehensive implementation (6,438 lines)
  - DataFrame-like API with row/column operations
  - Advanced filtering and querying
  - Sorting, grouping, and aggregation
- **NodesTable** and **EdgesTable** specialized implementations
- **TableArray** for collections of tables with bulk operations
- **GroupBy operations** with advanced aggregation functions
- **File I/O**: CSV, JSON, Parquet read/write support
- **Pandas compatibility layer** for seamless integration
- Rich display with Unicode box-drawing characters

#### **Array System Enhancement** 📈
- **BaseArray** (2,240 lines) with lazy evaluation patterns
- **NumArray** with SIMD acceleration for numeric operations
- **BoolArray** with optimized boolean operations
- **ArrayArray** for nested array collections
- **Lazy iterators** for memory-efficient processing
- **Memory profiling** and benchmark infrastructure
- Statistical operations (mean, std, median, quantiles)
- Advanced slicing and boolean indexing

#### **Import/Export Functionality** 📦
- **Graph bundle system** for complete graph serialization
  - Save/load with metadata preservation
  - Version compatibility checking
  - Validation and integrity reporting
- **NetworkX compatibility layer** for ecosystem integration
- **Import functions** for common graph formats
- **Export to scientific formats**: NumPy arrays, SciPy sparse, Pandas DataFrames
- CSV/JSON/Parquet support for nodes and edges tables

#### **Meta-API Discovery System** 🔍
- Complete API introspection and automated documentation
- **Meta-graph of API structure** (7,130-line JSON)
- Automatic test generator from discovered API signatures
- Interactive API visualization with HTML output
- Parameter type enhancement and validation
- API usage examples generator

#### **Neural Network Foundation** 🧠
- **Activation functions**: ReLU, LeakyReLU, Sigmoid, Tanh, Softmax, GELU, Swish
- **Auto-differentiation engine** with computational graph
- **Gradient computation** with backward pass
- **Convolutional operations** with padding and stride configuration
- **Operation fusion** for performance optimization
- Foundation for graph neural networks

#### **Display System Enhancements** ✨
- Unicode box-drawing for professional table display
- Smart truncation for large datasets with `…` indicators
- Multiple display themes (dark, light, minimal, publication, sleek)
- Rich `__repr__` for Jupyter notebooks
- Type annotations in column headers
- Summary statistics footer

#### **Performance Optimizations** ⚡
- SIMD operations for array processing (2-10x speedup)
- Memory pooling and efficient allocation
- Lazy evaluation throughout the stack
- Benchmark infrastructure with regression detection
- Performance monitoring CI/CD pipeline
- Profile-guided optimization support

---

### 🏗️ Changed

#### **Architecture Reorganization**
- FFI layer completely modularized:
  - `storage/` - Arrays, tables, matrices, accessors
  - `query/` - Query parsing and execution
  - `subgraphs/` - Subgraph implementations
  - `entities/` - Node and edge wrappers
  - `delegation/` - Delegation patterns and traits
- Core library reorganized:
  - `storage/` - All data structures
  - `state/` - Graph state management
  - `query/` - Query engine
  - `subgraphs/` - Subgraph operations
  - `viz/` - Visualization engine
  - `utils/` - Shared utilities

#### **API Improvements**
- Consistent return types across chaining operations
- Better error messages with context
- Improved type hints and documentation
- Delegation pattern for method forwarding
- Cleaner separation between core and FFI layers

---

### 🐛 Fixed

- Table display showing column names instead of actual data values
- Edge attribute access returning zeros instead of real values
- Subgraph accessor graph reference issues
- Memory leaks in AttrValue conversions
- PyGraphArray consistency issues
- Compilation warnings reduced from 247 to manageable levels

---

### 📊 Performance

- **SIMD acceleration**: 2-10x speedup for numeric array operations
- **Lazy evaluation**: Reduced memory allocations by deferring computation
- **Memory profiling**: Added tracking for optimization opportunities
- **Benchmark suite**: Comprehensive performance regression detection
- **CI monitoring**: Automated performance tracking in GitHub Actions

---

### 📚 Documentation

- 100+ planning and architecture documents in `documentation/`
- Comprehensive API reference with examples
- Usage tutorials for major features
- Testing strategy and coverage documentation
- Architecture decision records
- Visualization system guides
- Meta-API discovery reports
- Performance optimization guides

---

### 🔧 Developer Experience

- **Automatic stub generation** for Python type hints
- **Comprehensive error handling** with detailed context
- **Performance dashboards** for monitoring
- **CI/CD improvements** with automated testing
- **Memory profiling tools** for optimization
- **Benchmark infrastructure** with HTML reports
- **Test fixtures** for rapid development

---

### 📦 Dependencies

- Updated PyO3 to 0.20 for better Python interop
- Added tokio for async runtime (viz server)
- Added hyper for HTTP server (streaming viz)
- Added criterion for benchmarking
- Scientific computing stack: NumPy, SciPy, Pandas compatibility

---

### 🎯 Breaking Changes

**Minimal - Mostly Internal**:
- FFI module reorganization (internal, not user-facing)
- Some internal debugging APIs may have changed
- Graph bundle format updated (forward compatible)

**Python API remains largely compatible** - existing code should work with minimal changes.

---


### 🚀 Upgrade Guide

#### From v0.4.0

Most existing code will work without changes:

```python
import groggy as gr

# Existing code continues to work
g = gr.Graph()
g.add_node("alice", age=30)
g.add_edge("alice", "bob", weight=0.8)
subgraph = g.connected_components()[0]
```

**New capabilities available**:

```python
# Visualization
g.viz.show()  # Interactive visualization in Jupyter
g.viz.save("graph.html")  # Export to standalone HTML

# Type stubs for IDE autocomplete
g.nodes.  # <-- Full autocomplete available!

# Advanced table operations
table = g.nodes.table()
grouped = table.groupby("department").agg({"age": "mean"})
table.to_csv("nodes.csv")

# Matrix operations with auto-diff
import groggy.neural as gn
matrix = g.adjacency_matrix()
activation = gn.relu(matrix)
gradient = activation.backward()

# Meta-API discovery
from groggy.imports import discover_api
api_graph = discover_api(gr.Graph)
```

---

### 🙏 Acknowledgments

This release represents a fundamental transformation of Groggy into a comprehensive graph analytics platform with:
- Enterprise-grade testing (406 passing tests)
- Professional visualization system
- Neural network capabilities
- Complete IDE integration
- Comprehensive documentation

The 198,000+ lines of additions establish Groggy as a serious platform for graph analytics, machine learning, and data science.

---

### 🔗 Links

- **Full Changelog**: https://github.com/rollingstorms/groggy/compare/v0.4.0...v0.5.1
- **Documentation**: https://groggy.readthedocs.io
- **Issues**: https://github.com/rollingstorms/groggy/issues

---

## [0.4.0] - 2025-09-01

### 🏗️ Complete Architecture Overhaul

**Focus**: GraphEntity Foundation + Unified Traits System

See [RELEASE_NOTES_v0.4.0.md](documentation/releases/RELEASE_NOTES_v0.4.0.md) for complete details.

**Key achievements**:
- Universal GraphEntity trait system
- 2,087+ lines of specialized operation traits
- Fixed critical subgraph operations (BFS, DFS, shortest path)
- 4.5x connected components speedup
- Enhanced FFI architecture with pure delegation

**Statistics**: 112 files changed, 41,146 net lines added

---

## [0.3.0] - 2025-08-15

See [upcoming_release_notes_draft.md](documentation/releases/upcoming_release_notes_draft.md) for details.

**Key achievements**:
- GraphArray analytics system
- 48x node filtering performance improvement
- Lazy Rust view architecture
- Adjacency matrix API
- Enhanced query parser

---

## [0.2.1] - 2025-08-01

Minor bug fixes and performance improvements.

---

## [0.2.0] - 2025-07-15

Initial public release with core graph functionality.

---

[0.5.1]: https://github.com/rollingstorms/groggy/compare/v0.4.0...v0.5.1
[0.4.0]: https://github.com/rollingstorms/groggy/compare/v0.3.0...v0.4.0
[0.3.0]: https://github.com/rollingstorms/groggy/compare/v0.2.1...v0.3.0
[0.2.1]: https://github.com/rollingstorms/groggy/compare/v0.2.0...v0.2.1
[0.2.0]: https://github.com/rollingstorms/groggy/releases/tag/v0.2.0
