"""Tests for the PromptInjectionDetector class."""

from agentdojo.agent_pipeline.pi_detector import PromptInjectionDetector
from agentdojo.functions_runtime import EmptyEnv, FunctionCall, FunctionsRuntime
from agentdojo.types import ChatToolResultMessage, text_content_block_from_string


class MockDetector(PromptInjectionDetector):
    """Mock detector for testing that tracks all detected texts."""

    def __init__(self, mode="message", raise_on_injection=False):
        super().__init__(mode=mode, raise_on_injection=raise_on_injection)
        self.detected_texts = []

    def detect(self, tool_output: str) -> bool:
        """Store the detected text and always return False (no injection)."""
        self.detected_texts.append(tool_output)
        return False


def test_detector_extracts_correct_content_from_multiple_tool_messages():
    """Test that the detector extracts content from each individual tool message, not just the last one.
    
    This is a regression test for the bug where the detector was using messages[-1]["content"]
    instead of message["content"] in the loop, causing all tool messages to be checked with
    the content of the last message instead of their own content.
    """
    # Create a mock detector that tracks all texts it processes
    detector = MockDetector(mode="message")
    
    # Create three tool messages with different content
    tool_message_1 = ChatToolResultMessage(
        role="tool",
        tool_call=FunctionCall(function="test_func_1", args={}),
        content=[text_content_block_from_string("First tool output")],
        tool_call_id="call_1",
        error=None,
    )
    
    tool_message_2 = ChatToolResultMessage(
        role="tool",
        tool_call=FunctionCall(function="test_func_2", args={}),
        content=[text_content_block_from_string("Second tool output")],
        tool_call_id="call_2",
        error=None,
    )
    
    tool_message_3 = ChatToolResultMessage(
        role="tool",
        tool_call=FunctionCall(function="test_func_3", args={}),
        content=[text_content_block_from_string("Third tool output")],
        tool_call_id="call_3",
        error=None,
    )
    
    # Create a message list with all three tool messages
    messages = [tool_message_1, tool_message_2, tool_message_3]
    
    # Run the detector
    runtime = FunctionsRuntime()
    env = EmptyEnv()
    _, _, _, _, _ = detector.query("test query", runtime, env, messages, {})
    
    # The detector should have processed all three messages with their correct content
    # Since we iterate in reverse, we expect: Third, Second, First
    assert len(detector.detected_texts) == 3, f"Expected 3 texts, got {len(detector.detected_texts)}"
    assert detector.detected_texts[0] == "Third tool output", f"First detection should be 'Third tool output', got {detector.detected_texts[0]}"
    assert detector.detected_texts[1] == "Second tool output", f"Second detection should be 'Second tool output', got {detector.detected_texts[1]}"
    assert detector.detected_texts[2] == "First tool output", f"Third detection should be 'First tool output', got {detector.detected_texts[2]}"
    
    # Before the fix, all three would have been "Third tool output" because the bug
    # used messages[-1]["content"] instead of message["content"]
