"""
Copyright 2023-2023 VMware Inc.
SPDX-License-Identifier: Apache-2.0

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
"""

import json
import logging
import os
import time
from datetime import datetime
from enum import Enum
from json.decoder import JSONDecodeError

import click
import hcs_core.ctxp.data_util as data_util
import hcs_core.sglib.cli_options as cli
import httpx

from hcs_cli.service import admin, org_service, pki, vmm

# Create a logger
logger = logging.getLogger("migrate_edge")

global common_dict
common_dict = dict()
allowed_providers = {"azure", "aws"}


@click.command(name="migrate_edge", hidden="false")
@cli.org_id
@click.argument("edge_ids", type=str, required=False)
@click.option("--ca-label", type=click.Choice(["legacy", "omnissa"], case_sensitive=True), required=True)
@click.option("--file", type=click.Path(), required=False)
@click.option("--run-id", type=str, required=False)
@click.option("--verbose", "--v", type=bool, is_flag=True, required=False)
def migrate(org: str, edge_ids: str, ca_label: str, file: str, run_id: str, verbose: bool):
    """one-time edge migration for rootca - cert refresh"""

    if file:
        if not os.path.isfile(file):
            return f"invalid file path {file}", 1
        if os.path.getsize(file) <= 0:
            return f"no content in the file {file}", 1

    target_org_id = org
    if not file and not target_org_id:
        return "--org or --file is required", 1

    if file and target_org_id:
        return "both --org and --file command options not allowed. Specify only one of them.", 1

    if run_id:
        common_dict["RUN_ID"] = run_id
    else:
        common_dict["RUN_ID"] = str(datetime.now().strftime("%m%d%Y%H%M%S"))

    # log all command input
    logger.info("one-time edge migration - command input:")
    logger.info(f"file = {file}")
    logger.info(f"target_org_id = {target_org_id}")
    logger.info(f"edge_ids = {edge_ids}")
    logger.info(f"ca_label = {ca_label}")
    logger.info(f"verbose = {verbose}")
    logger.info(f"run_id = {common_dict['RUN_ID']}")

    logger.info("Starting one-time edge migration:\n")
    logger.info(f"default timeout for API calls: {os.environ.get('HCS_TIMEOUT', 30)} seconds")
    if file:
        logger.info("input file specified. file input takes the precedence. processing the file now")
        f = open(file)
        data = None
        try:
            data = json.load(f)
        except JSONDecodeError as error:
            logger.error(f"{file} file can't be parsed due to {error}")
            return
        logger.info(f"file content = {data}")
        if not data:
            return "invalid file content", 1
        for org_id in data.keys():
            logger.info(f"{org_id} -- {data[org_id]}")
            edge_id_list = data.get(org_id)
            edge_ids = None
            if edge_id_list:
                edge_ids = ",".join(edge_id_list)
            migrate_edges(org_id, edge_ids, ca_label, verbose)
    else:
        migrate_edges(target_org_id, edge_ids, ca_label, verbose)


def migrate_edges(target_org_id: str, edge_ids: str, ca_label: str, verbose: bool):
    org = org_service.details.get(target_org_id)
    if not org:
        logger.error(f"Invalid org id {target_org_id}")
        return
    logger.info(f"Migrating edges: target_org_id = {target_org_id}, edge_ids = {edge_ids}, ca_label = {ca_label}, verbose = {verbose}")
    if edge_ids:
        edges = admin.edge.items(org_id=target_org_id, search="id $in " + edge_ids)
    else:
        edges = admin.edge.items(org_id=target_org_id)
    for edge in edges:
        edge_id = edge["id"]
        if edge["providerLabel"].lower() not in allowed_providers:
            logger.warn(
                f"One-time edge migration is not supported on {edge.get('providerLabel')} provider edge {edge_id}, org: {target_org_id}"
            )
            continue

        logger.info("one-time edge migration on org_id: %s, edge_id: %s", target_org_id, edge_id)
        stats = dict()
        stats["org_id"] = target_org_id
        stats["edge_id"] = edge_id
        stats["edge_name"] = edge.get("name")
        stats["provider_label"] = edge.get("providerLabel")
        stats["is_fqdn_auto_generated"] = edge.get("fqdnAutoGenerated")

        try:
            logger.info("performing one-time edge migration on org_id: %s, edge_id: %s", target_org_id, edge_id)
            # check if edge mqtt url is already migrated to omnissa on the edge
            edge_mqtt_url = data_util.deep_get_attr(edge, "fqdn", raise_on_not_found=False)
            if edge.get("fqdnAutoGenerated") and edge.get("fqdnAutoGenerated") is True:
                if ".omnissa." in edge_mqtt_url:
                    logger.warn(
                        f"Edge({edge_id}): edge mqtt url already updated with omnissa url {edge_mqtt_url}. Edge mqtt url update is skipped."
                    )
                    stats["edge_fqdn_mig_status"] = "ALREADY_MIGRATED"
                else:
                    try:
                        admin.edge.rootca_migrate_edge_fqdn(edge_id, target_org_id, verbose)
                        time.sleep(10)
                        is_sync_complete = is_sync_complete_on_all_edge_templates(target_org_id, edge_id, verbose)
                        if is_sync_complete:
                            logger.info(f"Edge({edge_id}): Edge is updated with omnissa' edge mqtt url")
                            stats["edge_fqdn_mig_status"] = "MIGRATED"
                        else:
                            logger.error(
                                f"Edge({edge_id}): Edge is not updated with omnissa' edge mqtt url since sync not completed on all edge templates."
                            )
                            stats["edge_fqdn_mig_status"] = "FAILED"
                            stats["info"] = "Edge FQDN is updated but edge templates' sync not completed on all edge templates. "
                    except Exception as ex:
                        logger.error(f"Edge({edge_id}): failed updating edge FQDN due to {str(ex)}")
                        stats["edge_fqdn_mig_status"] = "FAILED"
                        stats["info"] = str(ex)
            else:
                stats["edge_fqdn_mig_status"] = "NOT_APPLICABLE"
                stats["info"] = "one-time edge FQDN update is not needed since edge FQDN is not auto generated. "

            # check if private endpoint url is already migrated to omnissa on the edge
            regional_mqtt_url = data_util.deep_get_attr(edge, "privateEndpointDetails.dnsRecord", raise_on_not_found=False)
            if not regional_mqtt_url:
                logger.warn(f"Edge({edge_id}): edge uses global mqtt url. privateEndpoint update is skipped.")
                stats["edge_pep_fqdn_mig_status"] = "NOT_APPLICABLE"
                msg = "one-time edge private link update is not needed since edge uses global mqtt url. "
                stats["info"] = stats["info"] + msg if stats.get("info") else msg
            elif ".omnissa." in regional_mqtt_url:
                logger.warn(
                    f"Edge({edge_id}): edge private endpoint url already updated with omnissa url {regional_mqtt_url}. private endpoint url update is skipped."
                )
                stats["edge_pep_fqdn_mig_status"] = "ALREADY_MIGRATED"
            else:
                admin.edge.rootca_migrate_private_endpoint_fqdn(edge_id, target_org_id, verbose)
                time.sleep(10)
                is_sync_complete = is_sync_complete_on_all_edge_templates(target_org_id, edge_id, verbose)
                if is_sync_complete:
                    logger.info(f"Edge({edge_id}): Edge is updated with omnissa' private endpoint url")
                    stats["edge_pep_fqdn_mig_status"] = "MIGRATED"
                else:
                    logger.error(
                        f"Edge({edge_id}): Edge is not updated with omnissa' private endpoint url since sync not completed on all edge templates."
                    )
                    stats["edge_pep_fqdn_mig_status"] = "FAILED"
                    msg = "Edge private endpoint FQDN is updated but edge templates' sync not completed on all edge templates. "
                    stats["info"] = stats["info"] + msg if stats.get("info") else msg

            edge_calabel = pki.certificate.get_edge_calabel(edge_id, target_org_id, verbose)
            logger.info(f"edge_calabel = {edge_calabel}")
            if edge_calabel and edge_calabel.get("caLabel") and edge_calabel.get("caLabel") == ca_label:
                logger.warn(f"Edge({edge_id}): PKI already has caLabel {ca_label}. caLabel update on pki is skipped.")
                stats["edge_pki_ca_label"] = edge_calabel.get("caLabel")
                stats["edge_pki_ca_label_mig_status"] = "ALREADY_MIGRATED"
            else:
                pki.certificate.update_edge_calabel(edge_id, target_org_id, ca_label, verbose)
                logger.info(f"Edge({edge_id}): PKI is updated with the caLabel {ca_label} for this edge")
                stats["edge_pki_ca_label_mig_status"] = "MIGRATED"
                stats["edge_pki_ca_label"] = ca_label
                logger.info("sleep 7 minutes for cache refresh in PKI")
                time.sleep(60 * 7)

            # perform uag twin sync operation
            try:
                res = admin.edge.rootca_sync_uag_twin_config(edge_id, target_org_id, verbose)
                if res.get("twinSyncStatusMap") and res.get("twinSyncStatusMap").get("UAG_MODULE") == "SYNC_NEEDED":
                    stats["uag_twin_sync_config_status"] = "SYNCED"
                    logger.info(f"Edge({edge_id}): UAG twin sync config request accepted - {str(res)}")
                else:
                    logger.warn(f"Edge({edge_id}): UAG twin sync config request failed due to {str(res)}")
                    stats["uag_twin_sync_config_status"] = "FAILED"
                    msg = f"uag twin sync config request failed due to {str(res)}"
                    stats["info"] = stats["info"] + msg if stats.get("info") else msg
            except Exception as ex:
                logger.error(f"Edge({edge_id}): UAG twin sync operation failed due to {str(ex)}")
                stats["uag_twin_sync_config_status"] = "FAILED"
                msg = f"uag twin sync operation failed due to {str(ex)}"
                stats["info"] = stats["info"] + msg if stats.get("info") else msg

            refresh_certs_on_all_uag_vms(target_org_id, edge, stats, verbose)

            send_event(stats, HocStatus.SUCCESS.value, verbose)
        except httpx.HTTPStatusError as ex:
            logger.error(f"Http Status Code: {ex.response.status_code}")
            logger.error(f"Url: {ex.request.url}")
            logger.error(f"Response Content: {ex.response.text}")
            stats["info"] = str(ex)
            send_event(stats, HocStatus.FAILURE.value, verbose)
            return
        except Exception as error:
            logger.error(f"Failed one-time edge migration on org_id: %s, edge_id: %s due to {str(error)}", target_org_id, edge_id)
            stats["info"] = str(error)
            send_event(stats, HocStatus.FAILURE.value, verbose)
            return
    logger.info("Finished one-time edge migration on org_id: %s", target_org_id)


def refresh_certs_on_all_uag_vms(org_id: str, edge: dict, stats: dict, verbose: bool):
    logger.info(f"cert refresh on all uag vms; org_id: {org_id}, edge_id: {edge['id']}")
    edge_id = edge["id"]
    try:
        # first get uag_id
        uags_dict = admin.uag.get_by_edge_deployment_id(edge_id, org_id)
        if uags_dict and uags_dict["content"] and uags_dict["content"][0]["id"]:
            uag_id = uags_dict["content"][0]["id"]
            logger.info(f"cert refresh on all uag vms; org_id: {org_id}, edge_id: {edge['id']}, uag_id: {uag_id}")
            # perform cert refresh on all uag vms
            res = admin.uag.refresh_certs_on_uag_vms(uag_id, org_id, "UPDATE_PKI_CERTIFICATE", verbose)
            logger.info(f"cert refresh on all uag vms; response = {res}")
            # get cert refresh status of all uag vms
            uag_vms_cert_refresh_result = dict()
            if res.results:
                for key, value in res.results.items():
                    vm_cert_ref_status = "fl"
                    if value and value.get("diagnosticData") and value.get("diagnosticData").startswith("Enable succeeded"):
                        vm_cert_ref_status = "su"
                    uag_vms_cert_refresh_result[key] = vm_cert_ref_status
            if res.errors:
                for key, value in res.errors.items():
                    uag_vms_cert_refresh_result[key] = "fl"
            # deduce cert refresh status across all uag vms
            uag_vms_cert_refresh_status = "REFRESHED"
            if len(uag_vms_cert_refresh_result) == 0:
                uag_vms_cert_refresh_status = "REFRESH_FAILED"
            else:
                for value in uag_vms_cert_refresh_result.values():
                    if value == "fl":
                        uag_vms_cert_refresh_status = "REFRESH_FAILED"
                        break
            logger.info(
                f"cert refresh on all uag vms; org_id: {org_id}, edge_id: {edge['id']}, uag_id: {uag_id}; uag_vms_cert_refresh_status: {uag_vms_cert_refresh_status}, detailed_status: {uag_vms_cert_refresh_result}"
            )
            stats["uag_vms_cert_refresh_status"] = uag_vms_cert_refresh_status
            stats["uag_vms_cert_refresh_status_details"] = str(uag_vms_cert_refresh_result)
    except Exception as ex:
        stats["uag_vms_cert_refresh_status"] = "REFRESH_FAILED"
        logger.error(f"cert refresh on all uag vms failed due to {str(ex)}")
        msg = f"cert refresh uag vms failed due to {str(ex)}"
        stats["info"] = stats["info"] + msg if stats.get("info") else msg


def is_sync_complete_on_all_edge_templates(target_org_id: str, edge_id: str, verbose: bool):
    for template in admin.template.items(org_id=target_org_id, search="edgeDeploymentId $eq " + edge_id):
        if "SYNC_COMPLETE" != template["syncStatus"]:
            return False
    return True


# "rca:edge:otmig" =  rootca:edge:onetimemigration
def send_event(stats: dict, status: str, verbose: bool):
    unique_id = stats.get("edge_id")
    if not unique_id:
        unique_id = str(int(round(time.time() * 1000)))

    migration_status = "NOT_MIGRATED"
    if (
        (
            stats.get("edge_fqdn_mig_status")
            and (
                stats.get("edge_fqdn_mig_status") == "MIGRATED"
                or stats.get("edge_fqdn_mig_status") == "ALREADY_MIGRATED"
                or stats.get("edge_fqdn_mig_status") == "NOT_APPLICABLE"
            )
        )
        and (
            stats.get("edge_pep_fqdn_mig_status")
            and (
                stats.get("edge_pep_fqdn_mig_status") == "MIGRATED"
                or stats.get("edge_pep_fqdn_mig_status") == "ALREADY_MIGRATED"
                or stats.get("edge_pep_fqdn_mig_status") == "NOT_APPLICABLE"
            )
        )
        and (
            stats.get("edge_pki_ca_label_mig_status")
            and (stats.get("edge_pki_ca_label_mig_status") == "MIGRATED" or stats.get("edge_pki_ca_label_mig_status") == "ALREADY_MIGRATED")
        )
        and (stats.get("uag_twin_sync_config_status") and stats.get("uag_twin_sync_config_status") == "SYNCED")
        and (stats.get("uag_vms_cert_refresh_status") and stats.get("uag_vms_cert_refresh_status") == "REFRESHED")
    ):
        migration_status = "MIGRATED"

    event = {
        hoc_fields["id"]: unique_id,
        hoc_fields["type"]: "rca:edge:otmig",
        hoc_fields["version"]: "1",
        hoc_fields["source"]: "vmm",
    }
    data = {
        hoc_fields["utcTime"]: int(round(time.time() * 1000)),
        hoc_fields["run_id"]: common_dict["RUN_ID"],
        hoc_fields["org_id"]: stats.get("org_id"),
        hoc_fields["edge_id"]: stats.get("edge_id"),
        hoc_fields["edge_name"]: stats.get("edge_name"),
        hoc_fields["provider_label"]: stats.get("provider_label"),
        hoc_fields["is_fqdn_auto_generated"]: stats.get("is_fqdn_auto_generated"),
        hoc_fields["edge_fqdn_mig_status"]: stats.get("edge_fqdn_mig_status"),
        hoc_fields["edge_pep_fqdn_mig_status"]: stats.get("edge_pep_fqdn_mig_status"),
        hoc_fields["uag_twin_sync_config_status"]: stats.get("uag_twin_sync_config_status"),
        hoc_fields["edge_pki_ca_label_mig_status"]: stats.get("edge_pki_ca_label_mig_status"),
        hoc_fields["edge_pki_ca_label"]: stats.get("edge_pki_ca_label"),
        hoc_fields["uag_vms_cert_refresh_status"]: stats.get("uag_vms_cert_refresh_status"),
        hoc_fields["uag_vms_cert_refresh_status_details"]: stats.get("uag_vms_cert_refresh_status_details"),
        hoc_fields["mig_status"]: migration_status,
        hoc_fields["status"]: status,
        hoc_fields["info"]: stats.get("info"),
    }

    if stats.get("info"):
        data[hoc_fields["info"]] = stats.get("info")

    event["d"] = data
    vmm.hoc_event.send_generic(event, verbose)


hoc_fields = dict()
# common HOC fields
hoc_fields["id"] = "i"
hoc_fields["type"] = "t"
hoc_fields["version"] = "v"
hoc_fields["source"] = "src"
hoc_fields["data"] = "d"

hoc_fields["utcTime"] = "utcTime"
hoc_fields["org_id"] = "oid"

# template stats HOC fields
hoc_fields["edge_id"] = "eid"
hoc_fields["edge_name"] = "ename"
hoc_fields["provider_label"] = "epl"
hoc_fields["is_fqdn_auto_generated"] = "fqdnautogen"
hoc_fields["edge_fqdn_mig_status"] = "efqdnmigst"
hoc_fields["edge_pep_fqdn_mig_status"] = "epepfqdnmigst"
hoc_fields["edge_pki_ca_label_mig_status"] = "epkicalblmigst"
hoc_fields["uag_twin_sync_config_status"] = "uagtscst"
hoc_fields["edge_pki_ca_label"] = "epkicalbl"
hoc_fields["uag_vms_cert_refresh_status"] = "uagvmscertrefst"
hoc_fields["uag_vms_cert_refresh_status_details"] = "uagvmscertrefstdetails"
hoc_fields["info"] = "info"
hoc_fields["run_id"] = "runid"
hoc_fields["mig_status"] = "migst"
hoc_fields["status"] = "s"


class HocStatus(str, Enum):
    SUCCESS = "su"
    FAILURE = "fl"
