"""
Generate template command implementation
"""

import os
from rich.console import Console
from rich.progress import Progress, SpinnerColumn, TextColumn
from rich.panel import Panel

from ...core.analyzer import LambdaAnalyzer

console = Console()


class GenerateCommand:
    """Handler for the generate-template command"""

    @staticmethod
    def register_parser(subparsers):
        """Register the generate-template command parser"""
        template_parser = subparsers.add_parser(
            'generate-template',
            help='Generate infrastructure template (SAM/CloudFormation) from analysis'
        )
        template_parser.add_argument('path', help='Path to Python file or directory')
        template_parser.add_argument(
            '--format',
            choices=['sam', 'cloudformation'],
            default='sam',
            help='Template format to generate'
        )
        template_parser.add_argument(
            '--output', '-o',
            default='template.yaml',
            help='Output template file path'
        )
        template_parser.add_argument(
            '--config', '-c',
            help='Path to configuration file'
        )
        template_parser.add_argument(
            '--preview',
            action='store_true',
            help='Show template preview in console'
        )
        template_parser.add_argument(
            '--dry-run',
            action='store_true',
            help='Generate template without saving to file'
        )

    @staticmethod
    def execute(args) -> int:
        """Execute the generate-template command"""
        try:
            with Progress(
                    SpinnerColumn(),
                    TextColumn("[progress.description]{task.description}"),
                    console=console
            ) as progress:
                task = progress.add_task("Analyzing functions and generating template...", total=None)

                analyzer = LambdaAnalyzer(config_path=args.config)

                if os.path.isfile(args.path):
                    result = analyzer.analyze_file(args.path)
                    results = [result]
                elif os.path.isdir(args.path):
                    results = analyzer.analyze_directory(args.path)
                else:
                    console.print(f"❌ [red]Path not found: {args.path}[/red]")
                    return 2

                # Generate template (this would need template modules)
                template = GenerateCommand._generate_template_placeholder(
                    results, args.format
                )

                progress.update(task, completed=1, total=1)

                # Save template unless dry run
                if not args.dry_run:
                    with open(args.output, 'w') as f:
                        f.write(template)
                    console.print(f"✅ [green]{args.format.upper()} template generated: {args.output}[/green]")
                else:
                    console.print("🔍 [yellow]Dry run mode - template not saved[/yellow]")

                # Show preview if requested
                if args.preview or args.dry_run:
                    console.print(f"\n📋 [bold]Template Preview:[/bold]")
                    preview_text = template[:1000] + "\n..." if len(template) > 1000 else template
                    console.print(Panel(
                        preview_text,
                        title=f"{args.format.upper()} Template",
                        border_style="blue"
                    ))

            return 0

        except Exception as e:
            console.print(f"\n❌ [red]Template generation failed: {str(e)}[/red]")
            if hasattr(args, 'verbose') and args.verbose:
                console.print_exception()
            return 1

    @staticmethod
    def _generate_template_placeholder(results, template_format):
        """Generate a placeholder template (real implementation would use template modules)"""
        all_services = set()
        all_env_vars = set()

        for result in results:
            all_services.update(result.services)
            all_env_vars.update(result.environment_variables)

        if template_format == 'sam':
            return f"""AWSTemplateFormatVersion: '2010-09-09'
Transform: AWS::Serverless-2016-10-31
Description: 'Generated by Lambda Analyzer'

Parameters:
{GenerateCommand._generate_parameters(all_env_vars)}

Globals:
  Function:
    Runtime: python3.9
    MemorySize: 512
    Timeout: 30

Resources:
{GenerateCommand._generate_sam_resources(results, all_services)}

Outputs:
  # Add outputs based on your needs
"""
        else:  # cloudformation
            return f"""AWSTemplateFormatVersion: '2010-09-09'
Description: 'Generated by Lambda Analyzer'

Parameters:
{GenerateCommand._generate_parameters(all_env_vars)}

Resources:
{GenerateCommand._generate_cf_resources(results, all_services)}

Outputs:
  # Add outputs based on your needs
"""

    @staticmethod
    def _generate_parameters(env_vars):
        """Generate template parameters"""
        if not env_vars:
            return "  # No parameters needed"

        params = []
        for var in sorted(env_vars):
            params.append(f"""  {var}:
    Type: String
    Description: Value for {var} environment variable""")

        return "\n".join(params)

    @staticmethod
    def _generate_sam_resources(results, services):
        """Generate SAM resources placeholder"""
        resources = []

        # Lambda functions
        for i, result in enumerate(results):
            func_name = f"Function{i + 1}" if len(results) > 1 else "Function"
            filename = os.path.basename(result.file_path).replace('.py', '')

            resources.append(f"""  {func_name}:
    Type: AWS::Serverless::Function
    Properties:
      CodeUri: .
      Handler: {filename}.lambda_handler
      Runtime: python3.9
      # Add IAM policies based on analysis results
      # Services detected: {', '.join(result.services) if result.services else 'None'}""")

        # Add service-specific resources
        if 's3' in services:
            resources.append("""  S3Bucket:
    Type: AWS::S3::Bucket
    Properties:
      BucketEncryption:
        ServerSideEncryptionConfiguration:
          - ServerSideEncryptionByDefault:
              SSEAlgorithm: AES256""")

        if 'dynamodb' in services:
            resources.append("""  DynamoDBTable:
    Type: AWS::DynamoDB::Table
    Properties:
      BillingMode: PAY_PER_REQUEST
      AttributeDefinitions:
        - AttributeName: id
          AttributeType: S
      KeySchema:
        - AttributeName: id
          KeyType: HASH""")

        return "\n".join(resources) if resources else "  # No resources generated"

    @staticmethod
    def _generate_cf_resources(results, services):
        """Generate CloudFormation resources placeholder"""
        # Similar to SAM but with full CloudFormation syntax
        return "  # CloudFormation resources would be generated here"