"""
CloudFormation template generator
"""

import yaml
import json
from typing import Dict, List, Any
from pathlib import Path


class CloudFormationTemplateGenerator:
    """Generates CloudFormation templates from analysis results"""

    def __init__(self, config: Dict):
        """Initialize with configuration"""
        self.config = config
        self.template_config = config.get('template', {})

    def generate(self, analysis_results: List) -> str:
        """
        Generate CloudFormation template from analysis results

        Args:
            analysis_results: List of AnalysisResult objects

        Returns:
            CloudFormation template as YAML string
        """
        template = {
            'AWSTemplateFormatVersion': '2010-09-09',
            'Description': 'CloudFormation Template generated by Lambda Analyzer',
            'Parameters': {},
            'Resources': {},
            'Outputs': {}
        }

        # Collect all environment variables and services
        all_env_vars = set()
        all_services = set()

        for result in analysis_results:
            all_env_vars.update(result.environment_variables)
            all_services.update(result.services)

        # Generate parameters
        template['Parameters'] = self._generate_parameters(all_env_vars)

        # Generate IAM roles first
        iam_resources = self._generate_iam_resources(analysis_results)
        template['Resources'].update(iam_resources)

        # Generate Lambda functions
        for i, result in enumerate(analysis_results):
            function_resources = self._generate_function_resources(result, i, len(analysis_results))
            template['Resources'].update(function_resources)

        # Generate supporting AWS resources
        support_resources = self._generate_support_resources(all_services)
        template['Resources'].update(support_resources)

        # Generate outputs
        template['Outputs'] = self._generate_outputs(analysis_results, all_services)

        return yaml.dump(template, default_flow_style=False, sort_keys=False)

    def _generate_parameters(self, env_vars: set) -> Dict[str, Any]:
        """Generate CloudFormation parameters"""
        parameters = {}

        for env_var in sorted(env_vars):
            parameters[f"{env_var}Parameter"] = {
                'Type': 'String',
                'Description': f'Value for {env_var} environment variable',
                'NoEcho': True if any(sensitive in env_var.lower()
                                      for sensitive in ['key', 'secret', 'password']) else False
            }

        parameters['Environment'] = {
            'Type': 'String',
            'Default': 'dev',
            'AllowedValues': ['dev', 'staging', 'prod'],
            'Description': 'Deployment environment'
        }

        return parameters

    def _generate_iam_resources(self, results: List) -> Dict[str, Any]:
        """Generate IAM roles and policies"""
        resources = {}

        for i, result in enumerate(results):
            if len(results) > 1:
                role_name = f"Function{i + 1}Role"
            else:
                role_name = "LambdaExecutionRole"

            # Basic Lambda execution role
            resources[role_name] = {
                'Type': 'AWS::IAM::Role',
                'Properties': {
                    'RoleName': f'!Sub ${{AWS::StackName}}-{role_name}',
                    'AssumeRolePolicyDocument': {
                        'Version': '2012-10-17',
                        'Statement': [{
                            'Effect': 'Allow',
                            'Principal': {'Service': 'lambda.amazonaws.com'},
                            'Action': 'sts:AssumeRole'
                        }]
                    },
                    'ManagedPolicyArns': [
                        'arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole'
                    ],
                    'Policies': []
                }
            }

            # Add custom policies based on analysis
            if result.iam_policy and result.iam_policy.get('Statement'):
                policy_name = f"{role_name}Policy"
                custom_policy = {
                    'PolicyName': policy_name,
                    'PolicyDocument': result.iam_policy
                }
                resources[role_name]['Properties']['Policies'].append(custom_policy)

        return resources

    def _generate_function_resources(self, result, index: int, total_count: int) -> Dict[str, Any]:
        """Generate Lambda function resources"""
        resources = {}

        if total_count > 1:
            function_name = f"Function{index + 1}"
            role_name = f"Function{index + 1}Role"
        else:
            function_name = "LambdaFunction"
            role_name = "LambdaExecutionRole"

        file_stem = Path(result.file_path).stem
        handler = f"{file_stem}.lambda_handler"

        function_config = {
            'Type': 'AWS::Lambda::Function',
            'Properties': {
                'FunctionName': f'!Sub ${{AWS::StackName}}-{function_name}',
                'Runtime': self.template_config.get('runtime', 'python3.9'),
                'Handler': handler,
                'Role': f'!GetAtt {role_name}.Arn',
                'Code': {
                    'ZipFile': f'# Placeholder for {file_stem}\n# Replace with actual deployment package'
                },
                'MemorySize': self.template_config.get('memory', 512),
                'Timeout': self.template_config.get('timeout', 30),
                'Environment': {
                    'Variables': {}
                },
                'Description': f'Function generated by Lambda Analyzer - Services: {", ".join(result.services)}'
            }
        }

        # Add environment variables
        for env_var in result.environment_variables:
            function_config['Properties']['Environment']['Variables'][env_var] = f"!Ref {env_var}Parameter"

        resources[function_name] = function_config

        # Generate event source mappings for supported triggers
        event_sources = self._generate_event_source_mappings(result, function_name)
        resources.update(event_sources)

        return resources

    def _generate_event_source_mappings(self, result, function_name: str) -> Dict[str, Any]:
        """Generate event source mappings for triggers"""
        resources = {}

        for i, trigger in enumerate(result.triggers):
            trigger_type = trigger['type']

            if trigger_type == 'sqs':
                mapping_name = f"{function_name}SQSMapping{i + 1}"
                resources[mapping_name] = {
                    'Type': 'AWS::Lambda::EventSourceMapping',
                    'Properties': {
                        'EventSourceArn': '!GetAtt SQSQueue.Arn',
                        'FunctionName': f'!Ref {function_name}',
                        'BatchSize': 10
                    }
                }
            elif trigger_type == 'dynamodb':
                mapping_name = f"{function_name}DynamoDBMapping{i + 1}"
                resources[mapping_name] = {
                    'Type': 'AWS::Lambda::EventSourceMapping',
                    'Properties': {
                        'EventSourceArn': '!GetAtt DynamoDBTable.StreamArn',
                        'FunctionName': f'!Ref {function_name}',
                        'StartingPosition': 'TRIM_HORIZON'
                    }
                }

        return resources

    def _generate_support_resources(self, services: set) -> Dict[str, Any]:
        """Generate supporting AWS resources"""
        resources = {}

        if 's3' in services:
            resources['S3Bucket'] = {
                'Type': 'AWS::S3::Bucket',
                'Properties': {
                    'BucketName': '!Sub ${AWS::StackName}-data-bucket',
                    'BucketEncryption': {
                        'ServerSideEncryptionConfiguration': [{
                            'ServerSideEncryptionByDefault': {
                                'SSEAlgorithm': 'AES256'
                            }
                        }]
                    },
                    'PublicAccessBlockConfiguration': {
                        'BlockPublicAcls': True,
                        'BlockPublicPolicy': True,
                        'IgnorePublicAcls': True,
                        'RestrictPublicBuckets': True
                    }
                }
            }

        if 'dynamodb' in services:
            resources['DynamoDBTable'] = {
                'Type': 'AWS::DynamoDB::Table',
                'Properties': {
                    'TableName': '!Sub ${AWS::StackName}-data-table',
                    'BillingMode': 'PAY_PER_REQUEST',
                    'AttributeDefinitions': [{
                        'AttributeName': 'id',
                        'AttributeType': 'S'
                    }],
                    'KeySchema': [{
                        'AttributeName': 'id',
                        'KeyType': 'HASH'
                    }],
                    'StreamSpecification': {
                        'StreamViewType': 'NEW_AND_OLD_IMAGES'
                    },
                    'PointInTimeRecoverySpecification': {
                        'PointInTimeRecoveryEnabled': True
                    }
                }
            }

        if 'sns' in services:
            resources['SNSTopic'] = {
                'Type': 'AWS::SNS::Topic',
                'Properties': {
                    'TopicName': '!Sub ${AWS::StackName}-notifications',
                    'KmsMasterKeyId': 'alias/aws/sns'
                }
            }

        if 'sqs' in services:
            resources['SQSQueue'] = {
                'Type': 'AWS::SQS::Queue',
                'Properties': {
                    'QueueName': '!Sub ${AWS::StackName}-processing-queue',
                    'KmsMasterKeyId': 'alias/aws/sqs',
                    'MessageRetentionPeriod': 1209600,
                    'VisibilityTimeoutSeconds': 180,
                    'RedrivePolicy': {
                        'deadLetterTargetArn': '!GetAtt SQSDeadLetterQueue.Arn',
                        'maxReceiveCount': 3
                    }
                }
            }

            resources['SQSDeadLetterQueue'] = {
                'Type': 'AWS::SQS::Queue',
                'Properties': {
                    'QueueName': '!Sub ${AWS::StackName}-dlq',
                    'KmsMasterKeyId': 'alias/aws/sqs',
                    'MessageRetentionPeriod': 1209600
                }
            }

        return resources

    def _generate_outputs(self, results: List, services: set) -> Dict[str, Any]:
        """Generate CloudFormation outputs"""
        outputs = {}

        # Function outputs
        for i, result in enumerate(results):
            if len(results) > 1:
                func_name = f"Function{i + 1}"
            else:
                func_name = "LambdaFunction"

            outputs[f"{func_name}Arn"] = {
                'Description': f'ARN of {func_name}',
                'Value': {'Fn::GetAtt': [func_name, 'Arn']},
                'Export': {
                    'Name': {
                        'Fn::Sub': f"${{{{AWS::StackName}}}}-{func_name}-Arn"
                    }
                }
            }

        # Service outputs
        if 's3' in services:
            outputs['S3BucketName'] = {
                'Description': 'Name of the S3 bucket',
                'Value': {'Ref': 'S3Bucket'}
            }

        if 'dynamodb' in services:
            outputs['DynamoDBTableName'] = {
                'Description': 'Name of the DynamoDB table',
                'Value': {'Ref': 'DynamoDBTable'}
            }

        if 'sns' in services:
            outputs['SNSTopicArn'] = {
                'Description': 'ARN of the SNS topic',
                'Value': {'Ref': 'SNSTopic'}
            }

        if 'sqs' in services:
            outputs['SQSQueueUrl'] = {
                'Description': 'URL of the SQS queue',
                'Value': {'Ref': 'SQSQueue'}
            }

        return outputs