"use strict";
// --------------------------------------------------------------------------------
// This file defines context keys that enable certain features that are
// implemented behind a flag in order to preserve backwards compatibility for
// existing apps. When a new app is initialized through `cdk init`, the CLI will
// automatically add enable these features by adding them to the generated
// `cdk.json` file. In the next major release of the CDK, these feature flags
// will be removed and will become the default behavior.
// See https://github.com/aws/aws-cdk-rfcs/blob/master/text/0055-feature-flags.md
// --------------------------------------------------------------------------------
Object.defineProperty(exports, "__esModule", { value: true });
exports.futureFlagDefault = exports.FUTURE_FLAGS = exports.S3_GRANT_WRITE_WITHOUT_ACL = exports.KMS_DEFAULT_KEY_POLICIES = exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME = exports.DOCKER_IGNORE_SUPPORT = exports.STACK_RELATIVE_EXPORTS_CONTEXT = exports.NEW_STYLE_STACK_SYNTHESIS_CONTEXT = exports.ENABLE_DIFF_NO_FAIL = exports.ENABLE_DIFF_NO_FAIL_CONTEXT = exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT = void 0;
/**
 * If this is set, multiple stacks can use the same stack name (e.g. deployed to
 * different environments). This means that the name of the synthesized template
 * file will be based on the construct path and not on the defined `stackName`
 * of the stack.
 *
 * This is a "future flag": the feature is disabled by default for backwards
 * compatibility, but new projects created using `cdk init` will have this
 * enabled through the generated `cdk.json`.
 */
exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT = '@aws-cdk/core:enableStackNameDuplicates';
/**
 * IF this is set, `cdk diff` will always exit with 0.
 *
 * Use `cdk diff --fail` to exit with 1 if there's a diff.
 */
exports.ENABLE_DIFF_NO_FAIL_CONTEXT = 'aws-cdk:enableDiffNoFail';
/** @deprecated use `ENABLE_DIFF_NO_FAIL_CONTEXT` */
exports.ENABLE_DIFF_NO_FAIL = exports.ENABLE_DIFF_NO_FAIL_CONTEXT;
/**
 * Switch to new stack synthesis method which enable CI/CD
 */
exports.NEW_STYLE_STACK_SYNTHESIS_CONTEXT = '@aws-cdk/core:newStyleStackSynthesis';
/**
 * Name exports based on the construct paths relative to the stack, rather than the global construct path
 *
 * Combined with the stack name this relative construct path is good enough to
 * ensure uniqueness, and makes the export names robust against refactoring
 * the location of the stack in the construct tree (specifically, moving the Stack
 * into a Stage).
 */
exports.STACK_RELATIVE_EXPORTS_CONTEXT = '@aws-cdk/core:stackRelativeExports';
/**
 * DockerImageAsset properly supports `.dockerignore` files by default
 *
 * If this flag is not set, the default behavior for `DockerImageAsset` is to use
 * glob semantics for `.dockerignore` files. If this flag is set, the default behavior
 * is standard Docker ignore semantics.
 *
 * This is a feature flag as the old behavior was technically incorrect but
 * users may have come to depend on it.
 */
exports.DOCKER_IGNORE_SUPPORT = '@aws-cdk/aws-ecr-assets:dockerIgnoreSupport';
/**
 * Secret.secretName for an "owned" secret will attempt to parse the secretName from the ARN,
 * rather than the default full resource name, which includes the SecretsManager suffix.
 *
 * If this flag is not set, Secret.secretName will include the SecretsManager suffix, which cannot be directly
 * used by SecretsManager.DescribeSecret, and must be parsed by the user first (e.g., Fn:Join, Fn:Select, Fn:Split).
 */
exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME = '@aws-cdk/aws-secretsmanager:parseOwnedSecretName';
/**
 * KMS Keys start with a default key policy that grants the account access to administer the key,
 * mirroring the behavior of the KMS SDK/CLI/Console experience. Users may override the default key
 * policy by specifying their own.
 *
 * If this flag is not set, the default key policy depends on the setting of the `trustAccountIdentities`
 * flag. If false (the default, for backwards-compatibility reasons), the default key policy somewhat
 * resemebles the default admin key policy, but with the addition of 'GenerateDataKey' permissions. If
 * true, the policy matches what happens when this feature flag is set.
 *
 * Additionally, if this flag is not set and the user supplies a custom key policy, this will be appended
 * to the key's default policy (rather than replacing it).
 */
exports.KMS_DEFAULT_KEY_POLICIES = '@aws-cdk/aws-kms:defaultKeyPolicies';
/**
 * Change the old 's3:PutObject*' permission to 's3:PutObject' on Bucket,
 * as the former includes 's3:PutObjectAcl',
 * which allows changing the visibility of an object written to the Bucket.
 * Use a feature flag to make sure existing customers who might be relying
 * on the overly-broad permissions are not broken.
 */
exports.S3_GRANT_WRITE_WITHOUT_ACL = '@aws-cdk/aws-s3:grantWriteWithoutAcl';
/**
 * This map includes context keys and values for feature flags that enable
 * capabilities "from the future", which we could not introduce as the default
 * behavior due to backwards compatibility for existing projects.
 *
 * New projects generated through `cdk init` will include these flags in their
 * generated `cdk.json` file.
 *
 * When we release the next major version of the CDK, we will flip the logic of
 * these features and clean up the `cdk.json` generated by `cdk init`.
 *
 * Tests must cover the default (disabled) case and the future (enabled) case.
 */
exports.FUTURE_FLAGS = {
    [exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT]: 'true',
    [exports.ENABLE_DIFF_NO_FAIL_CONTEXT]: 'true',
    [exports.STACK_RELATIVE_EXPORTS_CONTEXT]: 'true',
    [exports.DOCKER_IGNORE_SUPPORT]: true,
    [exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME]: true,
    [exports.KMS_DEFAULT_KEY_POLICIES]: true,
    [exports.S3_GRANT_WRITE_WITHOUT_ACL]: true,
};
/**
 * The set of defaults that should be applied if the feature flag is not
 * explicitly configured.
 */
const FUTURE_FLAGS_DEFAULTS = {
    [exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT]: false,
    [exports.ENABLE_DIFF_NO_FAIL_CONTEXT]: false,
    [exports.STACK_RELATIVE_EXPORTS_CONTEXT]: false,
    [exports.NEW_STYLE_STACK_SYNTHESIS_CONTEXT]: false,
    [exports.DOCKER_IGNORE_SUPPORT]: false,
    [exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME]: false,
    [exports.KMS_DEFAULT_KEY_POLICIES]: false,
    [exports.S3_GRANT_WRITE_WITHOUT_ACL]: false,
};
function futureFlagDefault(flag) {
    return FUTURE_FLAGS_DEFAULTS[flag];
}
exports.futureFlagDefault = futureFlagDefault;
//# sourceMappingURL=data:application/json;base64,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