"""
Module containing functions for converting models to LAMMPS data files.
"""

from pathlib import Path

from ase.io import write, read
from ase.io.lammpsrun import read_lammps_dump_text as ase_read_lammps_dump_text

import numpy as np

from lammpsinputbuilder.types import MoleculeFileFormat, Forcefield, \
    ElectrostaticMethod, GlobalInformation
from lammpsinputbuilder.quantities import LammpsUnitSystem


def extract_elements_from_data(data_path: str) -> str:
    """
    Get the list of elements from the data file to be used by airebo/reax
    """
    with open(data_path, 'r', encoding="utf-8") as f:
        lines = f.readlines()
    i = 0
    ini = -1
    fin = -1
    for line in lines:
        if line[:6] == 'Masses':
            ini = i + 2
        if line[:5] == 'Atoms':
            fin = i - 1
        i += 1

    if ini == -1 or fin == -1:
        raise ValueError(f"Could not find masses and atoms in {data_path}")
    elements = ''
    for line in lines[ini:fin]:
        elements += ' ' + line.split()[-1]
    return elements


def molecule_to_lammps_data_pbc(
        molecule_content: str,
        molecule_file_format: MoleculeFileFormat,
        job_folder: Path,
        data_filename: str) -> GlobalInformation:
    """
    Convert a molecule from XYZ or MOL2 format to a LAMMPS data file.
    TODO: add support for PBC/Shrink
    """

    global_information = GlobalInformation()

    # Save the molecule file in the job folder
    if molecule_file_format == MoleculeFileFormat.MOL2:
        molecule_path = job_folder / 'molecule.MOL2'
    elif molecule_file_format == MoleculeFileFormat.XYZ:
        molecule_path = job_folder / 'molecule.XYZ'
    elif molecule_file_format == MoleculeFileFormat.LAMMPS_DUMP_TEXT:
        molecule_path = job_folder / 'molecule.lammpstrj'
    else:   # Unreachable but kept in case the file format enum changes
        raise ValueError(f"Unknown molecule file format: {molecule_file_format}")

    with open(molecule_path, 'w', encoding="utf-8") as f:
        f.write(molecule_content)

    if molecule_file_format == MoleculeFileFormat.LAMMPS_DUMP_TEXT:
        with open(molecule_path, "r", encoding="utf-8") as f:
            atoms = ase_read_lammps_dump_text(f)
    else:
        atoms = read(molecule_path)

    # Default cell, will be overwritten when rewritting the data file.
    atoms.set_cell([500, 500, 500])

    global_information.set_atoms(atoms)

    # Write a temporary data file which doesn't contain mass information yet
    temp_data_path = job_folder / (str(data_filename) + '.temp')
    write(
        filename=temp_data_path,
        images=atoms,
        format='lammps-data',
        atom_style='full')

    # ASE always write the bounding starting from 0,0,0
    # We have to rewrite it manually to handle cases where positions can be in the negative
    # Doing it now avoir the need to translate back the trajectory later on to
    # match the user input.
    positions = np.array(atoms.get_positions())

    xcoords = positions[:, 0]
    ycoords = positions[:, 1]
    zcoords = positions[:, 2]

    xmin = xcoords.min()
    ymin = ycoords.min()
    zmin = zcoords.min()

    xmax = xcoords.max()
    ymax = ycoords.max()
    zmax = zcoords.max()

    padding = 50.0  # Adding 50A for now

    # TODO: check if writting the masses manually is necessary, ASE write has
    # the masses option.
    chemical_symbols = atoms.get_chemical_symbols()
    masses = atoms.get_masses()
    indexes = np.unique(chemical_symbols, return_index=True)[1]
    masses_u = [masses[index] for index in indexes]
    chemical_symbols_u = [chemical_symbols[index] for index in indexes]

    with open(temp_data_path, 'r', encoding="utf-8") as f:
        lines = f.readlines()

    data_path = job_folder / str(data_filename)
    with open(data_path, 'w', encoding="utf-8") as f:
        f.write('# Generated by LammpsInputBuilder\n')

        # Find the first line which start with a number
        start_offset = 0
        # for i in range(len(lines)):
        for i, line in enumerate(lines):
            # if len(lines[i].split()) > 0 and lines[i].split()[0].isdigit():
            if len(line.split()) > 0 and line.split()[0].isdigit():
                start_offset = i
                break

        if start_offset > len(lines) - 1:
            raise RuntimeError(
                "Could not find the first line in LAMMPS data file")

        # Copy the lines until the line ending with zhi is found
        offset = 0
        bbox_coords = [
            xmin - padding,
            xmax + padding,
            ymin - padding,
            ymax + padding,
            zmin - padding,
            zmax + padding]
        for i in range(start_offset, len(lines)):
            # f.write(lines[i])
            # if len(lines[i].split()) > 0 and lines[i].split()[-1] == 'zhi':
            #    break
            if len(lines[i].split()) > 0 and lines[i].split()[-1] == 'xhi':
                f.write(f"{bbox_coords[0]}\t{bbox_coords[1]} xlo xhi\n")
            elif len(lines[i].split()) > 0 and lines[i].split()[-1] == 'yhi':
                f.write(f"{bbox_coords[2]}\t{bbox_coords[3]} ylo yhi\n")
            elif len(lines[i].split()) > 0 and lines[i].split()[-1] == 'zhi':
                f.write(f"{bbox_coords[4]}\t{bbox_coords[5]} zlo zhi\n")
                break
            else:
                f.write(lines[i])
                offset += 1
        if offset > len(lines) - 1:
            raise RuntimeError("Could not find zhi in LAMMPS data file")
        global_information.set_bbox_coords(bbox_coords)

        # Insert the masses in the data file so it doesn't have to be added
        # separatly in the script file
        element_map = {}
        f.write('\n')
        f.write('Masses\n')
        f.write('\n')
        # for i in range(len(masses_u)):
        for i, mass in enumerate(masses_u):
            #f.write(f'{i + 1} {masses_u[i]} # {chemical_symbols_u[i]}\n')
            f.write(f'{i + 1} {mass} # {chemical_symbols_u[i]}\n')
            element_map[i + 1] = chemical_symbols_u[i]
        f.write('\n')
        f.write('Atoms # full\n')
        f.write('\n')
        global_information.set_element_table(element_map)

        # Find the offset of the Atoms section
        for i in range(offset, len(lines)):
            if lines[i].startswith('Atoms'):
                offset = i + 2
                break
        if offset > len(lines) - 1:
            raise RuntimeError("Could not find Atoms in LAMMPS data file")

        # Now that the masses are added, we can add the rest of the lines.
        for i in range(offset, len(lines)):
            f.write(lines[i])

    return global_information


def molecule_to_lammps_input(
        lammps_script_filename: Path,
        data_file_path: Path,
        job_folder: Path,
        ff_type: Forcefield,
        forcefield_name: str,
        global_information: GlobalInformation,
        electrostatic_method: ElectrostaticMethod) -> Path:
    lammps_script_file_path = job_folder / lammps_script_filename
    with open(lammps_script_file_path, "w", encoding="utf-8") as f:

        # Extract the simulation box
        cell_dims = global_information.get_bbox_dims()
        min_cell_dim = min([cell_dims[0], cell_dims[1], cell_dims[2]])

        # Get back the list of elements
        elements = extract_elements_from_data(data_file_path)

        script_content = "# -*- mode: lammps -*-\n"
        if ff_type == Forcefield.REAX:
            script_content += 'units          real\n'
            global_information.set_unit_style(LammpsUnitSystem.REAL)
        elif ff_type in [Forcefield.AIREBO, Forcefield.REBO, Forcefield.AIREBOM]:
            script_content += 'units          metal\n'
            global_information.set_unit_style(LammpsUnitSystem.METAL)
        else:
            raise NotImplementedError(f"Forcefield {ff_type} not supported")
        script_content += 'atom_style     full\n'
        script_content += 'atom_modify    map hash\n'
        script_content += 'newton         on\n'
        script_content += 'boundary       p p p\n'

        script_content += f'read_data       {data_file_path.name}\n'
        # for i in range(len(indexes)):
        #    script_content += f'mass           {i + 1} {masses_u[i]}\n'

        if ff_type == Forcefield.REAX:
            script_content += 'pair_style     reaxff NULL mincap 1000\n'
            script_content += f'pair_coeff     * * {forcefield_name}{elements}\n'
            if electrostatic_method == ElectrostaticMethod.ACKS2:
                script_content += 'fix            \
                    ReaxFFSpec all acks2/reaxff 1 0.0 10.0 1e-8 reaxff\n'
            elif electrostatic_method == ElectrostaticMethod.QEQ:
                script_content += 'fix            \
                    ReaxFFSpec all qeq/reaxff 1 0.0 10.0 1e-8 reaxff\n'
            else:
                raise NotImplementedError(
                    f"Electrostatic method {electrostatic_method} not supported.")
        elif ff_type == Forcefield.REBO:
            script_content += 'pair_style     rebo\n'
            script_content += f'pair_coeff     * * {forcefield_name}{elements}\n'
        elif ff_type == Forcefield.AIREBO:
            script_content += 'pair_style     airebo 3 1 1\n'
            # script_content+='pair_style     airebo 3 1 0\n'
            script_content += f'pair_coeff     * * {forcefield_name}{elements}\n'
        elif ff_type == Forcefield.AIREBOM:
            script_content += 'pair_style     airebo/morse 3 1 1\n'
            script_content += f'pair_coeff     * * {forcefield_name}{elements}\n'
        # script_content += 'neighbor       2.5 bin\n'
        # 2.5 is too large for small molecule like benzene. Trying to compute a
        # reasonable cell skin based on the simulation box
        script_content += f"neighbor       {min([2.5, min_cell_dim/2])} bin\n"
        script_content += 'neigh_modify   every 1 delay 0 check yes\n'

        if ff_type == Forcefield.REAX:
            script_content += 'compute reax   all pair reaxff\n'
            script_content += 'variable eb    equal c_reax[1]\n'
            script_content += 'variable ea    equal c_reax[2]\n'
            script_content += 'variable elp   equal c_reax[3]\n'
            script_content += 'variable emol  equal c_reax[4]\n'
            script_content += 'variable ev    equal c_reax[5]\n'
            script_content += 'variable epen  equal c_reax[6]\n'
            script_content += 'variable ecoa  equal c_reax[7]\n'
            script_content += 'variable ehb   equal c_reax[8]\n'
            script_content += 'variable et    equal c_reax[9]\n'
            script_content += 'variable eco   equal c_reax[10]\n'
            script_content += 'variable ew    equal c_reax[11]\n'
            script_content += 'variable ep    equal c_reax[12]\n'
            script_content += 'variable efi   equal c_reax[13]\n'
            script_content += 'variable eqeq  equal c_reax[14]\n'
            script_content += '\n'
        elif ff_type in [Forcefield.AIREBO, Forcefield.REBO, Forcefield.AIREBOM]:
            if ff_type == Forcefield.AIREBOM:
                script_content += 'compute reboPair all pair airebo/morse\n'
            elif ff_type == Forcefield.AIREBO:
                script_content += 'compute reboPair all pair airebo\n'
            else:
                script_content += 'compute reboPair all pair rebo\n'
            script_content += 'variable REBO     equal c_reboPair[1]\n'
            script_content += 'variable LJ       equal c_reboPair[2]\n'
            script_content += 'variable TORSION  equal c_reboPair[3]\n'

        f.write(script_content)

        return lammps_script_file_path
