import { IAspect } from 'aws-cdk-lib';
import { IConstruct } from 'constructs';
/**
 * The supported annotation types. Only error will stop deployment of restricted resources.
 */
export declare enum AnnotationType {
    WARNING = "warning",
    ERROR = "error",
    INFO = "info"
}
/**
 * The base aspect properties available to any aspect.
 *
 * JSII doesn't support an Omit when extending interfaces, so we create a base class to extend from.
 * This base class meets the needed properties for all non-base aspects.
 */
export interface AspectPropsBase {
    /**
     * The annotation text to use for the annotation.
     */
    readonly annotationText?: string;
    /**
     * The annotation type to use for the annotation.
     */
    readonly annotationType?: AnnotationType;
}
/**
 * The extended aspect properties available only to the base security aspects.
 *
 * These additional properties shouldn't be changed in aspects that already have clearly defined goals.
 * So, this extended properties interface is applied selectively to the base aspects.
 */
export interface AspectPropsExtended extends AspectPropsBase {
    /**
     * The restricted port. Defaults to restricting all ports and only checking sources.
     *
     *
     * @default undefined
     */
    readonly ports?: number[];
    /**
     * The restricted CIDRs for the given port.
     *
     *
     * @default ['0.0.0.0/0', '::/0']
     */
    readonly restrictedCidrs?: string[];
    /**
     * The restricted source security groups for the given port.
     *
     *
     * @default undefined
     */
    readonly restrictedSGs?: string[];
    /**
     * Whether any source is valid. This will ignore all other restrictions and only check the port.
     *
     *
     * @default false
     */
    readonly anySource?: boolean;
}
/**
 * @function anyInRange
 * Function to check if a number is in a range.
 *
 * Useful for determining if a port number in a security group is in part of a range since rules take a to and from port.
 *
 * @param num The number to check.
 * @param from The starting number of the range.
 * @param to The ending number of the range.
 */
export declare function anyInRange(num: number[], from: number, to: number): boolean;
/**
 * @function annotate
 * Function to annotate a construct node based on a defined annotation type.
 *
 * @param {IConstruct} node The construct node to annotate.
 * @param {string} annotationText The annotation text to use for the annotation.
 * @param {AnnotationType} annotationType The annotation type to use for the annotation.
 */
export declare function annotate(node: IConstruct, annotationText: string | undefined, annotationType: AnnotationType | undefined): void;
/**
 * The arguments for the checkRules function.
 * Extends the IAspectPropsBase interface which includes additional properties that can be used as args.
 */
export interface RuleCheckArgs extends AspectPropsExtended {
    /**
     * The node to check.
     */
    readonly node: IConstruct;
}
/**
 * @function checkRules
 *
 * @param {IRuleCheckArgs} args The arguments for the checkRules function.
 *
 * Function to check a node for security group rules and determine if they breaks the rules of a given aspect.
 */
export declare function checkRules(args: RuleCheckArgs): void;
/**
 * The base class for all security group aspects in the library.
 *
 * By default this will not restrict anything.
 */
export declare class SecurityGroupAspectBase implements IAspect {
    annotationText: string;
    annotationType: AnnotationType;
    ports: number[] | undefined;
    restrictedCidrs: string[] | undefined;
    restrictedSGs: string[] | undefined;
    anySource: boolean;
    constructor(props?: AspectPropsExtended);
    visit(node: IConstruct): void;
}
/**
 * The base aspect to determine if a security group allows inbound traffic from the public internet to any port.
 *
 * This inherits everything from the base SecurityGroupAspectBase class and sets a default set of CIDRS that match allowing all IPs on AWS.
 */
export declare class NoPublicIngressAspectBase extends SecurityGroupAspectBase implements IAspect {
    constructor(props?: AspectPropsBase);
}
/**
 * The same as the base NoPublicIngressAspectBase but with a more descriptive annotation.
 *
 * Blocks the ANY port from the public internet.
 */
export declare class NoPublicIngressAspect extends NoPublicIngressAspectBase implements IAspect {
    constructor(props?: AspectPropsBase);
}
/**
 * Aspect to determine if a security group allows inbound traffic from the public internet to the SSH port.
 */
export declare class NoPublicIngressSSHAspect extends NoPublicIngressAspectBase {
    constructor(props?: AspectPropsBase);
}
/**
 * CIS AWS Foundations Benchmark 4.1
 *
 * CIS recommends that no security group allow unrestricted ingress access to port 22. Removing unfettered connectivity to remote console services, such as SSH, reduces a server's exposure to risk.
 *
 * This aspect uses the NoPublicIngressSSHAspect with an alternate annotation text.
 */
export declare class CISAwsFoundationBenchmark4Dot1Aspect extends NoPublicIngressSSHAspect {
    constructor(props?: AspectPropsBase);
}
/**
 * Aspect to determine if a security group allows inbound traffic from the public internet to the RDP port.
 */
export declare class NoPublicIngressRDPAspect extends NoPublicIngressAspectBase {
    constructor(props?: AspectPropsBase);
}
/**
 * CIS AWS Foundations Benchmark 4.2
 *
 * CIS recommends that no security group allow unrestricted ingress access to port 3389. Removing unfettered connectivity to remote console services, such as RDP, reduces a server's exposure to risk.
 *
 * This aspect uses the NoPublicIngressRDPAspect with an alternate annotation text.
 */
export declare class CISAwsFoundationBenchmark4Dot2Aspect extends NoPublicIngressRDPAspect {
    constructor(props?: AspectPropsBase);
}
/**
 * Restricted common ports based on AWS Config rule https://docs.aws.amazon.com/config/latest/developerguide/restricted-common-ports.html
 */
export declare class AWSRestrictedCommonPortsAspect extends NoPublicIngressAspectBase {
    constructor(props?: AspectPropsBase);
}
/**
 * Aspect to restrict public access to common management ports.
 *
 * 22 - SSH
 * 3389 - RDP
 * 5985 - WinRM
 * 5986 - WinRM HTTPS
 */
export declare class NoPublicIngressCommonManagementPortsAspect extends NoPublicIngressAspectBase {
    constructor(props?: AspectPropsBase);
}
/**
 * Aspect to restrict any access to common management ports.
 *
 * 22 - SSH
 * 3389 - RDP
 * 5985 - WinRM
 * 5986 - WinRM HTTPS
 */
export declare class NoIngressCommonManagementPortsAspect extends SecurityGroupAspectBase {
    constructor(props?: AspectPropsBase);
}
/**
 * Aspect to restrict public access to common relational DB ports.
 *
 * 3306 - MySQL
 * 5432 - PostgreSQL
 * 1521 - Oracle
 * 1433 - SQL Server
 */
export declare class NoPublicIngressCommonRelationalDBPortsAspect extends NoPublicIngressAspectBase {
    constructor(props?: AspectPropsBase);
}
/**
 * Aspect to restrict any access to common relational DB ports.
 *
 * 3306 - MySQL
 * 5432 - PostgreSQL
 * 1521 - Oracle
 * 1433 - SQL Server
 */
export declare class NoIngressCommonRelationalDBPortsAspect extends SecurityGroupAspectBase {
    constructor(props?: AspectPropsBase);
}
/**
 * Aspect to restrict public access to common web ports.
 *
 * 80 - HTTP
 * 443 - HTTPS
 * 8080 - HTTP
 * 8443 - HTTPS
 */
export declare class NoPublicIngressCommonWebPortsAspect extends NoPublicIngressAspectBase {
    constructor(props?: AspectPropsBase);
}
/**
 * Aspect to restrict any access to common web ports.
 *
 * 80 - HTTP
 * 443 - HTTPS
 * 8080 - HTTP
 * 8443 - HTTPS
 */
export declare class NoIngressCommonWebPortsAspect extends SecurityGroupAspectBase {
    constructor(props?: AspectPropsBase);
}
