# This file is part of ast_error_detection.
# Copyright (C) 2025 Badmavasan Kirouchenassamy & Eva Chouaki.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or any later version.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program. If not, see <https://www.gnu.org/licenses/>.

import ast
from .convert_ast_to_custom_node import ast_to_custom_node
from .error_checks import get_customized_error_tags
from .zang_shasha_distance import distance
from .node import Node
from .error_annotation import ErrorAnnotation


def get_primary_code_errors(code1: str, code2: str):
    """
    Given two Python code snippets, compute their differences as error annotations using
    a reimplemented version of the Zhang-Shasha tree edit distance algorithm.

    This function follows these major steps:
      1. Parses the input code strings into Abstract Syntax Trees (ASTs).
      2. Converts each AST into a custom node structure to be processed.
      3. Computes the tree edit distance between the two custom node trees by calling the
         `distance` function, which implements the Zhang-Shasha algorithm.
      4. Processes the list of edit operations (generated by the `distance` call) to
         automatically generate error annotations, including detailed error paths.

    Important Notes:
      - If, in the future, there is a need to change the wrapper that processes the list of
        edit operations (i.e., to improve how error tags and paths are consolidated), the
        modifications should be applied to the block below where the `distance` function is called.
      - An alternative processing function may be imported from another module to handle
        these operations, should advanced or customized error annotation be required.

    Args:
        code1 (str): The first (incorrect) code snippet.
        code2 (str): The second (correct) code snippet.

    Returns:
        tuple: A tuple containing:
            - dist: The computed tree edit distance between the two code snippets. This can be an integer or a float.
            - errors (list): A list of error annotations that detail the differences between the two code snippets.
    """
    # Parse the AST for both code snippets
    symbolic_code1 = ast.parse(code1)
    symbolic_code2 = ast.parse(code2)

    # Convert AST to custom Node structure
    tree1 = ast_to_custom_node(symbolic_code1)
    tree2 = ast_to_custom_node(symbolic_code2)

    # Ensure both trees have valid roots
    if not tree1 or not tree2:
        raise ValueError("Failed to parse one or both code snippets.")

    # Zhang-Shasha Tree Edit Distance computation
    dist, ops = distance(
        tree1[0],
        tree2[0],
        get_children=Node.get_children,
    )

    # --- Critical Section for Future Modifications ---
    # The following call to the `concatenate_all_errors` function is a wrapper that processes the list of edit
    # operations given byt the distance function of Reimplemented Zhang Shasha Algorithm.
    # In such cases, consider calling an  alternative function (e.g. module error_annotation_v2
    # (initialized even in a different file)) that can handle act as a wrapper for specialized
    # processing of Zhan Shasha edit operations.

    error_annotation = ErrorAnnotation()
    errors = error_annotation.concatenate_all_errors(ops)

    # --- End of Critical Section ---

    return dist, errors


def get_typology_based_code_error(incorrect_code : str, correct_code_list: list[str]):
    """
    Compute customized code error annotations by applying a two-step wrapper process.

    This function acts as a secondary wrapper around the core code error extraction process.
    It performs the following steps:
      1. It first calls `get_primary_code_errors`, which internally uses the reimplemented Zhang-Shasha
         tree edit distance algorithm to calculate the differences between two Python code snippets.
         - The `get_primary_code_errors` function generates a detailed list of errors by calling
           the `concatenate_all_errors` method, resulting in a rich set of error data based on
           multiple criteria.
      2. It then applies a typology overlay by calling `get_customized_error_tags` with the
         original error list. This additional processing filters and refines the error tags according
         to a limited set of error types tailored for current services.

    Note:
      - Although the current two-step process (first generating all possible errors and then
        filtering them) is in use, it is not strictly necessary. A unified function that collects
        errors directly from the Zhang-Shasha edit operations could be used.
      - The two-step process is retained because the detailed error detection produced by
        `get_primary_code_errors` are currently more granular than needed for our services.
      - Future modifications may consolidate these steps if a simpler error collection process
        that aligns directly with service requirements is developed.

    Args:
        correct_code_list: Erroneous code snippet
        incorrect_code: List of possible correct code snippets.

    Returns:
        tuple: A tuple containing:
            - dist: The computed tree edit distance between the two code snippets.
            - typology_based_error_tags (list): A filtered list of errors after applying the
              typology rules overlay.
    """

    results = []
    for correct_code in correct_code_list:
        dist, primary_errors = get_primary_code_errors(incorrect_code, correct_code)
        typology_based_error_tags = get_customized_error_tags(primary_errors)
        results.append([dist, typology_based_error_tags])

    return min(results, key=lambda x: x[0])
