import numpy as np
from momlevel.eos.wright import density, drho_dtemp, drho_dsal, alpha, beta

rng = np.random.default_rng(123)

thetao = rng.normal(15.0, 5.0, (5, 5))
so = rng.normal(35.0, 1.5, (5, 5))
pressure = rng.normal(2000.0, 500.0, (5, 5))


def test_wright_density_scalar():
    assert np.allclose(density(18.0, 35.0, 200000.0), 1025.359957453976)


def test_wright_density_3D():
    result = density(thetao, so, pressure)

    reference = np.array(
        [
            [1026.77225958, 1027.8498461, 1025.60122596, 1026.20882763, 1024.87391971],
            [1024.88644021, 1025.19201528, 1026.75381182, 1023.79104261, 1025.88621966],
            [1026.0552234, 1028.37999356, 1026.51881832, 1027.81485778, 1024.42974495],
            [1026.6621244, 1022.59875294, 1028.3130563, 1026.12308827, 1024.81379696],
            [1027.02622475, 1024.91713466, 1023.57331842, 1027.21287132, 1024.2578034],
        ]
    )
    assert np.allclose(result, reference)


def test_wright_drho_dtemp():
    assert np.allclose(drho_dtemp(18.0, 35.0, 200000.0), -0.24680005918175105)


def test_wright_drho_dtemp_3D():
    result = drho_dtemp(thetao, so, pressure)

    reference = np.array(
        [
            [-0.17114065, -0.20603321, -0.27756054, -0.22950261, -0.26015658],
            [-0.24460594, -0.18559082, -0.24683748, -0.1985665, -0.2035094],
            [-0.22453698, -0.14625498, -0.27509023, -0.19063898, -0.26286275],
            [-0.22776725, -0.28269306, -0.19248364, -0.20462358, -0.23325416],
            [-0.10154182, -0.25610488, -0.28481589, -0.27352674, -0.25147033],
        ]
    )

    assert np.allclose(result, reference)


def test_wright_drho_dsal():
    assert np.allclose(drho_dsal(18.0, 35.0, 200000.0), 0.7652676800174607)


def test_wright_drho_dsal_3D():
    result = drho_dsal(thetao, so, pressure)

    reference = np.array(
        [
            [0.78009753, 0.77419326, 0.76091065, 0.76887919, 0.76295099],
            [0.76541526, 0.77612706, 0.76639166, 0.77265234, 0.7732611],
            [0.76962426, 0.78635121, 0.76192714, 0.77704721, 0.76221462],
            [0.76949764, 0.7579565, 0.77704819, 0.77322653, 0.76724224],
            [0.79503107, 0.76360797, 0.7583952, 0.76264792, 0.76385609],
        ]
    )

    assert np.allclose(result, reference)


def test_wright_alpha():
    assert np.allclose(alpha(18.0, 35.0, 200000.0), 0.0002406960183958898)


def test_wright_alpha_3D():
    result = alpha(thetao, so, pressure)

    reference = np.array(
        [
            [
                1.66678292e-04,
                2.00450692e-04,
                2.70632031e-04,
                2.23641234e-04,
                2.53842521e-04,
            ],
            [
                2.38666386e-04,
                1.81030305e-04,
                2.40405708e-04,
                1.93952172e-04,
                1.98374245e-04,
            ],
            [
                2.18835177e-04,
                1.42218806e-04,
                2.67983623e-04,
                1.85479884e-04,
                2.56594220e-04,
            ],
            [
                2.21852198e-04,
                2.76445731e-04,
                1.87183889e-04,
                1.99414264e-04,
                2.27606383e-04,
            ],
            [
                9.88697468e-05,
                2.49878625e-04,
                2.78256465e-04,
                2.66280487e-04,
                2.45514686e-04,
            ],
        ]
    )

    assert np.allclose(result, reference)


def test_wright_beta():
    assert np.allclose(beta(18.0, 35.0, 200000.0), 0.0007463405162784603)


def test_wright_beta_3D():
    result = beta(thetao, so, pressure)

    reference = np.array(
        [
            [0.00075976, 0.00075322, 0.00074192, 0.00074924, 0.00074443],
            [0.00074683, 0.00075706, 0.00074642, 0.0007547, 0.00075375],
            [0.00075008, 0.00076465, 0.00074224, 0.00075602, 0.00074404],
            [0.00074951, 0.00074121, 0.00075565, 0.00075354, 0.00074867],
            [0.00077411, 0.00074504, 0.00074093, 0.00074244, 0.00074577],
        ]
    )

    assert np.allclose(result, reference)
