import time
import json
import os
import re
from concurrent.futures.thread import ThreadPoolExecutor

from Tea.exceptions import TeaException
from computenestcli.common.decorator import retry_on_exception
from computenestcli.service.artifact import ArtifactService
from computenestcli.service.file import FileService
from computenestcli.service.credentials import CredentialsService
from computenestcli.common.util import Util
from computenestcli.processor.image import ImageProcessor
from computenestcli.common import constant
from computenestcli.common.context import Context
from computenestcli.base_log import get_user_logger

user_logger = get_user_logger()
TRUE = 'True'
FALSE = 'False'
UPDATE_ARTIFACT = 'UpdateArtifact'
AVAILABLE = 'Available'
DELIVERING = 'Delivering'
DATA = 'data'
RESULT = 'result'
IMAGE_BUILDER = 'ImageBuilder'
ACR_IMAGE_BUILDER = 'AcrImageBuilder'
HELM_CHART_BUILDER = 'HelmChartBuilder'
ARTIFACT = 'artifact'
DRAFT = 'draft'
REGION_ID = 'regionId'
COMMAND_CONTENT = 'CommandContent'
ENTITY_ALREADY_EXIST_DRAFT_ARTIFACT = 'EntityAlreadyExist.DraftArtifact'
ARTIFACT_VERSION_NOT_FOUND = 'EntityNotExist.ArtifactVersion'
ARTIFACT_STATUS_NOT_SUPPORT_OPERATION = 'OperationDenied'


class ArtifactProcessor:

    def __init__(self, context):
        self.context = context

    def _get_file_artifact_url(self, artifact_name):
        get_artifact_data = json.loads(ArtifactService.get_artifact(self.context, artifact_name).body.artifact_property)
        url = get_artifact_data.get(constant.URL)
        return url

    def _create_image_from_image_builder(self, data, artifact_data):
        data_image = data.get(IMAGE_BUILDER)
        id_image = artifact_data.get(constant.ARTIFACT_PROPERTY).get(constant.IMAGE_ID)
        id_image_match = Util.regular_expression(id_image)
        data_image_oos = data_image.get(id_image_match[1])
        region_id = data_image_oos[constant.REGION_ID]
        command_content = data_image_oos[COMMAND_CONTENT]
        # 识别命令语句中的占位符
        pattern = r'\$\{Artifact\.(.*?)\.ArtifactProperty\.Url\}'
        matches = re.findall(pattern, command_content)
        if matches:
            artifact_key = matches[0].strip()
            artifact_name = data[constant.ARTIFACT].get(artifact_key, {}).get(constant.ARTIFACT_NAME)
            url = self._get_file_artifact_url(artifact_name)
            parts = url.split("/")
            # 截取文件部署物下载链接的后半部分
            artifact_url = parts[-2] + "/" + parts[-1]
            placeholder = f'${{Artifact.{artifact_key}.ArtifactProperty.Url}}'
            # 替换真正的url
            command_content = command_content.replace(placeholder, artifact_url)
            data_image_oos[COMMAND_CONTENT] = command_content
        data_image_oos = Util.lowercase_first_letter(data_image_oos)
        region_id_image = data_image_oos[REGION_ID]
        image_context = Context(region_id_image, self.context.credentials)
        image_processor = ImageProcessor(image_context)
        return region_id, image_processor.process_image(data_image_oos)

    '''
    根据acrImageBuilder构造容器镜像部署物
    service_config为整个服务的配置，对应的acrImageBuilder结构如下：
    AcrImageBuilder:
      AcrImage_VcControllerManager:
        BuilderType: DockerRepo
        DockerRepoUrl: 'volcanosh/vc-controller-manager:v1.9.0'
        # DockerFilePath: 'resources/artifact_resources/acr_image/Dockerfile'
        RepoName: volcanosh/vc-controller-manager
        Tag: v1.9.0
        
    artifact_config对应的是单个部署物的配置，只包含value部分，对应的结构如下：
    Artifact_AcrImage_VcControllerManager:
      ArtifactType: AcrImage
      ArtifactName: VcControllerManager
      Description: VcControllerManager AcrImage部署物
      ArtifactProperty:
        RepoName: ${AcrImageBuilder.AcrImage_VcControllerManager.RepoName}
        Tag: ${AcrImageBuilder.AcrImage_VcControllerManager.Tag}
    '''

    def _create_acrimage_from_acrimage_builder(self, service_config, artifact_config, file_path_config):
        try:
            artifact_type = artifact_config.get(constant.ARTIFACT_TYPE)
            image_processor = ImageProcessor(self.context)
            acr_image_builder_dict = service_config.get(ACR_IMAGE_BUILDER)
            repo_name_var = artifact_config.get(constant.ARTIFACT_PROPERTY).get(constant.REPO_NAME)
            acr_image_builder_key = Util.regular_expression(repo_name_var)[1]
            acr_image_builder = acr_image_builder_dict.get(acr_image_builder_key)
            file_path = os.path.dirname(file_path_config)
            acr_image_name = acr_image_builder[constant.REPO_NAME]
            acr_image_tag = acr_image_builder[constant.TAG]
            build_type = acr_image_builder.get(constant.BUILD_TYPE)
            build_args = acr_image_builder.get(constant.DOCKER_BUILD_ARGS)
            if build_type == constant.DOCKER_REPO_TYPE:
                docker_repo_url = acr_image_builder[constant.DOCKER_REPO_URL]
                image_processor.process_acr_image(acr_image_name, acr_image_tag, file_path, docker_repo_url, build_type,
                                                  build_args, dockerfile_path=None)
                repo_id = self._get_repo_id(artifact_type, acr_image_name)
            elif acr_image_builder.get(constant.DOCKER_FILE_PATH):
                dockerfile_path = acr_image_builder[constant.DOCKER_FILE_PATH]
                file_path = os.path.abspath(file_path)
                file_path = os.path.dirname(file_path)
                image_processor.process_acr_image(acr_image_name, acr_image_tag, file_path,
                                                  None, constant.DOCKER_FILE_TYPE, build_args, dockerfile_path)
                repo_id = self._get_repo_id(artifact_type, acr_image_name)
            else:
                # 对应的容器镜像已上传到计算巢，校验是否存在
                repo_id = self._get_exist_acr_image_repo_id(artifact_type, acr_image_name, acr_image_tag)
        except FileNotFoundError as e:
            # 使用 logger.exception() 记录错误信息
            user_logger.error("An error occurred while trying to open the file.")
            raise
        return acr_image_name, repo_id, acr_image_tag

    def _get_repo_id(self, artifact_type, acr_image_name):
        response_body = ArtifactService.list_acr_image_repositories(self.context, artifact_type, acr_image_name).body
        if response_body and response_body.repositories:
            for repository in response_body.repositories:
                if repository.repo_name == acr_image_name:
                    return repository.repo_id
        raise Exception(f"No Repo found,repo_name: {acr_image_name}")

    def _create_helmchart_from_helmchart_builder(self, service_config, artifact_config, file_path_config):
        artifact_type = artifact_config.get(constant.ARTIFACT_TYPE)
        image_processor = ImageProcessor(self.context)
        helm_chart_data = service_config.get(HELM_CHART_BUILDER)
        helm_chart_var = artifact_config.get(constant.ARTIFACT_PROPERTY).get(constant.REPO_NAME)
        helm_chart_match = Util.regular_expression(helm_chart_var)
        helm_chart_builder = helm_chart_data.get(helm_chart_match[1])
        file_path = os.path.dirname(file_path_config)
        helm_chart_repo_name = helm_chart_builder[constant.REPO_NAME]
        helm_chart_tag = helm_chart_builder[constant.TAG]
        build_type = helm_chart_builder.get(constant.BUILD_TYPE)
        if build_type == constant.HELM_REPO_TYPE:
            helm_repo_url = helm_chart_builder[constant.HELM_REPO_URL]
            image_processor.process_helm_chart(file_path, helm_chart_repo_name, helm_chart_tag,
                                               helm_repo_url, build_type)
            helm_chart_repo_id = self._get_repo_id(artifact_type, helm_chart_repo_name)
        elif helm_chart_builder.get(constant.HELM_CHART_PATH):
            helm_chart_file_path = os.path.join(os.path.dirname(file_path_config),
                                                helm_chart_builder[constant.HELM_CHART_PATH])
            image_processor.process_helm_chart(helm_chart_file_path, helm_chart_repo_name, helm_chart_tag,
                                               None, constant.HELM_PACKAGE_TYPE)
            helm_chart_repo_id = self._get_repo_id(artifact_type, helm_chart_repo_name)
        # 若不存在认为线上仓库已存在相关容器镜像
        else:
            helm_chart_repo_id = self._get_exist_acr_image_repo_id(artifact_type, helm_chart_repo_name, helm_chart_tag)

        return helm_chart_repo_name, helm_chart_repo_id, helm_chart_tag

    def _get_exist_acr_image_repo_id(self, artifact_type, repo_name, tag):
        repo_id = self._get_repo_id(artifact_type, repo_name)
        # 查到repo_id后检索所有已存在的tag，检查提供的tag是否存在
        tags = ArtifactService.list_acr_image_tags(self.context, repo_id, artifact_type).body.images
        tag_values = [item.tag for item in tags]
        if tag not in tag_values:
            raise ValueError(
                f"Invalid or non-existent {artifact_type} version provided. repo_name:{repo_name}, tag:{tag}.")
        return repo_id

    def _replace_artifact_file_path_with_url(self, file_path):
        artifact_credentials = CredentialsService.get_artifact_repository_credentials(self.context, constant.FILE)
        file_artifact_url = FileService.put_file(artifact_credentials, file_path, ARTIFACT)
        return file_artifact_url

    def _release_artifact(self, artifact_id, artifact_name):
        ArtifactService.release_artifact(self.context, artifact_id)
        while True:
            # 定时检测部署物发布状态
            try:
                data_response = ArtifactService.get_artifact(self.context, artifact_name, 'draft')
                artifact_status = data_response.body.status
                if artifact_status == DELIVERING:
                    user_logger.info('The artifact is being released...')
            except TeaException as e:
                if e.code == ARTIFACT_VERSION_NOT_FOUND:
                    user_logger.info('The release is complete')
                    break
                else:
                    raise
            time.sleep(25)

    def get_artifact_detail(self, artifact_id, artifact_version):
        response = ArtifactService.get_artifact(self.context, '', artifact_version, artifact_id)
        artifact_type = response.body.artifact_type
        artifact_name = response.body.name
        description = response.body.description
        support_region_ids = response.body.support_region_ids
        artifact_property = json.loads(response.body.artifact_property)
        parameters = {
            constant.ARTIFACT_TYPE: artifact_type,
            constant.ARTIFACT_NAME: artifact_name,
            constant.DESCRIPTION: description,
            constant.ARTIFACT_PROPERTY: artifact_property,
            constant.SUPPORT_REGION_IDS: support_region_ids
        }
        return parameters

    @Util.measure_time
    def process(self, data_config, file_path_config, update_artifact=False, version_name=''):
        artifact_dict = data_config.get(constant.ARTIFACT)
        with ThreadPoolExecutor(max_workers=10, thread_name_prefix="process_image") as executor:
            future_list = []
            for artifact_config in artifact_dict.values():
                if constant.ARTIFACT_NAME not in artifact_config:
                    continue
                future = executor.submit(self._process_single_artifact, data_config, file_path_config,
                                         artifact_config, version_name, update_artifact)
                future_list.append(future)
            [future.result() for future in future_list]
        return artifact_dict

    def _process_single_artifact(self, data_config, file_path_config, artifact_config, version_name, update_artifact):
        artifact_type = artifact_config.get(constant.ARTIFACT_TYPE)
        artifact_name = artifact_config.get(constant.ARTIFACT_NAME)
        if version_name:
            artifact_config[constant.VERSION_NAME] = version_name
        artifact_config[UPDATE_ARTIFACT] = update_artifact
        artifact_data_list = ArtifactService.list_artifact(self.context, artifact_name)
        if len(artifact_data_list.body.artifacts) == 0:
            if artifact_type == constant.FILE:
                # 将相对路径替换成绝对路径
                file_path = os.path.join(os.path.dirname(file_path_config),
                                         artifact_config.get(constant.ARTIFACT_PROPERTY).get(constant.URL))
                # 将文件部署物的本地路径替换成Url
                artifact_config[constant.ARTIFACT_PROPERTY][
                    constant.URL] = self._replace_artifact_file_path_with_url(file_path)
            elif artifact_type == constant.ECS_IMAGE:
                if IMAGE_BUILDER in data_config:
                    # 利用oos模版创建镜像
                    region_id, image_id = self._create_image_from_image_builder(data_config, artifact_config)
                    artifact_config[constant.ARTIFACT_PROPERTY][constant.REGION_ID] = region_id
                    artifact_config[constant.ARTIFACT_PROPERTY][constant.IMAGE_ID] = image_id
            elif artifact_type == constant.ACR_IMAGE:
                acr_image_name, repo_id, acr_image_tag = self._create_acrimage_from_acrimage_builder(
                    data_config, artifact_config, file_path_config)
                artifact_config[constant.ARTIFACT_PROPERTY][constant.REPO_NAME] = acr_image_name
                artifact_config[constant.ARTIFACT_PROPERTY][constant.TAG] = acr_image_tag
                artifact_config[constant.ARTIFACT_PROPERTY][constant.REPO_ID] = repo_id
            elif artifact_type == constant.HELM_CHART:
                helm_chart_name, helm_chart_id, helm_chart_tag = self._create_helmchart_from_helmchart_builder(
                    data_config, artifact_config, file_path_config)
                artifact_config[constant.ARTIFACT_PROPERTY][constant.REPO_NAME] = helm_chart_name
                artifact_config[constant.ARTIFACT_PROPERTY][constant.TAG] = helm_chart_tag
                artifact_config[constant.ARTIFACT_PROPERTY][constant.REPO_ID] = helm_chart_id
            data_create_artifact = ArtifactService.create_artifact(self.context, artifact_config)
            artifact_id = data_create_artifact.body.artifact_id
            current_time = Util.get_current_time()
            user_logger.info("===========================")
            user_logger.info("Successfully created a new artifact!")
            user_logger.info(f"The artifact name: {artifact_name}")
            user_logger.info(f"The artifact id: {artifact_id}")
            user_logger.info(f"Completion time: {current_time}")
            user_logger.info("===========================")
            if artifact_type == constant.ACR_IMAGE or artifact_type == constant.HELM_CHART:
                self._release_artifact(artifact_id, artifact_name)
        elif not artifact_config.get(UPDATE_ARTIFACT):
            artifact_id = artifact_data_list.body.artifacts[0].artifact_id
            current_time = Util.get_current_time()
            user_logger.info("===========================")
            user_logger.info("No need to update the artifact")
            user_logger.info(f"The artifact name: {artifact_name}")
            user_logger.info(f"The artifact id: {artifact_id}")
            user_logger.info(f"Completion time: {current_time}")
            user_logger.info("===========================")
        else:
            if artifact_type == constant.FILE:
                file_url_existed = self._get_file_artifact_url(artifact_name)
                # 将相对路径替换成绝对路径
                file_path = os.path.join(os.path.dirname(file_path_config),
                                         artifact_config.get(constant.ARTIFACT_PROPERTY).get(constant.URL))
                result_artifact = FileService.check_file_repeat(file_url_existed, file_path)
                # 检查文件部署物是否重复，重复则不再上传，使用现有Url
                if result_artifact:
                    artifact_config[constant.ARTIFACT_PROPERTY][constant.URL] = file_url_existed.split('?')[0]
                else:
                    artifact_config[constant.ARTIFACT_PROPERTY][
                        constant.URL] = self._replace_artifact_file_path_with_url(file_path)
            elif artifact_type == constant.ECS_IMAGE:
                if IMAGE_BUILDER in data_config:
                    # 利用oos模版创建镜像
                    region_id, image_id = self._create_image_from_image_builder(data_config, artifact_config)
                    artifact_config[constant.ARTIFACT_PROPERTY][constant.REGION_ID] = region_id
                    artifact_config[constant.ARTIFACT_PROPERTY][constant.IMAGE_ID] = image_id
            elif artifact_type == constant.ACR_IMAGE:
                acr_image_name, repo_id, acr_image_tag = self._create_acrimage_from_acrimage_builder(
                    data_config, artifact_config, file_path_config)
                artifact_config[constant.ARTIFACT_PROPERTY][constant.REPO_NAME] = acr_image_name
                artifact_config[constant.ARTIFACT_PROPERTY][constant.TAG] = acr_image_tag
                artifact_config[constant.ARTIFACT_PROPERTY][constant.REPO_ID] = repo_id
            elif artifact_type == constant.HELM_CHART:
                helm_chart_name, helm_chart_id, helm_chart_tag = self._create_helmchart_from_helmchart_builder(
                    data_config, artifact_config, file_path_config)
                artifact_config[constant.ARTIFACT_PROPERTY][constant.REPO_NAME] = helm_chart_name
                artifact_config[constant.ARTIFACT_PROPERTY][constant.TAG] = helm_chart_tag
                artifact_config[constant.ARTIFACT_PROPERTY][constant.REPO_ID] = helm_chart_id
            artifact_id = artifact_data_list.body.artifacts[0].artifact_id
            self._create_or_update_artifact(artifact_config, artifact_id)
            current_time = Util.get_current_time()
            user_logger.info("===========================")
            user_logger.info("Successfully updated the artifact!")
            user_logger.info(f"The artifact name: {artifact_name}")
            user_logger.info(f"The artifact id: {artifact_id}")
            user_logger.info(f"Completion time: {current_time}")
            user_logger.info("===========================")
            if artifact_type == constant.ACR_IMAGE or artifact_type == constant.HELM_CHART:
                self._release_artifact(artifact_id, artifact_name)
        data_response = ArtifactService.list_artifact(self.context, artifact_name)
        max_version = int(data_response.body.artifacts[0].max_version)
        if artifact_config.get(constant.ARTIFACT_PROPERTY).get(constant.VERSION):
            version = artifact_config.get(constant.ARTIFACT_PROPERTY).get(constant.VERSION)
            if version == constant.BACK_VERSION:
                artifact_version = max_version - 1
            elif int(version) <= max_version:
                artifact_version = version
            else:
                raise ValueError("Wrong version")
        else:
            artifact_version = max_version
        artifact_config[constant.ARTIFACT_ID] = artifact_id
        artifact_config[constant.ARTIFACT_VERSION] = artifact_version

    @retry_on_exception(max_retries=10, delay=2, backoff=2, exceptions=(TeaException,))
    def _create_or_update_artifact(self, artifact_data, artifact_id):
        try:
            ArtifactService.create_artifact(self.context, artifact_data, artifact_id)
        except TeaException as e:
            if e.code == ENTITY_ALREADY_EXIST_DRAFT_ARTIFACT:
                try:
                    ArtifactService.update_artifact(self.context, artifact_data, artifact_id)
                except TeaException as e:
                    if e.code == ARTIFACT_STATUS_NOT_SUPPORT_OPERATION:
                        raise
            else:
                raise
