"""A SegmentationExtractor for Minian.

Classes
-------
MinianSegmentationExtractor
    A class for extracting segmentation from Minian output.
"""

import warnings
from pathlib import Path
from typing import Optional

import numpy as np
import pandas as pd
import zarr

from ...extraction_tools import FloatType, PathType
from ...segmentationextractor import SegmentationExtractor


class MinianSegmentationExtractor(SegmentationExtractor):
    """A SegmentationExtractor for Minian.

    This class inherits from the SegmentationExtractor class, having all
    its functionality specifically applied to the dataset output from
    the 'Minian' ROI segmentation method.

    Users can extract key information such as ROI traces, image masks,
    and timestamps from the output of the Minian pipeline
    (https://minian.readthedocs.io/en/stable/pipeline/notebook_5.html).
    More specifically, it reads the .zarr output files generated by Minian:
    A.zarr: Spatial footprints of cells. Should have dimensions ("unit_id", "height", "width"). --> image_masks
    C.zarr: Temporal components of cells. Should have dimensions "frame" and "unit_id". --> roi_response_denoised
    b.zarr: Spatial footprint of background. Should have dimensions ("height", "width"). --> background_image_masks
    f.zarr: Temporal dynamic of background. Should have dimension "frame". --> roi_response_neuropil
    b0.zarr: Baseline fluorescence for each cell. Should have dimensions ("frame", "unit_id") and same shape as C --> roi_response_baseline
    c0.zarr: Initial calcium decay, in theory triggered by calcium events happened before the recording starts. Should have dimensions ("frame", "unit_id") and same shape as C
    S.zarr: Deconvolved spikes for each cell. Should have dimensions ("frame", "unit_id") and same shape as C --> roi_response_deconvolved
    max_proj.zarr: the maximum projection --> summary_image

    Key features:
    - Extracts fluorescence traces (denoised, baseline, neuropil, deconvolved) for each ROI.
    - Retrieves ROI masks and background components.
    - Provides access to timestamps corresponding to calcium traces.
    - Retrieves maximum projection image.

    """

    extractor_name = "MinianSegmentation"

    def __init__(self, folder_path: PathType):
        """Initialize a MinianSegmentationExtractor instance.

        Parameters
        ----------
        folder_path: str or Path
            The location of the folder containing minian .zarr output.
        """
        SegmentationExtractor.__init__(self)
        self.folder_path = Path(folder_path)
        self._roi_response_denoised = self._read_trace_from_zarr_field(field="C")
        self._roi_response_baseline = self._read_trace_from_zarr_field(field="b0")
        self._roi_response_neuropil = self._read_trace_from_zarr_field(field="f")
        self._roi_response_deconvolved = self._read_trace_from_zarr_field(field="S")
        self._image_maximum_projection = np.array(self._read_zarr_group("/max_proj.zarr/max_proj"))
        self._image_masks = self._read_roi_image_mask_from_zarr()
        self._background_image_masks = self._read_background_image_mask_from_zarr()
        # Check for spatial-temporal component mismatches
        if (
            self._image_masks is not None
            and self._roi_response_denoised is None
            and self._roi_response_deconvolved is None
            and self._roi_response_baseline is None
        ):
            raise ValueError(
                "Spatial components (A.zarr) are available but no temporal components (C.zarr, S.zarr, b0.zarr) are associated. "
                "This means ROI masks exist but without any corresponding fluorescence traces."
            )

        if self._background_image_masks is not None and self._roi_response_neuropil is None:
            raise ValueError(
                "Background spatial components (b.zarr) are available but no background temporal component (f.zarr) is associated. "
                "This means background masks exist but without corresponding temporal dynamics."
            )

    def _read_zarr_group(self, zarr_group: str):
        """Read the zarr group.

        Parameters
        ----------
        zarr_group : str
            The zarr group path to read

        Returns
        -------
        zarr.Group or None
            The zarr group object specified by self.folder_path and zarr_group,
            or None if the group is not found.
        """
        if zarr_group not in zarr.open(self.folder_path, mode="r"):
            warnings.warn(f"Group '{zarr_group}' not found in the Zarr store.", UserWarning)
            return None
        else:
            return zarr.open(str(self.folder_path) + f"/{zarr_group}", "r")

    def _read_roi_image_mask_from_zarr(self):
        """Read the image masks from the zarr output.

        Returns
        -------
        image_masks: numpy.ndarray
            The image masks for each ROI.
        """
        dataset = self._read_zarr_group("/A.zarr")
        if dataset is None or "A" not in dataset:
            raise ValueError("No image masks found in A.zarr dataset.")
        else:
            # Transpose to (height, width, weight) format because A.zarr is stored as (weight, height, width)
            return np.transpose(dataset["A"], (1, 2, 0))

    def _read_background_image_mask_from_zarr(self):
        """Read the background image masks (b) from the zarr output.

        b.zarr structure is expected to be like:
        ├── b.zarr
        │   ├── animal () <U7
        │   ├── b (608, 608) float64
        │   ├── height (608,) int64
        │   ├── session () <U14
        │   ├── session_id () <U9
        │   ├── unit_id () int64
        │   └── width (608,) in

        Returns
        -------
        image_masks: numpy.ndarray
            The image masks for each background components.
        """
        dataset = self._read_zarr_group("/b.zarr")
        if dataset is None or "b" not in dataset:
            raise ValueError("No background image masks found in b.zarr dataset.")
        else:
            return np.expand_dims(dataset["b"], axis=2)

    def _read_trace_from_zarr_field(self, field: str):
        """Read the traces specified by the field from the zarr object.

        Parameters
        ----------
        field: str
            The field to read from the zarr object.

        Returns
        -------
        trace: numpy.ndarray
            The traces specified by the field.
        """
        dataset = self._read_zarr_group(f"/{field}.zarr")

        if dataset is None or field not in dataset:
            return None
        elif dataset[field].ndim == 2:
            return np.transpose(dataset[field])
        elif dataset[field].ndim == 1:
            return np.expand_dims(dataset[field], axis=1)

    def _read_timestamps_from_csv(self) -> np.ndarray:
        """Extract timestamps corresponding to frame indexes (not downsampled) of the stored denoised trace.

        Returns
        -------
        np.ndarray
            The timestamps of the denoised trace.
        """

    def get_native_timestamps(self, start_sample: Optional[int] = None, end_sample: Optional[int] = None) -> np.ndarray:
        """
        Get the native format timestamps from the CSV file.

        Parameters
        ----------
        start_sample : int, optional
            The starting sample index. If None, starts from the beginning.
        end_sample : int, optional
            The ending sample index. If None, goes to the end.

        Returns
        -------
        np.ndarray
            The native timestamps in seconds.
        """
        csv_file = self.folder_path / "timeStamps.csv"
        df = pd.read_csv(csv_file)
        frame_indexes = self._read_zarr_group("/C.zarr/frame")
        filtered_df = df[df["Frame Number"].isin(frame_indexes)]

        native_timestamps = filtered_df["Time Stamp (ms)"].to_numpy() * 1e-3

        # Set defaults
        if start_sample is None:
            start_sample = 0
        if end_sample is None:
            end_sample = self.get_num_samples()

        return native_timestamps[start_sample:end_sample]

    def sample_indices_to_time(self, sample_indices: FloatType | np.ndarray) -> FloatType | np.ndarray:
        """Convert user-inputted sample indices to times with units of seconds.

        Parameters
        ----------
        sample_indices: int or array-like
            The sample indices to be converted to times.

        Returns
        -------
        times: float or array-like
            The corresponding times in seconds.
        """
        # Default implementation
        if self._times is None:
            self._times = self.get_native_timestamps()

        return self._times[sample_indices]

    def has_time_vector(self) -> bool:
        if self._times is None:
            self._times = self.get_native_timestamps()
        return True  # The MinianSegmentationExtractor always has a time vector from the timestamps CSV.

    def get_frame_shape(self) -> tuple[int, int]:
        """Get the frame shape (height, width) from the zarr dataset.

        Returns
        -------
        tuple[int, int]
            The frame shape as (height, width).

        Raises
        ------
        ValueError
            If the A.zarr dataset or height/width dimensions are not found.
        """
        # First try to get frame shape from the zarr dataset
        dataset = self._read_zarr_group("/A.zarr")
        if dataset is None or "height" not in dataset or "width" not in dataset:
            # Fallback: try to infer from image masks if available
            if self._image_masks is not None:
                height, width, _ = self._image_masks.shape
                return (height, width)
            else:
                raise ValueError(
                    "Cannot determine frame shape: height/width dimensions not found, "
                    "and no image masks are available to infer frame shape."
                )

        height = dataset["height"].shape[0]
        width = dataset["width"].shape[0]
        return (height, width)

    def get_roi_ids(self) -> list:
        """Get a list of ROI ids from the A.zarr dataset.

        Returns
        -------
        roi_ids: list
            List of ROI ids.

        Raises
        ------
        Warnings
            If the A.zarr dataset is not found or does not contain 'unit_id'.
        """
        dataset = self._read_zarr_group("/A.zarr")
        if dataset is None or "unit_id" not in dataset:
            warnings.warn("No ROI ids found in A.zarr dataset. A range of indexes will be used as ROI ids", UserWarning)
            return super().get_roi_ids()  # Fallback to default implementation
        return list(dataset["unit_id"])

    def get_accepted_list(self) -> list:
        """Get a list of accepted ROI ids.

        Returns
        -------
        accepted_list: list
            List of accepted ROI ids.
        """
        return self.get_roi_ids()

    def get_rejected_list(self) -> list:
        """Get a list of rejected ROI ids.

        Returns
        -------
        rejected_list: list
            List of rejected ROI ids.
        """
        return list()

    def get_traces_dict(self) -> dict:
        """Get traces as a dictionary with key as the name of the ROiResponseSeries.

        Returns
        -------
        roi_response_dict: dict
            dictionary with key, values representing different types of RoiResponseSeries:
                Raw Fluorescence, DeltaFOverF, Denoised, Neuropil, Deconvolved, Background, etc.
        """
        return dict(
            denoised=self._roi_response_denoised,
            baseline=self._roi_response_baseline,
            neuropil=self._roi_response_neuropil,
            deconvolved=self._roi_response_deconvolved,
        )

    def get_images_dict(self) -> dict:
        """Get images as a dictionary with key as the name of the ROIResponseSeries.

        Returns
        -------
        _roi_image_dict: dict
            dictionary with key, values representing different types of Images used in segmentation:
                Mean, Correlation image
        """
        return dict(
            mean=self._image_mean,
            correlation=self._image_correlation,
            maximum_projection=self._image_maximum_projection,
        )

    def _get_session_id(self) -> str:
        """Get the session id from the A.zarr group.

        Returns
        -------
        session_id: str
            The session id.

        Raises
        ------
        Warnings
            If the A.zarr dataset is not found or does not contain 'session_id'.
        """
        dataset = self._read_zarr_group("/A.zarr")
        if dataset is None or "session_id" not in dataset:
            warnings.warn(
                "No 'session_id' found in A.zarr dataset. Returning an empty string for 'session_id'.", UserWarning
            )
            return ""
        return dataset["session_id"][()]

    def _get_subject_id(self) -> str:
        """Get the subject id from the A.zarr group.

        Returns
        -------
        subject_id: str
            The subject id.session_id

        Raises
        ------
        Warnings
            If the A.zarr dataset is not found or does not contain 'animal'.
        """
        dataset = self._read_zarr_group("/A.zarr")
        if dataset is None or "animal" not in dataset:
            warnings.warn(
                "No 'animal' found in A.zarr dataset. Returning an empty string for 'subject_id'.", UserWarning
            )
            return ""
        return dataset["animal"][()]
