#!/usr/bin/env python3

from setuptools import setup, find_packages

# Read the README file for long description
try:
    with open("README.md", "r", encoding="utf-8") as fh:
        long_description = fh.read()
except FileNotFoundError:
    long_description = "pysera - A Python library for radiomics feature extraction with multiprocessing support"


# Read requirements from requirements.txt
def read_requirements(filename="requirements-library.txt"):
    """Read requirements from requirements.txt, ignoring comments and blank lines."""
    with open(filename, "r", encoding="utf-8") as f:
        lines = f.readlines()
    return [
        line.strip()
        for line in lines
        if line.strip() and not line.strip().startswith("#")
    ]


setup(
    name="pysera",
    version="1.0.1",
    author="Mohammad R. Salmanpour, Amir Hossein Pouria",
    author_email="M.salmanpoor66@gmail.com",
    description="pysera (Python-based Standardized Extraction for Radiomics Analysis) is a comprehensive Python library for radiomics feature extraction from medical imaging data. It provides a simple, single-function API with built-in multiprocessing support and comprehensive report capabilities.",
    long_description=long_description,
    long_description_content_type="text/markdown",
    url="https://github.com/MohammadRSalmanpour/PySERA",
    packages=find_packages(),
    license="MIT",
    classifiers=[
        "Development Status :: 5 - Production/Stable",
        "Intended Audience :: Science/Research",
        "Intended Audience :: Healthcare Industry",
        "Topic :: Scientific/Engineering :: Medical Science Apps.",
        "Topic :: Scientific/Engineering :: Image Processing",
        "Programming Language :: Python :: 3",
        "Programming Language :: Python :: 3.8",
        "Programming Language :: Python :: 3.9",
        "Programming Language :: Python :: 3.10",
        "Programming Language :: Python :: 3.11",
        "Operating System :: OS Independent",
    ],
    python_requires=">=3.8",
    install_requires=read_requirements(),
    extras_require={
        "dev": [
            "pytest>=7.0",
            "pytest-cov>=4.0",
            "black>=23.0",
            "flake8>=6.0",
            "mypy>=1.0",
        ],
        "full": [
            "vtk>=9.5.0",
            "itk>=5.4.4.post1",
            "rt-utils>=1.2.7",
            "dcmrtstruct2nii>=5",
            "ReliefF>=0.1.2",
            "sklearn-relief>=1.0.0b2",
            "scikit-optimize>=0.10.2",
            "scikit-learn-extra>=0.3.0",
            "kmodes>=0.12.2"
        ]
    },
    entry_points={
        "console_scripts": [
            "pysera=pysera.cli:main",
        ],
    },
    include_package_data=True,
    package_data={
        "pysera": ["config/*.yaml", "config/*.json"],
    },
    keywords="medical-imaging Standardized-Radiomics-Feature-Extraction Quantitative-Analysis IBSI_Evaluation-Standardization "
             "healthcare",
    project_urls={
        "Bug Reports": "https://github.com/MohammadRSalmanpour/PySERA/issues",
        "Source": "https://github.com/MohammadRSalmanpour/PySERA",
        "Documentation": "https://pysera.readthedocs.io/",
    },
)
