# Transpiler Mate (c) 2025
# 
# Transpiler Mate is licensed under
# Creative Commons Attribution-ShareAlike 4.0 International.
# 
# You should have received a copy of the license along with this work.
# If not, see <https://creativecommons.org/licenses/by-sa/4.0/>.

from pydantic import (
  AnyUrl,
  BaseModel,
  computed_field
)
from typing import (
  Mapping,
  Optional,
  Sequence
)

class ScienceKeywordRecord(BaseModel):
  @computed_field
  @property
  def scheme(self) -> str:
      return 'GCMD Science Keywords'
  category: str
  topic: Optional[str] = None
  term: Optional[str] = None
  variable_level_1: Optional[str] = None
  variable_level_2: Optional[str] = None
  variable_level_3: Optional[str] = None
  detailed_variable: Optional[str] = None
  identifier: str
  @computed_field
  @property
  def uri(self) -> AnyUrl:
      return AnyUrl(f'https://cmr.earthdata.nasa.gov/kms/concept/{self.identifier}')
  @computed_field
  @property
  def hierarchy_list(self) -> Sequence[str]:
      hierarchy = [self.category]

      def _add(keyword: str | None):
          if keyword:
              hierarchy.append(keyword)

      _add(self.topic)
      _add(self.term)
      _add(self.variable_level_1)
      _add(self.variable_level_2)
      _add(self.variable_level_3)
      _add(self.detailed_variable)

      return hierarchy

KEYWORDS_INDEX: Mapping[str, ScienceKeywordRecord] = {
  'e9f67a66-e9fc-435c-b720-ae32a2c3d8f5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='',
    term='',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e9f67a66-e9fc-435c-b720-ae32a2c3d8f5'
  ),
  'a956d045-3b12-441c-8a18-fac7d33b2b4e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a956d045-3b12-441c-8a18-fac7d33b2b4e'
  ),
  'ca227ff0-4742-4e51-a763-4582fa28291c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='AGRICULTURAL AQUATIC SCIENCES',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ca227ff0-4742-4e51-a763-4582fa28291c'
  ),
  '8916dafb-5ad5-45c6-ab64-3500ea1e9577': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='AGRICULTURAL AQUATIC SCIENCES',
    variable_level_1='AQUACULTURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8916dafb-5ad5-45c6-ab64-3500ea1e9577'
  ),
  'c7112a64-be39-414a-9125-f63ab44ecb5b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='AGRICULTURAL AQUATIC SCIENCES',
    variable_level_1='FISHERIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c7112a64-be39-414a-9125-f63ab44ecb5b'
  ),
  '8495b76a-16ba-418c-bbb9-3c6bcfb12aba': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='AGRICULTURAL AQUATIC SCIENCES',
    variable_level_1='ICE STUPA',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8495b76a-16ba-418c-bbb9-3c6bcfb12aba'
  ),
  'afd084b9-1f4c-4eb5-a58e-689a360e7abf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='AGRICULTURAL CHEMICALS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='afd084b9-1f4c-4eb5-a58e-689a360e7abf'
  ),
  '18a8197e-3a3f-408c-9c51-e9fe89dd6b45': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='AGRICULTURAL CHEMICALS',
    variable_level_1='FERTILIZERS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='18a8197e-3a3f-408c-9c51-e9fe89dd6b45'
  ),
  '59a203f9-f818-42a6-8d00-4301385cafc3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='AGRICULTURAL CHEMICALS',
    variable_level_1='PESTICIDES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='59a203f9-f818-42a6-8d00-4301385cafc3'
  ),
  'b8018326-a186-4847-961d-8bd0727bbd5e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='AGRICULTURAL ENGINEERING',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b8018326-a186-4847-961d-8bd0727bbd5e'
  ),
  'f2f37978-d942-43d2-9c51-79e9f5bdfe24': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='AGRICULTURAL ENGINEERING',
    variable_level_1='AGRICULTURAL EQUIPMENT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f2f37978-d942-43d2-9c51-79e9f5bdfe24'
  ),
  'd53e1951-fb68-4ad8-8725-d19c10751da5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='AGRICULTURAL ENGINEERING',
    variable_level_1='FARM STRUCTURES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d53e1951-fb68-4ad8-8725-d19c10751da5'
  ),
  '25be3b9a-9d4c-4b5b-8d24-b1f519913d90': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='AGRICULTURAL PLANT SCIENCE',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='25be3b9a-9d4c-4b5b-8d24-b1f519913d90'
  ),
  'c31e6af1-d4fd-408c-9fca-bd0697a45891': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='AGRICULTURAL PLANT SCIENCE',
    variable_level_1='CROP EMERGENCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c31e6af1-d4fd-408c-9fca-bd0697a45891'
  ),
  'e210bb78-19b6-453c-8dcb-81e389601329': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='AGRICULTURAL PLANT SCIENCE',
    variable_level_1='CROP TYPE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e210bb78-19b6-453c-8dcb-81e389601329'
  ),
  'f12d8026-f24a-4413-91d0-4704c243c9e7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='AGRICULTURAL PLANT SCIENCE',
    variable_level_1='CROP/PLANT YIELDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f12d8026-f24a-4413-91d0-4704c243c9e7'
  ),
  '2dda92a8-6c26-4506-9881-43b6d9a83b18': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='AGRICULTURAL PLANT SCIENCE',
    variable_level_1='CROPPING SYSTEMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2dda92a8-6c26-4506-9881-43b6d9a83b18'
  ),
  'a756fd6b-6208-4af0-ac56-6ee914fc4597': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='AGRICULTURAL PLANT SCIENCE',
    variable_level_1='IRRIGATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a756fd6b-6208-4af0-ac56-6ee914fc4597'
  ),
  'dcd7a439-6021-4fc3-b3d8-a8936ef171f6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='AGRICULTURAL PLANT SCIENCE',
    variable_level_1='PLANT BREEDING AND GENETICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='dcd7a439-6021-4fc3-b3d8-a8936ef171f6'
  ),
  '213cefd8-806f-40f5-b3ca-05022cde9498': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='AGRICULTURAL PLANT SCIENCE',
    variable_level_1='PLANT DISEASES/DISORDERS/PESTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='213cefd8-806f-40f5-b3ca-05022cde9498'
  ),
  'c7570528-f2d5-42b0-b8e9-d12a2432e87e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='AGRICULTURAL PLANT SCIENCE',
    variable_level_1='RECLAMATION/REVEGETATION/RESTORATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c7570528-f2d5-42b0-b8e9-d12a2432e87e'
  ),
  'b376a9f9-585e-4567-ba1f-55ef45cfa8df': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='AGRICULTURAL PLANT SCIENCE',
    variable_level_1='WEEDS, NOXIOUS PLANTS OR INVASIVE PLANTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b376a9f9-585e-4567-ba1f-55ef45cfa8df'
  ),
  'c9f1a861-2173-4124-962c-759f71b6f131': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='ANIMAL COMMODITIES',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c9f1a861-2173-4124-962c-759f71b6f131'
  ),
  'a368da76-b191-4859-bd55-8643f4fab812': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='ANIMAL COMMODITIES',
    variable_level_1='DAIRY PRODUCTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a368da76-b191-4859-bd55-8643f4fab812'
  ),
  '1e2557c5-d232-48e4-8276-369a22ae6aae': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='ANIMAL COMMODITIES',
    variable_level_1='LIVESTOCK PRODUCTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1e2557c5-d232-48e4-8276-369a22ae6aae'
  ),
  'd3ce1677-f3a8-452e-91c8-0ff80e6a3f09': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='ANIMAL COMMODITIES',
    variable_level_1='POULTRY PRODUCTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d3ce1677-f3a8-452e-91c8-0ff80e6a3f09'
  ),
  'b41894fa-2e3e-475b-b8f0-b6ffdd2d6e9c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='ANIMAL SCIENCE',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b41894fa-2e3e-475b-b8f0-b6ffdd2d6e9c'
  ),
  '26089a3e-469d-44b3-a9aa-231d0a072ef9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='ANIMAL SCIENCE',
    variable_level_1='ANIMAL BREEDING AND GENETICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='26089a3e-469d-44b3-a9aa-231d0a072ef9'
  ),
  'e749bafe-9a0a-42cc-bed8-9b42e3e088c8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='ANIMAL SCIENCE',
    variable_level_1='ANIMAL DISEASES/DISORDERS/PESTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e749bafe-9a0a-42cc-bed8-9b42e3e088c8'
  ),
  '5d1b53b2-7d69-4b7c-903f-d8cf29430f93': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='ANIMAL SCIENCE',
    variable_level_1='ANIMAL ECOLOGY AND BEHAVIOR',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5d1b53b2-7d69-4b7c-903f-d8cf29430f93'
  ),
  'e5b724af-b661-406a-ae1f-7cd2730c0576': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='ANIMAL SCIENCE',
    variable_level_1='ANIMAL MANAGEMENT SYSTEMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e5b724af-b661-406a-ae1f-7cd2730c0576'
  ),
  '3c1c65c3-e1ef-4163-9695-c39ff7fb48da': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='ANIMAL SCIENCE',
    variable_level_1='ANIMAL MANURE AND WASTE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3c1c65c3-e1ef-4163-9695-c39ff7fb48da'
  ),
  'ca551e61-4b8c-46d5-8590-80cada40ebbd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='ANIMAL SCIENCE',
    variable_level_1='ANIMAL NUTRITION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ca551e61-4b8c-46d5-8590-80cada40ebbd'
  ),
  'f9cdf3ae-fe8b-4a19-a946-a8c8780d7894': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='ANIMAL SCIENCE',
    variable_level_1='ANIMAL PHYSIOLOGY AND BIOCHEMISTRY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f9cdf3ae-fe8b-4a19-a946-a8c8780d7894'
  ),
  '3c0bbd0f-6d4d-4036-afa9-03f9b4f8fba0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='ANIMAL SCIENCE',
    variable_level_1='ANIMAL YIELDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3c0bbd0f-6d4d-4036-afa9-03f9b4f8fba0'
  ),
  '2c31fc22-747a-476f-b76d-fec61220b5b1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='ANIMAL SCIENCE',
    variable_level_1='APICULTURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2c31fc22-747a-476f-b76d-fec61220b5b1'
  ),
  '06053150-d796-477b-b305-292442d658ed': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='ANIMAL SCIENCE',
    variable_level_1='SERICULTURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='06053150-d796-477b-b305-292442d658ed'
  ),
  'c1f9f5fa-245c-4055-81cf-5230c076c0ce': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FEED PRODUCTS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c1f9f5fa-245c-4055-81cf-5230c076c0ce'
  ),
  'cf9ef34d-ed39-4c8d-bf00-ca1b0bb11363': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FEED PRODUCTS',
    variable_level_1='FEED COMPOSITION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='cf9ef34d-ed39-4c8d-bf00-ca1b0bb11363'
  ),
  'b9957bbc-3c12-481d-86a0-0f6cf2bb8219': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FEED PRODUCTS',
    variable_level_1='FEED CONTAMINATION AND TOXICOLOGY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b9957bbc-3c12-481d-86a0-0f6cf2bb8219'
  ),
  'fec2eb53-bc69-4d35-849c-c2bedf5dc6cf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FEED PRODUCTS',
    variable_level_1='FEED PROCESSING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fec2eb53-bc69-4d35-849c-c2bedf5dc6cf'
  ),
  '9244fe19-b86f-4a8d-82bf-c52f804a77e3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FEED PRODUCTS',
    variable_level_1='FEED STORAGE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9244fe19-b86f-4a8d-82bf-c52f804a77e3'
  ),
  'b98f3a77-397d-41d7-9507-e7a3e47210b1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FOOD SCIENCE',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b98f3a77-397d-41d7-9507-e7a3e47210b1'
  ),
  'eb9b8c19-3b39-4865-bcfc-d2a12689094a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FOOD SCIENCE',
    variable_level_1='FOOD ADDITIVES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='eb9b8c19-3b39-4865-bcfc-d2a12689094a'
  ),
  'e86ea427-f735-4998-af16-9bd619df4974': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FOOD SCIENCE',
    variable_level_1='FOOD CONTAMINATION AND TOXICOLOGY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e86ea427-f735-4998-af16-9bd619df4974'
  ),
  '85d7c19a-6d05-446f-a490-382e7c199e09': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FOOD SCIENCE',
    variable_level_1='FOOD PACKAGING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='85d7c19a-6d05-446f-a490-382e7c199e09'
  ),
  'b153bcea-3114-4809-8e6f-f22cf9a3be87': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FOOD SCIENCE',
    variable_level_1='FOOD PROCESSING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b153bcea-3114-4809-8e6f-f22cf9a3be87'
  ),
  '3ec3b00e-52e1-4df9-99cd-c93120d97645': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FOOD SCIENCE',
    variable_level_1='FOOD QUALITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3ec3b00e-52e1-4df9-99cd-c93120d97645'
  ),
  'b3b14df8-5197-4a26-ae61-882fdba706f3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FOOD SCIENCE',
    variable_level_1='FOOD STORAGE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b3b14df8-5197-4a26-ae61-882fdba706f3'
  ),
  '22ec2f9b-1f1a-469b-bc09-851d58637ff4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FOREST SCIENCE',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='22ec2f9b-1f1a-469b-bc09-851d58637ff4'
  ),
  'b3a1e091-0bc2-4c9b-a89c-bd003fdd5889': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FOREST SCIENCE',
    variable_level_1='AFFORESTATION/REFORESTATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b3a1e091-0bc2-4c9b-a89c-bd003fdd5889'
  ),
  'b3fcccdd-745f-4299-94b3-e72e37f551be': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FOREST SCIENCE',
    variable_level_1='DEFOLIANTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b3fcccdd-745f-4299-94b3-e72e37f551be'
  ),
  '7ee9d286-0742-4844-b7eb-b7550d3f782b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FOREST SCIENCE',
    variable_level_1='FOREST CONSERVATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7ee9d286-0742-4844-b7eb-b7550d3f782b'
  ),
  'e5a8c6ed-5b59-40fe-a83b-18b39fb7c31b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FOREST SCIENCE',
    variable_level_1='FOREST FIRE SCIENCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e5a8c6ed-5b59-40fe-a83b-18b39fb7c31b'
  ),
  '23336b57-1ba3-42a6-9ec7-152285c55689': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FOREST SCIENCE',
    variable_level_1='FOREST HARVESTING AND ENGINEERING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='23336b57-1ba3-42a6-9ec7-152285c55689'
  ),
  'd2056285-8249-4c11-810b-783600030525': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FOREST SCIENCE',
    variable_level_1='FOREST MANAGEMENT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d2056285-8249-4c11-810b-783600030525'
  ),
  '31d01087-d5b8-4474-820c-d84d523dfb39': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FOREST SCIENCE',
    variable_level_1='FOREST MENSURATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='31d01087-d5b8-4474-820c-d84d523dfb39'
  ),
  '3676ebab-9aa0-43c2-94e5-5d59a34317d2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FOREST SCIENCE',
    variable_level_1='FOREST PRODUCTS/COMMODITIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3676ebab-9aa0-43c2-94e5-5d59a34317d2'
  ),
  'adeb4c27-a115-4ced-9827-5f022883f606': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FOREST SCIENCE',
    variable_level_1='FOREST PROTECTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='adeb4c27-a115-4ced-9827-5f022883f606'
  ),
  '49804617-d59b-4e97-8030-2c4ab79a3057': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FOREST SCIENCE',
    variable_level_1='FOREST YIELDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='49804617-d59b-4e97-8030-2c4ab79a3057'
  ),
  'be7f6de0-f51e-42bc-9a66-fff30d809a67': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FOREST SCIENCE',
    variable_level_1='REFORESTATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='be7f6de0-f51e-42bc-9a66-fff30d809a67'
  ),
  '8e115d45-acd4-4116-a0e1-3de8a5db23c2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='FOREST SCIENCE',
    variable_level_1='SILVICULTURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8e115d45-acd4-4116-a0e1-3de8a5db23c2'
  ),
  'd6560f20-3bef-41c6-8eec-9f913329b9ac': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='PLANT COMMODITIES',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d6560f20-3bef-41c6-8eec-9f913329b9ac'
  ),
  '63317fb1-01d9-4658-93e8-9800c5359454': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='PLANT COMMODITIES',
    variable_level_1='FIELD CROP PRODUCTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='63317fb1-01d9-4658-93e8-9800c5359454'
  ),
  '41b30b1b-5dbb-4ef8-849c-e1949ad04227': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='PLANT COMMODITIES',
    variable_level_1='FRUIT PRODUCTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='41b30b1b-5dbb-4ef8-849c-e1949ad04227'
  ),
  'd23b37cd-5e05-4356-b8b4-df6d7af236d6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='PLANT COMMODITIES',
    variable_level_1='HORTICULTURAL PRODUCTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d23b37cd-5e05-4356-b8b4-df6d7af236d6'
  ),
  'eb1627c2-0061-466c-9935-399e53a06024': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='PLANT COMMODITIES',
    variable_level_1='VEGETABLE PRODUCTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='eb1627c2-0061-466c-9935-399e53a06024'
  ),
  '199e3af8-4cf3-48ba-8b28-b9b54756b3db': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='199e3af8-4cf3-48ba-8b28-b9b54756b3db'
  ),
  '7367c08c-304f-4ce7-b716-975f835ba711': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='CALCIUM',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7367c08c-304f-4ce7-b716-975f835ba711'
  ),
  '9315c474-b65f-400d-beba-611c9a6a62cb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='CARBON',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9315c474-b65f-400d-beba-611c9a6a62cb'
  ),
  '5c05e69f-f6db-4296-abd3-3b07e6093579': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='CATION EXCHANGE CAPACITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5c05e69f-f6db-4296-abd3-3b07e6093579'
  ),
  'cac79930-334e-49c5-836b-4f2ee8e0b098': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='DENITRIFICATION RATE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='cac79930-334e-49c5-836b-4f2ee8e0b098'
  ),
  '7241d799-4f5c-4ae3-a4ec-2e9cdbf656aa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='ELECTRICAL CONDUCTIVITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7241d799-4f5c-4ae3-a4ec-2e9cdbf656aa'
  ),
  '8b3939b6-1c11-4a79-878e-0be1b231c528': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='HEAVY METALS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8b3939b6-1c11-4a79-878e-0be1b231c528'
  ),
  '7112e739-cb5d-427e-95bd-5419360e91d8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='HYDRAULIC CONDUCTIVITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7112e739-cb5d-427e-95bd-5419360e91d8'
  ),
  '60992683-3183-4510-9fce-e4611115315c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='IRON',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='60992683-3183-4510-9fce-e4611115315c'
  ),
  '83da5ac6-5981-4929-9e19-f46522c1babe': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='MACROFAUNA',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='83da5ac6-5981-4929-9e19-f46522c1babe'
  ),
  '79f18259-bd76-4c7b-bd18-cbd2edafd24f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='MAGNESIUM',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='79f18259-bd76-4c7b-bd18-cbd2edafd24f'
  ),
  '53231d78-471d-4afe-a435-b577b7d53b17': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='MICROFAUNA',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='53231d78-471d-4afe-a435-b577b7d53b17'
  ),
  'b3063d3a-af53-44f9-a532-4cea2880c198': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='MICROFLORA',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b3063d3a-af53-44f9-a532-4cea2880c198'
  ),
  '2473e776-4449-4351-9835-1507532ae60e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='MICRONUTRIENTS/TRACE ELEMENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2473e776-4449-4351-9835-1507532ae60e'
  ),
  '5ed7811a-2ba1-4985-9f1c-a78c802fa27f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='NITROGEN',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5ed7811a-2ba1-4985-9f1c-a78c802fa27f'
  ),
  '83cf51f6-8c03-4f6d-b605-fde9818c7805': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='ORGANIC MATTER',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='83cf51f6-8c03-4f6d-b605-fde9818c7805'
  ),
  'fb3ce3be-d830-407f-bd7c-58d66c24b6be': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='PERMAFROST',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fb3ce3be-d830-407f-bd7c-58d66c24b6be'
  ),
  '4962dabc-b426-4c84-8147-12e15645baff': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='PHOSPHORUS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4962dabc-b426-4c84-8147-12e15645baff'
  ),
  'c07fe67b-234e-4293-9f09-abaf9612c0e9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='POTASSIUM',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c07fe67b-234e-4293-9f09-abaf9612c0e9'
  ),
  'd0da93ff-af45-4e26-8b94-8b90d0e06438': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL ABSORPTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d0da93ff-af45-4e26-8b94-8b90d0e06438'
  ),
  'b69052b9-69ab-4294-aa04-5ad639d1b31d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL BIOGEOCHEMISTRY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b69052b9-69ab-4294-aa04-5ad639d1b31d'
  ),
  '62d5fb39-e9ee-47db-a426-1991537f8a4d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL BULK DENSITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='62d5fb39-e9ee-47db-a426-1991537f8a4d'
  ),
  '652349bd-f6f9-4c8d-8573-d71e05ad1208': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL CHEMISTRY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='652349bd-f6f9-4c8d-8573-d71e05ad1208'
  ),
  '2f57fd58-d8e4-4e6d-b8c3-2a9ef7e64f54': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL CLASSIFICATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2f57fd58-d8e4-4e6d-b8c3-2a9ef7e64f54'
  ),
  '3985ce6b-e0c3-42a8-b40f-9dd948350c6e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL COLOR',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3985ce6b-e0c3-42a8-b40f-9dd948350c6e'
  ),
  'e0c0af2a-1429-4248-8d5b-ccae510da0c9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL COMPACTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e0c0af2a-1429-4248-8d5b-ccae510da0c9'
  ),
  '25c5c222-c053-4081-ac0f-52e6c774198c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL CONSISTENCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='25c5c222-c053-4081-ac0f-52e6c774198c'
  ),
  'b09b4731-f357-4838-829b-f38c0f5075aa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL DEPTH',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b09b4731-f357-4838-829b-f38c0f5075aa'
  ),
  '36c862a7-7117-4fd2-8e33-0dda03097178': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL EROSION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='36c862a7-7117-4fd2-8e33-0dda03097178'
  ),
  'e4781de7-a4a4-4157-a549-4ac238d36512': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL FERTILITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e4781de7-a4a4-4157-a549-4ac238d36512'
  ),
  'd302aeaa-3a86-4ddf-9755-60b7bb4404a5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL GAS/AIR',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d302aeaa-3a86-4ddf-9755-60b7bb4404a5'
  ),
  '68033b72-7f8d-48a4-8f63-638e4e96fd23': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL HEAT BUDGET',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='68033b72-7f8d-48a4-8f63-638e4e96fd23'
  ),
  '1fc22c9d-cf29-4bd7-90b1-b0f6f139fd92': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL HORIZONS/PROFILE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1fc22c9d-cf29-4bd7-90b1-b0f6f139fd92'
  ),
  '6edf1b99-fe00-493e-b0d1-ad6b36b8da75': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL IMPEDANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6edf1b99-fe00-493e-b0d1-ad6b36b8da75'
  ),
  '0ab5ead8-6037-42b3-b3c0-0746f3645af6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL INFILTRATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0ab5ead8-6037-42b3-b3c0-0746f3645af6'
  ),
  '3b1d75b6-7559-4921-8edb-63f4dff370cf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL MECHANICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3b1d75b6-7559-4921-8edb-63f4dff370cf'
  ),
  '88e1a654-5cfd-423f-9350-0ef48d85e085': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL MOISTURE/WATER CONTENT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='88e1a654-5cfd-423f-9350-0ef48d85e085'
  ),
  '2a9bce94-c391-4834-96bb-a9685d3590b1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL PH',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2a9bce94-c391-4834-96bb-a9685d3590b1'
  ),
  '934bfe13-908b-40d9-b346-a347a8a6855e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL PLASTICITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='934bfe13-908b-40d9-b346-a347a8a6855e'
  ),
  'c26693ea-ca5a-44e8-9e8e-32427bc62aa0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL POROSITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c26693ea-ca5a-44e8-9e8e-32427bc62aa0'
  ),
  '5c6df811-bebf-4dae-a70f-f49fece3fa1e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL PRODUCTIVITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5c6df811-bebf-4dae-a70f-f49fece3fa1e'
  ),
  'db9b56da-e05f-4d58-b9d5-34edc83ca650': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL RESPIRATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='db9b56da-e05f-4d58-b9d5-34edc83ca650'
  ),
  '2b91245e-a779-42fa-89c2-303217463b95': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL ROOTING DEPTH',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2b91245e-a779-42fa-89c2-303217463b95'
  ),
  '3b54403e-25a1-43cc-97ac-7c14e73bda96': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL SALINITY/SOIL SODICITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3b54403e-25a1-43cc-97ac-7c14e73bda96'
  ),
  'e4daef1d-e672-41d0-bc6d-80c6b5c0799b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL STRUCTURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e4daef1d-e672-41d0-bc6d-80c6b5c0799b'
  ),
  '26f5bb2a-b872-41e8-922f-3a9a0e9f9bcd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL TEMPERATURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='26f5bb2a-b872-41e8-922f-3a9a0e9f9bcd'
  ),
  'afd1d3cb-d31d-4069-8cff-b592887aa18c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL TEXTURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='afd1d3cb-d31d-4069-8cff-b592887aa18c'
  ),
  '223ce1f2-e2f1-4612-8fce-b96b7d34710f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SOIL WATER HOLDING CAPACITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='223ce1f2-e2f1-4612-8fce-b96b7d34710f'
  ),
  'e3d3f76d-0ffe-4616-9988-0520e78cf842': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='SULFUR',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e3d3f76d-0ffe-4616-9988-0520e78cf842'
  ),
  '5c349776-dd95-483e-a5da-e8d1b1434985': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='AGRICULTURE',
    term='SOILS',
    variable_level_1='THERMAL CONDUCTIVITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5c349776-dd95-483e-a5da-e8d1b1434985'
  ),
  'c47f6052-634e-40ef-a5ac-13f69f6f4c2a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c47f6052-634e-40ef-a5ac-13f69f6f4c2a'
  ),
  '2e5a401b-1507-4f57-82b8-36557c13b154': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2e5a401b-1507-4f57-82b8-36557c13b154'
  ),
  '2e2b3e40-8775-41a7-a541-e482847289cb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='AEROSOL ABSORPTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2e2b3e40-8775-41a7-a541-e482847289cb'
  ),
  'f795b88f-1aba-4548-97f6-7b587e8ba451': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='AEROSOL BACKSCATTER',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f795b88f-1aba-4548-97f6-7b587e8ba451'
  ),
  '101c2cd2-b135-430b-a9c6-b710dee48d78': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='AEROSOL BACKSCATTER',
    variable_level_2='AEROSOL FRACTION',
    variable_level_3='',
    detailed_variable='',
    identifier='101c2cd2-b135-430b-a9c6-b710dee48d78'
  ),
  '40633fe2-5b32-4bdc-a17b-b1cfebc01ae7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='AEROSOL EXTINCTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='40633fe2-5b32-4bdc-a17b-b1cfebc01ae7'
  ),
  '449e2e03-8efd-42b6-8152-3602e4bab21d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='AEROSOL FORWARD SCATTER',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='449e2e03-8efd-42b6-8152-3602e4bab21d'
  ),
  'e83e3455-9b6e-4ece-943e-1ce9e570a1ec': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='AEROSOL OPTICAL CENTROID HEIGHT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e83e3455-9b6e-4ece-943e-1ce9e570a1ec'
  ),
  '61c3b720-abc8-4430-866c-f1da35d2cd0b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='AEROSOL OPTICAL DEPTH/THICKNESS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='61c3b720-abc8-4430-866c-f1da35d2cd0b'
  ),
  'ee84a17d-221f-4798-93f9-7cecc8400662': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='AEROSOL OPTICAL DEPTH/THICKNESS',
    variable_level_2='ABSORBING AEROSOL OPTICAL DEPTH/THICKNESS',
    variable_level_3='',
    detailed_variable='',
    identifier='ee84a17d-221f-4798-93f9-7cecc8400662'
  ),
  '6e7306a1-79a5-482e-b646-74b75a1eaa48': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='AEROSOL OPTICAL DEPTH/THICKNESS',
    variable_level_2='ANGSTROM EXPONENT',
    variable_level_3='',
    detailed_variable='',
    identifier='6e7306a1-79a5-482e-b646-74b75a1eaa48'
  ),
  '8d7e5d36-4d81-469d-9318-bf20ba3bae5c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='AEROSOL OPTICAL DEPTH/THICKNESS',
    variable_level_2='UV AEROSOL INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='8d7e5d36-4d81-469d-9318-bf20ba3bae5c'
  ),
  '7acd7fd4-a9ed-4af9-a0e6-1f6bccb17a3c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='AEROSOL OPTICAL DEPTH/THICKNESS',
    variable_level_2='UV AEROSOL OPTICAL DEPTH',
    variable_level_3='',
    detailed_variable='',
    identifier='7acd7fd4-a9ed-4af9-a0e6-1f6bccb17a3c'
  ),
  '02ea239e-4bca-4fda-ab87-be12c723c30a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='AEROSOL PARTICLE PROPERTIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='02ea239e-4bca-4fda-ab87-be12c723c30a'
  ),
  '59920ad4-85fb-4cee-ba56-f39bc5857a3d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='AEROSOL PARTICLE PROPERTIES',
    variable_level_2='AEROSOL CONCENTRATION',
    variable_level_3='',
    detailed_variable='',
    identifier='59920ad4-85fb-4cee-ba56-f39bc5857a3d'
  ),
  '41575931-3cc4-4c9d-97a7-dde82bb0e19e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='AEROSOL PARTICLE PROPERTIES',
    variable_level_2='AEROSOL SIZE DISTRIBUTION',
    variable_level_3='',
    detailed_variable='',
    identifier='41575931-3cc4-4c9d-97a7-dde82bb0e19e'
  ),
  '7db9eab3-4c7a-4471-a826-a306f178ad3e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='AEROSOL RADIANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7db9eab3-4c7a-4471-a826-a306f178ad3e'
  ),
  '86c75a9c-d6ff-4a33-aad2-5c42fbe87daf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='AEROSOL SINGLE SCATTERING ALBEDO',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='86c75a9c-d6ff-4a33-aad2-5c42fbe87daf'
  ),
  '3a1aa79b-2897-4ede-8682-3bd66be93a91': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='AEROSOL VERTICAL DISTRIBUTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3a1aa79b-2897-4ede-8682-3bd66be93a91'
  ),
  '48e0400c-727e-4794-b80c-efdf9f71e3ba': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='AMMONIUM AEROSOLS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='48e0400c-727e-4794-b80c-efdf9f71e3ba'
  ),
  '9c0288cc-864d-40f7-93af-6df413b404f5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='BLACK CARBON',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9c0288cc-864d-40f7-93af-6df413b404f5'
  ),
  '527f637c-aea5-4519-9293-d57e10a76bff': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='CARBONACEOUS AEROSOLS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='527f637c-aea5-4519-9293-d57e10a76bff'
  ),
  '0611b9fd-fd92-4c4d-87bb-bc2f22c548bc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='CHEMICAL COMPOSITION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0611b9fd-fd92-4c4d-87bb-bc2f22c548bc'
  ),
  '23cd8555-05a7-4fea-a3c1-765227f0d9d4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='CHEMICAL COMPOSITION',
    variable_level_2='ELEMENTAL CARBON',
    variable_level_3='',
    detailed_variable='',
    identifier='23cd8555-05a7-4fea-a3c1-765227f0d9d4'
  ),
  'a63f4fe6-51dc-4719-95e3-a09d111774c9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='CHEMICAL COMPOSITION',
    variable_level_2='NON-REFRACTORY AEROSOL ORGANIC MASS',
    variable_level_3='',
    detailed_variable='',
    identifier='a63f4fe6-51dc-4719-95e3-a09d111774c9'
  ),
  'bc6f9a64-0d00-4f39-9f1c-a4c25b373897': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='CHEMICAL COMPOSITION',
    variable_level_2='WATER-SOLUBLE AEROSOL ORGANIC MASS',
    variable_level_3='',
    detailed_variable='',
    identifier='bc6f9a64-0d00-4f39-9f1c-a4c25b373897'
  ),
  '27478148-b4b6-4c89-8829-08d2ee7bfe10': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='CLOUD CONDENSATION NUCLEI',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='27478148-b4b6-4c89-8829-08d2ee7bfe10'
  ),
  '5b775d6e-de6c-4b11-b986-3c3a32cbf66d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='DEPOSITION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5b775d6e-de6c-4b11-b986-3c3a32cbf66d'
  ),
  '1b6342c6-315b-4f4f-b4e3-d6902aaa3e85': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='DUST/ASH/SMOKE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1b6342c6-315b-4f4f-b4e3-d6902aaa3e85'
  ),
  '768cfa32-003d-47bd-ab3a-3e27e4ec2699': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='NITRATE PARTICLES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='768cfa32-003d-47bd-ab3a-3e27e4ec2699'
  ),
  '8929113a-ded5-4c39-b20f-7968ed114317': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='ORGANIC PARTICLES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8929113a-ded5-4c39-b20f-7968ed114317'
  ),
  '548a3f85-bf22-473b-b641-45c32d9c6a0c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='PARTICULATE MATTER',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='548a3f85-bf22-473b-b641-45c32d9c6a0c'
  ),
  '6a340e0c-1f2e-435d-acf2-427ebc0d5e4c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='PARTICULATE MATTER',
    variable_level_2='PARTICULATE MATTER (PM 1.0)',
    variable_level_3='',
    detailed_variable='',
    identifier='6a340e0c-1f2e-435d-acf2-427ebc0d5e4c'
  ),
  '1ffc2101-e4bc-4010-9a4c-b86c858d850f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='PARTICULATE MATTER',
    variable_level_2='PARTICULATE MATTER (PM 10)',
    variable_level_3='',
    detailed_variable='',
    identifier='1ffc2101-e4bc-4010-9a4c-b86c858d850f'
  ),
  '6ab81a2f-5e7e-4249-87d2-875c6a4a2a80': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='PARTICULATE MATTER',
    variable_level_2='PARTICULATE MATTER (PM 2.5)',
    variable_level_3='',
    detailed_variable='',
    identifier='6ab81a2f-5e7e-4249-87d2-875c6a4a2a80'
  ),
  'cf040a0f-f934-474f-8def-0623a15db69f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='SEA SALT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='cf040a0f-f934-474f-8def-0623a15db69f'
  ),
  'ca71b02b-4446-414c-8697-0950d7382cc4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='SULFATE PARTICLES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ca71b02b-4446-414c-8697-0950d7382cc4'
  ),
  '4f4db335-d418-43d2-9e72-b8312cba310c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AEROSOLS',
    variable_level_1='TOXIC AEROSOLS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4f4db335-d418-43d2-9e72-b8312cba310c'
  ),
  '77397026-09c9-44e0-b85f-77b2bc9b1630': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AIR QUALITY',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='77397026-09c9-44e0-b85f-77b2bc9b1630'
  ),
  '080389c4-68d4-41ee-ab89-070794038c8e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AIR QUALITY',
    variable_level_1='CARBON MONOXIDE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='080389c4-68d4-41ee-ab89-070794038c8e'
  ),
  '2a60df4a-a0d7-4e4b-b02a-372a083f0170': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AIR QUALITY',
    variable_level_1='EMISSIONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2a60df4a-a0d7-4e4b-b02a-372a083f0170'
  ),
  'c79453a3-ed2f-4ec4-9298-bf9fd11d08eb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AIR QUALITY',
    variable_level_1='LEAD',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c79453a3-ed2f-4ec4-9298-bf9fd11d08eb'
  ),
  'e5563c99-0fb6-43a9-8e20-6b47b1144394': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AIR QUALITY',
    variable_level_1='NITROGEN OXIDES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e5563c99-0fb6-43a9-8e20-6b47b1144394'
  ),
  'a2bdd7e8-145e-4bbf-b10a-ef7a87fcb1ad': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AIR QUALITY',
    variable_level_1='PARTICULATE MATTER (PM 1.0)',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a2bdd7e8-145e-4bbf-b10a-ef7a87fcb1ad'
  ),
  '2cd061e7-f351-46fa-8432-fb36faef3bbe': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AIR QUALITY',
    variable_level_1='PARTICULATE MATTER (PM 10)',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2cd061e7-f351-46fa-8432-fb36faef3bbe'
  ),
  '686bd6b6-8305-4e33-a334-ef5d4f46a230': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AIR QUALITY',
    variable_level_1='PARTICULATE MATTER (PM 2.5)',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='686bd6b6-8305-4e33-a334-ef5d4f46a230'
  ),
  'f9fe1bc0-88c5-4c26-9b4c-a9867d027685': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AIR QUALITY',
    variable_level_1='PARTICULATES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f9fe1bc0-88c5-4c26-9b4c-a9867d027685'
  ),
  'bad08657-da2b-4e2b-9804-25c5732bc795': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AIR QUALITY',
    variable_level_1='SMOG',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bad08657-da2b-4e2b-9804-25c5732bc795'
  ),
  'c3090318-c845-4242-bf2f-ff1631b88831': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AIR QUALITY',
    variable_level_1='SULFUR OXIDES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c3090318-c845-4242-bf2f-ff1631b88831'
  ),
  '426aee98-764c-4c21-ab65-1e9d4bd6b0d0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AIR QUALITY',
    variable_level_1='TROPOSPHERIC OZONE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='426aee98-764c-4c21-ab65-1e9d4bd6b0d0'
  ),
  '227cf2d4-968a-4312-89e6-8c6bcf616e5d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AIR QUALITY',
    variable_level_1='TURBIDITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='227cf2d4-968a-4312-89e6-8c6bcf616e5d'
  ),
  '9337898d-68dc-43d7-93a9-6afdb4ab1784': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AIR QUALITY',
    variable_level_1='VISIBILITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9337898d-68dc-43d7-93a9-6afdb4ab1784'
  ),
  '1f3c543d-9ca9-4db4-b4a5-d3e2fd71e4a4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='AIR QUALITY',
    variable_level_1='VOLATILE ORGANIC COMPOUNDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1f3c543d-9ca9-4db4-b4a5-d3e2fd71e4a4'
  ),
  '16bfcf54-f8e1-4c8e-9bd4-a1ac06ea95a0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ALTITUDE',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='16bfcf54-f8e1-4c8e-9bd4-a1ac06ea95a0'
  ),
  '5d703cfe-2f7c-4736-acbc-ec4e4f4f8eef': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ALTITUDE',
    variable_level_1='BAROMETRIC ALTITUDE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5d703cfe-2f7c-4736-acbc-ec4e4f4f8eef'
  ),
  'd6aec072-daf9-4f96-b667-6c7831cf6bdd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ALTITUDE',
    variable_level_1='GEOPOTENTIAL HEIGHT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d6aec072-daf9-4f96-b667-6c7831cf6bdd'
  ),
  'dacbf270-1734-4503-bab8-a32cdaff3012': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ALTITUDE',
    variable_level_1='MESOPAUSE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='dacbf270-1734-4503-bab8-a32cdaff3012'
  ),
  '765e92a7-8c14-47dc-bdd8-d85d132a11ee': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ALTITUDE',
    variable_level_1='PLANETARY BOUNDARY LAYER HEIGHT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='765e92a7-8c14-47dc-bdd8-d85d132a11ee'
  ),
  '2343baae-1c4a-4096-8cac-fea8ed7a984f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ALTITUDE',
    variable_level_1='STATION HEIGHT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2343baae-1c4a-4096-8cac-fea8ed7a984f'
  ),
  '82191e97-53ba-413d-9a08-acd8b848e0b0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ALTITUDE',
    variable_level_1='STRATOPAUSE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='82191e97-53ba-413d-9a08-acd8b848e0b0'
  ),
  'c3447c90-7490-4f04-89c1-c5274ba8f8f6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ALTITUDE',
    variable_level_1='TROPOPAUSE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c3447c90-7490-4f04-89c1-c5274ba8f8f6'
  ),
  '55b1e1e3-f6c1-439d-864c-f32a9ca8852c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERE/SURFACE INTERACTIONS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='55b1e1e3-f6c1-439d-864c-f32a9ca8852c'
  ),
  '0c18c5bf-1a5b-4633-8e1c-b93be226be77': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERE/SURFACE INTERACTIONS',
    variable_level_1='SURFACE MOISTURE FLUX',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0c18c5bf-1a5b-4633-8e1c-b93be226be77'
  ),
  'fe40d33a-a792-4e06-8b4a-70bd0210b5a0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERE/SURFACE INTERACTIONS',
    variable_level_1='SURFACE MOMENTUM FLUX',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fe40d33a-a792-4e06-8b4a-70bd0210b5a0'
  ),
  'b9c56939-c624-467d-b196-e56a5b660334': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b9c56939-c624-467d-b196-e56a5b660334'
  ),
  '19ab681c-bdd7-4793-bbdb-1ec498575314': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='CARBON AND HYDROCARBON COMPOUNDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='19ab681c-bdd7-4793-bbdb-1ec498575314'
  ),
  '04833f72-ac6d-40b0-b1ae-1f55eb25b5dd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='CARBON AND HYDROCARBON COMPOUNDS',
    variable_level_2='ACETYLENE',
    variable_level_3='',
    detailed_variable='',
    identifier='04833f72-ac6d-40b0-b1ae-1f55eb25b5dd'
  ),
  'c3b81888-8a39-4b3f-8033-4c077797bcba': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='CARBON AND HYDROCARBON COMPOUNDS',
    variable_level_2='ATMOSPHERIC CARBON DIOXIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='c3b81888-8a39-4b3f-8033-4c077797bcba'
  ),
  '03ddc432-906d-4469-bb00-179c828dbea4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='CARBON AND HYDROCARBON COMPOUNDS',
    variable_level_2='ATMOSPHERIC CARBON DIOXIDE',
    variable_level_3='CARBON DIOXIDE PROFILES',
    detailed_variable='',
    identifier='03ddc432-906d-4469-bb00-179c828dbea4'
  ),
  '194d8a3c-9cdd-45f3-8b4c-ce7830d9df46': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='CARBON AND HYDROCARBON COMPOUNDS',
    variable_level_2='ATMOSPHERIC CARBON DIOXIDE',
    variable_level_3='CARBON DIOXIDE SURFACE',
    detailed_variable='',
    identifier='194d8a3c-9cdd-45f3-8b4c-ce7830d9df46'
  ),
  'a65cfcfa-1028-4cc8-a4d5-9e78f487a612': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='CARBON AND HYDROCARBON COMPOUNDS',
    variable_level_2='ATMOSPHERIC CARBON DIOXIDE',
    variable_level_3='PARTIAL PRESSURE OF CARBON DIOXIDE',
    detailed_variable='',
    identifier='a65cfcfa-1028-4cc8-a4d5-9e78f487a612'
  ),
  '88a1b416-1589-45a4-9923-452975ec35c7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='CARBON AND HYDROCARBON COMPOUNDS',
    variable_level_2='ATMOSPHERIC CARBON MONOXIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='88a1b416-1589-45a4-9923-452975ec35c7'
  ),
  'ab3e5ad3-d2c0-4c63-b321-345307bda59d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='CARBON AND HYDROCARBON COMPOUNDS',
    variable_level_2='ATMOSPHERIC CARBON MONOXIDE',
    variable_level_3='CARBON MONOXIDE PROFILES',
    detailed_variable='',
    identifier='ab3e5ad3-d2c0-4c63-b321-345307bda59d'
  ),
  '3965bb3f-fa0e-475e-a48c-5703c9ab9fe5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='CARBON AND HYDROCARBON COMPOUNDS',
    variable_level_2='ATMOSPHERIC CARBON MONOXIDE',
    variable_level_3='CARBON MONOXIDE SURFACE',
    detailed_variable='',
    identifier='3965bb3f-fa0e-475e-a48c-5703c9ab9fe5'
  ),
  'cdab2cca-6767-427e-b464-09fe26ec59db': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='CARBON AND HYDROCARBON COMPOUNDS',
    variable_level_2='CHLORINATED HYDROCARBONS',
    variable_level_3='',
    detailed_variable='',
    identifier='cdab2cca-6767-427e-b464-09fe26ec59db'
  ),
  '94863274-c0fc-4386-9bac-5b6f5d1b9d06': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='CARBON AND HYDROCARBON COMPOUNDS',
    variable_level_2='DICARBON MONOXIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='94863274-c0fc-4386-9bac-5b6f5d1b9d06'
  ),
  '93a95204-8ded-4cde-8937-38e373c41df6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='CARBON AND HYDROCARBON COMPOUNDS',
    variable_level_2='ETHANE',
    variable_level_3='',
    detailed_variable='',
    identifier='93a95204-8ded-4cde-8937-38e373c41df6'
  ),
  'bc05d7d2-3c96-4bb6-b759-d45e3c673b86': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='CARBON AND HYDROCARBON COMPOUNDS',
    variable_level_2='FORMALDEHYDE',
    variable_level_3='',
    detailed_variable='',
    identifier='bc05d7d2-3c96-4bb6-b759-d45e3c673b86'
  ),
  'eb1bfadc-8aa6-477b-af6a-6c320aa21351': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='CARBON AND HYDROCARBON COMPOUNDS',
    variable_level_2='FORMIC ACID',
    variable_level_3='',
    detailed_variable='',
    identifier='eb1bfadc-8aa6-477b-af6a-6c320aa21351'
  ),
  '890297e4-b53b-4ceb-aeef-02f41e747311': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='CARBON AND HYDROCARBON COMPOUNDS',
    variable_level_2='GLYOXAL',
    variable_level_3='',
    detailed_variable='',
    identifier='890297e4-b53b-4ceb-aeef-02f41e747311'
  ),
  'af157837-bdbd-4a9a-b24e-6a79adfef57f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='CARBON AND HYDROCARBON COMPOUNDS',
    variable_level_2='HYDROGEN CYANIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='af157837-bdbd-4a9a-b24e-6a79adfef57f'
  ),
  'c6b2279c-804f-42bf-aa8a-0c81f9ecf6cd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='CARBON AND HYDROCARBON COMPOUNDS',
    variable_level_2='HYPOCHLOROUS MONOXIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='c6b2279c-804f-42bf-aa8a-0c81f9ecf6cd'
  ),
  '9154777e-2e33-49b5-a21e-0a2638c57528': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='CARBON AND HYDROCARBON COMPOUNDS',
    variable_level_2='ISOPRENE',
    variable_level_3='',
    detailed_variable='',
    identifier='9154777e-2e33-49b5-a21e-0a2638c57528'
  ),
  '7c892333-f4c4-4f81-b825-d6a86e107e9f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='CARBON AND HYDROCARBON COMPOUNDS',
    variable_level_2='METHANE',
    variable_level_3='',
    detailed_variable='',
    identifier='7c892333-f4c4-4f81-b825-d6a86e107e9f'
  ),
  '35721fc2-a968-487f-ad85-6307a18e4af6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='CARBON AND HYDROCARBON COMPOUNDS',
    variable_level_2='METHYL CYANIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='35721fc2-a968-487f-ad85-6307a18e4af6'
  ),
  '06d230f1-08f8-48cc-9bbd-5f2358a84d13': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='CARBON AND HYDROCARBON COMPOUNDS',
    variable_level_2='NON-METHANE HYDROCARBONS/VOLATILE ORGANIC COMPOUNDS',
    variable_level_3='',
    detailed_variable='',
    identifier='06d230f1-08f8-48cc-9bbd-5f2358a84d13'
  ),
  'd46a5046-e1c6-4a09-a2f1-db6a21eda611': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d46a5046-e1c6-4a09-a2f1-db6a21eda611'
  ),
  '39c478bd-620e-455c-904d-4621965e376c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='BROMINE MONOXIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='39c478bd-620e-455c-904d-4621965e376c'
  ),
  '1ecb1e7c-50fc-4951-b610-5140475d87ed': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='CARBON TETRACHLORIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='1ecb1e7c-50fc-4951-b610-5140475d87ed'
  ),
  'cf96b289-d316-4abc-8540-b8849e2f6140': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='CARBON TETRAFLUORIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='cf96b289-d316-4abc-8540-b8849e2f6140'
  ),
  'cbfabd7a-7032-4f67-acd8-8f6f1e026eff': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='CARBONYL FLUORIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='cbfabd7a-7032-4f67-acd8-8f6f1e026eff'
  ),
  'a56d397b-bff5-4a14-b54c-366470e023c7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='CHLORINE DIOXIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='a56d397b-bff5-4a14-b54c-366470e023c7'
  ),
  '6f96d1bd-f6ba-437a-9079-c575c4822248': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='CHLORINE MONOXIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='6f96d1bd-f6ba-437a-9079-c575c4822248'
  ),
  'a9104127-6846-4123-8ab0-b65c61a0018d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='CHLORINE NITRATE',
    variable_level_3='',
    detailed_variable='',
    identifier='a9104127-6846-4123-8ab0-b65c61a0018d'
  ),
  'e78ae4ce-807a-4417-ad6e-a458c6da6638': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='CHLOROFLUOROCARBONS',
    variable_level_3='',
    detailed_variable='',
    identifier='e78ae4ce-807a-4417-ad6e-a458c6da6638'
  ),
  '94472216-6cd7-434b-beec-17067fb69b2e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='CHLOROFLUOROCARBONS',
    variable_level_3='CFC-11',
    detailed_variable='',
    identifier='94472216-6cd7-434b-beec-17067fb69b2e'
  ),
  '97efdb7f-b2aa-4a6d-b338-30f3ad849c1f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='CHLOROFLUOROCARBONS',
    variable_level_3='CFC-12',
    detailed_variable='',
    identifier='97efdb7f-b2aa-4a6d-b338-30f3ad849c1f'
  ),
  '13588158-07b6-4294-a00c-fa095b6ad4fd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='HALOCARBONS',
    variable_level_3='',
    detailed_variable='',
    identifier='13588158-07b6-4294-a00c-fa095b6ad4fd'
  ),
  '33e3c858-25ee-4a5e-a938-93779679ed06': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='HALONS',
    variable_level_3='',
    detailed_variable='',
    identifier='33e3c858-25ee-4a5e-a938-93779679ed06'
  ),
  'f6b97280-74d0-4233-bd17-f9f3d9dd21c2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='HYDROCHLOROFLUOROCARBONS',
    variable_level_3='',
    detailed_variable='',
    identifier='f6b97280-74d0-4233-bd17-f9f3d9dd21c2'
  ),
  'f2464ebf-811a-43c2-bd50-9e80e03e07b3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='HYDROCHLOROFLUOROCARBONS',
    variable_level_3='HCFC-22',
    detailed_variable='',
    identifier='f2464ebf-811a-43c2-bd50-9e80e03e07b3'
  ),
  'ed5106fd-a73f-4203-87a3-9c9e7e85dcfc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='HYDROFLUOROCARBONS',
    variable_level_3='',
    detailed_variable='',
    identifier='ed5106fd-a73f-4203-87a3-9c9e7e85dcfc'
  ),
  '146a0a0b-1b42-41a6-b1f7-a27615b006a0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='HYDROGEN CHLORIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='146a0a0b-1b42-41a6-b1f7-a27615b006a0'
  ),
  'ff9f8056-84d6-4fbc-abe0-9b6e82ed3f5e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='HYDROGEN FLUORIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='ff9f8056-84d6-4fbc-abe0-9b6e82ed3f5e'
  ),
  '27d63fe6-9970-46fd-9b22-a58e52efc57b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='HYPOCHLOROUS ACID',
    variable_level_3='',
    detailed_variable='',
    identifier='27d63fe6-9970-46fd-9b22-a58e52efc57b'
  ),
  '228c14d1-e9bf-4c25-a67b-92c99bc2a8b7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='METHANOL',
    variable_level_3='',
    detailed_variable='',
    identifier='228c14d1-e9bf-4c25-a67b-92c99bc2a8b7'
  ),
  '9b6ca807-7719-48aa-864d-ebb45a519ff8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='METHYL BROMIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='9b6ca807-7719-48aa-864d-ebb45a519ff8'
  ),
  '676248f0-75cd-466d-93f1-351440027c82': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='METHYL CHLORIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='676248f0-75cd-466d-93f1-351440027c82'
  ),
  '50753922-d435-4c1a-a4ad-8caa9a67afcb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HALOCARBONS AND HALOGENS',
    variable_level_2='METHYL FLUORIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='50753922-d435-4c1a-a4ad-8caa9a67afcb'
  ),
  'd8dcfd36-f71c-499f-84f5-43da9fee26c5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HYDROGEN COMPOUNDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d8dcfd36-f71c-499f-84f5-43da9fee26c5'
  ),
  '4904a081-ffb5-430c-b659-08cd55d41818': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HYDROGEN COMPOUNDS',
    variable_level_2='DEUTERIUM OXIDE/HEAVY WATER',
    variable_level_3='',
    detailed_variable='',
    identifier='4904a081-ffb5-430c-b659-08cd55d41818'
  ),
  '3a79e7ec-4ab3-44c6-84bd-d0a67788453f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HYDROGEN COMPOUNDS',
    variable_level_2='HYDROGEN OXIDES',
    variable_level_3='',
    detailed_variable='',
    identifier='3a79e7ec-4ab3-44c6-84bd-d0a67788453f'
  ),
  '4a4379fb-1dbb-40ff-8c74-f2b8edda55ec': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HYDROGEN COMPOUNDS',
    variable_level_2='HYDROGEN OXIDES',
    variable_level_3='HYDROGEN OXIDE PROFILES',
    detailed_variable='',
    identifier='4a4379fb-1dbb-40ff-8c74-f2b8edda55ec'
  ),
  'f1550163-0584-4935-80bb-63c3b6ab8cfc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HYDROGEN COMPOUNDS',
    variable_level_2='HYDROGEN OXIDES',
    variable_level_3='HYDROGEN OXIDE SURFACE',
    detailed_variable='',
    identifier='f1550163-0584-4935-80bb-63c3b6ab8cfc'
  ),
  'f259ddd3-87d3-4a2f-b39d-ed3cce629f3d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HYDROGEN COMPOUNDS',
    variable_level_2='HYDROGEN-DEUTERIUM OXIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='f259ddd3-87d3-4a2f-b39d-ed3cce629f3d'
  ),
  'd8494f01-bcec-4232-ad78-fbd92c242e62': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HYDROGEN COMPOUNDS',
    variable_level_2='HYDROPEROXY',
    variable_level_3='',
    detailed_variable='',
    identifier='d8494f01-bcec-4232-ad78-fbd92c242e62'
  ),
  '5b49fd6d-3759-4b61-8b04-8309f38b2f90': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HYDROGEN COMPOUNDS',
    variable_level_2='HYDROXYL',
    variable_level_3='',
    detailed_variable='',
    identifier='5b49fd6d-3759-4b61-8b04-8309f38b2f90'
  ),
  'e073c9d4-5a61-436c-8890-2695c4e825eb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='HYDROGEN COMPOUNDS',
    variable_level_2='MOLECULAR HYDROGEN',
    variable_level_3='',
    detailed_variable='',
    identifier='e073c9d4-5a61-436c-8890-2695c4e825eb'
  ),
  '9e5ec924-2fd3-4cbb-a7eb-ffde114d0cb9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='NITROGEN COMPOUNDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9e5ec924-2fd3-4cbb-a7eb-ffde114d0cb9'
  ),
  '6a745a5e-829c-43f5-8d5a-6fb549e7b81b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='NITROGEN COMPOUNDS',
    variable_level_2='AMMONIA',
    variable_level_3='',
    detailed_variable='',
    identifier='6a745a5e-829c-43f5-8d5a-6fb549e7b81b'
  ),
  'b7bbed0f-24a1-44d8-a10d-92541cd2c05b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='NITROGEN COMPOUNDS',
    variable_level_2='ATMOSPHERIC NITRIC ACID',
    variable_level_3='',
    detailed_variable='',
    identifier='b7bbed0f-24a1-44d8-a10d-92541cd2c05b'
  ),
  '6c5a6bbe-a12f-4030-9220-2013db36cf47': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='NITROGEN COMPOUNDS',
    variable_level_2='CLOUD-SCREENED TOTAL COLUMN NITROGEN DIOXIDE (NO2)',
    variable_level_3='',
    detailed_variable='',
    identifier='6c5a6bbe-a12f-4030-9220-2013db36cf47'
  ),
  'd92ae6cc-989b-45b8-92d3-68008356c2b0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='NITROGEN COMPOUNDS',
    variable_level_2='CLOUD-SCREENED TROPOSHERIC COLUMN NO2',
    variable_level_3='',
    detailed_variable='',
    identifier='d92ae6cc-989b-45b8-92d3-68008356c2b0'
  ),
  '9ca9519d-c62b-42ea-8c91-cad06cfc59cb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='NITROGEN COMPOUNDS',
    variable_level_2='DINITROGEN PENTOXIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='9ca9519d-c62b-42ea-8c91-cad06cfc59cb'
  ),
  '3c3b37d4-b934-4057-b8e4-438523ae88e3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='NITROGEN COMPOUNDS',
    variable_level_2='MOLECULAR NITROGEN',
    variable_level_3='',
    detailed_variable='',
    identifier='3c3b37d4-b934-4057-b8e4-438523ae88e3'
  ),
  '82a60ed8-5414-4ce0-858c-c50b27b12bc8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='NITROGEN COMPOUNDS',
    variable_level_2='NITRIC OXIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='82a60ed8-5414-4ce0-858c-c50b27b12bc8'
  ),
  'ef36cb15-ad64-4bdc-9331-42cc5b493671': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='NITROGEN COMPOUNDS',
    variable_level_2='NITROGEN',
    variable_level_3='',
    detailed_variable='',
    identifier='ef36cb15-ad64-4bdc-9331-42cc5b493671'
  ),
  'f8e65155-27c1-483e-a9b8-85399897c3ae': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='NITROGEN COMPOUNDS',
    variable_level_2='NITROGEN DIOXIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='f8e65155-27c1-483e-a9b8-85399897c3ae'
  ),
  'e82ebd1c-8241-4ca0-95a9-a6e1432519cd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='NITROGEN COMPOUNDS',
    variable_level_2='NITROGEN OXIDES',
    variable_level_3='',
    detailed_variable='',
    identifier='e82ebd1c-8241-4ca0-95a9-a6e1432519cd'
  ),
  'cf08917f-4cef-456f-99b0-57dc468da877': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='NITROGEN COMPOUNDS',
    variable_level_2='NITROUS OXIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='cf08917f-4cef-456f-99b0-57dc468da877'
  ),
  'e66fdcc7-3a94-48a0-aa21-5964f9ddaf23': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='NITROGEN COMPOUNDS',
    variable_level_2='PEROXYACETYL NITRATE',
    variable_level_3='',
    detailed_variable='',
    identifier='e66fdcc7-3a94-48a0-aa21-5964f9ddaf23'
  ),
  'd44d3115-91d1-4655-9e6e-babfe39e1632': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='NITROGEN COMPOUNDS',
    variable_level_2='Peroxyacyl Nitrate',
    variable_level_3='',
    detailed_variable='',
    identifier='d44d3115-91d1-4655-9e6e-babfe39e1632'
  ),
  'c2b7b126-8737-4933-ba27-fe64226a0363': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='NITROGEN COMPOUNDS',
    variable_level_2='PEROXYNITRIC ACID',
    variable_level_3='',
    detailed_variable='',
    identifier='c2b7b126-8737-4933-ba27-fe64226a0363'
  ),
  '18d9ddbb-66cc-4b92-aa8d-2395ab3a17ce': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='NOBLE GAS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='18d9ddbb-66cc-4b92-aa8d-2395ab3a17ce'
  ),
  'a81f6b17-2a16-4600-84b6-bcbcc5c29d2f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='NOBLE GAS',
    variable_level_2='ATMOSPHERIC RADON',
    variable_level_3='',
    detailed_variable='',
    identifier='a81f6b17-2a16-4600-84b6-bcbcc5c29d2f'
  ),
  '0857df34-93ca-4c8e-a909-3621ea1dcbe7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='NOBLE GAS',
    variable_level_2='ATMOSPHERIC RADON',
    variable_level_3='RADON PROFILES',
    detailed_variable='',
    identifier='0857df34-93ca-4c8e-a909-3621ea1dcbe7'
  ),
  '7b6f32f2-7123-488b-9c9b-98acdc0c6f5a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='NOBLE GAS',
    variable_level_2='ATMOSPHERIC RADON',
    variable_level_3='RADON SURFACE',
    detailed_variable='',
    identifier='7b6f32f2-7123-488b-9c9b-98acdc0c6f5a'
  ),
  '4cc9b4fa-5097-447f-914c-eb90820938c6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='OXYGEN COMPOUNDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4cc9b4fa-5097-447f-914c-eb90820938c6'
  ),
  'dd316647-9043-40c3-9329-f22f9215fefa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='OXYGEN COMPOUNDS',
    variable_level_2='ATMOSPHERIC OZONE',
    variable_level_3='',
    detailed_variable='',
    identifier='dd316647-9043-40c3-9329-f22f9215fefa'
  ),
  '959878c8-b02a-4bb8-ad10-96ae31ebe59f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='OXYGEN COMPOUNDS',
    variable_level_2='ATMOSPHERIC OZONE',
    variable_level_3='OZONE PROFILES',
    detailed_variable='',
    identifier='959878c8-b02a-4bb8-ad10-96ae31ebe59f'
  ),
  'e00bfcb8-8968-400d-af2e-86a288f3443f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='OXYGEN COMPOUNDS',
    variable_level_2='ATMOSPHERIC OZONE',
    variable_level_3='OZONE SURFACE',
    detailed_variable='',
    identifier='e00bfcb8-8968-400d-af2e-86a288f3443f'
  ),
  'b9107ec3-c777-4e71-9046-55bd7ed57ef0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='OXYGEN COMPOUNDS',
    variable_level_2='ATMOSPHERIC OZONE',
    variable_level_3='TOTAL OZONE',
    detailed_variable='',
    identifier='b9107ec3-c777-4e71-9046-55bd7ed57ef0'
  ),
  '75662ed3-35c5-41ee-abba-51ce435d1b31': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='OXYGEN COMPOUNDS',
    variable_level_2='HYDROGEN PEROXIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='75662ed3-35c5-41ee-abba-51ce435d1b31'
  ),
  '61f4f3d0-7895-4cce-94e3-d249001d5ee8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='OXYGEN COMPOUNDS',
    variable_level_2='MOLECULAR OXYGEN',
    variable_level_3='',
    detailed_variable='',
    identifier='61f4f3d0-7895-4cce-94e3-d249001d5ee8'
  ),
  '6433e330-3797-4cf9-a8ba-d26d39624459': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='PHOTOCHEMISTRY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6433e330-3797-4cf9-a8ba-d26d39624459'
  ),
  '0fd2b083-e65c-443b-9794-2c355ebac06b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='PHOTOCHEMISTRY',
    variable_level_2='PHOTOLYSIS RATES',
    variable_level_3='',
    detailed_variable='',
    identifier='0fd2b083-e65c-443b-9794-2c355ebac06b'
  ),
  'b80a242d-d5f5-4a5f-976c-6f6fe2ab6b2c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='SULFUR COMPOUNDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b80a242d-d5f5-4a5f-976c-6f6fe2ab6b2c'
  ),
  'bde65cfd-faec-4656-bc27-22dfe30912b7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='SULFUR COMPOUNDS',
    variable_level_2='CARBONYL SULFIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='bde65cfd-faec-4656-bc27-22dfe30912b7'
  ),
  '5d282de9-162a-4aeb-a48d-4569fbbd5205': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='SULFUR COMPOUNDS',
    variable_level_2='DIMETHYL SULFIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='5d282de9-162a-4aeb-a48d-4569fbbd5205'
  ),
  '2ab4134d-1ac7-4421-a7a6-659a542aff4c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='SULFUR COMPOUNDS',
    variable_level_2='SULFATE',
    variable_level_3='',
    detailed_variable='',
    identifier='2ab4134d-1ac7-4421-a7a6-659a542aff4c'
  ),
  'f5717312-c3ca-4492-a166-9f17c6d9b273': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='SULFUR COMPOUNDS',
    variable_level_2='SULFUR DIOXIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='f5717312-c3ca-4492-a166-9f17c6d9b273'
  ),
  '25664d90-f32e-490f-97e4-720e3903d775': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='SULFUR COMPOUNDS',
    variable_level_2='SULFUR DIOXIDE',
    variable_level_3='SULFUR DIOXIDE PROFILES',
    detailed_variable='',
    identifier='25664d90-f32e-490f-97e4-720e3903d775'
  ),
  '022f3225-3b23-4e56-a6ea-c21f08f9e179': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='SULFUR COMPOUNDS',
    variable_level_2='SULFUR DIOXIDE',
    variable_level_3='SULFUR DIOXIDE SURFACE',
    detailed_variable='',
    identifier='022f3225-3b23-4e56-a6ea-c21f08f9e179'
  ),
  'b4c7dec8-6a9a-4424-8f80-4ad1ed0bc5ec': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='SULFUR COMPOUNDS',
    variable_level_2='SULFUR HEXAFLUORIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='b4c7dec8-6a9a-4424-8f80-4ad1ed0bc5ec'
  ),
  '958983a1-feee-4139-8a67-b059382e6c06': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='SULFUR COMPOUNDS',
    variable_level_2='SULFUR HEXAFLUORIDE',
    variable_level_3='SULFUR HEXAFLUORIDE PROFILES',
    detailed_variable='',
    identifier='958983a1-feee-4139-8a67-b059382e6c06'
  ),
  'fceaf7e0-c22b-40a1-ba1d-37616031dba6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='SULFUR COMPOUNDS',
    variable_level_2='SULFUR HEXAFLUORIDE',
    variable_level_3='SULFUR HEXAFLUORIDE SURFACE',
    detailed_variable='',
    identifier='fceaf7e0-c22b-40a1-ba1d-37616031dba6'
  ),
  'cc676fb2-cf17-413d-bb00-0b95d231f157': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='SULFUR COMPOUNDS',
    variable_level_2='SULFUR OXIDES',
    variable_level_3='',
    detailed_variable='',
    identifier='cc676fb2-cf17-413d-bb00-0b95d231f157'
  ),
  '2d36c283-2fe3-4a08-aeb3-6a8146e79bb3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='TRACE ELEMENTS/TRACE METALS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2d36c283-2fe3-4a08-aeb3-6a8146e79bb3'
  ),
  '4dd22dc9-1db4-4187-a2b7-f5b76d666055': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC CHEMISTRY',
    variable_level_1='TRACE GASES/TRACE SPECIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4dd22dc9-1db4-4187-a2b7-f5b76d666055'
  ),
  '0af72e0e-52a5-4695-9eaf-d6fbb7991039': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC ELECTRICITY',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0af72e0e-52a5-4695-9eaf-d6fbb7991039'
  ),
  '12b1cc7c-cb81-4851-9163-19c04a8ffd1c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC ELECTRICITY',
    variable_level_1='ATMOSPHERIC CONDUCTIVITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='12b1cc7c-cb81-4851-9163-19c04a8ffd1c'
  ),
  '41f27172-14f6-4940-9b7b-f3d4db69e0c6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC ELECTRICITY',
    variable_level_1='ELECTRIC FIELD',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='41f27172-14f6-4940-9b7b-f3d4db69e0c6'
  ),
  '637ac172-e624-4ae0-aac4-0d1adcc889a2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC ELECTRICITY',
    variable_level_1='LIGHTNING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='637ac172-e624-4ae0-aac4-0d1adcc889a2'
  ),
  'cac28264-0788-49a9-bb6a-c2251b0b325c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC ELECTRICITY',
    variable_level_1='TOTAL ELECTRON CONTENT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='cac28264-0788-49a9-bb6a-c2251b0b325c'
  ),
  '08fd82a1-4370-46a2-82ea-94c0f91498a7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC PRESSURE',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='08fd82a1-4370-46a2-82ea-94c0f91498a7'
  ),
  '7a4c13fb-3f2d-49d9-9158-ed84743355fc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC PRESSURE',
    variable_level_1='AIR MASS/DENSITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7a4c13fb-3f2d-49d9-9158-ed84743355fc'
  ),
  '178694aa-5f0a-4de5-a193-74e323dc6aa9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC PRESSURE',
    variable_level_1='ANTICYCLONES/CYCLONES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='178694aa-5f0a-4de5-a193-74e323dc6aa9'
  ),
  '9efbc088-ba8c-4c9c-a458-ad6ad63f4188': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC PRESSURE',
    variable_level_1='ATMOSPHERIC PRESSURE MEASUREMENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9efbc088-ba8c-4c9c-a458-ad6ad63f4188'
  ),
  '5d7e487d-0ec4-40ef-9811-401779c31794': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC PRESSURE',
    variable_level_1='DIFFERENTIAL PRESSURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5d7e487d-0ec4-40ef-9811-401779c31794'
  ),
  '7e6f7c15-32e7-4b6e-bd35-7bff4bc03caf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC PRESSURE',
    variable_level_1='GRAVITY WAVE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7e6f7c15-32e7-4b6e-bd35-7bff4bc03caf'
  ),
  'a5aa7055-642d-4442-9b4b-76a759e15257': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC PRESSURE',
    variable_level_1='HYDROSTATIC PRESSURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a5aa7055-642d-4442-9b4b-76a759e15257'
  ),
  'c0656cbc-5d94-4945-bbfd-1c8eabb059b2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC PRESSURE',
    variable_level_1='OSCILLATIONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c0656cbc-5d94-4945-bbfd-1c8eabb059b2'
  ),
  'f51a3caf-c5ec-496a-8dd3-854d9bb994e7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC PRESSURE',
    variable_level_1='PLANETARY BOUNDARY LAYER HEIGHT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f51a3caf-c5ec-496a-8dd3-854d9bb994e7'
  ),
  '6f262b9b-2cb8-4745-ae41-5fff23c72a1e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC PRESSURE',
    variable_level_1='PLANETARY/ROSSBY WAVES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6f262b9b-2cb8-4745-ae41-5fff23c72a1e'
  ),
  '011bed30-f5b6-4b46-a7ce-797851f24f24': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC PRESSURE',
    variable_level_1='PRESSURE ANOMALIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='011bed30-f5b6-4b46-a7ce-797851f24f24'
  ),
  'fa98caa0-54dc-465e-9bde-cdf4da905994': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC PRESSURE',
    variable_level_1='PRESSURE TENDENCY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fa98caa0-54dc-465e-9bde-cdf4da905994'
  ),
  'be027470-35ab-4ebb-a213-5f557cca71c8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC PRESSURE',
    variable_level_1='PRESSURE THICKNESS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='be027470-35ab-4ebb-a213-5f557cca71c8'
  ),
  '07ce145c-9936-4675-b4a7-8710e39aa391': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC PRESSURE',
    variable_level_1='SEA LEVEL PRESSURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='07ce145c-9936-4675-b4a7-8710e39aa391'
  ),
  '622c44b4-e307-4c11-af4d-8104de7086e5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC PRESSURE',
    variable_level_1='STATIC PRESSURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='622c44b4-e307-4c11-af4d-8104de7086e5'
  ),
  'b54de5cd-4475-4c7b-acbc-4eb529b9396e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC PRESSURE',
    variable_level_1='SURFACE PRESSURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b54de5cd-4475-4c7b-acbc-4eb529b9396e'
  ),
  'b13a29a1-47a0-4d8b-a017-398b364dc202': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC PRESSURE',
    variable_level_1='TOPOGRAPHIC WAVES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b13a29a1-47a0-4d8b-a017-398b364dc202'
  ),
  '4ad0c52d-6449-48ff-8678-adc6b2cebcb7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4ad0c52d-6449-48ff-8678-adc6b2cebcb7'
  ),
  '061f7fd0-67af-42bf-bc9f-5a007c146f65': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='ABSORPTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='061f7fd0-67af-42bf-bc9f-5a007c146f65'
  ),
  'ec839718-ba64-4bc5-8458-fae7390e11c4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='ACTINIC FLUX',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ec839718-ba64-4bc5-8458-fae7390e11c4'
  ),
  'bf22e55d-fbff-4eaf-8592-68be24e2bc32': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='AIRGLOW',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bf22e55d-fbff-4eaf-8592-68be24e2bc32'
  ),
  '48c16952-b6e0-40cd-b6dd-7cdbf5a443a1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='ALBEDO',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='48c16952-b6e0-40cd-b6dd-7cdbf5a443a1'
  ),
  '31a14270-6275-4155-961f-b78b60ee05f7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='ANISOTROPY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='31a14270-6275-4155-961f-b78b60ee05f7'
  ),
  '1ed8ac8d-3a66-4b86-be30-a5b79b3806d2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='ATMOSPHERIC EMITTED RADIATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1ed8ac8d-3a66-4b86-be30-a5b79b3806d2'
  ),
  '06a24fd2-38b6-4a4a-a0cf-1abf149283e2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='ATMOSPHERIC HEATING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='06a24fd2-38b6-4a4a-a0cf-1abf149283e2'
  ),
  '7cacbfdf-71a3-4fac-b690-9aa54e4060dd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='EARTH RADIATION BUDGET',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7cacbfdf-71a3-4fac-b690-9aa54e4060dd'
  ),
  '49c8770a-2eb7-40f1-aab0-9c12d3aed031': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='EMISSIVITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='49c8770a-2eb7-40f1-aab0-9c12d3aed031'
  ),
  '46a3c823-727d-4c3c-b09d-e3e3fcaa43a5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='HEAT FLUX',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='46a3c823-727d-4c3c-b09d-e3e3fcaa43a5'
  ),
  '6b3be650-6625-40b5-9b40-9e7c8a9fd336': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='INCOMING SOLAR RADIATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6b3be650-6625-40b5-9b40-9e7c8a9fd336'
  ),
  '68323795-3614-462f-8259-bd5293620799': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='LONGWAVE RADIATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='68323795-3614-462f-8259-bd5293620799'
  ),
  'bd6270d3-dd4d-41ed-a6c2-58abe926017c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='LONGWAVE RADIATION',
    variable_level_2='DOWNWELLING LONGWAVE RADIATION',
    variable_level_3='',
    detailed_variable='',
    identifier='bd6270d3-dd4d-41ed-a6c2-58abe926017c'
  ),
  '82f5ea3b-63f3-47a8-ad41-24cd5914afd1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='LONGWAVE RADIATION',
    variable_level_2='UPWELLING LONGWAVE RADIATION',
    variable_level_3='',
    detailed_variable='',
    identifier='82f5ea3b-63f3-47a8-ad41-24cd5914afd1'
  ),
  '50ee8910-449b-46c8-a59b-1cd76d632b44': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='NET RADIATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='50ee8910-449b-46c8-a59b-1cd76d632b44'
  ),
  '13723b5d-1945-4e62-8672-4535ffdddb87': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='OPTICAL DEPTH/THICKNESS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='13723b5d-1945-4e62-8672-4535ffdddb87'
  ),
  '006b1ea6-222d-4740-b220-03886d49cd81': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='OUTGOING LONGWAVE RADIATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='006b1ea6-222d-4740-b220-03886d49cd81'
  ),
  'a87d6473-3a03-4bc6-aa21-6157fae96b8e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='POLARIZED REFLECTANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a87d6473-3a03-4bc6-aa21-6157fae96b8e'
  ),
  '107582ef-a356-4afa-a9a4-4e1d2200c134': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='RADIATIVE FLUX',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='107582ef-a356-4afa-a9a4-4e1d2200c134'
  ),
  '4fad64ce-32fe-413d-8b55-c78000d1980c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='RADIATIVE FORCING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4fad64ce-32fe-413d-8b55-c78000d1980c'
  ),
  'bdfd401f-7eed-4a48-bd6f-f0c2a890594a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='REFLECTANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bdfd401f-7eed-4a48-bd6f-f0c2a890594a'
  ),
  'c1cb99a3-67a4-4ad8-855a-e98a1e0b23ac': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='REFLECTANCE',
    variable_level_2='TOP OF ATMOSPHERE (TOA) REFLECTANCE',
    variable_level_3='',
    detailed_variable='',
    identifier='c1cb99a3-67a4-4ad8-855a-e98a1e0b23ac'
  ),
  'ec9e0b6a-1315-4569-93bc-0f1190bb8c08': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='SCATTERING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ec9e0b6a-1315-4569-93bc-0f1190bb8c08'
  ),
  'a8f5c969-34e9-4284-afb5-ff2113f5f881': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='SHORTWAVE RADIATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a8f5c969-34e9-4284-afb5-ff2113f5f881'
  ),
  '74de6492-617b-4046-8a4e-1336c3238a7e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='SHORTWAVE RADIATION',
    variable_level_2='DOWNWELLING SHORTWAVE RADIATION',
    variable_level_3='',
    detailed_variable='',
    identifier='74de6492-617b-4046-8a4e-1336c3238a7e'
  ),
  'de7647c9-b129-4cba-afe4-63fa9998206e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='SOLAR IRRADIANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='de7647c9-b129-4cba-afe4-63fa9998206e'
  ),
  'e1af236f-ee88-4b10-8feb-70d9e09f90be': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='SOLAR IRRADIANCE',
    variable_level_2='SHORTWAVE DOWNWARD IRRADIANCE',
    variable_level_3='',
    detailed_variable='',
    identifier='e1af236f-ee88-4b10-8feb-70d9e09f90be'
  ),
  'a0f3474e-9a54-4a82-97c4-43864b48df4c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='SOLAR RADIATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a0f3474e-9a54-4a82-97c4-43864b48df4c'
  ),
  'b7a45c57-b652-469a-a3f2-8d38555bf478': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='SPECTRAL IRRADIANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b7a45c57-b652-469a-a3f2-8d38555bf478'
  ),
  '86c95fdb-17b9-4224-a020-b1aacbea00fd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='SUNSHINE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='86c95fdb-17b9-4224-a020-b1aacbea00fd'
  ),
  '714be1d7-2012-4a98-bdd5-02bbcadf69d8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='TRANSMITTANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='714be1d7-2012-4a98-bdd5-02bbcadf69d8'
  ),
  '90e7fd13-2da2-4ba6-9e0c-dbecdf7c2215': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='ULTRAVIOLET RADIATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='90e7fd13-2da2-4ba6-9e0c-dbecdf7c2215'
  ),
  '782b60de-9ac2-4e6c-9dfa-cd52c4cf1ea0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC RADIATION',
    variable_level_1='ULTRAVIOLET RADIATION',
    variable_level_2='UV SPECTRAL',
    variable_level_3='',
    detailed_variable='',
    identifier='782b60de-9ac2-4e6c-9dfa-cd52c4cf1ea0'
  ),
  '35e1f93b-99b3-4430-b477-0ecafa80d67a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='35e1f93b-99b3-4430-b477-0ecafa80d67a'
  ),
  'ff5d5c12-74d9-435d-9164-1c9d69f967d7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='ATMOSPHERIC STABILITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ff5d5c12-74d9-435d-9164-1c9d69f967d7'
  ),
  '25d73bcf-c8d4-4c0e-ac98-8f3e98677e73': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='ATMOSPHERIC TEMPERATURE INDICES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='25d73bcf-c8d4-4c0e-ac98-8f3e98677e73'
  ),
  '1d527151-57b2-49ed-9937-c1756a704ce9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='ATMOSPHERIC TEMPERATURE INDICES',
    variable_level_2='COMMON SENSE CLIMATE INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='1d527151-57b2-49ed-9937-c1756a704ce9'
  ),
  '2590519a-c2bb-448a-b2f3-d10aaa7e057c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='ATMOSPHERIC TEMPERATURE INDICES',
    variable_level_2='COOLING DEGREE DAYS',
    variable_level_3='',
    detailed_variable='',
    identifier='2590519a-c2bb-448a-b2f3-d10aaa7e057c'
  ),
  '2329bf96-d927-4993-95f9-93551d787ad7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='ATMOSPHERIC TEMPERATURE INDICES',
    variable_level_2='FREEZING INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='2329bf96-d927-4993-95f9-93551d787ad7'
  ),
  'a43f9a02-769d-4343-8790-fa29a0507f44': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='ATMOSPHERIC TEMPERATURE INDICES',
    variable_level_2='GROWING DEGREE DAYS',
    variable_level_3='',
    detailed_variable='',
    identifier='a43f9a02-769d-4343-8790-fa29a0507f44'
  ),
  '289ca013-0526-49e0-8b87-51513702e8f4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='ATMOSPHERIC TEMPERATURE INDICES',
    variable_level_2='HEAT INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='289ca013-0526-49e0-8b87-51513702e8f4'
  ),
  '349b4322-26ff-4b3c-90fb-b3b1afd20755': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='ATMOSPHERIC TEMPERATURE INDICES',
    variable_level_2='HEATING DEGREE DAYS',
    variable_level_3='',
    detailed_variable='',
    identifier='349b4322-26ff-4b3c-90fb-b3b1afd20755'
  ),
  '37ae8d4e-fe97-43d3-b8ee-a597e4ebfe87': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='ATMOSPHERIC TEMPERATURE INDICES',
    variable_level_2='RESIDENTIAL ENERGY DEMAND TEMPERATURE INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='37ae8d4e-fe97-43d3-b8ee-a597e4ebfe87'
  ),
  '1c441454-851f-48e0-abb3-053ae44c0d4e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='ATMOSPHERIC TEMPERATURE INDICES',
    variable_level_2='TEMPERATURE CONCENTRATION INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='1c441454-851f-48e0-abb3-053ae44c0d4e'
  ),
  '746c49af-3e36-4f0a-b488-e024314d6cfa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='ATMOSPHERIC TEMPERATURE INDICES',
    variable_level_2='THAWING INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='746c49af-3e36-4f0a-b488-e024314d6cfa'
  ),
  'd50d0685-f42f-4693-9458-eddb9ccf5704': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='ATMOSPHERIC TEMPERATURE INDICES',
    variable_level_2='WIND CHILL INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='d50d0685-f42f-4693-9458-eddb9ccf5704'
  ),
  '5a7bb095-4d12-4232-bc75-b8e82197cb92': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='SURFACE TEMPERATURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5a7bb095-4d12-4232-bc75-b8e82197cb92'
  ),
  'f634ab55-de40-4d0b-93bc-691bf5408ccb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='SURFACE TEMPERATURE',
    variable_level_2='AIR TEMPERATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='f634ab55-de40-4d0b-93bc-691bf5408ccb'
  ),
  'e9c3b6ca-a534-4f3e-82de-b8b921e8f312': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='SURFACE TEMPERATURE',
    variable_level_2='BOUNDARY LAYER TEMPERATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='e9c3b6ca-a534-4f3e-82de-b8b921e8f312'
  ),
  '6e923275-f9e3-4faf-8a7f-2c96f3d5a280': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='SURFACE TEMPERATURE',
    variable_level_2='DEICED TEMPERATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='6e923275-f9e3-4faf-8a7f-2c96f3d5a280'
  ),
  '0c28d9e4-c848-4628-9c00-45a540707b59': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='SURFACE TEMPERATURE',
    variable_level_2='DEW POINT TEMPERATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='0c28d9e4-c848-4628-9c00-45a540707b59'
  ),
  'a5e36040-cc5e-46d1-aeee-f49902e943b2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='SURFACE TEMPERATURE',
    variable_level_2='DEW POINT TEMPERATURE',
    variable_level_3='DEWPOINT DEPRESSION',
    detailed_variable='',
    identifier='a5e36040-cc5e-46d1-aeee-f49902e943b2'
  ),
  '5164162a-60eb-4c94-a0f0-2caaa3bb1754': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='SURFACE TEMPERATURE',
    variable_level_2='MAXIMUM/MINIMUM TEMPERATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='5164162a-60eb-4c94-a0f0-2caaa3bb1754'
  ),
  'ce6a6b3a-df4f-4bd7-a931-7ee874ee9efe': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='SURFACE TEMPERATURE',
    variable_level_2='MAXIMUM/MINIMUM TEMPERATURE',
    variable_level_3='24 HOUR MAXIMUM TEMPERATURE',
    detailed_variable='',
    identifier='ce6a6b3a-df4f-4bd7-a931-7ee874ee9efe'
  ),
  '5c7f35d5-a3ec-4010-b1c3-6e98ac29dc3f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='SURFACE TEMPERATURE',
    variable_level_2='MAXIMUM/MINIMUM TEMPERATURE',
    variable_level_3='24 HOUR MINIMUM TEMPERATURE',
    detailed_variable='',
    identifier='5c7f35d5-a3ec-4010-b1c3-6e98ac29dc3f'
  ),
  'e56bcf72-f331-4545-948f-73fe0193b1bd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='SURFACE TEMPERATURE',
    variable_level_2='MAXIMUM/MINIMUM TEMPERATURE',
    variable_level_3='6 HOUR MAXIMUM TEMPERATURE',
    detailed_variable='',
    identifier='e56bcf72-f331-4545-948f-73fe0193b1bd'
  ),
  'c9ab66f1-91c6-497a-b8d6-4688160b0e16': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='SURFACE TEMPERATURE',
    variable_level_2='MAXIMUM/MINIMUM TEMPERATURE',
    variable_level_3='6 HOUR MINIMUM TEMPERATURE',
    detailed_variable='',
    identifier='c9ab66f1-91c6-497a-b8d6-4688160b0e16'
  ),
  '7a0bd777-be0d-43c8-80eb-5ac58f4832de': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='SURFACE TEMPERATURE',
    variable_level_2='POTENTIAL TEMPERATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='7a0bd777-be0d-43c8-80eb-5ac58f4832de'
  ),
  '25fcdcb7-efd2-4d2f-ba57-92bbcc7ba69a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='SURFACE TEMPERATURE',
    variable_level_2='SKIN TEMPERATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='25fcdcb7-efd2-4d2f-ba57-92bbcc7ba69a'
  ),
  'a1588b7d-7307-4543-9908-76d7877c4010': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='SURFACE TEMPERATURE',
    variable_level_2='STATIC TEMPERATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='a1588b7d-7307-4543-9908-76d7877c4010'
  ),
  '7ca345d4-8e15-49ae-98a7-1c387f61ea85': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='SURFACE TEMPERATURE',
    variable_level_2='TEMPERATURE ANOMALIES',
    variable_level_3='',
    detailed_variable='',
    identifier='7ca345d4-8e15-49ae-98a7-1c387f61ea85'
  ),
  '449ad1fb-8010-43c7-b994-178a049d4cff': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='SURFACE TEMPERATURE',
    variable_level_2='TEMPERATURE TENDENCY',
    variable_level_3='',
    detailed_variable='',
    identifier='449ad1fb-8010-43c7-b994-178a049d4cff'
  ),
  'fd19a3f1-8eeb-49ab-bcaf-e7b4b267d415': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='SURFACE TEMPERATURE',
    variable_level_2='VIRTUAL TEMPERATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='fd19a3f1-8eeb-49ab-bcaf-e7b4b267d415'
  ),
  '926c1b80-6c11-40eb-ae7f-f5bcfdc43fac': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='UPPER AIR TEMPERATURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='926c1b80-6c11-40eb-ae7f-f5bcfdc43fac'
  ),
  '7f94b0e5-edc6-4724-bd84-404896e09afe': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='UPPER AIR TEMPERATURE',
    variable_level_2='BOUNDARY LAYER TEMPERATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='7f94b0e5-edc6-4724-bd84-404896e09afe'
  ),
  '1aa94028-9c0a-47be-8408-f7537c071530': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='UPPER AIR TEMPERATURE',
    variable_level_2='BOUNDARY LAYER TEMPERATURE',
    variable_level_3='BOUNDARY LAYER TEMPERATURE PROFILES',
    detailed_variable='',
    identifier='1aa94028-9c0a-47be-8408-f7537c071530'
  ),
  'b3e6afd7-35a6-4cdb-a066-654a17168253': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='UPPER AIR TEMPERATURE',
    variable_level_2='DEICED TEMPERATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='b3e6afd7-35a6-4cdb-a066-654a17168253'
  ),
  '76103e17-59c2-4458-972d-9ff9801e5d32': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='UPPER AIR TEMPERATURE',
    variable_level_2='DEW POINT TEMPERATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='76103e17-59c2-4458-972d-9ff9801e5d32'
  ),
  '86fb8a31-35f6-4d0e-b4b4-f9cecf961a47': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='UPPER AIR TEMPERATURE',
    variable_level_2='DEW POINT TEMPERATURE',
    variable_level_3='DEW POINT DEPRESSION',
    detailed_variable='',
    identifier='86fb8a31-35f6-4d0e-b4b4-f9cecf961a47'
  ),
  '1e76ccc7-2729-4de1-8c01-f295476ebb35': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='UPPER AIR TEMPERATURE',
    variable_level_2='TEMPERATURE ANOMALIES',
    variable_level_3='',
    detailed_variable='',
    identifier='1e76ccc7-2729-4de1-8c01-f295476ebb35'
  ),
  '72304037-ce59-451a-beeb-4258f3db296a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='UPPER AIR TEMPERATURE',
    variable_level_2='VERTICAL PROFILES',
    variable_level_3='',
    detailed_variable='',
    identifier='72304037-ce59-451a-beeb-4258f3db296a'
  ),
  '17ce714a-bd7e-41a2-ab3d-4865832f1f0a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='UPPER AIR TEMPERATURE',
    variable_level_2='VERTICAL PROFILES',
    variable_level_3='DRY ADIABATIC LAPSE RATE',
    detailed_variable='',
    identifier='17ce714a-bd7e-41a2-ab3d-4865832f1f0a'
  ),
  '050771bb-27a3-4e47-bd1b-724d1d73e20c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='UPPER AIR TEMPERATURE',
    variable_level_2='VERTICAL PROFILES',
    variable_level_3='ENVIRONMENTAL LAPSE RATE',
    detailed_variable='',
    identifier='050771bb-27a3-4e47-bd1b-724d1d73e20c'
  ),
  '4fa883a3-e312-4dbe-870e-3272de4ac76a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='UPPER AIR TEMPERATURE',
    variable_level_2='VERTICAL PROFILES',
    variable_level_3='INVERSION HEIGHT',
    detailed_variable='',
    identifier='4fa883a3-e312-4dbe-870e-3272de4ac76a'
  ),
  '65937e73-0cc0-4058-b7dc-12c418ba2ed5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='UPPER AIR TEMPERATURE',
    variable_level_2='VERTICAL PROFILES',
    variable_level_3='SATURATED ADIABATIC LAPSE RATE',
    detailed_variable='',
    identifier='65937e73-0cc0-4058-b7dc-12c418ba2ed5'
  ),
  '3afb06fa-96b7-4bf4-a6b7-b5fa626afc04': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC TEMPERATURE',
    variable_level_1='UPPER AIR TEMPERATURE',
    variable_level_2='VIRTUAL TEMPERATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='3afb06fa-96b7-4bf4-a6b7-b5fa626afc04'
  ),
  '286d2ae0-9d86-4ef0-a2b4-014843a98532': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='286d2ae0-9d86-4ef0-a2b4-014843a98532'
  ),
  '005d192a-95b9-4fc2-afed-f87da3c3dc33': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR INDICATORS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='005d192a-95b9-4fc2-afed-f87da3c3dc33'
  ),
  '31ca2413-f257-4ceb-849a-68538efecfee': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR INDICATORS',
    variable_level_2='ATMOSPHERIC RIVER',
    variable_level_3='',
    detailed_variable='',
    identifier='31ca2413-f257-4ceb-849a-68538efecfee'
  ),
  '731beb11-9418-40ec-8f2c-c4b320e8231a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR INDICATORS',
    variable_level_2='DEW POINT TEMPERATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='731beb11-9418-40ec-8f2c-c4b320e8231a'
  ),
  '427e5121-a142-41cb-a8e9-a70b7f98eb6a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR INDICATORS',
    variable_level_2='HUMIDITY',
    variable_level_3='',
    detailed_variable='',
    identifier='427e5121-a142-41cb-a8e9-a70b7f98eb6a'
  ),
  '6b61a904-b92d-45ee-9061-aa5e61c29dd2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR INDICATORS',
    variable_level_2='HUMIDITY',
    variable_level_3='ABSOLUTE HUMIDITY',
    detailed_variable='',
    identifier='6b61a904-b92d-45ee-9061-aa5e61c29dd2'
  ),
  'ea308986-ad35-4482-948c-5eb1a01be836': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR INDICATORS',
    variable_level_2='HUMIDITY',
    variable_level_3='HUMIDITY MIXING RATIO',
    detailed_variable='',
    identifier='ea308986-ad35-4482-948c-5eb1a01be836'
  ),
  'a249c68f-8249-4285-aad2-020b3c5aefc3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR INDICATORS',
    variable_level_2='HUMIDITY',
    variable_level_3='RELATIVE HUMIDITY',
    detailed_variable='',
    identifier='a249c68f-8249-4285-aad2-020b3c5aefc3'
  ),
  'ba2491a4-2498-4c9f-9adc-123078eef633': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR INDICATORS',
    variable_level_2='HUMIDITY',
    variable_level_3='SATURATION SPECIFIC HUMIDITY',
    detailed_variable='',
    identifier='ba2491a4-2498-4c9f-9adc-123078eef633'
  ),
  '811391d2-4113-4d52-9c88-47d56afda481': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR INDICATORS',
    variable_level_2='HUMIDITY',
    variable_level_3='SPECIFIC HUMIDITY',
    detailed_variable='',
    identifier='811391d2-4113-4d52-9c88-47d56afda481'
  ),
  '871f5bee-ea8d-44c0-8740-9b0153fa6ea4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR INDICATORS',
    variable_level_2='LAYERED PRECIPITABLE WATER',
    variable_level_3='',
    detailed_variable='',
    identifier='871f5bee-ea8d-44c0-8740-9b0153fa6ea4'
  ),
  '1a2332d9-fd69-4002-89a5-203d748a4e21': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR INDICATORS',
    variable_level_2='SATURATION VAPOR PRESSURE',
    variable_level_3='',
    detailed_variable='',
    identifier='1a2332d9-fd69-4002-89a5-203d748a4e21'
  ),
  'df1a03f5-1cb3-4c63-870a-5a09debdf065': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR INDICATORS',
    variable_level_2='STABLE ISOTOPES',
    variable_level_3='',
    detailed_variable='',
    identifier='df1a03f5-1cb3-4c63-870a-5a09debdf065'
  ),
  'c3a4eb4a-4619-43cd-b890-b567d01324ea': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR INDICATORS',
    variable_level_2='TOTAL PRECIPITABLE WATER',
    variable_level_3='',
    detailed_variable='',
    identifier='c3a4eb4a-4619-43cd-b890-b567d01324ea'
  ),
  '433ea253-243d-42e4-bc61-f85eb7a73879': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR INDICATORS',
    variable_level_2='VAPOR PRESSURE',
    variable_level_3='',
    detailed_variable='',
    identifier='433ea253-243d-42e4-bc61-f85eb7a73879'
  ),
  '15029eb0-6342-4066-8ac9-c50f7dbfb392': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR INDICATORS',
    variable_level_2='WATER VAPOR',
    variable_level_3='',
    detailed_variable='',
    identifier='15029eb0-6342-4066-8ac9-c50f7dbfb392'
  ),
  '4f58cf68-0d44-424a-88af-65c3edfd0945': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR INDICES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4f58cf68-0d44-424a-88af-65c3edfd0945'
  ),
  '07826fba-f581-4119-803e-14f3bfc2d14c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR INDICES',
    variable_level_2='HUMIDITY INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='07826fba-f581-4119-803e-14f3bfc2d14c'
  ),
  '425486f4-7b04-4b77-af40-563fe6ed4167': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR INDICES',
    variable_level_2='WATER VAPOR TRANSPORT INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='425486f4-7b04-4b77-af40-563fe6ed4167'
  ),
  '3c4fe00c-6fb1-403e-a053-3a0174a6dfe6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR PROCESSES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3c4fe00c-6fb1-403e-a053-3a0174a6dfe6'
  ),
  'd7fbbafe-fc73-4b63-9837-3d53d2370d9d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR PROCESSES',
    variable_level_2='CONDENSATION',
    variable_level_3='',
    detailed_variable='',
    identifier='d7fbbafe-fc73-4b63-9837-3d53d2370d9d'
  ),
  'b68ab978-6db6-49ee-84e2-5f37b461a998': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR PROCESSES',
    variable_level_2='EVAPORATION',
    variable_level_3='',
    detailed_variable='',
    identifier='b68ab978-6db6-49ee-84e2-5f37b461a998'
  ),
  '26fc4850-7ba9-44d8-a156-5c623e17b72f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR PROCESSES',
    variable_level_2='EVAPOTRANSPIRATION',
    variable_level_3='',
    detailed_variable='',
    identifier='26fc4850-7ba9-44d8-a156-5c623e17b72f'
  ),
  'f28060e0-1c51-41df-8451-6c98b3e77e8a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR PROCESSES',
    variable_level_2='EVAPOTRANSPIRATION',
    variable_level_3='EFFECTIVE EVAPOTRANSPIRATION',
    detailed_variable='',
    identifier='f28060e0-1c51-41df-8451-6c98b3e77e8a'
  ),
  '6045993e-a656-40c1-853c-9db1fbb49171': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR PROCESSES',
    variable_level_2='EVAPOTRANSPIRATION',
    variable_level_3='POTENTIAL EVAPOTRANSPIRATION',
    detailed_variable='',
    identifier='6045993e-a656-40c1-853c-9db1fbb49171'
  ),
  'd438f0a2-5a88-4d56-8bec-7c5e35249544': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR PROCESSES',
    variable_level_2='SUBLIMATION',
    variable_level_3='',
    detailed_variable='',
    identifier='d438f0a2-5a88-4d56-8bec-7c5e35249544'
  ),
  '293cdec2-44b7-488c-ae04-0722f0a9e8b9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR PROCESSES',
    variable_level_2='SUPERSATURATION',
    variable_level_3='',
    detailed_variable='',
    identifier='293cdec2-44b7-488c-ae04-0722f0a9e8b9'
  ),
  '5d8b1280-62a6-48f5-a9f6-ed18023e3481': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR PROCESSES',
    variable_level_2='WATER VAPOR CONVERGENCE',
    variable_level_3='',
    detailed_variable='',
    identifier='5d8b1280-62a6-48f5-a9f6-ed18023e3481'
  ),
  '957240ee-7ad8-4c62-9fd7-364371d247d7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR PROCESSES',
    variable_level_2='WATER VAPOR DIVERGENCE',
    variable_level_3='',
    detailed_variable='',
    identifier='957240ee-7ad8-4c62-9fd7-364371d247d7'
  ),
  '32a88fee-dfa9-4ef8-ab6d-cbc18426da53': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR PROCESSES',
    variable_level_2='WATER VAPOR FLUX',
    variable_level_3='',
    detailed_variable='',
    identifier='32a88fee-dfa9-4ef8-ab6d-cbc18426da53'
  ),
  '5cd8b242-ac18-4d9f-85d5-eb551792d7e9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR PROCESSES',
    variable_level_2='WATER VAPOR TENDENCY',
    variable_level_3='',
    detailed_variable='',
    identifier='5cd8b242-ac18-4d9f-85d5-eb551792d7e9'
  ),
  'acc824e7-8eea-4e7d-aa3d-757cda7e6ec9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR PROFILES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='acc824e7-8eea-4e7d-aa3d-757cda7e6ec9'
  ),
  '4f982a1f-63f0-4160-a19b-76e2c48cea4c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR PROFILES',
    variable_level_2='BOUNDARY LAYER WATER VAPOR PROFILES',
    variable_level_3='',
    detailed_variable='',
    identifier='4f982a1f-63f0-4160-a19b-76e2c48cea4c'
  ),
  'a088868e-8c40-4cea-bcd2-01e777b0563b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR PROFILES',
    variable_level_2='STRATOSPHERIC WATER VAPOR',
    variable_level_3='',
    detailed_variable='',
    identifier='a088868e-8c40-4cea-bcd2-01e777b0563b'
  ),
  '1b9a1873-c02f-4b6c-906e-5da8833354d4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR PROFILES',
    variable_level_2='VERTICALLY RESOLVED BACKSCATTER LIGHT',
    variable_level_3='',
    detailed_variable='',
    identifier='1b9a1873-c02f-4b6c-906e-5da8833354d4'
  ),
  '04c30b59-88ea-4311-8353-8896d4eba83f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR PROFILES',
    variable_level_2='WATER VAPOR CONCENTRATION PROFILES',
    variable_level_3='',
    detailed_variable='',
    identifier='04c30b59-88ea-4311-8353-8896d4eba83f'
  ),
  '9fccc013-4a58-438a-b1e4-cd625aeb8204': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WATER VAPOR',
    variable_level_1='WATER VAPOR PROFILES',
    variable_level_2='WATER VAPOR MIXING RATIO PROFILES',
    variable_level_3='',
    detailed_variable='',
    identifier='9fccc013-4a58-438a-b1e4-cd625aeb8204'
  ),
  'df160e31-ae45-41a4-9093-a80fe5303cea': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='df160e31-ae45-41a4-9093-a80fe5303cea'
  ),
  '1488b98d-6497-48b9-88db-6ee82a2e3ed3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='LOCAL WINDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1488b98d-6497-48b9-88db-6ee82a2e3ed3'
  ),
  '72c180e6-b3f3-4f9a-8d04-23f0b10735af': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='LOCAL WINDS',
    variable_level_2='DUST DEVILS',
    variable_level_3='',
    detailed_variable='',
    identifier='72c180e6-b3f3-4f9a-8d04-23f0b10735af'
  ),
  '31fe9edf-ec85-446f-a476-4bd24ee59ae2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='LOCAL WINDS',
    variable_level_2='LAND BREEZES',
    variable_level_3='',
    detailed_variable='',
    identifier='31fe9edf-ec85-446f-a476-4bd24ee59ae2'
  ),
  'b73a2e6a-7a8b-443e-98f4-5a77f3a9691c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='LOCAL WINDS',
    variable_level_2='MICROBURSTS',
    variable_level_3='',
    detailed_variable='',
    identifier='b73a2e6a-7a8b-443e-98f4-5a77f3a9691c'
  ),
  'a1df1d50-dd2b-4944-bda5-0cf1127e2f49': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='LOCAL WINDS',
    variable_level_2='OROGRAPHIC WINDS',
    variable_level_3='',
    detailed_variable='',
    identifier='a1df1d50-dd2b-4944-bda5-0cf1127e2f49'
  ),
  '5f55961d-45b8-4330-8eee-0b9a9eb4f309': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='LOCAL WINDS',
    variable_level_2='OROGRAPHIC WINDS',
    variable_level_3='ANABATIC WINDS',
    detailed_variable='',
    identifier='5f55961d-45b8-4330-8eee-0b9a9eb4f309'
  ),
  '2cf573dd-0ed7-4455-a233-5987b5a8b52a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='LOCAL WINDS',
    variable_level_2='OROGRAPHIC WINDS',
    variable_level_3='BORA WINDS',
    detailed_variable='',
    identifier='2cf573dd-0ed7-4455-a233-5987b5a8b52a'
  ),
  'c19501d9-bd86-4611-bd30-6a34dc763a35': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='LOCAL WINDS',
    variable_level_2='OROGRAPHIC WINDS',
    variable_level_3='FOEHN WINDS',
    detailed_variable='',
    identifier='c19501d9-bd86-4611-bd30-6a34dc763a35'
  ),
  'd7d48399-62ac-4eca-9c09-14b9094a9444': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='LOCAL WINDS',
    variable_level_2='OROGRAPHIC WINDS',
    variable_level_3='KATABATIC WINDS',
    detailed_variable='',
    identifier='d7d48399-62ac-4eca-9c09-14b9094a9444'
  ),
  '6520897f-c6b6-432e-b7d5-e99b33e6932e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='LOCAL WINDS',
    variable_level_2='OROGRAPHIC WINDS',
    variable_level_3='MOUNTAIN BREEZES',
    detailed_variable='',
    identifier='6520897f-c6b6-432e-b7d5-e99b33e6932e'
  ),
  '4d005bfc-597b-4a99-971f-21d3d44b7b91': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='LOCAL WINDS',
    variable_level_2='OROGRAPHIC WINDS',
    variable_level_3='VALLEY BREEZES',
    detailed_variable='',
    identifier='4d005bfc-597b-4a99-971f-21d3d44b7b91'
  ),
  '9cb8f1a4-5d2b-40d1-a7c3-c608bbe20a0b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='LOCAL WINDS',
    variable_level_2='SEA BREEZES',
    variable_level_3='',
    detailed_variable='',
    identifier='9cb8f1a4-5d2b-40d1-a7c3-c608bbe20a0b'
  ),
  '10685919-bc01-43e7-901a-b62ac44627f3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='SURFACE WINDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='10685919-bc01-43e7-901a-b62ac44627f3'
  ),
  '185b86e2-af35-42b2-b20d-f9ca6fdab493': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='SURFACE WINDS',
    variable_level_2='STORM RELATIVE WINDS',
    variable_level_3='',
    detailed_variable='',
    identifier='185b86e2-af35-42b2-b20d-f9ca6fdab493'
  ),
  '1e9bb112-5dc0-47a5-8c8a-b9cb07ece7c5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='SURFACE WINDS',
    variable_level_2='U/V WIND COMPONENTS',
    variable_level_3='',
    detailed_variable='',
    identifier='1e9bb112-5dc0-47a5-8c8a-b9cb07ece7c5'
  ),
  'e987550e-d443-48eb-93eb-0bc47a62d4b4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='SURFACE WINDS',
    variable_level_2='WIND DIRECTION',
    variable_level_3='',
    detailed_variable='',
    identifier='e987550e-d443-48eb-93eb-0bc47a62d4b4'
  ),
  'c455fcc4-e27d-44bc-96c6-f7a7b31911ff': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='SURFACE WINDS',
    variable_level_2='WIND DIRECTION TENDENCY',
    variable_level_3='',
    detailed_variable='',
    identifier='c455fcc4-e27d-44bc-96c6-f7a7b31911ff'
  ),
  'a92f49f3-e2ee-4ef4-b064-39311ffb95d3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='SURFACE WINDS',
    variable_level_2='WIND SPEED',
    variable_level_3='',
    detailed_variable='',
    identifier='a92f49f3-e2ee-4ef4-b064-39311ffb95d3'
  ),
  '69526601-5607-46e0-954a-251249de80fe': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='SURFACE WINDS',
    variable_level_2='WIND SPEED TENDENCY',
    variable_level_3='',
    detailed_variable='',
    identifier='69526601-5607-46e0-954a-251249de80fe'
  ),
  '592d49c4-e8ae-4ab4-bf24-ae4a896d0637': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='UPPER LEVEL WINDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='592d49c4-e8ae-4ab4-bf24-ae4a896d0637'
  ),
  '8bb1dca3-9793-4120-b0ea-f27a5b81f259': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='UPPER LEVEL WINDS',
    variable_level_2='BOUNDARY LAYER WINDS',
    variable_level_3='',
    detailed_variable='',
    identifier='8bb1dca3-9793-4120-b0ea-f27a5b81f259'
  ),
  '385af5fe-ad73-4e04-9d51-675599fb0576': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='UPPER LEVEL WINDS',
    variable_level_2='FLIGHT LEVEL WINDS',
    variable_level_3='',
    detailed_variable='',
    identifier='385af5fe-ad73-4e04-9d51-675599fb0576'
  ),
  'b30a6184-0d59-41de-92f0-8876582ef045': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='UPPER LEVEL WINDS',
    variable_level_2='STORM RELATIVE WINDS',
    variable_level_3='',
    detailed_variable='',
    identifier='b30a6184-0d59-41de-92f0-8876582ef045'
  ),
  'baa4b68a-96f9-4ab3-9a9f-3df1ee1d8ff0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='UPPER LEVEL WINDS',
    variable_level_2='U/V WIND COMPONENTS',
    variable_level_3='',
    detailed_variable='',
    identifier='baa4b68a-96f9-4ab3-9a9f-3df1ee1d8ff0'
  ),
  '272ffe8a-2949-4b58-bb81-52cb1c879f4a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='UPPER LEVEL WINDS',
    variable_level_2='WIND DIRECTION',
    variable_level_3='',
    detailed_variable='',
    identifier='272ffe8a-2949-4b58-bb81-52cb1c879f4a'
  ),
  '2a43bf40-7f23-4616-be1b-66940b7b7f4f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='UPPER LEVEL WINDS',
    variable_level_2='WIND DIRECTION TENDENCY',
    variable_level_3='',
    detailed_variable='',
    identifier='2a43bf40-7f23-4616-be1b-66940b7b7f4f'
  ),
  '661591b3-6685-4de7-a2a4-9ce8ae505044': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='UPPER LEVEL WINDS',
    variable_level_2='WIND SPEED',
    variable_level_3='',
    detailed_variable='',
    identifier='661591b3-6685-4de7-a2a4-9ce8ae505044'
  ),
  '1fe29b31-b9ff-4a6c-b474-09bd9502b5c5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='UPPER LEVEL WINDS',
    variable_level_2='WIND SPEED TENDENCY',
    variable_level_3='',
    detailed_variable='',
    identifier='1fe29b31-b9ff-4a6c-b474-09bd9502b5c5'
  ),
  '492ffe26-8fbe-4d7d-a537-495fb96bdcce': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND DYNAMICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='492ffe26-8fbe-4d7d-a537-495fb96bdcce'
  ),
  'ce546f0d-d2e1-43ed-b8e0-a9079c690c56': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND DYNAMICS',
    variable_level_2='ADVECTION',
    variable_level_3='',
    detailed_variable='',
    identifier='ce546f0d-d2e1-43ed-b8e0-a9079c690c56'
  ),
  'ebce0874-7635-4094-8ef4-968851873771': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND DYNAMICS',
    variable_level_2='CONVECTION',
    variable_level_3='',
    detailed_variable='',
    identifier='ebce0874-7635-4094-8ef4-968851873771'
  ),
  'a2cc8e02-3207-4c40-af41-9656404bac0a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND DYNAMICS',
    variable_level_2='CONVERGENCE',
    variable_level_3='',
    detailed_variable='',
    identifier='a2cc8e02-3207-4c40-af41-9656404bac0a'
  ),
  '5c58acfc-04ed-4cbf-8674-13c41b3e950d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND DYNAMICS',
    variable_level_2='DIVERGENCE',
    variable_level_3='',
    detailed_variable='',
    identifier='5c58acfc-04ed-4cbf-8674-13c41b3e950d'
  ),
  '8a12ec59-c8c8-4512-b123-16bca93771b0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND DYNAMICS',
    variable_level_2='HORIZONTAL WIND VELOCITY/SPEED',
    variable_level_3='',
    detailed_variable='',
    identifier='8a12ec59-c8c8-4512-b123-16bca93771b0'
  ),
  '84780569-bef5-41fd-901f-828418e390dd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND DYNAMICS',
    variable_level_2='OROGRAPHIC LIFTING',
    variable_level_3='',
    detailed_variable='',
    identifier='84780569-bef5-41fd-901f-828418e390dd'
  ),
  'eaeb5cdd-365f-4368-8e20-6defe111b3b4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND DYNAMICS',
    variable_level_2='STREAMFUNCTIONS',
    variable_level_3='',
    detailed_variable='',
    identifier='eaeb5cdd-365f-4368-8e20-6defe111b3b4'
  ),
  '226d05da-dd0b-4314-919a-0b259ce724b5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND DYNAMICS',
    variable_level_2='TURBULENCE',
    variable_level_3='',
    detailed_variable='',
    identifier='226d05da-dd0b-4314-919a-0b259ce724b5'
  ),
  '841a7ac7-5981-4e93-895f-1b57c3d892a0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND DYNAMICS',
    variable_level_2='VERTICAL WIND VELOCITY/SPEED',
    variable_level_3='',
    detailed_variable='',
    identifier='841a7ac7-5981-4e93-895f-1b57c3d892a0'
  ),
  '858a80ff-5aa4-4590-b2e2-e88a802a6ee4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND DYNAMICS',
    variable_level_2='VORTICITY',
    variable_level_3='',
    detailed_variable='',
    identifier='858a80ff-5aa4-4590-b2e2-e88a802a6ee4'
  ),
  '72edbeca-b608-4f2d-8aba-492c8e6615b8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND DYNAMICS',
    variable_level_2='VORTICITY',
    variable_level_3='POTENTIAL VORTICITY',
    detailed_variable='',
    identifier='72edbeca-b608-4f2d-8aba-492c8e6615b8'
  ),
  '9e2f502b-a2d5-4bc8-8c8f-489aa0c68177': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND DYNAMICS',
    variable_level_2='VORTICITY',
    variable_level_3='VORTICITY ADVECTION',
    detailed_variable='',
    identifier='9e2f502b-a2d5-4bc8-8c8f-489aa0c68177'
  ),
  '05cf5b56-0f86-4819-b713-1272b97b06c5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND DYNAMICS',
    variable_level_2='WIND SHEAR',
    variable_level_3='',
    detailed_variable='',
    identifier='05cf5b56-0f86-4819-b713-1272b97b06c5'
  ),
  'ef91f2b6-27e9-42ab-b8c6-4410aace0141': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND DYNAMICS',
    variable_level_2='WIND SHEAR',
    variable_level_3='HORIZONTAL WIND SHEAR',
    detailed_variable='',
    identifier='ef91f2b6-27e9-42ab-b8c6-4410aace0141'
  ),
  '1b0abf68-b069-4a0b-8081-35a36da9d4a7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND DYNAMICS',
    variable_level_2='WIND SHEAR',
    variable_level_3='VERTICAL WIND SHEAR',
    detailed_variable='',
    identifier='1b0abf68-b069-4a0b-8081-35a36da9d4a7'
  ),
  'ef034881-8bf4-403f-a4ee-c68771769c93': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND DYNAMICS',
    variable_level_2='WIND STRESS',
    variable_level_3='',
    detailed_variable='',
    identifier='ef034881-8bf4-403f-a4ee-c68771769c93'
  ),
  '25775905-dac3-4834-b709-f38a0a03b258': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND INDICES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='25775905-dac3-4834-b709-f38a0a03b258'
  ),
  '8251fedc-3910-4f18-9594-df2fbb9bb1d9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND INDICES',
    variable_level_2='GOES WIND INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='8251fedc-3910-4f18-9594-df2fbb9bb1d9'
  ),
  '17e33fba-625b-40eb-b51d-902a89ca5747': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND INDICES',
    variable_level_2='QUASI-BIENNIAL OSCILLATION (QBO) ZONAL WIND INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='17e33fba-625b-40eb-b51d-902a89ca5747'
  ),
  'dcc6cbbf-23a0-4ae7-bfbd-6207d35c741f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND PROFILES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='dcc6cbbf-23a0-4ae7-bfbd-6207d35c741f'
  ),
  'cd6f51f9-6ab4-4df4-a4d2-347e38fe80b6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND PROFILES',
    variable_level_2='LINE OF SIGHT WINDS',
    variable_level_3='',
    detailed_variable='',
    identifier='cd6f51f9-6ab4-4df4-a4d2-347e38fe80b6'
  ),
  '4478e3ea-ac49-4ea3-bcb8-e6b4e2190266': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND PROFILES',
    variable_level_2='VELOCITY AZIMUTH DISPLAY VERTICAL WIND PROFILES',
    variable_level_3='',
    detailed_variable='',
    identifier='4478e3ea-ac49-4ea3-bcb8-e6b4e2190266'
  ),
  '5be35f50-a1ea-40c5-8e0d-579dad1b9143': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND PROFILES',
    variable_level_2='WIND DIRECTION PROFILES',
    variable_level_3='',
    detailed_variable='',
    identifier='5be35f50-a1ea-40c5-8e0d-579dad1b9143'
  ),
  '1c93710e-cfaa-47c1-ba97-b2deb85620ca': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='ATMOSPHERIC WINDS',
    variable_level_1='WIND PROFILES',
    variable_level_2='WIND VELOCITY/SPEED PROFILES',
    variable_level_3='',
    detailed_variable='',
    identifier='1c93710e-cfaa-47c1-ba97-b2deb85620ca'
  ),
  '162e2243-3266-4999-b352-d8a1a9dc82ac': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='162e2243-3266-4999-b352-d8a1a9dc82ac'
  ),
  'cbb0d517-462a-46fe-a0e6-32555f7e7f23': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD DROPLET DISTRIBUTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='cbb0d517-462a-46fe-a0e6-32555f7e7f23'
  ),
  '62019831-aaba-4d63-a5cd-73138ccfa5d0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD DYNAMICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='62019831-aaba-4d63-a5cd-73138ccfa5d0'
  ),
  '49fd6f11-5682-4d27-8fc6-66bf3faadf39': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD DYNAMICS',
    variable_level_2='HEAT FLUX',
    variable_level_3='',
    detailed_variable='',
    identifier='49fd6f11-5682-4d27-8fc6-66bf3faadf39'
  ),
  'd8205171-15fc-4e2c-af05-2d384cb82e1e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD DYNAMICS',
    variable_level_2='IN-CLOUD VERTICAL VELOCITIES',
    variable_level_3='',
    detailed_variable='',
    identifier='d8205171-15fc-4e2c-af05-2d384cb82e1e'
  ),
  '925f563d-908a-4671-b750-23d0f3e42310': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD DYNAMICS',
    variable_level_2='MOISTURE FLUX',
    variable_level_3='',
    detailed_variable='',
    identifier='925f563d-908a-4671-b750-23d0f3e42310'
  ),
  '1dc6063b-892d-4879-8551-1e346dd3f2e7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD DYNAMICS',
    variable_level_2='MOISTURE FLUX',
    variable_level_3='DOWNWARD MOISTURE FLUX',
    detailed_variable='',
    identifier='1dc6063b-892d-4879-8551-1e346dd3f2e7'
  ),
  '49cad94d-0e93-44cb-a8a2-8e83d603463b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD DYNAMICS',
    variable_level_2='MOISTURE FLUX',
    variable_level_3='UPWARD MOISTURE FLUX',
    detailed_variable='',
    identifier='49cad94d-0e93-44cb-a8a2-8e83d603463b'
  ),
  '5bac3ef6-5e30-4f14-a5dc-8065c7fcba55': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD DYNAMICS',
    variable_level_2='RADIATIONAL COOLING',
    variable_level_3='',
    detailed_variable='',
    identifier='5bac3ef6-5e30-4f14-a5dc-8065c7fcba55'
  ),
  'c7259da4-18dd-4196-91ff-a68087978349': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD DYNAMICS',
    variable_level_2='RADIATIONAL DIVERGENCE',
    variable_level_3='',
    detailed_variable='',
    identifier='c7259da4-18dd-4196-91ff-a68087978349'
  ),
  'cfa49843-2d36-4709-8969-b176432adf78': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD DYNAMICS',
    variable_level_2='THETA-E ENTRAINMENT',
    variable_level_3='',
    detailed_variable='',
    identifier='cfa49843-2d36-4709-8969-b176432adf78'
  ),
  'ba4a9964-8323-45df-a372-b4e2f3eef9e5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD DYNAMICS',
    variable_level_2='VORTEX STREET',
    variable_level_3='',
    detailed_variable='',
    identifier='ba4a9964-8323-45df-a372-b4e2f3eef9e5'
  ),
  '2d00d3c4-2ef3-49f6-9261-6184f6517b4f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD DYNAMICS',
    variable_level_2='VORTEX STREET',
    variable_level_3='KARMAN VORTEX STREET',
    detailed_variable='',
    identifier='2d00d3c4-2ef3-49f6-9261-6184f6517b4f'
  ),
  'a997c21b-ca61-4e78-8828-aa3e144976c3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD DYNAMICS',
    variable_level_2='WATER VAPOR TRANSPORT',
    variable_level_3='',
    detailed_variable='',
    identifier='a997c21b-ca61-4e78-8828-aa3e144976c3'
  ),
  '0cfcbaa7-727b-4199-8cca-93824b427e9b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD MICROPHYSICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0cfcbaa7-727b-4199-8cca-93824b427e9b'
  ),
  'ebbf8642-3da1-4401-a779-3e56550a029d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD MICROPHYSICS',
    variable_level_2='CLOUD CONDENSATION NUCLEI',
    variable_level_3='',
    detailed_variable='',
    identifier='ebbf8642-3da1-4401-a779-3e56550a029d'
  ),
  '47812ef8-b64b-4988-9ae4-31f3581ae9a5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD MICROPHYSICS',
    variable_level_2='CLOUD DROPLET CONCENTRATION/SIZE',
    variable_level_3='',
    detailed_variable='',
    identifier='47812ef8-b64b-4988-9ae4-31f3581ae9a5'
  ),
  '05ac9d3e-bc44-41fa-ace0-c41bf3ebee97': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD MICROPHYSICS',
    variable_level_2='CLOUD LIQUID WATER/ICE',
    variable_level_3='',
    detailed_variable='',
    identifier='05ac9d3e-bc44-41fa-ace0-c41bf3ebee97'
  ),
  '804fb334-1c74-4070-bd5f-848014a6e220': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD MICROPHYSICS',
    variable_level_2='CLOUD MASS FLUX',
    variable_level_3='',
    detailed_variable='',
    identifier='804fb334-1c74-4070-bd5f-848014a6e220'
  ),
  '4bc483b1-dd64-4e97-bfd3-c0e755df6308': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD MICROPHYSICS',
    variable_level_2='CLOUD OPTICAL DEPTH/THICKNESS',
    variable_level_3='',
    detailed_variable='',
    identifier='4bc483b1-dd64-4e97-bfd3-c0e755df6308'
  ),
  'b709d6fc-f0cf-47de-bdbb-1cd875b5f3ab': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD MICROPHYSICS',
    variable_level_2='CLOUD PRECIPITABLE WATER',
    variable_level_3='',
    detailed_variable='',
    identifier='b709d6fc-f0cf-47de-bdbb-1cd875b5f3ab'
  ),
  '76bcb8e0-1c07-4783-9d15-3a22203f7849': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD MICROPHYSICS',
    variable_level_2='COLLISION RATE',
    variable_level_3='',
    detailed_variable='',
    identifier='76bcb8e0-1c07-4783-9d15-3a22203f7849'
  ),
  '63effad4-4323-486d-a81b-e0bf3264e5c9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD MICROPHYSICS',
    variable_level_2='DROPLET GROWTH',
    variable_level_3='',
    detailed_variable='',
    identifier='63effad4-4323-486d-a81b-e0bf3264e5c9'
  ),
  'ab702934-0959-45fc-a523-81e1aa0c09c8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD MICROPHYSICS',
    variable_level_2='DROPLET GROWTH',
    variable_level_3='ACCRETION',
    detailed_variable='',
    identifier='ab702934-0959-45fc-a523-81e1aa0c09c8'
  ),
  '889253e1-e189-4f75-bdc7-7e612b19e3ae': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD MICROPHYSICS',
    variable_level_2='DROPLET GROWTH',
    variable_level_3='ACCRETION',
    detailed_variable='RIMING',
    identifier='889253e1-e189-4f75-bdc7-7e612b19e3ae'
  ),
  '8e484ec4-50fd-4c08-9c96-6ad483e170ad': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD MICROPHYSICS',
    variable_level_2='DROPLET GROWTH',
    variable_level_3='AGGREGATION',
    detailed_variable='',
    identifier='8e484ec4-50fd-4c08-9c96-6ad483e170ad'
  ),
  'd5d64790-db29-451d-b022-a461dac06228': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD MICROPHYSICS',
    variable_level_2='DROPLET GROWTH',
    variable_level_3='COALESCENCE',
    detailed_variable='',
    identifier='d5d64790-db29-451d-b022-a461dac06228'
  ),
  '60bb6ddf-29b1-4dad-9c9f-27f040a43bba': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD MICROPHYSICS',
    variable_level_2='ICE NUCLEI',
    variable_level_3='',
    detailed_variable='',
    identifier='60bb6ddf-29b1-4dad-9c9f-27f040a43bba'
  ),
  '00d6fb2f-16d5-4949-afec-a1adbd600a58': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD MICROPHYSICS',
    variable_level_2='PARTICLE SIZE DISTRIBUTION',
    variable_level_3='',
    detailed_variable='',
    identifier='00d6fb2f-16d5-4949-afec-a1adbd600a58'
  ),
  '8d66dbbe-886e-449d-bfd2-93fc8d357ccd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD MICROPHYSICS',
    variable_level_2='SEDIMENTATION',
    variable_level_3='',
    detailed_variable='',
    identifier='8d66dbbe-886e-449d-bfd2-93fc8d357ccd'
  ),
  'bee9f657-c115-4d73-a10c-7e05e00db574': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD MICROPHYSICS',
    variable_level_2='SEDIMENTATION',
    variable_level_3='SEDIMENTATION RATE',
    detailed_variable='',
    identifier='bee9f657-c115-4d73-a10c-7e05e00db574'
  ),
  'c9e429cb-eff0-4dd3-9eca-527e0081f65c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD PROPERTIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c9e429cb-eff0-4dd3-9eca-527e0081f65c'
  ),
  '70f5b8a3-3f52-467c-957c-7ca98e1c6184': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD PROPERTIES',
    variable_level_2='CLOUD AMOUNT',
    variable_level_3='',
    detailed_variable='',
    identifier='70f5b8a3-3f52-467c-957c-7ca98e1c6184'
  ),
  '4c737490-1486-418f-81f4-c50c47da117d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD PROPERTIES',
    variable_level_2='CLOUD ASYMMETRY',
    variable_level_3='',
    detailed_variable='',
    identifier='4c737490-1486-418f-81f4-c50c47da117d'
  ),
  '1f0765e3-4ea3-42be-8ed5-3e26bdebb219': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD PROPERTIES',
    variable_level_2='CLOUD BASE HEIGHT',
    variable_level_3='',
    detailed_variable='',
    identifier='1f0765e3-4ea3-42be-8ed5-3e26bdebb219'
  ),
  '17f212af-e782-4196-b467-060699ecf4ca': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD PROPERTIES',
    variable_level_2='CLOUD BASE PRESSURE',
    variable_level_3='',
    detailed_variable='',
    identifier='17f212af-e782-4196-b467-060699ecf4ca'
  ),
  '5f5f4f7a-ea5f-40fe-ba73-8d5f7241e5fa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD PROPERTIES',
    variable_level_2='CLOUD BASE TEMPERATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='5f5f4f7a-ea5f-40fe-ba73-8d5f7241e5fa'
  ),
  '88dc0be1-7427-4a82-9fee-3b2bf84d002a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD PROPERTIES',
    variable_level_2='CLOUD CEILING',
    variable_level_3='',
    detailed_variable='',
    identifier='88dc0be1-7427-4a82-9fee-3b2bf84d002a'
  ),
  'b296b688-0ff0-4212-9b30-30e9fe413709': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD PROPERTIES',
    variable_level_2='CLOUD FRACTION',
    variable_level_3='',
    detailed_variable='',
    identifier='b296b688-0ff0-4212-9b30-30e9fe413709'
  ),
  'acb52274-6c0d-4241-a979-3fa3efca6702': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD PROPERTIES',
    variable_level_2='CLOUD FREQUENCY',
    variable_level_3='',
    detailed_variable='',
    identifier='acb52274-6c0d-4241-a979-3fa3efca6702'
  ),
  '57292a97-19be-4fae-b2f7-9fa0a3629b53': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD PROPERTIES',
    variable_level_2='CLOUD HEIGHT',
    variable_level_3='',
    detailed_variable='',
    identifier='57292a97-19be-4fae-b2f7-9fa0a3629b53'
  ),
  '2ca13dfa-c2b3-47de-8175-f0723151ef28': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD PROPERTIES',
    variable_level_2='CLOUD MIDLAYER TEMPERATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='2ca13dfa-c2b3-47de-8175-f0723151ef28'
  ),
  '0893cf38-fe6e-4ebc-95f4-db7d24c874db': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD PROPERTIES',
    variable_level_2='CLOUD TOP HEIGHT',
    variable_level_3='',
    detailed_variable='',
    identifier='0893cf38-fe6e-4ebc-95f4-db7d24c874db'
  ),
  'deddd6c7-b840-484f-aeca-253feefb8d7d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD PROPERTIES',
    variable_level_2='CLOUD TOP PHASE',
    variable_level_3='',
    detailed_variable='',
    identifier='deddd6c7-b840-484f-aeca-253feefb8d7d'
  ),
  '1a217e7e-74fa-438e-b4bd-5ad574d92e9d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD PROPERTIES',
    variable_level_2='CLOUD TOP PRESSURE',
    variable_level_3='',
    detailed_variable='',
    identifier='1a217e7e-74fa-438e-b4bd-5ad574d92e9d'
  ),
  '4dc3fcab-a947-47b9-b9a1-acb2a23ee478': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD PROPERTIES',
    variable_level_2='CLOUD TOP TEMPERATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='4dc3fcab-a947-47b9-b9a1-acb2a23ee478'
  ),
  'f2902c27-0872-4ea4-98b9-706855bcd7a3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD PROPERTIES',
    variable_level_2='CLOUD VERTICAL DISTRIBUTION',
    variable_level_3='',
    detailed_variable='',
    identifier='f2902c27-0872-4ea4-98b9-706855bcd7a3'
  ),
  '3487d350-a5a5-43d9-a60d-c1407dd2f0ce': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD RADIATIVE TRANSFER',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3487d350-a5a5-43d9-a60d-c1407dd2f0ce'
  ),
  'd2e93932-0231-4b23-af2f-217c6315a95e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD RADIATIVE TRANSFER',
    variable_level_2='ABSORPTION',
    variable_level_3='',
    detailed_variable='',
    identifier='d2e93932-0231-4b23-af2f-217c6315a95e'
  ),
  '576b5025-dc0e-4021-b8ff-6a7699a79b0c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD RADIATIVE TRANSFER',
    variable_level_2='CLOUD EMISSIVITY',
    variable_level_3='',
    detailed_variable='',
    identifier='576b5025-dc0e-4021-b8ff-6a7699a79b0c'
  ),
  '345ab082-59ac-4649-9a2a-a3bef0d26a06': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD RADIATIVE TRANSFER',
    variable_level_2='CLOUD RADIATIVE FORCING',
    variable_level_3='',
    detailed_variable='',
    identifier='345ab082-59ac-4649-9a2a-a3bef0d26a06'
  ),
  '8a6572c3-676a-41dd-851f-836ac9f1f1d9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD RADIATIVE TRANSFER',
    variable_level_2='CLOUD REFLECTANCE',
    variable_level_3='',
    detailed_variable='',
    identifier='8a6572c3-676a-41dd-851f-836ac9f1f1d9'
  ),
  '4d5273ad-febb-47f6-bdb7-ededf9f9eb1e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD RADIATIVE TRANSFER',
    variable_level_2='DROPLET GROWTH',
    variable_level_3='',
    detailed_variable='',
    identifier='4d5273ad-febb-47f6-bdb7-ededf9f9eb1e'
  ),
  '4b12439a-45fc-42fa-ae19-535826f6247b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD RADIATIVE TRANSFER',
    variable_level_2='EMISSION',
    variable_level_3='',
    detailed_variable='',
    identifier='4b12439a-45fc-42fa-ae19-535826f6247b'
  ),
  'c830ad5e-ac31-41cf-b8e2-277fe457d76d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD RADIATIVE TRANSFER',
    variable_level_2='SCATTERING',
    variable_level_3='',
    detailed_variable='',
    identifier='c830ad5e-ac31-41cf-b8e2-277fe457d76d'
  ),
  '29b61359-ebec-42c2-be05-2d7be2275954': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CLOUD TYPES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='29b61359-ebec-42c2-be05-2d7be2275954'
  ),
  '9a802ef3-680d-4bc6-a42e-aa84d5eb9908': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CONVECTIVE CLOUDS/SYSTEMS (OBSERVED/ANALYZED)',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9a802ef3-680d-4bc6-a42e-aa84d5eb9908'
  ),
  '4074eb32-a3de-494f-a722-2deeaab76b33': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CONVECTIVE CLOUDS/SYSTEMS (OBSERVED/ANALYZED)',
    variable_level_2='CLOUD CLUSTERS',
    variable_level_3='',
    detailed_variable='',
    identifier='4074eb32-a3de-494f-a722-2deeaab76b33'
  ),
  '7c4d5f8f-4809-4859-b379-3b8c379bc83c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CONVECTIVE CLOUDS/SYSTEMS (OBSERVED/ANALYZED)',
    variable_level_2='CUMULONIMBUS',
    variable_level_3='',
    detailed_variable='',
    identifier='7c4d5f8f-4809-4859-b379-3b8c379bc83c'
  ),
  'e1035388-6993-4143-966b-30ced627c2da': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CONVECTIVE CLOUDS/SYSTEMS (OBSERVED/ANALYZED)',
    variable_level_2='CUMULONIMBUS',
    variable_level_3='CUMULONIMBUS CALVUS',
    detailed_variable='',
    identifier='e1035388-6993-4143-966b-30ced627c2da'
  ),
  'eec3cec2-1649-4507-b91d-3a25ab2200ee': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CONVECTIVE CLOUDS/SYSTEMS (OBSERVED/ANALYZED)',
    variable_level_2='CUMULONIMBUS',
    variable_level_3='CUMULONIMBUS CAPILLATUS',
    detailed_variable='',
    identifier='eec3cec2-1649-4507-b91d-3a25ab2200ee'
  ),
  '52f4dfb0-4583-4d82-8cb7-813ffaadd783': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CONVECTIVE CLOUDS/SYSTEMS (OBSERVED/ANALYZED)',
    variable_level_2='CUMULONIMBUS',
    variable_level_3='CUMULONIMBUS CAPILLATUS',
    detailed_variable='CUMULONIMBUS INCUS',
    identifier='52f4dfb0-4583-4d82-8cb7-813ffaadd783'
  ),
  '772d8044-f11b-4b01-bc72-d7dd45cfe1b3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CONVECTIVE CLOUDS/SYSTEMS (OBSERVED/ANALYZED)',
    variable_level_2='CUMULONIMBUS',
    variable_level_3='PYROCUMULONIMBUS',
    detailed_variable='',
    identifier='772d8044-f11b-4b01-bc72-d7dd45cfe1b3'
  ),
  'e1dff4d5-2e5b-46e7-9804-9de29fdb36d9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CONVECTIVE CLOUDS/SYSTEMS (OBSERVED/ANALYZED)',
    variable_level_2='CUMULUS',
    variable_level_3='',
    detailed_variable='',
    identifier='e1dff4d5-2e5b-46e7-9804-9de29fdb36d9'
  ),
  '29d6cd82-4762-4316-9fdd-29430dae7ad9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CONVECTIVE CLOUDS/SYSTEMS (OBSERVED/ANALYZED)',
    variable_level_2='CUMULUS',
    variable_level_3='CUMULUS CASTELLANUS',
    detailed_variable='',
    identifier='29d6cd82-4762-4316-9fdd-29430dae7ad9'
  ),
  '3bafe2e0-c1a7-4bee-a02e-ac66964b4d7f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CONVECTIVE CLOUDS/SYSTEMS (OBSERVED/ANALYZED)',
    variable_level_2='CUMULUS',
    variable_level_3='CUMULUS CONGESTUS',
    detailed_variable='',
    identifier='3bafe2e0-c1a7-4bee-a02e-ac66964b4d7f'
  ),
  '79668331-c50d-49da-aea6-83c94545f9e3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CONVECTIVE CLOUDS/SYSTEMS (OBSERVED/ANALYZED)',
    variable_level_2='CUMULUS',
    variable_level_3='CUMULUS CONGESTUS',
    detailed_variable='TOWERING CUMULUS',
    identifier='79668331-c50d-49da-aea6-83c94545f9e3'
  ),
  '6aa7422c-ad66-40b6-90cd-750f9158daee': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CONVECTIVE CLOUDS/SYSTEMS (OBSERVED/ANALYZED)',
    variable_level_2='CUMULUS',
    variable_level_3='CUMULUS HUMILIS',
    detailed_variable='',
    identifier='6aa7422c-ad66-40b6-90cd-750f9158daee'
  ),
  'ef4de9ce-01ee-4bf0-8814-abefd1bad4b9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CONVECTIVE CLOUDS/SYSTEMS (OBSERVED/ANALYZED)',
    variable_level_2='CUMULUS',
    variable_level_3='CUMULUS HUMILIS',
    detailed_variable='FAIR WEATHER CUMULUS',
    identifier='ef4de9ce-01ee-4bf0-8814-abefd1bad4b9'
  ),
  'ba9a8dac-abb7-4580-938d-762b53bab71b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CONVECTIVE CLOUDS/SYSTEMS (OBSERVED/ANALYZED)',
    variable_level_2='CUMULUS',
    variable_level_3='CUMULUS MEDIOCRIS',
    detailed_variable='',
    identifier='ba9a8dac-abb7-4580-938d-762b53bab71b'
  ),
  '801846d5-622b-4937-83cf-9d387be73ac4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CONVECTIVE CLOUDS/SYSTEMS (OBSERVED/ANALYZED)',
    variable_level_2='CUMULUS',
    variable_level_3='PYROCUMULUS',
    detailed_variable='',
    identifier='801846d5-622b-4937-83cf-9d387be73ac4'
  ),
  '879cccd4-d375-40f6-8bee-6f58efd2dd61': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CONVECTIVE CLOUDS/SYSTEMS (OBSERVED/ANALYZED)',
    variable_level_2='DEEP CONVECTIVE CLOUD SYSTEMS',
    variable_level_3='',
    detailed_variable='',
    identifier='879cccd4-d375-40f6-8bee-6f58efd2dd61'
  ),
  'd13661da-d022-439a-bb27-dc2273f9dc88': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CONVECTIVE CLOUDS/SYSTEMS (OBSERVED/ANALYZED)',
    variable_level_2='MESOSCALE CONVECTIVE COMPLEX',
    variable_level_3='',
    detailed_variable='',
    identifier='d13661da-d022-439a-bb27-dc2273f9dc88'
  ),
  'c3ee0a52-266b-45e4-adad-d0675699676b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CONVECTIVE CLOUDS/SYSTEMS (OBSERVED/ANALYZED)',
    variable_level_2='PERCENT CONVECTIVE CLOUDS',
    variable_level_3='',
    detailed_variable='',
    identifier='c3ee0a52-266b-45e4-adad-d0675699676b'
  ),
  'ca7f5dbd-199e-4cc8-bc7b-550753ecbc93': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CONVECTIVE CLOUDS/SYSTEMS (OBSERVED/ANALYZED)',
    variable_level_2='PRECIPITATING CONVECTIVE CLOUD SYSTEMS',
    variable_level_3='',
    detailed_variable='',
    identifier='ca7f5dbd-199e-4cc8-bc7b-550753ecbc93'
  ),
  'c6024258-d344-4cd2-932b-31e5c81a9c4b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CONVECTIVE CLOUDS/SYSTEMS (OBSERVED/ANALYZED)',
    variable_level_2='SQUALL LINE',
    variable_level_3='',
    detailed_variable='',
    identifier='c6024258-d344-4cd2-932b-31e5c81a9c4b'
  ),
  'fe2e0b6f-3d7d-489a-b093-86ed0d233385': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='CONVECTIVE CLOUDS/SYSTEMS (OBSERVED/ANALYZED)',
    variable_level_2='TROPICAL OCEANIC CLOUD SYSTEMS',
    variable_level_3='',
    detailed_variable='',
    identifier='fe2e0b6f-3d7d-489a-b093-86ed0d233385'
  ),
  '04bc6942-12e0-413f-94d2-1ba7f5edf595': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='MESOSPHERIC CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='04bc6942-12e0-413f-94d2-1ba7f5edf595'
  ),
  '939c0a66-0340-425b-999a-44a09046ec93': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='MESOSPHERIC CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='NOCTILUCENT CLOUDS',
    variable_level_3='',
    detailed_variable='',
    identifier='939c0a66-0340-425b-999a-44a09046ec93'
  ),
  '0a7f50ce-4968-46c8-86a6-23ea13c1830c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='MESOSPHERIC CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='POLAR MESOSPHERIC CLOUDS',
    variable_level_3='',
    detailed_variable='',
    identifier='0a7f50ce-4968-46c8-86a6-23ea13c1830c'
  ),
  'd6ab88c0-5a97-4f5e-8e4c-1c6fc6ed368f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='STRATOSPHERIC CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d6ab88c0-5a97-4f5e-8e4c-1c6fc6ed368f'
  ),
  '9d3d400c-ded2-4b3c-8d0c-5a76e25be033': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='STRATOSPHERIC CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='POLAR STRATOSPHERIC CLOUDS/NACREOUS',
    variable_level_3='',
    detailed_variable='',
    identifier='9d3d400c-ded2-4b3c-8d0c-5a76e25be033'
  ),
  '705cd3a0-ea07-40c8-bfa1-9c26f22d13ba': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/HIGH-LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='705cd3a0-ea07-40c8-bfa1-9c26f22d13ba'
  ),
  'e59c154f-cdc9-4400-a0d2-af60df9e1b56': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/HIGH-LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='CIRROCUMULUS',
    variable_level_3='',
    detailed_variable='',
    identifier='e59c154f-cdc9-4400-a0d2-af60df9e1b56'
  ),
  'bf271f69-3294-44d6-bfa8-a8f54468ca30': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/HIGH-LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='CIRROSTRATUS',
    variable_level_3='',
    detailed_variable='',
    identifier='bf271f69-3294-44d6-bfa8-a8f54468ca30'
  ),
  '8ce319a5-9b49-49e3-8981-3ce512c7efb0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/HIGH-LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='CIRRUS/SYSTEMS',
    variable_level_3='',
    detailed_variable='',
    identifier='8ce319a5-9b49-49e3-8981-3ce512c7efb0'
  ),
  'e4f5faaa-36d9-4529-b667-7d4e39d3c67b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/HIGH-LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='CIRRUS/SYSTEMS',
    variable_level_3='CIRRUS CLOUD SYSTEMS',
    detailed_variable='',
    identifier='e4f5faaa-36d9-4529-b667-7d4e39d3c67b'
  ),
  'd6ba91a1-a5f4-47e3-9485-89348235acb9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/HIGH-LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='CIRRUS/SYSTEMS',
    variable_level_3='CIRRUS KELVIN-HELMHOLTZ COLOMBIAH',
    detailed_variable='',
    identifier='d6ba91a1-a5f4-47e3-9485-89348235acb9'
  ),
  'cf75769c-2430-4280-b9c2-ba384849a548': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/HIGH-LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='CONTRAILS',
    variable_level_3='',
    detailed_variable='',
    identifier='cf75769c-2430-4280-b9c2-ba384849a548'
  ),
  '31c8b1d1-1e46-4c40-a23f-0db327121eb7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/HIGH-LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='PILEUS',
    variable_level_3='',
    detailed_variable='',
    identifier='31c8b1d1-1e46-4c40-a23f-0db327121eb7'
  ),
  '20365b0a-f8df-437a-8b31-25557f7b4d82': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/LOW LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='20365b0a-f8df-437a-8b31-25557f7b4d82'
  ),
  '94668478-3b79-4819-847e-b154bf241aa3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/LOW LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='FOG',
    variable_level_3='',
    detailed_variable='',
    identifier='94668478-3b79-4819-847e-b154bf241aa3'
  ),
  '30c9e32c-7dfe-430f-bd06-4cfa844076e2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/LOW LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='FOG',
    variable_level_3='ADVECTION FOG',
    detailed_variable='',
    identifier='30c9e32c-7dfe-430f-bd06-4cfa844076e2'
  ),
  '09a1b23e-bd8b-4bb9-966b-2388328973d4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/LOW LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='FOG',
    variable_level_3='FRONTAL FOG',
    detailed_variable='',
    identifier='09a1b23e-bd8b-4bb9-966b-2388328973d4'
  ),
  'cd4f6e31-14b5-468a-a15c-5ac0ce97bf35': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/LOW LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='FOG',
    variable_level_3='ICE FOG',
    detailed_variable='',
    identifier='cd4f6e31-14b5-468a-a15c-5ac0ce97bf35'
  ),
  '99a0a2d2-5d77-4cf2-8fc0-90d12840b12d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/LOW LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='FOG',
    variable_level_3='RADIATION FOG',
    detailed_variable='',
    identifier='99a0a2d2-5d77-4cf2-8fc0-90d12840b12d'
  ),
  '50604404-fef4-4e17-a6a6-a88c0bd88c4f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/LOW LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='FOG',
    variable_level_3='STEAM FOG',
    detailed_variable='',
    identifier='50604404-fef4-4e17-a6a6-a88c0bd88c4f'
  ),
  '3eefb892-0453-48b6-b619-b8fa3e7bbfc8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/LOW LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='FOG',
    variable_level_3='UPSLOPE FOG',
    detailed_variable='',
    identifier='3eefb892-0453-48b6-b619-b8fa3e7bbfc8'
  ),
  'a3d37438-644d-448e-95ea-991d79b3a0f3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/LOW LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='NIMBOSTRATUS',
    variable_level_3='',
    detailed_variable='',
    identifier='a3d37438-644d-448e-95ea-991d79b3a0f3'
  ),
  '3375096a-7782-42e8-97d2-0febf63893e0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/LOW LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='STRATOCUMULUS',
    variable_level_3='',
    detailed_variable='',
    identifier='3375096a-7782-42e8-97d2-0febf63893e0'
  ),
  'b1d51b72-97d0-484c-b251-220f219965c2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/LOW LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='STRATOCUMULUS',
    variable_level_3='MARINE STRATOCUMULUS',
    detailed_variable='',
    identifier='b1d51b72-97d0-484c-b251-220f219965c2'
  ),
  '5857260b-1de6-47c2-8f66-5c0dbac42e32': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/LOW LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='STRATOCUMULUS',
    variable_level_3='STRATOCUMULUS CUMILIFORMIS',
    detailed_variable='',
    identifier='5857260b-1de6-47c2-8f66-5c0dbac42e32'
  ),
  '53378c8d-0324-4473-8bbd-231aed830d26': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/LOW LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='STRATOCUMULUS',
    variable_level_3='STRATOCUMULUS CUMILIFORMIS',
    detailed_variable='STRATOCUMULUS CASTELLANUS',
    identifier='53378c8d-0324-4473-8bbd-231aed830d26'
  ),
  '16e33a06-ae4b-48cd-be7f-ad44f3dfd23b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/LOW LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='STRATOCUMULUS',
    variable_level_3='STRATOCUMULUS CUMILIFORMIS',
    detailed_variable='STRATOCUMULUS DIURNALIS',
    identifier='16e33a06-ae4b-48cd-be7f-ad44f3dfd23b'
  ),
  '3adff54b-cfc7-4ba0-ba06-7a06b8c4876a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/LOW LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='STRATOCUMULUS',
    variable_level_3='STRATOCUMULUS CUMILIFORMIS',
    detailed_variable='STRATOCUMULUS MAMMATUS',
    identifier='3adff54b-cfc7-4ba0-ba06-7a06b8c4876a'
  ),
  '8fc75200-666d-4b59-a493-99e08e55e57d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/LOW LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='STRATOCUMULUS',
    variable_level_3='STRATOCUMULUS CUMILIFORMIS',
    detailed_variable='STRATOCUMULUS VESPERALIS',
    identifier='8fc75200-666d-4b59-a493-99e08e55e57d'
  ),
  'bec5166e-1822-40f7-8f07-4d5167d8a565': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/LOW LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='STRATOCUMULUS',
    variable_level_3='STRATOCUMULUS UNDULATAS',
    detailed_variable='',
    identifier='bec5166e-1822-40f7-8f07-4d5167d8a565'
  ),
  '12adeea5-f3e7-4f72-a029-f1e52411de18': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/LOW LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='STRATOCUMULUS',
    variable_level_3='STRATOCUMULUS UNDULATAS',
    detailed_variable='STRATOCUMULUS LENTICULARIS',
    identifier='12adeea5-f3e7-4f72-a029-f1e52411de18'
  ),
  '53ade6b0-1b8c-4766-b9c1-4d40d1f69482': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/LOW LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='STRATOCUMULUS',
    variable_level_3='STRATOCUMULUS UNDULATAS',
    detailed_variable='STRATOCUMULUS OPACUS',
    identifier='53ade6b0-1b8c-4766-b9c1-4d40d1f69482'
  ),
  '94fa2efc-0e7f-4dce-9f2a-0d6f34edcb92': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/LOW LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='STRATOCUMULUS',
    variable_level_3='STRATOCUMULUS UNDULATAS',
    detailed_variable='STRATOCUMULUS PERLUCIDUS',
    identifier='94fa2efc-0e7f-4dce-9f2a-0d6f34edcb92'
  ),
  'b962de0e-f115-4452-95be-7a4af6687bc3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/LOW LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='STRATOCUMULUS',
    variable_level_3='STRATOCUMULUS UNDULATAS',
    detailed_variable='STRATOCUMULUS TRANSLUCIDUS',
    identifier='b962de0e-f115-4452-95be-7a4af6687bc3'
  ),
  '8945d3c7-1c39-4a8b-b954-2a84da8ecc88': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/LOW LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='STRATUS',
    variable_level_3='',
    detailed_variable='',
    identifier='8945d3c7-1c39-4a8b-b954-2a84da8ecc88'
  ),
  'a413f88b-859c-4035-a45b-2faa9934156b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/MID-LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a413f88b-859c-4035-a45b-2faa9934156b'
  ),
  '01021105-60ed-479a-a35b-faa73e286264': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/MID-LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='ALTOCUMULUS',
    variable_level_3='',
    detailed_variable='',
    identifier='01021105-60ed-479a-a35b-faa73e286264'
  ),
  '44415a90-bfe0-447a-93c9-6e4badc6871c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/MID-LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='ALTOCUMULUS',
    variable_level_3='ALTOCUMULUS CASTELLANUS',
    detailed_variable='',
    identifier='44415a90-bfe0-447a-93c9-6e4badc6871c'
  ),
  'dbc7fed3-ad30-4e57-868e-bae478713a71': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/MID-LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='ALTOCUMULUS',
    variable_level_3='ALTOCUMULUS LENTICULARIS',
    detailed_variable='',
    identifier='dbc7fed3-ad30-4e57-868e-bae478713a71'
  ),
  'd5e72d73-22f6-4f4c-b937-b71d84960a1e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/MID-LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='ALTOCUMULUS',
    variable_level_3='ALTOCUMULUS LENTICULARIS',
    detailed_variable='LENTICULAR CLOUDS',
    identifier='d5e72d73-22f6-4f4c-b937-b71d84960a1e'
  ),
  '73e102fa-3089-42c3-bd0f-4682f73fff0f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/MID-LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='ALTOCUMULUS',
    variable_level_3='ALTOCUMULUS UNDULATUS',
    detailed_variable='',
    identifier='73e102fa-3089-42c3-bd0f-4682f73fff0f'
  ),
  'f58d0203-0070-422c-ab52-6ca8ffbb6362': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/MID-LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='ALTOSTRATUS',
    variable_level_3='',
    detailed_variable='',
    identifier='f58d0203-0070-422c-ab52-6ca8ffbb6362'
  ),
  '4da38a31-aac6-4080-96b0-c8ee2cb33158': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='CLOUDS',
    variable_level_1='TROPOSPHERIC/MID-LEVEL CLOUDS (OBSERVED/ANALYZED)',
    variable_level_2='ALTOSTRATUS',
    variable_level_3='ALTOSTRATUS UNDULATUS',
    detailed_variable='',
    identifier='4da38a31-aac6-4080-96b0-c8ee2cb33158'
  ),
  '16db3055-847d-4f06-a3aa-9e27e02170a5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PLANETARY BOUNDARY LAYER',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='16db3055-847d-4f06-a3aa-9e27e02170a5'
  ),
  '1532e590-a62d-46e3-8d03-2351bc48166a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1532e590-a62d-46e3-8d03-2351bc48166a'
  ),
  '2b3dc817-9238-482a-8c10-d34375f3d27d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='ACCUMULATIVE CONVECTIVE PRECIPITATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2b3dc817-9238-482a-8c10-d34375f3d27d'
  ),
  'c7477201-761f-4cd1-b986-3e99a0be866b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='ATMOSPHERIC PRECIPITATION INDICES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c7477201-761f-4cd1-b986-3e99a0be866b'
  ),
  'c6e7ddb6-1f7c-4364-8fb4-aabd1f4dcab4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='ATMOSPHERIC PRECIPITATION INDICES',
    variable_level_2='CENTRAL INDIAN PRECIPITATION INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='c6e7ddb6-1f7c-4364-8fb4-aabd1f4dcab4'
  ),
  '284738a2-4fcb-4eee-9ee7-5eac2378f46d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='ATMOSPHERIC PRECIPITATION INDICES',
    variable_level_2='ENSO PRECIPITATION INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='284738a2-4fcb-4eee-9ee7-5eac2378f46d'
  ),
  '3b024dec-76c2-4995-a9ad-7e2bf4feda72': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='ATMOSPHERIC PRECIPITATION INDICES',
    variable_level_2='STANDARDIZED PRECIPITATION INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='3b024dec-76c2-4995-a9ad-7e2bf4feda72'
  ),
  'dc9c73a3-689c-44b5-b8fe-a5229168193e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='ATMOSPHERIC PRECIPITATION INDICES',
    variable_level_2='WEIGHTED ANOMALY STANDARDIZED PRECIPITATION INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='dc9c73a3-689c-44b5-b8fe-a5229168193e'
  ),
  '6eaed241-db16-4a1a-a06c-893da5d98b45': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='DROPLET SIZE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6eaed241-db16-4a1a-a06c-893da5d98b45'
  ),
  '56f2cdbd-2a91-4267-97eb-1680e8582322': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='HYDROMETEORS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='56f2cdbd-2a91-4267-97eb-1680e8582322'
  ),
  '7d45f108-dda2-4341-b853-ee3a490aad59': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='LIQUID PRECIPITATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7d45f108-dda2-4341-b853-ee3a490aad59'
  ),
  '0ffab597-284f-4d1a-b026-a78a6604cec5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='LIQUID PRECIPITATION',
    variable_level_2='DRIZZLE',
    variable_level_3='',
    detailed_variable='',
    identifier='0ffab597-284f-4d1a-b026-a78a6604cec5'
  ),
  '88e39edc-bf9b-4c02-8a9d-83f9b6c01891': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='LIQUID PRECIPITATION',
    variable_level_2='DRIZZLE',
    variable_level_3='FREEZING DRIZZLE',
    detailed_variable='',
    identifier='88e39edc-bf9b-4c02-8a9d-83f9b6c01891'
  ),
  '09d991ca-020a-4d20-910a-747ea683e1f8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='LIQUID PRECIPITATION',
    variable_level_2='LIQUID SURFACE PRECIPITATION RATE',
    variable_level_3='',
    detailed_variable='',
    identifier='09d991ca-020a-4d20-910a-747ea683e1f8'
  ),
  '09a57dc7-3911-4a65-9f12-b819652b8671': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='LIQUID PRECIPITATION',
    variable_level_2='RAIN',
    variable_level_3='',
    detailed_variable='',
    identifier='09a57dc7-3911-4a65-9f12-b819652b8671'
  ),
  'f9405e92-0c1c-4443-9cc4-45d662d8b5f2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='LIQUID PRECIPITATION',
    variable_level_2='RAIN',
    variable_level_3='ACID RAIN',
    detailed_variable='',
    identifier='f9405e92-0c1c-4443-9cc4-45d662d8b5f2'
  ),
  'a90306f0-353c-4083-941a-0973a6fd6584': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='LIQUID PRECIPITATION',
    variable_level_2='RAIN',
    variable_level_3='FREEZING RAIN',
    detailed_variable='',
    identifier='a90306f0-353c-4083-941a-0973a6fd6584'
  ),
  'eca0080c-b001-4b6a-b978-f76415e28421': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='LIQUID WATER EQUIVALENT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='eca0080c-b001-4b6a-b978-f76415e28421'
  ),
  'cad5c02a-e771-434e-bef6-8dced38a68e8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='PRECIPITATION AMOUNT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='cad5c02a-e771-434e-bef6-8dced38a68e8'
  ),
  'feef8827-92a6-4d1d-b6a5-ecda38a32656': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='PRECIPITATION AMOUNT',
    variable_level_2='12 HOUR PRECIPITATION AMOUNT',
    variable_level_3='',
    detailed_variable='',
    identifier='feef8827-92a6-4d1d-b6a5-ecda38a32656'
  ),
  '12250935-8f40-4279-aada-2f22cbef1459': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='PRECIPITATION AMOUNT',
    variable_level_2='24 HOUR PRECIPITATION AMOUNT',
    variable_level_3='',
    detailed_variable='',
    identifier='12250935-8f40-4279-aada-2f22cbef1459'
  ),
  '039bbfd2-7653-4ba8-9003-b46d367c6038': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='PRECIPITATION AMOUNT',
    variable_level_2='3 AND 6 HOUR PRECIPITATION AMOUNT',
    variable_level_3='',
    detailed_variable='',
    identifier='039bbfd2-7653-4ba8-9003-b46d367c6038'
  ),
  '2f0f103a-4fe9-429f-a783-ba1d6e6a446a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='PRECIPITATION AMOUNT',
    variable_level_2='HOURLY PRECIPITATION AMOUNT',
    variable_level_3='',
    detailed_variable='',
    identifier='2f0f103a-4fe9-429f-a783-ba1d6e6a446a'
  ),
  '22a4ddef-90f0-4935-a13d-26b14723a956': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='PRECIPITATION ANOMALIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='22a4ddef-90f0-4935-a13d-26b14723a956'
  ),
  'd4449cf4-8d4e-4282-b84d-5098715389dd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='PRECIPITATION PROFILES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d4449cf4-8d4e-4282-b84d-5098715389dd'
  ),
  'db0ff132-48ed-429b-b7e7-6a173b380421': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='PRECIPITATION PROFILES',
    variable_level_2='CLOUD WATER PATH',
    variable_level_3='',
    detailed_variable='',
    identifier='db0ff132-48ed-429b-b7e7-6a173b380421'
  ),
  '7ec9473a-0d08-4fb1-b2e8-b83d590d710c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='PRECIPITATION PROFILES',
    variable_level_2='ICE WATER PATH',
    variable_level_3='',
    detailed_variable='',
    identifier='7ec9473a-0d08-4fb1-b2e8-b83d590d710c'
  ),
  '9985d211-1056-4a7a-a1c8-550923ea5a81': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='PRECIPITATION PROFILES',
    variable_level_2='LATENT HEAT FLUX',
    variable_level_3='',
    detailed_variable='',
    identifier='9985d211-1056-4a7a-a1c8-550923ea5a81'
  ),
  'ce105b93-42b1-4692-a8ef-dc10792f26bf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='PRECIPITATION PROFILES',
    variable_level_2='MELTING LAYER HEIGHT',
    variable_level_3='',
    detailed_variable='',
    identifier='ce105b93-42b1-4692-a8ef-dc10792f26bf'
  ),
  'e3973025-f274-44f1-9ff5-0d2fd7e006c2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='PRECIPITATION PROFILES',
    variable_level_2='RAIN TYPE',
    variable_level_3='',
    detailed_variable='',
    identifier='e3973025-f274-44f1-9ff5-0d2fd7e006c2'
  ),
  '4059c189-1ecb-4e0d-98d6-d67c0f76f275': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='PRECIPITATION PROFILES',
    variable_level_2='RAIN WATER PATH',
    variable_level_3='',
    detailed_variable='',
    identifier='4059c189-1ecb-4e0d-98d6-d67c0f76f275'
  ),
  'ac50c468-df2f-429c-8394-9d63efcc6f9d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='PRECIPITATION RATE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ac50c468-df2f-429c-8394-9d63efcc6f9d'
  ),
  '001002d4-28ec-4ee2-9ff6-99d83be2d705': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='PRECIPITATION RATE',
    variable_level_2='CONVECTIVE PRECIPITATION',
    variable_level_3='',
    detailed_variable='',
    identifier='001002d4-28ec-4ee2-9ff6-99d83be2d705'
  ),
  'd57a7ba2-7d23-472f-8d6c-674dec4e8fa0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='PRECIPITATION RATE',
    variable_level_2='FROZEN PRECIPITATION',
    variable_level_3='',
    detailed_variable='',
    identifier='d57a7ba2-7d23-472f-8d6c-674dec4e8fa0'
  ),
  'f29968d9-b911-45b5-b5b5-0a759a345ce9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='PRECIPITATION RATE',
    variable_level_2='SURFACE PRECIPITATION',
    variable_level_3='',
    detailed_variable='',
    identifier='f29968d9-b911-45b5-b5b5-0a759a345ce9'
  ),
  '30bd3a01-8cb0-4045-a998-582adbf97df9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='SNOW WATER EQUIVALENT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='30bd3a01-8cb0-4045-a998-582adbf97df9'
  ),
  '1906bb87-db16-46db-b814-e0b322356125': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='SOLID PRECIPITATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1906bb87-db16-46db-b814-e0b322356125'
  ),
  '6c8581e8-d49c-423e-9b38-3be406b64efa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='SOLID PRECIPITATION',
    variable_level_2='CONVECTIVE SURFACE PRECIPITATION RATE',
    variable_level_3='',
    detailed_variable='',
    identifier='6c8581e8-d49c-423e-9b38-3be406b64efa'
  ),
  '7118d286-6629-48e5-931f-052cd347395e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='SOLID PRECIPITATION',
    variable_level_2='HAIL',
    variable_level_3='',
    detailed_variable='',
    identifier='7118d286-6629-48e5-931f-052cd347395e'
  ),
  'cac27b59-7810-4132-87b4-53108663584e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='SOLID PRECIPITATION',
    variable_level_2='ICE PELLETS',
    variable_level_3='',
    detailed_variable='',
    identifier='cac27b59-7810-4132-87b4-53108663584e'
  ),
  '5beaf99c-0675-4af3-9236-f55d8d206d85': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='SOLID PRECIPITATION',
    variable_level_2='ICE PELLETS',
    variable_level_3='SLEET',
    detailed_variable='',
    identifier='5beaf99c-0675-4af3-9236-f55d8d206d85'
  ),
  '26087764-bd76-4a70-8dba-3c0cbadad6a7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='SOLID PRECIPITATION',
    variable_level_2='ICE PELLETS',
    variable_level_3='SMALL HAIL',
    detailed_variable='',
    identifier='26087764-bd76-4a70-8dba-3c0cbadad6a7'
  ),
  'b51b3708-a662-4cf1-bf13-e67f36b001c4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='SOLID PRECIPITATION',
    variable_level_2='SNOW',
    variable_level_3='',
    detailed_variable='',
    identifier='b51b3708-a662-4cf1-bf13-e67f36b001c4'
  ),
  '6a16461a-49b9-4887-802f-2320c6dc4dd2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='SOLID PRECIPITATION',
    variable_level_2='SNOW',
    variable_level_3='SNOW GRAINS',
    detailed_variable='',
    identifier='6a16461a-49b9-4887-802f-2320c6dc4dd2'
  ),
  'c2815464-48b7-4dc1-90d6-0ab5a8b7c82b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='SOLID PRECIPITATION',
    variable_level_2='SNOW',
    variable_level_3='SNOW PELLETS',
    detailed_variable='',
    identifier='c2815464-48b7-4dc1-90d6-0ab5a8b7c82b'
  ),
  '9466020a-db25-40ba-a76f-4720800efc92': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='TOTAL SURFACE PRECIPITATION RATE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9466020a-db25-40ba-a76f-4720800efc92'
  ),
  'e96f2d1a-432e-44e4-bc88-6f8f35ae88fb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='PRECIPITATION',
    variable_level_1='VIRGA',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e96f2d1a-432e-44e4-bc88-6f8f35ae88fb'
  ),
  '32d72476-ba7b-43fa-b054-84804278aa84': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='RADIO OCCULTATION PROFILES',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='32d72476-ba7b-43fa-b054-84804278aa84'
  ),
  '8fd7f9f6-9b95-4ba4-9df1-7b876b779a86': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='RADIO OCCULTATION PROFILES',
    variable_level_1='BENDING ANGLE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8fd7f9f6-9b95-4ba4-9df1-7b876b779a86'
  ),
  '547dd45b-91a6-4511-bdef-157be65d9b6a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='RADIO OCCULTATION PROFILES',
    variable_level_1='EXCESS PHASE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='547dd45b-91a6-4511-bdef-157be65d9b6a'
  ),
  'f4ec49af-b1c5-431c-968c-56890da96020': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='RADIO OCCULTATION PROFILES',
    variable_level_1='REFRACTIVITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f4ec49af-b1c5-431c-968c-56890da96020'
  ),
  'b7d562cf-9b9b-4461-900b-50423a8c4d29': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b7d562cf-9b9b-4461-900b-50423a8c4d29'
  ),
  '03bc515c-af45-4a15-b2a2-65270f0e72bd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='COLD WAVE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='03bc515c-af45-4a15-b2a2-65270f0e72bd'
  ),
  '12a896f3-993d-49f6-aafc-17378ffa3998': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='DROUGHTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='12a896f3-993d-49f6-aafc-17378ffa3998'
  ),
  'ebd54a3b-8b8b-40e8-94fd-d0b4352e0745': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='DROUGHTS',
    variable_level_2='DROUGHT DURATION',
    variable_level_3='',
    detailed_variable='',
    identifier='ebd54a3b-8b8b-40e8-94fd-d0b4352e0745'
  ),
  '0f7f8887-844d-45d8-9d45-7e48e15c50fd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='DROUGHTS',
    variable_level_2='DROUGHT FREQUENCY',
    variable_level_3='',
    detailed_variable='',
    identifier='0f7f8887-844d-45d8-9d45-7e48e15c50fd'
  ),
  '1a6b76b0-6c03-4021-92fb-66552edcf845': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='DROUGHTS',
    variable_level_2='DROUGHT SEVERITY',
    variable_level_3='',
    detailed_variable='',
    identifier='1a6b76b0-6c03-4021-92fb-66552edcf845'
  ),
  'da436e9b-60e5-4a5f-a50a-08794d62bca8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='EXTRATROPICAL CYCLONES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='da436e9b-60e5-4a5f-a50a-08794d62bca8'
  ),
  '10277cb5-5a11-47a2-8578-3ac1c7152cd2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='EXTRATROPICAL CYCLONES',
    variable_level_2='EXTRATROPICAL CYCLONE FREQUENCY',
    variable_level_3='',
    detailed_variable='',
    identifier='10277cb5-5a11-47a2-8578-3ac1c7152cd2'
  ),
  '2357d9ae-3376-4c4e-8533-6193bf177345': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='EXTRATROPICAL CYCLONES',
    variable_level_2='EXTRATROPICAL CYCLONE MOTION',
    variable_level_3='',
    detailed_variable='',
    identifier='2357d9ae-3376-4c4e-8533-6193bf177345'
  ),
  '7de1c2c0-89c2-4841-b0b8-158224c8ad22': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='EXTRATROPICAL CYCLONES',
    variable_level_2='EXTRATROPICAL CYCLONE TRACK',
    variable_level_3='',
    detailed_variable='',
    identifier='7de1c2c0-89c2-4841-b0b8-158224c8ad22'
  ),
  'a5ad4f63-7483-4f07-86c7-57037e5faf6c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='FOG',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a5ad4f63-7483-4f07-86c7-57037e5faf6c'
  ),
  '4539272a-f041-4fc6-883d-4c4c5bef1683': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='FREEZE/FROST',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4539272a-f041-4fc6-883d-4c4c5bef1683'
  ),
  '2cc64007-a443-45d8-bf9d-c9fae69f4554': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='FREEZE/FROST',
    variable_level_2='FIRST FREEZE/FROST DATE',
    variable_level_3='',
    detailed_variable='',
    identifier='2cc64007-a443-45d8-bf9d-c9fae69f4554'
  ),
  '53b7e7d6-2aeb-4636-bae1-c7cd92d3d541': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='FREEZE/FROST',
    variable_level_2='FIRST FREEZE/FROST PROBABILITY',
    variable_level_3='',
    detailed_variable='',
    identifier='53b7e7d6-2aeb-4636-bae1-c7cd92d3d541'
  ),
  '581d6ad6-2132-45cf-b6be-72341024587b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='FREEZE/FROST',
    variable_level_2='FIRST MODERATE FREEZE/FROST DATE',
    variable_level_3='',
    detailed_variable='',
    identifier='581d6ad6-2132-45cf-b6be-72341024587b'
  ),
  '5a0347ba-2684-4c4a-adc0-ddb63cbbde6b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='FREEZE/FROST',
    variable_level_2='FREEZE FREE PERIOD LENGTH',
    variable_level_3='',
    detailed_variable='',
    identifier='5a0347ba-2684-4c4a-adc0-ddb63cbbde6b'
  ),
  'fc768468-62d4-40fa-8880-a773a855a496': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='FREEZE/FROST',
    variable_level_2='LAST FREEZE/FROST DATE',
    variable_level_3='',
    detailed_variable='',
    identifier='fc768468-62d4-40fa-8880-a773a855a496'
  ),
  '22b3623a-66c6-4616-8a6a-139ce119f672': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='FREEZE/FROST',
    variable_level_2='LAST FREEZE/FROST PROBABILITY',
    variable_level_3='',
    detailed_variable='',
    identifier='22b3623a-66c6-4616-8a6a-139ce119f672'
  ),
  'a8cc5031-9c46-4a73-a999-68cdaec453a5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='FREEZE/FROST',
    variable_level_2='LAST MODERATE FREEZE/FROST DATE',
    variable_level_3='',
    detailed_variable='',
    identifier='a8cc5031-9c46-4a73-a999-68cdaec453a5'
  ),
  'a2ea1792-c011-4c7c-95c7-3bd648b1b57b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='HAIL STORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a2ea1792-c011-4c7c-95c7-3bd648b1b57b'
  ),
  '4ec71dc5-66bf-453b-883a-ad6d31b3edde': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='HAIL STORMS',
    variable_level_2='HAIL PATH',
    variable_level_3='',
    detailed_variable='',
    identifier='4ec71dc5-66bf-453b-883a-ad6d31b3edde'
  ),
  'ca820557-401e-4e5e-ac32-29fdbc0628b3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='HEAT WAVE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ca820557-401e-4e5e-ac32-29fdbc0628b3'
  ),
  '5ce75010-ec8a-4af7-9e34-3e49ef2fe10c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='ICE STORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5ce75010-ec8a-4af7-9e34-3e49ef2fe10c'
  ),
  '0df15471-3175-44c0-aa8b-5178dfeb27a0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='ICE STORMS',
    variable_level_2='TOTAL FREEZING RAIN ACCUMULATION',
    variable_level_3='',
    detailed_variable='',
    identifier='0df15471-3175-44c0-aa8b-5178dfeb27a0'
  ),
  'f24c4f33-5b89-4e8d-8de7-296078a7f18a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='LIGHTNING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f24c4f33-5b89-4e8d-8de7-296078a7f18a'
  ),
  'a6212424-1146-4a79-a14c-8ce88543b08b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='MONSOONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a6212424-1146-4a79-a14c-8ce88543b08b'
  ),
  'f6b314db-883a-4493-9140-b6afda949710': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='RAIN STORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f6b314db-883a-4493-9140-b6afda949710'
  ),
  'bc9215ae-58ec-481e-ba83-89376a298000': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='SNOW STORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bc9215ae-58ec-481e-ba83-89376a298000'
  ),
  '3d4f9f5a-912b-4dc1-b1c5-cd0fd9bbd3d3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='SNOW STORMS',
    variable_level_2='BLIZZARDS',
    variable_level_3='',
    detailed_variable='',
    identifier='3d4f9f5a-912b-4dc1-b1c5-cd0fd9bbd3d3'
  ),
  '63cad48d-d085-484f-9ae8-99e3b798671e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='SNOW STORMS',
    variable_level_2='BLOWING SNOW',
    variable_level_3='',
    detailed_variable='',
    identifier='63cad48d-d085-484f-9ae8-99e3b798671e'
  ),
  '12b7f57f-c295-4adf-97f5-43356f1270bf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='SNOW STORMS',
    variable_level_2='LAKE EFFECT SNOW',
    variable_level_3='',
    detailed_variable='',
    identifier='12b7f57f-c295-4adf-97f5-43356f1270bf'
  ),
  '7844ae66-f542-442f-8359-05014bc19831': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='Stability/Severe Weather Indices',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7844ae66-f542-442f-8359-05014bc19831'
  ),
  '00748b19-30cc-4d12-a7a3-0aa8b3be5a94': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='Stability/Severe Weather Indices',
    variable_level_2='CONVECTIVE AVAILABLE POTENTIAL ENERGY (CAPE)',
    variable_level_3='',
    detailed_variable='',
    identifier='00748b19-30cc-4d12-a7a3-0aa8b3be5a94'
  ),
  '1d8a8e42-0fc0-4ce1-a058-9fa961c9d4ac': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='Stability/Severe Weather Indices',
    variable_level_2='K-index (KI)',
    variable_level_3='',
    detailed_variable='',
    identifier='1d8a8e42-0fc0-4ce1-a058-9fa961c9d4ac'
  ),
  'f07365c3-a36e-4a28-8364-be3941fae000': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='Stability/Severe Weather Indices',
    variable_level_2='LIFTED INDEX (LI)',
    variable_level_3='',
    detailed_variable='',
    identifier='f07365c3-a36e-4a28-8364-be3941fae000'
  ),
  'bd0c62a2-5336-4b41-81e1-089ce118651a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='Stability/Severe Weather Indices',
    variable_level_2='SHOWALTER STABILITY INDEX (SI)',
    variable_level_3='',
    detailed_variable='',
    identifier='bd0c62a2-5336-4b41-81e1-089ce118651a'
  ),
  '77bcf3f2-8d61-4b18-9e2a-439310197c83': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='Stability/Severe Weather Indices',
    variable_level_2='TOTAL TOTALS INDEX (TT)',
    variable_level_3='',
    detailed_variable='',
    identifier='77bcf3f2-8d61-4b18-9e2a-439310197c83'
  ),
  'edfe982b-a5bb-4001-83fa-f46f90f69b79': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='SUBTROPICAL CYCLONES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='edfe982b-a5bb-4001-83fa-f46f90f69b79'
  ),
  '99ad9306-0a99-402a-961f-acb9255cb113': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='SUBTROPICAL CYCLONES',
    variable_level_2='SUBTROPICAL DEPRESSION',
    variable_level_3='',
    detailed_variable='',
    identifier='99ad9306-0a99-402a-961f-acb9255cb113'
  ),
  '241d4bbb-3965-4595-93d3-8fe8c89fdab1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='SUBTROPICAL CYCLONES',
    variable_level_2='SUBTROPICAL DEPRESSION',
    variable_level_3='SUBTROPICAL DEPRESSION TRACK',
    detailed_variable='',
    identifier='241d4bbb-3965-4595-93d3-8fe8c89fdab1'
  ),
  'ca133c4d-9751-4b92-a1ec-013ef625ad7b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='SUBTROPICAL CYCLONES',
    variable_level_2='SUBTROPICAL STORM',
    variable_level_3='',
    detailed_variable='',
    identifier='ca133c4d-9751-4b92-a1ec-013ef625ad7b'
  ),
  '308beca2-b3c8-4cbb-aa9c-e1be605ca785': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='SUBTROPICAL CYCLONES',
    variable_level_2='SUBTROPICAL STORM',
    variable_level_3='SUBTROPICAL STORM MOTION',
    detailed_variable='',
    identifier='308beca2-b3c8-4cbb-aa9c-e1be605ca785'
  ),
  'c1a196a3-4134-473a-819e-369ab9656abb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='SUBTROPICAL CYCLONES',
    variable_level_2='SUBTROPICAL STORM',
    variable_level_3='SUBTROPICAL STORM TRACK',
    detailed_variable='',
    identifier='c1a196a3-4134-473a-819e-369ab9656abb'
  ),
  'a200e677-384a-42d6-8519-1c7735f0adb9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TORNADOES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a200e677-384a-42d6-8519-1c7735f0adb9'
  ),
  '8fd6e7bc-df59-4637-b1e7-d6715fb3e8af': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TORNADOES',
    variable_level_2='DESTRUCTION POTENTIAL INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='8fd6e7bc-df59-4637-b1e7-d6715fb3e8af'
  ),
  'd866f0ba-c70a-4377-9f91-58ab402f6f8b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TORNADOES',
    variable_level_2='ENHANCED FUJITA SCALE RATING',
    variable_level_3='',
    detailed_variable='',
    identifier='d866f0ba-c70a-4377-9f91-58ab402f6f8b'
  ),
  'd9969cf1-6a1f-4f37-91bf-c746aeba81c4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TORNADOES',
    variable_level_2='STORM SYSTEM MOTION',
    variable_level_3='',
    detailed_variable='',
    identifier='d9969cf1-6a1f-4f37-91bf-c746aeba81c4'
  ),
  'd912e61f-6c95-449d-9bee-2eac2f599b8f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TORNADOES',
    variable_level_2='TORNADO CLIMATOLOGY',
    variable_level_3='',
    detailed_variable='',
    identifier='d912e61f-6c95-449d-9bee-2eac2f599b8f'
  ),
  '9a310897-86d4-4a31-9fe3-4b4ad45b3575': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TORNADOES',
    variable_level_2='TORNADO DENSITY',
    variable_level_3='',
    detailed_variable='',
    identifier='9a310897-86d4-4a31-9fe3-4b4ad45b3575'
  ),
  'de691f09-0ef3-4795-bac0-1ed15c3e7f8b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TORNADOES',
    variable_level_2='TORNADO FREQUENCY',
    variable_level_3='',
    detailed_variable='',
    identifier='de691f09-0ef3-4795-bac0-1ed15c3e7f8b'
  ),
  'b253d76b-d48a-4d7a-abbe-7d02f783176e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TORNADOES',
    variable_level_2='TORNADO PATH LENGTH',
    variable_level_3='',
    detailed_variable='',
    identifier='b253d76b-d48a-4d7a-abbe-7d02f783176e'
  ),
  '069ff99d-1455-4285-83d9-4f57fb0cb635': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TORNADOES',
    variable_level_2='TORNADO PATH WIDTH',
    variable_level_3='',
    detailed_variable='',
    identifier='069ff99d-1455-4285-83d9-4f57fb0cb635'
  ),
  'c3354d3b-44a4-4b1a-b1dd-1243bd1640be': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TORNADOES',
    variable_level_2='TORNADO VORTEX SIGNATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='c3354d3b-44a4-4b1a-b1dd-1243bd1640be'
  ),
  '2992d7d3-5ae6-4844-b0fa-4ad348e3a8c2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TORNADOES',
    variable_level_2='WATER SPOUT',
    variable_level_3='',
    detailed_variable='',
    identifier='2992d7d3-5ae6-4844-b0fa-4ad348e3a8c2'
  ),
  '06180441-d4bb-4fed-b36a-9b3cb2cac0fe': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='06180441-d4bb-4fed-b36a-9b3cb2cac0fe'
  ),
  '2ead8ea2-0357-4c95-9483-da8149855fd4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='ACCUMULATED CYCLONE ENERGY',
    variable_level_3='',
    detailed_variable='',
    identifier='2ead8ea2-0357-4c95-9483-da8149855fd4'
  ),
  '7067a3f8-2903-46b7-9189-af1189a15a43': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='ACCUMULATED CYCLONE ENERGY',
    variable_level_3='CYCLONES (SW INDIAN)',
    detailed_variable='',
    identifier='7067a3f8-2903-46b7-9189-af1189a15a43'
  ),
  'fb890034-3ae6-4c91-941c-ae1483a13528': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='ACCUMULATED CYCLONE ENERGY',
    variable_level_3='HURRICANES (N. ATLANTIC/E. PACIFIC)',
    detailed_variable='',
    identifier='fb890034-3ae6-4c91-941c-ae1483a13528'
  ),
  '5da932fa-2f4b-4f65-bad4-18c661816549': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='ACCUMULATED CYCLONE ENERGY',
    variable_level_3='SEVERE CYCLONIC STORMS (N. INDIAN)',
    detailed_variable='',
    identifier='5da932fa-2f4b-4f65-bad4-18c661816549'
  ),
  'e89e331c-ca8e-4c25-be34-c81017bd019f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='ACCUMULATED CYCLONE ENERGY',
    variable_level_3='SEVERE TROPICAL CYCLONES (SW PACIFIC/SE INDIAN)',
    detailed_variable='',
    identifier='e89e331c-ca8e-4c25-be34-c81017bd019f'
  ),
  '074c2800-e458-4fa0-bcae-7f400d970650': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='ACCUMULATED CYCLONE ENERGY',
    variable_level_3='TYPHOONS (WESTERN N. PACIFIC)',
    detailed_variable='',
    identifier='074c2800-e458-4fa0-bcae-7f400d970650'
  ),
  '923ab959-48ee-4db1-827a-3d672099e273': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='LANDFALL INTENSITY',
    variable_level_3='',
    detailed_variable='',
    identifier='923ab959-48ee-4db1-827a-3d672099e273'
  ),
  '0d7ea0fa-987a-4429-85e7-754ca638e504': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='LANDFALL INTENSITY',
    variable_level_3='CYCLONES (SW INDIAN)',
    detailed_variable='',
    identifier='0d7ea0fa-987a-4429-85e7-754ca638e504'
  ),
  '4354779d-94e6-4c38-973b-3a9bafa4eeb2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='LANDFALL INTENSITY',
    variable_level_3='HURRICANES (N. ATLANTIC/E. PACIFIC)',
    detailed_variable='',
    identifier='4354779d-94e6-4c38-973b-3a9bafa4eeb2'
  ),
  '7aa4aea2-0f5b-4490-967b-7e339eaec507': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='LANDFALL INTENSITY',
    variable_level_3='SEVERE CYCLONIC STORMS (N. INDIAN)',
    detailed_variable='',
    identifier='7aa4aea2-0f5b-4490-967b-7e339eaec507'
  ),
  'ab9dfb44-979e-495c-ad83-8d30a37018be': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='LANDFALL INTENSITY',
    variable_level_3='SEVERE TROPICAL CYCLONES (SW PACIFIC/SE INDIAN)',
    detailed_variable='',
    identifier='ab9dfb44-979e-495c-ad83-8d30a37018be'
  ),
  'dd5cbcc2-622a-4c3c-82b8-7e2869f8438a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='LANDFALL INTENSITY',
    variable_level_3='TYPHOONS (WESTERN N. PACIFIC)',
    detailed_variable='',
    identifier='dd5cbcc2-622a-4c3c-82b8-7e2869f8438a'
  ),
  'ba286b68-a400-4c29-bd24-b8ca99967968': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MAXIMUM 1-MINUTE SUSTAINED WIND',
    variable_level_3='',
    detailed_variable='',
    identifier='ba286b68-a400-4c29-bd24-b8ca99967968'
  ),
  '1ab5b26c-8560-412c-8b7b-80921aff9fe1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MAXIMUM 1-MINUTE SUSTAINED WIND',
    variable_level_3='CYCLONES (SW INDIAN)',
    detailed_variable='',
    identifier='1ab5b26c-8560-412c-8b7b-80921aff9fe1'
  ),
  '93f7b0c1-ea76-431f-8cb0-0599eb51f928': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MAXIMUM 1-MINUTE SUSTAINED WIND',
    variable_level_3='HURRICANES (N. ATLANTIC/E. PACIFIC)',
    detailed_variable='',
    identifier='93f7b0c1-ea76-431f-8cb0-0599eb51f928'
  ),
  '58ddb82c-fbb2-4910-8259-d9c2df2555da': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MAXIMUM 1-MINUTE SUSTAINED WIND',
    variable_level_3='SEVERE CYCLONIC STORMS (N. INDIAN)',
    detailed_variable='',
    identifier='58ddb82c-fbb2-4910-8259-d9c2df2555da'
  ),
  'dd54dfac-069b-4552-abfe-d182320189c7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MAXIMUM 1-MINUTE SUSTAINED WIND',
    variable_level_3='SEVERE TROPICAL CYCLONES (SW PACIFIC/SE INDIAN)',
    detailed_variable='',
    identifier='dd54dfac-069b-4552-abfe-d182320189c7'
  ),
  '53998f98-9bf6-4666-90c7-48f2e5730dae': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MAXIMUM 1-MINUTE SUSTAINED WIND',
    variable_level_3='TYPHOONS (WESTERN N. PACIFIC)',
    detailed_variable='',
    identifier='53998f98-9bf6-4666-90c7-48f2e5730dae'
  ),
  '106461af-377c-4dc0-bbd7-9769eba05321': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MAXIMUM SURFACE WIND',
    variable_level_3='',
    detailed_variable='',
    identifier='106461af-377c-4dc0-bbd7-9769eba05321'
  ),
  'b5965fe4-fc00-4d9b-93f8-f03a6a369304': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MAXIMUM SURFACE WIND',
    variable_level_3='CYCLONES (SW INDIAN)',
    detailed_variable='',
    identifier='b5965fe4-fc00-4d9b-93f8-f03a6a369304'
  ),
  '8807cdb6-56af-43d6-9efa-14d234d69374': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MAXIMUM SURFACE WIND',
    variable_level_3='HURRICANES (N. ATLANTIC/E. PACIFIC)',
    detailed_variable='',
    identifier='8807cdb6-56af-43d6-9efa-14d234d69374'
  ),
  '40f7445f-1741-418e-9831-e2e3322daf5a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MAXIMUM SURFACE WIND',
    variable_level_3='SEVERE CYCLONIC STORMS (N. INDIAN)',
    detailed_variable='',
    identifier='40f7445f-1741-418e-9831-e2e3322daf5a'
  ),
  'a8258a99-866f-4e34-80ab-25239546ffb2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MAXIMUM SURFACE WIND',
    variable_level_3='SEVERE TROPICAL CYCLONES (SW PACIFIC/SE INDIAN)',
    detailed_variable='',
    identifier='a8258a99-866f-4e34-80ab-25239546ffb2'
  ),
  '8e93861c-5f03-4892-96d7-cfac368e6c26': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MAXIMUM SURFACE WIND',
    variable_level_3='TYPHOONS (WESTERN N. PACIFIC)',
    detailed_variable='',
    identifier='8e93861c-5f03-4892-96d7-cfac368e6c26'
  ),
  '4b0e986f-5dce-48ca-8bad-794c97482553': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MAXIMUM WIND GUST',
    variable_level_3='',
    detailed_variable='',
    identifier='4b0e986f-5dce-48ca-8bad-794c97482553'
  ),
  'd5f307ab-e5df-4c84-84e7-42822e3a4864': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MAXIMUM WIND GUST',
    variable_level_3='CYCLONES (SW INDIAN)',
    detailed_variable='',
    identifier='d5f307ab-e5df-4c84-84e7-42822e3a4864'
  ),
  '2d2a56cb-a99c-4001-9f41-0e04037e0d41': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MAXIMUM WIND GUST',
    variable_level_3='HURRICANES (N. ATLANTIC/E. PACIFIC)',
    detailed_variable='',
    identifier='2d2a56cb-a99c-4001-9f41-0e04037e0d41'
  ),
  '6e28bebd-0c5d-4bf3-8770-84d79c75e33c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MAXIMUM WIND GUST',
    variable_level_3='SEVERE CYCLONIC STORMS (N. INDIAN)',
    detailed_variable='',
    identifier='6e28bebd-0c5d-4bf3-8770-84d79c75e33c'
  ),
  '4379a82a-c0fd-4d40-b1f3-3b516cac1a8e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MAXIMUM WIND GUST',
    variable_level_3='SEVERE TROPICAL CYCLONES (SW PACIFIC/SE INDIAN)',
    detailed_variable='',
    identifier='4379a82a-c0fd-4d40-b1f3-3b516cac1a8e'
  ),
  '536b666d-a4ad-4ec3-b7fc-282e884e53ee': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MAXIMUM WIND GUST',
    variable_level_3='TYPHOONS (WESTERN N. PACIFIC)',
    detailed_variable='',
    identifier='536b666d-a4ad-4ec3-b7fc-282e884e53ee'
  ),
  '38cefcb2-f5d6-4917-a87b-7cfba482e30d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MINIMUM CENTRAL PRESSURE',
    variable_level_3='',
    detailed_variable='',
    identifier='38cefcb2-f5d6-4917-a87b-7cfba482e30d'
  ),
  'ef467c3c-0aed-4aa8-bfa5-67721e83e557': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MINIMUM CENTRAL PRESSURE',
    variable_level_3='CYCLONES (SW INDIAN)',
    detailed_variable='',
    identifier='ef467c3c-0aed-4aa8-bfa5-67721e83e557'
  ),
  '3b1544bc-1711-4553-a643-5d8fba38a1f1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MINIMUM CENTRAL PRESSURE',
    variable_level_3='HURRICANES (N. ATLANTIC/E. PACIFIC)',
    detailed_variable='',
    identifier='3b1544bc-1711-4553-a643-5d8fba38a1f1'
  ),
  '50abff20-11a8-4aea-8425-c9a05b1d8d09': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MINIMUM CENTRAL PRESSURE',
    variable_level_3='SEVERE CYCLONIC STORMS (N. INDIAN)',
    detailed_variable='',
    identifier='50abff20-11a8-4aea-8425-c9a05b1d8d09'
  ),
  '5b70e02b-0ed2-42a0-9fe9-7a552d6819d1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MINIMUM CENTRAL PRESSURE',
    variable_level_3='SEVERE TROPICAL CYCLONES (SW PACIFIC/SE INDIAN)',
    detailed_variable='',
    identifier='5b70e02b-0ed2-42a0-9fe9-7a552d6819d1'
  ),
  'cfb85bf2-9920-4e3f-bce3-3d8f68ab1436': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='MINIMUM CENTRAL PRESSURE',
    variable_level_3='TYPHOONS (WESTERN N. PACIFIC)',
    detailed_variable='',
    identifier='cfb85bf2-9920-4e3f-bce3-3d8f68ab1436'
  ),
  'c17617a1-5d2b-426f-bfe0-d8c4d4b5cfad': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='PEAK INTENSITY',
    variable_level_3='',
    detailed_variable='',
    identifier='c17617a1-5d2b-426f-bfe0-d8c4d4b5cfad'
  ),
  'd038c99b-efbc-41f3-99a6-5d066fda5ecd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='PEAK INTENSITY',
    variable_level_3='CYCLONES (SW INDIAN)',
    detailed_variable='',
    identifier='d038c99b-efbc-41f3-99a6-5d066fda5ecd'
  ),
  '5730c1ba-7e4e-4d0e-adf3-053af4be97b4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='PEAK INTENSITY',
    variable_level_3='HURRICANES (N. ATLANTIC/E. PACIFIC)',
    detailed_variable='',
    identifier='5730c1ba-7e4e-4d0e-adf3-053af4be97b4'
  ),
  'b21b9b00-5da4-47fc-b2a8-fc2ecd5bd912': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='PEAK INTENSITY',
    variable_level_3='SEVERE CYCLONIC STORMS (N. INDIAN)',
    detailed_variable='',
    identifier='b21b9b00-5da4-47fc-b2a8-fc2ecd5bd912'
  ),
  'cbe89018-3eb6-4c8e-82c9-c540147a75e2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='PEAK INTENSITY',
    variable_level_3='SEVERE TROPICAL CYCLONES (SW PACIFIC/SE INDIAN)',
    detailed_variable='',
    identifier='cbe89018-3eb6-4c8e-82c9-c540147a75e2'
  ),
  '20da8cba-3546-4699-8809-01bffa6bccca': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='PEAK INTENSITY',
    variable_level_3='TYPHOONS (WESTERN N. PACIFIC)',
    detailed_variable='',
    identifier='20da8cba-3546-4699-8809-01bffa6bccca'
  ),
  '27847732-2a5a-4094-9ba5-3c56ae897f87': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='SAFFIR-SIMPSON SCALE AT LANDFALL (CATEGORY 1)',
    variable_level_3='',
    detailed_variable='',
    identifier='27847732-2a5a-4094-9ba5-3c56ae897f87'
  ),
  'c6ff6623-a24c-494c-804c-bc486b3de548': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='SAFFIR-SIMPSON SCALE AT LANDFALL (CATEGORY 1)',
    variable_level_3='HURRICANES (N. ATLANTIC/E. PACIFIC)',
    detailed_variable='',
    identifier='c6ff6623-a24c-494c-804c-bc486b3de548'
  ),
  'e282c375-ed1a-465b-b960-aa49118307ea': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='SAFFIR-SIMPSON SCALE AT LANDFALL (CATEGORY 2)',
    variable_level_3='',
    detailed_variable='',
    identifier='e282c375-ed1a-465b-b960-aa49118307ea'
  ),
  'fe4f3f33-7df3-439a-9382-d02140da29aa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='SAFFIR-SIMPSON SCALE AT LANDFALL (CATEGORY 2)',
    variable_level_3='HURRICANES (N. ATLANTIC/E. PACIFIC)',
    detailed_variable='',
    identifier='fe4f3f33-7df3-439a-9382-d02140da29aa'
  ),
  '530dfe77-5740-49e8-b994-9a6f82cf4adb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='SAFFIR-SIMPSON SCALE AT LANDFALL (CATEGORY 3)',
    variable_level_3='',
    detailed_variable='',
    identifier='530dfe77-5740-49e8-b994-9a6f82cf4adb'
  ),
  'd678b2d9-9956-45a9-9a9f-95450fb4ca46': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='SAFFIR-SIMPSON SCALE AT LANDFALL (CATEGORY 3)',
    variable_level_3='HURRICANES (N. ATLANTIC/E. PACIFIC)',
    detailed_variable='',
    identifier='d678b2d9-9956-45a9-9a9f-95450fb4ca46'
  ),
  'e691d1ab-6d20-4ad6-bea6-46587e94c4ff': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='SAFFIR-SIMPSON SCALE AT LANDFALL (CATEGORY 4)',
    variable_level_3='',
    detailed_variable='',
    identifier='e691d1ab-6d20-4ad6-bea6-46587e94c4ff'
  ),
  '91843b75-9519-456a-89a5-1b1c221ebd4e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='SAFFIR-SIMPSON SCALE AT LANDFALL (CATEGORY 4)',
    variable_level_3='HURRICANES (N. ATLANTIC/E. PACIFIC)',
    detailed_variable='',
    identifier='91843b75-9519-456a-89a5-1b1c221ebd4e'
  ),
  '978dd843-3a96-4d52-a7d6-31642503c267': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='SAFFIR-SIMPSON SCALE AT LANDFALL (CATEGORY 5)',
    variable_level_3='',
    detailed_variable='',
    identifier='978dd843-3a96-4d52-a7d6-31642503c267'
  ),
  '6f80bcdf-b778-4ecc-99aa-9f5779fd6f31': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='SAFFIR-SIMPSON SCALE AT LANDFALL (CATEGORY 5)',
    variable_level_3='HURRICANES (N. ATLANTIC/E. PACIFIC)',
    detailed_variable='',
    identifier='6f80bcdf-b778-4ecc-99aa-9f5779fd6f31'
  ),
  'ed710f29-031f-45b3-8fa7-aeb17e8ce564': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE CENTER',
    variable_level_3='',
    detailed_variable='',
    identifier='ed710f29-031f-45b3-8fa7-aeb17e8ce564'
  ),
  'eec57358-8166-443e-b595-cb831911cd42': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE FORCE WIND EXTENT',
    variable_level_3='',
    detailed_variable='',
    identifier='eec57358-8166-443e-b595-cb831911cd42'
  ),
  '00d89979-f1bb-4e95-b73e-6a0d8d924bd8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE FORCE WIND EXTENT',
    variable_level_3='CYCLONES (SW INDIAN)',
    detailed_variable='',
    identifier='00d89979-f1bb-4e95-b73e-6a0d8d924bd8'
  ),
  'd99d0464-db69-44fb-9b18-9469a08fe4b4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE FORCE WIND EXTENT',
    variable_level_3='HURRICANES (N. ATLANTIC/E. PACIFIC)',
    detailed_variable='',
    identifier='d99d0464-db69-44fb-9b18-9469a08fe4b4'
  ),
  'ed71cef0-0e5a-49a0-83c6-f7dd02215290': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE FORCE WIND EXTENT',
    variable_level_3='SEVERE CYCLONIC STORMS (N. INDIAN)',
    detailed_variable='',
    identifier='ed71cef0-0e5a-49a0-83c6-f7dd02215290'
  ),
  '713123e4-ebc8-49dd-bc8b-b9fbeaabeaad': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE FORCE WIND EXTENT',
    variable_level_3='SEVERE TROPICAL CYCLONES (SW PACIFIC/SE INDIAN)',
    detailed_variable='',
    identifier='713123e4-ebc8-49dd-bc8b-b9fbeaabeaad'
  ),
  '6ee22b9c-f418-4b77-bb6b-f70d3e44afbc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE FORCE WIND EXTENT',
    variable_level_3='TYPHOONS (WESTERN N. PACIFIC)',
    detailed_variable='',
    identifier='6ee22b9c-f418-4b77-bb6b-f70d3e44afbc'
  ),
  'cda34c9c-e59a-4dfb-9d2d-b8317e4b7f27': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE MOTION',
    variable_level_3='',
    detailed_variable='',
    identifier='cda34c9c-e59a-4dfb-9d2d-b8317e4b7f27'
  ),
  '63e53301-d263-4d09-a4be-f0c874646e23': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE MOTION',
    variable_level_3='CYCLONES (SW INDIAN)',
    detailed_variable='',
    identifier='63e53301-d263-4d09-a4be-f0c874646e23'
  ),
  'a8a40309-c4e5-46d7-ac39-1b7230766192': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE MOTION',
    variable_level_3='HURRICANES (N. ATLANTIC/E. PACIFIC)',
    detailed_variable='',
    identifier='a8a40309-c4e5-46d7-ac39-1b7230766192'
  ),
  '446a22b7-3ea1-43db-9176-47d4dac3ac93': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE MOTION',
    variable_level_3='SEVERE CYCLONIC STORMS (N. INDIAN)',
    detailed_variable='',
    identifier='446a22b7-3ea1-43db-9176-47d4dac3ac93'
  ),
  '7705e65c-90a1-451d-8898-ef5f170fa051': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE MOTION',
    variable_level_3='SEVERE TROPICAL CYCLONES (SW PACIFIC/SE INDIAN)',
    detailed_variable='',
    identifier='7705e65c-90a1-451d-8898-ef5f170fa051'
  ),
  '93ef0499-0b06-4f9a-885b-52e89563b3ec': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE MOTION',
    variable_level_3='TYPHOONS (WESTERN N. PACIFIC)',
    detailed_variable='',
    identifier='93ef0499-0b06-4f9a-885b-52e89563b3ec'
  ),
  '104ed5fa-f65a-442e-992c-88a4fe74a66c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE RADIUS',
    variable_level_3='',
    detailed_variable='',
    identifier='104ed5fa-f65a-442e-992c-88a4fe74a66c'
  ),
  'f4f4a7ad-73da-42f2-94f9-d9ecb81e0bf0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE RADIUS',
    variable_level_3='CYCLONES (SW INDIAN)',
    detailed_variable='',
    identifier='f4f4a7ad-73da-42f2-94f9-d9ecb81e0bf0'
  ),
  '41829fbf-2b76-4714-bf4a-e0d63b5472d5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE RADIUS',
    variable_level_3='HURRICANES (N. ATLANTIC/E. PACIFIC)',
    detailed_variable='',
    identifier='41829fbf-2b76-4714-bf4a-e0d63b5472d5'
  ),
  '9928589d-0714-4b88-a8ad-11126dd97521': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE RADIUS',
    variable_level_3='SEVERE CYCLONIC STORMS (N. INDIAN)',
    detailed_variable='',
    identifier='9928589d-0714-4b88-a8ad-11126dd97521'
  ),
  '5d51ef9b-f058-48ca-b1ea-c8d63a50a699': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE RADIUS',
    variable_level_3='SEVERE TROPICAL CYCLONES (SW PACIFIC/SE INDIAN)',
    detailed_variable='',
    identifier='5d51ef9b-f058-48ca-b1ea-c8d63a50a699'
  ),
  '09849cf3-df4d-40d3-a224-f30c6fe22c1f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE RADIUS',
    variable_level_3='TYPHOONS (WESTERN N. PACIFIC)',
    detailed_variable='',
    identifier='09849cf3-df4d-40d3-a224-f30c6fe22c1f'
  ),
  '10a9bb22-9119-4409-84c1-7c97ef31b1a1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE TRACK',
    variable_level_3='',
    detailed_variable='',
    identifier='10a9bb22-9119-4409-84c1-7c97ef31b1a1'
  ),
  '5d0a21f1-cc5d-481c-ad5f-7fe15deabc9c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE TRACK',
    variable_level_3='CYCLONES (SW INDIAN)',
    detailed_variable='',
    identifier='5d0a21f1-cc5d-481c-ad5f-7fe15deabc9c'
  ),
  '72de9813-4c72-45bc-a216-be6ebd08bb6c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE TRACK',
    variable_level_3='HURRICANES (N. ATLANTIC/E. PACIFIC)',
    detailed_variable='',
    identifier='72de9813-4c72-45bc-a216-be6ebd08bb6c'
  ),
  'e61fcc9f-bdb6-4dbc-94f2-52c4c64b6df9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE TRACK',
    variable_level_3='SEVERE CYCLONIC STORMS (N. INDIAN)',
    detailed_variable='',
    identifier='e61fcc9f-bdb6-4dbc-94f2-52c4c64b6df9'
  ),
  '8d27af08-6b2f-48d7-8e6b-bd57e93992ad': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE TRACK',
    variable_level_3='SEVERE TROPICAL CYCLONES (SW PACIFIC/SE INDIAN)',
    detailed_variable='',
    identifier='8d27af08-6b2f-48d7-8e6b-bd57e93992ad'
  ),
  'b5a681af-5005-4182-922e-528ec8d514f1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL CYCLONE TRACK',
    variable_level_3='TYPHOONS (WESTERN N. PACIFIC)',
    detailed_variable='',
    identifier='b5a681af-5005-4182-922e-528ec8d514f1'
  ),
  '74aac882-80ae-4ecd-9585-c541cd7a10fc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL DEPRESSION FREQUENCY',
    variable_level_3='',
    detailed_variable='',
    identifier='74aac882-80ae-4ecd-9585-c541cd7a10fc'
  ),
  '03e9cfd2-631c-42e6-b25c-b75f57e4ebb8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL DEPRESSION MOTION',
    variable_level_3='',
    detailed_variable='',
    identifier='03e9cfd2-631c-42e6-b25c-b75f57e4ebb8'
  ),
  '75c369df-2b9f-4328-8b1f-325d83ffb4cf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL DEPRESSION TRACK',
    variable_level_3='',
    detailed_variable='',
    identifier='75c369df-2b9f-4328-8b1f-325d83ffb4cf'
  ),
  '23c94a4c-db57-4d57-b24f-4dba24aa3cc6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL STORM FORCE WIND EXTENT',
    variable_level_3='',
    detailed_variable='',
    identifier='23c94a4c-db57-4d57-b24f-4dba24aa3cc6'
  ),
  'de9ffa22-76e3-469c-926b-2dee007702d0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL STORM FREQUENCY',
    variable_level_3='',
    detailed_variable='',
    identifier='de9ffa22-76e3-469c-926b-2dee007702d0'
  ),
  'ce15b57a-9b1b-4bb7-805e-b13defd9a851': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL STORM MOTION',
    variable_level_3='',
    detailed_variable='',
    identifier='ce15b57a-9b1b-4bb7-805e-b13defd9a851'
  ),
  '2a4bc557-ee60-4446-920a-25632f5b8b4d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TROPICAL STORM TRACK',
    variable_level_3='',
    detailed_variable='',
    identifier='2a4bc557-ee60-4446-920a-25632f5b8b4d'
  ),
  'c5846f45-863e-43a8-8816-95b3ff359e40': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='WEATHER/CLIMATE ADVISORIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c5846f45-863e-43a8-8816-95b3ff359e40'
  ),
  '0fd11f45-185a-4e38-8749-092ee09fab36': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='WEATHER/CLIMATE ADVISORIES',
    variable_level_2='Weather Forecast',
    variable_level_3='',
    detailed_variable='',
    identifier='0fd11f45-185a-4e38-8749-092ee09fab36'
  ),
  'c40071d2-6478-4edf-80bb-95c3886533b9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='WIND STORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c40071d2-6478-4edf-80bb-95c3886533b9'
  ),
  '4e845edf-3635-4665-9d9d-d7186c151cda': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='WIND STORMS',
    variable_level_2='DERECHO',
    variable_level_3='',
    detailed_variable='',
    identifier='4e845edf-3635-4665-9d9d-d7186c151cda'
  ),
  '530ca9b8-50f3-4bd6-82d4-c49fa688a977': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='WIND STORMS',
    variable_level_2='GALE',
    variable_level_3='',
    detailed_variable='',
    identifier='530ca9b8-50f3-4bd6-82d4-c49fa688a977'
  ),
  '9da19ae9-799f-4885-8fb8-564ca803639a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='WIND STORMS',
    variable_level_2='MICROBURST',
    variable_level_3='',
    detailed_variable='',
    identifier='9da19ae9-799f-4885-8fb8-564ca803639a'
  ),
  '27275638-546e-4181-b15c-ddc3524de3d5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='ATMOSPHERE',
    term='WEATHER EVENTS',
    variable_level_1='WIND STORMS',
    variable_level_2='SQUALL',
    variable_level_3='',
    detailed_variable='',
    identifier='27275638-546e-4181-b15c-ddc3524de3d5'
  ),
  'fbec5145-79e6-4ed0-a804-6228aa6daba5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fbec5145-79e6-4ed0-a804-6228aa6daba5'
  ),
  'abc6f016-d1f0-4725-b847-639de054d13f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='abc6f016-d1f0-4725-b847-639de054d13f'
  ),
  '70c0b882-3d34-4e2d-90bf-339ade328ee0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ACORN WORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='70c0b882-3d34-4e2d-90bf-339ade328ee0'
  ),
  '328d3442-34a0-496b-ae4d-87eb447058b8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ARROW WORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='328d3442-34a0-496b-ae4d-87eb447058b8'
  ),
  'bb87baf5-3844-4a56-865f-ea5ed420db06': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ARTHROPODS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bb87baf5-3844-4a56-865f-ea5ed420db06'
  ),
  'b32ca9df-f981-4696-bf97-c190175e47b7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ARTHROPODS',
    variable_level_2='CHELICERATES',
    variable_level_3='',
    detailed_variable='',
    identifier='b32ca9df-f981-4696-bf97-c190175e47b7'
  ),
  '973bd2bd-c201-4e8c-8c86-d2e849298310': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ARTHROPODS',
    variable_level_2='CHELICERATES',
    variable_level_3='ARACHNIDS',
    detailed_variable='',
    identifier='973bd2bd-c201-4e8c-8c86-d2e849298310'
  ),
  'f2044dcf-40da-4fcc-97ab-914343d885a5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ARTHROPODS',
    variable_level_2='CRUSTACEANS',
    variable_level_3='',
    detailed_variable='',
    identifier='f2044dcf-40da-4fcc-97ab-914343d885a5'
  ),
  'e20c4981-7cbe-4f5c-9139-78b22ee7bfb6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ARTHROPODS',
    variable_level_2='CRUSTACEANS',
    variable_level_3='AMPHIPODS',
    detailed_variable='',
    identifier='e20c4981-7cbe-4f5c-9139-78b22ee7bfb6'
  ),
  '57e04385-5f7b-432b-8f0b-b26fc9e3d77d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ARTHROPODS',
    variable_level_2='CRUSTACEANS',
    variable_level_3='BARNACLES',
    detailed_variable='',
    identifier='57e04385-5f7b-432b-8f0b-b26fc9e3d77d'
  ),
  '9443e3fb-5087-4aae-a311-35ab172c45ce': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ARTHROPODS',
    variable_level_2='CRUSTACEANS',
    variable_level_3='COPEPODS',
    detailed_variable='',
    identifier='9443e3fb-5087-4aae-a311-35ab172c45ce'
  ),
  'e631c681-5dad-48b2-83ce-943a1f0df47a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ARTHROPODS',
    variable_level_2='CRUSTACEANS',
    variable_level_3='DECAPODS',
    detailed_variable='',
    identifier='e631c681-5dad-48b2-83ce-943a1f0df47a'
  ),
  '4095dea9-3da1-4679-bccf-7fe637414910': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ARTHROPODS',
    variable_level_2='CRUSTACEANS',
    variable_level_3='EUPHAUSIIDS (KRILL)',
    detailed_variable='',
    identifier='4095dea9-3da1-4679-bccf-7fe637414910'
  ),
  'dcf06e40-74f1-4341-bc80-79dcd2e268b9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ARTHROPODS',
    variable_level_2='CRUSTACEANS',
    variable_level_3='ISOPODS',
    detailed_variable='',
    identifier='dcf06e40-74f1-4341-bc80-79dcd2e268b9'
  ),
  'bfda8569-896e-4efa-81e1-8f02af8b1017': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ARTHROPODS',
    variable_level_2='CRUSTACEANS',
    variable_level_3='MYSIDS',
    detailed_variable='',
    identifier='bfda8569-896e-4efa-81e1-8f02af8b1017'
  ),
  'e91ca626-7afa-4e36-8a28-f8df6fc9d797': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ARTHROPODS',
    variable_level_2='CRUSTACEANS',
    variable_level_3='OSTRACODS',
    detailed_variable='',
    identifier='e91ca626-7afa-4e36-8a28-f8df6fc9d797'
  ),
  '38e40180-1a2a-40a9-a030-04775dabbabb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ARTHROPODS',
    variable_level_2='HEXAPODS',
    variable_level_3='',
    detailed_variable='',
    identifier='38e40180-1a2a-40a9-a030-04775dabbabb'
  ),
  'b23a3120-0b90-434d-81e6-988f62034e22': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ARTHROPODS',
    variable_level_2='HEXAPODS',
    variable_level_3='ENTOGNATHA',
    detailed_variable='',
    identifier='b23a3120-0b90-434d-81e6-988f62034e22'
  ),
  '44e49605-e860-41d0-8ef8-cb74419f831d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ARTHROPODS',
    variable_level_2='HEXAPODS',
    variable_level_3='INSECTS',
    detailed_variable='',
    identifier='44e49605-e860-41d0-8ef8-cb74419f831d'
  ),
  '9b5474eb-2dc8-4a8e-90b4-872f9fda80d9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ARTHROPODS',
    variable_level_2='MYRIAPODS',
    variable_level_3='',
    detailed_variable='',
    identifier='9b5474eb-2dc8-4a8e-90b4-872f9fda80d9'
  ),
  '582efbf1-ae9c-47f4-8155-c445f3816dd8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ARTHROPODS',
    variable_level_2='MYRIAPODS',
    variable_level_3='CENTIPEDES',
    detailed_variable='',
    identifier='582efbf1-ae9c-47f4-8155-c445f3816dd8'
  ),
  'd6db51cf-d2d0-4203-94c8-c884579e0cb0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ARTHROPODS',
    variable_level_2='MYRIAPODS',
    variable_level_3='MILLIPEDES',
    detailed_variable='',
    identifier='d6db51cf-d2d0-4203-94c8-c884579e0cb0'
  ),
  'b560f23d-f190-4c41-8bd9-4650a83296af': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='BRYOZOANS/MOSS ANIMALS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b560f23d-f190-4c41-8bd9-4650a83296af'
  ),
  'f70d3181-c6b6-40ec-a583-6c9e44e1c4ad': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='BURROWS/SPOON WORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f70d3181-c6b6-40ec-a583-6c9e44e1c4ad'
  ),
  'b6164a29-8e14-4861-a30c-fefce375e284': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='CNIDARIANS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b6164a29-8e14-4861-a30c-fefce375e284'
  ),
  'ad557b31-fc70-4519-a8e4-3a5daf05f774': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='CNIDARIANS',
    variable_level_2='ANTHOZOANS/HEXACORALS',
    variable_level_3='',
    detailed_variable='',
    identifier='ad557b31-fc70-4519-a8e4-3a5daf05f774'
  ),
  'dd4de9c8-e078-43cf-a7d8-78f289c8618e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='CNIDARIANS',
    variable_level_2='ANTHOZOANS/HEXACORALS',
    variable_level_3='HARD OR STONY CORALS',
    detailed_variable='',
    identifier='dd4de9c8-e078-43cf-a7d8-78f289c8618e'
  ),
  '9c47c3f3-09ae-491f-994c-0322e2875a7e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='CNIDARIANS',
    variable_level_2='ANTHOZOANS/HEXACORALS',
    variable_level_3='SEA ANEMONES',
    detailed_variable='',
    identifier='9c47c3f3-09ae-491f-994c-0322e2875a7e'
  ),
  'cbdf4f94-efc6-4965-a329-5df989a9a211': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='CNIDARIANS',
    variable_level_2='ANTHOZOANS/OCTOCORALS',
    variable_level_3='',
    detailed_variable='',
    identifier='cbdf4f94-efc6-4965-a329-5df989a9a211'
  ),
  'c6b33bb7-9714-42b5-88d2-f16bd671b799': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='CNIDARIANS',
    variable_level_2='ANTHOZOANS/OCTOCORALS',
    variable_level_3='SEA FANS/SEA WHIPS',
    detailed_variable='',
    identifier='c6b33bb7-9714-42b5-88d2-f16bd671b799'
  ),
  '9f25a6bc-ccbd-44fd-a33f-36a2ed53827f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='CNIDARIANS',
    variable_level_2='ANTHOZOANS/OCTOCORALS',
    variable_level_3='SEA PENS',
    detailed_variable='',
    identifier='9f25a6bc-ccbd-44fd-a33f-36a2ed53827f'
  ),
  '7836e8bd-176d-4e2f-9ac1-7f9d9a152b4e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='CNIDARIANS',
    variable_level_2='ANTHOZOANS/OCTOCORALS',
    variable_level_3='SOFT CORALS',
    detailed_variable='',
    identifier='7836e8bd-176d-4e2f-9ac1-7f9d9a152b4e'
  ),
  'c2c891c2-aa15-40b8-bfae-f02f42d0c739': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='CNIDARIANS',
    variable_level_2='HYDROZOANS',
    variable_level_3='',
    detailed_variable='',
    identifier='c2c891c2-aa15-40b8-bfae-f02f42d0c739'
  ),
  'cb628a66-a10b-4ef1-9261-7ce63a9439dc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='CNIDARIANS',
    variable_level_2='JELLYFISHES',
    variable_level_3='',
    detailed_variable='',
    identifier='cb628a66-a10b-4ef1-9261-7ce63a9439dc'
  ),
  'acce07bc-4e22-48b8-8396-10628c13124f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='COMB JELLIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='acce07bc-4e22-48b8-8396-10628c13124f'
  ),
  '70892c25-4206-4673-9504-2876927d19a3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ECHINODERMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='70892c25-4206-4673-9504-2876927d19a3'
  ),
  '4c653917-a5d8-4572-a509-572e9fd2c63d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ECHINODERMS',
    variable_level_2='BRITTLE/BASKET STARS',
    variable_level_3='',
    detailed_variable='',
    identifier='4c653917-a5d8-4572-a509-572e9fd2c63d'
  ),
  'ec994afa-ecd4-4d25-9e8b-335cd982755c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ECHINODERMS',
    variable_level_2='SEA STARS',
    variable_level_3='',
    detailed_variable='',
    identifier='ec994afa-ecd4-4d25-9e8b-335cd982755c'
  ),
  '6972b6bd-2f7e-460d-b12a-914b7d1e029c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ECHINODERMS',
    variable_level_2='SEA URCHINS',
    variable_level_3='',
    detailed_variable='',
    identifier='6972b6bd-2f7e-460d-b12a-914b7d1e029c'
  ),
  '23193921-88ee-4ff2-b9ca-d4688aa4bda7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ENTOPROCTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='23193921-88ee-4ff2-b9ca-d4688aa4bda7'
  ),
  '8ce4bad9-f050-4b0c-845e-5e7569b6a2d2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='FLATWORMS/FLUKES/TAPEWORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8ce4bad9-f050-4b0c-845e-5e7569b6a2d2'
  ),
  'e05fac08-e3de-4f41-a3fa-29d322a99ac2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='GNATHOSTOMULIDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e05fac08-e3de-4f41-a3fa-29d322a99ac2'
  ),
  'e3425c65-ead7-4bf6-942c-7176a1469b58': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='HORSEHAIR WORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e3425c65-ead7-4bf6-942c-7176a1469b58'
  ),
  '28ae9814-61c2-4ca4-8bc5-d093c1ce5e83': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='LAMP SHELLS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='28ae9814-61c2-4ca4-8bc5-d093c1ce5e83'
  ),
  '470b8420-2a72-4a0c-9c87-e85c57bf01bb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='LORICIFERANS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='470b8420-2a72-4a0c-9c87-e85c57bf01bb'
  ),
  'd85c386f-e4f7-4e1c-a16e-34dbb12bb2be': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='MOLLUSKS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d85c386f-e4f7-4e1c-a16e-34dbb12bb2be'
  ),
  '2b24db47-ecf4-4559-aaff-aef150188b03': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='MOLLUSKS',
    variable_level_2='APLACOPHORANS',
    variable_level_3='',
    detailed_variable='',
    identifier='2b24db47-ecf4-4559-aaff-aef150188b03'
  ),
  '7da8400b-e2cf-4ab1-b2f0-5bc4b21c23b3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='MOLLUSKS',
    variable_level_2='BIVALVES',
    variable_level_3='',
    detailed_variable='',
    identifier='7da8400b-e2cf-4ab1-b2f0-5bc4b21c23b3'
  ),
  '9273a48a-7ca4-4f1a-9347-7f6599b5a7e3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='MOLLUSKS',
    variable_level_2='BIVALVES',
    variable_level_3='CLAMS',
    detailed_variable='',
    identifier='9273a48a-7ca4-4f1a-9347-7f6599b5a7e3'
  ),
  'bc60fbb8-f9e9-492e-9acf-0f47345cedf2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='MOLLUSKS',
    variable_level_2='BIVALVES',
    variable_level_3='MUSSELS',
    detailed_variable='',
    identifier='bc60fbb8-f9e9-492e-9acf-0f47345cedf2'
  ),
  'b6782a30-639e-4d70-8290-81683d248b1f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='MOLLUSKS',
    variable_level_2='BIVALVES',
    variable_level_3='OYSTERS',
    detailed_variable='',
    identifier='b6782a30-639e-4d70-8290-81683d248b1f'
  ),
  '955fae6f-6aae-460a-a952-1c0c30f5151e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='MOLLUSKS',
    variable_level_2='CEPHALOPODS',
    variable_level_3='',
    detailed_variable='',
    identifier='955fae6f-6aae-460a-a952-1c0c30f5151e'
  ),
  '0d07a910-06bf-4607-90f8-422e1f35cfa0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='MOLLUSKS',
    variable_level_2='CEPHALOPODS',
    variable_level_3='SQUIDS',
    detailed_variable='',
    identifier='0d07a910-06bf-4607-90f8-422e1f35cfa0'
  ),
  'e65cfeec-0da2-40d8-b80d-1c74d1a498fc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='MOLLUSKS',
    variable_level_2='CHITONS',
    variable_level_3='',
    detailed_variable='',
    identifier='e65cfeec-0da2-40d8-b80d-1c74d1a498fc'
  ),
  'd2db293d-2ed7-4831-9794-a2cf903e4d4d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='MOLLUSKS',
    variable_level_2='GASTROPODS',
    variable_level_3='',
    detailed_variable='',
    identifier='d2db293d-2ed7-4831-9794-a2cf903e4d4d'
  ),
  '3d179cd8-3d64-47a8-b665-ac1382053aff': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='PEANUT WORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3d179cd8-3d64-47a8-b665-ac1382053aff'
  ),
  'bb62f1cf-a6e5-4d9d-a3ab-c665b93ce072': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='PHORONIDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bb62f1cf-a6e5-4d9d-a3ab-c665b93ce072'
  ),
  '9c974992-0ec2-4c55-9ab9-e8158f446fe7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='PRIAPULANS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9c974992-0ec2-4c55-9ab9-e8158f446fe7'
  ),
  '0b5fd1dc-cfff-4bd8-9807-b5dd5ecf83fe': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='RIBBON WORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0b5fd1dc-cfff-4bd8-9807-b5dd5ecf83fe'
  ),
  '6b3f96de-62f8-482a-87a5-6efcc3414af7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ROTIFERS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6b3f96de-62f8-482a-87a5-6efcc3414af7'
  ),
  '2c1cf609-c70d-4811-8514-3ca45a8bb380': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='ROUNDWORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2c1cf609-c70d-4811-8514-3ca45a8bb380'
  ),
  'ab1952ae-de34-4299-ad10-9c9b2baf87f5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='SEGMENTED WORMS (ANNELIDS)',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ab1952ae-de34-4299-ad10-9c9b2baf87f5'
  ),
  'ea2c5c8f-6b57-4fcc-8c01-53343c706cef': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='SEGMENTED WORMS (ANNELIDS)',
    variable_level_2='BRISTLE WORMS',
    variable_level_3='',
    detailed_variable='',
    identifier='ea2c5c8f-6b57-4fcc-8c01-53343c706cef'
  ),
  '517e2978-223e-42a2-b889-9c60f7099859': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='SEGMENTED WORMS (ANNELIDS)',
    variable_level_2='EARTHWORMS',
    variable_level_3='',
    detailed_variable='',
    identifier='517e2978-223e-42a2-b889-9c60f7099859'
  ),
  'b845369b-bf6b-47a6-b56a-a11438604a39': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='SEGMENTED WORMS (ANNELIDS)',
    variable_level_2='LEECHES',
    variable_level_3='',
    detailed_variable='',
    identifier='b845369b-bf6b-47a6-b56a-a11438604a39'
  ),
  'af9520f0-6011-45e0-a1d8-bd8c3ed042b0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='SPINY-HEADED WORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='af9520f0-6011-45e0-a1d8-bd8c3ed042b0'
  ),
  'bfbfd84c-6bf0-412f-9e75-1bad5241c339': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='SPONGES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bfbfd84c-6bf0-412f-9e75-1bad5241c339'
  ),
  'c34af039-4868-41f0-aaf0-39e8e9554e03': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='TUNICATES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c34af039-4868-41f0-aaf0-39e8e9554e03'
  ),
  '8b1af14c-25f1-42bb-bba9-24ee5cee4e43': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='TUNICATES',
    variable_level_2='LARVACEANS',
    variable_level_3='',
    detailed_variable='',
    identifier='8b1af14c-25f1-42bb-bba9-24ee5cee4e43'
  ),
  'b63e1a64-661a-4228-8453-248076f612b7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='TUNICATES',
    variable_level_2='SALPS',
    variable_level_3='',
    detailed_variable='',
    identifier='b63e1a64-661a-4228-8453-248076f612b7'
  ),
  '9987f02e-2f2f-48ed-95ac-02514f02d7b0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='TUNICATES',
    variable_level_2='SEA SQUIRTS',
    variable_level_3='',
    detailed_variable='',
    identifier='9987f02e-2f2f-48ed-95ac-02514f02d7b0'
  ),
  'fb4834d9-7bfe-4283-86f7-931532baa79c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='WATER BEARS (TARDIGRADES)',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fb4834d9-7bfe-4283-86f7-931532baa79c'
  ),
  'fa4b61fa-32e7-420f-988f-df5527b6f935': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/INVERTEBRATES',
    variable_level_1='WHEEL WEAVERS (CYCLIOPHORANS)',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fa4b61fa-32e7-420f-988f-df5527b6f935'
  ),
  '14802b53-b702-438f-8c8a-f51506807ce6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='14802b53-b702-438f-8c8a-f51506807ce6'
  ),
  'a27837ae-62f7-4931-9da1-0bf63f4755fc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='AMPHIBIANS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a27837ae-62f7-4931-9da1-0bf63f4755fc'
  ),
  'db49ac33-d70a-488c-a1f2-9aa3706ba707': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='AMPHIBIANS',
    variable_level_2='FROGS/TOADS',
    variable_level_3='',
    detailed_variable='',
    identifier='db49ac33-d70a-488c-a1f2-9aa3706ba707'
  ),
  '1ac84a15-6f6b-48e0-b7ba-796813e5ff2c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='AMPHIBIANS',
    variable_level_2='SALAMANDERS',
    variable_level_3='',
    detailed_variable='',
    identifier='1ac84a15-6f6b-48e0-b7ba-796813e5ff2c'
  ),
  '1bf8f27a-d6ff-4cb6-acb7-7e5cce11e029': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='BIRDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1bf8f27a-d6ff-4cb6-acb7-7e5cce11e029'
  ),
  'b4e28ec2-c2a0-4eb4-9544-8eb227903d47': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='BIRDS',
    variable_level_2='ALBATROSSES/PETRELS AND ALLIES',
    variable_level_3='',
    detailed_variable='',
    identifier='b4e28ec2-c2a0-4eb4-9544-8eb227903d47'
  ),
  '39e33722-56b0-4928-a032-d4832f7136cc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='BIRDS',
    variable_level_2='CRANES AND ALLIES',
    variable_level_3='',
    detailed_variable='',
    identifier='39e33722-56b0-4928-a032-d4832f7136cc'
  ),
  'c310aa65-810b-4e36-9689-d37b1154fa1b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='BIRDS',
    variable_level_2='DUCKS/GEESE/SWANS',
    variable_level_3='',
    detailed_variable='',
    identifier='c310aa65-810b-4e36-9689-d37b1154fa1b'
  ),
  '3f51c987-e49b-4988-a05c-d2d9da82dd22': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='BIRDS',
    variable_level_2='EAGLES/FALCONS/HAWKS AND ALLIES',
    variable_level_3='',
    detailed_variable='',
    identifier='3f51c987-e49b-4988-a05c-d2d9da82dd22'
  ),
  '050ce2a5-f895-4600-a0a1-eb0e3adb09e1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='BIRDS',
    variable_level_2='GREBES',
    variable_level_3='',
    detailed_variable='',
    identifier='050ce2a5-f895-4600-a0a1-eb0e3adb09e1'
  ),
  'af67dee1-7c50-4e73-8db8-b1f421df67fb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='BIRDS',
    variable_level_2='HERONS/EGRETS AND ALLIES',
    variable_level_3='',
    detailed_variable='',
    identifier='af67dee1-7c50-4e73-8db8-b1f421df67fb'
  ),
  '70464ef6-7702-4b8d-bacc-50f44b0d6100': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='BIRDS',
    variable_level_2='IBISES/SPOONBILLS',
    variable_level_3='',
    detailed_variable='',
    identifier='70464ef6-7702-4b8d-bacc-50f44b0d6100'
  ),
  '4342be4c-fd26-4c02-b09f-e35ea4f34575': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='BIRDS',
    variable_level_2='LOONS',
    variable_level_3='',
    detailed_variable='',
    identifier='4342be4c-fd26-4c02-b09f-e35ea4f34575'
  ),
  '3a591a70-def2-4625-bf26-1151724dbcb4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='BIRDS',
    variable_level_2='PELICANS AND ALLIES',
    variable_level_3='',
    detailed_variable='',
    identifier='3a591a70-def2-4625-bf26-1151724dbcb4'
  ),
  'a463163e-8e86-4086-8b10-8fd6a95fca4a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='BIRDS',
    variable_level_2='PENGUINS',
    variable_level_3='',
    detailed_variable='',
    identifier='a463163e-8e86-4086-8b10-8fd6a95fca4a'
  ),
  'c8186508-c5dd-4282-86d2-b217643a87d8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='BIRDS',
    variable_level_2='PERCHING BIRDS',
    variable_level_3='',
    detailed_variable='',
    identifier='c8186508-c5dd-4282-86d2-b217643a87d8'
  ),
  '519a7291-55a2-44a4-8f01-ca7742ff69cc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='BIRDS',
    variable_level_2='SANDPIPERS',
    variable_level_3='',
    detailed_variable='',
    identifier='519a7291-55a2-44a4-8f01-ca7742ff69cc'
  ),
  'e8f25820-dd06-4d8d-9548-dcc30a871982': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='BIRDS',
    variable_level_2='WADERS/GULLS/AUKS AND ALLIES',
    variable_level_3='',
    detailed_variable='',
    identifier='e8f25820-dd06-4d8d-9548-dcc30a871982'
  ),
  'ea855d4c-f132-44f9-b31c-447e1101684d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='FISH',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ea855d4c-f132-44f9-b31c-447e1101684d'
  ),
  'e5404ad9-95bf-4851-9dbb-fecf7dc1e905': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='FISH',
    variable_level_2='LAMPREYS/HAGFISHES',
    variable_level_3='',
    detailed_variable='',
    identifier='e5404ad9-95bf-4851-9dbb-fecf7dc1e905'
  ),
  'c69cde73-7bd9-489b-a20b-bd23cfb82d92': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='FISH',
    variable_level_2='RAY-FINNED FISHES',
    variable_level_3='',
    detailed_variable='',
    identifier='c69cde73-7bd9-489b-a20b-bd23cfb82d92'
  ),
  '4309af7d-76c8-4856-8ed3-20693600228b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='FISH',
    variable_level_2='RAY-FINNED FISHES',
    variable_level_3='ANCHOVIES/HERRINGS',
    detailed_variable='',
    identifier='4309af7d-76c8-4856-8ed3-20693600228b'
  ),
  '89530978-556a-44fd-88fa-5d42ce9c8b91': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='FISH',
    variable_level_2='RAY-FINNED FISHES',
    variable_level_3='CATFISHES/MINNOWS',
    detailed_variable='',
    identifier='89530978-556a-44fd-88fa-5d42ce9c8b91'
  ),
  '340b843e-841e-40d9-97c2-d76cca66c65e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='FISH',
    variable_level_2='RAY-FINNED FISHES',
    variable_level_3='CODS/HADDOCKS',
    detailed_variable='',
    identifier='340b843e-841e-40d9-97c2-d76cca66c65e'
  ),
  '74b0e517-8cde-45ad-b6fa-0849d4355928': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='FISH',
    variable_level_2='RAY-FINNED FISHES',
    variable_level_3='FLOUNDERS',
    detailed_variable='',
    identifier='74b0e517-8cde-45ad-b6fa-0849d4355928'
  ),
  '4b9b8b32-93b4-4e8e-819d-55ee7f6a6480': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='FISH',
    variable_level_2='RAY-FINNED FISHES',
    variable_level_3='NEEDLEFISHES',
    detailed_variable='',
    identifier='4b9b8b32-93b4-4e8e-819d-55ee7f6a6480'
  ),
  '85b2ba83-b81c-45a6-98d1-07b36040fe45': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='FISH',
    variable_level_2='RAY-FINNED FISHES',
    variable_level_3='PERCH-LIKE FISHES',
    detailed_variable='',
    identifier='85b2ba83-b81c-45a6-98d1-07b36040fe45'
  ),
  '1369d226-2b9b-4f69-a197-6e24523b21e7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='FISH',
    variable_level_2='RAY-FINNED FISHES',
    variable_level_3='PERCH-LIKE FISHES',
    detailed_variable='TUNAS AND ALLIES',
    identifier='1369d226-2b9b-4f69-a197-6e24523b21e7'
  ),
  '3179b46a-ce20-453c-9ebf-18a070a91dec': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='FISH',
    variable_level_2='RAY-FINNED FISHES',
    variable_level_3='PUPFISHES',
    detailed_variable='',
    identifier='3179b46a-ce20-453c-9ebf-18a070a91dec'
  ),
  '0b8d5346-d10b-4e32-8179-7a51970c5e7f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='FISH',
    variable_level_2='RAY-FINNED FISHES',
    variable_level_3='SALMONS/TROUTS',
    detailed_variable='',
    identifier='0b8d5346-d10b-4e32-8179-7a51970c5e7f'
  ),
  'a0b5e8ee-e164-4713-97b7-a8e5100d5e9c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='FISH',
    variable_level_2='RAY-FINNED FISHES',
    variable_level_3='STICKLEBACKS',
    detailed_variable='',
    identifier='a0b5e8ee-e164-4713-97b7-a8e5100d5e9c'
  ),
  '01549b6d-91e1-40de-bd7c-5ed5ee59d14e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='FISH',
    variable_level_2='RAY-FINNED FISHES',
    variable_level_3='STURGEONS/PADDLEFISHES',
    detailed_variable='',
    identifier='01549b6d-91e1-40de-bd7c-5ed5ee59d14e'
  ),
  'ed019e00-9b0a-4bdc-89aa-606cc929bd9f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='FISH',
    variable_level_2='SHARKS/RAYS/CHIMAERAS',
    variable_level_3='',
    detailed_variable='',
    identifier='ed019e00-9b0a-4bdc-89aa-606cc929bd9f'
  ),
  'f5161094-3593-4bc1-85ea-c8c2ecab1d9a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='MAMMALS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f5161094-3593-4bc1-85ea-c8c2ecab1d9a'
  ),
  '9db9cb8c-7d18-4922-990c-b610d22356eb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='MAMMALS',
    variable_level_2='BATS',
    variable_level_3='',
    detailed_variable='',
    identifier='9db9cb8c-7d18-4922-990c-b610d22356eb'
  ),
  '7a00c50c-827c-4012-9afe-20972e6a00c6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='MAMMALS',
    variable_level_2='CARNIVORES',
    variable_level_3='',
    detailed_variable='',
    identifier='7a00c50c-827c-4012-9afe-20972e6a00c6'
  ),
  '6831004a-34d7-42f5-a903-6c84a5e7590f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='MAMMALS',
    variable_level_2='CARNIVORES',
    variable_level_3='BEARS',
    detailed_variable='',
    identifier='6831004a-34d7-42f5-a903-6c84a5e7590f'
  ),
  '8d01f599-3a98-44b9-889f-43df92386d12': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='MAMMALS',
    variable_level_2='CARNIVORES',
    variable_level_3='DOGS/FOXES/WOLVES',
    detailed_variable='',
    identifier='8d01f599-3a98-44b9-889f-43df92386d12'
  ),
  'd8973cd1-f3b4-4087-bf3b-25ac0732fb38': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='MAMMALS',
    variable_level_2='CARNIVORES',
    variable_level_3='MARTENS/WEASELS/WOLVERINES',
    detailed_variable='',
    identifier='d8973cd1-f3b4-4087-bf3b-25ac0732fb38'
  ),
  'eedb68d2-c487-4dc8-8292-c375e3e8b455': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='MAMMALS',
    variable_level_2='CARNIVORES',
    variable_level_3='OTTERS',
    detailed_variable='',
    identifier='eedb68d2-c487-4dc8-8292-c375e3e8b455'
  ),
  '35cf1beb-a654-4ceb-ab6b-7e505c2144e7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='MAMMALS',
    variable_level_2='CARNIVORES',
    variable_level_3='SEALS/SEA LIONS/WALRUSES',
    detailed_variable='',
    identifier='35cf1beb-a654-4ceb-ab6b-7e505c2144e7'
  ),
  '7f066677-c0f8-4bb1-91de-13954494a927': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='MAMMALS',
    variable_level_2='CETACEANS',
    variable_level_3='',
    detailed_variable='',
    identifier='7f066677-c0f8-4bb1-91de-13954494a927'
  ),
  '5e0ce993-df7c-46e2-9942-3a242df75705': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='MAMMALS',
    variable_level_2='CETACEANS',
    variable_level_3='BALEEN WHALES',
    detailed_variable='',
    identifier='5e0ce993-df7c-46e2-9942-3a242df75705'
  ),
  '1e2a4882-d1e9-4e1a-a2a5-efc449133bf5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='MAMMALS',
    variable_level_2='CETACEANS',
    variable_level_3='TOOTHED WHALES',
    detailed_variable='',
    identifier='1e2a4882-d1e9-4e1a-a2a5-efc449133bf5'
  ),
  'af5fb4da-260e-4e4e-a332-36dfd5084e5d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='MAMMALS',
    variable_level_2='DUGONGS/MANATEES',
    variable_level_3='',
    detailed_variable='',
    identifier='af5fb4da-260e-4e4e-a332-36dfd5084e5d'
  ),
  'de9598de-24cc-4f87-b3df-d9f3d4717d33': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='MAMMALS',
    variable_level_2='ELEPHANTS',
    variable_level_3='',
    detailed_variable='',
    identifier='de9598de-24cc-4f87-b3df-d9f3d4717d33'
  ),
  '7b0bc104-eed1-4bc1-b12b-3cf9add700da': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='MAMMALS',
    variable_level_2='EVEN-TOED UNGULATES',
    variable_level_3='',
    detailed_variable='',
    identifier='7b0bc104-eed1-4bc1-b12b-3cf9add700da'
  ),
  '33e2e026-e40b-4932-95a9-b2ca1a7aa407': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='MAMMALS',
    variable_level_2='EVEN-TOED UNGULATES',
    variable_level_3='CATTLE/SHEEP',
    detailed_variable='',
    identifier='33e2e026-e40b-4932-95a9-b2ca1a7aa407'
  ),
  '5557a4f3-8392-4df5-81a9-206c2a86da89': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='MAMMALS',
    variable_level_2='EVEN-TOED UNGULATES',
    variable_level_3='DEER/MOOSE',
    detailed_variable='',
    identifier='5557a4f3-8392-4df5-81a9-206c2a86da89'
  ),
  '108ece15-4588-4564-b803-dc1c17cf193e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='MAMMALS',
    variable_level_2='EVEN-TOED UNGULATES',
    variable_level_3='HOGS/PIGS',
    detailed_variable='',
    identifier='108ece15-4588-4564-b803-dc1c17cf193e'
  ),
  'fae29067-5d65-455a-a515-b1ac52881285': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='MAMMALS',
    variable_level_2='RODENTS',
    variable_level_3='',
    detailed_variable='',
    identifier='fae29067-5d65-455a-a515-b1ac52881285'
  ),
  '4cf1a3bd-20ce-42d7-95ac-9a4ece7be12c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='MARINE MAMMALS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4cf1a3bd-20ce-42d7-95ac-9a4ece7be12c'
  ),
  '557e8ffd-807d-41b7-9cd3-ccc3f2690cf5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='MARINE MAMMALS',
    variable_level_2='DOLPHINS',
    variable_level_3='',
    detailed_variable='',
    identifier='557e8ffd-807d-41b7-9cd3-ccc3f2690cf5'
  ),
  'a659fc9d-9d6a-4e47-b052-9270baa48dd4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='MARINE MAMMALS',
    variable_level_2='PINNIPED',
    variable_level_3='',
    detailed_variable='',
    identifier='a659fc9d-9d6a-4e47-b052-9270baa48dd4'
  ),
  '5d3725b6-743b-4dda-bb54-b64f201ec4d1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='REPTILES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5d3725b6-743b-4dda-bb54-b64f201ec4d1'
  ),
  '3f1803fa-3ada-4762-96e4-28966dfdcc83': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='REPTILES',
    variable_level_2='ALLIGATORS/CROCODILES',
    variable_level_3='',
    detailed_variable='',
    identifier='3f1803fa-3ada-4762-96e4-28966dfdcc83'
  ),
  '7dce336b-8596-45f0-bc76-f82b26e1405f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='REPTILES',
    variable_level_2='LIZARDS/SNAKES',
    variable_level_3='',
    detailed_variable='',
    identifier='7dce336b-8596-45f0-bc76-f82b26e1405f'
  ),
  '2037d286-6285-49df-aeb4-6e429b18d595': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='REPTILES',
    variable_level_2='TURTLES',
    variable_level_3='',
    detailed_variable='',
    identifier='2037d286-6285-49df-aeb4-6e429b18d595'
  ),
  'c9cb7c91-1b1d-42d8-b5f8-596e657138f9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='ANIMALS/VERTEBRATES',
    variable_level_1='REPTILES',
    variable_level_2='TURTLES',
    variable_level_3='SEA TURTLES',
    detailed_variable='',
    identifier='c9cb7c91-1b1d-42d8-b5f8-596e657138f9'
  ),
  '7437925f-7e10-4c96-af36-f3532ec24276': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='BACTERIA/ARCHAEA',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7437925f-7e10-4c96-af36-f3532ec24276'
  ),
  '166de4c9-89ad-4248-b771-512beb1705cf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='BACTERIA/ARCHAEA',
    variable_level_1='CYANOBACTERIA (BLUE-GREEN ALGAE)',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='166de4c9-89ad-4248-b771-512beb1705cf'
  ),
  '3546cb0a-27a2-4914-85cf-1774b5c4ed19': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='FUNGI',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3546cb0a-27a2-4914-85cf-1774b5c4ed19'
  ),
  'e85b6d64-a230-4c1d-99a5-c62be8af18c7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='FUNGI',
    variable_level_1='LICHENS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e85b6d64-a230-4c1d-99a5-c62be8af18c7'
  ),
  '14fa5360-320c-4d54-9bf6-9871a4b308d7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='FUNGI',
    variable_level_1='MUSHROOMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='14fa5360-320c-4d54-9bf6-9871a4b308d7'
  ),
  '05763c43-c6ed-4071-b868-2ea6c1335c12': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='FUNGI',
    variable_level_1='SLIME MOLDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='05763c43-c6ed-4071-b868-2ea6c1335c12'
  ),
  'ee2e5028-1963-4de1-a883-b9e546d682a4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='FUNGI',
    variable_level_1='YEASTS/TRUFFLES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ee2e5028-1963-4de1-a883-b9e546d682a4'
  ),
  '0b4081fa-5233-4484-bc82-706976defa0e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0b4081fa-5233-4484-bc82-706976defa0e'
  ),
  '5eda068f-97ea-474a-8a1b-b193f6901251': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='ANGIOSPERMS (FLOWERING PLANTS)',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5eda068f-97ea-474a-8a1b-b193f6901251'
  ),
  'f4211da2-9eaa-4bb3-86b1-c4595e9f2971': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='ANGIOSPERMS (FLOWERING PLANTS)',
    variable_level_2='DICOTS',
    variable_level_3='',
    detailed_variable='',
    identifier='f4211da2-9eaa-4bb3-86b1-c4595e9f2971'
  ),
  'b2957a8b-4c60-42aa-ac1c-56a88421702b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='ANGIOSPERMS (FLOWERING PLANTS)',
    variable_level_2='MONOCOTS',
    variable_level_3='',
    detailed_variable='',
    identifier='b2957a8b-4c60-42aa-ac1c-56a88421702b'
  ),
  'e36c5faa-c772-4bb0-aeca-b361e160ce9d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='ANGIOSPERMS (FLOWERING PLANTS)',
    variable_level_2='MONOCOTS',
    variable_level_3='SEAGRASS',
    detailed_variable='',
    identifier='e36c5faa-c772-4bb0-aeca-b361e160ce9d'
  ),
  '589875d3-4770-4fb3-871c-b37c7aff4b47': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='FERNS AND ALLIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='589875d3-4770-4fb3-871c-b37c7aff4b47'
  ),
  'c5ae3a71-d144-4b91-9cf0-e3cb27ce718f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='FERNS AND ALLIES',
    variable_level_2='CLUB MOSSES',
    variable_level_3='',
    detailed_variable='',
    identifier='c5ae3a71-d144-4b91-9cf0-e3cb27ce718f'
  ),
  '9818a5f0-bec9-47c0-b2ee-7e84c55466ed': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='FERNS AND ALLIES',
    variable_level_2='FERNS',
    variable_level_3='',
    detailed_variable='',
    identifier='9818a5f0-bec9-47c0-b2ee-7e84c55466ed'
  ),
  '5f5bbb69-57ea-4d8e-bd89-20478bc765d1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='FERNS AND ALLIES',
    variable_level_2='HORSETAILS',
    variable_level_3='',
    detailed_variable='',
    identifier='5f5bbb69-57ea-4d8e-bd89-20478bc765d1'
  ),
  'e76b3409-8be4-422b-8002-85bbfa846994': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='FERNS AND ALLIES',
    variable_level_2='WHISK FERNS',
    variable_level_3='',
    detailed_variable='',
    identifier='e76b3409-8be4-422b-8002-85bbfa846994'
  ),
  '566e22da-d72e-4663-89d2-ced5aea948ea': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='GYMNOSPERMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='566e22da-d72e-4663-89d2-ced5aea948ea'
  ),
  'b26769a1-f023-4ab1-bc21-78ef2a5fd185': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='GYMNOSPERMS',
    variable_level_2='CONIFERS',
    variable_level_3='',
    detailed_variable='',
    identifier='b26769a1-f023-4ab1-bc21-78ef2a5fd185'
  ),
  '2e7a8b01-ee3b-44e2-95ef-cf4603b05204': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='GYMNOSPERMS',
    variable_level_2='CYCADS',
    variable_level_3='',
    detailed_variable='',
    identifier='2e7a8b01-ee3b-44e2-95ef-cf4603b05204'
  ),
  'fdccf097-a2e1-4494-ad47-1c96a4d0d99a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='GYMNOSPERMS',
    variable_level_2='GINKGO',
    variable_level_3='',
    detailed_variable='',
    identifier='fdccf097-a2e1-4494-ad47-1c96a4d0d99a'
  ),
  'f0077bce-436c-432c-8d28-eb8d9cf2849b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='GYMNOSPERMS',
    variable_level_2='GNETOPS',
    variable_level_3='',
    detailed_variable='',
    identifier='f0077bce-436c-432c-8d28-eb8d9cf2849b'
  ),
  'e731c2a1-e4b0-42e9-bed9-bd911c9b496c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='MACROALGAE (SEAWEEDS)',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e731c2a1-e4b0-42e9-bed9-bd911c9b496c'
  ),
  '36e07e20-ce85-4418-83fd-6d718e55f370': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='MACROALGAE (SEAWEEDS)',
    variable_level_2='BROWN ALGAE',
    variable_level_3='',
    detailed_variable='',
    identifier='36e07e20-ce85-4418-83fd-6d718e55f370'
  ),
  '4fb63f34-f934-4a20-9d6e-ee57424f2391': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='MACROALGAE (SEAWEEDS)',
    variable_level_2='GREEN ALGAE',
    variable_level_3='',
    detailed_variable='',
    identifier='4fb63f34-f934-4a20-9d6e-ee57424f2391'
  ),
  '63015ca3-455b-4d91-b047-ff83a95d6bbe': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='MACROALGAE (SEAWEEDS)',
    variable_level_2='RED ALGAE',
    variable_level_3='',
    detailed_variable='',
    identifier='63015ca3-455b-4d91-b047-ff83a95d6bbe'
  ),
  'd3594523-ba0d-4275-b121-95039f905058': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='MICROALGAE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d3594523-ba0d-4275-b121-95039f905058'
  ),
  '502c9a41-ab95-4ae7-8e92-d1024b094f36': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='MICROALGAE',
    variable_level_2='CRYPTOMONADS',
    variable_level_3='',
    detailed_variable='',
    identifier='502c9a41-ab95-4ae7-8e92-d1024b094f36'
  ),
  'a14cfe48-9554-4e7c-9a2b-bf72834eafba': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='MICROALGAE',
    variable_level_2='DIATOMS',
    variable_level_3='',
    detailed_variable='',
    identifier='a14cfe48-9554-4e7c-9a2b-bf72834eafba'
  ),
  'b29cf79c-92a9-4160-aa8a-6917da79e298': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='MICROALGAE',
    variable_level_2='DINOFLAGELLATES',
    variable_level_3='',
    detailed_variable='',
    identifier='b29cf79c-92a9-4160-aa8a-6917da79e298'
  ),
  '0a454dc9-de56-4682-8688-36ffd547d42f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='MICROALGAE',
    variable_level_2='HAPTOPHYTES',
    variable_level_3='',
    detailed_variable='',
    identifier='0a454dc9-de56-4682-8688-36ffd547d42f'
  ),
  'ab7eb13f-5fcb-4afa-8819-c37d36feeec1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='MICROALGAE',
    variable_level_2='HAPTOPHYTES',
    variable_level_3='COCCOLITHOPHORES',
    detailed_variable='',
    identifier='ab7eb13f-5fcb-4afa-8819-c37d36feeec1'
  ),
  '934bd870-ffa8-41d8-8da9-214b73707168': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PLANTS',
    variable_level_1='MOSSES/HORNWORTS/LIVERWORTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='934bd870-ffa8-41d8-8da9-214b73707168'
  ),
  '6a2a2417-1a9c-4767-bffd-6b99f9747bab': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PROTISTS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6a2a2417-1a9c-4767-bffd-6b99f9747bab'
  ),
  '663a2ea2-e2bf-4209-ae9b-334c8222b106': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PROTISTS',
    variable_level_1='AMOEBOIDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='663a2ea2-e2bf-4209-ae9b-334c8222b106'
  ),
  '949f8a84-185a-42a0-89dc-48534b46f309': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PROTISTS',
    variable_level_1='AMOEBOIDS',
    variable_level_2='AMOEBAS',
    variable_level_3='',
    detailed_variable='',
    identifier='949f8a84-185a-42a0-89dc-48534b46f309'
  ),
  'd9750f06-3784-4058-941f-40289c8d9d8b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PROTISTS',
    variable_level_1='AMOEBOIDS',
    variable_level_2='FORAMINIFERS',
    variable_level_3='',
    detailed_variable='',
    identifier='d9750f06-3784-4058-941f-40289c8d9d8b'
  ),
  '9becd489-f8fb-4dbb-b920-6c8399100515': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PROTISTS',
    variable_level_1='AMOEBOIDS',
    variable_level_2='RADIOLARIANS',
    variable_level_3='',
    detailed_variable='',
    identifier='9becd489-f8fb-4dbb-b920-6c8399100515'
  ),
  '6f2a1cfb-13f4-444f-a6e6-2d8b29797253': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PROTISTS',
    variable_level_1='CILIATES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6f2a1cfb-13f4-444f-a6e6-2d8b29797253'
  ),
  'fdb04105-e8ba-4a83-9c35-ed3c931ccc9f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PROTISTS',
    variable_level_1='DIATOMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fdb04105-e8ba-4a83-9c35-ed3c931ccc9f'
  ),
  '2095acb5-14af-40fe-af22-e6af2e3528b5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PROTISTS',
    variable_level_1='FLAGELLATES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2095acb5-14af-40fe-af22-e6af2e3528b5'
  ),
  '802614f5-e178-4e5d-be64-a7e09ea736cb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PROTISTS',
    variable_level_1='FLAGELLATES',
    variable_level_2='CRYPTOMONADS',
    variable_level_3='',
    detailed_variable='',
    identifier='802614f5-e178-4e5d-be64-a7e09ea736cb'
  ),
  'a0176a92-3eff-4278-b8db-02148c990302': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PROTISTS',
    variable_level_1='FLAGELLATES',
    variable_level_2='DINOFLAGELLATES',
    variable_level_3='',
    detailed_variable='',
    identifier='a0176a92-3eff-4278-b8db-02148c990302'
  ),
  'dc7d2770-86a3-463c-a92b-c61516ffb32a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PROTISTS',
    variable_level_1='FLAGELLATES',
    variable_level_2='HAPTOPHYTES',
    variable_level_3='',
    detailed_variable='',
    identifier='dc7d2770-86a3-463c-a92b-c61516ffb32a'
  ),
  'e88cc54b-7a4b-4680-b441-4d10a4534cd9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PROTISTS',
    variable_level_1='FLAGELLATES',
    variable_level_2='HAPTOPHYTES',
    variable_level_3='COCCOLITHOPHORES',
    detailed_variable='',
    identifier='e88cc54b-7a4b-4680-b441-4d10a4534cd9'
  ),
  '81655dc5-83d3-4daf-81c8-dc1522e9906e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PROTISTS',
    variable_level_1='MACROALGAE (SEAWEEDS)',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='81655dc5-83d3-4daf-81c8-dc1522e9906e'
  ),
  'e2d18940-adf6-4bdd-ab4f-fe86e68278f4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PROTISTS',
    variable_level_1='MACROALGAE (SEAWEEDS)',
    variable_level_2='BROWN ALGAE',
    variable_level_3='',
    detailed_variable='',
    identifier='e2d18940-adf6-4bdd-ab4f-fe86e68278f4'
  ),
  '76557903-2ed7-4f0e-b8fc-df02798d724e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PROTISTS',
    variable_level_1='MACROALGAE (SEAWEEDS)',
    variable_level_2='GREEN ALGAE',
    variable_level_3='',
    detailed_variable='',
    identifier='76557903-2ed7-4f0e-b8fc-df02798d724e'
  ),
  'b9e718df-0a3a-46b6-a34f-4960e9449660': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PROTISTS',
    variable_level_1='MACROALGAE (SEAWEEDS)',
    variable_level_2='RED ALGAE',
    variable_level_3='',
    detailed_variable='',
    identifier='b9e718df-0a3a-46b6-a34f-4960e9449660'
  ),
  'a69dd814-e7c0-437f-ba2a-63500f68c9a3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PROTISTS',
    variable_level_1='PLANKTON',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a69dd814-e7c0-437f-ba2a-63500f68c9a3'
  ),
  '28dc7895-3365-4bab-9946-3b247f4137b0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PROTISTS',
    variable_level_1='PLANKTON',
    variable_level_2='PHYTOPLANKTON',
    variable_level_3='',
    detailed_variable='',
    identifier='28dc7895-3365-4bab-9946-3b247f4137b0'
  ),
  '98b35c6b-5d40-41d0-b29f-a6b159c03b78': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PROTISTS',
    variable_level_1='SLIME MOLDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='98b35c6b-5d40-41d0-b29f-a6b159c03b78'
  ),
  '32ffe87f-c0f0-4398-9a6a-755d7f87a5ff': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='PROTISTS',
    variable_level_1='SPOROZOANS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='32ffe87f-c0f0-4398-9a6a-755d7f87a5ff'
  ),
  '85510ccc-5dc9-44ff-871e-775e856714f8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOLOGICAL CLASSIFICATION',
    term='VIRUSES',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='85510ccc-5dc9-44ff-871e-775e856714f8'
  ),
  '91c64c46-d040-4daa-b26c-61952fdfaf50': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='91c64c46-d040-4daa-b26c-61952fdfaf50'
  ),
  '6bef0291-a9ca-4832-bbb4-80459dc1493f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6bef0291-a9ca-4832-bbb4-80459dc1493f'
  ),
  '8fb66b46-b998-4412-a541-d2acabdf484b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='COMMUNITY DYNAMICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8fb66b46-b998-4412-a541-d2acabdf484b'
  ),
  '4e366444-01ea-4517-9d93-56f55ddf41b7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='COMMUNITY DYNAMICS',
    variable_level_2='BIODIVERSITY FUNCTIONS',
    variable_level_3='',
    detailed_variable='',
    identifier='4e366444-01ea-4517-9d93-56f55ddf41b7'
  ),
  'f42c849c-7113-4c69-a01e-52ebc5e7b44d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='COMMUNITY DYNAMICS',
    variable_level_2='COMMUNITY STRUCTURE',
    variable_level_3='',
    detailed_variable='',
    identifier='f42c849c-7113-4c69-a01e-52ebc5e7b44d'
  ),
  'c09be13f-5dc2-4460-9055-1a7232aa41ae': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='COMMUNITY DYNAMICS',
    variable_level_2='GRAZING DYNAMICS / PLANT ECOLOGY',
    variable_level_3='',
    detailed_variable='',
    identifier='c09be13f-5dc2-4460-9055-1a7232aa41ae'
  ),
  'd3c5e3e3-97bf-4e74-9f8d-523dce5f9270': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='COMMUNITY DYNAMICS',
    variable_level_2='INDICATOR SPECIES',
    variable_level_3='',
    detailed_variable='',
    identifier='d3c5e3e3-97bf-4e74-9f8d-523dce5f9270'
  ),
  '7bfdbe8d-3945-4678-a90b-d2251f973955': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='COMMUNITY DYNAMICS',
    variable_level_2='INVASIVE SPECIES',
    variable_level_3='',
    detailed_variable='',
    identifier='7bfdbe8d-3945-4678-a90b-d2251f973955'
  ),
  'ad7abcce-b88e-46c7-be44-496d60c88f25': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='COMMUNITY DYNAMICS',
    variable_level_2='PLANT SUCCESSION',
    variable_level_3='',
    detailed_variable='',
    identifier='ad7abcce-b88e-46c7-be44-496d60c88f25'
  ),
  '1041545f-8a0a-4b33-9a97-b2f772f649f3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='COMMUNITY DYNAMICS',
    variable_level_2='SPECIES DISTRIBUTION',
    variable_level_3='',
    detailed_variable='',
    identifier='1041545f-8a0a-4b33-9a97-b2f772f649f3'
  ),
  '1a2a8cf8-6d7d-4ad6-b40c-4d9f7fed493f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='COMMUNITY DYNAMICS',
    variable_level_2='SPECIES DOMINANCE INDICES',
    variable_level_3='',
    detailed_variable='',
    identifier='1a2a8cf8-6d7d-4ad6-b40c-4d9f7fed493f'
  ),
  'b98b8823-3e95-4383-bbb0-414ee8832112': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='COMMUNITY DYNAMICS',
    variable_level_2='SPECIES RECRUITMENT',
    variable_level_3='',
    detailed_variable='',
    identifier='b98b8823-3e95-4383-bbb0-414ee8832112'
  ),
  '233a4d81-44f8-4b0e-8ad3-695f641729f8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOSYSTEM FUNCTIONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='233a4d81-44f8-4b0e-8ad3-695f641729f8'
  ),
  '9015e65f-bbae-4855-a4b6-1bfa601752bd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOSYSTEM FUNCTIONS',
    variable_level_2='BIOGEOCHEMICAL CYCLES',
    variable_level_3='',
    detailed_variable='',
    identifier='9015e65f-bbae-4855-a4b6-1bfa601752bd'
  ),
  'a0eb9268-0333-4442-9bc6-efbe338d9836': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOSYSTEM FUNCTIONS',
    variable_level_2='BIOMASS DYNAMICS',
    variable_level_3='',
    detailed_variable='',
    identifier='a0eb9268-0333-4442-9bc6-efbe338d9836'
  ),
  'e58872a8-6104-4ff8-bbca-4b00ba4b38e8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOSYSTEM FUNCTIONS',
    variable_level_2='CARBON SEQUESTRATION',
    variable_level_3='',
    detailed_variable='',
    identifier='e58872a8-6104-4ff8-bbca-4b00ba4b38e8'
  ),
  '7f8a1613-67b0-4d6a-a9ad-89097c27a052': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOSYSTEM FUNCTIONS',
    variable_level_2='CHEMOSYNTHESIS',
    variable_level_3='',
    detailed_variable='',
    identifier='7f8a1613-67b0-4d6a-a9ad-89097c27a052'
  ),
  'd6464d91-2373-456f-85a7-a5019bdb1076': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOSYSTEM FUNCTIONS',
    variable_level_2='CONSUMPTION RATES',
    variable_level_3='',
    detailed_variable='',
    identifier='d6464d91-2373-456f-85a7-a5019bdb1076'
  ),
  '560eac7e-d172-4a31-a659-a3e99d5f61ac': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOSYSTEM FUNCTIONS',
    variable_level_2='DECOMPOSITION',
    variable_level_3='',
    detailed_variable='',
    identifier='560eac7e-d172-4a31-a659-a3e99d5f61ac'
  ),
  '16e5beb3-e3ae-49a4-8fac-302fbbcdd39c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOSYSTEM FUNCTIONS',
    variable_level_2='EXCRETION RATES',
    variable_level_3='',
    detailed_variable='',
    identifier='16e5beb3-e3ae-49a4-8fac-302fbbcdd39c'
  ),
  '8b5ed8b8-c739-46cd-ba74-6cf560dc7986': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOSYSTEM FUNCTIONS',
    variable_level_2='FATTY ACID DESATURASE',
    variable_level_3='',
    detailed_variable='',
    identifier='8b5ed8b8-c739-46cd-ba74-6cf560dc7986'
  ),
  '4a55497b-8e07-431a-9af9-fece001f1dd7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOSYSTEM FUNCTIONS',
    variable_level_2='FOOD-WEB DYNAMICS',
    variable_level_3='',
    detailed_variable='',
    identifier='4a55497b-8e07-431a-9af9-fece001f1dd7'
  ),
  '7a33a978-8ef6-4313-b489-c06cfc6d9cec': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOSYSTEM FUNCTIONS',
    variable_level_2='NUTRIENT CYCLING',
    variable_level_3='',
    detailed_variable='',
    identifier='7a33a978-8ef6-4313-b489-c06cfc6d9cec'
  ),
  '5069167e-0b99-48af-9f81-8c765e112083': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOSYSTEM FUNCTIONS',
    variable_level_2='NUTRITIONAL CONSTRAINTS',
    variable_level_3='',
    detailed_variable='',
    identifier='5069167e-0b99-48af-9f81-8c765e112083'
  ),
  '5fb90409-f9b5-46bc-8a6a-7c42e250c7c3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOSYSTEM FUNCTIONS',
    variable_level_2='OXYGEN DEMAND',
    variable_level_3='',
    detailed_variable='',
    identifier='5fb90409-f9b5-46bc-8a6a-7c42e250c7c3'
  ),
  '07b53dde-6fea-4662-9d03-ccfd617ca710': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOSYSTEM FUNCTIONS',
    variable_level_2='PHOTOSYNTHESIS',
    variable_level_3='',
    detailed_variable='',
    identifier='07b53dde-6fea-4662-9d03-ccfd617ca710'
  ),
  '8e2c0e3a-4716-4211-9804-734a7a93adbe': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOSYSTEM FUNCTIONS',
    variable_level_2='POLYUNSATURATED FATTY ACID',
    variable_level_3='',
    detailed_variable='',
    identifier='8e2c0e3a-4716-4211-9804-734a7a93adbe'
  ),
  'ecd03762-df34-49b7-91f2-d8a51acd270e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOSYSTEM FUNCTIONS',
    variable_level_2='PRIMARY PRODUCTION',
    variable_level_3='',
    detailed_variable='',
    identifier='ecd03762-df34-49b7-91f2-d8a51acd270e'
  ),
  '29a64468-46a8-4dbc-955d-80b7b4cf9aaf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOSYSTEM FUNCTIONS',
    variable_level_2='RESPIRATION RATE',
    variable_level_3='',
    detailed_variable='',
    identifier='29a64468-46a8-4dbc-955d-80b7b4cf9aaf'
  ),
  '200e9b2d-0201-4f52-9a5e-6dc6c4668ec9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOSYSTEM FUNCTIONS',
    variable_level_2='SECONDARY PRODUCTION',
    variable_level_3='',
    detailed_variable='',
    identifier='200e9b2d-0201-4f52-9a5e-6dc6c4668ec9'
  ),
  'bd46a0bf-5c06-48af-a6c9-022417b1fffd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOSYSTEM FUNCTIONS',
    variable_level_2='TROPHIC DYNAMICS',
    variable_level_3='',
    detailed_variable='',
    identifier='bd46a0bf-5c06-48af-a6c9-022417b1fffd'
  ),
  'dd539b52-6de1-4b1b-a60c-fa5782f4d64b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOTOXICOLOGY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='dd539b52-6de1-4b1b-a60c-fa5782f4d64b'
  ),
  '8e89d525-161c-4e02-8ef8-4868e0cf8c57': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOTOXICOLOGY',
    variable_level_2='BIOAVAILABILITY',
    variable_level_3='',
    detailed_variable='',
    identifier='8e89d525-161c-4e02-8ef8-4868e0cf8c57'
  ),
  'a54dbc4f-c136-4648-9797-db00e62fe22b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOTOXICOLOGY',
    variable_level_2='SPECIES BIOACCUMULATION',
    variable_level_3='',
    detailed_variable='',
    identifier='a54dbc4f-c136-4648-9797-db00e62fe22b'
  ),
  '5518feb6-93a8-46fd-9e9a-25be3a832d6d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='ECOTOXICOLOGY',
    variable_level_2='TOXICITY LEVELS',
    variable_level_3='',
    detailed_variable='',
    identifier='5518feb6-93a8-46fd-9e9a-25be3a832d6d'
  ),
  '62c6d256-e6d4-4204-b7a8-e084dd52d30a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='FIRE ECOLOGY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='62c6d256-e6d4-4204-b7a8-e084dd52d30a'
  ),
  'a45abde1-4717-44d8-8c31-4db5b03d0758': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='FIRE ECOLOGY',
    variable_level_2='FIRE DISTURBANCE',
    variable_level_3='',
    detailed_variable='',
    identifier='a45abde1-4717-44d8-8c31-4db5b03d0758'
  ),
  '2a0a6319-80c4-49fd-8a40-553175aa8637': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='FIRE ECOLOGY',
    variable_level_2='FIRE DYNAMICS',
    variable_level_3='',
    detailed_variable='',
    identifier='2a0a6319-80c4-49fd-8a40-553175aa8637'
  ),
  '2bfd42f1-0453-4c33-a21e-74df3ad64813': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='FIRE ECOLOGY',
    variable_level_2='FIRE MODELS',
    variable_level_3='',
    detailed_variable='',
    identifier='2bfd42f1-0453-4c33-a21e-74df3ad64813'
  ),
  'e6f1ee58-fb71-42dd-b071-c1637da7e51f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='FIRE ECOLOGY',
    variable_level_2='FIRE OCCURRENCE',
    variable_level_3='',
    detailed_variable='',
    identifier='e6f1ee58-fb71-42dd-b071-c1637da7e51f'
  ),
  '53ea3f44-904c-42ad-8ac2-334d14b648c6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='FIRE ECOLOGY',
    variable_level_2='WILDLAND FUELS',
    variable_level_3='',
    detailed_variable='',
    identifier='53ea3f44-904c-42ad-8ac2-334d14b648c6'
  ),
  '58f39353-7e1c-4884-9501-376cd0377fbf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='58f39353-7e1c-4884-9501-376cd0377fbf'
  ),
  '5efc3bc4-6403-4e33-ba23-5418fbc026b1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='BIOLUMINESCENCE',
    variable_level_3='',
    detailed_variable='',
    identifier='5efc3bc4-6403-4e33-ba23-5418fbc026b1'
  ),
  'f75f9011-903e-4757-9fcf-fefac2599b59': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='DIURNAL MOVEMENTS',
    variable_level_3='',
    detailed_variable='',
    identifier='f75f9011-903e-4757-9fcf-fefac2599b59'
  ),
  'f930dcf2-ddb4-4242-9079-9c8d5ceeaa35': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='ENDANGERED SPECIES',
    variable_level_3='',
    detailed_variable='',
    identifier='f930dcf2-ddb4-4242-9079-9c8d5ceeaa35'
  ),
  'cf3d1728-7606-4561-a0dd-116b4dbec21f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='EVOLUTIONARY ADAPTATION',
    variable_level_3='',
    detailed_variable='',
    identifier='cf3d1728-7606-4561-a0dd-116b4dbec21f'
  ),
  'ddeb06af-5c36-428d-801e-e9f9a60ce429': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='EXOTIC SPECIES',
    variable_level_3='',
    detailed_variable='',
    identifier='ddeb06af-5c36-428d-801e-e9f9a60ce429'
  ),
  'f27f7bf4-53fd-41bb-8e7e-b771f48f3bcc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='EXTINCTION RATE',
    variable_level_3='',
    detailed_variable='',
    identifier='f27f7bf4-53fd-41bb-8e7e-b771f48f3bcc'
  ),
  'bcb43cdf-294e-463c-a114-a55bd54f0b48': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='GRAZING DYNAMICS/PLANT HERBIVORY',
    variable_level_3='',
    detailed_variable='',
    identifier='bcb43cdf-294e-463c-a114-a55bd54f0b48'
  ),
  'dfc20833-d79a-4976-91fd-db9f3efc7822': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='HIBERNATION',
    variable_level_3='',
    detailed_variable='',
    identifier='dfc20833-d79a-4976-91fd-db9f3efc7822'
  ),
  'cd9f44da-b3b4-4f9c-a21f-89b59a29b235': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='INDIGENOUS/NATIVE SPECIES',
    variable_level_3='',
    detailed_variable='',
    identifier='cd9f44da-b3b4-4f9c-a21f-89b59a29b235'
  ),
  'adf5f515-c7b4-4662-a144-580659957ce1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='MICROBIAL CHANGES',
    variable_level_3='',
    detailed_variable='',
    identifier='adf5f515-c7b4-4662-a144-580659957ce1'
  ),
  'a4ed794f-d7b6-4e53-b565-3b86fe584ba3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='MIGRATORY RATES/ROUTES',
    variable_level_3='',
    detailed_variable='',
    identifier='a4ed794f-d7b6-4e53-b565-3b86fe584ba3'
  ),
  '7f16bc53-9125-4b44-8cd0-edd7edf7217e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='MORPHOLOGICAL ADAPTATION',
    variable_level_3='',
    detailed_variable='',
    identifier='7f16bc53-9125-4b44-8cd0-edd7edf7217e'
  ),
  '87601d17-faca-42c2-a431-61cf67933095': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='MUTATION RATES',
    variable_level_3='',
    detailed_variable='',
    identifier='87601d17-faca-42c2-a431-61cf67933095'
  ),
  '003466f4-9ee7-4d3b-81ff-2013add292e2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='MUTUALISM',
    variable_level_3='',
    detailed_variable='',
    identifier='003466f4-9ee7-4d3b-81ff-2013add292e2'
  ),
  '80ae5fdc-c312-4fa1-bf7d-60346529976d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='NATURAL SELECTION',
    variable_level_3='',
    detailed_variable='',
    identifier='80ae5fdc-c312-4fa1-bf7d-60346529976d'
  ),
  '51f3e55c-b694-4028-86fe-604a52dc794f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='PARASITISM',
    variable_level_3='',
    detailed_variable='',
    identifier='51f3e55c-b694-4028-86fe-604a52dc794f'
  ),
  '45950ee6-adc2-4f39-96a7-c00bacd1ba9e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='POLLINATOR SPECIES',
    variable_level_3='',
    detailed_variable='',
    identifier='45950ee6-adc2-4f39-96a7-c00bacd1ba9e'
  ),
  'ad3a5f4f-4624-4a08-b875-6723c2615e90': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='POPULATION DYNAMICS',
    variable_level_3='',
    detailed_variable='',
    identifier='ad3a5f4f-4624-4a08-b875-6723c2615e90'
  ),
  'f173021d-afc4-4a8f-8432-30c0cf832e3b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='POST-BREEDING PERIODS',
    variable_level_3='',
    detailed_variable='',
    identifier='f173021d-afc4-4a8f-8432-30c0cf832e3b'
  ),
  '615e826e-a5da-4e94-b7df-ad3515c06135': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='RANGE CHANGES',
    variable_level_3='',
    detailed_variable='',
    identifier='615e826e-a5da-4e94-b7df-ad3515c06135'
  ),
  'abc96dce-cbae-43a4-b7c2-2ff02276b030': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='SCAVENGING',
    variable_level_3='',
    detailed_variable='',
    identifier='abc96dce-cbae-43a4-b7c2-2ff02276b030'
  ),
  '60bd0b0a-2d6f-4f3c-bf42-2c081ef48b72': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='SPECIES COMPETITION',
    variable_level_3='',
    detailed_variable='',
    identifier='60bd0b0a-2d6f-4f3c-bf42-2c081ef48b72'
  ),
  'c8be7a1c-0f83-4993-9d73-8c46e68cc705': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='SPECIES DISTRIBUTION',
    variable_level_3='',
    detailed_variable='',
    identifier='c8be7a1c-0f83-4993-9d73-8c46e68cc705'
  ),
  'fd06e0a2-f689-4b33-8a85-f38bf4966808': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='SPECIES LIFE HISTORY',
    variable_level_3='',
    detailed_variable='',
    identifier='fd06e0a2-f689-4b33-8a85-f38bf4966808'
  ),
  'b69d76ba-ad69-4418-8e5b-ebb659604dda': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='SPECIES PREDATION',
    variable_level_3='',
    detailed_variable='',
    identifier='b69d76ba-ad69-4418-8e5b-ebb659604dda'
  ),
  'fa68e752-f3a7-4361-a000-47c908545e49': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='SURVIVAL RATES',
    variable_level_3='',
    detailed_variable='',
    identifier='fa68e752-f3a7-4361-a000-47c908545e49'
  ),
  'e008a809-42eb-4694-aac2-db7b6027ee77': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='SYMBIOSIS',
    variable_level_3='',
    detailed_variable='',
    identifier='e008a809-42eb-4694-aac2-db7b6027ee77'
  ),
  '744c38f8-feeb-4e01-a909-33d75fefba82': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOLOGICAL DYNAMICS',
    variable_level_1='SPECIES/POPULATION INTERACTIONS',
    variable_level_2='USE/FEEDING HABITATS',
    variable_level_3='',
    detailed_variable='',
    identifier='744c38f8-feeb-4e01-a909-33d75fefba82'
  ),
  'f1a25060-330c-4f84-9633-ed59ae8c64bf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f1a25060-330c-4f84-9633-ed59ae8c64bf'
  ),
  'c4a619e9-88ba-4dc6-91a6-5f95284d6f80': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='ANTHROPOGENIC/HUMAN INFLUENCED ECOSYSTEMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c4a619e9-88ba-4dc6-91a6-5f95284d6f80'
  ),
  '38fb609b-2a10-4d4f-b2e8-7e51161ec974': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='ANTHROPOGENIC/HUMAN INFLUENCED ECOSYSTEMS',
    variable_level_2='AGRICULTURAL LANDS',
    variable_level_3='',
    detailed_variable='',
    identifier='38fb609b-2a10-4d4f-b2e8-7e51161ec974'
  ),
  '2c74f390-9d82-4903-98e0-bddf0d3247fb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='ANTHROPOGENIC/HUMAN INFLUENCED ECOSYSTEMS',
    variable_level_2='AGRICULTURAL LANDS',
    variable_level_3='CROPLAND',
    detailed_variable='',
    identifier='2c74f390-9d82-4903-98e0-bddf0d3247fb'
  ),
  '39fee18c-8572-4d72-a0ce-2a72942c4870': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='ANTHROPOGENIC/HUMAN INFLUENCED ECOSYSTEMS',
    variable_level_2='AGRICULTURAL LANDS',
    variable_level_3='FOREST PLANTATION',
    detailed_variable='',
    identifier='39fee18c-8572-4d72-a0ce-2a72942c4870'
  ),
  '46a26fc7-95f0-409e-8bfa-eb623b3a3f8d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='ANTHROPOGENIC/HUMAN INFLUENCED ECOSYSTEMS',
    variable_level_2='AGRICULTURAL LANDS',
    variable_level_3='PASTURE',
    detailed_variable='',
    identifier='46a26fc7-95f0-409e-8bfa-eb623b3a3f8d'
  ),
  '3c8b236c-de02-491b-a506-91ecdc324a1c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='ANTHROPOGENIC/HUMAN INFLUENCED ECOSYSTEMS',
    variable_level_2='AGRICULTURAL LANDS',
    variable_level_3='RANGELAND',
    detailed_variable='',
    identifier='3c8b236c-de02-491b-a506-91ecdc324a1c'
  ),
  '8f109871-e6ff-4cef-a5f8-5a3ad981923e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='ANTHROPOGENIC/HUMAN INFLUENCED ECOSYSTEMS',
    variable_level_2='RESOURCE DEVELOPMENT SITE',
    variable_level_3='',
    detailed_variable='',
    identifier='8f109871-e6ff-4cef-a5f8-5a3ad981923e'
  ),
  '7d8dcf2c-133f-47b2-9195-17dd263ec8a3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='ANTHROPOGENIC/HUMAN INFLUENCED ECOSYSTEMS',
    variable_level_2='RESOURCE DEVELOPMENT SITE',
    variable_level_3='MINING/DRILLING SITE',
    detailed_variable='',
    identifier='7d8dcf2c-133f-47b2-9195-17dd263ec8a3'
  ),
  '9ff1f885-108f-40cb-a054-4e076b8d648b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='ANTHROPOGENIC/HUMAN INFLUENCED ECOSYSTEMS',
    variable_level_2='RESOURCE DEVELOPMENT SITE',
    variable_level_3='SOLAR FARM',
    detailed_variable='',
    identifier='9ff1f885-108f-40cb-a054-4e076b8d648b'
  ),
  '39fa5f62-1c4e-4790-a768-1252c0b51c7b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='ANTHROPOGENIC/HUMAN INFLUENCED ECOSYSTEMS',
    variable_level_2='RESOURCE DEVELOPMENT SITE',
    variable_level_3='WATER IMPOUNDMENT',
    detailed_variable='',
    identifier='39fa5f62-1c4e-4790-a768-1252c0b51c7b'
  ),
  '0c603a5b-d5e9-4e87-a8dc-2af456678dba': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='ANTHROPOGENIC/HUMAN INFLUENCED ECOSYSTEMS',
    variable_level_2='RESOURCE DEVELOPMENT SITE',
    variable_level_3='WIND FARM',
    detailed_variable='',
    identifier='0c603a5b-d5e9-4e87-a8dc-2af456678dba'
  ),
  '3e59af3d-500b-4c66-a9a1-76db5cf4a00b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='ANTHROPOGENIC/HUMAN INFLUENCED ECOSYSTEMS',
    variable_level_2='URBAN LANDS',
    variable_level_3='',
    detailed_variable='',
    identifier='3e59af3d-500b-4c66-a9a1-76db5cf4a00b'
  ),
  'a0c33d15-b76c-4a0d-abb7-6919102b2977': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='ANTHROPOGENIC/HUMAN INFLUENCED ECOSYSTEMS',
    variable_level_2='URBAN LANDS',
    variable_level_3='CANAL',
    detailed_variable='',
    identifier='a0c33d15-b76c-4a0d-abb7-6919102b2977'
  ),
  '3bd03ca9-4a63-44f1-b368-36f2400776e6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='ANTHROPOGENIC/HUMAN INFLUENCED ECOSYSTEMS',
    variable_level_2='URBAN LANDS',
    variable_level_3='GARDEN',
    detailed_variable='',
    identifier='3bd03ca9-4a63-44f1-b368-36f2400776e6'
  ),
  '2b1f7993-2d54-40de-abc4-3909f619ad4e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='ANTHROPOGENIC/HUMAN INFLUENCED ECOSYSTEMS',
    variable_level_2='URBAN LANDS',
    variable_level_3='PARK',
    detailed_variable='',
    identifier='2b1f7993-2d54-40de-abc4-3909f619ad4e'
  ),
  'a9f2e036-f04f-46cc-a4e8-dfba30d9034c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='ANTHROPOGENIC/HUMAN INFLUENCED ECOSYSTEMS',
    variable_level_2='URBAN LANDS',
    variable_level_3='ROADSIDE',
    detailed_variable='',
    identifier='a9f2e036-f04f-46cc-a4e8-dfba30d9034c'
  ),
  'c6455081-132d-4661-bb5f-22edf2f90800': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='AQUATIC ECOSYSTEMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c6455081-132d-4661-bb5f-22edf2f90800'
  ),
  'ca8d77f2-9257-4298-9244-e81cd890f000': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='AQUATIC ECOSYSTEMS',
    variable_level_2='PLANKTON',
    variable_level_3='',
    detailed_variable='',
    identifier='ca8d77f2-9257-4298-9244-e81cd890f000'
  ),
  '235996b1-b1a8-4c20-bb1f-711fb1a0c952': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='AQUATIC ECOSYSTEMS',
    variable_level_2='PLANKTON',
    variable_level_3='PHYTOPLANKTON',
    detailed_variable='',
    identifier='235996b1-b1a8-4c20-bb1f-711fb1a0c952'
  ),
  '0399b52c-e3de-4dcc-9eb6-b1e3acf2cf1b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='AQUATIC ECOSYSTEMS',
    variable_level_2='PLANKTON',
    variable_level_3='ZOOPLANKTON',
    detailed_variable='',
    identifier='0399b52c-e3de-4dcc-9eb6-b1e3acf2cf1b'
  ),
  'b72c49a1-8276-4753-8c88-894bc7bbf60d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='AQUATIC ECOSYSTEMS',
    variable_level_2='WETLANDS',
    variable_level_3='',
    detailed_variable='',
    identifier='b72c49a1-8276-4753-8c88-894bc7bbf60d'
  ),
  '3e924e3a-eb5d-4f81-8981-1b9f622ddc82': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='AQUATIC ECOSYSTEMS',
    variable_level_2='WETLANDS',
    variable_level_3='ESTUARINE WETLANDS',
    detailed_variable='',
    identifier='3e924e3a-eb5d-4f81-8981-1b9f622ddc82'
  ),
  'dd22cc67-afd5-4b9e-8072-90651a191486': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='AQUATIC ECOSYSTEMS',
    variable_level_2='WETLANDS',
    variable_level_3='LACUSTRINE WETLANDS',
    detailed_variable='',
    identifier='dd22cc67-afd5-4b9e-8072-90651a191486'
  ),
  'bc320625-d9ba-41f5-9336-57e86fd878f3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='AQUATIC ECOSYSTEMS',
    variable_level_2='WETLANDS',
    variable_level_3='MARINE',
    detailed_variable='',
    identifier='bc320625-d9ba-41f5-9336-57e86fd878f3'
  ),
  '291a51b8-07e5-4a66-8140-d140d69843db': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='AQUATIC ECOSYSTEMS',
    variable_level_2='WETLANDS',
    variable_level_3='MARSHES',
    detailed_variable='',
    identifier='291a51b8-07e5-4a66-8140-d140d69843db'
  ),
  'd400ab07-bde9-40cc-b70a-63eda730eab2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='AQUATIC ECOSYSTEMS',
    variable_level_2='WETLANDS',
    variable_level_3='PALUSTRINE WETLANDS',
    detailed_variable='',
    identifier='d400ab07-bde9-40cc-b70a-63eda730eab2'
  ),
  'b70ef20c-7215-4a39-9479-dbff7c2fdca9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='AQUATIC ECOSYSTEMS',
    variable_level_2='WETLANDS',
    variable_level_3='PEATLANDS',
    detailed_variable='',
    identifier='b70ef20c-7215-4a39-9479-dbff7c2fdca9'
  ),
  '41446bdc-89f6-4d84-a2a4-005390757235': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='AQUATIC ECOSYSTEMS',
    variable_level_2='WETLANDS',
    variable_level_3='RIPARIAN WETLANDS',
    detailed_variable='',
    identifier='41446bdc-89f6-4d84-a2a4-005390757235'
  ),
  '6cec3b57-1a7f-404d-afde-4de045ef0dd2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='AQUATIC ECOSYSTEMS',
    variable_level_2='WETLANDS',
    variable_level_3='SWAMPS',
    detailed_variable='',
    identifier='6cec3b57-1a7f-404d-afde-4de045ef0dd2'
  ),
  'e72c39c5-5480-4602-bb37-216b5cc737dd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='AQUATIC ECOSYSTEMS',
    variable_level_2='WETLANDS',
    variable_level_3='VERNAL POOL',
    detailed_variable='',
    identifier='e72c39c5-5480-4602-bb37-216b5cc737dd'
  ),
  'ad73e951-fb5b-4a0b-b034-9469a8bfccaa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='FRESHWATER ECOSYSTEMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ad73e951-fb5b-4a0b-b034-9469a8bfccaa'
  ),
  '57a3a5a7-66b9-4a4a-82da-7b09d82c684a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='FRESHWATER ECOSYSTEMS',
    variable_level_2='LAKE/POND',
    variable_level_3='',
    detailed_variable='',
    identifier='57a3a5a7-66b9-4a4a-82da-7b09d82c684a'
  ),
  'b23b9a47-d2aa-4e67-84d6-5fe2527d6fb6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='FRESHWATER ECOSYSTEMS',
    variable_level_2='LAKE/POND',
    variable_level_3='MONTANE LAKE',
    detailed_variable='',
    identifier='b23b9a47-d2aa-4e67-84d6-5fe2527d6fb6'
  ),
  '06a2da0f-5234-4d29-905b-153d88657eb9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='FRESHWATER ECOSYSTEMS',
    variable_level_2='LAKE/POND',
    variable_level_3='SALINE LAKES',
    detailed_variable='',
    identifier='06a2da0f-5234-4d29-905b-153d88657eb9'
  ),
  '43d51c24-0523-4b65-919f-17618c7d72b4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='FRESHWATER ECOSYSTEMS',
    variable_level_2='RIVERS/STREAM',
    variable_level_3='',
    detailed_variable='',
    identifier='43d51c24-0523-4b65-919f-17618c7d72b4'
  ),
  '5f76c978-1c8a-496e-bc6a-78ff7656f014': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='FRESHWATER ECOSYSTEMS',
    variable_level_2='RIVERS/STREAM',
    variable_level_3='EPHEMERAL STREAM',
    detailed_variable='',
    identifier='5f76c978-1c8a-496e-bc6a-78ff7656f014'
  ),
  'de9222a5-c3bc-470d-86dc-8b426ce61b76': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='FRESHWATER ECOSYSTEMS',
    variable_level_2='RIVERS/STREAM',
    variable_level_3='HEADWATER STREAM',
    detailed_variable='',
    identifier='de9222a5-c3bc-470d-86dc-8b426ce61b76'
  ),
  '1b5d3b68-4f89-4772-b015-ce6f30cf0496': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='FRESHWATER ECOSYSTEMS',
    variable_level_2='RIVERS/STREAM',
    variable_level_3='INTERMITTENT STREAM',
    detailed_variable='',
    identifier='1b5d3b68-4f89-4772-b015-ce6f30cf0496'
  ),
  '0236a2e0-64d6-4763-bcd1-ea8bb3a117a1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='FRESHWATER ECOSYSTEMS',
    variable_level_2='RIVERS/STREAM',
    variable_level_3='PERENNIAL STREAM/RIVER',
    detailed_variable='',
    identifier='0236a2e0-64d6-4763-bcd1-ea8bb3a117a1'
  ),
  'bafaa203-0dc0-4167-a64a-d89ba16d8eb1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='FRESHWATER ECOSYSTEMS',
    variable_level_2='RIVERS/STREAM',
    variable_level_3='RIVER DELTA',
    detailed_variable='',
    identifier='bafaa203-0dc0-4167-a64a-d89ba16d8eb1'
  ),
  'f6350232-b1c7-458c-bc43-bda357ebb6db': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f6350232-b1c7-458c-bc43-bda357ebb6db'
  ),
  '1c286cb7-2668-4db3-a5ac-cb8b710bebc2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='ABYSSAL',
    variable_level_3='',
    detailed_variable='',
    identifier='1c286cb7-2668-4db3-a5ac-cb8b710bebc2'
  ),
  '290354cc-c670-4845-bb66-ef1974b1e2a2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='ABYSSAL',
    variable_level_3='COLD SEEP',
    detailed_variable='',
    identifier='290354cc-c670-4845-bb66-ef1974b1e2a2'
  ),
  'bee69b66-3921-4883-920f-6a0bd85b614f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='ABYSSAL',
    variable_level_3='HYDROTHERMAL VENT',
    detailed_variable='',
    identifier='bee69b66-3921-4883-920f-6a0bd85b614f'
  ),
  '09a78997-581b-4d1b-ae71-b2b3f96ef719': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='BENTHIC',
    variable_level_3='',
    detailed_variable='',
    identifier='09a78997-581b-4d1b-ae71-b2b3f96ef719'
  ),
  '47be68db-d10d-43e7-b150-61cfd3f06126': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='COASTAL',
    variable_level_3='',
    detailed_variable='',
    identifier='47be68db-d10d-43e7-b150-61cfd3f06126'
  ),
  'a61d1705-a6b7-4df3-9f8e-57e26029629c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='COASTAL',
    variable_level_3='BEACHES',
    detailed_variable='',
    identifier='a61d1705-a6b7-4df3-9f8e-57e26029629c'
  ),
  '8d38de3b-2d05-4ad2-a960-f47a66191319': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='COASTAL',
    variable_level_3='DUNES',
    detailed_variable='',
    identifier='8d38de3b-2d05-4ad2-a960-f47a66191319'
  ),
  'd609fc5c-8267-4e79-84ec-93629d52aba8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='COASTAL',
    variable_level_3='KELP FOREST',
    detailed_variable='',
    identifier='d609fc5c-8267-4e79-84ec-93629d52aba8'
  ),
  '879d286b-9ea6-4e4d-bdd1-56a4c7ca1531': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='COASTAL',
    variable_level_3='LAGOON',
    detailed_variable='',
    identifier='879d286b-9ea6-4e4d-bdd1-56a4c7ca1531'
  ),
  '7c666111-3297-474b-ba7b-c93db3a52cb0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='COASTAL',
    variable_level_3='MANGROVE SWAMP',
    detailed_variable='',
    identifier='7c666111-3297-474b-ba7b-c93db3a52cb0'
  ),
  '771b2919-ab55-4c71-8561-b4fb365da53f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='COASTAL',
    variable_level_3='MUDFLAT',
    detailed_variable='',
    identifier='771b2919-ab55-4c71-8561-b4fb365da53f'
  ),
  '80e51854-2f3f-447e-9786-6d2ccb0dd886': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='COASTAL',
    variable_level_3='ROCKY INTERTIDAL',
    detailed_variable='',
    identifier='80e51854-2f3f-447e-9786-6d2ccb0dd886'
  ),
  'fbe91a4f-4d27-4cfe-ba1b-69a62e359a3d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='COASTAL',
    variable_level_3='SALT MARSH',
    detailed_variable='',
    identifier='fbe91a4f-4d27-4cfe-ba1b-69a62e359a3d'
  ),
  '9d0e3045-943e-460c-8bef-1db6fbf76341': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='COASTAL',
    variable_level_3='SAV/SEA GRASS BED',
    detailed_variable='',
    identifier='9d0e3045-943e-460c-8bef-1db6fbf76341'
  ),
  'af953f41-ab6c-4569-9762-c46ad07118da': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='DEMERSAL',
    variable_level_3='',
    detailed_variable='',
    identifier='af953f41-ab6c-4569-9762-c46ad07118da'
  ),
  '5a1ebca4-057d-43b9-af6a-04f57b93f8bb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='ESTUARY',
    variable_level_3='',
    detailed_variable='',
    identifier='5a1ebca4-057d-43b9-af6a-04f57b93f8bb'
  ),
  '155e730b-4e22-4962-adc5-a4b92543a442': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='ESTUARY',
    variable_level_3='BRACKISH MARSH',
    detailed_variable='',
    identifier='155e730b-4e22-4962-adc5-a4b92543a442'
  ),
  '63cd8427-07bd-4a46-b725-ca65da4bf9b6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='ESTUARY',
    variable_level_3='MANGROVE SWAMP',
    detailed_variable='',
    identifier='63cd8427-07bd-4a46-b725-ca65da4bf9b6'
  ),
  '86987ad2-21d2-496b-9119-350b3fb17455': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='ESTUARY',
    variable_level_3='MUDFLAT',
    detailed_variable='',
    identifier='86987ad2-21d2-496b-9119-350b3fb17455'
  ),
  '5f6e1b08-caca-423b-80dc-7de3da7a2988': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='ESTUARY',
    variable_level_3='SAV/SEA GRASS BED',
    detailed_variable='',
    identifier='5f6e1b08-caca-423b-80dc-7de3da7a2988'
  ),
  '3d7ecc4f-e79e-40d1-8796-63059888bf5f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='PELAGIC',
    variable_level_3='',
    detailed_variable='',
    identifier='3d7ecc4f-e79e-40d1-8796-63059888bf5f'
  ),
  'eb958dfb-5e38-401f-8b42-5f1273c75a4a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='PELAGIC',
    variable_level_3='NERITIC ZONE',
    detailed_variable='',
    identifier='eb958dfb-5e38-401f-8b42-5f1273c75a4a'
  ),
  '02d78090-d0b5-490d-92a8-b593172ab232': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='PELAGIC',
    variable_level_3='OCEANIC ZONE',
    detailed_variable='',
    identifier='02d78090-d0b5-490d-92a8-b593172ab232'
  ),
  '367718c8-cc3b-4c94-a270-0a278afabb43': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='REEF',
    variable_level_3='',
    detailed_variable='',
    identifier='367718c8-cc3b-4c94-a270-0a278afabb43'
  ),
  'fa3bc02d-31a7-4456-b716-a8b8f8393c86': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='REEF',
    variable_level_3='CORAL REEF',
    detailed_variable='',
    identifier='fa3bc02d-31a7-4456-b716-a8b8f8393c86'
  ),
  '758c00c3-03a3-4cef-9248-ab392d789148': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='MARINE ECOSYSTEMS',
    variable_level_2='REEF',
    variable_level_3='OYSTER REEF',
    detailed_variable='',
    identifier='758c00c3-03a3-4cef-9248-ab392d789148'
  ),
  '9361962c-cfc7-4428-8843-b3502718c382': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9361962c-cfc7-4428-8843-b3502718c382'
  ),
  '76589134-8d93-4e45-8476-f04497181d14': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='ALPINE/TUNDRA',
    variable_level_3='',
    detailed_variable='',
    identifier='76589134-8d93-4e45-8476-f04497181d14'
  ),
  '944d9d09-4317-4e9a-9aa5-dc4282be406e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='ALPINE/TUNDRA',
    variable_level_3='ALPINE TUNDRA',
    detailed_variable='',
    identifier='944d9d09-4317-4e9a-9aa5-dc4282be406e'
  ),
  '46ecf46f-a710-4589-82b2-34aebf35c3c0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='ALPINE/TUNDRA',
    variable_level_3='ARCTIC TUNDRA',
    detailed_variable='',
    identifier='46ecf46f-a710-4589-82b2-34aebf35c3c0'
  ),
  '101950b9-00d3-4721-9af8-fa5d51b196c3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='ALPINE/TUNDRA',
    variable_level_3='SUBALPINE',
    detailed_variable='',
    identifier='101950b9-00d3-4721-9af8-fa5d51b196c3'
  ),
  '91f6a2e5-5862-46a9-ba6a-d76e06d9997c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='CAVE/SUBTERRANEAN',
    variable_level_3='',
    detailed_variable='',
    identifier='91f6a2e5-5862-46a9-ba6a-d76e06d9997c'
  ),
  '5d5426f6-e7ce-41c1-a3d3-b93adf748f0f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='DESERTS',
    variable_level_3='',
    detailed_variable='',
    identifier='5d5426f6-e7ce-41c1-a3d3-b93adf748f0f'
  ),
  '4f63746e-0e8b-4254-9d4a-a23a852f819f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='DESERTS',
    variable_level_3='DESERT SCRUB',
    detailed_variable='',
    identifier='4f63746e-0e8b-4254-9d4a-a23a852f819f'
  ),
  '46e4aaa4-349c-4049-a910-035391360010': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='FORESTS',
    variable_level_3='',
    detailed_variable='',
    identifier='46e4aaa4-349c-4049-a910-035391360010'
  ),
  'cafa8131-4a2d-4c8b-811c-0d64adf5fc06': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='FORESTS',
    variable_level_3='BOREAL FOREST/TIAGA',
    detailed_variable='',
    identifier='cafa8131-4a2d-4c8b-811c-0d64adf5fc06'
  ),
  '5d8236b5-bf5b-499f-a8e7-0cd80e00d261': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='FORESTS',
    variable_level_3='TEMPERATE CONIFEROUS FOREST',
    detailed_variable='',
    identifier='5d8236b5-bf5b-499f-a8e7-0cd80e00d261'
  ),
  'a59dc6dc-5348-4e8b-aec2-20cdeb38b617': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='FORESTS',
    variable_level_3='TEMPERATE DECIDUOUS FOREST',
    detailed_variable='',
    identifier='a59dc6dc-5348-4e8b-aec2-20cdeb38b617'
  ),
  '9cde47e7-325b-465e-93a6-ae4d459c7945': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='FORESTS',
    variable_level_3='TEMPERATE MIXED FOREST',
    detailed_variable='',
    identifier='9cde47e7-325b-465e-93a6-ae4d459c7945'
  ),
  '96ea0bde-7cf6-4601-8a49-116636f556cf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='FORESTS',
    variable_level_3='TEMPERATE RAINFOREST',
    detailed_variable='',
    identifier='96ea0bde-7cf6-4601-8a49-116636f556cf'
  ),
  '89bb4e2b-dd39-44ed-a4d3-2b205e9fa68a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='FORESTS',
    variable_level_3='TROPICAL RAINFOREST',
    detailed_variable='',
    identifier='89bb4e2b-dd39-44ed-a4d3-2b205e9fa68a'
  ),
  '142ea0c1-b77f-44da-8c64-ac7ee13fd5f6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='GRASSLANDS',
    variable_level_3='',
    detailed_variable='',
    identifier='142ea0c1-b77f-44da-8c64-ac7ee13fd5f6'
  ),
  'ddb4ca0c-9b19-442d-8bcc-e664544d3fe9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='GRASSLANDS',
    variable_level_3='MONTANE GRASSLAND',
    detailed_variable='',
    identifier='ddb4ca0c-9b19-442d-8bcc-e664544d3fe9'
  ),
  'd58dab07-f57e-47a9-8dcf-02a3e17f3533': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='GRASSLANDS',
    variable_level_3='SAVANNA',
    detailed_variable='',
    identifier='d58dab07-f57e-47a9-8dcf-02a3e17f3533'
  ),
  'fa3c6df8-a1e1-41d5-9de1-49b92e1ea455': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='ISLANDS',
    variable_level_3='',
    detailed_variable='',
    identifier='fa3c6df8-a1e1-41d5-9de1-49b92e1ea455'
  ),
  'de702fdd-3702-4164-a396-08082b0558c0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='KARST LANDSCAPE',
    variable_level_3='',
    detailed_variable='',
    identifier='de702fdd-3702-4164-a396-08082b0558c0'
  ),
  '99e09719-f1f8-439e-be4c-759242612a84': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='MONTANE HABITATS',
    variable_level_3='',
    detailed_variable='',
    identifier='99e09719-f1f8-439e-be4c-759242612a84'
  ),
  'f8d55ee4-1efb-4d83-b07f-1029ab0fa9e1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='SAVANNAS',
    variable_level_3='',
    detailed_variable='',
    identifier='f8d55ee4-1efb-4d83-b07f-1029ab0fa9e1'
  ),
  'e018b139-7e05-4155-8e2e-8d5603b5fe47': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='SHRUBLAND/SCRUB',
    variable_level_3='',
    detailed_variable='',
    identifier='e018b139-7e05-4155-8e2e-8d5603b5fe47'
  ),
  '0cc6527e-d162-4951-9db7-a6afe5c631c0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='SHRUBLAND/SCRUB',
    variable_level_3='CHAPARRAL',
    detailed_variable='',
    identifier='0cc6527e-d162-4951-9db7-a6afe5c631c0'
  ),
  '9409e1f9-f3a9-46fa-aaf9-0e685ca2adcb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='SHRUBLAND/SCRUB',
    variable_level_3='MONTANE SHRUBLAND',
    detailed_variable='',
    identifier='9409e1f9-f3a9-46fa-aaf9-0e685ca2adcb'
  ),
  '7da95c01-4b39-437e-a8d4-fd572e43f693': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='WETLANDS',
    variable_level_3='',
    detailed_variable='',
    identifier='7da95c01-4b39-437e-a8d4-fd572e43f693'
  ),
  '0e1f3f95-58b5-4f10-b239-850c66ed55ff': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='WETLANDS',
    variable_level_3='ESTUARINE WETLANDS',
    detailed_variable='',
    identifier='0e1f3f95-58b5-4f10-b239-850c66ed55ff'
  ),
  '686e66f7-27bf-4b67-b034-e0fdf0e47c0c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='WETLANDS',
    variable_level_3='LACUSTRINE WETLANDS',
    detailed_variable='',
    identifier='686e66f7-27bf-4b67-b034-e0fdf0e47c0c'
  ),
  '8ef6f360-10d0-4dc5-8fcb-c532eb23fe5d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='WETLANDS',
    variable_level_3='MARINE',
    detailed_variable='',
    identifier='8ef6f360-10d0-4dc5-8fcb-c532eb23fe5d'
  ),
  '419877cb-0c17-44b0-9b3d-a2283887a7a6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='WETLANDS',
    variable_level_3='MARSHES',
    detailed_variable='',
    identifier='419877cb-0c17-44b0-9b3d-a2283887a7a6'
  ),
  '6862d4d4-51fe-4fde-80eb-60d3ef08e88e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='WETLANDS',
    variable_level_3='PALUSTRINE WETLANDS',
    detailed_variable='',
    identifier='6862d4d4-51fe-4fde-80eb-60d3ef08e88e'
  ),
  'f3b5489d-6723-40bf-bd55-68a0f2fc1874': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='WETLANDS',
    variable_level_3='PEATLANDS',
    detailed_variable='',
    identifier='f3b5489d-6723-40bf-bd55-68a0f2fc1874'
  ),
  '1af675ae-9a65-4d91-970e-a8b9fcce0232': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='WETLANDS',
    variable_level_3='RIPARIAN WETLANDS',
    detailed_variable='',
    identifier='1af675ae-9a65-4d91-970e-a8b9fcce0232'
  ),
  '8c05bcf2-d13b-44fd-b1a2-5ec797b2f851': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='ECOSYSTEMS',
    variable_level_1='TERRESTRIAL ECOSYSTEMS',
    variable_level_2='WETLANDS',
    variable_level_3='SWAMPS',
    detailed_variable='',
    identifier='8c05bcf2-d13b-44fd-b1a2-5ec797b2f851'
  ),
  '8eb84f36-f355-458b-889f-b37cfa120654': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8eb84f36-f355-458b-889f-b37cfa120654'
  ),
  '4bde697e-39b1-4a42-9803-ee775b132869': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='BUILDING BLOCKS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4bde697e-39b1-4a42-9803-ee775b132869'
  ),
  '66034667-3a75-401d-894e-57f604d56571': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='BUILDING BLOCKS',
    variable_level_2='DNA',
    variable_level_3='',
    detailed_variable='',
    identifier='66034667-3a75-401d-894e-57f604d56571'
  ),
  '1993d124-f05d-4aea-a08a-a599438f9251': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='BUILDING BLOCKS',
    variable_level_2='DNA',
    variable_level_3='CONTIG',
    detailed_variable='',
    identifier='1993d124-f05d-4aea-a08a-a599438f9251'
  ),
  '4c0b1743-dfeb-4147-b89f-99386a370d42': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='BUILDING BLOCKS',
    variable_level_2='DNA',
    variable_level_3='ENVIRONMENTAL DNA (eDNA)',
    detailed_variable='',
    identifier='4c0b1743-dfeb-4147-b89f-99386a370d42'
  ),
  'a4f1da9a-eda0-4240-90a0-074d58a713ba': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='BUILDING BLOCKS',
    variable_level_2='PROTEIN',
    variable_level_3='',
    detailed_variable='',
    identifier='a4f1da9a-eda0-4240-90a0-074d58a713ba'
  ),
  '236cec51-c9e5-4584-90c4-0dac11c603cb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='BUILDING BLOCKS',
    variable_level_2='RNA',
    variable_level_3='',
    detailed_variable='',
    identifier='236cec51-c9e5-4584-90c4-0dac11c603cb'
  ),
  '9c40d795-abe1-42d4-b5ff-727d4843b3a4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='BUILDING BLOCKS',
    variable_level_2='RNA',
    variable_level_3='ENVIRONMENTAL RNA (eRNA)',
    detailed_variable='',
    identifier='9c40d795-abe1-42d4-b5ff-727d4843b3a4'
  ),
  '7d3c1302-4d88-4964-b966-f318bf2bcf74': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='BUILDING BLOCKS',
    variable_level_2='RNA',
    variable_level_3='TRANSCRIPTOME',
    detailed_variable='',
    identifier='7d3c1302-4d88-4964-b966-f318bf2bcf74'
  ),
  '9b01ff92-8e7f-4852-9ad1-bb91d6528ffd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='EPIGENOME/EPIGENETICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9b01ff92-8e7f-4852-9ad1-bb91d6528ffd'
  ),
  '1eb01003-dc1c-4743-be8b-22aaefd23a2c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='GENOME',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1eb01003-dc1c-4743-be8b-22aaefd23a2c'
  ),
  'dc777dfd-f24d-48b4-a137-387a24974d42': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='GENOME',
    variable_level_2='AMPLICON',
    variable_level_3='',
    detailed_variable='',
    identifier='dc777dfd-f24d-48b4-a137-387a24974d42'
  ),
  '4e3dcb15-d2db-4fa5-9f5f-f2c7a8a7ae34': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='GENOME',
    variable_level_2='GENOMICS',
    variable_level_3='',
    detailed_variable='',
    identifier='4e3dcb15-d2db-4fa5-9f5f-f2c7a8a7ae34'
  ),
  'fc3d3100-2f18-4e78-af4f-ca1ab8fd96ba': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='GENOME',
    variable_level_2='GENOTYPING',
    variable_level_3='',
    detailed_variable='',
    identifier='fc3d3100-2f18-4e78-af4f-ca1ab8fd96ba'
  ),
  'f978bbb1-08fb-4cc5-bb49-d8da0ee07bc4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='METABOLOMICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f978bbb1-08fb-4cc5-bb49-d8da0ee07bc4'
  ),
  '1a6ed5fe-608e-46a0-ab9e-1306de875ebe': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='METAGENOME',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1a6ed5fe-608e-46a0-ab9e-1306de875ebe'
  ),
  '02e08d51-9b6b-4412-9119-0a70207f8e87': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='METAGENOME',
    variable_level_2='AMPLICON SEQUENCE VARIANT (ASV)',
    variable_level_3='',
    detailed_variable='',
    identifier='02e08d51-9b6b-4412-9119-0a70207f8e87'
  ),
  '58631e49-648e-41ba-9409-db35dcc5e96c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='METAGENOME',
    variable_level_2='METAGENOME ASSEMBLED GENOMES (MAGs)',
    variable_level_3='',
    detailed_variable='',
    identifier='58631e49-648e-41ba-9409-db35dcc5e96c'
  ),
  '6aafe55d-2349-42b2-83b3-0dbdfdddd4ee': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='METAGENOME',
    variable_level_2='METAGENOMICS',
    variable_level_3='',
    detailed_variable='',
    identifier='6aafe55d-2349-42b2-83b3-0dbdfdddd4ee'
  ),
  '486922c4-afbb-4a85-8e5a-343fc4595ad3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='MICROBIOME',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='486922c4-afbb-4a85-8e5a-343fc4595ad3'
  ),
  'f8845bec-c628-4185-aac1-4af22481c024': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='MICROBIOME',
    variable_level_2='HOLOBIONT',
    variable_level_3='',
    detailed_variable='',
    identifier='f8845bec-c628-4185-aac1-4af22481c024'
  ),
  '929aa9c8-2001-4c7a-a5b8-e3eb7c4ff26f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='MITOGENOME',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='929aa9c8-2001-4c7a-a5b8-e3eb7c4ff26f'
  ),
  'd2f152e1-7398-44ad-b0a4-77ee6ef5bd6c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='OMICS METHODOLOGY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d2f152e1-7398-44ad-b0a4-77ee6ef5bd6c'
  ),
  'cb4d21ab-70a8-4111-be0a-e5508b256110': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='OMICS METHODOLOGY',
    variable_level_2='ABSOLUTE ABUNDANCE',
    variable_level_3='',
    detailed_variable='',
    identifier='cb4d21ab-70a8-4111-be0a-e5508b256110'
  ),
  '2af266cf-69f1-4326-b54f-d161bfb4c911': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='OMICS METHODOLOGY',
    variable_level_2='AMPLICON SEQUENCING',
    variable_level_3='',
    detailed_variable='',
    identifier='2af266cf-69f1-4326-b54f-d161bfb4c911'
  ),
  'fdbee1a3-10cd-4d07-ada0-dadbef9a5216': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='OMICS METHODOLOGY',
    variable_level_2='ASSEMBLY',
    variable_level_3='',
    detailed_variable='',
    identifier='fdbee1a3-10cd-4d07-ada0-dadbef9a5216'
  ),
  '3078eaeb-541e-44a0-9828-a1020f7bc6cf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='OMICS METHODOLOGY',
    variable_level_2='BIN/BINNING',
    variable_level_3='',
    detailed_variable='',
    identifier='3078eaeb-541e-44a0-9828-a1020f7bc6cf'
  ),
  '41f960a3-d695-4fa2-bfad-2d2887e1b063': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='OMICS METHODOLOGY',
    variable_level_2='BIOINFORMATICS',
    variable_level_3='',
    detailed_variable='',
    identifier='41f960a3-d695-4fa2-bfad-2d2887e1b063'
  ),
  'a272028a-5a84-4a52-b689-b45d5b6adfff': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='OMICS METHODOLOGY',
    variable_level_2='BLAST',
    variable_level_3='',
    detailed_variable='',
    identifier='a272028a-5a84-4a52-b689-b45d5b6adfff'
  ),
  'c39bf4db-70d3-4921-bd5e-e7c390a980bb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='OMICS METHODOLOGY',
    variable_level_2='DNA METABARCODING',
    variable_level_3='',
    detailed_variable='',
    identifier='c39bf4db-70d3-4921-bd5e-e7c390a980bb'
  ),
  'ad7ce68b-ee0a-4f2d-bbf7-fe3262186bfb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='OMICS METHODOLOGY',
    variable_level_2='DNA SEQUENCING',
    variable_level_3='',
    detailed_variable='',
    identifier='ad7ce68b-ee0a-4f2d-bbf7-fe3262186bfb'
  ),
  '20a8731e-4a5f-4f0a-ad96-3e28274cc90a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='OMICS METHODOLOGY',
    variable_level_2='DROPLET DIGITAL POLYMERASE CHAIN REACTION (ddPCR)',
    variable_level_3='',
    detailed_variable='',
    identifier='20a8731e-4a5f-4f0a-ad96-3e28274cc90a'
  ),
  '5804d1a5-c96d-4863-881d-6f763ecee262': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='OMICS METHODOLOGY',
    variable_level_2='HIGH THROUGHPUT SEQUENCING',
    variable_level_3='',
    detailed_variable='',
    identifier='5804d1a5-c96d-4863-881d-6f763ecee262'
  ),
  '83cdafe5-4bb9-4428-a9ad-80b7dcb01c42': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='OMICS METHODOLOGY',
    variable_level_2='MICROBIAL SOURCE TRACKING',
    variable_level_3='',
    detailed_variable='',
    identifier='83cdafe5-4bb9-4428-a9ad-80b7dcb01c42'
  ),
  'd1ef1ce2-7a50-446d-b744-73513700f351': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='OMICS METHODOLOGY',
    variable_level_2='OPERATIONAL TAXONOMIC UNIT (OTU)',
    variable_level_3='',
    detailed_variable='',
    identifier='d1ef1ce2-7a50-446d-b744-73513700f351'
  ),
  '7b989ab0-eaed-4698-87d8-f915c505b076': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='OMICS METHODOLOGY',
    variable_level_2='POLYMERASE CHAIN REACTION',
    variable_level_3='',
    detailed_variable='',
    identifier='7b989ab0-eaed-4698-87d8-f915c505b076'
  ),
  '692be87a-7812-44c7-be4b-7d067a3fa77d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='OMICS METHODOLOGY',
    variable_level_2='QUANTITATIVE POLYMERASE CHAIN REACTION (qPCR)',
    variable_level_3='',
    detailed_variable='',
    identifier='692be87a-7812-44c7-be4b-7d067a3fa77d'
  ),
  '9491a91d-44ee-4c2e-af49-96531091aa15': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='OMICS METHODOLOGY',
    variable_level_2='RAD-SEQ',
    variable_level_3='',
    detailed_variable='',
    identifier='9491a91d-44ee-4c2e-af49-96531091aa15'
  ),
  '3b9154a5-eea6-46ef-b17b-3ca6f37c3d3b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='OMICS METHODOLOGY',
    variable_level_2='RNA SEQUENCING',
    variable_level_3='',
    detailed_variable='',
    identifier='3b9154a5-eea6-46ef-b17b-3ca6f37c3d3b'
  ),
  '68c024c0-8308-42f1-baec-22165ede217c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='OMICS METHODOLOGY',
    variable_level_2='SANGER SEQUENCING',
    variable_level_3='',
    detailed_variable='',
    identifier='68c024c0-8308-42f1-baec-22165ede217c'
  ),
  '19291970-5f7d-47b0-9101-11b03a15d9ee': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='OMICS METHODOLOGY',
    variable_level_2='TRANSCRIPTOMIC ANALYSIS',
    variable_level_3='',
    detailed_variable='',
    identifier='19291970-5f7d-47b0-9101-11b03a15d9ee'
  ),
  '3f44b72b-eb44-4da9-9a13-a152700a043e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='OMICS METHODOLOGY',
    variable_level_2='WHOLE GENOME SEQUENCING',
    variable_level_3='',
    detailed_variable='',
    identifier='3f44b72b-eb44-4da9-9a13-a152700a043e'
  ),
  'dd6592c0-c95a-4e4f-8ae7-8bfae7de9a3e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='OMICS',
    variable_level_1='PROTEOMICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='dd6592c0-c95a-4e4f-8ae7-8bfae7de9a3e'
  ),
  'c7b5c02c-724d-4a19-b824-98180f3900c9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c7b5c02c-724d-4a19-b824-98180f3900c9'
  ),
  'a28eeef3-b252-4309-957b-860d2e0f97ef': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='AFFORESTATION/REFORESTATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a28eeef3-b252-4309-957b-860d2e0f97ef'
  ),
  '686feba9-87ba-474c-8280-7f67565cfb2f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='BIOMASS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='686feba9-87ba-474c-8280-7f67565cfb2f'
  ),
  'abbba948-9b77-4e19-a855-49a7fbc17696': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='CANOPY CHARACTERISTICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='abbba948-9b77-4e19-a855-49a7fbc17696'
  ),
  '42eb8baf-edbd-4791-8114-ca898ce5890f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='CANOPY CHARACTERISTICS',
    variable_level_2='VEGETATION HEIGHT',
    variable_level_3='',
    detailed_variable='',
    identifier='42eb8baf-edbd-4791-8114-ca898ce5890f'
  ),
  '2edf648a-6a71-44c3-9c1a-8fcdd2dcc61c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='CANOPY TRANSMITTANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2edf648a-6a71-44c3-9c1a-8fcdd2dcc61c'
  ),
  '6f6537f5-773f-4df1-862b-d9ab80eb5e04': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='CARBON',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6f6537f5-773f-4df1-862b-d9ab80eb5e04'
  ),
  '5e3999ec-d864-43fd-8d84-bd23630c405f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='CHLOROPHYLL',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5e3999ec-d864-43fd-8d84-bd23630c405f'
  ),
  'c59b0666-e20f-4134-847b-89719ed5621a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='CROWN',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c59b0666-e20f-4134-847b-89719ed5621a'
  ),
  'b7de16ed-c090-449b-81c1-44fe5b1195f0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='DECIDUOUS VEGETATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b7de16ed-c090-449b-81c1-44fe5b1195f0'
  ),
  'df597f06-8575-4726-acac-65b2bd432d59': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='DOMINANT SPECIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='df597f06-8575-4726-acac-65b2bd432d59'
  ),
  '16a7b4d6-e47f-4753-8803-f72edc4e1c5e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='EVERGREEN VEGETATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='16a7b4d6-e47f-4753-8803-f72edc4e1c5e'
  ),
  'f717330e-3656-4910-beed-d54cc9a19c2b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='EXOTIC VEGETATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f717330e-3656-4910-beed-d54cc9a19c2b'
  ),
  'a8d3f9a0-be0b-4690-86b9-ac64d951886a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='FOREST COMPOSITION/VEGETATION STRUCTURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a8d3f9a0-be0b-4690-86b9-ac64d951886a'
  ),
  '40766d01-bda1-420b-9fd1-fba6d6924f3f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='HERBIVORY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='40766d01-bda1-420b-9fd1-fba6d6924f3f'
  ),
  '536a5a5a-28bb-473a-aa95-6d2dd1e5098d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='IMPORTANCE VALUE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='536a5a5a-28bb-473a-aa95-6d2dd1e5098d'
  ),
  '0bfb8ae4-c08a-4d69-82d2-1b1b0d4acef6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='INDIGENOUS VEGETATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0bfb8ae4-c08a-4d69-82d2-1b1b0d4acef6'
  ),
  'bca1b724-3370-4a26-bcbc-3530ce4ddc97': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='LEAF CHARACTERISTICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bca1b724-3370-4a26-bcbc-3530ce4ddc97'
  ),
  'f829171e-8b22-4f93-8f71-7932dfd7a70b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='LEAF CHARACTERISTICS',
    variable_level_2='LEAF AREA INDEX (LAI)',
    variable_level_3='',
    detailed_variable='',
    identifier='f829171e-8b22-4f93-8f71-7932dfd7a70b'
  ),
  'afc54d28-de94-4674-9528-39f00bf74d6d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='LITTER CHARACTERISTICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='afc54d28-de94-4674-9528-39f00bf74d6d'
  ),
  'bf0ddf9c-39ba-4b2d-91ac-63021d644276': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='MACROPHYTES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bf0ddf9c-39ba-4b2d-91ac-63021d644276'
  ),
  'ed7c506e-b18e-4a93-ac03-4bdfe119b72f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='NITROGEN',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ed7c506e-b18e-4a93-ac03-4bdfe119b72f'
  ),
  '9bcb805c-718e-42c3-913d-174bdf06d4c1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='NUTRIENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9bcb805c-718e-42c3-913d-174bdf06d4c1'
  ),
  '47f4e7ac-b4ca-4ef9-824b-a36ea5510526': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='PHOSPHORUS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='47f4e7ac-b4ca-4ef9-824b-a36ea5510526'
  ),
  'db69ecb1-0738-4d82-943f-ae92093f500d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='PHOTOSYNTHETICALLY ACTIVE RADIATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='db69ecb1-0738-4d82-943f-ae92093f500d'
  ),
  '6079e5e4-4dee-4b32-aaa8-ae3231bcbadb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='PHOTOSYNTHETICALLY ACTIVE RADIATION',
    variable_level_2='FRACTION OF ABSORBED PHOTOSYNTHETICALLY ACTIVE RADIATION (FAPAR)',
    variable_level_3='',
    detailed_variable='',
    identifier='6079e5e4-4dee-4b32-aaa8-ae3231bcbadb'
  ),
  '3e801e91-897e-4528-8f4c-4ec527ad33cc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='PIGMENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3e801e91-897e-4528-8f4c-4ec527ad33cc'
  ),
  '0408bac9-c247-4b00-80de-f4665b813658': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='PLANT CHARACTERISTICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0408bac9-c247-4b00-80de-f4665b813658'
  ),
  '244a2eb6-bb6c-462c-978e-f9cfe96dffba': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='PLANT CHARACTERISTICS',
    variable_level_2='CROP DENSITY',
    variable_level_3='',
    detailed_variable='',
    identifier='244a2eb6-bb6c-462c-978e-f9cfe96dffba'
  ),
  'e95bcfcd-a7b6-4317-a678-fe713405f1c8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='PLANT CHARACTERISTICS',
    variable_level_2='CROP HEIGHT',
    variable_level_3='',
    detailed_variable='',
    identifier='e95bcfcd-a7b6-4317-a678-fe713405f1c8'
  ),
  'ff141ffe-05ea-4901-a243-e6186826b05c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='PLANT CHARACTERISTICS',
    variable_level_2='VEGETATION WATER CONTENT',
    variable_level_3='',
    detailed_variable='',
    identifier='ff141ffe-05ea-4901-a243-e6186826b05c'
  ),
  '3eb9aa2e-8314-4470-a7b5-8bfba901e1fe': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='PLANT CHARACTERISTICS',
    variable_level_2='XYLEM DIELECTRIC PERMITTIVITY',
    variable_level_3='',
    detailed_variable='',
    identifier='3eb9aa2e-8314-4470-a7b5-8bfba901e1fe'
  ),
  '3f45aadf-ec7c-43a1-a008-b24ca139837a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='PLANT PHENOLOGY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3f45aadf-ec7c-43a1-a008-b24ca139837a'
  ),
  '0f18f64b-dfac-4f93-9ded-89c8f249c2d1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='PLANT PHENOLOGY',
    variable_level_2='GROWTH STAGE',
    variable_level_3='',
    detailed_variable='',
    identifier='0f18f64b-dfac-4f93-9ded-89c8f249c2d1'
  ),
  'b0ad34ee-4b38-4a8d-a483-b3bfea66fa82': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='POLLEN',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b0ad34ee-4b38-4a8d-a483-b3bfea66fa82'
  ),
  '86dfb9ca-6587-4a91-b397-f220bb48a1eb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='RECLAMATION/REVEGETATION/RESTORATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='86dfb9ca-6587-4a91-b397-f220bb48a1eb'
  ),
  'fe6b37b9-f95a-491e-a58e-22aa66be9a9d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='REFORESTATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fe6b37b9-f95a-491e-a58e-22aa66be9a9d'
  ),
  '0a13efd5-d712-4b5d-abd4-9caf5914cfb6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='SOLAR INDUCED FLUORESCENCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0a13efd5-d712-4b5d-abd4-9caf5914cfb6'
  ),
  '0e06e528-e796-4b7c-9878-dbcb061d878d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='TREE RINGS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0e06e528-e796-4b7c-9878-dbcb061d878d'
  ),
  '5bdb3251-4811-439c-b172-9bbcd98e84b3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='VEGETATION COVER',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5bdb3251-4811-439c-b172-9bbcd98e84b3'
  ),
  'bf0f8e3a-d227-43d9-af73-ff36aeb079ba': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='VEGETATION DENSITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bf0f8e3a-d227-43d9-af73-ff36aeb079ba'
  ),
  '5ae1e11b-09c3-463d-9644-555a095a9333': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='VEGETATION FRACTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5ae1e11b-09c3-463d-9644-555a095a9333'
  ),
  'b7812c71-4b9e-4016-b4ba-dfcdb7e62365': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='VEGETATION INDEX',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b7812c71-4b9e-4016-b4ba-dfcdb7e62365'
  ),
  'b1d65d88-7bd0-491d-91ca-4102b89dc3e7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='VEGETATION INDEX',
    variable_level_2='LEAF AREA INDEX (LAI)',
    variable_level_3='',
    detailed_variable='',
    identifier='b1d65d88-7bd0-491d-91ca-4102b89dc3e7'
  ),
  '2297a00a-80f5-466e-b28e-b9ca42562d3f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='VEGETATION INDEX',
    variable_level_2='NORMALIZED DIFFERENCE VEGETATION INDEX (NDVI)',
    variable_level_3='',
    detailed_variable='',
    identifier='2297a00a-80f5-466e-b28e-b9ca42562d3f'
  ),
  'f094049b-fd89-4d01-b37f-7372b50c3309': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='VEGETATION INDEX',
    variable_level_2='PLANT AREA INDEX (PAI)',
    variable_level_3='',
    detailed_variable='',
    identifier='f094049b-fd89-4d01-b37f-7372b50c3309'
  ),
  'f97f5caf-206e-4310-8c9d-c4c1a09d3c62': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='VEGETATION OPTICAL DEPTH',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f97f5caf-206e-4310-8c9d-c4c1a09d3c62'
  ),
  'de0ace5c-fa2b-47ca-93db-79d8df7ab6f2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='VEGETATION SPECIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='de0ace5c-fa2b-47ca-93db-79d8df7ab6f2'
  ),
  '1f4e053f-c7a2-436d-b81c-8f87b36ca951': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='VEGETATION TEMPERATURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1f4e053f-c7a2-436d-b81c-8f87b36ca951'
  ),
  '7b4a6b86-6a74-4b01-b0ee-a9d7d60c72cb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='BIOSPHERE',
    term='VEGETATION',
    variable_level_1='VEGETATION WATER POTENTIAL',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7b4a6b86-6a74-4b01-b0ee-a9d7d60c72cb'
  ),
  '23703b6b-ee15-4512-b5b2-f441547e2edf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='23703b6b-ee15-4512-b5b2-f441547e2edf'
  ),
  '5273c8c2-d30b-4666-b2d5-0388ce2741d0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5273c8c2-d30b-4666-b2d5-0388ce2741d0'
  ),
  '8c4e2397-aed6-4ce4-9ead-08323e2f90ae': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='CLOUD INDICATORS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8c4e2397-aed6-4ce4-9ead-08323e2f90ae'
  ),
  '2111d240-315c-411b-8114-7ef9e89317e5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='CLOUD INDICATORS',
    variable_level_2='INCREASED/DECREASED CLOUD FRACTION',
    variable_level_3='',
    detailed_variable='',
    identifier='2111d240-315c-411b-8114-7ef9e89317e5'
  ),
  '83e9ddee-5887-4758-a3ba-5cb17a7d4ed5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='COMPOUND EXTREME EVENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='83e9ddee-5887-4758-a3ba-5cb17a7d4ed5'
  ),
  'b29b46ad-f05f-4144-b965-5f606ce96963': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='EXTREME WEATHER',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b29b46ad-f05f-4144-b965-5f606ce96963'
  ),
  'e4c806af-ab57-4fda-b7e9-29e3c65f6ec5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='EXTREME WEATHER',
    variable_level_2='EXTREME DROUGHT',
    variable_level_3='',
    detailed_variable='',
    identifier='e4c806af-ab57-4fda-b7e9-29e3c65f6ec5'
  ),
  'fc5a1b7a-5ee8-4d67-80f5-a57e3f1734ab': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='EXTREME WEATHER',
    variable_level_2='EXTREME PRECIPITATION',
    variable_level_3='',
    detailed_variable='',
    identifier='fc5a1b7a-5ee8-4d67-80f5-a57e3f1734ab'
  ),
  '079e6699-efbf-4358-9047-b668b459fc22': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='EXTREME WEATHER',
    variable_level_2='HEAT/COLD WAVE FREQUENCY/INTENSITY',
    variable_level_3='',
    detailed_variable='',
    identifier='079e6699-efbf-4358-9047-b668b459fc22'
  ),
  '7f95ceda-09fd-4ee3-9f30-bf38bf831e12': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='EXTREME WEATHER',
    variable_level_2='MONSOON ONSET/INTENSITY',
    variable_level_3='',
    detailed_variable='',
    identifier='7f95ceda-09fd-4ee3-9f30-bf38bf831e12'
  ),
  'a85b812e-e4d2-4dce-bf67-d89a3e1a9122': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='EXTREME WEATHER',
    variable_level_2='TROPICAL OR EXTRATROPICAL CYCLONE FREQUENCY/INTENSITY',
    variable_level_3='',
    detailed_variable='',
    identifier='a85b812e-e4d2-4dce-bf67-d89a3e1a9122'
  ),
  '12dc1f4f-2116-4b74-a1bd-bc61e8e57a5b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='FRESH WATER RIVER DISCHARGE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='12dc1f4f-2116-4b74-a1bd-bc61e8e57a5b'
  ),
  'b881cf8f-7260-4980-80bc-4b6ae3716c39': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='HUMIDITY INDICES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b881cf8f-7260-4980-80bc-4b6ae3716c39'
  ),
  'cdd7a31f-3244-494d-bc44-7b5f1ebb4bd7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='HUMIDITY INDICES',
    variable_level_2='HUMIDITY INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='cdd7a31f-3244-494d-bc44-7b5f1ebb4bd7'
  ),
  '5bdc74e2-ea3a-4d1d-b64e-9eaf3a879545': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='HUMIDITY INDICES',
    variable_level_2='TEMPERATURE-HUMIDITY INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='5bdc74e2-ea3a-4d1d-b64e-9eaf3a879545'
  ),
  'b9349099-8d45-4260-ab30-c891c3553a25': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='HUMIDITY INDICES',
    variable_level_2='WATER VAPOR TRANSPORT INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='b9349099-8d45-4260-ab30-c891c3553a25'
  ),
  'a219dbe6-c095-4002-9fbe-012b31da839c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='OCEAN ACIDIFICATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a219dbe6-c095-4002-9fbe-012b31da839c'
  ),
  'dbf8a0cf-1e9b-4bc4-95a2-819bb16af00c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='OCEAN OVERTURNING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='dbf8a0cf-1e9b-4bc4-95a2-819bb16af00c'
  ),
  '7d3e2368-75ba-43b9-bdce-bba2ff8d3e2c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='OCEAN UPWELLING INDICES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7d3e2368-75ba-43b9-bdce-bba2ff8d3e2c'
  ),
  '74ad118c-2f18-40fb-a26e-092390f52c20': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='OCEAN UPWELLING INDICES',
    variable_level_2='OCEAN COASTAL UPWELLING INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='74ad118c-2f18-40fb-a26e-092390f52c20'
  ),
  '873ed434-9407-4fd8-9660-41e50b0eb786': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='OCEAN UPWELLING/DOWNWELLING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='873ed434-9407-4fd8-9660-41e50b0eb786'
  ),
  '789939a6-3cd5-46f3-bdfd-5cdd6a012500': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='PRECIPITATION INDICATORS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='789939a6-3cd5-46f3-bdfd-5cdd6a012500'
  ),
  '279961c4-dac3-4188-917f-fa11982f957e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='PRECIPITATION INDICATORS',
    variable_level_2='PRECIPITATION TRENDS',
    variable_level_3='',
    detailed_variable='',
    identifier='279961c4-dac3-4188-917f-fa11982f957e'
  ),
  'c7c88080-660c-4913-8140-5f3bc91e295e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='PRECIPITATION INDICATORS',
    variable_level_2='PRECIPITATION VARIABILITY',
    variable_level_3='',
    detailed_variable='',
    identifier='c7c88080-660c-4913-8140-5f3bc91e295e'
  ),
  'e13b084e-d044-49c9-8791-f057f777fca3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='PRECIPITATION INDICATORS',
    variable_level_2='SAHEL STANDARDIZED RAINFALL',
    variable_level_3='',
    detailed_variable='',
    identifier='e13b084e-d044-49c9-8791-f057f777fca3'
  ),
  '52347642-9786-4b59-be77-02e9f307118d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='PRECIPITATION INDICES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='52347642-9786-4b59-be77-02e9f307118d'
  ),
  'd14d762c-4117-438a-9093-a098a0d0e4e6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='PRECIPITATION INDICES',
    variable_level_2='ENSO PRECIPITATION INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='d14d762c-4117-438a-9093-a098a0d0e4e6'
  ),
  'b8f0571c-4c19-4025-936c-936e9ac72e21': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='PRECIPITATION INDICES',
    variable_level_2='NORTHEAST BRAZIL RAINFALL ANOMALY',
    variable_level_3='',
    detailed_variable='',
    identifier='b8f0571c-4c19-4025-936c-936e9ac72e21'
  ),
  '7427fb2d-43b5-478a-960d-2ff9aa398462': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='PRECIPITATION INDICES',
    variable_level_2='STANDARDIZED PRECIPITATION INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='7427fb2d-43b5-478a-960d-2ff9aa398462'
  ),
  'aefbd3c5-6594-455b-a99d-7397a694bf8e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='PRECIPITATION INDICES',
    variable_level_2='WEIGHTED ANOMALY STANDARDIZED PRECIPITATION INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='aefbd3c5-6594-455b-a99d-7397a694bf8e'
  ),
  '536a86bd-3dd1-4f4a-9b4a-222a12746db5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SEA LEVEL RISE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='536a86bd-3dd1-4f4a-9b4a-222a12746db5'
  ),
  'eec5b471-bcc5-4d9b-8274-f3990e79ed84': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SEA LEVEL RISE',
    variable_level_2='EROSION',
    variable_level_3='',
    detailed_variable='',
    identifier='eec5b471-bcc5-4d9b-8274-f3990e79ed84'
  ),
  '9db10fb2-0ceb-412e-9936-a286c579fa9f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SEA LEVEL RISE',
    variable_level_2='INUNDATION',
    variable_level_3='',
    detailed_variable='',
    identifier='9db10fb2-0ceb-412e-9936-a286c579fa9f'
  ),
  'b83895e9-bac8-49fe-bcf5-8fe4d8fcaa16': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SEA SURFACE TEMPERATURE INDICES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b83895e9-bac8-49fe-bcf5-8fe4d8fcaa16'
  ),
  'ca418285-d1f2-4348-82e4-7fc59f8b60c8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SEA SURFACE TEMPERATURE INDICES',
    variable_level_2='ATLANTIC TRIPOLE SST',
    variable_level_3='',
    detailed_variable='',
    identifier='ca418285-d1f2-4348-82e4-7fc59f8b60c8'
  ),
  '5f2273b8-be30-45d5-a5d7-9bd947779c2e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SEA SURFACE TEMPERATURE INDICES',
    variable_level_2='CARIBBEAN INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='5f2273b8-be30-45d5-a5d7-9bd947779c2e'
  ),
  'ad5bde75-1f54-4f7e-a958-3adaf9f40639': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SEA SURFACE TEMPERATURE INDICES',
    variable_level_2='CENTRAL TROPICAL PACIFIC SST',
    variable_level_3='',
    detailed_variable='',
    identifier='ad5bde75-1f54-4f7e-a958-3adaf9f40639'
  ),
  'f59ce66b-a76d-467c-bab1-6264f9f3bb70': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SEA SURFACE TEMPERATURE INDICES',
    variable_level_2='CENTRAL TROPICAL PACIFIC SST',
    variable_level_3='NINO 4 INDEX',
    detailed_variable='',
    identifier='f59ce66b-a76d-467c-bab1-6264f9f3bb70'
  ),
  '01b96758-13f3-4cea-8447-decae36b1bde': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SEA SURFACE TEMPERATURE INDICES',
    variable_level_2='EAST CENTRAL TROPICAL PACIFIC SST',
    variable_level_3='',
    detailed_variable='',
    identifier='01b96758-13f3-4cea-8447-decae36b1bde'
  ),
  'a084d58c-c4f6-40fa-a645-96d9bef021aa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SEA SURFACE TEMPERATURE INDICES',
    variable_level_2='EAST CENTRAL TROPICAL PACIFIC SST',
    variable_level_3='NINO 3.4 INDEX',
    detailed_variable='',
    identifier='a084d58c-c4f6-40fa-a645-96d9bef021aa'
  ),
  '4b862c68-9cd9-4fee-942a-7cec0e6b05c2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SEA SURFACE TEMPERATURE INDICES',
    variable_level_2='EXTREME EASTERN TROPICAL PACIFIC SST',
    variable_level_3='',
    detailed_variable='',
    identifier='4b862c68-9cd9-4fee-942a-7cec0e6b05c2'
  ),
  '1c2e9a42-39d1-4b38-b752-3982f2a36ef4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SEA SURFACE TEMPERATURE INDICES',
    variable_level_2='EXTREME EASTERN TROPICAL PACIFIC SST',
    variable_level_3='NINO 1+2 INDEX',
    detailed_variable='',
    identifier='1c2e9a42-39d1-4b38-b752-3982f2a36ef4'
  ),
  '9c98dcbd-1dc8-4e0a-8ad1-0d11e88360eb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SEA SURFACE TEMPERATURE INDICES',
    variable_level_2='KAPLAN SST INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='9c98dcbd-1dc8-4e0a-8ad1-0d11e88360eb'
  ),
  'c58dc7fb-65d5-4309-8abe-160e8e845382': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SEA SURFACE TEMPERATURE INDICES',
    variable_level_2='NINO 3 INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='c58dc7fb-65d5-4309-8abe-160e8e845382'
  ),
  'd52674c3-0c78-4f35-9675-c2a8b3869b16': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SEA SURFACE TEMPERATURE INDICES',
    variable_level_2='NORTH TROPICAL ATLANTIC INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='d52674c3-0c78-4f35-9675-c2a8b3869b16'
  ),
  '70ed535b-a591-411d-80ca-9eafe10b3be8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SEA SURFACE TEMPERATURE INDICES',
    variable_level_2='OCEANIC NINO INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='70ed535b-a591-411d-80ca-9eafe10b3be8'
  ),
  'db1000b8-3b19-46fa-9d79-379379d654ac': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SEA SURFACE TEMPERATURE INDICES',
    variable_level_2='PACIFIC WARM POOL',
    variable_level_3='',
    detailed_variable='',
    identifier='db1000b8-3b19-46fa-9d79-379379d654ac'
  ),
  '58d71334-7fb5-4e05-85fd-9d2485854abe': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SEA SURFACE TEMPERATURE INDICES',
    variable_level_2='TRANS-NINO INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='58d71334-7fb5-4e05-85fd-9d2485854abe'
  ),
  '2cde80e8-3eb1-40e7-9305-e765dc8df5e2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SEA SURFACE TEMPERATURE INDICES',
    variable_level_2='TROPICAL NORTH ATLANTIC INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='2cde80e8-3eb1-40e7-9305-e765dc8df5e2'
  ),
  'ed2f3a3f-c841-41cf-9394-3a3254d13fc2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SEA SURFACE TEMPERATURE INDICES',
    variable_level_2='TROPICAL PACIFIC SST EOF',
    variable_level_3='',
    detailed_variable='',
    identifier='ed2f3a3f-c841-41cf-9394-3a3254d13fc2'
  ),
  '887e3bcc-ffd4-4f10-a91c-849783aac709': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SEA SURFACE TEMPERATURE INDICES',
    variable_level_2='TROPICAL SOUTH ATLANTIC INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='887e3bcc-ffd4-4f10-a91c-849783aac709'
  ),
  '309d2897-c74b-4de6-96fc-751a6935d549': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SEA SURFACE TEMPERATURE INDICES',
    variable_level_2='WESTERN HEMISPHERE WARM POOL',
    variable_level_3='',
    detailed_variable='',
    identifier='309d2897-c74b-4de6-96fc-751a6935d549'
  ),
  '1d8525f0-0cfc-4d59-8677-da5c8038deb7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='SURFACE SALINITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1d8525f0-0cfc-4d59-8677-da5c8038deb7'
  ),
  'b887d3e5-4280-43d2-a34e-0f63ac086b6a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b887d3e5-4280-43d2-a34e-0f63ac086b6a'
  ),
  '511e6c26-8806-4d88-9763-a136a6957042': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='ANTARCTIC OSCILLATION',
    variable_level_3='',
    detailed_variable='',
    identifier='511e6c26-8806-4d88-9763-a136a6957042'
  ),
  '98e5a7e4-b946-474a-8214-c1b7b3e5f976': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='ARCTIC OSCILLATION',
    variable_level_3='',
    detailed_variable='',
    identifier='98e5a7e4-b946-474a-8214-c1b7b3e5f976'
  ),
  'f141c968-94d4-4c42-8877-bbe34bb84b26': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='ATLANTIC MERIDIONAL MODE',
    variable_level_3='',
    detailed_variable='',
    identifier='f141c968-94d4-4c42-8877-bbe34bb84b26'
  ),
  'dcdb6cf1-48a7-488e-aeb8-e6c0b36752d4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='ATLANTIC MULTIDECADAL OSCILLATION LONG VERSION',
    variable_level_3='',
    detailed_variable='',
    identifier='dcdb6cf1-48a7-488e-aeb8-e6c0b36752d4'
  ),
  'a69f9faf-f730-4eba-9e38-0f72b0544bbe': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='BIVARIATE ENSO TIMESERIES INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='a69f9faf-f730-4eba-9e38-0f72b0544bbe'
  ),
  '2aeb8e10-b7f8-429e-b9f6-968ece330741': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='BLOCKING INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='2aeb8e10-b7f8-429e-b9f6-968ece330741'
  ),
  '47fb8f57-2ddd-4289-b8a5-af7ffa0ee031': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='EAST ATLANTIC JET PATTERN',
    variable_level_3='',
    detailed_variable='',
    identifier='47fb8f57-2ddd-4289-b8a5-af7ffa0ee031'
  ),
  '64d4ff80-59bb-4565-8759-e5223939abfd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='EAST ATLANTIC PATTERN',
    variable_level_3='',
    detailed_variable='',
    identifier='64d4ff80-59bb-4565-8759-e5223939abfd'
  ),
  'c58e035f-87c6-4aa5-8729-5a9c6270e73b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='EASTERN ATLANTIC WESTERN RUSSIA PATTERN',
    variable_level_3='',
    detailed_variable='',
    identifier='c58e035f-87c6-4aa5-8729-5a9c6270e73b'
  ),
  '0384fecd-9303-47f3-84e3-f01f58013fc3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='EASTERN PACIFIC OSCILLATION',
    variable_level_3='',
    detailed_variable='',
    identifier='0384fecd-9303-47f3-84e3-f01f58013fc3'
  ),
  '095a05c0-6220-4abd-9c1b-c4504a092d7d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='EL NINO SOUTHERN OSCILLATION (ENSO)',
    variable_level_3='',
    detailed_variable='',
    identifier='095a05c0-6220-4abd-9c1b-c4504a092d7d'
  ),
  '21389c4a-0d32-484a-95b9-db319a18f6ca': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='EQUATORIAL PACIFIC MERIDIONAL WIND ANOMALY INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='21389c4a-0d32-484a-95b9-db319a18f6ca'
  ),
  '57a381ef-56f6-48af-8974-822f5859979d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='EQUATORIAL PACIFIC ZONAL WIND ANOMALY INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='57a381ef-56f6-48af-8974-822f5859979d'
  ),
  '523c148f-bb4f-47d0-b176-0949ed59288a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='GLOBALLY INTEGRATED ANGULAR MOMENTUM',
    variable_level_3='',
    detailed_variable='',
    identifier='523c148f-bb4f-47d0-b176-0949ed59288a'
  ),
  '25d4368e-3b66-40d5-bac1-2343b127fa32': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='MADDEN-JULIAN OSCILLATION',
    variable_level_3='',
    detailed_variable='',
    identifier='25d4368e-3b66-40d5-bac1-2343b127fa32'
  ),
  'caddaef6-1a60-490a-938e-9107885f286f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='MULTIVARIATE ENSO INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='caddaef6-1a60-490a-938e-9107885f286f'
  ),
  'c5e1c055-768e-4aa3-a0a1-3adfda8ecdca': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='NORTH ATLANTIC OSCILLATION',
    variable_level_3='',
    detailed_variable='',
    identifier='c5e1c055-768e-4aa3-a0a1-3adfda8ecdca'
  ),
  '2295728d-0ee0-4c6f-9bb4-261b4b22322e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='NORTH PACIFIC OSCILLATION',
    variable_level_3='',
    detailed_variable='',
    identifier='2295728d-0ee0-4c6f-9bb4-261b4b22322e'
  ),
  'c6abcc08-7d59-4852-8c1a-82f464900333': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='NORTH PACIFIC PATTERN',
    variable_level_3='',
    detailed_variable='',
    identifier='c6abcc08-7d59-4852-8c1a-82f464900333'
  ),
  '77b2422e-ce52-465f-8841-5d04ebe536dc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='NORTHERN OSCILLATION INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='77b2422e-ce52-465f-8841-5d04ebe536dc'
  ),
  '2de06b90-4abe-4c71-a537-978679bf8aea': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='PACIFIC DECADAL OSCILLATION',
    variable_level_3='',
    detailed_variable='',
    identifier='2de06b90-4abe-4c71-a537-978679bf8aea'
  ),
  '233903dd-daec-474f-ac2e-cdcad84a85b5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='Pacific Transition Index',
    variable_level_3='',
    detailed_variable='',
    identifier='233903dd-daec-474f-ac2e-cdcad84a85b5'
  ),
  '0e53e397-7836-45ec-bc62-0d54f9f176e5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='PACIFIC/NORTH AMERICAN (PNA) PATTERN',
    variable_level_3='',
    detailed_variable='',
    identifier='0e53e397-7836-45ec-bc62-0d54f9f176e5'
  ),
  'ea64fa04-2822-4cc5-9014-f18ce1a1ef23': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='QUASI-BIENNIAL OSCILLATION',
    variable_level_3='',
    detailed_variable='',
    identifier='ea64fa04-2822-4cc5-9014-f18ce1a1ef23'
  ),
  'eaa0bc43-e283-4bf1-ba20-ca32850a66ef': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='SOUTHERN OSCILLATION INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='eaa0bc43-e283-4bf1-ba20-ca32850a66ef'
  ),
  '83b711e1-3fb5-4ef3-bafb-783e8239a4b5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='TROPICAL/NORTHERN HEMISPHERE PATTERN',
    variable_level_3='',
    detailed_variable='',
    identifier='83b711e1-3fb5-4ef3-bafb-783e8239a4b5'
  ),
  'bdb6eafa-f4e1-4536-b513-4c787f829722': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TELECONNECTIONS',
    variable_level_2='WEST PACIFIC INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='bdb6eafa-f4e1-4536-b513-4c787f829722'
  ),
  '2dcffd8f-2b01-4c68-a4f2-c4940d2709a3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TEMPERATURE INDICATORS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2dcffd8f-2b01-4c68-a4f2-c4940d2709a3'
  ),
  '3d997f01-8987-4fb1-a32e-d88d51f0a2c4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TEMPERATURE INDICATORS',
    variable_level_2='HIGHER MAXIMUM DAYTIME TEMPERATURES',
    variable_level_3='',
    detailed_variable='',
    identifier='3d997f01-8987-4fb1-a32e-d88d51f0a2c4'
  ),
  '93741006-ff2a-4ec2-bbd4-ff55301fabe0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TEMPERATURE INDICATORS',
    variable_level_2='HIGHER MINIMUM NIGHTTIME TEMPERATURES',
    variable_level_3='',
    detailed_variable='',
    identifier='93741006-ff2a-4ec2-bbd4-ff55301fabe0'
  ),
  '0eb1af15-7bd4-40c6-b8a4-666cbb61ff8c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TEMPERATURE INDICATORS',
    variable_level_2='STRATOSPHERIC TEMPERATURE ANOMALIES',
    variable_level_3='',
    detailed_variable='',
    identifier='0eb1af15-7bd4-40c6-b8a4-666cbb61ff8c'
  ),
  'ae247e59-db82-45ac-a9de-a9773ae4db40': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TEMPERATURE INDICATORS',
    variable_level_2='TEMPERATURE TRENDS',
    variable_level_3='',
    detailed_variable='',
    identifier='ae247e59-db82-45ac-a9de-a9773ae4db40'
  ),
  '7013bdc9-519d-42b6-827c-4b8013fbb726': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TEMPERATURE INDICATORS',
    variable_level_2='TEMPERATURE VARIABILITY',
    variable_level_3='',
    detailed_variable='',
    identifier='7013bdc9-519d-42b6-827c-4b8013fbb726'
  ),
  'f0e47cca-fa6e-44d0-b900-43920a3d0b91': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TEMPERATURE INDICATORS',
    variable_level_2='TROPOSPHERIC TEMPERATURE ANOMALIES',
    variable_level_3='',
    detailed_variable='',
    identifier='f0e47cca-fa6e-44d0-b900-43920a3d0b91'
  ),
  'e8580cbb-701a-4ab1-a40f-5fae4ae1ea24': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TEMPERATURE INDICES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e8580cbb-701a-4ab1-a40f-5fae4ae1ea24'
  ),
  'c7fa79e4-67a1-45da-b393-a1b89d54a1a5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TEMPERATURE INDICES',
    variable_level_2='COMMON SENSE CLIMATE INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='c7fa79e4-67a1-45da-b393-a1b89d54a1a5'
  ),
  'db0d03d7-1d08-42fb-b212-0da35b88e656': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TEMPERATURE INDICES',
    variable_level_2='COOLING DEGREE DAYS',
    variable_level_3='',
    detailed_variable='',
    identifier='db0d03d7-1d08-42fb-b212-0da35b88e656'
  ),
  'bc6d73a9-4943-4a3e-9f9d-9406fa54b0bc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TEMPERATURE INDICES',
    variable_level_2='FREEZING INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='bc6d73a9-4943-4a3e-9f9d-9406fa54b0bc'
  ),
  '6d808909-ce04-4401-a883-aff4d723d025': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TEMPERATURE INDICES',
    variable_level_2='GROWING DEGREE DAYS',
    variable_level_3='',
    detailed_variable='',
    identifier='6d808909-ce04-4401-a883-aff4d723d025'
  ),
  'fe2bc223-e503-4ca1-924c-d3fd5876721c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TEMPERATURE INDICES',
    variable_level_2='HEATING DEGREE DAYS',
    variable_level_3='',
    detailed_variable='',
    identifier='fe2bc223-e503-4ca1-924c-d3fd5876721c'
  ),
  'f19ff7fb-fd8b-433c-88e2-afc5dd3ee7b2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TEMPERATURE INDICES',
    variable_level_2='RESIDENTIAL ENERGY DEMAND TEMPERATURE INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='f19ff7fb-fd8b-433c-88e2-afc5dd3ee7b2'
  ),
  'd73be111-7aae-4a96-89c2-7b64c064893c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TEMPERATURE INDICES',
    variable_level_2='TEMPERATURE CONCENTRATION INDEX (TCI)',
    variable_level_3='',
    detailed_variable='',
    identifier='d73be111-7aae-4a96-89c2-7b64c064893c'
  ),
  '1e540a87-ffd9-4277-b8f2-683a58145b87': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ATMOSPHERIC/OCEAN INDICATORS',
    variable_level_1='TEMPERATURE INDICES',
    variable_level_2='THAWING INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='1e540a87-ffd9-4277-b8f2-683a58145b87'
  ),
  '76943142-e5a9-4ecf-b496-050dd3d97101': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='BIOSPHERIC INDICATORS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='76943142-e5a9-4ecf-b496-050dd3d97101'
  ),
  '379dd4c3-04d7-4f76-9bb9-83d0b8e1a2aa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='BIOSPHERIC INDICATORS',
    variable_level_1='BIRTH RATE DECLINE/INCREASE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='379dd4c3-04d7-4f76-9bb9-83d0b8e1a2aa'
  ),
  '0a07badb-1382-4f63-8344-7ba063b05534': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='BIOSPHERIC INDICATORS',
    variable_level_1='BREEDING PRODUCTIVITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0a07badb-1382-4f63-8344-7ba063b05534'
  ),
  '9a40bc0e-aece-4b4a-a87d-4869c8b903f8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='BIOSPHERIC INDICATORS',
    variable_level_1='CANOPY TEMPERATURE VARIABILITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9a40bc0e-aece-4b4a-a87d-4869c8b903f8'
  ),
  '853c3456-5397-4e16-ba67-51e4f4205db5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='BIOSPHERIC INDICATORS',
    variable_level_1='HYPOXIC CONDITIONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='853c3456-5397-4e16-ba67-51e4f4205db5'
  ),
  '6448f172-1560-4ea7-8826-8ac85dc820f3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='BIOSPHERIC INDICATORS',
    variable_level_1='INDICATOR SPECIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6448f172-1560-4ea7-8826-8ac85dc820f3'
  ),
  '93c8b32d-ab89-43e1-b58c-f1823fa7d118': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='BIOSPHERIC INDICATORS',
    variable_level_1='INVASIVE SPECIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='93c8b32d-ab89-43e1-b58c-f1823fa7d118'
  ),
  'cc5ab64b-11d0-4196-b7b3-f9c61e5e3ac6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='BIOSPHERIC INDICATORS',
    variable_level_1='PHENOLOGICAL CHANGES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='cc5ab64b-11d0-4196-b7b3-f9c61e5e3ac6'
  ),
  'f73cf4ee-2ae0-47b0-a294-5f8a8f694215': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='BIOSPHERIC INDICATORS',
    variable_level_1='PHENOLOGICAL CHANGES',
    variable_level_2='ANIMAL PHENOLOGICAL CHANGES',
    variable_level_3='',
    detailed_variable='',
    identifier='f73cf4ee-2ae0-47b0-a294-5f8a8f694215'
  ),
  'c6f81edb-b683-4356-93e6-5852766a5ee8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='BIOSPHERIC INDICATORS',
    variable_level_1='PHENOLOGICAL CHANGES',
    variable_level_2='PLANT PHENOLOGICAL CHANGES',
    variable_level_3='',
    detailed_variable='',
    identifier='c6f81edb-b683-4356-93e6-5852766a5ee8'
  ),
  'f5c63c23-f819-46e8-bc97-1e894424c00c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='BIOSPHERIC INDICATORS',
    variable_level_1='RANGE CHANGES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f5c63c23-f819-46e8-bc97-1e894424c00c'
  ),
  '0de668aa-cc97-482d-a0eb-cddcb1a705b6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='BIOSPHERIC INDICATORS',
    variable_level_1='SPECIES MIGRATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0de668aa-cc97-482d-a0eb-cddcb1a705b6'
  ),
  '1d9f0eb8-7233-4969-b40c-979d601ebaa7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='BIOSPHERIC INDICATORS',
    variable_level_1='VECTOR SPECIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1d9f0eb8-7233-4969-b40c-979d601ebaa7'
  ),
  '53fb0557-9f7f-4504-b0e8-adf329146c52': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CARBON FLUX',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='53fb0557-9f7f-4504-b0e8-adf329146c52'
  ),
  'de62c07e-96c6-44fb-a3a1-cd2902305691': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CLIMATE FEEDBACKS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='de62c07e-96c6-44fb-a3a1-cd2902305691'
  ),
  'fc77777e-614f-41f1-9b97-d5324fa99105': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CLIMATE FEEDBACKS',
    variable_level_1='ATMOSPHERIC FEEDBACKS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fc77777e-614f-41f1-9b97-d5324fa99105'
  ),
  '6a6bed83-f95a-44e6-8ae0-1371b532abc3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CLIMATE FEEDBACKS',
    variable_level_1='COUPLED SYSTEM FEEDBACKS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6a6bed83-f95a-44e6-8ae0-1371b532abc3'
  ),
  '3da6855e-9be8-4a79-826e-4ce984ed49a5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CLIMATE FEEDBACKS',
    variable_level_1='CRYOSPHERIC FEEDBACKS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3da6855e-9be8-4a79-826e-4ce984ed49a5'
  ),
  '514c891b-60b8-4a6f-adb3-0366c75588e9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CLIMATE FEEDBACKS',
    variable_level_1='LAND SURFACE FEEDBACKS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='514c891b-60b8-4a6f-adb3-0366c75588e9'
  ),
  '80d337f4-8e90-456a-9a5b-33e5f5c907ce': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CLIMATE FEEDBACKS',
    variable_level_1='OCEANIC FEEDBACKS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='80d337f4-8e90-456a-9a5b-33e5f5c907ce'
  ),
  '76b8c21c-c221-4724-86ef-c07222cb152b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='76b8c21c-c221-4724-86ef-c07222cb152b'
  ),
  'bb8c48bc-a36e-4f7e-afda-3244b058bc9c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='AVALANCHE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bb8c48bc-a36e-4f7e-afda-3244b058bc9c'
  ),
  '49d638f4-bdfa-4a6e-b154-cce1717d307f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='CLIMATE WARMING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='49d638f4-bdfa-4a6e-b154-cce1717d307f'
  ),
  '9fec9f47-c45d-4f15-8be5-d71424f33647': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='FIRN LIMIT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9fec9f47-c45d-4f15-8be5-d71424f33647'
  ),
  '2d79af4f-d15f-40cc-b0bf-8f5c8eb1fce5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='GLACIAL MEASUREMENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2d79af4f-d15f-40cc-b0bf-8f5c8eb1fce5'
  ),
  '83bd640d-cd05-49a8-9ec7-aab60820b126': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='GLACIAL MEASUREMENTS',
    variable_level_2='GLACIER ELEVATION/ICE SHEET ELEVATION',
    variable_level_3='',
    detailed_variable='',
    identifier='83bd640d-cd05-49a8-9ec7-aab60820b126'
  ),
  '6a2b291c-f78b-48c3-869f-9a4c8a40bc6c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='GLACIAL MEASUREMENTS',
    variable_level_2='GLACIER ELEVATION/ICE SHEET ELEVATION',
    variable_level_3='Hypsometry',
    detailed_variable='',
    identifier='6a2b291c-f78b-48c3-869f-9a4c8a40bc6c'
  ),
  '613c1fba-8710-47fb-a8e1-e4cd50bb97e1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='GLACIAL MEASUREMENTS',
    variable_level_2='GLACIER FACIES',
    variable_level_3='',
    detailed_variable='',
    identifier='613c1fba-8710-47fb-a8e1-e4cd50bb97e1'
  ),
  '6095d796-68e0-4c7d-aa4f-f2e5bd8c4916': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='GLACIAL MEASUREMENTS',
    variable_level_2='GLACIER MASS BALANCE/ICE SHEET MASS BALANCE',
    variable_level_3='',
    detailed_variable='',
    identifier='6095d796-68e0-4c7d-aa4f-f2e5bd8c4916'
  ),
  '4c9afaf7-4aec-440d-8084-6a482de09e7a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='GLACIAL MEASUREMENTS',
    variable_level_2='GLACIER MOTION/ICE SHEET MOTION',
    variable_level_3='',
    detailed_variable='',
    identifier='4c9afaf7-4aec-440d-8084-6a482de09e7a'
  ),
  '6a8a6fdb-c431-4d32-8cea-5849e2ee1f33': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='GLACIAL MEASUREMENTS',
    variable_level_2='GLACIER/ICE SHEET THICKNESS',
    variable_level_3='',
    detailed_variable='',
    identifier='6a8a6fdb-c431-4d32-8cea-5849e2ee1f33'
  ),
  'c3e4d439-bbb0-48c9-89eb-57d3a330627a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='GLACIAL MEASUREMENTS',
    variable_level_2='GLACIER/ICE SHEET TOPOGRAPHY',
    variable_level_3='',
    detailed_variable='',
    identifier='c3e4d439-bbb0-48c9-89eb-57d3a330627a'
  ),
  'fadd59e2-e1d2-44f6-9e41-0589eb953198': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='ICE DEPTH/THICKNESS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fadd59e2-e1d2-44f6-9e41-0589eb953198'
  ),
  'f4c1a555-4758-47ce-baa6-536730333833': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='ICE EDGES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f4c1a555-4758-47ce-baa6-536730333833'
  ),
  '50f0cf56-c119-4ac1-9a88-8eb04fa666ad': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='ICE EXTENT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='50f0cf56-c119-4ac1-9a88-8eb04fa666ad'
  ),
  '4733ef2c-e512-451b-8079-78ff7278e35c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='ICE FLOES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4733ef2c-e512-451b-8079-78ff7278e35c'
  ),
  '0ff2a38d-00f6-459d-ac9a-9a983bda602e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='ICE GROWTH/MELT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0ff2a38d-00f6-459d-ac9a-9a983bda602e'
  ),
  'fde8a54a-8aaa-45fd-bb66-3105e4c57102': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='RIVER ICE DEPTH/EXTENT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fde8a54a-8aaa-45fd-bb66-3105e4c57102'
  ),
  'ee0fce70-2097-4f5b-853a-c34e6cbff929': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='SALINITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ee0fce70-2097-4f5b-853a-c34e6cbff929'
  ),
  '1c0ebf89-f115-4e0d-9942-8ff8289bd330': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='SEA ICE CONCENTRATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1c0ebf89-f115-4e0d-9942-8ff8289bd330'
  ),
  '5d5cf73b-f833-4f8d-84a1-8a3840f3b4af': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='SEA ICE ELEVATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5d5cf73b-f833-4f8d-84a1-8a3840f3b4af'
  ),
  '8ef6560e-c699-49b4-bcb3-6db68506ca22': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='SNOW COVER',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8ef6560e-c699-49b4-bcb3-6db68506ca22'
  ),
  '008708ac-65a4-481a-8e03-640376f42f56': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='SNOW DEPTH',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='008708ac-65a4-481a-8e03-640376f42f56'
  ),
  '29f386e9-84fb-4e5d-9733-20233c63b1be': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='SNOW ENERGY BALANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='29f386e9-84fb-4e5d-9733-20233c63b1be'
  ),
  'b1be402f-336c-4f1f-8542-9807264a09a7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='CRYOSPHERIC INDICATORS',
    variable_level_1='SNOW MELT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b1be402f-336c-4f1f-8542-9807264a09a7'
  ),
  '897b3d65-709c-4739-9ba6-85911295d843': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ENVIRONMENTAL VULNERABILITY INDEX (EVI)',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='897b3d65-709c-4739-9ba6-85911295d843'
  ),
  '47200796-7541-4659-acf4-32b5303bcc1f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ENVIRONMENTAL VULNERABILITY INDEX (EVI)',
    variable_level_1='FIJI INDEX',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='47200796-7541-4659-acf4-32b5303bcc1f'
  ),
  '2c1e046e-2feb-4cb7-a6dd-f3753db7b5f5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ENVIRONMENTAL VULNERABILITY INDEX (EVI)',
    variable_level_1='SAMOA INDEX',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2c1e046e-2feb-4cb7-a6dd-f3753db7b5f5'
  ),
  '585182e9-6e5b-4ad6-96fe-065ffd31f7e8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='ENVIRONMENTAL VULNERABILITY INDEX (EVI)',
    variable_level_1='TUVALU INDEX',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='585182e9-6e5b-4ad6-96fe-065ffd31f7e8'
  ),
  '112e71ec-c0a1-49a8-82d7-bcb317b45860': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='112e71ec-c0a1-49a8-82d7-bcb317b45860'
  ),
  '27f92e4f-93f0-4740-bf99-2c44c0c3c23d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='DROUGHT DURATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='27f92e4f-93f0-4740-bf99-2c44c0c3c23d'
  ),
  '1a95af83-fe1b-4cea-931d-953a4e5965e6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='DROUGHT FREQUENCY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1a95af83-fe1b-4cea-931d-953a4e5965e6'
  ),
  'f50672b3-13d8-4206-b6c9-a1f9891ea470': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='DROUGHT INDICES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f50672b3-13d8-4206-b6c9-a1f9891ea470'
  ),
  '7e26f9e3-4c20-453d-bbc6-1970eca1ffb8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='DROUGHT INDICES',
    variable_level_2='PALMER DROUGHT CROP MOISTURE INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='7e26f9e3-4c20-453d-bbc6-1970eca1ffb8'
  ),
  'a43850a1-7b00-4993-80ff-753c2b5c4015': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='DROUGHT INDICES',
    variable_level_2='PALMER DROUGHT SEVERITY INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='a43850a1-7b00-4993-80ff-753c2b5c4015'
  ),
  '0365f0af-7843-4ba3-af8c-82d032c14f7e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='DROUGHT INDICES',
    variable_level_2='PALMER HYDROLOGICAL DROUGHT INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='0365f0af-7843-4ba3-af8c-82d032c14f7e'
  ),
  '0565650b-dce1-4ae8-8a7a-7ce25ac198c3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='DROUGHT INDICES',
    variable_level_2='PALMER Z INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='0565650b-dce1-4ae8-8a7a-7ce25ac198c3'
  ),
  '5bb5322d-f3f8-4ca4-91c6-311769936f96': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='DROUGHT SEVERITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5bb5322d-f3f8-4ca4-91c6-311769936f96'
  ),
  'b51f738c-7061-4ced-b216-53734ce4cb43': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='EROSION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b51f738c-7061-4ced-b216-53734ce4cb43'
  ),
  '7c1977bc-dfe7-4761-9b30-f42ec986d360': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='FIRE WEATHER INDEX',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7c1977bc-dfe7-4761-9b30-f42ec986d360'
  ),
  '16329a9b-72ea-4b46-b507-2ce389c63f50': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='FOREST FIRE DANGER INDEX',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='16329a9b-72ea-4b46-b507-2ce389c63f50'
  ),
  '36bdce45-37df-4475-9eed-73469a594edb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='LANDSLIDES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='36bdce45-37df-4475-9eed-73469a594edb'
  ),
  'ed8797be-661a-48c9-a7fe-2600b6c7c067': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='LENGTH OF GROWING SEASON',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ed8797be-661a-48c9-a7fe-2600b6c7c067'
  ),
  'f5824b8f-c3e7-4e56-96e8-cf4b5adefbf8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='LENGTH OF GROWING SEASON',
    variable_level_2='CROP HARVEST DATES',
    variable_level_3='',
    detailed_variable='',
    identifier='f5824b8f-c3e7-4e56-96e8-cf4b5adefbf8'
  ),
  '89428f01-000f-4d6f-a62b-29dd66b7e8c9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='LENGTH OF GROWING SEASON',
    variable_level_2='FIRST BLOOM DATE',
    variable_level_3='',
    detailed_variable='',
    identifier='89428f01-000f-4d6f-a62b-29dd66b7e8c9'
  ),
  '19f7b6be-1347-4c5c-bb8f-4a251369831e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='LENGTH OF GROWING SEASON',
    variable_level_2='FIRST LEAF DATE',
    variable_level_3='',
    detailed_variable='',
    identifier='19f7b6be-1347-4c5c-bb8f-4a251369831e'
  ),
  '226d4804-1a09-4d9b-a5c1-346f52a2e709': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='LENGTH OF GROWING SEASON',
    variable_level_2='FREEZE/FROST DATE',
    variable_level_3='',
    detailed_variable='',
    identifier='226d4804-1a09-4d9b-a5c1-346f52a2e709'
  ),
  'efc141a6-7d8e-45d5-b335-2fc122c62d78': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='LENGTH OF GROWING SEASON',
    variable_level_2='FREEZE/FROST PROBABILITY',
    variable_level_3='',
    detailed_variable='',
    identifier='efc141a6-7d8e-45d5-b335-2fc122c62d78'
  ),
  '738185b7-54d6-41a2-b31f-b8a4ee1dabe7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='LENGTH OF GROWING SEASON',
    variable_level_2='LENGTH OF FREEZE FREE PERIOD',
    variable_level_3='',
    detailed_variable='',
    identifier='738185b7-54d6-41a2-b31f-b8a4ee1dabe7'
  ),
  '32e1b1ec-fa69-47b5-b0d6-d71948e3997a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='SATELLITE SOIL MOISTURE INDEX',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='32e1b1ec-fa69-47b5-b0d6-d71948e3997a'
  ),
  'cb21c5cb-cc49-4328-a72d-94ccca1fa888': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='SOIL EROSION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='cb21c5cb-cc49-4328-a72d-94ccca1fa888'
  ),
  '27dd85c2-3403-438d-8b0c-8d424df60468': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='SOIL MOISTURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='27dd85c2-3403-438d-8b0c-8d424df60468'
  ),
  'b29ee2f4-b2ce-4b19-b8e3-2d74d071549b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='SOIL TEMPERATURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b29ee2f4-b2ce-4b19-b8e3-2d74d071549b'
  ),
  'c7503ec5-4e63-446a-9390-72c8a638a0af': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='SURFACE MOISTURE INDEX',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c7503ec5-4e63-446a-9390-72c8a638a0af'
  ),
  'd11df264-e70d-456c-9223-07f34e80b352': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='TREE LINE SHIFT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d11df264-e70d-456c-9223-07f34e80b352'
  ),
  '8d1157c4-d36b-40db-aa82-3603716f9988': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='LAND SURFACE/AGRICULTURE INDICATORS',
    variable_level_1='VEGETATION COVER',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8d1157c4-d36b-40db-aa82-3603716f9988'
  ),
  'c9a5b3eb-7556-41a8-a2b8-c015db80e5b2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c9a5b3eb-7556-41a8-a2b8-c015db80e5b2'
  ),
  '1efdd374-40a1-4118-a1da-61c647017ec9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='ALUMINUM-26 ANALYSIS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1efdd374-40a1-4118-a1da-61c647017ec9'
  ),
  '4bd6aafb-9240-4006-ada3-4b6a0501b612': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='BERYLLIUM-10 ANALYSIS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4bd6aafb-9240-4006-ada3-4b6a0501b612'
  ),
  '5553fe9d-ab0a-4305-86a6-1f7f697e15e4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='BIOLOGICAL RECORDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5553fe9d-ab0a-4305-86a6-1f7f697e15e4'
  ),
  '625ac4b3-a126-4c98-a061-3a780b942280': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='BIOLOGICAL RECORDS',
    variable_level_2='BIOMARKER',
    variable_level_3='',
    detailed_variable='',
    identifier='625ac4b3-a126-4c98-a061-3a780b942280'
  ),
  'afeb9962-d3e8-4260-ab2b-e62e11099e31': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='BIOLOGICAL RECORDS',
    variable_level_2='CORAL DEPOSITS',
    variable_level_3='',
    detailed_variable='',
    identifier='afeb9962-d3e8-4260-ab2b-e62e11099e31'
  ),
  'cffe377f-d840-4bcf-9223-8379b72defe7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='BIOLOGICAL RECORDS',
    variable_level_2='FAUNA',
    variable_level_3='',
    detailed_variable='',
    identifier='cffe377f-d840-4bcf-9223-8379b72defe7'
  ),
  '14c78811-5296-4095-9c44-26362914e798': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='BIOLOGICAL RECORDS',
    variable_level_2='MACROFOSSILS',
    variable_level_3='',
    detailed_variable='',
    identifier='14c78811-5296-4095-9c44-26362914e798'
  ),
  '0aa423e0-bc21-4d74-894d-a0dfcf17fae5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='BIOLOGICAL RECORDS',
    variable_level_2='MICROFOSSILS',
    variable_level_3='',
    detailed_variable='',
    identifier='0aa423e0-bc21-4d74-894d-a0dfcf17fae5'
  ),
  'fbd867cf-f7e8-4dbc-9fd2-2ccc0728350f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='BIOLOGICAL RECORDS',
    variable_level_2='PALEOVEGETATION',
    variable_level_3='',
    detailed_variable='',
    identifier='fbd867cf-f7e8-4dbc-9fd2-2ccc0728350f'
  ),
  '59719c53-a2b7-4200-9b3c-dfa5d39607f7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='BIOLOGICAL RECORDS',
    variable_level_2='POLLEN',
    variable_level_3='',
    detailed_variable='',
    identifier='59719c53-a2b7-4200-9b3c-dfa5d39607f7'
  ),
  '7bc06198-5546-40f5-97ac-b7b5b5503cfc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='BIOLOGICAL RECORDS',
    variable_level_2='POPULATION ABUNDANCE',
    variable_level_3='',
    detailed_variable='',
    identifier='7bc06198-5546-40f5-97ac-b7b5b5503cfc'
  ),
  '6444fc67-8cad-41c0-9ded-e93f604ba8b0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='BIOLOGICAL RECORDS',
    variable_level_2='TREE RINGS',
    variable_level_3='',
    detailed_variable='',
    identifier='6444fc67-8cad-41c0-9ded-e93f604ba8b0'
  ),
  'bd1834b0-4f8f-4616-b330-6205bff567c2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='BIOLOGICAL RECORDS',
    variable_level_2='TREE RINGS',
    variable_level_3='ISOTOPIC ANALYSIS',
    detailed_variable='',
    identifier='bd1834b0-4f8f-4616-b330-6205bff567c2'
  ),
  '437f13e8-0b8a-44a2-a7fd-5b41a00299db': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='BIOLOGICAL RECORDS',
    variable_level_2='TREE RINGS',
    variable_level_3='ISOTOPIC ANALYSIS',
    detailed_variable='CARBON ISOTOPE',
    identifier='437f13e8-0b8a-44a2-a7fd-5b41a00299db'
  ),
  '89387757-3548-4fe0-a383-d8f935f07c71': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='BIOLOGICAL RECORDS',
    variable_level_2='TREE RINGS',
    variable_level_3='ISOTOPIC ANALYSIS',
    detailed_variable='HYDROGEN ISOTOPE',
    identifier='89387757-3548-4fe0-a383-d8f935f07c71'
  ),
  'e91ff41a-5cf5-460b-b765-c553ca2a4ae2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='BIOLOGICAL RECORDS',
    variable_level_2='TREE RINGS',
    variable_level_3='ISOTOPIC ANALYSIS',
    detailed_variable='OXYGEN ISOTOPE',
    identifier='e91ff41a-5cf5-460b-b765-c553ca2a4ae2'
  ),
  '08a4f002-f368-414d-b923-83dd498452d8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='ICE CORE RECORDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='08a4f002-f368-414d-b923-83dd498452d8'
  ),
  'b53939ae-1264-409d-8434-3bb3d22b2848': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='ICE CORE RECORDS',
    variable_level_2='CARBON DIOXIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='b53939ae-1264-409d-8434-3bb3d22b2848'
  ),
  'a2987914-ed66-4b7c-964d-8eccf0174e57': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='ICE CORE RECORDS',
    variable_level_2='ELECTRICAL PROPERTIES',
    variable_level_3='',
    detailed_variable='',
    identifier='a2987914-ed66-4b7c-964d-8eccf0174e57'
  ),
  '42664b0d-26c2-44ad-b0a9-673ed2902f00': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='ICE CORE RECORDS',
    variable_level_2='ICE CORE AIR BUBBLES',
    variable_level_3='',
    detailed_variable='',
    identifier='42664b0d-26c2-44ad-b0a9-673ed2902f00'
  ),
  '302d7079-299a-4269-bd7e-d95009c9b46e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='ICE CORE RECORDS',
    variable_level_2='IONS',
    variable_level_3='',
    detailed_variable='',
    identifier='302d7079-299a-4269-bd7e-d95009c9b46e'
  ),
  'a0358b3e-0926-4b17-8b32-c1b15a73cba5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='ICE CORE RECORDS',
    variable_level_2='ISOTOPES',
    variable_level_3='',
    detailed_variable='',
    identifier='a0358b3e-0926-4b17-8b32-c1b15a73cba5'
  ),
  '6a0fc2ec-d1cf-43b5-8e97-6ab96811c02b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='ICE CORE RECORDS',
    variable_level_2='ISOTOPES',
    variable_level_3='ARGON ISOTOPES',
    detailed_variable='',
    identifier='6a0fc2ec-d1cf-43b5-8e97-6ab96811c02b'
  ),
  'e1138bec-7087-45f4-82b0-2e4029063381': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='ICE CORE RECORDS',
    variable_level_2='ISOTOPES',
    variable_level_3='NITROGEN ISOTOPES',
    detailed_variable='',
    identifier='e1138bec-7087-45f4-82b0-2e4029063381'
  ),
  'a1362cee-634d-40f4-b47f-901b328895c3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='ICE CORE RECORDS',
    variable_level_2='ISOTOPES',
    variable_level_3='OXYGEN ISOTOPES',
    detailed_variable='',
    identifier='a1362cee-634d-40f4-b47f-901b328895c3'
  ),
  '0948a59e-cc72-4e5d-b97d-4ea0335b0906': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='ICE CORE RECORDS',
    variable_level_2='METHANE',
    variable_level_3='',
    detailed_variable='',
    identifier='0948a59e-cc72-4e5d-b97d-4ea0335b0906'
  ),
  'bc90bc40-2a21-4a6f-9fb9-bf3ae5845157': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='ICE CORE RECORDS',
    variable_level_2='NITROUS OXIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='bc90bc40-2a21-4a6f-9fb9-bf3ae5845157'
  ),
  '15fdef7c-7fb7-4a1d-a24b-01164a8ba11a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='ICE CORE RECORDS',
    variable_level_2='PARTICULATE MATTER',
    variable_level_3='',
    detailed_variable='',
    identifier='15fdef7c-7fb7-4a1d-a24b-01164a8ba11a'
  ),
  '84b443b5-91d3-42d2-a48b-d2e157a39d5b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='ICE CORE RECORDS',
    variable_level_2='PARTICULATE MATTER',
    variable_level_3='MICROPARTICLE CONCENTRATION',
    detailed_variable='',
    identifier='84b443b5-91d3-42d2-a48b-d2e157a39d5b'
  ),
  'c736e45d-63f2-428b-abae-48f79d007703': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='ICE CORE RECORDS',
    variable_level_2='VOLCANIC DEPOSITS',
    variable_level_3='',
    detailed_variable='',
    identifier='c736e45d-63f2-428b-abae-48f79d007703'
  ),
  '2bedb6b3-6e92-42e2-b382-60e2a6aab8e9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='LAND RECORDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2bedb6b3-6e92-42e2-b382-60e2a6aab8e9'
  ),
  'f1f84fc8-d242-4f97-bb7d-77b68631273e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='LAND RECORDS',
    variable_level_2='BOREHOLES',
    variable_level_3='',
    detailed_variable='',
    identifier='f1f84fc8-d242-4f97-bb7d-77b68631273e'
  ),
  '482453d7-ffa4-4ae9-8158-9fa73bcf39ef': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='LAND RECORDS',
    variable_level_2='CAVE DEPOSITS',
    variable_level_3='',
    detailed_variable='',
    identifier='482453d7-ffa4-4ae9-8158-9fa73bcf39ef'
  ),
  'd2dc2330-0433-43f2-9154-dc399d24406c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='LAND RECORDS',
    variable_level_2='FIRE HISTORY',
    variable_level_3='',
    detailed_variable='',
    identifier='d2dc2330-0433-43f2-9154-dc399d24406c'
  ),
  'a090a598-3ae6-4fc4-b248-97ec5226702a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='LAND RECORDS',
    variable_level_2='FIRE HISTORY',
    variable_level_3='CHARCOAL SEDIMENT',
    detailed_variable='',
    identifier='a090a598-3ae6-4fc4-b248-97ec5226702a'
  ),
  'c9ba3275-2fe3-4619-b7c0-881d4f6fa34e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='LAND RECORDS',
    variable_level_2='FIRE HISTORY',
    variable_level_3='FIRE SCAR DATE',
    detailed_variable='',
    identifier='c9ba3275-2fe3-4619-b7c0-881d4f6fa34e'
  ),
  '2f257f83-bddd-41ce-ac78-5dac857b1be3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='LAND RECORDS',
    variable_level_2='GLACIAL RETREAT',
    variable_level_3='',
    detailed_variable='',
    identifier='2f257f83-bddd-41ce-ac78-5dac857b1be3'
  ),
  '3dfa8dcf-0df2-4654-ae3e-c97586265c3e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='LAND RECORDS',
    variable_level_2='GLACIATION',
    variable_level_3='',
    detailed_variable='',
    identifier='3dfa8dcf-0df2-4654-ae3e-c97586265c3e'
  ),
  '7557eddd-db2a-4f39-b1e2-91162f4fc92e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='LAND RECORDS',
    variable_level_2='ISOTOPES',
    variable_level_3='',
    detailed_variable='',
    identifier='7557eddd-db2a-4f39-b1e2-91162f4fc92e'
  ),
  'e0d88b2a-8563-443b-8756-73d744a41ee7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='LAND RECORDS',
    variable_level_2='LOESS',
    variable_level_3='',
    detailed_variable='',
    identifier='e0d88b2a-8563-443b-8756-73d744a41ee7'
  ),
  '219ba382-6c90-43d2-a6cf-2ddcc358f70e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='LAND RECORDS',
    variable_level_2='PALEOMAGNETIC DATA',
    variable_level_3='',
    detailed_variable='',
    identifier='219ba382-6c90-43d2-a6cf-2ddcc358f70e'
  ),
  '4fe601b0-314f-4f63-8ec1-3b96cc7263b8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='LAND RECORDS',
    variable_level_2='PALEOSOLS',
    variable_level_3='',
    detailed_variable='',
    identifier='4fe601b0-314f-4f63-8ec1-3b96cc7263b8'
  ),
  '8f4e90e0-aea0-40cd-b781-a6a69a6e6cb3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='LAND RECORDS',
    variable_level_2='RADIOCARBON',
    variable_level_3='',
    detailed_variable='',
    identifier='8f4e90e0-aea0-40cd-b781-a6a69a6e6cb3'
  ),
  '0960827a-ecdb-40a0-babc-fbd6df27bb53': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='LAND RECORDS',
    variable_level_2='SEDIMENTS',
    variable_level_3='',
    detailed_variable='',
    identifier='0960827a-ecdb-40a0-babc-fbd6df27bb53'
  ),
  '0d8fc6d8-eba5-4da1-9a78-d69c23d9d78d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='LAND RECORDS',
    variable_level_2='SEDIMENTS',
    variable_level_3='SEDIMENT THICKNESS',
    detailed_variable='',
    identifier='0d8fc6d8-eba5-4da1-9a78-d69c23d9d78d'
  ),
  'd845886d-0c44-4505-b5b9-d3fcd819208e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='LAND RECORDS',
    variable_level_2='STRATIGRAPHIC SEQUENCE',
    variable_level_3='',
    detailed_variable='',
    identifier='d845886d-0c44-4505-b5b9-d3fcd819208e'
  ),
  '52325c6e-1084-43c1-83b2-278bbe0201c6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='LAND RECORDS',
    variable_level_2='VOLCANIC DEPOSITS',
    variable_level_3='',
    detailed_variable='',
    identifier='52325c6e-1084-43c1-83b2-278bbe0201c6'
  ),
  '703d0c14-1978-4e7f-a51a-233c695823b9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='MASS EXTINCTIONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='703d0c14-1978-4e7f-a51a-233c695823b9'
  ),
  '5237fae3-c98e-4d4a-9013-d7c824b3862b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='OCEAN/LAKE RECORDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5237fae3-c98e-4d4a-9013-d7c824b3862b'
  ),
  '8f8c1808-ac5f-43e5-8397-dbb3d171144c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='OCEAN/LAKE RECORDS',
    variable_level_2='BOREHOLES',
    variable_level_3='',
    detailed_variable='',
    identifier='8f8c1808-ac5f-43e5-8397-dbb3d171144c'
  ),
  '47d6c670-db83-4975-b684-1be787811ac8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='OCEAN/LAKE RECORDS',
    variable_level_2='CORAL DEPOSITS',
    variable_level_3='',
    detailed_variable='',
    identifier='47d6c670-db83-4975-b684-1be787811ac8'
  ),
  'dc02e5fb-9ff3-483d-8c33-18db25a07eea': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='OCEAN/LAKE RECORDS',
    variable_level_2='ISOTOPES',
    variable_level_3='',
    detailed_variable='',
    identifier='dc02e5fb-9ff3-483d-8c33-18db25a07eea'
  ),
  '2c99427c-1a6a-4326-b072-0c12c87bd944': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='OCEAN/LAKE RECORDS',
    variable_level_2='LAKE LEVELS',
    variable_level_3='',
    detailed_variable='',
    identifier='2c99427c-1a6a-4326-b072-0c12c87bd944'
  ),
  '11e12021-f63e-4081-ae78-1bb19fe7b4bf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='OCEAN/LAKE RECORDS',
    variable_level_2='MACROFOSSILS',
    variable_level_3='',
    detailed_variable='',
    identifier='11e12021-f63e-4081-ae78-1bb19fe7b4bf'
  ),
  '6d00c961-de64-40ed-becd-3a95cae182e3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='OCEAN/LAKE RECORDS',
    variable_level_2='MICROFOSSILS',
    variable_level_3='',
    detailed_variable='',
    identifier='6d00c961-de64-40ed-becd-3a95cae182e3'
  ),
  '9713a1d5-8b03-4d38-b3b6-34578a1d5f39': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='OCEAN/LAKE RECORDS',
    variable_level_2='OXYGEN ISOTOPES',
    variable_level_3='',
    detailed_variable='',
    identifier='9713a1d5-8b03-4d38-b3b6-34578a1d5f39'
  ),
  '7ee90f7c-bdc6-403a-b447-2100d573cad6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='OCEAN/LAKE RECORDS',
    variable_level_2='PALEOMAGNETIC DATA',
    variable_level_3='',
    detailed_variable='',
    identifier='7ee90f7c-bdc6-403a-b447-2100d573cad6'
  ),
  'fe06f678-7155-4f93-9e28-4c083d60cccc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='OCEAN/LAKE RECORDS',
    variable_level_2='POLLEN',
    variable_level_3='',
    detailed_variable='',
    identifier='fe06f678-7155-4f93-9e28-4c083d60cccc'
  ),
  'a389bcd6-929d-43ac-9af1-5a20a4ddcbe2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='OCEAN/LAKE RECORDS',
    variable_level_2='RADIOCARBON',
    variable_level_3='',
    detailed_variable='',
    identifier='a389bcd6-929d-43ac-9af1-5a20a4ddcbe2'
  ),
  'b3764016-0b5d-48fb-be3e-4f1082cf13e7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='OCEAN/LAKE RECORDS',
    variable_level_2='SEDIMENTS',
    variable_level_3='',
    detailed_variable='',
    identifier='b3764016-0b5d-48fb-be3e-4f1082cf13e7'
  ),
  '6fb40553-a2ef-465a-b7d2-3401e3bfceac': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='OCEAN/LAKE RECORDS',
    variable_level_2='SEDIMENTS',
    variable_level_3='SEDIMENT THICKNESS',
    detailed_variable='',
    identifier='6fb40553-a2ef-465a-b7d2-3401e3bfceac'
  ),
  '417a6538-f89e-4f73-a89a-c2e5d2cd7667': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='OCEAN/LAKE RECORDS',
    variable_level_2='STRATIGRAPHIC SEQUENCE',
    variable_level_3='',
    detailed_variable='',
    identifier='417a6538-f89e-4f73-a89a-c2e5d2cd7667'
  ),
  '9db3b1cb-0d3d-4486-bf56-1e96b8691b01': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='OCEAN/LAKE RECORDS',
    variable_level_2='VARVE DEPOSITS',
    variable_level_3='',
    detailed_variable='',
    identifier='9db3b1cb-0d3d-4486-bf56-1e96b8691b01'
  ),
  '2f2d4df2-0701-4fe1-9d9b-e7e1c8678a8f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='OXYGEN ISOTOPE ANALYSIS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2f2d4df2-0701-4fe1-9d9b-e7e1c8678a8f'
  ),
  'dc3f297b-8471-4101-b70e-dc5765762061': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='PALEOCLIMATE FORCING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='dc3f297b-8471-4101-b70e-dc5765762061'
  ),
  'e0867ff5-2eb4-4959-b874-ac37c1b407e0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='PALEOCLIMATE FORCING',
    variable_level_2='CARBON DIOXIDE FORCING',
    variable_level_3='',
    detailed_variable='',
    identifier='e0867ff5-2eb4-4959-b874-ac37c1b407e0'
  ),
  '7cc62051-537c-4399-b9b9-b59c1a3e0773': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='PALEOCLIMATE FORCING',
    variable_level_2='ORBITAL CHANGE FORCING',
    variable_level_3='',
    detailed_variable='',
    identifier='7cc62051-537c-4399-b9b9-b59c1a3e0773'
  ),
  '250ce118-46e7-4dec-9e44-8054c9318cff': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='PALEOCLIMATE FORCING',
    variable_level_2='SOLAR FORCING',
    variable_level_3='',
    detailed_variable='',
    identifier='250ce118-46e7-4dec-9e44-8054c9318cff'
  ),
  '78c47e38-e842-4e31-81b2-44f44c52c692': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='PALEOCLIMATE FORCING',
    variable_level_2='VOLCANIC FORCING',
    variable_level_3='',
    detailed_variable='',
    identifier='78c47e38-e842-4e31-81b2-44f44c52c692'
  ),
  '6f6423e8-ab4e-4572-8982-d9c40f64e28b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6f6423e8-ab4e-4572-8982-d9c40f64e28b'
  ),
  '89e5b8c9-ef72-4e21-83c8-a7552f6871a4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_2='AIR TEMPERATURE RECONSTRUCTION',
    variable_level_3='',
    detailed_variable='',
    identifier='89e5b8c9-ef72-4e21-83c8-a7552f6871a4'
  ),
  '555b048d-8904-4a62-a85a-3af1aa14674e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_2='ATMOSPHERIC CIRCULATION RECONSTRUCTION',
    variable_level_3='',
    detailed_variable='',
    identifier='555b048d-8904-4a62-a85a-3af1aa14674e'
  ),
  '06bcba40-6046-4c0e-aa38-8f83410b93f0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_2='DROUGHT/PRECIPITATION RECONSTRUCTION',
    variable_level_3='',
    detailed_variable='',
    identifier='06bcba40-6046-4c0e-aa38-8f83410b93f0'
  ),
  '9f687ff2-52c0-496b-9a81-503a8c207823': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_2='GROUND WATER RECONSTRUCTION',
    variable_level_3='',
    detailed_variable='',
    identifier='9f687ff2-52c0-496b-9a81-503a8c207823'
  ),
  'ec4c1ae2-53f4-40ca-b0c3-e145f00e2583': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_2='LAKE LEVEL RECONSTRUCTION',
    variable_level_3='',
    detailed_variable='',
    identifier='ec4c1ae2-53f4-40ca-b0c3-e145f00e2583'
  ),
  '1ba98ab7-dee3-4b15-aea1-179ecd8f6e7d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_2='OCEAN SALINITY RECONSTRUCTION',
    variable_level_3='',
    detailed_variable='',
    identifier='1ba98ab7-dee3-4b15-aea1-179ecd8f6e7d'
  ),
  'b51093c5-5997-410c-899d-98d15ab5f5cc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_2='SEA LEVEL RECONSTRUCTION',
    variable_level_3='',
    detailed_variable='',
    identifier='b51093c5-5997-410c-899d-98d15ab5f5cc'
  ),
  'facdb262-04eb-47f9-b46e-ba7a379722ec': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_2='SEA SURFACE TEMPERATURE RECONSTRUCTION',
    variable_level_3='',
    detailed_variable='',
    identifier='facdb262-04eb-47f9-b46e-ba7a379722ec'
  ),
  '3b4ea1db-bb93-4eb8-ac08-4880a3a5e6d2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_2='SEDIMENTS',
    variable_level_3='',
    detailed_variable='',
    identifier='3b4ea1db-bb93-4eb8-ac08-4880a3a5e6d2'
  ),
  '88735956-6d46-41e1-8cbb-5dba20c33d8c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_2='SEDIMENTS',
    variable_level_3='SEDIMENT THICKNESS',
    detailed_variable='',
    identifier='88735956-6d46-41e1-8cbb-5dba20c33d8c'
  ),
  'fec6c2e4-ca15-426a-b344-36bba69e5c1f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_2='SOLAR FORCING/INSOLATION RECONSTRUCTION',
    variable_level_3='',
    detailed_variable='',
    identifier='fec6c2e4-ca15-426a-b344-36bba69e5c1f'
  ),
  'cde7aacb-0204-4a84-afcb-279cc3d0870c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_2='STREAMFLOW RECONSTRUCTION',
    variable_level_3='',
    detailed_variable='',
    identifier='cde7aacb-0204-4a84-afcb-279cc3d0870c'
  ),
  'c1c1890d-a6b0-4482-836b-a4b8ed0beee8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_2='VEGETATION RECONSTRUCTION',
    variable_level_3='',
    detailed_variable='',
    identifier='c1c1890d-a6b0-4482-836b-a4b8ed0beee8'
  ),
  '478092f3-7cdd-4136-84ec-cebf0d539480': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='PERMAFROST/METHANE RELEASE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='478092f3-7cdd-4136-84ec-cebf0d539480'
  ),
  '6971fecc-af14-4c97-82db-2b01c98453b9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='PLATE TECTONICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6971fecc-af14-4c97-82db-2b01c98453b9'
  ),
  '1cbefa2a-484e-4742-ad3d-d347d27272bd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='SPELEOTHEMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1cbefa2a-484e-4742-ad3d-d347d27272bd'
  ),
  '08bc1b7d-b27b-43e2-a728-4939efb88f08': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='PALEOCLIMATE INDICATORS',
    variable_level_1='VOLCANIC ACTIVITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='08bc1b7d-b27b-43e2-a728-4939efb88f08'
  ),
  '83ec2082-64bb-48d7-bcd2-454f082d608e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='REGIONAL CLIMATE LEVELS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='83ec2082-64bb-48d7-bcd2-454f082d608e'
  ),
  '273973f1-ccc5-4ae8-9872-17e249023c53': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='REGIONAL CLIMATE LEVELS',
    variable_level_1='MACROCLIMATE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='273973f1-ccc5-4ae8-9872-17e249023c53'
  ),
  'c4820c2b-37e0-491e-a268-e5b18e1a1062': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='REGIONAL CLIMATE LEVELS',
    variable_level_1='MESOCLIMATE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c4820c2b-37e0-491e-a268-e5b18e1a1062'
  ),
  '78369c58-beaa-46f2-b286-dd2540634556': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='REGIONAL CLIMATE LEVELS',
    variable_level_1='MICROCLIMATE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='78369c58-beaa-46f2-b286-dd2540634556'
  ),
  '3d64c047-c4fb-4981-bc91-d5dbc22337de': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='SUN-EARTH INTERACTIONS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3d64c047-c4fb-4981-bc91-d5dbc22337de'
  ),
  '3429bc72-0780-44c8-9743-92f84118279d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='SUN-EARTH INTERACTIONS',
    variable_level_1='SUNSPOT ACTIVITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3429bc72-0780-44c8-9743-92f84118279d'
  ),
  '22c14e35-48a4-40b5-a503-add48c2d4cd4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='SUN-EARTH INTERACTIONS',
    variable_level_1='SUNSPOT ACTIVITY',
    variable_level_2='LENGTH OF THE SOLAR CYCLE',
    variable_level_3='',
    detailed_variable='',
    identifier='22c14e35-48a4-40b5-a503-add48c2d4cd4'
  ),
  '3b230650-68ff-4e7a-9273-6e0b1083bdfa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='SUN-EARTH INTERACTIONS',
    variable_level_1='SUNSPOT ACTIVITY',
    variable_level_2='SOLAR FLUX',
    variable_level_3='',
    detailed_variable='',
    identifier='3b230650-68ff-4e7a-9273-6e0b1083bdfa'
  ),
  '9246fc12-17e7-4473-b9c0-c23e4bfc4eda': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='TERRESTRIAL HYDROSPHERE INDICATORS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9246fc12-17e7-4473-b9c0-c23e4bfc4eda'
  ),
  '915399a1-eb5b-475b-ae9a-ff45f1dcddc9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='TERRESTRIAL HYDROSPHERE INDICATORS',
    variable_level_1='FRESHWATER RUNOFF',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='915399a1-eb5b-475b-ae9a-ff45f1dcddc9'
  ),
  '56e7e412-b354-4ef4-8742-f1f5681c378a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='TERRESTRIAL HYDROSPHERE INDICATORS',
    variable_level_1='MOUNTAIN SNOW LINE SHIFT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='56e7e412-b354-4ef4-8742-f1f5681c378a'
  ),
  'ed0501c5-310c-42ab-b1eb-66e211f22803': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='TERRESTRIAL HYDROSPHERE INDICATORS',
    variable_level_1='PERMAFROST MELT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ed0501c5-310c-42ab-b1eb-66e211f22803'
  ),
  '64baed75-e3c0-4495-9bc9-c5b9373670f6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='TERRESTRIAL HYDROSPHERE INDICATORS',
    variable_level_1='RIVER/LAKE ICE BREAKUP',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='64baed75-e3c0-4495-9bc9-c5b9373670f6'
  ),
  '2adde197-3f0f-4eda-ae00-a337dfa853c3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='TERRESTRIAL HYDROSPHERE INDICATORS',
    variable_level_1='RIVER/LAKE ICE FREEZE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2adde197-3f0f-4eda-ae00-a337dfa853c3'
  ),
  '994cc55d-b789-4f03-98dc-4cd0f58ad12a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CLIMATE INDICATORS',
    term='TERRESTRIAL HYDROSPHERE INDICATORS',
    variable_level_1='SNOW COVER DEGRADATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='994cc55d-b789-4f03-98dc-4cd0f58ad12a'
  ),
  'fa0a36c3-2503-4662-98cd-7f3e74ce9f80': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fa0a36c3-2503-4662-98cd-7f3e74ce9f80'
  ),
  '376a1d5c-2496-4381-981f-bc047af92044': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='FROZEN GROUND',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='376a1d5c-2496-4381-981f-bc047af92044'
  ),
  '2e544263-d92f-46c2-9568-25e36d0b9825': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='FROZEN GROUND',
    variable_level_1='ACTIVE LAYER',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2e544263-d92f-46c2-9568-25e36d0b9825'
  ),
  '0cd7a96f-46e1-4d86-93d0-9cbb6fda61e3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='FROZEN GROUND',
    variable_level_1='CRYOSOLS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0cd7a96f-46e1-4d86-93d0-9cbb6fda61e3'
  ),
  '4931dcac-8b89-4bc9-ba59-469cfdcf6f12': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='FROZEN GROUND',
    variable_level_1='GROUND ICE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4931dcac-8b89-4bc9-ba59-469cfdcf6f12'
  ),
  'c82f3480-545f-4491-83f1-0477369ddcd8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='FROZEN GROUND',
    variable_level_1='PERMAFROST',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c82f3480-545f-4491-83f1-0477369ddcd8'
  ),
  'd8606e80-3d34-4540-a355-5f99737f7ab7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='FROZEN GROUND',
    variable_level_1='PERMAFROST',
    variable_level_2='PERMAFROST TEMPERATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='d8606e80-3d34-4540-a355-5f99737f7ab7'
  ),
  'b1ce822a-139b-4e11-8bbe-453f19501c36': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='FROZEN GROUND',
    variable_level_1='ROCK GLACIERS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b1ce822a-139b-4e11-8bbe-453f19501c36'
  ),
  '2a109b2f-947a-4c2c-9db9-ae315a53ef93': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='FROZEN GROUND',
    variable_level_1='SEASONALLY FROZEN GROUND',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2a109b2f-947a-4c2c-9db9-ae315a53ef93'
  ),
  '021714ad-1cae-441c-bb6f-4be866a0f742': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='FROZEN GROUND',
    variable_level_1='SOIL TEMPERATURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='021714ad-1cae-441c-bb6f-4be866a0f742'
  ),
  '78e5e44c-7832-456d-a599-893ea87ae695': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='FROZEN GROUND',
    variable_level_1='TALIK',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='78e5e44c-7832-456d-a599-893ea87ae695'
  ),
  '8603db51-3484-4439-8b3b-a06f48e8c686': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8603db51-3484-4439-8b3b-a06f48e8c686'
  ),
  '95fbaefd-1afe-4887-a1ba-fc338a8109bb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='ABLATION ZONES/ACCUMULATION ZONES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='95fbaefd-1afe-4887-a1ba-fc338a8109bb'
  ),
  'ab4b800d-820f-40cc-bb01-4e8835368d04': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='AGE AT ICE-THICKNESS-NORMALIZED DEPTHS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ab4b800d-820f-40cc-bb01-4e8835368d04'
  ),
  '9ce536e1-06c8-4817-af5f-b625cfe571a7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='AGE OF INTERNAL REFLECTIONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9ce536e1-06c8-4817-af5f-b625cfe571a7'
  ),
  '68d0f29d-cf46-4f8c-8cad-83817a7093bc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='BASAL SHEAR STRESS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='68d0f29d-cf46-4f8c-8cad-83817a7093bc'
  ),
  '79c7fd4c-9328-4a6f-82ed-bb012b570ecd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='BASINS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='79c7fd4c-9328-4a6f-82ed-bb012b570ecd'
  ),
  '7401d2b4-7a39-45ea-89f2-8b88cb0c22c4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='COASTLINE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7401d2b4-7a39-45ea-89f2-8b88cb0c22c4'
  ),
  '70541b66-c911-47fb-a99a-5638a9cb55d4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='DEPTHS AT SPECIFIC AGES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='70541b66-c911-47fb-a99a-5638a9cb55d4'
  ),
  '6159b9d9-4aa5-4dec-8146-0e47751449ff': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='FIRN',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6159b9d9-4aa5-4dec-8146-0e47751449ff'
  ),
  '0229714c-7960-4179-b671-30ceb9bf68bb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='FIRN',
    variable_level_2='FIRN AIR CONTENT',
    variable_level_3='',
    detailed_variable='',
    identifier='0229714c-7960-4179-b671-30ceb9bf68bb'
  ),
  'a30b2871-4cdc-418a-b00c-969b50008726': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='FIRN',
    variable_level_2='SNOW GRAIN SIZE',
    variable_level_3='',
    detailed_variable='',
    identifier='a30b2871-4cdc-418a-b00c-969b50008726'
  ),
  'ab319cdf-a34c-446c-9fc0-27605048364e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='GEOMETRY OF INTERNAL REFLECTIONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ab319cdf-a34c-446c-9fc0-27605048364e'
  ),
  '13bf19c5-087f-4fe0-87ea-ef6f7ecd5444': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='GLACIER ELEVATION/ICE SHEET ELEVATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='13bf19c5-087f-4fe0-87ea-ef6f7ecd5444'
  ),
  '399a84d1-ccf5-4167-a699-15eb7d1ad1e6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='GLACIER FACIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='399a84d1-ccf5-4167-a699-15eb7d1ad1e6'
  ),
  '9f408faa-a427-44e9-a194-b1b9caff1e6d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='GLACIER MASS BALANCE/ICE SHEET MASS BALANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9f408faa-a427-44e9-a194-b1b9caff1e6d'
  ),
  '73f3c797-2eed-4f0d-accf-7e8a36a3fa93': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='GLACIER MOTION/ICE SHEET MOTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='73f3c797-2eed-4f0d-accf-7e8a36a3fa93'
  ),
  '5034ba1f-7208-40a1-beeb-43aefe1c0c33': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='GLACIER THICKNESS/ICE SHEET THICKNESS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5034ba1f-7208-40a1-beeb-43aefe1c0c33'
  ),
  'bf19f1d1-ae18-4ff2-95f6-dc0ed812c568': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='GLACIER TOPOGRAPHY/ICE SHEET TOPOGRAPHY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bf19f1d1-ae18-4ff2-95f6-dc0ed812c568'
  ),
  '68eed887-8008-4352-b420-949457ab59ab': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='GLACIERS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='68eed887-8008-4352-b420-949457ab59ab'
  ),
  '8b705c00-50a7-438f-8fdd-c5799f7dab89': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='GLACIERS',
    variable_level_2='DEBRIS THICKNESS',
    variable_level_3='',
    detailed_variable='',
    identifier='8b705c00-50a7-438f-8fdd-c5799f7dab89'
  ),
  'c0fba127-5208-4e8c-b18f-349dc14fb3d3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='GLACIERS',
    variable_level_2='GLACIAL LAKE EXTENT',
    variable_level_3='',
    detailed_variable='',
    identifier='c0fba127-5208-4e8c-b18f-349dc14fb3d3'
  ),
  'b09f0809-2686-4e2b-bacf-e192760ca297': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='GLACIERS',
    variable_level_2='GLACIER ABLATION',
    variable_level_3='',
    detailed_variable='',
    identifier='b09f0809-2686-4e2b-bacf-e192760ca297'
  ),
  '7b8ad020-7eff-4639-b8c1-8b22e1535c00': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='GLACIERS',
    variable_level_2='GLACIER ACCUMULATION',
    variable_level_3='',
    detailed_variable='',
    identifier='7b8ad020-7eff-4639-b8c1-8b22e1535c00'
  ),
  'ae4be026-7471-4301-9264-675accce8340': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='GLACIERS',
    variable_level_2='GLACIER AREA',
    variable_level_3='',
    detailed_variable='',
    identifier='ae4be026-7471-4301-9264-675accce8340'
  ),
  '79b569df-5ed1-4791-b089-c46484069d81': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='GLACIERS',
    variable_level_2='GLACIER EXTENT',
    variable_level_3='',
    detailed_variable='',
    identifier='79b569df-5ed1-4791-b089-c46484069d81'
  ),
  '52d6b4d1-2daf-4e19-8129-c3e2dbf812d0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='GLACIERS',
    variable_level_2='GLACIER MASS',
    variable_level_3='',
    detailed_variable='',
    identifier='52d6b4d1-2daf-4e19-8129-c3e2dbf812d0'
  ),
  '6bd14a79-ac70-4f99-9107-a4c036d33bd7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='GLACIERS',
    variable_level_2='GLACIER MELT',
    variable_level_3='',
    detailed_variable='',
    identifier='6bd14a79-ac70-4f99-9107-a4c036d33bd7'
  ),
  '399025f7-315e-47db-af48-67771318a70b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='GLACIERS',
    variable_level_2='GLACIER REFREEZE',
    variable_level_3='',
    detailed_variable='',
    identifier='399025f7-315e-47db-af48-67771318a70b'
  ),
  '6c3aa715-61fd-47a1-804e-fb6d461792ea': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='GLACIERS',
    variable_level_2='GLACIER RUNOFF',
    variable_level_3='',
    detailed_variable='',
    identifier='6c3aa715-61fd-47a1-804e-fb6d461792ea'
  ),
  'd8e2bcae-7781-41b6-8d2d-7c82ae61be47': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='GLACIERS',
    variable_level_2='GLACIER TERMINUS',
    variable_level_3='',
    detailed_variable='',
    identifier='d8e2bcae-7781-41b6-8d2d-7c82ae61be47'
  ),
  '929f60ff-f938-48b0-86fd-c5c8c071c4bd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='GLACIERS',
    variable_level_2='GROUNDING LINE',
    variable_level_3='',
    detailed_variable='',
    identifier='929f60ff-f938-48b0-86fd-c5c8c071c4bd'
  ),
  'dbf0db20-7d18-4a67-9227-ea479fcf7c7d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='GLACIERS',
    variable_level_2='ICE STUPA',
    variable_level_3='',
    detailed_variable='',
    identifier='dbf0db20-7d18-4a67-9227-ea479fcf7c7d'
  ),
  'b6c9cc25-b989-4915-a5c6-43dff744b056': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='GLACIERS',
    variable_level_2='SUB-DEBRIS MELT ENHANCEMENT',
    variable_level_3='',
    detailed_variable='',
    identifier='b6c9cc25-b989-4915-a5c6-43dff744b056'
  ),
  '10b1872b-4a48-4360-a449-388e8988bca9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='ICE SHEETS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='10b1872b-4a48-4360-a449-388e8988bca9'
  ),
  '6d6a2b61-5d2c-4ec1-a164-34000f481588': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='ICE SHEETS',
    variable_level_2='ICE SHEET MEASUREMENTS',
    variable_level_3='',
    detailed_variable='',
    identifier='6d6a2b61-5d2c-4ec1-a164-34000f481588'
  ),
  '8a8fa93e-6424-46dd-ae97-d8afbac41b89': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='ICE SHEETS',
    variable_level_2='ICE SHEET MEASUREMENTS',
    variable_level_3='RIFTS',
    detailed_variable='',
    identifier='8a8fa93e-6424-46dd-ae97-d8afbac41b89'
  ),
  '94402f47-38ea-4798-98da-ea17599e092f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='ICE SHEETS',
    variable_level_2='SURFACE MORPHOLOGY',
    variable_level_3='',
    detailed_variable='',
    identifier='94402f47-38ea-4798-98da-ea17599e092f'
  ),
  '681e59ee-2006-454d-82d3-c9be49cc67a5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='ICE SHELVES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='681e59ee-2006-454d-82d3-c9be49cc67a5'
  ),
  '4d95ccc8-3ef9-40df-85e7-db36cb815499': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='GLACIERS/ICE SHEETS',
    variable_level_1='ICEBERGS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4d95ccc8-3ef9-40df-85e7-db36cb815499'
  ),
  'dbe8d9d3-1609-4128-9b45-1061a501401b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='LAND ICE/OCEAN CLASSIFICATION',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='dbe8d9d3-1609-4128-9b45-1061a501401b'
  ),
  '860e2af9-ce29-4f3f-b027-ae3747eb3e01': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='860e2af9-ce29-4f3f-b027-ae3747eb3e01'
  ),
  'd8bb5d3c-c1ab-4ee2-82ec-f1c44af6bba4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='FAST ICE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d8bb5d3c-c1ab-4ee2-82ec-f1c44af6bba4'
  ),
  'a4466cbe-b991-427b-97b8-fdc284b9ef21': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='FREEBOARD',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a4466cbe-b991-427b-97b8-fdc284b9ef21'
  ),
  '5569b7a3-3a4b-4799-8c68-98126757074b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='HEAT FLUX',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5569b7a3-3a4b-4799-8c68-98126757074b'
  ),
  'da442d88-426b-4469-8ebe-f2ec83f410d0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='HEAT FLUX',
    variable_level_2='LATENT HEAT FLUX',
    variable_level_3='',
    detailed_variable='',
    identifier='da442d88-426b-4469-8ebe-f2ec83f410d0'
  ),
  '8b333abf-9475-4c61-bcd9-65b40baaf213': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='HEAT FLUX',
    variable_level_2='LONGWAVE HEAT FLUX',
    variable_level_3='',
    detailed_variable='',
    identifier='8b333abf-9475-4c61-bcd9-65b40baaf213'
  ),
  '9a11c433-3cf6-4d6a-9038-259a77f94158': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='HEAT FLUX',
    variable_level_2='SENSIBLE HEAT FLUX',
    variable_level_3='',
    detailed_variable='',
    identifier='9a11c433-3cf6-4d6a-9038-259a77f94158'
  ),
  '2bc52086-0c6d-4d66-8b1f-72ca78297383': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='HEAT FLUX',
    variable_level_2='SHORTWAVE HEAT FLUX',
    variable_level_3='',
    detailed_variable='',
    identifier='2bc52086-0c6d-4d66-8b1f-72ca78297383'
  ),
  '1009557b-0d4b-4c13-81a0-fd95c15bf158': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='ICE DEFORMATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1009557b-0d4b-4c13-81a0-fd95c15bf158'
  ),
  'c7708bb6-a0fa-4905-b99d-c468da7d951a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='ICE DEPTH/THICKNESS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c7708bb6-a0fa-4905-b99d-c468da7d951a'
  ),
  '3bacb194-cf25-42ae-95af-54a6a53898ef': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='ICE DRAFT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3bacb194-cf25-42ae-95af-54a6a53898ef'
  ),
  '5fa04fa9-06c7-41c7-98f9-f92756f080ea': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='ICE EDGES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5fa04fa9-06c7-41c7-98f9-f92756f080ea'
  ),
  '63b37017-9d57-4247-af4e-2df36ee3ed03': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='ICE EXTENT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='63b37017-9d57-4247-af4e-2df36ee3ed03'
  ),
  'af0d756e-784e-4747-97d0-3425baf5d09b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='ICE FLOES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='af0d756e-784e-4747-97d0-3425baf5d09b'
  ),
  'd9667e73-30db-45f9-861c-e0a5caaf2bf0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='ICE GROWTH/MELT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d9667e73-30db-45f9-861c-e0a5caaf2bf0'
  ),
  'ce3a1edd-a2fe-4efd-8971-9dd7b97b6d79': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='ICE ROUGHNESS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ce3a1edd-a2fe-4efd-8971-9dd7b97b6d79'
  ),
  'f6e7aa9a-ae65-480e-84fa-b3a5d523e822': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='ICE TEMPERATURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f6e7aa9a-ae65-480e-84fa-b3a5d523e822'
  ),
  '6bfd4d52-fad4-470f-9da0-fa7df2a5b4aa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='ICE TYPES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6bfd4d52-fad4-470f-9da0-fa7df2a5b4aa'
  ),
  '1efe6ac1-d375-44c3-b8ec-d0ff2987a881': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='ICEBERGS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1efe6ac1-d375-44c3-b8ec-d0ff2987a881'
  ),
  'f0d4b06b-c498-4760-bc92-877e28f3a098': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='ISOTOPES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f0d4b06b-c498-4760-bc92-877e28f3a098'
  ),
  '4f0f606c-6bf8-4b8c-9431-d5696fe8a5f2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='LEADS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4f0f606c-6bf8-4b8c-9431-d5696fe8a5f2'
  ),
  '5315248e-623d-40fa-a0fd-e1a981b03290': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='MELT POND',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5315248e-623d-40fa-a0fd-e1a981b03290'
  ),
  '5d7ea074-225b-4221-b122-e6a085cdce24': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='PACK ICE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5d7ea074-225b-4221-b122-e6a085cdce24'
  ),
  '70acf223-7895-4cbe-aca6-815babb2b7ed': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='POLYNYAS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='70acf223-7895-4cbe-aca6-815babb2b7ed'
  ),
  'cece77b6-42bf-44f6-9193-050cbc5f4cf7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='REFLECTANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='cece77b6-42bf-44f6-9193-050cbc5f4cf7'
  ),
  '6bc39a6d-cc60-467a-9181-d8b4e02a1cb0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='SALINITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6bc39a6d-cc60-467a-9181-d8b4e02a1cb0'
  ),
  '3488309d-ef21-4d60-81a3-78fb99ffa756': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='SEA ICE AGE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3488309d-ef21-4d60-81a3-78fb99ffa756'
  ),
  '8012fda7-3ea4-4ef2-bb4e-0f66d4d9e850': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='SEA ICE CONCENTRATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8012fda7-3ea4-4ef2-bb4e-0f66d4d9e850'
  ),
  'a3f36d7c-4eed-4d7a-8902-a5fcdc1b6261': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='SEA ICE CONCENTRATION',
    variable_level_2='ICE FRACTION',
    variable_level_3='',
    detailed_variable='',
    identifier='a3f36d7c-4eed-4d7a-8902-a5fcdc1b6261'
  ),
  '91603432-2af5-4012-a670-e73ff2aaa7b9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='SEA ICE DYNAMICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='91603432-2af5-4012-a670-e73ff2aaa7b9'
  ),
  '139b0dae-27bb-42bd-8027-81fb9fd8f85d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='SEA ICE ELEVATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='139b0dae-27bb-42bd-8027-81fb9fd8f85d'
  ),
  'cfc3ed52-a7e2-4ad1-8330-1f97c7cb0203': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='SEA ICE MASS BALANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='cfc3ed52-a7e2-4ad1-8330-1f97c7cb0203'
  ),
  '1455c369-88e2-411b-83f7-c914b20609b1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='SEA ICE MOTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1455c369-88e2-411b-83f7-c914b20609b1'
  ),
  '1ad69005-4418-4f5c-bab5-580d13c5992e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='SEA ICE STRAIN RATES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1ad69005-4418-4f5c-bab5-580d13c5992e'
  ),
  '1e335f30-1dff-4746-ad1c-c873a1fdb320': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='SEA ICE STRENGTH',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1e335f30-1dff-4746-ad1c-c873a1fdb320'
  ),
  '63a4c6e1-9248-488d-95ab-88ac2fb0a21d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='SEA ICE STRESS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='63a4c6e1-9248-488d-95ab-88ac2fb0a21d'
  ),
  '2e8a0665-3488-40f3-ac96-f9d660e31a4f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='SEA ICE VOLUME BUDGET',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2e8a0665-3488-40f3-ac96-f9d660e31a4f'
  ),
  '1a7da9b0-2170-4329-997c-939af604b0bf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='SEA ICE VOLUME BUDGET',
    variable_level_2='ADVECTION',
    variable_level_3='',
    detailed_variable='',
    identifier='1a7da9b0-2170-4329-997c-939af604b0bf'
  ),
  '4125f82e-9be4-4441-80f2-0d39ccd3d1d8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='SEA ICE VOLUME BUDGET',
    variable_level_2='DIFFUSION',
    variable_level_3='',
    detailed_variable='',
    identifier='4125f82e-9be4-4441-80f2-0d39ccd3d1d8'
  ),
  '78de9de8-852d-4b14-8650-414b6e4bfe0a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='SEA ICE VOLUME BUDGET',
    variable_level_2='FLOODING',
    variable_level_3='',
    detailed_variable='',
    identifier='78de9de8-852d-4b14-8650-414b6e4bfe0a'
  ),
  'd693a3a9-99dd-4872-a2c0-a4b930d312e5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='SEA ICE VOLUME BUDGET',
    variable_level_2='ICE GROWTH/MELT',
    variable_level_3='',
    detailed_variable='',
    identifier='d693a3a9-99dd-4872-a2c0-a4b930d312e5'
  ),
  '4dff1a9b-23ee-4af4-a3c3-601cd2f52f56': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='SEA ICE VOLUME BUDGET',
    variable_level_2='SUBLIMATION',
    variable_level_3='',
    detailed_variable='',
    identifier='4dff1a9b-23ee-4af4-a3c3-601cd2f52f56'
  ),
  '7a55ceba-057a-408f-924f-9ea1d4675f26': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='SEA ICE/OCEAN CLASSIFICATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7a55ceba-057a-408f-924f-9ea1d4675f26'
  ),
  'aa645419-cff3-4f5b-84af-e3de41dd0d16': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='SNOW DEPTH',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='aa645419-cff3-4f5b-84af-e3de41dd0d16'
  ),
  '064f9784-697e-414c-b463-29cfd734e689': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SEA ICE',
    variable_level_1='SNOW MELT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='064f9784-697e-414c-b463-29cfd734e689'
  ),
  'aa35a52f-e3d9-41bd-abd2-ec7e1a8101d1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='aa35a52f-e3d9-41bd-abd2-ec7e1a8101d1'
  ),
  '41ebe049-230e-4ff7-acb1-43de68ace83e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='ALBEDO',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='41ebe049-230e-4ff7-acb1-43de68ace83e'
  ),
  'e1dbe955-7285-4df2-a854-07693fce44ec': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='AVALANCHE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e1dbe955-7285-4df2-a854-07693fce44ec'
  ),
  '1ce90ed3-d8f1-42f5-a609-e2b329839fed': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='BLOWING SNOW',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1ce90ed3-d8f1-42f5-a609-e2b329839fed'
  ),
  'c306d542-9be8-449d-ba33-28ad033c77aa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='DEPTH HOAR',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c306d542-9be8-449d-ba33-28ad033c77aa'
  ),
  'dafb67df-dc6d-40a0-8d94-e4621d2538ce': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='FREEZE/THAW',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='dafb67df-dc6d-40a0-8d94-e4621d2538ce'
  ),
  '99c2726c-3ad7-4f54-b5ce-d954f9780bd1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='FREEZE/THAW',
    variable_level_2='TRANSITION DIRECTION',
    variable_level_3='',
    detailed_variable='',
    identifier='99c2726c-3ad7-4f54-b5ce-d954f9780bd1'
  ),
  'ea936862-2c98-41e5-8514-6b7288a5f941': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='FROST',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ea936862-2c98-41e5-8514-6b7288a5f941'
  ),
  'e28676de-738d-4112-8897-ee585b7d1d84': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='ICE DEPTH/THICKNESS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e28676de-738d-4112-8897-ee585b7d1d84'
  ),
  '19409c76-09d4-455c-b1f1-dc2e647f7403': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='ICE EXTENT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='19409c76-09d4-455c-b1f1-dc2e647f7403'
  ),
  '19594c37-ef32-4b03-bda6-abf8a321fdb9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='ICE GROWTH/MELT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='19594c37-ef32-4b03-bda6-abf8a321fdb9'
  ),
  'd7a7e7b7-4084-4a29-a4f0-470ea477b486': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='ICE MELANGE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d7a7e7b7-4084-4a29-a4f0-470ea477b486'
  ),
  'dcb59899-1b98-478b-b23f-7485f4d93eec': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='ICE MELANGE',
    variable_level_2='ICE MELANGE VELOCITY',
    variable_level_3='',
    detailed_variable='',
    identifier='dcb59899-1b98-478b-b23f-7485f4d93eec'
  ),
  '4b85cc37-1577-43f6-8cfa-8da2c49eaece': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='ICE MOTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4b85cc37-1577-43f6-8cfa-8da2c49eaece'
  ),
  '481cc0cb-3e52-45e4-bf6e-3bb0c43e3392': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='ICE SURFACE TEMPERAT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='481cc0cb-3e52-45e4-bf6e-3bb0c43e3392'
  ),
  '3896f032-388f-408e-b988-bf7e100704ba': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='ICE VELOCITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3896f032-388f-408e-b988-bf7e100704ba'
  ),
  '8cb47594-3af6-4f4f-8ba1-4299a6d6887e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='LAKE ICE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8cb47594-3af6-4f4f-8ba1-4299a6d6887e'
  ),
  '9cb22e18-bad2-4644-88d4-106600acf0fa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='LAND ICE FRACTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9cb22e18-bad2-4644-88d4-106600acf0fa'
  ),
  '1f4cdbc4-0f65-4384-83c9-9422c280717d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='PERMAFROST',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1f4cdbc4-0f65-4384-83c9-9422c280717d'
  ),
  '00a21e9c-0c1d-4931-b9fa-b0204625a98a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='REFLECTANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='00a21e9c-0c1d-4931-b9fa-b0204625a98a'
  ),
  '2dce1d90-f958-4e96-b8d8-c8b0bc69d16e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='REFLECTANCE',
    variable_level_2='BIDIRECTIONAL REFLECTANCE DISTRIBUTION FUNCTION',
    variable_level_3='',
    detailed_variable='',
    identifier='2dce1d90-f958-4e96-b8d8-c8b0bc69d16e'
  ),
  'eddd1f51-a6ae-4c35-bac5-e68131fcb386': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='REFLECTANCE',
    variable_level_2='BIDIRECTIONAL REFLECTANCE FACTOR',
    variable_level_3='',
    detailed_variable='',
    identifier='eddd1f51-a6ae-4c35-bac5-e68131fcb386'
  ),
  '26bb68e4-d24f-4f99-b147-5e333ff1f2cd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='RELATIVE PERMITTIVITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='26bb68e4-d24f-4f99-b147-5e333ff1f2cd'
  ),
  '52e6600b-7a51-4267-8b62-e79034db3a48': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='RIVER ICE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='52e6600b-7a51-4267-8b62-e79034db3a48'
  ),
  '97fd3b62-917d-4946-8c8a-29d2a15bf6dd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW CLASSIFICATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='97fd3b62-917d-4946-8c8a-29d2a15bf6dd'
  ),
  'c8ff0035-4776-4eb9-8cc9-a63d380102c8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW COVER',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c8ff0035-4776-4eb9-8cc9-a63d380102c8'
  ),
  'ba2e2eff-77e0-4071-8884-b2af06e5fc7b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW DENSITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ba2e2eff-77e0-4071-8884-b2af06e5fc7b'
  ),
  '47bc8942-6fdd-4173-bf38-209e933d843f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW DEPTH',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='47bc8942-6fdd-4173-bf38-209e933d843f'
  ),
  'a3520db9-7bed-4f55-a9f6-028d52af6091': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW ENERGY BALANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a3520db9-7bed-4f55-a9f6-028d52af6091'
  ),
  '99506fd1-5f84-485d-8e26-03e4f7b55136': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW FACIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='99506fd1-5f84-485d-8e26-03e4f7b55136'
  ),
  '89453747-14cb-4824-8c93-b142dff28cdb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW FRACTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='89453747-14cb-4824-8c93-b142dff28cdb'
  ),
  'b5c86a31-0c3b-4217-8530-0aa5234a5f89': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW GRAIN SIZE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b5c86a31-0c3b-4217-8530-0aa5234a5f89'
  ),
  '58f98d2a-d7d6-47d4-b826-68fdc57e79bb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW MELT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='58f98d2a-d7d6-47d4-b826-68fdc57e79bb'
  ),
  '4dc6b614-36ad-4e3b-ac6f-af6e0aa6378b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW MICROSTRUCTURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4dc6b614-36ad-4e3b-ac6f-af6e0aa6378b'
  ),
  'a72b96ad-3755-4205-b353-66592c7bff54': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW MICROSTRUCTURE',
    variable_level_2='SPECIFIC SURFACE AREA',
    variable_level_3='',
    detailed_variable='',
    identifier='a72b96ad-3755-4205-b353-66592c7bff54'
  ),
  '76d95fbe-8fa9-485d-9d65-55340512c12d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW RADIATIVE FORCING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='76d95fbe-8fa9-485d-9d65-55340512c12d'
  ),
  '9e15c793-ede5-4089-8fb7-5bbb31ff7913': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW STRATIGRAPHY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9e15c793-ede5-4089-8fb7-5bbb31ff7913'
  ),
  '46c8754c-8adf-48e7-a14c-b91945d343bb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW VOLUME BUDGET',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='46c8754c-8adf-48e7-a14c-b91945d343bb'
  ),
  '7ee6c16d-8828-44d6-a014-63a9ac8eee37': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW VOLUME BUDGET',
    variable_level_2='ADVECTION',
    variable_level_3='',
    detailed_variable='',
    identifier='7ee6c16d-8828-44d6-a014-63a9ac8eee37'
  ),
  'db28e274-27f4-4006-a363-32a98eab859d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW VOLUME BUDGET',
    variable_level_2='DIFFUSION',
    variable_level_3='',
    detailed_variable='',
    identifier='db28e274-27f4-4006-a363-32a98eab859d'
  ),
  '2c2baf1f-3e73-443b-8b1d-d248bb438c1b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW VOLUME BUDGET',
    variable_level_2='PRECIPITATION',
    variable_level_3='',
    detailed_variable='',
    identifier='2c2baf1f-3e73-443b-8b1d-d248bb438c1b'
  ),
  '8e4f4500-e377-4b8c-8275-6194c4b7db8b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW VOLUME BUDGET',
    variable_level_2='SNOW MELT',
    variable_level_3='',
    detailed_variable='',
    identifier='8e4f4500-e377-4b8c-8275-6194c4b7db8b'
  ),
  '94790dac-6bf8-453b-b841-30b86bdcd491': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW VOLUME BUDGET',
    variable_level_2='SUBLIMATION',
    variable_level_3='',
    detailed_variable='',
    identifier='94790dac-6bf8-453b-b841-30b86bdcd491'
  ),
  '587e4d68-36f0-45b5-9978-4b3edd58a1c0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW WATER EQUIVALENT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='587e4d68-36f0-45b5-9978-4b3edd58a1c0'
  ),
  'dfe4b154-84e0-4005-81ce-90daf38c06e3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW/ICE CHEMISTRY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='dfe4b154-84e0-4005-81ce-90daf38c06e3'
  ),
  'c599ae55-3ca3-4c68-9ef0-2b3739bda726': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW/ICE MASS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c599ae55-3ca3-4c68-9ef0-2b3739bda726'
  ),
  '99bc6084-32bc-405a-b2e9-efd906fa370b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SNOW/ICE TEMPERATURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='99bc6084-32bc-405a-b2e9-efd906fa370b'
  ),
  '8ea93d94-9a95-4dc0-8ad9-08896541fea9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='SURFACE MELT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8ea93d94-9a95-4dc0-8ad9-08896541fea9'
  ),
  '27a8f83d-6fdb-43a8-bbdd-25a0ac9b3b5a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='VISIBLE SNOW DARKENING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='27a8f83d-6fdb-43a8-bbdd-25a0ac9b3b5a'
  ),
  '1da194ab-ba56-4dec-8931-7c85d706c7da': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='WETNESS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1da194ab-ba56-4dec-8931-7c85d706c7da'
  ),
  '067004b9-1628-4c00-8bfb-28f910b68d59': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='CRYOSPHERE',
    term='SNOW/ICE',
    variable_level_1='WHITEOUT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='067004b9-1628-4c00-8bfb-28f910b68d59'
  ),
  'fb93d937-c17c-45d0-a9e3-ca5c8a800ca8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fb93d937-c17c-45d0-a9e3-ca5c8a800ca8'
  ),
  '07a856fd-75e2-46e8-91eb-8a8562d3452f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='BOUNDARIES',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='07a856fd-75e2-46e8-91eb-8a8562d3452f'
  ),
  '1ae304de-252c-45da-8dd8-df99a281e4f4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='BOUNDARIES',
    variable_level_1='ADMINISTRATIVE DIVISIONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1ae304de-252c-45da-8dd8-df99a281e4f4'
  ),
  '1f908d8e-3a7f-4f1c-ade0-884b7cc2f23a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='BOUNDARIES',
    variable_level_1='BOUNDARY ENCROACHMENT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1f908d8e-3a7f-4f1c-ade0-884b7cc2f23a'
  ),
  '8064b11d-8f9f-4c89-94fd-8a7cba95bb64': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='BOUNDARIES',
    variable_level_1='BOUNDARY SURVEYS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8064b11d-8f9f-4c89-94fd-8a7cba95bb64'
  ),
  '3381412c-54f0-4911-85ef-81d669c896cf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='BOUNDARIES',
    variable_level_1='POLITICAL DIVISIONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3381412c-54f0-4911-85ef-81d669c896cf'
  ),
  '245c630a-8022-46ed-9a79-8f6cf99b0822': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='BOUNDARIES',
    variable_level_1='POLITICAL DIVISIONS',
    variable_level_2='COUNTRY BOUNDARIES',
    variable_level_3='',
    detailed_variable='',
    identifier='245c630a-8022-46ed-9a79-8f6cf99b0822'
  ),
  'ef04f170-4797-4db1-aff7-ad493b6a7cda': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='BOUNDARIES',
    variable_level_1='POLITICAL DIVISIONS',
    variable_level_2='STATE BOUNDARIES',
    variable_level_3='',
    detailed_variable='',
    identifier='ef04f170-4797-4db1-aff7-ad493b6a7cda'
  ),
  'f69374fe-eda2-4223-b130-096220251235': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='CLIMATE CHANGE RESPONSES',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f69374fe-eda2-4223-b130-096220251235'
  ),
  '223e1204-e305-43d2-80d0-8baed8c828c0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='CLIMATE CHANGE RESPONSES',
    variable_level_1='CLIMATE ADAPTATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='223e1204-e305-43d2-80d0-8baed8c828c0'
  ),
  'c4a6c571-406e-4023-b479-6a1fc30f184c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='CLIMATE CHANGE RESPONSES',
    variable_level_1='CLIMATE MITIGATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c4a6c571-406e-4023-b479-6a1fc30f184c'
  ),
  'b8dfcdca-5777-4374-8950-8225b90df0e4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='CLIMATE CHANGE RESPONSES',
    variable_level_1='CLIMATE RESILIENCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b8dfcdca-5777-4374-8950-8225b90df0e4'
  ),
  '6062293a-f372-4e65-9bab-27355f0ebc59': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='CLIMATE CHANGE RESPONSES',
    variable_level_1='SCENARIO PLANNING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6062293a-f372-4e65-9bab-27355f0ebc59'
  ),
  'cdbe5ef5-408d-489d-b6ff-4482ce4a99c7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ECONOMIC RESOURCES',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='cdbe5ef5-408d-489d-b6ff-4482ce4a99c7'
  ),
  '83741fb9-6f86-4670-abbb-c1f3b14a939d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ECONOMIC RESOURCES',
    variable_level_1='AGRICULTURE PRODUCTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='83741fb9-6f86-4670-abbb-c1f3b14a939d'
  ),
  '941c691a-3bff-4c58-854a-16c5529524e9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ECONOMIC RESOURCES',
    variable_level_1='AGRICULTURE PRODUCTION',
    variable_level_2='MONOCULTURE',
    variable_level_3='',
    detailed_variable='',
    identifier='941c691a-3bff-4c58-854a-16c5529524e9'
  ),
  '392d3da2-c03c-4aa5-bf60-417984f824a6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ECONOMIC RESOURCES',
    variable_level_1='AQUACULTURE PRODUCTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='392d3da2-c03c-4aa5-bf60-417984f824a6'
  ),
  'b73cee46-8e2c-4df9-b1ed-7f0aa98a04ac': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ECONOMIC RESOURCES',
    variable_level_1='ENERGY PRODUCTION/USE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b73cee46-8e2c-4df9-b1ed-7f0aa98a04ac'
  ),
  '99ed30c9-332c-4acf-8620-eab3c67bcc90': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ECONOMIC RESOURCES',
    variable_level_1='ENERGY PRODUCTION/USE',
    variable_level_2='BIOMASS ENERGY PRODUCTION/USE',
    variable_level_3='',
    detailed_variable='',
    identifier='99ed30c9-332c-4acf-8620-eab3c67bcc90'
  ),
  'c90081fb-f6c2-4f7c-a124-0cd432e92200': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ECONOMIC RESOURCES',
    variable_level_1='ENERGY PRODUCTION/USE',
    variable_level_2='COAL PRODUCTION/USE',
    variable_level_3='',
    detailed_variable='',
    identifier='c90081fb-f6c2-4f7c-a124-0cd432e92200'
  ),
  '05410006-351a-4877-96e8-f0a821161ecf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ECONOMIC RESOURCES',
    variable_level_1='ENERGY PRODUCTION/USE',
    variable_level_2='GEOTHERMAL ENERGY PRODUCTION/USE',
    variable_level_3='',
    detailed_variable='',
    identifier='05410006-351a-4877-96e8-f0a821161ecf'
  ),
  '7eba0eef-3a30-4282-a162-1f483370ddc4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ECONOMIC RESOURCES',
    variable_level_1='ENERGY PRODUCTION/USE',
    variable_level_2='HYDROELECTRIC ENERGY PRODUCTION/USE',
    variable_level_3='',
    detailed_variable='',
    identifier='7eba0eef-3a30-4282-a162-1f483370ddc4'
  ),
  'c346378a-09ee-428c-89c1-c94354cdc74f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ECONOMIC RESOURCES',
    variable_level_1='ENERGY PRODUCTION/USE',
    variable_level_2='HYDROGEN PRODUCTION/USE',
    variable_level_3='',
    detailed_variable='',
    identifier='c346378a-09ee-428c-89c1-c94354cdc74f'
  ),
  'd3b2e908-b732-480c-a9cb-2e981da52094': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ECONOMIC RESOURCES',
    variable_level_1='ENERGY PRODUCTION/USE',
    variable_level_2='METHANE PRODUCTION/USE',
    variable_level_3='',
    detailed_variable='',
    identifier='d3b2e908-b732-480c-a9cb-2e981da52094'
  ),
  '83bddfa5-d9ba-40f1-9a2f-1bee33559176': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ECONOMIC RESOURCES',
    variable_level_1='ENERGY PRODUCTION/USE',
    variable_level_2='NATURAL GAS PRODUCTION/USE',
    variable_level_3='',
    detailed_variable='',
    identifier='83bddfa5-d9ba-40f1-9a2f-1bee33559176'
  ),
  '582af998-1f5c-48a7-8cdd-70fe06bb9f17': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ECONOMIC RESOURCES',
    variable_level_1='ENERGY PRODUCTION/USE',
    variable_level_2='NUCLEAR ENERGY PRODUCTION/USE',
    variable_level_3='',
    detailed_variable='',
    identifier='582af998-1f5c-48a7-8cdd-70fe06bb9f17'
  ),
  'e5d17711-c9c1-42f6-96e4-c618c0df37cb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ECONOMIC RESOURCES',
    variable_level_1='ENERGY PRODUCTION/USE',
    variable_level_2='OIL PRODUCTION/USE',
    variable_level_3='',
    detailed_variable='',
    identifier='e5d17711-c9c1-42f6-96e4-c618c0df37cb'
  ),
  'e4774745-c565-4b9e-a642-6fa4a0b3b79b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ECONOMIC RESOURCES',
    variable_level_1='ENERGY PRODUCTION/USE',
    variable_level_2='PETROLEUM PRODUCTION/USE',
    variable_level_3='',
    detailed_variable='',
    identifier='e4774745-c565-4b9e-a642-6fa4a0b3b79b'
  ),
  '8b4f34c1-7aed-4833-811a-401382abd17c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ECONOMIC RESOURCES',
    variable_level_1='ENERGY PRODUCTION/USE',
    variable_level_2='SOLAR ENERGY PRODUCTION/USE',
    variable_level_3='',
    detailed_variable='',
    identifier='8b4f34c1-7aed-4833-811a-401382abd17c'
  ),
  '1eb6eeff-77f8-40b6-8e4a-2e4438f00b10': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ECONOMIC RESOURCES',
    variable_level_1='ENERGY PRODUCTION/USE',
    variable_level_2='TIDAL ENERGY PRODUCTION/USE',
    variable_level_3='',
    detailed_variable='',
    identifier='1eb6eeff-77f8-40b6-8e4a-2e4438f00b10'
  ),
  '62c1fec5-3512-4136-a060-ec2338a48296': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ECONOMIC RESOURCES',
    variable_level_1='ENERGY PRODUCTION/USE',
    variable_level_2='WAVE ENERGY PRODUCTION/USE',
    variable_level_3='',
    detailed_variable='',
    identifier='62c1fec5-3512-4136-a060-ec2338a48296'
  ),
  'b3a95e10-1c1d-41cf-8802-8bb1d3a41353': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ECONOMIC RESOURCES',
    variable_level_1='ENERGY PRODUCTION/USE',
    variable_level_2='WIND ENERGY PRODUCTION/USE',
    variable_level_3='',
    detailed_variable='',
    identifier='b3a95e10-1c1d-41cf-8802-8bb1d3a41353'
  ),
  '49da5018-59ec-4a60-9cb9-614ea6266ced': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ECONOMIC RESOURCES',
    variable_level_1='MARICULTURE PRODUCTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='49da5018-59ec-4a60-9cb9-614ea6266ced'
  ),
  '82fdb39c-4fe8-4e2b-9dcf-67ceb4c6d8b9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ECONOMIC RESOURCES',
    variable_level_1='TOURISM',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='82fdb39c-4fe8-4e2b-9dcf-67ceb4c6d8b9'
  ),
  'e6cf64ce-389f-479c-835a-eecd612d4d88': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ECONOMIC RESOURCES',
    variable_level_1='TOURISM',
    variable_level_2='ECOTOURISM',
    variable_level_3='',
    detailed_variable='',
    identifier='e6cf64ce-389f-479c-835a-eecd612d4d88'
  ),
  'd81b77be-0177-4e26-942c-aa911239482d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL GOVERNANCE/MANAGEMENT',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d81b77be-0177-4e26-942c-aa911239482d'
  ),
  'e8c24822-7d2d-48c6-9dca-df3860e9bd63': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL GOVERNANCE/MANAGEMENT',
    variable_level_1='CARBON CAPTURE AND STORAGE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e8c24822-7d2d-48c6-9dca-df3860e9bd63'
  ),
  '0e530a5f-1e75-4602-9659-98ff5c3d7076': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL GOVERNANCE/MANAGEMENT',
    variable_level_1='CARBON FOOTPRINT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0e530a5f-1e75-4602-9659-98ff5c3d7076'
  ),
  '079724fa-ff86-4195-aee0-51a4d6dd73bb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL GOVERNANCE/MANAGEMENT',
    variable_level_1='ENVIRONMENTAL ASSESSMENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='079724fa-ff86-4195-aee0-51a4d6dd73bb'
  ),
  '57df059e-578a-4371-9484-7a34d63edfa5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL GOVERNANCE/MANAGEMENT',
    variable_level_1='ENVIRONMENTAL REGULATIONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='57df059e-578a-4371-9484-7a34d63edfa5'
  ),
  '0ef4a2f0-8a29-4f5e-9396-b4f6a71c8bf6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL GOVERNANCE/MANAGEMENT',
    variable_level_1='FIRE MANAGEMENT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0ef4a2f0-8a29-4f5e-9396-b4f6a71c8bf6'
  ),
  '262e3568-c57b-4e28-a142-ad5e7b51dfb7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL GOVERNANCE/MANAGEMENT',
    variable_level_1='GEOENGINEERING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='262e3568-c57b-4e28-a142-ad5e7b51dfb7'
  ),
  '1595c0a9-63a8-433c-8515-044a977d73a7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL GOVERNANCE/MANAGEMENT',
    variable_level_1='GEOENGINEERING',
    variable_level_2='CARBON DIOXIDE REMOVAL',
    variable_level_3='',
    detailed_variable='',
    identifier='1595c0a9-63a8-433c-8515-044a977d73a7'
  ),
  '0f583845-c39e-471d-a590-8212a4358e1e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL GOVERNANCE/MANAGEMENT',
    variable_level_1='GEOENGINEERING',
    variable_level_2='SOLAR RADIATION MANAGEMENT',
    variable_level_3='',
    detailed_variable='',
    identifier='0f583845-c39e-471d-a590-8212a4358e1e'
  ),
  '2be0af28-a6b8-4fce-82e4-1ad86788a4d5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL GOVERNANCE/MANAGEMENT',
    variable_level_1='LAND MANAGEMENT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2be0af28-a6b8-4fce-82e4-1ad86788a4d5'
  ),
  '0ceb5ef1-5a07-4f93-8e86-d3cc2baf5768': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL GOVERNANCE/MANAGEMENT',
    variable_level_1='LAND MANAGEMENT',
    variable_level_2='LAND TENURE',
    variable_level_3='',
    detailed_variable='',
    identifier='0ceb5ef1-5a07-4f93-8e86-d3cc2baf5768'
  ),
  '1fd206a9-83a7-4f43-902d-003811080fed': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL GOVERNANCE/MANAGEMENT',
    variable_level_1='LAND MANAGEMENT',
    variable_level_2='LAND USE CLASSES',
    variable_level_3='',
    detailed_variable='',
    identifier='1fd206a9-83a7-4f43-902d-003811080fed'
  ),
  '5066def0-b14b-4a2c-b40f-dc9953860366': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL GOVERNANCE/MANAGEMENT',
    variable_level_1='LAND MANAGEMENT',
    variable_level_2='LAND USE/LAND COVER CLASSIFICATION',
    variable_level_3='',
    detailed_variable='',
    identifier='5066def0-b14b-4a2c-b40f-dc9953860366'
  ),
  '4dad174d-9419-4634-84f0-7eeb1d517241': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL GOVERNANCE/MANAGEMENT',
    variable_level_1='TREATY AGREEMENTS/RESULTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4dad174d-9419-4634-84f0-7eeb1d517241'
  ),
  '14555831-70ae-4650-8983-956d65595575': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL GOVERNANCE/MANAGEMENT',
    variable_level_1='WATER MANAGEMENT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='14555831-70ae-4650-8983-956d65595575'
  ),
  '96810430-e7e1-45eb-a4eb-8a7e17fe5076': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL GOVERNANCE/MANAGEMENT',
    variable_level_1='WATER MANAGEMENT',
    variable_level_2='GROUNDWATER MANAGEMENT',
    variable_level_3='',
    detailed_variable='',
    identifier='96810430-e7e1-45eb-a4eb-8a7e17fe5076'
  ),
  '873e35f5-908b-4418-861e-eab5d13a19a4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL GOVERNANCE/MANAGEMENT',
    variable_level_1='WATER MANAGEMENT',
    variable_level_2='STORMWATER MANAGEMENT',
    variable_level_3='',
    detailed_variable='',
    identifier='873e35f5-908b-4418-861e-eab5d13a19a4'
  ),
  'cbf64c32-99fa-4312-91a0-4fc85a6890bb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL GOVERNANCE/MANAGEMENT',
    variable_level_1='WATER MANAGEMENT',
    variable_level_2='WASTEWATER MANAGEMENT',
    variable_level_3='',
    detailed_variable='',
    identifier='cbf64c32-99fa-4312-91a0-4fc85a6890bb'
  ),
  'bdfa3404-c00d-45fe-a3c1-31389a831ffc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL GOVERNANCE/MANAGEMENT',
    variable_level_1='WATER MANAGEMENT',
    variable_level_2='WATER STORAGE',
    variable_level_3='',
    detailed_variable='',
    identifier='bdfa3404-c00d-45fe-a3c1-31389a831ffc'
  ),
  '73a50bfa-6a0a-4a36-ace4-2c424db05ab8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL GOVERNANCE/MANAGEMENT',
    variable_level_1='WATER MANAGEMENT',
    variable_level_2='WATER STORAGE',
    variable_level_3='ICE STUPA',
    detailed_variable='',
    identifier='73a50bfa-6a0a-4a36-ace4-2c424db05ab8'
  ),
  '3f4cfc81-7745-43d9-b313-f68cdf72359b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3f4cfc81-7745-43d9-b313-f68cdf72359b'
  ),
  'dbeff538-6857-4573-8d14-12009e0ee078': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='ACID DEPOSITION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='dbeff538-6857-4573-8d14-12009e0ee078'
  ),
  'd076e628-320a-477b-aad9-07d87ca04993': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='AGRICULTURAL EXPANSION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d076e628-320a-477b-aad9-07d87ca04993'
  ),
  'de89c42c-206a-4573-a2af-edffe5ddd6bf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='BIOCHEMICAL RELEASE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='de89c42c-206a-4573-a2af-edffe5ddd6bf'
  ),
  '9d7eed04-9c49-4024-8d0f-06474cc38bbc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='BIOMASS BURNING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9d7eed04-9c49-4024-8d0f-06474cc38bbc'
  ),
  'a5b074da-5e00-4fd9-9c40-cfec771263ee': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='CHEMICAL SPILLS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a5b074da-5e00-4fd9-9c40-cfec771263ee'
  ),
  '09cf34f3-5e20-4dc1-9b76-97afd856ebe0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='CIVIL DISTURBANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='09cf34f3-5e20-4dc1-9b76-97afd856ebe0'
  ),
  '40869a25-edea-4438-80f9-47c9e6910b9b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='CONSERVATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='40869a25-edea-4438-80f9-47c9e6910b9b'
  ),
  '912245ce-a81e-4d3b-b4fb-f71c8da63357': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='CONTAMINANT LEVELS/SPILLS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='912245ce-a81e-4d3b-b4fb-f71c8da63357'
  ),
  '69d84440-b806-4093-a659-f052185e22bd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='ELECTRIC/MAGNETIC FIELD EXPOSURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='69d84440-b806-4093-a659-f052185e22bd'
  ),
  'edfbff1e-b24b-40b9-be54-e1823b4d7f49': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='FOSSIL FUEL BURNING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='edfbff1e-b24b-40b9-be54-e1823b4d7f49'
  ),
  '083d79ba-b7fa-4a07-9c36-73540666d5c4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='GAS EXPLOSIONS/LEAKS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='083d79ba-b7fa-4a07-9c36-73540666d5c4'
  ),
  '6221fa7d-9407-4ffc-ab58-886038209254': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='GAS FLARING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6221fa7d-9407-4ffc-ab58-886038209254'
  ),
  'f03cac2e-6ee5-4734-9ab7-eb8521ba2654': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='GREENHOUSE GASES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f03cac2e-6ee5-4734-9ab7-eb8521ba2654'
  ),
  '0c4ffc6a-694d-4f33-bc18-06fefb68acdd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='HEAVY METALS CONCENTRATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0c4ffc6a-694d-4f33-bc18-06fefb68acdd'
  ),
  '07f7ea8e-cf94-4421-923a-539e12dbeb95': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='INDUSTRIAL EMISSIONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='07f7ea8e-cf94-4421-923a-539e12dbeb95'
  ),
  'c4a74ff8-0eb9-426d-b6bd-97cc951addd0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='LIQUID POLLUTANTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c4a74ff8-0eb9-426d-b6bd-97cc951addd0'
  ),
  'ed96cfbb-6fc5-4f02-baba-4c37b813c259': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='MARINE DEBRIS REMOVAL',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ed96cfbb-6fc5-4f02-baba-4c37b813c259'
  ),
  'e84f5068-2123-4594-aa07-bc1389346093': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='MECHANICAL DISTURBANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e84f5068-2123-4594-aa07-bc1389346093'
  ),
  '207a34e0-48c0-439a-a001-dcf664b61686': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='MINE DRAINAGE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='207a34e0-48c0-439a-a001-dcf664b61686'
  ),
  '48671d9e-a627-4034-baec-201bda5d166d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='NUCLEAR RADIATION EXPOSURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='48671d9e-a627-4034-baec-201bda5d166d'
  ),
  '82c3689a-6bbf-496f-b118-b6ab46a9d2c7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='OIL SPILLS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='82c3689a-6bbf-496f-b118-b6ab46a9d2c7'
  ),
  'c0fb4215-4f72-445f-af81-b3f44c44cd0e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='PRESCRIBED BURNS/FIRES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c0fb4215-4f72-445f-af81-b3f44c44cd0e'
  ),
  '835c5ec2-50e3-4bef-b380-9f74b143dac6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='SEWAGE DISPOSAL',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='835c5ec2-50e3-4bef-b380-9f74b143dac6'
  ),
  'ee3e296f-fdc0-4695-95af-dbe63f43b679': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='ENVIRONMENTAL IMPACTS',
    variable_level_1='WATER RESOURCES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ee3e296f-fdc0-4695-95af-dbe63f43b679'
  ),
  'f1682ed1-2d9c-41ec-9553-49b9ab55df9b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='HABITAT CONVERSION/FRAGMENTATION',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f1682ed1-2d9c-41ec-9553-49b9ab55df9b'
  ),
  '9a4715a7-1847-4fef-8116-494b36420fb7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='HABITAT CONVERSION/FRAGMENTATION',
    variable_level_1='DEFORESTATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9a4715a7-1847-4fef-8116-494b36420fb7'
  ),
  'dee57819-62c7-4f89-87e5-90a87a07820a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='HABITAT CONVERSION/FRAGMENTATION',
    variable_level_1='DESERTIFICATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='dee57819-62c7-4f89-87e5-90a87a07820a'
  ),
  'a66ec515-6a6e-487b-9004-2d19d6ffff04': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='HABITAT CONVERSION/FRAGMENTATION',
    variable_level_1='ECOLOGICAL CORRIDORS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a66ec515-6a6e-487b-9004-2d19d6ffff04'
  ),
  '32777aa3-a06a-4719-bbe5-7dcecb1a06f5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='HABITAT CONVERSION/FRAGMENTATION',
    variable_level_1='EUTROPHICATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='32777aa3-a06a-4719-bbe5-7dcecb1a06f5'
  ),
  '59b3849e-6704-402f-9a3e-512db10c2f51': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='HABITAT CONVERSION/FRAGMENTATION',
    variable_level_1='IRRIGATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='59b3849e-6704-402f-9a3e-512db10c2f51'
  ),
  '1ed2e238-e6dc-4a9f-a264-d52d567480f7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='HABITAT CONVERSION/FRAGMENTATION',
    variable_level_1='NIGHTTIME LIGHTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1ed2e238-e6dc-4a9f-a264-d52d567480f7'
  ),
  'aa4a9df3-0fed-4512-b158-ed369463e33a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='HABITAT CONVERSION/FRAGMENTATION',
    variable_level_1='RECLAMATION/REVEGETATION/RESTORATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='aa4a9df3-0fed-4512-b158-ed369463e33a'
  ),
  '39a032bf-c3bc-481b-9698-8be114fe85cb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='HABITAT CONVERSION/FRAGMENTATION',
    variable_level_1='REFORESTATION/REVEGETATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='39a032bf-c3bc-481b-9698-8be114fe85cb'
  ),
  'e759cacb-33f0-4564-b151-c7cfa5e85ed3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='HABITAT CONVERSION/FRAGMENTATION',
    variable_level_1='URBANIZATION/URBAN SPRAWL',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e759cacb-33f0-4564-b151-c7cfa5e85ed3'
  ),
  'fee25cad-7ffe-4ee2-a6f2-8116b8a0a707': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='HUMAN SETTLEMENTS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fee25cad-7ffe-4ee2-a6f2-8116b8a0a707'
  ),
  'bf703f22-9775-460d-86bd-149aaef1acde': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='HUMAN SETTLEMENTS',
    variable_level_1='ARCHAEOLOGICAL AREAS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bf703f22-9775-460d-86bd-149aaef1acde'
  ),
  'b1f63bf1-a547-4189-9c7e-66a8d11facc4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='HUMAN SETTLEMENTS',
    variable_level_1='COASTAL AREAS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b1f63bf1-a547-4189-9c7e-66a8d11facc4'
  ),
  'd83b4271-048c-4763-9d5c-b5ec1b1788f4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='HUMAN SETTLEMENTS',
    variable_level_1='RURAL AREAS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d83b4271-048c-4763-9d5c-b5ec1b1788f4'
  ),
  'dbb6d354-1ffe-4228-8ee4-69b2577a21f7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='HUMAN SETTLEMENTS',
    variable_level_1='RURAL AREAS',
    variable_level_2='TRAILS',
    variable_level_3='',
    detailed_variable='',
    identifier='dbb6d354-1ffe-4228-8ee4-69b2577a21f7'
  ),
  '2b4df9a9-ac03-4bdc-bee4-346045a75e05': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='HUMAN SETTLEMENTS',
    variable_level_1='TRIBAL LANDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2b4df9a9-ac03-4bdc-bee4-346045a75e05'
  ),
  'e4abd82b-b17a-4f16-be79-0093f2a09f7d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='HUMAN SETTLEMENTS',
    variable_level_1='URBAN AREAS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e4abd82b-b17a-4f16-be79-0093f2a09f7d'
  ),
  'd4313915-2d24-424c-a171-30ee9a6f4bb5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='INFRASTRUCTURE',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d4313915-2d24-424c-a171-30ee9a6f4bb5'
  ),
  'd7742082-5461-4610-9ced-e0ec3bb64697': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='INFRASTRUCTURE',
    variable_level_1='BUILDINGS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d7742082-5461-4610-9ced-e0ec3bb64697'
  ),
  'db692676-a2f6-4fd9-91b6-92ae4f9c04fd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='INFRASTRUCTURE',
    variable_level_1='COMMUNICATIONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='db692676-a2f6-4fd9-91b6-92ae4f9c04fd'
  ),
  '79b0b1d3-5279-4ce5-a387-6ecb4ee2a335': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='INFRASTRUCTURE',
    variable_level_1='CULTURAL FEATURES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='79b0b1d3-5279-4ce5-a387-6ecb4ee2a335'
  ),
  '12433114-d15a-46cf-aba9-ce4b569119ce': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='INFRASTRUCTURE',
    variable_level_1='ELECTRICITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='12433114-d15a-46cf-aba9-ce4b569119ce'
  ),
  'eeba88d2-20bf-43b1-bccf-b125485405f4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='INFRASTRUCTURE',
    variable_level_1='GREEN INFRASTRUCTURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='eeba88d2-20bf-43b1-bccf-b125485405f4'
  ),
  '50c9b6e3-4012-47c3-b0af-a140e4ba7b50': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='INFRASTRUCTURE',
    variable_level_1='OFFSHORE PLATFORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='50c9b6e3-4012-47c3-b0af-a140e4ba7b50'
  ),
  'a994de61-f16c-4b57-9dd0-b4862067203f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='INFRASTRUCTURE',
    variable_level_1='OIL RIGS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a994de61-f16c-4b57-9dd0-b4862067203f'
  ),
  'ee49d315-1fe5-42ce-a5f8-232450dfa408': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='INFRASTRUCTURE',
    variable_level_1='PIPELINES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ee49d315-1fe5-42ce-a5f8-232450dfa408'
  ),
  '37a6c8e2-f2ac-48a4-a4fa-d80f700f68db': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='INFRASTRUCTURE',
    variable_level_1='TRANSPORTATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='37a6c8e2-f2ac-48a4-a4fa-d80f700f68db'
  ),
  'f12c3e63-07ba-45dd-848f-55e96b497d27': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='INFRASTRUCTURE',
    variable_level_1='WATER CONTROL STRUCTURES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f12c3e63-07ba-45dd-848f-55e96b497d27'
  ),
  '2e372203-e14e-46cc-9db0-3b6613f61405': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='INFRASTRUCTURE',
    variable_level_1='WATER CONTROL STRUCTURES',
    variable_level_2='LEVEES',
    variable_level_3='',
    detailed_variable='',
    identifier='2e372203-e14e-46cc-9db0-3b6613f61405'
  ),
  '0692f19d-55bc-4fd7-8bf8-a0224e8715f9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='INFRASTRUCTURE',
    variable_level_1='WATER STORAGE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0692f19d-55bc-4fd7-8bf8-a0224e8715f9'
  ),
  'ec0e2762-f57a-4fdc-b395-c8d7d5590d18': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ec0e2762-f57a-4fdc-b395-c8d7d5590d18'
  ),
  'bb73336e-9113-426b-ac99-2b7c143b22ca': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='BIOLOGICAL HAZARDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bb73336e-9113-426b-ac99-2b7c143b22ca'
  ),
  '115d340f-cb5e-4436-bfa4-04a740988bf7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='DROUGHTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='115d340f-cb5e-4436-bfa4-04a740988bf7'
  ),
  'a1af404e-e108-4777-b726-7d2e068c632b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='DROUGHTS',
    variable_level_2='DROUGHT DURATION',
    variable_level_3='',
    detailed_variable='',
    identifier='a1af404e-e108-4777-b726-7d2e068c632b'
  ),
  'e54536fa-c78e-478c-893f-1da73baf2da7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='DROUGHTS',
    variable_level_2='DROUGHT FREQUENCY',
    variable_level_3='',
    detailed_variable='',
    identifier='e54536fa-c78e-478c-893f-1da73baf2da7'
  ),
  'd0a52ada-8d32-4d68-9daf-cc1c4d58aed3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='DROUGHTS',
    variable_level_2='DROUGHT SEVERITY',
    variable_level_3='',
    detailed_variable='',
    identifier='d0a52ada-8d32-4d68-9daf-cc1c4d58aed3'
  ),
  'b3406120-9faa-4c00-874e-ce8878ae129f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='EARTHQUAKES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b3406120-9faa-4c00-874e-ce8878ae129f'
  ),
  '79688737-55d1-4e38-970b-5ad150b74b34': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='EARTHQUAKES',
    variable_level_2='INDUCED EARTHQUAKES',
    variable_level_3='',
    detailed_variable='',
    identifier='79688737-55d1-4e38-970b-5ad150b74b34'
  ),
  '6bb02b3d-be70-47b0-93d7-eb0c926f5979': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='FAMINE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6bb02b3d-be70-47b0-93d7-eb0c926f5979'
  ),
  'fd03d204-4391-4e98-8142-8b8efa235231': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='FLOODS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fd03d204-4391-4e98-8142-8b8efa235231'
  ),
  'bb9c9be6-78c7-4fbd-9a35-a218276393ec': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='HEAT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bb9c9be6-78c7-4fbd-9a35-a218276393ec'
  ),
  '90edcb0a-78cc-4622-8182-4c6fe9646df4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='LAND DEFORMATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='90edcb0a-78cc-4622-8182-4c6fe9646df4'
  ),
  'ba064d3f-0327-49d2-9984-332de1a97146': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='LAND SUBSIDENCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ba064d3f-0327-49d2-9984-332de1a97146'
  ),
  'f81d3752-d97c-4caf-9a79-5709ee693158': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='LANDSLIDES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f81d3752-d97c-4caf-9a79-5709ee693158'
  ),
  'ad28623e-bb9b-433c-8fc1-2ab06dda58c4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='SEVERE STORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ad28623e-bb9b-433c-8fc1-2ab06dda58c4'
  ),
  'ff44a7b0-64b6-418a-9d74-1cbc3a4ae951': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='TORNADOES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ff44a7b0-64b6-418a-9d74-1cbc3a4ae951'
  ),
  '00fc45e0-400d-4024-a82a-4d6544735f64': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='00fc45e0-400d-4024-a82a-4d6544735f64'
  ),
  '0720043d-4d31-45ae-a37c-9ba5959bf97d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='CYCLONES',
    variable_level_3='',
    detailed_variable='',
    identifier='0720043d-4d31-45ae-a37c-9ba5959bf97d'
  ),
  '6314f68a-1f00-4e6d-9f06-b3e2ce4348e8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='HURRICANES',
    variable_level_3='',
    detailed_variable='',
    identifier='6314f68a-1f00-4e6d-9f06-b3e2ce4348e8'
  ),
  '4bee2d4d-d15d-4300-8804-626eff7ac0f3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='SEVERE CYCLONIC STORMS',
    variable_level_3='',
    detailed_variable='',
    identifier='4bee2d4d-d15d-4300-8804-626eff7ac0f3'
  ),
  '6f7996f7-5905-42e7-b9fd-c24c6328b5d9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='SEVERE TROPICAL CYCLONES',
    variable_level_3='',
    detailed_variable='',
    identifier='6f7996f7-5905-42e7-b9fd-c24c6328b5d9'
  ),
  'bd5c19e4-b25a-48b2-ad9d-4596a0ba67de': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='TROPICAL CYCLONES',
    variable_level_2='TYPHOONS',
    variable_level_3='',
    detailed_variable='',
    identifier='bd5c19e4-b25a-48b2-ad9d-4596a0ba67de'
  ),
  '768f266e-0807-49c6-a69e-c518de310331': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='TSUNAMIS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='768f266e-0807-49c6-a69e-c518de310331'
  ),
  '06c1281f-e306-4511-bdab-ed6c0694f0f9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='VOLCANIC ERUPTIONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='06c1281f-e306-4511-bdab-ed6c0694f0f9'
  ),
  '868b87a1-d8c2-49b3-8bbd-9cbbed115271': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='WILDFIRES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='868b87a1-d8c2-49b3-8bbd-9cbbed115271'
  ),
  '436b098d-e4d9-4fbd-9ede-05675e111eee': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='WILDFIRES',
    variable_level_2='BURNED AREA',
    variable_level_3='',
    detailed_variable='',
    identifier='436b098d-e4d9-4fbd-9ede-05675e111eee'
  ),
  '5e693789-87a8-4f94-9b5d-a50cecf55e24': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='NATURAL HAZARDS',
    variable_level_1='WILDFIRES',
    variable_level_2='WILDFIRE SUPPRESSION',
    variable_level_3='',
    detailed_variable='',
    identifier='5e693789-87a8-4f94-9b5d-a50cecf55e24'
  ),
  '085edf65-1c8c-414a-b8e4-a1a08ff08f22': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='POPULATION',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='085edf65-1c8c-414a-b8e4-a1a08ff08f22'
  ),
  '3fd888c4-2fd2-4ce1-8753-3158e2826ef7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='POPULATION',
    variable_level_1='MORTALITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3fd888c4-2fd2-4ce1-8753-3158e2826ef7'
  ),
  '611f0108-5706-43ca-bc39-38e528f6024b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='POPULATION',
    variable_level_1='MORTALITY',
    variable_level_2='INFANT MORTALITY RATES',
    variable_level_3='',
    detailed_variable='',
    identifier='611f0108-5706-43ca-bc39-38e528f6024b'
  ),
  '918c4136-bb4c-422b-9c15-8273307546d1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='POPULATION',
    variable_level_1='MORTALITY',
    variable_level_2='MORTALITY RATES',
    variable_level_3='',
    detailed_variable='',
    identifier='918c4136-bb4c-422b-9c15-8273307546d1'
  ),
  '9d6eda76-cf5d-4170-92ce-9ac9197832bf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='POPULATION',
    variable_level_1='NATALITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9d6eda76-cf5d-4170-92ce-9ac9197832bf'
  ),
  'd0931461-2e93-418c-b470-a218cadcf498': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='POPULATION',
    variable_level_1='NATALITY',
    variable_level_2='NATALITY RATES',
    variable_level_3='',
    detailed_variable='',
    identifier='d0931461-2e93-418c-b470-a218cadcf498'
  ),
  'ff1ace7a-7924-4198-8385-f7f7ede361c1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='POPULATION',
    variable_level_1='NATALITY',
    variable_level_2='POPULATION GENETICS/GENOMICS',
    variable_level_3='',
    detailed_variable='',
    identifier='ff1ace7a-7924-4198-8385-f7f7ede361c1'
  ),
  '9f0657f7-70d9-410f-a02e-8f36f569d1ec': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='POPULATION',
    variable_level_1='POPULATION BIOLOGY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9f0657f7-70d9-410f-a02e-8f36f569d1ec'
  ),
  'd2a5c7ec-ccf2-4ab7-8863-9063be91c022': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='POPULATION',
    variable_level_1='POPULATION DENSITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d2a5c7ec-ccf2-4ab7-8863-9063be91c022'
  ),
  'ae9f3a07-f23e-4116-b172-677435102b2f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='POPULATION',
    variable_level_1='POPULATION DISTRIBUTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ae9f3a07-f23e-4116-b172-677435102b2f'
  ),
  'd7ad5cff-75df-4bb6-92f0-b5d56da2a588': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='POPULATION',
    variable_level_1='POPULATION ESTIMATES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d7ad5cff-75df-4bb6-92f0-b5d56da2a588'
  ),
  'dd0b8bc9-90b3-4e7d-a021-e91dc676d622': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='POPULATION',
    variable_level_1='POPULATION SIZE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='dd0b8bc9-90b3-4e7d-a021-e91dc676d622'
  ),
  '35b7c7cd-49c8-476c-83f2-f2e1f4097307': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='POPULATION',
    variable_level_1='VULNERABLE POPULATIONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='35b7c7cd-49c8-476c-83f2-f2e1f4097307'
  ),
  'da2c70fd-d92b-45be-b159-b2c10cb387c6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='PUBLIC HEALTH',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='da2c70fd-d92b-45be-b159-b2c10cb387c6'
  ),
  '7d59f070-ccac-4a90-815b-edfca521779b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='PUBLIC HEALTH',
    variable_level_1='DISEASES/EPIDEMICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7d59f070-ccac-4a90-815b-edfca521779b'
  ),
  'b8a877b7-d867-4305-9053-3777e5dd330a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='PUBLIC HEALTH',
    variable_level_1='DISEASES/EPIDEMICS',
    variable_level_2='EPIDEMIOLOGY',
    variable_level_3='',
    detailed_variable='',
    identifier='b8a877b7-d867-4305-9053-3777e5dd330a'
  ),
  'd6cad59b-327e-4f3f-a664-706224c470f9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='PUBLIC HEALTH',
    variable_level_1='DISEASES/EPIDEMICS',
    variable_level_2='EPIDEMIOLOGY',
    variable_level_3='TELE-EPIDEMIOLOGY',
    detailed_variable='',
    identifier='d6cad59b-327e-4f3f-a664-706224c470f9'
  ),
  '007eeff3-1c96-4b54-aa35-2de5ebb9971a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='PUBLIC HEALTH',
    variable_level_1='DISEASES/EPIDEMICS',
    variable_level_2='FOODBORNE DISEASES',
    variable_level_3='',
    detailed_variable='',
    identifier='007eeff3-1c96-4b54-aa35-2de5ebb9971a'
  ),
  '9d92320e-b9b9-4ae8-8394-252eeda7ceb1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='PUBLIC HEALTH',
    variable_level_1='DISEASES/EPIDEMICS',
    variable_level_2='VECTOR-BORN DISEASES',
    variable_level_3='',
    detailed_variable='',
    identifier='9d92320e-b9b9-4ae8-8394-252eeda7ceb1'
  ),
  '68447296-6019-453b-9684-3cd3ff1530c9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='PUBLIC HEALTH',
    variable_level_1='DISEASES/EPIDEMICS',
    variable_level_2='WATERBORNE DISEASES',
    variable_level_3='',
    detailed_variable='',
    identifier='68447296-6019-453b-9684-3cd3ff1530c9'
  ),
  '5a47842e-785d-4cc4-b1c1-2147a9252c19': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='PUBLIC HEALTH',
    variable_level_1='ENVIRONMENTAL HEALTH FACTORS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5a47842e-785d-4cc4-b1c1-2147a9252c19'
  ),
  '6984e0a6-cb78-4f60-a31d-3ff8415e3829': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='PUBLIC HEALTH',
    variable_level_1='ENVIRONMENTAL HEALTH FACTORS',
    variable_level_2='AEROALLERGENS',
    variable_level_3='',
    detailed_variable='',
    identifier='6984e0a6-cb78-4f60-a31d-3ff8415e3829'
  ),
  '52153867-76e4-4beb-8f52-d8a69e90b9a3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='PUBLIC HEALTH',
    variable_level_1='ENVIRONMENTAL HEALTH FACTORS',
    variable_level_2='PARTICULATE MATTER (PM 1.0)',
    variable_level_3='',
    detailed_variable='',
    identifier='52153867-76e4-4beb-8f52-d8a69e90b9a3'
  ),
  '750a9b6e-4cbb-4e46-b2da-52ecd6d3a153': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='PUBLIC HEALTH',
    variable_level_1='ENVIRONMENTAL HEALTH FACTORS',
    variable_level_2='PARTICULATE MATTER (PM 10)',
    variable_level_3='',
    detailed_variable='',
    identifier='750a9b6e-4cbb-4e46-b2da-52ecd6d3a153'
  ),
  '764edcaa-d41a-4210-9b1e-f4e0f63e8329': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='PUBLIC HEALTH',
    variable_level_1='ENVIRONMENTAL HEALTH FACTORS',
    variable_level_2='PARTICULATE MATTER (PM 2.5)',
    variable_level_3='',
    detailed_variable='',
    identifier='764edcaa-d41a-4210-9b1e-f4e0f63e8329'
  ),
  '681812bd-c115-42b2-b717-f89715e89406': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='PUBLIC HEALTH',
    variable_level_1='ENVIRONMENTAL HEALTH FACTORS',
    variable_level_2='PARTICULATE MATTER CONCENTRATIONS',
    variable_level_3='',
    detailed_variable='',
    identifier='681812bd-c115-42b2-b717-f89715e89406'
  ),
  '5ce8b673-cdb9-4000-ad00-774d1c67c1b1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='PUBLIC HEALTH',
    variable_level_1='ENVIRONMENTAL HEALTH FACTORS',
    variable_level_2='Urban Heat Island',
    variable_level_3='',
    detailed_variable='',
    identifier='5ce8b673-cdb9-4000-ad00-774d1c67c1b1'
  ),
  '85a73755-cb84-40cb-a23e-2ed3811138f8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='PUBLIC HEALTH',
    variable_level_1='FOOD SECURITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='85a73755-cb84-40cb-a23e-2ed3811138f8'
  ),
  '1d1f1722-27ea-4021-922f-68b90c09bfa1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='PUBLIC HEALTH',
    variable_level_1='MALNUTRITION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1d1f1722-27ea-4021-922f-68b90c09bfa1'
  ),
  '4dcd46e9-4830-4de0-b75a-820729a6d787': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='PUBLIC HEALTH',
    variable_level_1='MALNUTRITION',
    variable_level_2='MALNUTRITION RATES',
    variable_level_3='',
    detailed_variable='',
    identifier='4dcd46e9-4830-4de0-b75a-820729a6d787'
  ),
  '3ad043a9-2ec8-401d-a727-5589a303ea4a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='PUBLIC HEALTH',
    variable_level_1='MENTAL HEALTH IMPACTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3ad043a9-2ec8-401d-a727-5589a303ea4a'
  ),
  '8a49484a-a9c8-411b-b911-7646f5323a7b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='PUBLIC HEALTH',
    variable_level_1='MORBIDITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8a49484a-a9c8-411b-b911-7646f5323a7b'
  ),
  '4b95ab99-4784-44aa-99f0-ecc677dbda65': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='PUBLIC HEALTH',
    variable_level_1='MORBIDITY',
    variable_level_2='MORBIDITY RATES',
    variable_level_3='',
    detailed_variable='',
    identifier='4b95ab99-4784-44aa-99f0-ecc677dbda65'
  ),
  '74851074-27ab-425b-9521-8d139b907b0d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='PUBLIC HEALTH',
    variable_level_1='RADIATION EXPOSURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='74851074-27ab-425b-9521-8d139b907b0d'
  ),
  '91df78a1-2e38-41d5-b88e-e235450c89fc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='PUBLIC HEALTH',
    variable_level_1='WATER TREATMENT DISRUPTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='91df78a1-2e38-41d5-b88e-e235450c89fc'
  ),
  'c8317644-4cb2-4e37-b536-c762f7e670ab': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='SOCIAL BEHAVIOR',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c8317644-4cb2-4e37-b536-c762f7e670ab'
  ),
  'aef9855c-70e1-4e22-aa25-8ccd23176d3b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='SOCIAL BEHAVIOR',
    variable_level_1='CONSERVATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='aef9855c-70e1-4e22-aa25-8ccd23176d3b'
  ),
  'd11d5e6d-fafb-4012-818c-8bfb984128f1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='SOCIAL BEHAVIOR',
    variable_level_1='CONSUMER BEHAVIOR',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d11d5e6d-fafb-4012-818c-8bfb984128f1'
  ),
  '507860e1-7494-438a-8537-b21da89efddf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='SOCIAL BEHAVIOR',
    variable_level_1='DISASTER RESPONSE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='507860e1-7494-438a-8537-b21da89efddf'
  ),
  '843a6584-e3f2-4a75-a003-cc430fd8c22c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='SOCIAL BEHAVIOR',
    variable_level_1='HAZARD MITIGATION/PLANNING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='843a6584-e3f2-4a75-a003-cc430fd8c22c'
  ),
  '859155e1-d2d3-41a3-8d44-91afa87d68b4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='SOCIAL BEHAVIOR',
    variable_level_1='PRESERVATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='859155e1-d2d3-41a3-8d44-91afa87d68b4'
  ),
  '9ee8acad-458e-45c1-a1d5-9b1649c82ea7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='SOCIAL BEHAVIOR',
    variable_level_1='RECREATIONAL ACTIVITIES/AREAS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9ee8acad-458e-45c1-a1d5-9b1649c82ea7'
  ),
  'b2f12641-19c8-4b26-9496-e79da5efcb85': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='SOCIAL BEHAVIOR',
    variable_level_1='RECYCLING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b2f12641-19c8-4b26-9496-e79da5efcb85'
  ),
  '33f20afe-5ce2-43e9-9676-c5f664fbc324': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='SOCIAL BEHAVIOR',
    variable_level_1='VULNERABILITY LEVELS/INDEX',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='33f20afe-5ce2-43e9-9676-c5f664fbc324'
  ),
  'a96e6cd6-0f35-491d-8198-7551d03e1cbc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='SOCIOECONOMICS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a96e6cd6-0f35-491d-8198-7551d03e1cbc'
  ),
  'c88a747b-2302-49c9-b747-f2faa21e2b6b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='SOCIOECONOMICS',
    variable_level_1='HOUSEHOLD INCOME',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c88a747b-2302-49c9-b747-f2faa21e2b6b'
  ),
  '92d8968b-617e-433d-ab9b-e269497c3f43': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='SOCIOECONOMICS',
    variable_level_1='INDUSTRIALIZATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='92d8968b-617e-433d-ab9b-e269497c3f43'
  ),
  'b37021a3-4d7f-4b94-b614-807d6981d2ad': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='SOCIOECONOMICS',
    variable_level_1='POVERTY LEVELS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b37021a3-4d7f-4b94-b614-807d6981d2ad'
  ),
  '2bf46486-3004-447e-b2c6-82c4aa13fc11': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='SOCIOECONOMICS',
    variable_level_1='PURCHASING POWER',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2bf46486-3004-447e-b2c6-82c4aa13fc11'
  ),
  '03d38261-1c90-491b-bc4e-cc4e703e1dff': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='SUSTAINABILITY',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='03d38261-1c90-491b-bc4e-cc4e703e1dff'
  ),
  '73266dd6-217a-432f-9237-176d3e94b39b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='SUSTAINABILITY',
    variable_level_1='ENVIRONMENTAL SUSTAINABILITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='73266dd6-217a-432f-9237-176d3e94b39b'
  ),
  '8d11c81c-ff5b-4cc0-9be2-8e73dddcb51b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='HUMAN DIMENSIONS',
    term='SUSTAINABILITY',
    variable_level_1='SUSTAINABLE DEVELOPMENT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8d11c81c-ff5b-4cc0-9be2-8e73dddcb51b'
  ),
  '6a426480-c58f-4b6b-8e35-0975b7f6edb5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6a426480-c58f-4b6b-8e35-0975b7f6edb5'
  ),
  'a246a8cf-e3f9-4045-af9f-dc97f6fe019a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='EROSION/SEDIMENTATION',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a246a8cf-e3f9-4045-af9f-dc97f6fe019a'
  ),
  '2f2f5764-d4e6-4bbb-bd6d-dda373018237': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='EROSION/SEDIMENTATION',
    variable_level_1='DEGRADATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2f2f5764-d4e6-4bbb-bd6d-dda373018237'
  ),
  'f6a5cc87-a333-4e99-88d4-bf7b5b1cf484': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='EROSION/SEDIMENTATION',
    variable_level_1='ENTRAINMENT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f6a5cc87-a333-4e99-88d4-bf7b5b1cf484'
  ),
  '1e2b1b67-a401-4fb6-9ee9-b022c1c023dc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='EROSION/SEDIMENTATION',
    variable_level_1='EROSION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1e2b1b67-a401-4fb6-9ee9-b022c1c023dc'
  ),
  'ea4aefeb-64cd-4408-83d8-8e0a672739b9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='EROSION/SEDIMENTATION',
    variable_level_1='LANDSLIDES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ea4aefeb-64cd-4408-83d8-8e0a672739b9'
  ),
  'ca2ffcd6-39e6-4eab-abc2-07eb4a197e3d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='EROSION/SEDIMENTATION',
    variable_level_1='SEDIMENT CHEMISTRY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ca2ffcd6-39e6-4eab-abc2-07eb4a197e3d'
  ),
  '807aff1a-6fe0-474a-a025-a0d0d8b17dbd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='EROSION/SEDIMENTATION',
    variable_level_1='SEDIMENT COMPOSITION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='807aff1a-6fe0-474a-a025-a0d0d8b17dbd'
  ),
  'e4ad5a76-7540-4433-ad82-9fe89259538b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='EROSION/SEDIMENTATION',
    variable_level_1='SEDIMENT TRANSPORT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e4ad5a76-7540-4433-ad82-9fe89259538b'
  ),
  'b41498cd-6b2b-47e3-afe7-0f05b4c0807d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='EROSION/SEDIMENTATION',
    variable_level_1='SEDIMENTATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b41498cd-6b2b-47e3-afe7-0f05b4c0807d'
  ),
  '26558c08-beca-4ef0-9ea3-b000504ece60': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='EROSION/SEDIMENTATION',
    variable_level_1='SEDIMENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='26558c08-beca-4ef0-9ea3-b000504ece60'
  ),
  '42b5ae5b-90d5-44f0-b331-6c22cdd45c3f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='EROSION/SEDIMENTATION',
    variable_level_1='STRATIGRAPHIC SEQUENCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='42b5ae5b-90d5-44f0-b331-6c22cdd45c3f'
  ),
  '69ff701a-674e-4b63-bb93-6ebe6cd95281': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='EROSION/SEDIMENTATION',
    variable_level_1='SUSPENDED SOLIDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='69ff701a-674e-4b63-bb93-6ebe6cd95281'
  ),
  'b07017e8-d714-45a6-b1fe-8c00230ec209': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='EROSION/SEDIMENTATION',
    variable_level_1='WEATHERING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b07017e8-d714-45a6-b1fe-8c00230ec209'
  ),
  '8073b62d-a2f3-4ad9-b619-de26f28877a7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='FROZEN GROUND',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8073b62d-a2f3-4ad9-b619-de26f28877a7'
  ),
  '10270ee0-8d85-4c75-9fa2-49e7a9755cb3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='FROZEN GROUND',
    variable_level_1='ACTIVE LAYER',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='10270ee0-8d85-4c75-9fa2-49e7a9755cb3'
  ),
  '1469f30b-6eb4-4186-b1ef-7dd25c34c592': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='FROZEN GROUND',
    variable_level_1='CRYOSOLS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1469f30b-6eb4-4186-b1ef-7dd25c34c592'
  ),
  '5b4dde5a-733e-4e55-97c9-2108b337cfeb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='FROZEN GROUND',
    variable_level_1='GROUND ICE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5b4dde5a-733e-4e55-97c9-2108b337cfeb'
  ),
  '5181e50a-b1d2-41b1-bde3-fd9b4da9b1bf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='FROZEN GROUND',
    variable_level_1='PERIGLACIAL PROCESSES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5181e50a-b1d2-41b1-bde3-fd9b4da9b1bf'
  ),
  'b6723314-3db7-4bdd-85ee-0b8507e6ae1b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='FROZEN GROUND',
    variable_level_1='PERMAFROST',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b6723314-3db7-4bdd-85ee-0b8507e6ae1b'
  ),
  'ee2af62b-9f76-440c-aa9b-77940468b8f4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='FROZEN GROUND',
    variable_level_1='ROCK GLACIERS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ee2af62b-9f76-440c-aa9b-77940468b8f4'
  ),
  '6fdd8021-3f6f-4f54-829c-26f744597309': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='FROZEN GROUND',
    variable_level_1='SEASONALLY FROZEN GROUND',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6fdd8021-3f6f-4f54-829c-26f744597309'
  ),
  '240ff021-6a9c-4603-983d-f135ee7e49ab': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='FROZEN GROUND',
    variable_level_1='SOIL TEMPERATURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='240ff021-6a9c-4603-983d-f135ee7e49ab'
  ),
  'c39710ae-423f-44c8-b969-9af8a1f912cf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='FROZEN GROUND',
    variable_level_1='TALIK',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c39710ae-423f-44c8-b969-9af8a1f912cf'
  ),
  'd35b9ba5-d018-48a5-8f0d-92b9c55b3279': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d35b9ba5-d018-48a5-8f0d-92b9c55b3279'
  ),
  'ed75fb8f-cb96-448e-ada5-dc48fbd0ebb1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN LANDFORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ed75fb8f-cb96-448e-ada5-dc48fbd0ebb1'
  ),
  '416221ec-04e1-4913-aacb-9045551949c4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='',
    detailed_variable='',
    identifier='416221ec-04e1-4913-aacb-9045551949c4'
  ),
  'f51acce1-eaf6-4de7-b279-5b58c3034aeb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='CRESCENTIC (BARCHAN/TRANSVERSE) DUNE',
    detailed_variable='',
    identifier='f51acce1-eaf6-4de7-b279-5b58c3034aeb'
  ),
  'cb6b9191-21ab-4a56-b43f-27e86f90f6d9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='DOME DUNE',
    detailed_variable='',
    identifier='cb6b9191-21ab-4a56-b43f-27e86f90f6d9'
  ),
  'b5ee3496-6910-4971-8539-5aa084bfa9e1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='LONGITUDINAL/LINEAR DUNE',
    detailed_variable='',
    identifier='b5ee3496-6910-4971-8539-5aa084bfa9e1'
  ),
  'c63be844-efa7-49f6-8089-c60111bbdffb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='PARABOLIC DUNE',
    detailed_variable='',
    identifier='c63be844-efa7-49f6-8089-c60111bbdffb'
  ),
  'dc9dea65-e574-4bbb-9945-cd6d1cdbf6c1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='STAR DUNE',
    detailed_variable='',
    identifier='dc9dea65-e574-4bbb-9945-cd6d1cdbf6c1'
  ),
  '1376f8a1-84f2-4797-a978-69ec520e2423': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN LANDFORMS',
    variable_level_2='RIPPLES',
    variable_level_3='',
    detailed_variable='',
    identifier='1376f8a1-84f2-4797-a978-69ec520e2423'
  ),
  '3ab3aa92-9cca-4660-a0ed-281fff07eede': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3ab3aa92-9cca-4660-a0ed-281fff07eede'
  ),
  'eb039da2-8af7-4d31-9ec9-0700251cfd5d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='ABRASION',
    variable_level_3='',
    detailed_variable='',
    identifier='eb039da2-8af7-4d31-9ec9-0700251cfd5d'
  ),
  'dec3d35a-3ffa-4bea-b239-f8e74b498fb2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='ABRASION',
    variable_level_3='VENTIFACTS',
    detailed_variable='',
    identifier='dec3d35a-3ffa-4bea-b239-f8e74b498fb2'
  ),
  'dabc0fc5-acac-48df-b32e-02c9166e8385': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='ABRASION',
    variable_level_3='YARDANGS',
    detailed_variable='',
    identifier='dabc0fc5-acac-48df-b32e-02c9166e8385'
  ),
  '03ea18fe-793d-48e0-aa44-e211376c73d8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='DEFLATION',
    variable_level_3='',
    detailed_variable='',
    identifier='03ea18fe-793d-48e0-aa44-e211376c73d8'
  ),
  '32f6083c-f6a2-40cf-8cf4-782b02b9df9e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='DEGRADATION',
    variable_level_3='',
    detailed_variable='',
    identifier='32f6083c-f6a2-40cf-8cf4-782b02b9df9e'
  ),
  'cfaf76ef-89e2-4dc8-a4eb-b3308ef8c52c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='SALTATION',
    variable_level_3='',
    detailed_variable='',
    identifier='cfaf76ef-89e2-4dc8-a4eb-b3308ef8c52c'
  ),
  '0b5e5a9b-5552-4e41-b1a1-9c01c52dff4b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='SEDIMENT TRANSPORT',
    variable_level_3='',
    detailed_variable='',
    identifier='0b5e5a9b-5552-4e41-b1a1-9c01c52dff4b'
  ),
  '5ce16b97-c91c-420c-9701-33d19d50b286': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='SEDIMENT TRANSPORT',
    variable_level_3='LOESS',
    detailed_variable='',
    identifier='5ce16b97-c91c-420c-9701-33d19d50b286'
  ),
  'abe3a81a-3bac-450b-8006-304bee055289': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='SEDIMENT TRANSPORT',
    variable_level_3='MONADNOCK',
    detailed_variable='',
    identifier='abe3a81a-3bac-450b-8006-304bee055289'
  ),
  '5fff607c-5df4-4f06-a541-896f7cbc1e4c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='',
    detailed_variable='',
    identifier='5fff607c-5df4-4f06-a541-896f7cbc1e4c'
  ),
  '9ea3e92d-f772-4f39-a615-08b0e062ee9d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='SEDIMENT CHEMISTRY',
    detailed_variable='',
    identifier='9ea3e92d-f772-4f39-a615-08b0e062ee9d'
  ),
  '02916754-4814-48ea-b8fc-ef50d7a7c5b5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='SEDIMENT COMPOSITION',
    detailed_variable='',
    identifier='02916754-4814-48ea-b8fc-ef50d7a7c5b5'
  ),
  'a7b04d56-2a44-4a94-8d94-1911a7110f9d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='STRATIGRAPHIC SEQUENCE',
    detailed_variable='',
    identifier='a7b04d56-2a44-4a94-8d94-1911a7110f9d'
  ),
  '82eac236-38ba-469d-837a-950ffa7e8316': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='WEATHERING',
    variable_level_3='',
    detailed_variable='',
    identifier='82eac236-38ba-469d-837a-950ffa7e8316'
  ),
  '0cff6e4b-e42a-4565-89ff-350adf41ed69': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0cff6e4b-e42a-4565-89ff-350adf41ed69'
  ),
  '128db882-0522-4a5e-ac69-81d05986a645': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='BARRIER ISLANDS',
    variable_level_3='',
    detailed_variable='',
    identifier='128db882-0522-4a5e-ac69-81d05986a645'
  ),
  '68b4238d-c10f-4f59-9c23-820563107d12': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='BEACHES',
    variable_level_3='',
    detailed_variable='',
    identifier='68b4238d-c10f-4f59-9c23-820563107d12'
  ),
  'c6244bfb-300f-4818-bf45-cf1a15e7e073': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='CORAL REEFS',
    variable_level_3='',
    detailed_variable='',
    identifier='c6244bfb-300f-4818-bf45-cf1a15e7e073'
  ),
  '6c78ed6a-2dbc-4ced-acc2-d0246e0afedd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='CORAL REEFS',
    variable_level_3='APRON REEF',
    detailed_variable='',
    identifier='6c78ed6a-2dbc-4ced-acc2-d0246e0afedd'
  ),
  '8bbf1177-c74b-4f11-8f7d-40c5785312a1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='CORAL REEFS',
    variable_level_3='ATOLL REEF',
    detailed_variable='',
    identifier='8bbf1177-c74b-4f11-8f7d-40c5785312a1'
  ),
  '9ea0dbd4-2af5-4520-a831-32ee04d02ecc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='CORAL REEFS',
    variable_level_3='BANK REEF',
    detailed_variable='',
    identifier='9ea0dbd4-2af5-4520-a831-32ee04d02ecc'
  ),
  'e125e285-b547-47ea-b566-5dffce2bbcbd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='CORAL REEFS',
    variable_level_3='BARRIER REEF',
    detailed_variable='',
    identifier='e125e285-b547-47ea-b566-5dffce2bbcbd'
  ),
  'b54234a2-3261-4c6e-8fd8-75230f3488c0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='CORAL REEFS',
    variable_level_3='FRINGING REEF',
    detailed_variable='',
    identifier='b54234a2-3261-4c6e-8fd8-75230f3488c0'
  ),
  'a722fea3-fe54-4995-8aec-407efe20dee9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='CORAL REEFS',
    variable_level_3='PATCH REEF',
    detailed_variable='',
    identifier='a722fea3-fe54-4995-8aec-407efe20dee9'
  ),
  '7f674559-6e36-4a13-ac0c-f61aa6a37d63': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='CORAL REEFS',
    variable_level_3='RIBBON REEF',
    detailed_variable='',
    identifier='7f674559-6e36-4a13-ac0c-f61aa6a37d63'
  ),
  'ec0692d8-1cce-4c89-a6ef-c35a5f812121': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='CORAL REEFS',
    variable_level_3='TABLE REEF',
    detailed_variable='',
    identifier='ec0692d8-1cce-4c89-a6ef-c35a5f812121'
  ),
  '11175bd5-ee63-4b13-aa03-bc5500a458c2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='CUSPATE FORELANDS',
    variable_level_3='',
    detailed_variable='',
    identifier='11175bd5-ee63-4b13-aa03-bc5500a458c2'
  ),
  '93647a7c-a881-4066-a696-c19053c7c30b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='DELTAS',
    variable_level_3='',
    detailed_variable='',
    identifier='93647a7c-a881-4066-a696-c19053c7c30b'
  ),
  '362993fc-743e-42bc-a011-459baea8f427': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='',
    detailed_variable='',
    identifier='362993fc-743e-42bc-a011-459baea8f427'
  ),
  'f8b39934-bdce-4f90-8b86-a001c0af8b76': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='CRESCENTIC (BARCHAN/TRANSVERSE)',
    detailed_variable='',
    identifier='f8b39934-bdce-4f90-8b86-a001c0af8b76'
  ),
  '8971f15d-bee3-4eaf-a7dd-ceb005448b37': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='DOME DUNE',
    detailed_variable='',
    identifier='8971f15d-bee3-4eaf-a7dd-ceb005448b37'
  ),
  '386f4f36-26bd-4193-aa25-0c0ec2e5baae': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='LONGITUDINAL/LINEAR DUNE',
    detailed_variable='',
    identifier='386f4f36-26bd-4193-aa25-0c0ec2e5baae'
  ),
  'db89fdd2-d911-4a75-9210-ce90db043358': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='PARABOLIC DUNE',
    detailed_variable='',
    identifier='db89fdd2-d911-4a75-9210-ce90db043358'
  ),
  '5a271522-fee4-4646-9c9a-a99385f00d9f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='STAR DUNE',
    detailed_variable='',
    identifier='5a271522-fee4-4646-9c9a-a99385f00d9f'
  ),
  '8d634619-aed2-4326-a73d-cec49ff74398': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='ESTUARIES',
    variable_level_3='',
    detailed_variable='',
    identifier='8d634619-aed2-4326-a73d-cec49ff74398'
  ),
  '7299f45f-eafb-4ed9-ae12-5e01c97c1530': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='FJORDS',
    variable_level_3='',
    detailed_variable='',
    identifier='7299f45f-eafb-4ed9-ae12-5e01c97c1530'
  ),
  '153080e1-2ab1-438a-8f1e-0cb6d5fe1242': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='HEADLANDS/BAYS/CAPE',
    variable_level_3='',
    detailed_variable='',
    identifier='153080e1-2ab1-438a-8f1e-0cb6d5fe1242'
  ),
  '49db8758-1282-45a0-ad3f-0f1e9d8abc44': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='INLETS',
    variable_level_3='',
    detailed_variable='',
    identifier='49db8758-1282-45a0-ad3f-0f1e9d8abc44'
  ),
  'e069e3fc-0c75-40ee-92d7-595991f8fdb4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='ISTHMUS',
    variable_level_3='',
    detailed_variable='',
    identifier='e069e3fc-0c75-40ee-92d7-595991f8fdb4'
  ),
  'd9483208-ff59-4293-9867-3f4895e58c9f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='LAGOONS',
    variable_level_3='',
    detailed_variable='',
    identifier='d9483208-ff59-4293-9867-3f4895e58c9f'
  ),
  '1e7daefd-fa73-4561-90cc-478ca37bcb9a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='RIA',
    variable_level_3='',
    detailed_variable='',
    identifier='1e7daefd-fa73-4561-90cc-478ca37bcb9a'
  ),
  'd541e4e1-2542-4716-b943-e080b0865e74': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='SALT MARSH',
    variable_level_3='',
    detailed_variable='',
    identifier='d541e4e1-2542-4716-b943-e080b0865e74'
  ),
  '575d5577-3107-4192-83a3-5a28ceea7a5d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='SEA ARCHES',
    variable_level_3='',
    detailed_variable='',
    identifier='575d5577-3107-4192-83a3-5a28ceea7a5d'
  ),
  'c702cd1d-48dd-4652-9ec6-cff6ff52b430': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='SEA CAVES',
    variable_level_3='',
    detailed_variable='',
    identifier='c702cd1d-48dd-4652-9ec6-cff6ff52b430'
  ),
  'a82477e6-b563-4135-90e8-c6977c7381be': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='SEA CLIFFS',
    variable_level_3='',
    detailed_variable='',
    identifier='a82477e6-b563-4135-90e8-c6977c7381be'
  ),
  'cae5bafd-10a7-4bcf-af1a-3e187ee5e955': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='SHOALS',
    variable_level_3='',
    detailed_variable='',
    identifier='cae5bafd-10a7-4bcf-af1a-3e187ee5e955'
  ),
  '4e5cf935-cf17-4947-bd1f-6816a855953a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='SHORELINES',
    variable_level_3='',
    detailed_variable='',
    identifier='4e5cf935-cf17-4947-bd1f-6816a855953a'
  ),
  '1815faf3-2411-4d2a-a3d5-1e5b0c50782b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='SOUNDS',
    variable_level_3='',
    detailed_variable='',
    identifier='1815faf3-2411-4d2a-a3d5-1e5b0c50782b'
  ),
  '4f25c039-56b9-47a9-9232-d80860da5990': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='SPITS AND BARS',
    variable_level_3='',
    detailed_variable='',
    identifier='4f25c039-56b9-47a9-9232-d80860da5990'
  ),
  '320e14a6-4882-4533-b1cf-55d49c8a6b37': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='TOMBOLOS',
    variable_level_3='',
    detailed_variable='',
    identifier='320e14a6-4882-4533-b1cf-55d49c8a6b37'
  ),
  '0c523ed2-d02e-4b02-bf21-2da4e171c959': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='WAVE-CUT NOTCH/PLATFORMS',
    variable_level_3='',
    detailed_variable='',
    identifier='0c523ed2-d02e-4b02-bf21-2da4e171c959'
  ),
  'e26803a0-82ea-40c4-a41a-9e222c9bd09a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e26803a0-82ea-40c4-a41a-9e222c9bd09a'
  ),
  '9ca8db82-9230-42e0-ad91-9068bc144855': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='ABRASION',
    variable_level_3='',
    detailed_variable='',
    identifier='9ca8db82-9230-42e0-ad91-9068bc144855'
  ),
  'ee016b0b-353b-4811-bfc2-5d32aed59f29': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='ACCRETION',
    variable_level_3='',
    detailed_variable='',
    identifier='ee016b0b-353b-4811-bfc2-5d32aed59f29'
  ),
  '4dee8110-972f-4665-bd28-a9e64de21a16': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='ATTRITION/WEATHERING',
    variable_level_3='',
    detailed_variable='',
    identifier='4dee8110-972f-4665-bd28-a9e64de21a16'
  ),
  '1e73dd30-abb5-4723-be3e-71706d2b1ea1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='CHEMICAL SOLUTION',
    variable_level_3='',
    detailed_variable='',
    identifier='1e73dd30-abb5-4723-be3e-71706d2b1ea1'
  ),
  '8b99d6c3-1751-43e6-81d1-92a7618cadb3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='DEPOSITION',
    variable_level_3='',
    detailed_variable='',
    identifier='8b99d6c3-1751-43e6-81d1-92a7618cadb3'
  ),
  'a2401a77-908f-4c03-abcc-d27d99586967': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='FLOODING',
    variable_level_3='',
    detailed_variable='',
    identifier='a2401a77-908f-4c03-abcc-d27d99586967'
  ),
  'ebb8ef06-0f73-48eb-bc22-47f36a729bc6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='HYDRAULIC ACTION',
    variable_level_3='',
    detailed_variable='',
    identifier='ebb8ef06-0f73-48eb-bc22-47f36a729bc6'
  ),
  'c344fddc-ffa8-4093-bcf5-bcfe7806c737': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='SALTATION',
    variable_level_3='',
    detailed_variable='',
    identifier='c344fddc-ffa8-4093-bcf5-bcfe7806c737'
  ),
  '2aad0e7e-1c96-4d87-adeb-4894225e2922': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='SEA LEVEL CHANGES',
    variable_level_3='',
    detailed_variable='',
    identifier='2aad0e7e-1c96-4d87-adeb-4894225e2922'
  ),
  '5e101ced-5d9f-4733-8768-38db92d83660': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='SEDIMENT TRANSPORT',
    variable_level_3='',
    detailed_variable='',
    identifier='5e101ced-5d9f-4733-8768-38db92d83660'
  ),
  '866aa07b-132c-4b93-9ced-d74b56b3016f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='',
    detailed_variable='',
    identifier='866aa07b-132c-4b93-9ced-d74b56b3016f'
  ),
  '786c08f1-f3ed-4edd-8ec9-a69313906426': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='SEDIMENT CHEMISTRY',
    detailed_variable='',
    identifier='786c08f1-f3ed-4edd-8ec9-a69313906426'
  ),
  '40d7f7e1-e11a-410b-a1b6-78c4a961d631': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='SEDIMENT COMPOSITION',
    detailed_variable='',
    identifier='40d7f7e1-e11a-410b-a1b6-78c4a961d631'
  ),
  '870803f5-8bbc-4e39-8372-ce21b0decb75': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='STRATIGRAPHIC SEQUENCE',
    detailed_variable='',
    identifier='870803f5-8bbc-4e39-8372-ce21b0decb75'
  ),
  '43b2798d-32ac-497f-8881-98d52422e3ac': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='SUBMERGENCE',
    variable_level_3='',
    detailed_variable='',
    identifier='43b2798d-32ac-497f-8881-98d52422e3ac'
  ),
  'af017320-085a-4e6c-81c2-38056cb55c7b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='SUBSIDENCE',
    variable_level_3='',
    detailed_variable='',
    identifier='af017320-085a-4e6c-81c2-38056cb55c7b'
  ),
  '62ecfc64-48d0-4373-9c44-599471703cf4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='SUSPENSION',
    variable_level_3='',
    detailed_variable='',
    identifier='62ecfc64-48d0-4373-9c44-599471703cf4'
  ),
  '406bfa8b-8522-4776-936a-1fda8b0cfe97': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='WAVE BREAKING',
    variable_level_3='',
    detailed_variable='',
    identifier='406bfa8b-8522-4776-936a-1fda8b0cfe97'
  ),
  'ab0138b8-6939-4ac1-aa5f-36073d52360b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='WAVE DIFFRACTION',
    variable_level_3='',
    detailed_variable='',
    identifier='ab0138b8-6939-4ac1-aa5f-36073d52360b'
  ),
  '097a7f54-df6e-4aeb-8d15-65d4bd24da64': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='WAVE EROSION',
    variable_level_3='',
    detailed_variable='',
    identifier='097a7f54-df6e-4aeb-8d15-65d4bd24da64'
  ),
  'df4a0112-3aba-41ca-816a-86129cacb6a5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='WAVE REFRACTION',
    variable_level_3='',
    detailed_variable='',
    identifier='df4a0112-3aba-41ca-816a-86129cacb6a5'
  ),
  '15d5b23c-f739-43b5-bf14-3063c0a59f2d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='WAVE SHOALING',
    variable_level_3='',
    detailed_variable='',
    identifier='15d5b23c-f739-43b5-bf14-3063c0a59f2d'
  ),
  'bdc0bd86-a3a3-48fa-b1fb-4ca5d13d4dde': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bdc0bd86-a3a3-48fa-b1fb-4ca5d13d4dde'
  ),
  '3207af29-bd29-4f85-9a08-2614579dd27f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='AIT',
    variable_level_3='',
    detailed_variable='',
    identifier='3207af29-bd29-4f85-9a08-2614579dd27f'
  ),
  'ff850d62-675c-4386-a375-fe4af92ec3ff': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='BAR',
    variable_level_3='',
    detailed_variable='',
    identifier='ff850d62-675c-4386-a375-fe4af92ec3ff'
  ),
  'a6fdb3c7-a0ea-4f7c-82e5-d72db09b6444': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='BAYOU',
    variable_level_3='',
    detailed_variable='',
    identifier='a6fdb3c7-a0ea-4f7c-82e5-d72db09b6444'
  ),
  'a78c946a-9529-4643-b002-1aa2ac9cfed6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='CANYON',
    variable_level_3='',
    detailed_variable='',
    identifier='a78c946a-9529-4643-b002-1aa2ac9cfed6'
  ),
  'e18c970a-d0c6-4430-b419-64cf718bc456': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='CONFLUENCE',
    variable_level_3='',
    detailed_variable='',
    identifier='e18c970a-d0c6-4430-b419-64cf718bc456'
  ),
  '47ea7cc6-2816-4d64-ad41-6ac1d11c2a33': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='CUTBANK',
    variable_level_3='',
    detailed_variable='',
    identifier='47ea7cc6-2816-4d64-ad41-6ac1d11c2a33'
  ),
  'daa297ec-4397-4caa-b563-634a71f62b8a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='DELTAS',
    variable_level_3='',
    detailed_variable='',
    identifier='daa297ec-4397-4caa-b563-634a71f62b8a'
  ),
  '1afe698e-d920-4756-8de4-482d2ef15a24': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='ENDORHEIC BASIN',
    variable_level_3='',
    detailed_variable='',
    identifier='1afe698e-d920-4756-8de4-482d2ef15a24'
  ),
  'ba37314d-ec38-4e67-bf30-7e1fdc6bfbad': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='FLOOD PLAIN',
    variable_level_3='',
    detailed_variable='',
    identifier='ba37314d-ec38-4e67-bf30-7e1fdc6bfbad'
  ),
  'b9b85df8-3b95-4baf-bd32-8bacd35dc9b5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='GULLY',
    variable_level_3='',
    detailed_variable='',
    identifier='b9b85df8-3b95-4baf-bd32-8bacd35dc9b5'
  ),
  '9d078d5c-62cb-46b5-a6f5-43678643a0ce': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='ISLAND',
    variable_level_3='',
    detailed_variable='',
    identifier='9d078d5c-62cb-46b5-a6f5-43678643a0ce'
  ),
  '8ca51b5e-0b7a-4b7a-b7e2-6e163e195e26': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='LACUSTRINE PLAIN',
    variable_level_3='',
    detailed_variable='',
    identifier='8ca51b5e-0b7a-4b7a-b7e2-6e163e195e26'
  ),
  '4c09d43f-68d5-469d-aaed-f9ef8968ef2e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='MARSH',
    variable_level_3='',
    detailed_variable='',
    identifier='4c09d43f-68d5-469d-aaed-f9ef8968ef2e'
  ),
  'b976b8e5-01b9-4bb3-ba4c-308f8fa0fb97': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='MEANDER',
    variable_level_3='',
    detailed_variable='',
    identifier='b976b8e5-01b9-4bb3-ba4c-308f8fa0fb97'
  ),
  'c5a9eb49-93c4-4fb5-9a02-5fa06ea8800f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='OX-BOW LAKE',
    variable_level_3='',
    detailed_variable='',
    identifier='c5a9eb49-93c4-4fb5-9a02-5fa06ea8800f'
  ),
  '71b4e773-40a4-47db-8449-7c13f4cc49d9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='PINGO',
    variable_level_3='',
    detailed_variable='',
    identifier='71b4e773-40a4-47db-8449-7c13f4cc49d9'
  ),
  '00e49fcb-d846-4b4e-8f45-b022c1713920': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='POINT BAR',
    variable_level_3='',
    detailed_variable='',
    identifier='00e49fcb-d846-4b4e-8f45-b022c1713920'
  ),
  '89228e69-5a64-4662-839f-cb3d2209fa41': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='POND',
    variable_level_3='',
    detailed_variable='',
    identifier='89228e69-5a64-4662-839f-cb3d2209fa41'
  ),
  'af058c0e-d40b-4e59-9f92-67b59fd1e2bd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='RIFFLE',
    variable_level_3='',
    detailed_variable='',
    identifier='af058c0e-d40b-4e59-9f92-67b59fd1e2bd'
  ),
  '87624706-e11f-4043-ac54-479ed94b8dac': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='RIVER',
    variable_level_3='',
    detailed_variable='',
    identifier='87624706-e11f-4043-ac54-479ed94b8dac'
  ),
  '620a9e6c-5851-48b7-93c5-a1706546f5d1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='SPRING',
    variable_level_3='',
    detailed_variable='',
    identifier='620a9e6c-5851-48b7-93c5-a1706546f5d1'
  ),
  '01a84bc1-a571-4d23-b57f-1b04fd9542a6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='STREAM',
    variable_level_3='',
    detailed_variable='',
    identifier='01a84bc1-a571-4d23-b57f-1b04fd9542a6'
  ),
  'd1964724-2481-417a-be5a-e0dedb111ab4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='STREAM TERRACE',
    variable_level_3='',
    detailed_variable='',
    identifier='d1964724-2481-417a-be5a-e0dedb111ab4'
  ),
  '8f6adff6-672d-4066-8c85-25418a7d0e00': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='SWAMP',
    variable_level_3='',
    detailed_variable='',
    identifier='8f6adff6-672d-4066-8c85-25418a7d0e00'
  ),
  'f4f9c238-2d7e-4529-944b-52389c13932c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='VALLEY',
    variable_level_3='',
    detailed_variable='',
    identifier='f4f9c238-2d7e-4529-944b-52389c13932c'
  ),
  '186a08ed-b6fc-4963-adb6-2c5113d133e5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='VALLEY',
    variable_level_3='V SHAPED VALLEY',
    detailed_variable='',
    identifier='186a08ed-b6fc-4963-adb6-2c5113d133e5'
  ),
  '97c6eb84-90a8-4b47-9a22-99c7c1369989': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='WATERFALL',
    variable_level_3='',
    detailed_variable='',
    identifier='97c6eb84-90a8-4b47-9a22-99c7c1369989'
  ),
  'feceb3aa-d3b4-49e0-ad85-3275acd604fb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='WATERSHED/DRAINAGE BASIN',
    variable_level_3='',
    detailed_variable='',
    identifier='feceb3aa-d3b4-49e0-ad85-3275acd604fb'
  ),
  '4b982bef-56fe-41e9-a131-af575a8fec6a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4b982bef-56fe-41e9-a131-af575a8fec6a'
  ),
  '6ae0d1f7-cc99-4da7-8446-e2dca16f546b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='ABRASION',
    variable_level_3='',
    detailed_variable='',
    identifier='6ae0d1f7-cc99-4da7-8446-e2dca16f546b'
  ),
  'b655ca30-361c-4434-a784-68b8ab99668d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='ATTRITION',
    variable_level_3='',
    detailed_variable='',
    identifier='b655ca30-361c-4434-a784-68b8ab99668d'
  ),
  'ae368822-4979-4feb-967b-ee7764639646': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='DEGRADATION',
    variable_level_3='',
    detailed_variable='',
    identifier='ae368822-4979-4feb-967b-ee7764639646'
  ),
  '0bb77741-598a-4f8c-8ce9-5aa0d61a0906': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='DOWNCUTTING',
    variable_level_3='',
    detailed_variable='',
    identifier='0bb77741-598a-4f8c-8ce9-5aa0d61a0906'
  ),
  '3288c7e0-20fa-4e05-80fa-cdb14c436c7e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='ENTRAINMENT',
    variable_level_3='',
    detailed_variable='',
    identifier='3288c7e0-20fa-4e05-80fa-cdb14c436c7e'
  ),
  'b13bf33a-2a06-489f-80ca-0c77b08588ec': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='HYDRAULIC ACTION',
    variable_level_3='',
    detailed_variable='',
    identifier='b13bf33a-2a06-489f-80ca-0c77b08588ec'
  ),
  'c09cb9dc-2916-40f0-9f2f-4bbb39d2e7c9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='LANDSLIDES',
    variable_level_3='',
    detailed_variable='',
    identifier='c09cb9dc-2916-40f0-9f2f-4bbb39d2e7c9'
  ),
  'e8ba38ce-fc48-44b7-8b78-03b69e068d46': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='SALTATION',
    variable_level_3='',
    detailed_variable='',
    identifier='e8ba38ce-fc48-44b7-8b78-03b69e068d46'
  ),
  '4a031bdf-a6c6-40b8-9c92-34cd83a5739e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='SEDIMENT TRANSPORT',
    variable_level_3='',
    detailed_variable='',
    identifier='4a031bdf-a6c6-40b8-9c92-34cd83a5739e'
  ),
  'b1de8d2f-cfe6-4358-a4cd-5b7e19d0e585': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='',
    detailed_variable='',
    identifier='b1de8d2f-cfe6-4358-a4cd-5b7e19d0e585'
  ),
  'ba0630cb-9a7e-4c4b-9675-c92aba7088ce': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='SEDIMENT CHEMISTRY',
    detailed_variable='',
    identifier='ba0630cb-9a7e-4c4b-9675-c92aba7088ce'
  ),
  'b976820e-6b8c-45f7-87a6-fa6474d39a35': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='SEDIMENT COMPOSITION',
    detailed_variable='',
    identifier='b976820e-6b8c-45f7-87a6-fa6474d39a35'
  ),
  '13090c27-d808-4e51-a8ca-2f2c7b958f57': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='SEDIMENT THICKNESS',
    detailed_variable='',
    identifier='13090c27-d808-4e51-a8ca-2f2c7b958f57'
  ),
  '62013604-1f53-45ab-8612-b3f3fb001db0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='SEDIMENTARY STRUCTURES',
    detailed_variable='',
    identifier='62013604-1f53-45ab-8612-b3f3fb001db0'
  ),
  '783a9466-a7f9-4294-8159-f84a977a2843': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='SEDIMENTARY TEXTURES',
    detailed_variable='',
    identifier='783a9466-a7f9-4294-8159-f84a977a2843'
  ),
  'e9c6d45a-787e-4099-bbf9-03d377cdb8d5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='STRATIGRAPHIC SEQUENCE',
    detailed_variable='',
    identifier='e9c6d45a-787e-4099-bbf9-03d377cdb8d5'
  ),
  '26352dff-a48b-4b4a-a442-3b5039cf55c0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='SUSPENSION',
    variable_level_3='',
    detailed_variable='',
    identifier='26352dff-a48b-4b4a-a442-3b5039cf55c0'
  ),
  '5d05853c-b709-484f-a406-03f64e643ea4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='WEATHERING',
    variable_level_3='',
    detailed_variable='',
    identifier='5d05853c-b709-484f-a406-03f64e643ea4'
  ),
  'b895f4b5-5273-49ef-883f-b67d9f199505': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b895f4b5-5273-49ef-883f-b67d9f199505'
  ),
  'd37d51e0-1bef-473a-9221-6713166762f9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='ARETES',
    variable_level_3='',
    detailed_variable='',
    identifier='d37d51e0-1bef-473a-9221-6713166762f9'
  ),
  'b3032f74-fcdf-41d7-8899-2f2b140209c9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='CIRQUES/COMBES',
    variable_level_3='',
    detailed_variable='',
    identifier='b3032f74-fcdf-41d7-8899-2f2b140209c9'
  ),
  'b1d30791-5871-474f-aedf-2d4aa51e2b92': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='CREVASSES',
    variable_level_3='',
    detailed_variable='',
    identifier='b1d30791-5871-474f-aedf-2d4aa51e2b92'
  ),
  'a78c3d42-ac89-4040-9f0d-4d864b8c4551': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='CREVASSES',
    variable_level_3='LONGITUDINAL CREVASSES',
    detailed_variable='',
    identifier='a78c3d42-ac89-4040-9f0d-4d864b8c4551'
  ),
  'f0bbea2f-2ef0-4e99-ad76-1aedbbedc016': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='CREVASSES',
    variable_level_3='MARGINAL CREVASSES',
    detailed_variable='',
    identifier='f0bbea2f-2ef0-4e99-ad76-1aedbbedc016'
  ),
  '6ba069ae-8561-44b7-9e59-d645d6bd725f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='CREVASSES',
    variable_level_3='TRANSVERSE CREVASSES',
    detailed_variable='',
    identifier='6ba069ae-8561-44b7-9e59-d645d6bd725f'
  ),
  '614309a2-4332-4695-aa77-d11794fe4733': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='DRUMLINS',
    variable_level_3='',
    detailed_variable='',
    identifier='614309a2-4332-4695-aa77-d11794fe4733'
  ),
  '158a8764-a6e4-4d28-a1b9-b2ab91e09995': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='ESKERS',
    variable_level_3='',
    detailed_variable='',
    identifier='158a8764-a6e4-4d28-a1b9-b2ab91e09995'
  ),
  '666d9a2b-aaa8-4789-a9d9-a6774e650fe4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='FJORDS',
    variable_level_3='',
    detailed_variable='',
    identifier='666d9a2b-aaa8-4789-a9d9-a6774e650fe4'
  ),
  '8c878cc0-d601-4371-af35-9db2c67d8de6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='GLACIAL HORNS',
    variable_level_3='',
    detailed_variable='',
    identifier='8c878cc0-d601-4371-af35-9db2c67d8de6'
  ),
  'c8ad9c7e-384d-42c9-a75c-813a67e4dbfa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='GLACIER STRIATIONS/GROOVES',
    variable_level_3='',
    detailed_variable='',
    identifier='c8ad9c7e-384d-42c9-a75c-813a67e4dbfa'
  ),
  '7335b131-0e86-41b3-a0bc-b28120a0a78a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='GLACIER/HANGING/U-SHAPED VALLEYS',
    variable_level_3='',
    detailed_variable='',
    identifier='7335b131-0e86-41b3-a0bc-b28120a0a78a'
  ),
  '0a009cb2-9883-48d3-8b91-21efb75b4347': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='GLACIER/ICE CAVES',
    variable_level_3='',
    detailed_variable='',
    identifier='0a009cb2-9883-48d3-8b91-21efb75b4347'
  ),
  '86c6042a-b7ca-4d97-b9d7-db22b1560810': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='ICE-DAMMED LAKES',
    variable_level_3='',
    detailed_variable='',
    identifier='86c6042a-b7ca-4d97-b9d7-db22b1560810'
  ),
  '6a8d6d83-1a3b-452c-90b8-f37b28bd7eb6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='KAME DELTA',
    variable_level_3='',
    detailed_variable='',
    identifier='6a8d6d83-1a3b-452c-90b8-f37b28bd7eb6'
  ),
  'a3407182-6908-4206-b2fd-4c39da4072ce': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='KAMES',
    variable_level_3='',
    detailed_variable='',
    identifier='a3407182-6908-4206-b2fd-4c39da4072ce'
  ),
  '0557b602-cf85-4b04-82a7-ca76f364e5f4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='KETTLE HOLES',
    variable_level_3='',
    detailed_variable='',
    identifier='0557b602-cf85-4b04-82a7-ca76f364e5f4'
  ),
  '2575cfaf-1a09-48b6-acb9-fda23b6f4719': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='MORAINES',
    variable_level_3='',
    detailed_variable='',
    identifier='2575cfaf-1a09-48b6-acb9-fda23b6f4719'
  ),
  'd3ad1ced-39fa-4e3a-a75d-58e5393a2abe': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='MORAINES',
    variable_level_3='LATERAL MORAINE',
    detailed_variable='',
    identifier='d3ad1ced-39fa-4e3a-a75d-58e5393a2abe'
  ),
  '063fae56-f066-4023-84c2-daff8261b7fc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='MORAINES',
    variable_level_3='MEDIAL MORAINE',
    detailed_variable='',
    identifier='063fae56-f066-4023-84c2-daff8261b7fc'
  ),
  '5c242e01-40c4-4fca-a99d-48e4064f6c6f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='MORAINES',
    variable_level_3='RECESSIONAL MORAINE',
    detailed_variable='',
    identifier='5c242e01-40c4-4fca-a99d-48e4064f6c6f'
  ),
  'b62123dd-1bf5-4222-a646-05d71d729c75': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='MORAINES',
    variable_level_3='RIBBED/ROGAN MORAINE',
    detailed_variable='',
    identifier='b62123dd-1bf5-4222-a646-05d71d729c75'
  ),
  '16bd425e-9a14-41ac-900e-5b5c4f713dda': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='MORAINES',
    variable_level_3='TERMINAL MORAINE',
    detailed_variable='',
    identifier='16bd425e-9a14-41ac-900e-5b5c4f713dda'
  ),
  '12e5921b-0d9b-4656-8d5c-d73abcf90a81': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='NUNATAKS',
    variable_level_3='',
    detailed_variable='',
    identifier='12e5921b-0d9b-4656-8d5c-d73abcf90a81'
  ),
  '2e62a5dd-5ea5-4cd6-8051-b5e162ef4e01': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='OUTWASH FANS/PLAINS',
    variable_level_3='',
    detailed_variable='',
    identifier='2e62a5dd-5ea5-4cd6-8051-b5e162ef4e01'
  ),
  'e96cea31-2bee-4d9d-bf4a-d0f469aa3bd4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='ROCHE MOUTONNEES/SHEEPBACK',
    variable_level_3='',
    detailed_variable='',
    identifier='e96cea31-2bee-4d9d-bf4a-d0f469aa3bd4'
  ),
  'fcbf8f96-ac53-41b6-9c98-a87425a4ec82': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='ROCK GLACIERS',
    variable_level_3='',
    detailed_variable='',
    identifier='fcbf8f96-ac53-41b6-9c98-a87425a4ec82'
  ),
  'bffac466-83aa-4060-a378-6d2d6e49f2a1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='TILL PLAINS',
    variable_level_3='',
    detailed_variable='',
    identifier='bffac466-83aa-4060-a378-6d2d6e49f2a1'
  ),
  'b409a30b-0e3f-4592-bec9-7d371797b4a9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b409a30b-0e3f-4592-bec9-7d371797b4a9'
  ),
  'ad793d5e-b75d-4d3e-a542-ad4b4075b141': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='ABLATION',
    variable_level_3='',
    detailed_variable='',
    identifier='ad793d5e-b75d-4d3e-a542-ad4b4075b141'
  ),
  'c4c96fc4-c75b-4e98-852d-b28fdf6b77a4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='ABRASION',
    variable_level_3='',
    detailed_variable='',
    identifier='c4c96fc4-c75b-4e98-852d-b28fdf6b77a4'
  ),
  'df77e4c7-0b22-4f14-afa5-11b1d335a315': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='CRUST REBOUND',
    variable_level_3='',
    detailed_variable='',
    identifier='df77e4c7-0b22-4f14-afa5-11b1d335a315'
  ),
  'eb31cb40-97cf-4445-8abd-d375391edf6f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='DEGRADATION',
    variable_level_3='',
    detailed_variable='',
    identifier='eb31cb40-97cf-4445-8abd-d375391edf6f'
  ),
  '45ee1fde-6b00-4aca-ac1b-6c13e2361467': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='DUMPING',
    variable_level_3='',
    detailed_variable='',
    identifier='45ee1fde-6b00-4aca-ac1b-6c13e2361467'
  ),
  '10ec2826-c4ac-4373-ad05-9bb4eb35b360': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='ENTRAINMENT',
    variable_level_3='',
    detailed_variable='',
    identifier='10ec2826-c4ac-4373-ad05-9bb4eb35b360'
  ),
  'e2ea5b37-7004-4943-ad69-ca39a57569a4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='FIRN FORMATION',
    variable_level_3='',
    detailed_variable='',
    identifier='e2ea5b37-7004-4943-ad69-ca39a57569a4'
  ),
  '24b052b6-5996-496d-9e91-1fdbda5897da': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='FREEZE/THAW',
    variable_level_3='',
    detailed_variable='',
    identifier='24b052b6-5996-496d-9e91-1fdbda5897da'
  ),
  'ee8dfdf6-0153-4067-ab3b-51794b01ee86': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='FREEZE/THAW',
    variable_level_3='BASAL ICE FREEZING',
    detailed_variable='',
    identifier='ee8dfdf6-0153-4067-ab3b-51794b01ee86'
  ),
  'aa74db50-4ae7-463b-903a-2a256f967ca8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='GLACIAL DISPLACEMENT',
    variable_level_3='',
    detailed_variable='',
    identifier='aa74db50-4ae7-463b-903a-2a256f967ca8'
  ),
  'b313436f-d925-48b7-a339-e6a08475b6e1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='GLACIAL DRIFT',
    variable_level_3='',
    detailed_variable='',
    identifier='b313436f-d925-48b7-a339-e6a08475b6e1'
  ),
  '6743ea28-0a6e-4d47-ac71-0c9cdf24ac25': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='GLACIAL GROWTH',
    variable_level_3='',
    detailed_variable='',
    identifier='6743ea28-0a6e-4d47-ac71-0c9cdf24ac25'
  ),
  '12bb9ec2-a706-436d-aa29-253495276052': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='GLACIAL STRIATION',
    variable_level_3='',
    detailed_variable='',
    identifier='12bb9ec2-a706-436d-aa29-253495276052'
  ),
  'f8b73efd-d313-41d8-995a-49b80bc8f248': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='GLACIER CRUST SUBSIDENCE',
    variable_level_3='',
    detailed_variable='',
    identifier='f8b73efd-d313-41d8-995a-49b80bc8f248'
  ),
  '2f8b965f-0a0e-427f-a696-ac6b4323744e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='PERIGLACIAL PROCESSES',
    variable_level_3='',
    detailed_variable='',
    identifier='2f8b965f-0a0e-427f-a696-ac6b4323744e'
  ),
  '53e26c7c-5e85-4dd0-a999-8de519bf9976': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='PLUCKING',
    variable_level_3='',
    detailed_variable='',
    identifier='53e26c7c-5e85-4dd0-a999-8de519bf9976'
  ),
  '92e348f3-9e6c-4e9e-a1bc-ee72d04755d6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='SCOURING',
    variable_level_3='',
    detailed_variable='',
    identifier='92e348f3-9e6c-4e9e-a1bc-ee72d04755d6'
  ),
  '8c6d4f39-6ae0-4c8d-ad48-2f82bb4e1541': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='SEDIMENT TRANSPORT',
    variable_level_3='',
    detailed_variable='',
    identifier='8c6d4f39-6ae0-4c8d-ad48-2f82bb4e1541'
  ),
  'b609e525-db71-4634-b569-f8aab5ad544e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='',
    detailed_variable='',
    identifier='b609e525-db71-4634-b569-f8aab5ad544e'
  ),
  '134fd984-9b26-4ceb-9084-3bffc0c5a321': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='SEDIMENT CHEMISTRY',
    detailed_variable='',
    identifier='134fd984-9b26-4ceb-9084-3bffc0c5a321'
  ),
  '7fa9d5ef-690e-4daf-8503-363b6b1cb6e4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='STRATIGRAPHIC SEQUENCE',
    detailed_variable='',
    identifier='7fa9d5ef-690e-4daf-8503-363b6b1cb6e4'
  ),
  '07825acf-619a-4689-b1ed-09c15166624c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='WEATHERING',
    variable_level_3='',
    detailed_variable='',
    identifier='07825acf-619a-4689-b1ed-09c15166624c'
  ),
  '590aa85e-bbce-40b2-8ffb-53d80a61c51a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST LANDFORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='590aa85e-bbce-40b2-8ffb-53d80a61c51a'
  ),
  'cdeff06c-28ec-4a4c-b522-4a46f1f9a239': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST LANDFORMS',
    variable_level_2='CAVES',
    variable_level_3='',
    detailed_variable='',
    identifier='cdeff06c-28ec-4a4c-b522-4a46f1f9a239'
  ),
  'ee347068-e1ff-4271-8726-8343f4f15614': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST LANDFORMS',
    variable_level_2='COCKPIT/TOWER KARST',
    variable_level_3='',
    detailed_variable='',
    identifier='ee347068-e1ff-4271-8726-8343f4f15614'
  ),
  'c9323363-ea07-479d-8b64-e3dbf298a7c5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST LANDFORMS',
    variable_level_2='KARST VALLEY',
    variable_level_3='',
    detailed_variable='',
    identifier='c9323363-ea07-479d-8b64-e3dbf298a7c5'
  ),
  'b9e8b2e3-ea76-4ce9-8a25-64ba0cdef913': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST LANDFORMS',
    variable_level_2='SINKHOLES (DOLINES)',
    variable_level_3='',
    detailed_variable='',
    identifier='b9e8b2e3-ea76-4ce9-8a25-64ba0cdef913'
  ),
  'd42a8f13-c438-4794-9f37-bd7870ec731d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST LANDFORMS',
    variable_level_2='UVALA',
    variable_level_3='',
    detailed_variable='',
    identifier='d42a8f13-c438-4794-9f37-bd7870ec731d'
  ),
  '9cd875b0-210b-458f-b208-1690f50820d0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST PROCESSES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9cd875b0-210b-458f-b208-1690f50820d0'
  ),
  '50fd29da-a846-4dd6-98e8-e826b75eeda7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST PROCESSES',
    variable_level_2='CAC03',
    variable_level_3='',
    detailed_variable='',
    identifier='50fd29da-a846-4dd6-98e8-e826b75eeda7'
  ),
  'c2920f06-fd42-47da-9989-3104f8fb7282': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST PROCESSES',
    variable_level_2='DISSOLVED CO2',
    variable_level_3='',
    detailed_variable='',
    identifier='c2920f06-fd42-47da-9989-3104f8fb7282'
  ),
  'e657b41f-4aa0-4816-b3c8-5b477812a0bc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST PROCESSES',
    variable_level_2='KARST HYDROLOGY',
    variable_level_3='',
    detailed_variable='',
    identifier='e657b41f-4aa0-4816-b3c8-5b477812a0bc'
  ),
  '0317caf8-af0a-4abe-89fb-fd1d9c33b9e7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST PROCESSES',
    variable_level_2='POROSITY',
    variable_level_3='',
    detailed_variable='',
    identifier='0317caf8-af0a-4abe-89fb-fd1d9c33b9e7'
  ),
  'd727d3c7-3a02-48c6-ab63-a4b3d3364783': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST PROCESSES',
    variable_level_2='WEATHERING',
    variable_level_3='',
    detailed_variable='',
    identifier='d727d3c7-3a02-48c6-ab63-a4b3d3364783'
  ),
  '9673dc0b-89c0-4f0c-b378-f1c8cb267c8f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9673dc0b-89c0-4f0c-b378-f1c8cb267c8f'
  ),
  '1a0e7a60-9c22-40c5-8424-55119b4db743': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='CALDERA',
    variable_level_3='',
    detailed_variable='',
    identifier='1a0e7a60-9c22-40c5-8424-55119b4db743'
  ),
  'a9f7bee8-fb32-40b1-9936-ecf6f6597b6b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='CINDER CONE',
    variable_level_3='',
    detailed_variable='',
    identifier='a9f7bee8-fb32-40b1-9936-ecf6f6597b6b'
  ),
  '181fb5a4-125b-445d-b65f-adf9a919c800': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='FAULTS',
    variable_level_3='',
    detailed_variable='',
    identifier='181fb5a4-125b-445d-b65f-adf9a919c800'
  ),
  '12d2d3ab-2f04-436f-abd4-e28517e6f86c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='FOLDS',
    variable_level_3='',
    detailed_variable='',
    identifier='12d2d3ab-2f04-436f-abd4-e28517e6f86c'
  ),
  '1a888a27-8715-46c8-9e11-a6bffba00078': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='GEYSER',
    variable_level_3='',
    detailed_variable='',
    identifier='1a888a27-8715-46c8-9e11-a6bffba00078'
  ),
  'f1fa2b28-dc04-4373-a6b6-bbcedfaabfb5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='GRABEN',
    variable_level_3='',
    detailed_variable='',
    identifier='f1fa2b28-dc04-4373-a6b6-bbcedfaabfb5'
  ),
  '8493f8c2-63c3-4e1c-b813-1f0f3893a30a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='HORST',
    variable_level_3='',
    detailed_variable='',
    identifier='8493f8c2-63c3-4e1c-b813-1f0f3893a30a'
  ),
  '78200b25-8c91-4f1b-82cc-ed79764cd647': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='LAVA DOME',
    variable_level_3='',
    detailed_variable='',
    identifier='78200b25-8c91-4f1b-82cc-ed79764cd647'
  ),
  'edb9d13d-27a1-4e9a-a32e-4e49b5e76836': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='LAVA PLAIN',
    variable_level_3='',
    detailed_variable='',
    identifier='edb9d13d-27a1-4e9a-a32e-4e49b5e76836'
  ),
  'b7ff366c-4322-47bf-b12e-c3fbfb05cf54': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='MAAR',
    variable_level_3='',
    detailed_variable='',
    identifier='b7ff366c-4322-47bf-b12e-c3fbfb05cf54'
  ),
  '8b7f66ea-d481-4641-9dbf-da90ca3ad9c9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='MOUNTAINS',
    variable_level_3='',
    detailed_variable='',
    identifier='8b7f66ea-d481-4641-9dbf-da90ca3ad9c9'
  ),
  '694e18ec-ceaf-4070-9763-f3ee6dbd6b5b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='PLATEAU',
    variable_level_3='',
    detailed_variable='',
    identifier='694e18ec-ceaf-4070-9763-f3ee6dbd6b5b'
  ),
  '97298feb-6991-4d68-8337-177460e436ad': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='RIDGE',
    variable_level_3='',
    detailed_variable='',
    identifier='97298feb-6991-4d68-8337-177460e436ad'
  ),
  'ca874a66-f3a8-4099-978c-4684944dc348': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='RIFT VALLEY',
    variable_level_3='',
    detailed_variable='',
    identifier='ca874a66-f3a8-4099-978c-4684944dc348'
  ),
  'f8a9104b-fe7b-4a60-94fc-6b5ef504db55': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='TUYA',
    variable_level_3='',
    detailed_variable='',
    identifier='f8a9104b-fe7b-4a60-94fc-6b5ef504db55'
  ),
  '7c2e1960-ae20-46a9-acf1-a3e71542fbb4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='VOLCANO',
    variable_level_3='',
    detailed_variable='',
    identifier='7c2e1960-ae20-46a9-acf1-a3e71542fbb4'
  ),
  '99b4792a-9ea3-4756-a4dc-b1b30c946b54': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC PROCESSES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='99b4792a-9ea3-4756-a4dc-b1b30c946b54'
  ),
  'bb554660-d608-467c-b265-b9b68eecfb37': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC PROCESSES',
    variable_level_2='EPEIROGENIC MOVEMENT',
    variable_level_3='',
    detailed_variable='',
    identifier='bb554660-d608-467c-b265-b9b68eecfb37'
  ),
  '4f0f52fb-b272-49c6-9425-690d9285c380': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC PROCESSES',
    variable_level_2='ISOSTATIC UPLIFT',
    variable_level_3='',
    detailed_variable='',
    identifier='4f0f52fb-b272-49c6-9425-690d9285c380'
  ),
  'f486acc8-0d0c-4322-bf5c-177bc632bd76': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC PROCESSES',
    variable_level_2='OROGENIC MOVEMENT',
    variable_level_3='',
    detailed_variable='',
    identifier='f486acc8-0d0c-4322-bf5c-177bc632bd76'
  ),
  '1b370af4-1887-4e7a-82a6-9acb9ce5dd5f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC PROCESSES',
    variable_level_2='RIFTING',
    variable_level_3='',
    detailed_variable='',
    identifier='1b370af4-1887-4e7a-82a6-9acb9ce5dd5f'
  ),
  '0d43cb88-dd6d-40ac-b241-b628a39ed2af': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC PROCESSES',
    variable_level_2='SUBDUCTION',
    variable_level_3='',
    detailed_variable='',
    identifier='0d43cb88-dd6d-40ac-b241-b628a39ed2af'
  ),
  '7d5472ba-ae65-45df-a19a-c762055eaead': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC PROCESSES',
    variable_level_2='TECTONIC UPLIFT',
    variable_level_3='',
    detailed_variable='',
    identifier='7d5472ba-ae65-45df-a19a-c762055eaead'
  ),
  'e5815f58-8232-4c7f-b50d-ea71d73891a9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='LAND USE/LAND COVER',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e5815f58-8232-4c7f-b50d-ea71d73891a9'
  ),
  'a71560e7-42c5-4cbe-96fc-368de3b05a5f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='LAND USE/LAND COVER',
    variable_level_1='DISTURBANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a71560e7-42c5-4cbe-96fc-368de3b05a5f'
  ),
  '2e69c08b-ee0f-426c-a8d2-dc50876f76c2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='LAND USE/LAND COVER',
    variable_level_1='LAND PRODUCTIVITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2e69c08b-ee0f-426c-a8d2-dc50876f76c2'
  ),
  'c77819e9-f62f-48dc-b924-e7a73b4dcda9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='LAND USE/LAND COVER',
    variable_level_1='LAND RESOURCES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c77819e9-f62f-48dc-b924-e7a73b4dcda9'
  ),
  'fe2f8240-4d8e-4b1f-b869-29fee59692f7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='LAND USE/LAND COVER',
    variable_level_1='LAND USE CLASSES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fe2f8240-4d8e-4b1f-b869-29fee59692f7'
  ),
  '2f72ebfd-bbbd-4cda-bfc1-0d650272b51d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='LAND USE/LAND COVER',
    variable_level_1='LAND USE/LAND COVER CHANGE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2f72ebfd-bbbd-4cda-bfc1-0d650272b51d'
  ),
  '75c312bc-79f9-4d74-a7c0-3c67c019196c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='LAND USE/LAND COVER',
    variable_level_1='LAND USE/LAND COVER CLASSIFICATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='75c312bc-79f9-4d74-a7c0-3c67c019196c'
  ),
  'e63844c1-015c-4776-b01c-e3e7d5dd3d0c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='LAND USE/LAND COVER',
    variable_level_1='LAND USE/LAND COVER CLASSIFICATION',
    variable_level_2='VEGETATION INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='e63844c1-015c-4776-b01c-e3e7d5dd3d0c'
  ),
  '7b43eda3-899a-4afa-89be-2dbe527834c2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='LAND USE/LAND COVER',
    variable_level_1='LAND USE/LAND COVER CLASSIFICATION',
    variable_level_2='VEGETATION INDEX',
    variable_level_3='NORMALIZED DIFFERENCE VEGETATION INDEX (NDVI)',
    detailed_variable='',
    identifier='7b43eda3-899a-4afa-89be-2dbe527834c2'
  ),
  'f7e776da-50a3-4bb2-bf51-b6ba6266a605': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='LAND USE/LAND COVER',
    variable_level_1='LAND/OCEAN/ICE FRACTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f7e776da-50a3-4bb2-bf51-b6ba6266a605'
  ),
  'b7586859-abb5-4449-9be6-72c613b6a084': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='LAND USE/LAND COVER',
    variable_level_1='LAND/OCEAN/ICE MASK',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b7586859-abb5-4449-9be6-72c613b6a084'
  ),
  '911f2d35-8678-42e6-a581-324d385f7d6b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='LAND USE/LAND COVER',
    variable_level_1='ROCK FRACTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='911f2d35-8678-42e6-a581-324d385f7d6b'
  ),
  '87be8c1f-9f92-45f9-ba9b-f50efa5bcca0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='LAND USE/LAND COVER',
    variable_level_1='SHADE FRACTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='87be8c1f-9f92-45f9-ba9b-f50efa5bcca0'
  ),
  'f36d71c6-f2ad-49c4-809f-09b4f0688412': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='LANDSCAPE',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f36d71c6-f2ad-49c4-809f-09b4f0688412'
  ),
  'e77c0096-05a7-47ff-8629-55d12c46bb6b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='LANDSCAPE',
    variable_level_1='LANDSCAPE ECOLOGY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e77c0096-05a7-47ff-8629-55d12c46bb6b'
  ),
  '36a4ac5a-1082-4922-9bca-934c06e54cda': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='LANDSCAPE',
    variable_level_1='LANDSCAPE MANAGEMENT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='36a4ac5a-1082-4922-9bca-934c06e54cda'
  ),
  '3b6fe940-7383-4bae-b436-fc487723bbcf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='LANDSCAPE',
    variable_level_1='LANDSCAPE PATTERNS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3b6fe940-7383-4bae-b436-fc487723bbcf'
  ),
  '6c44a08d-0a08-47f5-b819-7e561445e613': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='LANDSCAPE',
    variable_level_1='LANDSCAPE PROCESSES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6c44a08d-0a08-47f5-b819-7e561445e613'
  ),
  'd57dff3d-5f2f-425c-80bb-2a6fcb42d3fa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='LANDSCAPE',
    variable_level_1='RECLAMATION/REVEGETATION/RESTORATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d57dff3d-5f2f-425c-80bb-2a6fcb42d3fa'
  ),
  'dcb100c9-5b43-422a-a429-25cae9dbb170': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='LANDSCAPE',
    variable_level_1='REFORESTATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='dcb100c9-5b43-422a-a429-25cae9dbb170'
  ),
  '3526afb8-0dc9-43c7-8ad4-f34f250a1e91': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3526afb8-0dc9-43c7-8ad4-f34f250a1e91'
  ),
  '9409ee0f-2b72-4472-9d61-f8072981a6cb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='CALCIUM',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9409ee0f-2b72-4472-9d61-f8072981a6cb'
  ),
  'a7ae5843-479c-4055-b8fc-ba651e485750': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='CARBON',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a7ae5843-479c-4055-b8fc-ba651e485750'
  ),
  'f6a8db71-9686-46ec-a3ac-66ca4a0ec1bd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='CARBON',
    variable_level_2='NET ECOSYSTEM CO2 EXCHANGE (NEE)',
    variable_level_3='',
    detailed_variable='',
    identifier='f6a8db71-9686-46ec-a3ac-66ca4a0ec1bd'
  ),
  '39a39084-ae04-421c-892b-f554133ca4e6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='CARBON',
    variable_level_2='SOIL ORGANIC CARBON (SOC)',
    variable_level_3='',
    detailed_variable='',
    identifier='39a39084-ae04-421c-892b-f554133ca4e6'
  ),
  '9bad3c7b-daf6-428a-89bd-ce62b074dfcf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='CATION EXCHANGE CAPACITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9bad3c7b-daf6-428a-89bd-ce62b074dfcf'
  ),
  '0d0de6a7-c340-4e6b-b01d-ccbb6e7fa913': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='DENITRIFICATION RATE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0d0de6a7-c340-4e6b-b01d-ccbb6e7fa913'
  ),
  '781bf38b-2797-4415-8d5a-67e9f3a2f5fe': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='ELECTRICAL CONDUCTIVITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='781bf38b-2797-4415-8d5a-67e9f3a2f5fe'
  ),
  '49ee4fcc-a0ad-4638-aec9-90b4946d8922': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='HEAVY METALS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='49ee4fcc-a0ad-4638-aec9-90b4946d8922'
  ),
  'e2a88ac8-7bf3-408c-b2b4-b3217f9e4917': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='HYDRAULIC CONDUCTIVITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e2a88ac8-7bf3-408c-b2b4-b3217f9e4917'
  ),
  'ae575305-340d-474b-99a7-22b537f10ec8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='IRON',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ae575305-340d-474b-99a7-22b537f10ec8'
  ),
  '5c4b5f03-8e57-49f0-bb03-f8efafb837d3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='MACROFAUNA',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5c4b5f03-8e57-49f0-bb03-f8efafb837d3'
  ),
  '7b86bc20-ba2b-4cd0-8aa0-ed47663d9222': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='MAGNESIUM',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7b86bc20-ba2b-4cd0-8aa0-ed47663d9222'
  ),
  'e9555194-efd1-4427-b8e3-8fe6c49b8636': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='MICROFAUNA',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e9555194-efd1-4427-b8e3-8fe6c49b8636'
  ),
  'ed1b3fa6-173d-476c-9b35-c57335c0a473': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='MICROFLORA',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ed1b3fa6-173d-476c-9b35-c57335c0a473'
  ),
  'ac061db6-21c7-46fc-b5a8-9f61c795fdd6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='MICRONUTRIENTS/TRACE ELEMENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ac061db6-21c7-46fc-b5a8-9f61c795fdd6'
  ),
  'e1179e7f-59e5-465a-9879-6bda6985744e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='NITROGEN',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e1179e7f-59e5-465a-9879-6bda6985744e'
  ),
  '215f69b9-259a-4b82-9f8f-f96d4f5aaad2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='ORGANIC MATTER',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='215f69b9-259a-4b82-9f8f-f96d4f5aaad2'
  ),
  '08240c92-00b5-4f25-bf2e-8030531a78d2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='PERMAFROST',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='08240c92-00b5-4f25-bf2e-8030531a78d2'
  ),
  '9169ace5-0f04-4fc9-b38d-b89a786b9fe1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='PHOSPHORUS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9169ace5-0f04-4fc9-b38d-b89a786b9fe1'
  ),
  '8af17fd3-7c42-4698-9d60-e154ece5aebe': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='POTASSIUM',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8af17fd3-7c42-4698-9d60-e154ece5aebe'
  ),
  'c22818ce-07aa-4f77-8fe2-be1925743bac': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='RECLAMATION/REVEGETATION/RESTORATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c22818ce-07aa-4f77-8fe2-be1925743bac'
  ),
  'e497c2e3-cd21-4af9-9a5d-91da4e201631': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL ABSORPTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e497c2e3-cd21-4af9-9a5d-91da4e201631'
  ),
  'df5ff39e-b49f-4517-afc0-1842a1a6fdc7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL BIOGEOCHEMISTRY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='df5ff39e-b49f-4517-afc0-1842a1a6fdc7'
  ),
  '2c821621-f035-4c57-8dee-5f24968f959a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL BULK DENSITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2c821621-f035-4c57-8dee-5f24968f959a'
  ),
  'e273b634-62f5-4601-8b92-6550f6efeab8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL CHEMISTRY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e273b634-62f5-4601-8b92-6550f6efeab8'
  ),
  '14e51b6e-9d91-4af5-bb93-22842359d492': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL CLASSIFICATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='14e51b6e-9d91-4af5-bb93-22842359d492'
  ),
  '013b44e9-df5c-4ef8-a99f-7351d16bfd14': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL COLOR',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='013b44e9-df5c-4ef8-a99f-7351d16bfd14'
  ),
  'c9f8c1e9-dca8-4c2e-9537-65903d19cfe5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL COMPACTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c9f8c1e9-dca8-4c2e-9537-65903d19cfe5'
  ),
  '6ce3eeff-d222-4356-8cd2-50fbcbcbb295': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL CONSISTENCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6ce3eeff-d222-4356-8cd2-50fbcbcbb295'
  ),
  '60e783c1-4b33-4ab3-860b-8bd4ed00dc9f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL DEPTH',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='60e783c1-4b33-4ab3-860b-8bd4ed00dc9f'
  ),
  '6eef914d-ff9f-44b0-a3a6-3dcf911023d4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL EROSION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6eef914d-ff9f-44b0-a3a6-3dcf911023d4'
  ),
  'cdb10789-ef01-46bd-8047-86e550df0df4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL FERTILITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='cdb10789-ef01-46bd-8047-86e550df0df4'
  ),
  '76c23076-d9d5-4414-a69f-a830cecdd9ce': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL GAS/AIR',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='76c23076-d9d5-4414-a69f-a830cecdd9ce'
  ),
  'c6847d01-cbf9-491b-be59-c283d9072d95': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL HEAT BUDGET',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c6847d01-cbf9-491b-be59-c283d9072d95'
  ),
  '7a16aa40-c74b-4a69-a230-1edd1b453332': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL HORIZONS/PROFILE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7a16aa40-c74b-4a69-a230-1edd1b453332'
  ),
  'bcc72093-b2d4-47e8-9213-7f48172e0e95': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL IMPEDANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bcc72093-b2d4-47e8-9213-7f48172e0e95'
  ),
  '2283a2fe-19ec-4b1d-a553-20ec9713a658': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL INFILTRATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2283a2fe-19ec-4b1d-a553-20ec9713a658'
  ),
  'e9d5ae5a-0718-44f2-9694-b791b646a825': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL MECHANICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e9d5ae5a-0718-44f2-9694-b791b646a825'
  ),
  'bbe2ea34-8842-4a9f-9b0b-95dd3c71857f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL MOISTURE/WATER CONTENT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bbe2ea34-8842-4a9f-9b0b-95dd3c71857f'
  ),
  '4353d710-3d65-44b3-b988-26af1415646a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL MOISTURE/WATER CONTENT',
    variable_level_2='ROOT ZONE SOIL MOISTURE',
    variable_level_3='',
    detailed_variable='',
    identifier='4353d710-3d65-44b3-b988-26af1415646a'
  ),
  '09e712bf-7389-4980-8115-af4282469eb8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL MOISTURE/WATER CONTENT',
    variable_level_2='SURFACE SOIL MOISTURE',
    variable_level_3='',
    detailed_variable='',
    identifier='09e712bf-7389-4980-8115-af4282469eb8'
  ),
  '357193c5-154d-487b-a1c3-a1a90d15918c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL PH',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='357193c5-154d-487b-a1c3-a1a90d15918c'
  ),
  '2da4e52a-b43b-4ff0-9e4d-c98438a38c6d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL PLASTICITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2da4e52a-b43b-4ff0-9e4d-c98438a38c6d'
  ),
  '20f932b9-cc40-4462-879f-1c8d8c765152': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL POROSITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='20f932b9-cc40-4462-879f-1c8d8c765152'
  ),
  '1e7afff2-cd50-4d26-968b-bffd2d738edd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL PRODUCTIVITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1e7afff2-cd50-4d26-968b-bffd2d738edd'
  ),
  '283df7dc-58e0-41c5-80b1-e9cdeae9e79e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL PRODUCTIVITY',
    variable_level_2='GROSS PRIMARY PRODUCTION (GPP)',
    variable_level_3='',
    detailed_variable='',
    identifier='283df7dc-58e0-41c5-80b1-e9cdeae9e79e'
  ),
  'e699830a-0abf-45b2-8026-ac80e0269ea7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL RESPIRATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e699830a-0abf-45b2-8026-ac80e0269ea7'
  ),
  '3fb007fe-605f-4f07-be91-f723d7051ac3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL RESPIRATION',
    variable_level_2='AUTOTROPHIC RESPIRATION (Ra)',
    variable_level_3='',
    detailed_variable='',
    identifier='3fb007fe-605f-4f07-be91-f723d7051ac3'
  ),
  '8286d00a-e540-4e01-9e15-377711e5fe56': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL RESPIRATION',
    variable_level_2='HETEROTROPHIC RESPIRATION (Rh)',
    variable_level_3='',
    detailed_variable='',
    identifier='8286d00a-e540-4e01-9e15-377711e5fe56'
  ),
  '1b475201-a032-4a66-a3aa-a35605affaee': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL ROOTING DEPTH',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1b475201-a032-4a66-a3aa-a35605affaee'
  ),
  '9d7b0259-2d88-4e78-b2c2-131a02d05c15': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL SALINITY/SOIL SODICITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9d7b0259-2d88-4e78-b2c2-131a02d05c15'
  ),
  'aa25235a-596f-4504-89e1-4c625275700d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL STRUCTURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='aa25235a-596f-4504-89e1-4c625275700d'
  ),
  '0546b91a-294d-45d9-8b45-76aaad0cc024': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL TEMPERATURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0546b91a-294d-45d9-8b45-76aaad0cc024'
  ),
  'fb05c0c0-7fcd-470c-ba2b-755f04f5d811': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL TEXTURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fb05c0c0-7fcd-470c-ba2b-755f04f5d811'
  ),
  '7c00e468-6a43-49ef-891e-b0ce29e2ff36': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SOIL WATER HOLDING CAPACITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7c00e468-6a43-49ef-891e-b0ce29e2ff36'
  ),
  '742e6889-1ebf-4441-b803-4892c7176822': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='SULFUR',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='742e6889-1ebf-4441-b803-4892c7176822'
  ),
  'c67c1e1c-19f1-49de-8b2b-d5ce6f596323': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SOILS',
    variable_level_1='THERMAL CONDUCTIVITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c67c1e1c-19f1-49de-8b2b-d5ce6f596323'
  ),
  'cb5cc628-a1b5-459e-934f-881153a937b8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SURFACE RADIATIVE PROPERTIES',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='cb5cc628-a1b5-459e-934f-881153a937b8'
  ),
  '136b1de3-4b2e-49e6-80cd-cf2e9bac2c48': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SURFACE RADIATIVE PROPERTIES',
    variable_level_1='ALBEDO',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='136b1de3-4b2e-49e6-80cd-cf2e9bac2c48'
  ),
  '00c1d7b9-61d8-40ad-8c33-f27006832866': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SURFACE RADIATIVE PROPERTIES',
    variable_level_1='ANISOTROPY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='00c1d7b9-61d8-40ad-8c33-f27006832866'
  ),
  '4ee9d0c5-2e0c-486c-b89b-7b002d18c5f7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SURFACE RADIATIVE PROPERTIES',
    variable_level_1='EMISSIVITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4ee9d0c5-2e0c-486c-b89b-7b002d18c5f7'
  ),
  'f043c0a8-9cee-4c51-bf64-a4aaa34ab75d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SURFACE RADIATIVE PROPERTIES',
    variable_level_1='REFLECTANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f043c0a8-9cee-4c51-bf64-a4aaa34ab75d'
  ),
  'a228b67f-0791-470b-a4ca-71b8da279332': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SURFACE THERMAL PROPERTIES',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a228b67f-0791-470b-a4ca-71b8da279332'
  ),
  '9886e184-f4a5-4940-914d-10f98fe530bc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SURFACE THERMAL PROPERTIES',
    variable_level_1='HEAT FLUX',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9886e184-f4a5-4940-914d-10f98fe530bc'
  ),
  '05dd9887-0b58-45f2-b3ea-6e26bbee6990': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SURFACE THERMAL PROPERTIES',
    variable_level_1='HEAT FLUX',
    variable_level_2='LATENT HEAT FLUX',
    variable_level_3='',
    detailed_variable='',
    identifier='05dd9887-0b58-45f2-b3ea-6e26bbee6990'
  ),
  'dcefbf77-61b6-462b-a9dc-97f9035ac545': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SURFACE THERMAL PROPERTIES',
    variable_level_1='HEAT FLUX',
    variable_level_2='LONGWAVE HEAT FLUX',
    variable_level_3='',
    detailed_variable='',
    identifier='dcefbf77-61b6-462b-a9dc-97f9035ac545'
  ),
  'a213e661-4a55-4cee-a889-738f7bd6097c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SURFACE THERMAL PROPERTIES',
    variable_level_1='HEAT FLUX',
    variable_level_2='SENSIBLE HEAT FLUX',
    variable_level_3='',
    detailed_variable='',
    identifier='a213e661-4a55-4cee-a889-738f7bd6097c'
  ),
  'b514c2a8-ff5e-4f5a-95ab-5d06c61288c4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SURFACE THERMAL PROPERTIES',
    variable_level_1='HEAT FLUX',
    variable_level_2='SHORTWAVE HEAT FLUX',
    variable_level_3='',
    detailed_variable='',
    identifier='b514c2a8-ff5e-4f5a-95ab-5d06c61288c4'
  ),
  '8931329c-3f6d-4ba6-913c-27afa8d104c1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SURFACE THERMAL PROPERTIES',
    variable_level_1='LAND HEAT CAPACITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8931329c-3f6d-4ba6-913c-27afa8d104c1'
  ),
  'd559b900-eca6-42a4-9311-0297b2ef98ab': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SURFACE THERMAL PROPERTIES',
    variable_level_1='LAND SURFACE TEMPERATURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d559b900-eca6-42a4-9311-0297b2ef98ab'
  ),
  '40d6a3e7-89dd-4399-8fa5-bbc7a0917b4e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='SURFACE THERMAL PROPERTIES',
    variable_level_1='SKIN TEMPERATURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='40d6a3e7-89dd-4399-8fa5-bbc7a0917b4e'
  ),
  '3e822484-c94a-457b-a32f-376fcbd6fd35': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='TOPOGRAPHY',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3e822484-c94a-457b-a32f-376fcbd6fd35'
  ),
  '21474df3-f9a6-48ca-be15-bdb3611fe062': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='TOPOGRAPHY',
    variable_level_1='SURFACE ROUGHNESS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='21474df3-f9a6-48ca-be15-bdb3611fe062'
  ),
  '74ed1690-968e-444c-8a31-7b8344a2aad3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='TOPOGRAPHY',
    variable_level_1='TERRAIN ELEVATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='74ed1690-968e-444c-8a31-7b8344a2aad3'
  ),
  '05e52e24-b9ac-42cf-bdf9-1dcad56900e8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='TOPOGRAPHY',
    variable_level_1='TERRAIN ELEVATION',
    variable_level_2='BED ELEVATION',
    variable_level_3='',
    detailed_variable='',
    identifier='05e52e24-b9ac-42cf-bdf9-1dcad56900e8'
  ),
  '120f9132-a756-4f6f-a74c-78e94dfcd2a1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='TOPOGRAPHY',
    variable_level_1='TERRAIN ELEVATION',
    variable_level_2='CONTOUR MAPS',
    variable_level_3='',
    detailed_variable='',
    identifier='120f9132-a756-4f6f-a74c-78e94dfcd2a1'
  ),
  '395372ad-2883-4b6a-a481-6383a310ca47': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='TOPOGRAPHY',
    variable_level_1='TERRAIN ELEVATION',
    variable_level_2='DIGITAL ELEVATION/TERRAIN MODEL (DEM)',
    variable_level_3='',
    detailed_variable='',
    identifier='395372ad-2883-4b6a-a481-6383a310ca47'
  ),
  'a1f5c621-7b45-4889-8d02-b5ca6bf86c08': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='TOPOGRAPHY',
    variable_level_1='TERRAIN ELEVATION',
    variable_level_2='Digital Surface Model (DSM)',
    variable_level_3='',
    detailed_variable='',
    identifier='a1f5c621-7b45-4889-8d02-b5ca6bf86c08'
  ),
  'f7d2e34a-c5c2-4c21-9132-2472620dbda1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='TOPOGRAPHY',
    variable_level_1='TERRAIN ELEVATION',
    variable_level_2='TOPOGRAPHICAL RELIEF MAPS',
    variable_level_3='',
    detailed_variable='',
    identifier='f7d2e34a-c5c2-4c21-9132-2472620dbda1'
  ),
  '05bef198-cfff-48be-b0cb-14e296d38dbc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='LAND SURFACE',
    term='TOPOGRAPHY',
    variable_level_1='TOPOGRAPHIC EFFECTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='05bef198-cfff-48be-b0cb-14e296d38dbc'
  ),
  '91697b7d-8f2b-4954-850e-61d5f61c867d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='91697b7d-8f2b-4954-850e-61d5f61c867d'
  ),
  'f27ad52c-3dfd-4788-851a-427e60ae1b8f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='AQUATIC SCIENCES',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f27ad52c-3dfd-4788-851a-427e60ae1b8f'
  ),
  'f6c057c9-c789-4cd5-ba22-e9b08aae152b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='AQUATIC SCIENCES',
    variable_level_1='AQUACULTURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f6c057c9-c789-4cd5-ba22-e9b08aae152b'
  ),
  'fa57b0a0-9723-4195-bdd1-4f26aefa0e07': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='AQUATIC SCIENCES',
    variable_level_1='FISHERIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fa57b0a0-9723-4195-bdd1-4f26aefa0e07'
  ),
  'c16bda61-353b-4668-af2f-bbb98785b6fa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='BATHYMETRY/SEAFLOOR TOPOGRAPHY',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c16bda61-353b-4668-af2f-bbb98785b6fa'
  ),
  '0b011562-ee55-4ba0-a026-4faa7493ca5b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='BATHYMETRY/SEAFLOOR TOPOGRAPHY',
    variable_level_1='ABYSSAL HILLS/PLAINS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0b011562-ee55-4ba0-a026-4faa7493ca5b'
  ),
  '80d79c7e-6c64-4ada-bfcc-4093969758a5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='BATHYMETRY/SEAFLOOR TOPOGRAPHY',
    variable_level_1='BATHYMETRY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='80d79c7e-6c64-4ada-bfcc-4093969758a5'
  ),
  'd80c015f-a383-4883-8309-6aab1c39f5b6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='BATHYMETRY/SEAFLOOR TOPOGRAPHY',
    variable_level_1='BATHYMETRY',
    variable_level_2='COASTAL BATHYMETRY',
    variable_level_3='',
    detailed_variable='',
    identifier='d80c015f-a383-4883-8309-6aab1c39f5b6'
  ),
  'a91a00f7-05ed-4633-9fac-1772a48b6342': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='BATHYMETRY/SEAFLOOR TOPOGRAPHY',
    variable_level_1='CONTINENTAL MARGINS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a91a00f7-05ed-4633-9fac-1772a48b6342'
  ),
  '58c12630-a889-44c1-a951-56bbbe9758c9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='BATHYMETRY/SEAFLOOR TOPOGRAPHY',
    variable_level_1='FRACTURE ZONES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='58c12630-a889-44c1-a951-56bbbe9758c9'
  ),
  '73e02157-9df9-415f-93fc-cb457989ddb1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='BATHYMETRY/SEAFLOOR TOPOGRAPHY',
    variable_level_1='OCEAN PLATEAUS/RIDGES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='73e02157-9df9-415f-93fc-cb457989ddb1'
  ),
  'b6b51058-1111-4498-a9ac-e1515270fb27': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='BATHYMETRY/SEAFLOOR TOPOGRAPHY',
    variable_level_1='SEAFLOOR TOPOGRAPHY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b6b51058-1111-4498-a9ac-e1515270fb27'
  ),
  '83520258-413c-4842-93c0-58a23dc58638': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='BATHYMETRY/SEAFLOOR TOPOGRAPHY',
    variable_level_1='SEAMOUNTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='83520258-413c-4842-93c0-58a23dc58638'
  ),
  '8b22d265-0f46-46c1-b307-1957527c13bb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='BATHYMETRY/SEAFLOOR TOPOGRAPHY',
    variable_level_1='SUB-BOTTOM PROFILE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8b22d265-0f46-46c1-b307-1957527c13bb'
  ),
  '18ce5577-26e9-4b76-860b-1ba31cafa9d0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='BATHYMETRY/SEAFLOOR TOPOGRAPHY',
    variable_level_1='SUBMARINE CANYONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='18ce5577-26e9-4b76-860b-1ba31cafa9d0'
  ),
  '36040c6a-5e3a-49fe-b519-162fb77a0fd4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='BATHYMETRY/SEAFLOOR TOPOGRAPHY',
    variable_level_1='TRENCHES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='36040c6a-5e3a-49fe-b519-162fb77a0fd4'
  ),
  'ca477721-473b-40d7-a72b-4ffa963e48fb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='BATHYMETRY/SEAFLOOR TOPOGRAPHY',
    variable_level_1='WATER DEPTH',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ca477721-473b-40d7-a72b-4ffa963e48fb'
  ),
  'b6fd22ab-dca7-4dfa-8812-913453b5695b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b6fd22ab-dca7-4dfa-8812-913453b5695b'
  ),
  '7e28f2e0-a641-4085-be07-366ed6e701f4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='BARRIER ISLANDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7e28f2e0-a641-4085-be07-366ed6e701f4'
  ),
  '4ba798ce-ad0b-4809-94fa-ec1b8e294252': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='BEACHES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4ba798ce-ad0b-4809-94fa-ec1b8e294252'
  ),
  '22450240-b06c-4954-a8d6-c6b756dab92d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='BOTTOM COVER TYPE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='22450240-b06c-4954-a8d6-c6b756dab92d'
  ),
  '1fbf5df2-ab7c-43fc-9bb2-8eb3f8891f7b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='COASTAL ELEVATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1fbf5df2-ab7c-43fc-9bb2-8eb3f8891f7b'
  ),
  'ad497e7a-48fa-45e1-90a5-b052508bdb30': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='CORAL REEFS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ad497e7a-48fa-45e1-90a5-b052508bdb30'
  ),
  'f5df87b6-ed50-4da0-9ba5-7ce4c907bdb3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='CORAL REEFS',
    variable_level_2='CORAL BLEACHING',
    variable_level_3='',
    detailed_variable='',
    identifier='f5df87b6-ed50-4da0-9ba5-7ce4c907bdb3'
  ),
  '3edb3342-dab8-41d6-9f6a-28dd448528ec': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='CORAL REEFS',
    variable_level_2='CORAL REEF ASSESSMENT',
    variable_level_3='',
    detailed_variable='',
    identifier='3edb3342-dab8-41d6-9f6a-28dd448528ec'
  ),
  '7e24064a-7035-402a-ab9d-fa5e5c359720': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='CORAL REEFS',
    variable_level_2='CORAL REEF EXTENT',
    variable_level_3='',
    detailed_variable='',
    identifier='7e24064a-7035-402a-ab9d-fa5e5c359720'
  ),
  'f9f0f92b-7901-4dda-8d64-be4e845ce29b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='DELTAS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f9f0f92b-7901-4dda-8d64-be4e845ce29b'
  ),
  '6f7b2753-aed1-4783-a7cc-781d00d13a0f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='DUNES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6f7b2753-aed1-4783-a7cc-781d00d13a0f'
  ),
  'cd7a7748-7231-4a73-b85c-b5696066230a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='EROSION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='cd7a7748-7231-4a73-b85c-b5696066230a'
  ),
  'a7dcdedf-bcc5-4032-b70f-7fadf74d6144': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='ESTUARIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a7dcdedf-bcc5-4032-b70f-7fadf74d6144'
  ),
  'a90899c8-fe50-48e0-b92c-bb64f6ae681c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='FJORDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a90899c8-fe50-48e0-b92c-bb64f6ae681c'
  ),
  'f43cd776-c568-4d09-997c-0a8ad1022e06': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='INLETS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f43cd776-c568-4d09-997c-0a8ad1022e06'
  ),
  '82b62e59-6ea1-48e1-a402-bd386c5046eb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='INTERTIDAL ZONE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='82b62e59-6ea1-48e1-a402-bd386c5046eb'
  ),
  'c733c179-c12a-47e9-8e9a-817a5212446f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='LAGOONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c733c179-c12a-47e9-8e9a-817a5212446f'
  ),
  '5a090f0c-7466-47fd-b679-5dee947ab05c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='LOCAL SUBSIDENCE TRENDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5a090f0c-7466-47fd-b679-5dee947ab05c'
  ),
  'ccf07d90-b3a3-43d3-9249-a494bb48d1b6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='LONGSHORE CURRENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ccf07d90-b3a3-43d3-9249-a494bb48d1b6'
  ),
  '04c4a85f-91ce-4d64-9e19-b3e0897ff187': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='MANGROVES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='04c4a85f-91ce-4d64-9e19-b3e0897ff187'
  ),
  '30056645-a442-4ef6-ac76-c5bc27086d83': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='MARSHES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='30056645-a442-4ef6-ac76-c5bc27086d83'
  ),
  '488f4df2-712e-4fac-98d1-46ab134b84ee': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='ROCKY COASTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='488f4df2-712e-4fac-98d1-46ab134b84ee'
  ),
  'dffe5a35-09af-4413-bdd3-a5aedfeb49cc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='SALTWATER INTRUSION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='dffe5a35-09af-4413-bdd3-a5aedfeb49cc'
  ),
  '0afaaa5e-f88c-4c1f-95c1-1faa0148885a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='SEA LEVEL RISE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0afaaa5e-f88c-4c1f-95c1-1faa0148885a'
  ),
  '1ed24fe1-d0d5-46d1-8d22-8ac25d289c75': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='SEA SURFACE HEIGHT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1ed24fe1-d0d5-46d1-8d22-8ac25d289c75'
  ),
  'c5c34f0a-552e-45a6-91c1-9edb3a8deef9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='SEDIMENT TRANSPORT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c5c34f0a-552e-45a6-91c1-9edb3a8deef9'
  ),
  '9457740a-897b-4adc-96fb-f3e3aafa34ea': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='SEDIMENTATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9457740a-897b-4adc-96fb-f3e3aafa34ea'
  ),
  '4c2d2255-680d-47d6-adb2-179093593f8a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='SHOALS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4c2d2255-680d-47d6-adb2-179093593f8a'
  ),
  '1a740c3e-7032-4f72-93e8-d0ba343d82e0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='SHORELINE DISPLACEMENT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1a740c3e-7032-4f72-93e8-d0ba343d82e0'
  ),
  '1d3b4eb7-9931-44bf-8457-26847051b7a8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='SHORELINES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1d3b4eb7-9931-44bf-8457-26847051b7a8'
  ),
  '3472f70b-874f-4dc5-87db-4b3ebc4b9aaa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='SHORELINES',
    variable_level_2='SHORELINE MAPPING',
    variable_level_3='',
    detailed_variable='',
    identifier='3472f70b-874f-4dc5-87db-4b3ebc4b9aaa'
  ),
  '9edd23d0-68a9-4bae-8887-705058f48ba7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='STORM SURGE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9edd23d0-68a9-4bae-8887-705058f48ba7'
  ),
  '9ab67e8f-066e-47b8-838d-8cd5e7460119': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='COASTAL PROCESSES',
    variable_level_1='TIDAL HEIGHT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9ab67e8f-066e-47b8-838d-8cd5e7460119'
  ),
  '035d870c-9792-4a74-8e02-e03c9a671c8e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='GEOLOGICAL FEATURES',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='035d870c-9792-4a74-8e02-e03c9a671c8e'
  ),
  'e63e32cc-896c-47dd-9652-a01ba2ee3334': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='GEOLOGICAL FEATURES',
    variable_level_1='LAND/OCEAN/ICE FRACTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e63e32cc-896c-47dd-9652-a01ba2ee3334'
  ),
  '7a74347d-e372-4048-8012-c9be550e0e5e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='GEOLOGICAL FEATURES',
    variable_level_1='LAND/OCEAN/ICE MASK',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7a74347d-e372-4048-8012-c9be550e0e5e'
  ),
  '916b2963-6c1d-48ee-8f97-8606febf8db7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='HYDROGRAPHY',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='916b2963-6c1d-48ee-8f97-8606febf8db7'
  ),
  'ca154e02-a226-4cc7-8e4a-4474e7eb1eeb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE ENVIRONMENT MONITORING',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ca154e02-a226-4cc7-8e4a-4474e7eb1eeb'
  ),
  '43763945-ceea-4716-8e77-068393300a7e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE ENVIRONMENT MONITORING',
    variable_level_1='ENFORCEMENT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='43763945-ceea-4716-8e77-068393300a7e'
  ),
  '56e4dd42-e393-4aa2-b4d9-9e96d85c9768': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE ENVIRONMENT MONITORING',
    variable_level_1='MARINE OBSTRUCTIONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='56e4dd42-e393-4aa2-b4d9-9e96d85c9768'
  ),
  '8e5371ad-4e70-48cf-9109-bfe995b7230c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE ENVIRONMENT MONITORING',
    variable_level_1='MARINE SUBMERGED DEBRIS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8e5371ad-4e70-48cf-9109-bfe995b7230c'
  ),
  'e6c6507d-59dd-49f4-9afa-bb7393a718c6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE ENVIRONMENT MONITORING',
    variable_level_1='MARINE SURFACE ELEMENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e6c6507d-59dd-49f4-9afa-bb7393a718c6'
  ),
  'd594fc9c-556b-4eb5-9ec3-0d2126ca9cd5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE ENVIRONMENT MONITORING',
    variable_level_1='MARINE SURFACE ELEMENTS',
    variable_level_2='MARINE SURFACE DEBRIS',
    variable_level_3='',
    detailed_variable='',
    identifier='d594fc9c-556b-4eb5-9ec3-0d2126ca9cd5'
  ),
  'f81de12c-5f0c-4027-8ff1-de84d1bacb60': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE ENVIRONMENT MONITORING',
    variable_level_1='MARINE SURFACE ELEMENTS',
    variable_level_2='MARINE VESSELS',
    variable_level_3='',
    detailed_variable='',
    identifier='f81de12c-5f0c-4027-8ff1-de84d1bacb60'
  ),
  'bb04ee83-bf49-4f96-898d-20bb6e92bc93': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE GEOPHYSICS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bb04ee83-bf49-4f96-898d-20bb6e92bc93'
  ),
  'e31f905d-bd2a-4fe9-89d8-909e1d2b9b1a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE GEOPHYSICS',
    variable_level_1='MAGNETIC ANOMALIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e31f905d-bd2a-4fe9-89d8-909e1d2b9b1a'
  ),
  'ad09b215-e837-4d9f-acbc-2b45e5b81825': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE GEOPHYSICS',
    variable_level_1='MARINE GRAVITY FIELD',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ad09b215-e837-4d9f-acbc-2b45e5b81825'
  ),
  '7863ce31-0e06-42a5-bcf8-25981c44dec8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE GEOPHYSICS',
    variable_level_1='MARINE MAGNETICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7863ce31-0e06-42a5-bcf8-25981c44dec8'
  ),
  '78a4dbe2-2d6b-4562-988c-022c3a83f4c1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE GEOPHYSICS',
    variable_level_1='PLATE TECTONICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='78a4dbe2-2d6b-4562-988c-022c3a83f4c1'
  ),
  'ce4b2c6e-3d69-4cf1-8416-c36e5f9b1b2c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE SEDIMENTS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ce4b2c6e-3d69-4cf1-8416-c36e5f9b1b2c'
  ),
  'ff0108e2-8415-423c-85ed-07792dbef534': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE SEDIMENTS',
    variable_level_1='BIOGENIC SEDIMENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ff0108e2-8415-423c-85ed-07792dbef534'
  ),
  '14c8935f-8a46-4111-8f2e-bec8bbae5d13': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE SEDIMENTS',
    variable_level_1='BIOTURBATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='14c8935f-8a46-4111-8f2e-bec8bbae5d13'
  ),
  '4bfed15d-b8b4-4fb1-940b-ef342c4c2225': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE SEDIMENTS',
    variable_level_1='DIAGENESIS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4bfed15d-b8b4-4fb1-940b-ef342c4c2225'
  ),
  'd4f4b5d3-27b2-4b7d-bb69-733b67ac687a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE SEDIMENTS',
    variable_level_1='GEOTECHNICAL PROPERTIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d4f4b5d3-27b2-4b7d-bb69-733b67ac687a'
  ),
  '3d352f0f-f69f-44c4-b345-aa9230fbd6ca': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE SEDIMENTS',
    variable_level_1='HYDROGENOUS SEDIMENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3d352f0f-f69f-44c4-b345-aa9230fbd6ca'
  ),
  '676327f4-8354-4033-8081-9cab6651ac98': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE SEDIMENTS',
    variable_level_1='PARTICLE FLUX',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='676327f4-8354-4033-8081-9cab6651ac98'
  ),
  'f8411549-a72d-44cd-9b7b-6953ec22f8da': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE SEDIMENTS',
    variable_level_1='SEDIMENT CHEMISTRY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f8411549-a72d-44cd-9b7b-6953ec22f8da'
  ),
  '17008d04-394d-4de8-8834-dd0a3cd88093': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE SEDIMENTS',
    variable_level_1='SEDIMENT COMPOSITION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='17008d04-394d-4de8-8834-dd0a3cd88093'
  ),
  '6956aa5f-a54a-4335-a48c-8be363d267ac': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE SEDIMENTS',
    variable_level_1='SEDIMENT THICKNESS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6956aa5f-a54a-4335-a48c-8be363d267ac'
  ),
  'bd55adac-4182-4441-91e2-163aa77e1320': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE SEDIMENTS',
    variable_level_1='SEDIMENT TRANSPORT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bd55adac-4182-4441-91e2-163aa77e1320'
  ),
  '282ea985-efd0-4113-860d-b8221f6cc6f2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE SEDIMENTS',
    variable_level_1='SEDIMENTARY STRUCTURES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='282ea985-efd0-4113-860d-b8221f6cc6f2'
  ),
  'cddc37fd-8540-4c78-b567-add74e6b789b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE SEDIMENTS',
    variable_level_1='SEDIMENTARY TEXTURES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='cddc37fd-8540-4c78-b567-add74e6b789b'
  ),
  'a4eb3bc4-48a5-4ed2-a74b-ca87a58e90f5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE SEDIMENTS',
    variable_level_1='SEDIMENTATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a4eb3bc4-48a5-4ed2-a74b-ca87a58e90f5'
  ),
  '41b7293f-7f20-40ab-8bf7-b211c68146b9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE SEDIMENTS',
    variable_level_1='STRATIGRAPHIC SEQUENCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='41b7293f-7f20-40ab-8bf7-b211c68146b9'
  ),
  'bcf6975f-2a21-4a6c-9286-fb8f85d00901': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE SEDIMENTS',
    variable_level_1='SUSPENDED SOLIDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bcf6975f-2a21-4a6c-9286-fb8f85d00901'
  ),
  '31cf96eb-7fcd-490d-9e10-7f17dc12e1e3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE SEDIMENTS',
    variable_level_1='TERRIGENOUS SEDIMENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='31cf96eb-7fcd-490d-9e10-7f17dc12e1e3'
  ),
  '68e2c729-f729-4936-af2e-0ecf7ee7d231': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE SEDIMENTS',
    variable_level_1='TURBIDITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='68e2c729-f729-4936-af2e-0ecf7ee7d231'
  ),
  'e3b178eb-2d47-41db-aba1-43a05e9e9256': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE VOLCANISM',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e3b178eb-2d47-41db-aba1-43a05e9e9256'
  ),
  'bf3d6238-d0d6-4e73-82e6-5e38bc9291bb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE VOLCANISM',
    variable_level_1='BENTHIC HEAT FLOW',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bf3d6238-d0d6-4e73-82e6-5e38bc9291bb'
  ),
  'b677862b-7921-458f-a6db-0eb46469df33': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE VOLCANISM',
    variable_level_1='HYDROTHERMAL VENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b677862b-7921-458f-a6db-0eb46469df33'
  ),
  '9bb0de49-1812-400c-a73b-d2686dd9066a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE VOLCANISM',
    variable_level_1='ISLAND ARCS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9bb0de49-1812-400c-a73b-d2686dd9066a'
  ),
  'f345294c-36e6-4c76-b484-2204cc0bc3a2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE VOLCANISM',
    variable_level_1='MID-OCEAN RIDGES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f345294c-36e6-4c76-b484-2204cc0bc3a2'
  ),
  'f32afbca-dac6-41b1-a198-791c1fb57951': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='MARINE VOLCANISM',
    variable_level_1='RIFT VALLEYS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f32afbca-dac6-41b1-a198-791c1fb57951'
  ),
  '0517ae1f-7617-4f3b-80cb-649178032825': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN ACOUSTICS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0517ae1f-7617-4f3b-80cb-649178032825'
  ),
  '025c1e31-1a97-4a30-a887-0b9a5127fd4d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN ACOUSTICS',
    variable_level_1='ACOUSTIC ATTENUATION/TRANSMISSION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='025c1e31-1a97-4a30-a887-0b9a5127fd4d'
  ),
  '7bb3c4cd-cbb4-4c82-997b-d11ecc1cdb9f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN ACOUSTICS',
    variable_level_1='ACOUSTIC FREQUENCY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7bb3c4cd-cbb4-4c82-997b-d11ecc1cdb9f'
  ),
  '6a583047-6023-4b6a-ab25-b72529721a8c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN ACOUSTICS',
    variable_level_1='ACOUSTIC REFLECTIVITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6a583047-6023-4b6a-ab25-b72529721a8c'
  ),
  'b4a924bb-0d42-4169-bad7-3856f69f0c4a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN ACOUSTICS',
    variable_level_1='ACOUSTIC SCATTERING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b4a924bb-0d42-4169-bad7-3856f69f0c4a'
  ),
  '1295cf9a-c345-40eb-9b79-82bddc6acf50': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN ACOUSTICS',
    variable_level_1='ACOUSTIC TOMOGRAPHY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1295cf9a-c345-40eb-9b79-82bddc6acf50'
  ),
  'e4aae1a4-b4d5-4b13-9cc0-c0df6234ce3b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN ACOUSTICS',
    variable_level_1='ACOUSTIC VELOCITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e4aae1a4-b4d5-4b13-9cc0-c0df6234ce3b'
  ),
  'a74abbc1-dd75-4f22-bbec-7d45091a4593': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN ACOUSTICS',
    variable_level_1='AMBIENT NOISE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a74abbc1-dd75-4f22-bbec-7d45091a4593'
  ),
  '96f15c48-4ea1-4c68-92f0-d59218856bb5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN ACOUSTICS',
    variable_level_1='AMBIENT NOISE',
    variable_level_2='BIOLOGICAL AMBIENT NOISE',
    variable_level_3='',
    detailed_variable='',
    identifier='96f15c48-4ea1-4c68-92f0-d59218856bb5'
  ),
  'b016722f-5441-41c1-97c4-6612c87c4311': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN ACOUSTICS',
    variable_level_1='AMBIENT NOISE',
    variable_level_2='PHYSICAL AMBIENT NOISE',
    variable_level_3='',
    detailed_variable='',
    identifier='b016722f-5441-41c1-97c4-6612c87c4311'
  ),
  'f1f90445-4272-4390-92b1-2efc626a9ed1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN ACOUSTICS',
    variable_level_1='AMBIENT NOISE',
    variable_level_2='TOTAL AMBIENT NOISE',
    variable_level_3='',
    detailed_variable='',
    identifier='f1f90445-4272-4390-92b1-2efc626a9ed1'
  ),
  '6eb3919b-85ce-4988-8b78-9d0018fd8089': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6eb3919b-85ce-4988-8b78-9d0018fd8089'
  ),
  '4eab7956-e59e-4615-8d5c-39a16faa1f27': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='ALKALINITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4eab7956-e59e-4615-8d5c-39a16faa1f27'
  ),
  '64d17528-29b4-4e2e-843a-7f7035bb5717': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='AMMONIA',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='64d17528-29b4-4e2e-843a-7f7035bb5717'
  ),
  'f1e6caa5-2c97-407d-a0db-7bf01794d8e3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='BIOGEOCHEMICAL CYCLES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f1e6caa5-2c97-407d-a0db-7bf01794d8e3'
  ),
  '97636cf7-189f-4953-9807-64fbcc60f72c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='BIOMEDICAL CHEMICALS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='97636cf7-189f-4953-9807-64fbcc60f72c'
  ),
  '5c52009f-2c44-4db1-b62b-135c6181bad2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='CARBON',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5c52009f-2c44-4db1-b62b-135c6181bad2'
  ),
  '26afa886-4866-4536-be3a-6f9db9aacd97': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='CARBON DIOXIDE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='26afa886-4866-4536-be3a-6f9db9aacd97'
  ),
  '6bf8c40d-6bc0-410b-92a5-349bd88dc021': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='CARBON DIOXIDE',
    variable_level_2='CARBON DIOXIDE PARTIAL PRESSURE',
    variable_level_3='',
    detailed_variable='',
    identifier='6bf8c40d-6bc0-410b-92a5-349bd88dc021'
  ),
  '68f7ba1b-a2f9-41b6-9bc1-fd187942fbed': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='CARBONATE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='68f7ba1b-a2f9-41b6-9bc1-fd187942fbed'
  ),
  '7989eae1-8ea3-4039-af0c-9130de145449': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='CHLOROPHYLL',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7989eae1-8ea3-4039-af0c-9130de145449'
  ),
  '38219b66-2acd-4f77-a0fc-8241172c9001': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='DISSOLVED GASES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='38219b66-2acd-4f77-a0fc-8241172c9001'
  ),
  'a3c25ed5-d3e4-4b86-bd9a-6f78d5d2bc07': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='DISSOLVED SOLIDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a3c25ed5-d3e4-4b86-bd9a-6f78d5d2bc07'
  ),
  '6d8eb011-ffb5-4e18-ac59-2d8f84353734': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='HYDROCARBONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6d8eb011-ffb5-4e18-ac59-2d8f84353734'
  ),
  'b846063c-e218-4fc6-9866-0cdca24e9023': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='HYPOXIA',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b846063c-e218-4fc6-9866-0cdca24e9023'
  ),
  'd9b4f30d-bddd-4888-b66b-07d2dc09708b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='INORGANIC CARBON',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d9b4f30d-bddd-4888-b66b-07d2dc09708b'
  ),
  'b9cfc6af-a424-42b9-8e89-6b332262e841': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='INORGANIC MATTER',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b9cfc6af-a424-42b9-8e89-6b332262e841'
  ),
  '1dfb36a3-f985-4514-a1d0-cc73ca572922': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='MARINE GEOCHEMISTRY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1dfb36a3-f985-4514-a1d0-cc73ca572922'
  ),
  '4fde380a-38c5-4d46-bc80-4f2515a43983': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='NITRATE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4fde380a-38c5-4d46-bc80-4f2515a43983'
  ),
  '61740c18-f010-4384-8516-1eb33d75352e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='NITRIC ACID',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='61740c18-f010-4384-8516-1eb33d75352e'
  ),
  '941410da-0b7f-4ec6-a718-212194ced13f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='NITRITE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='941410da-0b7f-4ec6-a718-212194ced13f'
  ),
  'db5357c9-cc9d-4693-86fe-6bb88555d434': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='NITROGEN',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='db5357c9-cc9d-4693-86fe-6bb88555d434'
  ),
  '54054bc3-5faa-4b0d-b5dd-cf04595369b5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='NITROGEN DIOXIDE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='54054bc3-5faa-4b0d-b5dd-cf04595369b5'
  ),
  'd1c2bba5-799d-412b-80e0-fa04058416e3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='NITROUS OXIDE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d1c2bba5-799d-412b-80e0-fa04058416e3'
  ),
  '8dd7c9f0-51d0-4037-b1d0-a2517c1770ad': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='NUTRIENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8dd7c9f0-51d0-4037-b1d0-a2517c1770ad'
  ),
  '0f6a760e-999c-4275-9748-d682ad73fd58': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='NUTRIENTS',
    variable_level_2='NUTRIENT PROFILES',
    variable_level_3='',
    detailed_variable='',
    identifier='0f6a760e-999c-4275-9748-d682ad73fd58'
  ),
  '4d06267b-8f85-4b7b-9b2f-97a09f804d70': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='NUTRIENTS',
    variable_level_2='SURFACE NUTRIENTS',
    variable_level_3='',
    detailed_variable='',
    identifier='4d06267b-8f85-4b7b-9b2f-97a09f804d70'
  ),
  '080db90f-79ff-4900-941d-9c02fe2df862': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='OCEAN TRACERS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='080db90f-79ff-4900-941d-9c02fe2df862'
  ),
  'd3055f47-258e-4556-a885-54cd1fff4680': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='ORGANIC CARBON',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d3055f47-258e-4556-a885-54cd1fff4680'
  ),
  'b2bdeb71-81b5-43e6-a8b1-b09c215c8d1a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='ORGANIC MATTER',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b2bdeb71-81b5-43e6-a8b1-b09c215c8d1a'
  ),
  '18e0fad3-b6f4-4120-9221-f82fb2ffd384': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='ORGANIC MATTER',
    variable_level_2='COLORED DISSOLVED ORGANIC MATTER',
    variable_level_3='',
    detailed_variable='',
    identifier='18e0fad3-b6f4-4120-9221-f82fb2ffd384'
  ),
  '90aa8838-79bd-4b28-b518-8217e863c385': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='OXYGEN',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='90aa8838-79bd-4b28-b518-8217e863c385'
  ),
  '4433600b-f323-458a-b295-352f939aab6b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='PH',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4433600b-f323-458a-b295-352f939aab6b'
  ),
  '0b513d8c-bfd3-44ee-976e-42757b8375a2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='PHOSPHATE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0b513d8c-bfd3-44ee-976e-42757b8375a2'
  ),
  'ed925b43-db83-4cbb-8347-3dc0081bb8f4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='PIGMENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ed925b43-db83-4cbb-8347-3dc0081bb8f4'
  ),
  '37669b8c-1940-4330-b4e9-ee49ad3673b5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='PIGMENTS',
    variable_level_2='CHLOROPHYLL',
    variable_level_3='',
    detailed_variable='',
    identifier='37669b8c-1940-4330-b4e9-ee49ad3673b5'
  ),
  '6641ff15-36c8-4dbc-bf9c-176a08688173': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='RADIOCARBON',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6641ff15-36c8-4dbc-bf9c-176a08688173'
  ),
  'e9ed684e-5252-4091-a794-aaf6e5f249ed': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='RADIONUCLIDES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e9ed684e-5252-4091-a794-aaf6e5f249ed'
  ),
  'c91c8879-1b29-48e3-b4cd-a238af66cdaf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='SILICATE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c91c8879-1b29-48e3-b4cd-a238af66cdaf'
  ),
  '38dadd6d-6adb-44e2-b28a-fd18d797d052': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='STABLE ISOTOPES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='38dadd6d-6adb-44e2-b28a-fd18d797d052'
  ),
  '718fb499-8c55-4fa6-9a07-ac9155d4bc9d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='SUSPENDED SOLIDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='718fb499-8c55-4fa6-9a07-ac9155d4bc9d'
  ),
  '6c320188-da7b-4d52-8e99-57d7ac401841': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CHEMISTRY',
    variable_level_1='TRACE ELEMENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6c320188-da7b-4d52-8e99-57d7ac401841'
  ),
  'a031952d-9f00-4ba5-9966-5f87ab9dfdd4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a031952d-9f00-4ba5-9966-5f87ab9dfdd4'
  ),
  '0cb7f2c6-5e99-4781-8d4f-19ecbad2e2e0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='ADVECTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0cb7f2c6-5e99-4781-8d4f-19ecbad2e2e0'
  ),
  '81f51367-8467-4183-baea-6b526780fcc7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='BUOY POSITION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='81f51367-8467-4183-baea-6b526780fcc7'
  ),
  '10a9c153-f37d-48fe-920d-c790d946ab07': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='CONVECTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='10a9c153-f37d-48fe-920d-c790d946ab07'
  ),
  '6fe4680b-96e8-4304-ab32-c17a0769932c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='DIFFUSION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6fe4680b-96e8-4304-ab32-c17a0769932c'
  ),
  '13927300-c59c-491a-91f3-f1540bcb2d8d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='EDDIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='13927300-c59c-491a-91f3-f1540bcb2d8d'
  ),
  'fc95c990-47cb-4087-a08f-235dd1eb1260': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='EDDIES',
    variable_level_2='MESOSCALE EDDIES',
    variable_level_3='',
    detailed_variable='',
    identifier='fc95c990-47cb-4087-a08f-235dd1eb1260'
  ),
  '160c0b4d-6c03-4576-a4ea-f743a3a69d13': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='EDDIES',
    variable_level_2='SUBMESOSCALE EDDIES',
    variable_level_3='',
    detailed_variable='',
    identifier='160c0b4d-6c03-4576-a4ea-f743a3a69d13'
  ),
  '34033240-5675-4760-9fad-0c0d9b67b7ce': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='Finite Time Lyapunov Exponent',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='34033240-5675-4760-9fad-0c0d9b67b7ce'
  ),
  'bdd42024-d1a4-4fb2-a16a-06ac0cc1dedc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='FRESH WATER FLUX',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bdd42024-d1a4-4fb2-a16a-06ac0cc1dedc'
  ),
  '22b339b5-1af5-46e3-8191-d93729001eeb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='FRONTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='22b339b5-1af5-46e3-8191-d93729001eeb'
  ),
  'fc0a6bb2-27f0-48e8-89f1-ebfc7ccd4823': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='GYRES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fc0a6bb2-27f0-48e8-89f1-ebfc7ccd4823'
  ),
  'a110c0bd-0d41-4ac3-8463-16f97c57b266': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='Lagrangian feature',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a110c0bd-0d41-4ac3-8463-16f97c57b266'
  ),
  '510c5f78-e19e-4ce4-b59a-8937aeb84631': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='OCEAN CURRENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='510c5f78-e19e-4ce4-b59a-8937aeb84631'
  ),
  '811512a3-5138-43c5-99e5-d1373e2710a8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='OCEAN CURRENTS',
    variable_level_2='CURRENT PROFILES',
    variable_level_3='',
    detailed_variable='',
    identifier='811512a3-5138-43c5-99e5-d1373e2710a8'
  ),
  '39dfbc45-f1bc-496b-bb97-849aacf2feb7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='OCEAN CURRENTS',
    variable_level_2='HORIZONTAL CURRENT SHEAR',
    variable_level_3='',
    detailed_variable='',
    identifier='39dfbc45-f1bc-496b-bb97-849aacf2feb7'
  ),
  '7744f889-b25e-4d0e-bcf6-d94cbf63df22': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='OCEAN CURRENTS',
    variable_level_2='SPEED PROFILES',
    variable_level_3='',
    detailed_variable='',
    identifier='7744f889-b25e-4d0e-bcf6-d94cbf63df22'
  ),
  'abb21298-124f-4f12-92e8-affbb5c8fba8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='OCEAN CURRENTS',
    variable_level_2='SUBSURFACE CURRENTS',
    variable_level_3='',
    detailed_variable='',
    identifier='abb21298-124f-4f12-92e8-affbb5c8fba8'
  ),
  'b3647731-a71a-4af4-bfa2-e53b61efafeb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='OCEAN CURRENTS',
    variable_level_2='SURFACE CURRENTS',
    variable_level_3='',
    detailed_variable='',
    identifier='b3647731-a71a-4af4-bfa2-e53b61efafeb'
  ),
  '64bcd669-cbb0-41ff-a4bf-9ce1050d12c7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='OCEAN CURRENTS',
    variable_level_2='SURFACE SPEED',
    variable_level_3='',
    detailed_variable='',
    identifier='64bcd669-cbb0-41ff-a4bf-9ce1050d12c7'
  ),
  '499ad64c-bf58-498e-a885-fc99c6d76728': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='OCEAN CURRENTS',
    variable_level_2='VERTICAL CURRENT SHEAR',
    variable_level_3='',
    detailed_variable='',
    identifier='499ad64c-bf58-498e-a885-fc99c6d76728'
  ),
  'd96bcb09-f240-41cc-84d0-6af9fb3509de': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='OCEAN MASS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d96bcb09-f240-41cc-84d0-6af9fb3509de'
  ),
  '48ec6449-373c-41f6-8a61-8f1e9ed95737': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='OCEAN MIXED LAYER',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='48ec6449-373c-41f6-8a61-8f1e9ed95737'
  ),
  'c6f748f7-3a2a-4c76-90bd-8e8d7a691b21': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='SUBDUCTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c6f748f7-3a2a-4c76-90bd-8e8d7a691b21'
  ),
  'aa1bc71c-daeb-401e-9e29-ebde975482cf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='THERMOHALINE CIRCULATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='aa1bc71c-daeb-401e-9e29-ebde975482cf'
  ),
  'b9f343a1-0b8d-4e88-91bc-21f5d551963f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='TURBULENCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b9f343a1-0b8d-4e88-91bc-21f5d551963f'
  ),
  '75ab3537-34b1-4025-b758-7296626079ba': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='UPWELLING/DOWNWELLING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='75ab3537-34b1-4025-b758-7296626079ba'
  ),
  '55715ed3-471e-46a8-97b6-b463708a2cbe': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='VORTICITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='55715ed3-471e-46a8-97b6-b463708a2cbe'
  ),
  'aad49974-99ab-4623-a716-ea73e2f46ad1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='VORTICITY',
    variable_level_2='POTENTIAL VORTICITY',
    variable_level_3='',
    detailed_variable='',
    identifier='aad49974-99ab-4623-a716-ea73e2f46ad1'
  ),
  'ace6a51d-36af-4330-893f-d1fecc8ac904': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='VORTICITY',
    variable_level_2='RELATIVE VORTICITY',
    variable_level_3='',
    detailed_variable='',
    identifier='ace6a51d-36af-4330-893f-d1fecc8ac904'
  ),
  '113edd07-7b1a-4082-b054-b58d3f23b93a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='WATER MASSES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='113edd07-7b1a-4082-b054-b58d3f23b93a'
  ),
  '03fbea0a-74b9-4c78-8752-a588cff27f17': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN CIRCULATION',
    variable_level_1='WIND-DRIVEN CIRCULATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='03fbea0a-74b9-4c78-8752-a588cff27f17'
  ),
  '63bc0693-52eb-4ebd-a39e-e77e96409072': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN HEAT BUDGET',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='63bc0693-52eb-4ebd-a39e-e77e96409072'
  ),
  '5f6358aa-872c-4c1c-9388-4714138f034a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN HEAT BUDGET',
    variable_level_1='ADVECTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5f6358aa-872c-4c1c-9388-4714138f034a'
  ),
  'a9b6a001-42b2-48db-b132-62e69f03b8cb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN HEAT BUDGET',
    variable_level_1='BOWEN RATIO',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a9b6a001-42b2-48db-b132-62e69f03b8cb'
  ),
  '93c1a177-70e3-4c33-a183-baff7f401697': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN HEAT BUDGET',
    variable_level_1='CONDENSATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='93c1a177-70e3-4c33-a183-baff7f401697'
  ),
  '2ef42281-1e38-4391-b578-ba6a6158f0c2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN HEAT BUDGET',
    variable_level_1='CONDUCTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2ef42281-1e38-4391-b578-ba6a6158f0c2'
  ),
  '69d394f7-a792-4a17-8d7f-e60cd60dcda0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN HEAT BUDGET',
    variable_level_1='CONVECTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='69d394f7-a792-4a17-8d7f-e60cd60dcda0'
  ),
  '064d919e-3262-44c3-a636-8094bc963001': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN HEAT BUDGET',
    variable_level_1='DIFFUSION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='064d919e-3262-44c3-a636-8094bc963001'
  ),
  '881eea51-e32c-4174-a73f-d56c94122c2e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN HEAT BUDGET',
    variable_level_1='EVAPORATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='881eea51-e32c-4174-a73f-d56c94122c2e'
  ),
  'ee2cb9eb-f960-4e23-9e7c-be64d44a64e7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN HEAT BUDGET',
    variable_level_1='HEAT FLUX',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ee2cb9eb-f960-4e23-9e7c-be64d44a64e7'
  ),
  'cb02e3ec-d872-4944-b824-07fde2260599': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN HEAT BUDGET',
    variable_level_1='HEAT FLUX',
    variable_level_2='CONDUCTIVE HEAT FLUX',
    variable_level_3='',
    detailed_variable='',
    identifier='cb02e3ec-d872-4944-b824-07fde2260599'
  ),
  '258d6984-ff0c-40d5-9dc5-673d211e21e7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN HEAT BUDGET',
    variable_level_1='HEAT FLUX',
    variable_level_2='GEOTHERMAL HEAT FLUX',
    variable_level_3='',
    detailed_variable='',
    identifier='258d6984-ff0c-40d5-9dc5-673d211e21e7'
  ),
  'c7dc02a5-0db0-43cf-ac7a-8768b7ddda5f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN HEAT BUDGET',
    variable_level_1='HEAT FLUX',
    variable_level_2='LATENT HEAT FLUX',
    variable_level_3='',
    detailed_variable='',
    identifier='c7dc02a5-0db0-43cf-ac7a-8768b7ddda5f'
  ),
  '6a2d1d48-d2cc-4fe7-85f5-c98ab3a11262': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN HEAT BUDGET',
    variable_level_1='HEAT FLUX',
    variable_level_2='SENSIBLE HEAT FLUX',
    variable_level_3='',
    detailed_variable='',
    identifier='6a2d1d48-d2cc-4fe7-85f5-c98ab3a11262'
  ),
  '7ee32363-7c39-40c3-95b3-4f24e284abb6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN HEAT BUDGET',
    variable_level_1='HEAT FLUX',
    variable_level_2='TURBULENT HEAT FLUX',
    variable_level_3='',
    detailed_variable='',
    identifier='7ee32363-7c39-40c3-95b3-4f24e284abb6'
  ),
  'ed2e9f34-2358-4a2a-a83e-febba8989c5c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN HEAT BUDGET',
    variable_level_1='HEATING RATE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ed2e9f34-2358-4a2a-a83e-febba8989c5c'
  ),
  'bc891281-b24c-4310-b39b-81715d7dad08': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN HEAT BUDGET',
    variable_level_1='LONGWAVE RADIATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bc891281-b24c-4310-b39b-81715d7dad08'
  ),
  'd1426df9-7653-442b-8e38-fa28757ec748': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN HEAT BUDGET',
    variable_level_1='REFLECTANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d1426df9-7653-442b-8e38-fa28757ec748'
  ),
  '8d69bce7-efce-4efb-9870-6a6d3a2684fd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN HEAT BUDGET',
    variable_level_1='SHORTWAVE RADIATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8d69bce7-efce-4efb-9870-6a6d3a2684fd'
  ),
  '457883c4-b30c-4d26-bed8-6c2887ebbc90': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN OPTICS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='457883c4-b30c-4d26-bed8-6c2887ebbc90'
  ),
  'e501d002-d11e-4569-8c0d-e40ae5b45f65': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN OPTICS',
    variable_level_1='ABSORPTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e501d002-d11e-4569-8c0d-e40ae5b45f65'
  ),
  '4e8943e7-daf9-41f2-8a5e-b415b82e6381': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN OPTICS',
    variable_level_1='APHOTIC/PHOTIC ZONE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4e8943e7-daf9-41f2-8a5e-b415b82e6381'
  ),
  '71c78d69-9cfe-48e9-8dd2-9c75acf22283': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN OPTICS',
    variable_level_1='ATTENUATION/TRANSMISSION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='71c78d69-9cfe-48e9-8dd2-9c75acf22283'
  ),
  '90f97e5b-f883-4a34-a3bc-7dea8d96eb7d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN OPTICS',
    variable_level_1='BIOLUMINESCENCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='90f97e5b-f883-4a34-a3bc-7dea8d96eb7d'
  ),
  '15cc550b-068c-49f4-b082-bc2a43675606': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN OPTICS',
    variable_level_1='CHLOROPHYLL',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='15cc550b-068c-49f4-b082-bc2a43675606'
  ),
  '0f816677-9e94-4e3b-b409-513335769af8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN OPTICS',
    variable_level_1='CHLOROPHYLL',
    variable_level_2='CHLOROPHYLL CONCENTRATION',
    variable_level_3='',
    detailed_variable='',
    identifier='0f816677-9e94-4e3b-b409-513335769af8'
  ),
  '5f2ec7b9-3e8c-4d12-bba6-0f84c08729e0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN OPTICS',
    variable_level_1='EXTINCTION COEFFICIENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5f2ec7b9-3e8c-4d12-bba6-0f84c08729e0'
  ),
  'a60ae1b6-abfc-4905-8c09-772da7bb1a10': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN OPTICS',
    variable_level_1='FLUORESCENCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a60ae1b6-abfc-4905-8c09-772da7bb1a10'
  ),
  '87b074b4-9b73-4e69-b8c0-0f112b1cfa6d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN OPTICS',
    variable_level_1='GELBSTOFF',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='87b074b4-9b73-4e69-b8c0-0f112b1cfa6d'
  ),
  '40aacf7a-aba0-4ba2-bf85-ea7c39c3322c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN OPTICS',
    variable_level_1='IRRADIANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='40aacf7a-aba0-4ba2-bf85-ea7c39c3322c'
  ),
  '78f5a84f-1b5b-44a9-97e7-4a1996cd2e36': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN OPTICS',
    variable_level_1='OCEAN COLOR',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='78f5a84f-1b5b-44a9-97e7-4a1996cd2e36'
  ),
  '001f18d3-7e61-430b-9883-1960c6256fe5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN OPTICS',
    variable_level_1='OPTICAL DEPTH',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='001f18d3-7e61-430b-9883-1960c6256fe5'
  ),
  'b7410899-350a-4443-9430-c7fe1fa3a499': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN OPTICS',
    variable_level_1='PHOTOSYNTHETICALLY ACTIVE RADIATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b7410899-350a-4443-9430-c7fe1fa3a499'
  ),
  '68dacfbb-4f23-4325-b80f-4b09d41bd505': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN OPTICS',
    variable_level_1='RADIANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='68dacfbb-4f23-4325-b80f-4b09d41bd505'
  ),
  '4f7ad022-70ea-4254-b0ae-7a231fc2e46a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN OPTICS',
    variable_level_1='REFLECTANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4f7ad022-70ea-4254-b0ae-7a231fc2e46a'
  ),
  '20b41061-e6dc-47ef-b73b-00dc08a59618': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN OPTICS',
    variable_level_1='SCATTERING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='20b41061-e6dc-47ef-b73b-00dc08a59618'
  ),
  '954c2f25-3ec8-4774-ba34-fa4289f33f0e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN OPTICS',
    variable_level_1='SECCHI DEPTH',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='954c2f25-3ec8-4774-ba34-fa4289f33f0e'
  ),
  'f0d83687-bc0a-4491-bb3e-697f1018da13': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN OPTICS',
    variable_level_1='TURBIDITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f0d83687-bc0a-4491-bb3e-697f1018da13'
  ),
  'ad41b62a-141b-4207-887c-334367860cf4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN OPTICS',
    variable_level_1='WATER-LEAVING RADIANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ad41b62a-141b-4207-887c-334367860cf4'
  ),
  'bfa56100-6fb5-4e49-9633-298fa3b45508': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN PRESSURE',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bfa56100-6fb5-4e49-9633-298fa3b45508'
  ),
  '73311948-541c-4960-ae2b-2e82a79aa621': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN PRESSURE',
    variable_level_1='OCEAN BOTTOM PRESSURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='73311948-541c-4960-ae2b-2e82a79aa621'
  ),
  'e5bca08d-ecb3-4b85-8acd-fed782875aa2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN PRESSURE',
    variable_level_1='SEA LEVEL PRESSURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e5bca08d-ecb3-4b85-8acd-fed782875aa2'
  ),
  'dd025312-0d27-44e0-ae05-7cfcc1aa17f0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN PRESSURE',
    variable_level_1='WATER PRESSURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='dd025312-0d27-44e0-ae05-7cfcc1aa17f0'
  ),
  '251c87cd-03b3-464f-8390-8ede2fec28fc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN TEMPERATURE',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='251c87cd-03b3-464f-8390-8ede2fec28fc'
  ),
  'a76f878d-c6fb-49bf-9165-3cac5fb61d80': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN TEMPERATURE',
    variable_level_1='OCEAN BARRIER LAYER',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a76f878d-c6fb-49bf-9165-3cac5fb61d80'
  ),
  '64074461-95d0-4538-869a-0114e39216aa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN TEMPERATURE',
    variable_level_1='OCEAN MIXED LAYER',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='64074461-95d0-4538-869a-0114e39216aa'
  ),
  'f952e80e-77de-4dc8-aa6b-0f3be186aba5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN TEMPERATURE',
    variable_level_1='OCEAN TEMPERATURE PROFILES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f952e80e-77de-4dc8-aa6b-0f3be186aba5'
  ),
  'e02b0b50-a0f2-4c47-841b-9689fdb99121': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN TEMPERATURE',
    variable_level_1='POTENTIAL TEMPERATURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e02b0b50-a0f2-4c47-841b-9689fdb99121'
  ),
  'bd24a9a9-7d52-4c29-b2a0-6cefd216ae78': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN TEMPERATURE',
    variable_level_1='SEA SURFACE TEMPERATURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bd24a9a9-7d52-4c29-b2a0-6cefd216ae78'
  ),
  'e4d58a7f-7eaa-4f75-996a-18238c698063': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN TEMPERATURE',
    variable_level_1='SEA SURFACE TEMPERATURE',
    variable_level_2='SEA SURFACE FOUNDATION TEMPERATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='e4d58a7f-7eaa-4f75-996a-18238c698063'
  ),
  'cd5a4729-ea4a-4ce1-8f5a-ec6a76d31055': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN TEMPERATURE',
    variable_level_1='SEA SURFACE TEMPERATURE',
    variable_level_2='SEA SURFACE SKIN TEMPERATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='cd5a4729-ea4a-4ce1-8f5a-ec6a76d31055'
  ),
  '68a09c56-be36-4100-8757-3a6eec7dc251': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN TEMPERATURE',
    variable_level_1='SEA SURFACE TEMPERATURE',
    variable_level_2='SEA SURFACE SUBSKIN TEMPERATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='68a09c56-be36-4100-8757-3a6eec7dc251'
  ),
  '904f3b34-20c8-4eb8-bf68-6304edecf945': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN TEMPERATURE',
    variable_level_1='SEA SURFACE TEMPERATURE',
    variable_level_2='SEA SURFACE TEMPERATURE ANOMALY',
    variable_level_3='',
    detailed_variable='',
    identifier='904f3b34-20c8-4eb8-bf68-6304edecf945'
  ),
  '68772b70-e493-48d5-b063-00b9d2dd4078': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN TEMPERATURE',
    variable_level_1='THERMOCLINE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='68772b70-e493-48d5-b063-00b9d2dd4078'
  ),
  '46206e8c-8def-406f-9e62-da4e74633a58': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN TEMPERATURE',
    variable_level_1='WATER TEMPERATURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='46206e8c-8def-406f-9e62-da4e74633a58'
  ),
  '54b47174-d035-4b9c-99a5-27b39c7f0f17': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN VOLUME BUDGET',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='54b47174-d035-4b9c-99a5-27b39c7f0f17'
  ),
  '30fd009d-df91-47ba-8800-2f2771f15e80': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN VOLUME BUDGET',
    variable_level_1='ADVECTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='30fd009d-df91-47ba-8800-2f2771f15e80'
  ),
  '4fc9280e-f273-4280-a732-93ef4ceea418': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN VOLUME BUDGET',
    variable_level_1='DIFFUSION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4fc9280e-f273-4280-a732-93ef4ceea418'
  ),
  '9e8257c6-5c14-4707-995a-d31409265407': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN VOLUME BUDGET',
    variable_level_1='EVAPORATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9e8257c6-5c14-4707-995a-d31409265407'
  ),
  '32d0ca35-bff8-4e63-9c5b-c70d1593824a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN VOLUME BUDGET',
    variable_level_1='ICE GROWTH/MELT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='32d0ca35-bff8-4e63-9c5b-c70d1593824a'
  ),
  '6c71e621-aae6-436b-9405-5dc6ed2a527e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN VOLUME BUDGET',
    variable_level_1='PRECIPITATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6c71e621-aae6-436b-9405-5dc6ed2a527e'
  ),
  '67290503-94b9-4517-b5b6-063bba2bee27': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN VOLUME BUDGET',
    variable_level_1='RUNOFF',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='67290503-94b9-4517-b5b6-063bba2bee27'
  ),
  '18323b62-5f66-4878-843b-cbde545dd775': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN VOLUME BUDGET',
    variable_level_1='SNOW MELT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='18323b62-5f66-4878-843b-cbde545dd775'
  ),
  'a04804d5-1064-48fd-a7a7-8da8e10399e1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a04804d5-1064-48fd-a7a7-8da8e10399e1'
  ),
  'dc9fcd27-58ac-4705-a522-6475d59cfb81': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='GRAVITY WAVES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='dc9fcd27-58ac-4705-a522-6475d59cfb81'
  ),
  '41764af0-1264-4adb-881d-44991489344c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='ROSSBY/PLANETARY WAVES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='41764af0-1264-4adb-881d-44991489344c'
  ),
  '11aca777-8a01-42ce-b076-b3059c3d8cae': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='SEA STATE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='11aca777-8a01-42ce-b076-b3059c3d8cae'
  ),
  '2b4963ba-1a7a-419d-97ef-eacaa14688e0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='SEICHES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2b4963ba-1a7a-419d-97ef-eacaa14688e0'
  ),
  '1ac6850e-9266-4e90-ba83-b6a6cc4ae365': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='SIGNIFICANT WAVE HEIGHT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1ac6850e-9266-4e90-ba83-b6a6cc4ae365'
  ),
  '0bf50cd4-8a97-468c-8e73-047e3e09a03d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='STORM SURGE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0bf50cd4-8a97-468c-8e73-047e3e09a03d'
  ),
  'a90526a9-5476-45bc-9a15-73ac2dfc62ab': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='SURF BEAT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a90526a9-5476-45bc-9a15-73ac2dfc62ab'
  ),
  '4e4d3c18-cdd4-474a-a936-6e127ec526f7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='SWELLS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4e4d3c18-cdd4-474a-a936-6e127ec526f7'
  ),
  'b23597aa-ccb2-40be-920d-5663769cd502': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='SWELLS',
    variable_level_2='SWELL DIRECTION',
    variable_level_3='',
    detailed_variable='',
    identifier='b23597aa-ccb2-40be-920d-5663769cd502'
  ),
  '5e9ad407-bd70-43ae-a901-6b07f100db27': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='SWELLS',
    variable_level_2='SWELL HEIGHT',
    variable_level_3='',
    detailed_variable='',
    identifier='5e9ad407-bd70-43ae-a901-6b07f100db27'
  ),
  '7a8920f3-e531-47e0-bb23-a9f816cfb7bf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='SWELLS',
    variable_level_2='SWELL PERIOD',
    variable_level_3='',
    detailed_variable='',
    identifier='7a8920f3-e531-47e0-bb23-a9f816cfb7bf'
  ),
  '3dd99ea6-51bd-4b78-bf2e-d5aeca7f5bc8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='TOPOGRAPHIC WAVES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3dd99ea6-51bd-4b78-bf2e-d5aeca7f5bc8'
  ),
  '7a79a3f3-1817-4c9f-8485-550a022b5a8d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='TSUNAMIS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7a79a3f3-1817-4c9f-8485-550a022b5a8d'
  ),
  '037ce518-b71f-4599-b37f-feab9cc9809d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='WAVE DIRECTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='037ce518-b71f-4599-b37f-feab9cc9809d'
  ),
  '09b326df-79b3-41b8-8998-e06344b0fe0d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='WAVE FETCH',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='09b326df-79b3-41b8-8998-e06344b0fe0d'
  ),
  '0d91f6d9-44c4-4418-90b0-00feb09c6fc0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='WAVE FREQUENCY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0d91f6d9-44c4-4418-90b0-00feb09c6fc0'
  ),
  '0fc68280-1361-43e1-bc5a-40c49e9679b7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='WAVE HEIGHT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0fc68280-1361-43e1-bc5a-40c49e9679b7'
  ),
  '5daa972e-b47c-4050-97f1-1e628401fb97': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='WAVE LENGTH',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5daa972e-b47c-4050-97f1-1e628401fb97'
  ),
  '5377fb64-b10a-4284-9b7b-be77b4c16fe5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='WAVE OVERTOPPING',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5377fb64-b10a-4284-9b7b-be77b4c16fe5'
  ),
  '99ea6719-b751-4a4f-95d4-aaa02e961bc1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='WAVE PERIOD',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='99ea6719-b751-4a4f-95d4-aaa02e961bc1'
  ),
  '9a4816c1-dba8-4ae4-9c3b-7f98a4ac245b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='WAVE RUNUP',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9a4816c1-dba8-4ae4-9c3b-7f98a4ac245b'
  ),
  '4dd520ea-30fc-416d-b98c-340fd23431d3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='WAVE SETUP',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4dd520ea-30fc-416d-b98c-340fd23431d3'
  ),
  'e79ff727-c598-4a1c-8b4f-b6019fcf386b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='WAVE SPECTRA',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e79ff727-c598-4a1c-8b4f-b6019fcf386b'
  ),
  'd02bae1c-b05e-4c56-b964-7f49610efc3b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='WAVE SPEED',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d02bae1c-b05e-4c56-b964-7f49610efc3b'
  ),
  'e52114b2-adbc-4e3e-9c87-1a7f245fe5ef': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='WAVE SPEED/DIRECTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e52114b2-adbc-4e3e-9c87-1a7f245fe5ef'
  ),
  'a4f0e0d2-4bcb-4675-b874-e6e0f3a8c462': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='WAVE TYPES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a4f0e0d2-4bcb-4675-b874-e6e0f3a8c462'
  ),
  '0c9adb35-b203-42d7-8ccf-b7f2079db7ce': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WAVES',
    variable_level_1='WIND WAVES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='0c9adb35-b203-42d7-8ccf-b7f2079db7ce'
  ),
  '346cade5-801a-4afc-9652-48d02905bc4f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WINDS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='346cade5-801a-4afc-9652-48d02905bc4f'
  ),
  'b59e188c-49b8-41b3-94c4-0bc1dbb554fe': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WINDS',
    variable_level_1='CONVERGENCE/DIVERGENCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b59e188c-49b8-41b3-94c4-0bc1dbb554fe'
  ),
  'fbc53539-ce4e-4e3e-bbd2-8270386616b4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WINDS',
    variable_level_1='SURFACE WINDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fbc53539-ce4e-4e3e-bbd2-8270386616b4'
  ),
  'd78e5503-d78e-466d-97bb-e68d6e768a9d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WINDS',
    variable_level_1='SURFACE WINDS',
    variable_level_2='WIND DIRECTION',
    variable_level_3='',
    detailed_variable='',
    identifier='d78e5503-d78e-466d-97bb-e68d6e768a9d'
  ),
  'a7ce84a3-8329-4eb7-b5de-72d2dea8c6bf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WINDS',
    variable_level_1='SURFACE WINDS',
    variable_level_2='WIND SPEED',
    variable_level_3='',
    detailed_variable='',
    identifier='a7ce84a3-8329-4eb7-b5de-72d2dea8c6bf'
  ),
  '13aeaea0-ab45-4148-abcf-c6becf7a8934': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WINDS',
    variable_level_1='TURBULENCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='13aeaea0-ab45-4148-abcf-c6becf7a8934'
  ),
  'ab1e152c-eab9-400a-a90f-15cb64ed2a75': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WINDS',
    variable_level_1='VERTICAL WIND MOTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ab1e152c-eab9-400a-a90f-15cb64ed2a75'
  ),
  '253ccaf2-dd4c-4fc1-923d-1aea542a51b0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WINDS',
    variable_level_1='VORTICITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='253ccaf2-dd4c-4fc1-923d-1aea542a51b0'
  ),
  'd571e1f5-7449-4052-943b-94d76f762677': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WINDS',
    variable_level_1='WIND CHILL',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d571e1f5-7449-4052-943b-94d76f762677'
  ),
  '855c22f5-d1e0-4ccf-81bd-c8120e7c4055': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WINDS',
    variable_level_1='WIND SHEAR',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='855c22f5-d1e0-4ccf-81bd-c8120e7c4055'
  ),
  '91d73256-925d-4d04-9b55-aaf088080cac': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WINDS',
    variable_level_1='WIND STRESS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='91d73256-925d-4d04-9b55-aaf088080cac'
  ),
  'b9a716e9-970e-44e0-9faf-66647f5a59ed': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='OCEAN WINDS',
    variable_level_1='WIND VELOCITY/SPEED',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b9a716e9-970e-44e0-9faf-66647f5a59ed'
  ),
  'ea213be5-fe37-4179-9a9b-030c2bf42cf5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='PRECIPITATION',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ea213be5-fe37-4179-9a9b-030c2bf42cf5'
  ),
  '4ca02520-1345-475c-9a54-b562a042c4e1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='PRECIPITATION',
    variable_level_1='LIQUID PRECIPITATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4ca02520-1345-475c-9a54-b562a042c4e1'
  ),
  '5953d403-41e8-48f7-a4a8-06aaa633e12a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='PRECIPITATION',
    variable_level_1='LIQUID PRECIPITATION',
    variable_level_2='RAIN',
    variable_level_3='',
    detailed_variable='',
    identifier='5953d403-41e8-48f7-a4a8-06aaa633e12a'
  ),
  'c40d3dbc-5d7e-434d-996e-120ba44a5d44': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='PRECIPITATION',
    variable_level_1='SOLID PRECIPITATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c40d3dbc-5d7e-434d-996e-120ba44a5d44'
  ),
  '09c21315-faba-446c-b060-136705972347': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='PRECIPITATION',
    variable_level_1='SOLID PRECIPITATION',
    variable_level_2='SNOW',
    variable_level_3='',
    detailed_variable='',
    identifier='09c21315-faba-446c-b060-136705972347'
  ),
  'a46016d7-e571-403a-ab37-7223fd74e68e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a46016d7-e571-403a-ab37-7223fd74e68e'
  ),
  '7041e51c-e2de-405a-b154-6016f624f54f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='CONDUCTIVITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7041e51c-e2de-405a-b154-6016f624f54f'
  ),
  '9709d1cb-e165-4aa5-be87-daa2989aac31': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='CONDUCTIVITY',
    variable_level_2='CONDUCTIVITY PROFILES',
    variable_level_3='',
    detailed_variable='',
    identifier='9709d1cb-e165-4aa5-be87-daa2989aac31'
  ),
  'a819235a-68b0-46f2-9d96-49b73fd31092': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='CONDUCTIVITY',
    variable_level_2='SURFACE CONDUCTIVITY',
    variable_level_3='',
    detailed_variable='',
    identifier='a819235a-68b0-46f2-9d96-49b73fd31092'
  ),
  '007ab607-2ee1-484d-85fb-0bfb89f18c9b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='DENSITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='007ab607-2ee1-484d-85fb-0bfb89f18c9b'
  ),
  '41926d67-161a-4add-bb12-66038c919efb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='DESALINIZATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='41926d67-161a-4add-bb12-66038c919efb'
  ),
  '04305c55-14f0-42a3-a099-79eb326946d7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='HALOCLINE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='04305c55-14f0-42a3-a099-79eb326946d7'
  ),
  '1a4e5774-7d4a-4ce7-9a4c-e2c72c8c377f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='OCEAN SALINITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1a4e5774-7d4a-4ce7-9a4c-e2c72c8c377f'
  ),
  'f1964fd8-9ab6-4f36-b761-131ff79a12bc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='OCEAN SALINITY',
    variable_level_2='ABSOLUTE SALINITY',
    variable_level_3='',
    detailed_variable='',
    identifier='f1964fd8-9ab6-4f36-b761-131ff79a12bc'
  ),
  '8c0570f6-c5d1-4675-99be-68d9d3b9d90c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='OCEAN SALINITY',
    variable_level_2='OCEAN SALINITY BUDGET',
    variable_level_3='',
    detailed_variable='',
    identifier='8c0570f6-c5d1-4675-99be-68d9d3b9d90c'
  ),
  '470e3f31-86af-4a9b-9279-ce1ba125d1dd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='OCEAN SALINITY',
    variable_level_2='OCEAN SALINITY BUDGET',
    variable_level_3='ADVECTION',
    detailed_variable='',
    identifier='470e3f31-86af-4a9b-9279-ce1ba125d1dd'
  ),
  '73432d3c-341a-48e7-a765-30395ce588be': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='OCEAN SALINITY',
    variable_level_2='OCEAN SALINITY BUDGET',
    variable_level_3='DIFFUSION',
    detailed_variable='',
    identifier='73432d3c-341a-48e7-a765-30395ce588be'
  ),
  '1cc632bd-8ed6-46ae-8948-15d7b3e1524a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='OCEAN SALINITY',
    variable_level_2='OCEAN SALINITY BUDGET',
    variable_level_3='EVAPORATION',
    detailed_variable='',
    identifier='1cc632bd-8ed6-46ae-8948-15d7b3e1524a'
  ),
  'ac5ba325-b613-4f2e-ae7d-f81e478f5091': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='OCEAN SALINITY',
    variable_level_2='OCEAN SALINITY BUDGET',
    variable_level_3='ICE GROWTH/MELT',
    detailed_variable='',
    identifier='ac5ba325-b613-4f2e-ae7d-f81e478f5091'
  ),
  'bfa93e9c-392f-4f59-8cc8-0866e24531f5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='OCEAN SALINITY',
    variable_level_2='OCEAN SALINITY BUDGET',
    variable_level_3='PRECIPITATION',
    detailed_variable='',
    identifier='bfa93e9c-392f-4f59-8cc8-0866e24531f5'
  ),
  '132ef849-3da3-4252-8f70-8dd36e790844': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='OCEAN SALINITY',
    variable_level_2='OCEAN SALINITY BUDGET',
    variable_level_3='RUNOFF',
    detailed_variable='',
    identifier='132ef849-3da3-4252-8f70-8dd36e790844'
  ),
  'e22f7a00-3f3e-48ce-82c5-f69203239570': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='OCEAN SALINITY',
    variable_level_2='OCEAN SALINITY BUDGET',
    variable_level_3='SNOW MELT',
    detailed_variable='',
    identifier='e22f7a00-3f3e-48ce-82c5-f69203239570'
  ),
  '972d17d7-7dea-4df2-bec5-24e8ca873dbd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='OCEAN SALINITY',
    variable_level_2='OCEAN SALINITY PROFILES',
    variable_level_3='',
    detailed_variable='',
    identifier='972d17d7-7dea-4df2-bec5-24e8ca873dbd'
  ),
  '1544c1fe-58dd-4b19-bf9e-457b4f21ef29': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='OCEAN SALINITY',
    variable_level_2='OCEAN SURFACE SALINITY',
    variable_level_3='',
    detailed_variable='',
    identifier='1544c1fe-58dd-4b19-bf9e-457b4f21ef29'
  ),
  '9c778b59-6ed9-442e-897e-c48ce5baa3b0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='OCEAN SALINITY',
    variable_level_2='PRACTICAL SALINITY',
    variable_level_3='',
    detailed_variable='',
    identifier='9c778b59-6ed9-442e-897e-c48ce5baa3b0'
  ),
  '376dde0f-a750-4138-91ce-8ca635bf05bd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='OCEAN SALINITY',
    variable_level_2='SALT FLUX',
    variable_level_3='',
    detailed_variable='',
    identifier='376dde0f-a750-4138-91ce-8ca635bf05bd'
  ),
  'fe4a246b-4614-422b-8ca5-0481ee417318': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='POTENTIAL DENSITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fe4a246b-4614-422b-8ca5-0481ee417318'
  ),
  '2ad73f85-8bad-4e5a-a902-e83eee910b5e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='PYCNOCLINE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2ad73f85-8bad-4e5a-a902-e83eee910b5e'
  ),
  '7e95b5fc-1d58-431a-af36-948b29fa870d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='SALINITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7e95b5fc-1d58-431a-af36-948b29fa870d'
  ),
  '15f87fbc-b972-403f-97c0-15f387a13efe': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SALINITY/DENSITY',
    variable_level_1='SALT TRANSPORT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='15f87fbc-b972-403f-97c0-15f387a13efe'
  ),
  'd73e969a-4b66-4713-8d63-fa3cbb1e25e3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d73e969a-4b66-4713-8d63-fa3cbb1e25e3'
  ),
  'ae1c9b54-caf2-4726-b180-5c6544f09111': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='HEAT FLUX',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ae1c9b54-caf2-4726-b180-5c6544f09111'
  ),
  '3cdebef6-902d-4c1a-9d7e-7609f8ee6ef6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='ICE DEFORMATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3cdebef6-902d-4c1a-9d7e-7609f8ee6ef6'
  ),
  'a735d8ca-182c-4307-9305-186a065e84a4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='ICE DEPTH/THICKNESS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a735d8ca-182c-4307-9305-186a065e84a4'
  ),
  'ae4869cc-65f4-4f24-a77b-b77637f8818c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='ICE DRAFT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ae4869cc-65f4-4f24-a77b-b77637f8818c'
  ),
  'f0cd20bd-41e8-4ca0-9ae3-7c602c251858': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='ICE EDGES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f0cd20bd-41e8-4ca0-9ae3-7c602c251858'
  ),
  '87feb47e-aee3-42f1-8c39-5109d9d5422e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='ICE EXTENT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='87feb47e-aee3-42f1-8c39-5109d9d5422e'
  ),
  'aa15804c-5f7f-40cc-b949-aa3e4418fc27': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='ICE FLOES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='aa15804c-5f7f-40cc-b949-aa3e4418fc27'
  ),
  '89fc22ca-326e-468c-ad3d-171c4ad34977': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='ICE GROWTH/MELT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='89fc22ca-326e-468c-ad3d-171c4ad34977'
  ),
  'a6c3e78f-f408-4b72-941a-f40e3d83dd60': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='ICE ROUGHNESS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a6c3e78f-f408-4b72-941a-f40e3d83dd60'
  ),
  '2a664e2d-4e50-463a-af9f-b14b86eb42a7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='ICE TEMPERATURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2a664e2d-4e50-463a-af9f-b14b86eb42a7'
  ),
  'f5d7cafc-13bf-4ec8-bc6e-a6d850fae5c8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='ICE TYPES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f5d7cafc-13bf-4ec8-bc6e-a6d850fae5c8'
  ),
  '1151dc7e-7441-4a21-95b6-1d03a1053f60': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='ICEBERGS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1151dc7e-7441-4a21-95b6-1d03a1053f60'
  ),
  '9d99408d-0d8b-4642-a2cb-edee8319fe1d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='ISOTOPES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9d99408d-0d8b-4642-a2cb-edee8319fe1d'
  ),
  'f523f73f-efcc-4193-b9e3-1161ed7f4881': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='LEADS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f523f73f-efcc-4193-b9e3-1161ed7f4881'
  ),
  '7b5939eb-4101-40bc-8dfe-810f6e9e5d6f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='MELT POND',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7b5939eb-4101-40bc-8dfe-810f6e9e5d6f'
  ),
  'ea85ea0b-1b7d-464a-9f8c-1f80383ffc51': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='PACK ICE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ea85ea0b-1b7d-464a-9f8c-1f80383ffc51'
  ),
  '10a128a6-12d4-4bce-b25d-2ffc464182f4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='POLYNYAS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='10a128a6-12d4-4bce-b25d-2ffc464182f4'
  ),
  '8ed9f39d-986e-4b36-83f9-f29f6a4df89b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='REFLECTANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8ed9f39d-986e-4b36-83f9-f29f6a4df89b'
  ),
  '04fa9023-ab68-4dd0-a82e-abe685105a53': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='SALINITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='04fa9023-ab68-4dd0-a82e-abe685105a53'
  ),
  'b6085d71-a7ee-4b65-9c9c-ff374bdc3974': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='SEA ICE AGE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b6085d71-a7ee-4b65-9c9c-ff374bdc3974'
  ),
  '354c4f50-ccf9-4714-85ee-b6c028521bef': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='SEA ICE AREA',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='354c4f50-ccf9-4714-85ee-b6c028521bef'
  ),
  'bb27bbb7-7bc4-4e38-833a-30e0a7861ccc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='SEA ICE CONCENTRATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bb27bbb7-7bc4-4e38-833a-30e0a7861ccc'
  ),
  'defd2c00-64e3-4986-9061-feade19f972f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='SEA ICE DYNAMICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='defd2c00-64e3-4986-9061-feade19f972f'
  ),
  '6e2f1371-05b1-41db-a6d9-bccd7cc2b3da': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='SEA ICE ELEVATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6e2f1371-05b1-41db-a6d9-bccd7cc2b3da'
  ),
  '4519a99a-ffa7-436b-888a-c742c82b9ed1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='SEA ICE MASS BALANCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4519a99a-ffa7-436b-888a-c742c82b9ed1'
  ),
  'a47ab696-7ed9-4374-8965-c8996e61463d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='SEA ICE MOTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a47ab696-7ed9-4374-8965-c8996e61463d'
  ),
  'ab411758-5abe-4a89-9319-97eba1510cda': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='SEA ICE MOTION',
    variable_level_2='SEA ICE DIRECTION',
    variable_level_3='',
    detailed_variable='',
    identifier='ab411758-5abe-4a89-9319-97eba1510cda'
  ),
  'ad4646f8-bf09-4751-9072-e6cec59af253': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='SEA ICE MOTION',
    variable_level_2='SEA ICE SPEED',
    variable_level_3='',
    detailed_variable='',
    identifier='ad4646f8-bf09-4751-9072-e6cec59af253'
  ),
  '48d9b511-8e99-4b6d-a0e8-e87b71bd172e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='SEA ICE ORIGIN',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='48d9b511-8e99-4b6d-a0e8-e87b71bd172e'
  ),
  '15547b03-1b99-4c60-92eb-216a2908f504': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='SEA ICE TOPOGRAPHY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='15547b03-1b99-4c60-92eb-216a2908f504'
  ),
  '32929f40-ee7f-411d-8d2d-1d2cd9b78b09': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='SEA ICE VOLUME',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='32929f40-ee7f-411d-8d2d-1d2cd9b78b09'
  ),
  'dc2d2e73-4028-41d2-96f9-0b800fa95ea4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='SEA ICE/OCEAN CLASSIFICATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='dc2d2e73-4028-41d2-96f9-0b800fa95ea4'
  ),
  '5575125b-7f15-4d46-ba47-f86de96a1a25': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='SNOW DEPTH',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5575125b-7f15-4d46-ba47-f86de96a1a25'
  ),
  '32259124-81f7-4845-b2fb-6435d7bb5804': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA ICE',
    variable_level_1='SNOW MELT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='32259124-81f7-4845-b2fb-6435d7bb5804'
  ),
  '68f93a0c-1525-4f5a-9545-5d94191a3dbf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA SURFACE TOPOGRAPHY',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='68f93a0c-1525-4f5a-9545-5d94191a3dbf'
  ),
  '940550d2-1d9f-4c28-b9ba-857c2dc8ef95': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA SURFACE TOPOGRAPHY',
    variable_level_1='DYNAMIC TOPOGRAPHY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='940550d2-1d9f-4c28-b9ba-857c2dc8ef95'
  ),
  '7e1fc68e-5a7e-4a59-8ae6-3fa15bdae12d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA SURFACE TOPOGRAPHY',
    variable_level_1='DYNAMIC TOPOGRAPHY',
    variable_level_2='ABSOLUTE DYNAMIC TOPOGRAPHY',
    variable_level_3='',
    detailed_variable='',
    identifier='7e1fc68e-5a7e-4a59-8ae6-3fa15bdae12d'
  ),
  'cf89619d-c67d-43f0-a217-c8684ce7c984': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA SURFACE TOPOGRAPHY',
    variable_level_1='DYNAMIC TOPOGRAPHY',
    variable_level_2='MEAN DYNAMIC TOPOGRAPHY',
    variable_level_3='',
    detailed_variable='',
    identifier='cf89619d-c67d-43f0-a217-c8684ce7c984'
  ),
  '70082342-c777-49e9-88e5-a4a77728d3cc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA SURFACE TOPOGRAPHY',
    variable_level_1='MEAN SEA SURFACE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='70082342-c777-49e9-88e5-a4a77728d3cc'
  ),
  '9ac7a1c5-4179-47bc-8589-ebaa90d6cbd1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA SURFACE TOPOGRAPHY',
    variable_level_1='SEA LEVEL',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9ac7a1c5-4179-47bc-8589-ebaa90d6cbd1'
  ),
  'f3ea8884-87a8-4a12-96d5-98e21a9fa2c7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA SURFACE TOPOGRAPHY',
    variable_level_1='SEA LEVEL',
    variable_level_2='MEAN SEA LEVEL',
    variable_level_3='',
    detailed_variable='',
    identifier='f3ea8884-87a8-4a12-96d5-98e21a9fa2c7'
  ),
  '0fde8353-9773-4948-b206-9c273c2100c8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA SURFACE TOPOGRAPHY',
    variable_level_1='SEA LEVEL',
    variable_level_2='SEA LEVEL ANOMALY',
    variable_level_3='',
    detailed_variable='',
    identifier='0fde8353-9773-4948-b206-9c273c2100c8'
  ),
  '5c0b448c-7eb4-4e8c-8403-260cbb6114bb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA SURFACE TOPOGRAPHY',
    variable_level_1='SEA SURFACE HEIGHT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5c0b448c-7eb4-4e8c-8403-260cbb6114bb'
  ),
  '3798a6c9-9b39-4e22-bee4-be80d39049fe': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA SURFACE TOPOGRAPHY',
    variable_level_1='SEA SURFACE HEIGHT',
    variable_level_2='SEA SURFACE HEIGHT ANOMALY (SSHA)',
    variable_level_3='',
    detailed_variable='',
    identifier='3798a6c9-9b39-4e22-bee4-be80d39049fe'
  ),
  '52a32bd3-d701-49e1-a827-67b3d96d8e56': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='SEA SURFACE TOPOGRAPHY',
    variable_level_1='SEA SURFACE SLOPE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='52a32bd3-d701-49e1-a827-67b3d96d8e56'
  ),
  'e3bef663-6116-4f15-995c-38c7cdc9652c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='TIDES',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e3bef663-6116-4f15-995c-38c7cdc9652c'
  ),
  '062be713-9c35-458e-86e2-26cea9415f5d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='TIDES',
    variable_level_1='STORM SURGE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='062be713-9c35-458e-86e2-26cea9415f5d'
  ),
  'f4f40ec7-e698-4e11-b406-a0fa7f4b530c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='TIDES',
    variable_level_1='TIDAL COMPONENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f4f40ec7-e698-4e11-b406-a0fa7f4b530c'
  ),
  '54ab2e0e-8e36-48e8-b020-ea9a5b453373': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='TIDES',
    variable_level_1='TIDAL CURRENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='54ab2e0e-8e36-48e8-b020-ea9a5b453373'
  ),
  '9afcf69c-f56f-45a9-afd9-6f929850326b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='TIDES',
    variable_level_1='TIDAL HEIGHT',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9afcf69c-f56f-45a9-afd9-6f929850326b'
  ),
  'a5a6266a-9457-4acf-b140-fcdc8bc00a00': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='TIDES',
    variable_level_1='TIDAL RANGE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a5a6266a-9457-4acf-b140-fcdc8bc00a00'
  ),
  '1ee8a323-f0ba-4a21-b597-50890c527c8e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='WATER QUALITY',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1ee8a323-f0ba-4a21-b597-50890c527c8e'
  ),
  '940cfe3b-e5b2-47f0-8e86-e3a5bd231a3f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='WATER QUALITY',
    variable_level_1='ALGAL BLOOMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='940cfe3b-e5b2-47f0-8e86-e3a5bd231a3f'
  ),
  'ba506291-2799-4877-a886-8e906704a060': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='WATER QUALITY',
    variable_level_1='ALGAL BLOOMS',
    variable_level_2='HARMFUL ALGAL BLOOM (HABs)',
    variable_level_3='',
    detailed_variable='',
    identifier='ba506291-2799-4877-a886-8e906704a060'
  ),
  'bce85eb7-e9fc-48ed-9595-9d45c4482728': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='WATER QUALITY',
    variable_level_1='ALGAL BLOOMS',
    variable_level_2='HARMFUL ALGAL BLOOM (HABs)',
    variable_level_3='CELL CONCENTRATION',
    detailed_variable='',
    identifier='bce85eb7-e9fc-48ed-9595-9d45c4482728'
  ),
  '631935dd-0a9f-4627-bbb1-c224ac0a7766': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='WATER QUALITY',
    variable_level_1='ALGAL BLOOMS',
    variable_level_2='HARMFUL ALGAL BLOOM (HABs)',
    variable_level_3='TOXIN CONCENTRATION',
    detailed_variable='',
    identifier='631935dd-0a9f-4627-bbb1-c224ac0a7766'
  ),
  'f1ee3e81-09b9-48d4-81d9-5faeb90430cc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='WATER QUALITY',
    variable_level_1='OCEAN CONTAMINANTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f1ee3e81-09b9-48d4-81d9-5faeb90430cc'
  ),
  'ff13560a-161c-4ac9-b79c-4910936cf465': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='WATER QUALITY',
    variable_level_1='SEA SURFACE CONTAMINANTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ff13560a-161c-4ac9-b79c-4910936cf465'
  ),
  '4e0ac490-817b-4735-93a3-ab6775486023': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='OCEANS',
    term='WATER QUALITY',
    variable_level_1='SEA SURFACE CONTAMINANTS',
    variable_level_2='MICROPLASTIC CONCENTRATION',
    variable_level_3='',
    detailed_variable='',
    identifier='4e0ac490-817b-4735-93a3-ab6775486023'
  ),
  'c7245882-84a1-4192-acfa-a758b5b9c151': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c7245882-84a1-4192-acfa-a758b5b9c151'
  ),
  'dba19648-3f52-48ba-b00b-8527d44c4d74': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='ICE CORE RECORDS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='dba19648-3f52-48ba-b00b-8527d44c4d74'
  ),
  '37dac8df-b04b-4561-91fb-886e1bded2c1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='ICE CORE RECORDS',
    variable_level_1='CARBON DIOXIDE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='37dac8df-b04b-4561-91fb-886e1bded2c1'
  ),
  '01a4a324-cad3-441d-b0f1-02dc9742784a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='ICE CORE RECORDS',
    variable_level_1='ELECTRICAL PROPERTIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='01a4a324-cad3-441d-b0f1-02dc9742784a'
  ),
  '3643618f-3af3-4c69-8beb-2ad14141a176': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='ICE CORE RECORDS',
    variable_level_1='ICE CORE AIR BUBBLES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3643618f-3af3-4c69-8beb-2ad14141a176'
  ),
  '591d2038-5c5e-47bf-a551-1b28f33d1f05': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='ICE CORE RECORDS',
    variable_level_1='IONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='591d2038-5c5e-47bf-a551-1b28f33d1f05'
  ),
  '096f466a-a86a-42ac-93f7-05a799910817': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='ICE CORE RECORDS',
    variable_level_1='ISOTOPES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='096f466a-a86a-42ac-93f7-05a799910817'
  ),
  'daab3b2a-1fe7-4ad9-8340-1a7cfa54a2ac': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='ICE CORE RECORDS',
    variable_level_1='METHANE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='daab3b2a-1fe7-4ad9-8340-1a7cfa54a2ac'
  ),
  'ddbd1be1-2a1b-4aea-a085-6a63208a75c0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='ICE CORE RECORDS',
    variable_level_1='NITROUS OXIDE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ddbd1be1-2a1b-4aea-a085-6a63208a75c0'
  ),
  '63c7d604-707e-4c38-8baf-19a620e61917': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='ICE CORE RECORDS',
    variable_level_1='PARTICULATE MATTER',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='63c7d604-707e-4c38-8baf-19a620e61917'
  ),
  'aeaa43ab-5ccf-4df7-b8ad-b9f9f4249551': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='ICE CORE RECORDS',
    variable_level_1='POTASSIUM',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='aeaa43ab-5ccf-4df7-b8ad-b9f9f4249551'
  ),
  'c692e8e4-b920-4eb5-86c3-b5f6121fec4b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='ICE CORE RECORDS',
    variable_level_1='SODIUM',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c692e8e4-b920-4eb5-86c3-b5f6121fec4b'
  ),
  '4487cb97-df49-421f-8c00-5c5f12dd8af1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='ICE CORE RECORDS',
    variable_level_1='VELOCITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4487cb97-df49-421f-8c00-5c5f12dd8af1'
  ),
  '2f4ccb5c-7b99-442c-9054-964070d95f7b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='ICE CORE RECORDS',
    variable_level_1='VOLCANIC DEPOSITS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2f4ccb5c-7b99-442c-9054-964070d95f7b'
  ),
  '486f2c33-2401-4292-9d74-8756ee95211f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='LAND RECORDS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='486f2c33-2401-4292-9d74-8756ee95211f'
  ),
  '5a63fa7f-2971-4874-a920-394df07d218e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='LAND RECORDS',
    variable_level_1='BOREHOLES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5a63fa7f-2971-4874-a920-394df07d218e'
  ),
  '1651d2e2-4483-42fc-aef2-fd49e650eff1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='LAND RECORDS',
    variable_level_1='CAVE DEPOSITS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='1651d2e2-4483-42fc-aef2-fd49e650eff1'
  ),
  '8c615709-df55-4b09-a5a9-1fabb133fe1a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='LAND RECORDS',
    variable_level_1='GLACIATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8c615709-df55-4b09-a5a9-1fabb133fe1a'
  ),
  '61f57065-8f47-45c7-8319-f6115153a6ad': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='LAND RECORDS',
    variable_level_1='ISOTOPES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='61f57065-8f47-45c7-8319-f6115153a6ad'
  ),
  '733234ec-053b-4595-811a-b221e6afb35e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='LAND RECORDS',
    variable_level_1='LOESS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='733234ec-053b-4595-811a-b221e6afb35e'
  ),
  'd412deec-d4ef-4c97-ac1c-f92ddb6964c6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='LAND RECORDS',
    variable_level_1='MACROFOSSILS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d412deec-d4ef-4c97-ac1c-f92ddb6964c6'
  ),
  '98e15316-0055-4392-8825-c38f447d6582': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='LAND RECORDS',
    variable_level_1='MICROFOSSILS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='98e15316-0055-4392-8825-c38f447d6582'
  ),
  'f2ceb98b-4b5d-4ee6-b033-e987d2f820f1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='LAND RECORDS',
    variable_level_1='PALEOMAGNETIC DATA',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f2ceb98b-4b5d-4ee6-b033-e987d2f820f1'
  ),
  'b54e01eb-02d9-413a-baf1-40a6e59d9eae': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='LAND RECORDS',
    variable_level_1='PALEOSOLS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b54e01eb-02d9-413a-baf1-40a6e59d9eae'
  ),
  'e4871f3e-bc88-4380-b7b7-3a18afccc2bd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='LAND RECORDS',
    variable_level_1='PALEOVEGETATION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e4871f3e-bc88-4380-b7b7-3a18afccc2bd'
  ),
  '14c8721e-4d05-4aaa-90be-8607ae2f84b1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='LAND RECORDS',
    variable_level_1='POLLEN',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='14c8721e-4d05-4aaa-90be-8607ae2f84b1'
  ),
  'bf0db125-0182-42e7-81c9-6ed55a05ddd0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='LAND RECORDS',
    variable_level_1='RADIOCARBON',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bf0db125-0182-42e7-81c9-6ed55a05ddd0'
  ),
  '858d3f93-f2eb-4d2a-87c5-68018f206a47': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='LAND RECORDS',
    variable_level_1='SEDIMENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='858d3f93-f2eb-4d2a-87c5-68018f206a47'
  ),
  'ac45c059-9555-45ee-ad20-d58514578f1e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='LAND RECORDS',
    variable_level_1='STRATIGRAPHIC SEQUENCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ac45c059-9555-45ee-ad20-d58514578f1e'
  ),
  '84510f18-a4e6-434c-a54c-44cc995e1af2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='LAND RECORDS',
    variable_level_1='TREE RINGS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='84510f18-a4e6-434c-a54c-44cc995e1af2'
  ),
  '902e9bf8-85d4-431b-aa15-e0d7abe0f17c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='LAND RECORDS',
    variable_level_1='TREE RINGS',
    variable_level_2='SEA SALT',
    variable_level_3='',
    detailed_variable='',
    identifier='902e9bf8-85d4-431b-aa15-e0d7abe0f17c'
  ),
  '9761565c-2126-49cd-b4c4-cd3bcb5dbbde': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='LAND RECORDS',
    variable_level_1='VOLCANIC DEPOSITS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='9761565c-2126-49cd-b4c4-cd3bcb5dbbde'
  ),
  '45325a01-2522-48d3-bffa-0edf1a934d48': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='OCEAN/LAKE RECORDS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='45325a01-2522-48d3-bffa-0edf1a934d48'
  ),
  'e001c431-c204-419e-af64-cc8978132abf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='OCEAN/LAKE RECORDS',
    variable_level_1='BOREHOLES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e001c431-c204-419e-af64-cc8978132abf'
  ),
  '296b7bc4-c031-48ea-bb6d-99f7c971c953': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='OCEAN/LAKE RECORDS',
    variable_level_1='CORAL DEPOSITS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='296b7bc4-c031-48ea-bb6d-99f7c971c953'
  ),
  '56589fec-7573-42df-b853-2754cdc9e1b7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='OCEAN/LAKE RECORDS',
    variable_level_1='ISOTOPES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='56589fec-7573-42df-b853-2754cdc9e1b7'
  ),
  '77cbdebf-eddf-42b5-8603-e939eccd1780': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='OCEAN/LAKE RECORDS',
    variable_level_1='LAKE LEVELS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='77cbdebf-eddf-42b5-8603-e939eccd1780'
  ),
  'f986b716-d26c-4c98-8166-b415229186ff': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='OCEAN/LAKE RECORDS',
    variable_level_1='MACROFOSSILS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f986b716-d26c-4c98-8166-b415229186ff'
  ),
  '6e872413-f416-43dd-a960-942ef892ae59': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='OCEAN/LAKE RECORDS',
    variable_level_1='MICROFOSSILS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6e872413-f416-43dd-a960-942ef892ae59'
  ),
  'a65ec029-86a7-4c3b-b2b3-ee26353aaf36': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='OCEAN/LAKE RECORDS',
    variable_level_1='OXYGEN ISOTOPES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a65ec029-86a7-4c3b-b2b3-ee26353aaf36'
  ),
  'd42bf3e3-3eda-471a-adb5-ddf1240cd474': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='OCEAN/LAKE RECORDS',
    variable_level_1='PALEOMAGNETIC DATA',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d42bf3e3-3eda-471a-adb5-ddf1240cd474'
  ),
  'adcd37fe-9f4a-4d8d-8f79-489775707ea2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='OCEAN/LAKE RECORDS',
    variable_level_1='POLLEN',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='adcd37fe-9f4a-4d8d-8f79-489775707ea2'
  ),
  '23822618-39a2-4b2b-9162-37bb0651c118': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='OCEAN/LAKE RECORDS',
    variable_level_1='RADIOCARBON',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='23822618-39a2-4b2b-9162-37bb0651c118'
  ),
  'a088eccf-1160-4f01-ab71-53e720264ecd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='OCEAN/LAKE RECORDS',
    variable_level_1='SEDIMENT CORE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a088eccf-1160-4f01-ab71-53e720264ecd'
  ),
  'd324729f-cc0d-4943-8d1a-d38335120c00': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='OCEAN/LAKE RECORDS',
    variable_level_1='SEDIMENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d324729f-cc0d-4943-8d1a-d38335120c00'
  ),
  '4722fc1e-f93f-4aa1-854a-2b8a82920008': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='OCEAN/LAKE RECORDS',
    variable_level_1='STRATIGRAPHIC SEQUENCE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4722fc1e-f93f-4aa1-854a-2b8a82920008'
  ),
  '4ad60c3b-f72e-4f54-9d3e-0048373c166d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='OCEAN/LAKE RECORDS',
    variable_level_1='VARVE DEPOSITS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4ad60c3b-f72e-4f54-9d3e-0048373c166d'
  ),
  '350c9923-fa80-4f83-8724-2886ac559ac0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='350c9923-fa80-4f83-8724-2886ac559ac0'
  ),
  'cb49a2e7-bd89-4d3a-974a-8776a763a4ae': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_1='AIR TEMPERATURE RECONSTRUCTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='cb49a2e7-bd89-4d3a-974a-8776a763a4ae'
  ),
  '2bdfbf06-c583-4e51-a595-dbc6143d95e0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_1='ATMOSPHERIC CIRCULATION RECONSTRUCTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2bdfbf06-c583-4e51-a595-dbc6143d95e0'
  ),
  'cc063daf-1db5-4597-9de2-0501a5593947': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_1='DROUGHT/PRECIPITATION RECONSTRUCTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='cc063daf-1db5-4597-9de2-0501a5593947'
  ),
  'cf0e53d3-c8ae-4baa-8a31-672a2252f285': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_1='GROUND WATER RECONSTRUCTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='cf0e53d3-c8ae-4baa-8a31-672a2252f285'
  ),
  'fed291ec-8f7d-4131-a5cd-dc04706f61b0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_1='LAKE LEVEL RECONSTRUCTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fed291ec-8f7d-4131-a5cd-dc04706f61b0'
  ),
  '80a6803a-5bf3-4439-b13f-0909e0ea40f9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_1='OCEAN SALINITY RECONSTRUCTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='80a6803a-5bf3-4439-b13f-0909e0ea40f9'
  ),
  'e240565d-d265-474b-a25b-34059526ae44': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_1='SEA LEVEL RECONSTRUCTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e240565d-d265-474b-a25b-34059526ae44'
  ),
  'e7b30694-5d05-404b-9748-b8f6adc3491d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_1='SEA SURFACE TEMPERATURE RECONSTRUCTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e7b30694-5d05-404b-9748-b8f6adc3491d'
  ),
  '4f07a511-1c78-4b2b-8c6a-f4aeedb0f5b6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_1='SOLAR FORCING/INSOLATION RECONSTRUCTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4f07a511-1c78-4b2b-8c6a-f4aeedb0f5b6'
  ),
  '7859191e-732b-47cf-b1a3-fc7a934509ce': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_1='STREAMFLOW RECONSTRUCTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='7859191e-732b-47cf-b1a3-fc7a934509ce'
  ),
  '91600c9b-397e-4855-b21e-b9e97e6d5261': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='PALEOCLIMATE',
    term='PALEOCLIMATE RECONSTRUCTIONS',
    variable_level_1='VEGETATION RECONSTRUCTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='91600c9b-397e-4855-b21e-b9e97e6d5261'
  ),
  '2b9ad978-d986-4d63-b477-0f5efc8ace72': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='2b9ad978-d986-4d63-b477-0f5efc8ace72'
  ),
  'e3fa1998-b003-4d55-a92e-16b42ac0fc17': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='EARTH GASES/LIQUIDS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e3fa1998-b003-4d55-a92e-16b42ac0fc17'
  ),
  '96bbae63-81c1-43b4-90f0-52731e2b52ca': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='EARTH GASES/LIQUIDS',
    variable_level_1='HYDROGEN GAS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='96bbae63-81c1-43b4-90f0-52731e2b52ca'
  ),
  '833b6958-fc93-473c-aadb-bb65da7578e5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='EARTH GASES/LIQUIDS',
    variable_level_1='HYDROGEN GAS',
    variable_level_2='HYDROGEN GAS VERTICAL/GEOGRAPHIC DISTRIBUTION',
    variable_level_3='',
    detailed_variable='',
    identifier='833b6958-fc93-473c-aadb-bb65da7578e5'
  ),
  '72eb280a-d5d0-4c5e-b789-8f1a8cf8bdac': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='EARTH GASES/LIQUIDS',
    variable_level_1='NATURAL GAS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='72eb280a-d5d0-4c5e-b789-8f1a8cf8bdac'
  ),
  '58769369-608c-4482-924b-207454a5fb1c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='EARTH GASES/LIQUIDS',
    variable_level_1='NATURAL GAS',
    variable_level_2='NATURAL GAS VERTICAL/GEOGRAPHIC DISTRIBUTION',
    variable_level_3='',
    detailed_variable='',
    identifier='58769369-608c-4482-924b-207454a5fb1c'
  ),
  '44d0ad8f-fe22-4d17-bc47-c0b728a82baf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='EARTH GASES/LIQUIDS',
    variable_level_1='PETROLEUM',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='44d0ad8f-fe22-4d17-bc47-c0b728a82baf'
  ),
  '37f3fdb8-a82f-4bff-bda4-cca12a683d6f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='EARTH GASES/LIQUIDS',
    variable_level_1='PETROLEUM',
    variable_level_2='MICROFOSSIL',
    variable_level_3='',
    detailed_variable='',
    identifier='37f3fdb8-a82f-4bff-bda4-cca12a683d6f'
  ),
  'f9e3595d-29b6-462a-8eb6-a06e5a02b081': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='EARTH GASES/LIQUIDS',
    variable_level_1='PETROLEUM',
    variable_level_2='PETROLEUM VERTICAL/GEOGRAPHIC DISTRIBUTION',
    variable_level_3='',
    detailed_variable='',
    identifier='f9e3595d-29b6-462a-8eb6-a06e5a02b081'
  ),
  '906e647b-2683-4ae7-9986-1aea15582b52': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='906e647b-2683-4ae7-9986-1aea15582b52'
  ),
  'b472632f-8e67-4892-9896-1c14c5089682': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='BIOGEOCHEMICAL PROCESSES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b472632f-8e67-4892-9896-1c14c5089682'
  ),
  '14d972b3-a587-4994-a021-f1e620b02341': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='BIOGEOCHEMICAL PROCESSES',
    variable_level_2='CHEMICAL DECOMPOSITION',
    variable_level_3='',
    detailed_variable='',
    identifier='14d972b3-a587-4994-a021-f1e620b02341'
  ),
  '8c6adb44-54c5-42f1-ae19-602e248ff9d9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='BIOGEOCHEMICAL PROCESSES',
    variable_level_2='HYDROLYSIS',
    variable_level_3='',
    detailed_variable='',
    identifier='8c6adb44-54c5-42f1-ae19-602e248ff9d9'
  ),
  'd73ed320-cd5b-4994-a26a-dac5a2fc394f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='BIOGEOCHEMICAL PROCESSES',
    variable_level_2='NITRIFICATION',
    variable_level_3='',
    detailed_variable='',
    identifier='d73ed320-cd5b-4994-a26a-dac5a2fc394f'
  ),
  'e6fb1b81-8ffc-486f-b1a1-2f292af8cee6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='GEOCHEMICAL PROCESSES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='e6fb1b81-8ffc-486f-b1a1-2f292af8cee6'
  ),
  '3e934184-42bd-45ff-b9c1-5c5321fd066f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='GEOCHEMICAL PROCESSES',
    variable_level_2='BIODEGRATION',
    variable_level_3='',
    detailed_variable='',
    identifier='3e934184-42bd-45ff-b9c1-5c5321fd066f'
  ),
  'b2a9741a-f978-46ac-83ad-e92ff07a637c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='GEOCHEMICAL PROCESSES',
    variable_level_2='CARBONATE FORMATION',
    variable_level_3='',
    detailed_variable='',
    identifier='b2a9741a-f978-46ac-83ad-e92ff07a637c'
  ),
  '84b29fbe-8200-4d21-a1a3-fe84fa4cb132': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='GEOCHEMICAL PROCESSES',
    variable_level_2='CHEMICAL FIXATION',
    variable_level_3='',
    detailed_variable='',
    identifier='84b29fbe-8200-4d21-a1a3-fe84fa4cb132'
  ),
  '7e140a1e-385d-4dd3-8b08-6239b082e35e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='GEOCHEMICAL PROCESSES',
    variable_level_2='CHEMICAL WEATHERING',
    variable_level_3='',
    detailed_variable='',
    identifier='7e140a1e-385d-4dd3-8b08-6239b082e35e'
  ),
  '7b60ab41-92e7-4550-821b-0ab7ebd3d7c8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='GEOCHEMICAL PROCESSES',
    variable_level_2='DECOMPOSITION',
    variable_level_3='',
    detailed_variable='',
    identifier='7b60ab41-92e7-4550-821b-0ab7ebd3d7c8'
  ),
  '2b1f870b-c679-4b6d-b02e-3eb005f0648d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='GEOCHEMICAL PROCESSES',
    variable_level_2='HYDRATION',
    variable_level_3='',
    detailed_variable='',
    identifier='2b1f870b-c679-4b6d-b02e-3eb005f0648d'
  ),
  'ccbf4ef8-955b-4337-a45b-95affc360173': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='GEOCHEMICAL PROCESSES',
    variable_level_2='ION EXCHANGE',
    variable_level_3='',
    detailed_variable='',
    identifier='ccbf4ef8-955b-4337-a45b-95affc360173'
  ),
  '524cbe78-9c1f-4ef3-8aa9-0481476c253e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='GEOCHEMICAL PROCESSES',
    variable_level_2='MINERAL DISSOLUTION',
    variable_level_3='',
    detailed_variable='',
    identifier='524cbe78-9c1f-4ef3-8aa9-0481476c253e'
  ),
  '9c2f3bee-4629-4607-9962-12fe919594a0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='GEOCHEMICAL PROCESSES',
    variable_level_2='OXIDATION/REDUCTION',
    variable_level_3='',
    detailed_variable='',
    identifier='9c2f3bee-4629-4607-9962-12fe919594a0'
  ),
  '048df94e-841d-4f4d-a5c5-6683d1d07aa6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='GEOCHEMICAL PROPERTIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='048df94e-841d-4f4d-a5c5-6683d1d07aa6'
  ),
  '12ed4fa0-27cc-4e05-a2b7-bbf2fde871f6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='GEOCHEMICAL PROPERTIES',
    variable_level_2='CHEMICAL CONCENTRATIONS',
    variable_level_3='',
    detailed_variable='',
    identifier='12ed4fa0-27cc-4e05-a2b7-bbf2fde871f6'
  ),
  '849edfe2-9ed7-4211-8f57-9c8ccff0a4ea': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='GEOCHEMICAL PROPERTIES',
    variable_level_2='ISOTOPE MEASUREMENTS',
    variable_level_3='',
    detailed_variable='',
    identifier='849edfe2-9ed7-4211-8f57-9c8ccff0a4ea'
  ),
  'f7998303-d145-452d-bcff-770f62038909': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='GEOCHEMICAL PROPERTIES',
    variable_level_2='ISOTOPE RATIOS',
    variable_level_3='',
    detailed_variable='',
    identifier='f7998303-d145-452d-bcff-770f62038909'
  ),
  '441ce068-91f2-4412-8893-c0096d8f9079': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='GEOCHEMICAL PROPERTIES',
    variable_level_2='ISOTOPES',
    variable_level_3='',
    detailed_variable='',
    identifier='441ce068-91f2-4412-8893-c0096d8f9079'
  ),
  '211d289d-fae7-4815-9f3d-28a5afc7b3a9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='GEOCHEMICAL PROPERTIES',
    variable_level_2='ISOTOPIC AGE',
    variable_level_3='',
    detailed_variable='',
    identifier='211d289d-fae7-4815-9f3d-28a5afc7b3a9'
  ),
  '2dc96cc9-a128-4dc8-b8c8-1d799201b5c6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='GEOCHEMICAL PROPERTIES',
    variable_level_2='ROCK-EVAL PRYOLYSIS',
    variable_level_3='',
    detailed_variable='',
    identifier='2dc96cc9-a128-4dc8-b8c8-1d799201b5c6'
  ),
  '5cef2f41-a17a-4eff-8ce4-328593e1b703': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='MARINE GEOCHEMICAL PROCESSES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5cef2f41-a17a-4eff-8ce4-328593e1b703'
  ),
  '4efb531e-3c6c-4469-9215-d55a8a6ce9da': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='MARINE GEOCHEMICAL PROCESSES',
    variable_level_2='CHEMICAL DECOMPOSITION',
    variable_level_3='',
    detailed_variable='',
    identifier='4efb531e-3c6c-4469-9215-d55a8a6ce9da'
  ),
  '6628bfb9-c0e1-4281-9b1a-d213a9d5b2d8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='MARINE GEOCHEMICAL PROCESSES',
    variable_level_2='DISSOLUTION',
    variable_level_3='',
    detailed_variable='',
    identifier='6628bfb9-c0e1-4281-9b1a-d213a9d5b2d8'
  ),
  'f956cc7c-da39-4eac-98ab-ba6207181b7d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOCHEMISTRY',
    variable_level_1='MARINE GEOCHEMICAL PROCESSES',
    variable_level_2='MINERAL DISSOLUTION',
    variable_level_3='',
    detailed_variable='',
    identifier='f956cc7c-da39-4eac-98ab-ba6207181b7d'
  ),
  '5498572c-aaed-4c08-8aad-8b297057e9c9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEODETICS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='5498572c-aaed-4c08-8aad-8b297057e9c9'
  ),
  '14b19e68-0fb3-43b1-a102-537c4e33c338': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEODETICS',
    variable_level_1='COORDINATE REFERENCE SYSTEM',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='14b19e68-0fb3-43b1-a102-537c4e33c338'
  ),
  'bb5ca226-fdb1-4fab-9988-7486c643635b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEODETICS',
    variable_level_1='COORDINATE REFERENCE SYSTEM',
    variable_level_2='COUNTRY/REGIONAL COORDINATE REFERENCE SYSTEM',
    variable_level_3='',
    detailed_variable='',
    identifier='bb5ca226-fdb1-4fab-9988-7486c643635b'
  ),
  'e0a2edbb-8a94-4f47-918a-fe9f93aba5f4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEODETICS',
    variable_level_1='COORDINATE REFERENCE SYSTEM',
    variable_level_2='GLOBAL COORDINATE REFERENCE SYSTEM',
    variable_level_3='',
    detailed_variable='',
    identifier='e0a2edbb-8a94-4f47-918a-fe9f93aba5f4'
  ),
  'bc640e63-70c1-4228-b2dc-6aa1ac6edfa6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEODETICS',
    variable_level_1='ELLIPSOID CHARACTERISTICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='bc640e63-70c1-4228-b2dc-6aa1ac6edfa6'
  ),
  '6bbbf7b0-434b-4dbc-9fe8-e5e31fe99614': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEODETICS',
    variable_level_1='GEOID CHARACTERISTICS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6bbbf7b0-434b-4dbc-9fe8-e5e31fe99614'
  ),
  'abaf0bfd-4acf-4ac0-a570-85cccb78032e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEODETICS',
    variable_level_1='TECHNIQUES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='abaf0bfd-4acf-4ac0-a570-85cccb78032e'
  ),
  '846a5d37-085b-4ae0-9d4d-23b88b6fc304': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEODETICS',
    variable_level_1='TECHNIQUES',
    variable_level_2='DORIS (DOPPLER ORBITOGRAPHY AND RADIOPOSITIONING BY INTEGRATED SATELLITE)',
    variable_level_3='',
    detailed_variable='',
    identifier='846a5d37-085b-4ae0-9d4d-23b88b6fc304'
  ),
  '93b2fed5-883f-4d82-9d62-19654d0765ea': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEODETICS',
    variable_level_1='TECHNIQUES',
    variable_level_2='GNSS (GLOBAL NAVIGATION SATELLITE SYSTEMS)',
    variable_level_3='',
    detailed_variable='',
    identifier='93b2fed5-883f-4d82-9d62-19654d0765ea'
  ),
  'de965e45-2275-44d6-9193-9387881b538c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEODETICS',
    variable_level_1='TECHNIQUES',
    variable_level_2='GNSS-R (RONGOWAI GLOBAL NAVIGATION SATELLITE SYSTEM REFLECTOMETRY)',
    variable_level_3='',
    detailed_variable='',
    identifier='de965e45-2275-44d6-9193-9387881b538c'
  ),
  '22f57987-7a8d-46d8-9b3d-37761755f8b2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEODETICS',
    variable_level_1='TECHNIQUES',
    variable_level_2='SLR (SATELLITE LASER RANGING)',
    variable_level_3='',
    detailed_variable='',
    identifier='22f57987-7a8d-46d8-9b3d-37761755f8b2'
  ),
  '06382d35-5885-4d34-a586-8b44868250a8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEODETICS',
    variable_level_1='TECHNIQUES',
    variable_level_2='VLBI (VERY LONG BASELINE INTERFEROMETRY)',
    variable_level_3='',
    detailed_variable='',
    identifier='06382d35-5885-4d34-a586-8b44868250a8'
  ),
  '910013d7-1e6a-4d1a-9921-be32d792a290': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMAGNETISM',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='910013d7-1e6a-4d1a-9921-be32d792a290'
  ),
  '3202dab6-144a-4bfb-9bda-9d07e5ee7ec2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMAGNETISM',
    variable_level_1='ELECTRICAL FIELD',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3202dab6-144a-4bfb-9bda-9d07e5ee7ec2'
  ),
  'd55d29e8-9015-4c23-b137-528eb298aa49': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMAGNETISM',
    variable_level_1='ELECTRICAL FIELD',
    variable_level_2='ELECTRICAL ANOMALIES',
    variable_level_3='',
    detailed_variable='',
    identifier='d55d29e8-9015-4c23-b137-528eb298aa49'
  ),
  '84d77f98-d5a2-4da8-9ba6-0b15e082d050': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMAGNETISM',
    variable_level_1='ELECTRICAL FIELD',
    variable_level_2='ELECTRICAL INTENSITY',
    variable_level_3='',
    detailed_variable='',
    identifier='84d77f98-d5a2-4da8-9ba6-0b15e082d050'
  ),
  '02290e22-24ae-40f6-96f1-0c6c76a145af': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMAGNETISM',
    variable_level_1='GEOMAGNETIC FORECASTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='02290e22-24ae-40f6-96f1-0c6c76a145af'
  ),
  '2d3d9a57-44e8-43c0-98b4-b4891c994862': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMAGNETISM',
    variable_level_1='GEOMAGNETIC FORECASTS',
    variable_level_2='GEOMAGNETIC ACTIVITY',
    variable_level_3='',
    detailed_variable='',
    identifier='2d3d9a57-44e8-43c0-98b4-b4891c994862'
  ),
  '4b7decec-e378-4824-aecf-9fe509392efd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMAGNETISM',
    variable_level_1='GEOMAGNETIC FORECASTS',
    variable_level_2='GEOMAGNETIC STORM CATEGORY',
    variable_level_3='',
    detailed_variable='',
    identifier='4b7decec-e378-4824-aecf-9fe509392efd'
  ),
  '9a46a62c-952d-4253-8249-7375c14068a2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMAGNETISM',
    variable_level_1='GEOMAGNETIC FORECASTS',
    variable_level_2='TOTAL INTENSITY',
    variable_level_3='',
    detailed_variable='',
    identifier='9a46a62c-952d-4253-8249-7375c14068a2'
  ),
  'ae35f430-6534-49de-8b4c-edfc1e98870a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMAGNETISM',
    variable_level_1='GEOMAGNETIC INDICES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ae35f430-6534-49de-8b4c-edfc1e98870a'
  ),
  '5fd5ccc2-5edb-4823-940d-03a290a5c5fc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMAGNETISM',
    variable_level_1='GEOMAGNETIC INDICES',
    variable_level_2='AA INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='5fd5ccc2-5edb-4823-940d-03a290a5c5fc'
  ),
  '31f77d6b-72f7-45e6-93be-8ac5fd5dc373': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMAGNETISM',
    variable_level_1='GEOMAGNETIC INDICES',
    variable_level_2='AE INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='31f77d6b-72f7-45e6-93be-8ac5fd5dc373'
  ),
  'b3283844-d867-4c2f-9917-a72bc06fd9ef': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMAGNETISM',
    variable_level_1='GEOMAGNETIC INDICES',
    variable_level_2='AM INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='b3283844-d867-4c2f-9917-a72bc06fd9ef'
  ),
  'cdb4b514-75c4-4a1f-a4ad-1855fbd396ab': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMAGNETISM',
    variable_level_1='GEOMAGNETIC INDICES',
    variable_level_2='DST INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='cdb4b514-75c4-4a1f-a4ad-1855fbd396ab'
  ),
  '40386eea-beb0-4b83-906b-75c6bfa24b73': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMAGNETISM',
    variable_level_1='GEOMAGNETIC INDICES',
    variable_level_2='KP INDEX',
    variable_level_3='',
    detailed_variable='',
    identifier='40386eea-beb0-4b83-906b-75c6bfa24b73'
  ),
  '204b482b-449b-42c9-a5bb-f6da42bee3a4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMAGNETISM',
    variable_level_1='MAGNETIC FIELD',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='204b482b-449b-42c9-a5bb-f6da42bee3a4'
  ),
  'ee421700-0fe2-420c-9a07-91e8ae9fb524': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMAGNETISM',
    variable_level_1='MAGNETIC FIELD',
    variable_level_2='GEOMAGNETIC INDUCTION',
    variable_level_3='',
    detailed_variable='',
    identifier='ee421700-0fe2-420c-9a07-91e8ae9fb524'
  ),
  '65ae8ab2-489b-44bf-bf5b-43cf957b70c0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMAGNETISM',
    variable_level_1='MAGNETIC FIELD',
    variable_level_2='MAGNETIC ANOMALIES',
    variable_level_3='',
    detailed_variable='',
    identifier='65ae8ab2-489b-44bf-bf5b-43cf957b70c0'
  ),
  'f311eac7-5c85-4a8f-90c2-abcff3eec92d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMAGNETISM',
    variable_level_1='MAGNETIC FIELD',
    variable_level_2='MAGNETIC DECLINATION',
    variable_level_3='',
    detailed_variable='',
    identifier='f311eac7-5c85-4a8f-90c2-abcff3eec92d'
  ),
  'f0b7311e-df08-45fa-8dd5-33b6f74a66d9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMAGNETISM',
    variable_level_1='MAGNETIC FIELD',
    variable_level_2='MAGNETIC INCLINATION',
    variable_level_3='',
    detailed_variable='',
    identifier='f0b7311e-df08-45fa-8dd5-33b6f74a66d9'
  ),
  'd817911a-685b-4c9f-bdc7-2411b8c0a7af': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMAGNETISM',
    variable_level_1='MAGNETIC FIELD',
    variable_level_2='MAGNETIC INTENSITY',
    variable_level_3='',
    detailed_variable='',
    identifier='d817911a-685b-4c9f-bdc7-2411b8c0a7af'
  ),
  '720969dd-e966-41aa-af94-ee41cdf60390': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMAGNETISM',
    variable_level_1='PALEOMAGNETISM',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='720969dd-e966-41aa-af94-ee41cdf60390'
  ),
  'fd631e31-fe6f-462e-a3f6-c07b4b736ac7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMAGNETISM',
    variable_level_1='REFERENCE FIELDS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fd631e31-fe6f-462e-a3f6-c07b4b736ac7'
  ),
  'b5cb1fab-7281-478f-bb3b-ff04f900b3fc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='b5cb1fab-7281-478f-bb3b-ff04f900b3fc'
  ),
  '26637389-f4f6-47a0-9c3d-17e93ab99dea': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN LANDFORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='26637389-f4f6-47a0-9c3d-17e93ab99dea'
  ),
  'e43473a1-4392-48e3-9e56-8a4dcad8d7a2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='',
    detailed_variable='',
    identifier='e43473a1-4392-48e3-9e56-8a4dcad8d7a2'
  ),
  '5c80c047-c02e-4c15-83ac-26b8b1a8f114': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='CRESCENTIC (BARCHAN/TRANSVERSE) DUNE',
    detailed_variable='',
    identifier='5c80c047-c02e-4c15-83ac-26b8b1a8f114'
  ),
  'd5ff7545-0eec-4cad-90a5-019e03cdac47': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='DOME DUNE',
    detailed_variable='',
    identifier='d5ff7545-0eec-4cad-90a5-019e03cdac47'
  ),
  'db2d6cfb-70c3-4568-99a0-a25b3c3879dd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='LONGITUDINAL/LINEAR DUNE',
    detailed_variable='',
    identifier='db2d6cfb-70c3-4568-99a0-a25b3c3879dd'
  ),
  '4cce9a44-da57-4169-b89f-6b1460fcedb9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='PARABOLIC DUNE',
    detailed_variable='',
    identifier='4cce9a44-da57-4169-b89f-6b1460fcedb9'
  ),
  'ce087840-ec71-4575-bca9-e807151cc376': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='STAR DUNE',
    detailed_variable='',
    identifier='ce087840-ec71-4575-bca9-e807151cc376'
  ),
  'cae41424-161f-4378-a1a4-62cd76c61143': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN LANDFORMS',
    variable_level_2='RIPPLES',
    variable_level_3='',
    detailed_variable='',
    identifier='cae41424-161f-4378-a1a4-62cd76c61143'
  ),
  'f15b2ad3-f658-420b-99b4-41588646d9b7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f15b2ad3-f658-420b-99b4-41588646d9b7'
  ),
  'f6e19e2e-555a-4d40-9833-c7513d92c813': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='ABRASION',
    variable_level_3='',
    detailed_variable='',
    identifier='f6e19e2e-555a-4d40-9833-c7513d92c813'
  ),
  '2c15738b-839f-4b68-85bc-ece41e4ac6c9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='ABRASION',
    variable_level_3='VENTIFACTS',
    detailed_variable='',
    identifier='2c15738b-839f-4b68-85bc-ece41e4ac6c9'
  ),
  '59d1b0f7-ef02-4fa4-8d47-7eda39794713': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='ABRASION',
    variable_level_3='YARDANGS',
    detailed_variable='',
    identifier='59d1b0f7-ef02-4fa4-8d47-7eda39794713'
  ),
  'd415cb15-7586-464c-8707-9a5623a61cee': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='DEFLATION',
    variable_level_3='',
    detailed_variable='',
    identifier='d415cb15-7586-464c-8707-9a5623a61cee'
  ),
  'baf70c0f-fd59-4d4b-ae03-b664e0352ff7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='DEGRADATION',
    variable_level_3='',
    detailed_variable='',
    identifier='baf70c0f-fd59-4d4b-ae03-b664e0352ff7'
  ),
  '78778362-5d08-4cd7-9131-159cad561e54': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='SALTATION',
    variable_level_3='',
    detailed_variable='',
    identifier='78778362-5d08-4cd7-9131-159cad561e54'
  ),
  'fe2d9f93-ee9c-4d1e-af28-0c15ee762019': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='SEDIMENT TRANSPORT',
    variable_level_3='',
    detailed_variable='',
    identifier='fe2d9f93-ee9c-4d1e-af28-0c15ee762019'
  ),
  'a83052ef-9b98-4cb3-9bed-b0c9059812e5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='SEDIMENT TRANSPORT',
    variable_level_3='LOESS',
    detailed_variable='',
    identifier='a83052ef-9b98-4cb3-9bed-b0c9059812e5'
  ),
  '8167592d-13bf-4225-9822-29e68bcd1b37': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='SEDIMENT TRANSPORT',
    variable_level_3='MONADNOCK',
    detailed_variable='',
    identifier='8167592d-13bf-4225-9822-29e68bcd1b37'
  ),
  '22ba30ec-a4e2-4547-bad7-4d5f9917625d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='',
    detailed_variable='',
    identifier='22ba30ec-a4e2-4547-bad7-4d5f9917625d'
  ),
  '34ea8c99-ff34-495b-b986-92a78b74a8e9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='SEDIMENT CHEMISTRY',
    detailed_variable='',
    identifier='34ea8c99-ff34-495b-b986-92a78b74a8e9'
  ),
  '6e5a6d68-5f99-4f0d-bde3-9f24268af426': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='SEDIMENT COMPOSITION',
    detailed_variable='',
    identifier='6e5a6d68-5f99-4f0d-bde3-9f24268af426'
  ),
  'a08cce11-9407-4b1f-b13e-0df87da03612': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='STRATIGRAPHIC SEQUENCE',
    detailed_variable='',
    identifier='a08cce11-9407-4b1f-b13e-0df87da03612'
  ),
  '7a67a5af-42be-4aa7-8cb1-e1fc0de074cc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='AEOLIAN PROCESSES',
    variable_level_2='WEATHERING',
    variable_level_3='',
    detailed_variable='',
    identifier='7a67a5af-42be-4aa7-8cb1-e1fc0de074cc'
  ),
  'c58320e6-3f1d-4c36-9bee-6bad73404c21': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c58320e6-3f1d-4c36-9bee-6bad73404c21'
  ),
  '6e3135e9-6be6-4995-a5df-022f6a0cf45b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='BARRIER ISLANDS',
    variable_level_3='',
    detailed_variable='',
    identifier='6e3135e9-6be6-4995-a5df-022f6a0cf45b'
  ),
  '6a5d3e4d-86d1-4863-bfe6-f8e2899fab0e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='BEACHES',
    variable_level_3='',
    detailed_variable='',
    identifier='6a5d3e4d-86d1-4863-bfe6-f8e2899fab0e'
  ),
  'dff4d4af-e1e0-4991-884b-a1c088a802b2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='CORAL REEFS',
    variable_level_3='',
    detailed_variable='',
    identifier='dff4d4af-e1e0-4991-884b-a1c088a802b2'
  ),
  '0e566bce-90bf-4a0a-a000-5bb5fb430788': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='CORAL REEFS',
    variable_level_3='APRON REEF',
    detailed_variable='',
    identifier='0e566bce-90bf-4a0a-a000-5bb5fb430788'
  ),
  '8c89ede4-94d8-4fd4-a3df-f9d42e9835eb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='CORAL REEFS',
    variable_level_3='ATOLL REEF',
    detailed_variable='',
    identifier='8c89ede4-94d8-4fd4-a3df-f9d42e9835eb'
  ),
  '57417a5e-4d86-4fb6-81d6-68bf9a3d1148': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='CORAL REEFS',
    variable_level_3='BANK REEF',
    detailed_variable='',
    identifier='57417a5e-4d86-4fb6-81d6-68bf9a3d1148'
  ),
  '5fde7781-d4f6-41a8-8428-f428b70c02dc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='CORAL REEFS',
    variable_level_3='BARRIER REEF',
    detailed_variable='',
    identifier='5fde7781-d4f6-41a8-8428-f428b70c02dc'
  ),
  '0f5c48d1-5189-495d-b5a7-7ad596f0a5c4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='CORAL REEFS',
    variable_level_3='FRINGING REEF',
    detailed_variable='',
    identifier='0f5c48d1-5189-495d-b5a7-7ad596f0a5c4'
  ),
  'a1451fce-9e69-4f2d-b2cf-27238a7577ce': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='CORAL REEFS',
    variable_level_3='PATCH REEF',
    detailed_variable='',
    identifier='a1451fce-9e69-4f2d-b2cf-27238a7577ce'
  ),
  'b428ba89-4638-4989-90c1-f5e4f0d0a6f6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='CORAL REEFS',
    variable_level_3='RIBBON REEF',
    detailed_variable='',
    identifier='b428ba89-4638-4989-90c1-f5e4f0d0a6f6'
  ),
  '5f4dc81d-0893-4eb9-b82a-6a070836aa16': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='CORAL REEFS',
    variable_level_3='TABLE REEF',
    detailed_variable='',
    identifier='5f4dc81d-0893-4eb9-b82a-6a070836aa16'
  ),
  '0c51bdb0-54b0-4d0d-afd0-35ef7458ccb7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='CUSPATE FORELANDS',
    variable_level_3='',
    detailed_variable='',
    identifier='0c51bdb0-54b0-4d0d-afd0-35ef7458ccb7'
  ),
  'b37b1bdf-6392-4a80-891a-14f177ba2ca2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='DELTAS',
    variable_level_3='',
    detailed_variable='',
    identifier='b37b1bdf-6392-4a80-891a-14f177ba2ca2'
  ),
  '0c279e58-9ad3-4748-816a-de8cabeaf0c4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='',
    detailed_variable='',
    identifier='0c279e58-9ad3-4748-816a-de8cabeaf0c4'
  ),
  'fa47cab9-0aa4-4e16-8115-972f7f543920': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='CRESCENTIC (BARCHAN/TRANSVERSE) DUNE',
    detailed_variable='',
    identifier='fa47cab9-0aa4-4e16-8115-972f7f543920'
  ),
  '16a1f6b8-ea67-43fe-a47c-aad5250b4f59': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='DOME DUNE',
    detailed_variable='',
    identifier='16a1f6b8-ea67-43fe-a47c-aad5250b4f59'
  ),
  '1810b08a-9377-4f01-a3cf-fdd549ad8ebf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='LONGITUDINAL/LINEAR DUNE',
    detailed_variable='',
    identifier='1810b08a-9377-4f01-a3cf-fdd549ad8ebf'
  ),
  '174fa36e-3a06-40a1-bc95-87e6799bdead': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='PARABOLIC DUNE',
    detailed_variable='',
    identifier='174fa36e-3a06-40a1-bc95-87e6799bdead'
  ),
  '94ab11dc-b70b-4705-bb25-c4d430722d28': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='DUNES',
    variable_level_3='STAR DUNE',
    detailed_variable='',
    identifier='94ab11dc-b70b-4705-bb25-c4d430722d28'
  ),
  '127fdf1d-9985-4a27-9b6c-ad54380fd299': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='ESTUARIES',
    variable_level_3='',
    detailed_variable='',
    identifier='127fdf1d-9985-4a27-9b6c-ad54380fd299'
  ),
  'c9291bc7-784d-486a-95fa-f08fa1edcad9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='FJORDS',
    variable_level_3='',
    detailed_variable='',
    identifier='c9291bc7-784d-486a-95fa-f08fa1edcad9'
  ),
  '860e25fa-e63a-4fd0-bde9-4f596b4a5929': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='HEADLANDS/BAYS/CAPE',
    variable_level_3='',
    detailed_variable='',
    identifier='860e25fa-e63a-4fd0-bde9-4f596b4a5929'
  ),
  '356a245d-418a-4560-9eb1-d12f8f155f66': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='INLETS',
    variable_level_3='',
    detailed_variable='',
    identifier='356a245d-418a-4560-9eb1-d12f8f155f66'
  ),
  'ca9d9064-91c8-4c49-b388-e5f7290a3234': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='ISTHMUS',
    variable_level_3='',
    detailed_variable='',
    identifier='ca9d9064-91c8-4c49-b388-e5f7290a3234'
  ),
  '081d131a-6bef-47dc-adb3-f96da9123f93': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='LAGOONS',
    variable_level_3='',
    detailed_variable='',
    identifier='081d131a-6bef-47dc-adb3-f96da9123f93'
  ),
  '8d4c5e9c-bdab-48c9-89da-1eb4b9a528ab': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='RIA',
    variable_level_3='',
    detailed_variable='',
    identifier='8d4c5e9c-bdab-48c9-89da-1eb4b9a528ab'
  ),
  '85f409fd-9d81-4cac-84ed-fb0bb4599924': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='SALT MARSH',
    variable_level_3='',
    detailed_variable='',
    identifier='85f409fd-9d81-4cac-84ed-fb0bb4599924'
  ),
  '4321cb64-0997-438f-92fb-45169503c01f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='SEA ARCHES',
    variable_level_3='',
    detailed_variable='',
    identifier='4321cb64-0997-438f-92fb-45169503c01f'
  ),
  '521f883e-18be-4f28-b5fe-c1f887b4233a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='SEA CAVES',
    variable_level_3='',
    detailed_variable='',
    identifier='521f883e-18be-4f28-b5fe-c1f887b4233a'
  ),
  '01400b09-68a3-4e3e-b076-1687e30bed56': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='SEA CLIFFS',
    variable_level_3='',
    detailed_variable='',
    identifier='01400b09-68a3-4e3e-b076-1687e30bed56'
  ),
  '94b575b8-eac4-433d-aa74-d781b650f452': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='SHOALS',
    variable_level_3='',
    detailed_variable='',
    identifier='94b575b8-eac4-433d-aa74-d781b650f452'
  ),
  '57e6b119-567b-44d0-9d93-278ed5c21c47': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='SHORELINES',
    variable_level_3='',
    detailed_variable='',
    identifier='57e6b119-567b-44d0-9d93-278ed5c21c47'
  ),
  'c5b85924-9e3f-4106-b389-1ab4486bd233': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='SOUNDS',
    variable_level_3='',
    detailed_variable='',
    identifier='c5b85924-9e3f-4106-b389-1ab4486bd233'
  ),
  '62ef0883-8311-4485-947a-2691b456b667': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='SPITS AND BARS',
    variable_level_3='',
    detailed_variable='',
    identifier='62ef0883-8311-4485-947a-2691b456b667'
  ),
  '30f556c4-7531-4758-9e51-8adc6b2e0e8a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='TOMBOLOS',
    variable_level_3='',
    detailed_variable='',
    identifier='30f556c4-7531-4758-9e51-8adc6b2e0e8a'
  ),
  'ee1d9786-33e9-46dc-b859-25d18e9c8a88': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL LANDFORMS',
    variable_level_2='WAVE-CUT NOTCH/PLATFORMS',
    variable_level_3='',
    detailed_variable='',
    identifier='ee1d9786-33e9-46dc-b859-25d18e9c8a88'
  ),
  '672d6958-4bbc-4b33-adc8-927e4348908b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='672d6958-4bbc-4b33-adc8-927e4348908b'
  ),
  'fd29bf77-df38-4b80-8148-8184fa41d843': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='ABRASION',
    variable_level_3='',
    detailed_variable='',
    identifier='fd29bf77-df38-4b80-8148-8184fa41d843'
  ),
  '8b232049-ce98-4a34-8f00-2366335508e4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='ACCRETION',
    variable_level_3='',
    detailed_variable='',
    identifier='8b232049-ce98-4a34-8f00-2366335508e4'
  ),
  '36b178ad-4f20-41ce-89d1-4ee8567a3cf2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='ATTRITION/WEATHERING',
    variable_level_3='',
    detailed_variable='',
    identifier='36b178ad-4f20-41ce-89d1-4ee8567a3cf2'
  ),
  'bb891ee1-6c7b-4ec0-b2fa-6fb67a2df2a3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='CHEMICAL SOLUTION',
    variable_level_3='',
    detailed_variable='',
    identifier='bb891ee1-6c7b-4ec0-b2fa-6fb67a2df2a3'
  ),
  '6a11e5e5-e6a3-42dd-b793-141ce99932e1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='DEPOSITION',
    variable_level_3='',
    detailed_variable='',
    identifier='6a11e5e5-e6a3-42dd-b793-141ce99932e1'
  ),
  'fb5c09ec-c924-4deb-8294-8a27697a4550': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='FLOODING',
    variable_level_3='',
    detailed_variable='',
    identifier='fb5c09ec-c924-4deb-8294-8a27697a4550'
  ),
  '8fde8c6c-97d4-41a6-9e20-f862faafcd88': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='HYDRAULIC ACTION',
    variable_level_3='',
    detailed_variable='',
    identifier='8fde8c6c-97d4-41a6-9e20-f862faafcd88'
  ),
  '872459ca-da1e-448f-9bf4-383b628f4609': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='SALTATION',
    variable_level_3='',
    detailed_variable='',
    identifier='872459ca-da1e-448f-9bf4-383b628f4609'
  ),
  '6c958ab4-ab98-438e-86d4-1e6a6d0580da': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='SEA LEVEL CHANGE',
    variable_level_3='',
    detailed_variable='',
    identifier='6c958ab4-ab98-438e-86d4-1e6a6d0580da'
  ),
  '1088e9e2-dadd-4d20-a2db-ef7df32c6d42': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='SEDIMENT TRANSPORT',
    variable_level_3='',
    detailed_variable='',
    identifier='1088e9e2-dadd-4d20-a2db-ef7df32c6d42'
  ),
  '2cca0a13-3c6f-4617-aca9-bff7f8142c52': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='',
    detailed_variable='',
    identifier='2cca0a13-3c6f-4617-aca9-bff7f8142c52'
  ),
  '9f4548ad-ec40-4d79-a973-552b2541a62d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='SEDIMENT CHEMISTRY',
    detailed_variable='',
    identifier='9f4548ad-ec40-4d79-a973-552b2541a62d'
  ),
  '17d6838d-e05e-4f0f-a751-7dbd00d2a80a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='SEDIMENT COMPOSITION',
    detailed_variable='',
    identifier='17d6838d-e05e-4f0f-a751-7dbd00d2a80a'
  ),
  '1203f04d-cd90-4f46-b2c1-998a3c182250': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='STRATIGRAPHIC SEQUENCE',
    detailed_variable='',
    identifier='1203f04d-cd90-4f46-b2c1-998a3c182250'
  ),
  '87186c13-548e-4ea8-ba79-38cff394eb59': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='SUBMERGENCE',
    variable_level_3='',
    detailed_variable='',
    identifier='87186c13-548e-4ea8-ba79-38cff394eb59'
  ),
  'b3657e71-acd1-4be4-9c70-a54e074a40a4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='SUBSIDENCE',
    variable_level_3='',
    detailed_variable='',
    identifier='b3657e71-acd1-4be4-9c70-a54e074a40a4'
  ),
  '15c6332d-f6f2-45a4-9485-bb55471c0090': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='SUSPENSION',
    variable_level_3='',
    detailed_variable='',
    identifier='15c6332d-f6f2-45a4-9485-bb55471c0090'
  ),
  'a8c37cb5-9426-41fd-b192-53b4c3ae1ba3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='WAVE BREAKING',
    variable_level_3='',
    detailed_variable='',
    identifier='a8c37cb5-9426-41fd-b192-53b4c3ae1ba3'
  ),
  '5cbfc557-f3a6-4558-9954-ce37f0510952': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='WAVE DIFFRACTION',
    variable_level_3='',
    detailed_variable='',
    identifier='5cbfc557-f3a6-4558-9954-ce37f0510952'
  ),
  '86405d6d-eb37-4aa5-a525-bf6a23fd131d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='WAVE EROSION',
    variable_level_3='',
    detailed_variable='',
    identifier='86405d6d-eb37-4aa5-a525-bf6a23fd131d'
  ),
  '2f5ceedb-afb6-47e6-8eac-8f220ef0b564': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='WAVE EROSION',
    variable_level_3='DEGRADATION',
    detailed_variable='',
    identifier='2f5ceedb-afb6-47e6-8eac-8f220ef0b564'
  ),
  'b43d2d47-c86e-41b6-81bd-be803db536da': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='WAVE REFRACTION',
    variable_level_3='',
    detailed_variable='',
    identifier='b43d2d47-c86e-41b6-81bd-be803db536da'
  ),
  'f8accc20-818e-47a1-962d-80b7ec7f6d92': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='COASTAL PROCESSES',
    variable_level_2='WAVE SHOALING',
    variable_level_3='',
    detailed_variable='',
    identifier='f8accc20-818e-47a1-962d-80b7ec7f6d92'
  ),
  'cb5193ab-2d7a-4b35-b7ec-f16ce78ae270': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='cb5193ab-2d7a-4b35-b7ec-f16ce78ae270'
  ),
  'd8b04023-b9c4-42bc-a986-ab6c4f32ba28': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='AIT',
    variable_level_3='',
    detailed_variable='',
    identifier='d8b04023-b9c4-42bc-a986-ab6c4f32ba28'
  ),
  '6c061296-2c92-4aa4-b9d1-6ecf0efde876': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='BAR',
    variable_level_3='',
    detailed_variable='',
    identifier='6c061296-2c92-4aa4-b9d1-6ecf0efde876'
  ),
  '244bd4be-a3d2-4c02-b576-ae9f2f9e544f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='BAYOU',
    variable_level_3='',
    detailed_variable='',
    identifier='244bd4be-a3d2-4c02-b576-ae9f2f9e544f'
  ),
  'e25ce36c-eacd-447a-9d73-ccc8a7e3a328': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='CANYON',
    variable_level_3='',
    detailed_variable='',
    identifier='e25ce36c-eacd-447a-9d73-ccc8a7e3a328'
  ),
  'cbd9ee43-24f8-45ab-a39b-2ff34be81c51': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='CONFLUENCE',
    variable_level_3='',
    detailed_variable='',
    identifier='cbd9ee43-24f8-45ab-a39b-2ff34be81c51'
  ),
  '8dcff6c3-a6b3-479e-96e2-63191d10ac2d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='CUTBANK',
    variable_level_3='',
    detailed_variable='',
    identifier='8dcff6c3-a6b3-479e-96e2-63191d10ac2d'
  ),
  'ad535c83-3b93-4632-8aaa-7dfba8bb125a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='DELTAS',
    variable_level_3='',
    detailed_variable='',
    identifier='ad535c83-3b93-4632-8aaa-7dfba8bb125a'
  ),
  '4a2a2f6d-9735-4bee-9d1a-21dcd0352c6b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='ENDORHERIC BASIN',
    variable_level_3='',
    detailed_variable='',
    identifier='4a2a2f6d-9735-4bee-9d1a-21dcd0352c6b'
  ),
  'd71f94cb-e773-487a-a8ff-9c5f11c1dbc4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='FLOOD PLAIN',
    variable_level_3='',
    detailed_variable='',
    identifier='d71f94cb-e773-487a-a8ff-9c5f11c1dbc4'
  ),
  '2f8ad9b0-adb8-4022-8c95-bca68e7a87a5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='GULLY',
    variable_level_3='',
    detailed_variable='',
    identifier='2f8ad9b0-adb8-4022-8c95-bca68e7a87a5'
  ),
  '74caea9b-6023-438b-af3d-bb9d948036f1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='ISLAND',
    variable_level_3='',
    detailed_variable='',
    identifier='74caea9b-6023-438b-af3d-bb9d948036f1'
  ),
  '588d868d-05a4-4dac-9fb3-770b54ce39e5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='LACUSTRINE PLAIN',
    variable_level_3='',
    detailed_variable='',
    identifier='588d868d-05a4-4dac-9fb3-770b54ce39e5'
  ),
  '88adcca6-2bc8-443a-9f25-c9aded577615': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='MARSH',
    variable_level_3='',
    detailed_variable='',
    identifier='88adcca6-2bc8-443a-9f25-c9aded577615'
  ),
  'c6f77e54-069e-454f-8260-e150bc29547a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='MEANDER',
    variable_level_3='',
    detailed_variable='',
    identifier='c6f77e54-069e-454f-8260-e150bc29547a'
  ),
  '12233807-f6cd-410d-b607-ecbfbd545464': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='OX-BOW LAKE',
    variable_level_3='',
    detailed_variable='',
    identifier='12233807-f6cd-410d-b607-ecbfbd545464'
  ),
  '4a0c46ff-2d07-442d-b141-6156d9ea4a2e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='PINGO',
    variable_level_3='',
    detailed_variable='',
    identifier='4a0c46ff-2d07-442d-b141-6156d9ea4a2e'
  ),
  'dd0de414-6663-4280-94cf-bda7fea736cc': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='POINT BAR',
    variable_level_3='',
    detailed_variable='',
    identifier='dd0de414-6663-4280-94cf-bda7fea736cc'
  ),
  '5f292d99-b14a-4f18-bbe0-8025d04cae50': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='POND',
    variable_level_3='',
    detailed_variable='',
    identifier='5f292d99-b14a-4f18-bbe0-8025d04cae50'
  ),
  '5df6e78f-6dd4-4fc8-a88e-9e575dbca2eb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='RIFFLE',
    variable_level_3='',
    detailed_variable='',
    identifier='5df6e78f-6dd4-4fc8-a88e-9e575dbca2eb'
  ),
  'bb6b3b76-c496-464b-bd20-1b22296aae15': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='RIVER',
    variable_level_3='',
    detailed_variable='',
    identifier='bb6b3b76-c496-464b-bd20-1b22296aae15'
  ),
  'b498a5cb-f77d-4485-8174-81dec28cee0e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='SPRING',
    variable_level_3='',
    detailed_variable='',
    identifier='b498a5cb-f77d-4485-8174-81dec28cee0e'
  ),
  '1d2d0777-b47e-45ee-ac85-2d7b9f6e4ffd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='STREAM',
    variable_level_3='',
    detailed_variable='',
    identifier='1d2d0777-b47e-45ee-ac85-2d7b9f6e4ffd'
  ),
  '74ce5e8a-038a-471e-a27a-be5b1f17b72f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='STREAM TERRACE',
    variable_level_3='',
    detailed_variable='',
    identifier='74ce5e8a-038a-471e-a27a-be5b1f17b72f'
  ),
  '4811065d-7aed-45e0-ac31-6417123be10e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='SWAMP',
    variable_level_3='',
    detailed_variable='',
    identifier='4811065d-7aed-45e0-ac31-6417123be10e'
  ),
  '87b01c3a-f64f-4764-8cb8-c40ebcd5a989': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='VALLEY',
    variable_level_3='',
    detailed_variable='',
    identifier='87b01c3a-f64f-4764-8cb8-c40ebcd5a989'
  ),
  'fdb4c687-916e-48ec-858e-6009cc763de3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='VALLEY',
    variable_level_3='V SHAPED VALLEY',
    detailed_variable='',
    identifier='fdb4c687-916e-48ec-858e-6009cc763de3'
  ),
  '948dea97-9843-4895-b59b-cb55f07a41b4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='WATERFALL',
    variable_level_3='',
    detailed_variable='',
    identifier='948dea97-9843-4895-b59b-cb55f07a41b4'
  ),
  '97a71326-75ac-422f-941e-c0c2897dd46b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL LANDFORMS',
    variable_level_2='WATERSHED/DRAINAGE BASINS',
    variable_level_3='',
    detailed_variable='',
    identifier='97a71326-75ac-422f-941e-c0c2897dd46b'
  ),
  '6f47ae88-f28f-43e3-be6a-34f86b15fe19': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='6f47ae88-f28f-43e3-be6a-34f86b15fe19'
  ),
  'efacd4f6-59ea-4019-8265-8cc81ecc99c0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='ABRASION',
    variable_level_3='',
    detailed_variable='',
    identifier='efacd4f6-59ea-4019-8265-8cc81ecc99c0'
  ),
  '9eedd20e-fce3-4fb2-9871-c0a327565ad9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='ATTRITION',
    variable_level_3='',
    detailed_variable='',
    identifier='9eedd20e-fce3-4fb2-9871-c0a327565ad9'
  ),
  '800606ea-9890-4475-af7b-100f529858d1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='DEGRADATION',
    variable_level_3='',
    detailed_variable='',
    identifier='800606ea-9890-4475-af7b-100f529858d1'
  ),
  'aff6bb19-84d0-40ed-8b81-a2210c468283': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='DOWNCUTTING',
    variable_level_3='',
    detailed_variable='',
    identifier='aff6bb19-84d0-40ed-8b81-a2210c468283'
  ),
  'e89704aa-91a0-4888-bb33-a9073eff7119': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='ENTRAINMENT',
    variable_level_3='',
    detailed_variable='',
    identifier='e89704aa-91a0-4888-bb33-a9073eff7119'
  ),
  '267eca20-09a7-46ad-89f4-111ccb3fd16d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='HYDRAULIC ACTION',
    variable_level_3='',
    detailed_variable='',
    identifier='267eca20-09a7-46ad-89f4-111ccb3fd16d'
  ),
  '54e5d072-5a2c-471b-bca0-7e4ca32a2001': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='LANDSLIDE',
    variable_level_3='',
    detailed_variable='',
    identifier='54e5d072-5a2c-471b-bca0-7e4ca32a2001'
  ),
  '93596daf-d2d3-4bb8-9626-9db100c402de': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='SALTATION',
    variable_level_3='',
    detailed_variable='',
    identifier='93596daf-d2d3-4bb8-9626-9db100c402de'
  ),
  '0c33b48d-1dd1-4309-bcd4-1ce3d0e24b46': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='SEDIMENT TRANSPORT',
    variable_level_3='',
    detailed_variable='',
    identifier='0c33b48d-1dd1-4309-bcd4-1ce3d0e24b46'
  ),
  '984e15c6-7eac-45b8-b098-ad82eab6be6e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='',
    detailed_variable='',
    identifier='984e15c6-7eac-45b8-b098-ad82eab6be6e'
  ),
  '09b4427b-9e8b-413a-83cd-f087b284cf61': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='SEDIMENT CHEMISTRY',
    detailed_variable='',
    identifier='09b4427b-9e8b-413a-83cd-f087b284cf61'
  ),
  '17747820-39de-4908-bb3d-8c2f94ddd6f4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='SEDIMENT COMPOSITION',
    detailed_variable='',
    identifier='17747820-39de-4908-bb3d-8c2f94ddd6f4'
  ),
  '103f1165-1008-4caa-bf77-5259ae1a7a36': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='STRATIGRAPHIC SEQUENCE',
    detailed_variable='',
    identifier='103f1165-1008-4caa-bf77-5259ae1a7a36'
  ),
  '8009663e-73c7-403e-b849-f40d2c3e3de8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='SUSPENSION',
    variable_level_3='',
    detailed_variable='',
    identifier='8009663e-73c7-403e-b849-f40d2c3e3de8'
  ),
  '6f47d087-21dc-41bc-955e-6eb2db8890cd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='FLUVIAL PROCESSES',
    variable_level_2='WEATHERING',
    variable_level_3='',
    detailed_variable='',
    identifier='6f47d087-21dc-41bc-955e-6eb2db8890cd'
  ),
  '3c78951a-0293-4fb0-baff-ec7372fe784d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='3c78951a-0293-4fb0-baff-ec7372fe784d'
  ),
  '8e73bff6-c2f9-46a6-963b-8ef09dd7f5f3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='ARETES',
    variable_level_3='',
    detailed_variable='',
    identifier='8e73bff6-c2f9-46a6-963b-8ef09dd7f5f3'
  ),
  'ae3b0c3d-35a1-4c94-ba72-ffe1a641902e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='CIRQUES/COMBES',
    variable_level_3='',
    detailed_variable='',
    identifier='ae3b0c3d-35a1-4c94-ba72-ffe1a641902e'
  ),
  'e0d85cf0-b477-47df-a067-18e28a3e228f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='CREVASSES',
    variable_level_3='',
    detailed_variable='',
    identifier='e0d85cf0-b477-47df-a067-18e28a3e228f'
  ),
  '429d0eba-2689-4674-9a8a-d88c4058b1bf': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='CREVASSES',
    variable_level_3='LONGITUDINAL CREVASSES',
    detailed_variable='',
    identifier='429d0eba-2689-4674-9a8a-d88c4058b1bf'
  ),
  'bc803dca-2fdb-4dc0-bf02-f0b9399d6816': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='CREVASSES',
    variable_level_3='MARGINAL CREVASSES',
    detailed_variable='',
    identifier='bc803dca-2fdb-4dc0-bf02-f0b9399d6816'
  ),
  '45101ace-ce83-4b56-bea6-c4eca9c693dd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='CREVASSES',
    variable_level_3='TRANSVERSE CREVASSES',
    detailed_variable='',
    identifier='45101ace-ce83-4b56-bea6-c4eca9c693dd'
  ),
  '4be9b544-68fa-45ea-89f1-a44a9f5929e5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='DRUMLINS',
    variable_level_3='',
    detailed_variable='',
    identifier='4be9b544-68fa-45ea-89f1-a44a9f5929e5'
  ),
  '5e012809-98cf-468f-bdf7-7cea8569d3ab': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='ESKERS',
    variable_level_3='',
    detailed_variable='',
    identifier='5e012809-98cf-468f-bdf7-7cea8569d3ab'
  ),
  '6aed82cb-be90-4e58-ae33-14943ea555be': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='FJORDS',
    variable_level_3='',
    detailed_variable='',
    identifier='6aed82cb-be90-4e58-ae33-14943ea555be'
  ),
  '5477fad4-789b-436d-a01e-610aa8efa592': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='GLACIAL HORNS',
    variable_level_3='',
    detailed_variable='',
    identifier='5477fad4-789b-436d-a01e-610aa8efa592'
  ),
  'b2d3b8a4-4861-4c21-b875-97084b6e75aa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='GLACIER STRIATIONS/GROOVES',
    variable_level_3='',
    detailed_variable='',
    identifier='b2d3b8a4-4861-4c21-b875-97084b6e75aa'
  ),
  'd23f75ed-29ea-4aa2-8785-fd3a3726bc33': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='GLACIER/HANGING/U-SHAPED VALLEYS',
    variable_level_3='',
    detailed_variable='',
    identifier='d23f75ed-29ea-4aa2-8785-fd3a3726bc33'
  ),
  '93b60653-f7bb-46f3-8f65-69221267018c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='GLACIER/ICE CAVES',
    variable_level_3='',
    detailed_variable='',
    identifier='93b60653-f7bb-46f3-8f65-69221267018c'
  ),
  'ee565a8c-72b9-44a4-b25d-efefd1a28d8d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='ICE-DAMMED LAKES',
    variable_level_3='',
    detailed_variable='',
    identifier='ee565a8c-72b9-44a4-b25d-efefd1a28d8d'
  ),
  '3f86db44-f853-4eb3-b4e3-4aaee481043a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='KAME DELTA',
    variable_level_3='',
    detailed_variable='',
    identifier='3f86db44-f853-4eb3-b4e3-4aaee481043a'
  ),
  '89541868-0ea0-47c6-b81e-a0c4981f2d62': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='KAMES',
    variable_level_3='',
    detailed_variable='',
    identifier='89541868-0ea0-47c6-b81e-a0c4981f2d62'
  ),
  '6d3722bb-29c0-4fb6-90c3-3f3a144b9941': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='KETTLE HOLES',
    variable_level_3='',
    detailed_variable='',
    identifier='6d3722bb-29c0-4fb6-90c3-3f3a144b9941'
  ),
  '4f590d94-110c-4762-9171-aba6d24af6a0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='MORAINES',
    variable_level_3='',
    detailed_variable='',
    identifier='4f590d94-110c-4762-9171-aba6d24af6a0'
  ),
  'a4f0e7c2-711e-4675-b7c8-f5430905aa89': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='MORAINES',
    variable_level_3='LATERAL MORAINE',
    detailed_variable='',
    identifier='a4f0e7c2-711e-4675-b7c8-f5430905aa89'
  ),
  '9d6c8fac-a5cd-4fbc-8283-1bc256c12a43': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='MORAINES',
    variable_level_3='MEDIAL MORAINE',
    detailed_variable='',
    identifier='9d6c8fac-a5cd-4fbc-8283-1bc256c12a43'
  ),
  'c4f0d15c-1f9b-40f3-b5d4-da1d6ebe6da8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='MORAINES',
    variable_level_3='RECESSIONAL MORAINE',
    detailed_variable='',
    identifier='c4f0d15c-1f9b-40f3-b5d4-da1d6ebe6da8'
  ),
  'a389919c-a6da-465e-b074-ea29b66a686b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='MORAINES',
    variable_level_3='RIBBED/ROGAN MORAINE',
    detailed_variable='',
    identifier='a389919c-a6da-465e-b074-ea29b66a686b'
  ),
  '7886c3eb-e86e-4a84-9f2f-e398ecc82b2d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='MORAINES',
    variable_level_3='TERMINAL MORAINE',
    detailed_variable='',
    identifier='7886c3eb-e86e-4a84-9f2f-e398ecc82b2d'
  ),
  '3b8bdda1-2415-47ea-b4cf-c802fa44c496': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='NUNATAKS',
    variable_level_3='',
    detailed_variable='',
    identifier='3b8bdda1-2415-47ea-b4cf-c802fa44c496'
  ),
  'a8bfc8ad-42f2-43cc-b161-20058037bb95': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='OUTWASH FANS/PLAINS',
    variable_level_3='',
    detailed_variable='',
    identifier='a8bfc8ad-42f2-43cc-b161-20058037bb95'
  ),
  '691cb42a-9de2-4f49-b1b4-9a4be80abd2b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='ROCHE MOUNTONNEES/SHEEPBACK',
    variable_level_3='',
    detailed_variable='',
    identifier='691cb42a-9de2-4f49-b1b4-9a4be80abd2b'
  ),
  '2d98cbaf-8c82-46e6-9962-a5e63918fe66': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='ROCK GLACIERS',
    variable_level_3='',
    detailed_variable='',
    identifier='2d98cbaf-8c82-46e6-9962-a5e63918fe66'
  ),
  '2bea72da-2cf3-403c-adb9-9d963eb71536': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL LANDFORMS',
    variable_level_2='TILL PLAINS',
    variable_level_3='',
    detailed_variable='',
    identifier='2bea72da-2cf3-403c-adb9-9d963eb71536'
  ),
  'd7b62912-5970-46b1-be45-6a603c9a6979': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d7b62912-5970-46b1-be45-6a603c9a6979'
  ),
  '99db4dca-4d07-48fd-8ba3-393532d04aa6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='ABLATION',
    variable_level_3='',
    detailed_variable='',
    identifier='99db4dca-4d07-48fd-8ba3-393532d04aa6'
  ),
  '8f57f4b0-5177-4362-81e8-ced75d37d1aa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='ABRASION',
    variable_level_3='',
    detailed_variable='',
    identifier='8f57f4b0-5177-4362-81e8-ced75d37d1aa'
  ),
  'c06e70c0-616c-44f2-a884-ad0252e29e37': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='CRUST REBOUND',
    variable_level_3='',
    detailed_variable='',
    identifier='c06e70c0-616c-44f2-a884-ad0252e29e37'
  ),
  'e60bfab8-01a8-4d0b-ae95-5d9014c71717': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='DEGRADATION',
    variable_level_3='',
    detailed_variable='',
    identifier='e60bfab8-01a8-4d0b-ae95-5d9014c71717'
  ),
  'b6d56c3f-daa4-4c2f-9c56-4cecdf3d9fcd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='DUMPING',
    variable_level_3='',
    detailed_variable='',
    identifier='b6d56c3f-daa4-4c2f-9c56-4cecdf3d9fcd'
  ),
  '1dc7ed2f-2834-4044-8caa-117ce12389af': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='ENTRAINMENT',
    variable_level_3='',
    detailed_variable='',
    identifier='1dc7ed2f-2834-4044-8caa-117ce12389af'
  ),
  '5b66d75f-331f-49d0-ad97-12f6535ce93a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='FIRN FORMATION',
    variable_level_3='',
    detailed_variable='',
    identifier='5b66d75f-331f-49d0-ad97-12f6535ce93a'
  ),
  'f7849055-fa5c-437c-a8c6-08c7db3a3b0a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='FREEZE/THAW',
    variable_level_3='',
    detailed_variable='',
    identifier='f7849055-fa5c-437c-a8c6-08c7db3a3b0a'
  ),
  'a72c8430-0b33-4167-b189-1309cc2048c5': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='FREEZE/THAW',
    variable_level_3='BASAL ICE FREEZING',
    detailed_variable='',
    identifier='a72c8430-0b33-4167-b189-1309cc2048c5'
  ),
  '5ddbaf71-b279-42cf-b250-faaefb627f66': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='GLACIAL DISPLACEMENT',
    variable_level_3='',
    detailed_variable='',
    identifier='5ddbaf71-b279-42cf-b250-faaefb627f66'
  ),
  '5cd3ad48-ade6-4306-a7de-4e68ecdf6bc7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='GLACIAL DRIFT',
    variable_level_3='',
    detailed_variable='',
    identifier='5cd3ad48-ade6-4306-a7de-4e68ecdf6bc7'
  ),
  '4be0198b-b88c-44db-b887-6cc7f5cd68f8': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='GLACIAL GROWTH',
    variable_level_3='',
    detailed_variable='',
    identifier='4be0198b-b88c-44db-b887-6cc7f5cd68f8'
  ),
  '114e9f84-8bc5-4863-abd2-55b80ed2af11': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='GLACIAL STRIATION',
    variable_level_3='',
    detailed_variable='',
    identifier='114e9f84-8bc5-4863-abd2-55b80ed2af11'
  ),
  'b87c5264-13c6-4716-acf3-51b2576dc1e9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='GLACIER CRUST SUBSIDENCE',
    variable_level_3='',
    detailed_variable='',
    identifier='b87c5264-13c6-4716-acf3-51b2576dc1e9'
  ),
  'fa0f38f3-2faa-4cd7-a848-22f3d96ab210': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='PERIGLACIAL PROCESSES',
    variable_level_3='',
    detailed_variable='',
    identifier='fa0f38f3-2faa-4cd7-a848-22f3d96ab210'
  ),
  'c4619d3d-f852-4899-9e33-9fd6d4096351': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='PLUCKING',
    variable_level_3='',
    detailed_variable='',
    identifier='c4619d3d-f852-4899-9e33-9fd6d4096351'
  ),
  '7ca88385-d0cf-439c-9a12-86b926b71582': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='SCOURING',
    variable_level_3='',
    detailed_variable='',
    identifier='7ca88385-d0cf-439c-9a12-86b926b71582'
  ),
  '791b7271-3a30-46ee-98e0-bc8239389950': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='SEDIMENT TRANSPORT',
    variable_level_3='',
    detailed_variable='',
    identifier='791b7271-3a30-46ee-98e0-bc8239389950'
  ),
  'd8f33f0a-137c-49ac-aebf-f8a8b0540a09': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='',
    detailed_variable='',
    identifier='d8f33f0a-137c-49ac-aebf-f8a8b0540a09'
  ),
  '7d10ff6d-efde-4f97-866b-7d771dd32b25': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='SEDIMENT CHEMISTRY',
    detailed_variable='',
    identifier='7d10ff6d-efde-4f97-866b-7d771dd32b25'
  ),
  'c25fef4a-f346-4831-8015-7853886c4fc7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='SEDIMENTATION',
    variable_level_3='STRATIGRAPHIC SEQUENCE',
    detailed_variable='',
    identifier='c25fef4a-f346-4831-8015-7853886c4fc7'
  ),
  '580ef100-0fb8-456c-a9ca-565d11392a26': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='GLACIAL PROCESSES',
    variable_level_2='WEATHERING',
    variable_level_3='',
    detailed_variable='',
    identifier='580ef100-0fb8-456c-a9ca-565d11392a26'
  ),
  'ac2d1035-1896-42c1-861b-042a917b6889': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST LANDFORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ac2d1035-1896-42c1-861b-042a917b6889'
  ),
  '631c5fb8-5e44-48f8-b937-a5f393d0832d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST LANDFORMS',
    variable_level_2='CAVES',
    variable_level_3='',
    detailed_variable='',
    identifier='631c5fb8-5e44-48f8-b937-a5f393d0832d'
  ),
  'a20151df-e7cf-43e0-9745-ffc965f97ef7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST LANDFORMS',
    variable_level_2='COCKPIT/TOWER KARST',
    variable_level_3='',
    detailed_variable='',
    identifier='a20151df-e7cf-43e0-9745-ffc965f97ef7'
  ),
  'c319a44c-b21a-491f-9cf0-65868507576c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST LANDFORMS',
    variable_level_2='KARST VALLEY',
    variable_level_3='',
    detailed_variable='',
    identifier='c319a44c-b21a-491f-9cf0-65868507576c'
  ),
  '7f298307-73f6-4f10-96a2-db381f357cb6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST LANDFORMS',
    variable_level_2='SINKHOLES (DOLINES)',
    variable_level_3='',
    detailed_variable='',
    identifier='7f298307-73f6-4f10-96a2-db381f357cb6'
  ),
  '40d9bf88-e7e2-4137-81fc-4721d67ce520': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST LANDFORMS',
    variable_level_2='UVALA',
    variable_level_3='',
    detailed_variable='',
    identifier='40d9bf88-e7e2-4137-81fc-4721d67ce520'
  ),
  '63846997-4a3f-41e1-9241-6d5053360d7a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST PROCESSES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='63846997-4a3f-41e1-9241-6d5053360d7a'
  ),
  '9902dc89-61fb-4a1e-becf-c8138122d2c4': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST PROCESSES',
    variable_level_2='CACO3',
    variable_level_3='',
    detailed_variable='',
    identifier='9902dc89-61fb-4a1e-becf-c8138122d2c4'
  ),
  '613abf26-7625-4134-8961-7a59fe82efc9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST PROCESSES',
    variable_level_2='DISSOLVED CO2',
    variable_level_3='',
    detailed_variable='',
    identifier='613abf26-7625-4134-8961-7a59fe82efc9'
  ),
  '05172a3b-cdc0-4e97-af29-e38cd4f271c6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST PROCESSES',
    variable_level_2='KARST HYDROLOGY',
    variable_level_3='',
    detailed_variable='',
    identifier='05172a3b-cdc0-4e97-af29-e38cd4f271c6'
  ),
  'bbfe00ab-ab63-40e0-8752-8f47d17c1d39': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST PROCESSES',
    variable_level_2='KARST HYDROLOGY',
    variable_level_3='SUBSURFACE DRAINAGE',
    detailed_variable='',
    identifier='bbfe00ab-ab63-40e0-8752-8f47d17c1d39'
  ),
  '07f6c977-077b-47f2-962c-00dadcd9f555': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST PROCESSES',
    variable_level_2='POROSITY',
    variable_level_3='',
    detailed_variable='',
    identifier='07f6c977-077b-47f2-962c-00dadcd9f555'
  ),
  '60dc0787-9e7e-4e0d-8023-d916da5d0836': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='KARST PROCESSES',
    variable_level_2='WEATHERING',
    variable_level_3='',
    detailed_variable='',
    identifier='60dc0787-9e7e-4e0d-8023-d916da5d0836'
  ),
  '46172bbe-8bf0-49a0-848f-129c089aeb8e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='46172bbe-8bf0-49a0-848f-129c089aeb8e'
  ),
  '5d9d1d85-b402-4f84-ab5c-03a49fc68c25': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='CALDERA',
    variable_level_3='',
    detailed_variable='',
    identifier='5d9d1d85-b402-4f84-ab5c-03a49fc68c25'
  ),
  '7c394040-91f1-4438-a50a-3118254f5989': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='CINDER CONE',
    variable_level_3='',
    detailed_variable='',
    identifier='7c394040-91f1-4438-a50a-3118254f5989'
  ),
  '6107d1c4-5aea-4bfa-861d-d77083a4476e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='FAULTS',
    variable_level_3='',
    detailed_variable='',
    identifier='6107d1c4-5aea-4bfa-861d-d77083a4476e'
  ),
  'a2a3893c-de51-4ca7-a952-e9a43dd961a1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='FOLDS',
    variable_level_3='',
    detailed_variable='',
    identifier='a2a3893c-de51-4ca7-a952-e9a43dd961a1'
  ),
  'ea580c65-2f66-4745-bbb6-dde61279ecfa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='GEYSER',
    variable_level_3='',
    detailed_variable='',
    identifier='ea580c65-2f66-4745-bbb6-dde61279ecfa'
  ),
  '524f075d-e875-4c9d-9e46-91f2a0b12168': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='GRABEN',
    variable_level_3='',
    detailed_variable='',
    identifier='524f075d-e875-4c9d-9e46-91f2a0b12168'
  ),
  'bf3fbdaa-cefb-4a54-8a4e-ee0a862795fb': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='HORST',
    variable_level_3='',
    detailed_variable='',
    identifier='bf3fbdaa-cefb-4a54-8a4e-ee0a862795fb'
  ),
  '33a0cd6c-a8e4-4187-a2f3-7eb4bf62808d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='LAVA DOME',
    variable_level_3='',
    detailed_variable='',
    identifier='33a0cd6c-a8e4-4187-a2f3-7eb4bf62808d'
  ),
  'dc18db4d-2184-453e-ba0a-86c83a9bede0': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='LAVA PLAIN',
    variable_level_3='',
    detailed_variable='',
    identifier='dc18db4d-2184-453e-ba0a-86c83a9bede0'
  ),
  'c1f717e9-da1a-4e85-ba2b-01986d53674d': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='MAAR',
    variable_level_3='',
    detailed_variable='',
    identifier='c1f717e9-da1a-4e85-ba2b-01986d53674d'
  ),
  'c34ea556-10bd-4665-9f22-68b5d05c9aea': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='MOUNTAINS',
    variable_level_3='',
    detailed_variable='',
    identifier='c34ea556-10bd-4665-9f22-68b5d05c9aea'
  ),
  '0baf564f-f942-4aeb-9b75-30b838f28f3f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='PLATEAU',
    variable_level_3='',
    detailed_variable='',
    identifier='0baf564f-f942-4aeb-9b75-30b838f28f3f'
  ),
  'ca091be1-4762-49ec-859b-a1a2fcb8e038': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='RIDGE',
    variable_level_3='',
    detailed_variable='',
    identifier='ca091be1-4762-49ec-859b-a1a2fcb8e038'
  ),
  '0bd4d492-4911-4a6a-afaa-34899a80294b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='RIFT VALLEY',
    variable_level_3='',
    detailed_variable='',
    identifier='0bd4d492-4911-4a6a-afaa-34899a80294b'
  ),
  'a355aafc-f0ce-4774-afc3-82b41df5f022': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='TUYA',
    variable_level_3='',
    detailed_variable='',
    identifier='a355aafc-f0ce-4774-afc3-82b41df5f022'
  ),
  'cefe2205-809c-4386-915e-a8737ae8e68e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC LANDFORMS',
    variable_level_2='VOLCANO',
    variable_level_3='',
    detailed_variable='',
    identifier='cefe2205-809c-4386-915e-a8737ae8e68e'
  ),
  'f0bd7eeb-9004-4e40-a649-f6010d8a4303': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC PROCESSES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='f0bd7eeb-9004-4e40-a649-f6010d8a4303'
  ),
  'ebcd5f14-9468-493b-b0e6-de5afda2621a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC PROCESSES',
    variable_level_2='EPEIROGENIC MOVEMENT',
    variable_level_3='',
    detailed_variable='',
    identifier='ebcd5f14-9468-493b-b0e6-de5afda2621a'
  ),
  'ca464924-4299-46ea-8cae-fd9bad49c1b1': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC PROCESSES',
    variable_level_2='ISOSTATIC UPLIFT',
    variable_level_3='',
    detailed_variable='',
    identifier='ca464924-4299-46ea-8cae-fd9bad49c1b1'
  ),
  '46d188a9-1099-4d72-b466-6e839297320e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC PROCESSES',
    variable_level_2='OROGENIC MOVEMENT',
    variable_level_3='',
    detailed_variable='',
    identifier='46d188a9-1099-4d72-b466-6e839297320e'
  ),
  '9c207e15-9947-4849-bdf4-c1893a7f800a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC PROCESSES',
    variable_level_2='RIFTING',
    variable_level_3='',
    detailed_variable='',
    identifier='9c207e15-9947-4849-bdf4-c1893a7f800a'
  ),
  '44dd98d0-a0d0-46b2-bb98-ed887ce7fa60': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC PROCESSES',
    variable_level_2='SUBDUCTION',
    variable_level_3='',
    detailed_variable='',
    identifier='44dd98d0-a0d0-46b2-bb98-ed887ce7fa60'
  ),
  '4bc109b5-6788-4f64-8238-745bab3910dd': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOMORPHIC LANDFORMS/PROCESSES',
    variable_level_1='TECTONIC PROCESSES',
    variable_level_2='TECTONIC UPLIFT',
    variable_level_3='',
    detailed_variable='',
    identifier='4bc109b5-6788-4f64-8238-745bab3910dd'
  ),
  'ec2bf43d-2525-439e-bbbe-0db758e71965': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOTHERMAL DYNAMICS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ec2bf43d-2525-439e-bbbe-0db758e71965'
  ),
  '33d1810f-40c6-4b37-ac90-7435ef5fa507': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOTHERMAL DYNAMICS',
    variable_level_1='GEOTHERMAL ENERGY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='33d1810f-40c6-4b37-ac90-7435ef5fa507'
  ),
  '9d258088-e5bd-42b9-a281-e566da10ea74': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOTHERMAL DYNAMICS',
    variable_level_1='GEOTHERMAL ENERGY',
    variable_level_2='ENERGY DISTRIBUTION',
    variable_level_3='',
    detailed_variable='',
    identifier='9d258088-e5bd-42b9-a281-e566da10ea74'
  ),
  '64461e23-b3c1-4b99-b879-84e54bacdb24': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOTHERMAL DYNAMICS',
    variable_level_1='GEOTHERMAL ENERGY',
    variable_level_2='ENERGY OUTPUT',
    variable_level_3='',
    detailed_variable='',
    identifier='64461e23-b3c1-4b99-b879-84e54bacdb24'
  ),
  'cacfd8f0-b83a-46b7-b324-52ce1b55baa9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOTHERMAL DYNAMICS',
    variable_level_1='GEOTHERMAL TEMPERATURE',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='cacfd8f0-b83a-46b7-b324-52ce1b55baa9'
  ),
  '1d2ac206-0977-4145-b334-baa6e13a0db6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOTHERMAL DYNAMICS',
    variable_level_1='GEOTHERMAL TEMPERATURE',
    variable_level_2='AMBIENT TEMPERATURE',
    variable_level_3='',
    detailed_variable='',
    identifier='1d2ac206-0977-4145-b334-baa6e13a0db6'
  ),
  '99d567bb-7767-4ea4-a135-a611eac6a669': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOTHERMAL DYNAMICS',
    variable_level_1='GEOTHERMAL TEMPERATURE',
    variable_level_2='TEMPERATURE GRADIENT',
    variable_level_3='',
    detailed_variable='',
    identifier='99d567bb-7767-4ea4-a135-a611eac6a669'
  ),
  'f4573e47-3cce-49ec-98d3-b5b3bb51371e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOTHERMAL DYNAMICS',
    variable_level_1='GEOTHERMAL TEMPERATURE',
    variable_level_2='TEMPERATURE GRADIENT',
    variable_level_3='TEMPERATURE GRADIENT RATE',
    detailed_variable='',
    identifier='f4573e47-3cce-49ec-98d3-b5b3bb51371e'
  ),
  '321d9086-fc85-40a3-a2e0-d24bc6765345': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GEOTHERMAL DYNAMICS',
    variable_level_1='GEOTHERMAL TEMPERATURE',
    variable_level_2='TEMPERATURE PROFILES',
    variable_level_3='',
    detailed_variable='',
    identifier='321d9086-fc85-40a3-a2e0-d24bc6765345'
  ),
  '221386f6-ef9b-4990-82b3-f990b0fe39fa': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GRAVITY/GRAVITATIONAL FIELD',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='221386f6-ef9b-4990-82b3-f990b0fe39fa'
  ),
  '8b39b880-f385-4dab-a563-24064b43be7e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GRAVITY/GRAVITATIONAL FIELD',
    variable_level_1='CONTROL SURVEYS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='8b39b880-f385-4dab-a563-24064b43be7e'
  ),
  '122f7d15-7e5c-4249-992c-c753c80cf05b': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GRAVITY/GRAVITATIONAL FIELD',
    variable_level_1='CRUSTAL MOTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='122f7d15-7e5c-4249-992c-c753c80cf05b'
  ),
  '5dee7d0e-e13e-4974-9750-79d5cd886c7a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GRAVITY/GRAVITATIONAL FIELD',
    variable_level_1='CRUSTAL MOTION',
    variable_level_2='ISOSTATIC ADJUSTMENTS',
    variable_level_3='',
    detailed_variable='',
    identifier='5dee7d0e-e13e-4974-9750-79d5cd886c7a'
  ),
  'aa6c2fe7-3261-4fd8-bed4-81403bc49086': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GRAVITY/GRAVITATIONAL FIELD',
    variable_level_1='CRUSTAL MOTION',
    variable_level_2='OCEAN CRUST DEFORMATION',
    variable_level_3='',
    detailed_variable='',
    identifier='aa6c2fe7-3261-4fd8-bed4-81403bc49086'
  ),
  '56b4cbe5-e5f7-4e61-8c48-bbb858b505e6': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GRAVITY/GRAVITATIONAL FIELD',
    variable_level_1='GRAVITATIONAL FIELD',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='56b4cbe5-e5f7-4e61-8c48-bbb858b505e6'
  ),
  '69af3046-08e0-4c24-981d-803c0412ce58': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GRAVITY/GRAVITATIONAL FIELD',
    variable_level_1='GRAVITY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='69af3046-08e0-4c24-981d-803c0412ce58'
  ),
  'fb7eeee0-9ad1-40f8-baa2-df7dc3acb6d3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GRAVITY/GRAVITATIONAL FIELD',
    variable_level_1='GRAVITY ANOMALIES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='fb7eeee0-9ad1-40f8-baa2-df7dc3acb6d3'
  ),
  'c44b078d-ec95-47d5-9a43-ba8475e568d2': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GRAVITY/GRAVITATIONAL FIELD',
    variable_level_1='POLAR MOTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='c44b078d-ec95-47d5-9a43-ba8475e568d2'
  ),
  '9d184041-9848-4f76-affd-74f4e4fd7462': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GRAVITY/GRAVITATIONAL FIELD',
    variable_level_1='POLAR MOTION',
    variable_level_2='ANNUAL ELLIPTICAL COMPONENT',
    variable_level_3='',
    detailed_variable='',
    identifier='9d184041-9848-4f76-affd-74f4e4fd7462'
  ),
  'a983aad3-c72a-49e8-8de9-e0aaf35e14b3': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GRAVITY/GRAVITATIONAL FIELD',
    variable_level_1='POLAR MOTION',
    variable_level_2='CHANDLER CIRCULAR COMPONENT',
    variable_level_3='',
    detailed_variable='',
    identifier='a983aad3-c72a-49e8-8de9-e0aaf35e14b3'
  ),
  '05225982-60ab-4772-a0b7-f67c3b853ab9': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GRAVITY/GRAVITATIONAL FIELD',
    variable_level_1='ROTATIONAL MOTION/VARIATIONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='05225982-60ab-4772-a0b7-f67c3b853ab9'
  ),
  'd5d9bd6a-92c4-49ac-bddf-0077cf804ea7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GRAVITY/GRAVITATIONAL FIELD',
    variable_level_1='ROTATIONAL MOTION/VARIATIONS',
    variable_level_2='ROTATIONAL RATE/SPEED',
    variable_level_3='',
    detailed_variable='',
    identifier='d5d9bd6a-92c4-49ac-bddf-0077cf804ea7'
  ),
  '4bb526d7-2c14-43bc-a2a7-f166b5c41a3a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GRAVITY/GRAVITATIONAL FIELD',
    variable_level_1='ROTATIONAL MOTION/VARIATIONS',
    variable_level_2='TIDAL FRICTION',
    variable_level_3='',
    detailed_variable='',
    identifier='4bb526d7-2c14-43bc-a2a7-f166b5c41a3a'
  ),
  '71278ba7-9a13-43ba-9ec3-62ae2b39de88': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GRAVITY/GRAVITATIONAL FIELD',
    variable_level_1='SATELLITE ORBITS/REVOLUTION',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='71278ba7-9a13-43ba-9ec3-62ae2b39de88'
  ),
  '96427b44-91a8-4ace-8276-0117948878ee': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GRAVITY/GRAVITATIONAL FIELD',
    variable_level_1='SATELLITE ORBITS/REVOLUTION',
    variable_level_2='ANGLE OF ELEVATION',
    variable_level_3='',
    detailed_variable='',
    identifier='96427b44-91a8-4ace-8276-0117948878ee'
  ),
  '025d666e-a5bb-48b5-9890-129e60104611': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GRAVITY/GRAVITATIONAL FIELD',
    variable_level_1='SATELLITE ORBITS/REVOLUTION',
    variable_level_2='ANGLE OF INCLINATION',
    variable_level_3='',
    detailed_variable='',
    identifier='025d666e-a5bb-48b5-9890-129e60104611'
  ),
  'e709d2f9-c110-4e71-b4da-ff1a7c382d99': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GRAVITY/GRAVITATIONAL FIELD',
    variable_level_1='SATELLITE ORBITS/REVOLUTION',
    variable_level_2='ORBIT TYPE',
    variable_level_3='',
    detailed_variable='',
    identifier='e709d2f9-c110-4e71-b4da-ff1a7c382d99'
  ),
  '53eeb68a-615d-42d0-9c6b-ddfe0d0eb2c7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GRAVITY/GRAVITATIONAL FIELD',
    variable_level_1='SATELLITE ORBITS/REVOLUTION',
    variable_level_2='ORBIT VELOCITY',
    variable_level_3='',
    detailed_variable='',
    identifier='53eeb68a-615d-42d0-9c6b-ddfe0d0eb2c7'
  ),
  'e72ba365-ea43-42ef-acd1-05ac5c46f29a': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='GRAVITY/GRAVITATIONAL FIELD',
    variable_level_1='SATELLITE ORBITS/REVOLUTION',
    variable_level_2='ORBITAL POSITION',
    variable_level_3='',
    detailed_variable='',
    identifier='e72ba365-ea43-42ef-acd1-05ac5c46f29a'
  ),
  'ba8d7f68-ad3a-4874-bc75-312b24b1b1ac': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='ROCKS/MINERALS/CRYSTALS',
    variable_level_1='',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='ba8d7f68-ad3a-4874-bc75-312b24b1b1ac'
  ),
  'd4ac49a1-9ba5-4a90-a033-2ef317028352': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='ROCKS/MINERALS/CRYSTALS',
    variable_level_1='AGE DETERMINATIONS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='d4ac49a1-9ba5-4a90-a033-2ef317028352'
  ),
  '4beaeec9-0750-44e6-8fb4-8d0085efc82e': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='ROCKS/MINERALS/CRYSTALS',
    variable_level_1='BEDROCK LITHOLOGY',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='4beaeec9-0750-44e6-8fb4-8d0085efc82e'
  ),
  'da22144c-634d-4007-aba9-e636a9f2fa3f': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='ROCKS/MINERALS/CRYSTALS',
    variable_level_1='ELEMENTS',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='da22144c-634d-4007-aba9-e636a9f2fa3f'
  ),
  '2440389a-d0d9-445a-9dce-908900f0c3a7': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='ROCKS/MINERALS/CRYSTALS',
    variable_level_1='ELEMENTS',
    variable_level_2='MAJOR ELEMENTS',
    variable_level_3='',
    detailed_variable='',
    identifier='2440389a-d0d9-445a-9dce-908900f0c3a7'
  ),
  '63cf1bca-72c7-4f5e-8018-3d22befa7147': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='ROCKS/MINERALS/CRYSTALS',
    variable_level_1='ELEMENTS',
    variable_level_2='MINOR ELEMENTS',
    variable_level_3='',
    detailed_variable='',
    identifier='63cf1bca-72c7-4f5e-8018-3d22befa7147'
  ),
  '334f47c1-fd13-483f-b493-e69a9e93d553': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='ROCKS/MINERALS/CRYSTALS',
    variable_level_1='ELEMENTS',
    variable_level_2='RADIOACTIVE ELEMENTS',
    variable_level_3='',
    detailed_variable='',
    identifier='334f47c1-fd13-483f-b493-e69a9e93d553'
  ),
  'c3c898d7-14db-4536-bd86-f8f222167195': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='ROCKS/MINERALS/CRYSTALS',
    variable_level_1='ELEMENTS',
    variable_level_2='TRACE ELEMENTS',
    variable_level_3='',
    detailed_variable='',
    identifier='c3c898d7-14db-4536-bd86-f8f222167195'
  ),
  'a654a922-8b69-46f2-be40-d4d830ce999c': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='ROCKS/MINERALS/CRYSTALS',
    variable_level_1='GAS HYDRATES',
    variable_level_2='',
    variable_level_3='',
    detailed_variable='',
    identifier='a654a922-8b69-46f2-be40-d4d830ce999c'
  ),
  '43561874-c5c4-47d5-8daf-e99fab694042': ScienceKeywordRecord(
    category='EARTH SCIENCE',
    topic='SOLID EARTH',
    term='ROCKS/MINERALS/CRYSTALS',
    variable_level_1='GAS HYDRATES',
    variable