"""Utils function for Yuma."""

from __future__ import annotations

import copy
from pathlib import Path
from typing import TYPE_CHECKING, Any, cast

from ftag import Label
from ftag.hdf5 import H5Reader

from puma.hlplots.results import Tagger
from puma.utils import logger

if TYPE_CHECKING:  # pragma: no cover
    from puma.hlplots.results import Results


def combine_suffixes(suffixes: list) -> str:
    """Combines a list of suffixes into a single suffix.

    Parameters
    ----------
    suffixes : list
        List of suffixes

    Returns
    -------
    str
        Combined suffix string
    """
    clean_suffixes = [s for s in suffixes if s not in {"", None}]
    if len(clean_suffixes) == 0:
        return ""
    return "_".join([s for s in suffixes if s not in {"", None}])


def get_include_exclude_str(include_taggers: dict, all_taggers: dict) -> str:
    """Generates the name of the plot, based on the included taggers.

    Parameters
    ----------
    include_taggers : dict
        Taggers that are to included
    all_taggers : dict
        All taggers

    Returns
    -------
    str
        Name of the plot based on the included taggers
    """
    if len(include_taggers) == len(all_taggers):
        return ""

    return "taggers_" + "_".join([t.yaml_name for t in include_taggers.values()])


def get_included_taggers(
    results: Results,
    plot_config: dict[str, Any],
) -> tuple[dict[str, Tagger], dict[str, Tagger], str]:
    """Converts 'include_taggers' or 'exclude_taggers' into the taggers to include.

    Parameters
    ----------
    results : Results
        Results object from which the taggers are taken
    plot_config : dict
        Plot config dict

    Returns
    -------
    tuple[dict | Any | None, dict, str]
        The included taggers as dict, all taggers as dict and the include_exclude string

    Raises
    ------
    ValueError
        If no tagger is left in the include taggers
        If the reference is not in the included taggers
    """
    # Make the type explicit for mypy
    all_taggers = cast(dict[str, Tagger], results.taggers)
    all_tagger_names = [t.yaml_name for t in all_taggers.values()]

    # Read config into separate, typed locals
    incl = cast(list[str] | None, plot_config.get("include_taggers"))
    excl = cast(list[str] | None, plot_config.get("exclude_taggers"))

    include_taggers: dict[str, Tagger]
    if not incl and not excl:
        include_taggers = dict(all_taggers)  # copy to avoid mutating upstream
    elif incl:
        assert all(
            t in all_tagger_names for t in incl
        ), f"Not all taggers are in the results: {incl}"
        include_taggers = {k: v for k, v in all_taggers.items() if v.yaml_name in incl}
    else:
        assert excl is not None
        assert all(
            t in all_tagger_names for t in excl
        ), f"Not all excluded taggers are in the results: {excl}"
        include_taggers = {k: v for k, v in all_taggers.items() if v.yaml_name not in excl}

    if len(include_taggers) == 0:
        raise ValueError(
            "No taggers included in plot, check that 'exclude_taggers' doesn't exclude "
            "all taggers, or that at least 1 tagger is defined in 'include_taggers'"
        )
    logger.debug("Include taggers: %s", include_taggers)

    # Ensure a reference is set
    if not any(t.reference for t in include_taggers.values()):
        ref_name = cast(str | None, plot_config.get("reference"))
        if ref_name is not None:
            if ref_name not in [t.yaml_name for t in include_taggers.values()]:
                raise ValueError(
                    f"Reference {ref_name} not in included taggers {list(include_taggers.keys())}"
                )
            ref_key = next(k for k, t in include_taggers.items() if t.yaml_name == ref_name)
        else:
            ref_key = next(iter(include_taggers.keys()))
            logger.info("No reference set for plot, using %s as reference", ref_key)

        include_taggers[ref_key] = copy.deepcopy(include_taggers[ref_key])
        include_taggers[ref_key].reference = True

    return (
        include_taggers,
        all_taggers,
        get_include_exclude_str(include_taggers, all_taggers),
    )


def get_tagger_name(name: str, sample_path: Path, key: str, flavours: list[Label]):
    """Attempts to return the name of the tagger if it is not specified in the config
    file by looking at available variable names, and the key of the tagger in the config.

    Parameters
    ----------
    name : str
        The name of the tagger, if specified in the config file
    sample_path : Path
        The path to the sample file
    key : str
        The key of the tagger in the config file, if multiple taggers are found in the
        sample path, this is used to select the correct tagger
    flavours : list[Label]
        The flavours of the tagger, used to identify the correct tagger. A 'valid'
        tagger in the file is one where tagger_p{flav} exists for all flavours defined
        in this list

    Returns
    -------
    str
        The name of the tagger to use

    Raises
    ------
    ValueError
        If no valid tagger was found
    """
    if name:
        return name

    reader = H5Reader(sample_path)
    jet_vars = reader.dtypes()["jets"].names
    req_keys = [f"_p{flav.name[:-4]}" for flav in flavours]

    potential_taggers: dict[str, list] = {}

    # Identify potential taggers
    for var in jet_vars:
        for suffix in req_keys:
            if var.endswith(suffix):
                base_name = var.rsplit(suffix, 1)[0]
                if base_name in potential_taggers:
                    potential_taggers[base_name].append(suffix)
                else:
                    potential_taggers[base_name] = [suffix]

    # Check if any base name has all three suffixes
    valid_taggers = [
        base for base, suffixes in potential_taggers.items() if set(suffixes) == set(req_keys)
    ]

    if len(valid_taggers) == 0:
        raise ValueError("No valid tagger found.")
    if len(valid_taggers) > 1:
        if key in valid_taggers:
            return key
        raise ValueError(
            f"Multiple valid taggers found: {', '.join(valid_taggers)} in file "
            f"{sample_path}, please specify the tagger name in the config file"
        )
    return valid_taggers[0]
