# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at https://mozilla.org/MPL/2.0/.
# SPDX-License-Identifier: MPL-2.0
from typing import List, Dict, Any
from VeraGridEngine.Utils.Symbolic.block import Block
from VeraGridEngine.Devices.Dynamic.rms_template import RmsModelTemplate


class DynamicModelHost:
    """
    This class serves to give flexible access to either a template or a custom model
    """

    def __init__(self):

        self._template: Block | None = None

        # a custom model always exits although it may be empty
        self._custom_model: Block = Block()

    @property
    def template(self):
        return self._template

    @template.setter
    def template(self, val: Block | RmsModelTemplate):

        if isinstance(val, RmsModelTemplate):
            self._template = val.block

        elif isinstance(val, Block):
            self._template = val

        else:
            raise ValueError(f"Cannot set template with {val}")

    @property
    def custom_model(self):
        return self._custom_model

    @property
    def model(self) -> Block:
        """
        Returns whatever is available with preference to the custom model if any
        :return: DynamicModel (even if it is empty)
        """
        if self.template is None:
            return self.custom_model
        else:
            return self.template

    @model.setter
    def model(self, val: Block | RmsModelTemplate):

        if isinstance(val, RmsModelTemplate):
            self.template = val.block

        elif isinstance(val, Block):
            self._custom_model = val

        else:
            raise ValueError(f"Cannot set model with {val}")

    def to_dict(self) -> Dict[str, int | Dict[str, List[Dict[str, Any]]]]:
        """
        Generate a dictionary to save
        :return: Data to save
        """
        return {
            "template": self.template.uid if self.template is not None else None,
            "custom_model": self.custom_model.to_dict()
        }

    def parse(self, data: Dict[str, str | Dict[str, List[Dict[str, Any]]]],
              models_dict: Dict[str, RmsModelTemplate]):
        """
        Parse the data
        :param data: data generated by to_dict
        :param models_dict: dictionary of DynamicModel to find the template reference
        """
        template_id = data.get("template", None)
        if template_id is not None:
            self.template = models_dict.get(template_id, None)

        custom_data = data.get("custom_model", None)
        self._custom_model = Block.parse(data=custom_data)

    def empty(self):
        if self._template is None:
            return self._custom_model.empty()
        else:
            return self._template.empty()

    def __eq__(self, other):
        if isinstance(other, DynamicModelHost):

            if self.template is None:
                if other.template is None:
                    return self.custom_model == other.custom_model
                else:
                    return False
            else:
                if other.template is None:
                    return False
                else:
                    return self.template.uid == other.template.uid
        else:
            return False

    def copy(self) -> "DynamicModelHost":
        """
        Deep copy of DynamicModelHost
        :return: DynamicModelHost
        """
        obj = DynamicModelHost()
        obj._custom_model = self._custom_model.copy()
        obj._template = self._template
        return obj

