// Copyright 2021 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "stim/diagram/timeline/timeline_ascii_drawer.h"

#include "gtest/gtest.h"

#include "stim/circuit/circuit.test.h"
#include "stim/gen/circuit_gen_params.h"
#include "stim/gen/gen_rep_code.h"
#include "stim/gen/gen_surface_code.h"

using namespace stim;
using namespace stim_draw_internal;

TEST(circuit_diagram_timeline_text, single_qubit_gates) {
    Circuit circuit(R"CIRCUIT(
        I 0
        X 1
        Y 2
        Z 3
        C_XYZ 0
        C_ZYX 1
        H 2
        H_XY 3
        H_XZ 0
        H_YZ 1
        S 2
        SQRT_X 3
        SQRT_X_DAG 0
        SQRT_Y 1
        SQRT_Y_DAG 2
        SQRT_Z 3
        SQRT_Z_DAG 0
        S_DAG 1
        H 2 0 3
    )CIRCUIT");
    ASSERT_EQ("\n" + DiagramTimelineAsciiDrawer::make_diagram(circuit).str() + "\n", R"DIAGRAM(
q0: -I-C_XYZ-H------SQRT_X_DAG-S_DAG-H-

q1: -X-C_ZYX-H_YZ---SQRT_Y-----S_DAG---

q2: -Y-H-----S------SQRT_Y_DAG-H-------

q3: -Z-H_XY--SQRT_X-S----------------H-
)DIAGRAM");
}

TEST(circuit_diagram_timeline_text, two_qubits_gates) {
    Circuit circuit(R"CIRCUIT(
        CNOT 0 1
        CX 2 3
        CY 4 5 5 4
        CZ 0 2
        ISWAP 1 3
        ISWAP_DAG 2 4
        SQRT_XX 3 5
        SQRT_XX_DAG 0 5
        SQRT_YY 3 4 4 3
        SQRT_YY_DAG 0 1
        SQRT_ZZ 2 3
        SQRT_ZZ_DAG 4 5
        SWAP 0 1
        XCX 2 3
        XCY 3 4
        XCZ 0 1
        YCX 2 3
        YCY 4 5
        YCZ 0 1
        ZCX 2 3
        ZCY 4 5
        ZCZ 0 5 2 3 1 4
        CXSWAP 0 1
        SWAPCX 0 1
    )CIRCUIT");
    ASSERT_EQ("\n" + DiagramTimelineAsciiDrawer::make_diagram(circuit).str() + "\n", R"DIAGRAM(
q0: -@-@-------------------------SQRT_XX_DAG---------SQRT_YY_DAG-SWAP----------X-Y---@-----ZSWAP-XSWAP-
     | |                         |                   |           |             | |   |     |     |
q1: -X-|-ISWAP-------------------|-------------------SQRT_YY_DAG-SWAP----------@-@---|---@-XSWAP-ZSWAP-
       | |                       |                                                   |   |
q2: -@-@-|-----ISWAP_DAG---------|-------------------------------SQRT_ZZ-----X---Y-@-|-@-|-------------
     |   |     |                 |                               |           |   | | | | |
q3: -X---ISWAP-|---------SQRT_XX-|-----------SQRT_YY-SQRT_YY-----SQRT_ZZ-----X-X-X-X-|-@-|-------------
               |         |       |           |       |                         |     |   |
q4: -@-Y-------ISWAP_DAG-|-------|-----------SQRT_YY-SQRT_YY-----SQRT_ZZ_DAG---Y-Y-@-|---@-------------
     | |                 |       |                               |               | | |
q5: -Y-@-----------------SQRT_XX-SQRT_XX_DAG---------------------SQRT_ZZ_DAG-----Y-Y-@-----------------
)DIAGRAM");
}

TEST(circuit_diagram_timeline_text, noise_gates) {
    Circuit circuit(R"CIRCUIT(
        DEPOLARIZE1(0.125) 0 1
        DEPOLARIZE2(0.125) 0 2 4 5
        X_ERROR(0.125) 0 1 2
        Y_ERROR(0.125) 0 1 4
        Z_ERROR(0.125) 2 3 5
    )CIRCUIT");
    ASSERT_EQ("\n" + DiagramTimelineAsciiDrawer::make_diagram(circuit).str() + "\n", R"DIAGRAM(
q0: -DEPOLARIZE1(0.125)-DEPOLARIZE2(0.125)-X_ERROR(0.125)-Y_ERROR(0.125)-
                        |
q1: -DEPOLARIZE1(0.125)-|------------------X_ERROR(0.125)-Y_ERROR(0.125)-
                        |
q2: --------------------DEPOLARIZE2(0.125)-X_ERROR(0.125)-Z_ERROR(0.125)-

q3: ------------------------------------------------------Z_ERROR(0.125)-

q4: --------------------DEPOLARIZE2(0.125)----------------Y_ERROR(0.125)-
                        |
q5: --------------------DEPOLARIZE2(0.125)----------------Z_ERROR(0.125)-
)DIAGRAM");

    circuit = Circuit(R"CIRCUIT(
        E(0.25) X1 X2
        CORRELATED_ERROR(0.125) X1 Y2 Z3
        ELSE_CORRELATED_ERROR(0.25) X2 Y4 Z3
        ELSE_CORRELATED_ERROR(0.25) X5
    )CIRCUIT");
    ASSERT_EQ("\n" + DiagramTimelineAsciiDrawer::make_diagram(circuit).str() + "\n", R"DIAGRAM(
q0: --------------------------------------------------------------------------------------

q1: -E[X](0.25)-E[X](0.125)---------------------------------------------------------------
     |          |
q2: -E[X](0.25)-E[Y](0.125)-ELSE_CORRELATED_ERROR[X](0.25)--------------------------------
                |           |
q3: ------------E[Z](0.125)-ELSE_CORRELATED_ERROR[Z](0.25)--------------------------------
                            |
q4: ------------------------ELSE_CORRELATED_ERROR[Y](0.25)--------------------------------

q5: -------------------------------------------------------ELSE_CORRELATED_ERROR[X](0.25)-
)DIAGRAM");

    circuit = Circuit(R"CIRCUIT(
        PAULI_CHANNEL_1(0.125,0.25,0.125) 0 1 2 3
        PAULI_CHANNEL_2(0.01,0.01,0.01,0.02,0.01,0.01,0.01,0.01,0.01,0.01,0.01,0.01,0.01,0.01,0.01) 0 1 2 4
    )CIRCUIT");
    ASSERT_EQ("\n" + DiagramTimelineAsciiDrawer::make_diagram(circuit).str() + "\n", R"DIAGRAM(
q0: -PAULI_CHANNEL_1(0.125,0.25,0.125)-PAULI_CHANNEL_2[0](0.01,0.01,0.01,0.02,0.01,0.01,0.01,0.01,0.01,0.01,0.01,0.01,0.01,0.01,0.01)-
                                       |
q1: -PAULI_CHANNEL_1(0.125,0.25,0.125)-PAULI_CHANNEL_2[1](0.01,0.01,0.01,0.02,0.01,0.01,0.01,0.01,0.01,0.01,0.01,0.01,0.01,0.01,0.01)-

q2: -PAULI_CHANNEL_1(0.125,0.25,0.125)-PAULI_CHANNEL_2[0](0.01,0.01,0.01,0.02,0.01,0.01,0.01,0.01,0.01,0.01,0.01,0.01,0.01,0.01,0.01)-
                                       |
q3: -PAULI_CHANNEL_1(0.125,0.25,0.125)-|----------------------------------------------------------------------------------------------
                                       |
q4: -----------------------------------PAULI_CHANNEL_2[1](0.01,0.01,0.01,0.02,0.01,0.01,0.01,0.01,0.01,0.01,0.01,0.01,0.01,0.01,0.01)-
)DIAGRAM");
}

TEST(circuit_diagram_timeline_text, collapsing) {
    Circuit circuit(R"CIRCUIT(
        R 0
        RX 1
        RY 2
        RZ 3
        M(0.001) 0 1
        MR 1 0
        MRX 1 2
        MRY 0 3 1
        MRZ 0
        MX 1
        MY 2
        MZ 3
        MPP X0*Y2 Z3 X1 Z2*Y3
    )CIRCUIT");
    ASSERT_EQ("\n" + DiagramTimelineAsciiDrawer::make_diagram(circuit).str() + "\n", R"DIAGRAM(
q0: -R--M(0.001):rec[0]-MR:rec[3]-MRY:rec[6]-MR:rec[9]-------------MPP[X]:rec[13]----------------
                                                                   |
q1: -RX-M(0.001):rec[1]-MR:rec[2]-MRX:rec[4]-MRY:rec[8]-MX:rec[10]-|--------------MPP[X]:rec[15]-
                                                                   |
q2: -RY---------------------------MRX:rec[5]------------MY:rec[11]-MPP[Y]:rec[13]-MPP[Z]:rec[16]-
                                                                                  |
q3: -R----------------------------MRY:rec[7]------------M:rec[12]--MPP[Z]:rec[14]-MPP[Y]:rec[16]-
)DIAGRAM");
}

TEST(circuit_diagram_timeline_text, measurement_looping) {
    Circuit circuit(R"CIRCUIT(
        M 0
        REPEAT 100 {
            M 1
            REPEAT 5 {
                M 2
            }
            REPEAT 7 {
                MPP X3*Y4
            }
        }
    )CIRCUIT");
    ASSERT_EQ("\n" + DiagramTimelineAsciiDrawer::make_diagram(circuit).str() + "\n", R"DIAGRAM(
              /REP 100                  /REP 5                        \ /REP 7                             \ \
q0: -M:rec[0]-|-------------------------|-----------------------------|-|----------------------------------|-|-
              |                         |                             | |                                  | |
q1: ----------|--------M:rec[1+iter*13]-|-----------------------------|-|----------------------------------|-|-
              |                         |                             | |                                  | |
q2: ----------|-------------------------|------M:rec[2+iter*13+iter2]-|-|----------------------------------|-|-
              |                         |                             | |                                  | |
q3: ----------|-------------------------|-----------------------------|-|------MPP[X]:rec[7+iter*13+iter2]-|-|-
              |                         |                             | |      |                           | |
q4: ----------|-------------------------|-----------------------------|-|------MPP[Y]:rec[7+iter*13+iter2]-|-|-
              \                         \                             / \                                  / /
)DIAGRAM");
}

TEST(circuit_diagram_timeline_text, repeat) {
    auto circuit = Circuit(R"CIRCUIT(
        H 0 1 2
        REPEAT 5 {
            RX 2
            REPEAT 100 {
                H 0 1 3 3
            }
        }
    )CIRCUIT");
    ASSERT_EQ("\n" + DiagramTimelineAsciiDrawer::make_diagram(circuit).str() + "\n", R"DIAGRAM(
       /REP 5    /REP 100     \ \
q0: -H-|---------|--------H---|-|-
       |         |            | |
q1: -H-|---------|--------H---|-|-
       |         |            | |
q2: -H-|------RX-|------------|-|-
       |         |            | |
q3: ---|---------|--------H-H-|-|-
       \         \            / /
)DIAGRAM");
}

TEST(circuit_diagram_timeline_text, classical_feedback) {
    auto circuit = Circuit(R"CIRCUIT(
        M 0
        CX rec[-1] 1
        YCZ 2 sweep[5]
    )CIRCUIT");
    ASSERT_EQ("\n" + DiagramTimelineAsciiDrawer::make_diagram(circuit).str() + "\n", R"DIAGRAM(
q0: -M:rec[0]---

q1: -X^rec[0]---

q2: -Y^sweep[5]-
)DIAGRAM");
}

TEST(circuit_diagram_timeline_text, lattice_surgery_cnot) {
    auto circuit = Circuit(R"CIRCUIT(
        R 2
        MPP X1*X2
        MPP Z0*Z2
        MX 2
        CZ rec[-3] 0
        CX rec[-2] 1
        CZ rec[-1] 0
    )CIRCUIT");
    ASSERT_EQ("\n" + DiagramTimelineAsciiDrawer::make_diagram(circuit).str() + "\n", R"DIAGRAM(
q0: -----------------MPP[Z]:rec[1]-Z^rec[0]--Z^rec[2]-
                     |
q1: ---MPP[X]:rec[0]-|-------------X^rec[1]-----------
       |             |
q2: -R-MPP[X]:rec[0]-MPP[Z]:rec[1]-MX:rec[2]----------
)DIAGRAM");

    ASSERT_EQ("\n" + DiagramTimelineAsciiDrawer::make_diagram(circuit).transposed().str() + "\n", R"DIAGRAM(
    q0:           q1:           q2:
      |             |             |
      |             |             R
      |             |             |
      |       MPP[X]:rec[0]-MPP[X]:rec[0]
      |             |             |
MPP[Z]:rec[1]---------------MPP[Z]:rec[1]
      |             |             |
  Z^rec[0]      X^rec[1]      MX:rec[2]
      |             |             |
  Z^rec[2]          |             |
      |             |             |
)DIAGRAM");
}

TEST(circuit_diagram_timeline_text, tick) {
    auto circuit = Circuit(R"CIRCUIT(
        H 0 0
        TICK
        H 0 1
        TICK
        H 0
        REPEAT 1 {
            H 0 1
            TICK
            H 0
            S 0
        }
        H 0 0
        SQRT_X 0
        TICK
        H 0 0
    )CIRCUIT");
    ASSERT_EQ("\n" + DiagramTimelineAsciiDrawer::make_diagram(circuit).str() + "\n", R"DIAGRAM(
     /-\     /REP 1   /-\ \ /--------\ /-\
q0: -H-H-H-H-|------H-H-S-|-H-H-SQRT_X-H-H-
             |            |
q1: -----H---|------H-----|----------------
     \-/     \        \-/ / \--------/ \-/
)DIAGRAM");
}

TEST(circuit_diagram_timeline_text, shifted_coords) {
    auto circuit = Circuit(R"CIRCUIT(
        QUBIT_COORDS(1, 2) 1
        DETECTOR(4, 5, 6)
        SHIFT_COORDS(10, 20, 30, 40)
        QUBIT_COORDS(1, 2) 2
        DETECTOR(4, 5, 6)
        REPEAT 100 {
            QUBIT_COORDS(7, 8) 3 4
            DETECTOR(9, 10, 11)
            SHIFT_COORDS(0, 200, 300, 400)
        }
        QUBIT_COORDS(1, 2) 5
        DETECTOR(4, 5, 6)
    )CIRCUIT");
    ASSERT_EQ("\n" + DiagramTimelineAsciiDrawer::make_diagram(circuit).str() + "\n", R"DIAGRAM(
                                                  /REP 100                                                  \
q0: -DETECTOR(4,5,6):D0=1-DETECTOR(14,25,36):D1=1-|--------DETECTOR(19,30+iter*200,41+iter*300):D[2+iter]=1-|-DETECTOR(14,20025,30036):D102=1-
                                                  |                                                         |
q1: -QUBIT_COORDS(1,2)----------------------------|---------------------------------------------------------|---------------------------------
                                                  |                                                         |
q2: -QUBIT_COORDS(11,22)--------------------------|---------------------------------------------------------|---------------------------------
                                                  |                                                         |
q3: ----------------------------------------------|--------QUBIT_COORDS(17,28+iter*200)---------------------|---------------------------------
                                                  |                                                         |
q4: ----------------------------------------------|--------QUBIT_COORDS(17,28+iter*200)---------------------|---------------------------------
                                                  |                                                         |
q5: ----------------------------------------------|---------------------------------------------------------|-QUBIT_COORDS(11,20022)----------
                                                  \                                                         /
)DIAGRAM");
}

TEST(circuit_diagram_timeline_text, detector_pseudo_targets) {
    auto circuit = Circuit(R"CIRCUIT(
        M 0 1 2 3 4 5
        REPEAT 100 {
            M 1 2
        }
        DETECTOR(1) rec[-1]
        DETECTOR(2) rec[-2]
        DETECTOR(3) rec[-3]
        DETECTOR(4) rec[-4]
        DETECTOR(5) rec[-1] rec[-2]
        OBSERVABLE_INCLUDE(100) rec[-201] rec[-203]
    )CIRCUIT");
    ASSERT_EQ("\n" + DiagramTimelineAsciiDrawer::make_diagram(circuit).str() + "\n", R"DIAGRAM(
              /REP 100                 \
q0: -M:rec[0]-|------------------------|----------------------------------------------------------------------------------------
              |                        |
q1: -M:rec[1]-|--------M:rec[6+iter*2]-|-DETECTOR(2):D1=rec[204]-DETECTOR(4):D3=rec[202]-DETECTOR(5):D4=rec[205]*rec[204]-------
              |                        |
q2: -M:rec[2]-|--------M:rec[7+iter*2]-|-DETECTOR(1):D0=rec[205]-DETECTOR(3):D2=rec[203]----------------------------------------
              |                        |
q3: -M:rec[3]-|------------------------|-------------------------------------------------OBSERVABLE_INCLUDE:L100*=rec[5]*rec[3]-
              |                        |
q4: -M:rec[4]-|------------------------|----------------------------------------------------------------------------------------
              |                        |
q5: -M:rec[5]-|------------------------|----------------------------------------------------------------------------------------
              \                        /
)DIAGRAM");
}

TEST(circuit_diagram_timeline_text, surface_code) {
    CircuitGenParameters params(10, 3, "unrotated_memory_z");
    auto circuit = generate_surface_code_circuit(params).circuit;
    ASSERT_EQ("\n" + DiagramTimelineAsciiDrawer::make_diagram(circuit).str() + "\n", R"DIAGRAM(
      /-----------------\     /-------\ /-------\     /----------------------------------\ /REP 9       /-------\ /-------\     /-----------------------------------------------------------------------------------\ \ /------------------------------------------------------------------------------------------------------------------\
 q0: -QUBIT_COORDS(0,0)-R-----------------@-------X----------------------------------------|------------------------@-------X-----------------------------------------------------------------------------------------|-M:rec[120]---------------------------------------------------------OBSERVABLE_INCLUDE:L0*=rec[122]*rec[121]*rec[120]-
                                          |       |                                        |                        |       |                                                                                         |
 q1: -QUBIT_COORDS(1,0)-R-H-@-@-----------|-------@-H-MR:rec[0]----------------------------|--------H-@-@-----------|-------@-H-MR:rec[12+iter*12]-DETECTOR(1,0,1+iter):D[6+iter*12]=rec[12+iter*12]*rec[0+iter*12]---|----------------------------------------------------------------------------------------------------------------------
                            | |           |                                                |          | |           |                                                                                                 |
 q2: -QUBIT_COORDS(2,0)-R---X-|-----------|-@-----X----------------------------------------|----------X-|-----------|-@-----X-----------------------------------------------------------------------------------------|-M:rec[121]-----------------------------------------------------------------------------------------------------------
                              |           | |     |                                        |            |           | |     |                                                                                         |
 q3: -QUBIT_COORDS(3,0)-R-H-@-|-@---------|-|-----@-H-MR:rec[1]----------------------------|--------H-@-|-@---------|-|-----@-H-MR:rec[13+iter*12]-DETECTOR(3,0,1+iter):D[7+iter*12]=rec[13+iter*12]*rec[1+iter*12]---|----------------------------------------------------------------------------------------------------------------------
                            | | |         | |                                              |          | | |         | |                                                                                               |
 q4: -QUBIT_COORDS(4,0)-R---X-|-|---------|-|-@--------------------------------------------|----------X-|-|---------|-|-@---------------------------------------------------------------------------------------------|-M:rec[122]-----------------------------------------------------------------------------------------------------------
                              | |         | | |                                            |            | |         | | |                                                                                             |
 q5: -QUBIT_COORDS(0,1)-R---X-|-|-X-------X-|-|-------MR:rec[2]--DETECTOR(0,1,0):D0=rec[2]-|----------X-|-|-X-------X-|-|-------MR:rec[14+iter*12]-DETECTOR(0,1,1+iter):D[8+iter*12]=rec[14+iter*12]*rec[2+iter*12]---|-DETECTOR(0,1,10):D114=rec[125]*rec[123]*rec[120]*rec[110]------------------------------------------------------------
                            | | | |         | |                                            |          | | | |         | |                                                                                             |
 q6: -QUBIT_COORDS(1,1)-R---@-X-|-|-----X---|-|---@----------------------------------------|----------@-X-|-|-----X---|-|---@-----------------------------------------------------------------------------------------|-M:rec[123]-----------------------------------------------------------------------------------------------------------
                                | |     |   | |   |                                        |              | |     |   | |   |                                                                                         |
 q7: -QUBIT_COORDS(2,1)-R---X---|-|-X---|---X-|---X---MR:rec[3]--DETECTOR(2,1,0):D2=rec[3]-|----------X---|-|-X---|---X-|---X---MR:rec[15+iter*12]-DETECTOR(2,1,1+iter):D[9+iter*12]=rec[15+iter*12]*rec[3+iter*12]---|-DETECTOR(2,1,10):D116=rec[126]*rec[124]*rec[123]*rec[121]*rec[111]---------------------------------------------------
                            |   | | |   |     |                                            |          |   | | |   |     |                                                                                             |
 q8: -QUBIT_COORDS(3,1)-R---@---X-|-|---|-X---|---@----------------------------------------|----------@---X-|-|---|-X---|---@-----------------------------------------------------------------------------------------|-M:rec[124]-----------------------------------------------------------------------------------------------------------
                                  | |   | |   |   |                                        |                | |   | |   |   |                                                                                         |
 q9: -QUBIT_COORDS(4,1)-R---------|-|-X-|-|---X---X---MR:rec[4]--DETECTOR(4,1,0):D4=rec[4]-|----------------|-|-X-|-|---X---X---MR:rec[16+iter*12]-DETECTOR(4,1,1+iter):D[10+iter*12]=rec[16+iter*12]*rec[4+iter*12]--|-DETECTOR(4,1,10):D118=rec[127]*rec[124]*rec[122]*rec[112]------------------------------------------------------------
                                  | | | | |                                                |                | | | | |                                                                                                 |
q10: -QUBIT_COORDS(0,2)-R---------@-|-|-|-|-@-----X----------------------------------------|----------------@-|-|-|-|-@-----X-----------------------------------------------------------------------------------------|-M:rec[125]-----------------------------------------------------------------------------------------------------------
                                    | | | | |     |                                        |                  | | | | |     |                                                                                         |
q11: -QUBIT_COORDS(1,2)-R-H-@-@-----|-|-@-|-|-----@-H-MR:rec[5]----------------------------|--------H-@-@-----|-|-@-|-|-----@-H-MR:rec[17+iter*12]-DETECTOR(1,2,1+iter):D[11+iter*12]=rec[17+iter*12]*rec[5+iter*12]--|----------------------------------------------------------------------------------------------------------------------
                            | |     | |   | |                                              |          | |     | |   | |                                                                                               |
q12: -QUBIT_COORDS(2,2)-R---X-|-----@-|---|-|-@---X----------------------------------------|----------X-|-----@-|---|-|-@---X-----------------------------------------------------------------------------------------|-M:rec[126]-----------------------------------------------------------------------------------------------------------
                              |       |   | | |   |                                        |            |       |   | | |   |                                                                                         |
q13: -QUBIT_COORDS(3,2)-R-H-@-|-@-----|---@-|-|---@-H-MR:rec[6]----------------------------|--------H-@-|-@-----|---@-|-|---@-H-MR:rec[18+iter*12]-DETECTOR(3,2,1+iter):D[12+iter*12]=rec[18+iter*12]*rec[6+iter*12]--|----------------------------------------------------------------------------------------------------------------------
                            | | |     |     | |                                            |          | | |     |     | |                                                                                             |
q14: -QUBIT_COORDS(4,2)-R---X-|-|-----@-----|-|-@------------------------------------------|----------X-|-|-----@-----|-|-@-------------------------------------------------------------------------------------------|-M:rec[127]-----------------------------------------------------------------------------------------------------------
                              | |           | | |                                          |            | |           | | |                                                                                           |
q15: -QUBIT_COORDS(0,3)-R---X-|-|-X---------X-|-|-----MR:rec[7]--DETECTOR(0,3,0):D1=rec[7]-|----------X-|-|-X---------X-|-|-----MR:rec[19+iter*12]-DETECTOR(0,3,1+iter):D[13+iter*12]=rec[19+iter*12]*rec[7+iter*12]--|-DETECTOR(0,3,10):D115=rec[130]*rec[128]*rec[125]*rec[115]------------------------------------------------------------
                            | | | |           | |                                          |          | | | |           | |                                                                                           |
q16: -QUBIT_COORDS(1,3)-R---@-X-|-|-----X-----|-|-@----------------------------------------|----------@-X-|-|-----X-----|-|-@-----------------------------------------------------------------------------------------|-M:rec[128]-----------------------------------------------------------------------------------------------------------
                                | |     |     | | |                                        |              | |     |     | | |                                                                                         |
q17: -QUBIT_COORDS(2,3)-R---X---|-|-X---|-----X-|-X---MR:rec[8]--DETECTOR(2,3,0):D3=rec[8]-|----------X---|-|-X---|-----X-|-X---MR:rec[20+iter*12]-DETECTOR(2,3,1+iter):D[14+iter*12]=rec[20+iter*12]*rec[8+iter*12]--|-DETECTOR(2,3,10):D117=rec[131]*rec[129]*rec[128]*rec[126]*rec[116]---------------------------------------------------
                            |   | | |   |       |                                          |          |   | | |   |       |                                                                                           |
q18: -QUBIT_COORDS(3,3)-R---@---X-|-|---|-X-----|-@----------------------------------------|----------@---X-|-|---|-X-----|-@-----------------------------------------------------------------------------------------|-M:rec[129]-----------------------------------------------------------------------------------------------------------
                                  | |   | |     | |                                        |                | |   | |     | |                                                                                         |
q19: -QUBIT_COORDS(4,3)-R---------|-|-X-|-|-----X-X---MR:rec[9]--DETECTOR(4,3,0):D5=rec[9]-|----------------|-|-X-|-|-----X-X---MR:rec[21+iter*12]-DETECTOR(4,3,1+iter):D[15+iter*12]=rec[21+iter*12]*rec[9+iter*12]--|-DETECTOR(4,3,10):D119=rec[132]*rec[129]*rec[127]*rec[117]------------------------------------------------------------
                                  | | | | |                                                |                | | | | |                                                                                                 |
q20: -QUBIT_COORDS(0,4)-R---------@-|-|-|-|-------X----------------------------------------|----------------@-|-|-|-|-------X-----------------------------------------------------------------------------------------|-M:rec[130]-----------------------------------------------------------------------------------------------------------
                                    | | | |       |                                        |                  | | | |       |                                                                                         |
q21: -QUBIT_COORDS(1,4)-R-H-@-------|-|-@-|-------@-H-MR:rec[10]---------------------------|--------H-@-------|-|-@-|-------@-H-MR:rec[22+iter*12]-DETECTOR(1,4,1+iter):D[16+iter*12]=rec[22+iter*12]*rec[10+iter*12]-|----------------------------------------------------------------------------------------------------------------------
                            |       | |   |                                                |          |       | |   |                                                                                                 |
q22: -QUBIT_COORDS(2,4)-R---X-------@-|---|-------X----------------------------------------|----------X-------@-|---|-------X-----------------------------------------------------------------------------------------|-M:rec[131]-----------------------------------------------------------------------------------------------------------
                                      |   |       |                                        |                    |   |       |                                                                                         |
q23: -QUBIT_COORDS(3,4)-R-H-@---------|---@-------@-H-MR:rec[11]---------------------------|--------H-@---------|---@-------@-H-MR:rec[23+iter*12]-DETECTOR(3,4,1+iter):D[17+iter*12]=rec[23+iter*12]*rec[11+iter*12]-|----------------------------------------------------------------------------------------------------------------------
                            |         |                                                    |          |         |                                                                                                     |
q24: -QUBIT_COORDS(4,4)-R---X---------@----------------------------------------------------|----------X---------@-----------------------------------------------------------------------------------------------------|-M:rec[132]-----------------------------------------------------------------------------------------------------------
      \-----------------/     \-------/ \-------/     \----------------------------------/ \            \-------/ \-------/     \-----------------------------------------------------------------------------------/ / \------------------------------------------------------------------------------------------------------------------/
)DIAGRAM");
}

TEST(circuit_diagram_timeline_text, repetition_code) {
    CircuitGenParameters params(10, 9, "memory");
    auto circuit = generate_rep_code_circuit(params).circuit;
    ASSERT_EQ("\n" + DiagramTimelineAsciiDrawer::make_diagram(circuit).str() + "\n", R"DIAGRAM(
            /--------------------------------\ /REP 9       /-----------------------------------------------------------------------------\ \ /---------------------------------------------------\
 q0: -R-@--------------------------------------|--------@-----------------------------------------------------------------------------------|-M:rec[80]-DETECTOR(1,10):D80=rec[81]*rec[80]*rec[72]--
        |                                      |        |                                                                                   |
 q1: -R-X-X-MR:rec[0]-DETECTOR(1,0):D0=rec[0]--|--------X-X-MR:rec[8+iter*8]--DETECTOR(1,1+iter):D[8+iter*8]=rec[8+iter*8]*rec[0+iter*8]----|-------------------------------------------------------
          |                                    |          |                                                                                 |
 q2: -R-@-@------------------------------------|--------@-@---------------------------------------------------------------------------------|-M:rec[81]-DETECTOR(3,10):D81=rec[82]*rec[81]*rec[73]--
        |                                      |        |                                                                                   |
 q3: -R-X-X-MR:rec[1]-DETECTOR(3,0):D1=rec[1]--|--------X-X-MR:rec[9+iter*8]--DETECTOR(3,1+iter):D[9+iter*8]=rec[9+iter*8]*rec[1+iter*8]----|-------------------------------------------------------
          |                                    |          |                                                                                 |
 q4: -R-@-@------------------------------------|--------@-@---------------------------------------------------------------------------------|-M:rec[82]-DETECTOR(5,10):D82=rec[83]*rec[82]*rec[74]--
        |                                      |        |                                                                                   |
 q5: -R-X-X-MR:rec[2]-DETECTOR(5,0):D2=rec[2]--|--------X-X-MR:rec[10+iter*8]-DETECTOR(5,1+iter):D[10+iter*8]=rec[10+iter*8]*rec[2+iter*8]--|-------------------------------------------------------
          |                                    |          |                                                                                 |
 q6: -R-@-@------------------------------------|--------@-@---------------------------------------------------------------------------------|-M:rec[83]-DETECTOR(7,10):D83=rec[84]*rec[83]*rec[75]--
        |                                      |        |                                                                                   |
 q7: -R-X-X-MR:rec[3]-DETECTOR(7,0):D3=rec[3]--|--------X-X-MR:rec[11+iter*8]-DETECTOR(7,1+iter):D[11+iter*8]=rec[11+iter*8]*rec[3+iter*8]--|-------------------------------------------------------
          |                                    |          |                                                                                 |
 q8: -R-@-@------------------------------------|--------@-@---------------------------------------------------------------------------------|-M:rec[84]-DETECTOR(9,10):D84=rec[85]*rec[84]*rec[76]--
        |                                      |        |                                                                                   |
 q9: -R-X-X-MR:rec[4]-DETECTOR(9,0):D4=rec[4]--|--------X-X-MR:rec[12+iter*8]-DETECTOR(9,1+iter):D[12+iter*8]=rec[12+iter*8]*rec[4+iter*8]--|-------------------------------------------------------
          |                                    |          |                                                                                 |
q10: -R-@-@------------------------------------|--------@-@---------------------------------------------------------------------------------|-M:rec[85]-DETECTOR(11,10):D85=rec[86]*rec[85]*rec[77]-
        |                                      |        |                                                                                   |
q11: -R-X-X-MR:rec[5]-DETECTOR(11,0):D5=rec[5]-|--------X-X-MR:rec[13+iter*8]-DETECTOR(11,1+iter):D[13+iter*8]=rec[13+iter*8]*rec[5+iter*8]-|-------------------------------------------------------
          |                                    |          |                                                                                 |
q12: -R-@-@------------------------------------|--------@-@---------------------------------------------------------------------------------|-M:rec[86]-DETECTOR(13,10):D86=rec[87]*rec[86]*rec[78]-
        |                                      |        |                                                                                   |
q13: -R-X-X-MR:rec[6]-DETECTOR(13,0):D6=rec[6]-|--------X-X-MR:rec[14+iter*8]-DETECTOR(13,1+iter):D[14+iter*8]=rec[14+iter*8]*rec[6+iter*8]-|-------------------------------------------------------
          |                                    |          |                                                                                 |
q14: -R-@-@------------------------------------|--------@-@---------------------------------------------------------------------------------|-M:rec[87]-DETECTOR(15,10):D87=rec[88]*rec[87]*rec[79]-
        |                                      |        |                                                                                   |
q15: -R-X-X-MR:rec[7]-DETECTOR(15,0):D7=rec[7]-|--------X-X-MR:rec[15+iter*8]-DETECTOR(15,1+iter):D[15+iter*8]=rec[15+iter*8]*rec[7+iter*8]-|-------------------------------------------------------
          |                                    |          |                                                                                 |
q16: -R---@------------------------------------|----------@---------------------------------------------------------------------------------|-M:rec[88]-OBSERVABLE_INCLUDE:L0*=rec[88]--------------
            \--------------------------------/ \            \-----------------------------------------------------------------------------/ / \---------------------------------------------------/
)DIAGRAM");
}

TEST(circuit_diagram_timeline_text, repetition_code_transposed) {
    CircuitGenParameters params(10, 3, "memory");
    auto circuit = generate_rep_code_circuit(params).circuit;
    ASSERT_EQ("\n" + DiagramTimelineAsciiDrawer::make_diagram(circuit).transposed().str() + "\n", R"DIAGRAM(
                         q0:                                                q1:                                                q2:                                                q3:                                          q4:
                          |                                                  |                                                  |                                                  |                                            |
                          R                                                  R                                                  R                                                  R                                            R
                          |                                                  |                                                  |                                                  |                                            |
                          @--------------------------------------------------X                                                  @--------------------------------------------------X                                            |
                          |                                                  |                                                  |                                                  |                                            |
                          |                                                  X--------------------------------------------------@                                                  X--------------------------------------------@
                          |                                                  |                                                  |                                                  |                                            |
/                         |                                              MR:rec[0]                                              |                                              MR:rec[1]                                        |               \
|                         |                                                  |                                                  |                                                  |                                            |               |
\                         |                                       DETECTOR(1,0):D0=rec[0]                                       |                                       DETECTOR(3,0):D1=rec[1]                                 |               /
                          |                                                  |                                                  |                                                  |                                            |
/REP 9------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------\
                          |                                                  |                                                  |                                                  |                                            |
                          |                                                  |                                                  |                                                  |                                            |
                          |                                                  |                                                  |                                                  |                                            |
                          @--------------------------------------------------X                                                  @--------------------------------------------------X                                            |
                          |                                                  |                                                  |                                                  |                                            |
                          |                                                  X--------------------------------------------------@                                                  X--------------------------------------------@
                          |                                                  |                                                  |                                                  |                                            |
/                         |                                           MR:rec[2+iter*2]                                          |                                           MR:rec[3+iter*2]                                    |               \
|                         |                                                  |                                                  |                                                  |                                            |               |
\                         |                      DETECTOR(1,1+iter):D[2+iter*2]=rec[2+iter*2]*rec[0+iter*2]                     |                      DETECTOR(3,1+iter):D[3+iter*2]=rec[3+iter*2]*rec[1+iter*2]               |               /
                          |                                                  |                                                  |                                                  |                                            |
\-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------/
                          |                                                  |                                                  |                                                  |                                            |
/                     M:rec[20]                                              |                                              M:rec[21]                                              |                                        M:rec[22]           \
|                         |                                                  |                                                  |                                                  |                                            |               |
\     DETECTOR(1,10):D20=rec[21]*rec[20]*rec[18]                             |                              DETECTOR(3,10):D21=rec[22]*rec[21]*rec[19]                             |                              OBSERVABLE_INCLUDE:L0*=rec[22]/
                          |                                                  |                                                  |                                                  |                                            |
)DIAGRAM");
}

TEST(circuit_diagram_timeline_text, test_circuit_all_ops) {
    auto circuit = generate_test_circuit_with_all_operations();
    ASSERT_EQ("\n" + DiagramTimelineAsciiDrawer::make_diagram(circuit).str() + "\n", R"DIAGRAM(
      /-------------------\ /--------------\ /---------------------\ /-----------------------------------------------------------------------------------------------------------------------------------------------------\ /------------------------------------------------------\                        /REP 3 /---\ \   /--------------------------------------------------------------------------------------------------------------------------\
 q0: -QUBIT_COORDS(1,2,3)-I-C_XYZ-SQRT_X-----ZSWAP-----SQRT_XX-----X------------DEPOLARIZE1(0.02)---------------X_ERROR(0.01)------------------------------------------------------------------------------------------------MPP[X]:rec[2]-MPP[Z]:rec[3]-SPP[X]-SPP_DAG[X]------------MRX:rec[4]-MXX:rec[11]-|------H-@---|---MR:rec[15]-X_ERROR(0.1)-MR(0.01):rec[16]-DETECTOR(2,4,6):D0=rec[16]-OBSERVABLE_INCLUDE:L0*=rec[16]-MPAD:rec[17]-MPAD:rec[19]-MRX:rec[20]----X^rec[24]--
                                             |         |           |                                                                                                                                                         |             |             |      |                                |           |        |   |
 q1: ---------------------X-C_ZYX-SQRT_X_DAG-XSWAP-----SQRT_XX-----X-E[X](0.01)-DEPOLARIZE2(0.03)---------------Y_ERROR(0.02)------------------------------------------------------------------------------------------------MPP[Y]:rec[2]-MPP[Z]:rec[3]-SPP[Y]-SPP_DAG[Y]------------MRY:rec[5]-MXX---------|--------X-S-|------------------------------------------------------------------------------------------------------MPAD:rec[18]--------------MY:rec[21]-----Y^sweep[0]-
                                                                     |          |                                                                                                                                            |                           |      |                                            |            |
 q2: ---------------------Y-H_XY--SQRT_Y-----ISWAP-----SQRT_XX_DAG-X-E[Y](0.01)-DEPOLARIZE2(0.03)---------------Z_ERROR(0.03)------------------------------------------------------------------------------------------------MPP[Z]:rec[2]---------------SPP[Z]-SPP_DAG[Z]-SPP_DAG[X]-MR:rec[6]--MXX:rec[12]-|------------|--------------------------------------------------------------------------------------------------------------------------------MZZ:rec[22]----Z^rec[24]--
                                             |         |           | |                                                                                                                                                                                                                           |           |            |                                                                                                                                |
 q3: ---------------------Z-H-----SQRT_Y_DAG-ISWAP-----SQRT_XX_DAG-Y-E[Z](0.01)-PAULI_CHANNEL_1(0.01,0.02,0.03)-HERALDED_ERASE(0.04):rec[0]--------------------------------------------------------------------------------------------------------------SPP[X]-----------------------MX:rec[7]--MXX---------|------------|--------------------------------------------------------------------------------------------------------------------------------MZZ-----------------------
                                                                                                                                                                                                                                                                                                             |            |
 q4: -----------------------H_YZ--S----------ISWAP_DAG-SQRT_YY-----X------------ELSE_CORRELATED_ERROR[X](0.02)--PAULI_CHANNEL_2[0](0.001,0.002,0.003,0.004,0.005,0.006,0.007,0.008,0.009,0.01,0.011,0.012,0.013,0.014,0.015)----------------------------------------------------------MY:rec[8]--MYY:rec[13]-|------------|--------------------------------------------------------------------------------------------------------------------------------MYY:rec[23]---------------
                                             |         |           |            |                               |                                                                                                                                                                                |           |            |                                                                                                                                |
 q5: -----------------------------S_DAG------ISWAP_DAG-SQRT_YY-----@------------|-------------------------------PAULI_CHANNEL_2[1](0.001,0.002,0.003,0.004,0.005,0.006,0.007,0.008,0.009,0.01,0.011,0.012,0.013,0.014,0.015)----------------------------------------------------------M:rec[9]---MYY---------|------------|--------------------------------------------------------------------------------------------------------------------------------MYY-----------------------
                                                                                |                                                                                                                                                                                                                            |            |
 q6: ----------------------------------------SWAP------SQRT_YY_DAG-Y------------ELSE_CORRELATED_ERROR[Z](0.02)--HERALDED_PAULI_CHANNEL_1(0.01,0.02,0.03,0.04):rec[1]------------------------------------------------------------------------------------------------------------------M:rec[10]--MZZ:rec[14]-|------------|--------------------------------------------------------------------------------------------------------------------------------MPP[X]:rec[24]------------
                                             |         |           |            |                                                                                                                                                                                                                |           |            |                                                                                                                                |
 q7: ----------------------------------------SWAP------SQRT_YY_DAG-X------------ELSE_CORRELATED_ERROR[Y](0.02)------------------------------------------------------------------------------------------------------------------------------------------------------------------------RX---------MZZ---------|------------|--------------------------------------------------------------------------------------------------------------------------------MPP[Y]:rec[24]------------
                                                                                                                                                                                                                                                                                                             |            |                                                                                                                                |
 q8: ----------------------------------------XSWAP-----SQRT_ZZ-----Y------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------RY---------------------|------------|--------------------------------------------------------------------------------------------------------------------------------MPP[Z]:rec[24]------------
                                             |         |           |                                                                                                                                                                                                                                         |            |
 q9: ----------------------------------------ZSWAP-----SQRT_ZZ-----Y------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------R----------------------|------------|----------------------------------------------------------------------------------------------------------------------------------------------------------
                                                                                                                                                                                                                                                                                                             |            |
q10: ----------------------------------------ZSWAP-----SQRT_ZZ_DAG-Y-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|------------|----------------------------------------------------------------------------------------------------------------------------------------------------------
                                             |         |           |                                                                                                                                                                                                                                         |            |
q11: ----------------------------------------ZSWAP-----SQRT_ZZ_DAG-@-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|------------|----------------------------------------------------------------------------------------------------------------------------------------------------------
                                                                                                                                                                                                                                                                                                             |            |
q12: --------------------------------------------------------------@-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|------------|----------------------------------------------------------------------------------------------------------------------------------------------------------
                                                                   |                                                                                                                                                                                                                                         |            |
q13: --------------------------------------------------------------X-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|------------|----------------------------------------------------------------------------------------------------------------------------------------------------------
                                                                                                                                                                                                                                                                                                             |            |
q14: --------------------------------------------------------------@-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|------------|----------------------------------------------------------------------------------------------------------------------------------------------------------
                                                                   |                                                                                                                                                                                                                                         |            |
q15: --------------------------------------------------------------Y-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|------------|----------------------------------------------------------------------------------------------------------------------------------------------------------
                                                                                                                                                                                                                                                                                                             |            |
q16: --------------------------------------------------------------@-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|------------|----------------------------------------------------------------------------------------------------------------------------------------------------------
                                                                   |                                                                                                                                                                                                                                         |            |
q17: --------------------------------------------------------------@-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|------------|----------------------------------------------------------------------------------------------------------------------------------------------------------
      \-------------------/ \--------------/ \---------------------/ \-----------------------------------------------------------------------------------------------------------------------------------------------------/ \------------------------------------------------------/                        \      \---/ /   \--------------------------------------------------------------------------------------------------------------------------/
)DIAGRAM");
}
