"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AlarmFactory = exports.CompositeAlarmOperator = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const strings_1 = require("../strings");
const AlarmNamingStrategy_1 = require("./AlarmNamingStrategy");
const IAlarmAnnotationStrategy_1 = require("./IAlarmAnnotationStrategy");
const NoopAlarmActionStrategy_1 = require("./NoopAlarmActionStrategy");
const DefaultDatapointsToAlarm = 3;
var CompositeAlarmOperator;
(function (CompositeAlarmOperator) {
    /**
     * trigger only if all the alarms are triggered
     */
    CompositeAlarmOperator[CompositeAlarmOperator["AND"] = 0] = "AND";
    /**
     * trigger if any of the alarms is triggered
     */
    CompositeAlarmOperator[CompositeAlarmOperator["OR"] = 1] = "OR";
})(CompositeAlarmOperator = exports.CompositeAlarmOperator || (exports.CompositeAlarmOperator = {}));
class AlarmFactory {
    constructor(alarmScope, props) {
        this.alarmScope = alarmScope;
        this.globalAlarmDefaults = props.globalAlarmDefaults;
        this.globalMetricDefaults = props.globalMetricDefaults;
        this.alarmNamingStrategy = new AlarmNamingStrategy_1.AlarmNamingStrategy(props.globalAlarmDefaults.alarmNamePrefix, props.localAlarmNamePrefix, props.globalAlarmDefaults.dedupeStringProcessor);
    }
    addAlarm(metric, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
        let adjustedMetric = metric;
        if (props.period) {
            // Adjust metric period for the alarm
            adjustedMetric = adjustedMetric.with({ period: props.period });
        }
        if (adjustedMetric.label) {
            // Annotations do not support dynamic labels, so we have to remove them from metric name
            adjustedMetric = adjustedMetric.with({
                label: strings_1.removeBracketsWithDynamicLabels(adjustedMetric.label),
            });
        }
        const actionsEnabled = this.determineActionsEnabled(props.actionsEnabled, props.disambiguator);
        const action = (_c = (_b = props.actionOverride) !== null && _b !== void 0 ? _b : this.globalAlarmDefaults.action) !== null && _c !== void 0 ? _c : NoopAlarmActionStrategy_1.noopAction();
        const alarmName = this.alarmNamingStrategy.getName(props);
        const alarmNameSuffix = props.alarmNameSuffix;
        const alarmLabel = this.alarmNamingStrategy.getWidgetLabel(props);
        const alarmDescription = this.generateDescription(props.alarmDescription, props.alarmDescriptionOverride, props.runbookLink, props.documentationLink);
        const dedupeString = this.alarmNamingStrategy.getDedupeString(props);
        const evaluateLowSampleCountPercentile = (_d = props.evaluateLowSampleCountPercentile) !== null && _d !== void 0 ? _d : true;
        const datapointsToAlarm = (_f = (_e = props.datapointsToAlarm) !== null && _e !== void 0 ? _e : this.globalAlarmDefaults.datapointsToAlarm) !== null && _f !== void 0 ? _f : DefaultDatapointsToAlarm;
        const evaluationPeriods = (_h = (_g = props.evaluationPeriods) !== null && _g !== void 0 ? _g : this.globalAlarmDefaults.evaluationPeriods) !== null && _h !== void 0 ? _h : datapointsToAlarm;
        if (evaluationPeriods < datapointsToAlarm) {
            throw new Error(`evaluationPeriods must be greater than or equal to datapointsToAlarm for ${alarmName}`);
        }
        const alarm = adjustedMetric.createAlarm(this.alarmScope, alarmName, {
            alarmName,
            alarmDescription,
            threshold: props.threshold,
            comparisonOperator: props.comparisonOperator,
            treatMissingData: props.treatMissingData,
            // default value (undefined) means "evaluate"
            evaluateLowSampleCountPercentile: evaluateLowSampleCountPercentile
                ? undefined
                : "ignore",
            datapointsToAlarm,
            evaluationPeriods,
            actionsEnabled,
        });
        action.addAlarmActions({
            alarm,
            action,
            dedupeString,
            disambiguator: props.disambiguator,
            customTags: (_j = props.customTags) !== null && _j !== void 0 ? _j : [],
            customParams: (_k = props.customParams) !== null && _k !== void 0 ? _k : {},
        });
        const annotation = this.createAnnotation({
            alarm,
            action,
            metric: adjustedMetric,
            evaluationPeriods,
            datapointsToAlarm,
            dedupeString,
            fillAlarmRange: (_l = props.fillAlarmRange) !== null && _l !== void 0 ? _l : false,
            comparisonOperator: props.comparisonOperator,
            threshold: props.threshold,
            disambiguator: props.disambiguator,
            customTags: (_m = props.customTags) !== null && _m !== void 0 ? _m : [],
            customParams: (_o = props.customParams) !== null && _o !== void 0 ? _o : {},
        });
        return {
            alarm,
            action,
            alarmName,
            alarmNameSuffix,
            alarmLabel,
            alarmDescription,
            customTags: props.customTags,
            customParams: props.customParams,
            alarmRuleWhenOk: aws_cloudwatch_1.AlarmRule.fromAlarm(alarm, aws_cloudwatch_1.AlarmState.OK),
            alarmRuleWhenAlarming: aws_cloudwatch_1.AlarmRule.fromAlarm(alarm, aws_cloudwatch_1.AlarmState.ALARM),
            alarmRuleWhenInsufficientData: aws_cloudwatch_1.AlarmRule.fromAlarm(alarm, aws_cloudwatch_1.AlarmState.INSUFFICIENT_DATA),
            dedupeString,
            annotation,
            disambiguator: props.disambiguator,
        };
    }
    addCompositeAlarm(alarms, props) {
        var _b, _c, _d;
        const actionsEnabled = this.determineActionsEnabled(props === null || props === void 0 ? void 0 : props.actionsEnabled, props === null || props === void 0 ? void 0 : props.disambiguator);
        const action = (_c = (_b = props.actionOverride) !== null && _b !== void 0 ? _b : this.globalAlarmDefaults.action) !== null && _c !== void 0 ? _c : NoopAlarmActionStrategy_1.noopAction();
        const namingInput = { alarmNameSuffix: "Composite", ...props };
        const alarmName = this.alarmNamingStrategy.getName(namingInput);
        const alarmDescription = this.generateDescription((_d = props === null || props === void 0 ? void 0 : props.alarmDescription) !== null && _d !== void 0 ? _d : "Composite alarm", props === null || props === void 0 ? void 0 : props.alarmDescriptionOverride, props === null || props === void 0 ? void 0 : props.runbookLink, props === null || props === void 0 ? void 0 : props.documentationLink);
        const dedupeString = this.alarmNamingStrategy.getDedupeString(namingInput);
        const alarmRule = this.determineCompositeAlarmRule(alarms, props);
        const alarm = new aws_cloudwatch_1.CompositeAlarm(this.alarmScope, alarmName, {
            compositeAlarmName: alarmName,
            alarmDescription,
            alarmRule,
            actionsEnabled,
        });
        action.addAlarmActions({
            alarm,
            action,
            dedupeString,
            disambiguator: props === null || props === void 0 ? void 0 : props.disambiguator,
            customTags: props === null || props === void 0 ? void 0 : props.customTags,
            customParams: props === null || props === void 0 ? void 0 : props.customParams,
        });
        return alarm;
    }
    determineCompositeAlarmRule(alarms, props) {
        var _b;
        const alarmRules = alarms.map((alarm) => alarm.alarmRuleWhenAlarming);
        const operator = (_b = props.compositeOperator) !== null && _b !== void 0 ? _b : CompositeAlarmOperator.OR;
        switch (operator) {
            case CompositeAlarmOperator.AND:
                return aws_cloudwatch_1.AlarmRule.allOf(...alarmRules);
            case CompositeAlarmOperator.OR:
                return aws_cloudwatch_1.AlarmRule.anyOf(...alarmRules);
            default:
                throw new Error("Unsupported composite alarm operator: " + operator);
        }
    }
    determineActionsEnabled(actionsEnabled, disambiguator) {
        var _b;
        if (actionsEnabled !== undefined) {
            // alarm-specific override to true or false
            return actionsEnabled;
        }
        if (typeof this.globalAlarmDefaults.actionsEnabled === "boolean") {
            // global defaults: boolean value
            return this.globalAlarmDefaults.actionsEnabled;
        }
        if (disambiguator) {
            // global defaults: disambiguator based value
            return (_b = this.globalAlarmDefaults.actionsEnabled[disambiguator]) !== null && _b !== void 0 ? _b : false;
        }
        return false;
    }
    get shouldUseDefaultDedupeForError() {
        var _b;
        return (_b = this.globalAlarmDefaults.useDefaultDedupeForError) !== null && _b !== void 0 ? _b : true;
    }
    get shouldUseDefaultDedupeForLatency() {
        var _b;
        return (_b = this.globalAlarmDefaults.useDefaultDedupeForLatency) !== null && _b !== void 0 ? _b : true;
    }
    generateDescription(alarmDescription, alarmDescriptionOverride, runbookLinkOverride, documentationLinkOverride) {
        const parts = [alarmDescriptionOverride !== null && alarmDescriptionOverride !== void 0 ? alarmDescriptionOverride : alarmDescription];
        const runbookLink = runbookLinkOverride !== null && runbookLinkOverride !== void 0 ? runbookLinkOverride : this.globalAlarmDefaults.runbookLink;
        const documentationLink = documentationLinkOverride !== null && documentationLinkOverride !== void 0 ? documentationLinkOverride : this.globalAlarmDefaults.documentationLink;
        if (runbookLink) {
            parts.push(`Runbook: ${runbookLink}`);
        }
        if (documentationLink) {
            parts.push(`Documentation: ${documentationLink}`);
        }
        return parts.join(" \r\n");
    }
    createAnnotation(props) {
        var _b;
        const strategy = (_b = this.globalAlarmDefaults.annotationStrategy) !== null && _b !== void 0 ? _b : new IAlarmAnnotationStrategy_1.DefaultAlarmAnnotationStrategy();
        return strategy.createAnnotation(props);
    }
}
exports.AlarmFactory = AlarmFactory;
_a = JSII_RTTI_SYMBOL_1;
AlarmFactory[_a] = { fqn: "cdk-monitoring-constructs.AlarmFactory", version: "1.6.2" };
//# sourceMappingURL=data:application/json;base64,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