"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsConsoleUrlFactory = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const monitoring_1 = require("../../monitoring");
class AwsConsoleUrlFactory {
    constructor(props) {
        this.awsAccountId = props.awsAccountId;
        this.awsAccountRegion = props.awsAccountRegion;
    }
    getAwsConsoleUrl(destinationUrl) {
        if (this.awsAccountId) {
            return aws_cdk_lib_1.Lazy.uncachedString({
                produce: (context) => {
                    if (destinationUrl) {
                        return this.getResolvedDestinationUrl(context, destinationUrl);
                    }
                    // simply go to the home page of the AWS console
                    return `https://${this.awsAccountRegion}.console.aws.amazon.com`;
                },
            });
        }
        return undefined;
    }
    getCloudWatchLogGroupUrl(logGroupName) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/cloudwatch/home?region=${region}#logsV2:log-groups/log-group/${logGroupName}`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getCodeBuildProjectUrl(projectName) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/codebuild/home?region=${region}#/projects/${projectName}`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getSnsTopicUrl(topicArn) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/sns/v3/home?region=${region}#/topic/${topicArn}`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getSqsQueueUrl(queueUrl) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/sqs/v2/home?region=${region}#/queues/${queueUrl}`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getLambdaFunctionUrl(functionName) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/lambda/home?region=${region}#/functions/${functionName}`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getDynamoTableUrl(tableName) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/dynamodb/home?region=${region}#tables:selected=${tableName}`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getStateMachineUrl(stateMachineArn) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/states/home?region=${region}#/statemachines/view/${stateMachineArn}`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getKinesisDataStreamUrl(streamName) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/kinesis/home?region=${region}#/streams/details/${streamName}/monitoring`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getKinesisFirehoseDeliveryStreamUrl(streamName) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/firehose/home?region=${region}#/details/${streamName}/monitoring`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getKinesisAnalyticsUrl(application) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/kinesisanalytics/home?region=${region}#/details?applicationName=${application}`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getS3BucketUrl(bucketName) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://s3.console.aws.amazon.com/s3/buckets/${bucketName}?region=${region}&tab=metrics`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getApiGatewayUrl(restApiId) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/apigateway/home?region=${region}#/apis/${restApiId}/dashboard`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getRdsClusterUrl(clusterId) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/rds/home?region=${region}#database:id=${clusterId};is-cluster=true;tab=monitoring`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getRedshiftClusterUrl(clusterId) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/redshiftv2/home?region=${region}#cluster-details?cluster=${clusterId}`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getOpenSearchClusterUrl(domainName) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/es/home?region=${region}#domain:resource=${domainName};action=dashboard;tab=TAB_CLUSTER_HEALTH_ID_V2`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getCloudFrontDistributionUrl(distributionId) {
        const destinationUrl = `https://console.aws.amazon.com/cloudfront/v2/home#/monitoring/${distributionId}`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getElastiCacheClusterUrl(clusterId, clusterType) {
        const region = this.awsAccountRegion;
        switch (clusterType) {
            case monitoring_1.ElastiCacheClusterType.MEMCACHED:
                const destinationUrlMemcached = `https://${region}.console.aws.amazon.com/elasticache/home?region=${region}#memcached-nodes:id=${clusterId};nodes`;
                return this.getAwsConsoleUrl(destinationUrlMemcached);
            case monitoring_1.ElastiCacheClusterType.REDIS:
                const destinationUrlRedis = `https://${region}.console.aws.amazon.com/elasticache/home?region=${region}#redis-shards:redis-id=${clusterId}`;
                return this.getAwsConsoleUrl(destinationUrlRedis);
            default:
                throw new Error(`Invalid cache type: ${clusterType}`);
        }
    }
    /**
     * Resolves a destination URL within a resolution context.
     * @param context The resolution context.
     * @param destinationUrl The destination URL to resolve since it may contain CDK tokens.
     * @see https://docs.aws.amazon.com/cdk/latest/guide/tokens.html
     */
    getResolvedDestinationUrl(context, destinationUrl) {
        let resolvedDestination = context.resolve(destinationUrl);
        if (typeof resolvedDestination !== "string") {
            // keep unresolved since we do not know how to deal with objects
            resolvedDestination = destinationUrl;
        }
        return resolvedDestination;
    }
}
exports.AwsConsoleUrlFactory = AwsConsoleUrlFactory;
_a = JSII_RTTI_SYMBOL_1;
AwsConsoleUrlFactory[_a] = { fqn: "cdk-monitoring-constructs.AwsConsoleUrlFactory", version: "1.6.2" };
//# sourceMappingURL=data:application/json;base64,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